# $NetBSD: web.site.mk,v 1.76 2023/06/06 01:14:10 riastradh Exp $
# FreeBSD: www/share/mk/web.site.mk,v 1.43 2002/08/07 03:17:10 trhodes Exp

#
# Build and install a web site.
#
# The following variables may be set by htdocs Makefiles:
#
# SUBDIR
#	The list of subdirectories that will be processed. Each of these
#	directories must have a Makefile that understands the basic
#	targets like "all", "clean", "cleandir", "install".
#
# HTMLDOCS
#	The list of basenames of *.html files that will be installed
#	directly.
#
# GENDOCS
#	The list of filenames that will be generated in this directory.
#	They are generated in the "all" target, installed in the "install"
#	target and removed in the "clean" target.
#
# XMLDOCS
#	(See web.xml.mk.)
#
# DATA
#	The list of files that should be copied to the destination
#	by the "install" target.
#
# CLEANFILES
#	List of additional files that will be deleted when "make clean"
#	is run.
#
#
# The following targets are provided by this file and are expected to be
# called by the user.
#
# all (default):
#	Generates the web site from the source files.
#
# clean:
#	Removes automatically generated files.
#
# cleandir:
#	Removes automatically generated files, some more than clean.
#
# install:
#	Installs the generated files into DESTDIR.
#	Support for installation is not yet complete.
#
# lint:
#	Runs some consistency checks in the current directory.
#

.MAIN: all

#
# include default settings and /etc/mk.conf first
#
.include "web.prefs.mk"

LOCALBASE?=	/usr/pkg
PREFIX?=	${LOCALBASE}

DESTDIR?=	${HOME}/public_html

WEBOWN?=	${USER}
WEBGRP?=	www
WEBMODE?=	664

AWK?=		/usr/bin/awk
CAT?=		/bin/cat
CP?=		/bin/cp
CVS?=		/usr/bin/cvs
ECHO?=		echo
FGREP?=		${GREP} -F
FIND?=		/usr/bin/find
GREP?=		/usr/bin/grep
LN?=		/bin/ln
M4?=		/usr/bin/m4
MKDIR?=		/bin/mkdir
MV?=		/bin/mv
PERL?=		${PREFIX}/bin/perl
PKG_INFO?=	${exists(${LOCALBASE}/sbin/pkg_info):?${PREFIX}/sbin/pkg_info:/usr/sbin/pkg_info}
PRINTF?=	/usr/bin/printf
RM?=		/bin/rm
SED?=		/usr/bin/sed
SETENV?=	/usr/bin/env
SH?=		/bin/sh
SORT?=		/usr/bin/sort
TOUCH?=		/usr/bin/touch

HTML2TXT?=	${PREFIX}/bin/lynx
HTML2TXTOPTS?=	-force_html -dump -nolist ${HTML2TXTFLAGS}
ISPELL?=	ispell
ISPELLOPTS?=	-l -p ${WEB_PREFIX}/en/share/dict/words ${ISPELLFLAGS}

# for docbook-website DTD
#
# Note: must be before web.xml.mk.
#
AUTOLAYOUTFILE?=${WEB_PREFIX}/autolayout.xml
LAYOUTFILE?=	${WEB_PREFIX}/layout.xml

#
# Include the transformation rules.
#
.include "${.PARSEDIR}/web.xml.mk"

lint: ${VALIDATE_DOCS}

#
# Install dirs derived from the above.
#

DOCINSTALLDIR=	${DESTDIR}/${HTDOCS_SUBDIR}

# A flag to the install(1) command.
COPY=	-c

##################################################################
# Transformation rules

.html.txt:
	@${ECHO} "[html2txt] ${.IMPSRC} -> ${.TARGET}"
	${RUN} ${HTML2TXT} ${HTML2TXTOPTS} ${.IMPSRC} > ${.TARGET}

#
# HTML files are just copied.
#
DATA+=		${HTMLDOCS:=.html}

##################################################################
# Targets

#
# If no target is specified, .MAIN is made
#
.MAIN: all

#
# Build most everything
#
all: check-errors pre-all ${GENDOCS} ${DATA} _PROGSUBDIR

#
# Make sure autolayout.xml is up-to-date first
#
pre-all: ${AUTOLAYOUTFILE} ${WEB_PREFIX}/docs/Hardware/Machines/DEC/vax

.if defined(AUTOLAYOUTDIR)
autolayout.xml: ${AUTOLAYOUTFILE} ${LAYOUTFILE}
.endif

${AUTOLAYOUTFILE}: ${LAYOUTFILE}
.if defined(AUTOLAYOUTDIR)
	@(cd docs/Hardware/Machines/DEC/vax && ${MAKE} full.xml)
	@${ECHO} "[xsltproc] layout.xml -> autolayout.xml"
	${RUN} ${XSLTPROC} --nonet -o ${.TARGET}.tmp \
		${WEB_PREFIX}/share/xsl/autolayout.xsl \
		${LAYOUTFILE}
	${RUN} ${XMLLINT} --xinclude --nonet --noent --nsclean --noxincludenode \
		${.TARGET}.tmp \
		| ${GREP} -v '!ENTITY' > ${.TARGET} \
		|| (${RM} -f ${.TARGET}.tmp && false)
.  if !defined(DEBUG)
	${RUN} ${RM} -f ${.TARGET}.tmp
.  endif

CLEANFILES+=	${AUTOLAYOUTFILE} ${AUTOLAYOUTFILE}.tmp
.else
	@${ECHO} "htdocs/layout.xml has changed - rebuilding htdocs/autolayout.xml..."
	${RUN} cd ${WEB_PREFIX} && ${MAKE} autolayout.xml
.endif

spellcheck:
.for _entry in ${GENDOCS}
	@echo "Spellcheck ${_entry}"
	${RUN} ${HTML2TXT} ${HTML2TXTOPTS} ${.CURDIR}/${_entry} | ${ISPELL} ${ISPELLOPTS}
.endfor

#
# Clean things up
#
clean: _PROGSUBDIR do-clean
cleandir: _PROGSUBDIR do-clean

.PHONY: do-clean
do-clean:
	${RM} -f Errs errs mklog ${GENDOCS} ${CLEANFILES}

#
# Installing a website to another directory.
#

# Commands to install files.
INSTALL_WEB?=	\
	${INSTALL} ${COPY} ${INSTALLFLAGS} -o ${WEBOWN} -g ${WEBGRP} -m ${WEBMODE}

_ALLINSTALL+=	${GENDOCS} ${DATA}

.if !target(do-install)
do-install: ${_ALLINSTALL}
.  if !empty(_ALLINSTALL)
	${MKDIR} -p ${DOCINSTALLDIR}
.    for entry in ${_ALLINSTALL}
.      if exists(${.CURDIR}/${entry})
	${INSTALL_WEB} ${.CURDIR}/${entry} ${DOCINSTALLDIR}
.      else
	${INSTALL_WEB} ${entry} ${DOCINSTALLDIR}
.      endif
.    endfor
.    if defined(INDEXLINK) && !empty(INDEXLINK)
	cd ${DOCINSTALLDIR}; ${LN} -fs ${INDEXLINK} index.html
.    endif
.  endif
.endif

install: _PROGSUBDIR
install: do-install

#
# This recursively calls make in subdirectories.
#
_PROGSUBDIR: .USE
.if defined(SUBDIR) && !empty(SUBDIR:M*)
.  for entry in ${SUBDIR}
	@${ECHO} "===> Making ${.TARGET} in ${DIRPRFX}${entry}"
	${RUN} cd ${.CURDIR}/${entry};					\
	${MAKE} ${.TARGET} DIRPRFX=${DIRPRFX}${entry}/
.  endfor
.endif

.include <bsd.obj.mk>

ERRORS?=	# none
WARNINGS?=	# none
.PHONY: check-errors

check-errors:
.if !empty(ERRORS:M*)
	@${PRINTF} "ERROR: %s\\n" ${ERRORS}
.endif
.if !empty(WARNINGS:M*)
	@${PRINTF} "WARNING: %s\\n" ${WARNINGS}
.endif
.if !empty(ERRORS:M*)
	@false
.endif

.PHONY: show-var
show-var:
	@${ECHO} ${VARNAME}=${${VARNAME}:Q}

# THE END
