/*
 * Copyright (c) 2005 Jacob Meuser <jakemsr@jakemsr.com>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/*
 *  $Id: bsdavplay.c,v 1.21 2006/03/30 09:21:26 jakemsr Exp $
 */

#include "includes.h"

#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/mman.h>
#include <sys/shm.h>

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/extensions/XShm.h>
#include <X11/extensions/Xvlib.h>
#include <X11/Xatom.h>

#include "bsdav.h"


struct video {
	Display		*dpy;
	Window		 window;
	Window		 rootwindow;
	XEvent		 event;
	GC		 gc;
	Atom		 wmdelwin;
	XWindowAttributes *norm_winatt;
	XvPortID	 port;
	XvImage		*xv_image;
	XShmSegmentInfo	 shminfo;
	char		*pixels;
	unsigned int	 adaptor;
	int		 display_width;
	int		 display_height;
	int		 screen_id;
	int		 format;
	int		 norm;
	int		 width;
	int		 height;
	struct bsdav_ratio aspect;
	struct bsdav_ratio fps;
	size_t		 buffer_size;
	uint8_t		*buffer;
	int		 full_screen;
	int		 aspect_lock;
	long		 frame_count;
};

struct audio {
	char		 dev[FILENAME_MAX];
	int	 	 fd;
	int	 	 format;
	int		 channels;
	long		 sample_rate;
	int		 samples_per_frame;
	size_t		 buffer_size;
	uint8_t		*buffer;
	long		 frame_count;
};

struct bsdavplay {
	struct bsdav_ringbuf *vid_rb;
	struct video	*vid;
	struct audio	*aud;
	struct timeval	 start_ts;
	char		 in_file[FILENAME_MAX];
	FILE		*in_fd;
	int		 shutdown;
	int		 verbose;
	int		 pause;
	int		 do_audio;
	int		 do_video;
};


extern char *__progname;
extern int errno;

/* getopt externs */
extern char *optarg;
extern int opterr;
extern int optind;
extern int optopt;
extern int optreset;

int open_files(struct bsdavplay *);
int init_buffers(struct bsdavplay *);
int play_audio_buf(struct bsdavplay *);
int stream(struct bsdavplay *);
int set_video_format(struct bsdavplay *);
int create_win(struct bsdavplay *);
int play_video_buf(struct bsdavplay *, int);
int display_event(struct bsdavplay *);
int resize_window(struct bsdavplay *, int);

void cleanup(struct bsdavplay *);
void usage(void);

void list_xv_enc(struct bsdavplay *);


void
usage(void)
{
extern char *__progname;

	fprintf(stderr,
		"Usage: %s [-AVlv] [-a vaspect] [-b buffers] [-C achans] [-e vformat]\n"
		"	   [-E aformat] [-f afile] [-h vheight] [-i input] [-n vnorm]\n"
		"          [-r vrate] [-R arate] [-w vwidth]\n",
		__progname);
	return;
}


void
list_xv_enc(struct bsdavplay *avp)
{
XvImageFormatValues *xvformats;
XvAdaptorInfo *ainfo;
unsigned int p;
int i, j, k;
int num_xvformats;
int num_adaptors;


	if ((avp->vid->dpy = XOpenDisplay(NULL)) == NULL) {
		warnx("cannot open display %s", XDisplayName(NULL));
		return;
	}

	if (Success != XvQueryExtension(avp->vid->dpy, &p, &p, &p, &p, &p)) {
		warnx("Xv not available");
		return;
	}
	avp->vid->screen_id = DefaultScreen(avp->vid->dpy);

	avp->vid->rootwindow = DefaultRootWindow(avp->vid->dpy);

	if (Success != XvQueryAdaptors(avp->vid->dpy, avp->vid->rootwindow,
	    &num_adaptors, &ainfo)) {
		warnx("no Xv adaptors present");
		return;
	}

	printf("Video Formats\n");
	for (i = 0; i < num_adaptors; i++) {
		if ((ainfo[i].type & XvInputMask) &&
		    (ainfo[i].type & XvImageMask)) {
			printf("  Xv Adaptor %d: %s\n", i, ainfo[i].name);
			printf("    %-12s  %16s\n", "name", "bits per pixel");
			xvformats = XvListImageFormats(avp->vid->dpy,
			    ainfo[i].base_id, &num_xvformats);
			for (j = 0; j < num_xvformats; j++)
				for (k = 0; bsdav_vid_fmts[k].name; k++)
					if (xvformats[j].id ==
					    bsdav_vid_fmts[k].xv_id)
						printf("    %12s  %16d\n",
						    bsdav_vid_fmts[k].name,
						    bsdav_vid_fmts[k].bpp);
			if (xvformats != NULL)
				XFree(xvformats);
		}
	}
	XvFreeAdaptorInfo(ainfo);

	return;
}


int
open_files(struct bsdavplay *avp)
{
unsigned int p;

	if (avp->verbose > 1)
		warnx("opening input: %s", avp->in_file);

	if ((avp->in_fd = fopen(avp->in_file, "r")) == NULL) {
		warn("%s", avp->in_file);
		return(1);
	}

	if (avp->do_video == 1) {
		if ((avp->vid->dpy = XOpenDisplay(NULL)) == NULL) {
			warnx("cannot open display %s", XDisplayName(NULL));
			return(1);
		}

		if (Success != XvQueryExtension(avp->vid->dpy, &p, &p, &p,
		    &p, &p)) {
			warnx("Xv not available");
			return(1);
		}
		avp->vid->screen_id = DefaultScreen(avp->vid->dpy);
	}

	if (avp->do_audio == 1) {
		if ((avp->aud->fd = open(avp->aud->dev, O_WRONLY)) < 0) {
			warnx("could not open %s", avp->aud->dev);
			close(avp->aud->fd);
			return(1);
		}
	}

	return(0);
}


int
set_video_format(struct bsdavplay *avp)
{
XvImageFormatValues *xvformats;
XvAdaptorInfo *ainfo;
int i, j, k;
int num_xvformats;
int num_adaptors;


	if (avp->vid->width == 0)
		avp->vid->width = bsdav_vid_nrms[avp->vid->norm].width;
	if (avp->vid->height == 0)
		avp->vid->height = bsdav_vid_nrms[avp->vid->norm].height;

	avp->vid->rootwindow = DefaultRootWindow(avp->vid->dpy);

	if (Success != XvQueryAdaptors(avp->vid->dpy, avp->vid->rootwindow,
	    &num_adaptors, &ainfo)) {
		warnx("no Xv adaptors present");
		return(1);
	}

	for (i = 0; i < num_adaptors && !avp->vid->port; i++) {
		if ((ainfo[i].type & XvInputMask) &&
		    (ainfo[i].type & XvImageMask)) {
			if (avp->verbose > 1)
				warnx("adaptor %d can PutImage", i);
			xvformats = XvListImageFormats(avp->vid->dpy,
			    ainfo[i].base_id, &num_xvformats);
			for (j = 0; j < num_xvformats; j++) {
				if (xvformats[j].id ==
				    bsdav_vid_fmts[avp->vid->format].xv_id) {
					avp->vid->adaptor = i;
					break;
				}
			}
			if (xvformats != NULL)
				XFree(xvformats);
		}
		for (k = 0; k < ainfo[i].num_ports; ++k) {
			if (Success == XvGrabPort(avp->vid->dpy,
			    ainfo[i].base_id, CurrentTime)) {
				avp->vid->port = ainfo[i].base_id;
				if (avp->verbose > 1)
					warnx("grabbed Xv port %ld",
					    avp->vid->port);
				break;
			}
		}
	}

	XvFreeAdaptorInfo(ainfo);

	if (avp->vid->port == 0) {
		warnx("could not find valid Xv port");
		return(1);
	}

	return(0);
}


int
create_win(struct bsdavplay * avp)
{
XGCValues values;
XTextProperty WinName;
XSizeHints szhints;
XWMHints wmhints;
char *name;
extern char *__progname;

	name = __progname;
	XStringListToTextProperty(&name, 1, &WinName);

	szhints.flags = PSize | PMaxSize | PMinSize;
	szhints.width = avp->vid->display_width;
	szhints.height = avp->vid->display_height;
	szhints.max_width = 2048;  /* get this from xv_init + DisplayWidth */
	szhints.max_height = 2048;  /* get this from xv_init + DisplayHeight*/
	szhints.min_width = 160;
	szhints.min_height = 120;

	wmhints.flags = InputHint | StateHint;
	wmhints.input = True;
	wmhints.initial_state = NormalState;

	if (avp->vid->full_screen == 1) {
		avp->vid->display_width = DisplayWidth(avp->vid->dpy,
		    avp->vid->screen_id);
		avp->vid->display_height = DisplayHeight(avp->vid->dpy,
		    avp->vid->screen_id);
	} else {
		avp->vid->display_width = avp->vid->width;
		avp->vid->display_height = avp->vid->height;
	}

	avp->vid->window = XCreateSimpleWindow(avp->vid->dpy,
	    avp->vid->rootwindow, 0, 0, avp->vid->display_width,
	    avp->vid->display_height, 0,
	    XWhitePixel(avp->vid->dpy, avp->vid->screen_id),
	    XBlackPixel(avp->vid->dpy, avp->vid->screen_id));

	XSetWMProperties(avp->vid->dpy, avp->vid->window, &WinName, &WinName,
	    NULL, 0, &szhints, &wmhints, NULL);

	XSelectInput(avp->vid->dpy, avp->vid->window,
	    KeyPressMask | ButtonPressMask | StructureNotifyMask);

	avp->vid->wmdelwin = XInternAtom(avp->vid->dpy, "WM_DELETE_WINDOW",
	    False);
	XSetWMProtocols(avp->vid->dpy, avp->vid->window, &avp->vid->wmdelwin,
	    1);

	XMapRaised(avp->vid->dpy, avp->vid->window);

	avp->vid->gc = XCreateGC(avp->vid->dpy, avp->vid->window, 0, &values);

	avp->vid->xv_image = XvShmCreateImage(avp->vid->dpy, avp->vid->port,
	    bsdav_vid_fmts[avp->vid->format].xv_id, avp->vid->pixels,
	    avp->vid->width, avp->vid->height, &avp->vid->shminfo);

	avp->vid->shminfo.shmid = shmget(IPC_PRIVATE, avp->vid->buffer_size,
	    IPC_CREAT | 0777);

	if (avp->vid->shminfo.shmid < 0) {
		warn("shmget");
		return(1);
	}
	avp->vid->xv_image->data = avp->vid->pixels =
	    avp->vid->shminfo.shmaddr = shmat(avp->vid->shminfo.shmid, 0, 0);
	if (avp->vid->shminfo.shmaddr == (void *)-1) {
		warn("shmat");
		return(1);
	}
	XShmAttach(avp->vid->dpy, &avp->vid->shminfo);

	XSync(avp->vid->dpy, False);

	return(0);
}


int
init_buffers(struct bsdavplay *avp)
{

	if (avp->do_video == 1) {
		if ((avp->vid->buffer =
		    malloc(avp->vid->buffer_size)) == NULL) {
			warn("avp->vid->buffer, %llu", (unsigned long long)avp->vid->buffer_size);
			return(1);
		}

		if ((avp->vid->norm_winatt =
		    malloc(sizeof(XWindowAttributes))) == NULL) {
			warn("avp->vid->norm_winatt");
			return(1);
		}

		if (bsdav_init_ringbuf(avp->vid_rb, avp->vid->buffer_size) != 0) {
			warnx("could not allocate memory for ringbuffer");
			return(1);
		}

	}

	if (avp->do_audio == 1) {
		avp->aud->samples_per_frame = avp->aud->buffer_size /
		    avp->aud->channels /
		    (bsdav_aud_fmts[avp->aud->format].bps / 8);
	}

	if ((avp->aud->buffer = malloc(avp->aud->buffer_size)) == NULL) {
		warn("avp->aud->buffer");
		return(1);
	}


	return(0);
}


int
play_video_buf(struct bsdavplay *avp, int buf)
{
	memcpy(avp->vid->pixels, avp->vid_rb->bufs[buf].buf,
	    avp->vid_rb->bufs[buf].size);

	XvShmPutImage(avp->vid->dpy, avp->vid->port, avp->vid->window,
	    avp->vid->gc, avp->vid->xv_image, 0, 0, avp->vid->width,
	    avp->vid->height, 0, 0, avp->vid->display_width,
	    avp->vid->display_height, True);

	avp->vid_rb->bufs[buf].size = 0;

	avp->vid_rb->bufs_off--;

	return(0);
}


int
play_audio_buf(struct bsdavplay *avp)
{
size_t	left;
off_t	offset;
ssize_t	wrote;

	for (left = avp->aud->buffer_size, offset = 0; left > 0;) {
		wrote = write(avp->aud->fd, avp->aud->buffer + offset, left);
		if (wrote == 0) {
			if (avp->verbose > 1)
				warnx("audio play: wrote == 0");
		}
		if (wrote < 0) {
			if (errno == EINTR) {
				wrote = 0;
				if (avp->verbose > 1)
					warn("audio play");
			} else if (errno == EAGAIN) {
				wrote = 0;
				if (avp->verbose > 1)
					warn("audio play blocked");
			} else {
				warn("audio play");
				return(1);
			}
		}
		if (wrote > left) {
			warnx("audio: write returns more bytes than requested");
			warnx("audio: requested: %llu, returned: %lld",
			    (unsigned long long)left, (long long)wrote);
			return(1);
		}
		offset += wrote;
		left -= wrote;
	}
	return(left);
}


int
stream(struct bsdavplay *avp)
{
struct bsdav_frame_header frmhdr;
struct timeval now_tv;
struct timeval run_tv;
struct timeval start_tv;
struct timeval end_tv;
struct timeval diff_tv;
struct timeval frame_ts;
struct timeval sleep_tv;
struct timeval aud_fill_tv;
struct timeval aud_fill_time;
long	 aud_fill_usecs;
double	 run_time;
double	 aud_fps;
long	 sleep_time;
size_t	 hw_buf_size;
int	 count;
int	 sequence;
int	 fill_bufs;
int	 rolling;

	avp->pause = 0;

	avp->vid->frame_count = 0;
	avp->aud->frame_count = 0;
	sequence = 200;
	count = 0;
	sleep_time = 0;
	fill_bufs = 0;
	rolling = 0;
	avp->vid_rb->buf_in = -1;
	avp->vid_rb->buf_out = 0;


	if (avp->do_audio == 1) {
		hw_buf_size = bsdav_get_audhw_buf_size(avp->aud->fd,
		    BSDAV_AUDMODE_PLAY);
		fill_bufs = hw_buf_size / avp->aud->buffer_size / 2;

		if (avp->verbose > 1)
			warnx("fill_bufs = %d", fill_bufs);

		aud_fps = avp->aud->sample_rate *
		    bsdav_aud_fmts[avp->aud->format].bps / 8 *
		    avp->aud->channels / avp->aud->buffer_size;
		aud_fill_usecs = 1000000 / aud_fps * fill_bufs;
		aud_fill_tv.tv_sec = 0;
		aud_fill_tv.tv_usec = aud_fill_usecs;

		timeradd(&avp->start_ts, &aud_fill_tv, &aud_fill_time);

		if (avp->verbose > 2) {
			warnx("aud_fill_usecs = %ld", aud_fill_usecs);
			warnx("aud_fill_time.tv_sec = %ld", aud_fill_time.tv_sec);
			warnx("aud_fill_time.tv_usec = %ld", aud_fill_time.tv_usec);
		}
	} else {
		aud_fill_usecs = 50000;
		rolling = 1;
	}

	gettimeofday(&start_tv, NULL);
	timersub(&start_tv, &avp->start_ts, &diff_tv);

	while (avp->shutdown == 0) {

		count++;

		if (avp->do_video == 1)
			display_event(avp);

		while ((avp->pause == 1) && (avp->shutdown == 0)) {
			usleep(50000);
			if (avp->do_video == 1)
				display_event(avp);
		}

		if (avp->shutdown == 1)
			break;

		if (bsdav_read_frame_header(avp->in_fd, &frmhdr) != 0) {
			if (feof(avp->in_fd))
				warnx("EOF   ");
			else
				warnx("bsdav_read_frame_header failed   ");
			break;
		}

		if (avp->verbose > 3)
			bsdav_dump_frame_header(&frmhdr);

		switch (frmhdr.type) {
		case BSDAV_FRMTYP_VIDEO:
			if (bsdav_read_frame_data(avp->in_fd,
			    avp->vid->buffer, (size_t)frmhdr.size,
			    avp->do_video == 1 ? 0 : 1) != 0) {
				if (feof(avp->in_fd))
					warnx("unexpected EOF!   ");
				else
					warnx("video read error   ");
				avp->shutdown = 1;
				break;
			}
			if (avp->do_video == 0)
				break;
			frame_ts.tv_sec = frmhdr.tssec;
			frame_ts.tv_usec = frmhdr.tsusec;
			if (bsdav_rb_buf_in(avp->vid_rb, avp->vid->buffer,
			    (size_t)frmhdr.size, frame_ts) != 0) {
				warnx("failed adding video frame to rb   ");
				avp->shutdown = 1;
				break;
			}
			/* put some data in the audio hardware buffer */
			/* before waiting on time stamps */
			if (rolling == 0) {
				if (timercmp(&frame_ts, &aud_fill_time, <) == 0) {
					avp->vid_rb->bufs_off =
					    avp->vid_rb->buf_in;
					if (avp->verbose > 1)
						warnx("bufs_off = %d   ",
						    avp->vid_rb->bufs_off);
					rolling = 1;
				}
			}
			avp->vid_rb->buf_out = avp->vid_rb->buf_in - avp->vid_rb->bufs_off;
			if (avp->vid_rb->buf_out < 0)
				avp->vid_rb->buf_out += avp->vid_rb->num_bufs;
			break;

		case BSDAV_FRMTYP_AUDIO:
			if (bsdav_read_frame_data(avp->in_fd,
			    avp->aud->buffer, (size_t)frmhdr.size, 0) != 0) {
				if (feof(avp->in_fd))
					warnx("unexpected EOF!   ");
				else
					warnx("audio read error   ");
				avp->shutdown = 1;
				break;
			}
			break;

		case BSDAV_FRMTYP_LAST:
			avp->shutdown = 1;
			break;

		default:
			warnx("invalid frame type   ");
			avp->shutdown = 1;
			break;
		}

		if (avp->shutdown == 1)
			break;

		sleep_time = 0;
		if ((rolling == 1) && (avp->do_video == 1) &&
		    (avp->vid_rb->bufs[avp->vid_rb->buf_out].size > 0)) {
			timeradd(&avp->vid_rb->bufs[avp->vid_rb->buf_out].ts,
			    &diff_tv, &frame_ts);
			gettimeofday(&now_tv, NULL);
			timersub(&frame_ts, &now_tv, &sleep_tv);
			sleep_time = sleep_tv.tv_sec * 1000000 +
			    sleep_tv.tv_usec;
		}

		if (sleep_time > 0) {
			if (sleep_time > 30000)
				if (avp->verbose > 1)
					warnx("sleep_time = %ld   ",
					   sleep_time);
			usleep((useconds_t)sleep_time);
		} else if (sleep_time < 0) {
			if (-(sleep_time) > aud_fill_usecs) {
				if (avp->verbose > 1)
				 	warnx("%ld usecs late!   ",
					    -(sleep_time));
				/* pause, etc */
				diff_tv.tv_usec -= sleep_time;
				/* let audio catch up as well */
				diff_tv.tv_usec -= aud_fill_usecs / 2;
			}
		}

		if ((avp->do_video == 1) && (rolling == 1) &&
		    (avp->vid_rb->bufs[avp->vid_rb->buf_out].size > 0)) {
			if (play_video_buf(avp, avp->vid_rb->buf_out) != 0) {
				warnx("video play error   ");
				avp->shutdown = 1;
			} else
				avp->vid->frame_count++;
		}

		if ((avp->do_audio == 1) &&
		    (frmhdr.type == BSDAV_FRMTYP_AUDIO)) {
			if (play_audio_buf(avp) != 0) {
				warnx("audio play error   ");
				avp->shutdown = 1;
			} else
				avp->aud->frame_count++;
		}

		if (avp->shutdown == 1)
			break;

		if ((count == sequence) && (avp->verbose > 1)) {
			count = 0;
			gettimeofday(&now_tv, NULL);
			timersub(&now_tv, &diff_tv, &end_tv);
			timersub(&end_tv, &avp->start_ts, &run_tv);
			run_time = run_tv.tv_sec + (double)run_tv.tv_usec /
			    1000000;
			fprintf(stderr, "%s: sec: %08.3f,", __progname,
			    run_time);
			if (avp->do_video == 1)
				fprintf(stderr, " vf: %06ld, fps: %0.3f,",
				    avp->vid->frame_count,
				    ((double)avp->vid->frame_count) /
				    run_time);
			if (avp->do_audio == 1)
				fprintf(stderr,
				    " as: %09ld, srate: %ld",
		    		    avp->aud->frame_count *
				    avp->aud->samples_per_frame,
				    (long)((avp->aud->frame_count - fill_bufs) *
				    avp->aud->samples_per_frame / run_time));
			fprintf(stderr, "\r");
			fflush(stderr);
		}
	}

	/*  last character was probably a carriage return */
	if (avp->verbose > 1)
		fprintf(stderr, "\n");

	gettimeofday(&now_tv, NULL);
	timersub(&now_tv, &diff_tv, &end_tv);
	timersub(&end_tv, &avp->start_ts, &run_tv);
	run_time = run_tv.tv_sec + (double)run_tv.tv_usec / 1000000;

	if (avp->verbose > 0) {
		warnx("total time: %f", run_time);
		if (avp->do_video == 1) {
			warnx("video: frames: %ld", avp->vid->frame_count);
			warnx("video: average fps: %f",
			    ((double)avp->vid->frame_count) / run_time);
		}
		if (avp->do_audio == 1) {
			warnx("audio: samples: %ld", avp->aud->frame_count *
			    avp->aud->samples_per_frame);
			warnx("audio: average rate: %f Hz",
			    ((avp->aud->frame_count - fill_bufs) *
			    avp->aud->samples_per_frame) / run_time);
		}
	}

	return(0);
}


int
display_event(struct bsdavplay *avp)
{
char str;

	if (XPending(avp->vid->dpy) > 0) {
		XNextEvent(avp->vid->dpy, &avp->vid->event);
		switch (avp->vid->event.type) {
		case KeyPress:
			if (avp->verbose > 1)
				warnx("got keypress event");
			XLookupString(&avp->vid->event.xkey, &str, 1, NULL,
			    NULL);
			switch (str) {
			case 'a':
				if (avp->vid->aspect_lock == 1) {
					avp->vid->aspect_lock = 0;
				} else {
					avp->vid->aspect_lock = 1;
					resize_window(avp, 0);
				}
				break;
			case 'f':
				resize_window(avp, 1);
				break;
			case 'p':
				if (avp->pause == 1)
					avp->pause = 0;
				else
					avp->pause = 1;
				break;
			case 'q':
				avp->shutdown = 1;
				break;
			case '+':
				if (avp->vid_rb->bufs_off >=
				    avp->vid_rb->num_bufs - 1)
					warnx("cannot buffer any more frames");
				else
					avp->vid_rb->bufs_off++;
				if (avp->verbose > 1)
					warnx("buffering %d frames",
					    avp->vid_rb->bufs_off);
				break;
			case '-':
				if (avp->vid_rb->bufs_off == 0)
					warnx("cannot buffer any fewer frames");
				else
					avp->vid_rb->bufs_off--;
				if (avp->verbose > 1)
					warnx("buffering %d frames",
					    avp->vid_rb->bufs_off);
				break;
			default:
				break;
			}
			break;
		case ClientMessage:
			if (avp->verbose > 1)
				warnx("got ClientMessage event");
			if (avp->vid->event.xclient.data.l[0] ==
			    avp->vid->wmdelwin) {
				avp->shutdown = 1;
				break;
			}
			break;
		case ConfigureNotify:
			if (avp->verbose > 1)
				warnx("got ConigureNotify event");
			resize_window(avp, 0);
			break;
		default:
			break;
		}
	}
	XSync(avp->vid->dpy, False);

	return(0);
}


int
resize_window(struct bsdavplay *avp, int fullscreen)
{
XWindowAttributes *winatt;
double aspect;
int new_width;
int new_height;
int new_x;
int new_y;

	aspect = (double)avp->vid->aspect.n / avp->vid->aspect.d;

	if (fullscreen == 1) {
		if (avp->vid->full_screen == 1) {
			new_width = avp->vid->norm_winatt->width;
			new_height = avp->vid->norm_winatt->height;
			new_x = avp->vid->norm_winatt->x;
			new_y = avp->vid->norm_winatt->y;
			avp->vid->full_screen = 0;
		} else {
			XGetWindowAttributes(avp->vid->dpy, avp->vid->window,
			    avp->vid->norm_winatt);
			new_width = DisplayWidth(avp->vid->dpy,
			    avp->vid->screen_id);
			new_height = DisplayHeight(avp->vid->dpy,
			    avp->vid->screen_id);
			new_x = 0;
			new_y = 0;
			if (avp->vid->aspect_lock == 1) {
				if (new_width <= new_height * aspect) {
					new_width -= new_width % 16;
					new_height = new_width / aspect;
				} else {
					new_height -= new_height % 16;
					new_width = new_height * aspect;
				}
			}
			avp->vid->full_screen = 1;
		}
		XMoveResizeWindow(avp->vid->dpy, avp->vid->window, new_x,
		    new_y, new_width, new_height);
	} else {
		if ((winatt = malloc(sizeof(XWindowAttributes))) == NULL) {
			warn("winatt");
			return(1);
		}
		XGetWindowAttributes(avp->vid->dpy, avp->vid->window, winatt);
		new_x = winatt->x;
		new_y = winatt->y;
		new_width = winatt->width;
		new_height = winatt->height;
		free(winatt);
		if (avp->vid->aspect_lock == 1) {
			new_width = (new_width + (new_height * aspect)) / 2;
			new_width -= new_width % 16;
			new_height = new_width / aspect;
		}
		XResizeWindow(avp->vid->dpy, avp->vid->window, new_width,
		    new_height);
	}
	XSync(avp->vid->dpy, False);
	avp->vid->display_width = new_width;
	avp->vid->display_height = new_height;
	XNextEvent(avp->vid->dpy, &avp->vid->event);
	XSync(avp->vid->dpy, True);

	return(0);
}


void
cleanup(struct bsdavplay *avp)
{
	if (avp->in_fd != NULL)
		fclose(avp->in_fd);
	avp->in_fd = NULL;

	if (avp->vid->shminfo.shmaddr != NULL)
		shmdt(avp->vid->shminfo.shmaddr);
	avp->vid->shminfo.shmaddr = NULL;

	if (avp->vid->shminfo.shmid > 0)
		shmctl(avp->vid->shminfo.shmid, IPC_RMID, 0);
	avp->vid->shminfo.shmid = 0;

	if (avp->vid->xv_image != NULL)
		XFree(avp->vid->xv_image);
	avp->vid->xv_image = NULL;

	if (avp->vid->gc != NULL)
		XFreeGC(avp->vid->dpy, avp->vid->gc);
	avp->vid->gc = NULL;

	if (avp->vid->window != 0)
		XDestroyWindow(avp->vid->dpy, avp->vid->window);
	avp->vid->window = 0;

	if (avp->vid->port != 0)
		XvUngrabPort(avp->vid->dpy, avp->vid->port, CurrentTime);
	avp->vid->port = 0;

	if (avp->vid->dpy != NULL)
		XCloseDisplay(avp->vid->dpy);
	avp->vid->dpy = NULL;

	if (avp->vid_rb != NULL)
		bsdav_free_ringbuf(avp->vid_rb);

}


int
main(int argc, char *argv[])
{
struct bsdav_stream_header strhdr;
struct bsdavplay *avp;
const char *errstr;
int	 ch;
int	 sret;
int	 err;
int	 lflag;

	if ((avp = malloc(sizeof(struct bsdavplay))) == NULL) {
		warn("avp");
		cleanup(avp);
		exit(1);
	}
	memset(avp, 0, sizeof(struct bsdavplay));

	if ((avp->vid = malloc(sizeof(struct video))) == NULL) {
		warn("avp->vid");
		cleanup(avp);
		exit(1);
	}
	memset(avp->vid, 0, sizeof(struct video));

	if ((avp->aud = malloc(sizeof(struct audio))) == NULL) {
		warn("avp->aud");
		cleanup(avp);
		exit(1);
	}
	memset(avp->aud, 0, sizeof(struct audio));

	if ((avp->vid_rb = malloc(sizeof(struct bsdav_ringbuf))) == NULL) {
		warn("avp->vid_rb");
		cleanup(avp);
		exit(1);
	}
	memset(avp->vid_rb, 0, sizeof(struct bsdav_ringbuf));

	/* defaults */
	avp->vid->norm = BSDAV_VIDNORM_NONE;
	avp->vid->format = BSDAV_VIDFMT_NONE;
	avp->vid->width = 0;
	avp->vid->height = 0;
	snprintf(avp->aud->dev, FILENAME_MAX, "/dev/audio");
	avp->aud->fd = -1;	
	avp->aud->format = BSDAV_AUDFMT_NONE;
	avp->aud->channels = 0;
	avp->aud->sample_rate = 0;

	avp->vid->aspect.n = 4;
	avp->vid->aspect.d = 3;
	avp->vid->aspect_lock = 1;

	avp->in_fd = NULL;
	avp->shutdown = 0;
	avp->do_audio = 1;
	avp->do_video = 1;
	avp->verbose = 0;

	avp->vid_rb->num_bufs = 10;
	avp->vid_rb->bufs_off = 0;

	lflag = 0;
	err = 0;

	while ((ch = getopt(argc, argv, "AVlva:b:C:e:E:f:h:i:n:r:R:w:")) != -1) {
		switch (ch) {
		case 'A':
			avp->do_audio = 0;
			break;
		case 'a':
			if (bsdav_parse_ratio(optarg, &avp->vid->aspect) != 0) {
				warnx("invalid vaspect: %s", optarg);
				err++;
			} else
				if (avp->vid->aspect.n == 0)
					avp->vid->aspect_lock = 0;
			break;
		case 'b':
			avp->vid_rb->num_bufs = (int)strtonum(optarg, 1, 20,
			    &errstr);
			if (errstr != NULL) {
				warnx("number of buffers is %s: %s:", errstr, optarg);
				err++;
			}
			break;
		case 'C':
			avp->aud->channels = (int)strtonum(optarg, 1, 2, &errstr);
			if (errstr != NULL) {
				warnx("achans is %s: %s:", errstr, optarg);
				err++;
			}
			break;
		case 'e':
			avp->vid->format = bsdav_find_vid_fmt(optarg);
			if (avp->vid->format < 0) {
				warnx("invalid video encoding: %s", optarg);
				err++;
			}
			break;
		case 'E':
			avp->aud->format = bsdav_find_aud_fmt(optarg);
			if (avp->aud->format < 0) {
				warnx("invalid audio encoding: %s", optarg);
				err++;
			}
			break;
		case 'f':
			sret = snprintf(avp->aud->dev, FILENAME_MAX, optarg);
			if (sret >= FILENAME_MAX) {
				warnx("file name is too long");
				err++;
			}
			break;
		case 'h':
			avp->vid->height = (int)strtonum(optarg, 1, UINT_MAX,
			    &errstr);
			if (errstr != NULL) {
				warnx("vheight is %s: %s", errstr, optarg);
				err++;
			}
			break;
		case 'i':
			sret = snprintf(avp->in_file, FILENAME_MAX,
			    optarg);
			if (sret >= FILENAME_MAX) {
				warnx("input file name is too long");
				err++;
			}
			break;
		case 'l':
			lflag++;
			break;
		case 'n':
			avp->vid->norm = bsdav_find_vid_norm(optarg);
			if (avp->vid->norm < 0) {
				warnx("invalid video norm: %s", optarg);
				err++;
			}
			break;
		case 'r':
			if (bsdav_parse_ratio(optarg, &avp->vid->fps) != 0) {
				warnx("invalid vrate: %s", optarg);
				err++;
			}
			break;
		case 'R':
			avp->aud->sample_rate = (int)strtonum(optarg, 0, 96000, &errstr);
			if (errstr != NULL) {
				warnx("arate is %s: %s", errstr, optarg);
				err++;
			}
			break;
		case 'V':
			avp->do_video = 0;
			break;
		case 'v':
			avp->verbose++;
			break;
		case 'w':
			avp->vid->width = (int)strtonum(optarg, 1, UINT_MAX,
			    &errstr);
			if (errstr != NULL) {
				warnx("vwidth is %s: %s", errstr, optarg);
				err++;
			}
			break;
		default:
			err++;
			break;
		}
		if (err > 0)
			break;
	}
	argc -= optind;
	argv += optind;

	if (err > 0) {
		usage();
		cleanup(avp);
		exit(1);
	}

	if (lflag > 0) {
		printf("\n");
		bsdav_list_audio_formats(avp->aud->dev, avp->aud->fd);
		printf("\n");
		list_xv_enc(avp);
		printf("\n");
		cleanup(avp);
		exit(0);
	} else if ((avp->do_video == 0) && (avp->do_audio == 0)) {
		warnx("playing neither audio nor video, exiting");
		cleanup(avp);
		exit(1);
	}

	if (open_files(avp) != 0) {
		warnx("failed opening files");
		cleanup(avp);
		exit(1);
	}

	if (bsdav_read_stream_header(avp->in_fd, &strhdr) != 0) {
		if (feof(avp->in_fd))
			warnx("EOF");
		else
			warnx("read stream header failed");
		cleanup(avp);
		exit(1);
	}

	avp->start_ts.tv_sec = strhdr.tssec;
	avp->start_ts.tv_usec = strhdr.tsusec;

	if (avp->verbose > 2)
		bsdav_dump_stream_header(&strhdr);

	if (avp->vid->format == BSDAV_VIDFMT_NONE)
		avp->vid->format = strhdr.vidfmt;
	if (avp->vid->width == 0)
		avp->vid->width = strhdr.vidwth;
	if (avp->vid->height == 0)
		avp->vid->height = strhdr.vidhgt;

	if ((avp->vid->format == BSDAV_VIDFMT_NONE) ||
	    (avp->vid->width == 0) || (avp->vid->height == 0))
		avp->do_video = 0;

	if (avp->aud->format == BSDAV_AUDFMT_NONE)
		avp->aud->format = strhdr.audfmt;
	if (avp->aud->channels == 0)
		avp->aud->channels = strhdr.audchn;
	if (avp->aud->sample_rate == 0)
		avp->aud->sample_rate = strhdr.audsrt;

	if (avp->do_video == 1) {
		avp->vid->buffer_size = strhdr.vidmfs;
		if (set_video_format(avp)) {
			warnx("failed to set video format");
			cleanup(avp);
			exit(1);
		}
	}

	avp->aud->buffer_size = strhdr.audmfs;

	if ((avp->aud->format == BSDAV_AUDFMT_NONE) ||
	    (avp->aud->channels == 0) || (avp->aud->sample_rate == 0) ||
	    (avp->aud->buffer_size == 0))
		avp->do_audio = 0;

	if (avp->do_audio == 1) {
		if (bsdav_audio_init(avp->aud->fd, BSDAV_AUDMODE_PLAY,
		    avp->aud->format, avp->aud->channels,
		    avp->aud->sample_rate)) {
			warnx("failed to set audio format");
			cleanup(avp);
			exit(1);
		}
	}

	if (init_buffers(avp)) {
		warnx("failed to init buffers");
		cleanup(avp);
		exit(1);
	}

	if (avp->do_video == 1) {
		if (create_win(avp)) {
			warnx("failed to create window");
			cleanup(avp);
			exit(1);
		}
	}

	if (stream(avp) != 0) {
		warnx("failed to stream");
		cleanup(avp);
		exit(1);
	}

	cleanup(avp);

	return (0);
}
