// FinalBurn NEO ZX Spectrum driver.  NEO edition!
// media: .tap, .z80 snapshots
// input: kbd, kempston: joy1, sinclair intf.2: joy1 & joy2

#include "tiles_generic.h"
#include "z80_intf.h"
#include "ay8910.h"
#include <math.h>
#include "biquad.h"

#if defined (_MSC_VER)
#define strcasecmp stricmp
#endif

static INT32 SpecMode = 0;
#define SPEC_TAP	(1 << 0)
#define SPEC_Z80	(1 << 1)
#define SPEC_128K   (1 << 2)
#define SPEC_PLUS2  (1 << 3) // +2a (Amstrad model)
#define SPEC_INVES	(1 << 4) // Spanish clone (non-contended ula)
#define SPEC_AY8910	(1 << 5)

static UINT8 SpecInputKbd[0x10][0x05] = {
	{ 0, 0, 0, 0, 0 }, // Shift, Z, X, C, V
	{ 0, 0, 0, 0, 0 }, // A, S, D, F, G
	{ 0, 0, 0, 0, 0 }, // Q, W, E, R, T
	{ 0, 0, 0, 0, 0 }, // 1, 2, 3, 4, 5
	{ 0, 0, 0, 0, 0 }, // 0, 9, 8, 7, 6
	{ 0, 0, 0, 0, 0 }, // P, O, I, U, Y
	{ 0, 0, 0, 0, 0 }, // Enter, L, K, J, H
	{ 0, 0, 0, 0, 0 }, // Space, Sym, M, N, B
	{ 0, 0, 0, 0, 0 }, // Kempston joy1
	{ 0, 0, 0, 0, 0 }, // Sinclair intf.2 joy1
	{ 0, 0, 0, 0, 0 }, // Sinclair intf.2 joy2
	// 128K+ Caps Shift (Secondary Membrane)
	{ 0, 0, 0, 0, 0 }, // Edit, CapsLock, True Vid, Inv Vid, Cursor Left
	{ 0, 0, 0, 0, 0 }, // Del, Graph, Cursor Right, Cursor Up, Cursor Down
	{ 0, 0, 0, 0, 0 }, // Break, Ext Mode
	// 128K+ Symbol Shift (Secondary Membrane)
	{ 0, 0, 0, 0, 0 }, // Quote, Semicolon
	{ 0, 0, 0, 0, 0 }  // Period, Comma
};

static UINT8 SpecDips[1];
static UINT8 SpecInput[0x10];
static UINT8 SpecReset;

static UINT8 *AllMem;
static UINT8 *MemEnd;
static UINT8 *RamStart;
static UINT8 *RamEnd;
static UINT8 *SpecZ80Rom;
static UINT8 *SpecZ80Ram;
static UINT8 *SpecVideoRam;
static UINT8 *SpecSnapshotData;
static INT32  SpecSnapshotDataLen;
static UINT8 *SpecTAP;
static INT32  SpecTAPLen;

static UINT32 *SpecPalette;
static UINT8 SpecRecalc;

static UINT8 ula_attr;
static UINT8 ula_scr;
static UINT8 ula_byte;
static UINT8 ula_border;
static UINT8 ula_flash;
static INT32 ula_last_cyc;
static INT32 nExtraCycles;

static INT16 *Buzzer;

static INT32 SpecRamPage; // calculated from Spec128kMapper*
static INT32 SpecRomPage; // calculated from Spec128kMapper*
static INT32 Spec128kMapper;
static INT32 Spec128kMapper2;

static INT32 SpecScanlines;
static INT32 SpecCylesPerScanline;
static INT32 SpecContention;

static INT16 *dacbuf; // for dc offset filter
static INT16 dac_lastin;
static INT16 dac_lastout;

static struct BurnInputInfo SpecInputList[] =
{
	{"P1 Up",			BIT_DIGITAL,	&SpecInputKbd[8][3],	"p1 up"				},
	{"P1 Down",			BIT_DIGITAL,	&SpecInputKbd[8][2],	"p1 down"			},
	{"P1 Left",			BIT_DIGITAL,	&SpecInputKbd[8][1],	"p1 left"			},
	{"P1 Right",		BIT_DIGITAL,	&SpecInputKbd[8][0],	"p1 right"			},
	{"P1 Fire",			BIT_DIGITAL,	&SpecInputKbd[8][4],	"p1 fire 1"			},

	{"P2 Up",			BIT_DIGITAL,	&SpecInputKbd[10][3],	"p2 up"				},
	{"P2 Down",			BIT_DIGITAL,	&SpecInputKbd[10][2],	"p2 down"			},
	{"P2 Left",			BIT_DIGITAL,	&SpecInputKbd[10][0],	"p2 left"			},
	{"P2 Right",		BIT_DIGITAL,	&SpecInputKbd[10][1],	"p2 right"			},
	{"P2 Fire",			BIT_DIGITAL,	&SpecInputKbd[10][4],	"p2 fire 1"			},

	{"ENTER",			BIT_DIGITAL,	&SpecInputKbd[6][0],	"keyb_enter"		},
	{"SPACE",			BIT_DIGITAL,	&SpecInputKbd[7][0],	"keyb_space"		},
	{"CAPS SHIFT",		BIT_DIGITAL,	&SpecInputKbd[0][0],	"keyb_left_shift"	},
	{"SYMBOL SHIFT",	BIT_DIGITAL,	&SpecInputKbd[7][1],	"keyb_right_shift"	},
	{"Q",				BIT_DIGITAL,	&SpecInputKbd[2][0],	"keyb_Q"			},
	{"W",				BIT_DIGITAL,	&SpecInputKbd[2][1],	"keyb_W"			},
	{"E",				BIT_DIGITAL,	&SpecInputKbd[2][2],	"keyb_E"			},
	{"R",				BIT_DIGITAL,	&SpecInputKbd[2][3],	"keyb_R"			},
	{"T",				BIT_DIGITAL,	&SpecInputKbd[2][4],	"keyb_T"			},
	{"Y",				BIT_DIGITAL,	&SpecInputKbd[5][4],	"keyb_Y"			},
	{"U",				BIT_DIGITAL,	&SpecInputKbd[5][3],	"keyb_U"			},
	{"I",				BIT_DIGITAL,	&SpecInputKbd[5][2],	"keyb_I"			},
	{"O",				BIT_DIGITAL,	&SpecInputKbd[5][1],	"keyb_O"			},
	{"P",				BIT_DIGITAL,	&SpecInputKbd[5][0],	"keyb_P"			},
	{"A",				BIT_DIGITAL,	&SpecInputKbd[1][0],	"keyb_A"			},
	{"S",				BIT_DIGITAL,	&SpecInputKbd[1][1],	"keyb_S"			},
	{"D",				BIT_DIGITAL,	&SpecInputKbd[1][2],	"keyb_D"			},
	{"F",				BIT_DIGITAL,	&SpecInputKbd[1][3],	"keyb_F"			},
	{"G",				BIT_DIGITAL,	&SpecInputKbd[1][4],	"keyb_G"			},
	{"H",				BIT_DIGITAL,	&SpecInputKbd[6][4],	"keyb_H"			},
	{"J",				BIT_DIGITAL,	&SpecInputKbd[6][3],	"keyb_J"			},
	{"K",				BIT_DIGITAL,	&SpecInputKbd[6][2],	"keyb_K"			},
	{"L",				BIT_DIGITAL,	&SpecInputKbd[6][1],	"keyb_L"			},
	{"Z",				BIT_DIGITAL,	&SpecInputKbd[0][1],	"keyb_Z"			},
	{"X",				BIT_DIGITAL,	&SpecInputKbd[0][2],	"keyb_X"			},
	{"C",				BIT_DIGITAL,	&SpecInputKbd[0][3],	"keyb_C"			},
	{"V",				BIT_DIGITAL,	&SpecInputKbd[0][4],	"keyb_V"			},
	{"B",				BIT_DIGITAL,	&SpecInputKbd[7][4],	"keyb_B"			},
	{"N",				BIT_DIGITAL,	&SpecInputKbd[7][3],	"keyb_N"			},
	{"M",				BIT_DIGITAL,	&SpecInputKbd[7][2],	"keyb_M"			},
	{"1",				BIT_DIGITAL,	&SpecInputKbd[3][0],	"keyb_1"			},
	{"2",				BIT_DIGITAL,	&SpecInputKbd[3][1],	"keyb_2"			},
	{"3",				BIT_DIGITAL,	&SpecInputKbd[3][2],	"keyb_3"			},
	{"4",				BIT_DIGITAL,	&SpecInputKbd[3][3],	"keyb_4"			},
	{"5",				BIT_DIGITAL,	&SpecInputKbd[3][4],	"keyb_5"			},
	{"6",				BIT_DIGITAL,	&SpecInputKbd[4][4],	"keyb_6"			},
	{"7",				BIT_DIGITAL,	&SpecInputKbd[4][3],	"keyb_7"			},
	{"8",				BIT_DIGITAL,	&SpecInputKbd[4][2],	"keyb_8"			},
	{"9",				BIT_DIGITAL,	&SpecInputKbd[4][1],	"keyb_9"			},
	{"0",				BIT_DIGITAL,	&SpecInputKbd[4][0],	"keyb_0"			},

	{"EDIT",			BIT_DIGITAL,	&SpecInputKbd[11][0],	"keyb_insert"		},
	{"CAPS LOCK",		BIT_DIGITAL,	&SpecInputKbd[11][1],	"keyb_caps_lock"	},
	{"TRUE VID",		BIT_DIGITAL,	&SpecInputKbd[11][2],	"keyb_home"			},
	{"INV VID",			BIT_DIGITAL,	&SpecInputKbd[11][3],	"keyb_end"			},

	{"DEL",				BIT_DIGITAL,	&SpecInputKbd[12][0],	"keyb_backspace"	},
	{"GRAPH",			BIT_DIGITAL,	&SpecInputKbd[12][1],	"keyb_left_alt"		},

	{"Cursor Up",		BIT_DIGITAL,	&SpecInputKbd[12][3],	"keyb_cursor_up"	}, // don't auto-map these cursors by default
	{"Cursor Down",		BIT_DIGITAL,	&SpecInputKbd[12][4],	"keyb_cursor_down"	}, // causes trouble w/games when using cursors
	{"Cursor Left",		BIT_DIGITAL,	&SpecInputKbd[11][4],	"keyb_cursor_left"	}, // as joystick! (yie ar kungfu, ...)
	{"Cursor Right",	BIT_DIGITAL,	&SpecInputKbd[12][2],	"keyb_cursor_right"	},

	{"BREAK",			BIT_DIGITAL,	&SpecInputKbd[13][0],	"keyb_pause"		},
	{"EXT MODE",		BIT_DIGITAL,	&SpecInputKbd[13][1],	"keyb_left_ctrl"	},

	{"\"",				BIT_DIGITAL,	&SpecInputKbd[14][0],	"keyb_apost"		},
	{";",				BIT_DIGITAL,	&SpecInputKbd[14][1],	"keyb_colon"		},

	{".",				BIT_DIGITAL,	&SpecInputKbd[15][2],	"keyb_stop"			},
	{",",				BIT_DIGITAL,	&SpecInputKbd[15][3],	"keyb_comma"		},

	{"Reset",			BIT_DIGITAL,	&SpecReset,				"reset"				},
	{"Dip A",			BIT_DIPSWITCH,	SpecDips + 0,			"dip"				},
};

STDINPUTINFO(Spec)

static struct BurnDIPInfo SpecDIPList[]=
{
	DIP_OFFSET(0x43)

	{0, 0xfe, 0   , 2   , "Hardware Version (48K)"	},
	{0, 0x01, 0x80, 0x00, "Issue 2"					},
	{0, 0x01, 0x80, 0x80, "Issue 3"					},

	{0, 0xfe, 0   , 7   , "Joystick Config"			},
	{0, 0x01, 0x5f, 0x00, "Kempston"				},
	{0, 0x01, 0x5f, 0x01, "Sinclair Interface 2"	},
	{0, 0x01, 0x5f, 0x02, "QAOPM"					},
	{0, 0x01, 0x5f, 0x04, "QAOP Space"				},
	{0, 0x01, 0x5f, 0x08, "Cursor Keys"				},
	{0, 0x01, 0x5f, 0x40, "Jumping Jack"			},
	{0, 0x01, 0x5f, 0x10, "Disabled"				},

	{0, 0xfe, 0   , 2   , "Interface 2 Joyport"		},
	{0, 0x01, 0x20, 0x00, "Normal"					},
	{0, 0x01, 0x20, 0x20, "Swapped: Joy1 <-> Joy2"	},
};

static struct BurnDIPInfo SpecDefaultDIPList[]=
{
	{0, 0xff, 0xff, 0x80, NULL						}, // Issue 3 + Kempston (Blinky's Scary School requires issue 3)
};

static struct BurnDIPInfo SpecIssue2DIPList[]=
{
	{0, 0xff, 0xff, 0x00, NULL						}, // Issue 2 + Kempston (Abu Simbel (English/Gremlin) requires issue 2)
};

static struct BurnDIPInfo SpecIntf2DIPList[]=
{
	{0, 0xff, 0xff, 0x81, NULL						}, // Sinclair Interface 2 (2 Joysticks)
};

static struct BurnDIPInfo SpecIntf2SwappedDIPList[]=
{
	{0, 0xff, 0xff, 0xa1, NULL						}, // Sinclair Interface 2 (2 Joysticks), Swapped (Joy1 <-> Joy2)
};

static struct BurnDIPInfo SpecQAOPMDIPList[]=
{
	{0, 0xff, 0xff, 0x82, NULL						}, // Kempston mapped to QAOPM (moon ranger)
};

static struct BurnDIPInfo SpecQAOPSpaceDIPList[]=
{
	{0, 0xff, 0xff, 0x84, NULL						}, // Kempston mapped to QAOPSpace (jet pack bob)
};

static struct BurnDIPInfo SpecJJackDIPList[]=
{
	{0, 0xff, 0xff, 0xc0, NULL						}, // Kempston mapped to space, symbol shift, caps shift (jumping jack)
};

static struct BurnDIPInfo SpecCursorKeysDIPList[]=
{
	{0, 0xff, 0xff, 0x88, NULL						}, // Kempston mapped to Cursor Keys
};

STDDIPINFOEXT(Spec, SpecDefault, Spec)
STDDIPINFOEXT(SpecIssue2, SpecIssue2, Spec)
STDDIPINFOEXT(SpecIntf2, SpecIntf2, Spec)
STDDIPINFOEXT(SpecIntf2Swap, SpecIntf2Swapped, Spec)
STDDIPINFOEXT(SpecQAOPM, SpecQAOPM, Spec)
STDDIPINFOEXT(SpecQAOPSpace, SpecQAOPSpace, Spec)
STDDIPINFOEXT(SpecJJack, SpecJJack, Spec)
STDDIPINFOEXT(SpecCursorKeys, SpecCursorKeys, Spec)

static void spectrum128_bank(); // forwards
static void spectrum_loadz80();
static void ula_init(INT32 scanlines, INT32 cyc_scanline, INT32 contention);
static void update_ula(INT32 cycle);

// Spectrum 48k tap-loading robot -dink
static INT32 CASFrameCounter = 0; // for autoloading
static INT32 CASAutoLoadPos = 0;
static INT32 CASAutoLoadTicker = 0;
static void SpecTAPReset(); // forward

static void SpecLoadTAP()
{
	CASAutoLoadPos = 0;
	CASAutoLoadTicker = 0;
	CASFrameCounter = 0;
}

static UINT8* FindInput(char *str)
{
	for (INT32 i = 0; SpecInputList[i].szName != NULL; i++) {
		if (!strcmp(str, SpecInputList[i].szName)) {
			return SpecInputList[i].pVal;
		}
	}
	return NULL;
}

static void SetInput(char *str, INT32 data)
{
	UINT8 *x = FindInput(str);
	if (x) {
		x[0] = data;
	}
}

static void TAPAutoLoadTick()
{
	const UINT8 TAPLoader[2][10] = { { "J\"\"\n\0" }, { "\n\0" } }; // 48k, 128k
	const INT32 KeyDelay = 12; // frames 0-4: press key, 5-11: delay, 11: next character.

	if (CASAutoLoadPos == 0xff) return;

	UINT8 c = TAPLoader[((SpecMode & SPEC_128K) ? 1 : 0)][CASAutoLoadPos];
	if (!c) {
		CASAutoLoadPos = 0xff;
		return;
	}

	if ((CASAutoLoadTicker % KeyDelay) < 5) {
		switch (c) {
			case '\"': {
				SetInput("SYMBOL SHIFT", 1);
				SetInput("P", 1);
				break;
			}
			case 'J': {
				SetInput("J", 1);
				break;
			}
			case '\n': {
				SetInput("ENTER", 1);
				break;
			}
		}
	}

	if ((CASAutoLoadTicker % KeyDelay) == KeyDelay - 1) CASAutoLoadPos++;
	CASAutoLoadTicker++;
}

static void TAPAutoLoadRobot()
{
	if (SpecMode & SPEC_TAP && CASFrameCounter > 90) {
		TAPAutoLoadTick();
	}
	CASFrameCounter++;
}
// End TAP Robot

// Oversampling Buzzer-DAC
static INT32 buzzer_last_update;
static INT32 buzzer_last_data;
static INT32 buzzer_data_len;
static INT32 buzzer_data_frame;
static INT32 buzzer_data_frame_minute;

static const INT32 buzzer_oversample = 3000;

static BIQ biquad[2]; // snd/biquad.h

static void BuzzerInit() // keep in DoReset()!
{
	biquad[0].init(FILT_LOWPASS, nBurnSoundRate, 7000, 0.554, 0.0);
	biquad[1].init(FILT_LOWPASS, nBurnSoundRate, 8000, 0.554, 0.0);

	buzzer_data_frame_minute = (SpecCylesPerScanline * SpecScanlines * 50.00);
	buzzer_data_frame = ((double)(SpecCylesPerScanline * SpecScanlines) * nBurnSoundRate * buzzer_oversample) / buzzer_data_frame_minute;
}

static void BuzzerExit()
{
	biquad[0].exit();
	biquad[1].exit();
}

static void BuzzerAdd(INT16 data)
{
	data *= (1 << 12);

	if (data != buzzer_last_data) {
		INT32 len = ((double)(ZetTotalCycles() - buzzer_last_update) * nBurnSoundRate * buzzer_oversample) / buzzer_data_frame_minute;
		if (len > 0)
		{
			for (INT32 i = buzzer_data_len; i < buzzer_data_len+len; i++) {
				// if len goes over buzzer_data_frame, wrap around to the beginning.
				Buzzer[i % buzzer_data_frame] = buzzer_last_data;
			}
			buzzer_data_len += len;
		}

		buzzer_last_data = data;
		buzzer_last_update = ZetTotalCycles();
	}
}

static void BuzzerRender(INT16 *dest)
{
	INT32 buzzer_data_pos = 0;

	// fill buffer (if needed)
	if (buzzer_data_len < buzzer_data_frame) {
		for (INT32 i = buzzer_data_len; i < buzzer_data_frame; i++) {
			Buzzer[i] = buzzer_last_data;
		}
		buzzer_data_len = buzzer_data_frame;
	}

	// average + mixdown
	for (INT32 i = 0; i < nBurnSoundLen; i++) {
		INT32 sample = 0;
		for (INT32 j = 0; j < buzzer_oversample; j++) {
			sample += Buzzer[buzzer_data_pos++];
		}
		sample = (INT32)(biquad[1].filter(biquad[0].filter((double)sample / buzzer_oversample)));
		dest[0] = BURN_SND_CLIP(sample);
		dest[1] = BURN_SND_CLIP(sample);
		dest += 2;
	}

	buzzer_data_len = 0;
	buzzer_last_update = 0;
}

// end Oversampling Buzzer-DAC

static INT32 MemIndex()
{
	UINT8 *Next; Next = AllMem;

	SpecZ80Rom              = Next; Next += 0x04000 * 4;
	SpecSnapshotData        = Next; Next += 0x20000;
	SpecTAP                 = Next; Next += 0x800000;

	RamStart                = Next;
	SpecZ80Ram              = Next; Next += 0x20000;
	RamEnd                  = Next;

	SpecPalette             = (UINT32*)Next; Next += 0x00010 * sizeof(UINT32);
	dacbuf                  = (INT16*)Next; Next += 0x800 * 2 * sizeof(INT16);

	Buzzer                  = (INT16*)Next; Next += 1000 * buzzer_oversample * sizeof(INT16);

	MemEnd                  = Next;

	return 0;
}

static INT32 SpecDoReset()
{
	ZetOpen(0);
	ZetReset();
	if (SpecMode & SPEC_AY8910) {
		AY8910Reset(0);
	}
	ZetClose();

	BuzzerInit();

	SpecVideoRam = SpecZ80Ram;
	Spec128kMapper = 0;
	Spec128kMapper2 = 0;
	ula_border = 0;

	if (SpecMode & SPEC_128K) {
		ZetOpen(0);
		spectrum128_bank();
		ZetClose();
	}

	if (SpecMode & SPEC_Z80) {
		spectrum_loadz80();
	}

	if (SpecMode & SPEC_TAP) {
		SpecLoadTAP(); // reset robot
		SpecTAPReset(); // reset tap engine
	}

	dac_lastin = 0;
	dac_lastout = 0;

	ula_last_cyc = 0;
	ula_byte = 0xff;
	ula_attr = 0x00;
	ula_scr = 0x00;

	nExtraCycles = 0;

	return 0;
}

static UINT8 __fastcall SpecZ80Read(UINT16 address)
{
	if (address < 0x4000) {
		return SpecZ80Rom[address];
	}

	if (address >= 0x4000 && address <= 0x7fff) {
		return SpecZ80Ram[address & 0x3fff];
	}

	if (address >= 0x8000 && address <= 0xffff) {
		return SpecZ80Ram[0x4000 + (address & 0x7fff)];
	}

	bprintf(0, _T("mr %x\n"), address);

	return 0;
}

static void __fastcall SpecZ80Write(UINT16 address, UINT8 data)
{
	if (address < 0x4000) return;

	if (address >= 0x4000 && address <= 0x7fff) {
		SpecZ80Ram[address & 0x3fff] = data;
		return;
	}

	if (address >= 0x8000 && address <= 0xffff) {
		SpecZ80Ram[0x4000 + (address & 0x7fff)] = data;
		return;
	}

	bprintf(0, _T("mw %x %x\n"), address, data);
}

// the 128K has a secondary kbd matrix that added more keys to the keyboard
// which were normally accessed by pressing Caps Shift / Symbol Shift.
// When one of those keys are pressed - it basically uses a little hw hack
// to hold the caps/symbol shift button. -dink
static INT32 check_caps_shift()
{
	INT32 ret = 0;
	for (INT32 i = 11; i <= 13; i++) {
		// check if a 2ndary matrix button that needs CAPS Shift has been pressed
		if (SpecInput[i] != 0x1f) ret = 1;
	}
	return ret;
}

static INT32 check_symbol_shift()
{
	INT32 ret = 0;
	for (INT32 i = 14; i <= 15; i++) {
		// check if a 2ndary matrix button that needs Symbol Shift has been pressed
		if (SpecInput[i] != 0x1f) ret = 1;
	}
	return ret;
}

static UINT8 read_keyboard(UINT16 address)
{
	UINT8 keytmp = 0x1f;

	for (INT32 i = 0; i < 8; i++) { // process all kbd rows
		if (~address & (1 << (i + 8))) {
			switch (i) {
				case 0:
					keytmp &= SpecInput[i];
					if (check_caps_shift()) keytmp &= ~(1 << 0);
					break;

				case 3:
					keytmp &= SpecInput[i] & SpecInput[11] & SpecInput[10]; // caps shift0, intf2 joy2
					break;

				case 4:
					keytmp &= SpecInput[i] & SpecInput[12] & SpecInput[9]; // caps shift1, intf2 joy1
					break;

				case 5:
					keytmp &= SpecInput[i] & SpecInput[14]; // symbol shift0
					break;

				case 7:
					keytmp &= SpecInput[i] & SpecInput[13] & SpecInput[15]; // caps shift2, symbol shift1
					if (check_symbol_shift()) keytmp &= ~(1 << 1);
					break;

				default:
					keytmp &= SpecInput[i];
					break;
			}
		}
	}

	if (SpecMode & SPEC_128K || SpecDips[0] & 0x80) {	// 128K or Issue 3
		keytmp |= (ula_border & 0x10) ? 0xe0 : 0xa0;
	} else {											// Issue 2
		keytmp |= (ula_border & 0x18) ? 0xe0 : 0xa0;
	}

	return keytmp;
}

static UINT8 __fastcall SpecZ80PortRead(UINT16 address)
{
	if (~address & 0x0001) { // keyboard
		return read_keyboard(address);
	}

	if ((address & 0x1f) == 0x1f && (address & 0x20) == 0) {
		// Kempston only pulls A5 low - Kempston RE: https://www.youtube.com/watch?v=4e1MlxPVyD4
		return SpecInput[8]; // kempston (returns 0xff when disabled)
	}

	if ((address & 0xc002) == 0xc000 && (SpecMode & SPEC_AY8910)) {
		return AY8910Read(0);
	}

	update_ula(ZetTotalCycles()); // get up-to-date ula_byte!

	return ula_byte; // Floating Memory
}

static void __fastcall SpecZ80PortWrite(UINT16 address, UINT8 data)
{
	if (~address & 0x0001) {
		BuzzerAdd((data & 0x10) >> 4);

		ula_border = data;
		return;
	}

	if (SpecMode & SPEC_AY8910) {
		switch (address & 0xc002) {
			case 0x8000: AY8910Write(0, 1, data); return;
			case 0xc000: AY8910Write(0, 0, data); return;
		}
	}

	if ((address & 0xff) == 0xfd) return; // Ignore (Jetpac writes here due to a bug in the game code, and it's the reason it won't work on 128k)

	//bprintf(0, _T("pw %x %x\n"), address, data);
}

static void spectrum128_bank()
{
	SpecVideoRam = SpecZ80Ram + ((5 + ((Spec128kMapper & 0x08) >> 2)) << 14);

	SpecRamPage = Spec128kMapper & 0x07;
	SpecRomPage = ((Spec128kMapper & 0x10) >> 4) * 0x4000;

	if (SpecMode & SPEC_PLUS2) {
		SpecRomPage += ((Spec128kMapper2 & 4) >> 1) * 0x4000;

		if (Spec128kMapper2 & 0x01) { // +2a/+3 "Special" paging mode
			const INT32 special_map[4][4] = { { 0, 1, 2, 3 },  { 4, 5, 6, 7 },  { 4, 5, 6, 3 },  { 4, 7, 6, 3 } };
			const INT32 special_sel = (Spec128kMapper2 >> 1) & 3;

			ZetMapMemory(SpecZ80Ram + (special_map[special_sel][0] << 14), 0x0000, 0x3fff, MAP_RAM);
			ZetMapMemory(SpecZ80Ram + (special_map[special_sel][1] << 14), 0x4000, 0x7fff, MAP_RAM);
			ZetMapMemory(SpecZ80Ram + (special_map[special_sel][2] << 14), 0x8000, 0xbfff, MAP_RAM);
			ZetMapMemory(SpecZ80Ram + (special_map[special_sel][3] << 14), 0xc000, 0xffff, MAP_RAM);
		} else {
			ZetUnmapMemory(0x0000, 0xffff, MAP_RAM);
		}
	}

	Z80Contention_set_bank(SpecRamPage);
}

static UINT8 __fastcall SpecSpec128Z80Read(UINT16 address)
{
	if (address < 0x4000) {
		return SpecZ80Rom[SpecRomPage + address];
	}

	if (address >= 0x4000 && address <= 0x7fff) {
		return SpecZ80Ram[(5 << 14) + (address & 0x3fff)];
	}

	if (address >= 0x8000 && address <= 0xbfff) {
		return SpecZ80Ram[(2 << 14) + (address & 0x3fff)];
	}

	if (address >= 0xc000 && address <= 0xffff) {
		return SpecZ80Ram[(SpecRamPage << 14) + (address & 0x3fff)];
	}

	bprintf(0, _T("mr %x\n"), address);

	return 0;
}

static void __fastcall SpecSpec128Z80Write(UINT16 address, UINT8 data)
{
	if (address < 0x4000) return; // ROM

	if (address >= 0x4000 && address <= 0x7fff) {
		SpecZ80Ram[(5 << 14) + (address & 0x3fff)] = data;
		return;
	}

	if (address >= 0x8000 && address <= 0xbfff) {
		SpecZ80Ram[(2 << 14) + (address & 0x3fff)] = data;
		return;
	}

	if (address >= 0xc000 && address <= 0xffff) {
		SpecZ80Ram[(SpecRamPage << 14) + (address & 0x3fff)] = data;
		return;
	}

	bprintf(0, _T("mw %x %x\n"), address, data);
}

static UINT8 __fastcall SpecSpec128Z80PortRead(UINT16 address)
{
	if (~address & 0x0001) { // keyboard
		return read_keyboard(address);
	}

	if ((address & 0x1f) == 0x1f && (address & 0x20) == 0) {
		// Kempston only pulls A5 low - Kempston RE: https://www.youtube.com/watch?v=4e1MlxPVyD4
		return SpecInput[8]; // kempston (returns 0xff when disabled)
	}

	if ((address & 0xc002) == 0xc000 && (SpecMode & SPEC_AY8910)) {
		return AY8910Read(0);
	}

	if ((address & 0x8002) == 0x0000) {
		// todo: figure out what 0x7ffd / 3ffd read does
		//bprintf(0, _T("reading %x (%x)\n"), address, Spec128kMapper);
	}

	update_ula(ZetTotalCycles()); // get up-to-date ula_byte!

	return ula_byte; // Floating Memory
}

static void __fastcall SpecSpec128Z80PortWrite(UINT16 address, UINT8 data)
{
	if (~address & 0x0001) {
		BuzzerAdd((data & 0x10) >> 4);

		ula_border = data;
		// needs to fall through!!
	}

	if (SpecMode & SPEC_PLUS2) { // 128k +2a / +3
		if ((address & 0xc002) == 0x4000) { // 7ffd
			if (Spec128kMapper & 0x20) return; // memory lock-out latch

			Spec128kMapper = data;
			//bprintf(0, _T("7ffd: %x\t%x\n"), address, data);
			spectrum128_bank();
			return;
		}

		if ((address & 0xf002) == 0x1000) { // 1ffd
			if (Spec128kMapper & 0x20) return; // memory lock-out latch

			//bprintf(0, _T("1ffd: %x\t%x\n"), address, data);
			Spec128kMapper2 = data;
			spectrum128_bank();
			return;
		}
	} else { // standard 128k
		if ((address & 0x8002) == 0x0000) { // 7ffd
			if (Spec128kMapper & 0x20) return; // memory lock-out latch

			Spec128kMapper = data;
			//bprintf(0, _T("7ffd: %x\t%x\n"), address, data);
			spectrum128_bank();
			return;
		}
	}

	if (SpecMode & SPEC_AY8910) {
		switch (address & 0xc002) {
			case 0x8000: AY8910Write(0, 1, data); return;
			case 0xc000: AY8910Write(0, 0, data); return;
		}
	}

	if (address == 0xff3b || address == 0xbf3b) return; // ignore (some games check for "ula plus" here)

	//bprintf(0, _T("pw %x %x\n"), address, data);
}

// Spectrum TAP loader (c) 2020 dink
#define DEBUG_TAP 0
#define BLKNUM 0x200
static UINT8 *SpecTAPBlock[BLKNUM];
static INT32 SpecTAPBlockLen[BLKNUM];

static INT32 SpecTAPBlocks = 0; // 1-based
static INT32 SpecTAPBlocknum = 0; // 0-based
static INT32 SpecTAPPos = 0; // 0-based
static INT32 SpecTAPLoading = 0;

static void SpecTAPReset()
{
	SpecTAPBlocknum = 0;
	SpecTAPPos = 0;
	SpecTAPLoading = 0;
}

static void SpecTAPInit()
{
	for (INT32 i = 0; i < BLKNUM; i++) {
		SpecTAPBlock[i] = NULL;
		SpecTAPBlockLen[i] = 0;
	}
	SpecTAPBlocks = 0;
	SpecTAPBlocknum = 0;
	if (DEBUG_TAP) {
		bprintf(0, _T("**  - Spectrum TAP Loader -\n"));
		bprintf(0, _T("Block#\tLength\tOffset\n"));
	}
	for (INT32 i = 0; i < SpecTAPLen;) {
		INT32 block_size = SpecTAP[i+0] | (SpecTAP[i+1] << 8);

		if (block_size) {
			if (DEBUG_TAP) {
				bprintf(0, _T("%x\t%d\t%x\n"), SpecTAPBlocks, block_size, i+2);
			}

			SpecTAPBlock[SpecTAPBlocks] = &SpecTAP[i+2];
			SpecTAPBlockLen[SpecTAPBlocks] = block_size-2;
			SpecTAPBlocks++;
			if (SpecTAPBlocks >= BLKNUM) {
				bprintf(PRINT_ERROR, _T(".TAP Loader: Tape blocks exceeded.\n"));
				break;
			}
		}

		i += block_size + 2;
	}
}

static INT32 SpecTAPDMACallback()
{
	if (~SpecMode & SPEC_TAP || SpecTAPBlocks == 0) return 0;

	UINT8 *data = SpecTAPBlock[SpecTAPBlocknum];

	INT32 transfer_ok = 0;
	INT32 carry_val = 0;
	INT32 checksum = 0;
	INT32 offset = 0;
	UINT8 byte = 0;

	ActiveZ80EXAF();
	INT32 tap_block = data[0];
	INT32 cpu_block = ActiveZ80GetAF() >> 8;
	INT32 address = ActiveZ80GetIX();
	INT32 length = ActiveZ80GetDE();
	INT32 length_unadjusted = length;

	// If anything is changed here, try the following testcases:
	// Chickin Chase, Alter Ego, V

	if (DEBUG_TAP) {
		bprintf(0, _T("TAP blocknum %d\n"), SpecTAPBlocknum);
		bprintf(0, _T("TAP blocklen %d\n"), SpecTAPBlockLen[SpecTAPBlocknum]);
		bprintf(0, _T("TAP blocktype %x\n"), tap_block);
		bprintf(0, _T("CPU blocktype %x\n"), cpu_block);
		bprintf(0, _T("CPU address %x\n"), address);
		bprintf(0, _T("CPU length %x\n"), length);
	}
	if (length > SpecTAPBlockLen[SpecTAPBlocknum]) {
		bprintf(0, _T("CPU Requested length %x > tape block length %x, adjusting.\n"), length, SpecTAPBlockLen[SpecTAPBlocknum]);
		length = SpecTAPBlockLen[SpecTAPBlocknum];
	}
	if (cpu_block == tap_block) { // we found our block! :)
		if (ActiveZ80GetCarry()) {
			if (DEBUG_TAP) {
				bprintf(0, _T("loading data\n"));
			}
			// load
			offset = 0;
			checksum = tap_block;
			while (offset < length) {
				if (offset+1 > SpecTAPBlockLen[SpecTAPBlocknum]) {
					bprintf(0, _T(".TAP Loader: trying to read past block.  offset %x  blocklen %x\n"), offset, SpecTAPBlockLen[SpecTAPBlocknum]);
					carry_val = 0;
					break;
				}
				byte = data[offset+1];
				ZetWriteByte((address + offset) & 0xffff, data[offset+1]);
				checksum ^= data[offset+1];
				offset++;
			}
			if (DEBUG_TAP) {
				bprintf(0, _T("end dma, checksum %x  tap checksum %x\n"), checksum, data[offset+1]);
			}
			carry_val = (checksum == data[offset+1]);
			transfer_ok = 1;
		}
	}

	ActiveZ80SetCarry(carry_val);
	ActiveZ80SetIX((address + offset) & 0xffff);
	if (transfer_ok) ActiveZ80SetDE(length_unadjusted - SpecTAPBlockLen[SpecTAPBlocknum]);
	ActiveZ80SetHL((checksum << 8) | byte);
	ActiveZ80SetA(0);
	ActiveZ80SetPC(0x05e2);

	SpecTAPBlocknum = (SpecTAPBlocknum + 1) % SpecTAPBlocks;

	return 0;
}

static void snapshot_write_ram(UINT32 address, UINT8 data)
{
	if (address >= 0x4000 && address < 0x24000) { // ignore writes to bios area (snapshot)
		SpecZ80Ram[address - 0x4000] = data;
	} else {
		bprintf(PRINT_ERROR, _T(".z80, snapshot_write_ram(%x, %x).\n"));
	}
}

static void z80_rle_decompress(UINT8 *source, UINT32 dest, UINT16 size)
{
	while (size > 0) {
		if (size > 2 && source[0] == 0xed && source[1] == 0xed) {
			UINT8 count = source[2];
			UINT8 data = source[3];

			if (count == 0)	{
				bprintf(PRINT_ERROR, _T(".z80 rle_decompress: zero length rle-block? eek. (bad .z80 file)\n"));
				return;
			}

			if (count > size) {
				bprintf(PRINT_ERROR, _T(".z80 rle_decompress: count > size, eek. (bad .z80 file)\n"));
				count = size;
			}

			for (INT32 i = 0; i < count; i++) {
				snapshot_write_ram(dest, data);
				dest++;
			}

			source += 4;
			size -= count;
		} else {
			snapshot_write_ram(dest, source[0]);

			dest++;
			source++;
			size--;
		}
	}
}

static UINT16 mem2uint16(INT32 posi, INT32 bigendian)
{
	UINT8 *umem = (UINT8*)&SpecSnapshotData[posi];

	if (bigendian) {
		return (umem[0] << 8) + umem[1];
	} else {
		return (umem[1] << 8) + umem[0];
	}
}

static UINT32 page_to_mem(INT32 page, INT32 is128k)
{
	if (is128k) {
		switch (page) {
			case 3: return 0x4000;
			case 4: return 0x8000;
			case 5: return 0xc000;
			case 6: return 0x10000;
			case 7: return 0x14000;
			case 8: return 0x18000;
			case 9: return 0x1c000;
			case 10: return 0x20000; // ram ends at 0x24000 (0-4000 = rom)
			default: return 0x0000;
		}
	} else {
		switch (page) {
			case 4: return 0x8000;
			case 5: return 0xc000;
			case 8: return 0x4000;
			default: return 0x0000;
		}
	}

	return 0x0000;
}

static void spectrum_loadz80()
{
	ZetSetAF(0, mem2uint16(0, 1));
	ZetSetBC(0, mem2uint16(2, 0));
	ZetSetHL(0, mem2uint16(4, 0));
	UINT16 v1PC = mem2uint16(6, 0);
	ZetSetSP(0, mem2uint16(8, 0));
	ZetSetI(0, (SpecSnapshotData[10] & 0xff));
	ZetSetR(0, (SpecSnapshotData[11] & 0x7f) | ((SpecSnapshotData[12] & 0x01) << 7));
	ula_border = (ula_border & 0xf8) | ((SpecSnapshotData[12] & 0x0e) >> 1);
	ZetSetDE(0, mem2uint16(13, 0));
	ZetSetBC2(0, mem2uint16(15, 0));
	ZetSetDE2(0, mem2uint16(17, 0));
	ZetSetHL2(0, mem2uint16(19, 0));
	ZetSetAF2(0, mem2uint16(21, 1));
	ZetSetIY(0, mem2uint16(23, 0));
	ZetSetIX(0, mem2uint16(25, 0));
	ZetSetIFF1(0, (SpecSnapshotData[27]) ? 1 : 0);
	ZetSetIFF2(0, (SpecSnapshotData[28]) ? 1 : 0);
	ZetSetIM(0, (SpecSnapshotData[29] & 0x03));

	if (v1PC != 0) { // version 1 (48k) snapshot
		bprintf(0, _T(".z80 version 1 - 48k snapshot - "));

		ZetSetPC(0, v1PC);

		if (SpecSnapshotData[12] & 0x20) {
			bprintf(0, _T(".z80: rle-compressed\n"));
			z80_rle_decompress(SpecSnapshotData + 30, 0x4000, 0xc000);
		} else {
			bprintf(0, _T(".z80: un-compressed\n")); // testcase: Horace Goes Skiing
			for (INT32 i = 0x4000; i < 0xc000; i++) {
				snapshot_write_ram(i, SpecSnapshotData[30 + (i - 0x4000)]);
			}
		}
	} else {
		INT32 v2_v3_headerlen = mem2uint16(30, 0);
		INT32 hwmode = SpecSnapshotData[34];
		INT32 v2 = (v2_v3_headerlen == 23);
		INT32 is_128k = ((v2) ? (hwmode > 2) : (hwmode > 3));

		bprintf(0, _T(".z80 version %d - "), (v2) ? 2 : 3);
		if (is_128k) {
			bprintf(0, _T("128k\n"));
			if (~SpecMode & SPEC_128K) {
				bprintf(PRINT_ERROR, _T(".z80 Error: loading 128k snapshot on 48k hw!\n"));
				return;
			}
		} else {
			bprintf(0, _T("48k\n"));
		}

		ZetSetPC(0, mem2uint16(32, 0));

		if (SpecMode & SPEC_AY8910 && SpecSnapshotData[37] & (1<<2)) { // AY8910
			bprintf(0, _T(".z80 contains AY8910 registers\n"));
			ZetOpen(0);
			for (INT32 i = 0; i < 0x10; i++) { // write regs
				AY8910Write(0, 0, i);
				AY8910Write(0, 1, SpecSnapshotData[39 + i]);
			}
			AY8910Write(0, 0, SpecSnapshotData[38]); // write latch
			ZetClose();
		}

		INT32 offset = v2_v3_headerlen + 32;
		while (offset < SpecSnapshotDataLen)
		{
			UINT16 length = mem2uint16(offset, 0);
			UINT32 dest = page_to_mem(SpecSnapshotData[offset + 2], is_128k);

			if (length == 0xffff) {
				length = 0x4000;
				bprintf(0, _T(".z80: copying $%x uncompressed bytes to %x\n"), length, dest);
				for (INT32 i = 0; i < length; i++)
					snapshot_write_ram(dest + i, SpecSnapshotData[offset + 3 + i]);
			} else {
				bprintf(0, _T(".z80: decompressing $%x bytes to %x\n"), length, dest);
				z80_rle_decompress(&SpecSnapshotData[offset + 3], dest, 0x4000);
			}

			offset += 3 + length;
		}

		if (is_128k) {
			Spec128kMapper = (SpecSnapshotData[35] & 0xff);
			ZetOpen(0);
			spectrum128_bank();
			ZetClose();
		}
	}
}

static INT32 BurnGetLength(INT32 rom_index)
{
	struct BurnRomInfo ri = { "", 0, 0, 0 };
	BurnDrvGetRomInfo(&ri, 0);
	return ri.nLen;
}

struct s_modes {
	INT32 mode;
	TCHAR text[40];
};

static s_modes speccy_modes[] = {
	{ SPEC_TAP,		_T(".tap file")						},
	{ SPEC_Z80,		_T(".z80 file")						},
	{ SPEC_128K,	_T("128K")							},
	{ SPEC_PLUS2,	_T("+2a")							},
	{ SPEC_INVES,	_T("Investronica (un-contended)")	},
	{ SPEC_AY8910,	_T("AY-8910 PSG")					},
	{ -1,			_T("")								}
};

static void PrintSpeccyMode()
{
	bprintf(0, _T("Speccy Init w/ "));

	for (INT32 i = 0; speccy_modes[i].mode != -1; i++) {
		if (SpecMode & speccy_modes[i].mode) {
			bprintf(0, _T("%s, "), speccy_modes[i].text);
		}
	}

	bprintf(0, _T("...\n"));
}


static INT32 SpectrumInit(INT32 Mode)
{
	SpecMode = Mode;

	SpecMode |= SPEC_AY8910; // Add an AY-8910!

	PrintSpeccyMode();

	BurnSetRefreshRate(50.0);

	BurnAllocMemIndex();

	INT32 nRet = 0;

	if (SpecMode & SPEC_Z80) {
		// Snapshot
		SpecSnapshotDataLen = BurnGetLength(0);
		nRet = BurnLoadRom(SpecSnapshotData + 0x00000, 0, 1); if (nRet != 0) return 1;
		nRet = BurnLoadRom(SpecZ80Rom + 0x00000, 0x80, 1); if (nRet != 0) return 1;
	}
	else if (SpecMode & SPEC_TAP) {
		// TAP
		SpecTAPLen = BurnGetLength(0);
		nRet = BurnLoadRom(SpecTAP + 0x00000, 0, 1); if (nRet != 0) return 1;
		nRet = BurnLoadRom(SpecZ80Rom + 0x00000, 0x80, 1); if (nRet != 0) return 1;

		SpecTAPInit();
	} else {
		// System
		nRet = BurnLoadRom(SpecZ80Rom + 0x00000, 0, 1); if (nRet != 0) return 1;
	}

	ZetInit(0);
	ZetOpen(0);
	ZetSetReadHandler(SpecZ80Read);
	ZetSetWriteHandler(SpecZ80Write);
	ZetSetInHandler(SpecZ80PortRead);
	ZetSetOutHandler(SpecZ80PortWrite);
	if (SpecMode & SPEC_TAP) {
		bprintf(0, _T("**  Spectrum: Using TAP file (len 0x%x) - DMA Loader\n"), SpecTAPLen);
		z80_set_spectrum_tape_callback(SpecTAPDMACallback);
	}
	if (~SpecMode & SPEC_INVES) {
		Z80InitContention(48, &update_ula);
	}
	ZetClose();

	AY8910Init(0, 17734475 / 10, 0);
	AY8910SetAllRoutes(0, 0.20, BURN_SND_ROUTE_BOTH);
	AY8910SetBuffered(ZetTotalCycles, 224*312*50);

	// Init Buzzer (in DoReset!)

	GenericTilesInit();

	ula_init(312, 224, 14335);

	SpecDoReset();

	return 0;
}

static INT32 Spectrum128Init(INT32 Mode)
{
	SpecMode = Mode;

	SpecMode |= SPEC_AY8910; // Add an AY-8910! (always on-board w/128K)

	BurnSetRefreshRate(50.0);

	BurnAllocMemIndex();

	PrintSpeccyMode();

	INT32 nRet = 0;

	if (SpecMode & SPEC_Z80) {
		// Snapshot
		SpecSnapshotDataLen = BurnGetLength(0);
		nRet = BurnLoadRom(SpecSnapshotData + 0x00000, 0, 1); if (nRet != 0) return 1;
		nRet = BurnLoadRom(SpecZ80Rom + 0x00000, 0x80, 1); if (nRet != 0) return 1;
		nRet = BurnLoadRom(SpecZ80Rom + 0x04000, 0x81, 1); if (nRet != 0) return 1;
		if (SpecMode & SPEC_PLUS2) {
			nRet = BurnLoadRom(SpecZ80Rom + 0x08000, 0x82, 1); if (nRet != 0) return 1;
			nRet = BurnLoadRom(SpecZ80Rom + 0x0c000, 0x83, 1); if (nRet != 0) return 1;
		}
	}
	else if (SpecMode & SPEC_TAP) {
		// TAP
		SpecTAPLen = BurnGetLength(0);
		nRet = BurnLoadRom(SpecTAP + 0x00000, 0, 1); if (nRet != 0) return 1;
		nRet = BurnLoadRom(SpecZ80Rom + 0x00000, 0x80, 1); if (nRet != 0) return 1;
		nRet = BurnLoadRom(SpecZ80Rom + 0x04000, 0x81, 1); if (nRet != 0) return 1;
		if (SpecMode & SPEC_PLUS2) {
			nRet = BurnLoadRom(SpecZ80Rom + 0x08000, 0x82, 1); if (nRet != 0) return 1;
			nRet = BurnLoadRom(SpecZ80Rom + 0x0c000, 0x83, 1); if (nRet != 0) return 1;
		}

		SpecTAPInit();
	} else {
		// System
		nRet = BurnLoadRom(SpecZ80Rom + 0x00000, 0, 1); if (nRet != 0) return 1;
		nRet = BurnLoadRom(SpecZ80Rom + 0x04000, 1, 1); if (nRet != 0) return 1;
		if (SpecMode & SPEC_PLUS2) {
			nRet = BurnLoadRom(SpecZ80Rom + 0x08000, 2, 1); if (nRet != 0) return 1;
			nRet = BurnLoadRom(SpecZ80Rom + 0x0c000, 3, 1); if (nRet != 0) return 1;
		}
	}

	ZetInit(0);
	ZetOpen(0);
	ZetSetReadHandler(SpecSpec128Z80Read);
	ZetSetWriteHandler(SpecSpec128Z80Write);
	ZetSetInHandler(SpecSpec128Z80PortRead);
	ZetSetOutHandler(SpecSpec128Z80PortWrite);
	if (SpecMode & SPEC_TAP) {
		bprintf(0, _T("**  Spectrum 128k: Using TAP file (len 0x%x) - DMA Loader\n"), SpecTAPLen);
		z80_set_spectrum_tape_callback(SpecTAPDMACallback);
	}
	if (~SpecMode & SPEC_INVES) {
		Z80InitContention((SpecMode & SPEC_PLUS2) ? 1282 : 128, &update_ula);
	}
	ZetClose();

	AY8910Init(0, 17734475 / 10, 0);
	AY8910SetAllRoutes(0, 0.20, BURN_SND_ROUTE_BOTH);
	AY8910SetBuffered(ZetTotalCycles, 228*311*50);

	// Init Buzzer (in DoReset!)

	GenericTilesInit();

	ula_init(311, 228, 14361);

	SpecDoReset();

	return 0;
}

static INT32 get_type()
{
	char *rn = NULL;

	if (!BurnDrvGetRomName(&rn, 0, 0))
	{
		INT32 len = strlen(rn);

		if (len > 4) {
			if (!strcasecmp(".z80", rn + (len-4))) {
				return SPEC_Z80;
			}
			if (!strcasecmp(".tap", rn + (len-4))) {
				return SPEC_TAP;
			}
		}
	}

	return 0;
}

static INT32 SpecInit()
{
	return SpectrumInit(get_type());
}

static INT32 Spec128KInit()
{
	return Spectrum128Init(SPEC_128K | get_type());
}

static INT32 Spec128KPlus2Init()
{
	return Spectrum128Init(SPEC_128K | SPEC_PLUS2 | get_type());
}

static INT32 Spec128KInvesInit()
{
	return Spectrum128Init(SPEC_128K | SPEC_INVES | get_type());
}

static INT32 SpecExit()
{
	ZetExit();

	if (SpecMode & SPEC_AY8910) AY8910Exit(0);

	BuzzerExit();

	GenericTilesExit();

	BurnFreeMemIndex();

	return 0;
}

static void SpecCalcPalette()
{
	UINT32 spec_pal[0x10] = {
		0x000000, 0x0000bf, 0xbf0000, 0xbf00bf,
		0x00bf00, 0x00bfbf, 0xbfbf00, 0xbfbfbf,
		0x000000, 0x0000ff, 0xff0000, 0xff00ff,
		0x00ff00, 0x00ffff, 0xffff00, 0xffffff
	};

	for (INT32 i = 0; i < 0x10; i++) {
		SpecPalette[i] = BurnHighCol((spec_pal[i] >> 16) & 0xff, (spec_pal[i] >> 8) & 0xff, spec_pal[i] & 0xff, 0);
	}
}

static INT32 SpecDraw()
{
	if (SpecRecalc) {
		SpecCalcPalette();
		SpecRecalc = 0;
	}

	BurnTransferCopy(SpecPalette);

	return 0;
}

// dink's ULA simulator 2000 SSEI (super special edition intense)
static INT32 CONT_START;
static INT32 CONT_END;
static INT32 BORDER_START;
static INT32 BORDER_END;

static void ula_init(INT32 scanlines, INT32 cyc_scanline, INT32 contention)
{
	SpecScanlines = scanlines;
	SpecCylesPerScanline = cyc_scanline;
	SpecContention = contention;
	CONT_START  = SpecContention; // 48k; 14335 / 128k: 14361
	CONT_END    = (CONT_START + 192 * SpecCylesPerScanline);

	BORDER_START= SpecContention + 3; // start at first vis. pixel (top-left)
	BORDER_START-=(SpecCylesPerScanline * 16) + 8; // go back 16 lines
	BORDER_END  = SpecCylesPerScanline * (16+256+16);
}

#if 0
static UINT8 ula_snow()
{
	extern int counter;
	return (ActiveZ80GetI() >= 0x40 && ActiveZ80GetI() <= 0x7f && (ActiveZ80GetLastOp() & counter) == 0/* && (rand() & 0x187)==0x181*/) ? (ActiveZ80GetR() & 0x7f) : 0x00;
}
#endif

static void ula_run_cyc(INT32 cyc, INT32 draw_screen)
{
	ula_byte = 0xff; // open-bus byte 0xff unless ula fetches screen data

	// borders (top + sides + bottom) - note: only drawing 16px borders here!
	if (cyc >= BORDER_START && cyc <= BORDER_END) {
		INT32 offset = cyc - BORDER_START;
		INT32 x = ((offset) % SpecCylesPerScanline) * 2;
		INT32 y =  (offset) / SpecCylesPerScanline;
		INT32 border = ula_border & 0x07;

		INT32 draw = 0;

		// top border
		if (y >= 0 && y < 16 && x >= 0 && x < nScreenWidth) {
			draw = 1;
		}

		// side borders
		if (y >= 16 && y < 16+192+16 && ((x >= 0 && x < 16) || (x >= 16+256 && x < 16+256+16)) && x < nScreenWidth) {
			draw = 1;
		}

		// bottom border
		if (y >= 16 + 192 && y < 16+192+16 && x >= 0 && x < nScreenWidth) {
			draw = 1;
		}

		if (draw) {
			if (draw_screen && ((x & 7) == 0) && x <= nScreenWidth - 8) {
				for (INT32 xx = x; xx < x + 8; xx++) {
					pTransDraw[y * nScreenWidth + xx] = border;
				}
			}
		}
	}

	// active area
	if (cyc > CONT_START && cyc <= CONT_END) {
		INT32 offset = cyc - (CONT_START+1); // get to start of contention pattern
		INT32 x = (offset) % SpecCylesPerScanline;
		INT32 y = (offset) / SpecCylesPerScanline;

		if (x < 128) {
			switch (x & 7) {
				default: // 0, 1, 6, 7
					ula_byte = 0xff;
					break;

				case 2:
				case 4:
					ula_scr = SpecVideoRam[((y & 0xc0) << 5) | ((y & 0x38) << 2) | ((y & 0x07) << 8) | (x >> 2)];// | ula_snow()<<1];
					ula_byte = ula_scr;
					break;

				case 3:
				case 5:
					ula_attr = SpecVideoRam[0x1800 | ((y & 0xf8) << 2) | (x >> 2)];
					ula_byte = ula_attr;

					UINT16 *dst = pTransDraw + ((y + 16) * nScreenWidth) + (((x << 1) + 16) & ~7);
					UINT8 ink = (ula_attr & 0x07) + ((ula_attr >> 3) & 0x08);
					UINT8 pap = (ula_attr >> 3) & 0x0f;

					if (ula_flash & 0x10 && ula_attr & 0x80) ula_scr = ~ula_scr;

					if (draw_screen) {
						*dst++ = (ula_scr & (1 << 7)) ? ink : pap;
						*dst++ = (ula_scr & (1 << 6)) ? ink : pap;
						*dst++ = (ula_scr & (1 << 5)) ? ink : pap;
						*dst++ = (ula_scr & (1 << 4)) ? ink : pap;
						*dst++ = (ula_scr & (1 << 3)) ? ink : pap;
						*dst++ = (ula_scr & (1 << 2)) ? ink : pap;
						*dst++ = (ula_scr & (1 << 1)) ? ink : pap;
						*dst++ = (ula_scr & (1 << 0)) ? ink : pap;
					}
					break;
			}
		}
	}
}

static void update_ula(INT32 cycle)
{
	//bprintf(0, _T("update_ula:  %d   last %d\n"), cycle, ula_last_cyc);
	if (cycle == -1) {
		// next frame!
		ula_last_cyc = 0;
		return;
	}

	for (INT32 i = ula_last_cyc+1; i <= cycle; i++) {
		ula_run_cyc(i, 1);
	}

	ula_last_cyc = cycle;
}

static void mix_dcblock(INT16 *inbuf, INT16 *outbuf, INT32 sample_nums)
{
	INT16 out;

	for (INT32 sample = 0; sample < sample_nums; sample++)
	{
		INT16 result = inbuf[sample * 2 + 0]; // source sample

		// dc block
		out = result - dac_lastin + 0.998 * dac_lastout;
		dac_lastin = result;
		dac_lastout = out;

		out *= 2.5;

		// add to stream (+include ay if Spec128)
		if (SpecMode & SPEC_AY8910) {
			outbuf[sample * 2 + 0] = BURN_SND_CLIP(outbuf[sample * 2 + 0] + out);
			outbuf[sample * 2 + 1] = BURN_SND_CLIP(outbuf[sample * 2 + 1] + out);
		} else {
			outbuf[sample * 2 + 0] = BURN_SND_CLIP(out);
			outbuf[sample * 2 + 1] = BURN_SND_CLIP(out);
		}
	}
}

static void SwapByte(UINT8 &a, UINT8 &b)
{ // a <-> b, using xor
	a = a ^ b;
	b = a ^ b;
	a = a ^ b;
}

static INT32 SpecFrame()
{
	if (SpecReset) SpecDoReset();

	if (SpecMode & SPEC_TAP) TAPAutoLoadRobot();

	{
		// Init keyboard matrix & 128k secondary matrix (caps/shifts lines) active-low (0x1f)
		SpecInput[0] = SpecInput[1] = SpecInput[2] = SpecInput[3] = SpecInput[4] = SpecInput[5] = SpecInput[6] = SpecInput[7] = 0x1f;
		SpecInput[11] = SpecInput[12] = SpecInput[13] = SpecInput[14] = SpecInput[15] = 0x1f;

		SpecInput[8] = 0x00; // kempston joy (active high)
		SpecInput[9] = SpecInput[10] = 0x1f; // intf2 joy (active low)

		if (SpecDips[0] & 0x01) { // use intf2: map kempston joy1 to intf2 joy1
			SpecInputKbd[9][1] = SpecInputKbd[8][3];
			SpecInputKbd[9][2] = SpecInputKbd[8][2];
			SpecInputKbd[9][4] = SpecInputKbd[8][1];
			SpecInputKbd[9][3] = SpecInputKbd[8][0];
			SpecInputKbd[9][0] = SpecInputKbd[8][4];

			if (SpecDips[0] & 0x20) { // swap intf2 joy1 <-> joy2
				SwapByte(SpecInputKbd[9][1], SpecInputKbd[10][3]);
				SwapByte(SpecInputKbd[9][2], SpecInputKbd[10][2]);
				SwapByte(SpecInputKbd[9][4], SpecInputKbd[10][0]);
				SwapByte(SpecInputKbd[9][3], SpecInputKbd[10][1]);
				SwapByte(SpecInputKbd[9][0], SpecInputKbd[10][4]);
			}
		}

		if (SpecDips[0] & (0x02|0x04)) { // map Kempston to QAOPM/QAOP SPACE
			SpecInputKbd[2][0] |= SpecInputKbd[8][3]; // Up -> Q
			SpecInputKbd[1][0] |= SpecInputKbd[8][2]; // Down -> A
			SpecInputKbd[5][1] |= SpecInputKbd[8][1]; // Left -> O
			SpecInputKbd[5][0] |= SpecInputKbd[8][0]; // Right -> P

			switch (SpecDips[0] & (0x02|0x04)) {
				case 0x02:
					SpecInputKbd[7][2] |= SpecInputKbd[8][4]; // Fire -> M
					break;
				case 0x04:
					SpecInputKbd[7][0] |= SpecInputKbd[8][4]; // Fire -> Space
					break;
			}
		}

		if (SpecDips[0] & 0x08) { // map kempston joy1 to Cursor Keys
			SpecInputKbd[4][3] |= SpecInputKbd[8][3]; // Up -> 7
			SpecInputKbd[4][4] |= SpecInputKbd[8][2]; // Down -> 6
			SpecInputKbd[3][4] |= SpecInputKbd[8][1]; // Left -> 5
			SpecInputKbd[4][2] |= SpecInputKbd[8][0]; // Right -> 8
			SpecInputKbd[4][0] |= SpecInputKbd[8][4]; // Fire -> 0
		}

		if (SpecDips[0] & 0x40) { // map kempston joy1 to Jumping Jack keys
			SpecInputKbd[7][1] |= SpecInputKbd[8][1]; // Left -> Symbol Shift
			SpecInputKbd[7][0] |= SpecInputKbd[8][0]; // Right -> Space
			SpecInputKbd[0][0] |= SpecInputKbd[8][4]; // Fire -> Caps Shift
		}

		// Compile the input matrix
		for (INT32 i = 0; i < 5; i++) {
			for (INT32 j = 0; j < 0x10; j++) { // 8x5 keyboard matrix
				SpecInput[j] ^= (SpecInputKbd[j][i] & 1) << i;
			}
		}

		// Disable inactive hw
		if ((SpecDips[0] & 0x1f) != 0x00) { // kempston not selected
			SpecInput[8] = 0xff; // kempston joy (active high, though none present returns 0xff)
		}
		if ((SpecDips[0] & 0x1f) != 0x01) { // intf2 not selected
			SpecInput[9] = SpecInput[10] = 0x1f; // intf2 joy (active low)
		}
	}

	INT32 nCyclesDo = 0;

	ZetNewFrame();
	ZetOpen(0);
	ZetIdle(nExtraCycles);
	nExtraCycles = 0;

	const INT32 IRQ_LENGTH = 38; // 48k 32, 128k 36 (z80 core takes care of this - only allowing irq for proper length of machine, so value here is OK)

	for (INT32 i = 0; i < SpecScanlines; i++) {
		if (i == 0) {
			ZetSetIRQLine(0, CPU_IRQSTATUS_ACK);
			ZetRun(IRQ_LENGTH);
			ZetSetIRQLine(0, CPU_IRQSTATUS_NONE);
			ula_flash = (ula_flash + 1) & 0x1f;
		}

		nCyclesDo += SpecCylesPerScanline;
		ZetRun(nCyclesDo - ZetTotalCycles());

		if (SpecMode & SPEC_INVES) {
			update_ula(ZetTotalCycles());
		}
	}

	if (SpecMode & SPEC_INVES) {
		update_ula(-1); // notify start next frame
	}

	if (pBurnDraw) {
		SpecDraw();
	}

	if (pBurnSoundOut) {
		if (SpecMode & SPEC_AY8910) {
			AY8910Render(pBurnSoundOut, nBurnSoundLen);
		}

		BuzzerRender(dacbuf);
	    mix_dcblock(dacbuf, pBurnSoundOut, nBurnSoundLen);
	}

	INT32 tot_frame = SpecScanlines * SpecCylesPerScanline;
	nExtraCycles = ZetTotalCycles() - tot_frame;

	ZetClose();

	return 0;
}

static INT32 SpecScan(INT32 nAction, INT32* pnMin)
{
	if (pnMin != NULL) {
		*pnMin = 0x029744;
	}

	if (nAction & ACB_MEMORY_RAM) {
		ScanVar(RamStart, RamEnd-RamStart, "All RAM");
	}

	if (nAction & ACB_DRIVER_DATA) {
		ZetScan(nAction);

		if (SpecMode & SPEC_AY8910) {
			AY8910Scan(nAction, pnMin);
		}

		SCAN_VAR(ula_attr);
		SCAN_VAR(ula_scr);
		SCAN_VAR(ula_byte);
		SCAN_VAR(ula_border);
		SCAN_VAR(ula_flash);
		SCAN_VAR(ula_last_cyc);

		SCAN_VAR(Spec128kMapper);
		SCAN_VAR(Spec128kMapper2);

		SCAN_VAR(nExtraCycles);

		if (SpecMode & SPEC_TAP) {
			// .TAP
			SCAN_VAR(SpecTAPBlocknum);
			// .TAP Robot
			SCAN_VAR(CASAutoLoadPos);
			SCAN_VAR(CASAutoLoadTicker);
			SCAN_VAR(CASFrameCounter);
		}
	}

	if (nAction & ACB_WRITE && SpecMode & SPEC_128K) {
		ZetOpen(0);
		spectrum128_bank();
		ZetClose();
	}

	return 0;
}

static INT32 SpectrumGetZipName(char** pszName, UINT32 i)
{
	static char szFilename[MAX_PATH];
	char* pszGameName = NULL;

	if (pszName == NULL) {
		return 1;
	}

	if (i == 0) {
		pszGameName = BurnDrvGetTextA(DRV_NAME);
	} else {
		if (i == 1 && BurnDrvGetTextA(DRV_BOARDROM)) {
			pszGameName = BurnDrvGetTextA(DRV_BOARDROM);
		} else {
			pszGameName = BurnDrvGetTextA(DRV_PARENT);
		}
	}

	if (pszGameName == NULL || i > 2) {
		*pszName = NULL;
		return 1;
	}

	// remove leader
	memset(szFilename, 0, MAX_PATH);
	for (UINT32 j = 0; j < (strlen(pszGameName) - 5); j++) {
		szFilename[j] = pszGameName[j + 5];
	}

	*pszName = szFilename;

	return 0;
}

// BIOS Handling
static struct BurnRomInfo emptyRomDesc[] = {
	{ "", 0, 0, 0 },
};

static struct BurnRomInfo SpectrumRomDesc[] = {
	{ "spectrum.rom",  0x04000, 0xddee531f, BRF_BIOS },
};

STD_ROM_PICK(Spectrum)
STD_ROM_FN(Spectrum)

static struct BurnRomInfo Spec128RomDesc[] = {
	{ "zx128_0.rom",   0x04000, 0xe76799d2, BRF_BIOS },
	{ "zx128_1.rom",   0x04000, 0xb96a36be, BRF_BIOS },
};

STD_ROM_PICK(Spec128)
STD_ROM_FN(Spec128)

static struct BurnRomInfo Spec1282aRomDesc[] = {
	{ "zx128_2a_0.rom",   0x04000, 0x30c9f490, BRF_BIOS },
	{ "zx128_2a_1.rom",   0x04000, 0xa7916b3f, BRF_BIOS },
	{ "zx128_2a_2.rom",   0x04000, 0xc9a0b748, BRF_BIOS },
	{ "zx128_2a_3.rom",   0x04000, 0xb88fd6e3, BRF_BIOS },
};

STD_ROM_PICK(Spec1282a)
STD_ROM_FN(Spec1282a)

struct BurnDriver BurnSpecSpectrumBIOS = {
	"spec_spectrum", NULL, NULL, NULL, "1982",
	"ZX Spectrum BIOS\0", "BIOS Only", "Sinclair Research Limited", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_BOARDROM, 0, HARDWARE_SPECTRUM, GBF_BIOS, 0,
	SpectrumGetZipName, SpectrumRomInfo, SpectrumRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

struct BurnDriver BurnSpecSpectrum = {
	"spec_spec48k", NULL, NULL, NULL, "1982",
	"ZX Spectrum 48K\0", NULL, "Sinclair Research Limited", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MISC, 0,
	SpectrumGetZipName, SpectrumRomInfo, SpectrumRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

struct BurnDriver BurnSpecSpec128BIOS = {
	"spec_spec128", NULL, NULL, NULL, "1986",
	"ZX Spectrum 128 BIOS\0", "BIOS Only", "Sinclair Research Limited", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_BOARDROM, 0, HARDWARE_SPECTRUM, GBF_BIOS, 0,
	SpectrumGetZipName, Spec128RomInfo, Spec128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

struct BurnDriver BurnSpecSpec1282aBIOS = {
	"spec_spec1282a", NULL, NULL, NULL, "1986",
	"ZX Spectrum 128 +2a BIOS\0", "BIOS Only", "Sinclair Research Limited", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_BOARDROM, 0, HARDWARE_SPECTRUM, GBF_BIOS, 0,
	SpectrumGetZipName, Spec1282aRomInfo, Spec1282aRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KPlus2Init, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

struct BurnDriver BurnSpecSpec128 = {
	"spec_spec128k", NULL, NULL, NULL, "1986",
	"ZX Spectrum 128K\0", NULL, "Sinclair Research Limited", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MISC, 0,
	SpectrumGetZipName, Spec128RomInfo, Spec128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

struct BurnDriver BurnSpecSpec1282a = {
	"spec_spec128k2a", NULL, NULL, NULL, "1986",
	"ZX Spectrum 128K +2a\0", NULL, "Sinclair Research Limited", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MISC, 0,
	SpectrumGetZipName, Spec1282aRomInfo, Spec1282aRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KPlus2Init, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// games

// 180 (128K)

static struct BurnRomInfo Spec180RomDesc[] = {
	{ "180 (1986)(Mastertronic Added Dimension)[128K].tap", 47982, 0xccbd1a31, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec180, Spec180, Spec128)
STD_ROM_FN(Spec180)

struct BurnDriver BurnSpec180 = {
	"spec_180", NULL, "spec_spec128", NULL, "1986",
	"180 (128K)\0", NULL, "MAD (Mastertronic Added Dimension)", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, Spec180RomInfo, Spec180RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 1942 (48K)

static struct BurnRomInfo Spec1942RomDesc[] = {
	{ "1942 (1986)(Elite Systems).z80", 0x08a78, 0x82b77807, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec1942, Spec1942, Spectrum)
STD_ROM_FN(Spec1942)

struct BurnDriver BurnSpec1942 = {
	"spec_1942", NULL, "spec_spectrum", NULL, "1986",
	"1942 (48K)\0", NULL, "Elite Systems", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, Spec1942RomInfo, Spec1942RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 1943 - The Battle of Midway (48K)

static struct BurnRomInfo Spec1943RomDesc[] = {
	{ "1943 - The Battle of Midway (1988)(Go!).z80", 0x0996f, 0xc97f9144, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec1943, Spec1943, Spectrum)
STD_ROM_FN(Spec1943)

struct BurnDriver BurnSpec1943 = {
	"spec_1943", NULL, "spec_spectrum", NULL, "1988",
	"1943 - The Battle of Midway (48K)\0", NULL, "Go!", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, Spec1943RomInfo, Spec1943RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 720 Degrees (48K)

static struct BurnRomInfo Spec720degRomDesc[] = {
	{ "720 Degrees (1987)(U.S. Gold).tap", 50644, 0x842e1279, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec720deg, Spec720deg, Spectrum)
STD_ROM_FN(Spec720deg)

struct BurnDriver BurnSpec720deg = {
	"spec_720deg", NULL, "spec_spectrum", NULL, "1986",
	"720 Degrees (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, Spec720degRomInfo, Spec720degRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Abu Simbel Profanation (Spanish) (48K)

static struct BurnRomInfo SpecabusimprdRomDesc[] = {
	{ "Abu Simbel Profanation (1985)(Dinamic Software).tap", 46362, 0x7849893d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specabusimprd, Specabusimprd, Spectrum)
STD_ROM_FN(Specabusimprd)

struct BurnDriver BurnSpecabusimprd = {
	"spec_abusimprd", "spec_abusimpr", "spec_spectrum", NULL, "1985",
	"Abu Simbel Profanation (Spanish) (48K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecabusimprdRomInfo, SpecabusimprdRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Abu Simbel Profanation (English) (48K)

static struct BurnRomInfo SpecabusimprRomDesc[] = {
	{ "Abu Simbel Profanation (1987)(Gremlin Graphics Software)[re-release].tap", 47312, 0x0635946a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specabusimpr, Specabusimpr, Spectrum)
STD_ROM_FN(Specabusimpr)

struct BurnDriver BurnSpecabusimpr = {
	"spec_abusimpr", NULL, "spec_spectrum", NULL, "1987",
	"Abu Simbel Profanation (English) (48K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecabusimprRomInfo, SpecabusimprRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIssue2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ace of Aces (48K)

static struct BurnRomInfo SpecaceofacesRomDesc[] = {
	{ "Ace of Aces (1986)(U.S. Gold).tap", 89410, 0xd8b0115b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specaceofaces, Specaceofaces, Spectrum)
STD_ROM_FN(Specaceofaces)

struct BurnDriver BurnSpecaceofaces = {
	"spec_aceofaces", NULL, "spec_spectrum", NULL, "1986",
	"Ace of Aces (48K)\0", "Keys from 1 to 4 to change view - 'M' to display map", "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_SIM, 0,
	SpectrumGetZipName, SpecaceofacesRomInfo, SpecaceofacesRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Action Fighter (128K)

static struct BurnRomInfo SpecafighterRomDesc[] = {
	{ "Action Fighter (1989)(Firebird Software)[128K].z80", 0x0a81a, 0x55f30b2a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specafighter, Specafighter, Spec128)
STD_ROM_FN(Specafighter)

struct BurnDriver BurnSpecafighter = {
	"spec_afighter", NULL, "spec_spec128", NULL, "1989",
	"Action Fighter (128K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecafighterRomInfo, SpecafighterRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Addams Family, The (128K)

static struct BurnRomInfo SpecaddfamthRomDesc[] = {
	{ "Addams Family, The 128K (1991)(Ocean).tap", 131557, 0x10ca8414, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specaddfamth, Specaddfamth, Spec128)
STD_ROM_FN(Specaddfamth)

struct BurnDriver BurnSpecaddfamth = {
	"spec_addfamth", NULL, "spec_spec128", NULL, "1991",
	"Addams Family, The (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecaddfamthRomInfo, SpecaddfamthRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Adidas Championship Football (128K)

static struct BurnRomInfo SpecadichafoRomDesc[] = {
	{ "Adidas Championship Football 128K (1990)(Ocean).tap", 106547, 0x2b13ba5a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specadichafo, Specadichafo, Spec128)
STD_ROM_FN(Specadichafo)

struct BurnDriver BurnSpecadichafo = {
	"spec_adichafo", NULL, "spec_spec128", NULL, "1990",
	"Adidas Championship Football (128K)\0", "Set '2. Sinclair' to use controller", "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL, 0,
	SpectrumGetZipName, SpecadichafoRomInfo, SpecadichafoRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Advanced Lawnmower Simulator II (128K)

static struct BurnRomInfo Specadvlawn2RomDesc[] = {
	{ "Advanced Lawnmower Simulator II 128K (1990)(J. Aron).tap", 7424, 0x20f80b05, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specadvlawn2, Specadvlawn2, Spec128)
STD_ROM_FN(Specadvlawn2)

struct BurnDriver BurnSpecadvlawn2 = {
	"spec_advlawn2", NULL, "spec_spec128", NULL, "1990",
	"Advanced Lawnmower Simulator II (128K)\0", NULL, "J. Aron Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specadvlawn2RomInfo, Specadvlawn2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Adventures of Buratino, The (Russian) (128K)

static struct BurnRomInfo SpecburatinoRomDesc[] = {
	{ "Adventures of Buratino, The (1993)(Copper Feet)(128k).z80", 0x096ec, 0x3a0cb189, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specburatino, Specburatino, Spec128)
STD_ROM_FN(Specburatino)

struct BurnDriver BurnSpecburatino = {
	"spec_buratino", NULL, "spec_spec128", NULL, "1993",
	"Adventures of Buratino, The (Russian) (128K)\0", "Choose '4' to set Kempston controller", "Copper Feet", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecburatinoRomInfo, SpecburatinoRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// After the War - Part 1 (Spanish) (48K)

static struct BurnRomInfo Specafterthewar1RomDesc[] = {
	{ "After the War - Part 1 ES (1989)(Dinamic).tap", 47283, 0x74ef00c2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specafterthewar1, Specafterthewar1, Spectrum)
STD_ROM_FN(Specafterthewar1)

struct BurnDriver BurnSpecafterthewar1 = {
	"spec_afterthewar1", NULL, "spec_spectrum", NULL, "1989",
	"After the War - Part 1 (Spanish) (48K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, Specafterthewar1RomInfo, Specafterthewar1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// After the War - Part 2 (Spanish) (48K)

static struct BurnRomInfo Specafterthewar2RomDesc[] = {
	{ "After the War - Part 2 ES (1989)(Dinamic).tap", 47483, 0xd36c48ad, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specafterthewar2, Specafterthewar2, Spectrum)
STD_ROM_FN(Specafterthewar2)

struct BurnDriver BurnSpecafterthewar2 = {
	"spec_afterthewar2", "spec_afterthewar1", "spec_spectrum", NULL, "1989",
	"After the War - Part 2 (Spanish) (48K)\0", "Password: 94656981", "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, Specafterthewar2RomInfo, Specafterthewar2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Airwolf (48K)

static struct BurnRomInfo SpecairwolfRomDesc[] = {
	{ "Airwolf (1985)(Elite Systems).tap", 48924, 0x762ff36d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specairwolf, Specairwolf, Spectrum)
STD_ROM_FN(Specairwolf)

struct BurnDriver BurnSpecairwolf = {
	"spec_airwolf", NULL, "spec_spectrum", NULL, "1984",
	"Airwolf (48K)\0", NULL, "Elite Systems", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecairwolfRomInfo, SpecairwolfRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Airwolf II (48K)

static struct BurnRomInfo Specairwolf2RomDesc[] = {
	{ "Airwolf II (1987)(Hit-Pak).tap", 42265, 0x655b28d9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specairwolf2, Specairwolf2, Spectrum)
STD_ROM_FN(Specairwolf2)

struct BurnDriver BurnSpecairwolf2 = {
	"spec_airwolf2", NULL, "spec_spectrum", NULL, "1987",
	"Airwolf II (48K)\0", NULL, "Hit-Pak", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, Specairwolf2RomInfo, Specairwolf2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Alien Syndrome (48K)

static struct BurnRomInfo SpecaliensynRomDesc[] = {
	{ "Alien Syndrome (1988)(ACE).tap", 74240, 0xed57e69d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specaliensyn, Specaliensyn, Spectrum)
STD_ROM_FN(Specaliensyn)

struct BurnDriver BurnSpecaliensyn = {
	"spec_aliensyn", NULL, "spec_spectrum", NULL, "1987-88",
	"Alien Syndrome (48K)\0", NULL, "ACE Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecaliensynRomInfo, SpecaliensynRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Altered Beast (128K)

static struct BurnRomInfo SpecaltbeastRomDesc[] = {
	{ "Altered Beast (1989)(Activision).tap", 191417, 0xf4ce807b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specaltbeast, Specaltbeast, Spec128)
STD_ROM_FN(Specaltbeast)

struct BurnDriver BurnSpecaltbeast = {
	"spec_altbeast", NULL, "spec_spec128", NULL, "1989",
	"Altered Beast (128K)\0", NULL, "Activision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecaltbeastRomInfo, SpecaltbeastRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Andy Capp (128K)

static struct BurnRomInfo SpecandycappRomDesc[] = {
	{ "Andy Capp (1988)(Mirrorsoft).tap", 54909, 0x7c81909f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specandycapp, Specandycapp, Spec128)
STD_ROM_FN(Specandycapp)

struct BurnDriver BurnSpecandycapp = {
	"spec_andycapp", NULL, "spec_spec128", NULL, "1988",
	"Andy Capp (128K)\0", NULL, "Mirrorsoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecandycappRomInfo, SpecandycappRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Angel Nieto Pole 500cc (Spanish) (128K)

static struct BurnRomInfo Specanpole128RomDesc[] = {
	{ "Angel Nieto Pole 500cc 128K ES (1990)(Opera Soft).tap", 68099, 0x010c9f97, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specanpole128, Specanpole128, Spec1282a)
STD_ROM_FN(Specanpole128)

struct BurnDriver BurnSpecanpole128 = {
	"spec_anpole128", NULL, "spec_spec1282a", NULL, "1990",
	"Angel Nieto Pole 500cc (Spanish) (128K)\0", "Press '6', '7' and ENTER to choose and set options", "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, Specanpole128RomInfo, Specanpole128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KPlus2Init, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Angel Nieto Pole 500cc (Spanish) (48K)

static struct BurnRomInfo Specanpole48RomDesc[] = {
	{ "Angel Nieto Pole 500cc 48K ES (1990)(Opera Soft).tap", 68296, 0xf3e448a8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specanpole48, Specanpole48, Spectrum)
STD_ROM_FN(Specanpole48)

struct BurnDriver BurnSpecanpole48 = {
	"spec_anpole48", "spec_anpole128", "spec_spectrum", NULL, "1990",
	"Angel Nieto Pole 500cc (Spanish) (48K)\0", "Press '6', '7' and ENTER to choose and set options", "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, Specanpole48RomInfo, Specanpole48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// APB - All Points Bulletin (128K)

static struct BurnRomInfo SpecapbRomDesc[] = {
	{ "APB - All Points Bulletin 128K (1989)(Domark).tap", 75630, 0x1d6e4ca9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specapb, Specapb, Spec128)
STD_ROM_FN(Specapb)

struct BurnDriver BurnSpecapb = {
	"spec_apb", NULL, "spec_spec128", NULL, "1989",
	"APB - All Points Bulletin (128K)\0", NULL, "Domark", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecapbRomInfo, SpecapbRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Arkanoid (48K)

static struct BurnRomInfo SpecarkanoidRomDesc[] = {
	{ "Arkanoid (1987)(Imagine Software).z80", 0x08ad3, 0x6fa4f00f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specarkanoid, Specarkanoid, Spectrum)
STD_ROM_FN(Specarkanoid)

struct BurnDriver BurnSpecarkanoid = {
	"spec_arkanoid", NULL, "spec_spectrum", NULL, "1987",
	"Arkanoid (48K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_BREAKOUT, 0,
	SpectrumGetZipName, SpecarkanoidRomInfo, SpecarkanoidRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Arkanoid II - Revenge of Doh (48K)

static struct BurnRomInfo Specarkanoid2_48RomDesc[] = {
	{ "Arkanoid - Revenge of Doh 48K (1988)(Imagine).tap", 40118, 0xcb8aa807, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specarkanoid2_48, Specarkanoid2_48, Spectrum)
STD_ROM_FN(Specarkanoid2_48)

struct BurnDriver BurnSpecarkanoid2_48 = {
	"spec_arkanoid2_48", "spec_arkanoid2", "spec_spectrum", NULL, "1988",
	"Arkanoid II - Revenge of Doh (48K)\0", "Z: left - M: right - ENTER: fire and start", "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_BREAKOUT, 0,
	SpectrumGetZipName, Specarkanoid2_48RomInfo, Specarkanoid2_48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Arkanoid II - Revenge of Doh (128K)

static struct BurnRomInfo Specarkanoid2RomDesc[] = {
	{ "Arkanoid - Revenge of Doh 128K (1988)(Imagine).tap", 43938, 0xfd20d4e8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specarkanoid2, Specarkanoid2, Spec128)
STD_ROM_FN(Specarkanoid2)

struct BurnDriver BurnSpecarkanoid2 = {
	"spec_arkanoid2", NULL, "spec_spec128", NULL, "1988",
	"Arkanoid II - Revenge of Doh (128K)\0", "Z: left - M: right - ENTER: fire and start", "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_BREAKOUT, 0,
	SpectrumGetZipName, Specarkanoid2RomInfo, Specarkanoid2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Army Moves (128K)

static struct BurnRomInfo Specarmymvs128RomDesc[] = {
	{ "Army Moves 128K (1987)(Imagine).tap", 68998, 0xdd386d25, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specarmymvs128, Specarmymvs128, Spec128)
STD_ROM_FN(Specarmymvs128)

struct BurnDriver BurnSpecarmymvs128 = {
	"spec_armymvs128", NULL, "spec_spec128", NULL, "1986-87",
	"Army Moves (128K)\0", "Password Part 2: 27351", "Imagine & Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT | GBF_PLATFORM, 0,
	SpectrumGetZipName, Specarmymvs128RomInfo, Specarmymvs128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Army Moves (48K)

static struct BurnRomInfo Specarmymvs48RomDesc[] = {
	{ "Army Moves 48K (1987)(Imagine).tap", 77664, 0x7f80f524, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specarmymvs48, Specarmymvs48, Spectrum)
STD_ROM_FN(Specarmymvs48)

struct BurnDriver BurnSpecarmymvs48 = {
	"spec_armymvs48", "spec_armymvs128", "spec_spectrum", NULL, "1986-87",
	"Army Moves (48K)\0", "Password Part 2: 27351", "Imagine & Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT | GBF_PLATFORM, 0,
	SpectrumGetZipName, Specarmymvs48RomInfo, Specarmymvs48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Aspar GP Master (Spanish) (48K)

static struct BurnRomInfo SpecaspargpmasRomDesc[] = {
	{ "Aspar GP Master ES (1988)(Dinamic).tap", 43743, 0x2a967d97, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specaspargpmas, Specaspargpmas, Spectrum)
STD_ROM_FN(Specaspargpmas)

struct BurnDriver BurnSpecaspargpmas = {
	"spec_aspargpmas", NULL, "spec_spectrum", NULL, "1988",
	"Aspar GP Master (Spanish) (48K)\0", "Set '3. Teclado' to use controller", "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecaspargpmasRomInfo, SpecaspargpmasRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Atic Atac (48K)

static struct BurnRomInfo SpecaticatacRomDesc[] = {
	{ "Atic Atac (1983)(Ultimate Play The Game).z80", 0x08994, 0x56520bdf, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specaticatac, Specaticatac, Spectrum)
STD_ROM_FN(Specaticatac)

struct BurnDriver BurnSpecaticatac = {
	"spec_aticatac", NULL, "spec_spectrum", NULL, "1983",
	"Atic Atac (48K)\0", NULL, "Ultimate Play The Game", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV | GBF_MAZE, 0,
	SpectrumGetZipName, SpecaticatacRomInfo, SpecaticatacRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// ATV Simulator - All Terrain Vehicle (48K)

static struct BurnRomInfo SpecatvsimRomDesc[] = {
	{ "ATV Simulator (1987)(Codemasters).tap", 47048, 0x373a50ad, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specatvsim, Specatvsim, Spectrum)
STD_ROM_FN(Specatvsim)

struct BurnDriver BurnSpecatvsim = {
	"spec_atvsim", NULL, "spec_spectrum", NULL, "1987",
	"ATV Simulator - All Terrain Vehicle (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecatvsimRomInfo, SpecatvsimRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Auf Wiedersehen Monty (128K)

static struct BurnRomInfo SpecaufwiemoRomDesc[] = {
	{ "Auf Wiedersehen Monty 128K (1987)(Gremlin Graphics).tap", 70498, 0x52564ec0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specaufwiemo, Specaufwiemo, Spec128)
STD_ROM_FN(Specaufwiemo)

struct BurnDriver BurnSpecaufwiemo = {
	"spec_aufwiemo", NULL, "spec_spec128", NULL, "1987",
	"Auf Wiedersehen Monty (128K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecaufwiemoRomInfo, SpecaufwiemoRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Avalon (48K)

static struct BurnRomInfo SpecavalonRomDesc[] = {
	{ "Avalon (1984)(Hewson Consultants).z80", 0x083f2, 0xe65ee95c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specavalon, Specavalon, Spectrum)
STD_ROM_FN(Specavalon)

struct BurnDriver BurnSpecavalon = {
	"spec_avalon", NULL, "spec_spectrum", NULL, "1984",
	"Avalon (48K)\0", NULL, "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecavalonRomInfo, SpecavalonRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Avenger (128K)

static struct BurnRomInfo SpecavengerRomDesc[] = {
	{ "Avenger 128K (1986)(Gremlin Graphics).tap", 54471, 0x9d484c92, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specavenger, Specavenger, Spec128)
STD_ROM_FN(Specavenger)

struct BurnDriver BurnSpecavenger = {
	"spec_avenger", NULL, "spec_spec128", NULL, "1986",
	"Avenger (128K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_MAZE, 0,
	SpectrumGetZipName, SpecavengerRomInfo, SpecavengerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Back to Skool (48K)

static struct BurnRomInfo SpecbackskooRomDesc[] = {
	{ "Back to Skool (1985)(Microsphere).z80", 0x0afb3, 0x6bf68f3d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbackskoo, Specbackskoo, Spectrum)
STD_ROM_FN(Specbackskoo)

struct BurnDriver BurnSpecbackskoo = {
	"spec_backskoo", NULL, "spec_spectrum", NULL, "1985",
	"Back to Skool (48K)\0", NULL, "Microsphere", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecbackskooRomInfo, SpecbackskooRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecCursorKeysDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Back to the Future (48K)

static struct BurnRomInfo SpecbackfutuRomDesc[] = {
	{ "Back to the Future 48K (1985)(Electric Dreams).tap", 50449, 0xf29bbae6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbackfutu, Specbackfutu, Spectrum)
STD_ROM_FN(Specbackfutu)

struct BurnDriver BurnSpecbackfutu = {
	"spec_backfutu", NULL, "spec_spectrum", NULL, "1985",
	"Back to the Future (48K)\0", NULL, "Electric Dreams", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecbackfutuRomInfo, SpecbackfutuRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Back to the Future II (128K)

static struct BurnRomInfo Specbackfut2RomDesc[] = {
	{ "Back to the Future II (1990)(Image Works).tap", 0x39d95, 0x464a3359, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbackfut2, Specbackfut2, Spec128)
STD_ROM_FN(Specbackfut2)

struct BurnDriver BurnSpecbackfut2 = {
	"spec_backfut2", NULL, "spec_spec128", NULL, "1990",
	"Back to the Future II (128K)\0", "Set '2 Sinclair' to use controller", "Image Works", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specbackfut2RomInfo, Specbackfut2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Back to the Future III (128K)

static struct BurnRomInfo Specbackfut3RomDesc[] = {
	{ "Back to the Future III (1991)(Image Works).tap", 0x3661a, 0x5de2c954, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbackfut3, Specbackfut3, Spec128)
STD_ROM_FN(Specbackfut3)

struct BurnDriver BurnSpecbackfut3 = {
	"spec_backfut3", NULL, "spec_spec128", NULL, "1991",
	"Back to the Future III (128K)\0", NULL, "Image Works", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specbackfut3RomInfo, Specbackfut3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Badlands (128K)

static struct BurnRomInfo SpecbadlandsRomDesc[] = {
	{ "Badlands (1990)(Domark)(48K-128K).tap", 33672, 0x5df53959, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbadlands, Specbadlands, Spec128)
STD_ROM_FN(Specbadlands)

struct BurnDriver BurnSpecbadlands = {
	"spec_badlands", NULL, "spec_spec128", NULL, "1990",
	"Badlands (128K)\0", NULL, "Domark", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecbadlandsRomInfo, SpecbadlandsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Barbarian: The Ultimate Warrior - Part 2 (48K)

static struct BurnRomInfo Specbarbpt2RomDesc[] = {
	{ "Barbarian - 1 Player (1987)(Palace Software).z80", 0x0a596, 0x5f06bc26, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbarbpt2, Specbarbpt2, Spectrum)
STD_ROM_FN(Specbarbpt2)

struct BurnDriver BurnSpecbarbpt2 = {
	"spec_barbpt2", "spec_barbpt1", "spec_spectrum", NULL, "1987",
	"Barbarian: The Ultimate Warrior - Part 2 (48K)\0", "only one player mode", "Palace Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, Specbarbpt2RomInfo, Specbarbpt2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Barbarian: The Ultimate Warrior - Part 1 (48K)

static struct BurnRomInfo Specbarbpt1RomDesc[] = {
	{ "Barbarian - 2 Players (1987)(Palace Software).z80", 0x0a2b2, 0xa856967f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbarbpt1, Specbarbpt1, Spectrum)
STD_ROM_FN(Specbarbpt1)

struct BurnDriver BurnSpecbarbpt1 = {
	"spec_barbpt1", NULL, "spec_spectrum", NULL, "1987",
	"Barbarian: The Ultimate Warrior - Part 1 (48K)\0", "one and two player mode", "Palace Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, Specbarbpt1RomInfo, Specbarbpt1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Barbarian (Psygnosis) (48K)

static struct BurnRomInfo SpecbarbarianRomDesc[] = {
	{ "Barbarian (1988)(Melbourne House).tap", 46467, 0x51bf72c1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbarbarian, Specbarbarian, Spectrum)
STD_ROM_FN(Specbarbarian)

struct BurnDriver BurnSpecbarbarian = {
	"spec_barbarian", NULL, "spec_spectrum", NULL, "1988",
	"Barbarian (Psygnosis) (48K)\0", NULL, "Melbourne House", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT | GBF_ADV, 0,
	SpectrumGetZipName, SpecbarbarianRomInfo, SpecbarbarianRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Barbarian II - The Dungeon of Drax (128K)

static struct BurnRomInfo Specbarbarn2RomDesc[] = {
	{ "Barbarian II - The Dungeon of Drax (1988)(Palace Software)[128K].z80", 0x1ac6c, 0x2215c3b7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbarbarn2, Specbarbarn2, Spec128)
STD_ROM_FN(Specbarbarn2)

struct BurnDriver BurnSpecbarbarn2 = {
	"spec_barbarn2", NULL, "spec_spec128", NULL, "1988",
	"Barbarian II - The Dungeon of Drax (128K)\0", "Press 'N' to through doors", "Palace Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, Specbarbarn2RomInfo, Specbarbarn2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Basil the Great Mouse Detective (48K)

static struct BurnRomInfo SpecbasilmouseRomDesc[] = {
	{ "Basil the Great Mouse Detective (1987)(Gremlin Graphics).tap", 49699, 0xcf9534f4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbasilmouse, Specbasilmouse, Spectrum)
STD_ROM_FN(Specbasilmouse)

struct BurnDriver BurnSpecbasilmouse = {
	"spec_basilmouse", NULL, "spec_spectrum", NULL, "1987",
	"Basil the Great Mouse Detective (48K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecbasilmouseRomInfo, SpecbasilmouseRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Batman - The Caped Crusader - Part 1 - A Bird in the Hand (48K)

static struct BurnRomInfo SpecbatmanccRomDesc[] = {
	{ "Batman - The Caped Crusader - A Bird in the Hand (1988)(Ocean).tap", 48076, 0x3102cef1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbatmancc, Specbatmancc, Spectrum)
STD_ROM_FN(Specbatmancc)

struct BurnDriver BurnSpecbatmancc = {
	"spec_batmancc", NULL, "spec_spectrum", NULL, "1988",
	"Batman - The Caped Crusader - Part 1 - A Bird in the Hand (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecbatmanccRomInfo, SpecbatmanccRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Batman - The Caped Crusader - Part 2 - A Fete Worse than Death (48K)

static struct BurnRomInfo Specbatmancc2RomDesc[] = {
	{ "Batman - The Caped Crusader - A Fete Worse than Death (1988)(Ocean).tap", 47806, 0x8591332f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbatmancc2, Specbatmancc2, Spectrum)
STD_ROM_FN(Specbatmancc2)

struct BurnDriver BurnSpecbatmancc2 = {
	"spec_batmancc2", "spec_batmancc", "spec_spectrum", NULL, "1988",
	"Batman - The Caped Crusader - Part 2 - A Fete Worse than Death (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, Specbatmancc2RomInfo, Specbatmancc2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Batman - The Movie (128K)

static struct BurnRomInfo SpecbatmanmvRomDesc[] = {
	{ "Batman - The Movie 128K (1989)(Ocean).tap", 121528, 0x910b0bf4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbatmanmv, Specbatmanmv, Spec128)
STD_ROM_FN(Specbatmanmv)

struct BurnDriver BurnSpecbatmanmv = {
	"spec_batmanmv", NULL, "spec_spec128", NULL, "1989",
	"Batman - The Movie (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecbatmanmvRomInfo, SpecbatmanmvRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Batman - The Puaj Edition (48K)

static struct BurnRomInfo SpecbatmanpeRomDesc[] = {
	{ "Batman - The Puaj Edition (1989)(Micro House).z80", 0x086f6, 0x1ec85d88, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbatmanpe, Specbatmanpe, Spectrum)
STD_ROM_FN(Specbatmanpe)

struct BurnDriver BurnSpecbatmanpe = {
	"spec_batmanpe", NULL, "spec_spectrum", NULL, "1989",
	"Batman - The Puaj Edition (48K)\0", NULL, "Micro House", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE | GBF_ADV, 0,
	SpectrumGetZipName, SpecbatmanpeRomInfo, SpecbatmanpeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Batman (48K)

static struct BurnRomInfo Specbatman_48RomDesc[] = {
	{ "Batman (1986)(Ocean)(48K-128K).tap", 47819, 0x898d7592, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbatman_48, Specbatman_48, Spectrum)
STD_ROM_FN(Specbatman_48)

struct BurnDriver BurnSpecbatman_48 = {
	"spec_batman_48", "spec_batman", "spec_spectrum", NULL, "1986",
	"Batman (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, Specbatman_48RomInfo, Specbatman_48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Batman (128K)

static struct BurnRomInfo SpecbatmanRomDesc[] = {
	{ "Batman (1986)(Ocean)(48K-128K).tap", 47819, 0x898d7592, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbatman, Specbatman, Spec128)
STD_ROM_FN(Specbatman)

struct BurnDriver BurnSpecbatman = {
	"spec_batman", NULL, "spec_spec128", NULL, "1986",
	"Batman (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecbatmanRomInfo, SpecbatmanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Beach Buggy Simulator (48K)

static struct BurnRomInfo SpecbeabugsiRomDesc[] = {
	{ "Beach Buggy Simulator 48K (1988)(Silverbird).tap", 48064, 0x4f4f62f2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbeabugsi, Specbeabugsi, Spectrum)
STD_ROM_FN(Specbeabugsi)

struct BurnDriver BurnSpecbeabugsi = {
	"spec_beabugsi", NULL, "spec_spectrum", NULL, "1988",
	"Beach Buggy Simulator (48K)\0", NULL, "Silverbird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecbeabugsiRomInfo, SpecbeabugsiRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Beyond the Ice Palace (128K)

static struct BurnRomInfo SpecbeyicepaRomDesc[] = {
	{ "Beyond the Ice Palace (1988)(Elite Systems).tap", 46226, 0x565cc58c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbeyicepa, Specbeyicepa, Spec128)
STD_ROM_FN(Specbeyicepa)

struct BurnDriver BurnSpecbeyicepa = {
	"spec_beyicepa", NULL, "spec_spec128", NULL, "1988",
	"Beyond the Ice Palace (128K)\0", NULL, "Elite Systems", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecbeyicepaRomInfo, SpecbeyicepaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bionic Commando (48K)

static struct BurnRomInfo Specbionicc_48RomDesc[] = {
	{ "Bionic Commandos (1988)(Go!).tap", 97616, 0x9093dd3d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbionicc_48, Specbionicc_48, Spectrum)
STD_ROM_FN(Specbionicc_48)

struct BurnDriver BurnSpecbionicc_48 = {
	"spec_bionicc_48", "spec_bionicc", "spec_spectrum", NULL, "1988",
	"Bionic Commando (48K)\0", NULL, "Go!", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, Specbionicc_48RomInfo, Specbionicc_48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bionic Commando (128K)

static struct BurnRomInfo SpecbioniccRomDesc[] = {
	{ "Bionic Commando (1988)(Go!)[128K].z80", 0x166f5, 0x8eb507eb, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbionicc, Specbionicc, Spec128)
STD_ROM_FN(Specbionicc)

struct BurnDriver BurnSpecbionicc = {
	"spec_bionicc", NULL, "spec_spec128", NULL, "1988",
	"Bionic Commando (128K)\0", NULL, "Go!", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecbioniccRomInfo, SpecbioniccRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Black Tiger (128K)

static struct BurnRomInfo SpecblktigerRomDesc[] = {
	{ "Black Tiger (1989)(U.S. Gold).tap", 122820, 0x12912cc8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specblktiger, Specblktiger, Spec128)
STD_ROM_FN(Specblktiger)

struct BurnDriver BurnSpecblktiger = {
	"spec_blktiger", NULL, "spec_spec128", NULL, "1989",
	"Black Tiger (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecblktigerRomInfo, SpecblktigerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Blinky's Scary School (128K)

static struct BurnRomInfo SpecblinkysRomDesc[] = {
	{ "Blinky's Scary School (1990)(Zeppelin Games).tap", 53665, 0x6fcf88b5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specblinkys, Specblinkys, Spec128)
STD_ROM_FN(Specblinkys)

struct BurnDriver BurnSpecblinkys = {
	"spec_blinkys", NULL, "spec_spec128", NULL, "1990",
	"Blinky's Scary School (128K)\0", NULL, "Zeppelin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecblinkysRomInfo, SpecblinkysRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// BMX Freestyle Simulator (48K)

static struct BurnRomInfo Specbmxfrees48RomDesc[] = {
	{ "BMX Freestyle Simulator (1989)(Codemasters).tap", 41824, 0x711d4d6a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbmxfrees48, Specbmxfrees48, Spectrum)
STD_ROM_FN(Specbmxfrees48)

struct BurnDriver BurnSpecbmxfrees48 = {
	"spec_bmxfrees48", "spec_bmxfrees", "spec_spectrum", NULL, "1989",
	"BMX Freestyle Simulator (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, Specbmxfrees48RomInfo, Specbmxfrees48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// BMX Freestyle Simulator (128K)

static struct BurnRomInfo SpecbmxfreesRomDesc[] = {
	{ "BMX Freestyle Simulator (1989)(Codemasters).tap", 41824, 0x711d4d6a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbmxfrees, Specbmxfrees, Spec128)
STD_ROM_FN(Specbmxfrees)

struct BurnDriver BurnSpecbmxfrees = {
	"spec_bmxfrees", NULL, "spec_spec128", NULL, "1989",
	"BMX Freestyle Simulator (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecbmxfreesRomInfo, SpecbmxfreesRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bomb Jack (48K)

static struct BurnRomInfo Specbombjack_48RomDesc[] = {
	{ "Bomb Jack (1986)(Elite Systems).z80", 0x0a30a, 0x00e95211, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbombjack_48, Specbombjack_48, Spectrum)
STD_ROM_FN(Specbombjack_48)

struct BurnDriver BurnSpecbombjack_48 = {
	"spec_bombjack_48", NULL, "spec_spectrum", NULL, "1986",
	"Bomb Jack (48K)\0", NULL, "Elite Systems", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Specbombjack_48RomInfo, Specbombjack_48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bomb Jack II (48K)

static struct BurnRomInfo Specbmbjack2RomDesc[] = {
	{ "Bomb Jack II (1987)(Elite Systems).z80", 0x0b1b2, 0x6327f471, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbmbjack2, Specbmbjack2, Spectrum)
STD_ROM_FN(Specbmbjack2)

struct BurnDriver BurnSpecbmbjack2 = {
	"spec_bmbjack2", NULL, "spec_spectrum", NULL, "1987",
	"Bomb Jack II (48K)\0", NULL, "Elite Systems", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Specbmbjack2RomInfo, Specbmbjack2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Boulder Dash (48K)

static struct BurnRomInfo SpecbouldashRomDesc[] = {
	{ "Boulder Dash (1984)(Front Runner).z80", 0x04849, 0x5d71133f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbouldash, Specbouldash, Spectrum)
STD_ROM_FN(Specbouldash)

struct BurnDriver BurnSpecbouldash = {
	"spec_bouldash", NULL, "spec_spectrum", NULL, "1984",
	"Boulder Dash (48K)\0", NULL, "Front Runner", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecbouldashRomInfo, SpecbouldashRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Boulder Dash II - Rockford's Riot (48K)

static struct BurnRomInfo Specbouldsh2RomDesc[] = {
	{ "Boulder Dash II - Rockford's Riot (1985)(Prism Leisure).z80", 0x057a4, 0xbcab1101, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbouldsh2, Specbouldsh2, Spectrum)
STD_ROM_FN(Specbouldsh2)

struct BurnDriver BurnSpecbouldsh2 = {
	"spec_bouldsh2", NULL, "spec_spectrum", NULL, "1985",
	"Boulder Dash II - Rockford's Riot (48K)\0", NULL, "Prism Leisure", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specbouldsh2RomInfo, Specbouldsh2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Boulder Dash III (48K)

static struct BurnRomInfo Specbouldsh3RomDesc[] = {
	{ "Boulder Dash III (1986)(Prism Leisure).z80", 0x04982, 0xb61f2fae, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbouldsh3, Specbouldsh3, Spectrum)
STD_ROM_FN(Specbouldsh3)

struct BurnDriver BurnSpecbouldsh3 = {
	"spec_bouldsh3", NULL, "spec_spectrum", NULL, "1986",
	"Boulder Dash III (48K)\0", NULL, "Prism Leisure", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specbouldsh3RomInfo, Specbouldsh3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Boulder Dash V (48K)

static struct BurnRomInfo Specbouldsh4RomDesc[] = {
	{ "Boulder Dash V (1992)(Too Trek Moscow S.N.G.).z80", 0x08721, 0x6e9b68bf, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbouldsh4, Specbouldsh4, Spectrum)
STD_ROM_FN(Specbouldsh4)

struct BurnDriver BurnSpecbouldsh4 = {
	"spec_bouldsh4", NULL, "spec_spectrum", NULL, "1992",
	"Boulder Dash V (48K)\0", NULL, "Too Trek Moscow S.N.G.", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specbouldsh4RomInfo, Specbouldsh4RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Boulder Dash VI (48K)

static struct BurnRomInfo Specbouldsh5RomDesc[] = {
	{ "Boulder Dash VI (1992)(Too Trek Moscow S.N.G.).z80", 0x08792, 0x6abfc981, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbouldsh5, Specbouldsh5, Spectrum)
STD_ROM_FN(Specbouldsh5)

struct BurnDriver BurnSpecbouldsh5 = {
	"spec_bouldsh5", NULL, "spec_spectrum", NULL, "1992",
	"Boulder Dash VI (48K)\0", NULL, "Too Trek Moscow S.N.G.", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specbouldsh5RomInfo, Specbouldsh5RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bounder (48K)

static struct BurnRomInfo SpecbounderRomDesc[] = {
	{ "Bounder 48K (1986)(Gremlin Graphics).tap", 56669, 0x9308e223, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbounder, Specbounder, Spectrum)
STD_ROM_FN(Specbounder)

struct BurnDriver BurnSpecbounder = {
	"spec_bounder", NULL, "spec_spectrum", NULL, "1986",
	"Bounder (48K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecbounderRomInfo, SpecbounderRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bruce Lee (48K)

static struct BurnRomInfo SpecbruceleeRomDesc[] = {
	{ "Bruce Lee (1984)(U.S. Gold).z80", 0x08ceb, 0x8298df22, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbrucelee, Specbrucelee, Spectrum)
STD_ROM_FN(Specbrucelee)

struct BurnDriver BurnSpecbrucelee = {
	"spec_brucelee", NULL, "spec_spectrum", NULL, "1984",
	"Bruce Lee (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecbruceleeRomInfo, SpecbruceleeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Boss, The (48K)

static struct BurnRomInfo SpecbosstheRomDesc[] = {
	{ "Boss, The (1984)(Peaksoft).z80", 0x09035, 0x9b7242b2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbossthe, Specbossthe, Spectrum)
STD_ROM_FN(Specbossthe)

struct BurnDriver BurnSpecbossthe = {
	"spec_bossthe", NULL, "spec_spectrum", NULL, "1984",
	"Boss, The (48K)\0", NULL, "Peaksoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_STRATEGY, 0,
	SpectrumGetZipName, SpecbosstheRomInfo, SpecbosstheRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bubble Bobble - The Adventure (128K)

static struct BurnRomInfo SpecbubothadRomDesc[] = {
	{ "Bubble Bobble - The Adventure 128K (1993)(AP's Adventures).tap", 73681, 0x856eb771, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbubothad, Specbubothad, Spec128)
STD_ROM_FN(Specbubothad)

struct BurnDriver BurnSpecbubothad = {
	"spec_bubothad", NULL, "spec_spec128", NULL, "1993",
	"Bubble Bobble - The Adventure (128K)\0", NULL, "AP's Adventures", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecbubothadRomInfo, SpecbubothadRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bubble Bobble (128K)

static struct BurnRomInfo SpecbublboblRomDesc[] = {
	{ "Bubble Bobble (1987)(Firebird).tap", 53224, 0xdce45529, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbublbobl, Specbublbobl, Spec128)
STD_ROM_FN(Specbublbobl)

struct BurnDriver BurnSpecbublbobl = {
	"spec_bublbobl", NULL, "spec_spec128", NULL, "1987",
	"Bubble Bobble (128K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecbublboblRomInfo, SpecbublboblRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bubble Dizzy (48K)

static struct BurnRomInfo Specbubbdizz48RomDesc[] = {
	{ "Bubble Dizzy (1991)(Code Masters)(48K-128K).tap", 50433, 0x8e6625af, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbubbdizz48, Specbubbdizz48, Spectrum)
STD_ROM_FN(Specbubbdizz48)

struct BurnDriver BurnSpecbubbdizz_48 = {
	"spec_bubbdizz48", "spec_bubbdizz128", "spec_spectrum", NULL, "1991",
	"Bubble Dizzy (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specbubbdizz48RomInfo, Specbubbdizz48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bubble Dizzy (128K)

static struct BurnRomInfo Specbubbdizz128RomDesc[] = {
	{ "Bubble Dizzy (1991)(Code Masters)(48K-128K).tap", 50433, 0x8e6625af, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbubbdizz128, Specbubbdizz128, Spec128)
STD_ROM_FN(Specbubbdizz128)

struct BurnDriver BurnSpecbubbdizz128 = {
	"spec_bubbdizz128", NULL, "spec_spec128", NULL, "1991",
	"Bubble Dizzy (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specbubbdizz128RomInfo, Specbubbdizz128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Buck Rogers - Planet of Zoon (48K)

static struct BurnRomInfo SpecbuckrogersRomDesc[] = {
	{ "Buck Rogers - Planet of Zoom 48K (1985)(U.S. Gold).z80", 30201, 0xe8b65171, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbuckrogers, Specbuckrogers, Spectrum)
STD_ROM_FN(Specbuckrogers)

struct BurnDriver BurnSpecbuckrogers = {
	"spec_buckrogers", NULL, "spec_spectrum", NULL, "1984",
	"Buck Rogers - Planet of Zoom (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecbuckrogersRomInfo, SpecbuckrogersRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bullseye (Macsen) (48K)

static struct BurnRomInfo SpecbulleyeRomDesc[] = {
	{ "Bulls Eye (1984)(Macsen Software).z80", 0x09944, 0xce764dd1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbulleye, Specbulleye, Spectrum)
STD_ROM_FN(Specbulleye)

struct BurnDriver BurnSpecbulleye = {
	"spec_bulleye", NULL, "spec_spectrum", NULL, "1984",
	"Bullseye (Macsen) (48K)\0", NULL, "Macsen Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecbulleyeRomInfo, SpecbulleyeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bullseye (Mastertronic) (48K)

static struct BurnRomInfo SpecbullseymRomDesc[] = {
	{ "Bullseye (1982)(Mastertronic)[aka Darts].z80", 0x0935f, 0x1caf68ca, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbullseym, Specbullseym, Spectrum)
STD_ROM_FN(Specbullseym)

struct BurnDriver BurnSpecbullseym = {
	"spec_bullseym", NULL, "spec_spectrum", NULL, "1982",
	"Bullseye (Mastertronic) (48K)\0", NULL, "Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecbullseymRomInfo, SpecbullseymRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Cabal (128K)

static struct BurnRomInfo SpeccabalRomDesc[] = {
	{ "Cabal 128K (1989)(Ocean).tap", 111703, 0x798f3c55, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccabal, Speccabal, Spec128)
STD_ROM_FN(Speccabal)

struct BurnDriver BurnSpeccabal = {
	"spec_cabal", NULL, "spec_spec128", NULL, "1988",
	"Cabal (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpeccabalRomInfo, SpeccabalRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Camelot Warriors (48K)

static struct BurnRomInfo SpeccamelotwRomDesc[] = {
	{ "Camelot Warriors (1986)(Dinamic).tap", 41677, 0x27fdde63, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccamelotw, Speccamelotw, Spectrum)
STD_ROM_FN(Speccamelotw)

struct BurnDriver BurnSpeccamelotw = {
	"spec_camelotw", NULL, "spec_spectrum", NULL, "1986",
	"Camelot Warriors (48K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpeccamelotwRomInfo, SpeccamelotwRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPMDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Capitan Trueno, El - Part 1 (Spanish) (48K)

static struct BurnRomInfo Speccaptrueno1RomDesc[] = {
	{ "Capitan Trueno, El - Part 1 ES (1989)(Dinamic).tap", 48603, 0x8e123899, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccaptrueno1, Speccaptrueno1, Spectrum)
STD_ROM_FN(Speccaptrueno1)

struct BurnDriver BurnSpeccaptrueno1 = {
	"spec_captrueno1", NULL, "spec_spectrum", NULL, "1989",
	"Capitan Trueno, El - Part 1 (Spanish) (48K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT | GBF_PLATFORM, 0,
	SpectrumGetZipName, Speccaptrueno1RomInfo, Speccaptrueno1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Capitan Trueno, El - Part 2 (Spanish) (48K)

static struct BurnRomInfo Speccaptrueno2RomDesc[] = {
	{ "Capitan Trueno, El - Part 2 ES (1989)(Dinamic).tap", 46209, 0x61b80ea8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccaptrueno2, Speccaptrueno2, Spectrum)
STD_ROM_FN(Speccaptrueno2)

struct BurnDriver BurnSpeccaptrueno2 = {
	"spec_captrueno2", "spec_captrueno1", "spec_spectrum", NULL, "1989",
	"Capitan Trueno, El - Part 2 (Spanish) (48K)\0", "Password: 270653", "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT | GBF_PLATFORM, 0,
	SpectrumGetZipName, Speccaptrueno2RomInfo, Speccaptrueno2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Carlos Sainz (Spanish) (128K)

static struct BurnRomInfo SpeccarlossainzRomDesc[] = {
	{ "Carlos Sainz 128K (1990)(Zigurat).tap", 48193, 0x3974f703, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccarlossainz, Speccarlossainz, Spec128)
STD_ROM_FN(Speccarlossainz)

struct BurnDriver BurnSpeccarlossainz = {
	"spec_carlossainz", NULL, "spec_spec128", NULL, "1990",
	"Carlos Sainz (Spanish) (128K)\0", NULL, "Zigurat Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpeccarlossainzRomInfo, SpeccarlossainzRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Cashdash (48K)

static struct BurnRomInfo SpeccashdashRomDesc[] = {
	{ "Cash Dash (19xx)(Tynesoft).tap", 47712, 0xa278579f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccashdash, Speccashdash, Spectrum)
STD_ROM_FN(Speccashdash)

struct BurnDriver BurnSpeccashdash = {
	"spec_cashdash", NULL, "spec_spectrum", NULL, "19xx",
	"Cashdash (48K)\0", NULL, "Tynesoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpeccashdashRomInfo, SpeccashdashRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Cauldron (48K)

static struct BurnRomInfo SpeccauldronRomDesc[] = {
	{ "Cauldron (1985)(Palace).z80", 0x09f7b, 0xadea0ad1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccauldron, Speccauldron, Spectrum)
STD_ROM_FN(Speccauldron)

struct BurnDriver BurnSpeccauldron = {
	"spec_cauldron", NULL, "spec_spectrum", NULL, "1985",
	"Cauldron (48K)\0", NULL, "Palace Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT | GBF_ADV, 0,
	SpectrumGetZipName, SpeccauldronRomInfo, SpeccauldronRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Cauldron II - The Pumpkin Strikes Back (48K)

static struct BurnRomInfo Speccauldrn2RomDesc[] = {
	{ "Cauldron II (1986)(Palace).z80", 0x09349, 0xc73307f0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccauldrn2, Speccauldrn2, Spectrum)
STD_ROM_FN(Speccauldrn2)

struct BurnDriver BurnSpeccauldrn2 = {
	"spec_cauldrn2", NULL, "spec_spectrum", NULL, "1986",
	"Cauldron II - The Pumpkin Strikes Back (48K)\0", NULL, "Palace Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, Speccauldrn2RomInfo, Speccauldrn2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Canyon Warrior (128K)

static struct BurnRomInfo SpeccanywarrRomDesc[] = {
	{ "Canyon Warrior (1989)(Mastertronic Plus).tap", 52451, 0x5574dd40, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccanywarr, Speccanywarr, Spec128)
STD_ROM_FN(Speccanywarr)

struct BurnDriver BurnSpeccanywarr = {
	"spec_canywarr", NULL, "spec_spec128", NULL, "1989",
	"Canyon Warrior (128K)\0", NULL, "Mastertronic Plus", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpeccanywarrRomInfo, SpeccanywarrRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Championship Jet Ski Simulator - Easy (48K)

static struct BurnRomInfo SpecchjesksiRomDesc[] = {
	{ "Championship Jet Ski Simulator - Easy (1989)(Codemasters).z80", 0x08e58, 0xff247fe4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specchjesksi, Specchjesksi, Spectrum)
STD_ROM_FN(Specchjesksi)

struct BurnDriver BurnSpecchjesksi = {
	"spec_chjesksi", NULL, "spec_spectrum", NULL, "1989",
	"Championship Jet Ski Simulator - Easy (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecchjesksiRomInfo, SpecchjesksiRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Championship Jet Ski Simulator - Hard (48K)

static struct BurnRomInfo SpecchjesksihRomDesc[] = {
	{ "Championship Jet Ski Simulator - Hard (1989)(Codemasters).z80", 0x08f27, 0xe54f4b7a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specchjesksih, Specchjesksih, Spectrum)
STD_ROM_FN(Specchjesksih)

struct BurnDriver BurnSpecchjesksih = {
	"spec_chjesksih", "spec_chjesksi", "spec_spectrum", NULL, "1989",
	"Championship Jet Ski Simulator - Hard (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecchjesksihRomInfo, SpecchjesksihRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Championship Sprint (48K)

static struct BurnRomInfo SpecchamspriRomDesc[] = {
	{ "Championship Sprint (1988)(Electric Dreams Software).z80", 0x07073, 0xf557d7f7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specchamspri, Specchamspri, Spectrum)
STD_ROM_FN(Specchamspri)

struct BurnDriver BurnSpecchamspri = {
	"spec_chamspri", NULL, "spec_spectrum", NULL, "1988",
	"Championship Sprint (48K)\0", NULL, "Electric Dreams", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecchamspriRomInfo, SpecchamspriRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Chase H.Q. (48K)

static struct BurnRomInfo Specchasehq48RomDesc[] = {
	{ "Chase H.Q. (1989)(Ocean).tap", 118340, 0x236a9884, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specchasehq48, Specchasehq48, Spectrum)
STD_ROM_FN(Specchasehq48)

struct BurnDriver BurnSpecchasehq48 = {
	"spec_chasehq48", "spec_chasehq128", "spec_spectrum", NULL, "1989",
	"Chase H.Q. (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, Specchasehq48RomInfo, Specchasehq48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Chase H.Q. (128K)

static struct BurnRomInfo Specchasehq128RomDesc[] = {
	{ "Chase H.Q. (1989)(Ocean).tap", 118340, 0x236a9884, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specchasehq128, Specchasehq128, Spec128)
STD_ROM_FN(Specchasehq128)

struct BurnDriver BurnSpecchasehq128 = {
	"spec_chasehq128", NULL, "spec_spec128", NULL, "1989",
	"Chase H.Q. (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, Specchasehq128RomInfo, Specchasehq128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Chase H.Q. II - Special Criminal Investigations (128K)

static struct BurnRomInfo Specchasehq2RomDesc[] = {
	{ "Chase H.Q. II - Special Criminal Investigations 128K (1990)(Ocean).tap", 122661, 0xfd47cf79, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specchasehq2, Specchasehq2, Spec128)
STD_ROM_FN(Specchasehq2)

struct BurnDriver BurnSpecchasehq2 = {
	"spec_chasehq2", NULL, "spec_spec128", NULL, "1990",
	"Chase H.Q. II - Special Criminal Investigations (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING | GBF_SHOOT, 0,
	SpectrumGetZipName, Specchasehq2RomInfo, Specchasehq2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Chequered Flag (48K)

static struct BurnRomInfo SpeccheqflagRomDesc[] = {
	{ "Chequered Flag (1982)(Sinclair Research).z80", 0x08e6f, 0xbe6e657f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccheqflag, Speccheqflag, Spectrum)
STD_ROM_FN(Speccheqflag)

struct BurnDriver BurnSpeccheqflag = {
	"spec_cheqflag", NULL, "spec_spectrum", NULL, "1982",
	"Chequered Flag (48K)\0", "'A-F' to steer, '0' to accelerate, 'I' to brake and 'M-N' to shift gears", "Sinclair Research", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpeccheqflagRomInfo, SpeccheqflagRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Chicago 30's (Spanish) (48K)

static struct BurnRomInfo Specchicago30RomDesc[] = {
	{ "Chicago 30's ES 48K (1988)(Topo Soft).tap", 48259, 0xfbcd39d9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specchicago30, Specchicago30, Spectrum)
STD_ROM_FN(Specchicago30)

struct BurnDriver BurnSpecchicago30 = {
	"spec_chicago30", NULL, "spec_spectrum", NULL, "1988",
	"Chicago 30's (Spanish) (48K)\0", NULL, "Topo Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, Specchicago30RomInfo, Specchicago30RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Chronos - A Tapestry of Time (48K)

static struct BurnRomInfo SpecchronosRomDesc[] = {
	{ "Chronos - A Tapestry of Time 48K (1987)(Mastertronic).z80", 42112, 0x9a40d9c4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specchronos, Specchronos, Spectrum)
STD_ROM_FN(Specchronos)

struct BurnDriver BurnSpecchronos = {
	"spec_chronos", NULL, "spec_spectrum", NULL, "1987",
	"Chronos - A Tapestry of Time (48K)\0", NULL, "Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecchronosRomInfo, SpecchronosRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Chuckie Egg (48K)

static struct BurnRomInfo SpecchuckeggRomDesc[] = {
	{ "Chuckie Egg 48K (1983)(A'n'F Software).tap", 20645, 0x15adf085, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specchuckegg, Specchuckegg, Spectrum)
STD_ROM_FN(Specchuckegg)

struct BurnDriver BurnSpecchuckegg = {
	"spec_chuckegg", NULL, "spec_spectrum", NULL, "1983",
	"Chuckie Egg (48K)\0", "Set '3' key type in Instructions 'I' menu to use controller", "A'n'F Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 4, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecchuckeggRomInfo, SpecchuckeggRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPMDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Chuckie Egg 2 (48K)

static struct BurnRomInfo Specchuckeg2RomDesc[] = {
	{ "Chuckie Egg 2 48K (1985)(A'n'F Software).tap", 49012, 0x7917ac74, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specchuckeg2, Specchuckeg2, Spectrum)
STD_ROM_FN(Specchuckeg2)

struct BurnDriver BurnSpecchuckeg2 = {
	"spec_chuckeg2", NULL, "spec_spectrum", NULL, "1985",
	"Chuckie Egg 2 (48K)\0", "At title press 'Space' and wait for loading time, then press 'Space' again", "A'n'F Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Specchuckeg2RomInfo, Specchuckeg2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// CJ's Elephant Antics (128K)

static struct BurnRomInfo Speccjseleph128RomDesc[] = {
	{ "CJ's Elephant Antics 128K (1991)(Codemasters).tap", 60962, 0xa44b3da6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccjseleph128, Speccjseleph128, Spec128)
STD_ROM_FN(Speccjseleph128)

struct BurnDriver BurnSpeccjseleph128 = {
	"spec_cjseleph128", NULL, "spec_spec128", NULL, "1991",
	"CJ's Elephant Antics (128K)\0", "Press '3' to use controller", "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Speccjseleph128RomInfo, Speccjseleph128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// CJ In the USA (128K)

static struct BurnRomInfo SpeccjiiiusaRomDesc[] = {
	{ "CJ In the USA (1991)(Codemasters)(128k).tap", 64000, 0x455F660B, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccjiiiusa, Speccjiiiusa, Spec128)
STD_ROM_FN(Speccjiiiusa)

struct BurnDriver BurnSpeccjiiiusa = {
	"spec_cjiiiusa", NULL, "spec_spec128", NULL, "1991",
	"CJ In the USA (128K)\0", "In main menu: SPACE to choose, ENTER to select", "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpeccjiiiusaRomInfo, SpeccjiiiusaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Coliseum (Spanish) (48K)

static struct BurnRomInfo SpeccoliseumRomDesc[] = {
	{ "Coliseum 48K ES (1988)(Topo Soft).tap", 52375, 0x50c20efc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccoliseum, Speccoliseum, Spectrum)
STD_ROM_FN(Speccoliseum)

struct BurnDriver BurnSpeccoliseum = {
	"spec_coliseum", NULL, "spec_spectrum", NULL, "1988",
	"Coliseum (Spanish) (48K)\0", NULL, "Topo Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpeccoliseumRomInfo, SpeccoliseumRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Combat School (48K)

static struct BurnRomInfo Speccschool48RomDesc[] = {
	{ "Combat School 48K (1987)(Ocean).tap", 109676, 0x14c7b9ea, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccschool48, Speccschool48, Spectrum)
STD_ROM_FN(Speccschool48)

struct BurnDriver BurnSpeccschool48 = {
	"spec_cschool48", "spec_cschool128", "spec_spectrum", NULL, "1987",
	"Combat School (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_SPORTSMISC, 0,
	SpectrumGetZipName, Speccschool48RomInfo, Speccschool48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Combat School (128K)

static struct BurnRomInfo Speccschool128RomDesc[] = {
	{ "Combat School 128K (1987)(Ocean).tap", 118171, 0xa45d89c2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccschool128, Speccschool128, Spec128)
STD_ROM_FN(Speccschool128)

struct BurnDriver BurnSpeccschool128 = {
	"spec_cschool128", NULL, "spec_spec128", NULL, "1987",
	"Combat School (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_SPORTSMISC, 0,
	SpectrumGetZipName, Speccschool128RomInfo, Speccschool128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Commando (French) (48K)

static struct BurnRomInfo SpeccommandofRomDesc[] = {
	{ "Commando (1984)(Loriciels)(fr).z80", 0x0a352, 0xc2b0cbfe, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccommandof, Speccommandof, Spectrum)
STD_ROM_FN(Speccommandof)

struct BurnDriver BurnSpeccommandof = {
	"spec_commandof", NULL, "spec_spectrum", NULL, "1984",
	"Commando (French) (48K)\0", NULL, "Loriciels", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpeccommandofRomInfo, SpeccommandofRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Commando (Pocket Money Software) (48K)

static struct BurnRomInfo SpeccommandopRomDesc[] = {
	{ "Commando (1984)(Pocket Money Software).z80", 0x04924, 0xeaac033a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccommandop, Speccommandop, Spectrum)
STD_ROM_FN(Speccommandop)

struct BurnDriver BurnSpeccommandop = {
	"spec_commandop", NULL, "spec_spectrum", NULL, "1984",
	"Commando (Pocket Money Software) (48K)\0", NULL, "Pocket Money Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpeccommandopRomInfo, SpeccommandopRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Commando (Elite Systems) (48K)

static struct BurnRomInfo SpeccommandoRomDesc[] = {
	{ "Commando (1985)(Elite Systems).tap", 48833, 0x90a44ff9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccommando, Speccommando, Spectrum)
STD_ROM_FN(Speccommando)

struct BurnDriver BurnSpeccommando = {
	"spec_commando", NULL, "spec_spectrum", NULL, "1985",
	"Commando (Elite Systems) (48K)\0", NULL, "Elite Systems", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpeccommandoRomInfo, SpeccommandoRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Continental Circus (128K)

static struct BurnRomInfo SpeccontcircRomDesc[] = {
	{ "Continental Circus (1989)(Virgin Games).tap", 127564, 0xbe5c0729, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccontcirc, Speccontcirc, Spec128)
STD_ROM_FN(Speccontcirc)

struct BurnDriver BurnSpeccontcirc = {
	"spec_contcirc", NULL, "spec_spec128", NULL, "1989",
	"Continental Circus (128K)\0", NULL, "Virgin Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpeccontcircRomInfo, SpeccontcircRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Corsarios (Part 1 of 2) (128K)

static struct BurnRomInfo Speccorsarios1RomDesc[] = {
	{ "Corsarios (Part 1 of 2) (1989)(Opera Soft)(128k).z80", 0x09dae, 0xb7d7624e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccorsarios1, Speccorsarios1, Spec128)
STD_ROM_FN(Speccorsarios1)

struct BurnDriver BurnSpeccorsarios1 = {
	"spec_corsarios1", NULL, "spec_spec128", NULL, "1989",
	"Corsarios (Part 1 of 2) (128K)\0", NULL, "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, Speccorsarios1RomInfo, Speccorsarios1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Corsarios (Part 2 of 2) (128K)

static struct BurnRomInfo Speccorsarios2RomDesc[] = {
	{ "Corsarios (Part 2 of 2) (1989)(Opera Soft)(128k).z80", 0x09fa3, 0xceb25c2a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccorsarios2, Speccorsarios2, Spec128)
STD_ROM_FN(Speccorsarios2)

struct BurnDriver BurnSpeccorsarios2 = {
	"spec_corsarios2", "spec_corsarios1", "spec_spec128", NULL, "1989",
	"Corsarios (Part 2 of 2) (128K)\0", NULL, "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, Speccorsarios2RomInfo, Speccorsarios2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Crystal Castles Diamond Plateaus in Space (48K)

static struct BurnRomInfo SpeccryscastRomDesc[] = {
	{ "Crystal Castles Diamond Plateaus in Space (1986)(U.S. Gold).tap", 46669, 0xf6e3d62a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccryscast, Speccryscast, Spectrum)
STD_ROM_FN(Speccryscast)

struct BurnDriver BurnSpeccryscast = {
	"spec_cryscast", NULL, "spec_spectrum", NULL, "1986",
	"Crystal Castles Diamond Plateaus in Space (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpeccryscastRomInfo, SpeccryscastRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Cybernoid - The Fighting Machine (48K)

static struct BurnRomInfo SpecCybernoid48RomDesc[] = {
	{ "Cybernoid - The Fighting Machine (1988)(Hewson Consultants).tap", 46731, 0x965806fc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCybernoid48, SpecCybernoid48, Spectrum)
STD_ROM_FN(SpecCybernoid48)

struct BurnDriver BurnSpecCybernoid48 = {
	"spec_cybernoid48", "spec_cybernoid128", "spec_spectrum", NULL, "1988",
	"Cybernoid - The Fighting Machine (48K)\0", NULL, "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecCybernoid48RomInfo, SpecCybernoid48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Cybernoid - The Fighting Machine (128K)

static struct BurnRomInfo SpecCybernoid128RomDesc[] = {
	{ "Cybernoid - The Fighting Machine (1988)(Hewson Consultants).tap", 46731, 0x965806fc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCybernoid128, SpecCybernoid128, Spec128)
STD_ROM_FN(SpecCybernoid128)

struct BurnDriver BurnSpecCybernoid128 = {
	"spec_cybernoid128", NULL, "spec_spec128", NULL, "1988",
	"Cybernoid - The Fighting Machine (128K)\0", NULL, "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecCybernoid128RomInfo, SpecCybernoid128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Cybernoid II - The Revenge (48K)

static struct BurnRomInfo SpecCybernoidii48RomDesc[] = {
	{ "Cybernoid II - The Revenge 48K (1988)(Hewson Consultants).tap", 48032, 0xa66dd375, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCybernoidii48, SpecCybernoidii48, Spectrum)
STD_ROM_FN(SpecCybernoidii48)

struct BurnDriver BurnSpecCybernoidii48 = {
	"spec_cybernoidii48", "spec_cybernoidii128", "spec_spectrum", NULL, "1988",
	"Cybernoid II - The Revenge (48K)\0", NULL, "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecCybernoidii48RomInfo, SpecCybernoidii48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Cybernoid II - The Revenge (128K)

static struct BurnRomInfo SpecCybernoidii128RomDesc[] = {
	{ "Cybernoid II - The Revenge 128K (1988)(Hewson Consultants).tap", 48033, 0x5d933be6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCybernoidii128, SpecCybernoidii128, Spec128)
STD_ROM_FN(SpecCybernoidii128)

struct BurnDriver BurnSpecCybernoidii128 = {
	"spec_cybernoidii128", NULL, "spec_spec128", NULL, "1988",
	"Cybernoid II - The Revenge (128K)\0", NULL, "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecCybernoidii128RomInfo, SpecCybernoidii128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Daley Thompson's Decathlon - Day 1 (48K)

static struct BurnRomInfo SpecdtdecthnRomDesc[] = {
	{ "Daley Thompson's Decathlon - Day 1 (1984)(Ocean).z80", 0x08bbe, 0xf31094d1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdtdecthn, Specdtdecthn, Spectrum)
STD_ROM_FN(Specdtdecthn)

struct BurnDriver BurnSpecdtdecthn = {
	"spec_dtdecthn", NULL, "spec_spectrum", NULL, "1984",
	"Daley Thompson's Decathlon - Day 1 (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecdtdecthnRomInfo, SpecdtdecthnRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Daley Thompson's Decathlon - Day 2 (48K)

static struct BurnRomInfo Specdtdecthn2RomDesc[] = {
	{ "Daley Thompson's Decathlon - Day 2 (1984)(Ocean).z80", 0x08a49, 0x500ca1a5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdtdecthn2, Specdtdecthn2, Spectrum)
STD_ROM_FN(Specdtdecthn2)

struct BurnDriver BurnSpecdtdecthn2 = {
	"spec_dtdecthn2", "spec_dtdecthn", "spec_spectrum", NULL, "1984",
	"Daley Thompson's Decathlon - Day 2 (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, Specdtdecthn2RomInfo, Specdtdecthn2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Daley Thompson's Olympic Challenge (128K)

static struct BurnRomInfo SpecdatholchRomDesc[] = {
	{ "Daley Thompson's Olympic Challenge (1988)(Ocean)[128K].z80", 0x1d5dc, 0xfc2d513f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdatholch, Specdatholch, Spec128)
STD_ROM_FN(Specdatholch)

struct BurnDriver BurnSpecdatholch = {
	"spec_datholch", NULL, "spec_spec128", NULL, "1988",
	"Daley Thompson's Olympic Challenge (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecdatholchRomInfo, SpecdatholchRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Daley Thompson's Super-Test (48K)

static struct BurnRomInfo SpecDtsupertest48RomDesc[] = {
	{ "Daley Thompson's Super-Test 48K (1985)(Ocean).tap", 49345, 0x3c3d0d1e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDtsupertest48, SpecDtsupertest48, Spectrum)
STD_ROM_FN(SpecDtsupertest48)

struct BurnDriver BurnSpecDtsupertest48 = {
	"spec_dtsupertest48", "spec_dtsupertest128", "spec_spectrum", NULL, "1985",
	"Daley Thompson's Super-Test (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecDtsupertest48RomInfo, SpecDtsupertest48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Daley Thompson's Super-Test (128K)

static struct BurnRomInfo SpecDtsupertest128RomDesc[] = {
	{ "Daley Thompson's Super-Test 128K (1985)(Ocean).tap", 128381, 0xe8ab5d26, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDtsupertest128, SpecDtsupertest128, Spec128)
STD_ROM_FN(SpecDtsupertest128)

struct BurnDriver BurnSpecDtsupertest128 = {
	"spec_dtsupertest128", NULL, "spec_spec128", NULL, "1985",
	"Daley Thompson's Super-Test (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecDtsupertest128RomInfo, SpecDtsupertest128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dan Dare - Pilot of the Future (48K)

static struct BurnRomInfo SpecdandareRomDesc[] = {
	{ "Dan Dare - Pilot of the Future (1986)(Virgin Games).z80", 42412, 0x9378e2c3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdandare, Specdandare, Spectrum)
STD_ROM_FN(Specdandare)

struct BurnDriver BurnSpecdandare = {
	"spec_dandare", NULL, "spec_spectrum", NULL, "1986",
	"Dan Dare - Pilot of the Future (48K)\0", NULL, "Virgin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_ADV, 0,
	SpectrumGetZipName, SpecdandareRomInfo, SpecdandareRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dan Dare II - Mekon's Revenge (48K)

static struct BurnRomInfo Specdandare2RomDesc[] = {
	{ "Dan Dare II - Mekon's Revenge (1988)(Virgin Games).z80", 44497, 0x03159016, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdandare2, Specdandare2, Spectrum)
STD_ROM_FN(Specdandare2)

struct BurnDriver BurnSpecdandare2 = {
	"spec_dandare2", NULL, "spec_spectrum", NULL, "1988",
	"Dan Dare II - Mekon's Revenge (48K)\0", NULL, "Virgin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_ADV, 0,
	SpectrumGetZipName, Specdandare2RomInfo, Specdandare2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dan Dare III - The Escape (48K)

static struct BurnRomInfo Specdandare3RomDesc[] = {
	{ "Dan Dare III - The Escape (1990)(Virgin Games).z80", 42660, 0xad0767cb, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdandare3, Specdandare3, Spectrum)
STD_ROM_FN(Specdandare3)

struct BurnDriver BurnSpecdandare3 = {
	"spec_dandare3", NULL, "spec_spectrum", NULL, "1990",
	"Dan Dare III - The Escape (48K)\0", NULL, "Virgin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_ADV, 0,
	SpectrumGetZipName, Specdandare3RomInfo, Specdandare3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Darius+ (128K)

static struct BurnRomInfo SpecdariusRomDesc[] = {
	{ "Darius+ (1990)(The Edge Software).tap", 0x35789, 0xe47f46f1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdarius, Specdarius, Spec128)
STD_ROM_FN(Specdarius)

struct BurnDriver BurnSpecdarius = {
	"spec_darius", NULL, "spec_spec128", NULL, "1990",
	"Darius+ (128K)\0", NULL, "The Edge", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecdariusRomInfo, SpecdariusRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Darkman (128K)

static struct BurnRomInfo SpecdarkmanRomDesc[] = {
	{ "Darkman 128K (1991)(Ocean).tap", 181471, 0x697be425, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdarkman, Specdarkman, Spec128)
STD_ROM_FN(Specdarkman)

struct BurnDriver BurnSpecdarkman = {
	"spec_darkman", NULL, "spec_spec128", NULL, "1991",
	"Darkman (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecdarkmanRomInfo, SpecdarkmanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Deathchase (48K)

static struct BurnRomInfo SpecdeathchaRomDesc[] = {
	{ "Deathchase (1983)(Micromega).z80", 0x02bed, 0x040769a4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdeathcha, Specdeathcha, Spectrum)
STD_ROM_FN(Specdeathcha)

struct BurnDriver BurnSpecdeathcha = {
	"spec_deathcha", NULL, "spec_spectrum", NULL, "1983",
	"Deathchase (48K)\0", NULL, "Micromega", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecdeathchaRomInfo, SpecdeathchaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dizzy - Down the Rapids (128K)

static struct BurnRomInfo SpecdizdowraRomDesc[] = {
	{ "Dizzy - Down the Rapids (1991)(Codemasters)[128K].z80", 0x0a17e, 0x426abaa3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdizdowra, Specdizdowra, Spec128)
STD_ROM_FN(Specdizdowra)

struct BurnDriver BurnSpecdizdowra = {
	"spec_dizdowra", NULL, "spec_spec128", NULL, "1991",
	"Dizzy - Down the Rapids (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecdizdowraRomInfo, SpecdizdowraRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dizzy - The Ultimate Cartoon Adventure (Russian) (48K) (Hack)

static struct BurnRomInfo Specdizzy1ruRomDesc[] = {
	{ "Dizzy - The Ultimate Cartoon Adventure (1987)(Codemasters)[ru].tap", 48813, 0x85067eff, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdizzy1ru, Specdizzy1ru, Spectrum)
STD_ROM_FN(Specdizzy1ru)

struct BurnDriver BurnSpecdizzy1ru = {
	"spec_dizzy1ru", "spec_dizzy1", "spec_spectrum", NULL, "199?",
	"Dizzy - The Ultimate Cartoon Adventure (Russian) (48K) (Hack)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, Specdizzy1ruRomInfo, Specdizzy1ruRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dizzy - The Ultimate Cartoon Adventure (48K)

static struct BurnRomInfo Specdizzy1RomDesc[] = {
	{ "Dizzy - The Ultimate Cartoon Adventure (1987)(Codemasters).tap", 48901, 0xda1ce6bf, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdizzy1, Specdizzy1, Spectrum)
STD_ROM_FN(Specdizzy1)

struct BurnDriver BurnSpecdizzy1 = {
	"spec_dizzy1", NULL, "spec_spectrum", NULL, "1987",
	"Dizzy - The Ultimate Cartoon Adventure (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, Specdizzy1RomInfo, Specdizzy1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dizzy - The Ultimate Cartoon Adventure - Crash Special Edition (48K)

static struct BurnRomInfo Specdizzy1cseRomDesc[] = {
	{ "Dizzy - The Ultimate Cartoon Adventure - Crash Special Edition (1987)(Codemasters).tap", 48581, 0x83d585be, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdizzy1cse, Specdizzy1cse, Spectrum)
STD_ROM_FN(Specdizzy1cse)

struct BurnDriver BurnSpecdizzy1cse = {
	"spec_dizzy1cse", "spec_dizzy1", "spec_spectrum", NULL, "1987",
	"Dizzy - The Ultimate Cartoon Adventure - Crash Special Edition (48K)\0", "Z: left - X: right - SPACE: jump - ENTER: action", "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, Specdizzy1cseRomInfo, Specdizzy1cseRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dizzy II - Treasure Island Dizzy (Russian) (128K) (Hack)

static struct BurnRomInfo Specdizzy2ruRomDesc[] = {
	{ "Dizzy 2 - Treasure Island Dizzy 128K (1988)(Codemasters)[ru].z80", 57508, 0xccc7f01b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdizzy2ru, Specdizzy2ru, Spec128)
STD_ROM_FN(Specdizzy2ru)

struct BurnDriver BurnSpecdizzy2ru = {
	"spec_dizzy2ru", "spec_dizzy2ay", "spec_spec128", NULL, "199?",
	"Dizzy II - Treasure Island Dizzy (Russian) (128K) (Hack)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, Specdizzy2ruRomInfo, Specdizzy2ruRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dizzy II - Treasure Island Dizzy (48K)

static struct BurnRomInfo Specdizzy2RomDesc[] = {
	{ "Dizzy 2 - Treasure Island Dizzy 48K (1988)(Codemasters).tap", 44195, 0x3c680bbd, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdizzy2, Specdizzy2, Spectrum)
STD_ROM_FN(Specdizzy2)

struct BurnDriver BurnSpecdizzy2 = {
	"spec_dizzy2", "spec_dizzy2ay", "spec_spectrum", NULL, "1988",
	"Dizzy II - Treasure Island Dizzy (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, Specdizzy2RomInfo, Specdizzy2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dizzy II - Treasure Island Dizzy (128K)

static struct BurnRomInfo Specdizzy2ayRomDesc[] = {
	{ "Dizzy 2 - Treasure Island Dizzy 128K (1988)(Codemasters).tap", 57678, 0x0fd65ba2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdizzy2ay, Specdizzy2ay, Spec128)
STD_ROM_FN(Specdizzy2ay)

struct BurnDriver BurnSpecdizzy2ay = {
	"spec_dizzy2ay", NULL, "spec_spec128", NULL, "1988",
	"Dizzy II - Treasure Island Dizzy (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, Specdizzy2ayRomInfo, Specdizzy2ayRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dizzy III - Fantasy World Dizzy (Russian) (128K) (Hack)

static struct BurnRomInfo Specdizzy3ruRomDesc[] = {
	{ "Dizzy 3 - Fantasy World Dizzy (1989)(Codemasters)[ru].tap", 64532, 0x2a83e0d8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdizzy3ru, Specdizzy3ru, Spec128)
STD_ROM_FN(Specdizzy3ru)

struct BurnDriver BurnSpecdizzy3ru = {
	"spec_dizzy3ru", "spec_dizzy3", "spec_spec128", NULL, "199?",
	"Dizzy III - Fantasy World Dizzy (Russian) (128K) (Hack)\0", NULL, "unknown", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, Specdizzy3ruRomInfo, Specdizzy3ruRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dizzy III - Fantasy World Dizzy (128K)

static struct BurnRomInfo Specdizzy3RomDesc[] = {
	{ "Dizzy 3 - Fantasy World Dizzy (1989)(Codemasters).tap", 62876, 0x2080a324, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdizzy3, Specdizzy3, Spec128)
STD_ROM_FN(Specdizzy3)

struct BurnDriver BurnSpecdizzy3 = {
	"spec_dizzy3", NULL, "spec_spec128", NULL, "1989",
	"Dizzy III - Fantasy World Dizzy (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, Specdizzy3RomInfo, Specdizzy3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dizzy IV - Magicland Dizzy (48K)

static struct BurnRomInfo Specdizzy4_48RomDesc[] = {
	{ "Dizzy 4 - Magicland Dizzy 48K (1990)(Codemasters).tap", 56260, 0xbd5a1fca, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdizzy4_48, Specdizzy4_48, Spectrum)
STD_ROM_FN(Specdizzy4_48)

struct BurnDriver BurnSpecdizzy4_48 = {
	"spec_dizzy4_48", "spec_dizzy4", "spec_spectrum", NULL, "1989",
	"Dizzy IV - Magicland Dizzy (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, Specdizzy4_48RomInfo, Specdizzy4_48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dizzy IV - Magicland Dizzy (128K)

static struct BurnRomInfo Specdizzy4RomDesc[] = {
	{ "Dizzy 4 - Magicland Dizzy 128K (1990)(Codemasters).tap", 64959, 0x52c74596, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdizzy4, Specdizzy4, Spec128)
STD_ROM_FN(Specdizzy4)

struct BurnDriver BurnSpecdizzy4 = {
	"spec_dizzy4", NULL, "spec_spec128", NULL, "1990",
	"Dizzy IV - Magicland Dizzy (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, Specdizzy4RomInfo, Specdizzy4RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dizzy IV - Magicland Dizzy (Russian) (128K) (Hack)

static struct BurnRomInfo Specdizzy4ruRomDesc[] = {
	{ "Dizzy 4 - Magicland Dizzy 128K (1990)(Codemasters)[ru].tap", 58305, 0x328a1777, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdizzy4ru, Specdizzy4ru, Spec128)
STD_ROM_FN(Specdizzy4ru)

struct BurnDriver BurnSpecdizzy4ru = {
	"spec_dizzy4ru", "spec_dizzy4", "spec_spec128", NULL, "199?",
	"Dizzy IV - Magicland Dizzy (Russian) (128K) (Hack)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, Specdizzy4ruRomInfo, Specdizzy4ruRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dizzy V - Spellbound Dizzy (128K)

static struct BurnRomInfo Specdizzy5RomDesc[] = {
	{ "Dizzy 5 - Spellbound Dizzy 128K (1991)(Codemasters).tap", 96808, 0xb759620a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdizzy5, Specdizzy5, Spec128)
STD_ROM_FN(Specdizzy5)

struct BurnDriver BurnSpecdizzy5 = {
	"spec_dizzy5", NULL, "spec_spec128", NULL, "1991",
	"Dizzy V - Spellbound Dizzy (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, Specdizzy5RomInfo, Specdizzy5RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dizzy VI - Prince of the Yolkfolk (Russian) (48K) (Hack)

static struct BurnRomInfo Specdizzy6ruRomDesc[] = {
	{ "Dizzy 6 - Dizzy, Prince of the YolkFolk (1991)(Codemasters)[ru].tap", 41101, 0xc1462db4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdizzy6ru, Specdizzy6ru, Spectrum)
STD_ROM_FN(Specdizzy6ru)

struct BurnDriver BurnSpecdizzy6ru = {
	"spec_dizzy6ru", "spec_dizzy6", "spec_spectrum", NULL, "199?",
	"Dizzy VI - Prince of the Yolkfolk (Russian) (48K) (Hack)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, Specdizzy6ruRomInfo, Specdizzy6ruRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dizzy VI - Prince of the Yolkfolk (128K)

static struct BurnRomInfo Specdizzy6RomDesc[] = {
	{ "Dizzy 6 - Dizzy, Prince of the YolkFolk 128K (1991)(Codemasters).tap", 47088, 0xc46222fc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdizzy6, Specdizzy6, Spec128)
STD_ROM_FN(Specdizzy6)

struct BurnDriver BurnSpecdizzy6 = {
	"spec_dizzy6", NULL, "spec_spec128", NULL, "1991",
	"Dizzy VI - Prince of the Yolkfolk (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, Specdizzy6RomInfo, Specdizzy6RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dizzy VII - Crystal Kingdom Dizzy (128K)

static struct BurnRomInfo Specdizzy7RomDesc[] = {
	{ "Dizzy 7 - Crystal Kingdom Dizzy 128K (1992)(Codemasters).z80", 47389, 0x16fb82f0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdizzy7, Specdizzy7, Spec128)
STD_ROM_FN(Specdizzy7)

struct BurnDriver BurnSpecdizzy7 = {
	"spec_dizzy7", NULL, "spec_spec128", NULL, "1992",
	"Dizzy VII - Crystal Kingdom Dizzy (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, Specdizzy7RomInfo, Specdizzy7RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dizzy VII - Crystal Kingdom Dizzy (Russian) (128K) (Hack)

static struct BurnRomInfo Specdizzy7ruRomDesc[] = {
	{ "Dizzy 7 - Crystal Kingdom Dizzy 128K (1992)(Codemasters)[ru].z80", 61746, 0xd6b0801d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdizzy7ru, Specdizzy7ru, Spec128)
STD_ROM_FN(Specdizzy7ru)

struct BurnDriver BurnSpecdizzy7ru = {
	"spec_dizzy7ru", "spec_dizzy7", "spec_spec128", NULL, "1993",
	"Dizzy VII - Crystal Kingdom Dizzy (Russian) (128K) (Hack)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, Specdizzy7ruRomInfo, Specdizzy7ruRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Donkey Kong (48K)

static struct BurnRomInfo SpecdkongRomDesc[] = {
	{ "Donkey Kong (1986)(Ocean).tap", 57131, 0xf54f6058, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdkong, Specdkong, Spectrum)
STD_ROM_FN(Specdkong)

struct BurnDriver BurnSpecdkong = {
	"spec_dkong", NULL, "spec_spectrum", NULL, "1986",
	"Donkey Kong (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecdkongRomInfo, SpecdkongRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Donkey Kong Jr. (48K)

static struct BurnRomInfo SpecdkongjrRomDesc[] = {
	{ "Donkey Kong Jr. (19xx)(Sir Clive and Mr ZX).z80", 0x02a1d, 0x91569bef, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdkongjr, Specdkongjr, Spectrum)
STD_ROM_FN(Specdkongjr)

struct BurnDriver BurnSpecdkongjr = {
	"spec_dkongjr", NULL, "spec_spectrum", NULL, "19xx",
	"Donkey Kong Jr. (48K)\0", NULL, "Sir Clive and Mr ZX", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecdkongjrRomInfo, SpecdkongjrRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Doom (Demo) (128K)

static struct BurnRomInfo SpecdoomdemoRomDesc[] = {
	{ "Doom (demo) (1996)(Digital Reality)[128K].z80", 0x11d9a, 0xb310b6f1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdoomdemo, Specdoomdemo, Spec128)
STD_ROM_FN(Specdoomdemo)

struct BurnDriver BurnSpecdoomdemo = {
	"spec_doomdemo", NULL, "spec_spec128", NULL, "1996",
	"Doom (Demo) (128K)\0", NULL, "Digital Reality", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecdoomdemoRomInfo, SpecdoomdemoRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Double Dragon III - The Rosetta Stone (128K)

static struct BurnRomInfo Specddragon3RomDesc[] = {
	{ "Double Dragon III - The Rosetta Stone 128K (1991)(Storm).tap", 312619, 0x09d741f5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specddragon3, Specddragon3, Spec128)
STD_ROM_FN(Specddragon3)

struct BurnDriver BurnSpecddragon3 = {
	"spec_ddragon3", NULL, "spec_spec128", NULL, "1991",
	"Double Dragon III - The Rosetta Stone (128K)\0", NULL, "Storm Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, Specddragon3RomInfo, Specddragon3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Draconus (48K)

static struct BurnRomInfo SpecDraconusRomDesc[] = {
	{ "Draconus 48K (1988)(Zeppelin Games).tap", 39793, 0x6aeb0e66, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDraconus, SpecDraconus, Spectrum)
STD_ROM_FN(SpecDraconus)

struct BurnDriver BurnSpecDraconus = {
	"spec_draconus", NULL, "spec_spectrum", NULL, "1988",
	"Draconus (48K)\0", "Press ENTER for game menu", "Zeppelin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecDraconusRomInfo, SpecDraconusRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dragon Breed (48K)

static struct BurnRomInfo SpecdrgbreedRomDesc[] = {
	{ "Dragon Breed (1990)(Activision).tap", 126452, 0xc7158fdf, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdrgbreed, Specdrgbreed, Spectrum)
STD_ROM_FN(Specdrgbreed)

struct BurnDriver BurnSpecdrgbreed = {
	"spec_drgbreed", NULL, "spec_spectrum", NULL, "1989",
	"Dragon Breed (48K)\0", NULL, "Activision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecdrgbreedRomInfo, SpecdrgbreedRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dragon Ninja (48K)

static struct BurnRomInfo Specdninja48RomDesc[] = {
	{ "Dragon Ninja 48K (1988)(Imagine).tap", 95634, 0x5ba91604, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdninja48, Specdninja48, Spectrum)
STD_ROM_FN(Specdninja48)

struct BurnDriver BurnSpecdninja48 = {
	"spec_dninja48", "spec_dninja128", "spec_spectrum", NULL, "1988",
	"Dragon Ninja (48K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, Specdninja48RomInfo, Specdninja48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dragon Ninja (128K)

static struct BurnRomInfo Specdninja128RomDesc[] = {
	{ "Dragon Ninja 128K (1988)(Imagine).tap", 118559, 0x636223a9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdninja128, Specdninja128, Spec128)
STD_ROM_FN(Specdninja128)

struct BurnDriver BurnSpecdninja128 = {
	"spec_dninja128", NULL, "spec_spec128", NULL, "1988",
	"Dragon Ninja (128K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, Specdninja128RomInfo, Specdninja128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dragontorc (48K)

static struct BurnRomInfo SpecdragontoRomDesc[] = {
	{ "Dragontorc (1985)(Hewson Consultants).z80", 0x09d26, 0xc2e5c32e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdragonto, Specdragonto, Spectrum)
STD_ROM_FN(Specdragonto)

struct BurnDriver BurnSpecdragonto = {
	"spec_dragonto", NULL, "spec_spectrum", NULL, "1985",
	"Dragontorc (48K)\0", "At title-screen press 'S' to start game", "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecdragontoRomInfo, SpecdragontoRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Driller (48K)

static struct BurnRomInfo SpecdrillerRomDesc[] = {
	{ "Driller (1987)(Incentive).tap", 55483, 0x0387f1c0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdriller, Specdriller, Spectrum)
STD_ROM_FN(Specdriller)

struct BurnDriver BurnSpecdriller = {
	"spec_driller", NULL, "spec_spectrum", NULL, "1987",
	"Driller (48K)\0", NULL, "Incentive", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_ADV, 0,
	SpectrumGetZipName, SpecdrillerRomInfo, SpecdrillerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Druid (48K)

static struct BurnRomInfo SpecDruidRomDesc[] = {
	{ "Druid (1986)(Firebird).tap", 45791, 0x19fb94a7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDruid, SpecDruid, Spectrum)
STD_ROM_FN(SpecDruid)

struct BurnDriver BurnSpecDruid = {
	"spec_druid", NULL, "spec_spectrum", NULL, "1986",
	"Druid (48K)\0", "Press '2' to set Kempston Joystick", "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_MAZE, 0,
	SpectrumGetZipName, SpecDruidRomInfo, SpecDruidRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Druid II - Enlightenment (48K)

static struct BurnRomInfo SpecDruid2RomDesc[] = {
	{ "Druid II - Enlightenment (1987)(Firebird).tap", 58884, 0x8a5474aa, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDruid2, SpecDruid2, Spectrum)
STD_ROM_FN(SpecDruid2)

struct BurnDriver BurnSpecDruid2 = {
	"spec_druid2", NULL, "spec_spectrum", NULL, "1987",
	"Druid II - Enlightenment (48K)\0", "Press 'J' to set Kempston Joystick", "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_MAZE, 0,
	SpectrumGetZipName, SpecDruid2RomInfo, SpecDruid2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dynamite Dan (48K)

static struct BurnRomInfo SpecdynadanRomDesc[] = {
	{ "Dynamite Dan (1985)(Mirrorsoft).z80", 0x0a6a8, 0x218460b1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdynadan, Specdynadan, Spectrum)
STD_ROM_FN(Specdynadan)

struct BurnDriver BurnSpecdynadan = {
	"spec_dynadan", NULL, "spec_spectrum", NULL, "1985",
	"Dynamite Dan (48K)\0", NULL, "Mirrorsoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecdynadanRomInfo, SpecdynadanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dynamite Dan II - Dr. Blitzen and the Islands of Arcanum (48K)

static struct BurnRomInfo Specdynadan2RomDesc[] = {
	{ "Dynamite Dan II - Dr Blitzen And The Islands Of Arcanum (1985)(Mirrorsoft).z80", 0x0aaee, 0xdf00027d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdynadan2, Specdynadan2, Spectrum)
STD_ROM_FN(Specdynadan2)

struct BurnDriver BurnSpecdynadan2 = {
	"spec_dynadan2", NULL, "spec_spectrum", NULL, "1986",
	"Dynamite Dan II - Dr. Blitzen and the Islands of Arcanum (48K)\0", NULL, "Mirrorsoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Specdynadan2RomInfo, Specdynadan2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Elite (48K-128K)

static struct BurnRomInfo SpeceliteRomDesc[] = {
	{ "Elite (1985)(Firebird Software)(128k).z80", 0x0a8f4, 0xd914f0df, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specelite, Specelite, Spec128)
STD_ROM_FN(Specelite)

struct BurnDriver BurnSpecelite = {
	"spec_elite", NULL, "spec_spec128", NULL, "1985",
	"Elite (48K-128K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SIM | GBF_STRATEGY, 0,
	SpectrumGetZipName, SpeceliteRomInfo, SpeceliteRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Elven Warrior (128K)

static struct BurnRomInfo SpecelvewarrRomDesc[] = {
	{ "Elven Warrior (1989)(Players Premier).tap", 51605, 0x10bcb8c8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specelvewarr, Specelvewarr, Spec128)
STD_ROM_FN(Specelvewarr)

struct BurnDriver BurnSpecelvewarr = {
	"spec_elvewarr", NULL, "spec_spec128", NULL, "1989",
	"Elven Warrior (128K)\0", NULL, "Players Premier", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecelvewarrRomInfo, SpecelvewarrRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Emilio Butragueno Futbol (Spanish) (48K)

static struct BurnRomInfo SpecemilbutrafutbolRomDesc[] = {
	{ "Emilio Butragueno Futbol (1988)(Ocean Software - Topo Soft).tap", 35014, 0x99569c38, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specemilbutrafutbol, Specemilbutrafutbol, Spectrum)
STD_ROM_FN(Specemilbutrafutbol)

struct BurnDriver BurnSpecemilbutrafutbol = {
	"spec_emilbutrafutbol", NULL, "spec_spectrum", NULL, "1988",
	"Emilio Butragueno Futbol (Spanish) (48K)\0", NULL, "Topo Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL, 0,
	SpectrumGetZipName, SpecemilbutrafutbolRomInfo, SpecemilbutrafutbolRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Emilio Sanchez Vicario Grand Slam (Spanish) (48K)

static struct BurnRomInfo SpecemilsanchgslamRomDesc[] = {
	{ "Emilio Sanchez Vicario Grand Slam ES (1990)(Zigurat).tap", 46879, 0x22f0f155, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specemilsanchgslam, Specemilsanchgslam, Spectrum)
STD_ROM_FN(Specemilsanchgslam)

struct BurnDriver BurnSpecemilsanchgslam = {
	"spec_emilsanchgslam", NULL, "spec_spectrum", NULL, "1989",
	"Emilio Sanchez Vicario Grand Slam (Spanish) (48K)\0", NULL, "Zigurat Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecemilsanchgslamRomInfo, SpecemilsanchgslamRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// E-motion (128K)

static struct BurnRomInfo SpecEmotionRomDesc[] = {
	{ "E-motion 128K (1990)(U.S. Gold).z80", 44420, 0xcc29fe26, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecEmotion, SpecEmotion, Spec128)
STD_ROM_FN(SpecEmotion)

struct BurnDriver BurnSpecEmotion = {
	"spec_emotion", NULL, "spec_spec128", NULL, "1990",
	"E-motion (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION | GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecEmotionRomInfo, SpecEmotionRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Enduro Racer (128K)

static struct BurnRomInfo SpecenduroRomDesc[] = {
	{ "Enduro Racer (1987)(Activision).tap", 49670, 0x0e34f8bf, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specenduro, Specenduro, Spec128)
STD_ROM_FN(Specenduro)

struct BurnDriver BurnSpecenduro = {
	"spec_enduro", NULL, "spec_spec128", NULL, "1987",
	"Enduro Racer (128K)\0", NULL, "Activision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecenduroRomInfo, SpecenduroRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Espada Sagrada, La (Spanish) (128K)

static struct BurnRomInfo SpecespadasagradaRomDesc[] = {
	{ "Espada Sagrada, La 128K (1990)(Topo Soft).tap", 93687, 0x60affc15, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specespadasagrada, Specespadasagrada, Spec128)
STD_ROM_FN(Specespadasagrada)

struct BurnDriver BurnSpecespadasagrada = {
	"spec_espadasagrada", NULL, "spec_spec128", NULL, "1990",
	"Espada Sagrada, La (Spanish) (128K)\0", NULL, "Topo Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecespadasagradaRomInfo, SpecespadasagradaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Exolon (48K)

static struct BurnRomInfo Specexolon48RomDesc[] = {
	{ "Exolon 48K (1987)(Hewson Consultants).tap", 44921, 0xedd68b67, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specexolon48, Specexolon48, Spectrum)
STD_ROM_FN(Specexolon48)

struct BurnDriver BurnSpecexolon48 = {
	"spec_exolon48", "spec_exolon128", "spec_spectrum", NULL, "1987",
	"Exolon (48K)\0", NULL, "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, Specexolon48RomInfo, Specexolon48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Exolon (128K)

static struct BurnRomInfo Specexolon128RomDesc[] = {
	{ "Exolon 128K (1987)(Hewson Consultants).tap", 42957, 0x7094b901, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specexolon128, Specexolon128, Spec128)
STD_ROM_FN(Specexolon128)

struct BurnDriver BurnSpecexolon128 = {
	"spec_exolon128", NULL, "spec_spec128", NULL, "1987",
	"Exolon (128K)\0", NULL, "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, Specexolon128RomInfo, Specexolon128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Fantasy Zone, The (128K)

static struct BurnRomInfo SpecfntzoneRomDesc[] = {
	{ "Fantasy Zone, The (19xx)(-)[128K].z80", 0x05c6c, 0xe09d79d8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specfntzone, Specfntzone, Spec128)
STD_ROM_FN(Specfntzone)

struct BurnDriver BurnSpecfntzone = {
	"spec_fntzone", NULL, "spec_spec128", NULL, "19xx",
	"Fantasy Zone, The (128K)\0", "Adult game", "Unknown", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MISC, 0,
	SpectrumGetZipName, SpecfntzoneRomInfo, SpecfntzoneRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Fast Food Dizzy (48K)

static struct BurnRomInfo Specffdizzy48RomDesc[] = {
	{ "Fast Food Dizzy 48K (1989)(Codemasters).tap", 42067, 0x9c820493, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specffdizzy48, Specffdizzy48, Spectrum)
STD_ROM_FN(Specffdizzy48)

struct BurnDriver BurnSpecffdizzy48 = {
	"spec_ffdizzy48", "spec_ffdizzy128", "spec_spectrum", NULL, "1989",
	"Fast Food Dizzy (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_MAZE | GBF_ACTION, 0,
	SpectrumGetZipName, Specffdizzy48RomInfo, Specffdizzy48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Fast Food Dizzy (128K)

static struct BurnRomInfo Specffdizzy128RomDesc[] = {
	{ "Fast Food Dizzy 128K (1989)(Codemasters).tap", 53170, 0x025c92cb, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specffdizzy128, Specffdizzy128, Spec128)
STD_ROM_FN(Specffdizzy128)

struct BurnDriver BurnSpecffdizzy128 = {
	"spec_ffdizzy128", NULL, "spec_spec128", NULL, "1989",
	"Fast Food Dizzy (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE | GBF_ACTION, 0,
	SpectrumGetZipName, Specffdizzy128RomInfo, Specffdizzy128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Fernando Martin Basket Master (Spanish) (48K)

static struct BurnRomInfo SpecfernmartbasketmRomDesc[] = {
	{ "Fernando Martin Basket Master ES 48K (1987)(Dinamic).tap", 47196, 0xf802ceb0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specfernmartbasketm, Specfernmartbasketm, Spectrum)
STD_ROM_FN(Specfernmartbasketm)

struct BurnDriver BurnSpecfernmartbasketm = {
	"spec_fernmartbasketm", NULL, "spec_spectrum", NULL, "1987",
	"Fernando Martin Basket Master (Spanish) (48K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecfernmartbasketmRomInfo, SpecfernmartbasketmRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Feud (48K)

static struct BurnRomInfo SpecfeudRomDesc[] = {
	{ "Feud (1987)(Bulldog).tap", 47901, 0xf69e87b4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specfeud, Specfeud, Spectrum)
STD_ROM_FN(Specfeud)

struct BurnDriver BurnSpecfeud = {
	"spec_feud", NULL, "spec_spectrum", NULL, "1987",
	"Feud (48K)\0", NULL, "Bulldog", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_ADV, 0,
	SpectrumGetZipName, SpecfeudRomInfo, SpecfeudRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Final Matrix, The (48K)

static struct BurnRomInfo SpecfinalmatrixRomDesc[] = {
	{ "Final Matrix, The (1987)(Gremlin Graphics).tap", 67727, 0xdb92a506, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specfinalmatrix, Specfinalmatrix, Spectrum)
STD_ROM_FN(Specfinalmatrix)

struct BurnDriver BurnSpecfinalmatrix = {
	"spec_finalmatrix", NULL, "spec_spectrum", NULL, "1987",
	"Final Matrix, The (48K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE | GBF_ACTION, 0,
	SpectrumGetZipName, SpecfinalmatrixRomInfo, SpecfinalmatrixRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Firelord (48K)

static struct BurnRomInfo SpecFirelordRomDesc[] = {
	{ "Firelord (1986)(Hewson Consultants).tap", 48559, 0xe586189f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFirelord, SpecFirelord, Spectrum)
STD_ROM_FN(SpecFirelord)

struct BurnDriver BurnSpecFirelord = {
	"spec_firelord", NULL, "spec_spectrum", NULL, "1986",
	"Firelord (48K)\0", NULL, "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV | GBF_ACTION, 0,
	SpectrumGetZipName, SpecFirelordRomInfo, SpecFirelordRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Flying Shark (128K)

static struct BurnRomInfo SpecfsharkRomDesc[] = {
	{ "Flying Shark 128K (1987)(Firebird).tap", 36523, 0x634f50aa, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specfshark, Specfshark, Spec128)
STD_ROM_FN(Specfshark)

struct BurnDriver BurnSpecfshark = {
	"spec_fshark", NULL, "spec_spec128", NULL, "1987",
	"Flying Shark (128K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecfsharkRomInfo, SpecfsharkRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Football Director (48K)

static struct BurnRomInfo SpecfootdireRomDesc[] = {
	{ "Football Director (1986)(D&H Games).tap", 47945, 0x71643b24, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specfootdire, Specfootdire, Spectrum)
STD_ROM_FN(Specfootdire)

struct BurnDriver BurnSpecfootdire = {
	"spec_footdire", NULL, "spec_spectrum", NULL, "1986",
	"Football Director (48K)\0", NULL, "D&H Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL | GBF_STRATEGY, 0,
	SpectrumGetZipName, SpecfootdireRomInfo, SpecfootdireRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Football Manager - World Cup Edition (48K)

static struct BurnRomInfo SpecftmanwcRomDesc[] = {
	{ "Football Manager - World Cup Edition (1990)(Addictive Games).tap", 41462, 0x52f22bdb, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specftmanwc, Specftmanwc, Spectrum)
STD_ROM_FN(Specftmanwc)

struct BurnDriver BurnSpecftmanwc = {
	"spec_ftmanwc", NULL, "spec_spectrum", NULL, "1990",
	"Football Manager - World Cup Edition (48K)\0", NULL, "Addictive Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL | GBF_STRATEGY, 0,
	SpectrumGetZipName, SpecftmanwcRomInfo, SpecftmanwcRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Football Manager (48K)

static struct BurnRomInfo SpecftmanRomDesc[] = {
	{ "Football Manager (1982)(Addictive Games).tap", 35582, 0x9ff2f05d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specftman, Specftman, Spectrum)
STD_ROM_FN(Specftman)

struct BurnDriver BurnSpecftman = {
	"spec_ftman", NULL, "spec_spectrum", NULL, "1982",
	"Football Manager (48K)\0", NULL, "Addictive Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL | GBF_STRATEGY, 0,
	SpectrumGetZipName, SpecftmanRomInfo, SpecftmanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Football Manager 2 (48K)

static struct BurnRomInfo Specfootman2RomDesc[] = {
	{ "Football Manager 2 (1988)(Addictive Games).tap", 50435, 0x060ddf75, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specfootman2, Specfootman2, Spectrum)
STD_ROM_FN(Specfootman2)

struct BurnDriver BurnSpecfootman2 = {
	"spec_footman2", NULL, "spec_spectrum", NULL, "1988",
	"Football Manager 2 (48K)\0", NULL, "Addictive Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL | GBF_STRATEGY, 0,
	SpectrumGetZipName, Specfootman2RomInfo, Specfootman2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Football Manager 3 (48K)

static struct BurnRomInfo Specfootman3RomDesc[] = {
	{ "Football Manager 3 (1991)(Addictive Games).tap", 46747, 0x5e1a8c16, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specfootman3, Specfootman3, Spectrum)
STD_ROM_FN(Specfootman3)

struct BurnDriver BurnSpecfootman3 = {
	"spec_footman3", NULL, "spec_spectrum", NULL, "1991",
	"Football Manager 3 (48K)\0", NULL, "Addictive Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL | GBF_STRATEGY, 0,
	SpectrumGetZipName, Specfootman3RomInfo, Specfootman3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Footballer of the Year (48K)

static struct BurnRomInfo SpecfotyRomDesc[] = {
	{ "Footballer of the Year (1986)(Gremlin Graphics Software).z80", 0x08137, 0x2af522d0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specfoty, Specfoty, Spectrum)
STD_ROM_FN(Specfoty)

struct BurnDriver BurnSpecfoty = {
	"spec_foty", NULL, "spec_spectrum", NULL, "1986",
	"Footballer of the Year (48K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL | GBF_STRATEGY, 0,
	SpectrumGetZipName, SpecfotyRomInfo, SpecfotyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Footballer of the Year 2 (48K)

static struct BurnRomInfo Specfoty2RomDesc[] = {
	{ "Footballer of the Year 2 (1987)(Gremlin Graphics Software).z80", 0x08d8f, 0x3722f534, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specfoty2, Specfoty2, Spectrum)
STD_ROM_FN(Specfoty2)

struct BurnDriver BurnSpecfoty2 = {
	"spec_foty2", NULL, "spec_spectrum", NULL, "1987",
	"Footballer of the Year 2 (48K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL | GBF_STRATEGY, 0,
	SpectrumGetZipName, Specfoty2RomInfo, Specfoty2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Forgotten Worlds (128K)

static struct BurnRomInfo SpecforgottnRomDesc[] = {
	{ "Forgotten Worlds (1989)(U.S. Gold)[128K].z80", 0x0e045, 0x33ef767e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specforgottn, Specforgottn, Spec128)
STD_ROM_FN(Specforgottn)

struct BurnDriver BurnSpecforgottn = {
	"spec_forgottn", NULL, "spec_spec128", NULL, "1989",
	"Forgotten Worlds (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecforgottnRomInfo, SpecforgottnRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Fred (48K)

static struct BurnRomInfo SpecfredRomDesc[] = {
	{ "Fred (1984)(Quicksilva).tap", 31242, 0x8c4bd9a9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specfred, Specfred, Spectrum)
STD_ROM_FN(Specfred)

struct BurnDriver BurnSpecfred = {
	"spec_fred", NULL, "spec_spectrum", NULL, "1984",
	"Fred (48K)\0", NULL, "Quicksilva", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecfredRomInfo, SpecfredRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Freddy Hardest (Spanish) (48K)

static struct BurnRomInfo SpecfredhardRomDesc[] = {
	{ "Freddy Hardest ES (1987)(Dinamic).tap", 95231, 0xff6d65f0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specfredhard, Specfredhard, Spectrum)
STD_ROM_FN(Specfredhard)

struct BurnDriver BurnSpecfredhard = {
	"spec_fredhard", NULL, "spec_spectrum", NULL, "1987",
	"Freddy Hardest (Spanish) (48K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecfredhardRomInfo, SpecfredhardRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Frogger (48K)

static struct BurnRomInfo SpecfroggerRomDesc[] = {
	{ "Frogger 48K (1983)(A'n'F Software).tap", 16497, 0xab533972, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specfrogger, Specfrogger, Spectrum)
STD_ROM_FN(Specfrogger)

struct BurnDriver BurnSpecfrogger = {
	"spec_frogger", NULL, "spec_spectrum", NULL, "1983",
	"Frogger (48K)\0", NULL, "A'n'F Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecfroggerRomInfo, SpecfroggerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Galaxian (48K)

static struct BurnRomInfo SpecgalaxianRomDesc[] = {
	{ "Galaxian 48K (1984)(Atarisoft).tap", 20304, 0x10276234, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specgalaxian, Specgalaxian, Spectrum)
STD_ROM_FN(Specgalaxian)

struct BurnDriver BurnSpecgalaxian = {
	"spec_galaxian", NULL, "spec_spectrum", NULL, "1984",
	"Galaxian (48K)\0", "Q: left - W: right - P: fire", "Atarisoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecgalaxianRomInfo, SpecgalaxianRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Game Over - Part 1 (48K)

static struct BurnRomInfo Specgameoverp1RomDesc[] = {
	{ "Game Over - Part 1 (1987)(Dinamic).tap", 40891, 0x9bc5e9b4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specgameoverp1, Specgameoverp1, Spectrum)
STD_ROM_FN(Specgameoverp1)

struct BurnDriver BurnSpecgameoverp1 = {
	"spec_gameoverp1", NULL, "spec_spectrum", NULL, "1987",
	"Game Over - Part 1 (48K)\0", NULL, "Imagine & Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, Specgameoverp1RomInfo, Specgameoverp1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Game Over - Part 2 (48K)

static struct BurnRomInfo Specgameoverp2RomDesc[] = {
	{ "Game Over - Part 2 (1987)(Dinamic).tap", 41167, 0x2657f897, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specgameoverp2, Specgameoverp2, Spectrum)
STD_ROM_FN(Specgameoverp2)

struct BurnDriver BurnSpecgameoverp2 = {
	"spec_gameoverp2", "spec_gameoverp1", "spec_spectrum", NULL, "1987",
	"Game Over - Part 2 (48K)\0", "Password: 18024", "Imagine & Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, Specgameoverp2RomInfo, Specgameoverp2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Garfield - Big, Fat, Hairy Deal (48K)

static struct BurnRomInfo SpecGarfieldRomDesc[] = {
	{ "Garfield - Big, Fat, Hairy Deal (1988)(The Edge).tap", 43785, 0x45c3ceb8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGarfield, SpecGarfield, Spectrum)
STD_ROM_FN(SpecGarfield)

struct BurnDriver BurnSpecGarfield = {
	"spec_garfield", NULL, "spec_spectrum", NULL, "1988",
	"Garfield - Big Fat Hairy Deal (48K)\0", NULL, "The Edge", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecGarfieldRomInfo, SpecGarfieldRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Gary Lineker's Hot-Shot! (48K)

static struct BurnRomInfo SpecglinhtRomDesc[] = {
	{ "Gary Lineker's Hot-Shot! (1988)(Gremlin Graphics Software).z80", 0x08a0a, 0x18e1d943, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specglinht, Specglinht, Spectrum)
STD_ROM_FN(Specglinht)

struct BurnDriver BurnSpecglinht = {
	"spec_glinht", NULL, "spec_spectrum", NULL, "1988",
	"Gary Lineker's Hot-Shot! (48K)\0", "Press 'Z' and 'X' to select and 'K' to change options", "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecglinhtRomInfo, SpecglinhtRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Gary Lineker's Super Skills (128K)

static struct BurnRomInfo SpecglssRomDesc[] = {
	{ "Gary Lineker's Super Skills (1988)(Gremlin Graphics Software)[128K].z80", 0x12e6f, 0xfcb98fd1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specglss, Specglss, Spec128)
STD_ROM_FN(Specglss)

struct BurnDriver BurnSpecglss = {
	"spec_glss", NULL, "spec_spec128", NULL, "1988",
	"Gary Lineker's Super Skills (128K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 4, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecglssRomInfo, SpecglssRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Gary Lineker's Super Star Soccer (48K)

static struct BurnRomInfo SpecglsssRomDesc[] = {
	{ "Gary Lineker's Super Star Soccer 48K (1987)(Gremlin Graphics).tap", 47904, 0x4eda2a1e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specglsss, Specglsss, Spectrum)
STD_ROM_FN(Specglsss)

struct BurnDriver BurnSpecglsss = {
	"spec_glsss", NULL, "spec_spectrum", NULL, "1987",
	"Gary Lineker's Super Star Soccer (48K)\0", "Press '2' to use controller", "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_STRATEGY, 0,
	SpectrumGetZipName, SpecglsssRomInfo, SpecglsssRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ghosts 'n' Goblins (48K)

static struct BurnRomInfo Specgng_48RomDesc[] = {
	{ "Ghosts 'n' Goblins (1986)(Elite Systems).z80", 0x0b805, 0xdc252529, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specgng_48, Specgng_48, Spectrum)
STD_ROM_FN(Specgng_48)

struct BurnDriver BurnSpecgng_48 = {
	"spec_gng_48", NULL, "spec_spectrum", NULL, "1986",
	"Ghosts 'n' Goblins (48K)\0", NULL, "Elite Systems", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, Specgng_48RomInfo, Specgng_48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ghostbusters (48K)

static struct BurnRomInfo Specghostb_48RomDesc[] = {
	{ "Ghostbusters 48K (1984)(Activision).tap", 39790, 0x25deb7c0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specghostb_48, Specghostb_48, Spectrum)
STD_ROM_FN(Specghostb_48)

struct BurnDriver BurnSpecghostb_48 = {
	"spec_ghostb_48", "spec_ghostb", "spec_spectrum", NULL, "1984",
	"Ghostbusters (48K)\0", NULL, "Activision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specghostb_48RomInfo, Specghostb_48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ghostbusters (128K)

static struct BurnRomInfo SpecghostbRomDesc[] = {
	{ "Ghostbusters 128K (1984)(Activision).tap", 69552, 0xc4c790a1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specghostb, Specghostb, Spec128)
STD_ROM_FN(Specghostb)

struct BurnDriver BurnSpecghostb = {
	"spec_ghostb", NULL, "spec_spec128", NULL, "1984",
	"Ghostbusters (128K)\0", NULL, "Activision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecghostbRomInfo, SpecghostbRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ghostbusters II (128K) (Trainer)

static struct BurnRomInfo Specghostb2RomDesc[] = {
	{ "Ghostbusters II (1989)(Activision)[t].tap", 174896, 0x1594430c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specghostb2, Specghostb2, Spec128)
STD_ROM_FN(Specghostb2)

struct BurnDriver BurnSpecghostb2 = {
	"spec_ghostb2", NULL, "spec_spec128", NULL, "1989",
	"Ghostbusters II (128K) (Trainer)\0", NULL, "Activision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specghostb2RomInfo, Specghostb2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ghouls 'n' Ghosts (128K)

static struct BurnRomInfo SpecgngRomDesc[] = {
	{ "Ghouls 'n' Ghosts (1989)(U.S. Gold)[128K].z80", 0x1a8d2, 0x1b626fe8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specgng, Specgng, Spec128)
STD_ROM_FN(Specgng)

struct BurnDriver BurnSpecgng = {
	"spec_gng", NULL, "spec_spec128", NULL, "1989",
	"Ghouls 'n' Ghosts (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecgngRomInfo, SpecgngRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// G-LOC (128K)

static struct BurnRomInfo SpecglocRomDesc[] = {
	{ "G-LOC (1992)(U.S. Gold)[128K].z80", 0x16f39, 0xeeae7278, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specgloc, Specgloc, Spec128)
STD_ROM_FN(Specgloc)

struct BurnDriver BurnSpecgloc = {
	"spec_gloc", NULL, "spec_spec128", NULL, "1992",
	"G-LOC (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecglocRomInfo, SpecglocRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Great Escape, The (48K)

static struct BurnRomInfo SpecgreatescRomDesc[] = {
	{ "Great Escape, The (1986)(Ocean).tap", 50366, 0x0d3be279, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specgreatesc, Specgreatesc, Spectrum)
STD_ROM_FN(Specgreatesc)

struct BurnDriver BurnSpecgreatesc = {
	"spec_greatesc", NULL, "spec_spectrum", NULL, "1986",
	"Great Escape, The (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecgreatescRomInfo, SpecgreatescRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Green Beret (48K)

static struct BurnRomInfo Specgberet_48RomDesc[] = {
	{ "Green Beret (1986)(Imagine Software).z80", 0x0ad53, 0x55f36544, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specgberet_48, Specgberet_48, Spectrum)
STD_ROM_FN(Specgberet_48)

struct BurnDriver BurnSpecgberet_48 = {
	"spec_gberet_48", NULL, "spec_spectrum", NULL, "1986",
	"Green Beret (48K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, Specgberet_48RomInfo, Specgberet_48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Gryzor (48K)

static struct BurnRomInfo Specgryzor48RomDesc[] = {
	{ "Gryzor 48K (1987)(Ocean).tap", 49774, 0x1b11c206, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specgryzor48, Specgryzor48, Spectrum)
STD_ROM_FN(Specgryzor48)

struct BurnDriver BurnSpecgryzor48 = {
	"spec_gryzor48", "spec_gryzor128", "spec_spectrum", NULL, "1987",
	"Gryzor (48K)\0", "SPACE: jump - SPACE+down: climb down", "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, Specgryzor48RomInfo, Specgryzor48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Gryzor (128K)

static struct BurnRomInfo Specgryzor128RomDesc[] = {
	{ "Gryzor 128K (1987)(Ocean).tap", 115219, 0x941d84c9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specgryzor128, Specgryzor128, Spec128)
STD_ROM_FN(Specgryzor128)

struct BurnDriver BurnSpecgryzor128 = {
	"spec_gryzor128", NULL, "spec_spec128", NULL, "1987",
	"Gryzor (128K)\0", "SPACE: jump - SPACE+down: climb down", "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, Specgryzor128RomInfo, Specgryzor128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hammer Boy (Part 1) (128K)

static struct BurnRomInfo Spechammerboy1RomDesc[] = {
	{ "Hammer Boy (1991)(Dinamic Software)(Part 1 of 2)(128k).tap", 48943, 0x1770245b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spechammerboy1, Spechammerboy1, Spec128)
STD_ROM_FN(Spechammerboy1)

struct BurnDriver BurnSpechammerboy1 = {
	"spec_hammerboy1", NULL, "spec_spec128", NULL, "1991",
	"Hammer Boy (Part 1) (128K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Spechammerboy1RomInfo, Spechammerboy1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hammer Boy (Part 2) (128K)

static struct BurnRomInfo Spechammerboy2RomDesc[] = {
	{ "Hammer Boy (1991)(Dinamic Software)(Part 2 of 2)(128k).tap", 48943, 0x858a0af4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spechammerboy2, Spechammerboy2, Spec128)
STD_ROM_FN(Spechammerboy2)

struct BurnDriver BurnSpechammerboy2 = {
	"spec_hammerboy2", "spec_hammerboy1", "spec_spec128", NULL, "1991",
	"Hammer Boy (Part 2) (128K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Spechammerboy2RomInfo, Spechammerboy2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// H.A.T.E. - Hostile All Terrain Encounter (48K)

static struct BurnRomInfo SpechatehateRomDesc[] = {
	{ "H.A.T.E. - Hostile All Terrain Encounter (1989)(Vortex Software).tap", 42128, 0x0fcfc02e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spechatehate, Spechatehate, Spectrum)
STD_ROM_FN(Spechatehate)

struct BurnDriver BurnSpechatehate = {
	"spec_hatehate", NULL, "spec_spectrum", NULL, "1989",
	"H.A.T.E. - Hostile All Terrain Encounter (48K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpechatehateRomInfo, SpechatehateRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hard Drivin' (128K)

static struct BurnRomInfo SpecharddrivRomDesc[] = {
	{ "Hard Drivin' 128K (1989)(Domark).tap", 47211, 0xf55dc984, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specharddriv, Specharddriv, Spec128)
STD_ROM_FN(Specharddriv)

struct BurnDriver BurnSpecharddriv = {
	"spec_harddriv", NULL, "spec_spec128", NULL, "1989",
	"Hard Drivin' (128K)\0", NULL, "Domark", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecharddrivRomInfo, SpecharddrivRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Head over Heels (128K)

static struct BurnRomInfo SpecheadheelRomDesc[] = {
	{ "Head over Heels (1987)(Ocean)[128K].z80", 0x0c5a5, 0x9f148037, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specheadheel, Specheadheel, Spec128)
STD_ROM_FN(Specheadheel)

struct BurnDriver BurnSpecheadheel = {
	"spec_headheel", NULL, "spec_spec128", NULL, "1987",
	"Head over Heels (128K)\0", "Carry: ENTER - Fire: X - Swop: S", "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecheadheelRomInfo, SpecheadheelRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Heart Broken (48K)

static struct BurnRomInfo SpechrtbbrknRomDesc[] = {
	{ "Heart Broken 48K (1989)(Atlantis Software).tap", 54856, 0x34ed13a7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spechrtbbrkn, Spechrtbbrkn, Spectrum)
STD_ROM_FN(Spechrtbbrkn)

struct BurnDriver BurnSpechrtbbrkn = {
	"spec_hrtbbrkn", NULL, "spec_spectrum", NULL, "1989",
	"Heart Broken (48K)\0", NULL, "Atlantis Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpechrtbbrknRomInfo, SpechrtbbrknRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Heartland (48K)

static struct BurnRomInfo SpecheartlanRomDesc[] = {
	{ "Heartland (1986)(Odin Computer Graphics).tap", 42923, 0xd756125c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specheartlan, Specheartlan, Spectrum)
STD_ROM_FN(Specheartlan)

struct BurnDriver BurnSpecheartlan = {
	"spec_heartlan", NULL, "spec_spectrum", NULL, "1986",
	"Heartland (48K)\0", NULL, "Odin Computer Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_ADV, 0,
	SpectrumGetZipName, SpecheartlanRomInfo, SpecheartlanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Highway Encounter (48K)

static struct BurnRomInfo SpechighencoRomDesc[] = {
	{ "Highway Encounter (1985)(Vortex Software).z80", 0x0a36c, 0x7946eec7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spechighenco, Spechighenco, Spectrum)
STD_ROM_FN(Spechighenco)

struct BurnDriver BurnSpechighenco = {
	"spec_highenco", NULL, "spec_spectrum", NULL, "1985",
	"Highway Encounter (48K)\0", NULL, "Vortex Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpechighencoRomInfo, SpechighencoRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hobbit, The v1.0 (48K)

static struct BurnRomInfo Spechobbit2RomDesc[] = {
	{ "Hobbit, The v1.0 (1982)(Melbourne House).z80", 0x0a4c3, 0x46c20d35, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spechobbit2, Spechobbit2, Spectrum)
STD_ROM_FN(Spechobbit2)

struct BurnDriver BurnSpechobbit2 = {
	"spec_hobbit2", "spec_hobbit", "spec_spectrum", NULL, "1982",
	"Hobbit, The v1.0 (48K)\0", NULL, "Melbourne House", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, Spechobbit2RomInfo, Spechobbit2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hobbit, The v1.2 (48K)

static struct BurnRomInfo SpechobbitRomDesc[] = {
	{ "Hobbit, The v1.2 (1982)(Melbourne House).z80", 0x0b1ee, 0x10231c84, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spechobbit, Spechobbit, Spectrum)
STD_ROM_FN(Spechobbit)

struct BurnDriver BurnSpechobbit = {
	"spec_hobbit", NULL, "spec_spectrum", NULL, "1982",
	"Hobbit, The v1.2 (48K)\0", NULL, "Melbourne House", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpechobbitRomInfo, SpechobbitRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Horace Goes Skiing (48K)

static struct BurnRomInfo SpechoraceskRomDesc[] = {
	{ "Horace Goes Skiing (1982)(Sinclair Research).z80", 0x0c01e, 0x02cd124b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spechoracesk, Spechoracesk, Spectrum)
STD_ROM_FN(Spechoracesk)

struct BurnDriver BurnSpechoracesk = {
	"spec_horacesk", NULL, "spec_spectrum", NULL, "1982",
	"Horace Goes Skiing (48K)\0", NULL, "Sinclair Research", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpechoraceskRomInfo, SpechoraceskRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// How to be a Complete Bastard (128K)

static struct BurnRomInfo SpechowbastRomDesc[] = {
	{ "How to be a Complete Bastard (1987)(Virgin Games)[128K].z80", 0x0e728, 0x7460da43, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spechowbast, Spechowbast, Spec128)
STD_ROM_FN(Spechowbast)

struct BurnDriver BurnSpechowbast = {
	"spec_howbast", NULL, "spec_spec128", NULL, "1987",
	"How to be a Complete Bastard (128K)\0", NULL, "Virgin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpechowbastRomInfo, SpechowbastRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hudson Hawk (128K)

static struct BurnRomInfo SpechudshawkRomDesc[] = {
	{ "Hudson Hawk 128K (1991)(Ocean).tap", 135956, 0xbccb6e2a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spechudshawk, Spechudshawk, Spec128)
STD_ROM_FN(Spechudshawk)

struct BurnDriver BurnSpechudshawk = {
	"spec_hudshawk", NULL, "spec_spec128", NULL, "1991",
	"Hudson Hawk (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpechudshawkRomInfo, SpechudshawkRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hunchback (48K)

static struct BurnRomInfo SpechunchbacRomDesc[] = {
	{ "Hunchback 48K (1984)(Ocean).tap", 43449, 0x3fd8f859, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spechunchbac, Spechunchbac, Spectrum)
STD_ROM_FN(Spechunchbac)

struct BurnDriver BurnSpechunchbac = {
	"spec_hunchbac", NULL, "spec_spectrum", NULL, "1984",
	"Hunchback (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpechunchbacRomInfo, SpechunchbacRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hunchback II - Quasimodo's Revenge (48K)

static struct BurnRomInfo Spechnchbac2RomDesc[] = {
	{ "Hunchback II - Quasimodo's Revenge 48K (1984)(Ocean).tap", 49221, 0x65a9fcbe, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spechnchbac2, Spechnchbac2, Spectrum)
STD_ROM_FN(Spechnchbac2)

struct BurnDriver BurnSpechnchbac2 = {
	"spec_hnchbac2", NULL, "spec_spectrum", NULL, "1985",
	"Hunchback II - Quasimodo's Revenge (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Spechnchbac2RomInfo, Spechnchbac2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// I, Ball (48K)

static struct BurnRomInfo SpeciballRomDesc[] = {
	{ "I, Ball 48K (1987)(Firebird).tap", 53775, 0x4257b017, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speciball, Speciball, Spectrum)
STD_ROM_FN(Speciball)

struct BurnDriver BurnSpeciball = {
	"spec_iball", NULL, "spec_spectrum", NULL, "1987",
	"I, Ball (48K)\0", "Press '3' to use controller", "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpeciballRomInfo, SpeciballRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Incredible Shrinking Fireman, The (48K)

static struct BurnRomInfo SpecincshrfRomDesc[] = {
	{ "Incredible Shrinking Fireman, The 48K (1986)(Mastertronic).tap", 46523, 0x1940b148, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specincshrf, Specincshrf, Spectrum)
STD_ROM_FN(Specincshrf)

struct BurnDriver BurnSpecincshrf = {
	"spec_incshrf", NULL, "spec_spectrum", NULL, "1986",
	"Incredible Shrinking Fireman, The (48K)\0", NULL, "Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecincshrfRomInfo, SpecincshrfRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Indoor Soccer (48K)

static struct BurnRomInfo SpecindrsoccrRomDesc[] = {
	{ "Indoor Soccer (1986)(Magnificent 7 Software).z80", 0x070ed, 0x85a1a21f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specindrsoccr, Specindrsoccr, Spectrum)
STD_ROM_FN(Specindrsoccr)

struct BurnDriver BurnSpecindrsoccr = {
	"spec_indrsoccr", NULL, "spec_spectrum", NULL, "1986",
	"Indoor Soccer (48K)\0", NULL, "Magnificent 7 Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL, 0,
	SpectrumGetZipName, SpecindrsoccrRomInfo, SpecindrsoccrRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Into the Eagle's Nest (48K-128K)

static struct BurnRomInfo SpecinteagnRomDesc[] = {
	{ "Into the Eagle's Nest (1987)(Pandora).tap", 40894, 0x289b4c8e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specinteagn, Specinteagn, Spec128)
STD_ROM_FN(Specinteagn)

struct BurnDriver BurnSpecinteagn = {
	"spec_inteagn", NULL, "spec_spec128", NULL, "1987",
	"Into the Eagle's Nest (48K-128K)\0", NULL, "Pandora", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE | GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecinteagnRomInfo, SpecinteagnRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ivan 'Ironman' Stewart's Super Off Road Racer (128K)

static struct BurnRomInfo SpecironmanRomDesc[] = {
	{ "Ivan 'Ironman' Stewart's Super Off Road Racer 128K (1990)(Virgin Games).tap", 40051, 0x68cf1ab9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specironman, Specironman, Spec128)
STD_ROM_FN(Specironman)

struct BurnDriver BurnSpecironman = {
	"spec_ironman", NULL, "spec_spec128", NULL, "1990",
	"Ivan 'Ironman' Stewart's Super Off Road Racer (128K)\0", "Press 6, 7 and ENTER to choose the initial settings", "Virgin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecironmanRomInfo, SpecironmanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Jack the Nipper (48K)

static struct BurnRomInfo SpecjacknipperRomDesc[] = {
	{ "Jack the Nipper 48K (1986)(Gremlin Graphics).tap", 46966, 0x729f41ab, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specjacknipper, Specjacknipper, Spectrum)
STD_ROM_FN(Specjacknipper)

struct BurnDriver BurnSpecjacknipper = {
	"spec_jacknipper", NULL, "spec_spectrum", NULL, "1986",
	"Jack the Nipper (48K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecjacknipperRomInfo, SpecjacknipperRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Jack the Nipper II - In Coconut Capers (48K)

static struct BurnRomInfo Specjacknipper2RomDesc[] = {
	{ "Jack the Nipper II - In Coconut Capers 48K (1987)(Gremlin Graphics).z80", 43126, 0xdc5995ff, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specjacknipper2, Specjacknipper2, Spectrum)
STD_ROM_FN(Specjacknipper2)

struct BurnDriver BurnSpecjacknipper2 = {
	"spec_jacknipper2", NULL, "spec_spectrum", NULL, "1987",
	"Jack the Nipper II - In Coconut Capers (48K)\0", "Set '4 Interface II' to use controller", "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Specjacknipper2RomInfo, Specjacknipper2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Jet Set Willy (48K)

static struct BurnRomInfo SpecjswillyRomDesc[] = {
	{ "Jet Set Willy 48K (1984)(Software Projects).tap", 33280, 0x17ec8978, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specjswilly, Specjswilly, Spectrum)
STD_ROM_FN(Specjswilly)

struct BurnDriver BurnSpecjswilly = {
	"spec_jswilly", NULL, "spec_spectrum", NULL, "1984",
	"Jet Set Willy (48K)\0", NULL, "Software Projects", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecjswillyRomInfo, SpecjswillyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecCursorKeysDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Jet Set Willy II - The Final Frontier (48K)

static struct BurnRomInfo Specjswilly2RomDesc[] = {
	{ "Jet Set Willy II - The Final Frontier 48K (1985)(Software Projects).tap", 49881, 0x85a4ac92, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specjswilly2, Specjswilly2, Spectrum)
STD_ROM_FN(Specjswilly2)

struct BurnDriver BurnSpecjswilly2 = {
	"spec_jswilly2", NULL, "spec_spectrum", NULL, "1985",
	"Jet Set Willy II - The Final Frontier (48K)\0", NULL, "Software Projects", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Specjswilly2RomInfo, Specjswilly2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPMDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Jet Set Willy III (48K)

static struct BurnRomInfo Specjswilly3RomDesc[] = {
	{ "Jet Set Willy III (1985)(MB - APG Software).z80", 0x08061, 0xbde7b5ae, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specjswilly3, Specjswilly3, Spectrum)
STD_ROM_FN(Specjswilly3)

struct BurnDriver BurnSpecjswilly3 = {
	"spec_jswilly3", NULL, "spec_spectrum", NULL, "1985",
	"Jet Set Willy III (48K)\0", NULL, "MB - APG Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Specjswilly3RomInfo, Specjswilly3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Jet Pac (16K)

static struct BurnRomInfo SpecjetpacRomDesc[] = {
	{ "Jet Pac 16K (1983)(Ultimate Play The Game).tap",     15366, 0xeb8ce1c8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specjetpac, Specjetpac, Spectrum)
STD_ROM_FN(Specjetpac)

struct BurnDriver BurnSpecjetpac = {
	"spec_jetpac", NULL, "spec_spectrum", NULL, "1983",
	"Jet Pac (16K)\0", NULL, "Ultimate Play The Game", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecjetpacRomInfo, SpecjetpacRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Jetsons The Computer Game (128K)

static struct BurnRomInfo SpecJetsonsRomDesc[] = {
	{ "Jetsons - The Computer Game 128K (1992)(Hi-Tec).tap", 117628, 0xdf6fe05a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecJetsons, SpecJetsons, Spec128)
STD_ROM_FN(SpecJetsons)

struct BurnDriver BurnSpecJetsons = {
	"spec_jetsons", NULL, "spec_spec128", NULL, "1992",
	"Jetsons The Computer Game (128K)\0", "Mega Trainer version", "Hi-Tec Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecJetsonsRomInfo, SpecJetsonsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Joe Blade (128K)

static struct BurnRomInfo Specjoebld128RomDesc[] = {
	{ "Joe Blade (1987)(Players)(48K-128K).tap", 50660, 0x27e8fa83, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specjoebld128, Specjoebld128, Spec128)
STD_ROM_FN(Specjoebld128)

struct BurnDriver BurnSpecjoebld128 = {
	"spec_joebld128", NULL, "spec_spec128", NULL, "1987",
	"Joe Blade (128K)\0", NULL, "Players Premier", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_ADV, 0,
	SpectrumGetZipName, Specjoebld128RomInfo, Specjoebld128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Joe Blade (48K)

static struct BurnRomInfo Specjoebld48RomDesc[] = {
	{ "Joe Blade (1987)(Players)(48K-128K).tap", 50660, 0x27e8fa83, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specjoebld48, Specjoebld48, Spectrum)
STD_ROM_FN(Specjoebld48)

struct BurnDriver BurnSpecjoebld48 = {
	"spec_joebld48", "spec_joebld128", "spec_spectrum", NULL, "1987",
	"Joe Blade (48K)\0", NULL, "Players Premier", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_ADV, 0,
	SpectrumGetZipName, Specjoebld48RomInfo, Specjoebld48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Joe Blade II (128K)

static struct BurnRomInfo Specjoebld2RomDesc[] = {
	{ "Joe Blade 2 (1988)(Players Premier)(128k).z80", 0x0bb13, 0x20a19599, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specjoebld2, Specjoebld2, Spec128)
STD_ROM_FN(Specjoebld2)

struct BurnDriver BurnSpecjoebld2 = {
	"spec_joebld2", NULL, "spec_spec128", NULL, "1988",
	"Joe Blade II (128K)\0", NULL, "Players Premier", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_ADV, 0,
	SpectrumGetZipName, Specjoebld2RomInfo, Specjoebld2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Joe Blade III (128K)

static struct BurnRomInfo Specjoebld3RomDesc[] = {
	{ "Joe Blade 3 (1989)(Players Premier)(128k).z80", 0x0b400, 0x422359d9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specjoebld3, Specjoebld3, Spec128)
STD_ROM_FN(Specjoebld3)

struct BurnDriver BurnSpecjoebld3 = {
	"spec_joebld3", NULL, "spec_spec128", NULL, "1989",
	"Joe Blade III (128K)\0", NULL, "Players Premier", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_ADV, 0,
	SpectrumGetZipName, Specjoebld3RomInfo, Specjoebld3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Kick Off (48K)

static struct BurnRomInfo SpeckickoffRomDesc[] = {
	{ "Kick Off 48K (1989)(Anco).tap", 47041, 0xf54acc07, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speckickoff, Speckickoff, Spectrum)
STD_ROM_FN(Speckickoff)

struct BurnDriver BurnSpeckickoff = {
	"spec_kickoff", NULL, "spec_spectrum", NULL, "1989",
	"Kick Off (48K)\0", NULL, "Anco", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL, 0,
	SpectrumGetZipName, SpeckickoffRomInfo, SpeckickoffRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Kick Off 2 (128K)

static struct BurnRomInfo SpecKickoff2RomDesc[] = {
	{ "Kick Off 2 (1990)(Anco).tap", 77052, 0x74a921aa, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecKickoff2, SpecKickoff2, Spec128)
STD_ROM_FN(SpecKickoff2)

struct BurnDriver BurnSpecKickoff2 = {
	"spec_kickoff2", NULL, "spec_spec128", NULL, "1990",
	"Kick Off 2 (128K)\0", "Use Q, A and C to choose and select options", "Anco", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL, 0,
	SpectrumGetZipName, SpecKickoff2RomInfo, SpecKickoff2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Kick Off World Cup Edition (128K)

static struct BurnRomInfo SpeckickoffwRomDesc[] = {
	{ "Kick Off World Cup Edition (1990)(Anco Software)[128K].z80", 0x09ba6, 0x2a01e70e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speckickoffw, Speckickoffw, Spec128)
STD_ROM_FN(Speckickoffw)

struct BurnDriver BurnSpeckickoffw = {
	"spec_kickoffw", NULL, "spec_spec128", NULL, "1990",
	"Kick Off World Cup Edition (128K)\0", NULL, "Anco", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL, 0,
	SpectrumGetZipName, SpeckickoffwRomInfo, SpeckickoffwRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Kikstart 2 + Course Designer (48K)

static struct BurnRomInfo Speckikstrt2RomDesc[] = {
	{ "Kikstart 2 + Course Designer (1988)(Mastertronic).tap", 35230, 0xf4beb546, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speckikstrt2, Speckikstrt2, Spectrum)
STD_ROM_FN(Speckikstrt2)

struct BurnDriver BurnSpeckikstrt2 = {
	"spec_kikstrt2", NULL, "spec_spectrum", NULL, "1988",
	"Kikstart 2 + Course Designer (48K)\0", "Set '4. CURSOR' to use controller", "Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Speckikstrt2RomInfo, Speckikstrt2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecCursorKeysDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Knight Lore (48K)

static struct BurnRomInfo SpecknigloreRomDesc[] = {
	{ "Knight Lore (1984)(Ultimate Play The Game).z80", 0x08c4c, 0x137ffdb2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speckniglore, Speckniglore, Spectrum)
STD_ROM_FN(Speckniglore)

struct BurnDriver BurnSpeckniglore = {
	"spec_kniglore", NULL, "spec_spectrum", NULL, "1984",
	"Knight Lore (48K)\0", NULL, "Ultimate Play The Game", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecknigloreRomInfo, SpecknigloreRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Kong (Ocean) (48K)

static struct BurnRomInfo SpeckongRomDesc[] = {
	{ "Kong 48K (1983)(Ocean).tap", 45603, 0xd5a04266, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speckong, Speckong, Spectrum)
STD_ROM_FN(Speckong)

struct BurnDriver BurnSpeckong = {
	"spec_kong", NULL, "spec_spectrum", NULL, "1984",
	"Kong (Ocean) (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpeckongRomInfo, SpeckongRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Kong 2 - Kong Strikes Back (48K)

static struct BurnRomInfo Speckong2RomDesc[] = {
	{ "Kong Strikes Back! 48K (1984)(Ocean).tap", 32685, 0x134e89ea, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speckong2, Speckong2, Spectrum)
STD_ROM_FN(Speckong2)

struct BurnDriver BurnSpeckong2 = {
	"spec_kong2", NULL, "spec_spectrum", NULL, "1985",
	"Kong 2 - Kong Strikes Back (48K)\0", "Press 'J' for controller menu", "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Speckong2RomInfo, Speckong2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Kung-Fu Master (48K)

static struct BurnRomInfo SpeckungfumRomDesc[] = {
	{ "Kung-Fu Master (1986)(U.S. Gold).z80", 0x0a7a1, 0x7a375d9e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speckungfum, Speckungfum, Spectrum)
STD_ROM_FN(Speckungfum)

struct BurnDriver BurnSpeckungfum = {
	"spec_kungfum", NULL, "spec_spectrum", NULL, "1986",
	"Kung-Fu Master (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpeckungfumRomInfo, SpeckungfumRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Kwik Snax Dizzy (128K)

static struct BurnRomInfo SpecksdizzyRomDesc[] = {
	{ "Kwik Snax Dizzy 128K (1990)(Codemasters).tap", 100738, 0x0b234319, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specksdizzy, Specksdizzy, Spec128)
STD_ROM_FN(Specksdizzy)

struct BurnDriver BurnSpecksdizzy = {
	"spec_ksdizzy", NULL, "spec_spec128", NULL, "1990",
	"Kwik Snax Dizzy (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE | GBF_ACTION, 0,
	SpectrumGetZipName, SpecksdizzyRomInfo, SpecksdizzyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Last Ninja 2 - Back with a Vengeance (128K)

static struct BurnRomInfo Speclninja2RomDesc[] = {
	{ "Last Ninja 2 - Back with a Vengeance (1988)(System 3).tap", 233900, 0xa6be2baa, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speclninja2, Speclninja2, Spec128)
STD_ROM_FN(Speclninja2)

struct BurnDriver BurnSpeclninja2 = {
	"spec_lninja2", NULL, "spec_spec128", NULL, "1988",
	"Last Ninja 2 - Back with a Vengeance (128K)\0", NULL, "System 3", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_ADV, 0,
	SpectrumGetZipName, Speclninja2RomInfo, Speclninja2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Line of Fire (128K)

static struct BurnRomInfo SpeclinefireRomDesc[] = {
	{ "Line of Fire 128K (1990)(U.S. Gold).tap", 165694, 0x4f1c0e0e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speclinefire, Speclinefire, Spec128)
STD_ROM_FN(Speclinefire)

struct BurnDriver BurnSpeclinefire = {
	"spec_linefire", NULL, "spec_spec128", NULL, "1990",
	"Line of Fire (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpeclinefireRomInfo, SpeclinefireRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Livingstone, I Presume (128K)

static struct BurnRomInfo SpeclivipresRomDesc[] = {
	{ "Livingstone, I Presume (1987)(Opera Soft)(128k).z80", 0x0aff2, 0x50f57c95, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speclivipres, Speclivipres, Spec128)
STD_ROM_FN(Speclivipres)

struct BurnDriver BurnSpeclivipres = {
	"spec_livipres", NULL, "spec_spec128", NULL, "1987",
	"Livingstone, I Presume (128K)\0", NULL, "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpeclivipresRomInfo, SpeclivipresRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Livingstone Supongo II (I Premuse II) - Part 1 (48K)

static struct BurnRomInfo Speclivipres21RomDesc[] = {
	{ "Livingstone Supongo II - part 1 48K (1989)(Opera Soft).z80", 42326, 0x995fb9a8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speclivipres21, Speclivipres21, Spectrum)
STD_ROM_FN(Speclivipres21)

struct BurnDriver BurnSpeclivipres21 = {
	"spec_livipres21", NULL, "spec_spectrum", NULL, "1989",
	"Livingstone Supongo II (I Premuse II) - Part 1 (48K)\0", NULL, "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Speclivipres21RomInfo, Speclivipres21RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Livingstone Supongo II (I Premuse II) - Part 2 (48K)

static struct BurnRomInfo Speclivipres22RomDesc[] = {
	{ "Livingstone Supongo II - part 2 48K (1989)(Opera Soft).z80", 36481, 0x851ca3c8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speclivipres22, Speclivipres22, Spectrum)
STD_ROM_FN(Speclivipres22)

struct BurnDriver BurnSpeclivipres22 = {
	"spec_livipres22", "spec_livipres21", "spec_spectrum", NULL, "1989",
	"Livingstone Supongo II (I Premuse II) - Part 2 (48K)\0", "Password: 15215", "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Speclivipres22RomInfo, Speclivipres22RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Lode Runner (Part 1 of 2) (48K)

static struct BurnRomInfo Specloderunn_48RomDesc[] = {
	{ "Lode Runner (1984)(Software Projects)(Part 1 of 2).z80", 0x07db5, 0x60fd844e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specloderunn_48, Specloderunn_48, Spectrum)
STD_ROM_FN(Specloderunn_48)

struct BurnDriver BurnSpecloderunn_48 = {
	"spec_loderunn_48", "spec_loderunn", "spec_spectrum", NULL, "1984",
	"Lode Runner (Part 1 of 2) (48K)\0", NULL, "Software Projects", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Specloderunn_48RomInfo, Specloderunn_48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Lode Runner (Part 2 of 2) (48K)

static struct BurnRomInfo Specloderunn2_48RomDesc[] = {
	{ "Lode Runner (1984)(Software Projects)(Part 2 of 2).z80", 0x07c15, 0x94d3a6c5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specloderunn2_48, Specloderunn2_48, Spectrum)
STD_ROM_FN(Specloderunn2_48)

struct BurnDriver BurnSpecloderunn2_48 = {
	"spec_loderunn2_48", "spec_loderunn", "spec_spectrum", NULL, "1984",
	"Lode Runner (Part 2 of 2) (48K)\0", NULL, "Software Projects", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Specloderunn2_48RomInfo, Specloderunn2_48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Lode Runner (128K)

static struct BurnRomInfo SpecloderunnRomDesc[] = {
	{ "Lode Runner (1984)(Software Projects)[128K].z80", 0x0fab1, 0x937fee1b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specloderunn, Specloderunn, Spec128)
STD_ROM_FN(Specloderunn)

struct BurnDriver BurnSpecloderunn = {
	"spec_loderunn", NULL, "spec_spec128", NULL, "1984",
	"Lode Runner (128K)\0", NULL, "Software Projects", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecloderunnRomInfo, SpecloderunnRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Lode Runner v2 (48K)

static struct BurnRomInfo Specloderun2RomDesc[] = {
	{ "Lode Runner v2 (1984)(Software Projects).z80", 0x07bf5, 0x4a3ca5b1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specloderun2, Specloderun2, Spectrum)
STD_ROM_FN(Specloderun2)

struct BurnDriver BurnSpecloderun2 = {
	"spec_loderun2", "spec_loderunn", "spec_spectrum", NULL, "1984",
	"Lode Runner v2 (48K)\0", NULL, "Software Projects", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Specloderun2RomInfo, Specloderun2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Lotus Esprit Turbo Challenge (128K)

static struct BurnRomInfo SpeclotustcRomDesc[] = {
	{ "Lotus Esprit Turbo Challenge (1990)(Gremlin Graphics Software)(128k).z80", 0x159b6, 0xe2a3503d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speclotustc, Speclotustc, Spec128)
STD_ROM_FN(Speclotustc)

struct BurnDriver BurnSpeclotustc = {
	"spec_lotustc", NULL, "spec_spec128", NULL, "1990",
	"Lotus Esprit Turbo Challenge (128K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpeclotustcRomInfo, SpeclotustcRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Lunar Jetman (48K)

static struct BurnRomInfo SpecljetmanRomDesc[] = {
	{ "Lunar Jetman (1983)(Ultimate Play The Game).z80", 0x08e6a, 0x914bc877, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specljetman, Specljetman, Spectrum)
STD_ROM_FN(Specljetman)

struct BurnDriver BurnSpecljetman = {
	"spec_ljetman", NULL, "spec_spectrum", NULL, "1983",
	"Lunar Jetman (48K)\0", NULL, "Ultimate Play The Game", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecljetmanRomInfo, SpecljetmanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Lunar Rescue (48K)

static struct BurnRomInfo SpeclrescueRomDesc[] = {
	{ "Lunar Rescue (1984)(Lyversoft).z80", 0x046c2, 0x6aac917a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speclrescue, Speclrescue, Spectrum)
STD_ROM_FN(Speclrescue)

struct BurnDriver BurnSpeclrescue = {
	"spec_lrescue", NULL, "spec_spectrum", NULL, "1984",
	"Lunar Rescue (48K)\0", NULL, "Lyversoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpeclrescueRomInfo, SpeclrescueRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mad Mix Game (Spanish) (128K)

static struct BurnRomInfo SpecmadmixgameRomDesc[] = {
	{ "Mad Mix Game (1988)(Topo Soft)(es)(128k).z80", 0x0a9ad, 0xc35eb329, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmadmixgame, Specmadmixgame, Spec128)
STD_ROM_FN(Specmadmixgame)

struct BurnDriver BurnSpecmadmixgame = {
	"spec_madmixgame", NULL, "spec_spec128", NULL, "1988",
	"Mad Mix Game (Spanish) (128K)\0", NULL, "Topo Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE | GBF_ACTION, 0,
	SpectrumGetZipName, SpecmadmixgameRomInfo, SpecmadmixgameRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Manchester United Europe (128K)

static struct BurnRomInfo SpecmanutdeuRomDesc[] = {
	{ "Manchester United Europe (1991)(Krisalis).tap", 79751, 0xc8b8d2aa, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmanutdeu, Specmanutdeu, Spec128)
STD_ROM_FN(Specmanutdeu)

struct BurnDriver BurnSpecmanutdeu = {
	"spec_manutdeu", NULL, "spec_spec128", NULL, "1991",
	"Manchester United Europe (128K)\0", NULL, "Krisalis Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL, 0,
	SpectrumGetZipName, SpecmanutdeuRomInfo, SpecmanutdeuRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Manic Miner (Bug-Byte Software) (48K)

static struct BurnRomInfo SpecmminerRomDesc[] = {
	{ "Manic Miner (1983)(Bug-Byte Software).z80", 0x06834, 0x024b1971, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmminer, Specmminer, Spectrum)
STD_ROM_FN(Specmminer)

struct BurnDriver BurnSpecmminer = {
	"spec_mminer", NULL, "spec_spectrum", NULL, "1983",
	"Manic Miner (Bug-Byte Software) (48K)\0", NULL, "Bug-Byte Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecmminerRomInfo, SpecmminerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Manic Miner (Software Projects) (48K)

static struct BurnRomInfo SpecmminerspRomDesc[] = {
	{ "Manic Miner (1983)(Software Projects).z80", 0x05ff0, 0x2187b9dd, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmminersp, Specmminersp, Spectrum)
STD_ROM_FN(Specmminersp)

struct BurnDriver BurnSpecmminersp = {
	"spec_mminersp", NULL, "spec_spectrum", NULL, "1983",
	"Manic Miner (Software Projects) (48K)\0", NULL, "Software Projects", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecmminerspRomInfo, SpecmminerspRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Marble Madness - Deluxe Edition (48K)

static struct BurnRomInfo SpecmarblemRomDesc[] = {
	{ "Marble Madness - Deluxe Edition (1987)(Melbourne House).z80", 0x0793d, 0x3d08e9ec, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmarblem, Specmarblem, Spectrum)
STD_ROM_FN(Specmarblem)

struct BurnDriver BurnSpecmarblem = {
	"spec_marblem", NULL, "spec_spectrum", NULL, "1987",
	"Marble Madness - Deluxe Edition (48K)\0", NULL, "Melbourne House", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecmarblemRomInfo, SpecmarblemRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mario Bros (48K)

static struct BurnRomInfo SpecmaribrosRomDesc[] = {
	{ "Mario Bros (1987)(Ocean).z80", 0x08f38, 0xe42c245c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmaribros, Specmaribros, Spectrum)
STD_ROM_FN(Specmaribros)

struct BurnDriver BurnSpecmaribros = {
	"spec_maribros", NULL, "spec_spectrum", NULL, "1987",
	"Mario Bros (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecmaribrosRomInfo, SpecmaribrosRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPMDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Match Day (48K)

static struct BurnRomInfo SpecmatchdayRomDesc[] = {
	{ "Match Day (1985)(Ocean).z80", 0x0a809, 0x59d3bc21, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmatchday, Specmatchday, Spectrum)
STD_ROM_FN(Specmatchday)

struct BurnDriver BurnSpecmatchday = {
	"spec_matchday", NULL, "spec_spectrum", NULL, "1985",
	"Match Day (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL, 0,
	SpectrumGetZipName, SpecmatchdayRomInfo, SpecmatchdayRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Match Day II (48K)

static struct BurnRomInfo Specmatchdy2RomDesc[] = {
	{ "Match Day II (1987)(Ocean).z80", 0x09469, 0x910131c2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmatchdy2, Specmatchdy2, Spectrum)
STD_ROM_FN(Specmatchdy2)

struct BurnDriver BurnSpecmatchdy2 = {
	"spec_matchdy2", NULL, "spec_spectrum", NULL, "1987",
	"Match Day II (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL, 0,
	SpectrumGetZipName, Specmatchdy2RomInfo, Specmatchdy2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Match of the Day (128K)

static struct BurnRomInfo SpecmtchotdRomDesc[] = {
	{ "Match of the Day (1992)(Zeppelin Games)[128K].z80", 0x0cd0a, 0xeb11c05c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmtchotd, Specmtchotd, Spec128)
STD_ROM_FN(Specmtchotd)

struct BurnDriver BurnSpecmtchotd = {
	"spec_mtchotd", NULL, "spec_spec128", NULL, "1992",
	"Match of the Day (128K)\0", NULL, "Zeppelin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_STRATEGY, 0,
	SpectrumGetZipName, SpecmtchotdRomInfo, SpecmtchotdRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Merlin (48K)

static struct BurnRomInfo SpecmerlinRomDesc[] = {
	{ "Merlin 48K (1988)(Firebird).z80", 41557, 0xa37c1a27, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmerlin, Specmerlin, Spectrum)
STD_ROM_FN(Specmerlin)

struct BurnDriver BurnSpecmerlin = {
	"spec_merlin", NULL, "spec_spectrum", NULL, "1987",
	"Merlin (48K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecmerlinRomInfo, SpecmerlinRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Metal Army (48K)

static struct BurnRomInfo SpecmetaarmyRomDesc[] = {
	{ "Metal Army (1988)(Players).tap", 40251, 0xdbced42f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmetaarmy, Specmetaarmy, Spectrum)
STD_ROM_FN(Specmetaarmy)

struct BurnDriver BurnSpecmetaarmy = {
	"spec_metaarmy", NULL, "spec_spectrum", NULL, "1988",
	"Metal Army (48K)\0", NULL, "Players Premier", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecmetaarmyRomInfo, SpecmetaarmyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// MicroProse Soccer - Six-a-Side Challenge 128K

static struct BurnRomInfo SpecMicrsocc6sdRomDesc[] = {
	{ "MicroProse Soccer - 6aside Challenge 128K (1989)(MicroProse).tap", 98198, 0x3c0a7349, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMicrsocc6sd, SpecMicrsocc6sd, Spec128)
STD_ROM_FN(SpecMicrsocc6sd)

struct BurnDriver BurnSpecMicrsocc6sd = {
	"spec_micrsocc6sd", "spec_micrsoccint", "spec_spec128", NULL, "1989",
	"MicroProse Soccer - Six-a-Side Challenge (128K)\0", NULL, "MicroProse", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL, 0,
	SpectrumGetZipName, SpecMicrsocc6sdRomInfo, SpecMicrsocc6sdRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// MicroProse Soccer - International Challenge (128K)

static struct BurnRomInfo SpecMicrsoccintRomDesc[] = {
	{ "MicroProse Soccer - Int. Challenge 128K (1989)(MicroProse).tap", 98198, 0x2d03b320, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMicrsoccint, SpecMicrsoccint, Spec128)
STD_ROM_FN(SpecMicrsoccint)

struct BurnDriver BurnSpecMicrsoccint = {
	"spec_micrsoccint", NULL, "spec_spec128", NULL, "1989",
	"MicroProse Soccer - International Challenge (128K)\0", NULL, "MicroProse", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL, 0,
	SpectrumGetZipName, SpecMicrsoccintRomInfo, SpecMicrsoccintRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// MicroProse Soccer (48K)

static struct BurnRomInfo SpecMicrsocc_48RomDesc[] = {
	{ "MicroProse Soccer 48K (1989)(MicroProse).tap", 36576, 0x68a260d1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMicrsocc_48, SpecMicrsocc_48, Spectrum)
STD_ROM_FN(SpecMicrsocc_48)

struct BurnDriver BurnSpecMicrsocc_48 = {
	"spec_micrsocc_48", "spec_micrsoccint", "spec_spectrum", NULL, "1989",
	"MicroProse Soccer (48K)\0", NULL, "MicroProse", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL, 0,
	SpectrumGetZipName, SpecMicrsocc_48RomInfo, SpecMicrsocc_48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Midnight Resistance (48K)

static struct BurnRomInfo Specmresist_48RomDesc[] = {
	{ "Midnight Resistance (1990)(Ocean).z80", 0x0a2e3, 0x70b26b8c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmresist_48, Specmresist_48, Spectrum)
STD_ROM_FN(Specmresist_48)

struct BurnDriver BurnSpecmresist_48 = {
	"spec_mresist_48", "spec_mresist", "spec_spectrum", NULL, "1990",
	"Midnight Resistance (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, Specmresist_48RomInfo, Specmresist_48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Midnight Resistance (128K)

static struct BurnRomInfo SpecmresistRomDesc[] = {
	{ "Midnight Resistance (1990)(Ocean)[128K].z80", 0x1b641, 0x614f8f38, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmresist, Specmresist, Spec128)
STD_ROM_FN(Specmresist)

struct BurnDriver BurnSpecmresist = {
	"spec_mresist", NULL, "spec_spec128", NULL, "1990",
	"Midnight Resistance (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecmresistRomInfo, SpecmresistRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Monty on the Run (128K)

static struct BurnRomInfo SpecmontrunRomDesc[] = {
	{ "Monty on the Run (1985)(Gremlin Graphics)(128k).z80", 0x09f68, 0x330a393a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmontrun, Specmontrun, Spec128)
STD_ROM_FN(Specmontrun)

struct BurnDriver BurnSpecmontrun = {
	"spec_montrun", NULL, "spec_spec128", NULL, "1985",
	"Monty on the Run (128K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecmontrunRomInfo, SpecmontrunRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Moon Cresta - Review (48K)

static struct BurnRomInfo SpecmoocrereRomDesc[] = {
	{ "Moon Cresta - Review (1985)(Incentive Software).z80", 0x0147f, 0x5db616b9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmoocrere, Specmoocrere, Spectrum)
STD_ROM_FN(Specmoocrere)

struct BurnDriver BurnSpecmoocrere = {
	"spec_moocrere", "spec_mooncrst", "spec_spectrum", NULL, "1985",
	"Moon Cresta - Review (48K)\0", NULL, "Incentive Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_MISC, 0,
	SpectrumGetZipName, SpecmoocrereRomInfo, SpecmoocrereRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Moon Cresta (48K)

static struct BurnRomInfo SpecmooncrstRomDesc[] = {
	{ "Moon Cresta (1985)(Incentive Software).z80", 0x0979b, 0xaf817ac8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmooncrst, Specmooncrst, Spectrum)
STD_ROM_FN(Specmooncrst)

struct BurnDriver BurnSpecmooncrst = {
	"spec_mooncrst", NULL, "spec_spectrum", NULL, "1985",
	"Moon Cresta (48K)\0", NULL, "Incentive Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecmooncrstRomInfo, SpecmooncrstRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// MoonTorc (48K)

static struct BurnRomInfo SpecmoontorcRomDesc[] = {
	{ "Moontorc (1991)(Atlantis).tap", 58279, 0x0fe238a1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmoontorc, Specmoontorc, Spectrum)
STD_ROM_FN(Specmoontorc)

struct BurnDriver BurnSpecmoontorc = {
	"spec_moontorc", NULL, "spec_spectrum", NULL, "1991",
	"MoonTorc (48K)\0", NULL, "Atlantis Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecmoontorcRomInfo, SpecmoontorcRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPMDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Moonwalker - The Computer Game (128K)

static struct BurnRomInfo SpecmoonwalkRomDesc[] = {
	{ "Moonwalker - The Computer Game (1989)(U.S. Gold).tap", 170496, 0x0c84679e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmoonwalk, Specmoonwalk, Spec128)
STD_ROM_FN(Specmoonwalk)

struct BurnDriver BurnSpecmoonwalk = {
	"spec_moonwalk", NULL, "spec_spec128", NULL, "1989",
	"Moonwalker - The Computer Game (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecmoonwalkRomInfo, SpecmoonwalkRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mot - Part 1 (Spanish) (48K)

static struct BurnRomInfo Specmotopera1RomDesc[] = {
	{ "Mot (1989)(Opera Soft)(Sp)(48k)[Part 1 of 3].tap", 49084, 0x66889cb5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmotopera1, Specmotopera1, Spectrum)
STD_ROM_FN(Specmotopera1)

struct BurnDriver BurnSpecmotopera1 = {
	"spec_motopera1", NULL, "spec_spectrum", NULL, "1989",
	"Mot - Part 1 (Spanish) (48K)\0", NULL, "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specmotopera1RomInfo, Specmotopera1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mot - Part 2 (Spanish) (48K)

static struct BurnRomInfo Specmotopera2RomDesc[] = {
	{ "Mot (1989)(Opera Soft)(Sp)(48k)[Part 2 of 3].tap", 49084, 0x9b20327d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmotopera2, Specmotopera2, Spectrum)
STD_ROM_FN(Specmotopera2)

struct BurnDriver BurnSpecmotopera2 = {
	"spec_motopera2", "spec_motopera1", "spec_spectrum", NULL, "1989",
	"Mot - Part 2 (Spanish) (48K)\0", NULL, "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specmotopera2RomInfo, Specmotopera2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mot - Part 3 (Spanish) (48K)

static struct BurnRomInfo Specmotopera3RomDesc[] = {
	{ "Mot (1989)(Opera Soft)(Sp)(48k)[Part 3 of 3].tap", 49084, 0x0b659f33, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmotopera3, Specmotopera3, Spectrum)
STD_ROM_FN(Specmotopera3)

struct BurnDriver BurnSpecmotopera3 = {
	"spec_motopera3", "spec_motopera1", "spec_spectrum", NULL, "1989",
	"Mot - Part 3 (Spanish) (48K)\0", NULL, "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specmotopera3RomInfo, Specmotopera3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ms. Pac-Man (48K)

static struct BurnRomInfo SpecmspacmanRomDesc[] = {
	{ "Ms. Pac-Man (1984)(Atarisoft).z80", 0x08e51, 0x168677eb, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmspacman, Specmspacman, Spectrum)
STD_ROM_FN(Specmspacman)

struct BurnDriver BurnSpecmspacman = {
	"spec_mspacman", NULL, "spec_spectrum", NULL, "1984",
	"Ms. Pac-Man (48K)\0", NULL, "Atarisoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_MAZE | GBF_ACTION, 0,
	SpectrumGetZipName, SpecmspacmanRomInfo, SpecmspacmanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mundial de Futbol (Spanish) (128K)

static struct BurnRomInfo SpecmundialfutbolRomDesc[] = {
	{ "Mundial de Futbol ES (1990)(Opera Soft).tap", 98168, 0xe41fa6c3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmundialfutbol, Specmundialfutbol, Spec128)
STD_ROM_FN(Specmundialfutbol)

struct BurnDriver BurnSpecmundialfutbol = {
	"spec_mundialfutbol", NULL, "spec_spec128", NULL, "1990",
	"Mundial de Futbol (Spanish) (128K)\0", NULL, "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL, 0,
	SpectrumGetZipName, SpecmundialfutbolRomInfo, SpecmundialfutbolRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mutan Zone (Spanish) (Part 1 of 2) (128K)

static struct BurnRomInfo SpecmutazoneRomDesc[] = {
	{ "Mutan Zone (1989)(Opera Soft)(es)(Part 1 of 2)(128k).z80", 0x0ae17, 0xc83a4c4d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmutazone, Specmutazone, Spec128)
STD_ROM_FN(Specmutazone)

struct BurnDriver BurnSpecmutazone = {
	"spec_mutazone", NULL, "spec_spec128", NULL, "1989",
	"Mutan Zone (Spanish) (Part 1 of 2) (128K)\0", NULL, "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecmutazoneRomInfo, SpecmutazoneRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mutan Zone (Spanish) (Part 2 of 2) (128K)

static struct BurnRomInfo Specmutazone2RomDesc[] = {
	{ "Mutan Zone (1989)(Opera Soft)(es)(Part 2 of 2)(128k).z80", 0x0adf1, 0x24ee7c22, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmutazone2, Specmutazone2, Spec128)
STD_ROM_FN(Specmutazone2)

struct BurnDriver BurnSpecmutazone2 = {
	"spec_mutazone2", "spec_mutazone", "spec_spec128", NULL, "1989",
	"Mutan Zone (Spanish) (Part 2 of 2) (128K)\0", NULL, "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, Specmutazone2RomInfo, Specmutazone2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Navy Moves - Part 1 (Spanish) (48K)

static struct BurnRomInfo Specnavymoves1RomDesc[] = {
	{ "Navy Moves ES - Part 1 (1988)(Dinamic).tap", 47056, 0xb5090163, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specnavymoves1, Specnavymoves1, Spectrum)
STD_ROM_FN(Specnavymoves1)

struct BurnDriver BurnSpecnavymoves1 = {
	"spec_navymoves1", NULL, "spec_spectrum", NULL, "1988",
	"Navy Moves - Part 1 (Spanish) (48K)\0", "Set '3-Sinclair' to use controller", "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specnavymoves1RomInfo, Specnavymoves1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Navy Moves - Part 2 (Spanish) (48K)

static struct BurnRomInfo Specnavymoves2RomDesc[] = {
	{ "Navy Moves ES - Part 2 (1988)(Dinamic).tap", 46510, 0xc89abc9d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specnavymoves2, Specnavymoves2, Spectrum)
STD_ROM_FN(Specnavymoves2)

struct BurnDriver BurnSpecnavymoves2 = {
	"spec_navymoves2", "spec_navymoves1", "spec_spectrum", NULL, "1988",
	"Navy Moves - Part 2 (Spanish) (48K)\0", "Password: 63723", "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, Specnavymoves2RomInfo, Specnavymoves2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Navy SEALs - Part 1 (128K)

static struct BurnRomInfo SpecnvysealsRomDesc[] = {
	{ "Navy SEALs 128K (1991)(Ocean)(Part1).tap", 136806, 0x01b1a341, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specnvyseals, Specnvyseals, Spec128)
STD_ROM_FN(Specnvyseals)

struct BurnDriver BurnSpecnvyseals = {
	"spec_nvyseals", NULL, "spec_spec128", NULL, "1991",
	"Navy SEALs - Part 1 (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecnvysealsRomInfo, SpecnvysealsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Navy SEALs - Part 2 (128K)

static struct BurnRomInfo Specnvyseals2RomDesc[] = {
	{ "Navy SEALs 128K (1991)(Ocean)(Part2).tap", 136805, 0x6b7f3a21, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specnvyseals2, Specnvyseals2, Spec128)
STD_ROM_FN(Specnvyseals2)

struct BurnDriver BurnSpecnvyseals2 = {
	"spec_nvyseals2", "spec_nvyseals", "spec_spec128", NULL, "1991",
	"Navy SEALs - Part 2 (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, Specnvyseals2RomInfo, Specnvyseals2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Nebulus (48K)

static struct BurnRomInfo SpecnebulusRomDesc[] = {
	{ "Nebulus 48K (1987)(Hewson Consultants).tap", 44437, 0x46353552, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specnebulus, Specnebulus, Spectrum)
STD_ROM_FN(Specnebulus)

struct BurnDriver BurnSpecnebulus = {
	"spec_nebulus", NULL, "spec_spectrum", NULL, "1987",
	"Nebulus (48K)\0", NULL, "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ACTION, 0,
	SpectrumGetZipName, SpecnebulusRomInfo, SpecnebulusRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Nemesis - The Final Challenge (48K)

static struct BurnRomInfo SpecnemesisRomDesc[] = {
	{ "Nemesis (1987)(The Hit Squad)[re-release].z80", 0x0a71d, 0xad42684d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specnemesis, Specnemesis, Spectrum)
STD_ROM_FN(Specnemesis)

struct BurnDriver BurnSpecnemesis = {
	"spec_nemesis", NULL, "spec_spectrum", NULL, "1987",
	"Nemesis - The Final Challenge (48K)\0", NULL, "Konami - The Hit Squad", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecnemesisRomInfo, SpecnemesisRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Olli & Lissa - The Ghost of Shilmoore Castle (48K)

static struct BurnRomInfo SpecollilsaRomDesc[] = {
	{ "Olli & Lissa - The Ghost Of Shilmoore Castle (1986)(Firebird).z80", 0x09f58, 0xed38d8ad, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specollilsa, Specollilsa, Spectrum)
STD_ROM_FN(Specollilsa)

struct BurnDriver BurnSpecollilsa = {
	"spec_ollilsa", NULL, "spec_spectrum", NULL, "1986",
	"Olli & Lissa - The Ghost of Shilmoore Castle (48K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecollilsaRomInfo, SpecollilsaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Olli & Lissa II - Halloween (48K)

static struct BurnRomInfo Specollilsa2RomDesc[] = {
	{ "Olli & Lissa II - Halloween (1987)(Firebird).z80", 0x0a6b1, 0x99e59116, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specollilsa2, Specollilsa2, Spectrum)
STD_ROM_FN(Specollilsa2)

struct BurnDriver BurnSpecollilsa2 = {
	"spec_ollilsa2", NULL, "spec_spectrum", NULL, "1987",
	"Olli & Lissa II - Halloween (48K)\0", NULL, "Silverbird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Specollilsa2RomInfo, Specollilsa2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Olli & Lissa III - The Candlelight Adventure (48K)

static struct BurnRomInfo Specollilsa3RomDesc[] = {
	{ "Olli & Lissa 3 (1989)(Cartoon Time).tap", 44893, 0xc5703df4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specollilsa3, Specollilsa3, Spectrum)
STD_ROM_FN(Specollilsa3)

struct BurnDriver BurnSpecollilsa3 = {
	"spec_ollilsa3", NULL, "spec_spectrum", NULL, "1989",
	"Olli & Lissa III - The Candlelight Adventure (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Specollilsa3RomInfo, Specollilsa3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Operation Thunderbolt (128K)

static struct BurnRomInfo SpecothunderRomDesc[] = {
	{ "Operation Thunderbolt 128K (1989)(Ocean).tap", 138426, 0x16257514, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specothunder, Specothunder, Spec128)
STD_ROM_FN(Specothunder)

struct BurnDriver BurnSpecothunder = {
	"spec_othunder", NULL, "spec_spec128", NULL, "1989",
	"Operation Thunderbolt (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecothunderRomInfo, SpecothunderRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Operation Wolf (128K)

static struct BurnRomInfo SpecopwolfRomDesc[] = {
	{ "Operation Wolf (1988)(Ocean)[128K].z80", 0x1da84, 0xa7cd683f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specopwolf, Specopwolf, Spec128)
STD_ROM_FN(Specopwolf)

struct BurnDriver BurnSpecopwolf = {
	"spec_opwolf", NULL, "spec_spec128", NULL, "1988",
	"Operation Wolf (128K)\0", "Set '2. Sinclair 1' to use controller - Press SPACE for grenades", "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecopwolfRomInfo, SpecopwolfRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Pac-Land (128K)

static struct BurnRomInfo SpecpaclandRomDesc[] = {
	{ "Pac-Land (1989)(Grandslam Entertainments)[128K].z80", 0x0b9d7, 0x78fe5d59, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specpacland, Specpacland, Spec128)
STD_ROM_FN(Specpacland)

struct BurnDriver BurnSpecpacland = {
	"spec_pacland", NULL, "spec_spec128", NULL, "1989",
	"Pac-Land (128K)\0", "ENTER to add coins, 1 or 2 to play", "Grandslam Entertainments", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecpaclandRomInfo, SpecpaclandRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Pac-Man (Atarisoft) (48K)

static struct BurnRomInfo SpecpacmanRomDesc[] = {
	{ "Pac-Man (1983)(Atarisoft).z80", 0x035fc, 0xe3c56f6b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specpacman, Specpacman, Spectrum)
STD_ROM_FN(Specpacman)

struct BurnDriver BurnSpecpacman = {
	"spec_pacman", NULL, "spec_spectrum", NULL, "1983",
	"Pac-Man (Atarisoft) (48K)\0", NULL, "Atarisoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE | GBF_ACTION, 0,
	SpectrumGetZipName, SpecpacmanRomInfo, SpecpacmanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Pac-Mania (128K)

static struct BurnRomInfo SpecpacmaniaRomDesc[] = {
	{ "Pac-Mania (1988)(Grandslam Entertainments)[128K].z80", 0x0a813, 0x938c60ff, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specpacmania, Specpacmania, Spec128)
STD_ROM_FN(Specpacmania)

struct BurnDriver BurnSpecpacmania = {
	"spec_pacmania", NULL, "spec_spec128", NULL, "1988",
	"Pac-Mania (128K)\0", NULL, "Grandslam Entertainments", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE | GBF_ACTION, 0,
	SpectrumGetZipName, SpecpacmaniaRomInfo, SpecpacmaniaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Pang (128K)

static struct BurnRomInfo SpecpangRomDesc[] = {
	{ "Pang 128K (1990)(Ocean).tap", 100344, 0x06ce4e01, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specpang, Specpang, Spec128)
STD_ROM_FN(Specpang)

struct BurnDriver BurnSpecpang = {
	"spec_pang", NULL, "spec_spec128", NULL, "1990",
	"Pang (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecpangRomInfo, SpecpangRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Panic Dizzy (128K)

static struct BurnRomInfo SpecpandizzyRomDesc[] = {
	{ "Panic Dizzy (1991)(Codemasters).tap", 48017, 0xdbc18295, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specpandizzy, Specpandizzy, Spec128)
STD_ROM_FN(Specpandizzy)

struct BurnDriver BurnSpecpandizzy = {
	"spec_pandizzy", NULL, "spec_spec128", NULL, "1991",
	"Panic Dizzy (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecpandizzyRomInfo, SpecpandizzyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Paperboy (48K)

static struct BurnRomInfo SpecpaperboyRomDesc[] = {
	{ "Paperboy (1986)(Elite Systems).tap", 51430, 0xac1c032c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specpaperboy, Specpaperboy, Spectrum)
STD_ROM_FN(Specpaperboy)

struct BurnDriver BurnSpecpaperboy = {
	"spec_paperboy", NULL, "spec_spectrum", NULL, "1986",
	"Paperboy (48K)\0", NULL, "Elite Systems", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecpaperboyRomInfo, SpecpaperboyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Paperboy 2 (48K)

static struct BurnRomInfo Specpaperby2RomDesc[] = {
	{ "Paperboy 2 (1992)(Mindscape International).z80", 39870, 0x523b2b3b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specpaperby2, Specpaperby2, Spectrum)
STD_ROM_FN(Specpaperby2)

struct BurnDriver BurnSpecpaperby2 = {
	"spec_paperby2", NULL, "spec_spectrum", NULL, "1992",
	"Paperboy 2 (48K)\0", NULL, "Mindscape International", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specpaperby2RomInfo, Specpaperby2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Passing Shot (128K)

static struct BurnRomInfo SpecpassshotRomDesc[] = {
	{ "Passing Shot (1989)(Image Works).tap", 40394, 0x574083d5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specpassshot, Specpassshot, Spec128)
STD_ROM_FN(Specpassshot)

struct BurnDriver BurnSpecpassshot = {
	"spec_passshot", NULL, "spec_spec128", NULL, "1989",
	"Passing Shot (128K)\0", "Press '1' to music and '2' to sound effects", "Image Works", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecpassshotRomInfo, SpecpassshotRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Peter Shilton's Handball Maradona (48K)

static struct BurnRomInfo SpecpshandmRomDesc[] = {
	{ "Peter Shilton's Handball Maradona (1986)(Grandslam Entertainments).z80", 40447, 0x5d4a8e4d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specpshandm, Specpshandm, Spectrum)
STD_ROM_FN(Specpshandm)

struct BurnDriver BurnSpecpshandm = {
	"spec_pshandm", NULL, "spec_spectrum", NULL, "1986",
	"Peter Shilton's Handball Maradona (48K)\0", NULL, "Grandslam Entertainments", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL, 0,
	SpectrumGetZipName, SpecpshandmRomInfo, SpecpshandmRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Game Over II - Part 1 (aka Phantis) (48K)

static struct BurnRomInfo Specgameover21RomDesc[] = {
	{ "Game Over 2 Part 1 (1988)(Dinamic)[aka Phantis].tap", 48141, 0x8420b778, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specgameover21, Specgameover21, Spectrum)
STD_ROM_FN(Specgameover21)

struct BurnDriver BurnSpecgameover21 = {
	"spec_gameover21", NULL, "spec_spectrum", NULL, "1988",
	"Game Over II - Part 1 (aka Phantis) (48K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, Specgameover21RomInfo, Specgameover21RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Game Over II - Part 2 (aka Phantis) (48K)

static struct BurnRomInfo Specgameover22RomDesc[] = {
	{ "Game Over 2 Part 2 (1988)(Dinamic)[aka Phantis].tap", 48155, 0x223202b2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specgameover22, Specgameover22, Spectrum)
STD_ROM_FN(Specgameover22)

struct BurnDriver BurnSpecgameover22 = {
	"spec_gameover22", "spec_gameover21", "spec_spectrum", NULL, "1988",
	"Game Over II - Part 2 (aka Phantis) (48K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specgameover22RomInfo, Specgameover22RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Phantis - Part 1 (Spanish) (48K)

static struct BurnRomInfo Specphantis1RomDesc[] = {
	{ "Phantis - Part 1 ES (1987)(Dinamic).tap", 48529, 0xca456c10, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specphantis1, Specphantis1, Spectrum)
STD_ROM_FN(Specphantis1)

struct BurnDriver BurnSpecphantis1 = {
	"spec_phantis1", NULL, "spec_spectrum", NULL, "1987",
	"Phantis - Part 1 (Spanish) (48K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, Specphantis1RomInfo, Specphantis1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Phantis - Part 2 (Spanish) (48K)

static struct BurnRomInfo Specphantis2RomDesc[] = {
	{ "Phantis - Part 2 ES (1987)(Dinamic).tap", 48529, 0xb9045126, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specphantis2, Specphantis2, Spectrum)
STD_ROM_FN(Specphantis2)

struct BurnDriver BurnSpecphantis2 = {
	"spec_phantis2", "spec_phantis1", "spec_spectrum", NULL, "1987",
	"Phantis - Part 2 (Spanish) (48K)\0", "Password: 18757", "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specphantis2RomInfo, Specphantis2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Phoenix - Text Adventure (48K)

static struct BurnRomInfo SpecphoenixRomDesc[] = {
	{ "Phoenix (1991)(Zenobi).tap", 37722, 0x8cf4e0e2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specphoenix, Specphoenix, Spectrum)
STD_ROM_FN(Specphoenix)

struct BurnDriver BurnSpecphoenix = {
	"spec_phoenix", NULL, "spec_spectrum", NULL, "1991",
	"Phoenix - Text Adventure (48K)\0", NULL, "Zenobi Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecphoenixRomInfo, SpecphoenixRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Pippo (48K)

static struct BurnRomInfo SpecpippoRomDesc[] = {
	{ "Pippo 48K (1986)(Mastertronic).tap", 42906, 0xbf464d08, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specpippo, Specpippo, Spectrum)
STD_ROM_FN(Specpippo)

struct BurnDriver BurnSpecpippo = {
	"spec_pippo", NULL, "spec_spectrum", NULL, "1986",
	"Pippo (48K)\0", NULL, "Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecpippoRomInfo, SpecpippoRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Pixy the Microdot 2 (48K)

static struct BurnRomInfo Specpixymicrod2RomDesc[] = {
	{ "Pixy the Microdot 2 (1992)(Your Sinclair).z80", 40323, 0x9de92490, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specpixymicrod2, Specpixymicrod2, Spectrum)
STD_ROM_FN(Specpixymicrod2)

struct BurnDriver BurnSpecpixymicrod2 = {
	"spec_pixymicrod2", NULL, "spec_spectrum", NULL, "1992",
	"Pixy the Microdot 2 (48K)\0", NULL, "Your Sinclair", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specpixymicrod2RomInfo, Specpixymicrod2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Platoon (128K)

static struct BurnRomInfo SpecplatoonRomDesc[] = {
	{ "Platoon (1988)(Ocean)[128K].z80", 106785, 0xaa4d4d13, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specplatoon, Specplatoon, Spec128)
STD_ROM_FN(Specplatoon)

struct BurnDriver BurnSpecplatoon = {
	"spec_platoon", NULL, "spec_spec128", NULL, "1988",
	"Platoon (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecplatoonRomInfo, SpecplatoonRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Plotting (128K)

static struct BurnRomInfo SpecplottingRomDesc[] = {
	{ "Plotting (1990)(Ocean)[128K].z80", 40883, 0xdd65a0b5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specplotting, Specplotting, Spec128)
STD_ROM_FN(Specplotting)

struct BurnDriver BurnSpecplotting = {
	"spec_plotting", NULL, "spec_spec128", NULL, "1990",
	"Plotting (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecplottingRomInfo, SpecplottingRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Pole Position (Atarisoft) (48K)

static struct BurnRomInfo SpecpoleposaRomDesc[] = {
	{ "Pole Position (1984)(Atarisoft).tap", 47760, 0xfc80c755, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specpoleposa, Specpoleposa, Spectrum)
STD_ROM_FN(Specpoleposa)

struct BurnDriver BurnSpecpoleposa = {
	"spec_poleposa", NULL, "spec_spectrum", NULL, "1984",
	"Pole Position (48K)\0", "Up or down to shift gear, fire button to brake", "Atarisoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecpoleposaRomInfo, SpecpoleposaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Poli Diaz (Spanish) (128K)

static struct BurnRomInfo SpecpolidiazRomDesc[] = {
	{ "Poli Diaz (1990)(Opera Soft)(es)(128k).z80", 0x0b232, 0x635c6283, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specpolidiaz, Specpolidiaz, Spec128)
STD_ROM_FN(Specpolidiaz)

struct BurnDriver BurnSpecpolidiaz = {
	"spec_polidiaz", NULL, "spec_spec128", NULL, "1990",
	"Poli Diaz (Spanish) (128K)\0", NULL, "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpecpolidiazRomInfo, SpecpolidiazRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Power Drift (128K)

static struct BurnRomInfo SpecpdriftRomDesc[] = {
	{ "Power Drift (1989)(Activision)[128K].z80", 0x160f6, 0x34a7f74a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specpdrift, Specpdrift, Spec128)
STD_ROM_FN(Specpdrift)

struct BurnDriver BurnSpecpdrift = {
	"spec_pdrift", NULL, "spec_spec128", NULL, "1989",
	"Power Drift (128K)\0", NULL, "Activision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecpdriftRomInfo, SpecpdriftRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Puzznic (128K)

static struct BurnRomInfo SpecpuzznicRomDesc[] = {
	{ "Puzznic (1990)(Ocean).tap", 33679, 0xeceabdd0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specpuzznic, Specpuzznic, Spec128)
STD_ROM_FN(Specpuzznic)

struct BurnDriver BurnSpecpuzznic = {
	"spec_puzznic", NULL, "spec_spec128", NULL, "1990",
	"Puzznic (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecpuzznicRomInfo, SpecpuzznicRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Quartet (48K)

static struct BurnRomInfo SpecquartetRomDesc[] = {
	{ "Quartet (1987)(Activision).z80", 0x08a9e, 0x45711e73, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specquartet, Specquartet, Spectrum)
STD_ROM_FN(Specquartet)

struct BurnDriver BurnSpecquartet = {
	"spec_quartet", NULL, "spec_spectrum", NULL, "1987",
	"Quartet (48K)\0", "To start a game session press 'S' and after ENTER", "Activision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 4, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecquartetRomInfo, SpecquartetRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Quazatron (48K)

static struct BurnRomInfo SpecquazatroRomDesc[] = {
	{ "Quazatron (1986)(Hewson Consultants).z80", 0x07e39, 0xdf931658, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specquazatro, Specquazatro, Spectrum)
STD_ROM_FN(Specquazatro)

struct BurnDriver BurnSpecquazatro = {
	"spec_quazatro", NULL, "spec_spectrum", NULL, "1986",
	"Quazatron (48K)\0", NULL, "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecquazatroRomInfo, SpecquazatroRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rainbow Islands - The Story of Bubble Bobble 2 (48K)

static struct BurnRomInfo Specrbisland_48RomDesc[] = {
	{ "Rainbow Islands - The Story of Bubble Bobble 2 (1990)(Ocean).z80", 0x07da0, 0xad2c841f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrbisland_48, Specrbisland_48, Spectrum)
STD_ROM_FN(Specrbisland_48)

struct BurnDriver BurnSpecrbisland_48 = {
	"spec_rbisland_48", "spec_rbisland", "spec_spectrum", NULL, "1990",
	"Rainbow Islands - The Story of Bubble Bobble 2 (48K)\0", "Press ENTER to add 'coins'", "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Specrbisland_48RomInfo, Specrbisland_48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rainbow Islands - The Story of Bubble Bobble 2 (128K)

static struct BurnRomInfo SpecrbislandRomDesc[] = {
	{ "Rainbow Islands - The Story of Bubble Bobble 2 (1990)(Ocean)[128K].z80", 0x161bb, 0x0211cd1d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrbisland, Specrbisland, Spec128)
STD_ROM_FN(Specrbisland)

struct BurnDriver BurnSpecrbisland = {
	"spec_rbisland", NULL, "spec_spec128", NULL, "1990",
	"Rainbow Islands - The Story of Bubble Bobble 2 (128K)\0", "Press ENTER to add 'coins'", "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecrbislandRomInfo, SpecrbislandRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// R.A.M. (Spanish) (48K)

static struct BurnRomInfo SpecramtopoRomDesc[] = {
	{ "R.A.M. ES (1990)(Topo Soft).tap", 62593, 0xeaaf3bd1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specramtopo, Specramtopo, Spectrum)
STD_ROM_FN(Specramtopo)

struct BurnDriver BurnSpecramtopo = {
	"spec_ramtopo", NULL, "spec_spectrum", NULL, "1990",
	"R.A.M. (Spanish) (48K)\0", NULL, "Topo Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecramtopoRomInfo, SpecramtopoRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rambo - First Blood Part II (48K)

static struct BurnRomInfo SpecramboRomDesc[] = {
	{ "Rambo - First Blood Part II (1985)(Ocean).tap", 49392, 0x769660f9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrambo, Specrambo, Spectrum)
STD_ROM_FN(Specrambo)

struct BurnDriver BurnSpecrambo = {
	"spec_rambo", NULL, "spec_spectrum", NULL, "1985",
	"Rambo - First Blood Part II (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecramboRomInfo, SpecramboRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rambo III (128K)

static struct BurnRomInfo Specrambo3RomDesc[] = {
	{ "Rambo III 128K (1988)(Ocean).tap", 95128, 0x63e49eee, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrambo3, Specrambo3, Spec128)
STD_ROM_FN(Specrambo3)

struct BurnDriver BurnSpecrambo3 = {
	"spec_rambo3", NULL, "spec_spec128", NULL, "1988",
	"Rambo III (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_ADV, 0,
	SpectrumGetZipName, Specrambo3RomInfo, Specrambo3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rampage (48K)

static struct BurnRomInfo SpecrampageRomDesc[] = {
	{ "Rampage (1988)(Activision).z80", 0x094eb, 0x3735beaf, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrampage, Specrampage, Spectrum)
STD_ROM_FN(Specrampage)

struct BurnDriver BurnSpecrampage = {
	"spec_rampage", NULL, "spec_spectrum", NULL, "1988",
	"Rampage (48K)\0", NULL, "Activision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 3, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecrampageRomInfo, SpecrampageRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rastan Saga (128K)

static struct BurnRomInfo SpecrastanRomDesc[] = {
	{ "Rastan Saga 128K (1988)(Imagine).tap", 190502, 0xf8c66b9c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrastan, Specrastan, Spec128)
STD_ROM_FN(Specrastan)

struct BurnDriver BurnSpecrastan = {
	"spec_rastan", NULL, "spec_spec128", NULL, "1988",
	"Rastan Saga (128K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecrastanRomInfo, SpecrastanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Renegade (48K)

static struct BurnRomInfo Specrenegade48RomDesc[] = {
	{ "Renegade 48K (1987)(Imagine).tap", 48977, 0x0cd81e69, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrenegade48, Specrenegade48, Spectrum)
STD_ROM_FN(Specrenegade48)

struct BurnDriver BurnSpecrenegade48 = {
	"spec_renegade48", "spec_renegade128", "spec_spectrum", NULL, "1987",
	"Renegade (48K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, Specrenegade48RomInfo, Specrenegade48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Renegade (128K)

static struct BurnRomInfo Specrenegade128RomDesc[] = {
	{ "Renegade 128K (1987)(Imagine).z80", 93965, 0xcd930d9a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrenegade128, Specrenegade128, Spec128)
STD_ROM_FN(Specrenegade128)

struct BurnDriver BurnSpecrenegade128 = {
	"spec_renegade128", NULL, "spec_spec128", NULL, "1987",
	"Renegade (128K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, Specrenegade128RomInfo, Specrenegade128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Renegade II - Target Renegade (128K)

static struct BurnRomInfo Specrenegad2RomDesc[] = {
	{ "Renegade II - Target Renegade (1988)(Imagine Software)[128K].z80", 108880, 0x25d57e2c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrenegad2, Specrenegad2, Spec128)
STD_ROM_FN(Specrenegad2)

struct BurnDriver BurnSpecrenegad2 = {
	"spec_renegad2", NULL, "spec_spec128", NULL, "1988",
	"Renegade II - Target Renegade (128K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, Specrenegad2RomInfo, Specrenegad2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Renegade III - The Final Chapter (128K)

static struct BurnRomInfo Specrenegad3RomDesc[] = {
	{ "Renegade III - The Final Chapter 128K (1989)(Imagine).tap", 79244, 0xd0c2761b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrenegad3, Specrenegad3, Spec128)
STD_ROM_FN(Specrenegad3)

struct BurnDriver BurnSpecrenegad3 = {
	"spec_renegad3", NULL, "spec_spec128", NULL, "1989",
	"Renegade III - The Final Chapter (128K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, Specrenegad3RomInfo, Specrenegad3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rex (aka Zenith) - Part 1 (128K)

static struct BurnRomInfo SpecrexRomDesc[] = {
	{ "Rex (aka Zenith) - Part 1 (1988)(Martech Games).tap", 48077, 0x7284eb38, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrex, Specrex, Spec128)
STD_ROM_FN(Specrex)

struct BurnDriver BurnSpecrex = {
	"spec_rex", NULL, "spec_spec128", NULL, "1988",
	"Rex (aka Zenith) - Part 1 (128K)\0", NULL, "Martech Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecrexRomInfo, SpecrexRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rex (aka Zenith) - Part 2 (128K)

static struct BurnRomInfo Specrex2RomDesc[] = {
	{ "Rex (aka Zenith) - Part 2 (1988)(Martech Games).tap", 47681, 0x7a54760c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrex2, Specrex2, Spec128)
STD_ROM_FN(Specrex2)

struct BurnDriver BurnSpecrex2 = {
	"spec_rex2", "spec_rex", "spec_spec128", NULL, "1988",
	"Rex (aka Zenith) - Part 2 (128K)\0", "Password: 8881808384829507", "Martech Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, Specrex2RomInfo, Specrex2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rick Dangerous (128K)

static struct BurnRomInfo SpecrickdangRomDesc[] = {
	{ "Rick Dangerous 128K (1989)(Firebird).tap", 48387, 0x48b1600d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrickdang, Specrickdang, Spec128)
STD_ROM_FN(Specrickdang)

struct BurnDriver BurnSpecrickdang = {
	"spec_rickdang", NULL, "spec_spec128", NULL, "1989",
	"Rick Dangerous (128K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecrickdangRomInfo, SpecrickdangRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rick Dangerous 2 (128K)

static struct BurnRomInfo Specrickdang2RomDesc[] = {
	{ "Rick Dangerous 2 (1990)(Micro Style).tap", 0x26599, 0x5cf79480, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrickdang2, Specrickdang2, Spec128)
STD_ROM_FN(Specrickdang2)

struct BurnDriver BurnSpecrickdang2 = {
	"spec_rickdang2", NULL, "spec_spec128", NULL, "1990",
	"Rick Dangerous 2 (128K)\0", NULL, "Micro Style", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Specrickdang2RomInfo, Specrickdang2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Robocop (128K)

static struct BurnRomInfo SpecrobocopRomDesc[] = {
	{ "Robocop (1988)(Ocean)[128K].z80", 0x1cbf8, 0xdcc4bf16, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrobocop, Specrobocop, Spec128)
STD_ROM_FN(Specrobocop)

struct BurnDriver BurnSpecrobocop = {
	"spec_robocop", NULL, "spec_spec128", NULL, "1988",
	"Robocop (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecrobocopRomInfo, SpecrobocopRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Robocop 2 (128K)

static struct BurnRomInfo Specrobocop2RomDesc[] = {
	{ "Robocop 2 (1990)(Ocean)[128K].z80", 0x1c73e, 0xe9b44bc7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrobocop2, Specrobocop2, Spec128)
STD_ROM_FN(Specrobocop2)

struct BurnDriver BurnSpecrobocop2 = {
	"spec_robocop2", NULL, "spec_spec128", NULL, "1990",
	"Robocop 2 (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, Specrobocop2RomInfo, Specrobocop2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Robocop 3 (128K)

static struct BurnRomInfo Specrobocop3RomDesc[] = {
	{ "Robocop 3 (1992)(Ocean)[128K].z80", 0x1ac8a, 0x21b5c6b7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrobocop3, Specrobocop3, Spec128)
STD_ROM_FN(Specrobocop3)

struct BurnDriver BurnSpecrobocop3 = {
	"spec_robocop3", NULL, "spec_spec128", NULL, "1992",
	"Robocop 3 (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, Specrobocop3RomInfo, Specrobocop3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rock 'n Roller (Spanish) (48K)

static struct BurnRomInfo SpecrocknrollerRomDesc[] = {
	{ "Rock'n Roller ES (1988)(Topo Soft).z80", 43965, 0x16da9309, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrocknroller, Specrocknroller, Spectrum)
STD_ROM_FN(Specrocknroller)

struct BurnDriver BurnSpecrocknroller = {
	"spec_rocknroller", NULL, "spec_spectrum", NULL, "1988",
	"Rock 'n Roller (Spanish) (48K)\0", NULL, "Topo Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecrocknrollerRomInfo, SpecrocknrollerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rock Star Ate my Hamster (128K)

static struct BurnRomInfo SpecrockshamRomDesc[] = {
	{ "Rock Star Ate My Hamster! 128K (1989)(Codemasters).tap", 67285, 0xba0efbf5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrocksham, Specrocksham, Spec128)
STD_ROM_FN(Specrocksham)

struct BurnDriver BurnSpecrocksham = {
	"spec_rocksham", NULL, "spec_spec128", NULL, "1989",
	"Rock Star Ate my Hamster (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_STRATEGY, 0,
	SpectrumGetZipName, SpecrockshamRomInfo, SpecrockshamRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rolling Thunder (48K)

static struct BurnRomInfo SpecrollthunRomDesc[] = {
	{ "Rolling Thunder 48K (1988)(U.S. Gold).tap", 42913, 0x796b68c6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrollthun, Specrollthun, Spectrum)
STD_ROM_FN(Specrollthun)

struct BurnDriver BurnSpecrollthun = {
	"spec_rollthun", NULL, "spec_spectrum", NULL, "1988",
	"Rolling Thunder (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecrollthunRomInfo, SpecrollthunRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// R-Type (128K)

static struct BurnRomInfo SpecrtypeRomDesc[] = {
	{ "R-Type (1988)(Electric Dreams Software).tap", 0x206d7, 0xa03330a0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrtype, Specrtype, Spec128)
STD_ROM_FN(Specrtype)

struct BurnDriver BurnSpecrtype = {
	"spec_rtype", NULL, "spec_spec128", NULL, "1988",
	"R-Type (128K)\0", NULL, "Electric Dreams", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecrtypeRomInfo, SpecrtypeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ruff and Reddy in the Space Adventure (128K)

static struct BurnRomInfo SpecRuffreddyRomDesc[] = {
	{ "Ruff and Reddy in the Space Adventure (1990)(Hi-Tec).tap", 54708, 0xffe8fec1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecRuffreddy, SpecRuffreddy, Spec128)
STD_ROM_FN(SpecRuffreddy)

struct BurnDriver BurnSpecRuffreddy = {
	"spec_ruffreddy", NULL, "spec_spec128", NULL, "1990",
	"Ruff and Reddy in the Space Adventure (128K)\0", NULL, "Hi-Tec Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecRuffreddyRomInfo, SpecRuffreddyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Run the Gauntlet (128K)

static struct BurnRomInfo SpecrungauntRomDesc[] = {
	{ "Run the Gauntlet (1989)(Ocean).z80", 0x1a162, 0xe2e3b97c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrungaunt, Specrungaunt, Spec128)
STD_ROM_FN(Specrungaunt)

struct BurnDriver BurnSpecrungaunt = {
	"spec_rungaunt", NULL, "spec_spec128", NULL, "1989",
	"Run the Gauntlet (128K)\0", "During the game press 'H' (Pause) then '2' to set controller", "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecrungauntRomInfo, SpecrungauntRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rygar (48K)

static struct BurnRomInfo SpecrygarRomDesc[] = {
	{ "Rygar - Legendary Warrior (1987)(U.S. Gold).tap", 44813, 0xc64bbc03, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrygar, Specrygar, Spectrum)
STD_ROM_FN(Specrygar)

struct BurnDriver BurnSpecrygar = {
	"spec_rygar", NULL, "spec_spectrum", NULL, "1987",
	"Rygar - Legendary Warrior (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecrygarRomInfo, SpecrygarRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Satan - Part 1 (48K)

static struct BurnRomInfo SpecsatanRomDesc[] = {
	{ "Satan Part 1 (1990)(Dinamic).tap", 48395, 0x2098b208, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsatan, Specsatan, Spectrum)
STD_ROM_FN(Specsatan)

struct BurnDriver BurnSpecsatan = {
	"spec_satan", NULL, "spec_spectrum", NULL, "1990",
	"Satan - Part 1 (48K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecsatanRomInfo, SpecsatanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Satan - Part 2 (48K)

static struct BurnRomInfo Specsatan2RomDesc[] = {
	{ "Satan Part 2 (1990)(Dinamic).tap", 47945, 0xb6586a99, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsatan2, Specsatan2, Spectrum)
STD_ROM_FN(Specsatan2)

struct BurnDriver BurnSpecsatan2 = {
	"spec_satan2", "spec_satan", "spec_spectrum", NULL, "1990",
	"Satan - Part 2 (48K)\0", "Password: 01020304", "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Specsatan2RomInfo, Specsatan2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// S.D.I. - Strategic Defence Initiative (48K)

static struct BurnRomInfo SpecsdisdiRomDesc[] = {
	{ "S.D.I. -  Strategic Defense Initiative 48K (1988)(Activision).tap", 48053, 0xaedbad94, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsdisdi, Specsdisdi, Spectrum)
STD_ROM_FN(Specsdisdi)

struct BurnDriver BurnSpecsdisdi = {
	"spec_sdisdi", NULL, "spec_spectrum", NULL, "1988",
	"S.D.I. - Strategic Defence Initiative (48K)\0", NULL, "Activision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecsdisdiRomInfo, SpecsdisdiRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Saboteur (Durell Software) (48K)

static struct BurnRomInfo Specsabot_48RomDesc[] = {
	{ "Saboteur 48K (1985)(Durell).z80", 39664, 0xa7a89067, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsabot_48, Specsabot_48, Spectrum)
STD_ROM_FN(Specsabot_48)

struct BurnDriver BurnSpecsabot_48 = {
	"spec_sabot_48", NULL, "spec_spectrum", NULL, "1985",
	"Saboteur! (48K)\0", NULL, "Durell Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specsabot_48RomInfo, Specsabot_48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Saboteur II - Avenging Angel (Trainer) (48K)

static struct BurnRomInfo Specsabotur2_48RomDesc[] = {
	{ "Saboteur II - Avenging Angel (1987)(Durell).z80", 0x0a996, 0x4f0d8f73, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsabotur2_48, Specsabotur2_48, Spectrum)
STD_ROM_FN(Specsabotur2_48)

struct BurnDriver BurnSpecsabotur2_48 = {
	"spec_sabotur2_48", "spec_sabotur2", "spec_spectrum", NULL, "1987",
	"Saboteur II - Avenging Angel (Trainer) (48K)\0", NULL, "Durell Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specsabotur2_48RomInfo, Specsabotur2_48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Saboteur II - Avenging Angel (48K)

static struct BurnRomInfo Specsabotur2std_48RomDesc[] = {
	{ "Saboteur II - Avenging Angel (1987)(Durell)(Standard).z80", 0x09ab5, 0x8904b9ba, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsabotur2std_48, Specsabotur2std_48, Spectrum)
STD_ROM_FN(Specsabotur2std_48)

struct BurnDriver BurnSpecsabotur2std_48 = {
	"spec_sabotur2std_48", "spec_sabotur2", "spec_spectrum", NULL, "1987",
	"Saboteur II - Avenging Angel (48K)\0", NULL, "Durell Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specsabotur2std_48RomInfo, Specsabotur2std_48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Saboteur II - Avenging Angel (128K)

static struct BurnRomInfo Specsabotur2RomDesc[] = {
	{ "Saboteur II - Avenging Angel (1987)(Durell)(128k).z80", 0x0b790, 0x7aad77db, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsabotur2, Specsabotur2, Spec128)
STD_ROM_FN(Specsabotur2)

struct BurnDriver BurnSpecsabotur2 = {
	"spec_sabotur2", NULL, "spec_spec128", NULL, "1987",
	"Saboteur II - Avenging Angel (128K)\0", NULL, "Durell Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specsabotur2RomInfo, Specsabotur2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Salamander (48K)

static struct BurnRomInfo SpecsalamandRomDesc[] = {
	{ "Salamander (1987)(Imagine Software).z80", 0x0a680, 0x5ae35d91, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsalamand, Specsalamand, Spectrum)
STD_ROM_FN(Specsalamand)

struct BurnDriver BurnSpecsalamand = {
	"spec_salamand", NULL, "spec_spectrum", NULL, "1987",
	"Salamander (48K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_HORSHOOT | GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecsalamandRomInfo, SpecsalamandRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Savage - Intro (48K)

static struct BurnRomInfo SpecsavageiRomDesc[] = {
	{ "Savage, The (1988)(Firebird)[intro].z80", 31023, 0xe606ebf0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsavagei, Specsavagei, Spectrum)
STD_ROM_FN(Specsavagei)

struct BurnDriver BurnSpecsavagei = {
	"spec_savagei", NULL, "spec_spectrum", NULL, "1988",
	"Savage - Intro (48K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MISC, 0,
	SpectrumGetZipName, SpecsavageiRomInfo, SpecsavageiRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Savage - Part 1 (48K)

static struct BurnRomInfo Specsavage1RomDesc[] = {
	{ "Savage, The (1988)(Firebird)[Part 1 of 3].z80", 37764, 0x3db88855, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsavage1, Specsavage1, Spectrum)
STD_ROM_FN(Specsavage1)

struct BurnDriver BurnSpecsavage1 = {
	"spec_savage1", "spec_savagei", "spec_spectrum", NULL, "1988",
	"Savage - Part 1 (48K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, Specsavage1RomInfo, Specsavage1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Savage - Part 2 (48K)

static struct BurnRomInfo Specsavage2RomDesc[] = {
	{ "Savage, The (1988)(Firebird)[Part 2 of 3].z80", 0x09a20, 0x4f8ddec1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsavage2, Specsavage2, Spectrum)
STD_ROM_FN(Specsavage2)

struct BurnDriver BurnSpecsavage2 = {
	"spec_savage2", "spec_savagei", "spec_spectrum", NULL, "1988",
	"Savage - Part 2 (48K)\0", "Level code: SABATTA", "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, Specsavage2RomInfo, Specsavage2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Savage - Part 3  (48K)

static struct BurnRomInfo Specsavage3RomDesc[] = {
	{ "Savage, The (1988)(Firebird)[Part 3 of 3].z80", 0x08475, 0xe994f627, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsavage3, Specsavage3, Spectrum)
STD_ROM_FN(Specsavage3)

struct BurnDriver BurnSpecsavage3 = {
	"spec_savage3", "spec_savagei", "spec_spectrum", NULL, "1988",
	"Savage - Part 3 (48K)\0", "Level code: FERGUS", "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, Specsavage3RomInfo, Specsavage3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Scramble Spirits (128K)

static struct BurnRomInfo SpecscrspirtRomDesc[] = {
	{ "Scramble Spirits 128K (1990)(Grandslam).tap", 105793, 0x05c9e40e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specscrspirt, Specscrspirt, Spec128)
STD_ROM_FN(Specscrspirt)

struct BurnDriver BurnSpecscrspirt = {
	"spec_scrspirt", NULL, "spec_spec128", NULL, "1990",
	"Scramble Spirits (128K)\0", "NB: in Kempston controller the x-asis is reversed", "Grandslam Entertainments", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecscrspirtRomInfo, SpecscrspirtRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Seymour At The Movies (48K)

static struct BurnRomInfo Specseymmovi_48RomDesc[] = {
	{ "Seymour At The Movies (1991)(Codemasters)(48K).z80", 0x0b3c7, 0xc91a5fa5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specseymmovi_48, Specseymmovi_48, Spectrum)
STD_ROM_FN(Specseymmovi_48)

struct BurnDriver BurnSpecseymmovi = {
	"spec_seymmovi_48", "spec_seymmovi", "spec_spectrum", NULL, "1991",
	"Seymour at the Movies (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, Specseymmovi_48RomInfo, Specseymmovi_48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Seymour At The Movies (128K)

static struct BurnRomInfo SpecSeymmoviRomDesc[] = {
	{ "Seymour At the Movies (1991)(Codemasters)(128K).z80", 79527, 0xd1a72ae0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSeymmovi, SpecSeymmovi, Spec128)
STD_ROM_FN(SpecSeymmovi)

struct BurnDriver BurnSpecSeymmovi = {
	"spec_seymmovi", NULL, "spec_spec128", NULL, "1991",
	"Seymour At The Movies (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecSeymmoviRomInfo, SpecSeymmoviRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Seymour Take One (48K)

static struct BurnRomInfo SpecseytakonRomDesc[] = {
	{ "Seymour Take One (1991)(Codemasters).z80", 0x0858a, 0x21645f38, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specseytakon, Specseytakon, Spectrum)
STD_ROM_FN(Specseytakon)

struct BurnDriver BurnSpecseytakon = {
	"spec_seytakon", NULL, "spec_spectrum", NULL, "1991",
	"Seymour Take One (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecseytakonRomInfo, SpecseytakonRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Shadow Warriors (128K)

static struct BurnRomInfo SpecshadwarrRomDesc[] = {
	{ "Shadow Warriors (1990)(Ocean).tap", 130491, 0xa9adbe6d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specshadwarr, Specshadwarr, Spec128)
STD_ROM_FN(Specshadwarr)

struct BurnDriver BurnSpecshadwarr = {
	"spec_shadwarr", NULL, "spec_spec128", NULL, "1990",
	"Shadow Warriors (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecshadwarrRomInfo, SpecshadwarrRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Shinobi (128K)

static struct BurnRomInfo SpecshinobiRomDesc[] = {
	{ "Shinobi (1989)(Mastertronic Plus).tap", 231589, 0x195feaad, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specshinobi, Specshinobi, Spec128)
STD_ROM_FN(Specshinobi)

struct BurnDriver BurnSpecshinobi = {
	"spec_shinobi", NULL, "spec_spec128", NULL, "1989",
	"Shinobi (128K)\0", NULL, "Mastertronic Plus", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecshinobiRomInfo, SpecshinobiRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Silkworm (128K)

static struct BurnRomInfo SpecsilkwormRomDesc[] = {
	{ "Silkworm 128K (1989)(Mastertronic Plus).tap", 44612, 0x1980dccc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsilkworm, Specsilkworm, Spec128)
STD_ROM_FN(Specsilkworm)

struct BurnDriver BurnSpecsilkworm = {
	"spec_silkworm", NULL, "spec_spec128", NULL, "1989",
	"Silkworm (128K)\0", NULL, "Virgin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecsilkwormRomInfo, SpecsilkwormRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sim City (48K)

static struct BurnRomInfo SpecsimcityRomDesc[] = {
	{ "Sim City (1989)(Infogrames).z80", 34657, 0xd7281b2e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsimcity, Specsimcity, Spectrum)
STD_ROM_FN(Specsimcity)

struct BurnDriver BurnSpecsimcity = {
	"spec_simcity", NULL, "spec_spectrum", NULL, "1989",
	"Sim City (48K)\0", NULL, "Infogrames", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_STRATEGY, 0,
	SpectrumGetZipName, SpecsimcityRomInfo, SpecsimcityRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Simulador Profesional de Tenis (Spanish) (48K-128K)

static struct BurnRomInfo SpecsimulprotenisRomDesc[] = {
	{ "Simulador Profesional de Tenis ES 128K (1990)(Dinamic).tap", 53988, 0x2a031f63, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsimulprotenis, Specsimulprotenis, Spec128)
STD_ROM_FN(Specsimulprotenis)

struct BurnDriver BurnSpecsimulprotenis = {
	"spec_simulprotenis", NULL, "spec_spec128", NULL, "1990",
	"Simulador Profesional de Tenis (Spanish) (48K-128K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecsimulprotenisRomInfo, SpecsimulprotenisRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sir Fred (48K)

static struct BurnRomInfo SpecsirfredRomDesc[] = {
	{ "Sir Fred (1986)(Mikro-Gen).tap", 49152, 0x4a4f8a17, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsirfred, Specsirfred, Spectrum)
STD_ROM_FN(Specsirfred)

struct BurnDriver BurnSpecsirfred = {
	"spec_sirfred", NULL, "spec_spectrum", NULL, "1986",
	"Sir Fred (48K)\0", NULL, "Mikro-Gen", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecsirfredRomInfo, SpecsirfredRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sirwood - Intro (128K)

static struct BurnRomInfo SpecsirwoodintroRomDesc[] = {
	{ "Sirwood - Intro (1990)(Opera Soft).tap", 38955, 0x4a37ad11, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsirwoodintro, Specsirwoodintro, Spec128)
STD_ROM_FN(Specsirwoodintro)

struct BurnDriver BurnSpecsirwoodintro = {
	"spec_sirwoodintro", NULL, "spec_spec128", NULL, "1990",
	"Sirwood - Intro (128K)\0", NULL, "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecsirwoodintroRomInfo, SpecsirwoodintroRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sirwood - Part 1 (128K)

static struct BurnRomInfo Specsirwoodg1RomDesc[] = {
	{ "Sirwood - Level 1 (1990)(Opera Soft).tap", 46340, 0xd75b2f05, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsirwoodg1, Specsirwoodg1, Spec128)
STD_ROM_FN(Specsirwoodg1)

struct BurnDriver BurnSpecsirwoodg1 = {
	"spec_sirwoodg1", "spec_sirwoodintro", "spec_spec128", NULL, "1990",
	"Sirwood - Part 1 (128K)\0", "G: quit current game - M: pause on/off", "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specsirwoodg1RomInfo, Specsirwoodg1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sirwood - Part 2 (128K)

static struct BurnRomInfo Specsirwoodg2RomDesc[] = {
	{ "Sirwood - Level 2 (1990)(Opera Soft).tap", 47993, 0x8e0f2b4d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsirwoodg2, Specsirwoodg2, Spec128)
STD_ROM_FN(Specsirwoodg2)

struct BurnDriver BurnSpecsirwoodg2 = {
	"spec_sirwoodg2", "spec_sirwoodintro", "spec_spec128", NULL, "1990",
	"Sirwood - Part 2 (128K)\0", "G: quit current game - M: pause on/off", "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specsirwoodg2RomInfo, Specsirwoodg2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sirwood - Part 3 (128K)

static struct BurnRomInfo Specsirwoodg3RomDesc[] = {
	{ "Sirwood - Level 3 (1990)(Opera Soft).tap", 45506, 0xb58dc313, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsirwoodg3, Specsirwoodg3, Spec128)
STD_ROM_FN(Specsirwoodg3)

struct BurnDriver BurnSpecsirwoodg3 = {
	"spec_sirwoodg3", "spec_sirwoodintro", "spec_spec128", NULL, "1990",
	"Sirwood - Part 3 (128K)\0", "G: quit current game - M: pause on/off", "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specsirwoodg3RomInfo, Specsirwoodg3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sito Pons 500cc Grand Prix (Spanish) (48K)

static struct BurnRomInfo SpecsitoponsgpRomDesc[] = {
	{ "Sito Pons 500cc Grand Prix ES 48K (1990)(Zigurat).tap", 49795, 0x1afbe22f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsitoponsgp, Specsitoponsgp, Spectrum)
STD_ROM_FN(Specsitoponsgp)

struct BurnDriver BurnSpecsitoponsgp = {
	"spec_sitoponsgp", NULL, "spec_spectrum", NULL, "1990",
	"Sito Pons 500cc Grand Prix (Spanish) (48K)\0", "Press 'Z' to quit the game session", "Zigurat Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecsitoponsgpRomInfo, SpecsitoponsgpRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Skool Daze (48K)

static struct BurnRomInfo SpecskoldazeRomDesc[] = {
	{ "Skool Daze (1985)(Microsphere).z80", 0x0a5f5, 0x4034c78b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specskoldaze, Specskoldaze, Spectrum)
STD_ROM_FN(Specskoldaze)

struct BurnDriver BurnSpecskoldaze = {
	"spec_skoldaze", NULL, "spec_spectrum", NULL, "1985",
	"Skool Daze (48K)\0", NULL, "Microsphere", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecskoldazeRomInfo, SpecskoldazeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecCursorKeysDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Slap Fight (48K)

static struct BurnRomInfo SpecslapfighRomDesc[] = {
	{ "Slap Fight (1987)(Imagine).tap", 47183, 0xda714b33, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specslapfigh, Specslapfigh, Spectrum)
STD_ROM_FN(Specslapfigh)

struct BurnDriver BurnSpecslapfigh = {
	"spec_slapfigh", NULL, "spec_spectrum", NULL, "1987",
	"Slap Fight (48K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecslapfighRomInfo, SpecslapfighRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Slightly Magic (128K)

static struct BurnRomInfo SpecslightmgRomDesc[] = {
	{ "Slightly Magic (1990)(Codemasters)[128K].z80", 0x11bed, 0x12aaa197, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specslightmg, Specslightmg, Spec128)
STD_ROM_FN(Specslightmg)

struct BurnDriver BurnSpecslightmg = {
	"spec_slightmg", NULL, "spec_spec128", NULL, "1990",
	"Slightly Magic (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecslightmgRomInfo, SpecslightmgRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Smash TV (128K)

static struct BurnRomInfo SpecsmashtvRomDesc[] = {
	{ "Smash TV (1991)(Ocean)[128K].z80", 0x0e6c1, 0x2f90973d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsmashtv, Specsmashtv, Spec128)
STD_ROM_FN(Specsmashtv)

struct BurnDriver BurnSpecsmashtv = {
	"spec_smashtv", NULL, "spec_spec128", NULL, "1991",
	"Smash TV (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecsmashtvRomInfo, SpecsmashtvRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Soldier of Fortune (48K)

static struct BurnRomInfo SpecsoldfortRomDesc[] = {
	{ "Soldier Of Fortune (1988)(Firebird).z80", 0x0adff, 0xc3dc26df, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsoldfort, Specsoldfort, Spectrum)
STD_ROM_FN(Specsoldfort)

struct BurnDriver BurnSpecsoldfort = {
	"spec_soldfort", NULL, "spec_spectrum", NULL, "1988",
	"Soldier of Fortune (48K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecsoldfortRomInfo, SpecsoldfortRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sol Negro - Special Edition - Part 1 (Spanish) (48K)

static struct BurnRomInfo Specsolnegro1RomDesc[] = {
	{ "Sol Negro - Special Edition - Part 1 (1989)(Opera Soft).z80", 42525, 0x27b7e34f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsolnegro1, Specsolnegro1, Spectrum)
STD_ROM_FN(Specsolnegro1)

struct BurnDriver BurnSpecsolnegro1 = {
	"spec_solnegro1", NULL, "spec_spectrum", NULL, "1989",
	"Sol Negro - Special Edition - Part 1 (Spanish) (48K)\0", "AY8912 PSG supported", "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, Specsolnegro1RomInfo, Specsolnegro1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sol Negro - Special Edition - Part 2 (Spanish) (48K)

static struct BurnRomInfo Specsolnegro2RomDesc[] = {
	{ "Sol Negro - Special Edition - Part 2 (1989)(Opera Soft).z80", 37075, 0xaf912284, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsolnegro2, Specsolnegro2, Spectrum)
STD_ROM_FN(Specsolnegro2)

struct BurnDriver BurnSpecsolnegro2 = {
	"spec_solnegro2", "spec_solnegro1", "spec_spectrum", NULL, "1989",
	"Sol Negro - Special Edition - Part 2 (Spanish) (48K)\0", "AY8912 PSG supported", "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, Specsolnegro2RomInfo, Specsolnegro2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Solomon's Key (48K)

static struct BurnRomInfo SpecsolomonRomDesc[] = {
	{ "Solomon's Key 48K (1987)(U.S. Gold).tap", 47085, 0xf8e955e0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsolomon, Specsolomon, Spectrum)
STD_ROM_FN(Specsolomon)

struct BurnDriver BurnSpecsolomon = {
	"spec_solomon", NULL, "spec_spectrum", NULL, "1987",
	"Solomon's Key (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecsolomonRomInfo, SpecsolomonRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Space Harrier (48K)

static struct BurnRomInfo SpecsharrierRomDesc[] = {
	{ "Space Harrier (1986)(Elite Systems).z80", 0x0b439, 0xd33b7f51, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsharrier, Specsharrier, Spectrum)
STD_ROM_FN(Specsharrier)

struct BurnDriver BurnSpecsharrier = {
	"spec_sharrier", NULL, "spec_spectrum", NULL, "1986",
	"Space Harrier (48K)\0", NULL, "Elite Systems", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecsharrierRomInfo, SpecsharrierRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Space Harrier II (128K)

static struct BurnRomInfo Specsharrir2RomDesc[] = {
	{ "Space Harrier II 128K (1990)(Grandslam).tap", 149856, 0x69d5d562, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsharrir2, Specsharrir2, Spec128)
STD_ROM_FN(Specsharrir2)

struct BurnDriver BurnSpecsharrir2 = {
	"spec_sharrir2", NULL, "spec_spec128", NULL, "1990",
	"Space Harrier II (128K)\0", NULL, "Grandslam Entertainments", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, Specsharrir2RomInfo, Specsharrir2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Spy Hunter (48K)

static struct BurnRomInfo SpecspyhuntRomDesc[] = {
	{ "Spy Hunter (1985)(U.S. Gold).tap", 32547, 0x3febdfd4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specspyhunt, Specspyhunt, Spectrum)
STD_ROM_FN(Specspyhunt)

struct BurnDriver BurnSpecspyhunt = {
	"spec_spyhunt", NULL, "spec_spectrum", NULL, "1985",
	"Spy Hunter (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecspyhuntRomInfo, SpecspyhuntRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Spy vs Spy (48K)

static struct BurnRomInfo SpecspyvspyRomDesc[] = {
	{ "Spy vs Spy (1985)(Beyond Software).z80", 40858, 0xa5fc636b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specspyvspy, Specspyvspy, Spectrum)
STD_ROM_FN(Specspyvspy)

struct BurnDriver BurnSpecspyvspy = {
	"spec_spyvspy", NULL, "spec_spectrum", NULL, "1985",
	"Spy vs Spy (48K)\0", NULL, "Beyond Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION | GBF_STRATEGY, 0,
	SpectrumGetZipName, SpecspyvspyRomInfo, SpecspyvspyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecCursorKeysDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Spy vs Spy II - The Island Caper (48K)

static struct BurnRomInfo Specspyvspy2RomDesc[] = {
	{ "Spy vs Spy II - The Island Caper (1987)(Databyte).z80", 0x09350, 0xe5133176, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specspyvspy2, Specspyvspy2, Spectrum)
STD_ROM_FN(Specspyvspy2)

struct BurnDriver BurnSpecspyvspy2 = {
	"spec_spyvspy2", NULL, "spec_spectrum", NULL, "1987",
	"Spy vs Spy II - The Island Caper (48K)\0", NULL, "Databyte", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION | GBF_STRATEGY, 0,
	SpectrumGetZipName, Specspyvspy2RomInfo, Specspyvspy2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Spy vs Spy III - Arctic Antics (48K)

static struct BurnRomInfo Specspyvspy3RomDesc[] = {
	{ "Spy vs Spy III - Arctic Antics (1988)(Databyte).z80", 0x07a46, 0x9bf7db2a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specspyvspy3, Specspyvspy3, Spectrum)
STD_ROM_FN(Specspyvspy3)

struct BurnDriver BurnSpecspyvspy3 = {
	"spec_spyvspy3", NULL, "spec_spectrum", NULL, "1988",
	"Spy vs Spy III - Arctic Antics (48K)\0", NULL, "Databyte", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION | GBF_STRATEGY, 0,
	SpectrumGetZipName, Specspyvspy3RomInfo, Specspyvspy3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Star Farce (48K)

static struct BurnRomInfo SpecstarfarcRomDesc[] = {
	{ "Star Farce (1988)(Mastertronic).tap", 44969, 0xb7915492, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specstarfarc, Specstarfarc, Spectrum)
STD_ROM_FN(Specstarfarc)

struct BurnDriver BurnSpecstarfarc = {
	"spec_starfarc", NULL, "spec_spectrum", NULL, "1988",
	"Star Farce (48K)\0", NULL, "Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecstarfarcRomInfo, SpecstarfarcRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Starquake (48K)

static struct BurnRomInfo SpecstarquakRomDesc[] = {
	{ "Starquake (1985)(Bubble Bus).tap", 56203, 0x16e3e1b2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specstarquak, Specstarquak, Spectrum)
STD_ROM_FN(Specstarquak)

struct BurnDriver BurnSpecstarquak = {
	"spec_starquak", NULL, "spec_spectrum", NULL, "1985",
	"Starquake (48K)\0", NULL, "Bubblebus Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecstarquakRomInfo, SpecstarquakRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Stop the Express (48K)

static struct BurnRomInfo SpecstopexprRomDesc[] = {
	{ "Stop The Express (1983)(Hudson Soft).tap", 22521, 0x413eeba8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specstopexpr, Specstopexpr, Spectrum)
STD_ROM_FN(Specstopexpr)

struct BurnDriver BurnSpecstopexpr = {
	"spec_stopexpr", NULL, "spec_spectrum", NULL, "1983",
	"Stop the Express (48K)\0", NULL, "Sinclair Research", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecstopexprRomInfo, SpecstopexprRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Stormlord (48K)

static struct BurnRomInfo SpecstormlorRomDesc[] = {
	{ "Stormlord 48K (1989)(Hewson Consultants).tap", 46265, 0x6bef026d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specstormlor, Specstormlor, Spectrum)
STD_ROM_FN(Specstormlor)

struct BurnDriver BurnSpecstormlor = {
	"spec_stormlor", "spec_stormlor128", "spec_spectrum", NULL, "1989",
	"Stormlord (48K)\0", NULL, "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecstormlorRomInfo, SpecstormlorRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Stormlord (128K)

static struct BurnRomInfo Specstormlor128RomDesc[] = {
	{ "Stormlord 128K (1989)(Hewson Consultants).tap", 46266, 0xc91e464f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specstormlor128, Specstormlor128, Spec128)
STD_ROM_FN(Specstormlor128)

struct BurnDriver BurnSpecstormlor128 = {
	"spec_stormlor128", NULL, "spec_spec128", NULL, "1989",
	"Stormlord (128K)\0", NULL, "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Specstormlor128RomInfo, Specstormlor128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Street Fighter (Trainer) (48K)

static struct BurnRomInfo SpecsfightRomDesc[] = {
	{ "Street Fighter (1988)(Go!).tap", 70328, 0x293c4d02, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsfight, Specsfight, Spectrum)
STD_ROM_FN(Specsfight)

struct BurnDriver BurnSpecsfight = {
	"spec_sfight", NULL, "spec_spectrum", NULL, "1988",
	"Street Fighter (Trainer) (48K)\0", NULL, "Go!", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpecsfightRomInfo, SpecsfightRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Strider (128K)

static struct BurnRomInfo SpecstriderRomDesc[] = {
	{ "Strider (1989)(U.S. Gold).tap", 121097, 0x4e28a067, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specstrider, Specstrider, Spec128)
STD_ROM_FN(Specstrider)

struct BurnDriver BurnSpecstrider = {
	"spec_strider", NULL, "spec_spec128", NULL, "1989",
	"Strider (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecstriderRomInfo, SpecstriderRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Strider II (128K)

static struct BurnRomInfo Specstrider2RomDesc[] = {
	{ "Strider II (1990)(U.S. Gold).tap", 0x21e24, 0xd1b065fc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specstrider2, Specstrider2, Spec128)
STD_ROM_FN(Specstrider2)

struct BurnDriver BurnSpecstrider2 = {
	"spec_strider2", NULL, "spec_spec128", NULL, "1990",
	"Strider II (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, Specstrider2RomInfo, Specstrider2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Striker (128K)

static struct BurnRomInfo SpecstrikerRomDesc[] = {
	{ "Striker (1989)(Cult Games)[128K].z80", 0x090e9, 0xf7144c4d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specstriker, Specstriker, Spec128)
STD_ROM_FN(Specstriker)

struct BurnDriver BurnSpecstriker = {
	"spec_striker", NULL, "spec_spec128", NULL, "1989",
	"Striker (128K)\0", NULL, "Cult Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_STRATEGY | GBF_SPORTSFOOTBALL, 0,
	SpectrumGetZipName, SpecstrikerRomInfo, SpecstrikerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Stunt Car Racer (128K)

static struct BurnRomInfo SpecstuntcarRomDesc[] = {
	{ "Stunt Car Racer 128K (1989)(Micro Style).tap", 55835, 0xfdd995ec, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specstuntcar, Specstuntcar, Spec128)
STD_ROM_FN(Specstuntcar)

struct BurnDriver BurnSpecstuntcar = {
	"spec_stuntcar", NULL, "spec_spec128", NULL, "1989",
	"Stunt Car Racer (128K)\0", NULL, "Micro Style", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecstuntcarRomInfo, SpecstuntcarRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Subbuteo - The Computer Game (128K)

static struct BurnRomInfo SpecsubbuteoRomDesc[] = {
	{ "Subbuteo - The Computer Game 128K (1990)(Electronic Zoo).tap", 76987, 0x9a6211f8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsubbuteo, Specsubbuteo, Spec128)
STD_ROM_FN(Specsubbuteo)

struct BurnDriver BurnSpecsubbuteo = {
	"spec_subbuteo", NULL, "spec_spec128", NULL, "1990",
	"Subbuteo - The Computer Game (128K)\0", NULL, "Electronic Zoo", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL | GBF_STRATEGY, 0,
	SpectrumGetZipName, SpecsubbuteoRomInfo, SpecsubbuteoRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Super Hang-On - Part 1 - Beginner (48K)

static struct BurnRomInfo SpecshangonRomDesc[] = {
	{ "Super Hang-On - Part 1 - Beginner (1986)(Electric Dreams Software).z80", 0x08818, 0x6021b420, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specshangon, Specshangon, Spectrum)
STD_ROM_FN(Specshangon)

struct BurnDriver BurnSpecshangon = {
	"spec_shangon", NULL, "spec_spectrum", NULL, "1986",
	"Super Hang-On - Part 1 - Beginner (48K)\0", NULL, "Electric Dreams", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecshangonRomInfo, SpecshangonRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Super Hang-On - Part 2 - Junior (48K)

static struct BurnRomInfo Specshangon2RomDesc[] = {
	{ "Super Hang-On - Part 2 - Junior (1986)(Electric Dreams Software).z80", 0x08690, 0xd8180d70, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specshangon2, Specshangon2, Spectrum)
STD_ROM_FN(Specshangon2)

struct BurnDriver BurnSpecshangon2 = {
	"spec_shangon2", "spec_shangon", "spec_spectrum", NULL, "1986",
	"Super Hang-On - Part 2 - Junior (48K)\0", NULL, "Electric Dreams", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, Specshangon2RomInfo, Specshangon2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Super Hang-On - Part 3 - Senior (48K)

static struct BurnRomInfo Specshangon3RomDesc[] = {
	{ "Super Hang-On - Part 3 - Senior (1986)(Electric Dreams Software).z80", 0x08adc, 0x363567ec, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specshangon3, Specshangon3, Spectrum)
STD_ROM_FN(Specshangon3)

struct BurnDriver BurnSpecshangon3 = {
	"spec_shangon3", "spec_shangon", "spec_spectrum", NULL, "1986",
	"Super Hang-On - Part 3 - Senior (48K)\0", NULL, "Electric Dreams", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, Specshangon3RomInfo, Specshangon3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Super Hang-On - Part 4 - Expert (48K)

static struct BurnRomInfo Specshangon4RomDesc[] = {
	{ "Super Hang-On - Part 4 - Expert (1986)(Electric Dreams Software).z80", 0x0868a, 0x8cb2ac52, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specshangon4, Specshangon4, Spectrum)
STD_ROM_FN(Specshangon4)

struct BurnDriver BurnSpecshangon4 = {
	"spec_shangon4", "spec_shangon", "spec_spectrum", NULL, "1986",
	"Super Hang-On - Part 4 - Expert (48K)\0", NULL, "Electric Dreams", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, Specshangon4RomInfo, Specshangon4RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Super Monaco GP (128K)

static struct BurnRomInfo SpecsmgpRomDesc[] = {
	{ "Super Monaco GP 128K (1991)(U.S. Gold).tap", 123409, 0x14cba3ef, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsmgp, Specsmgp, Spec128)
STD_ROM_FN(Specsmgp)

struct BurnDriver BurnSpecsmgp = {
	"spec_smgp", NULL, "spec_spec128", NULL, "1991",
	"Super Monaco GP (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecsmgpRomInfo, SpecsmgpRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Super Sprint (48K)

static struct BurnRomInfo SpecsupespriRomDesc[] = {
	{ "Super Sprint (1987)(Activision).z80", 0x07a7e, 0x52ee2754, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsupespri, Specsupespri, Spectrum)
STD_ROM_FN(Specsupespri)

struct BurnDriver BurnSpecsupespri = {
	"spec_supespri", NULL, "spec_spectrum", NULL, "1987",
	"Super Sprint (48K)\0", NULL, "Activision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecsupespriRomInfo, SpecsupespriRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// SWIV (128K)

static struct BurnRomInfo SpecswivRomDesc[] = {
	{ "SWIV (1991)(Storm Software).tap", 0x2378e, 0xbaf9be65, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specswiv, Specswiv, Spec128)
STD_ROM_FN(Specswiv)

struct BurnDriver BurnSpecswiv = {
	"spec_swiv", NULL, "spec_spec128", NULL, "1991",
	"SWIV (128K)\0", NULL, "Storm Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecswivRomInfo, SpecswivRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Technician Ted (48K)

static struct BurnRomInfo Spectechted_48RomDesc[] = {
	{ "Technician Ted (1984)(Hewson Consultants).z80", 0x0a2af, 0x90e4eaee, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spectechted_48, Spectechted_48, Spectrum)
STD_ROM_FN(Spectechted_48)

struct BurnDriver BurnSpectechted_48 = {
	"spec_techted_48", NULL, "spec_spectrum", NULL, "1984",
	"Technician Ted (48K)\0", NULL, "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Spectechted_48RomInfo, Spectechted_48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Teenage Mutant Hero Turtles - The Coin-Op (128K)

static struct BurnRomInfo SpectmhtarcRomDesc[] = {
	{ "Teenage Mutant Hero Turtles - The Coin-Op (1991)(Image Works).tap", 155263, 0xa1ea1547, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spectmhtarc, Spectmhtarc, Spec128)
STD_ROM_FN(Spectmhtarc)

struct BurnDriver BurnSpectmhtarc = {
	"spec_tmhtarc", NULL, "spec_spec128", NULL, "1991",
	"Teenage Mutant Hero Turtles - The Coin-Op (128K)\0", NULL, "Image Works", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpectmhtarcRomInfo, SpectmhtarcRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Teenage Mutant Hero Turtles (48K)

static struct BurnRomInfo SpectmhtRomDesc[] = {
	{ "Teenage Mutant Hero Turtles (1990)(Image Works).tap", 43029, 0xd96767fb, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spectmht, Spectmht, Spectrum)
STD_ROM_FN(Spectmht)

struct BurnDriver BurnSpectmht = {
	"spec_tmht", NULL, "spec_spectrum", NULL, "1990",
	"Teenage Mutant Hero Turtles (48K)\0", NULL, "Image Works", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpectmhtRomInfo, SpectmhtRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Terra Cresta (48K)

static struct BurnRomInfo SpecterracrestaRomDesc[] = {
	{ "Terra Cresta (1986)(Imagine).tap", 48178, 0x55a50359, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specterracresta, Specterracresta, Spectrum)
STD_ROM_FN(Specterracresta)

struct BurnDriver BurnSpecterracresta = {
	"spec_terracresta", NULL, "spec_spectrum", NULL, "1986",
	"Terra Cresta (48K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecterracrestaRomInfo, SpecterracrestaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Tetris (Mirrorsoft) (128K)

static struct BurnRomInfo SpectetrisRomDesc[] = {
	{ "Tetris 128K (1988)(Mirrorsoft).tap", 38370, 0x6e160a6c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spectetris, Spectetris, Spec128)
STD_ROM_FN(Spectetris)

struct BurnDriver BurnSpectetris = {
	"spec_tetris", NULL, "spec_spec128", NULL, "1987-88",
	"Tetris (Mirrorsoft) (128K)\0", "Press 'O' for options", "Mirrorsoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpectetrisRomInfo, SpectetrisRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Three Weeks in Paradise (48K)

static struct BurnRomInfo Spec3weekspr_48RomDesc[] = {
	{ "Three Weeks In Paradise (1986)(Mikro-Gen).z80", 0x0a49e, 0x3c7ac0a9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec3weekspr_48, Spec3weekspr_48, Spectrum)
STD_ROM_FN(Spec3weekspr_48)

struct BurnDriver BurnSpec3weekspr_48 = {
	"spec_3weekspr_48", "spec_3weekspr", "spec_spectrum", NULL, "1986",
	"Three Weeks in Paradise (48K)\0", NULL, "Mikro-Gen", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, Spec3weekspr_48RomInfo, Spec3weekspr_48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Three Weeks in Paradise (128K)

static struct BurnRomInfo Spec3weeksprRomDesc[] = {
	{ "Three Weeks In Paradise (1986)(Mikro-Gen)(128k).z80", 0x0e06c, 0xf21d8b5d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec3weekspr, Spec3weekspr, Spec128)
STD_ROM_FN(Spec3weekspr)

struct BurnDriver BurnSpec3weekspr = {
	"spec_3weekspr", NULL, "spec_spec128", NULL, "1986",
	"Three Weeks in Paradise (128K)\0", NULL, "Mikro-Gen", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, Spec3weeksprRomInfo, Spec3weeksprRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Thunder Blade (128K)

static struct BurnRomInfo SpecthndrbldRomDesc[] = {
	{ "Thunder Blade (1988)(U.S. Gold).tap", 114186, 0xa56877b6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specthndrbld, Specthndrbld, Spec128)
STD_ROM_FN(Specthndrbld)

struct BurnDriver BurnSpecthndrbld = {
	"spec_thndrbld", NULL, "spec_spec128", NULL, "1988",
	"Thunder Blade (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecthndrbldRomInfo, SpecthndrbldRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Tiger Road (128K)

static struct BurnRomInfo SpectigeroadRomDesc[] = {
	{ "Tiger Road (1988)(Go!)[128K].z80", 0x0b726, 0xede04afd, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spectigeroad, Spectigeroad, Spec128)
STD_ROM_FN(Spectigeroad)

struct BurnDriver BurnSpectigeroad = {
	"spec_tigeroad", NULL, "spec_spec128", NULL, "1988",
	"Tiger Road (128K)\0", "Press '4' to use controller", "Go!", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpectigeroadRomInfo, SpectigeroadRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Time Scanner (128K)

static struct BurnRomInfo SpectimescanRomDesc[] = {
	{ "Time Scanner 128K (1989)(Activision).tap", 113308, 0x3698495a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spectimescan, Spectimescan, Spec128)
STD_ROM_FN(Spectimescan)

struct BurnDriver BurnSpectimescan = {
	"spec_timescan", NULL, "spec_spec128", NULL, "1989",
	"Time Scanner (128K)\0", "Flippers: W and O - Tilt: - Q, P and SPACE", "Activision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PINBALL, 0,
	SpectrumGetZipName, SpectimescanRomInfo, SpectimescanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Titanic - Part 1 (Spanish) (48K)

static struct BurnRomInfo Spectitanictopo1RomDesc[] = {
	{ "Titanic - part 1 ES (1988)(Topo Soft).tap", 50546, 0x797a006e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spectitanictopo1, Spectitanictopo1, Spectrum)
STD_ROM_FN(Spectitanictopo1)

struct BurnDriver BurnSpectitanictopo1 = {
	"spec_titanictopo1", NULL, "spec_spectrum", NULL, "1988",
	"Titanic - Part 1 (Spanish) (48K)\0", NULL, "Topo Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Spectitanictopo1RomInfo, Spectitanictopo1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Titanic - Part 2 (Spanish) (48K)

static struct BurnRomInfo Spectitanictopo2RomDesc[] = {
	{ "Titanic - part 2 ES (1988)(Topo Soft).tap", 46204, 0x547d088f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spectitanictopo2, Spectitanictopo2, Spectrum)
STD_ROM_FN(Spectitanictopo2)

struct BurnDriver BurnSpectitanictopo2 = {
	"spec_titanictopo2", "spec_titanictopo1", "spec_spectrum", NULL, "1988",
	"Titanic - Part 2 (Spanish) (48K)\0", "Password: SUSIE", "Topo Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Spectitanictopo2RomInfo, Spectitanictopo2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Total Recall (128K)

static struct BurnRomInfo SpectotrcallRomDesc[] = {
	{ "Total Recall (1991)(Ocean)[128K].z80", 0x17197, 0xab3503be, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spectotrcall, Spectotrcall, Spec128)
STD_ROM_FN(Spectotrcall)

struct BurnDriver BurnSpectotrcall = {
	"spec_totrcall", NULL, "spec_spec128", NULL, "1991",
	"Total Recall (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpectotrcallRomInfo, SpectotrcallRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Track and Field (48K)

static struct BurnRomInfo SpectracfielRomDesc[] = {
	{ "Track and Field (1988)(Ocean).z80", 0x05787, 0x7ddee010, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spectracfiel, Spectracfiel, Spectrum)
STD_ROM_FN(Spectracfiel)

struct BurnDriver BurnSpectracfiel = {
	"spec_tracfiel", NULL, "spec_spectrum", NULL, "1988",
	"Track and Field (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpectracfielRomInfo, SpectracfielRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Trantor - The Last Stormtrooper (48K)

static struct BurnRomInfo SpectrantorRomDesc[] = {
	{ "Trantor - The Last Stormtrooper 48K (1987)(Go!).tap", 47841, 0xd385ae7e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spectrantor, Spectrantor, Spectrum)
STD_ROM_FN(Spectrantor)

struct BurnDriver BurnSpectrantor = {
	"spec_trantor", NULL, "spec_spectrum", NULL, "1987",
	"Trantor - The Last Stormtrooper (48K)\0", NULL, "Go!", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpectrantorRomInfo, SpectrantorRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Trap Door, The (48K)

static struct BurnRomInfo SpectradoothRomDesc[] = {
	{ "Trap Door, The 48K (1986)(Piranha).tap", 42677, 0x0d070aa6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spectradooth, Spectradooth, Spectrum)
STD_ROM_FN(Spectradooth)

struct BurnDriver BurnSpectradooth = {
	"spec_tradooth", NULL, "spec_spectrum", NULL, "1986",
	"Trap Door, The (48K)\0", NULL, "Piranha", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpectradoothRomInfo, SpectradoothRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Underwurlde (48K)

static struct BurnRomInfo SpecunderwurRomDesc[] = {
	{ "Underwurlde (1984)(Ultimate Play The Game).tap", 46026, 0x0a46e7cf, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specunderwur, Specunderwur, Spectrum)
STD_ROM_FN(Specunderwur)

struct BurnDriver BurnSpecunderwur = {
	"spec_underwur", NULL, "spec_spectrum", NULL, "1984",
	"Underwurlde (48K)\0", NULL, "Ultimate Play The Game", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecunderwurRomInfo, SpecunderwurRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Untouchables, The (128K)

static struct BurnRomInfo SpecuntouchbRomDesc[] = {
	{ "Untouchables, The (1989)(Ocean).tap", 182932, 0x81349cbf, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specuntouchb, Specuntouchb, Spec128)
STD_ROM_FN(Specuntouchb)

struct BurnDriver BurnSpecuntouchb = {
	"spec_untouchb", NULL, "spec_spec128", NULL, "1989",
	"Untouchables, The (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecuntouchbRomInfo, SpecuntouchbRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Vindicator, The (128K)

static struct BurnRomInfo SpecvindtheRomDesc[] = {
	{ "Vindicator, The (1988)(Imagine Software)[128K].z80", 0x1b692, 0x57c8a81d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specvindthe, Specvindthe, Spec128)
STD_ROM_FN(Specvindthe)

struct BurnDriver BurnSpecvindthe = {
	"spec_vindthe", NULL, "spec_spec128", NULL, "1988",
	"Vindicator, The (128K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_SHOOT, 0,
	SpectrumGetZipName, SpecvindtheRomInfo, SpecvindtheRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Vindicators (128K)

static struct BurnRomInfo SpecvindicatRomDesc[] = {
	{ "Vindicators (1989)(Domark).tap", 0x0ffc4, 0xc4b6502e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specvindicat, Specvindicat, Spec128)
STD_ROM_FN(Specvindicat)

struct BurnDriver BurnSpecvindicat = {
	"spec_vindicat", NULL, "spec_spec128", NULL, "1989",
	"Vindicators (128K)\0", NULL, "Domark", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecvindicatRomInfo, SpecvindicatRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Wacky Darts (48K)

static struct BurnRomInfo Specwackdarts_48RomDesc[] = {
	{ "Wacky Darts (1991)(Codemasters)(48K).z80", 0x0ae9d, 0x6214a4ce, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specwackdarts_48, Specwackdarts_48, Spectrum)
STD_ROM_FN(Specwackdarts_48)

struct BurnDriver BurnSpecwackdart_48 = {
	"spec_wackdarts_48", "spec_wackydarts", "spec_spectrum", NULL, "1991",
	"Wacky Darts (48K)\0", "In options menu: keys '6', '7' and ENTER to choose and set", "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 8, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, Specwackdarts_48RomInfo, Specwackdarts_48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Wacky Races (128K)

static struct BurnRomInfo SpecwackraceRomDesc[] = {
	{ "Wacky Races 128K (1992)(Hi-Tec).tap", 149897, 0xc46f4db9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specwackrace, Specwackrace, Spec128)
STD_ROM_FN(Specwackrace)

struct BurnDriver BurnSpecwackrace = {
	"spec_wackrace", NULL, "spec_spec128", NULL, "1992",
	"Wacky Races (128K)\0", NULL, "Hi-Tec Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecwackraceRomInfo, SpecwackraceRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Wanted! Monty Mole (48K)

static struct BurnRomInfo SpecwanmonmoRomDesc[] = {
	{ "Wanted - Monty Mole (1984)(Gremlin Graphics).tap", 49411, 0xa50a88cc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specwanmonmo, Specwanmonmo, Spectrum)
STD_ROM_FN(Specwanmonmo)

struct BurnDriver BurnSpecwanmonmo = {
	"spec_wanmonmo", NULL, "spec_spectrum", NULL, "1984",
	"Wanted! Monty Mole (48K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecwanmonmoRomInfo, SpecwanmonmoRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Way of the Exploding Fist, The (48K)

static struct BurnRomInfo SpecwayexplfRomDesc[] = {
	{ "Way of the Exploding Fist, The (1985)(Melbourne House).tap", 45962, 0x7e81d64e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specwayexplf, Specwayexplf, Spectrum)
STD_ROM_FN(Specwayexplf)

struct BurnDriver BurnSpecwayexplf = {
	"spec_wayexplf", NULL, "spec_spectrum", NULL, "1985",
	"Way of the Exploding Fist, The (48K)\0", "Press '1' or '2' to one-two players mode and '0' to redefine controls", "Melbourne House", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpecwayexplfRomInfo, SpecwayexplfRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// WEC Le Mans (128K)

static struct BurnRomInfo SpecweclemansRomDesc[] = {
	{ "WEC Le Mans (1989)(Imagine)(48K-128K).tap", 112833, 0x835ac96a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specweclemans, Specweclemans, Spec128)
STD_ROM_FN(Specweclemans)

struct BurnDriver BurnSpecweclemans = {
	"spec_weclemans", NULL, "spec_spec128", NULL, "1988-89",
	"WEC Le Mans (128K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecweclemansRomInfo, SpecweclemansRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Wild West Seymour (48K)

static struct BurnRomInfo Specwwseymr_48RomDesc[] = {
	{ "Wild West Seymour (1992)(Codemasters).z80", 0x0b4ab, 0xbf324d91, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specwwseymr_48, Specwwseymr_48, Spectrum)
STD_ROM_FN(Specwwseymr_48)

struct BurnDriver BurnSpecwwseymr_48 = {
	"spec_wwseymr_48", "spec_wwseymr", "spec_spectrum", NULL, "1992",
	"Wild West Seymour (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, Specwwseymr_48RomInfo, Specwwseymr_48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Wild West Seymour (128K)

static struct BurnRomInfo SpecwwseymrRomDesc[] = {
	{ "Wild West Seymour (1992)(Codemasters)(128k).z80", 0x141c6, 0x01dc3cee, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specwwseymr, Specwwseymr, Spec128)
STD_ROM_FN(Specwwseymr)

struct BurnDriver BurnSpecwwseymr = {
	"spec_wwseymr", NULL, "spec_spec128", NULL, "1992",
	"Wild West Seymour (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecwwseymrRomInfo, SpecwwseymrRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Wonder Boy (128K)

static struct BurnRomInfo SpecwboyRomDesc[] = {
	{ "Wonder Boy (1987)(Activision)[128K].z80", 0x11193, 0xb492a055, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specwboy, Specwboy, Spec128)
STD_ROM_FN(Specwboy)

struct BurnDriver BurnSpecwboy = {
	"spec_wboy", NULL, "spec_spec128", NULL, "1987",
	"Wonder Boy (128K)\0", NULL, "Activision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecwboyRomInfo, SpecwboyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Xenon (128K)

static struct BurnRomInfo SpecxenonRomDesc[] = {
	{ "Xenon (1989)(Melbourne House).tap", 69556, 0xb9201d1c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specxenon, Specxenon, Spec128)
STD_ROM_FN(Specxenon)

struct BurnDriver BurnSpecxenon = {
	"spec_xenon", NULL, "spec_spec128", NULL, "1989",
	"Xenon (128K)\0", NULL, "Melbourne House", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecxenonRomInfo, SpecxenonRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Xevious (48K)

static struct BurnRomInfo SpecxeviousRomDesc[] = {
	{ "Xevious (1986)(U.S. Gold).tap", 44672, 0xf40c7c95, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specxevious, Specxevious, Spectrum)
STD_ROM_FN(Specxevious)

struct BurnDriver BurnSpecxevious = {
	"spec_xevious", NULL, "spec_spectrum", NULL, "1986",
	"Xevious (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecxeviousRomInfo, SpecxeviousRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Yie Ar Kung-Fu (48K)

static struct BurnRomInfo Specyiarkufu_48RomDesc[] = {
	{ "Yie Ar Kung-Fu 48K (1985)(Imagine).tap", 49930, 0xb2ca9ba8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specyiarkufu_48, Specyiarkufu_48, Spectrum)
STD_ROM_FN(Specyiarkufu_48)

struct BurnDriver BurnSpecyiarkufu_48 = {
	"spec_yiarkufu_48", "spec_yiarkufu", "spec_spectrum", NULL, "1985",
	"Yie Ar Kung-Fu (48K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, Specyiarkufu_48RomInfo, Specyiarkufu_48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Yie Ar Kung-Fu (128K)

static struct BurnRomInfo SpecyiarkufuRomDesc[] = {
	{ "Yie Ar Kung-Fu (1985)(Imagine Software)[128K].z80", 0x157f7, 0xf7c52002, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specyiarkufu, Specyiarkufu, Spec128)
STD_ROM_FN(Specyiarkufu)

struct BurnDriver BurnSpecyiarkufu = {
	"spec_yiarkufu", NULL, "spec_spec128", NULL, "1986",
	"Yie Ar Kung-Fu (128K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpecyiarkufuRomInfo, SpecyiarkufuRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Yie Ar Kung-Fu II (48K)

static struct BurnRomInfo Specyiarkuf2RomDesc[] = {
	{ "Yie Ar Kung-Fu 2 (1986)(Imagine).tap", 41343, 0x8a149697, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specyiarkuf2, Specyiarkuf2, Spectrum)
STD_ROM_FN(Specyiarkuf2)

struct BurnDriver BurnSpecyiarkuf2 = {
	"spec_yiarkuf2", NULL, "spec_spectrum", NULL, "1986",
	"Yie Ar Kung-Fu II (48K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, Specyiarkuf2RomInfo, Specyiarkuf2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Zaxxon (48K)

static struct BurnRomInfo SpeczaxxonRomDesc[] = {
	{ "Zaxxon (1985)(U.S. Gold).tap", 49889, 0x6d169dbf, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speczaxxon, Speczaxxon, Spectrum)
STD_ROM_FN(Speczaxxon)

struct BurnDriver BurnSpeczaxxon = {
	"spec_zaxxon", NULL, "spec_spectrum", NULL, "1985",
	"Zaxxon (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpeczaxxonRomInfo, SpeczaxxonRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Lorna (128K)

static struct BurnRomInfo SpecLornaRomDesc[] = {
	{ "Lorna (1990)(Topo Soft)(Sp).tap", 0x1cf8a, 0xa09afe9a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecLorna, SpecLorna, Spec128)
STD_ROM_FN(SpecLorna)

struct BurnDriver BurnSpecLorna = {
	"spec_lorna", NULL, "spec_spec128", NULL, "1990",
	"Lorna (128K)\0", NULL, "Topo Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecLornaRomInfo, SpecLornaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInvesInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Myth - History In The Making (128K)

static struct BurnRomInfo SpecMythRomDesc[] = {
	{ "Myth - History In The Making (1989)(System 3 Software).tap", 0x2f76e, 0xbc8f2fe4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMyth, SpecMyth, Spec128)
STD_ROM_FN(SpecMyth)

struct BurnDriver BurnSpecMyth = {
	"spec_myth", NULL, "spec_spec128", NULL, "1989",
	"Myth - History In The Making (128K)\0", NULL, "System 3", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecMythRomInfo, SpecMythRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Gremlins 2 - The New Batch (48K)

static struct BurnRomInfo SpecGremlins2RomDesc[] = {
	{ "Gremlins 2 - The New Batch (1990)(Topo Soft - Elite Systems).tap", 0x2716c, 0xb1402dfe, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGremlins2, SpecGremlins2, Spectrum)
STD_ROM_FN(SpecGremlins2)

struct BurnDriver BurnSpecGremlins2 = {
	"spec_gremlins2", NULL, "spec_spectrum", NULL, "1990",
	"Gremlins 2 - The New Batch (48K)\0", "In options menu use Q, A and Space", "Topo Soft - Elite Systems", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecGremlins2RomInfo, SpecGremlins2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Narco Police (128K)

static struct BurnRomInfo SpecNarcopolRomDesc[] = {
	{ "Narco Police (1991)(Rajsoft).tap", 0x1cf59, 0xd47256a3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecNarcopol, SpecNarcopol, Spec128)
STD_ROM_FN(SpecNarcopol)

struct BurnDriver BurnSpecNarcopol = {
	"spec_narcopol", NULL, "spec_spec128", NULL, "1991",
	"Narco Police (128K)\0", NULL, "Rajsoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecNarcopolRomInfo, SpecNarcopolRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Outrun (128K)

static struct BurnRomInfo SpecOutrunRomDesc[] = {
	{ "Outrun (1987)(US Gold).tap", 0x1f5ca, 0x1c4dde6b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecOutrun, SpecOutrun, Spec128)
STD_ROM_FN(SpecOutrun)

struct BurnDriver BurnSpecOutrun = {
	"spec_outrun", NULL, "spec_spec128", NULL, "1987",
	"Outrun (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecOutrunRomInfo, SpecOutrunRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Outrun Europa (128K)

static struct BurnRomInfo SpecOutruneuRomDesc[] = {
	{ "Outrun Europa (1991)(US Gold).tap", 0x495d0, 0x3240887e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecOutruneu, SpecOutruneu, Spec128)
STD_ROM_FN(SpecOutruneu)

struct BurnDriver BurnSpecOutruneu = {
	"spec_outruneu", NULL, "spec_spec128", NULL, "1991",
	"Outrun Europa (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecOutruneuRomInfo, SpecOutruneuRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// New Zealand Story, The (128K)

static struct BurnRomInfo SpecTnzsRomDesc[] = {
	{ "New Zealand Story, The (1989)(Ocean Software).tap", 0x2b800, 0xe6f32d53, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTnzs, SpecTnzs, Spec128)
STD_ROM_FN(SpecTnzs)

struct BurnDriver BurnSpecTnzs = {
	"spec_tnzs", NULL, "spec_spec128", NULL, "1989",
	"New Zealand Story, The (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecTnzsRomInfo, SpecTnzsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Turbo Outrun (128K)

static struct BurnRomInfo SpecToutrunRomDesc[] = {
	{ "Turbo Outrun (1990)(US Gold).tap", 0x5a75a, 0x80a6916d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecToutrun, SpecToutrun, Spec128)
STD_ROM_FN(SpecToutrun)

struct BurnDriver BurnSpecToutrun = {
	"spec_toutrun", NULL, "spec_spec128", NULL, "1990",
	"Turbo Outrun (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecToutrunRomInfo, SpecToutrunRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Astro Marine Corps - Part 1 (Spanish) (128K)

static struct BurnRomInfo SpecastmarcsRomDesc[] = {
	{ "Astro Marine Corps (1989)(Dinamic Software)(es)(Part 1 of 2).z80", 0x0c38f, 0x643f916a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specastmarcs, Specastmarcs, Spec128)
STD_ROM_FN(Specastmarcs)

struct BurnDriver BurnSpecastmarcs = {
	"spec_astmarcs", NULL, "spec_spec128", NULL, "1989",
	"Astro Marine Corps - Part 1 (Spanish) (128K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecastmarcsRomInfo, SpecastmarcsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Astro Marine Corps - Part 2 (Spanish) (128K)

static struct BurnRomInfo Specastmarcs2RomDesc[] = {
	{ "Astro Marine Corps (1989)(Dinamic Software)(es)(Part 2 of 2).z80", 0x0bcce, 0x295074fd, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specastmarcs2, Specastmarcs2, Spec128)
STD_ROM_FN(Specastmarcs2)

struct BurnDriver BurnSpecastmarcs2 = {
	"spec_astmarcs2", "spec_astmarcs", "spec_spec128", NULL, "1989",
	"Astro Marine Corps - Part 2 (Spanish) (128K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, Specastmarcs2RomInfo, Specastmarcs2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Choy-Lee-Fut Kung-Fu Warrior (Spanish) (48K)

static struct BurnRomInfo SpecclfkfwRomDesc[] = {
	{ "Choy-Lee-Fut Kung-Fu Warrior (1990)(Positive)(es).tap", 0x0bf47, 0x4af1bacf, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specclfkfw, Specclfkfw, Spectrum)
STD_ROM_FN(Specclfkfw)

struct BurnDriver BurnSpecclfkfw = {
	"spec_clfkfw", NULL, "spec_spectrum", NULL, "1990",
	"Choy-Lee-Fut Kung-Fu Warrior (Spanish) (48K)\0", NULL, "Positive", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpecclfkfwRomInfo, SpecclfkfwRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Crosswize (48K)

static struct BurnRomInfo SpeccrosswizeRomDesc[] = {
	{ "Crosswize (1988)(Firebird Software).tap", 0x0c4e7, 0x8a6d909b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccrosswize, Speccrosswize, Spectrum)
STD_ROM_FN(Speccrosswize)

struct BurnDriver BurnSpeccrosswize = {
	"spec_crosswize", NULL, "spec_spectrum", NULL, "1988",
	"Crosswize (48K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpeccrosswizeRomInfo, SpeccrosswizeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Goody (Spanish) (128K)

static struct BurnRomInfo SpecgoodyesRomDesc[] = {
	{ "Goody (1987)(Opera Soft)(es).tap", 0x0bdae, 0x2fd4f11b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specgoodyes, Specgoodyes, Spec128)
STD_ROM_FN(Specgoodyes)

struct BurnDriver BurnSpecgoodyes = {
	"spec_goodyes", NULL, "spec_spec128", NULL, "1987",
	"Goody (Spanish) (128K)\0", NULL, "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecgoodyesRomInfo, SpecgoodyesRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ice Breaker (Spanish) (128K)

static struct BurnRomInfo SpecicebreakerRomDesc[] = {
	{ "Ice Breaker (1990)(Topo Soft)(es).tap", 0x127d5, 0x62ce5b80, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specicebreaker, Specicebreaker, Spec128)
STD_ROM_FN(Specicebreaker)

struct BurnDriver BurnSpecicebreaker = {
	"spec_icebreaker", NULL, "spec_spec128", NULL, "1990",
	"Ice Breaker (Spanish) (128K)\0", NULL, "Topo Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecicebreakerRomInfo, SpecicebreakerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// International Karate+ (128K)

static struct BurnRomInfo SpecintkarateplusRomDesc[] = {
	{ "International Karate+ (1987)(System 3 Software).tap", 0x0d173, 0x097593e6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specintkarateplus, Specintkarateplus, Spec128)
STD_ROM_FN(Specintkarateplus)

struct BurnDriver BurnSpecintkarateplus = {
	"spec_intkarateplus", NULL, "spec_spec128", NULL, "1987",
	"International Karate+ (128K)\0", NULL, "System 3", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpecintkarateplusRomInfo, SpecintkarateplusRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Robin of the Wood (128K)

static struct BurnRomInfo SpecrobinofwoodRomDesc[] = {
	{ "Robin of the Wood (1985)(Odin Computer Graphics)(128k).tap", 0x0f37e, 0xce069366, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrobinofwood, Specrobinofwood, Spec128)
STD_ROM_FN(Specrobinofwood)

struct BurnDriver BurnSpecrobinofwood = {
	"spec_robinofwood", NULL, "spec_spec128", NULL, "1985",
	"Robin of the Wood (128K)\0", NULL, "Odin Computer Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_ADV, 0,
	SpectrumGetZipName, SpecrobinofwoodRomInfo, SpecrobinofwoodRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sidewize (48K)

static struct BurnRomInfo SpecsidewizeRomDesc[] = {
	{ "Sidewize (1987)(Firebird Software).tap", 0x0a2c8, 0xdb066855, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsidewize, Specsidewize, Spectrum)
STD_ROM_FN(Specsidewize)

struct BurnDriver BurnSpecsidewize = {
	"spec_sidewize", NULL, "spec_spectrum", NULL, "1987",
	"Sidewize (48K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecsidewizeRomInfo, SpecsidewizeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Test Drive II - The Duel (128K)

static struct BurnRomInfo Spectestdrv2RomDesc[] = {
	{ "Test Drive II - The Duel (1989)(Accolade).z80", 0x0a8cd, 0x4047f7de, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spectestdrv2, Spectestdrv2, Spec128)
STD_ROM_FN(Spectestdrv2)

struct BurnDriver BurnSpectestdrv2 = {
	"spec_testdrv2", NULL, "spec_spec128", NULL, "1989",
	"Test Drive II - The Duel (128K)\0", NULL, "Accolade", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, Spectestdrv2RomInfo, Spectestdrv2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Plot, The (48K)

static struct BurnRomInfo SpecplottheRomDesc[] = {
	{ "Plot, The (1988)(Firebird Software).tap", 0x0af68, 0x1adf1093, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specplotthe, Specplotthe, Spectrum)
STD_ROM_FN(Specplotthe)

struct BurnDriver BurnSpecplotthe = {
	"spec_plotthe", NULL, "spec_spectrum", NULL, "1988",
	"Plot, The (48K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecplottheRomInfo, SpecplottheRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Tintin on the Moon (48K)

static struct BurnRomInfo SpectintmoonRomDesc[] = {
	{ "Tintin on the Moon (1989)(Infogrames).tap", 0x09ca6, 0x1959b7a0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spectintmoon, Spectintmoon, Spectrum)
STD_ROM_FN(Spectintmoon)

struct BurnDriver BurnSpectintmoon = {
	"spec_tintmoon", NULL, "spec_spectrum", NULL, "1989",
	"Tintin on the Moon (48K)\0", NULL, "Infogrames", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpectintmoonRomInfo, SpectintmoonRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Turbo Cup Challenge (128K)

static struct BurnRomInfo SpecturbocupchRomDesc[] = {
	{ "Turbo Cup Challenge (1989)(Loriciels).tap", 0x0b19c, 0xb81e5707, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specturbocupch, Specturbocupch, Spec128)
STD_ROM_FN(Specturbocupch)

struct BurnDriver BurnSpecturbocupch = {
	"spec_turbocupch", NULL, "spec_spec128", NULL, "1989",
	"Turbo Cup Challenge (128K)\0", NULL, "Loriciels", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecturbocupchRomInfo, SpecturbocupchRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Vigilante (48K)

static struct BurnRomInfo SpecvigilanteRomDesc[] = {
	{ "Vigilante (1989)(U.S. Gold).tap", 0x1b03a, 0x372bf21f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specvigilante, Specvigilante, Spectrum)
STD_ROM_FN(Specvigilante)

struct BurnDriver BurnSpecvigilante = {
	"spec_vigilante", NULL, "spec_spectrum", NULL, "1989",
	"Vigilante (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecvigilanteRomInfo, SpecvigilanteRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Impossamole (128K)

static struct BurnRomInfo SpecimpossamoleRomDesc[] = {
	{ "Impossamole (1990)(Gremlin Graphics Software).tap", 0x25438, 0xf1c505c7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specimpossamole, Specimpossamole, Spec128)
STD_ROM_FN(Specimpossamole)

struct BurnDriver BurnSpecimpossamole = {
	"spec_impossamole", NULL, "spec_spec128", NULL, "1990",
	"Impossamole (128K)\0", NULL, "The Edge", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecimpossamoleRomInfo, SpecimpossamoleRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Saint Dragon (128K)

static struct BurnRomInfo SpecstdragonRomDesc[] = {
	{ "St. Dragon (1990)(Storm Software).tap", 0x2e158, 0x6bb077c8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specstdragon, Specstdragon, Spec128)
STD_ROM_FN(Specstdragon)

struct BurnDriver BurnSpecstdragon = {
	"spec_stdragon", NULL, "spec_spec128", NULL, "1990",
	"Saint Dragon (128K)\0", NULL, "Storm Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecstdragonRomInfo, SpecstdragonRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPMDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Stormlord II: Deliverance (128K)

static struct BurnRomInfo Specstormlord2RomDesc[] = {
	{ "Stormlord II - Deliverance (1990)(Hewson Consultants).tap", 128474, 0x2e4e34c6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specstormlord2, Specstormlord2, Spec128)
STD_ROM_FN(Specstormlord2)

struct BurnDriver BurnSpecstormlord2 = {
	"spec_stormlord2", NULL, "spec_spec128", NULL, "1990",
	"Stormlord II: Deliverance (128K)\0", NULL, "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Specstormlord2RomInfo, Specstormlord2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Desperado 2 (Spanish) (128K)

static struct BurnRomInfo Specdesperd2RomDesc[] = {
	{ "Desperado 2 (1991)(Topo Soft)(es).tap", 0x18bc6, 0xf1aca40a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdesperd2, Specdesperd2, Spec128)
STD_ROM_FN(Specdesperd2)

struct BurnDriver BurnSpecdesperd2 = {
	"spec_desperd2", NULL, "spec_spec128", NULL, "1991",
	"Desperado 2 (Spanish) (128K)\0", "Password Zona 2: CLAVE", "Topo Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_SHOOT, 0,
	SpectrumGetZipName, Specdesperd2RomInfo, Specdesperd2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mad Mix 2 - En el Castillo de los Fantasmas (Spanish) (128K)

static struct BurnRomInfo SpecmamiencaRomDesc[] = {
	{ "Mad Mix 2 - En el Castillo de los Fantasmas (1990)(Topo Soft)(es).tap", 0x13ae2, 0x2e343a0c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmamienca, Specmamienca, Spec128)
STD_ROM_FN(Specmamienca)

struct BurnDriver BurnSpecmamienca = {
	"spec_mamienca", NULL, "spec_spec128", NULL, "1990",
	"Mad Mix 2 - En el Castillo de los Fantasmas (Spanish) (128K)\0", NULL, "Topo Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE | GBF_ACTION, 0,
	SpectrumGetZipName, SpecmamiencaRomInfo, SpecmamiencaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Paris-Dakar (Spanish) (48K)

static struct BurnRomInfo SpecparisdakaresRomDesc[] = {
	{ "Paris-Dakar (1988)(Zigurat Software)(es).tap", 0x0f960, 0x15728fdf, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specparisdakares, Specparisdakares, Spectrum)
STD_ROM_FN(Specparisdakares)

struct BurnDriver BurnSpecparisdakares = {
	"spec_parisdakares", NULL, "spec_spectrum", NULL, "1988",
	"Paris-Dakar (Spanish) (48K)\0", NULL, "Zigurat Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecparisdakaresRomInfo, SpecparisdakaresRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Perico Delgado Maillot Amarillo (Spanish) (48K)

static struct BurnRomInfo SpecpericodelgadoRomDesc[] = {
	{ "Perico Delgado Maillot Amarillo (1989)(Topo Soft)(es).tap", 0x2497f, 0x284b616d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specpericodelgado, Specpericodelgado, Spectrum)
STD_ROM_FN(Specpericodelgado)

struct BurnDriver BurnSpecpericodelgado = {
	"spec_pericodelgado", NULL, "spec_spectrum", NULL, "1989",
	"Perico Delgado Maillot Amarillo (Spanish) (48K)\0", NULL, "Topo Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecpericodelgadoRomInfo, SpecpericodelgadoRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Soviet (Part 1 of 2) (128K)

static struct BurnRomInfo Specsovietpart1RomDesc[] = {
	{ "Soviet (1990)(Opera Soft)(Part 1 of 2).tap", 0x15ec0, 0x407510c3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsovietpart1, Specsovietpart1, Spec128)
STD_ROM_FN(Specsovietpart1)

struct BurnDriver BurnSpecsovietpart1 = {
	"spec_sovietpart1", NULL, "spec_spec128", NULL, "1990",
	"Soviet (Part 1 of 2) (128K)\0", "Set '1 keyboard' to use controller", "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, Specsovietpart1RomInfo, Specsovietpart1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Soviet (Part 2 of 2) (128K)

static struct BurnRomInfo Specsovietpart2RomDesc[] = {
	{ "Soviet (1990)(Opera Soft)(Part 2 of 2).tap", 0x186c4, 0x7db72ee5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsovietpart2, Specsovietpart2, Spec128)
STD_ROM_FN(Specsovietpart2)

struct BurnDriver BurnSpecsovietpart2 = {
	"spec_sovietpart2", "spec_sovietpart1", "spec_spec128", NULL, "1990",
	"Soviet (Part 2 of 2) (128K)\0", "Set '1 keyboard' to use controller", "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, Specsovietpart2RomInfo, Specsovietpart2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Tour 91 (Spanish) (128K)

static struct BurnRomInfo Spectour91RomDesc[] = {
	{ "Tour 91 (1991)(Topo Soft)(es).tap", 0x26f73, 0xa27369fc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spectour91, Spectour91, Spec128)
STD_ROM_FN(Spectour91)

struct BurnDriver BurnSpectour91 = {
	"spec_tour91", NULL, "spec_spec128", NULL, "1991",
	"Tour 91 (Spanish) (128K)\0", NULL, "Topo Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, Spectour91RomInfo, Spectour91RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Viaje al Centro de la Tierra (Spanish) (128K)

static struct BurnRomInfo SpecviajecentterraRomDesc[] = {
	{ "Viaje al Centro de la Tierra (1989)(Topo Soft).tap", 0x1eaf1, 0xdbc5e51b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specviajecentterra, Specviajecentterra, Spec128)
STD_ROM_FN(Specviajecentterra)

struct BurnDriver BurnSpecviajecentterra = {
	"spec_viajecentterra", NULL, "spec_spec128", NULL, "1989",
	"Viaje al Centro de la Tierra (Spanish) (128K)\0", NULL, "Topo Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecviajecentterraRomInfo, SpecviajecentterraRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// WWF WrestleMania (128K)

static struct BurnRomInfo SpecwwfwresmaniaRomDesc[] = {
	{ "WWF WrestleMania (1991)(Ocean Software).tap", 0x331b1, 0xb377350f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specwwfwresmania, Specwwfwresmania, Spec128)
STD_ROM_FN(Specwwfwresmania)

struct BurnDriver BurnSpecwwfwresmania = {
	"spec_wwfwresmania", NULL, "spec_spec128", NULL, "1991",
	"WWF WrestleMania (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpecwwfwresmaniaRomInfo, SpecwwfwresmaniaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Zona 0 (Spanish) (128K)

static struct BurnRomInfo Speczona0RomDesc[] = {
	{ "Zona 0 (1991)(Topo Soft)(es).tap", 0x12b84, 0xdf7e9568, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speczona0, Speczona0, Spec128)
STD_ROM_FN(Speczona0)

struct BurnDriver BurnSpeczona0 = {
	"spec_zona0", NULL, "spec_spec128", NULL, "1991",
	"Zona 0 (Spanish) (128K)\0", NULL, "Topo Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Speczona0RomInfo, Speczona0RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Double Dragon (48K)

static struct BurnRomInfo Specddragon1RomDesc[] = {
	{ "Double Dragon (1989)(Melbourne House).tap", 0x29ffb, 0xa38e0c54, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specddragon1, Specddragon1, Spec128)
STD_ROM_FN(Specddragon1)

struct BurnDriver BurnSpecddragon1 = {
	"spec_ddragon1", NULL, "spec_spec128", NULL, "1989",
	"Double Dragon (48K)\0", NULL, "Melbourne House", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, Specddragon1RomInfo, Specddragon1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Double Dragon II: The Revenge (128K)

static struct BurnRomInfo Specddragon2RomDesc[] = {
	{ "Double Dragon II - The Revenge (1989)(Virgin Games).tap", 0x2d64f, 0xaf872e35, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specddragon2, Specddragon2, Spec128)
STD_ROM_FN(Specddragon2)

struct BurnDriver BurnSpecddragon2 = {
	"spec_ddragon2", NULL, "spec_spec128", NULL, "1989",
	"Double Dragon II: The Revenge (128K)\0", NULL, "Virgin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, Specddragon2RomInfo, Specddragon2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mortadelo y Filemon II (Part 1 of 2) (Spanish)

static struct BurnRomInfo Specmortfilemon21RomDesc[] = {
	{ "Mortadelo y Filemon II (1990)(Dro Soft)(Part 1 of 2)(es).tap", 0x0bba9, 0x3735008e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmortfilemon21, Specmortfilemon21, Spec128)
STD_ROM_FN(Specmortfilemon21)

struct BurnDriver BurnSpecmortfilemon21 = {
	"spec_mortfilemon21", NULL, "spec_spec128", NULL, "1990",
	"Mortadelo y Filemon II (Part 1 of 2) (Spanish)\0", NULL, "Dro Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Specmortfilemon21RomInfo, Specmortfilemon21RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mortadelo y Filemon II (Part 2 of 2) (Spanish)

static struct BurnRomInfo Specmortfilemon22RomDesc[] = {
	{ "Mortadelo y Filemon II (1990)(Dro Soft)(Part 2 of 2)(es).tap", 0x0b6a3, 0x28472d0e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmortfilemon22, Specmortfilemon22, Spec128)
STD_ROM_FN(Specmortfilemon22)

struct BurnDriver BurnSpecmortfilemon22 = {
	"spec_mortfilemon22", "spec_mortfilemon21", "spec_spec128", NULL, "1990",
	"Mortadelo y Filemon II (Part 2 of 2) (Spanish)\0", NULL, "Dro Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Specmortfilemon22RomInfo, Specmortfilemon22RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Desperado (48K)

static struct BurnRomInfo SpecDesperadoRomDesc[] = {
	{ "Desperado (1986)(Central Solutions).tap", 0x9a0a, 0x488f2cd5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDesperado, SpecDesperado, Spectrum)
STD_ROM_FN(SpecDesperado)

struct BurnDriver BurnSpecDesperado = {
	"spec_desperado", NULL, "spec_spectrum", NULL, "1986",
	"Desperado (48K)\0", NULL, "Central Solutions", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecDesperadoRomInfo, SpecDesperadoRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mikie (48K)

static struct BurnRomInfo SpecMikieRomDesc[] = {
	{ "Mikie (1985)(Imagine Software).tap", 0xbc79, 0x45203b87, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMikie, SpecMikie, Spectrum)
STD_ROM_FN(SpecMikie)

struct BurnDriver BurnSpecMikie = {
	"spec_mikie", NULL, "spec_spectrum", NULL, "1985",
	"Mikie (48K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecMikieRomInfo, SpecMikieRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Super Trux (128K)

static struct BurnRomInfo SpecsupertruxRomDesc[] = {
	{ "Super Trux (1988)(Elite Systems).tap", 0xdd35, 0x65fcf302, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsupertrux, Specsupertrux, Spec128)
STD_ROM_FN(Specsupertrux)

struct BurnDriver BurnSpecsupertrux = {
	"spec_supertrux", NULL, "spec_spec128", NULL, "1988",
	"Super Trux (128K)\0", "Set 'S SINCLAIR' to use controller", "Elite Systems", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecsupertruxRomInfo, SpecsupertruxRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Glider Rider (128K)

static struct BurnRomInfo SpecGliderrRomDesc[] = {
	{ "Glider Rider (1987)(Quicksilva).tap", 0xfe51, 0x2789b5bb, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGliderr, SpecGliderr, Spec128)
STD_ROM_FN(SpecGliderr)

struct BurnDriver BurnSpecGliderr = {
	"spec_gliderr", NULL, "spec_spec128", NULL, "1987",
	"Glider Rider (128K)\0", NULL, "Quicksilva", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecGliderrRomInfo, SpecGliderrRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// North & South (128K)

static struct BurnRomInfo SpecnorthnsouthRomDesc[] = {
	{ "North & South (1991)(Infogrames)(128k).z80", 0x1bf6b, 0x8ca87f0d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specnorthnsouth, Specnorthnsouth, Spec128)
STD_ROM_FN(Specnorthnsouth)

struct BurnDriver BurnSpecnorthnsouth = {
	"spec_northnsouth", NULL, "spec_spec128", NULL, "1991",
	"North & South (128K)\0", "Use 'O', 'P' and SPACE for set language", "Infogrames", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_STRATEGY, 0,
	SpectrumGetZipName, SpecnorthnsouthRomInfo, SpecnorthnsouthRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Afteroids (Spanish) (48K)

static struct BurnRomInfo SpecafteroidsRomDesc[] = {
	{ "Afteroids (1988)(Zigurat Software)(es).tap", 0x0c30b, 0x53f59e66, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specafteroids, Specafteroids, Spectrum)
STD_ROM_FN(Specafteroids)

struct BurnDriver BurnSpecafteroids = {
	"spec_afteroids", NULL, "spec_spectrum", NULL, "1988",
	"Afteroids (Spanish) (48K)\0", NULL, "Zigurat Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecafteroidsRomInfo, SpecafteroidsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Humphrey (Spanish) (48K)

static struct BurnRomInfo SpechumphreyRomDesc[] = {
	{ "Humphrey (1988)(Zigurat Software)(es).tap", 0x0c30b, 0xdd0455ba, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spechumphrey, Spechumphrey, Spectrum)
STD_ROM_FN(Spechumphrey)

struct BurnDriver BurnSpechumphrey = {
	"spec_humphrey", NULL, "spec_spectrum", NULL, "1988",
	"Humphrey (Spanish) (48K)\0", NULL, "Zigurat Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpechumphreyRomInfo, SpechumphreyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Shadow Dancer (128K)

static struct BurnRomInfo SpecshaddancerRomDesc[] = {
	{ "Shadow Dancer (1991)(U.S. Gold).tap", 0x239f7, 0xfaa90676, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specshaddancer, Specshaddancer, Spec128)
STD_ROM_FN(Specshaddancer)

struct BurnDriver BurnSpecshaddancer = {
	"spec_shaddancer", NULL, "spec_spec128", NULL, "1991",
	"Shadow Dancer (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecshaddancerRomInfo, SpecshaddancerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Airborne Ranger (48K)

static struct BurnRomInfo SpecairbornerangerRomDesc[] = {
	{ "Airborne Ranger (1988)(MicroProse Software).tap", 0x40e88, 0xa82ed5c5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specairborneranger, Specairborneranger, Spectrum)
STD_ROM_FN(Specairborneranger)

struct BurnDriver BurnSpecairborneranger = {
	"spec_airborneranger", NULL, "spec_spectrum", NULL, "1988",
	"Airborne Ranger (48K)\0", NULL, "MicroProse Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_STRATEGY, 0,
	SpectrumGetZipName, SpecairbornerangerRomInfo, SpecairbornerangerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Project Stealth Fighter (128K)

static struct BurnRomInfo SpecprojstealthfighterRomDesc[] = {
	{ "Project Stealth Fighter (1990)(MicroProse Software).tap", 0x2bde9, 0xf0b7dc66, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specprojstealthfighter, Specprojstealthfighter, Spec128)
STD_ROM_FN(Specprojstealthfighter)

struct BurnDriver BurnSpecprojstealthfighter = {
	"spec_projstealthfighter", NULL, "spec_spec128", NULL, "1990",
	"Project Stealth Fighter (128K)\0", NULL, "MicroProse Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SIM, 0,
	SpectrumGetZipName, SpecprojstealthfighterRomInfo, SpecprojstealthfighterRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Golden Axe (128K)

static struct BurnRomInfo Specgoldenaxe1RomDesc[] = {
	{ "Golden Axe (1990)(Virgin Games).z80", 0x1b859, 0x3f32f82f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specgoldenaxe1, Specgoldenaxe1, Spec128)
STD_ROM_FN(Specgoldenaxe1)

struct BurnDriver BurnSpecgoldenaxe1 = {
	"spec_goldenaxe1", NULL, "spec_spec128", NULL, "1990",
	"Golden Axe (128K)\0", NULL, "Virgin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, Specgoldenaxe1RomInfo, Specgoldenaxe1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Gunship (128K)

static struct BurnRomInfo SpecgunshipRomDesc[] = {
	{ "Gunship (1987)(MicroProse Software).tap", 0x193ca, 0x01e4ed88, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specgunship, Specgunship, Spec128)
STD_ROM_FN(Specgunship)

struct BurnDriver BurnSpecgunship = {
	"spec_gunship", NULL, "spec_spec128", NULL, "1987",
	"Gunship (128K)\0", NULL, "MicroProse Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SIM, 0,
	SpectrumGetZipName, SpecgunshipRomInfo, SpecgunshipRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Desperado - Gunsmoke (Trainer) (48K)

static struct BurnRomInfo Specdesperado1trnRomDesc[] = {
	{ "Desperado (1987)(Topo Soft)(Trainer).tap", 149093, 0xff12d868, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdesperado1trn, Specdesperado1trn, Spectrum)
STD_ROM_FN(Specdesperado1trn)

struct BurnDriver BurnSpecdesperado1trn = {
	"spec_desperado1trn", NULL, "spec_spectrum", NULL, "1987",
	"Desperado - Gunsmoke (Trainer) (48K)\0", NULL, "Topo Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, Specdesperado1trnRomInfo, Specdesperado1trnRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Star Wars (48K)

static struct BurnRomInfo SpecStarwarsRomDesc[] = {
	{ "Star Wars.tap", 0xa323, 0x14f2595f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecStarwars, SpecStarwars, Spectrum)
STD_ROM_FN(SpecStarwars)

struct BurnDriver BurnSpecStarwars = {
	"spec_starwars", NULL, "spec_spectrum", NULL, "1987",
	"Star Wars (48K)\0", NULL, "Domark", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecStarwarsRomInfo, SpecStarwarsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Action Force (48K)

static struct BurnRomInfo SpecActionforceRomDesc[] = {
	{ "Action Force (1987)(Virgin Games).tap", 47303, 0x3ed87428, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecActionforce, SpecActionforce, Spectrum)
STD_ROM_FN(SpecActionforce)

struct BurnDriver BurnSpecActionforce = {
	"spec_actionforce", NULL, "spec_spectrum", NULL, "1987",
	"Action Force (48K)\0", NULL, "Virgin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecActionforceRomInfo, SpecActionforceRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Action Force II (128K)

static struct BurnRomInfo Specactionforce2RomDesc[] = {
	{ "Action Force II (1988)(Virgin Games).tap", 46813, 0x3d4139b7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specactionforce2, Specactionforce2, Spec128)
STD_ROM_FN(Specactionforce2)

struct BurnDriver BurnSpecactionforce2 = {
	"spec_actionforce2", NULL, "spec_spec128", NULL, "1988",
	"Action Force II (128K)\0", "On black screen press SPACE to start", "Virgin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, Specactionforce2RomInfo, Specactionforce2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecCursorKeysDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Action Reflex (48K)

static struct BurnRomInfo SpecactionreflexRomDesc[] = {
	{ "Action Reflex (1986)(Mirrorsoft).tap", 41597, 0xaada6209, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specactionreflex, Specactionreflex, Spectrum)
STD_ROM_FN(Specactionreflex)

struct BurnDriver BurnSpecactionreflex = {
	"spec_actionreflex", NULL, "spec_spectrum", NULL, "1986",
	"Action Reflex (48K)\0", NULL, "Mirrorsoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecactionreflexRomInfo, SpecactionreflexRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hostages (128K)

static struct BurnRomInfo SpechostagesRomDesc[] = {
	{ "Hostages (1990)(Infogrames).tap", 222554, 0x171b72b5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spechostages, Spechostages, Spec128)
STD_ROM_FN(Spechostages)

struct BurnDriver BurnSpechostages = {
	"spec_hostages", NULL, "spec_spec128", NULL, "1990",
	"Hostages (128K)\0", NULL, "Infogrames", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MISC | GBF_SHOOT, 0,
	SpectrumGetZipName, SpechostagesRomInfo, SpechostagesRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ranarama (48K)

static struct BurnRomInfo SpecranaramaRomDesc[] = {
	{ "Ranarama (1987)(Hewson Consultants).z80", 38541, 0x4894e422, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specranarama, Specranarama, Spectrum)
STD_ROM_FN(Specranarama)

struct BurnDriver BurnSpecranarama = {
	"spec_ranarama", NULL, "spec_spectrum", NULL, "1987",
	"Ranarama (48K)\0", NULL, "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE | GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecranaramaRomInfo, SpecranaramaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Zynaps (48K)

static struct BurnRomInfo SpeczynapsRomDesc[] = {
	{ "Zynaps (1987)(Hewson Consultants).tap", 42473, 0x2e272efc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speczynaps, Speczynaps, Spectrum)
STD_ROM_FN(Speczynaps)

struct BurnDriver BurnSpeczynaps = {
	"spec_zynaps", NULL, "spec_spectrum", NULL, "1987",
	"Zynaps (48K)\0", NULL, "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpeczynapsRomInfo, SpeczynapsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Uridium (48K)

static struct BurnRomInfo SpecuridiumRomDesc[] = {
	{ "Uridium (1986)(Hewson Consultants).tap", 27747, 0x28bdbbfc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specuridium, Specuridium, Spectrum)
STD_ROM_FN(Specuridium)

struct BurnDriver BurnSpecuridium = {
	"spec_uridium", NULL, "spec_spectrum", NULL, "1986",
	"Uridium (48K)\0", NULL, "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecuridiumRomInfo, SpecuridiumRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Netherworld (128K)

static struct BurnRomInfo SpecnetherworldRomDesc[] = {
	{ "Netherworld (1988)(Hewson Consultants).tap", 44343, 0x48db7496, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specnetherworld, Specnetherworld, Spec128)
STD_ROM_FN(Specnetherworld)

struct BurnDriver BurnSpecnetherworld = {
	"spec_netherworld", NULL, "spec_spec128", NULL, "1988",
	"Netherworld (128K)\0", NULL, "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecnetherworldRomInfo, SpecnetherworldRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Marauder (128K)

static struct BurnRomInfo SpecmarauderRomDesc[] = {
	{ "Marauder (1988)(Hewson Consultants).tap", 54339, 0xcbd7f723, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmarauder, Specmarauder, Spec128)
STD_ROM_FN(Specmarauder)

struct BurnDriver BurnSpecmarauder = {
	"spec_marauder", NULL, "spec_spec128", NULL, "1988",
	"Marauder (128K)\0", NULL, "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecmarauderRomInfo, SpecmarauderRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Kraal (48K)

static struct BurnRomInfo SpeckraalRomDesc[] = {
	{ "Kraal (1990)(Hewson Consultants).tap", 57614, 0x84276314, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speckraal, Speckraal, Spectrum)
STD_ROM_FN(Speckraal)

struct BurnDriver BurnSpeckraal = {
	"spec_kraal", NULL, "spec_spectrum", NULL, "1990",
	"Kraal (48K)\0", "Use Keyboard", "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpeckraalRomInfo, SpeckraalRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sentinel, The (48K)

static struct BurnRomInfo SpecthesentinelRomDesc[] = {
	{ "Sentinel, The (1987)(Firebird).tap", 45932, 0xc6957a83, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specthesentinel, Specthesentinel, Spectrum)
STD_ROM_FN(Specthesentinel)

struct BurnDriver BurnSpecthesentinel = {
	"spec_thesentinel", NULL, "spec_spectrum", NULL, "1987",
	"Sentinel, The (48K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecthesentinelRomInfo, SpecthesentinelRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Indiana Jones and the Last Crusade (128K)

static struct BurnRomInfo SpecindylastcrusadeRomDesc[] = {
	{ "Indiana Jones and the Last Crusade (1989)(U.S. Gold).tap", 145470, 0xe43fcead, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specindylastcrusade, Specindylastcrusade, Spec128)
STD_ROM_FN(Specindylastcrusade)

struct BurnDriver BurnSpecindylastcrusade = {
	"spec_indylastcrusade", NULL, "spec_spec128", NULL, "1989",
	"Indiana Jones and the Last Crusade (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecindylastcrusadeRomInfo, SpecindylastcrusadeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Indiana Jones and the Temple of Doom (48K)

static struct BurnRomInfo SpecindytempledoomRomDesc[] = {
	{ "Indiana Jones and the Temple of Doom (1987)(U.S. Gold).tap", 76529, 0x2c9fa63e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specindytempledoom, Specindytempledoom, Spectrum)
STD_ROM_FN(Specindytempledoom)

struct BurnDriver BurnSpecindytempledoom = {
	"spec_indytempledoom", NULL, "spec_spectrum", NULL, "1987",
	"Indiana Jones and the Temple of Doom (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecindytempledoomRomInfo, SpecindytempledoomRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Spindizzy (48K)

static struct BurnRomInfo SpecspindizzyRomDesc[] = {
	{ "Spindizzy (1986)(Electric Dreams).tap", 47877, 0x568a095a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specspindizzy, Specspindizzy, Spectrum)
STD_ROM_FN(Specspindizzy)

struct BurnDriver BurnSpecspindizzy = {
	"spec_spindizzy", NULL, "spec_spectrum", NULL, "1986",
	"Spindizzy (48K)\0", NULL, "Electric Dreams", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecspindizzyRomInfo, SpecspindizzyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hopping Mad (128K)

static struct BurnRomInfo SpechoppingmadRomDesc[] = {
	{ "Hopping Mad (1988)(Elite Systems).tap", 40347, 0x620ba84c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spechoppingmad, Spechoppingmad, Spec128)
STD_ROM_FN(Spechoppingmad)

struct BurnDriver BurnSpechoppingmad = {
	"spec_hoppingmad", NULL, "spec_spec128", NULL, "1988",
	"Hopping Mad (128K)\0", NULL, "Elite Systems", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpechoppingmadRomInfo, SpechoppingmadRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Indiana Jones and the Fate of Atlantis (128K)

static struct BurnRomInfo SpecindyfateatlantisRomDesc[] = {
	{ "Indiana Jones and the Fate of Atlantis (1992)(U.S. Gold).tap", 366350, 0xb6341899, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specindyfateatlantis, Specindyfateatlantis, Spec128)
STD_ROM_FN(Specindyfateatlantis)

struct BurnDriver BurnSpecindyfateatlantis = {
	"spec_indyfateatlantis", NULL, "spec_spec128", NULL, "1992",
	"Indiana Jones and the Fate of Atlantis (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecindyfateatlantisRomInfo, SpecindyfateatlantisRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Black Lamp (128K)

static struct BurnRomInfo SpecblacklampRomDesc[] = {
	{ "Black Lamp (1988)(Firebird Software).tap", 47656, 0xaffec634, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specblacklamp, Specblacklamp, Spec128)
STD_ROM_FN(Specblacklamp)

struct BurnDriver BurnSpecblacklamp = {
	"spec_blacklamp", NULL, "spec_spec128", NULL, "1988",
	"Black Lamp (128K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecblacklampRomInfo, SpecblacklampRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dark Fusion (128K)

static struct BurnRomInfo SpecdarkfusionRomDesc[] = {
	{ "Dark Fusion (1988)(Gremlin Graphics).tap", 65624, 0x87c7d4d6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdarkfusion, Specdarkfusion, Spec128)
STD_ROM_FN(Specdarkfusion)

struct BurnDriver BurnSpecdarkfusion = {
	"spec_darkfusion", NULL, "spec_spec128", NULL, "1988",
	"Dark Fusion (128K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecdarkfusionRomInfo, SpecdarkfusionRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Switchblade (128K)

static struct BurnRomInfo SpecswitchbladeRomDesc[] = {
	{ "Switchblade (1991)(Gremlin Graphics).tap", 52937, 0xa050f9d6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specswitchblade, Specswitchblade, Spec128)
STD_ROM_FN(Specswitchblade)

struct BurnDriver BurnSpecswitchblade = {
	"spec_switchblade", NULL, "spec_spec128", NULL, "1991",
	"Switchblade (128K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecswitchbladeRomInfo, SpecswitchbladeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// E-SWAT (128K)

static struct BurnRomInfo SpeceswatRomDesc[] = {
	{ "E-SWAT (1990)(U.S. Gold).tap", 213337, 0x721ab30f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speceswat, Speceswat, Spec128)
STD_ROM_FN(Speceswat)

struct BurnDriver BurnSpeceswat = {
	"spec_eswat", NULL, "spec_spec128", NULL, "1990",
	"E-SWAT (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpeceswatRomInfo, SpeceswatRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Final Fight (128K)

static struct BurnRomInfo SpecfinalfightRomDesc[] = {
	{ "Final Fight (1991)(U.S. Gold).tap", 199327, 0x8993d2c6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specfinalfight, Specfinalfight, Spec128)
STD_ROM_FN(Specfinalfight)

struct BurnDriver BurnSpecfinalfight = {
	"spec_finalfight", NULL, "spec_spec128", NULL, "1991",
	"Final Fight (128K)\0", "Press 'A' to use controller", "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecfinalfightRomInfo, SpecfinalfightRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ikari Warriors (128K)

static struct BurnRomInfo SpecikariwarriorsRomDesc[] = {
	{ "Ikari Warriors (1988)(Elite Systems).tap", 47889, 0xf3fac369, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specikariwarriors, Specikariwarriors, Spec128)
STD_ROM_FN(Specikariwarriors)

struct BurnDriver BurnSpecikariwarriors = {
	"spec_ikariwarriors", NULL, "spec_spec128", NULL, "1988",
	"Ikari Warriors (128K)\0", NULL, "Elite Systems", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecikariwarriorsRomInfo, SpecikariwarriorsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mercs (128K)

static struct BurnRomInfo SpecmercsRomDesc[] = {
	{ "Mercs (1991)(U.S. Gold).tap", 191310, 0x8b4f3dc9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmercs, Specmercs, Spec128)
STD_ROM_FN(Specmercs)

struct BurnDriver BurnSpecmercs = {
	"spec_mercs", NULL, "spec_spec128", NULL, "1991",
	"Mercs (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecmercsRomInfo, SpecmercsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Turrican (48K)

static struct BurnRomInfo Specturrican1RomDesc[] = {
	{ "Turrican (1990)(Rainbow Arts).tap", 266448, 0xc5cfad1a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specturrican1, Specturrican1, Spectrum)
STD_ROM_FN(Specturrican1)

struct BurnDriver BurnSpecturrican1 = {
	"spec_turrican1", NULL, "spec_spectrum", NULL, "1990",
	"Turrican (48K)\0", NULL, "Rainbow Arts", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_RUNGUN, 0,
	SpectrumGetZipName, Specturrican1RomInfo, Specturrican1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Turrican II (128K)

static struct BurnRomInfo Specturrican2RomDesc[] = {
	{ "Turrican II (1991)(Rainbow Arts).tap", 357318, 0x48f546ce, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specturrican2, Specturrican2, Spec128)
STD_ROM_FN(Specturrican2)

struct BurnDriver BurnSpecturrican2 = {
	"spec_turrican2", NULL, "spec_spec128", NULL, "1991",
	"Turrican II (128K)\0", NULL, "Rainbow Arts", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_RUNGUN, 0,
	SpectrumGetZipName, Specturrican2RomInfo, Specturrican2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Tetris 2 (128K)

static struct BurnRomInfo Spectetris2RomDesc[] = {
	{ "Tetris 2 (1990)(Fuxoft).tap", 24510, 0xeb9220fd, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spectetris2, Spectetris2, Spec128)
STD_ROM_FN(Spectetris2)

struct BurnDriver BurnSpectetris2 = {
	"spec_tetris2", NULL, "spec_spec128", NULL, "1990",
	"Tetris 2 (128K)\0", NULL, "Fuxoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, Spectetris2RomInfo, Spectetris2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 007 - Licence to Kill (128K)

static struct BurnRomInfo SpeclicencetokillRomDesc[] = {
	{ "Licence to Kill (1989)(Domark).tap", 48133, 0x44505823, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speclicencetokill, Speclicencetokill, Spec128)
STD_ROM_FN(Speclicencetokill)

struct BurnDriver BurnSpeclicencetokill = {
	"spec_licencetokill", NULL, "spec_spec128", NULL, "1989",
	"007 - Licence to Kill (128K)\0", NULL, "Domark", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpeclicencetokillRomInfo, SpeclicencetokillRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Finders Keepers (48K)

static struct BurnRomInfo SpecfinderskeepersRomDesc[] = {
	{ "Finders Keepers (1985)(Mastertronic).tap", 44174, 0x06f43742, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specfinderskeepers, Specfinderskeepers, Spectrum)
STD_ROM_FN(Specfinderskeepers)

struct BurnDriver BurnSpecfinderskeepers = {
	"spec_finderskeepers", NULL, "spec_spectrum", NULL, "1985",
	"Finders Keepers (48K)\0", NULL, "Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecfinderskeepersRomInfo, SpecfinderskeepersRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Spellbound (128K)

static struct BurnRomInfo SpecspellboundRomDesc[] = {
	{ "Spellbound (1985)(Mastertronic).tap", 112432, 0x275e1ec1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specspellbound, Specspellbound, Spec128)
STD_ROM_FN(Specspellbound)

struct BurnDriver BurnSpecspellbound = {
	"spec_spellbound", NULL, "spec_spec128", NULL, "1985",
	"Spellbound (128K)\0", "A, Z and SPACE to choose and set main options", "Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecspellboundRomInfo, SpecspellboundRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Knight Tyme (128K)

static struct BurnRomInfo SpecknighttymeRomDesc[] = {
	{ "Knight Tyme (1986)(Mastertronic).tap", 119348, 0x79c80fac, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specknighttyme, Specknighttyme, Spec128)
STD_ROM_FN(Specknighttyme)

struct BurnDriver BurnSpecknighttyme = {
	"spec_knighttyme", NULL, "spec_spec128", NULL, "1986",
	"Knight Tyme (128K)\0", "Options menu = A, Z, Space", "Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecknighttymeRomInfo, SpecknighttymeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Stormbringer (128K)

static struct BurnRomInfo SpecstormbringerRomDesc[] = {
	{ "Stormbringer (1987)(Mastertronic).tap", 102898, 0x8819d367, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specstormbringer, Specstormbringer, Spec128)
STD_ROM_FN(Specstormbringer)

struct BurnDriver BurnSpecstormbringer = {
	"spec_stormbringer", NULL, "spec_spec128", NULL, "1987",
	"Stormbringer (128K)\0", NULL, "Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecstormbringerRomInfo, SpecstormbringerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hades Nebula (48K)

static struct BurnRomInfo SpechadesnebulaRomDesc[] = {
	{ "Hades Nebula (1987)(Nexus Productions).tap", 43264, 0xbeac467c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spechadesnebula, Spechadesnebula, Spectrum)
STD_ROM_FN(Spechadesnebula)

struct BurnDriver BurnSpechadesnebula = {
	"spec_hadesnebula", NULL, "spec_spectrum", NULL, "1987",
	"Hades Nebula (48K)\0", NULL, "Nexus Productions", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpechadesnebulaRomInfo, SpechadesnebulaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Arc of Yesod, The (128K)

static struct BurnRomInfo SpecarcofyesodRomDesc[] = {
	{ "Arc of Yesod, The (1985)(Thor Computer Software).tap", 74016, 0xb4c5351b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specarcofyesod, Specarcofyesod, Spec128)
STD_ROM_FN(Specarcofyesod)

struct BurnDriver BurnSpecarcofyesod = {
	"spec_arcofyesod", NULL, "spec_spec128", NULL, "1985",
	"Arc of Yesod, The (128K)\0", NULL, "Thor Computer Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecarcofyesodRomInfo, SpecarcofyesodRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Nodes of Yesod (128K)

static struct BurnRomInfo SpecNoyesod128RomDesc[] = {
	{ "Nodes of Yesod 128K (1985)(Odin Computer Graphics).tap", 83920, 0xdea25751, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecNoyesod128, SpecNoyesod128, Spec128)
STD_ROM_FN(SpecNoyesod128)

struct BurnDriver BurnSpecNoyesod128 = {
	"spec_noyesod128", NULL, "spec_spec128", NULL, "1985",
	"Nodes of Yesod (128K)\0", NULL, "Odin Computer Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecNoyesod128RomInfo, SpecNoyesod128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Nodes of Yesod (48K)

static struct BurnRomInfo SpecNoyesod48RomDesc[] = {
	{ "Nodes of Yesod 48K (1985)(Odin Computer Graphics).tap", 36898, 0x1b5fe585, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecNoyesod48, SpecNoyesod48, Spectrum)
STD_ROM_FN(SpecNoyesod48)

struct BurnDriver BurnSpecNoyesod48 = {
	"spec_noyesod48", "spec_noyesod128", "spec_spectrum", NULL, "1985",
	"Nodes of Yesod (48K)\0", NULL, "Odin Computer Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecNoyesod48RomInfo, SpecNoyesod48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Blood Brothers (Module 1) (128K)

static struct BurnRomInfo Specbloodbrothers1RomDesc[] = {
	{ "Blood Brothers (Module 1)(1988)(Gremlin Graphics).z80", 42931, 0xb32fcea7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbloodbrothers1, Specbloodbrothers1, Spec128)
STD_ROM_FN(Specbloodbrothers1)

struct BurnDriver BurnSpecbloodbrothers1 = {
	"spec_bloodbrothers1", NULL, "spec_spec128", NULL, "1988",
	"Blood Brothers (Module 1) (128K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_PLATFORM, 0,
	SpectrumGetZipName, Specbloodbrothers1RomInfo, Specbloodbrothers1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Blood Brothers (Module 2) (128K)

static struct BurnRomInfo Specbloodbrothers2RomDesc[] = {
	{ "Blood Brothers (Module 2)(1988)(Gremlin Graphics).z80", 42790, 0x3fc07585, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbloodbrothers2, Specbloodbrothers2, Spec128)
STD_ROM_FN(Specbloodbrothers2)

struct BurnDriver BurnSpecbloodbrothers2 = {
	"spec_bloodbrothers2", "spec_bloodbrothers1", "spec_spec128", NULL, "1988",
	"Blood Brothers (Module 2) (128K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_PLATFORM, 0,
	SpectrumGetZipName, Specbloodbrothers2RomInfo, Specbloodbrothers2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Blood Brothers (Module 3) (128K)

static struct BurnRomInfo Specbloodbrothers3RomDesc[] = {
	{ "Blood Brothers (Module 3)(1988)(Gremlin Graphics).z80", 42935, 0x735098ec, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbloodbrothers3, Specbloodbrothers3, Spec128)
STD_ROM_FN(Specbloodbrothers3)

struct BurnDriver BurnSpecbloodbrothers3 = {
	"spec_bloodbrothers3", "spec_bloodbrothers1", "spec_spec128", NULL, "1988",
	"Blood Brothers (Module 3) (128K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_PLATFORM, 0,
	SpectrumGetZipName, Specbloodbrothers3RomInfo, Specbloodbrothers3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Destiny Mission (128K)

static struct BurnRomInfo SpecdestinymissionRomDesc[] = {
	{ "Destiny Mission (1990)(Williams Technology).tap", 42759, 0x71528fa3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdestinymission, Specdestinymission, Spec128)
STD_ROM_FN(Specdestinymission)

struct BurnDriver BurnSpecdestinymission = {
	"spec_destinymission", NULL, "spec_spec128", NULL, "1990",
	"Destiny Mission (128K)\0", NULL, "Williams Technology", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecdestinymissionRomInfo, SpecdestinymissionRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Frost Byte (48K)

static struct BurnRomInfo SpecfrostbyteRomDesc[] = {
	{ "Frost Byte (1986)(Mikro-Gen).tap", 49465, 0x895fc6d8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specfrostbyte, Specfrostbyte, Spectrum)
STD_ROM_FN(Specfrostbyte)

struct BurnDriver BurnSpecfrostbyte = {
	"spec_frostbyte", NULL, "spec_spectrum", NULL, "1986",
	"Frost Byte (48K)\0", NULL, "Mikro-Gen", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_MAZE, 0,
	SpectrumGetZipName, SpecfrostbyteRomInfo, SpecfrostbyteRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Gauntlet III - The Final Quest (128K)

static struct BurnRomInfo Specgauntlet3RomDesc[] = {
	{ "Gauntlet III - The Final Quest 128K (1991)(U.S. Gold).tap", 352550, 0x9dd3f5fc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specgauntlet3, Specgauntlet3, Spec128)
STD_ROM_FN(Specgauntlet3)

struct BurnDriver BurnSpecgauntlet3 = {
	"spec_gauntlet3", NULL, "spec_spec128", NULL, "1991",
	"Gauntlet III - The Final Quest (128K)\0", "Set 'Interface 2' to use controller", "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_MAZE | GBF_RUNGUN, 0,
	SpectrumGetZipName, Specgauntlet3RomInfo, Specgauntlet3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hydrofool (128K)

static struct BurnRomInfo SpechydrofoolRomDesc[] = {
	{ "Hydrofool (1987)(Faster Than Light).tap", 45889, 0x82e3db69, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spechydrofool, Spechydrofool, Spec128)
STD_ROM_FN(Spechydrofool)

struct BurnDriver BurnSpechydrofool = {
	"spec_hydrofool", NULL, "spec_spec128", NULL, "1987",
	"Hydrofool (128K)\0", NULL, "Faster Than Light", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpechydrofoolRomInfo, SpechydrofoolRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Kong's Revenge - Part 1 (Spanish) (128K)

static struct BurnRomInfo Speckongsrevenge1RomDesc[] = {
	{ "Kong's Revenge (Part 1)(1991)(Zigurat Software).tap", 56868, 0x20a7e7ce, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speckongsrevenge1, Speckongsrevenge1, Spec128)
STD_ROM_FN(Speckongsrevenge1)

struct BurnDriver BurnSpeckongsrevenge1 = {
	"spec_kongsrevenge1", NULL, "spec_spec128", NULL, "1991",
	"Kong's Revenge - Part 1 (Spanish) (128K)\0", NULL, "Zigurat Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Speckongsrevenge1RomInfo, Speckongsrevenge1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Kong's Revenge - Part 2 (Spanish) (128K)

static struct BurnRomInfo Speckongsrevenge2RomDesc[] = {
	{ "Kong's Revenge (Part 2)(1991)(Zigurat Software).tap", 56868, 0xdf77b41f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speckongsrevenge2, Speckongsrevenge2, Spec128)
STD_ROM_FN(Speckongsrevenge2)

struct BurnDriver BurnSpeckongsrevenge2 = {
	"spec_kongsrevenge2", "spec_kongsrevenge1", "spec_spec128", NULL, "1991",
	"Kong's Revenge - Part 2 (Spanish) (128K)\0", "Password: CHIRIPA - use num-keys from 1 to 7 to set password", "Zigurat Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, Speckongsrevenge2RomInfo, Speckongsrevenge2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// North Star (48K)

static struct BurnRomInfo SpecnorthstarRomDesc[] = {
	{ "North Star (1988)(Gremlin Graphics).tap", 49369, 0x61d487db, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specnorthstar, Specnorthstar, Spectrum)
STD_ROM_FN(Specnorthstar)

struct BurnDriver BurnSpecnorthstar = {
	"spec_northstar", NULL, "spec_spectrum", NULL, "1988",
	"North Star (48K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecnorthstarRomInfo, SpecnorthstarRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Athena (48K)

static struct BurnRomInfo SpecathenaRomDesc[] = {
	{ "Athena (1987)(Imagine).tap", 102859, 0xb457ae80, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specathena, Specathena, Spectrum)
STD_ROM_FN(Specathena)

struct BurnDriver BurnSpecathena = {
	"spec_athena", NULL, "spec_spectrum", NULL, "1987",
	"Athena (48K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecathenaRomInfo, SpecathenaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Psycho Hopper (48K)

static struct BurnRomInfo SpecpsychohopperRomDesc[] = {
	{ "Psycho Hopper (1990)(Mastertronic Plus).tap", 50178, 0xbeda1169, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specpsychohopper, Specpsychohopper, Spectrum)
STD_ROM_FN(Specpsychohopper)

struct BurnDriver BurnSpecpsychohopper = {
	"spec_psychohopper", NULL, "spec_spectrum", NULL, "1989-90",
	"Psycho Hopper (48K)\0", "Set '3 Sinclair' to use controller", "Mastertronic Plus", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecpsychohopperRomInfo, SpecpsychohopperRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Psycho Pigs U.X.B. (128K)

static struct BurnRomInfo SpecpsychopigsRomDesc[] = {
	{ "Psycho Pigs U.X.B. (1988)(U.S. Gold).tap", 50190, 0x3632f18a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specpsychopigs, Specpsychopigs, Spec128)
STD_ROM_FN(Specpsychopigs)

struct BurnDriver BurnSpecpsychopigs = {
	"spec_psychopigs", NULL, "spec_spec128", NULL, "1988",
	"Psycho Pigs U.X.B. (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecpsychopigsRomInfo, SpecpsychopigsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Snoopy (48K)

static struct BurnRomInfo SpecsnoopyRomDesc[] = {
	{ "Snoopy (1990)(The Edge).tap", 40149, 0xa24e7539, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsnoopy, Specsnoopy, Spectrum)
STD_ROM_FN(Specsnoopy)

struct BurnDriver BurnSpecsnoopy = {
	"spec_snoopy", NULL, "spec_spectrum", NULL, "1990",
	"Snoopy (48K)\0", NULL, "The Edge", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecsnoopyRomInfo, SpecsnoopyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Stainless Steel (48K)

static struct BurnRomInfo SpecstainlesssteelRomDesc[] = {
	{ "Stainless Steel (1986)(Mikro-Gen).tap", 44663, 0xc01c4113, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specstainlesssteel, Specstainlesssteel, Spectrum)
STD_ROM_FN(Specstainlesssteel)

struct BurnDriver BurnSpecstainlesssteel = {
	"spec_stainlesssteel", NULL, "spec_spectrum", NULL, "1986",
	"Stainless Steel (48K)\0", NULL, "Mikro-Gen", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecstainlesssteelRomInfo, SpecstainlesssteelRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Tai Chi Tortoise (128K)

static struct BurnRomInfo SpectaichitortoiseRomDesc[] = {
	{ "Tai Chi Tortoise (1991)(Zeppelin Games).tap", 29553, 0x5a60396a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spectaichitortoise, Spectaichitortoise, Spec128)
STD_ROM_FN(Spectaichitortoise)

struct BurnDriver BurnSpectaichitortoise = {
	"spec_taichitortoise", NULL, "spec_spec128", NULL, "1991",
	"Tai Chi Tortoise (128K)\0", NULL, "Zeppelin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpectaichitortoiseRomInfo, SpectaichitortoiseRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Towdie (English) (128K)

static struct BurnRomInfo SpectowdieenRomDesc[] = {
	{ "Towdie (English)(1994)(Ultrasoft).tap", 55179, 0x20bd2e70, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spectowdieen, Spectowdieen, Spec128)
STD_ROM_FN(Spectowdieen)

struct BurnDriver BurnSpectowdieen = {
	"spec_towdieen", NULL, "spec_spec128", NULL, "1994-2011",
	"Towdie (English) (128K)\0", NULL, "Ultrasoft - Softhouse", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpectowdieenRomInfo, SpectowdieenRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPMDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Towdie (Slovak) (128K)

static struct BurnRomInfo SpectowdieskRomDesc[] = {
	{ "Towdie (Slovak)(1994)(Ultrasoft).tap", 55187, 0x04ba91b1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spectowdiesk, Spectowdiesk, Spec128)
STD_ROM_FN(Spectowdiesk)

struct BurnDriver BurnSpectowdiesk = {
	"spec_towdiesk", "spec_towdieen", "spec_spec128", NULL, "1994",
	"Towdie (Slovak) (128K)\0", NULL, "Ultrasoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpectowdieskRomInfo, SpectowdieskRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPMDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Typhoon (128K)

static struct BurnRomInfo SpectyphoonRomDesc[] = {
	{ "Typhoon (1988)(Imagine Software).tap", 130475, 0x1ab3aa4a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spectyphoon, Spectyphoon, Spec128)
STD_ROM_FN(Spectyphoon)

struct BurnDriver BurnSpectyphoon = {
	"spec_typhoon", NULL, "spec_spec128", NULL, "1988",
	"Typhoon (128K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpectyphoonRomInfo, SpectyphoonRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Silent Shadow (48K)

static struct BurnRomInfo SpecsilentshadowRomDesc[] = {
	{ "Silent Shadow (1988)(Topo Soft).tap", 138528, 0x76e0c5b8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsilentshadow, Specsilentshadow, Spectrum)
STD_ROM_FN(Specsilentshadow)

struct BurnDriver BurnSpecsilentshadow = {
	"spec_silentshadow", NULL, "spec_spectrum", NULL, "1988",
	"Silent Shadow (48K)\0", "Set '1. KEYS' to use controller", "Topo Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecsilentshadowRomInfo, SpecsilentshadowRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Night Breed (128K)

static struct BurnRomInfo SpecnbreedRomDesc[] = {
	{ "Night Breed (1990)(Ocean).tap", 188653, 0x5701dcaf, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specnbreed, Specnbreed, Spec128)
STD_ROM_FN(Specnbreed)

struct BurnDriver BurnSpecnbreed = {
	"spec_nbreed", NULL, "spec_spec128", NULL, "1990",
	"Night Breed (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecnbreedRomInfo, SpecnbreedRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rescate en el Golfo - Part 1 (128K)

static struct BurnRomInfo Specrescategolfo1RomDesc[] = {
	{ "Rescate en el Golfo (1990)(Opera Soft)(Part 1 of 2).z80", 34779, 0xd7add860, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrescategolfo1, Specrescategolfo1, Spec128)
STD_ROM_FN(Specrescategolfo1)

struct BurnDriver BurnSpecrescategolfo1 = {
	"spec_rescategolfo1", NULL, "spec_spec128", NULL, "1990",
	"Rescate en el Golfo - Part 1 (128K)\0", NULL, "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, Specrescategolfo1RomInfo, Specrescategolfo1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rescate en el Golfo - Part 2 (128K)

static struct BurnRomInfo Specrescategolfo2RomDesc[] = {
	{ "Rescate en el Golfo (1990)(Opera Soft)(Part 2 of 2).z80", 37620, 0x370d2085, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrescategolfo2, Specrescategolfo2, Spec128)
STD_ROM_FN(Specrescategolfo2)

struct BurnDriver BurnSpecrescategolfo2 = {
	"spec_rescategolfo2", "spec_rescategolfo1", "spec_spec128", NULL, "1990",
	"Rescate en el Golfo - Part 2 (128K)\0", NULL, "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, Specrescategolfo2RomInfo, Specrescategolfo2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bobby Bearing (48K)

static struct BurnRomInfo SpecbobbybearingRomDesc[] = {
	{ "Bobby Bearing (1986)(The Edge).tap", 27778, 0x6b6895d2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbobbybearing, Specbobbybearing, Spectrum)
STD_ROM_FN(Specbobbybearing)

struct BurnDriver BurnSpecbobbybearing = {
	"spec_bobbybearing", NULL, "spec_spectrum", NULL, "1986",
	"Bobby Bearing (48K)\0", "6, 7 and 0 to select control type", "The Edge", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE, 0,
	SpectrumGetZipName, SpecbobbybearingRomInfo, SpecbobbybearingRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Incredible Shrinking Sphere (48K)

static struct BurnRomInfo SpecincshrinksphereRomDesc[] = {
	{ "Incredible Shrinking Sphere (1989)(Electric Dreams).tap", 42625, 0xf89b035d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specincshrinksphere, Specincshrinksphere, Spectrum)
STD_ROM_FN(Specincshrinksphere)

struct BurnDriver BurnSpecincshrinksphere = {
	"spec_incshrinksphere", NULL, "spec_spectrum", NULL, "1989",
	"Incredible Shrinking Sphere (48K)\0", NULL, "Electric Dreams", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE, 0,
	SpectrumGetZipName, SpecincshrinksphereRomInfo, SpecincshrinksphereRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Alien Storm (128K)

static struct BurnRomInfo SpecAlienstormRomDesc[] = {
	{ "Alien Storm (1991)(U.S. Gold).tap", 252760, 0x8aa6b74d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAlienstorm, SpecAlienstorm, Spec128)
STD_ROM_FN(SpecAlienstorm)

struct BurnDriver BurnSpecAlienstorm = {
	"spec_alienstorm", NULL, "spec_spec128", NULL, "1991",
	"Alien Storm (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SCRFIGHT | GBF_SHOOT, 0,
	SpectrumGetZipName, SpecAlienstormRomInfo, SpecAlienstormRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Guerrilla War (128K)

static struct BurnRomInfo SpecGuerrillawarRomDesc[] = {
	{ "Guerrilla War (1988)(Imagine Software).z80", 91016, 0xaf3aabbe, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGuerrillawar, SpecGuerrillawar, Spec128)
STD_ROM_FN(SpecGuerrillawar)

struct BurnDriver BurnSpecGuerrillawar = {
	"spec_guerrillawar", NULL, "spec_spec128", NULL, "1988",
	"Guerrilla War (128K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecGuerrillawarRomInfo, SpecGuerrillawarRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Predator (Trainer) (128K)

static struct BurnRomInfo SpecPredatorRomDesc[] = {
	{ "Predator 128K (1987)(Activision)[t +2].tap", 196738, 0xb7916d8c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPredator, SpecPredator, Spec128)
STD_ROM_FN(SpecPredator)

struct BurnDriver BurnSpecPredator = {
	"spec_predator", NULL, "spec_spec128", NULL, "1987",
	"Predator (Trainer) (128K)\0", NULL, "Activision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecPredatorRomInfo, SpecPredatorRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Predator 2 (128K)

static struct BurnRomInfo SpecPredator2RomDesc[] = {
	{ "Predator 2 (1991)(Image Works).tap", 120822, 0xa287648e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPredator2, SpecPredator2, Spec128)
STD_ROM_FN(SpecPredator2)

struct BurnDriver BurnSpecPredator2 = {
	"spec_predator2", NULL, "spec_spec128", NULL, "1991",
	"Predator 2 (128K)\0", NULL, "Image Works", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecPredator2RomInfo, SpecPredator2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Street Fighter II (128K)

static struct BurnRomInfo SpecStreetfighter2RomDesc[] = {
	{ "Street Fighter II (1993)(Go).tap", 519072, 0x052f9f1f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecStreetfighter2, SpecStreetfighter2, Spec128)
STD_ROM_FN(SpecStreetfighter2)

struct BurnDriver BurnSpecStreetfighter2 = {
	"spec_streetfighter2", NULL, "spec_spec128", NULL, "1993",
	"Street Fighter II (128K)\0", NULL, "Go", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpecStreetfighter2RomInfo, SpecStreetfighter2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Terminator 2 - Judgement Day (128K)

static struct BurnRomInfo SpecTerminator2RomDesc[] = {
	{ "Terminator 2 - Judgement Day (1991)(Ocean Software).tap", 105018, 0x23bc34fc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTerminator2, SpecTerminator2, Spec128)
STD_ROM_FN(SpecTerminator2)

struct BurnDriver BurnSpecTerminator2 = {
	"spec_terminator2", NULL, "spec_spec128", NULL, "1991",
	"Terminator 2 - Judgement Day (128K)\0", NULL, "Ocean Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpecTerminator2RomInfo, SpecTerminator2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Karateka (Spanish) (128K)

static struct BurnRomInfo SpeckaratekaRomDesc[] = {
	{ "Karateka (1990)(Dro Soft).tap", 67058, 0x61d5c8ac, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speckarateka, Speckarateka, Spec128)
STD_ROM_FN(Speckarateka)

struct BurnDriver BurnSpeckarateka = {
	"spec_karateka", NULL, "spec_spec128", NULL, "1990",
	"Karateka (Spanish) (128K)\0", NULL, "Dro Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpeckaratekaRomInfo, SpeckaratekaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Gunfright (48K)

static struct BurnRomInfo SpecGunfrightRomDesc[] = {
	{ "Gunfright (1986)(Ultimate Play The Game).tap", 43004, 0x2fe5588c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGunfright, SpecGunfright, Spectrum)
STD_ROM_FN(SpecGunfright)

struct BurnDriver BurnSpecGunfright = {
	"spec_gunfright", NULL, "spec_spectrum", NULL, "1986",
	"Gunfright (48K)\0", NULL, "Ultimate Play The Game", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecGunfrightRomInfo, SpecGunfrightRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Arcadia (16K)

static struct BurnRomInfo SpecArcadiaRomDesc[] = {
	{ "Arcadia (1982)(Imagine)(16K).tap", 8522, 0x63ef88a9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecArcadia, SpecArcadia, Spectrum)
STD_ROM_FN(SpecArcadia)

struct BurnDriver BurnSpecArcadia = {
	"spec_arcadia", NULL, "spec_spectrum", NULL, "1982",
	"Arcadia (16K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecArcadiaRomInfo, SpecArcadiaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Batty (48K)

static struct BurnRomInfo SpecBattyRomDesc[] = {
	{ "Batty (1987)(Hit-Pak).tap", 30142, 0x83ed1c93, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBatty, SpecBatty, Spectrum)
STD_ROM_FN(SpecBatty)

struct BurnDriver BurnSpecBatty = {
	"spec_batty", NULL, "spec_spectrum", NULL, "1987",
	"Batty (48K)\0", "Press 'A' for 1P options and 'B' for 2P options", "Hit-Pak", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_BREAKOUT, 0,
	SpectrumGetZipName, SpecBattyRomInfo, SpecBattyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Cookie (16K)

static struct BurnRomInfo SpecCookieRomDesc[] = {
	{ "Cookie (1983)(Ultimate Play The Game)(16K).tap", 15646, 0xfffe62ca, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCookie, SpecCookie, Spec128)
STD_ROM_FN(SpecCookie)

struct BurnDriver BurnSpecCookie = {
	"spec_cookie", NULL, "spec_spec128", NULL, "1983",
	"Cookie (16K)\0", NULL, "Ultimate Play The Game", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecCookieRomInfo, SpecCookieRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Day in the Life, A (48K)

static struct BurnRomInfo SpecDayinthelifeRomDesc[] = {
	{ "Day in the Life, A (1985)(Micromega).tap", 50099, 0x80e1e221, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDayinthelife, SpecDayinthelife, Spectrum)
STD_ROM_FN(SpecDayinthelife)

struct BurnDriver BurnSpecDayinthelife = {
	"spec_dayinthelife", NULL, "spec_spectrum", NULL, "1985",
	"Day in the Life, A (48K)\0", NULL, "Micromega", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecDayinthelifeRomInfo, SpecDayinthelifeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Firefly (128K)

static struct BurnRomInfo SpecFireflyRomDesc[] = {
	{ "Firefly (1988)(Ocean).tap", 45885, 0x4c890b6f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFirefly, SpecFirefly, Spec128)
STD_ROM_FN(SpecFirefly)

struct BurnDriver BurnSpecFirefly = {
	"spec_firefly", NULL, "spec_spec128", NULL, "1988",
	"Firefly (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_STRATEGY, 0,
	SpectrumGetZipName, SpecFireflyRomInfo, SpecFireflyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Gauntlet (128K)

static struct BurnRomInfo SpecGauntletRomDesc[] = {
	{ "Gauntlet (1987)(U.S. Gold)(48K-128K).tap", 150888, 0x173177cc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGauntlet, SpecGauntlet, Spec128)
STD_ROM_FN(SpecGauntlet)

struct BurnDriver BurnSpecGauntlet = {
	"spec_gauntlet", NULL, "spec_spec128", NULL, "1986",
	"Gauntlet (128K)\0", "In menu use Cursor Keys 5-6-7-8 + SPACE", "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_MAZE | GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecGauntletRomInfo, SpecGauntletRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hyper Active (48K)

static struct BurnRomInfo SpecHyperactiveRomDesc[] = {
	{ "Hyper Active (1988)(Sinclair User).tap", 25305, 0xb7aa30e8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHyperactive, SpecHyperactive, Spectrum)
STD_ROM_FN(SpecHyperactive)

struct BurnDriver BurnSpecHyperactive = {
	"spec_hyperactive", NULL, "spec_spectrum", NULL, "1988",
	"Hyper Active (48K)\0", NULL, "Sinclair User", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecHyperactiveRomInfo, SpecHyperactiveRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sabre Wulf (48K)

static struct BurnRomInfo SpecSabrewulfRomDesc[] = {
	{ "Sabre Wulf (1984)(Ultimate Play The Game).z80", 39457, 0x75fe1462, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSabrewulf, SpecSabrewulf, Spectrum)
STD_ROM_FN(SpecSabrewulf)

struct BurnDriver BurnSpecSabrewulf = {
	"spec_sabrewulf", NULL, "spec_spectrum", NULL, "1984",
	"Sabre Wulf (48K)\0", NULL, "Ultimate Play The Game", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecSabrewulfRomInfo, SpecSabrewulfRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// The Biz (48K)

static struct BurnRomInfo SpecThebizRomDesc[] = {
	{ "The Biz (1984)(Virgin Games).tap", 47268, 0x6773d204, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecThebiz, SpecThebiz, Spectrum)
STD_ROM_FN(SpecThebiz)

struct BurnDriver BurnSpecThebiz = {
	"spec_thebiz", NULL, "spec_spectrum", NULL, "1984",
	"The Biz (48K)\0", NULL, "Virgin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_STRATEGY, 0,
	SpectrumGetZipName, SpecThebizRomInfo, SpecThebizRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Trashman (48K)

static struct BurnRomInfo SpecTrashmanRomDesc[] = {
	{ "Trashman (1984)(New Generation).tap", 33845, 0x79aac256, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTrashman, SpecTrashman, Spectrum)
STD_ROM_FN(SpecTrashman)

struct BurnDriver BurnSpecTrashman = {
	"spec_trashman", NULL, "spec_spectrum", NULL, "1984",
	"Trashman (48K)\0", NULL, "New Generation", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecTrashmanRomInfo, SpecTrashmanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Aliens (UK Version) (48K)

static struct BurnRomInfo SpecAliensukRomDesc[] = {
	{ "Aliens UK (1986)(Electric Dreams).tap", 48291, 0xdb83e425, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAliensuk, SpecAliensuk, Spectrum)
STD_ROM_FN(SpecAliensuk)

struct BurnDriver BurnSpecAliensuk = {
	"spec_aliensuk", NULL, "spec_spectrum", NULL, "1986",
	"Aliens (UK Version) (48K)\0", "Press 'Space' to walk through doors", "Electric Dreams", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV | GBF_SHOOT, 0,
	SpectrumGetZipName, SpecAliensukRomInfo, SpecAliensukRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecCursorKeysDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Aliens (US Version) (48K)

static struct BurnRomInfo SpecAliensusRomDesc[] = {
	{ "Aliens US (1987)(Electric Dreams).tap", 192788, 0xd7c22d0d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAliensus, SpecAliensus, Spectrum)
STD_ROM_FN(SpecAliensus)

struct BurnDriver BurnSpecAliensus = {
	"spec_aliensus", NULL, "spec_spectrum", NULL, "1987",
	"Aliens (US Version) (48K)\0", NULL, "Electric Dreams", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecAliensusRomInfo, SpecAliensusRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Casanova (48K)

static struct BurnRomInfo SpecCasanovaRomDesc[] = {
	{ "Casanova (1989)(Iber Software).tap", 45005, 0x1c66d96d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCasanova, SpecCasanova, Spectrum)
STD_ROM_FN(SpecCasanova)

struct BurnDriver BurnSpecCasanova = {
	"spec_casanova", NULL, "spec_spectrum", NULL, "1989",
	"Casanova (48K)\0", NULL, "Iber Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecCasanovaRomInfo, SpecCasanovaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Human Killing Machine (48K)

static struct BurnRomInfo SpecHumankillmachineRomDesc[] = {
	{ "Human Killing Machine (1989)(U.S. Gold).tap", 83940, 0xfac2e471, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHumankillmachine, SpecHumankillmachine, Spectrum)
STD_ROM_FN(SpecHumankillmachine)

struct BurnDriver BurnSpecHumankillmachine = {
	"spec_humankillmachine", NULL, "spec_spectrum", NULL, "1988",
	"Human Killing Machine (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpecHumankillmachineRomInfo, SpecHumankillmachineRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hypsys - Part 1 (48K)

static struct BurnRomInfo SpecHypsys1RomDesc[] = {
	{ "Hypsys - Part 1 (1989)(Dro Soft).tap", 49162, 0x0d76e3f1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHypsys1, SpecHypsys1, Spectrum)
STD_ROM_FN(SpecHypsys1)

struct BurnDriver BurnSpecHypsys1 = {
	"spec_hypsys1", NULL, "spec_spectrum", NULL, "1989",
	"Hypsys - Part 1 (48K)\0", NULL, "Dro Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecHypsys1RomInfo, SpecHypsys1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hypsys - Part 2 (48K)

static struct BurnRomInfo SpecHypsys2RomDesc[] = {
	{ "Hypsys - Part 2 (1989)(Dro Soft).tap", 49809, 0x97b3d2d9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHypsys2, SpecHypsys2, Spectrum)
STD_ROM_FN(SpecHypsys2)

struct BurnDriver BurnSpecHypsys2 = {
	"spec_hypsys2", "spec_hypsys1", "spec_spectrum", NULL, "1989",
	"Hypsys - Part 2 (48K)\0", NULL, "Dro Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecHypsys2RomInfo, SpecHypsys2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Karnov (128K)

static struct BurnRomInfo SpecKarnovRomDesc[] = {
	{ "Karnov (1988)(Electric Dreams).tap", 234536, 0x95d92161, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecKarnov, SpecKarnov, Spec128)
STD_ROM_FN(SpecKarnov)

struct BurnDriver BurnSpecKarnov = {
	"spec_karnov", NULL, "spec_spec128", NULL, "1988",
	"Karnov (128K)\0", NULL, "Electric Dreams", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecKarnovRomInfo, SpecKarnovRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Kendo Warrior (128K)

static struct BurnRomInfo SpecKendowarriorRomDesc[] = {
	{ "Kendo Warrior (1989)(Byte Back).tap", 48724, 0xb4d3cb13, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecKendowarrior, SpecKendowarrior, Spec128)
STD_ROM_FN(SpecKendowarrior)

struct BurnDriver BurnSpecKendowarrior = {
	"spec_kendowarrior", NULL, "spec_spec128", NULL, "1989",
	"Kendo Warrior (128K)\0", NULL, "Byte Back", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecKendowarriorRomInfo, SpecKendowarriorRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Thunderbirds - Mission 1: Mine Menace (128K)

static struct BurnRomInfo SpecThundbirds1RomDesc[] = {
	{ "Thunderbirds - Part 1 (1989)(Grandslam Entertainments).tap", 52409, 0xa914189f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecThundbirds1, SpecThundbirds1, Spec128)
STD_ROM_FN(SpecThundbirds1)

struct BurnDriver BurnSpecThundbirds1 = {
	"spec_thundbirds1", NULL, "spec_spec128", NULL, "1989",
	"Thunderbirds - Mission 1: Mine Menace (128K)\0", "Press SPACE to change character", "Grandslam Entertainments", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecThundbirds1RomInfo, SpecThundbirds1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Thunderbirds - Mission 2: Sub Crash (128K)

static struct BurnRomInfo SpecThundbirds2RomDesc[] = {
	{ "Thunderbirds - Part 2 (1989)(Grandslam Entertainments).tap", 52409, 0x0869b04e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecThundbirds2, SpecThundbirds2, Spec128)
STD_ROM_FN(SpecThundbirds2)

struct BurnDriver BurnSpecThundbirds2 = {
	"spec_thundbirds2", "spec_thundbirds1", "spec_spec128", NULL, "1989",
	"Thunderbirds - Mission 2: Sub Crash (128K)\0", "Press SPACE to change character", "Grandslam Entertainments", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecThundbirds2RomInfo, SpecThundbirds2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Thunderbirds - Mission 3: Bank Job (128K)

static struct BurnRomInfo SpecThundbirds3RomDesc[] = {
	{ "Thunderbirds - Part 3 (1989)(Grandslam Entertainments).tap", 52409, 0xb156bc82, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecThundbirds3, SpecThundbirds3, Spec128)
STD_ROM_FN(SpecThundbirds3)

struct BurnDriver BurnSpecThundbirds3 = {
	"spec_thundbirds3", "spec_thundbirds1", "spec_spec128", NULL, "1989",
	"Thunderbirds - Mission 3: Bank Job (128K)\0", "Press SPACE to change character", "Grandslam Entertainments", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecThundbirds3RomInfo, SpecThundbirds3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Thunderbirds - Mission 4: Evil Hood (128K)

static struct BurnRomInfo SpecThundbirds4RomDesc[] = {
	{ "Thunderbirds - Part 4 (1989)(Grandslam Entertainments).tap", 52409, 0x9aafecf7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecThundbirds4, SpecThundbirds4, Spec128)
STD_ROM_FN(SpecThundbirds4)

struct BurnDriver BurnSpecThundbirds4 = {
	"spec_thundbirds4", "spec_thundbirds1", "spec_spec128", NULL, "1989",
	"Thunderbirds - Mission 4: Evil Hood (128K)\0", "Press SPACE to change character", "Grandslam Entertainments", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecThundbirds4RomInfo, SpecThundbirds4RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Paradise Cafe (Portuguese) (48K)

static struct BurnRomInfo SpecparadisecafeRomDesc[] = {
	{ "Paradise Cafe PT (1985)(Damatta)[adult].z80", 36975, 0xd0016cf2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specparadisecafe, Specparadisecafe, Spectrum)
STD_ROM_FN(Specparadisecafe)

struct BurnDriver BurnSpecparadisecafe = {
	"spec_paradisecafe", NULL, "spec_spectrum", NULL, "1985",
	"Paradise Cafe (Portuguese) (48K)\0", "Adult game", "Damatta", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MISC, 0,
	SpectrumGetZipName, SpecparadisecafeRomInfo, SpecparadisecafeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Xecutor (48K)

static struct BurnRomInfo SpecxecutorRomDesc[] = {
	{ "Xecutor (1987)(ACE Software).tap", 42063, 0xd1047062, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specxecutor, Specxecutor, Spectrum)
STD_ROM_FN(Specxecutor)

struct BurnDriver BurnSpecxecutor = {
	"spec_xecutor", "spec_xecutorremix", "spec_spectrum", NULL, "1987",
	"Xecutor (48K)\0", NULL, "ACE Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecxecutorRomInfo, SpecxecutorRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dominator (48K)

static struct BurnRomInfo SpecdominatorRomDesc[] = {
	{ "Dominator (1989)(System 3).tap", 67437, 0x3af9c9f7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdominator, Specdominator, Spectrum)
STD_ROM_FN(Specdominator)

struct BurnDriver BurnSpecdominator = {
	"spec_dominator", NULL, "spec_spectrum", NULL, "1987",
	"Dominator (48K)\0", NULL, "System 3", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT | GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecdominatorRomInfo, SpecdominatorRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sword of the Samurai (48K)

static struct BurnRomInfo SpecSwordsamuraiRomDesc[] = {
	{ "Sword of the Samurai (1992)(Zeppelin Games Ltd).tap", 41439, 0x42e23e9f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSwordsamurai, SpecSwordsamurai, Spectrum)
STD_ROM_FN(SpecSwordsamurai)

struct BurnDriver BurnSpecSwordsamurai = {
	"spec_swordsamurai", NULL, "spec_spectrum", NULL, "1992",
	"Sword of the Samurai (48K)\0", NULL, "Zeppelin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecSwordsamuraiRomInfo, SpecSwordsamuraiRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// California Games (48K)

static struct BurnRomInfo SpecCaliforniagamesRomDesc[] = {
	{ "California Games (1987)(U.S. Gold).tap", 168996, 0x52f1cd37, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCaliforniagames, SpecCaliforniagames, Spectrum)
STD_ROM_FN(SpecCaliforniagames)

struct BurnDriver BurnSpecCaliforniagames = {
	"spec_californiagames", NULL, "spec_spectrum", NULL, "1987",
	"California Games (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 4, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecCaliforniagamesRomInfo, SpecCaliforniagamesRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Deflektor (48K)

static struct BurnRomInfo SpecDeflektorRomDesc[] = {
	{ "Deflektor (1987)(Gremlin Graphics).tap", 28048, 0x04bf9aff, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDeflektor, SpecDeflektor, Spectrum)
STD_ROM_FN(SpecDeflektor)

struct BurnDriver BurnSpecDeflektor = {
	"spec_deflektor", NULL, "spec_spectrum", NULL, "1987",
	"Deflektor (48K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecDeflektorRomInfo, SpecDeflektorRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Games - Summer Edition, The (128K)

static struct BurnRomInfo SpecGamessummeredRomDesc[] = {
	{ "Games - Summer Edition, The (1989)(U.S. Gold).tap", 354208, 0xe8b9a3d0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGamessummered, SpecGamessummered, Spec128)
STD_ROM_FN(SpecGamessummered)

struct BurnDriver BurnSpecGamessummered = {
	"spec_gamessummered", NULL, "spec_spec128", NULL, "1989",
	"Games - Summer Edition, The (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecGamessummeredRomInfo, SpecGamessummeredRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Games - Winter Edition, The (48K)

static struct BurnRomInfo SpecGameswinteredRomDesc[] = {
	{ "Games - Winter Edition, The (1988)(U.S. Gold).tap", 146925, 0x15935233, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGameswintered, SpecGameswintered, Spectrum)
STD_ROM_FN(SpecGameswintered)

struct BurnDriver BurnSpecGameswintered = {
	"spec_gameswintered", NULL, "spec_spectrum", NULL, "1988",
	"Games - Winter Edition, The (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecGameswinteredRomInfo, SpecGameswinteredRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Goonies, The (48K)

static struct BurnRomInfo SpecGooniesRomDesc[] = {
	{ "Goonies, The (1986)(U.S. Gold).tap", 48377, 0x9ed8efdf, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGoonies, SpecGoonies, Spectrum)
STD_ROM_FN(SpecGoonies)

struct BurnDriver BurnSpecGoonies = {
	"spec_goonies", NULL, "spec_spectrum", NULL, "1986",
	"Goonies, The (48K)\0", "It's not a malfunction, the bg-music is horrible!", "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecGooniesRomInfo, SpecGooniesRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hyper Sports (48K)

static struct BurnRomInfo SpecHypersportsRomDesc[] = {
	{ "Hyper Sports (1985)(Imagine).tap", 48751, 0xe3a072a5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHypersports, SpecHypersports, Spectrum)
STD_ROM_FN(SpecHypersports)

struct BurnDriver BurnSpecHypersports = {
	"spec_hypersports", NULL, "spec_spectrum", NULL, "1985",
	"Hyper Sports (48K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecHypersportsRomInfo, SpecHypersportsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Lemmings (48K)

static struct BurnRomInfo SpecLemmingsRomDesc[] = {
	{ "Lemmings (1991)(Psygnosis).tap", 544963, 0x3a5fa27a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecLemmings, SpecLemmings, Spectrum)
STD_ROM_FN(SpecLemmings)

struct BurnDriver BurnSpecLemmings = {
	"spec_lemmings", NULL, "spec_spectrum", NULL, "1991",
	"Lemmings (48K)\0", NULL, "Psygnosis", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecLemmingsRomInfo, SpecLemmingsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// P-47 Thunderbolt (128K)

static struct BurnRomInfo SpecP47thunderboltRomDesc[] = {
	{ "P-47 Thunderbolt (1990)(Firebird).z80", 72894, 0x9efaf7fc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecP47thunderbolt, SpecP47thunderbolt, Spec128)
STD_ROM_FN(SpecP47thunderbolt)

struct BurnDriver BurnSpecP47thunderbolt = {
	"spec_p47thunderbolt", NULL, "spec_spec128", NULL, "1990",
	"P-47 Thunderbolt (128K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecP47thunderboltRomInfo, SpecP47thunderboltRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Summer Games (48K)

static struct BurnRomInfo SpecSummergamesRomDesc[] = {
	{ "Summer Games (1988)(U.S. Gold).tap", 103495, 0xa03cb5cb, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSummergames, SpecSummergames, Spectrum)
STD_ROM_FN(SpecSummergames)

struct BurnDriver BurnSpecSummergames = {
	"spec_summergames", NULL, "spec_spectrum", NULL, "1988",
	"Summer Games (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 4, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecSummergamesRomInfo, SpecSummergamesRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Summer Games II (48K)

static struct BurnRomInfo SpecSummergames2RomDesc[] = {
	{ "Summer Games II (1988)(U.S. Gold).tap", 189121, 0xdbe0691e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSummergames2, SpecSummergames2, Spectrum)
STD_ROM_FN(SpecSummergames2)

struct BurnDriver BurnSpecSummergames2 = {
	"spec_summergames2", NULL, "spec_spectrum", NULL, "1988",
	"Summer Games II (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecSummergames2RomInfo, SpecSummergames2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Super Wonder Boy (48K)

static struct BurnRomInfo SpecSuperwonderboyRomDesc[] = {
	{ "Super Wonder Boy (1989)(Activision).tap", 169846, 0x179244d3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSuperwonderboy, SpecSuperwonderboy, Spectrum)
STD_ROM_FN(SpecSuperwonderboy)

struct BurnDriver BurnSpecSuperwonderboy = {
	"spec_superwonderboy", NULL, "spec_spectrum", NULL, "1989",
	"Super Wonder Boy (48K)\0", NULL, "Activision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecSuperwonderboyRomInfo, SpecSuperwonderboyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Tai-Pan (128K)

static struct BurnRomInfo SpecTaipanRomDesc[] = {
	{ "Tai-Pan 128K (1987)(Ocean).z80", 82362, 0x3a80cd98, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTaipan, SpecTaipan, Spec128)
STD_ROM_FN(SpecTaipan)

struct BurnDriver BurnSpecTaipan = {
	"spec_taipan", NULL, "spec_spec128", NULL, "1987",
	"Tai-Pan (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecTaipanRomInfo, SpecTaipanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// U.N. Squadron (128K)

static struct BurnRomInfo SpecUnsquadronRomDesc[] = {
	{ "UN Squadron (1990)(U.S Gold).tap", 303274, 0x12571e3a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecUnsquadron, SpecUnsquadron, Spec128)
STD_ROM_FN(SpecUnsquadron)

struct BurnDriver BurnSpecUnsquadron = {
	"spec_unsquadron", NULL, "spec_spec128", NULL, "1990",
	"U.N. Squadron (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecUnsquadronRomInfo, SpecUnsquadronRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Winter Games (128K)

static struct BurnRomInfo SpecWintergamesRomDesc[] = {
	{ "Winter Games (1986)(U.S. Gold).tap", 119630, 0x5ab9f21c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecWintergames, SpecWintergames, Spec128)
STD_ROM_FN(SpecWintergames)

struct BurnDriver BurnSpecWintergames = {
	"spec_wintergames", NULL, "spec_spec128", NULL, "1986",
	"Winter Games (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 4, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecWintergamesRomInfo, SpecWintergamesRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// World Games (128K)

static struct BurnRomInfo SpecWorldgamesRomDesc[] = {
	{ "World Games (1987)(U.S. Gold).tap", 224518, 0xab9ccddb, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecWorldgames, SpecWorldgames, Spec128)
STD_ROM_FN(SpecWorldgames)

struct BurnDriver BurnSpecWorldgames = {
	"spec_worldgames", NULL, "spec_spec128", NULL, "1987",
	"World Games (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 4, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecWorldgamesRomInfo, SpecWorldgamesRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Xecutor - Remix (128K)

static struct BurnRomInfo SpecXecutorremixRomDesc[] = {
	{ "Xecutor - Remix (1987)(ACE Software).z80", 77548, 0x97ef7538, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecXecutorremix, SpecXecutorremix, Spec128)
STD_ROM_FN(SpecXecutorremix)

struct BurnDriver BurnSpecXecutorremix = {
	"spec_xecutorremix", NULL, "spec_spec128", NULL, "1987",
	"Xecutor - Remix (128K)\0", NULL, "ACE Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecXecutorremixRomInfo, SpecXecutorremixRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 100 - Stolnik (128K)

static struct BurnRomInfo Spec100stolnikRomDesc[] = {
	{ "100 - Stolnik (1995)(Power Of Sound).z80", 43905, 0x8de30bdc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec100stolnik, Spec100stolnik, Spec128)
STD_ROM_FN(Spec100stolnik)

struct BurnDriver BurnSpec100stolnik = {
	"spec_100stolnik", NULL, "spec_spec128", NULL, "1995",
	"100 - Stolnik (128K)\0", NULL, "Power Of Sound", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, Spec100stolnikRomInfo, Spec100stolnikRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 10th Frame (48K)

static struct BurnRomInfo Spec10thframeRomDesc[] = {
	{ "10th Frame (1987)(U.S. Gold).tap", 20061, 0x4a6d1860, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec10thframe, Spec10thframe, Spectrum)
STD_ROM_FN(Spec10thframe)

struct BurnDriver BurnSpec10thframe = {
	"spec_10thframe", NULL, "spec_spectrum", NULL, "1987",
	"10th Frame (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 8, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, Spec10thframeRomInfo, Spec10thframeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 2112 AD (48K)

static struct BurnRomInfo Spec2112adRomDesc[] = {
	{ "2112 AD (1985)(Design Design).tap", 45602, 0xca5a76f1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec2112ad, Spec2112ad, Spectrum)
STD_ROM_FN(Spec2112ad)

struct BurnDriver BurnSpec2112ad = {
	"spec_2112ad", NULL, "spec_spectrum", NULL, "1985",
	"2112 AD (48K)\0", "Q, A, N, M and 0 to play", "Design Design", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, Spec2112adRomInfo, Spec2112adRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 21 Erotic (Czech) (48K)

static struct BurnRomInfo Spec21eroticRomDesc[] = {
	{ "21 Erotic (1995)(JardaSoft).tap", 10336, 0x80cb68c0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec21erotic, Spec21erotic, Spectrum)
STD_ROM_FN(Spec21erotic)

struct BurnDriver BurnSpec21erotic = {
	"spec_21erotic", NULL, "spec_spectrum", NULL, "1995",
	"21 Erotic (Czech) (48K)\0", "Adult game", "JardaSoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_CASINO, 0,
	SpectrumGetZipName, Spec21eroticRomInfo, Spec21eroticRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 4 Soccer Simulators - 11-a-Side Soccer (128K)

static struct BurnRomInfo Spec4soccersims1RomDesc[] = {
	{ "4 Soccer Simulators - 11-a-Side Soccer (1989)(Codemasters).z80", 44128, 0x7409c0ec, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec4soccersims1, Spec4soccersims1, Spec128)
STD_ROM_FN(Spec4soccersims1)

struct BurnDriver BurnSpec4soccersims1 = {
	"spec_4soccersims1", NULL, "spec_spec128", NULL, "1989",
	"4 Soccer Simulators - 11-a-Side Soccer (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 4, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL, 0,
	SpectrumGetZipName, Spec4soccersims1RomInfo, Spec4soccersims1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 4 Soccer Simulators - Indoor Soccer (128K)

static struct BurnRomInfo Spec4soccersims2RomDesc[] = {
	{ "4 Soccer Simulators - Indoor Soccer (1989)(Codemasters).z80", 43850, 0xfa35acd5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec4soccersims2, Spec4soccersims2, Spec128)
STD_ROM_FN(Spec4soccersims2)

struct BurnDriver BurnSpec4soccersims2 = {
	"spec_4soccersims2", NULL, "spec_spec128", NULL, "1989",
	"4 Soccer Simulators - Indoor Soccer (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 4, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL, 0,
	SpectrumGetZipName, Spec4soccersims2RomInfo, Spec4soccersims2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 4 Soccer Simulators - Soccer Skills (128K)

static struct BurnRomInfo Spec4soccersims3RomDesc[] = {
	{ "4 Soccer Simulators - Soccer Skills (1989)(Codemasters).z80", 45681, 0x387466c5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec4soccersims3, Spec4soccersims3, Spec128)
STD_ROM_FN(Spec4soccersims3)

struct BurnDriver BurnSpec4soccersims3 = {
	"spec_4soccersims3", NULL, "spec_spec128", NULL, "1989",
	"4 Soccer Simulators - Soccer Skills (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL, 0,
	SpectrumGetZipName, Spec4soccersims3RomInfo, Spec4soccersims3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 4 Soccer Simulators - Street Soccer (128K)

static struct BurnRomInfo Spec4soccersims4RomDesc[] = {
	{ "4 Soccer Simulators - Street Soccer (1989)(Codemasters).z80", 45164, 0xb2e7be10, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec4soccersims4, Spec4soccersims4, Spec128)
STD_ROM_FN(Spec4soccersims4)

struct BurnDriver BurnSpec4soccersims4 = {
	"spec_4soccersims4", NULL, "spec_spec128", NULL, "1989",
	"4 Soccer Simulators - Street Soccer (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 4, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL, 0,
	SpectrumGetZipName, Spec4soccersims4RomInfo, Spec4soccersims4RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 911 TS - Tiger Shark (48K)

static struct BurnRomInfo Spec911tsRomDesc[] = {
	{ "911 TS (1985)(Elite Systems).tap", 47921, 0x303db330, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec911ts, Spec911ts, Spectrum)
STD_ROM_FN(Spec911ts)

struct BurnDriver BurnSpec911ts = {
	"spec_911ts", NULL, "spec_spectrum", NULL, "1985",
	"911 TS - Tiger Shark (48K)\0", "Z, X, O, and P to play", "Elite Systems", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, Spec911tsRomInfo, Spec911tsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Advanced Pinball Simulator (128K)

static struct BurnRomInfo SpecAdvancedpinballsimRomDesc[] = {
	{ "Advanced Pinball Simulator (1990)(Codemasters).tap", 51760, 0xfca7d79b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAdvancedpinballsim, SpecAdvancedpinballsim, Spec128)
STD_ROM_FN(SpecAdvancedpinballsim)

struct BurnDriver BurnSpecAdvancedpinballsim = {
	"spec_advancedpinballsim", NULL, "spec_spec128", NULL, "1990",
	"Advanced Pinball Simulator (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 3, HARDWARE_SPECTRUM, GBF_PINBALL, 0,
	SpectrumGetZipName, SpecAdvancedpinballsimRomInfo, SpecAdvancedpinballsimRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Afterburner (128K)

static struct BurnRomInfo SpecAfterburnerRomDesc[] = {
	{ "Afterburner (1989)(Activision).tap", 102813, 0xb40696fb, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAfterburner, SpecAfterburner, Spec128)
STD_ROM_FN(SpecAfterburner)

struct BurnDriver BurnSpecAfterburner = {
	"spec_afterburner", NULL, "spec_spec128", NULL, "1989",
	"Afterburner (128K)\0", NULL, "Activision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecAfterburnerRomInfo, SpecAfterburnerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Alien 8 (48K)

static struct BurnRomInfo SpecAlien8RomDesc[] = {
	{ "Alien 8 (1985)(Ultimate Play The Game).tap", 47264, 0x043d8dae, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAlien8, SpecAlien8, Spectrum)
STD_ROM_FN(SpecAlien8)

struct BurnDriver BurnSpecAlien8 = {
	"spec_alien8", NULL, "spec_spectrum", NULL, "1985",
	"Alien 8 (48K)\0", NULL, "Ultimate Play The Game", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecAlien8RomInfo, SpecAlien8RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Arcade Flight Simulator (128K)

static struct BurnRomInfo SpecArcadeflightsimRomDesc[] = {
	{ "Arcade Flight Simulator (1989)(Codemasters).tap", 48526, 0x834a6586, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecArcadeflightsim, SpecArcadeflightsim, Spec128)
STD_ROM_FN(SpecArcadeflightsim)

struct BurnDriver BurnSpecArcadeflightsim = {
	"spec_arcadeflightsim", NULL, "spec_spec128", NULL, "1989",
	"Arcade Flight Simulator (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecArcadeflightsimRomInfo, SpecArcadeflightsimRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Battle Ships (48K)

static struct BurnRomInfo SpecBattleshipsRomDesc[] = {
	{ "Battle Ships 48K (1987)(Hit-Pak).z80", 38283, 0xfb32ab1a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBattleships, SpecBattleships, Spectrum)
STD_ROM_FN(SpecBattleships)

struct BurnDriver BurnSpecBattleships = {
	"spec_battleships", NULL, "spec_spectrum", NULL, "1987",
	"Battle Ships (48K)\0", NULL, "Hit-Pak", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_STRATEGY, 0,
	SpectrumGetZipName, SpecBattleshipsRomInfo, SpecBattleshipsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bigfoot (48K)

static struct BurnRomInfo SpecBigfootRomDesc[] = {
	{ "Bigfoot (1988)(Codemasters).tap", 39134, 0x5f68a506, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBigfoot, SpecBigfoot, Spectrum)
STD_ROM_FN(SpecBigfoot)

struct BurnDriver BurnSpecBigfoot = {
	"spec_bigfoot", NULL, "spec_spectrum", NULL, "1988",
	"Bigfoot (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecBigfootRomInfo, SpecBigfootRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Big Nose's American Adventure (48K)

static struct BurnRomInfo SpecBignoseamericanadvRomDesc[] = {
	{ "Big Nose's American Adventure (1992)(Codemasters).tap", 47470, 0x64006917, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBignoseamericanadv, SpecBignoseamericanadv, Spectrum)
STD_ROM_FN(SpecBignoseamericanadv)

struct BurnDriver BurnSpecBignoseamericanadv = {
	"spec_bignoseamericanadv", NULL, "spec_spectrum", NULL, "1992",
	"Big Nose's American Adventure (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecBignoseamericanadvRomInfo, SpecBignoseamericanadvRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Big Trouble in Little China (48K)

static struct BurnRomInfo SpecBigtroubleinlittlechinaRomDesc[] = {
	{ "Big Trouble in Little China (1986)(Electric Dreams).tap", 51364, 0x5ba135d3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBigtroubleinlittlechina, SpecBigtroubleinlittlechina, Spectrum)
STD_ROM_FN(SpecBigtroubleinlittlechina)

struct BurnDriver BurnSpecBigtroubleinlittlechina = {
	"spec_bigtroubleinlittlechina", NULL, "spec_spectrum", NULL, "1986",
	"Big Trouble in Little China (48K)\0", NULL, "Electric Dreams", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecBigtroubleinlittlechinaRomInfo, SpecBigtroubleinlittlechinaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Blade Warrior (48K)

static struct BurnRomInfo SpecBladewarriorRomDesc[] = {
	{ "Blade Warrior (1988)(Codemasters).tap", 41844, 0xd8bff571, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBladewarrior, SpecBladewarrior, Spectrum)
STD_ROM_FN(SpecBladewarrior)

struct BurnDriver BurnSpecBladewarrior = {
	"spec_bladewarrior", NULL, "spec_spectrum", NULL, "1988",
	"Blade Warrior (48K)\0", "Select 'Sinclair Stick' to use controller", "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecBladewarriorRomInfo, SpecBladewarriorRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// BMX Simulator 2 - Dirt Biking (128K)

static struct BurnRomInfo SpecBmxsim21RomDesc[] = {
	{ "BMX Simulator 2 - Dirt Biking (1989)(Codemasters).z80", 48248, 0xab7d26d5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBmxsim21, SpecBmxsim21, Spec128)
STD_ROM_FN(SpecBmxsim21)

struct BurnDriver BurnSpecBmxsim21 = {
	"spec_bmxsim21", NULL, "spec_spec128", NULL, "1989",
	"BMX Simulator 2 - Dirt Biking (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 4, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecBmxsim21RomInfo, SpecBmxsim21RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecCursorKeysDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// BMX Simulator 2 - Quarry Racing (128K)

static struct BurnRomInfo SpecBmxsim22RomDesc[] = {
	{ "BMX Simulator 2 - Quarry Racing (1989)(Codemasters).z80", 48362, 0xd75318bf, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBmxsim22, SpecBmxsim22, Spec128)
STD_ROM_FN(SpecBmxsim22)

struct BurnDriver BurnSpecBmxsim22 = {
	"spec_bmxsim22", NULL, "spec_spec128", NULL, "1989",
	"BMX Simulator 2 - Quarry Racing (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 4, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecBmxsim22RomInfo, SpecBmxsim22RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecCursorKeysDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bounty Hunter (128K)

static struct BurnRomInfo SpecBountyhunterRomDesc[] = {
	{ "Bounty Hunter (1989)(Codemasters).tap", 47005, 0x91c9c3ba, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBountyhunter, SpecBountyhunter, Spec128)
STD_ROM_FN(SpecBountyhunter)

struct BurnDriver BurnSpecBountyhunter = {
	"spec_bountyhunter", NULL, "spec_spec128", NULL, "1989",
	"Bounty Hunter (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecBountyhunterRomInfo, SpecBountyhunterRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Jackal (48K)

static struct BurnRomInfo SpecJackalRomDesc[] = {
	{ "Jackal (1986)(Konami).tap", 26861, 0x954fae85, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecJackal, SpecJackal, Spectrum)
STD_ROM_FN(SpecJackal)

struct BurnDriver BurnSpecJackal = {
	"spec_jackal", NULL, "spec_spectrum", NULL, "1986",
	"Jackal (48K)\0", NULL, "Konami", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecJackalRomInfo, SpecJackalRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Laser Squad (48K)

static struct BurnRomInfo SpecLasersquadRomDesc[] = {
	{ "Laser Squad 48K (1988)(Blade Software).tap", 156755, 0x6a065f74, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecLasersquad, SpecLasersquad, Spectrum)
STD_ROM_FN(SpecLasersquad)

struct BurnDriver BurnSpecLasersquad = {
	"spec_lasersquad", NULL, "spec_spectrum", NULL, "1988",
	"Laser Squad (48K)\0", NULL, "Blade Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_STRATEGY, 0,
	SpectrumGetZipName, SpecLasersquadRomInfo, SpecLasersquadRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Road Race (48K)

static struct BurnRomInfo SpecRoadraceRomDesc[] = {
	{ "Road Race (1987)(Your Sinclair).tap", 46100, 0xa1daf3de, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecRoadrace, SpecRoadrace, Spectrum)
STD_ROM_FN(SpecRoadrace)

struct BurnDriver BurnSpecRoadrace = {
	"spec_roadrace", NULL, "spec_spectrum", NULL, "1987",
	"Road Race (48K)\0", NULL, "Your Sinclair", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecRoadraceRomInfo, SpecRoadraceRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rod-Land (128K)

static struct BurnRomInfo SpecRodlandRomDesc[] = {
	{ "Rod-Land (1991)(Storm).tap", 117150, 0xcd3ecd05, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecRodland, SpecRodland, Spec128)
STD_ROM_FN(SpecRodland)

struct BurnDriver BurnSpecRodland = {
	"spec_rodland", NULL, "spec_spec128", NULL, "1991",
	"Rod-Land (128K)\0", NULL, "Storm", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecRodlandRomInfo, SpecRodlandRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Shadow Of The Beast (128K)

static struct BurnRomInfo SpecShadowofthebeastRomDesc[] = {
	{ "Shadow Of The Beast (1990)(Gremlin Graphics).z80", 114104, 0x0507010d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecShadowofthebeast, SpecShadowofthebeast, Spec128)
STD_ROM_FN(SpecShadowofthebeast)

struct BurnDriver BurnSpecShadowofthebeast = {
	"spec_shadowofthebeast", NULL, "spec_spec128", NULL, "1990",
	"Shadow Of The Beast (128K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecShadowofthebeastRomInfo, SpecShadowofthebeastRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Space Crusade (128K)

static struct BurnRomInfo SpecSpacecrusadeRomDesc[] = {
	{ "Space Crusade (1992)(Gremlin Graphics).tap", 70110, 0xa46b2859, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSpacecrusade, SpecSpacecrusade, Spec128)
STD_ROM_FN(SpecSpacecrusade)

struct BurnDriver BurnSpecSpacecrusade = {
	"spec_spacecrusade", NULL, "spec_spec128", NULL, "1992",
	"Space Crusade (128K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_STRATEGY, 0,
	SpectrumGetZipName, SpecSpacecrusadeRomInfo, SpecSpacecrusadeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Super Car Trans Am - American Turbo King (128K)

static struct BurnRomInfo SpecSupercartransamRomDesc[] = {
	{ "Super Car Trans Am - American Turbo King (1989)(Virgin Mastertronic).tap", 43374, 0xeb2361c5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSupercartransam, SpecSupercartransam, Spec128)
STD_ROM_FN(SpecSupercartransam)

struct BurnDriver BurnSpecSupercartransam = {
	"spec_supercartransam", NULL, "spec_spec128", NULL, "1989",
	"Super Car Trans Am - American Turbo King (128K)\0", NULL, "Virgin Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecSupercartransamRomInfo, SpecSupercartransamRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sword Slayer (128K)

static struct BurnRomInfo SpecSwordslayerRomDesc[] = {
	{ "Sword Slayer (1988)(Players).tap", 102652, 0x0a989566, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSwordslayer, SpecSwordslayer, Spec128)
STD_ROM_FN(SpecSwordslayer)

struct BurnDriver BurnSpecSwordslayer = {
	"spec_swordslayer", NULL, "spec_spec128", NULL, "1988",
	"Sword Slayer (128K)\0", NULL, "Players Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpecSwordslayerRomInfo, SpecSwordslayerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 1st Division Manager (128K)

static struct BurnRomInfo Spec1stdivmanagerRomDesc[] = {
	{ "1st Division Manager (1991)(Codemasters).z80", 47446, 0x57fed4fa, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec1stdivmanager, Spec1stdivmanager, Spec128)
STD_ROM_FN(Spec1stdivmanager)

struct BurnDriver BurnSpec1stdivmanager = {
	"spec_1stdivmanager", NULL, "spec_spec128", NULL, "1991",
	"1st Division Manager (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_STRATEGY, 0,
	SpectrumGetZipName, Spec1stdivmanagerRomInfo, Spec1stdivmanagerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 750cc Grand Prix (128K)

static struct BurnRomInfo Spec750ccRomDesc[] = {
	{ "750cc Grand Prix (1991)(Code Masters).tap", 82394, 0xc9073683, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec750cc, Spec750cc, Spec128)
STD_ROM_FN(Spec750cc)

struct BurnDriver BurnSpec750cc = {
	"spec_750cc", NULL, "spec_spec128", NULL, "1991",
	"750cc Grand Prix (128K)\0", NULL, "Code Masters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, Spec750ccRomInfo, Spec750ccRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Billy the Kid (48K)

static struct BurnRomInfo SpecBillythekidRomDesc[] = {
	{ "Billy the Kid (1989)(Virgin Mastertronic).tap", 34510, 0xbc4e886c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBillythekid, SpecBillythekid, Spectrum)
STD_ROM_FN(SpecBillythekid)

struct BurnDriver BurnSpecBillythekid = {
	"spec_billythekid", NULL, "spec_spectrum", NULL, "1989",
	"Billy the Kid (48K)\0", "Press Space and Enter to select options", "Virgin Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecBillythekidRomInfo, SpecBillythekidRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Brainache (48K)

static struct BurnRomInfo SpecBrainacheRomDesc[] = {
	{ "Brainache (1987)(Codemasters).tap", 49898, 0x28592736, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBrainache, SpecBrainache, Spectrum)
STD_ROM_FN(SpecBrainache)

struct BurnDriver BurnSpecBrainache = {
	"spec_brainache", NULL, "spec_spectrum", NULL, "1987",
	"Brainache (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecBrainacheRomInfo, SpecBrainacheRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bronx Street Cop (128K)

static struct BurnRomInfo SpecBronxstcopRomDesc[] = {
	{ "Bronx Street Cop (1989)(Virgin Mastertronic).tap", 45318, 0x9884b2f8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBronxstcop, SpecBronxstcop, Spec128)
STD_ROM_FN(SpecBronxstcop)

struct BurnDriver BurnSpecBronxstcop = {
	"spec_bronxstcop", NULL, "spec_spec128", NULL, "1989",
	"Bronx Street Cop (128K)\0", NULL, "Virgin Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecBronxstcopRomInfo, SpecBronxstcopRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// By Fair Means...or Foul (48K)

static struct BurnRomInfo SpecByfairmeansorfoulRomDesc[] = {
	{ "By Fair Means Or Foul (1989)(Superior).z80", 44540, 0x8f8c6d0e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecByfairmeansorfoul, SpecByfairmeansorfoul, Spectrum)
STD_ROM_FN(SpecByfairmeansorfoul)

struct BurnDriver BurnSpecByfairmeansorfoul = {
	"spec_byfairmeansorfoul", NULL, "spec_spectrum", NULL, "1989",
	"By Fair Means...or Foul (48K)\0", NULL, "Superior", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpecByfairmeansorfoulRomInfo, SpecByfairmeansorfoulRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Captain Dynamo (128K)

static struct BurnRomInfo SpecCaptaindynamoRomDesc[] = {
	{ "Captain Dynamo (1992)(Codemasters).tap", 68500, 0x62855d75, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCaptaindynamo, SpecCaptaindynamo, Spec128)
STD_ROM_FN(SpecCaptaindynamo)

struct BurnDriver BurnSpecCaptaindynamo = {
	"spec_captaindynamo", NULL, "spec_spec128", NULL, "1992",
	"Captain Dynamo (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecCaptaindynamoRomInfo, SpecCaptaindynamoRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Cue Boy (128K)

static struct BurnRomInfo SpecCueboyRomDesc[] = {
	{ "Cue Boy (1992)(Codemasters).tap", 54475, 0x2b625a7e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCueboy, SpecCueboy, Spec128)
STD_ROM_FN(SpecCueboy)

struct BurnDriver BurnSpecCueboy = {
	"spec_cueboy", NULL, "spec_spec128", NULL, "1992",
	"Cue Boy (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecCueboyRomInfo, SpecCueboyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Death Stalker (128K)

static struct BurnRomInfo SpecDeathstalkerRomDesc[] = {
	{ "Death Stalker (1988)(Codemasters).tap", 50809, 0x95e38b12, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDeathstalker, SpecDeathstalker, Spec128)
STD_ROM_FN(SpecDeathstalker)

struct BurnDriver BurnSpecDeathstalker = {
	"spec_deathstalker", NULL, "spec_spec128", NULL, "1988",
	"Death Stalker (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecDeathstalkerRomInfo, SpecDeathstalkerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dizzy 3 & a Half (48K)

static struct BurnRomInfo SpecDizzy3andahalfRomDesc[] = {
	{ "Dizzy 3 & a Half (1988)(Codemasters).tap", 25241, 0x8a82c837, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDizzy3andahalf, SpecDizzy3andahalf, Spectrum)
STD_ROM_FN(SpecDizzy3andahalf)

struct BurnDriver BurnSpecDizzy3andahalf = {
	"spec_dizzy3andahalf", NULL, "spec_spectrum", NULL, "1988",
	"Dizzy III (Three).. and a Half (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecDizzy3andahalfRomInfo, SpecDizzy3andahalfRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// DJ Puff's Volcanic Capers (128K)

static struct BurnRomInfo SpecDjpuffRomDesc[] = {
	{ "DJ Puff (1992)(Codemasters).z80", 66937, 0x38340567, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDjpuff, SpecDjpuff, Spec128)
STD_ROM_FN(SpecDjpuff)

struct BurnDriver BurnSpecDjpuff = {
	"spec_djpuff", NULL, "spec_spec128", NULL, "1992",
	"DJ Puff's Volcanic Capers (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecDjpuffRomInfo, SpecDjpuffRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// F-16 Fighting Falcon (128K)

static struct BurnRomInfo SpecF16falconRomDesc[] = {
	{ "F-16 Fighting Falcon (1990)(Virgin Mastertronic).tap", 39117, 0x2071bb7c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecF16falcon, SpecF16falcon, Spec128)
STD_ROM_FN(SpecF16falcon)

struct BurnDriver BurnSpecF16falcon = {
	"spec_f16falcon", NULL, "spec_spec128", NULL, "1990",
	"F-16 Fighting Falcon (128K)\0", NULL, "Virgin Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecF16falconRomInfo, SpecF16falconRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Fairlight - A Prelude (128K)

static struct BurnRomInfo SpecFairlight128RomDesc[] = {
	{ "Fairlight - A Prelude 128K (1985)(The Edge).z80", 70829, 0x5fdc3ff9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFairlight128, SpecFairlight128, Spec128)
STD_ROM_FN(SpecFairlight128)

struct BurnDriver BurnSpecFairlight128 = {
	"spec_fairlight128", NULL, "spec_spec128", NULL, "1985",
	"Fairlight - A Prelude (128K)\0", "Press '9' to set controller", "The Edge", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecFairlight128RomInfo, SpecFairlight128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Fairlight - A Prelude (48K)

static struct BurnRomInfo SpecFairlight48RomDesc[] = {
	{ "Fairlight - A Prelude 48K (1985)(The Edge).tap", 50258, 0x12342db8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFairlight48, SpecFairlight48, Spectrum)
STD_ROM_FN(SpecFairlight48)

struct BurnDriver BurnSpecFairlight48 = {
	"spec_fairlight48", "spec_fairlight128", "spec_spectrum", NULL, "1985",
	"Fairlight - A Prelude (48K)\0", "Press '9' to set controller", "The Edge", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecFairlight48RomInfo, SpecFairlight48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Fairlight II (128K)

static struct BurnRomInfo SpecFairlight2RomDesc[] = {
	{ "Fairlight II 128K (1986)(The Edge).tap", 97674, 0x5fac6ab2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFairlight2, SpecFairlight2, Spec128)
STD_ROM_FN(SpecFairlight2)

struct BurnDriver BurnSpecFairlight2 = {
	"spec_fairlight2", NULL, "spec_spec128", NULL, "1986",
	"Fairlight II - A Trail of Darkness (128K)\0", "Press '9' to set controller", "The Edge", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecFairlight2RomInfo, SpecFairlight2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Fantastic Voyage

static struct BurnRomInfo SpecFantvoyageRomDesc[] = {
	{ "Fantastic Voyage (1984)(Quicksilva Ltd.).tap", 42032, 0x8c6d5727, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFantvoyage, SpecFantvoyage, Spectrum)
STD_ROM_FN(SpecFantvoyage)

struct BurnDriver BurnSpecFantvoyage = {
	"spec_fantvoyage", NULL, "spec_spectrum", NULL, "1984",
	"Fantastic Voyage (48K)\0", NULL, "Quicksilva Ltd.", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE, 0,
	SpectrumGetZipName, SpecFantvoyageRomInfo, SpecFantvoyageRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Frankenstein Jnr (48K)

static struct BurnRomInfo SpecFrankensteinjrRomDesc[] = {
	{ "Frankenstein Jnr (1990)(Codemasters).tap", 43412, 0xf549d96d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFrankensteinjr, SpecFrankensteinjr, Spectrum)
STD_ROM_FN(SpecFrankensteinjr)

struct BurnDriver BurnSpecFrankensteinjr = {
	"spec_frankensteinjr", NULL, "spec_spectrum", NULL, "1990",
	"Frankenstein Jnr (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecFrankensteinjrRomInfo, SpecFrankensteinjrRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Freddy Hardest in South Manhattan (Spanish) (48K)

static struct BurnRomInfo SpecFreddyhardestsouthmanRomDesc[] = {
	{ "Freddy Hardest in South Manhattan (1989)(Dinamic Software).tap", 61774, 0xffbc6590, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFreddyhardestsouthman, SpecFreddyhardestsouthman, Spectrum)
STD_ROM_FN(SpecFreddyhardestsouthman)

struct BurnDriver BurnSpecFreddyhardestsouthman = {
	"spec_freddyhardestsouthman", NULL, "spec_spectrum", NULL, "1989",
	"Freddy Hardest in South Manhattan (Spanish) (48K)\0", "Set 'S - Sinclair' to use controller", "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecFreddyhardestsouthmanRomInfo, SpecFreddyhardestsouthmanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Fruit Machine Simulator - Cash Bash (48K)

static struct BurnRomInfo SpecFruitmachinesim1RomDesc[] = {
	{ "Fruit Machine Simulator - Cash Bash (1987)(Codemasters).tap", 47967, 0x0f58266f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFruitmachinesim1, SpecFruitmachinesim1, Spectrum)
STD_ROM_FN(SpecFruitmachinesim1)

struct BurnDriver BurnSpecFruitmachinesim1 = {
	"spec_fruitmachinesim1", NULL, "spec_spectrum", NULL, "1987",
	"Fruit Machine Simulator - Cash Bash (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_CASINO, 0,
	SpectrumGetZipName, SpecFruitmachinesim1RomInfo, SpecFruitmachinesim1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Fruit Machine Simulator 2 - Mega Trek (128K)

static struct BurnRomInfo SpecFruitmachinesim2RomDesc[] = {
	{ "Fruit Machine Simulator 2 - Mega Trek (1990)(Codemasters).tap", 49545, 0xbb3c551e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFruitmachinesim2, SpecFruitmachinesim2, Spec128)
STD_ROM_FN(SpecFruitmachinesim2)

struct BurnDriver BurnSpecFruitmachinesim2 = {
	"spec_fruitmachinesim2", NULL, "spec_spec128", NULL, "1990",
	"Fruit Machine Simulator 2 - Mega Trek (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_CASINO, 0,
	SpectrumGetZipName, SpecFruitmachinesim2RomInfo, SpecFruitmachinesim2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ghost Hunters (48K)

static struct BurnRomInfo SpecGhosthuntersRomDesc[] = {
	{ "Ghost Hunters (1987)(Codemasters).tap", 48896, 0xdfa7ffcb, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGhosthunters, SpecGhosthunters, Spectrum)
STD_ROM_FN(SpecGhosthunters)

struct BurnDriver BurnSpecGhosthunters = {
	"spec_ghosthunters", NULL, "spec_spectrum", NULL, "1987",
	"Ghost Hunters (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecGhosthuntersRomInfo, SpecGhosthuntersRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Grand Prix Simulator (48K)

static struct BurnRomInfo SpecGpsimulator1RomDesc[] = {
	{ "Grand Prix Simulator (1987)(Codemasters).tap", 55033, 0x379b30e6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGpsimulator1, SpecGpsimulator1, Spec128)
STD_ROM_FN(SpecGpsimulator1)

struct BurnDriver BurnSpecGpsimulator1 = {
	"spec_gpsimulator1", NULL, "spec_spec128", NULL, "1987",
	"Grand Prix Simulator (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecGpsimulator1RomInfo, SpecGpsimulator1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Grand Prix Simulator 2 (128K)

static struct BurnRomInfo SpecGpsimii128RomDesc[] = {
	{ "Grand Prix Simulator 2 (1989)(Codemasters)(48K-128K).tap", 53178, 0x2e2014ef, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGpsimii128, SpecGpsimii128, Spec128)
STD_ROM_FN(SpecGpsimii128)

struct BurnDriver BurnSpecGpsimii128 = {
	"spec_gpsimii128", NULL, "spec_spec128", NULL, "1989",
	"Grand Prix Simulator 2 (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 3, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecGpsimii128RomInfo, SpecGpsimii128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Grand Prix Simulator 2 (48K)

static struct BurnRomInfo SpecGpsimii48RomDesc[] = {
	{ "Grand Prix Simulator 2 (1989)(Codemasters)(48K-128K).tap", 53178, 0x2e2014ef, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGpsimii48, SpecGpsimii48, Spectrum)
STD_ROM_FN(SpecGpsimii48)

struct BurnDriver BurnSpecGpsimii48 = {
	"spec_gpsimii48", "spec_gpsimii128", "spec_spectrum", NULL, "1989",
	"Grand Prix Simulator 2 (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 3, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecGpsimii48RomInfo, SpecGpsimii48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Grell and Falla (128K)

static struct BurnRomInfo SpecGrellandfallaRomDesc[] = {
	{ "Grell and Falla (1987)(Codemasters).z80", 92473, 0x8929d289, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGrellandfalla, SpecGrellandfalla, Spec128)
STD_ROM_FN(SpecGrellandfalla)

struct BurnDriver BurnSpecGrellandfalla = {
	"spec_grellandfalla", NULL, "spec_spec128", NULL, "1992",
	"Grell and Falla (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecGrellandfallaRomInfo, SpecGrellandfallaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hit Squad, The (48K)

static struct BurnRomInfo SpecHitsquadRomDesc[] = {
	{ "Hit Squad, The (1988)(Codemasters).z80", 40124, 0x8afa17ab, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHitsquad, SpecHitsquad, Spectrum)
STD_ROM_FN(SpecHitsquad)

struct BurnDriver BurnSpecHitsquad = {
	"spec_hitsquad", NULL, "spec_spectrum", NULL, "1988",
	"Hit Squad, The (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecHitsquadRomInfo, SpecHitsquadRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hundra (48K)

static struct BurnRomInfo SpecHundraRomDesc[] = {
	{ "Hundra (1988)(Mastertronic).tap", 48229, 0xeeb32b33, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHundra, SpecHundra, Spec128)
STD_ROM_FN(SpecHundra)

struct BurnDriver BurnSpecHundra = {
	"spec_hundra", NULL, "spec_spec128", NULL, "1988",
	"Hundra (48K)\0", NULL, "Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecHundraRomInfo, SpecHundraRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// International 3D Tennis (128K)

static struct BurnRomInfo SpecInt3dtennisRomDesc[] = {
	{ "International 3D Tennis (1990)(Palace).tap", 85714, 0x6d60e471, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecInt3dtennis, SpecInt3dtennis, Spec128)
STD_ROM_FN(SpecInt3dtennis)

struct BurnDriver BurnSpecInt3dtennis = {
	"spec_int3dtennis", NULL, "spec_spec128", NULL, "1990",
	"International 3D Tennis (128K)\0", NULL, "Palace Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecInt3dtennisRomInfo, SpecInt3dtennisRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// International Rugby (48K)

static struct BurnRomInfo SpecIntrugbyRomDesc[] = {
	{ "International Rugby (1987)(Codemasters).tap", 40161, 0xbaf57b6c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecIntrugby, SpecIntrugby, Spectrum)
STD_ROM_FN(SpecIntrugby)

struct BurnDriver BurnSpecIntrugby = {
	"spec_intrugby", NULL, "spec_spectrum", NULL, "1987",
	"International Rugby (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecIntrugbyRomInfo, SpecIntrugbyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// International Speedway (128K)

static struct BurnRomInfo SpecIntspeedwayRomDesc[] = {
	{ "International Speedway (1988)(Silverbird).z80", 35832, 0x7d1e8f3a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecIntspeedway, SpecIntspeedway, Spec128)
STD_ROM_FN(SpecIntspeedway)

struct BurnDriver BurnSpecIntspeedway = {
	"spec_intspeedway", NULL, "spec_spec128", NULL, "1988",
	"International Speedway (128K)\0", NULL, "Silverbird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecIntspeedwayRomInfo, SpecIntspeedwayRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Italian Supercar (128K)

static struct BurnRomInfo SpecItaliansupercarRomDesc[] = {
	{ "Italian Supercar (1990)(Codemasters).tap", 48649, 0x0e28c70e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecItaliansupercar, SpecItaliansupercar, Spec128)
STD_ROM_FN(SpecItaliansupercar)

struct BurnDriver BurnSpecItaliansupercar = {
	"spec_italiansupercar", NULL, "spec_spec128", NULL, "1990",
	"Italian Supercar (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecItaliansupercarRomInfo, SpecItaliansupercarRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Jet Bike Simulator - Lakes Standard (48K)

static struct BurnRomInfo SpecJetbikesim1RomDesc[] = {
	{ "Jet Bike Simulator - Lakes Standard (1988)(Codemasters).z80", 37457, 0x4732d30c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecJetbikesim1, SpecJetbikesim1, Spectrum)
STD_ROM_FN(SpecJetbikesim1)

struct BurnDriver BurnSpecJetbikesim1 = {
	"spec_jetbikesim1", NULL, "spec_spectrum", NULL, "1988",
	"Jet Bike Simulator - Lakes Standard (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecJetbikesim1RomInfo, SpecJetbikesim1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Jet Bike Simulator - Coasts Standard (48K)

static struct BurnRomInfo SpecJetbikesim2RomDesc[] = {
	{ "Jet Bike Simulator - Coasts Standard (1988)(Codemasters).z80", 37757, 0x2311bdb7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecJetbikesim2, SpecJetbikesim2, Spectrum)
STD_ROM_FN(SpecJetbikesim2)

struct BurnDriver BurnSpecJetbikesim2 = {
	"spec_jetbikesim2", NULL, "spec_spectrum", NULL, "1988",
	"Jet Bike Simulator - Coasts Standard (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecJetbikesim2RomInfo, SpecJetbikesim2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Jet Bike Simulator - Docks Standard (48K)

static struct BurnRomInfo SpecJetbikesim3RomDesc[] = {
	{ "Jet Bike Simulator - Docks Standard (1988)(Codemasters).z80", 37266, 0x94a6c7d8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecJetbikesim3, SpecJetbikesim3, Spectrum)
STD_ROM_FN(SpecJetbikesim3)

struct BurnDriver BurnSpecJetbikesim3 = {
	"spec_jetbikesim3", NULL, "spec_spectrum", NULL, "1988",
	"Jet Bike Simulator - Docks Standard (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecJetbikesim3RomInfo, SpecJetbikesim3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Jet Bike Simulator - Lakes Expert (48K)

static struct BurnRomInfo SpecJetbikesim4RomDesc[] = {
	{ "Jet Bike Simulator - Lakes Expert (1988)(Codemasters).z80", 37906, 0x3681dac9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecJetbikesim4, SpecJetbikesim4, Spectrum)
STD_ROM_FN(SpecJetbikesim4)

struct BurnDriver BurnSpecJetbikesim4 = {
	"spec_jetbikesim4", "spec_jetbikesim1", "spec_spectrum", NULL, "1988",
	"Jet Bike Simulator - Lakes Expert (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecJetbikesim4RomInfo, SpecJetbikesim4RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Jet Bike Simulator - Coasts Expert (48K)

static struct BurnRomInfo SpecJetbikesim5RomDesc[] = {
	{ "Jet Bike Simulator - Coasts Expert (1988)(Codemasters).z80", 38206, 0xc3d570e4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecJetbikesim5, SpecJetbikesim5, Spectrum)
STD_ROM_FN(SpecJetbikesim5)

struct BurnDriver BurnSpecJetbikesim5 = {
	"spec_jetbikesim5", "spec_jetbikesim2", "spec_spectrum", NULL, "1988",
	"Jet Bike Simulator - Coasts Expert (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecJetbikesim5RomInfo, SpecJetbikesim5RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Jet Bike Simulator - Docks Expert (48K)

static struct BurnRomInfo SpecJetbikesim6RomDesc[] = {
	{ "Jet Bike Simulator - Docks Expert (1988)(Codemasters).z80", 37673, 0x8e1e9ee5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecJetbikesim6, SpecJetbikesim6, Spectrum)
STD_ROM_FN(SpecJetbikesim6)

struct BurnDriver BurnSpecJetbikesim6 = {
	"spec_jetbikesim6", "spec_jetbikesim3", "spec_spectrum", NULL, "1988",
	"Jet Bike Simulator - Docks Expert (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecJetbikesim6RomInfo, SpecJetbikesim6RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Jungle Warfare (48K)

static struct BurnRomInfo SpecJunglewarfareRomDesc[] = {
	{ "Jungle Warfare (1989)(Codemasters).tap", 43631, 0xeb977230, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecJunglewarfare, SpecJunglewarfare, Spec128)
STD_ROM_FN(SpecJunglewarfare)

struct BurnDriver BurnSpecJunglewarfare = {
	"spec_junglewarfare", NULL, "spec_spec128", NULL, "1989",
	"Jungle Warfare (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecJunglewarfareRomInfo, SpecJunglewarfareRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Kamikaze (128K)

static struct BurnRomInfo SpecKamikazeRomDesc[] = {
	{ "Kamikaze (1991)(Codemasters).tap", 47767, 0xd35b3009, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecKamikaze, SpecKamikaze, Spec128)
STD_ROM_FN(SpecKamikaze)

struct BurnDriver BurnSpecKamikaze = {
	"spec_kamikaze", NULL, "spec_spec128", NULL, "1991",
	"Kamikaze (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecKamikazeRomInfo, SpecKamikazeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// KGB Super Spy (48K)

static struct BurnRomInfo SpecKgbsuperspyRomDesc[] = {
	{ "KGB Super Spy (1989)(Codemasters).tap", 45049, 0xa879a856, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecKgbsuperspy, SpecKgbsuperspy, Spectrum)
STD_ROM_FN(SpecKgbsuperspy)

struct BurnDriver BurnSpecKgbsuperspy = {
	"spec_kgbsuperspy", NULL, "spec_spectrum", NULL, "1989",
	"KGB Super Spy (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecKgbsuperspyRomInfo, SpecKgbsuperspyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Escape from the Planet of the Robot Monsters (128K)

static struct BurnRomInfo SpecEscplanetrobotmonRomDesc[] = {
	{ "Escape from the Planet of the Robot Monsters (1990)(Domark).tap", 111578, 0x9f95584c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecEscplanetrobotmon, SpecEscplanetrobotmon, Spec128)
STD_ROM_FN(SpecEscplanetrobotmon)

struct BurnDriver BurnSpecEscplanetrobotmon = {
	"spec_escplanetrobotmon", NULL, "spec_spec128", NULL, "1990",
	"Escape from the Planet of the Robot Monsters (128K)\0", NULL, "Domark", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecEscplanetrobotmonRomInfo, SpecEscplanetrobotmonRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Gauntlet II (128K)

static struct BurnRomInfo SpecGauntlet2RomDesc[] = {
	{ "Gauntlet II (1987)(U.S. Gold).tap", 135163, 0x400e869d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGauntlet2, SpecGauntlet2, Spec128)
STD_ROM_FN(SpecGauntlet2)

struct BurnDriver BurnSpecGauntlet2 = {
	"spec_gauntlet2", NULL, "spec_spec128", NULL, "1987",
	"Gauntlet II (128K)\0", "In menu use Cursor Keys 5-6-7-8 + SPACE", "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_MAZE | GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecGauntlet2RomInfo, SpecGauntlet2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Last Vampire, The (48K)

static struct BurnRomInfo SpecLastvampireRomDesc[] = {
	{ "Last Vampire, The (1990)(Atlantis).tap", 50418, 0x9c48dd1a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecLastvampire, SpecLastvampire, Spectrum)
STD_ROM_FN(SpecLastvampire)

struct BurnDriver BurnSpecLastvampire = {
	"spec_lastvampire", NULL, "spec_spectrum", NULL, "1990",
	"Last Vampire, The (48K)\0", NULL, "Atlantis Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecLastvampireRomInfo, SpecLastvampireRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Michel Futbol Master - Championship (Spanish) (48K)

static struct BurnRomInfo SpecMichelfutbolchampRomDesc[] = {
	{ "Michel Futbol Master (1989)(Dinamic).tap", 42768, 0x38824765, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMichelfutbolchamp, SpecMichelfutbolchamp, Spectrum)
STD_ROM_FN(SpecMichelfutbolchamp)

struct BurnDriver BurnSpecMichelfutbolchamp = {
	"spec_michelfutbolchamp", NULL, "spec_spectrum", NULL, "1989",
	"Michel Futbol Master - Championship (Spanish) (48K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL, 0,
	SpectrumGetZipName, SpecMichelfutbolchampRomInfo, SpecMichelfutbolchampRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Michel Futbol Master - Super Skills (Spanish) (48K)

static struct BurnRomInfo SpecMichelfutbolskillsRomDesc[] = {
	{ "Michel Super Skills (1989)(Dinamic).tap", 43013, 0x035d9ad3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMichelfutbolskills, SpecMichelfutbolskills, Spectrum)
STD_ROM_FN(SpecMichelfutbolskills)

struct BurnDriver BurnSpecMichelfutbolskills = {
	"spec_michelfutbolskills", NULL, "spec_spectrum", NULL, "1989",
	"Michel Futbol Master - Super Skills (Spanish) (48K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL, 0,
	SpectrumGetZipName, SpecMichelfutbolskillsRomInfo, SpecMichelfutbolskillsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// NARC (128K)

static struct BurnRomInfo SpecNarcRomDesc[] = {
	{ "NARC (1990)(Ocean).tap", 356350, 0x3c9ccc9c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecNarc, SpecNarc, Spec128)
STD_ROM_FN(SpecNarc)

struct BurnDriver BurnSpecNarc = {
	"spec_narc", NULL, "spec_spec128", NULL, "1990",
	"NARC (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecNarcRomInfo, SpecNarcRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Simpsons - Bart vs. the Space Mutants, The (128K)

static struct BurnRomInfo SpecSimpsonsbartvsspacemutRomDesc[] = {
	{ "Simpsons - Bart vs. the Space Mutants, The (1991)(Ocean).tap", 351987, 0x382137b2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSimpsonsbartvsspacemut, SpecSimpsonsbartvsspacemut, Spec128)
STD_ROM_FN(SpecSimpsonsbartvsspacemut)

struct BurnDriver BurnSpecSimpsonsbartvsspacemut = {
	"spec_simpsonsbartvsspacemut", NULL, "spec_spec128", NULL, "1991",
	"Simpsons - Bart vs. the Space Mutants, The (128K)\0", "Press '0' to start the game", "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecSimpsonsbartvsspacemutRomInfo, SpecSimpsonsbartvsspacemutRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Skateball (48K)

static struct BurnRomInfo SpecSkateballRomDesc[] = {
	{ "Skateball (1988)(Ubi Soft).tap", 49399, 0xe816c833, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSkateball, SpecSkateball, Spectrum)
STD_ROM_FN(SpecSkateball)

struct BurnDriver BurnSpecSkateball = {
	"spec_skateball", NULL, "spec_spectrum", NULL, "1988",
	"Skateball (48K)\0", NULL, "Ubi Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecSkateballRomInfo, SpecSkateballRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sly Spy - Secret Agent (128K)

static struct BurnRomInfo SpecSlyspyRomDesc[] = {
	{ "Sly Spy - Secret Agent (1990)(Ocean).tap", 170067, 0x1276744e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSlyspy, SpecSlyspy, Spec128)
STD_ROM_FN(SpecSlyspy)

struct BurnDriver BurnSpecSlyspy = {
	"spec_slyspy", NULL, "spec_spec128", NULL, "1990",
	"Sly Spy - Secret Agent (128K)\0", "Select 'Interface II' to use controller", "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecSlyspyRomInfo, SpecSlyspyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Steg the Slug (128K)

static struct BurnRomInfo SpecStegRomDesc[] = {
	{ "Steg (1992)(Codemasters).tap", 51271, 0x15a4d49e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSteg, SpecSteg, Spec128)
STD_ROM_FN(SpecSteg)

struct BurnDriver BurnSpecSteg = {
	"spec_steg", NULL, "spec_spec128", NULL, "1992",
	"Steg the Slug (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecStegRomInfo, SpecStegRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Toobin' (128K)

static struct BurnRomInfo SpecToobinRomDesc[] = {
	{ "Toobin' (1989)(Domark).tap", 53121, 0xa2acce35, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecToobin, SpecToobin, Spec128)
STD_ROM_FN(SpecToobin)

struct BurnDriver BurnSpecToobin = {
	"spec_toobin", NULL, "spec_spec128", NULL, "1989",
	"Toobin' (128K)\0", NULL, "Domark", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecToobinRomInfo, SpecToobinRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// V (Visitors) (48K)

static struct BurnRomInfo SpecVoceanRomDesc[] = {
	{ "V (1986)(Ocean).tap", 56344, 0x07f2dfd6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecVocean, SpecVocean, Spectrum)
STD_ROM_FN(SpecVocean)

struct BurnDriver BurnSpecVocean = {
	"spec_vocean", NULL, "spec_spectrum", NULL, "1986",
	"V (Visitors) (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecVoceanRomInfo, SpecVoceanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Where Time Stood Still (128K)

static struct BurnRomInfo SpecWheretimestoodstillRomDesc[] = {
	{ "Where Time Stood Still (1988)(Ocean).tap", 110751, 0xfa89cab1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecWheretimestoodstill, SpecWheretimestoodstill, Spec128)
STD_ROM_FN(SpecWheretimestoodstill)

struct BurnDriver BurnSpecWheretimestoodstill = {
	"spec_wheretimestoodstill", NULL, "spec_spec128", NULL, "1988",
	"Where Time Stood Still (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecWheretimestoodstillRomInfo, SpecWheretimestoodstillRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Wizball (128K)

static struct BurnRomInfo SpecWizballRomDesc[] = {
	{ "Wizball (1987)(Ocean).tap", 34256, 0x70cf4de6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecWizball, SpecWizball, Spec128)
STD_ROM_FN(SpecWizball)

struct BurnDriver BurnSpecWizball = {
	"spec_wizball", NULL, "spec_spec128", NULL, "1987",
	"Wizball (128K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecWizballRomInfo, SpecWizballRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Aaargh! (48K)

static struct BurnRomInfo SpecAaarghRomDesc[] = {
	{ "Aaargh! (1989)(Binary Design).tap", 96865, 0x11417dd7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAaargh, SpecAaargh, Spectrum)
STD_ROM_FN(SpecAaargh)

struct BurnDriver BurnSpecAaargh = {
	"spec_aaargh", NULL, "spec_spectrum", NULL, "1989",
	"Aaargh! (48K)\0", NULL, "Binary Design", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecAaarghRomInfo, SpecAaarghRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Agent Orange (48K)

static struct BurnRomInfo SpecAgentorangeRomDesc[] = {
	{ "Agent Orange (1987)(Icon Design).tap", 43566, 0x112aedd9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAgentorange, SpecAgentorange, Spectrum)
STD_ROM_FN(SpecAgentorange)

struct BurnDriver BurnSpecAgentorange = {
	"spec_agentorange", NULL, "spec_spectrum", NULL, "1987",
	"Agent Orange (48K)\0", NULL, "Icon Design", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecAgentorangeRomInfo, SpecAgentorangeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ah Diddums (48K)

static struct BurnRomInfo SpecAhdiddumsRomDesc[] = {
	{ "Ah Diddums (1983)(Imagine).tap", 8490, 0xe5b88c3b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAhdiddums, SpecAhdiddums, Spec128)
STD_ROM_FN(SpecAhdiddums)

struct BurnDriver BurnSpecAhdiddums = {
	"spec_ahdiddums", NULL, "spec_spec128", NULL, "1983",
	"Ah Diddums (48K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecAhdiddumsRomInfo, SpecAhdiddumsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Alien Evolution (128K)

static struct BurnRomInfo SpecAlienevolutionRomDesc[] = {
	{ "Alien Evolution (1987)(Gremlin).tap", 49510, 0xa16b45f5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAlienevolution, SpecAlienevolution, Spec128)
STD_ROM_FN(SpecAlienevolution)

struct BurnDriver BurnSpecAlienevolution = {
	"spec_alienevolution", NULL, "spec_spec128", NULL, "1987",
	"Alien Evolution (128K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecAlienevolutionRomInfo, SpecAlienevolutionRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Aqua Plane (48K)

static struct BurnRomInfo SpecAquaplaneRomDesc[] = {
	{ "Aqua Plane (1983)(Quicksilva).tap", 13553, 0x3ef11315, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAquaplane, SpecAquaplane, Spectrum)
STD_ROM_FN(SpecAquaplane)

struct BurnDriver BurnSpecAquaplane = {
	"spec_aquaplane", NULL, "spec_spectrum", NULL, "1983",
	"Aqua Plane (48K)\0", "Press ENTER to skip 'loading screen'", "Quicksilva", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecAquaplaneRomInfo, SpecAquaplaneRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Atom Ant (128K)

static struct BurnRomInfo SpecAtomantRomDesc[] = {
	{ "Atom Ant (1990)(Hi-Tec).tap", 61884, 0xbb656d10, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAtomant, SpecAtomant, Spec128)
STD_ROM_FN(SpecAtomant)

struct BurnDriver BurnSpecAtomant = {
	"spec_atomant", NULL, "spec_spec128", NULL, "1990",
	"Atom Ant (128K)\0", NULL, "Hi-Tec Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecAtomantRomInfo, SpecAtomantRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// AutoCrash (Spanish) (128K)

static struct BurnRomInfo SpecAutocrashRomDesc[] = {
	{ "Autocrash (1991)(Zigurat).tap", 49550, 0x02fe192c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAutocrash, SpecAutocrash, Spec128)
STD_ROM_FN(SpecAutocrash)

struct BurnDriver BurnSpecAutocrash = {
	"spec_autocrash", NULL, "spec_spec128", NULL, "1991",
	"AutoCrash (Spanish) (128K)\0", NULL, "Zigurat Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecAutocrashRomInfo, SpecAutocrashRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// B.C. Bill (48K)

static struct BurnRomInfo SpecBcbillRomDesc[] = {
	{ "B.C. Bill (1984)(Imagine).tap", 30496, 0x61b2d9bf, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBcbill, SpecBcbill, Spectrum)
STD_ROM_FN(SpecBcbill)

struct BurnDriver BurnSpecBcbill = {
	"spec_bcbill", NULL, "spec_spectrum", NULL, "1984",
	"B.C. Bill (48K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecBcbillRomInfo, SpecBcbillRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// BC's Quest for Tires (48K)

static struct BurnRomInfo SpecBcquesttiresRomDesc[] = {
	{ "BC's Quest for Tires (1985)(Software Projects).tap", 43450, 0xfcb53b87, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBcquesttires, SpecBcquesttires, Spectrum)
STD_ROM_FN(SpecBcquesttires)

struct BurnDriver BurnSpecBcquesttires = {
	"spec_bcquesttires", NULL, "spec_spectrum", NULL, "1985",
	"BC's Quest for Tires (48K)\0", "Press 'P' for control options", "Software Projects", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecBcquesttiresRomInfo, SpecBcquesttiresRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Blob the Cop (48K)

static struct BurnRomInfo SpecBlobthecopRomDesc[] = {
	{ "Blob the Cop (1989)(Players Software).tap", 42196, 0x09b3e306, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBlobthecop, SpecBlobthecop, Spectrum)
STD_ROM_FN(SpecBlobthecop)

struct BurnDriver BurnSpecBlobthecop = {
	"spec_blobthecop", NULL, "spec_spectrum", NULL, "1989",
	"Blob the Cop (48K)\0", "ALT + SPACE for control options", "Players Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecBlobthecopRomInfo, SpecBlobthecopRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bubbler (48K)

static struct BurnRomInfo SpecBubblerRomDesc[] = {
	{ "Bubbler (1987)(Ultimate Play The Game).tap", 41660, 0x8e3e4b5a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBubbler, SpecBubbler, Spectrum)
STD_ROM_FN(SpecBubbler)

struct BurnDriver BurnSpecBubbler = {
	"spec_bubbler", NULL, "spec_spectrum", NULL, "1987",
	"Bubbler (48K)\0", "Press '1' for P1 options and '2' for P2 options", "Ultimate Play The Game", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecBubblerRomInfo, SpecBubblerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Buggy Ranger (48K)

static struct BurnRomInfo SpecBuggyrangerRomDesc[] = {
	{ "Buggy Ranger (1990)(Dinamic).tap", 50964, 0xc7f9d095, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBuggyranger, SpecBuggyranger, Spectrum)
STD_ROM_FN(SpecBuggyranger)

struct BurnDriver BurnSpecBuggyranger = {
	"spec_buggyranger", NULL, "spec_spectrum", NULL, "1990",
	"Buggy Ranger (48K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecBuggyrangerRomInfo, SpecBuggyrangerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Chain Reaction (128K)

static struct BurnRomInfo SpecChainreactionRomDesc[] = {
	{ "Chain Reaction (1987)(Durell).tap", 51862, 0x42c4cd17, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecChainreaction, SpecChainreaction, Spec128)
STD_ROM_FN(SpecChainreaction)

struct BurnDriver BurnSpecChainreaction = {
	"spec_chainreaction", NULL, "spec_spec128", NULL, "1987",
	"Chain Reaction (128K)\0", NULL, "Durell Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_ADV, 0,
	SpectrumGetZipName, SpecChainreactionRomInfo, SpecChainreactionRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Chickin Chase (Il Galletto) (Italian) (48K)

static struct BurnRomInfo SpecChickinchaseRomDesc[] = {
	{ "Chickin Chase (1985)(Firebird).tap", 50446, 0x5762f239, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecChickinchase, SpecChickinchase, Spectrum)
STD_ROM_FN(SpecChickinchase)

struct BurnDriver BurnSpecChickinchase = {
	"spec_chickinchase", NULL, "spec_spectrum", NULL, "1985",
	"Il Galletto - Chickin Chase (Italian) (48K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecChickinchaseRomInfo, SpecChickinchaseRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Comando Quatro (48K)

static struct BurnRomInfo SpecComandoquatroRomDesc[] = {
	{ "Comando Quatro (1989)(Zigurat).tap", 49485, 0x11a8ebc0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecComandoquatro, SpecComandoquatro, Spectrum)
STD_ROM_FN(SpecComandoquatro)

struct BurnDriver BurnSpecComandoquatro = {
	"spec_comandoquatro", NULL, "spec_spectrum", NULL, "1989",
	"Comando Quatro (48K)\0", NULL, "Zigurat Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecComandoquatroRomInfo, SpecComandoquatroRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Cosmic Wartoad (48K)

static struct BurnRomInfo SpecCosmicwartoadRomDesc[] = {
	{ "Cosmic Wartoad (1985)(Ocean).tap", 45221, 0x81e8ae65, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCosmicwartoad, SpecCosmicwartoad, Spectrum)
STD_ROM_FN(SpecCosmicwartoad)

struct BurnDriver BurnSpecCosmicwartoad = {
	"spec_cosmicwartoad", NULL, "spec_spectrum", NULL, "1985",
	"Cosmic Wartoad (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecCosmicwartoadRomInfo, SpecCosmicwartoadRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dark Star (48K)

static struct BurnRomInfo SpecDarkstarRomDesc[] = {
	{ "Dark Star (1984)(Design Design).tap", 45980, 0x38a11d60, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDarkstar, SpecDarkstar, Spectrum)
STD_ROM_FN(SpecDarkstar)

struct BurnDriver BurnSpecDarkstar = {
	"spec_darkstar", NULL, "spec_spectrum", NULL, "1984",
	"Dark Star (48K)\0", NULL, "Design Design", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecDarkstarRomInfo, SpecDarkstarRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dea Tenebrarum (Spanish) (48K)

static struct BurnRomInfo SpecDeatenebrarumRomDesc[] = {
	{ "Dea Tenebrarum (1988)(System 4).tap", 48584, 0xef594c3b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDeatenebrarum, SpecDeatenebrarum, Spectrum)
STD_ROM_FN(SpecDeatenebrarum)

struct BurnDriver BurnSpecDeatenebrarum = {
	"spec_deatenebrarum", NULL, "spec_spectrum", NULL, "1988",
	"Dea Tenebrarum (Spanish) (48K)\0", NULL, "System 4", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecDeatenebrarumRomInfo, SpecDeatenebrarumRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Denizen (128K)

static struct BurnRomInfo SpecDenizenRomDesc[] = {
	{ "Denizen (1988)(Players).tap", 112972, 0x6ae6dff4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDenizen, SpecDenizen, Spec128)
STD_ROM_FN(SpecDenizen)

struct BurnDriver BurnSpecDenizen = {
	"spec_denizen", NULL, "spec_spec128", NULL, "1988",
	"Denizen (128K)\0", NULL, "Players Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_MAZE, 0,
	SpectrumGetZipName, SpecDenizenRomInfo, SpecDenizenRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Disco Dan (48K)

static struct BurnRomInfo SpecDiscodanRomDesc[] = {
	{ "Disco Dan (1984)(Gem Software).tap", 19931, 0x8ba92db3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDiscodan, SpecDiscodan, Spectrum)
STD_ROM_FN(SpecDiscodan)

struct BurnDriver BurnSpecDiscodan = {
	"spec_discodan", NULL, "spec_spectrum", NULL, "1984",
	"Disco Dan (48K) \0", "Set '1. Cursor Joystick' to use controller", "Gem Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecDiscodanRomInfo, SpecDiscodanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecCursorKeysDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Duet (48K)

static struct BurnRomInfo SpecDuetRomDesc[] = {
	{ "Duet (1987)(Elite).tap", 41164, 0x80308e30, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDuet, SpecDuet, Spectrum)
STD_ROM_FN(SpecDuet)

struct BurnDriver BurnSpecDuet = {
	"spec_duet", NULL, "spec_spectrum", NULL, "1987",
	"Duet (48K)\0", "Q, A, E, R and Z to control player 1", "Elite Systems", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_MAZE, 0,
	SpectrumGetZipName, SpecDuetRomInfo, SpecDuetRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Eliminator (128K)

static struct BurnRomInfo SpecEliminatorRomDesc[] = {
	{ "Eliminator (1988)(Hewson).tap", 44964, 0x4e5643c0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecEliminator, SpecEliminator, Spec128)
STD_ROM_FN(SpecEliminator)

struct BurnDriver BurnSpecEliminator = {
	"spec_eliminator", NULL, "spec_spec128", NULL, "1988",
	"Eliminator (128K)\0", NULL, "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_RACING, 0,
	SpectrumGetZipName, SpecEliminatorRomInfo, SpecEliminatorRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Fearless Frank (48K)

static struct BurnRomInfo SpecFearlessfrankRomDesc[] = {
	{ "Fearless Frank (1984)(Visions).tap", 40614, 0x573889ab, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFearlessfrank, SpecFearlessfrank, Spectrum)
STD_ROM_FN(SpecFearlessfrank)

struct BurnDriver BurnSpecFearlessfrank = {
	"spec_fearlessfrank", NULL, "spec_spectrum", NULL, "1984",
	"Fearless Frank (48K)\0", NULL, "Visions", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecFearlessfrankRomInfo, SpecFearlessfrankRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Foxx Fights Back (128K)

static struct BurnRomInfo SpecFoxxfightsbackRomDesc[] = {
	{ "Foxx Fights Back (1988)(Image Works).tap", 47395, 0x7c9ae4b5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFoxxfightsback, SpecFoxxfightsback, Spec128)
STD_ROM_FN(SpecFoxxfightsback)

struct BurnDriver BurnSpecFoxxfightsback = {
	"spec_foxxfightsback", NULL, "spec_spec128", NULL, "1988",
	"Foxx Fights Back (128K)\0", NULL, "Image Works", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecFoxxfightsbackRomInfo, SpecFoxxfightsbackRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Star Blade (48K)

static struct BurnRomInfo SpecStarbladeRomDesc[] = {
	{ "Star Blade (1995)(Barry Addams).z80", 38880, 0x96e096c6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecStarblade, SpecStarblade, Spectrum)
STD_ROM_FN(SpecStarblade)

struct BurnDriver BurnSpecStarblade = {
	"spec_starblade", NULL, "spec_spectrum", NULL, "1995",
	"Star Blade (48K)\0", NULL, "Barry Addams", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecStarbladeRomInfo, SpecStarbladeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// UVE Atac (48K)

static struct BurnRomInfo SpecUveatacRomDesc[] = {
	{ "UVE Atac (1985)(MicroHobby).tap", 17979, 0x78f60aea, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecUveatac, SpecUveatac, Spectrum)
STD_ROM_FN(SpecUveatac)

struct BurnDriver BurnSpecUveatac = {
	"spec_uveatac", NULL, "spec_spectrum", NULL, "1985",
	"UVE Atac (48K)\0", NULL, "MicroHobby", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecUveatacRomInfo, SpecUveatacRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Zub (128K)

static struct BurnRomInfo SpecZubRomDesc[] = {
	{ "Zub (1986)(Mastertronic Added Dimension).tap", 62034, 0xe48404af, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecZub, SpecZub, Spec128)
STD_ROM_FN(SpecZub)

struct BurnDriver BurnSpecZub = {
	"spec_zub", NULL, "spec_spec128", NULL, "1986",
	"Zub (128K)\0", NULL, "MAD (Mastertronic Added Dimension)", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecZubRomInfo, SpecZubRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Gunhead (128K)

static struct BurnRomInfo SpecGunheadRomDesc[] = {
	{ "Gunhead (1990)(Hellenic).tap", 47800, 0xd1c99fb1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGunhead, SpecGunhead, Spec128)
STD_ROM_FN(SpecGunhead)

struct BurnDriver BurnSpecGunhead = {
	"spec_gunhead", NULL, "spec_spec128", NULL, "1990",
	"Gunhead (128K)\0", NULL, "Hellenic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecGunheadRomInfo, SpecGunheadRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPMDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hunter (128K)

static struct BurnRomInfo SpecHunterRomDesc[] = {
	{ "Hunter (1991)(Stuffduck).tap", 47590, 0xac3c997b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHunter, SpecHunter, Spec128)
STD_ROM_FN(SpecHunter)

struct BurnDriver BurnSpecHunter = {
	"spec_hunter", NULL, "spec_spec128", NULL, "1991",
	"Hunter (128K)\0", NULL, "Stuffduck", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecHunterRomInfo, SpecHunterRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Jackson City (48K)

static struct BurnRomInfo SpecJacksoncityRomDesc[] = {
	{ "Jackson City (1990)(Diabolic).tap", 42878, 0x320d1aec, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecJacksoncity, SpecJacksoncity, Spectrum)
STD_ROM_FN(SpecJacksoncity)

struct BurnDriver BurnSpecJacksoncity = {
	"spec_jacksoncity", NULL, "spec_spectrum", NULL, "1990",
	"Jackson City (48K)\0", NULL, "Diabolic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecJacksoncityRomInfo, SpecJacksoncityRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Laser Zone (48K)

static struct BurnRomInfo SpecLaserzoneRomDesc[] = {
	{ "Laser Zone (1983)(Quicksilva).tap", 17800, 0x701e7685, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecLaserzone, SpecLaserzone, Spectrum)
STD_ROM_FN(SpecLaserzone)

struct BurnDriver BurnSpecLaserzone = {
	"spec_laserzone", NULL, "spec_spectrum", NULL, "1983",
	"Laser Zone (48K)\0", NULL, "Quicksilva", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecLaserzoneRomInfo, SpecLaserzoneRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// MASK (48K-128K)

static struct BurnRomInfo SpecMaskRomDesc[] = {
	{ "Mask (1987)(Gremlin).z80", 92283, 0xcdd8c4b1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMask, SpecMask, Spec128)
STD_ROM_FN(SpecMask)

struct BurnDriver BurnSpecMask = {
	"spec_mask", NULL, "spec_spec128", NULL, "1987",
	"MASK (48K-128K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecMaskRomInfo, SpecMaskRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Master Blaster (48K)

static struct BurnRomInfo SpecMasterblasterRomDesc[] = {
	{ "Master Blaster (1990)(Cybadyne).tap", 49304, 0x83f104ce, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMasterblaster, SpecMasterblaster, Spectrum)
STD_ROM_FN(SpecMasterblaster)

struct BurnDriver BurnSpecMasterblaster = {
	"spec_masterblaster", NULL, "spec_spectrum", NULL, "1990",
	"Master Blaster (48K)\0", NULL, "Cybadyne", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecMasterblasterRomInfo, SpecMasterblasterRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mean Streak (128K)

static struct BurnRomInfo SpecMeanstreakRomDesc[] = {
	{ "Mean Streak (1987)(Mirrorsoft).tap", 66912, 0xeaa7cbc4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMeanstreak, SpecMeanstreak, Spec128)
STD_ROM_FN(SpecMeanstreak)

struct BurnDriver BurnSpecMeanstreak = {
	"spec_meanstreak", NULL, "spec_spec128", NULL, "1987",
	"Mean Streak (128K)\0", NULL, "Mirrorsoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecMeanstreakRomInfo, SpecMeanstreakRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mega Apocalypse (128K)

static struct BurnRomInfo SpecMegaapocalypseRomDesc[] = {
	{ "Mega Apocalypse (1988)(Martech).tap", 45336, 0x989130c0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMegaapocalypse, SpecMegaapocalypse, Spec128)
STD_ROM_FN(SpecMegaapocalypse)

struct BurnDriver BurnSpecMegaapocalypse = {
	"spec_megaapocalypse", NULL, "spec_spec128", NULL, "1988",
	"Mega Apocalypse (128K)\0", NULL, "Martech", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecMegaapocalypseRomInfo, SpecMegaapocalypseRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mega Phoenix (128K)

static struct BurnRomInfo SpecMegaphoenixRomDesc[] = {
	{ "Megaphoenix (1991)(Dinamic).tap", 43229, 0x81937ff0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMegaphoenix, SpecMegaphoenix, Spec128)
STD_ROM_FN(SpecMegaphoenix)

struct BurnDriver BurnSpecMegaphoenix = {
	"spec_megaphoenix", NULL, "spec_spec128", NULL, "1991",
	"Mega Phoenix (128K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecMegaphoenixRomInfo, SpecMegaphoenixRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Moon Strike (48K)

static struct BurnRomInfo SpecMoonstrikeRomDesc[] = {
	{ "Moon Strike (1987)(Mirrorsoft).tap", 49633, 0xbd0e5cbd, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMoonstrike, SpecMoonstrike, Spectrum)
STD_ROM_FN(SpecMoonstrike)

struct BurnDriver BurnSpecMoonstrike = {
	"spec_moonstrike", NULL, "spec_spectrum", NULL, "1987",
	"Moon Strike (48K)\0", "Press Enter to skip intro screen", "Mirrorsoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecMoonstrikeRomInfo, SpecMoonstrikeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Paratroopers (16K)

static struct BurnRomInfo SpecParatroopersRomDesc[] = {
	{ "Paratroopers (1983)(Rabbit).tap", 21732, 0xc19b3a5d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecParatroopers, SpecParatroopers, Spectrum)
STD_ROM_FN(SpecParatroopers)

struct BurnDriver BurnSpecParatroopers = {
	"spec_paratroopers", NULL, "spec_spectrum", NULL, "1983",
	"Paratroopers (16K)\0", NULL, "Rabbit", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecParatroopersRomInfo, SpecParatroopersRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Phileas Fogg's Balloon Battles (48K)

static struct BurnRomInfo SpecPhileasfoggRomDesc[] = {
	{ "Phileas Fogg's Balloon Battles (1991)(Zeppelin).tap", 48594, 0xc311768c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPhileasfogg, SpecPhileasfogg, Spectrum)
STD_ROM_FN(SpecPhileasfogg)

struct BurnDriver BurnSpecPhileasfogg = {
	"spec_phileasfogg", NULL, "spec_spectrum", NULL, "1991",
	"Phileas Fogg's Balloon Battles (48K)\0", NULL, "Zeppelin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecPhileasfoggRomInfo, SpecPhileasfoggRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Quick Draw McGraw (128K)

static struct BurnRomInfo SpecQuickdrawRomDesc[] = {
	{ "Quick Draw McGraw (1991)(Hi-Tec).tap", 53434, 0x435650ba, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecQuickdraw, SpecQuickdraw, Spec128)
STD_ROM_FN(SpecQuickdraw)

struct BurnDriver BurnSpecQuickdraw = {
	"spec_quickdraw", NULL, "spec_spec128", NULL, "1991",
	"Quick Draw McGraw (128K)\0", NULL, "Hi-Tec Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecQuickdrawRomInfo, SpecQuickdrawRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Seahawk (48K)

static struct BurnRomInfo SpecSeahawkRomDesc[] = {
	{ "Seahawk (1990)(Atlantis Software).tap", 48448, 0x32c8705a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSeahawk, SpecSeahawk, Spectrum)
STD_ROM_FN(SpecSeahawk)

struct BurnDriver BurnSpecSeahawk = {
	"spec_seahawk", NULL, "spec_spectrum", NULL, "1990",
	"Seahawk (48K)\0", NULL, "Atlantis Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecSeahawkRomInfo, SpecSeahawkRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPMDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Shadow Skimmer (48K)

static struct BurnRomInfo SpecShadowskimmerRomDesc[] = {
	{ "Shadow Skimmer (1987)(The Edge).z80", 41483, 0xc7d0a073, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecShadowskimmer, SpecShadowskimmer, Spectrum)
STD_ROM_FN(SpecShadowskimmer)

struct BurnDriver BurnSpecShadowskimmer = {
	"spec_shadowskimmer", NULL, "spec_spectrum", NULL, "1987",
	"Shadow Skimmer (48K)\0", NULL, "The Edge", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecShadowskimmerRomInfo, SpecShadowskimmerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Shark (128K)

static struct BurnRomInfo SpecSharkRomDesc[] = {
	{ "Shark (1989)(Players Premier).tap", 101962, 0x014ce429, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecShark, SpecShark, Spec128)
STD_ROM_FN(SpecShark)

struct BurnDriver BurnSpecShark = {
	"spec_shark", NULL, "spec_spec128", NULL, "1989",
	"Shark (128K)\0", NULL, "Players Premier", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecSharkRomInfo, SpecSharkRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Star Firebirds (48K)

static struct BurnRomInfo SpecStarfirebirdsRomDesc[] = {
	{ "Star Firebirds (1985)(Firebird).z80", 34053, 0xe38dec78, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecStarfirebirds, SpecStarfirebirds, Spectrum)
STD_ROM_FN(SpecStarfirebirds)

struct BurnDriver BurnSpecStarfirebirds = {
	"spec_starfirebirds", NULL, "spec_spectrum", NULL, "1985",
	"Star Firebirds (48K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecStarfirebirdsRomInfo, SpecStarfirebirdsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Tomcat (48K)

static struct BurnRomInfo SpecTomcatRomDesc[] = {
	{ "Tomcat (1989)(Players).tap", 41255, 0xef75de5c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTomcat, SpecTomcat, Spectrum)
STD_ROM_FN(SpecTomcat)

struct BurnDriver BurnSpecTomcat = {
	"spec_tomcat", NULL, "spec_spectrum", NULL, "1989",
	"Tomcat (48K)\0", NULL, "Players Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecTomcatRomInfo, SpecTomcatRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// War Machine (128K)

static struct BurnRomInfo SpecWarmachineRomDesc[] = {
	{ "War Machine (1989)(Players Premier).tap", 54991, 0x7ac04bb1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecWarmachine, SpecWarmachine, Spec128)
STD_ROM_FN(SpecWarmachine)

struct BurnDriver BurnSpecWarmachine = {
	"spec_warmachine", NULL, "spec_spec128", NULL, "1989",
	"War Machine (128K)\0", NULL, "Players Premier", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecWarmachineRomInfo, SpecWarmachineRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Who Dares Wins II (48K)

static struct BurnRomInfo SpecWhodareswinsiiRomDesc[] = {
	{ "Who Dares Wins II (1986)(Alligata).tap", 48631, 0x02dc6931, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecWhodareswinsii, SpecWhodareswinsii, Spectrum)
STD_ROM_FN(SpecWhodareswinsii)

struct BurnDriver BurnSpecWhodareswinsii = {
	"spec_whodareswinsii", NULL, "spec_spectrum", NULL, "1986",
	"Who Dares Wins II (48K)\0", NULL, "Alligata", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecWhodareswinsiiRomInfo, SpecWhodareswinsiiRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 007 - Live and Let Die (128K)

static struct BurnRomInfo Spec007liveandletdieRomDesc[] = {
	{ "007 - Live and Let Die (1988)(Domark).tap", 62379, 0x5f21da6b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec007liveandletdie, Spec007liveandletdie, Spec128)
STD_ROM_FN(Spec007liveandletdie)

struct BurnDriver BurnSpec007liveandletdie = {
	"spec_007liveandletdie", NULL, "spec_spec128", NULL, "1988",
	"007 - Live and Let Die (128K)\0", NULL, "Domark", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, Spec007liveandletdieRomInfo, Spec007liveandletdieRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 007 - The Spy Who Loved Me (128K)

static struct BurnRomInfo Spec007spywholovedmeRomDesc[] = {
	{ "007 - The Spy Who Loved Me (1990)(Domark).tap", 129851, 0x629c140c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec007spywholovedme, Spec007spywholovedme, Spec128)
STD_ROM_FN(Spec007spywholovedme)

struct BurnDriver BurnSpec007spywholovedme = {
	"spec_007spywholovedme", NULL, "spec_spec128", NULL, "1990",
	"007 - The Spy Who Loved Me (128K)\0", NULL, "Domark", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING | GBF_VERSHOOT, 0,
	SpectrumGetZipName, Spec007spywholovedmeRomInfo, Spec007spywholovedmeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 2088 (48K)

static struct BurnRomInfo Spec2088RomDesc[] = {
	{ "2088 (1988)(Zeppelin Games).tap", 28306, 0xdecaeb6e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec2088, Spec2088, Spectrum)
STD_ROM_FN(Spec2088)

struct BurnDriver BurnSpec2088 = {
	"spec_2088", NULL, "spec_spectrum", NULL, "1988",
	"2088 (48K)\0", NULL, "Zeppelin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, Spec2088RomInfo, Spec2088RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 3D Starfighter (48K)

static struct BurnRomInfo Spec3dstarfighterRomDesc[] = {
	{ "3D Starfighter (1988)(Codemasters).tap", 48891, 0x4f3f2fbe, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec3dstarfighter, Spec3dstarfighter, Spectrum)
STD_ROM_FN(Spec3dstarfighter)

struct BurnDriver BurnSpec3dstarfighter = {
	"spec_3dstarfighter", NULL, "spec_spectrum", NULL, "1988",
	"3D Starfighter (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, Spec3dstarfighterRomInfo, Spec3dstarfighterRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 3D-Tetris (48K)

static struct BurnRomInfo Spec3dtetrisRomDesc[] = {
	{ "3D-Tetris (1989)(Antic).z80", 19178, 0xe46b5e70, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec3dtetris, Spec3dtetris, Spectrum)
STD_ROM_FN(Spec3dtetris)

struct BurnDriver BurnSpec3dtetris = {
	"spec_3dtetris", NULL, "spec_spectrum", NULL, "1989",
	"3D-Tetris (48K)\0", NULL, "Antic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, Spec3dtetrisRomInfo, Spec3dtetrisRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 5 in 1 Tetris (128K)

static struct BurnRomInfo Spec5in1tetrisRomDesc[] = {
	{ "5 in 1 Tetris (1995)(Mortal Kombat).z80", 28526, 0x996128b2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec5in1tetris, Spec5in1tetris, Spec128)
STD_ROM_FN(Spec5in1tetris)

struct BurnDriver BurnSpec5in1tetris = {
	"spec_5in1tetris", NULL, "spec_spec128", NULL, "1995",
	"5 in 1 Tetris (128K)\0", NULL, "Mortal Kombat", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, Spec5in1tetrisRomInfo, Spec5in1tetrisRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ammytris (128K)

static struct BurnRomInfo SpecAmmytrisRomDesc[] = {
	{ "Ammytris (1995)(Navigator).tap", 19901, 0x24de1080, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAmmytris, SpecAmmytris, Spec128)
STD_ROM_FN(SpecAmmytris)

struct BurnDriver BurnSpecAmmytris = {
	"spec_ammytris", NULL, "spec_spec128", NULL, "1995",
	"Ammytris (128K)\0", NULL, "Navigator", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecAmmytrisRomInfo, SpecAmmytrisRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dragon Spirit (128K)

static struct BurnRomInfo SpecDragonspiritRomDesc[] = {
	{ "Dragon Spirit (1989)(Domark).tap", 164427, 0xadc4c7cf, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDragonspirit, SpecDragonspirit, Spec128)
STD_ROM_FN(SpecDragonspirit)

struct BurnDriver BurnSpecDragonspirit = {
	"spec_dragonspirit", NULL, "spec_spec128", NULL, "1989",
	"Dragon Spirit (128K)\0", NULL, "Domark", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecDragonspiritRomInfo, SpecDragonspiritRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Little Puff in Dragonland (128K)

static struct BurnRomInfo SpecLittlepuffRomDesc[] = {
	{ "Little Puff in Dragonland (1990)(Codemasters).tap", 42827, 0x30a03963, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecLittlepuff, SpecLittlepuff, Spec128)
STD_ROM_FN(SpecLittlepuff)

struct BurnDriver BurnSpecLittlepuff = {
	"spec_littlepuff", NULL, "spec_spec128", NULL, "1990",
	"Little Puff in Dragonland (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecLittlepuffRomInfo, SpecLittlepuffRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Miami Chase (128K)

static struct BurnRomInfo SpecMiamichaseRomDesc[] = {
	{ "Miami Chase (1991)(Codemasters).tap", 48671, 0x24cb897c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMiamichase, SpecMiamichase, Spec128)
STD_ROM_FN(SpecMiamichase)

struct BurnDriver BurnSpecMiamichase = {
	"spec_miamichase", NULL, "spec_spec128", NULL, "1991",
	"Miami Chase (128K)\0", "Use keys K, M and L to select options", "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_RACING, 0,
	SpectrumGetZipName, SpecMiamichaseRomInfo, SpecMiamichaseRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mig 29 Soviet Fighter (128K)

static struct BurnRomInfo SpecMig29sovfighterRomDesc[] = {
	{ "Mig 29 Soviet Fighter (1989)(Codemasters).tap", 49442, 0x93a448fd, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMig29sovfighter, SpecMig29sovfighter, Spec128)
STD_ROM_FN(SpecMig29sovfighter)

struct BurnDriver BurnSpecMig29sovfighter = {
	"spec_mig29sovfighter", NULL, "spec_spec128", NULL, "1989",
	"Mig 29 Soviet Fighter (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecMig29sovfighterRomInfo, SpecMig29sovfighterRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mission Jupiter (48K)

static struct BurnRomInfo SpecMissionjupiterRomDesc[] = {
	{ "Mission Jupiter (1987)(Codemasters).tap", 45152, 0xf136a5b9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMissionjupiter, SpecMissionjupiter, Spectrum)
STD_ROM_FN(SpecMissionjupiter)

struct BurnDriver BurnSpecMissionjupiter = {
	"spec_missionjupiter", NULL, "spec_spectrum", NULL, "1987",
	"Mission Jupiter (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecMissionjupiterRomInfo, SpecMissionjupiterRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Monte Carlo Casino (128K)

static struct BurnRomInfo SpecMontecarlocasinoRomDesc[] = {
	{ "Monte Carlo Casino (1989)(Codemasters).tap", 41821, 0x8d459dc5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMontecarlocasino, SpecMontecarlocasino, Spec128)
STD_ROM_FN(SpecMontecarlocasino)

struct BurnDriver BurnSpecMontecarlocasino = {
	"spec_montecarlocasino", NULL, "spec_spec128", NULL, "1989",
	"Monte Carlo Casino (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_CASINO, 0,
	SpectrumGetZipName, SpecMontecarlocasinoRomInfo, SpecMontecarlocasinoRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Moto Cross Simulator (128K)

static struct BurnRomInfo SpecMotocrosssimRomDesc[] = {
	{ "Moto Cross Simulator (1989)(Codemasters).tap", 51225, 0x7abd5f87, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMotocrosssim, SpecMotocrosssim, Spec128)
STD_ROM_FN(SpecMotocrosssim)

struct BurnDriver BurnSpecMotocrosssim = {
	"spec_motocrosssim", NULL, "spec_spec128", NULL, "1989",
	"Moto Cross Simulator (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecMotocrosssimRomInfo, SpecMotocrosssimRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mountain Bike Simulator (128K)

static struct BurnRomInfo SpecMountainbikesimRomDesc[] = {
	{ "Mountain Bike Simulator (1991)(Codemasters).tap", 53006, 0xd2738989, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMountainbikesim, SpecMountainbikesim, Spec128)
STD_ROM_FN(SpecMountainbikesim)

struct BurnDriver BurnSpecMountainbikesim = {
	"spec_mountainbikesim", NULL, "spec_spec128", NULL, "1991",
	"Mountain Bike Simulator (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecMountainbikesimRomInfo, SpecMountainbikesimRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Murray Mouse Super Cop (48K)

static struct BurnRomInfo SpecMurraymousesupercopRomDesc[] = {
	{ "Murray Mouse Super Cop (1992)(Codemasters).tap", 32045, 0x67637ddf, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMurraymousesupercop, SpecMurraymousesupercop, Spectrum)
STD_ROM_FN(SpecMurraymousesupercop)

struct BurnDriver BurnSpecMurraymousesupercop = {
	"spec_murraymousesupercop", NULL, "spec_spectrum", NULL, "1992",
	"Murray Mouse Super Cop (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecMurraymousesupercopRomInfo, SpecMurraymousesupercopRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Necris Dome (48K)

static struct BurnRomInfo SpecNecrisdomeRomDesc[] = {
	{ "Necris Dome (1987)(Codemasters).tap", 48602, 0x731d02fb, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecNecrisdome, SpecNecrisdome, Spectrum)
STD_ROM_FN(SpecNecrisdome)

struct BurnDriver BurnSpecNecrisdome = {
	"spec_necrisdome", NULL, "spec_spectrum", NULL, "1987",
	"Necris Dome (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecNecrisdomeRomInfo, SpecNecrisdomeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ninja Massacre (128K)

static struct BurnRomInfo SpecNinjamassacreRomDesc[] = {
	{ "Ninja Massacre (1989)(Codemasters).tap", 59148, 0x2b28000a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecNinjamassacre, SpecNinjamassacre, Spec128)
STD_ROM_FN(SpecNinjamassacre)

struct BurnDriver BurnSpecNinjamassacre = {
	"spec_ninjamassacre", NULL, "spec_spec128", NULL, "1989",
	"Ninja Massacre (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_MAZE, 0,
	SpectrumGetZipName, SpecNinjamassacreRomInfo, SpecNinjamassacreRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Operation Gunship (128K)

static struct BurnRomInfo SpecOperationgunshipRomDesc[] = {
	{ "Operation Gunship (1989)(Codemasters).tap", 52455, 0xee1f687d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecOperationgunship, SpecOperationgunship, Spec128)
STD_ROM_FN(SpecOperationgunship)

struct BurnDriver BurnSpecOperationgunship = {
	"spec_operationgunship", NULL, "spec_spec128", NULL, "1989",
	"Operation Gunship (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecOperationgunshipRomInfo, SpecOperationgunshipRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Paris To Dakar (128K)

static struct BurnRomInfo SpecParistodakarRomDesc[] = {
	{ "Paris To Dakar (1991)(Codemasters).tap", 73017, 0xcb4a5edc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecParistodakar, SpecParistodakar, Spec128)
STD_ROM_FN(SpecParistodakar)

struct BurnDriver BurnSpecParistodakar = {
	"spec_paristodakar", NULL, "spec_spec128", NULL, "1991",
	"Paris To Dakar (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecParistodakarRomInfo, SpecParistodakarRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Phantomas (48K)

static struct BurnRomInfo SpecPhantomasRomDesc[] = {
	{ "Phantomas (1986)(Dinamic).tap", 48127, 0x34c8919a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPhantomas, SpecPhantomas, Spectrum)
STD_ROM_FN(SpecPhantomas)

struct BurnDriver BurnSpecPhantomas = {
	"spec_phantomas", NULL, "spec_spectrum", NULL, "1986",
	"Phantomas (48K)\0", "6, 7 and Enter to select control type", "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecPhantomasRomInfo, SpecPhantomasRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Phantomas 2 (48K)

static struct BurnRomInfo SpecPhantomas2RomDesc[] = {
	{ "Phantomas 2 (1986)(Dinamic).tap", 49300, 0x7e1fecc2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPhantomas2, SpecPhantomas2, Spectrum)
STD_ROM_FN(SpecPhantomas2)

struct BurnDriver BurnSpecPhantomas2 = {
	"spec_phantomas2", NULL, "spec_spectrum", NULL, "1986",
	"Phantomas 2 (48K)\0", "6, 7 and Enter to select control type", "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecPhantomas2RomInfo, SpecPhantomas2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Prince Clumsy (48K)

static struct BurnRomInfo SpecPrinceclumsyRomDesc[] = {
	{ "Prince Clumsy (1990)(Codemasters).z80", 36725, 0xc5ab98f3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPrinceclumsy, SpecPrinceclumsy, Spectrum)
STD_ROM_FN(SpecPrinceclumsy)

struct BurnDriver BurnSpecPrinceclumsy = {
	"spec_princeclumsy", NULL, "spec_spectrum", NULL, "1990",
	"Prince Clumsy (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecPrinceclumsyRomInfo, SpecPrinceclumsyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Professional BMX Simulator - Expert (128K)

static struct BurnRomInfo SpecProbmxsimexpertRomDesc[] = {
	{ "Professional BMX Simulator - Expert (1988)(Codemasters).tap", 81052, 0x5f895fc1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecProbmxsimexpert, SpecProbmxsimexpert, Spec128)
STD_ROM_FN(SpecProbmxsimexpert)

struct BurnDriver BurnSpecProbmxsimexpert = {
	"spec_probmxsimexpert", "spec_probmxsimstandard", "spec_spec128", NULL, "1988",
	"Professional BMX Simulator - Expert (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 4, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecProbmxsimexpertRomInfo, SpecProbmxsimexpertRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Professional BMX Simulator - Standard (128K)

static struct BurnRomInfo SpecProbmxsimstandardRomDesc[] = {
	{ "Professional BMX Simulator - Standard (1988)(Codemasters).tap", 81052, 0x7468e20b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecProbmxsimstandard, SpecProbmxsimstandard, Spec128)
STD_ROM_FN(SpecProbmxsimstandard)

struct BurnDriver BurnSpecProbmxsimstandard = {
	"spec_probmxsimstandard", NULL, "spec_spec128", NULL, "1988",
	"Professional BMX Simulator - Standard (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 4, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecProbmxsimstandardRomInfo, SpecProbmxsimstandardRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Pro Golf Simulator (128K)

static struct BurnRomInfo SpecProgolfsimRomDesc[] = {
	{ "Pro Golf Simulator (1990)(Codemasters).tap", 48603, 0x6e5bed28, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecProgolfsim, SpecProgolfsim, Spec128)
STD_ROM_FN(SpecProgolfsim)

struct BurnDriver BurnSpecProgolfsim = {
	"spec_progolfsim", NULL, "spec_spec128", NULL, "1990",
	"Pro Golf Simulator (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecProgolfsimRomInfo, SpecProgolfsimRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Pro PowerBoat Simulator (128K)

static struct BurnRomInfo SpecPropowerboatsimRomDesc[] = {
	{ "Pro PowerBoat Simulator (1989)(Codemasters).tap", 49394, 0x5627c3f7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPropowerboatsim, SpecPropowerboatsim, Spec128)
STD_ROM_FN(SpecPropowerboatsim)

struct BurnDriver BurnSpecPropowerboatsim = {
	"spec_propowerboatsim", NULL, "spec_spec128", NULL, "1989",
	"Pro PowerBoat Simulator (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecPropowerboatsimRomInfo, SpecPropowerboatsimRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Pro Skateboard Simulator (48K)

static struct BurnRomInfo SpecProskateboardsimRomDesc[] = {
	{ "Pro Skateboard Simulator 48K (1989)(Codemasters).tap", 56277, 0xf3a7d031, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecProskateboardsim, SpecProskateboardsim, Spectrum)
STD_ROM_FN(SpecProskateboardsim)

struct BurnDriver BurnSpecProskateboardsim = {
	"spec_proskateboardsim", NULL, "spec_spectrum", NULL, "1989",
	"Pro Skateboard Simulator (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecProskateboardsimRomInfo, SpecProskateboardsimRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Professional Ski Simulator (48K)

static struct BurnRomInfo SpecProskisimRomDesc[] = {
	{ "Professional Ski Simulator (1987)(Codemasters).tap", 48913, 0xce33d499, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecProskisim, SpecProskisim, Spectrum)
STD_ROM_FN(SpecProskisim)

struct BurnDriver BurnSpecProskisim = {
	"spec_proskisim", NULL, "spec_spectrum", NULL, "1987",
	"Professional Ski Simulator (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecProskisimRomInfo, SpecProskisimRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Professional Snooker Simulator (48K)

static struct BurnRomInfo SpecProsnookersimRomDesc[] = {
	{ "Professional Snooker Simulator (1987)(Codemasters).tap", 22922, 0xc8b22bca, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecProsnookersim, SpecProsnookersim, Spectrum)
STD_ROM_FN(SpecProsnookersim)

struct BurnDriver BurnSpecProsnookersim = {
	"spec_prosnookersim", NULL, "spec_spectrum", NULL, "1987",
	"Professional Snooker Simulator (48K)\0", "Q, W, E, SPACE and ENTER to play", "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecProsnookersimRomInfo, SpecProsnookersimRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecCursorKeysDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Pro Tennis Simulator (128K)

static struct BurnRomInfo SpecProtennissimRomDesc[] = {
	{ "Pro Tennis Simulator (1990)(Codemasters).z80", 27039, 0x9e79b85c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecProtennissim, SpecProtennissim, Spec128)
STD_ROM_FN(SpecProtennissim)

struct BurnDriver BurnSpecProtennissim = {
	"spec_protennissim", NULL, "spec_spec128", NULL, "1990",
	"Pro Tennis Simulator (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecProtennissimRomInfo, SpecProtennissimRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Pub Trivia (128K)

static struct BurnRomInfo SpecPubtriviaRomDesc[] = {
	{ "Pub Trivia (1989)(Codemasters).z80", 100743, 0xbafbf528, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPubtrivia, SpecPubtrivia, Spec128)
STD_ROM_FN(SpecPubtrivia)

struct BurnDriver BurnSpecPubtrivia = {
	"spec_pubtrivia", NULL, "spec_spec128", NULL, "1989",
	"Pub Trivia (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 4, HARDWARE_SPECTRUM, GBF_QUIZ, 0,
	SpectrumGetZipName, SpecPubtriviaRomInfo, SpecPubtriviaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Race Against Time, The (48K)

static struct BurnRomInfo SpecRaceagainsttimeRomDesc[] = {
	{ "Race Against Time, The (1988)(Codemasters).tap", 48905, 0xeec9a042, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecRaceagainsttime, SpecRaceagainsttime, Spectrum)
STD_ROM_FN(SpecRaceagainsttime)

struct BurnDriver BurnSpecRaceagainsttime = {
	"spec_raceagainsttime", NULL, "spec_spectrum", NULL, "1988",
	"Race Against Time, The (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecRaceagainsttimeRomInfo, SpecRaceagainsttimeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rally Cross Simulator (128K)

static struct BurnRomInfo SpecRallycrosssimRomDesc[] = {
	{ "Rally Cross Simulator (1990)(Codemasters).tap", 111041, 0xa3f861ee, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecRallycrosssim, SpecRallycrosssim, Spec128)
STD_ROM_FN(SpecRallycrosssim)

struct BurnDriver BurnSpecRallycrosssim = {
	"spec_rallycrosssim", NULL, "spec_spec128", NULL, "1990",
	"Rally Cross Simulator (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecRallycrosssimRomInfo, SpecRallycrosssimRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Robin Hood - Legend Quest (128K)

static struct BurnRomInfo SpecRobinhoodlegendquestRomDesc[] = {
	{ "Robin Hood - Legend Quest (1993)(Codemasters).tap", 79970, 0x548b209e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecRobinhoodlegendquest, SpecRobinhoodlegendquest, Spec128)
STD_ROM_FN(SpecRobinhoodlegendquest)

struct BurnDriver BurnSpecRobinhoodlegendquest = {
	"spec_robinhoodlegendquest", NULL, "spec_spec128", NULL, "1993",
	"Robin Hood - Legend Quest (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecRobinhoodlegendquestRomInfo, SpecRobinhoodlegendquestRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rock Star Goes Bizarre (128K)

static struct BurnRomInfo SpecRockstargoesbizarreRomDesc[] = {
	{ "Rock Star Goes Bizarre (1989)(Codemasters).tap", 68929, 0x4dc0ff5a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecRockstargoesbizarre, SpecRockstargoesbizarre, Spec128)
STD_ROM_FN(SpecRockstargoesbizarre)

struct BurnDriver BurnSpecRockstargoesbizarre = {
	"spec_rockstargoesbizarre", NULL, "spec_spec128", NULL, "1989",
	"Rock Star Goes Bizarre (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_STRATEGY, 0,
	SpectrumGetZipName, SpecRockstargoesbizarreRomInfo, SpecRockstargoesbizarreRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rogue Trooper (48K)

static struct BurnRomInfo SpecRoguetrooperRomDesc[] = {
	{ "Rogue Trooper (1986)(Piranha).tap", 55832, 0x6e1e885e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecRoguetrooper, SpecRoguetrooper, Spectrum)
STD_ROM_FN(SpecRoguetrooper)

struct BurnDriver BurnSpecRoguetrooper = {
	"spec_roguetrooper", NULL, "spec_spectrum", NULL, "1986",
	"Rogue Trooper (48K)\0", NULL, "Piranha", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_ADV, 0,
	SpectrumGetZipName, SpecRoguetrooperRomInfo, SpecRoguetrooperRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// S.A.S. Combat Simulator (48K)

static struct BurnRomInfo SpecSascombatsimRomDesc[] = {
	{ "S.A.S. Combat Simulator (1988)(Codemasters).tap", 42914, 0x6324df81, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSascombatsim, SpecSascombatsim, Spectrum)
STD_ROM_FN(SpecSascombatsim)

struct BurnDriver BurnSpecSascombatsim = {
	"spec_sascombatsim", NULL, "spec_spectrum", NULL, "1988",
	"S.A.S. Combat Simulator (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecSascombatsimRomInfo, SpecSascombatsimRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// S.A.S. Operation Thunderflash (48K)

static struct BurnRomInfo SpecSasoperationthunderflashRomDesc[] = {
	{ "S.A.S. Operation Thunderflash (1986)(Sparklers).tap", 48345, 0x7013a78b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSasoperationthunderflash, SpecSasoperationthunderflash, Spectrum)
STD_ROM_FN(SpecSasoperationthunderflash)

struct BurnDriver BurnSpecSasoperationthunderflash = {
	"spec_sasoperationthunderflash", NULL, "spec_spectrum", NULL, "1986",
	"S.A.S. Operation Thunderflash (48K)\0", NULL, "Sparklers", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecSasoperationthunderflashRomInfo, SpecSasoperationthunderflashRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Scooby and Scrappy-Doo (48K)

static struct BurnRomInfo SpecSsdooRomDesc[] = {
	{ "Scooby and Scrappy-Doo (1991)(Hi-Tec).tap", 47868, 0x5e04acd0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSsdoo, SpecSsdoo, Spectrum)
STD_ROM_FN(SpecSsdoo)

struct BurnDriver BurnSpecSsdoo = {
	"spec_ssdoo", NULL, "spec_spectrum", NULL, "1991",
	"Scooby and Scrappy-Doo (48K)\0", NULL, "Hi-Tec Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecSsdooRomInfo, SpecSsdooRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sergeant Seymour Robotcop (128K)

static struct BurnRomInfo SpecSergeantseymourrobotcopRomDesc[] = {
	{ "Sergeant Seymour Robotcop (1992)(Codemasters).tap", 35548, 0x2ecf649d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSergeantseymourrobotcop, SpecSergeantseymourrobotcop, Spec128)
STD_ROM_FN(SpecSergeantseymourrobotcop)

struct BurnDriver BurnSpecSergeantseymourrobotcop = {
	"spec_sergeantseymourrobotcop", NULL, "spec_spec128", NULL, "1992",
	"Sergeant Seymour Robotcop (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecSergeantseymourrobotcopRomInfo, SpecSergeantseymourrobotcopRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sky High Stuntman (128K)

static struct BurnRomInfo SpecSkyhighstuntmanRomDesc[] = {
	{ "Sky High Stuntman (1991)(Codemasters).tap", 50525, 0x45b71f16, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSkyhighstuntman, SpecSkyhighstuntman, Spec128)
STD_ROM_FN(SpecSkyhighstuntman)

struct BurnDriver BurnSpecSkyhighstuntman = {
	"spec_skyhighstuntman", NULL, "spec_spec128", NULL, "1991",
	"Sky High Stuntman (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecSkyhighstuntmanRomInfo, SpecSkyhighstuntmanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Soccer Pinball (128K)

static struct BurnRomInfo SpecSoccerpinballRomDesc[] = {
	{ "Soccer Pinball (1992)(Codemasters).tap", 61677, 0xc1c3e3b6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSoccerpinball, SpecSoccerpinball, Spec128)
STD_ROM_FN(SpecSoccerpinball)

struct BurnDriver BurnSpecSoccerpinball = {
	"spec_soccerpinball", NULL, "spec_spec128", NULL, "1992",
	"Soccer Pinball (128K)\0", "Z = left flipper, M = right flipper, Space = shoot", "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_PINBALL, 0,
	SpectrumGetZipName, SpecSoccerpinballRomInfo, SpecSoccerpinballRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Spike in Transylvania (128K)

static struct BurnRomInfo SpecSpikeintransylvaniaRomDesc[] = {
	{ "Spike in Transylvania (1991)(Codemasters).tap", 54232, 0x2e3cf5ca, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSpikeintransylvania, SpecSpikeintransylvania, Spec128)
STD_ROM_FN(SpecSpikeintransylvania)

struct BurnDriver BurnSpecSpikeintransylvania = {
	"spec_spikeintransylvania", NULL, "spec_spec128", NULL, "1991",
	"Spike in Transylvania (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecSpikeintransylvaniaRomInfo, SpecSpikeintransylvaniaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Star Dragon (128K)

static struct BurnRomInfo SpecStardragonRomDesc[] = {
	{ "Star Dragon (1991)(Ultrasoft).tap", 34469, 0xf7a81a91, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecStardragon, SpecStardragon, Spec128)
STD_ROM_FN(SpecStardragon)

struct BurnDriver BurnSpecStardragon = {
	"spec_stardragon", NULL, "spec_spec128", NULL, "1991",
	"Star Dragon (128K)\0", NULL, "Ultrasoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecStardragonRomInfo, SpecStardragonRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Star Paws (128K)

static struct BurnRomInfo SpecStarpawsRomDesc[] = {
	{ "Star Paws (1988)(Software Projects).z80", 50264, 0x14d2c355, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecStarpaws, SpecStarpaws, Spec128)
STD_ROM_FN(SpecStarpaws)

struct BurnDriver BurnSpecStarpaws = {
	"spec_starpaws", NULL, "spec_spec128", NULL, "1988",
	"Star Paws (128K)\0", NULL, "Software Projects", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecStarpawsRomInfo, SpecStarpawsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Star Runner (48K)

static struct BurnRomInfo SpecStarrunnerRomDesc[] = {
	{ "Star Runner (1987)(Codemasters).tap", 56277, 0xfbde3432, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecStarrunner, SpecStarrunner, Spectrum)
STD_ROM_FN(SpecStarrunner)

struct BurnDriver BurnSpecStarrunner = {
	"spec_starrunner", NULL, "spec_spectrum", NULL, "1987",
	"Star Runner (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecStarrunnerRomInfo, SpecStarrunnerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Street Gang Football (48K)

static struct BurnRomInfo SpecStreetgangfootballRomDesc[] = {
	{ "Street Gang Football (1989)(Codemasters).tap", 43537, 0x580c88d2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecStreetgangfootball, SpecStreetgangfootball, Spectrum)
STD_ROM_FN(SpecStreetgangfootball)

struct BurnDriver BurnSpecStreetgangfootball = {
	"spec_streetgangfootball", NULL, "spec_spectrum", NULL, "1989",
	"Street Gang Football (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL, 0,
	SpectrumGetZipName, SpecStreetgangfootballRomInfo, SpecStreetgangfootballRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Stryker - In the Crypts of Trogan (128K)

static struct BurnRomInfo SpecStrykercryptsoftroganRomDesc[] = {
	{ "Stryker - In the Crypts of Trogan (1992)(Codemasters).tap", 90060, 0xd24f539d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecStrykercryptsoftrogan, SpecStrykercryptsoftrogan, Spec128)
STD_ROM_FN(SpecStrykercryptsoftrogan)

struct BurnDriver BurnSpecStrykercryptsoftrogan = {
	"spec_strykercryptsoftrogan", NULL, "spec_spec128", NULL, "1992",
	"Stryker - In the Crypts of Trogan (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecStrykercryptsoftroganRomInfo, SpecStrykercryptsoftroganRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Stunt Man Seymour (128K)

static struct BurnRomInfo SpecStuntmanseymourRomDesc[] = {
	{ "Stunt Man Seymour (1992)(Codemasters).tap", 51574, 0x098c5ba8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecStuntmanseymour, SpecStuntmanseymour, Spec128)
STD_ROM_FN(SpecStuntmanseymour)

struct BurnDriver BurnSpecStuntmanseymour = {
	"spec_stuntmanseymour", NULL, "spec_spec128", NULL, "1992",
	"Stunt Man Seymour (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecStuntmanseymourRomInfo, SpecStuntmanseymourRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Superbike Trans-Am (128K)

static struct BurnRomInfo SpecSuperbiketransamRomDesc[] = {
	{ "Superbike Trans-Am (1989)(Codemasters).z80", 40870, 0x28322d99, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSuperbiketransam, SpecSuperbiketransam, Spec128)
STD_ROM_FN(SpecSuperbiketransam)

struct BurnDriver BurnSpecSuperbiketransam = {
	"spec_superbiketransam", NULL, "spec_spec128", NULL, "1989",
	"Superbike Trans-Am (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecSuperbiketransamRomInfo, SpecSuperbiketransamRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Super Dragon Slayer (128K)

static struct BurnRomInfo SpecSuperdragonslayerRomDesc[] = {
	{ "Super Dragon Slayer (1989)(Codemasters).tap", 42607, 0x95581414, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSuperdragonslayer, SpecSuperdragonslayer, Spec128)
STD_ROM_FN(SpecSuperdragonslayer)

struct BurnDriver BurnSpecSuperdragonslayer = {
	"spec_superdragonslayer", NULL, "spec_spec128", NULL, "1989",
	"Super Dragon Slayer (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecSuperdragonslayerRomInfo, SpecSuperdragonslayerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Super G-Man (48K)

static struct BurnRomInfo SpecSupergmanRomDesc[] = {
	{ "Super G-Man (1987)(Codemasters).tap", 44091, 0x4e83dd70, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSupergman, SpecSupergman, Spectrum)
STD_ROM_FN(SpecSupergman)

struct BurnDriver BurnSpecSupergman = {
	"spec_supergman", NULL, "spec_spectrum", NULL, "1987",
	"Super G-Man (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecSupergmanRomInfo, SpecSupergmanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Super Hero (48K)

static struct BurnRomInfo SpecSuperheroRomDesc[] = {
	{ "Super Hero (1988)(Codemasters).tap", 51573, 0x26d19439, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSuperhero, SpecSuperhero, Spectrum)
STD_ROM_FN(SpecSuperhero)

struct BurnDriver BurnSpecSuperhero = {
	"spec_superhero", NULL, "spec_spectrum", NULL, "1988",
	"Super Hero (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecSuperheroRomInfo, SpecSuperheroRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Super Robin Hood (48K)

static struct BurnRomInfo SpecSuperrobinhoodRomDesc[] = {
	{ "Super Robin Hood (1987)(Codemasters).tap", 47900, 0x61d90998, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSuperrobinhood, SpecSuperrobinhood, Spectrum)
STD_ROM_FN(SpecSuperrobinhood)

struct BurnDriver BurnSpecSuperrobinhood = {
	"spec_superrobinhood", NULL, "spec_spectrum", NULL, "1987",
	"Super Robin Hood (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecSuperrobinhoodRomInfo, SpecSuperrobinhoodRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Super Scramble Simulator (128K)

static struct BurnRomInfo SpecSuperscramblesimRomDesc[] = {
	{ "Super Scramble Simulator (1989)(Gremlin Graphics).tap", 56996, 0x1454d1d7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSuperscramblesim, SpecSuperscramblesim, Spec128)
STD_ROM_FN(SpecSuperscramblesim)

struct BurnDriver BurnSpecSuperscramblesim = {
	"spec_superscramblesim", NULL, "spec_spec128", NULL, "1989",
	"Super Scramble Simulator (128K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecSuperscramblesimRomInfo, SpecSuperscramblesimRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Super Seymour Saves the Planet (128K)

static struct BurnRomInfo SpecSuperseymoursavestheplanetRomDesc[] = {
	{ "Super Seymour Saves the Planet (1991)(Codemasters).tap", 65869, 0xd422deed, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSuperseymoursavestheplanet, SpecSuperseymoursavestheplanet, Spec128)
STD_ROM_FN(SpecSuperseymoursavestheplanet)

struct BurnDriver BurnSpecSuperseymoursavestheplanet = {
	"spec_superseymoursavestheplanet", NULL, "spec_spec128", NULL, "1991",
	"Super Seymour Saves the Planet (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecSuperseymoursavestheplanetRomInfo, SpecSuperseymoursavestheplanetRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Super Stuntman (48K)

static struct BurnRomInfo SpecSuperstuntmanRomDesc[] = {
	{ "Super Stuntman (1988)(Codemasters).tap", 43549, 0x984b4f12, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSuperstuntman, SpecSuperstuntman, Spectrum)
STD_ROM_FN(SpecSuperstuntman)

struct BurnDriver BurnSpecSuperstuntman = {
	"spec_superstuntman", NULL, "spec_spectrum", NULL, "1988",
	"Super Stuntman (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecSuperstuntmanRomInfo, SpecSuperstuntmanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Super Tank Simulator (128K)

static struct BurnRomInfo SpecSupertanksimRomDesc[] = {
	{ "Super Tank Simulator (1989)(Codemasters).tap", 44848, 0xd88ed48a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSupertanksim, SpecSupertanksim, Spec128)
STD_ROM_FN(SpecSupertanksim)

struct BurnDriver BurnSpecSupertanksim = {
	"spec_supertanksim", NULL, "spec_spec128", NULL, "1989",
	"Super Tank Simulator (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecSupertanksimRomInfo, SpecSupertanksimRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Tarzan (128K)

static struct BurnRomInfo SpecTarzanRomDesc[] = {
	{ "Tarzan (1986)(Martech Games).tap", 56277, 0xe95f4ac1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTarzan, SpecTarzan, Spec128)
STD_ROM_FN(SpecTarzan)

struct BurnDriver BurnSpecTarzan = {
	"spec_tarzan", NULL, "spec_spec128", NULL, "1986",
	"Tarzan (128K)\0", NULL, "Martech Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecTarzanRomInfo, SpecTarzanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Tarzan Goes Ape (48K)

static struct BurnRomInfo SpecTarzangoesapeRomDesc[] = {
	{ "Tarzan Goes Ape (1991)(Codemasters).tap", 47470, 0x558dadb9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTarzangoesape, SpecTarzangoesape, Spectrum)
STD_ROM_FN(SpecTarzangoesape)

struct BurnDriver BurnSpecTarzangoesape = {
	"spec_tarzangoesape", NULL, "spec_spectrum", NULL, "1991",
	"Tarzan Goes Ape (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecTarzangoesapeRomInfo, SpecTarzangoesapeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Terra Cognita - The Hollow Moon (48K)

static struct BurnRomInfo SpecTerracognitaRomDesc[] = {
	{ "Terra Cognita - The Hollow Moon (1986)(Codemasters).tap", 39094, 0x52ef2dd2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTerracognita, SpecTerracognita, Spectrum)
STD_ROM_FN(SpecTerracognita)

struct BurnDriver BurnSpecTerracognita = {
	"spec_terracognita", NULL, "spec_spectrum", NULL, "1986",
	"Terra Cognita - The Hollow Moon (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecTerracognitaRomInfo, SpecTerracognitaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 007 - The Living Daylights (48K)

static struct BurnRomInfo SpecThelivingdaylightsRomDesc[] = {
	{ "The Living Daylights (1987)(Domark).tap", 38808, 0x9945b8ca, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecThelivingdaylights, SpecThelivingdaylights, Spectrum)
STD_ROM_FN(SpecThelivingdaylights)

struct BurnDriver BurnSpecThelivingdaylights = {
	"spec_thelivingdaylights", NULL, "spec_spectrum", NULL, "1987",
	"007 - The Living Daylights (48K)\0", NULL, "Domark", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_SHOOT, 0,
	SpectrumGetZipName, SpecThelivingdaylightsRomInfo, SpecThelivingdaylightsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Tilt (128K)

static struct BurnRomInfo SpecTiltRomDesc[] = {
	{ "Tilt (1990)(Codemasters).tap", 39893, 0xa03922c0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTilt, SpecTilt, Spec128)
STD_ROM_FN(SpecTilt)

struct BurnDriver BurnSpecTilt = {
	"spec_tilt", NULL, "spec_spec128", NULL, "1990",
	"Tilt (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecTiltRomInfo, SpecTiltRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Tornado ECR (48K)

static struct BurnRomInfo SpecTornadoecrRomDesc[] = {
	{ "Tornado ECR (1991)(Codemasters).tap", 47225, 0x7fe902cb, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTornadoecr, SpecTornadoecr, Spectrum)
STD_ROM_FN(SpecTornadoecr)

struct BurnDriver BurnSpecTornadoecr = {
	"spec_tornadoecr", NULL, "spec_spectrum", NULL, "1991",
	"Tornado ECR (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecTornadoecrRomInfo, SpecTornadoecrRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Transmuter (48K)

static struct BurnRomInfo SpecTransmuterRomDesc[] = {
	{ "Transmuter (1987)(Codemasters).tap", 48599, 0x626052a5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTransmuter, SpecTransmuter, Spectrum)
STD_ROM_FN(SpecTransmuter)

struct BurnDriver BurnSpecTransmuter = {
	"spec_transmuter", NULL, "spec_spectrum", NULL, "1987",
	"Transmuter (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecTransmuterRomInfo, SpecTransmuterRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Turbo Girl (48K)

static struct BurnRomInfo SpecTurbogirlRomDesc[] = {
	{ "Turbo Girl (1988)(Dinamic).tap", 48800, 0x358aaac4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTurbogirl, SpecTurbogirl, Spectrum)
STD_ROM_FN(SpecTurbogirl)

struct BurnDriver BurnSpecTurbogirl = {
	"spec_turbogirl", NULL, "spec_spectrum", NULL, "1988",
	"Turbo Girl (48K)\0", "Press SPACE to jump", "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecTurbogirlRomInfo, SpecTurbogirlRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Turbo the Tortoise (128K)

static struct BurnRomInfo SpecTurbothetortoiseRomDesc[] = {
	{ "Turbo the Tortoise (1992)(Hi-Tec Software).tap", 74768, 0x55513db0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTurbothetortoise, SpecTurbothetortoise, Spec128)
STD_ROM_FN(SpecTurbothetortoise)

struct BurnDriver BurnSpecTurbothetortoise = {
	"spec_turbothetortoise", NULL, "spec_spec128", NULL, "1992",
	"Turbo the Tortoise (128K)\0", NULL, "Hi-Tec Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecTurbothetortoiseRomInfo, SpecTurbothetortoiseRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Twin Turbo V8 (128K)

static struct BurnRomInfo SpecTwinturbov8RomDesc[] = {
	{ "Twin Turbo V8 (1989)(Codemasters).tap", 49133, 0x98b8d8ff, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTwinturbov8, SpecTwinturbov8, Spec128)
STD_ROM_FN(SpecTwinturbov8)

struct BurnDriver BurnSpecTwinturbov8 = {
	"spec_twinturbov8", NULL, "spec_spec128", NULL, "1989",
	"Twin Turbo V8 (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecTwinturbov8RomInfo, SpecTwinturbov8RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Vixen - Part 1 (128K)

static struct BurnRomInfo SpecVixen1RomDesc[] = {
	{ "Vixen - Part 1 (1988)(Martech Games).tap", 49543, 0xe5d1c4a9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecVixen1, SpecVixen1, Spec128)
STD_ROM_FN(SpecVixen1)

struct BurnDriver BurnSpecVixen1 = {
	"spec_vixen1", NULL, "spec_spec128", NULL, "1988",
	"Vixen - Part 1 (128K)\0", NULL, "Martech Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecVixen1RomInfo, SpecVixen1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Vixen - Part 2 (128K)

static struct BurnRomInfo SpecVixen2RomDesc[] = {
	{ "Vixen - Part 2 (1988)(Martech Games).tap", 49547, 0xba54ec0b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecVixen2, SpecVixen2, Spec128)
STD_ROM_FN(SpecVixen2)

struct BurnDriver BurnSpecVixen2 = {
	"spec_vixen2", "spec_vixen1", "spec_spec128", NULL, "1988",
	"Vixen - Part 2 (128K)\0", NULL, "Martech Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecVixen2RomInfo, SpecVixen2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Vixen - Part 3 (128K)

static struct BurnRomInfo SpecVixen3RomDesc[] = {
	{ "Vixen - Part 3 (1988)(Martech Games).tap", 49547, 0x1d0845d9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecVixen3, SpecVixen3, Spec128)
STD_ROM_FN(SpecVixen3)

struct BurnDriver BurnSpecVixen3 = {
	"spec_vixen3", "spec_vixen1", "spec_spec128", NULL, "1988",
	"Vixen - Part 3 (128K)\0", NULL, "Martech Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecVixen3RomInfo, SpecVixen3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Wacky Darts (128K)

static struct BurnRomInfo SpecWackydartsRomDesc[] = {
	{ "Wacky Darts (1991)(Codemasters)(128K).z80", 58652, 0x6790db75, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecWackydarts, SpecWackydarts, Spec128)
STD_ROM_FN(SpecWackydarts)

struct BurnDriver BurnSpecWackydarts = {
	"spec_wackydarts", NULL, "spec_spec128", NULL, "1991",
	"Wacky Darts (128K)\0", "In options menu: keys '6', '7' and ENTER to choose and set", "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 8, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecWackydartsRomInfo, SpecWackydartsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// White Heat (48K)

static struct BurnRomInfo SpecWhiteheatRomDesc[] = {
	{ "White Heat (1987)(Codemasters).tap", 35611, 0x37f5048a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecWhiteheat, SpecWhiteheat, Spectrum)
STD_ROM_FN(SpecWhiteheat)

struct BurnDriver BurnSpecWhiteheat = {
	"spec_whiteheat", NULL, "spec_spectrum", NULL, "1987",
	"White Heat (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecWhiteheatRomInfo, SpecWhiteheatRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Wizard Willy (128K)

static struct BurnRomInfo SpecWizardwillyRomDesc[] = {
	{ "Wizard Willy (1990)(Codemasters).tap", 45968, 0x42cd6c9d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecWizardwilly, SpecWizardwilly, Spec128)
STD_ROM_FN(SpecWizardwilly)

struct BurnDriver BurnSpecWizardwilly = {
	"spec_wizardwilly", NULL, "spec_spec128", NULL, "1990",
	"Wizard Willy (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecWizardwillyRomInfo, SpecWizardwillyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Wrestling Superstars (128K)

static struct BurnRomInfo SpecWrestlingsuperstarsRomDesc[] = {
	{ "Wrestling Superstars (1993)(Codemasters).z80", 41173, 0x0b21ab8e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecWrestlingsuperstars, SpecWrestlingsuperstars, Spec128)
STD_ROM_FN(SpecWrestlingsuperstars)

struct BurnDriver BurnSpecWrestlingsuperstars = {
	"spec_wrestlingsuperstars", NULL, "spec_spec128", NULL, "1992",
	"Wrestling Superstars (128K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpecWrestlingsuperstarsRomInfo, SpecWrestlingsuperstarsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hero Quest (128K)

static struct BurnRomInfo SpecHeroquestRomDesc[] = {
	{ "Hero Quest (1991)(Gremlin Graphics).tap", 111870, 0x76faec29, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHeroquest, SpecHeroquest, Spec128)
STD_ROM_FN(SpecHeroquest)

struct BurnDriver BurnSpecHeroquest = {
	"spec_heroquest", NULL, "spec_spec128", NULL, "1991",
	"Hero Quest (128K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RPG, 0,
	SpectrumGetZipName, SpecHeroquestRomInfo, SpecHeroquestRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hero Quest - Return of the Witch Lord (128K)

static struct BurnRomInfo SpecHeroquestwitchlordRomDesc[] = {
	{ "Hero Quest - Return of the Witch Lord (1991)(Gremlin Graphics).tap", 131173, 0x65141180, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHeroquestwitchlord, SpecHeroquestwitchlord, Spec128)
STD_ROM_FN(SpecHeroquestwitchlord)

struct BurnDriver BurnSpecHeroquestwitchlord = {
	"spec_heroquestwitchlord", NULL, "spec_spec128", NULL, "1991",
	"Hero Quest - Return of the Witch Lord (128K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RPG, 0,
	SpectrumGetZipName, SpecHeroquestwitchlordRomInfo, SpecHeroquestwitchlordRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// ATF - Advanced Tactical Fighter (128K)

static struct BurnRomInfo SpecAtf128kRomDesc[] = {
	{ "ATF 128K (1988)(Digital Integration).z80", 43769, 0x23f0731e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAtf128k, SpecAtf128k, Spec128)
STD_ROM_FN(SpecAtf128k)

struct BurnDriver BurnSpecAtf128k = {
	"spec_atf128k", NULL, "spec_spec128", NULL, "1988",
	"ATF - Advanced Tactical Fighter (128K)\0", NULL, "Digital Integration", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SIM | GBF_SHOOT, 0,
	SpectrumGetZipName, SpecAtf128kRomInfo, SpecAtf128kRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hercules - Slayer of the Damned (128K)

static struct BurnRomInfo SpecHercslayerRomDesc[] = {
	{ "Hercules - Slayer of the Damned (1988)(Gremlin Graphics).tap", 52958, 0xc6ca45e6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHercslayer, SpecHercslayer, Spec128)
STD_ROM_FN(SpecHercslayer)

struct BurnDriver BurnSpecHercslayer = {
	"spec_hercslayer", NULL, "spec_spec128", NULL, "1988",
	"Hercules - Slayer of the Damned (128K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpecHercslayerRomInfo, SpecHercslayerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Klax (128K)

static struct BurnRomInfo SpecKlaxRomDesc[] = {
	{ "Klax 128K (1990)(Domark).tap", 138496, 0x591ded31, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecKlax, SpecKlax, Spec128)
STD_ROM_FN(SpecKlax)

struct BurnDriver BurnSpecKlax = {
	"spec_klax", NULL, "spec_spec128", NULL, "1990",
	"Klax (128K)\0", NULL, "Domark", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecKlaxRomInfo, SpecKlaxRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mickey Mouse - The Computer Game (48K)

static struct BurnRomInfo SpecMickeymouseRomDesc[] = {
	{ "Mickey Mouse - The Computer Game 48K (1988)(Gremlin Graphics).z80", 43296, 0xaa993b54, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMickeymouse, SpecMickeymouse, Spectrum)
STD_ROM_FN(SpecMickeymouse)

struct BurnDriver BurnSpecMickeymouse = {
	"spec_mickeymouse", NULL, "spec_spectrum", NULL, "1988",
	"Mickey Mouse - The Computer Game (48K)\0", "SPACE: change weapon", "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecMickeymouseRomInfo, SpecMickeymouseRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Road Blasters (128K)

static struct BurnRomInfo SpecRblasterRomDesc[] = {
	{ "Road Blasters 128K (1988)(U.S. Gold).tap", 125428, 0x44e22515, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecRblaster, SpecRblaster, Spec128)
STD_ROM_FN(SpecRblaster)

struct BurnDriver BurnSpecRblaster = {
	"spec_rblaster", NULL, "spec_spec128", NULL, "1988",
	"Road Blasters (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING | GBF_SHOOT, 0,
	SpectrumGetZipName, SpecRblasterRomInfo, SpecRblasterRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Road Runner and Wile E. Coyote (48K)

static struct BurnRomInfo SpecRoadwile48RomDesc[] = {
	{ "Road Runner and Wile E. Coyote (1991)(Hi-Tec).tap", 50312, 0x766bd6a4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecRoadwile48, SpecRoadwile48, Spectrum)
STD_ROM_FN(SpecRoadwile48)

struct BurnDriver BurnSpecRoadwile = {
	"spec_roadwile48", "spec_roadwile128", "spec_spectrum", NULL, "1991",
	"Road Runner and Wile E. Coyote (48K)\0", NULL, "Hi-Tec Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecRoadwile48RomInfo, SpecRoadwile48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Road Runner and Wile E. Coyote (128K)

static struct BurnRomInfo SpecRoadwile128RomDesc[] = {
	{ "Road Runner and Wile E. Coyote (1991)(Hi-Tec).tap", 50312, 0x766bd6a4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecRoadwile128, SpecRoadwile128, Spec128)
STD_ROM_FN(SpecRoadwile128)

struct BurnDriver BurnSpecRoadwile128 = {
	"spec_roadwile128", NULL, "spec_spec128", NULL, "1991",
	"Road Runner and Wile E. Coyote (128K)\0", NULL, "Hi-Tec Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecRoadwile128RomInfo, SpecRoadwile128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Samurai Warrior - Usagi Yojimbo (128K)

static struct BurnRomInfo SpecSamwarrior128kRomDesc[] = {
	{ "Samurai Warrior - Usagi Yojimbo 128k (1988)(Firebird).tap", 53473, 0x989e9ffd, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSamwarrior128k, SpecSamwarrior128k, Spec128)
STD_ROM_FN(SpecSamwarrior128k)

struct BurnDriver BurnSpecSamwarrior128k = {
	"spec_samwarrior128k", NULL, "spec_spec128", NULL, "1988",
	"Samurai Warrior - Usagi Yojimbo (128K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecSamwarrior128kRomInfo, SpecSamwarrior128kRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Thundercats (128K)

static struct BurnRomInfo SpecthundcatsRomDesc[] = {
	{ "Thundercats (1987)(Elite Systems).tap", 94147, 0x76e19a9c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specthundcats, Specthundcats, Spec128)
STD_ROM_FN(Specthundcats)

struct BurnDriver BurnSpecthundcats = {
	"spec_thundcats", NULL, "spec_spec128", NULL, "1987",
	"Thundercats (128K)\0", NULL, "Elite Systems", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT | GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecthundcatsRomInfo, SpecthundcatsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hammerfist (48K)

static struct BurnRomInfo SpecHammerfistRomDesc[] = {
	{ "Hammerfist (1990)(Activision).tap", 188831, 0xc6ad835b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHammerfist, SpecHammerfist, Spectrum)
STD_ROM_FN(SpecHammerfist)

struct BurnDriver BurnSpecHammerfist = {
	"spec_hammerfist", NULL, "spec_spectrum", NULL, "1990",
	"Hammerfist (48K)\0", NULL, "Activision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecHammerfistRomInfo, SpecHammerfistRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hong Kong Phooey (48K)

static struct BurnRomInfo SpecHkphooeyRomDesc[] = {
	{ "Hong Kong Phooey 48K (1990)(Hi-Tec).z80", 37410, 0x444a5064, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHkphooey, SpecHkphooey, Spectrum)
STD_ROM_FN(SpecHkphooey)

struct BurnDriver BurnSpecHkphooey = {
	"spec_hkphooey", NULL, "spec_spectrum", NULL, "1990",
	"Hong Kong Phooey (48K)\0", NULL, "Hi-Tec Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecHkphooeyRomInfo, SpecHkphooeyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hysteria (48K)

static struct BurnRomInfo SpecHysteriaRomDesc[] = {
	{ "Hysteria 48K (1987)(Software Projects).tap", 47426, 0x70fcbb59, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHysteria, SpecHysteria, Spectrum)
STD_ROM_FN(SpecHysteria)

struct BurnDriver BurnSpecHysteria = {
	"spec_hysteria", NULL, "spec_spectrum", NULL, "1987",
	"Hysteria (48K)\0", NULL, "Software Projects", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecHysteriaRomInfo, SpecHysteriaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// International Karate - Stage 1 (48K)

static struct BurnRomInfo SpecIntkarates1RomDesc[] = {
	{ "International Karate stage 1 (1985)(System 3).tap", 41266, 0xefa7860d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecIntkarates1, SpecIntkarates1, Spectrum)
STD_ROM_FN(SpecIntkarates1)

struct BurnDriver BurnSpecIntkarates1 = {
	"spec_intkarates1", NULL, "spec_spectrum", NULL, "1985",
	"International Karate - Stage 1 (48K)\0", NULL, "System 3", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpecIntkarates1RomInfo, SpecIntkarates1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// International Karate - Stage 2 (48K)

static struct BurnRomInfo SpecIntkarates2RomDesc[] = {
	{ "International Karate stage 2 (1985)(System 3).tap", 47713, 0x89bfad75, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecIntkarates2, SpecIntkarates2, Spectrum)
STD_ROM_FN(SpecIntkarates2)

struct BurnDriver BurnSpecIntkarates2 = {
	"spec_intkarates2", "spec_intkarates1", "spec_spectrum", NULL, "1985",
	"International Karate - Stage 2 (48K)\0", NULL, "System 3", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpecIntkarates2RomInfo, SpecIntkarates2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Pentagram (48K)

static struct BurnRomInfo SpecPentagramRomDesc[] = {
	{ "Pentagram 48K (1986)(Ultimate).tap", 38524, 0x4d43b15a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPentagram, SpecPentagram, Spectrum)
STD_ROM_FN(SpecPentagram)

struct BurnDriver BurnSpecPentagram = {
	"spec_pentagram", NULL, "spec_spectrum", NULL, "1986",
	"Pentagram (48K)\0", NULL, "Ultimate", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecPentagramRomInfo, SpecPentagramRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Pyjamarama (48K)

static struct BurnRomInfo SpecPyjamaramaRomDesc[] = {
	{ "Pyjamarama 48K (1984)(Mikro-Gen).z80", 42069, 0x6801d962, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPyjamarama, SpecPyjamarama, Spectrum)
STD_ROM_FN(SpecPyjamarama)

struct BurnDriver BurnSpecPyjamarama = {
	"spec_pyjamarama", NULL, "spec_spectrum", NULL, "1984",
	"Pyjamarama (48K)\0", NULL, "Mikro-Gen", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecPyjamaramaRomInfo, SpecPyjamaramaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Scooby Doo in the Castle Mystery (48K)

static struct BurnRomInfo SpecScoobydooRomDesc[] = {
	{ "Scooby Doo in the Castle Mystery 48K (1986)(Elite Systems).tap", 46057, 0xf1436e40, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecScoobydoo, SpecScoobydoo, Spectrum)
STD_ROM_FN(SpecScoobydoo)

struct BurnDriver BurnSpecScoobydoo = {
	"spec_scoobydoo", NULL, "spec_spectrum", NULL, "1986",
	"Scooby Doo in the Castle Mystery (48K)\0", "Press 'J' to controller menu", "Elite Systems", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecScoobydooRomInfo, SpecScoobydooRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Super Cars (128K)

static struct BurnRomInfo SpecSupercarsRomDesc[] = {
	{ "Super Cars 128K (1990)(Gremlin Graphics).z80", 88269, 0x5a2d86ca, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSupercars, SpecSupercars, Spec128)
STD_ROM_FN(SpecSupercars)

struct BurnDriver BurnSpecSupercars = {
	"spec_supercars", NULL, "spec_spec128", NULL, "1990",
	"Super Cars (128K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecSupercarsRomInfo, SpecSupercarsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Total Eclipse (48K)

static struct BurnRomInfo SpecTeclipseRomDesc[] = {
	{ "Total Eclipse 48K (1988)(Incentive).tap", 50769, 0x1e312308, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTeclipse, SpecTeclipse, Spectrum)
STD_ROM_FN(SpecTeclipse)

struct BurnDriver BurnSpecTeclipse = {
	"spec_teclipse", NULL, "spec_spectrum", NULL, "1988",
	"Total Eclipse (48K)\0", "P and L: look up and down - S: step size - SPACE: draw/replace gunsight", "Incentive", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_ADV, 0,
	SpectrumGetZipName, SpecTeclipseRomInfo, SpecTeclipseRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Thanatos (128K)

static struct BurnRomInfo SpecThanatos128RomDesc[] = {
	{ "Thanatos 128K (1986)(Durell).tap", 49266, 0x4ce5e54e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecThanatos128, SpecThanatos128, Spec128)
STD_ROM_FN(SpecThanatos128)

struct BurnDriver BurnSpecThanatos128 = {
	"spec_thanatos128", NULL, "spec_spec128", NULL, "1986",
	"Thanatos (128K)\0", NULL, "Durell Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecThanatos128RomInfo, SpecThanatos128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Thanatos (48K)

static struct BurnRomInfo SpecThanatos48RomDesc[] = {
	{ "Thanatos 48K (1986)(Durell).tap", 43309, 0xefe49f3e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecThanatos48, SpecThanatos48, Spectrum)
STD_ROM_FN(SpecThanatos48)

struct BurnDriver BurnSpecThanatos48 = {
	"spec_thanatos48", "spec_thanatos128", "spec_spectrum", NULL, "1986",
	"Thanatos (48K)\0", NULL, "Durell Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecThanatos48RomInfo, SpecThanatos48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Alchemist (48K)

static struct BurnRomInfo SpecAlchemistRomDesc[] = {
	{ "Alchemist (1983)(Imagine).z80", 37807, 0xe6f0df00, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAlchemist, SpecAlchemist, Spectrum)
STD_ROM_FN(SpecAlchemist)

struct BurnDriver BurnSpecAlchemist = {
	"spec_alchemist", NULL, "spec_spectrum", NULL, "1983",
	"Alchemist (48K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecAlchemistRomInfo, SpecAlchemistRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Buggy Boy (128K)

static struct BurnRomInfo SpecBuggyboyRomDesc[] = {
	{ "Buggy Boy 128K (1988)(Elite Systems).tap", 53512, 0x9f71d593, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBuggyboy, SpecBuggyboy, Spec128)
STD_ROM_FN(SpecBuggyboy)

struct BurnDriver BurnSpecBuggyboy = {
	"spec_buggyboy", NULL, "spec_spec128", NULL, "1988",
	"Buggy Boy (128K)\0", "Use 'P', 'L' and SPACE to set options", "Elite Systems", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecBuggyboyRomInfo, SpecBuggyboyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Flash Gordon (48K)

static struct BurnRomInfo SpecFlashgordonRomDesc[] = {
	{ "Flash Gordon (1987)(Mastertronic Added Dimension).tap", 76147, 0x6fd40ff2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFlashgordon, SpecFlashgordon, Spectrum)
STD_ROM_FN(SpecFlashgordon)

struct BurnDriver BurnSpecFlashgordon = {
	"spec_flashgordon", NULL, "spec_spectrum", NULL, "1987",
	"Flash Gordon (48K)\0", NULL, "Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_ADV, 0,
	SpectrumGetZipName, SpecFlashgordonRomInfo, SpecFlashgordonRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Garfield - Winter's Tail (48K)

static struct BurnRomInfo SpecGarfield2RomDesc[] = {
	{ "Garfield - Winter's Tail 48K (1990)(The Edge).tap", 58470, 0x74f9f23e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGarfield2, SpecGarfield2, Spectrum)
STD_ROM_FN(SpecGarfield2)

struct BurnDriver BurnSpecGarfield2 = {
	"spec_garfield2", NULL, "spec_spectrum", NULL, "1990",
	"Garfield - Winter's Tail (48K)\0", NULL, "The Edge", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecGarfield2RomInfo, SpecGarfield2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Kinetik (48K)

static struct BurnRomInfo SpecKinetikRomDesc[] = {
	{ "Kinetik 48K (1987)(Firebird).z80", 43221, 0x39a23892, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecKinetik, SpecKinetik, Spectrum)
STD_ROM_FN(SpecKinetik)

struct BurnDriver BurnSpecKinetik = {
	"spec_kinetik", NULL, "spec_spectrum", NULL, "1987",
	"Kinetik (48K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecKinetikRomInfo, SpecKinetikRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// LED Storm - Rally 2011 (128K)

static struct BurnRomInfo SpecLedstormRomDesc[] = {
	{ "LED Storm 128K (1988)(Go!).tap", 86247, 0xd20b9e67, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecLedstorm, SpecLedstorm, Spec128)
STD_ROM_FN(SpecLedstorm)

struct BurnDriver BurnSpecLedstorm = {
	"spec_ledstorm", NULL, "spec_spec128", NULL, "1988",
	"LED Storm - Rally 2011 (128K)\0", NULL, "Go!", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_RACING, 0,
	SpectrumGetZipName, SpecLedstormRomInfo, SpecLedstormRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// S.T.U.N. Runner (128K)

static struct BurnRomInfo SpecStunrunRomDesc[] = {
	{ "S.T.U.N. Runner 128K (1990)(Domark).tap", 60868, 0x3aec4137, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecStunrun, SpecStunrun, Spec128)
STD_ROM_FN(SpecStunrun)

struct BurnDriver BurnSpecStunrun = {
	"spec_stunrun", NULL, "spec_spec128", NULL, "1990",
	"S.T.U.N. Runner (128K)\0", NULL, "Domark", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING | GBF_SHOOT, 0,
	SpectrumGetZipName, SpecStunrunRomInfo, SpecStunrunRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Starglider (128K)

static struct BurnRomInfo SpecStarglider128kRomDesc[] = {
	{ "Starglider 128K (1986)(Rainbird).tap", 113280, 0x5403a025, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecStarglider128k, SpecStarglider128k, Spec128)
STD_ROM_FN(SpecStarglider128k)

struct BurnDriver BurnSpecStarglider128k = {
	"spec_starglider128k", NULL, "spec_spec128", NULL, "1986",
	"Starglider (128K)\0", NULL, "Rainbird", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_SIM, 0,
	SpectrumGetZipName, SpecStarglider128kRomInfo, SpecStarglider128kRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Starglider (48K)

static struct BurnRomInfo SpecStarglider48kRomDesc[] = {
	{ "Starglider 48K (1986)(Rainbird).tap", 49104, 0x48258fa3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecStarglider48k, SpecStarglider48k, Spectrum)
STD_ROM_FN(SpecStarglider48k)

struct BurnDriver BurnSpecStarglider48k = {
	"spec_starglider48k", "spec_starglider128k", "spec_spectrum", NULL, "1986",
	"Starglider (48K)\0", NULL, "Rainbird", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_SIM, 0,
	SpectrumGetZipName, SpecStarglider48kRomInfo, SpecStarglider48kRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Viz - The Computer Game (128K)

static struct BurnRomInfo SpecVizRomDesc[] = {
	{ "Viz - The Computer Game 128K (1991)(Virgin Games).tap", 167642, 0x005482f2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecViz, SpecViz, Spec128)
STD_ROM_FN(SpecViz)

struct BurnDriver BurnSpecViz = {
	"spec_viz", NULL, "spec_spec128", NULL, "1991",
	"Viz - The Computer Game (128K)\0", NULL, "Virgin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MISC, 0,
	SpectrumGetZipName, SpecVizRomInfo, SpecVizRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Yogi Bear (48K)

static struct BurnRomInfo SpecYogibearRomDesc[] = {
	{ "Yogi Bear 48K (1987)(Piranha).z80", 43670, 0x437fa194, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecYogibear, SpecYogibear, Spectrum)
STD_ROM_FN(SpecYogibear)

struct BurnDriver BurnSpecYogibear = {
	"spec_yogibear", NULL, "spec_spectrum", NULL, "1987",
	"Yogi Bear (48K)\0", NULL, "Piranha", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecYogibearRomInfo, SpecYogibearRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Yogi's Great Escape (48K)

static struct BurnRomInfo SpecYogigreatescapeRomDesc[] = {
	{ "Yogi's Great Escape (1990)(Hi-Tec).tap", 54731, 0xd6afad15, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecYogigreatescape, SpecYogigreatescape, Spectrum)
STD_ROM_FN(SpecYogigreatescape)

struct BurnDriver BurnSpecYogigreatescape = {
	"spec_yogigreatescape", NULL, "spec_spectrum", NULL, "1990",
	"Yogi's Great Escape (48K)\0", NULL, "Hi-Tec Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecYogigreatescapeRomInfo, SpecYogigreatescapeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Yogi Bear and Friends - The Greed Monster (128K)

static struct BurnRomInfo SpecYogigreedmonsterRomDesc[] = {
	{ "Yogi Bear and Friends - The Greed Monster (1990)(Hi-Tec).tap", 56556, 0x601b7f1f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecYogigreedmonster, SpecYogigreedmonster, Spec128)
STD_ROM_FN(SpecYogigreedmonster)

struct BurnDriver BurnSpecYogigreedmonster = {
	"spec_yogigreedmonster", NULL, "spec_spec128", NULL, "1990",
	"Yogi Bear and Friends - The Greed Monster (128K)\0", NULL, "Hi-Tec Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecYogigreedmonsterRomInfo, SpecYogigreedmonsterRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Confuzion (48K)

static struct BurnRomInfo SpecConfuzionRomDesc[] = {
	{ "Confuzion (1985)(Incentive).tap", 49343, 0x9d0f6c6c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecConfuzion, SpecConfuzion, Spectrum)
STD_ROM_FN(SpecConfuzion)

struct BurnDriver BurnSpecConfuzion = {
	"spec_confuzion", NULL, "spec_spectrum", NULL, "1985",
	"Confuzion (48K)\0", "Press R to invert commands", "Incentive", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecConfuzionRomInfo, SpecConfuzionRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Count Duckula in No Sax Please - We're Egyptian (48K)

static struct BurnRomInfo SpecCountduckulaRomDesc[] = {
	{ "Count Duckula in No Sax Please 48K (1989)(Alternative).tap", 48615, 0x2185ec3a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCountduckula, SpecCountduckula, Spectrum)
STD_ROM_FN(SpecCountduckula)

struct BurnDriver BurnSpecCountduckula = {
	"spec_countduckula", NULL, "spec_spectrum", NULL, "1989",
	"Count Duckula in No Sax Please - We're Egyptian (48K)\0", NULL, "Alternative", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecCountduckulaRomInfo, SpecCountduckulaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Gemini Wing (128K)

static struct BurnRomInfo SpecGeminiwingRomDesc[] = {
	{ "Gemini Wing (1989)(Virgin Mastertronic).tap", 144667, 0x171e71ed, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGeminiwing, SpecGeminiwing, Spec128)
STD_ROM_FN(SpecGeminiwing)

struct BurnDriver BurnSpecGeminiwing = {
	"spec_geminiwing", NULL, "spec_spec128", NULL, "1989",
	"Gemini Wing (128K)\0", NULL, "Virgin Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecGeminiwingRomInfo, SpecGeminiwingRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Gunstar (48K)

static struct BurnRomInfo SpecGunstarRomDesc[] = {
	{ "Gunstar 48K (1987)(Firebird).tap", 47715, 0x23aedf6b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGunstar, SpecGunstar, Spectrum)
STD_ROM_FN(SpecGunstar)

struct BurnDriver BurnSpecGunstar = {
	"spec_gunstar", NULL, "spec_spectrum", NULL, "1987",
	"Gunstar (48K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecGunstarRomInfo, SpecGunstarRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Xenophobe (128K)

static struct BurnRomInfo SpecXenophobeRomDesc[] = {
	{ "Xenophobe 128K (1989)(Micro Style).tap", 213558, 0xb32da9b4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecXenophobe, SpecXenophobe, Spec128)
STD_ROM_FN(SpecXenophobe)

struct BurnDriver BurnSpecXenophobe = {
	"spec_xenophobe", NULL, "spec_spec128", NULL, "1989",
	"Xenophobe (128K)\0", NULL, "Micro Style", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecXenophobeRomInfo, SpecXenophobeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sacred Armour of Antiriad, The (48K)

static struct BurnRomInfo SpecAntiriadRomDesc[] = {
	{ "Sacred Armour of Antiriad, The (1986)(Palace).z80", 41012, 0x1abe0529, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAntiriad, SpecAntiriad, Spectrum)
STD_ROM_FN(SpecAntiriad)

struct BurnDriver BurnSpecAntiriad = {
	"spec_antiriad", NULL, "spec_spectrum", NULL, "1986",
	"Sacred Armour of Antiriad, The (48K)\0", NULL, "Palace Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecAntiriadRomInfo, SpecAntiriadRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Impossible Mission (48K)

static struct BurnRomInfo SpecImpmissionRomDesc[] = {
	{ "Impossible Mission (1985)(U.S. Gold).tap", 54480, 0xf3cddbf9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecImpmission, SpecImpmission, Spectrum)
STD_ROM_FN(SpecImpmission)

struct BurnDriver BurnSpecImpmission = {
	"spec_impmission", NULL, "spec_spectrum", NULL, "1985",
	"Impossible Mission (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecImpmissionRomInfo, SpecImpmissionRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Spherical (48K)

static struct BurnRomInfo SpecSphericalRomDesc[] = {
	{ "Spherical 48K (1989)(Rainbow Arts).tap", 154622, 0xc03fc493, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSpherical, SpecSpherical, Spectrum)
STD_ROM_FN(SpecSpherical)

struct BurnDriver BurnSpecSpherical = {
	"spec_spherical", NULL, "spec_spectrum", NULL, "1989",
	"Spherical (48K)\0", NULL, "Rainbow Arts", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecSphericalRomInfo, SpecSphericalRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Barmy Burgers (48K)

static struct BurnRomInfo SpecBarmyburgRomDesc[] = {
	{ "Barmy Burgers (1983)(Blaby Computer Games).z80", 28248, 0x138faacd, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBarmyburg, SpecBarmyburg, Spectrum)
STD_ROM_FN(SpecBarmyburg)

struct BurnDriver BurnSpecBarmyburg = {
	"spec_barmyburg", NULL, "spec_spectrum", NULL, "1983",
	"Barmy Burgers (48K)\0", NULL, "Blaby Computer Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecBarmyburgRomInfo, SpecBarmyburgRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// D.N.A. Warrior (128K)

static struct BurnRomInfo SpecDnawarriorRomDesc[] = {
	{ "D.N.A. Warrior 128K (1989)(Cascade Games).tap", 52639, 0x86a9a591, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDnawarrior, SpecDnawarrior, Spec128)
STD_ROM_FN(SpecDnawarrior)

struct BurnDriver BurnSpecDnawarrior = {
	"spec_dnawarrior", NULL, "spec_spec128", NULL, "1989",
	"D.N.A. Warrior (128K)\0", NULL, "Cascade Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecDnawarriorRomInfo, SpecDnawarriorRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Defenders of the Earth (128K)

static struct BurnRomInfo SpecDefearthRomDesc[] = {
	{ "Defenders of the Earth 128K (1990)(Enigma Variations).z80", 92616, 0xd0283da9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDefearth, SpecDefearth, Spec128)
STD_ROM_FN(SpecDefearth)

struct BurnDriver BurnSpecDefearth = {
	"spec_defearth", NULL, "spec_spec128", NULL, "1990",
	"Defenders of the Earth (128K)\0", NULL, "Enigma Variations", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecDefearthRomInfo, SpecDefearthRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Equinox (48K)

static struct BurnRomInfo SpecEquinoxRomDesc[] = {
	{ "Equinox (1986)(Mikro-Gen).z80", 45770, 0xaaa2864d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecEquinox, SpecEquinox, Spectrum)
STD_ROM_FN(SpecEquinox)

struct BurnDriver BurnSpecEquinox = {
	"spec_equinox", NULL, "spec_spectrum", NULL, "1986",
	"Equinox (48K)\0", NULL, "Mikro-Gen", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_MAZE, 0,
	SpectrumGetZipName, SpecEquinoxRomInfo, SpecEquinoxRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Everyone's a Wally (48K)

static struct BurnRomInfo SpecEverywallyRomDesc[] = {
	{ "Everyone's a Wally (1985)(Mikro-Gen).tap", 40178, 0x94e1bce9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecEverywally, SpecEverywally, Spectrum)
STD_ROM_FN(SpecEverywally)

struct BurnDriver BurnSpecEverywally = {
	"spec_everywally", NULL, "spec_spectrum", NULL, "1985",
	"Everyone's a Wally (48K)\0", NULL, "Mikro-Gen", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecEverywallyRomInfo, SpecEverywallyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Grand Prix Circuit (128K)

static struct BurnRomInfo SpecGpcircuitRomDesc[] = {
	{ "Grand Prix Circuit 128K (1990)(Accolade).z80", 60323, 0x1203a59d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGpcircuit, SpecGpcircuit, Spec128)
STD_ROM_FN(SpecGpcircuit)

struct BurnDriver BurnSpecGpcircuit = {
	"spec_gpcircuit", NULL, "spec_spec128", NULL, "1990",
	"Grand Prix Circuit (128K)\0", NULL, "Accolade", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecGpcircuitRomInfo, SpecGpcircuitRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Impossible Mission II (48K)

static struct BurnRomInfo SpecImpmission2RomDesc[] = {
	{ "Impossible Mission II (1988)(U.S. Gold).z80", 44256, 0xea95490a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecImpmission2, SpecImpmission2, Spec128)
STD_ROM_FN(SpecImpmission2)

struct BurnDriver BurnSpecImpmission2 = {
	"spec_impmission2", NULL, "spec_spec128", NULL, "1988",
	"Impossible Mission II (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecImpmission2RomInfo, SpecImpmission2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Impossaball (48K)

static struct BurnRomInfo SpecImpossaballRomDesc[] = {
	{ "Impossaball (1987)(Hewson Consultants).tap", 45993, 0x158525a4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecImpossaball, SpecImpossaball, Spectrum)
STD_ROM_FN(SpecImpossaball)

struct BurnDriver BurnSpecImpossaball = {
	"spec_impossaball", NULL, "spec_spectrum", NULL, "1987",
	"Impossaball (48K)\0", NULL, "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecImpossaballRomInfo, SpecImpossaballRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mr Heli (48K)

static struct BurnRomInfo SpecMrheliRomDesc[] = {
	{ "Mr Heli 48K (1989)(Firebird).tap", 126724, 0xbf869699, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMrheli, SpecMrheli, Spectrum)
STD_ROM_FN(SpecMrheli)

struct BurnDriver BurnSpecMrheli = {
	"spec_mrheli", NULL, "spec_spectrum", NULL, "1989",
	"Mr Heli (48K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecMrheliRomInfo, SpecMrheliRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Zorro (48K)

static struct BurnRomInfo SpecZorroRomDesc[] = {
	{ "Zorro (1986)(U.S. Gold).z80", 37618, 0x9f7b2443, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecZorro, SpecZorro, Spectrum)
STD_ROM_FN(SpecZorro)

struct BurnDriver BurnSpecZorro = {
	"spec_zorro", NULL, "spec_spectrum", NULL, "1986",
	"Zorro (48K)\0", "After options menu press ENTER to begin the game", "U.S. Gold - DataSoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecZorroRomInfo, SpecZorroRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Cobra (48K)

static struct BurnRomInfo SpecCobraRomDesc[] = {
	{ "Cobra (1986)(Ocean).z80", 42218, 0x28a20fe7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCobra, SpecCobra, Spectrum)
STD_ROM_FN(SpecCobra)

struct BurnDriver BurnSpecCobra = {
	"spec_cobra", NULL, "spec_spectrum", NULL, "1986",
	"Cobra (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecCobraRomInfo, SpecCobraRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Extreme (48K)

static struct BurnRomInfo SpecExtremeRomDesc[] = {
	{ "Extreme (1991)(Digital Integration).tap", 54612, 0xdc41f6f2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecExtreme, SpecExtreme, Spectrum)
STD_ROM_FN(SpecExtreme)

struct BurnDriver BurnSpecExtreme = {
	"spec_extreme", NULL, "spec_spectrum", NULL, "1991",
	"Extreme (48K)\0", NULL, "Digital Integration", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecExtremeRomInfo, SpecExtremeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Superman - The Man of Steel (48K)

static struct BurnRomInfo SpecSupermanRomDesc[] = {
	{ "Superman - The Man of Steel (1989)(Tynesoft).tap", 300217, 0x578fc970, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSuperman, SpecSuperman, Spectrum)
STD_ROM_FN(SpecSuperman)

struct BurnDriver BurnSpecSuperman = {
	"spec_superman", NULL, "spec_spectrum", NULL, "1989",
	"Superman - The Man of Steel (48K)\0", "'6', '7' and ENTER to select options", "Tynesoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecSupermanRomInfo, SpecSupermanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 3D Snooker (48K)

static struct BurnRomInfo Spec3dsnookerRomDesc[] = {
	{ "3D Snooker (1990)(Players Premier).z80", 29049, 0x8dfff22d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec3dsnooker, Spec3dsnooker, Spectrum)
STD_ROM_FN(Spec3dsnooker)

struct BurnDriver BurnSpec3dsnooker = {
	"spec_3dsnooker", NULL, "spec_spectrum", NULL, "1990",
	"3D Snooker (48K)\0", NULL, "Players Premier", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, Spec3dsnookerRomInfo, Spec3dsnookerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Comando Tracer (Spanish) (48K-128K)

static struct BurnRomInfo SpecCotracerRomDesc[] = {
	{ "Comando Tracer (1989)(Dinamic).tap", 41721, 0xd394ef8c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCotracer, SpecCotracer, Spec128)
STD_ROM_FN(SpecCotracer)

struct BurnDriver BurnSpecCotracer = {
	"spec_cotracer", NULL, "spec_spec128", NULL, "1988",
	"Comando Tracer (Spanish) (48K-128K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecCotracerRomInfo, SpecCotracerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Flintstones, the (128K)

static struct BurnRomInfo SpecFlintstonesRomDesc[] = {
	{ "Flintstones, the (1988)(Grandslam).tap", 76902, 0xba15a0de, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFlintstones, SpecFlintstones, Spec128)
STD_ROM_FN(SpecFlintstones)

struct BurnDriver BurnSpecFlintstones = {
	"spec_flintstones", NULL, "spec_spec128", NULL, "1988",
	"Flintstones, the (128K)\0", NULL, "Grandslam Entertainments", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV | GBF_MISC, 0,
	SpectrumGetZipName, SpecFlintstonesRomInfo, SpecFlintstonesRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Jungle Warrior (48K)

static struct BurnRomInfo SpecJwarriorRomDesc[] = {
	{ "Jungle Warrior 48K ES (1990)(Zigurat).tap", 46516, 0x7eaa77fe, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecJwarrior, SpecJwarrior, Spectrum)
STD_ROM_FN(SpecJwarrior)

struct BurnDriver BurnSpecJwarrior = {
	"spec_jwarrior", NULL, "spec_spectrum", NULL, "1990",
	"Jungle Warrior (Spanish) (48K)\0", NULL, "Zigurat Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecJwarriorRomInfo, SpecJwarriorRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mermaid Madness (48K)

static struct BurnRomInfo SpecMermadnesRomDesc[] = {
	{ "Mermaid Madness 48K (1986)(Electric Dreams).tap", 47263, 0x3b981595, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMermadnes, SpecMermadnes, Spectrum)
STD_ROM_FN(SpecMermadnes)

struct BurnDriver BurnSpecMermadnes = {
	"spec_mermadnes", NULL, "spec_spectrum", NULL, "1986",
	"Mermaid Madness (48K)\0", NULL, "Electric Dreams", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE | GBF_ADV, 0,
	SpectrumGetZipName, SpecMermadnesRomInfo, SpecMermadnesRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Metabolis (48K)

static struct BurnRomInfo SpecMetabolisRomDesc[] = {
	{ "Metabolis 48K (1985)(Gremlin Graphics).tap", 56590, 0xd65e3628, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMetabolis, SpecMetabolis, Spectrum)
STD_ROM_FN(SpecMetabolis)

struct BurnDriver BurnSpecMetabolis = {
	"spec_metabolis", NULL, "spec_spectrum", NULL, "1985",
	"Metabolis (48K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE | GBF_ACTION, 0,
	SpectrumGetZipName, SpecMetabolisRomInfo, SpecMetabolisRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mystical (128K)

static struct BurnRomInfo SpecMysticalRomDesc[] = {
	{ "Mystical (1991)(Infogrames).tap", 260015, 0xf7df7363, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMystical, SpecMystical, Spec128)
STD_ROM_FN(SpecMystical)

struct BurnDriver BurnSpecMystical = {
	"spec_mystical", NULL, "spec_spec128", NULL, "1991",
	"Mystical (128K)\0", NULL, "Infogrames", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecMysticalRomInfo, SpecMysticalRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Nemesis the Warlock (128K)

static struct BurnRomInfo SpecNwarlockRomDesc[] = {
	{ "Nemesis the Warlock 128K (1987)(Martech Games).tap", 70991, 0xe8b462f8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecNwarlock, SpecNwarlock, Spec128)
STD_ROM_FN(SpecNwarlock)

struct BurnDriver BurnSpecNwarlock = {
	"spec_nwarlock", NULL, "spec_spec128", NULL, "1987",
	"Nemesis the Warlock (128K)\0", "at intro press Fire and Space to view 'credits+music' screen", "Martech Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecNwarlockRomInfo, SpecNwarlockRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sorcery (48K)

static struct BurnRomInfo SpecSorceryRomDesc[] = {
	{ "Sorcery (1984)(Virgin Games).tap", 35062, 0xac2adaae, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSorcery, SpecSorcery, Spectrum)
STD_ROM_FN(SpecSorcery)

struct BurnDriver BurnSpecSorcery = {
	"spec_sorcery", "spec_sorceryplus", "spec_spectrum", NULL, "1984",
	"Sorcery (48K)\0", "Q: left - A: right - 0: jump", "Virgin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecSorceryRomInfo, SpecSorceryRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Xeno (48K)

static struct BurnRomInfo SpecXenoRomDesc[] = {
	{ "Xeno (1986)(A'n'F).z80", 37264, 0x4d10e27a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecXeno, SpecXeno, Spectrum)
STD_ROM_FN(SpecXeno)

struct BurnDriver BurnSpecXeno = {
	"spec_xeno", NULL, "spec_spectrum", NULL, "1986",
	"Xeno (48K)\0", NULL, "A'n'F Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecXenoRomInfo, SpecXenoRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Yabba Dabba Doo! - Flintstones (48K)

static struct BurnRomInfo SpecYabbadooRomDesc[] = {
	{ "Yabba Dabba Doo! 48K (1985)(Quicksilva).tap", 45016, 0xc13ebf67, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecYabbadoo, SpecYabbadoo, Spectrum)
STD_ROM_FN(SpecYabbadoo)

struct BurnDriver BurnSpecYabbadoo = {
	"spec_yabbadoo", NULL, "spec_spectrum", NULL, "1985",
	"Yabba Dabba Doo! - Flintstones (48K)\0", NULL, "Quicksilva", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecYabbadooRomInfo, SpecYabbadooRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Magic Carpet (16K)

static struct BurnRomInfo SpecMcarpetRomDesc[] = {
	{ "Magic Carpet 16K (1985)(Mastertronic).z80", 11221, 0x05e22cf1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMcarpet, SpecMcarpet, Spectrum)
STD_ROM_FN(SpecMcarpet)

struct BurnDriver BurnSpecMcarpet = {
	"spec_mcarpet", NULL, "spec_spectrum", NULL, "1985",
	"Magic Carpet (16K)\0", NULL, "Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecMcarpetRomInfo, SpecMcarpetRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Munsters, The (128K)

static struct BurnRomInfo SpecMunstersRomDesc[] = {
	{ "Munsters, The (1989)(Again Again).tap", 78341, 0x37451ece, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMunsters, SpecMunsters, Spec128)
STD_ROM_FN(SpecMunsters)

struct BurnDriver BurnSpecMunsters = {
	"spec_munsters", NULL, "spec_spec128", NULL, "1989",
	"Munsters, The (128K)\0", NULL, "Again Again", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_ADV, 0,
	SpectrumGetZipName, SpecMunstersRomInfo, SpecMunstersRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Nuclear Bowls (48K)

static struct BurnRomInfo SpecNubowlsRomDesc[] = {
	{ "Nuclear Bowls 48K (1986)(Zigurat).tap", 49570, 0x8a383a82, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecNubowls, SpecNubowls, Spectrum)
STD_ROM_FN(SpecNubowls)

struct BurnDriver BurnSpecNubowls = {
	"spec_nubowls", NULL, "spec_spectrum", NULL, "1986",
	"Nuclear Bowls (48K)\0", NULL, "Zigurat Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecNubowlsRomInfo, SpecNubowlsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Popeye (48K)

static struct BurnRomInfo SpecPopeyeRomDesc[] = {
	{ "Popeye (1985)(DK'Tronics).z80", 30572, 0x8582b9b7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPopeye, SpecPopeye, Spectrum)
STD_ROM_FN(SpecPopeye)

struct BurnDriver BurnSpecPopeye = {
	"spec_popeye", NULL, "spec_spectrum", NULL, "1985",
	"Popeye (48K)\0", NULL, "DK'Tronics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecPopeyeRomInfo, SpecPopeyeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Poseidon - Planet Eleven (48K)

static struct BurnRomInfo SpecPoseidonRomDesc[] = {
	{ "Poseidon - Planet Eleven 48K (1990)(Hi-Tec).z80", 35217, 0x73a2cdf2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPoseidon, SpecPoseidon, Spectrum)
STD_ROM_FN(SpecPoseidon)

struct BurnDriver BurnSpecPoseidon = {
	"spec_poseidon", NULL, "spec_spectrum", NULL, "1990",
	"Poseidon - Planet Eleven (48K)\0", NULL, "Hi-Tec Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_ADV, 0,
	SpectrumGetZipName, SpecPoseidonRomInfo, SpecPoseidonRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Tranz Am (16K)

static struct BurnRomInfo SpecTranzamRomDesc[] = {
	{ "Tranz Am (1983)(Ultimate Play The Game).z80", 14205, 0x00eaadb6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTranzam, SpecTranzam, Spectrum)
STD_ROM_FN(SpecTranzam)

struct BurnDriver BurnSpecTranzam = {
	"spec_tranzam", NULL, "spec_spectrum", NULL, "1983",
	"Tranz Am (16K)\0", NULL, "Ultimate Play The Game", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecTranzamRomInfo, SpecTranzamRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Trom (16K)

static struct BurnRomInfo SpecTromRomDesc[] = {
	{ "Trom 16K (1983)(DK'Tronics).z80", 10147, 0xfd8cc4f1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTrom, SpecTrom, Spectrum)
STD_ROM_FN(SpecTrom)

struct BurnDriver BurnSpecTrom = {
	"spec_trom", NULL, "spec_spectrum", NULL, "1983",
	"Trom (16K)\0", NULL, "DK'Tronics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecTromRomInfo, SpecTromRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Edd the Duck (48K)

static struct BurnRomInfo SpecEddduckRomDesc[] = {
	{ "Edd the Duck 48K (1990)(Impulze).z80", 29944, 0x720cac09, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecEddduck, SpecEddduck, Spectrum)
STD_ROM_FN(SpecEddduck)

struct BurnDriver BurnSpecEddduck = {
	"spec_eddduck", NULL, "spec_spectrum", NULL, "1990",
	"Edd the Duck (48K)\0", NULL, "Impulze", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecEddduckRomInfo, SpecEddduckRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Masters of the Universe - The Movie (48K)

static struct BurnRomInfo SpecMastersRomDesc[] = {
	{ "Masters of the Universe - The Movie 48K (1987)(Gremlin Graphics).tap", 42086, 0x8bcea841, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMasters, SpecMasters, Spectrum)
STD_ROM_FN(SpecMasters)

struct BurnDriver BurnSpecMasters = {
	"spec_masters", NULL, "spec_spectrum", NULL, "1987",
	"Masters of the Universe - The Movie (48K)\0", "Trainer Pokes", "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecMastersRomInfo, SpecMastersRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Empire Strikes Back, The (48K)

static struct BurnRomInfo SpecStarwars2RomDesc[] = {
	{ "Empire Strikes Back, The 48K (1988)(Domark).z80", 40675, 0x2f091fd7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecStarwars2, SpecStarwars2, Spectrum)
STD_ROM_FN(SpecStarwars2)

struct BurnDriver BurnSpecStarwars2 = {
	"spec_starwars2", NULL, "spec_spectrum", NULL, "1988",
	"Empire Strikes Back, The (48K)\0", NULL, "Domark", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecStarwars2RomInfo, SpecStarwars2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Asterix and the Magic Cauldron (48K)

static struct BurnRomInfo SpecAsterixRomDesc[] = {
	{ "Asterix and the Magic Cauldron (1986)(Melbourne House).tap", 48507, 0x38447018, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAsterix, SpecAsterix, Spectrum)
STD_ROM_FN(SpecAsterix)

struct BurnDriver BurnSpecAsterix = {
	"spec_asterix", NULL, "spec_spectrum", NULL, "1986",
	"Asterix and the Magic Cauldron (48K)\0", NULL, "Melbourne House", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecAsterixRomInfo, SpecAsterixRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Captain Planet and the Planeteers (48K)

static struct BurnRomInfo SpecCaptplanetRomDesc[] = {
	{ "Captain Planet and the Planeteers (1991)(Mindscape International).tap", 61255, 0x0e629f1d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCaptplanet, SpecCaptplanet, Spectrum)
STD_ROM_FN(SpecCaptplanet)

struct BurnDriver BurnSpecCaptplanet = {
	"spec_captplanet", NULL, "spec_spectrum", NULL, "1991",
	"Captain Planet and the Planeteers (48K)\0", NULL, "Mindscape International", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecCaptplanetRomInfo, SpecCaptplanetRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Centi-Bug (16K)

static struct BurnRomInfo SpecCentibugRomDesc[] = {
	{ "Centi-Bug 16K (1983)(DK'Tronics).z80", 9769, 0x8c90f40d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCentibug, SpecCentibug, Spectrum)
STD_ROM_FN(SpecCentibug)

struct BurnDriver BurnSpecCentibug = {
	"spec_centibug", NULL, "spec_spectrum", NULL, "1983",
	"Centi-Bug - aka Centipede (16K)\0", "Press 'K' to select Kempston Joystick", "DK'Tronics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecCentibugRomInfo, SpecCentibugRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Deep, The (48K)

static struct BurnRomInfo SpecDeepRomDesc[] = {
	{ "Deep, The (1988)(U.S. Gold).tap", 30931, 0xac4f747e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDeep, SpecDeep, Spectrum)
STD_ROM_FN(SpecDeep)

struct BurnDriver BurnSpecDeep = {
	"spec_deep", NULL, "spec_spectrum", NULL, "1988",
	"Deep, The (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecDeepRomInfo, SpecDeepRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Deviants (128K)

static struct BurnRomInfo SpecDeviantsRomDesc[] = {
	{ "Deviants 128K (1987)(Players).z80", 42765, 0x2d05732e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDeviants, SpecDeviants, Spec128)
STD_ROM_FN(SpecDeviants)

struct BurnDriver BurnSpecDeviants = {
	"spec_deviants", NULL, "spec_spec128", NULL, "1987",
	"Deviants (128K)\0", NULL, "Players Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecDeviantsRomInfo, SpecDeviantsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Invasion Force (16K)

static struct BurnRomInfo SpecInvforceRomDesc[] = {
	{ "Invasion Force 16K (1982)(Artic Computing).tap", 9216, 0x14bf4579, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecInvforce, SpecInvforce, Spectrum)
STD_ROM_FN(SpecInvforce)

struct BurnDriver BurnSpecInvforce = {
	"spec_invforce", NULL, "spec_spectrum", NULL, "1982",
	"Invasion Force (16K)\0", "Use A, S and F to play", "Artic Computing", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecInvforceRomInfo, SpecInvforceRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Metro-Cross (48K)

static struct BurnRomInfo SpecMetrocrossRomDesc[] = {
	{ "Metro-Cross (1987)(U.S. Gold).tap", 44446, 0x6e1d27d3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMetrocross, SpecMetrocross, Spectrum)
STD_ROM_FN(SpecMetrocross)

struct BurnDriver BurnSpecMetrocross = {
	"spec_metrocross", NULL, "spec_spectrum", NULL, "1987",
	"Metro-Cross (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecMetrocrossRomInfo, SpecMetrocrossRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Moonsweeper (48K)

static struct BurnRomInfo SpecMoonsweeperRomDesc[] = {
	{ "Moonsweeper (1983)(CheetahSoft).z80", 26570, 0x71b11ba8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMoonsweeper, SpecMoonsweeper, Spectrum)
STD_ROM_FN(SpecMoonsweeper)

struct BurnDriver BurnSpecMoonsweeper = {
	"spec_moonsweeper", NULL, "spec_spectrum", NULL, "1983",
	"Moonsweeper (48K)\0", NULL, "CheetahSoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecMoonsweeperRomInfo, SpecMoonsweeperRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Penetrator (48K)

static struct BurnRomInfo SpecPenetratorRomDesc[] = {
	{ "Penetrator 48K (1983)(Melbourne House).tap", 40029, 0x7efe1706, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPenetrator, SpecPenetrator, Spectrum)
STD_ROM_FN(SpecPenetrator)

struct BurnDriver BurnSpecPenetrator = {
	"spec_penetrator", NULL, "spec_spectrum", NULL, "1982-83",
	"Penetrator (48K)\0", NULL, "Melbourne House", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecPenetratorRomInfo, SpecPenetratorRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Xeno II (16K)

static struct BurnRomInfo SpecXeno2RomDesc[] = {
	{ "Xeno II 16K (1983)(Anirog).z80", 23368, 0x0406cb03, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecXeno2, SpecXeno2, Spectrum)
STD_ROM_FN(SpecXeno2)

struct BurnDriver BurnSpecXeno2 = {
	"spec_xeno2", NULL, "spec_spectrum", NULL, "1983",
	"Xeno II (16K)\0", NULL, "Anirog", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecXeno2RomInfo, SpecXeno2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 3D Starstrike (48K)

static struct BurnRomInfo Spec3dstarstrikeRomDesc[] = {
	{ "3D Starstrike (1984)(Realtime Games).z80", 45116, 0x8bcdf6a3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec3dstarstrike, Spec3dstarstrike, Spectrum)
STD_ROM_FN(Spec3dstarstrike)

struct BurnDriver BurnSpec3dstarstrike = {
	"spec_3dstarstrike", NULL, "spec_spectrum", NULL, "1984",
	"3D Starstrike (48K)\0", NULL, "Realtime Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, Spec3dstarstrikeRomInfo, Spec3dstarstrikeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Toyota Celica GT Rally (128K)

static struct BurnRomInfo SpecCelicagtRomDesc[] = {
	{ "Toyota Celica GT Rally (1991)(Gremlin Graphics).tap", 97948, 0x153240dc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCelicagt, SpecCelicagt, Spec128)
STD_ROM_FN(SpecCelicagt)

struct BurnDriver BurnSpecCelicagt = {
	"spec_celicagt", NULL, "spec_spec128", NULL, "1991",
	"Toyota Celica GT Rally (128K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecCelicagtRomInfo, SpecCelicagtRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Chevy Chase (128K)

static struct BurnRomInfo SpecChevychaseRomDesc[] = {
	{ "Chevy Chase 128K (1991)(Hi-Tec).z80", 85403, 0x6d88b22c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecChevychase, SpecChevychase, Spec128)
STD_ROM_FN(SpecChevychase)

struct BurnDriver BurnSpecChevychase = {
	"spec_chevychase", NULL, "spec_spec128", NULL, "1991",
	"Chevy Chase (128K)\0", NULL, "Hi-Tec Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecChevychaseRomInfo, SpecChevychaseRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Chip's Challenge (128K)

static struct BurnRomInfo SpecChipchalRomDesc[] = {
	{ "Chip's Challenge 128K (1990)(U.S. Gold)(128K).tap", 142333, 0xfcb9eb9a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecChipchal, SpecChipchal, Spec128)
STD_ROM_FN(SpecChipchal)

struct BurnDriver BurnSpecChipchal = {
	"spec_chipchal", NULL, "spec_spec128", NULL, "1990",
	"Chip's Challenge (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecChipchalRomInfo, SpecChipchalRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Star Bowls (48K)

static struct BurnRomInfo SpecStarbowlsRomDesc[] = {
	{ "Star Bowls (1991)(Zigurat Software).tap", 47077, 0x10eb1995, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecStarbowls, SpecStarbowls, Spectrum)
STD_ROM_FN(SpecStarbowls)

struct BurnDriver BurnSpecStarbowls = {
	"spec_starbowls", NULL, "spec_spectrum", NULL, "1991",
	"Star Bowls (48K)\0", NULL, "Zigurat Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecStarbowlsRomInfo, SpecStarbowlsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bloody (Spanish) (48K)

static struct BurnRomInfo SpecBloodyRomDesc[] = {
	{ "Bloody ES (1987)(Genesis Soft).tap", 28033, 0xae89fdd6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBloody, SpecBloody, Spectrum)
STD_ROM_FN(SpecBloody)

struct BurnDriver BurnSpecBloody = {
	"spec_bloody", NULL, "spec_spectrum", NULL, "1987",
	"Bloody (Spanish) (48K)\0", NULL, "Genesis Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecBloodyRomInfo, SpecBloodyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bounces (48K)

static struct BurnRomInfo SpecBouncesRomDesc[] = {
	{ "Bounces (1986)(Beyond).z80", 35649, 0xbd312347, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBounces, SpecBounces, Spectrum)
STD_ROM_FN(SpecBounces)

struct BurnDriver BurnSpecBounces = {
	"spec_bounces", NULL, "spec_spectrum", NULL, "1986",
	"Bounces (48K)\0", NULL, "Beyond", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_BALLPADDLE, 0,
	SpectrumGetZipName, SpecBouncesRomInfo, SpecBouncesRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// BraveStarr (48K)

static struct BurnRomInfo SpecBravestarrRomDesc[] = {
	{ "BraveStarr (1987)(Go!).tap", 47812, 0xe4da9cc1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBravestarr, SpecBravestarr, Spectrum)
STD_ROM_FN(SpecBravestarr)

struct BurnDriver BurnSpecBravestarr = {
	"spec_bravestarr", NULL, "spec_spectrum", NULL, "1987",
	"BraveStarr (48K)\0", NULL, "Go!", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecBravestarrRomInfo, SpecBravestarrRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Fury, The (128K)

static struct BurnRomInfo SpecFuryRomDesc[] = {
	{ "Fury, The (1988)(Martech Games).tap", 47954, 0x7ecbb308, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFury, SpecFury, Spec128)
STD_ROM_FN(SpecFury)

struct BurnDriver BurnSpecFury = {
	"spec_fury", NULL, "spec_spec128", NULL, "1988",
	"Fury, The (128K)\0", NULL, "Martech Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecFuryRomInfo, SpecFuryRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Gonzzalezz Part 1 (Spanish) (128K)

static struct BurnRomInfo SpecGonzzalezz1RomDesc[] = {
	{ "Gonzzalezz ES Part 1 (1989)(Opera Soft).tap", 48329, 0xf9309baa, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGonzzalezz1, SpecGonzzalezz1, Spec128)
STD_ROM_FN(SpecGonzzalezz1)

struct BurnDriver BurnSpecGonzzalezz1 = {
	"spec_gonzzalezz1", NULL, "spec_spec128", NULL, "1989",
	"Gonzzalezz Part 1 (Spanish) (128K)\0", NULL, "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecGonzzalezz1RomInfo, SpecGonzzalezz1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Gonzzalezz Part 2 (Spanish) (128K)

static struct BurnRomInfo SpecGonzzalezz2RomDesc[] = {
	{ "Gonzzalezz ES Part 2 (1989)(Opera Soft).tap", 48329, 0xd9041f4b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGonzzalezz2, SpecGonzzalezz2, Spec128)
STD_ROM_FN(SpecGonzzalezz2)

struct BurnDriver BurnSpecGonzzalezz2 = {
	"spec_gonzzalezz2", "spec_gonzzalezz1", "spec_spec128", NULL, "1989",
	"Gonzzalezz Part 2 (Spanish) (128K)\0", NULL, "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecGonzzalezz2RomInfo, SpecGonzzalezz2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sanxion - The Spectrum Remix (128K)

static struct BurnRomInfo SpecSanxion128RomDesc[] = {
	{ "Sanxion - The Spectrum Remix (1989)(Thalamus)(128K).tap", 48009, 0x0cb4ad4d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSanxion128, SpecSanxion128, Spec128)
STD_ROM_FN(SpecSanxion128)

struct BurnDriver BurnSpecSanxion128 = {
	"spec_sanxion128", NULL, "spec_spec128", NULL, "1989",
	"Sanxion - The Spectrum Remix (128K)\0", NULL, "Thalamus", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecSanxion128RomInfo, SpecSanxion128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sanxion - The Spectrum Remix (48K)

static struct BurnRomInfo SpecSanxion48RomDesc[] = {
	{ "Sanxion - The Spectrum Remix (1989)(Thalamus)(48K-128K).z80", 19615, 0x97a506fa, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSanxion48, SpecSanxion48, Spectrum)
STD_ROM_FN(SpecSanxion48)

struct BurnDriver BurnSpecSanxion48 = {
	"spec_sanxion48", "spec_sanxion128", "spec_spectrum", NULL, "1989",
	"Sanxion - The Spectrum Remix (48K)\0", NULL, "Thalamus", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecSanxion48RomInfo, SpecSanxion48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Scuba Dive (48K)

static struct BurnRomInfo SpecScubadiveRomDesc[] = {
	{ "Scuba Dive (1983)(Durell).z80", 28778, 0xfc1543e0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecScubadive, SpecScubadive, Spectrum)
STD_ROM_FN(SpecScubadive)

struct BurnDriver BurnSpecScubadive = {
	"spec_scubadive", NULL, "spec_spectrum", NULL, "1983",
	"Scuba Dive (48K)\0", NULL, "Durell Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecScubadiveRomInfo, SpecScubadiveRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Tusker (128K)

static struct BurnRomInfo SpecTuskerRomDesc[] = {
	{ "Tusker 128K (1989)(System 3).tap", 135053, 0x4802c727, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTusker, SpecTusker, Spec128)
STD_ROM_FN(SpecTusker)

struct BurnDriver BurnSpecTusker = {
	"spec_tusker", NULL, "spec_spec128", NULL, "1989",
	"Tusker (128K)\0", NULL, "System 3", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecTuskerRomInfo, SpecTuskerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// West Bank (48K)

static struct BurnRomInfo SpecWestbankRomDesc[] = {
	{ "West Bank (1985)(Gremlin Graphics).tap", 56654, 0xd211a5a9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecWestbank, SpecWestbank, Spectrum)
STD_ROM_FN(SpecWestbank)

struct BurnDriver BurnSpecWestbank = {
	"spec_westbank", NULL, "spec_spectrum", NULL, "1985",
	"West Bank (48K)\0", "1-2-3 to shoot, O-P to move", "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecWestbankRomInfo, SpecWestbankRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Treinador de Futebol (Portuguese) (48K)

static struct BurnRomInfo SpecTrfutebolRomDesc[] = {
	{ "Treinador de Futebol PT (1983)(Microbyte Software).z80", 34576, 0x49dbc0b8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTrfutebol, SpecTrfutebol, Spectrum)
STD_ROM_FN(SpecTrfutebol)

struct BurnDriver BurnSpecTrfutebol = {
	"spec_trfutebol", NULL, "spec_spectrum", NULL, "1983",
	"Treinador de Futebol (Portuguese) (48K)\0", NULL, "Microbyte Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_STRATEGY, 0,
	SpectrumGetZipName, SpecTrfutebolRomInfo, SpecTrfutebolRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 3D Lunattack (48K)

static struct BurnRomInfo SpecLunattackRomDesc[] = {
	{ "3D Lunattack (1984)(Hewson Consultants).z80", 16399, 0x0b58cfda, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecLunattack, SpecLunattack, Spectrum)
STD_ROM_FN(SpecLunattack)

struct BurnDriver BurnSpecLunattack = {
	"spec_lunattack", NULL, "spec_spectrum", NULL, "1984",
	"3D Lunattack (48K)\0", NULL, "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecLunattackRomInfo, SpecLunattackRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecCursorKeysDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Oh Mummy (16K)

static struct BurnRomInfo SpecOhmummyRomDesc[] = {
	{ "Oh Mummy 16K (1984)(Gem Software).tap", 11384, 0x575531e4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecOhmummy, SpecOhmummy, Spectrum)
STD_ROM_FN(SpecOhmummy)

struct BurnDriver BurnSpecOhmummy = {
	"spec_ohmummy", NULL, "spec_spectrum", NULL, "1984-86",
	"Oh Mummy (16K)\0", "A, Z, K, and L to play", "Gem Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE, 0,
	SpectrumGetZipName, SpecOhmummyRomInfo, SpecOhmummyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Percy the Potty Pigeon (48K)

static struct BurnRomInfo SpecPottypigeonRomDesc[] = {
	{ "Percy the Potty Pigeon (1984)(Gremlin Graphics).tap", 49061, 0x243fb1ea, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPottypigeon, SpecPottypigeon, Spectrum)
STD_ROM_FN(SpecPottypigeon)

struct BurnDriver BurnSpecPottypigeon = {
	"spec_pottypigeon", NULL, "spec_spectrum", NULL, "1984",
	"Percy the Potty Pigeon (48K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecPottypigeonRomInfo, SpecPottypigeonRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Pulsoid (128K)

static struct BurnRomInfo SpecPulsoidsRomDesc[] = {
	{ "Pulsoids (1988)(Mastertronic).tap", 35011, 0x9d1fc1e3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPulsoids, SpecPulsoids, Spec128)
STD_ROM_FN(SpecPulsoids)

struct BurnDriver BurnSpecPulsoids = {
	"spec_pulsoids", NULL, "spec_spec128", NULL, "1988",
	"Pulsoid (128K)\0", NULL, "Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_BREAKOUT, 0,
	SpectrumGetZipName, SpecPulsoidsRomInfo, SpecPulsoidsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Curse of Sherwood, The (48K)

static struct BurnRomInfo SpecSherwoodRomDesc[] = {
	{ "Curse of Sherwood, The (1987)(Mastertronic).z80", 39360, 0x72701ae3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSherwood, SpecSherwood, Spectrum)
STD_ROM_FN(SpecSherwood)

struct BurnDriver BurnSpecSherwood = {
	"spec_sherwood", NULL, "spec_spectrum", NULL, "1987",
	"Curse of Sherwood, The (48K)\0", NULL, "Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_ADV, 0,
	SpectrumGetZipName, SpecSherwoodRomInfo, SpecSherwoodRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Aquasquad (48K)

static struct BurnRomInfo SpecAquasquadRomDesc[] = {
	{ "Aquasquad (1988)(Atlantis).tap", 55215, 0x8d07b136, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAquasquad, SpecAquasquad, Spectrum)
STD_ROM_FN(SpecAquasquad)

struct BurnDriver BurnSpecAquasquad = {
	"spec_aquasquad", NULL, "spec_spectrum", NULL, "1988",
	"Aquasquad (48K)\0", NULL, "Atlantis Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE | GBF_SHOOT, 0,
	SpectrumGetZipName, SpecAquasquadRomInfo, SpecAquasquadRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ballbreaker (128K)

static struct BurnRomInfo SpecBallbreak128RomDesc[] = {
	{ "Ballbreaker 128K (1988)(CRL Group).z80", 38627, 0x2fca1d2e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBallbreak128, SpecBallbreak128, Spec128)
STD_ROM_FN(SpecBallbreak128)

struct BurnDriver BurnSpecBallbreak128 = {
	"spec_ballbreak128", NULL, "spec_spec128", NULL, "1988",
	"Ballbreaker (128K)\0", "A, Z and SPACE to select options", "CRL Group", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_BREAKOUT, 0,
	SpectrumGetZipName, SpecBallbreak128RomInfo, SpecBallbreak128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ballbreaker (48K)

static struct BurnRomInfo SpecBallbreak48RomDesc[] = {
	{ "Ballbreaker 48K (1988)(CRL Group).z80", 37090, 0x09eecf5a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBallbreak48, SpecBallbreak48, Spectrum)
STD_ROM_FN(SpecBallbreak48)

struct BurnDriver BurnSpecBallbreak48 = {
	"spec_ballbreak48", "spec_ballbreak128", "spec_spectrum", NULL, "1988",
	"Ballbreaker (48K)\0", "A, Z and SPACE to select options", "CRL Group", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_BREAKOUT, 0,
	SpectrumGetZipName, SpecBallbreak48RomInfo, SpecBallbreak48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bosconian '87 (128K)

static struct BurnRomInfo SpecBosconianRomDesc[] = {
	{ "Bosconian '87 (1987)(Mastertronic).tap", 46840, 0x3cc3e412, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBosconian, SpecBosconian, Spec128)
STD_ROM_FN(SpecBosconian)

struct BurnDriver BurnSpecBosconian = {
	"spec_bosconian", NULL, "spec_spec128", NULL, "1987",
	"Bosconian '87 (128K)\0", NULL, "Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecBosconianRomInfo, SpecBosconianRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Nonamed (48K)

static struct BurnRomInfo SpecNonamedRomDesc[] = {
	{ "Nonamed 48K (1987)(Dinamic Software).tap", 44233, 0xdc3f2124, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecNonamed, SpecNonamed, Spectrum)
STD_ROM_FN(SpecNonamed)

struct BurnDriver BurnSpecNonamed = {
	"spec_nonamed", NULL, "spec_spectrum", NULL, "1987",
	"Nonamed (48K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecNonamedRomInfo, SpecNonamedRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// World Series Basketball (48K)

static struct BurnRomInfo SpecWsbasketRomDesc[] = {
	{ "World Series Basketball (1985)(Imagine).z80", 40058, 0x611090db, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecWsbasket, SpecWsbasket, Spectrum)
STD_ROM_FN(SpecWsbasket)

struct BurnDriver BurnSpecWsbasket = {
	"spec_wsbasket", NULL, "spec_spectrum", NULL, "1985",
	"World Series Basketball (48K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecWsbasketRomInfo, SpecWsbasketRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Top Cat in Beverly Hills Cats (48K)

static struct BurnRomInfo SpecTopcatRomDesc[] = {
	{ "Top Cat in Beverly Hills Cats 48K (1991)(Hi-Tec).z80", 43855, 0x08eb7b04, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTopcat, SpecTopcat, Spectrum)
STD_ROM_FN(SpecTopcat)

struct BurnDriver BurnSpecTopcat = {
	"spec_topcat", NULL, "spec_spectrum", NULL, "1991",
	"Top Cat in Beverly Hills Cats (48K) (HB)\0", NULL, "Hi-Tec Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecTopcatRomInfo, SpecTopcatRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bangers and Mash (128K)

static struct BurnRomInfo SpecBangmashRomDesc[] = {
	{ "Bangers and Mash (1990)(Alternative).tap", 74560, 0xd5225cf2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBangmash, SpecBangmash, Spec128)
STD_ROM_FN(SpecBangmash)

struct BurnDriver BurnSpecBangmash = {
	"spec_bangmash", NULL, "spec_spec128", NULL, "1990",
	"Bangers and Mash (128K)\0", "Z, X, L and ENTER to move main character", "Alternative", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecBangmashRomInfo, SpecBangmashRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Blazing Thunder (48K)

static struct BurnRomInfo SpecBlazthundRomDesc[] = {
	{ "Blazing Thunder 48K (1990)(Hi-Tec).z80", 41048, 0x41afcb85, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBlazthund, SpecBlazthund, Spectrum)
STD_ROM_FN(SpecBlazthund)

struct BurnDriver BurnSpecBlazthund = {
	"spec_blazthund", NULL, "spec_spectrum", NULL, "1990",
	"Blazing Thunder (48K)\0", NULL, "Hi-Tec Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecBlazthundRomInfo, SpecBlazthundRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Death Pit (48K)

static struct BurnRomInfo SpecDeathpitRomDesc[] = {
	{ "Death Pit (1985)(Durell).tap", 47673, 0x9ee83a80, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDeathpit, SpecDeathpit, Spectrum)
STD_ROM_FN(SpecDeathpit)

struct BurnDriver BurnSpecDeathpit = {
	"spec_deathpit", NULL, "spec_spectrum", NULL, "1985",
	"Death Pit (48K)\0", NULL, "Durell Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecDeathpitRomInfo, SpecDeathpitRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Future Knight (128K)

static struct BurnRomInfo SpecFutknightRomDesc[] = {
	{ "Future Knight (1986)(Gremlin Graphics).tap", 53471, 0xfc50df19, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFutknight, SpecFutknight, Spec128)
STD_ROM_FN(SpecFutknight)

struct BurnDriver BurnSpecFutknight = {
	"spec_futknight", NULL, "spec_spec128", NULL, "1986",
	"Future Knight (128K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecFutknightRomInfo, SpecFutknightRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sub Chase (16K)

static struct BurnRomInfo SpecSubchaseRomDesc[] = {
	{ "Sub Chase 16K (1983)(Gem Software).z80", 7701, 0x211cd8c0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSubchase, SpecSubchase, Spectrum)
STD_ROM_FN(SpecSubchase)

struct BurnDriver BurnSpecSubchase = {
	"spec_subchase", NULL, "spec_spectrum", NULL, "1983",
	"Sub Chase (16K)\0", "Q and P to move, keys from 1 to 9 to use depth charges", "Gem Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecSubchaseRomInfo, SpecSubchaseRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Abadia del Crimen, La (Spanish) (128K)

static struct BurnRomInfo SpecAbadcrimRomDesc[] = {
	{ "Abadia del Crimen, La 128K (1988)(Opera Soft).tap", 99182, 0x7bd1ab11, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAbadcrim, SpecAbadcrim, Spec128)
STD_ROM_FN(SpecAbadcrim)

struct BurnDriver BurnSpecAbadcrim = {
	"spec_abadcrim", NULL, "spec_spec128", NULL, "1988",
	"Abadia del Crimen, La (Spanish) (128K)\0", "To use main character: A, K, L and SPACE", "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecAbadcrimRomInfo, SpecAbadcrimRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ad Astra (48K)

static struct BurnRomInfo SpecAdastraRomDesc[] = {
	{ "Ad Astra (1984)(Gargoyle Games).tap", 45791, 0xb4ff89d7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAdastra, SpecAdastra, Spectrum)
STD_ROM_FN(SpecAdastra)

struct BurnDriver BurnSpecAdastra = {
	"spec_adastra", NULL, "spec_spectrum", NULL, "1984",
	"Ad Astra (48K)\0", NULL, "Gargoyle Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecAdastraRomInfo, SpecAdastraRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Reckless Rufus (128K)

static struct BurnRomInfo SpecReckrufusRomDesc[] = {
	{ "Reckless Rufus 128K (1992)(Alternative).tap", 54968, 0x43172f9b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecReckrufus, SpecReckrufus, Spec128)
STD_ROM_FN(SpecReckrufus)

struct BurnDriver BurnSpecReckrufus = {
	"spec_reckrufus", NULL, "spec_spec128", NULL, "1992",
	"Reckless Rufus (128K)\0", NULL, "Alternative", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecReckrufusRomInfo, SpecReckrufusRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Benny Hill's Madcap Chase! (48K)

static struct BurnRomInfo SpecBennyhilRomDesc[] = {
	{ "Benny Hill's Madcap Chase! (1985)(DK'Tronics).tap", 33223, 0xe7a95cef, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBennyhil, SpecBennyhil, Spectrum)
STD_ROM_FN(SpecBennyhil)

struct BurnDriver BurnSpecBennyhil = {
	"spec_bennyhil", NULL, "spec_spectrum", NULL, "1985",
	"Benny Hill's Madcap Chase! (48K)\0", NULL, "DK'Tronics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MISC | GBF_ADV, 0,
	SpectrumGetZipName, SpecBennyhilRomInfo, SpecBennyhilRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Benny Hill's Madcap Chase! (128K) (Hack)

static struct BurnRomInfo SpecBennyhil128RomDesc[] = {
	{ "Benny Hill's Madcap Chase! 128K (2017)(AGDLabs).z80", 36822, 0x0c432e6c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBennyhil128, SpecBennyhil128, Spec128)
STD_ROM_FN(SpecBennyhil128)

struct BurnDriver BurnSpecBennyhil128 = {
	"spec_bennyhil128", "spec_bennyhil", "spec_spec128", NULL, "2017",
	"Benny Hill's Madcap Chase! (128K) (Hack)\0", "AY-PSG music", "AGDLabs", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_SPECTRUM, GBF_MISC | GBF_ADV, 0,
	SpectrumGetZipName, SpecBennyhil128RomInfo, SpecBennyhil128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Chiller (48K)

static struct BurnRomInfo SpecChillerRomDesc[] = {
	{ "Chiller 48K (1985)(Mastertronic).tap", 23016, 0x8ade8ac6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecChiller, SpecChiller, Spectrum)
STD_ROM_FN(SpecChiller)

struct BurnDriver BurnSpecChiller = {
	"spec_chiller", NULL, "spec_spectrum", NULL, "1985",
	"Chiller (48K)\0", NULL, "Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecChillerRomInfo, SpecChillerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Crazy Cars (128K)

static struct BurnRomInfo SpecCrazycarsRomDesc[] = {
	{ "Crazy Cars (1988)(Titus).tap", 44388, 0xe65d2bd3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCrazycars, SpecCrazycars, Spec128)
STD_ROM_FN(SpecCrazycars)

struct BurnDriver BurnSpecCrazycars = {
	"spec_crazycars", NULL, "spec_spec128", NULL, "1988",
	"Crazy Cars (128K)\0", NULL, "Titus", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecCrazycarsRomInfo, SpecCrazycarsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Crazy Cars II (128K)

static struct BurnRomInfo SpecCrazycars2RomDesc[] = {
	{ "Crazy Cars II (1988)(Titus).tap", 41208, 0x0b4a0be1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCrazycars2, SpecCrazycars2, Spec128)
STD_ROM_FN(SpecCrazycars2)

struct BurnDriver BurnSpecCrazycars2 = {
	"spec_crazycars2", NULL, "spec_spec128", NULL, "1988",
	"Crazy Cars II (128K)\0", NULL, "Titus", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecCrazycars2RomInfo, SpecCrazycars2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Delta Charge! (128K)

static struct BurnRomInfo SpecDeltaRomDesc[] = {
	{ "Delta Charge! (1990)(Thalamus).tap", 56562, 0x3f0b7368, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDelta, SpecDelta, Spec128)
STD_ROM_FN(SpecDelta)

struct BurnDriver BurnSpecDelta = {
	"spec_delta", NULL, "spec_spec128", NULL, "1990",
	"Delta Charge! (128K)\0", "Press 2 to set Kempston controller", "Thalamus - Twilight", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecDeltaRomInfo, SpecDeltaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Knight Force (128K)

static struct BurnRomInfo SpecKnightforceRomDesc[] = {
	{ "Knight Force (1990)(Titus).tap", 82635, 0xf1ed8323, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecKnightforce, SpecKnightforce, Spec128)
STD_ROM_FN(SpecKnightforce)

struct BurnDriver BurnSpecKnightforce = {
	"spec_knightforce", NULL, "spec_spec128", NULL, "1990",
	"Knight Force (128K)\0", NULL, "Titus", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecKnightforceRomInfo, SpecKnightforceRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Armageddon (16K)

static struct BurnRomInfo SpecArmageddonRomDesc[] = {
	{ "Armageddon 16K (1983)(Ocean).tap", 13845, 0x398b3aa5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecArmageddon, SpecArmageddon, Spectrum)
STD_ROM_FN(SpecArmageddon)

struct BurnDriver BurnSpecArmageddon = {
	"spec_armageddon", NULL, "spec_spectrum", NULL, "1983",
	"Armageddon (16K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecArmageddonRomInfo, SpecArmageddonRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bounty Bob Strikes Back! (48K)

static struct BurnRomInfo SpecBountybobRomDesc[] = {
	{ "Bounty Bob Strikes Back! 48K (1984)(U.S. Gold).z80", 45436, 0xde0b2094, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBountybob, SpecBountybob, Spectrum)
STD_ROM_FN(SpecBountybob)

struct BurnDriver BurnSpecBountybob = {
	"spec_bountybob", NULL, "spec_spectrum", NULL, "1984",
	"Bounty Bob Strikes Back! (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecBountybobRomInfo, SpecBountybobRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPMDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Brainstorm (48K)

static struct BurnRomInfo SpecBrainstormRomDesc[] = {
	{ "Brainstorm 48K (1985)(Bubble Bus).tap", 48514, 0xf52bcc89, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBrainstorm, SpecBrainstorm, Spectrum)
STD_ROM_FN(SpecBrainstorm)

struct BurnDriver BurnSpecBrainstorm = {
	"spec_brainstorm", NULL, "spec_spectrum", NULL, "1985",
	"Brainstorm (48K)\0", NULL, "Bubble Bus Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE | GBF_SHOOT, 0,
	SpectrumGetZipName, SpecBrainstormRomInfo, SpecBrainstormRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Don Quijote de la Mancha - Parte I (Spanish) (48K)

static struct BurnRomInfo SpecDonquijote1RomDesc[] = {
	{ "Don Quijote de la Mancha (1987)(Dinamic)(ES)(Side A).tap", 49387, 0x1ac98966, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDonquijote1, SpecDonquijote1, Spectrum)
STD_ROM_FN(SpecDonquijote1)

struct BurnDriver BurnSpecDonquijote1 = {
	"spec_donquijote1", NULL, "spec_spectrum", NULL, "1987",
	"Don Quijote de la Mancha - Parte I (Spanish) (48K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecDonquijote1RomInfo, SpecDonquijote1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Don Quijote de la Mancha - Parte II (Spanish) (48K)

static struct BurnRomInfo SpecDonquijote2RomDesc[] = {
	{ "Don Quijote de la Mancha (1987)(Dinamic)(ES)(Side B).tap", 49387, 0x042db349, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDonquijote2, SpecDonquijote2, Spectrum)
STD_ROM_FN(SpecDonquijote2)

struct BurnDriver BurnSpecDonquijote2 = {
	"spec_donquijote2", "spec_donquijote1", "spec_spectrum", NULL, "1987",
	"Don Quijote de la Mancha - Parte II (Spanish) (48K)\0", "Password: EL INGENIOSO HIDALGO", "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecDonquijote2RomInfo, SpecDonquijote2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Last Duel (48K)

static struct BurnRomInfo SpecLastduelRomDesc[] = {
	{ "Last Duel (1989)(Go!).tap", 171660, 0xca2ae42a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecLastduel, SpecLastduel, Spectrum)
STD_ROM_FN(SpecLastduel)

struct BurnDriver BurnSpecLastduel = {
	"spec_lastduel", NULL, "spec_spectrum", NULL, "1989",
	"Last Duel (48K)\0", NULL, "Go!", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RACING | GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecLastduelRomInfo, SpecLastduelRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Aventura Espacial, La - Parte 1 - Explorando Mundos (Spanish) (48K)

static struct BurnRomInfo SpecAvespacial1RomDesc[] = {
	{ "Aventura Espacial, La - Explorando Mundos (1990)(Aventuras AD).z80", 45676, 0xc46932a5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAvespacial1, SpecAvespacial1, Spectrum)
STD_ROM_FN(SpecAvespacial1)

struct BurnDriver BurnSpecAvespacial1 = {
	"spec_avespacial1", NULL, "spec_spectrum", NULL, "1990",
	"Aventura Espacial, La - Explorando Mundos (Spanish) (48K)\0", NULL, "Aventuras A.D.", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecAvespacial1RomInfo, SpecAvespacial1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Aventura Espacial, La - Parte 2 - En los Dominios del Cerebelo (Spanish) (48K)

static struct BurnRomInfo SpecAvespacial2RomDesc[] = {
	{ "Aventura Espacial, La - En los Dominios del Cerebelo (1990)(Aventuras AD).z80", 40915, 0x06d2fd5c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAvespacial2, SpecAvespacial2, Spectrum)
STD_ROM_FN(SpecAvespacial2)

struct BurnDriver BurnSpecAvespacial2 = {
	"spec_avespacial2", "spec_avespacial1", "spec_spectrum", NULL, "1990",
	"Aventura Espacial, La - En los Dominios del Cerebelo (Spanish) (48K)\0", "PASSWORD: CANES VENATICI", "Aventuras A.D.", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecAvespacial2RomInfo, SpecAvespacial2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Aventura Original - Parte 1 - La Busqueda (Spanish) (48K)

static struct BurnRomInfo SpecAvoriginal1RomDesc[] = {
	{ "Aventura Original - Parte 1 - La Busqueda (1989)(Aventuras AD).z80", 46750, 0xce246950, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAvoriginal1, SpecAvoriginal1, Spectrum)
STD_ROM_FN(SpecAvoriginal1)

struct BurnDriver BurnSpecAvoriginal1 = {
	"spec_avoriginal1", NULL, "spec_spectrum", NULL, "1989",
	"Aventura Original - Parte 1 - La Busqueda (Spanish) (48K)\0", NULL, "Aventuras A.D.", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecAvoriginal1RomInfo, SpecAvoriginal1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Aventura Original - Parte 2 - El Encuentro (Spanish) (48K)

static struct BurnRomInfo SpecAvoriginal2RomDesc[] = {
	{ "Aventura Original - Parte 2 - El Encuentro (1989)(Aventuras AD).z80", 47212, 0x126a3c19, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAvoriginal2, SpecAvoriginal2, Spectrum)
STD_ROM_FN(SpecAvoriginal2)

struct BurnDriver BurnSpecAvoriginal2 = {
	"spec_avoriginal2", "spec_avoriginal1", "spec_spectrum", NULL, "1989",
	"Aventura Original - Parte 2 - El Encuentro (Spanish) (48K)\0", "PASSWORD: TIMACUS", "Aventuras A.D.", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecAvoriginal2RomInfo, SpecAvoriginal2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Troll (128K)

static struct BurnRomInfo SpecTrollRomDesc[] = {
	{ "Troll 128K (1988)(Palace Software).tap", 47794, 0xd3af2582, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTroll, SpecTroll, Spec128)
STD_ROM_FN(SpecTroll)

struct BurnDriver BurnSpecTroll = {
	"spec_troll", NULL, "spec_spec128", NULL, "1988",
	"Troll (128K)\0", NULL, "Palace Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecTrollRomInfo, SpecTrollRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Yenght - La Fuente de la Juventud (Spanish) (48K)

static struct BurnRomInfo SpecYenghtRomDesc[] = {
	{ "Yenght - La Fuente de la Juventud ES (1984)(Dinamic).tap", 45958, 0x36df229a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecYenght, SpecYenght, Spectrum)
STD_ROM_FN(SpecYenght)

struct BurnDriver BurnSpecYenght = {
	"spec_yenght", NULL, "spec_spectrum", NULL, "1984",
	"Yenght - La Fuente de la Juventud (Spanish) (48K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecYenghtRomInfo, SpecYenghtRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Yeti (48K)

static struct BurnRomInfo SpecYetiRomDesc[] = {
	{ "Yeti 48K (1988)(Destiny).tap", 48009, 0x0536775c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecYeti, SpecYeti, Spectrum)
STD_ROM_FN(SpecYeti)

struct BurnDriver BurnSpecYeti = {
	"spec_yeti", NULL, "spec_spectrum", NULL, "1988",
	"Yeti (48K)\0", NULL, "Destiny", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecYetiRomInfo, SpecYetiRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Cobra's Arc (Spanish) (48K)

static struct BurnRomInfo SpecCobrarcRomDesc[] = {
	{ "Cobra's Arc ES (1986)(Dinamic).tap", 49106, 0xed2e3bb4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCobrarc, SpecCobrarc, Spectrum)
STD_ROM_FN(SpecCobrarc)

struct BurnDriver BurnSpecCobrarc = {
	"spec_cobrarc", NULL, "spec_spectrum", NULL, "1986",
	"Cobra's Arc (Spanish) (48K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecCobrarcRomInfo, SpecCobrarcRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Templos Sagrados, Los - Parte 1 (Spanish)

static struct BurnRomInfo SpecSagradosp1RomDesc[] = {
	{ "Templos Sagrados, Los - Parte 1 (1991)(Aventuras AD).tap", 45975, 0xcaebb204, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSagradosp1, SpecSagradosp1, Spectrum)
STD_ROM_FN(SpecSagradosp1)

struct BurnDriver BurnSpecSagradosp1 = {
	"spec_sagradosp1", NULL, "spec_spectrum", NULL, "1991",
	"Templos Sagrados, Los - Parte 1 (Spanish)\0", NULL, "Aventuras A.D.", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecSagradosp1RomInfo, SpecSagradosp1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Templos Sagrados, Los - Parte 2 (Spanish)

static struct BurnRomInfo SpecSagradosp2RomDesc[] = {
	{ "Templos Sagrados, Los - Parte 2 (1991)(Aventuras AD).tap", 45936, 0x52e03e35, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSagradosp2, SpecSagradosp2, Spectrum)
STD_ROM_FN(SpecSagradosp2)

struct BurnDriver BurnSpecSagradosp2 = {
	"spec_sagradosp2", "spec_sagradosp1", "spec_spectrum", NULL, "1991",
	"Templos Sagrados, Los - Parte 2 (Spanish)\0", "PASSWORD: TUR KOS BON", "Aventuras A.D.", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecSagradosp2RomInfo, SpecSagradosp2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Score 3020 (Spanish) (48K)

static struct BurnRomInfo SpecScore3020RomDesc[] = {
	{ "Score 3020 ES 48K (1989)(Topo Soft).z80", 35177, 0x3f505d11, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecScore3020, SpecScore3020, Spectrum)
STD_ROM_FN(SpecScore3020)

struct BurnDriver BurnSpecScore3020 = {
	"spec_score3020", NULL, "spec_spectrum", NULL, "1989",
	"Score 3020 (Spanish) (48K)\0", NULL, "Topo Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PINBALL, 0,
	SpectrumGetZipName, SpecScore3020RomInfo, SpecScore3020RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Aknadach (Czech) (48K)

static struct BurnRomInfo SpecAknadachczRomDesc[] = {
	{ "Aknadach CZ 48K (1990)(Proxima).z80", 40314, 0xe221fe1b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAknadachcz, SpecAknadachcz, Spectrum)
STD_ROM_FN(SpecAknadachcz)

struct BurnDriver BurnSpecAknadachcz = {
	"spec_aknadachcz", NULL, "spec_spectrum", NULL, "1990",
	"Aknadach (Czech) (48K)\0", NULL, "Proxima", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecAknadachczRomInfo, SpecAknadachczRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Aknadach (English) (48K) (Hack)

static struct BurnRomInfo SpecAknadachenRomDesc[] = {
	{ "Aknadach EN 48K (2011)(Softhouse).z80", 40286, 0xf369810f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAknadachen, SpecAknadachen, Spectrum)
STD_ROM_FN(SpecAknadachen)

struct BurnDriver BurnSpecAknadachen = {
	"spec_aknadachen", "spec_aknadachcz", "spec_spectrum", NULL, "2011",
	"Aknadach (English) (48K) (Hack)\0", NULL, "Softhouse", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecAknadachenRomInfo, SpecAknadachenRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Castle Master (48K)

static struct BurnRomInfo SpecCastlemasterRomDesc[] = {
	{ "Castle Master (1990)(Domark).tap", 50816, 0x63e17cd0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCastlemaster, SpecCastlemaster, Spectrum)
STD_ROM_FN(SpecCastlemaster)

struct BurnDriver BurnSpecCastlemaster = {
	"spec_castlemaster", NULL, "spec_spectrum", NULL, "1990",
	"Castle Master (48K)\0", NULL, "Domark", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecCastlemasterRomInfo, SpecCastlemasterRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Days of Thunder (128K)

static struct BurnRomInfo SpecDaysthunderRomDesc[] = {
	{ "Days of Thunder 128K (1990)(Mindscape International).tap", 95028, 0x954ba9b3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDaysthunder, SpecDaysthunder, Spec128)
STD_ROM_FN(SpecDaysthunder)

struct BurnDriver BurnSpecDaysthunder = {
	"spec_daysthunder", NULL, "spec_spec128", NULL, "1990",
	"Days of Thunder (128K)\0", NULL, "Mindscape International", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecDaysthunderRomInfo, SpecDaysthunderRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mega-Corp - Parte 1 (Spanish) (48K)

static struct BurnRomInfo SpecMegacorp1RomDesc[] = {
	{ "Mega-Corp - Parte 1 ES (1987)(Dinamic).tap", 49246, 0x1fe3c489, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMegacorp1, SpecMegacorp1, Spectrum)
STD_ROM_FN(SpecMegacorp1)

struct BurnDriver BurnSpecMegacorp1 = {
	"spec_megacorp1", NULL, "spec_spectrum", NULL, "1987",
	"Mega-Corp - Parte 1 (Spanish) (48K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecMegacorp1RomInfo, SpecMegacorp1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mega-Corp - Parte 2 (Spanish) (48K)

static struct BurnRomInfo SpecMegacorp2RomDesc[] = {
	{ "Mega-Corp - Parte 2 ES (1987)(Dinamic).tap", 49246, 0x0d83942c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMegacorp2, SpecMegacorp2, Spectrum)
STD_ROM_FN(SpecMegacorp2)

struct BurnDriver BurnSpecMegacorp2 = {
	"spec_megacorp2", "spec_megacorp1", "spec_spectrum", NULL, "1987",
	"Mega-Corp - Parte 2 (Spanish) (48K)\0", "Password: REBECA", "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecMegacorp2RomInfo, SpecMegacorp2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Q10 Tank Buster (48K)

static struct BurnRomInfo SpecQ10tbRomDesc[] = {
	{ "Q10 Tank Buster 48K (1992)(Zeppelin Games).tap", 43523, 0x1fad3a24, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecQ10tb, SpecQ10tb, Spectrum)
STD_ROM_FN(SpecQ10tb)

struct BurnDriver BurnSpecQ10tb = {
	"spec_q10tb", NULL, "spec_spectrum", NULL, "1992",
	"Q10 Tank Buster (48K)\0", NULL, "Zeppelin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecQ10tbRomInfo, SpecQ10tbRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Captain America Defies the Doom Tube (48K)

static struct BurnRomInfo SpecCaptamericaRomDesc[] = {
	{ "Captain America Defies the Doom Tube 48K (1988)(Go!).z80", 47363, 0x6cb2fdb2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCaptamerica, SpecCaptamerica, Spectrum)
STD_ROM_FN(SpecCaptamerica)

struct BurnDriver BurnSpecCaptamerica = {
	"spec_captamerica", NULL, "spec_spectrum", NULL, "1988",
	"Captain America Defies the Doom Tube (48K)\0", NULL, "Go!", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_ADV, 0,
	SpectrumGetZipName, SpecCaptamericaRomInfo, SpecCaptamericaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Jail Break (48K)

static struct BurnRomInfo SpecJailbreakRomDesc[] = {
	{ "Jail Break 48K (1987)(Konami).z80", 44911, 0x9ee208aa, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecJailbreak, SpecJailbreak, Spectrum)
STD_ROM_FN(SpecJailbreak)

struct BurnDriver BurnSpecJailbreak = {
	"spec_jailbreak", NULL, "spec_spectrum", NULL, "1987",
	"Jail Break (48K)\0", NULL, "Konami", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecJailbreakRomInfo, SpecJailbreakRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Jasper! (48K)

static struct BurnRomInfo SpecJasperRomDesc[] = {
	{ "Jasper! 48K (1984)(Micromega).tap", 50132, 0xc2bd0b3e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecJasper, SpecJasper, Spectrum)
STD_ROM_FN(SpecJasper)

struct BurnDriver BurnSpecJasper = {
	"spec_jasper", NULL, "spec_spectrum", NULL, "1984",
	"Jasper! (48K)\0", "Use A, S, O and L to move the main character", "Micromega", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecJasperRomInfo, SpecJasperRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Jahangir Khan's World Championship Squash - Club Game (128K)

static struct BurnRomInfo SpecJksquashcgRomDesc[] = {
	{ "Jahangir Khan's World Championship Squash - Club Game 128K (1991)(Krisalis).tap", 102623, 0xf94e2be3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecJksquashcg, SpecJksquashcg, Spec128)
STD_ROM_FN(SpecJksquashcg)

struct BurnDriver BurnSpecJksquashcg = {
	"spec_jksquashcg", NULL, "spec_spec128", NULL, "1991",
	"Jahangir Khan's World Championship Squash - Club Game (128K)\0", NULL, "Krisalis", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecJksquashcgRomInfo, SpecJksquashcgRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Jahangir Khan's World Championship Squash - Tournament Game (128K)

static struct BurnRomInfo SpecJksquashtgRomDesc[] = {
	{ "Jahangir Khan's World Championship Squash - Tournament Game 128K (1991)(Krisalis).tap", 109108, 0xc5b34d11, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecJksquashtg, SpecJksquashtg, Spec128)
STD_ROM_FN(SpecJksquashtg)

struct BurnDriver BurnSpecJksquashtg = {
	"spec_jksquashtg", "spec_jksquashcg", "spec_spec128", NULL, "1991",
	"Jahangir Khan's World Championship Squash - Tournament Game (128K)\0", NULL, "Krisalis", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecJksquashtgRomInfo, SpecJksquashtgRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// King Arthur's Quest (48K)

static struct BurnRomInfo SpecKaquestRomDesc[] = {
	{ "King Arthur's Quest 48K (1984)(Hill MacGibbon).tap", 51142, 0x461d1af1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecKaquest, SpecKaquest, Spectrum)
STD_ROM_FN(SpecKaquest)

struct BurnDriver BurnSpecKaquest = {
	"spec_kaquest", NULL, "spec_spectrum", NULL, "1984",
	"King Arthur's Quest (48K)\0", "Use numeric keys to play", "Hill MacGibbon", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecKaquestRomInfo, SpecKaquestRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Eidolon, The (128K)

static struct BurnRomInfo SpecEidolonRomDesc[] = {
	{ "Eidolon, The 128K (1986)(Activision - LucasfilmGames).tap", 107749, 0x00677105, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecEidolon, SpecEidolon, Spec128)
STD_ROM_FN(SpecEidolon)

struct BurnDriver BurnSpecEidolon = {
	"spec_eidolon", NULL, "spec_spec128", NULL, "1986",
	"Eidolon, The (128K)\0", NULL, "Activision - Lucasfilm Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_ADV, 0,
	SpectrumGetZipName, SpecEidolonRomInfo, SpecEidolonRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Pakacuda (16K)

static struct BurnRomInfo SpecPakacudaRomDesc[] = {
	{ "Pakacuda 16K (1983)(Rabbit).tap", 15372, 0x789f0e0b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPakacuda, SpecPakacuda, Spectrum)
STD_ROM_FN(SpecPakacuda)

struct BurnDriver BurnSpecPakacuda = {
	"spec_pakacuda", NULL, "spec_spectrum", NULL, "1983",
	"Pakacuda (16K)\0", "Select 'N'ormal cursor keys to use controller", "Rabbit Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE | GBF_ACTION, 0,
	SpectrumGetZipName, SpecPakacudaRomInfo, SpecPakacudaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecCursorKeysDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rocco (48K)

static struct BurnRomInfo SpecRoccoRomDesc[] = {
	{ "Rocco 48K (1985)(Gremlin Graphics).z80", 38035, 0xdae7f761, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecRocco, SpecRocco, Spectrum)
STD_ROM_FN(SpecRocco)

struct BurnDriver BurnSpecRocco = {
	"spec_rocco", "spec_rocky", "spec_spectrum", NULL, "1985",
	"Rocco (48K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpecRoccoRomInfo, SpecRoccoRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rocky (48K)

static struct BurnRomInfo SpecRockyRomDesc[] = {
	{ "Rocky 48K (1985)(Dinamic).z80", 38026, 0x79a0f6b2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecRocky, SpecRocky, Spectrum)
STD_ROM_FN(SpecRocky)

struct BurnDriver BurnSpecRocky = {
	"spec_rocky", NULL, "spec_spectrum", NULL, "1985",
	"Rocky (48K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpecRockyRomInfo, SpecRockyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sinbad and the Golden Ship - Part 1 (48K)

static struct BurnRomInfo SpecSinbadgs1RomDesc[] = {
	{ "Sinbad and the Golden Ship Part1 48K (1986)(Mastervision).tap", 55928, 0x3d445332, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSinbadgs1, SpecSinbadgs1, Spectrum)
STD_ROM_FN(SpecSinbadgs1)

struct BurnDriver BurnSpecSinbadgs1 = {
	"spec_sinbadgs1", NULL, "spec_spectrum", NULL, "1986",
	"Sinbad and the Golden Ship - Part 1 (48K)\0", NULL, "Mastervision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecSinbadgs1RomInfo, SpecSinbadgs1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sinbad and the Golden Ship - Part 2 (48K)

static struct BurnRomInfo SpecSinbadgs2RomDesc[] = {
	{ "Sinbad and the Golden Ship Part2 48K (1986)(Mastervision).tap", 48955, 0xff490432, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSinbadgs2, SpecSinbadgs2, Spectrum)
STD_ROM_FN(SpecSinbadgs2)

struct BurnDriver BurnSpecSinbadgs2 = {
	"spec_sinbadgs2", "spec_sinbadgs1", "spec_spectrum", NULL, "1986",
	"Sinbad and the Golden Ship - Part 2 (48K)\0", NULL, "Mastervision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecSinbadgs2RomInfo, SpecSinbadgs2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 3D Maze of Gold (16K)

static struct BurnRomInfo Spec3dmazegoldRomDesc[] = {
	{ "3D Maze of Gold 16K (1982)(Gilsoft International).tap", 7885, 0xc2da7962, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec3dmazegold, Spec3dmazegold, Spec128)
STD_ROM_FN(Spec3dmazegold)

struct BurnDriver BurnSpec3dmazegold = {
	"spec_3dmazegold", NULL, "spec_spec128", NULL, "1982",
	"3D Maze of Gold (16K)\0", NULL, "Gilsoft International", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE, 0,
	SpectrumGetZipName, Spec3dmazegoldRomInfo, Spec3dmazegoldRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecCursorKeysDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Averno (Spanish) (48K)

static struct BurnRomInfo SpecAvernoRomDesc[] = {
	{ "Averno ES 48K (1989)(P.J.).tap", 22871, 0xef4758f8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAverno, SpecAverno, Spectrum)
STD_ROM_FN(SpecAverno)

struct BurnDriver BurnSpecAverno = {
	"spec_averno", NULL, "spec_spectrum", NULL, "1989",
	"Averno (Spanish) (48K)\0", NULL, "P.J. Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecAvernoRomInfo, SpecAvernoRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecCursorKeysDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Zybex (48K)

static struct BurnRomInfo SpeczybexRomDesc[] = {
	{ "Zybex 48K (1989)(Zeppelin Games).z80", 34544, 0xa79c6137, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speczybex, Speczybex, Spectrum)
STD_ROM_FN(Speczybex)

struct BurnDriver BurnSpeczybex = {
	"spec_zybex", NULL, "spec_spectrum", NULL, "1989",
	"Zybex (48K)\0", NULL, "Zeppelin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpeczybexRomInfo, SpeczybexRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Breakthru (48K)

static struct BurnRomInfo SpecBreakthruRomDesc[] = {
	{ "Breakthru 48K (1986)(U.S. Gold).tap", 54622, 0xcd98cbc8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBreakthru, SpecBreakthru, Spectrum)
STD_ROM_FN(SpecBreakthru)

struct BurnDriver BurnSpecBreakthru = {
	"spec_breakthru", NULL, "spec_spectrum", NULL, "1986",
	"Breakthru (48K)\0", "Press 'SPACE' to jump", "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecBreakthruRomInfo, SpecBreakthruRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Count Duckula II (128K)

static struct BurnRomInfo SpecCountduck2128RomDesc[] = {
	{ "Count Duckula 2 128K (1992)(Alternative).tap", 96837, 0xe5c979e1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCountduck2128, SpecCountduck2128, Spec128)
STD_ROM_FN(SpecCountduck2128)

struct BurnDriver BurnSpecCountduck2128 = {
	"spec_countduck2128", NULL, "spec_spec128", NULL, "1992",
	"Count Duckula II (128K)\0", NULL, "Alternative", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecCountduck2128RomInfo, SpecCountduck2128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Count Duckula II (48K)

static struct BurnRomInfo SpecCountduck248RomDesc[] = {
	{ "Count Duckula 2 48K (1992)(Alternative).tap", 47605, 0x7b53965d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCountduck248, SpecCountduck248, Spectrum)
STD_ROM_FN(SpecCountduck248)

struct BurnDriver BurnSpecCountduck248 = {
	"spec_countduck248", "spec_countduck2128", "spec_spectrum", NULL, "1992",
	"Count Duckula II (48K)\0", NULL, "Alternative", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecCountduck248RomInfo, SpecCountduck248RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ghost's Revenge (16K)

static struct BurnRomInfo SpecGhostrevRomDesc[] = {
	{ "Ghost's Revenge 16K (1983)(Micromania).tap", 15558, 0x4eb8d249, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGhostrev, SpecGhostrev, Spectrum)
STD_ROM_FN(SpecGhostrev)

struct BurnDriver BurnSpecGhostrev = {
	"spec_ghostrev", NULL, "spec_spectrum", NULL, "1983",
	"Ghost's Revenge (16K)\0", NULL, "Micromania", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE | GBF_ACTION, 0,
	SpectrumGetZipName, SpecGhostrevRomInfo, SpecGhostrevRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Soldier of Light (48K)

static struct BurnRomInfo SpecSoldoflightRomDesc[] = {
	{ "Soldier of Light 48K (1988)(ACE).tap", 35934, 0xf417f933, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSoldoflight, SpecSoldoflight, Spectrum)
STD_ROM_FN(SpecSoldoflight)

struct BurnDriver BurnSpecSoldoflight = {
	"spec_soldoflight", NULL, "spec_spectrum", NULL, "1988",
	"Soldier of Light (48K)\0", NULL, "ACE", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecSoldoflightRomInfo, SpecSoldoflightRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Atomic Robo-Kid (Demo) (48K)

static struct BurnRomInfo SpecAtomrobokidRomDesc[] = {
	{ "Atomic Robo-Kid (demo) (1991)(Activision).z80", 44259, 0x2f0c3726, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAtomrobokid, SpecAtomrobokid, Spectrum)
STD_ROM_FN(SpecAtomrobokid)

struct BurnDriver BurnSpecAtomrobokid = {
	"spec_atomrobokid", NULL, "spec_spectrum", NULL, "1991",
	"Atomic Robo-Kid (Demo) (48K)\0", "Unfinished - no sound", "Activision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecAtomrobokidRomInfo, SpecAtomrobokidRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Attack of the Killer Tomatoes (48K)

static struct BurnRomInfo SpecAktomatoesRomDesc[] = {
	{ "Attack of the Killer Tomatoes 48K (1986)(Global).tap", 38305, 0xd3d3b7aa, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAktomatoes, SpecAktomatoes, Spectrum)
STD_ROM_FN(SpecAktomatoes)

struct BurnDriver BurnSpecAktomatoes = {
	"spec_aktomatoes", NULL, "spec_spectrum", NULL, "1986",
	"Attack of the Killer Tomatoes (48K)\0", NULL, "Global Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecAktomatoesRomInfo, SpecAktomatoesRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// BMX Simulator (48K)

static struct BurnRomInfo SpecBmxsimRomDesc[] = {
	{ "BMX Simulator 48K (1987)(Codemasters).z80", 43885, 0x61e994c0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBmxsim, SpecBmxsim, Spectrum)
STD_ROM_FN(SpecBmxsim)

struct BurnDriver BurnSpecBmxsim = {
	"spec_bmxsim", NULL, "spec_spectrum", NULL, "1987",
	"BMX Simulator (48K)\0", NULL, "Codemasters", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecBmxsimRomInfo, SpecBmxsimRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bonanza Bros. (128K)

static struct BurnRomInfo SpecBonanzabrosRomDesc[] = {
	{ "Bonanza Bros. 128K (1992)(U.S. Gold).tap", 132823, 0xb99578d6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBonanzabros, SpecBonanzabros, Spec128)
STD_ROM_FN(SpecBonanzabros)

struct BurnDriver BurnSpecBonanzabros = {
	"spec_bonanzabros", NULL, "spec_spec128", NULL, "1992",
	"Bonanza Bros. (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecBonanzabrosRomInfo, SpecBonanzabrosRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Deep Strike (48K)

static struct BurnRomInfo SpecDeepstrikeRomDesc[] = {
	{ "Deep Strike 48K (1986)(Durell).tap", 44086, 0x9af55985, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDeepstrike, SpecDeepstrike, Spectrum)
STD_ROM_FN(SpecDeepstrike)

struct BurnDriver BurnSpecDeepstrike = {
	"spec_deepstrike", NULL, "spec_spectrum", NULL, "1986",
	"Deep Strike (48K)\0", NULL, "Durell Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecDeepstrikeRomInfo, SpecDeepstrikeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dragon's Lair (48K)

static struct BurnRomInfo SpecDragonslairRomDesc[] = {
	{ "Dragon's Lair 48K (1986)(Software Projects).tap", 64368, 0x0f532df7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDragonslair, SpecDragonslair, Spectrum)
STD_ROM_FN(SpecDragonslair)

struct BurnDriver BurnSpecDragonslair = {
	"spec_dragonslair", NULL, "spec_spectrum", NULL, "1986",
	"Dragon's Lair (48K)\0", "Press 'J' to set Joystick", "Software Projects", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecDragonslairRomInfo, SpecDragonslairRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dragon's Lair II - Escape from Singe's Castle (128K)

static struct BurnRomInfo SpecDragonslair2RomDesc[] = {
	{ "Dragon's Lair II - Escape from Singe's Castle 128K (1987)(Software Projects).tap", 119432, 0xb2b86318, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDragonslair2, SpecDragonslair2, Spec128)
STD_ROM_FN(SpecDragonslair2)

struct BurnDriver BurnSpecDragonslair2 = {
	"spec_dragonslair2", NULL, "spec_spec128", NULL, "1987",
	"Dragon's Lair II - Escape from Singe's Castle (128K)\0", NULL, "Software Projects", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecDragonslair2RomInfo, SpecDragonslair2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Popeye 2 (48K)

static struct BurnRomInfo SpecPopeye2RomDesc[] = {
	{ "Popeye 2 48K (1991)(Alternative).tap", 26112, 0x13458bc7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPopeye2, SpecPopeye2, Spectrum)
STD_ROM_FN(SpecPopeye2)

struct BurnDriver BurnSpecPopeye2 = {
	"spec_popeye2", NULL, "spec_spectrum", NULL, "1991",
	"Popeye 2 (48K)\0", NULL, "Alternative", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecPopeye2RomInfo, SpecPopeye2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Popeye 3 - Wrestle Crazy (128K)

static struct BurnRomInfo SpecPopeye3RomDesc[] = {
	{ "Popeye 3 - Wrestle Crazy 128K (1992)(Alternative).tap", 124204, 0xaa3d43e3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPopeye3, SpecPopeye3, Spec128)
STD_ROM_FN(SpecPopeye3)

struct BurnDriver BurnSpecPopeye3 = {
	"spec_popeye3", NULL, "spec_spec128", NULL, "1992",
	"Popeye 3 - Wrestle Crazy (128K)\0", NULL, "Alternative", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpecPopeye3RomInfo, SpecPopeye3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rabbit Rabbit (16K)

static struct BurnRomInfo SpecRabbitrabbitRomDesc[] = {
	{ "Rabbit Rabbit 16K (1983)(ZX Computing).tap", 4937, 0xc741bf71, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecRabbitrabbit, SpecRabbitrabbit, Spectrum)
STD_ROM_FN(SpecRabbitrabbit)

struct BurnDriver BurnSpecRabbitrabbit = {
	"spec_rabbitrabbit", NULL, "spec_spectrum", NULL, "1983",
	"Rabbit Rabbit (16K)\0", NULL, "ZX Computing", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE, 0,
	SpectrumGetZipName, SpecRabbitrabbitRomInfo, SpecRabbitrabbitRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecCursorKeysDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Super Space Invaders (128K)

static struct BurnRomInfo SpecSsinvadersRomDesc[] = {
	{ "Super Space Invaders 128K (1991)(Domark).z80", 104637, 0x2a36ad52, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSsinvaders, SpecSsinvaders, Spec128)
STD_ROM_FN(SpecSsinvaders)

struct BurnDriver BurnSpecSsinvaders = {
	"spec_ssinvaders", NULL, "spec_spec128", NULL, "1991",
	"Super Space Invaders (128K)\0", "In main menu: 'A' to move, 'Q' to select", "Domark", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecSsinvadersRomInfo, SpecSsinvadersRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Booty (48K)

static struct BurnRomInfo SpecBootyRomDesc[] = {
	{ "Booty 48K (1984)(Firebird).tap", 45734, 0xa387b4bf, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBooty, SpecBooty, Spectrum)
STD_ROM_FN(SpecBooty)

struct BurnDriver BurnSpecBooty = {
	"spec_booty", NULL, "spec_spectrum", NULL, "1984",
	"Booty (48K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecBootyRomInfo, SpecBootyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// F-16 Combat Pilot (128K)

static struct BurnRomInfo SpecF16pilotRomDesc[] = {
	{ "F-16 Combat Pilot 128K (1991)(Digital Integration).tap", 102967, 0xe6ac9aee, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecF16pilot, SpecF16pilot, Spec128)
STD_ROM_FN(SpecF16pilot)

struct BurnDriver BurnSpecF16pilot = {
	"spec_f16pilot", NULL, "spec_spec128", NULL, "1991",
	"F-16 Combat Pilot (128K)\0", NULL, "Digital Integration", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SIM, 0,
	SpectrumGetZipName, SpecF16pilotRomInfo, SpecF16pilotRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Fiendish Freddy's Big Top o'Fun (128K)

static struct BurnRomInfo SpecFiendfreddyRomDesc[] = {
	{ "Fiendish Freddy's Big Top o'Fun 128K (1990)(Mindscape International).tap", 464247, 0x2214888c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFiendfreddy, SpecFiendfreddy, Spec128)
STD_ROM_FN(SpecFiendfreddy)

struct BurnDriver BurnSpecFiendfreddy = {
	"spec_fiendfreddy", NULL, "spec_spec128", NULL, "1990",
	"Fiendish Freddy's Big Top o'Fun (128K)\0", NULL, "Mindscape International", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 5, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecFiendfreddyRomInfo, SpecFiendfreddyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Muncher Eats Chewits, The (128K)

static struct BurnRomInfo SpecMuncherRomDesc[] = {
	{ "Muncher Eats Chewits, The 128K (1988)(Gremlin Graphics).tap", 120973, 0x16db8cd7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMuncher, SpecMuncher, Spec128)
STD_ROM_FN(SpecMuncher)

struct BurnDriver BurnSpecMuncher = {
	"spec_muncher", NULL, "spec_spec128", NULL, "1988",
	"Muncher Eats Chewits, The (128K)\0", "Set 'Sinclair Joystick' to use controller", "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecMuncherRomInfo, SpecMuncherRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Virus (48K)

static struct BurnRomInfo SpecVirusRomDesc[] = {
	{ "Virus 48K (1988)(Firebird).tap", 42183, 0x7a2c590b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecVirus, SpecVirus, Spectrum)
STD_ROM_FN(SpecVirus)

struct BurnDriver BurnSpecVirus = {
	"spec_virus", NULL, "spec_spectrum", NULL, "1988",
	"Virus (48K)\0", "Press 'R' to controller menu and ENTER to display in-game map", "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecVirusRomInfo, SpecVirusRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Exterminator (128K)

static struct BurnRomInfo SpecExterminatorRomDesc[] = {
	{ "Exterminator 128K (1991)(Audiogenic).tap", 129464, 0xdfeed3ed, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecExterminator, SpecExterminator, Spec128)
STD_ROM_FN(SpecExterminator)

struct BurnDriver BurnSpecExterminator = {
	"spec_exterminator", NULL, "spec_spec128", NULL, "1991",
	"Exterminator (128K)\0", NULL, "Audiogenic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecExterminatorRomInfo, SpecExterminatorRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2SwapDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Huxley Pig - Part 1 (48K)

static struct BurnRomInfo SpecHuxleyp1RomDesc[] = {
	{ "Huxley Pig part1 48K (1991)(Alternative).tap", 48133, 0x4d61bad5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHuxleyp1, SpecHuxleyp1, Spectrum)
STD_ROM_FN(SpecHuxleyp1)

struct BurnDriver BurnSpecHuxleyp1 = {
	"spec_huxleyp1", NULL, "spec_spectrum", NULL, "1991",
	"Huxley Pig - Part 1 (48K)\0", NULL, "Alternative", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecHuxleyp1RomInfo, SpecHuxleyp1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Huxley Pig - Part 2 (48K)

static struct BurnRomInfo SpecHuxleyp2RomDesc[] = {
	{ "Huxley Pig part2 48K (1991)(Alternative).tap", 48133, 0x2b75d8be, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHuxleyp2, SpecHuxleyp2, Spectrum)
STD_ROM_FN(SpecHuxleyp2)

struct BurnDriver BurnSpecHuxleyp2 = {
	"spec_huxleyp2", "spec_huxleyp1", "spec_spectrum", NULL, "1991",
	"Huxley Pig - Part 2 (48K)\0", "Password: 5BJV7BYN", "Alternative", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecHuxleyp2RomInfo, SpecHuxleyp2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Jaws (128K)

static struct BurnRomInfo SpecJawsRomDesc[] = {
	{ "Jaws 128K (1989)(Screen 7).tap", 70245, 0x5234e440, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecJaws, SpecJaws, Spec128)
STD_ROM_FN(SpecJaws)

struct BurnDriver BurnSpecJaws = {
	"spec_jaws", NULL, "spec_spec128", NULL, "1989",
	"Jaws (128K)\0", NULL, "Screen 7", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecJawsRomInfo, SpecJawsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Oriental Games (128K)

static struct BurnRomInfo SpecOrgamesRomDesc[] = {
	{ "Oriental Games 128K (1990)(Micro Style).tap", 138887, 0x4e03b386, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecOrgames, SpecOrgames, Spec128)
STD_ROM_FN(SpecOrgames)

struct BurnDriver BurnSpecOrgames = {
	"spec_orgames", NULL, "spec_spec128", NULL, "1990",
	"Oriental Games (128K)\0", NULL, "Micro Style", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpecOrgamesRomInfo, SpecOrgamesRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Pole Position (C&VG) (48K)

static struct BurnRomInfo SpecPoleposcvgRomDesc[] = {
	{ "Pole Position (1984)(C&VG).tap", 11325, 0x049b1e06, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPoleposcvg, SpecPoleposcvg, Spectrum)
STD_ROM_FN(SpecPoleposcvg)

struct BurnDriver BurnSpecPoleposcvg = {
	"spec_poleposcvg", NULL, "spec_spectrum", NULL, "1984",
	"Pole Position (C&VG) (48K)\0", NULL, "Computer & Video Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecPoleposcvgRomInfo, SpecPoleposcvgRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecCursorKeysDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Roland's Rat Race (48K)

static struct BurnRomInfo SpecRolandratRomDesc[] = {
	{ "Roland's Rat Race 48K (1985)(Ocean).tap", 44022, 0x44edc93e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecRolandrat, SpecRolandrat, Spectrum)
STD_ROM_FN(SpecRolandrat)

struct BurnDriver BurnSpecRolandrat = {
	"spec_rolandrat", NULL, "spec_spectrum", NULL, "1985",
	"Roland's Rat Race (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecRolandratRomInfo, SpecRolandratRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Souls of Darkon (48K)

static struct BurnRomInfo SpecSoulsdarkonRomDesc[] = {
	{ "Souls of Darkon (1985)(Taskset).tap", 48819, 0x7ad0d630, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSoulsdarkon, SpecSoulsdarkon, Spectrum)
STD_ROM_FN(SpecSoulsdarkon)

struct BurnDriver BurnSpecSoulsdarkon = {
	"spec_soulsdarkon", NULL, "spec_spectrum", NULL, "1985",
	"Souls of Darkon (48K)\0", NULL, "Taskset", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecSoulsdarkonRomInfo, SpecSoulsdarkonRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Time Machine (48K)

static struct BurnRomInfo SpecTimemachineRomDesc[] = {
	{ "Time Machine 48K (1990)(Activision).z80", 41590, 0x5cf41e5c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTimemachine, SpecTimemachine, Spectrum)
STD_ROM_FN(SpecTimemachine)

struct BurnDriver BurnSpecTimemachine = {
	"spec_timemachine", NULL, "spec_spectrum", NULL, "1990",
	"Time Machine (48K)\0", NULL, "Activision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecTimemachineRomInfo, SpecTimemachineRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Chichen Itza - Part 1 (Spanish) (48K)

static struct BurnRomInfo SpecChichenitza1RomDesc[] = {
	{ "Chichen Itza - Part 1 ES (1992)(Aventuras AD).tap", 48166, 0x14e54546, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecChichenitza1, SpecChichenitza1, Spectrum)
STD_ROM_FN(SpecChichenitza1)

struct BurnDriver BurnSpecChichenitza1 = {
	"spec_chichenitza1", NULL, "spec_spectrum", NULL, "1992",
	"Chichen Itza - Part 1 (Spanish) (48K)\0", NULL, "Aventuras A.D.", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecChichenitza1RomInfo, SpecChichenitza1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Chichen Itza - Part 2 (Spanish) (48K)

static struct BurnRomInfo SpecChichenitza2RomDesc[] = {
	{ "Chichen Itza - Part 2 ES (1992)(Aventuras AD).tap", 48166, 0x0a229edb, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecChichenitza2, SpecChichenitza2, Spectrum)
STD_ROM_FN(SpecChichenitza2)

struct BurnDriver BurnSpecChichenitza2 = {
	"spec_chichenitza2", "spec_chichenitza1", "spec_spectrum", NULL, "1992",
	"Chichen Itza - Part 2 (Spanish) (48K)\0", "Password: RICO PAPASITO", "Aventuras A.D.", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecChichenitza2RomInfo, SpecChichenitza2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Diosa de Cozumel, La (Spanish) (48K)

static struct BurnRomInfo SpecCozumelRomDesc[] = {
	{ "Diosa de Cozumel, La ES (1990)(Aventuras AD).tap", 95528, 0x225d77d8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCozumel, SpecCozumel, Spectrum)
STD_ROM_FN(SpecCozumel)

struct BurnDriver BurnSpecCozumel = {
	"spec_cozumel", NULL, "spec_spectrum", NULL, "1990",
	"Diosa de Cozumel, La (Spanish) (48K)\0", NULL, "Aventuras A.D.", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecCozumelRomInfo, SpecCozumelRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Train - Escape to Normandy, The (48K)

static struct BurnRomInfo SpecEscnormandyRomDesc[] = {
	{ "Train - Escape to Normandy, The 48K (1988)(Electronic Arts).z80", 44448, 0xccbf6d84, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecEscnormandy, SpecEscnormandy, Spectrum)
STD_ROM_FN(SpecEscnormandy)

struct BurnDriver BurnSpecEscnormandy = {
	"spec_escnormandy", NULL, "spec_spectrum", NULL, "1988",
	"Train - Escape to Normandy, The (48K)\0", NULL, "Electronic Arts", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SIM | GBF_SHOOT, 0,
	SpectrumGetZipName, SpecEscnormandyRomInfo, SpecEscnormandyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Inspector Gadget and the Circus of Fear (48K)

static struct BurnRomInfo SpecInspgadgetRomDesc[] = {
	{ "Inspector Gadget and the Circus of Fear (1987)(Melbourne House).tap", 44729, 0xc9dfdd10, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecInspgadget, SpecInspgadget, Spec128)
STD_ROM_FN(SpecInspgadget)

struct BurnDriver BurnSpecInspgadget = {
	"spec_inspgadget", NULL, "spec_spec128", NULL, "1987",
	"Inspector Gadget and the Circus of Fear (48K)\0", NULL, "Melbourne House", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecInspgadgetRomInfo, SpecInspgadgetRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ninja Spirit (128K)

static struct BurnRomInfo SpecNinjaspiritRomDesc[] = {
	{ "Ninja Spirit 128K (1990)(Activision).tap", 113048, 0x19b240f6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecNinjaspirit, SpecNinjaspirit, Spec128)
STD_ROM_FN(SpecNinjaspirit)

struct BurnDriver BurnSpecNinjaspirit = {
	"spec_ninjaspirit", NULL, "spec_spec128", NULL, "1990",
	"Ninja Spirit (128K)\0", NULL, "Activision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecNinjaspiritRomInfo, SpecNinjaspiritRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ninja Warriors, The (128K)

static struct BurnRomInfo SpecNinjawarrRomDesc[] = {
	{ "Ninja Warriors, The 128K (1989)(Virgin Games).tap", 112187, 0xe870c241, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecNinjawarr, SpecNinjawarr, Spec128)
STD_ROM_FN(SpecNinjawarr)

struct BurnDriver BurnSpecNinjawarr = {
	"spec_ninjawarr", NULL, "spec_spec128", NULL, "1989",
	"Ninja Warriors, The (128K)\0", NULL, "Virgin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecNinjawarrRomInfo, SpecNinjawarrRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Super Cycle (48K)

static struct BurnRomInfo SpecSupercycleRomDesc[] = {
	{ "Super Cycle (1987)(U.S. Gold).z80", 32852, 0x7aeedc00, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSupercycle, SpecSupercycle, Spectrum)
STD_ROM_FN(SpecSupercycle)

struct BurnDriver BurnSpecSupercycle = {
	"spec_supercycle", NULL, "spec_spectrum", NULL, "1987",
	"Super Cycle (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecSupercycleRomInfo, SpecSupercycleRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Terrorballs (48K)

static struct BurnRomInfo SpecTerrorballsRomDesc[] = {
	{ "Terrorballs 48K (1987)(Whammo Productions).tap", 42249, 0x86ae441b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTerrorballs, SpecTerrorballs, Spectrum)
STD_ROM_FN(SpecTerrorballs)

struct BurnDriver BurnSpecTerrorballs = {
	"spec_terrorballs", NULL, "spec_spectrum", NULL, "1987",
	"Terrorballs (48K)\0", NULL, "Whammo Productions", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecTerrorballsRomInfo, SpecTerrorballsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// International Ninja Rabbits (128K)

static struct BurnRomInfo SpecIntninjarabRomDesc[] = {
	{ "International Ninja Rabbits (1991)(Micro Value).tap", 87287, 0xf55bc727, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecIntninjarab, SpecIntninjarab, Spec128)
STD_ROM_FN(SpecIntninjarab)

struct BurnDriver BurnSpecIntninjarab = {
	"spec_intninjarab", NULL, "spec_spec128", NULL, "1991",
	"International Ninja Rabbits (128K)\0", NULL, "Micro Value", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecIntninjarabRomInfo, SpecIntninjarabRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ninja Commando (48K)

static struct BurnRomInfo SpecNinjacomRomDesc[] = {
	{ "Ninja Commando 48K (1989)(Zeppelin Games).tap", 40413, 0x60e678a2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecNinjacom, SpecNinjacom, Spectrum)
STD_ROM_FN(SpecNinjacom)

struct BurnDriver BurnSpecNinjacom = {
	"spec_ninjacom", NULL, "spec_spectrum", NULL, "1989",
	"Ninja Commando (48K)\0", NULL, "Zeppelin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecNinjacomRomInfo, SpecNinjacomRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ninja Scooter Simulator (48K)

static struct BurnRomInfo SpecNinjascooterRomDesc[] = {
	{ "Ninja Scooter Simulator 48K (1987)(Sysoft).tap", 46884, 0xf9173adb, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecNinjascooter, SpecNinjascooter, Spectrum)
STD_ROM_FN(SpecNinjascooter)

struct BurnDriver BurnSpecNinjascooter = {
	"spec_ninjascooter", NULL, "spec_spectrum", NULL, "1987",
	"Ninja Scooter Simulator (48K)\0", NULL, "Sysoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecNinjascooterRomInfo, SpecNinjascooterRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Starglider 2 - The Egrons Strike Back (128K)

static struct BurnRomInfo SpecStarglider2RomDesc[] = {
	{ "Starglider 2 - The Egrons Strike Back 128K (1989)(Rainbird).tap", 95040, 0x667e1ddf, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecStarglider2, SpecStarglider2, Spec128)
STD_ROM_FN(SpecStarglider2)

struct BurnDriver BurnSpecStarglider2 = {
	"spec_starglider2", NULL, "spec_spec128", NULL, "1989",
	"Starglider 2 - The Egrons Strike Back (128K)\0", NULL, "Rainbird", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_SIM, 0,
	SpectrumGetZipName, SpecStarglider2RomInfo, SpecStarglider2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Turbo Skate Fighter (48K)

static struct BurnRomInfo SpecTurboskateRomDesc[] = {
	{ "Turbo Skate Fighter 48K (1989)(Zeppelin Games).tap", 32414, 0x4c658b72, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTurboskate, SpecTurboskate, Spectrum)
STD_ROM_FN(SpecTurboskate)

struct BurnDriver BurnSpecTurboskate = {
	"spec_turboskate", NULL, "spec_spectrum", NULL, "1989",
	"Turbo Skate Fighter (48K)\0", NULL, "Zeppelin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecTurboskateRomInfo, SpecTurboskateRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Beach Volley (128K)

static struct BurnRomInfo SpecBvolleyRomDesc[] = {
	{ "Beach Volley (1989)(Ocean).tap", 93507, 0x73056f7e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBvolley, SpecBvolley, Spec128)
STD_ROM_FN(SpecBvolley)

struct BurnDriver BurnSpecBvolley = {
	"spec_bvolley", NULL, "spec_spec128", NULL, "1989",
	"Beach Volley (128K)\0", "'Redefine Keys' to use controller", "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecBvolleyRomInfo, SpecBvolleyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecCursorKeysDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Contact Sam Cruise (48K)

static struct BurnRomInfo SpecContscruiseRomDesc[] = {
	{ "Contact Sam Cruise (1986)(Microsphere).tap", 56267, 0x3175c3e3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecContscruise, SpecContscruise, Spectrum)
STD_ROM_FN(SpecContscruise)

struct BurnDriver BurnSpecContscruise = {
	"spec_contscruise", NULL, "spec_spectrum", NULL, "1986",
	"Contact Sam Cruise (48K)\0", "After the game has started, press 'J' and 'K' to use controller", "Microsphere", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecContscruiseRomInfo, SpecContscruiseRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dalek Attack - Doctor Who (Trainer) (128K)

static struct BurnRomInfo SpecDalekattackRomDesc[] = {
	{ "Dalek Attack (128K) (1992)(Admiral)(Trainer).tap", 119370, 0x5b3cd868, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDalekattack, SpecDalekattack, Spec128)
STD_ROM_FN(SpecDalekattack)

struct BurnDriver BurnSpecDalekattack = {
	"spec_dalekattack", NULL, "spec_spec128", NULL, "1993",
	"Dalek Attack - Doctor Who (Trainer) (128K)\0", NULL, "Admiral Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecDalekattackRomInfo, SpecDalekattackRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Egghead (48K)

static struct BurnRomInfo SpecEggheadRomDesc[] = {
	{ "Egghead (1989)(Crash).tap", 48076, 0x794d5953, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecEgghead, SpecEgghead, Spectrum)
STD_ROM_FN(SpecEgghead)

struct BurnDriver BurnSpecEgghead = {
	"spec_egghead", NULL, "spec_spectrum", NULL, "1989",
	"Egghead (48K)\0", "AY8912 PSG supported", "Crash", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecEggheadRomInfo, SpecEggheadRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Egghead II - Egghead to the Rescue (48K)

static struct BurnRomInfo SpecEgghead2RomDesc[] = {
	{ "Egghead II - Egghead to the Rescue (1990)(Crash).tap", 46328, 0xb854984a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecEgghead2, SpecEgghead2, Spectrum)
STD_ROM_FN(SpecEgghead2)

struct BurnDriver BurnSpecEgghead2 = {
	"spec_egghead2", NULL, "spec_spectrum", NULL, "1990",
	"Egghead II - Egghead to the Rescue (48K)\0", "AY8912 PSG supported", "Crash", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecEgghead2RomInfo, SpecEgghead2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Flunky (48K)

static struct BurnRomInfo SpecFlunkyRomDesc[] = {
	{ "Flunky 48K (1987)(Piranha).tap", 43939, 0x5f8ab2e3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFlunky, SpecFlunky, Spectrum)
STD_ROM_FN(SpecFlunky)

struct BurnDriver BurnSpecFlunky = {
	"spec_flunky", NULL, "spec_spectrum", NULL, "1987",
	"Flunky (48K)\0", "In the Menu Room, use the 'bar' to cycle and the 'buzzer' to set options", "Piranha", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecFlunkyRomInfo, SpecFlunkyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Krakout (48K)

static struct BurnRomInfo SpecKrakoutRomDesc[] = {
	{ "Krakout 48K (1987)(Gremlin Graphics).tap", 49145, 0x71d2a0f6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecKrakout, SpecKrakout, Spectrum)
STD_ROM_FN(SpecKrakout)

struct BurnDriver BurnSpecKrakout = {
	"spec_krakout", NULL, "spec_spectrum", NULL, "1987",
	"Krakout (48K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_BREAKOUT, 0,
	SpectrumGetZipName, SpecKrakoutRomInfo, SpecKrakoutRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Legend of Kage, The (48K)

static struct BurnRomInfo SpecLegkageRomDesc[] = {
	{ "Legend of Kage, The 48K (1987)(Imagine).tap", 46959, 0x8433d421, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecLegkage, SpecLegkage, Spectrum)
STD_ROM_FN(SpecLegkage)

struct BurnDriver BurnSpecLegkage = {
	"spec_legkage", NULL, "spec_spectrum", NULL, "1987",
	"Legend of Kage, The (48K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecLegkageRomInfo, SpecLegkageRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// M.O.V.I.E. (48K)

static struct BurnRomInfo SpecMovieRomDesc[] = {
	{ "MOVIE 48K (1986)(Imagine).tap", 48594, 0x4f413f00, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMovie, SpecMovie, Spectrum)
STD_ROM_FN(SpecMovie)

struct BurnDriver BurnSpecMovie = {
	"spec_movie", NULL, "spec_spectrum", NULL, "1986",
	"M.O.V.I.E. (48K)\0", "Press 'O' to change options", "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecMovieRomInfo, SpecMovieRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Race, The (128K)

static struct BurnRomInfo SpecTheraceRomDesc[] = {
	{ "Race, The 128K (1990)(Players Premier).tap", 52939, 0x68b09f7c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTherace, SpecTherace, Spec128)
STD_ROM_FN(SpecTherace)

struct BurnDriver BurnSpecTherace = {
	"spec_therace", NULL, "spec_spec128", NULL, "1990",
	"Race, The (128K)\0", NULL, "Players Premier", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION | GBF_RACING, 0,
	SpectrumGetZipName, SpecTheraceRomInfo, SpecTheraceRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Astro Blaster (16K)

static struct BurnRomInfo SpecAstroblastRomDesc[] = {
	{ "Astro Blaster 16K (1983)(Quicksilva).tap", 7664, 0x2db0e6e9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAstroblast, SpecAstroblast, Spectrum)
STD_ROM_FN(SpecAstroblast)

struct BurnDriver BurnSpecAstroblast = {
	"spec_astroblast", NULL, "spec_spectrum", NULL, "1983",
	"Astro Blaster (16K)\0", NULL, "Quicksilva", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecAstroblastRomInfo, SpecAstroblastRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Block-Dizzy (48K)

static struct BurnRomInfo SpecBlkdizzyRomDesc[] = {
	{ "Block-Dizzy 48K (1992)(Your Sinclair).tap", 39286, 0x639b4fe8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBlkdizzy, SpecBlkdizzy, Spectrum)
STD_ROM_FN(SpecBlkdizzy)

struct BurnDriver BurnSpecBlkdizzy = {
	"spec_blkdizzy", NULL, "spec_spectrum", NULL, "1992",
	"Block-Dizzy (48K)\0", "Set Controls>>Sinclair to use controller", "Your Sinclair", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecBlkdizzyRomInfo, SpecBlkdizzyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Gerry the Germ Goes Body Poppin (48K)

static struct BurnRomInfo SpecGerrygermRomDesc[] = {
	{ "Gerry the Germ Goes Body Poppin (1985)(Firebird).tap", 49339, 0x4bf4b0b0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGerrygerm, SpecGerrygerm, Spectrum)
STD_ROM_FN(SpecGerrygerm)

struct BurnDriver BurnSpecGerrygerm = {
	"spec_gerrygerm", NULL, "spec_spectrum", NULL, "1985",
	"Gerry the Germ Goes Body Poppin (48K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecGerrygermRomInfo, SpecGerrygermRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Gift from the Gods (48K)

static struct BurnRomInfo SpecGiftgodsRomDesc[] = {
	{ "Gift from the Gods (1984)(Ocean).tap", 48083, 0xb6b38137, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGiftgods, SpecGiftgods, Spectrum)
STD_ROM_FN(SpecGiftgods)

struct BurnDriver BurnSpecGiftgods = {
	"spec_giftgods", NULL, "spec_spectrum", NULL, "1984",
	"Gift from the Gods (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_ADV, 0,
	SpectrumGetZipName, SpecGiftgodsRomInfo, SpecGiftgodsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// GNONI (Spanish) (48K)

static struct BurnRomInfo SpecGnonimhRomDesc[] = {
	{ "Gnoni ES 48K (1988)(MicroHobby).tap", 17473, 0x14adb16f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGnonimh, SpecGnonimh, Spectrum)
STD_ROM_FN(SpecGnonimh)

struct BurnDriver BurnSpecGnonimh = {
	"spec_gnonimh", NULL, "spec_spectrum", NULL, "1988",
	"GNONI (Spanish) (48K)\0", "NB: when use 'Kempston Joys' the fire button crashes the game", "MicroHobby", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecGnonimhRomInfo, SpecGnonimhRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 3D-Tanx (16K)

static struct BurnRomInfo Spec3dtanxRomDesc[] = {
	{ "3D-Tanx 16K (1982)(DK'Tronics).tap", 16185, 0xbae4a458, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec3dtanx, Spec3dtanx, Spectrum)
STD_ROM_FN(Spec3dtanx)

struct BurnDriver BurnSpec3dtanx = {
	"spec_3dtanx", NULL, "spec_spectrum", NULL, "1982",
	"3D-Tanx (16K)\0", NULL, "DK'Tronics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 6, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, Spec3dtanxRomInfo, Spec3dtanxRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bomber (16K)

static struct BurnRomInfo SpecBomberRomDesc[] = {
	{ "Bomber 16K (1983)(Spectrum Computing).tap", 4346, 0x76e3760f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBomber, SpecBomber, Spectrum)
STD_ROM_FN(SpecBomber)

struct BurnDriver BurnSpecBomber = {
	"spec_bomber", NULL, "spec_spectrum", NULL, "1983",
	"Bomber (16K)\0", NULL, "Spectrum Computing", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecBomberRomInfo, SpecBomberRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Budokan - The Martial Spirit (48K)

static struct BurnRomInfo SpecBudokanRomDesc[] = {
	{ "Budokan - The Martial Spirit 48K (1991)(Dro Soft).tap", 110405, 0x7672dc9d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBudokan, SpecBudokan, Spectrum)
STD_ROM_FN(SpecBudokan)

struct BurnDriver BurnSpecBudokan = {
	"spec_budokan", NULL, "spec_spectrum", NULL, "1991",
	"Budokan - The Martial Spirit (48K)\0", "Select 'Teclado' to use controller", "Dro Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpecBudokanRomInfo, SpecBudokanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Fighter Bomber (48K)

static struct BurnRomInfo SpecFightbombRomDesc[] = {
	{ "Fighter Bomber 48K (1990)(Activision).tap", 163083, 0xa32a5d92, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFightbomb, SpecFightbomb, Spectrum)
STD_ROM_FN(SpecFightbomb)

struct BurnDriver BurnSpecFightbomb = {
	"spec_fightbomb", NULL, "spec_spectrum", NULL, "1990",
	"Fighter Bomber (48K)\0", NULL, "Activision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SIM, 0,
	SpectrumGetZipName, SpecFightbombRomInfo, SpecFightbombRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// High Steel (128K)

static struct BurnRomInfo SpecHsteel128RomDesc[] = {
	{ "High Steel 128K (1989)(Screen 7).tap", 46853, 0x0461a6b8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHsteel128, SpecHsteel128, Spec128)
STD_ROM_FN(SpecHsteel128)

struct BurnDriver BurnSpecHsteel128 = {
	"spec_hsteel128", NULL, "spec_spec128", NULL, "1989",
	"High Steel (128K)\0", "Use SPACE and ENTER to choose and set options", "Screen 7", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecHsteel128RomInfo, SpecHsteel128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// High Steel (48K)

static struct BurnRomInfo SpecHsteel48RomDesc[] = {
	{ "High Steel 48K (1989)(Screen 7).tap", 46598, 0x7afa1809, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHsteel48, SpecHsteel48, Spectrum)
STD_ROM_FN(SpecHsteel48)

struct BurnDriver BurnSpecHsteel48 = {
	"spec_hsteel48", "spec_hsteel128", "spec_spectrum", NULL, "1989",
	"High Steel (48K)\0", "Use SPACE and ENTER to choose and set options", "Screen 7", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecHsteel48RomInfo, SpecHsteel48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Carrier Command (128K)

static struct BurnRomInfo SpecCarriercmdRomDesc[] = {
	{ "Carrier Command 128K (1989)(Rainbird).tap", 122111, 0x041d3f43, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCarriercmd, SpecCarriercmd, Spec128)
STD_ROM_FN(SpecCarriercmd)

struct BurnDriver BurnSpecCarriercmd = {
	"spec_carriercmd", NULL, "spec_spec128", NULL, "1989",
	"Carrier Command (128K)\0", NULL, "Rainbird", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SIM, 0,
	SpectrumGetZipName, SpecCarriercmdRomInfo, SpecCarriercmdRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Fire and Forget (128K)

static struct BurnRomInfo SpecFireforgetRomDesc[] = {
	{ "Fire and Forget (1988)(Titus).tap", 46724, 0x7685ab3a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFireforget, SpecFireforget, Spec128)
STD_ROM_FN(SpecFireforget)

struct BurnDriver BurnSpecFireforget = {
	"spec_fireforget", NULL, "spec_spec128", NULL, "1988",
	"Fire and Forget (128K)\0", NULL, "Titus", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecFireforgetRomInfo, SpecFireforgetRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Psi Chess (48K)

static struct BurnRomInfo SpecPsichessRomDesc[] = {
	{ "Psi Chess (1986)(The Edge).tap", 49152, 0xdeca5027, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPsichess, SpecPsichess, Spectrum)
STD_ROM_FN(SpecPsichess)

struct BurnDriver BurnSpecPsichess = {
	"spec_psichess", NULL, "spec_spectrum", NULL, "1986",
	"Psi Chess (48K)\0", "S+2/S+3: 2D/3D view - M+1: start game - J+2: use controller", "The Edge", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_STRATEGY, 0,
	SpectrumGetZipName, SpecPsichessRomInfo, SpecPsichessRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Techno-Cop (128K)

static struct BurnRomInfo SpecTechcop128RomDesc[] = {
	{ "Techno-Cop 128K (1988)(Gremlin Graphics).tap", 69984, 0x323627a4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTechcop128, SpecTechcop128, Spec128)
STD_ROM_FN(SpecTechcop128)

struct BurnDriver BurnSpecTechcop128 = {
	"spec_techcop128", NULL, "spec_spec128", NULL, "1988",
	"Techno-Cop (128K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecTechcop128RomInfo, SpecTechcop128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Techno-Cop (48K)

static struct BurnRomInfo SpecTechcop48RomDesc[] = {
	{ "Techno-Cop 48K (1988)(Gremlin Graphics).tap", 49374, 0x10fca563, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTechcop48, SpecTechcop48, Spectrum)
STD_ROM_FN(SpecTechcop48)

struct BurnDriver BurnSpecTechcop48 = {
	"spec_techcop48", "spec_techcop128", "spec_spectrum", NULL, "1988",
	"Techno-Cop (48K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecTechcop48RomInfo, SpecTechcop48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hydra (128K)

static struct BurnRomInfo SpecHydraRomDesc[] = {
	{ "Hydra 128K (1991)(Domark).tap", 133588, 0x78856d99, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHydra, SpecHydra, Spec128)
STD_ROM_FN(SpecHydra)

struct BurnDriver BurnSpecHydra = {
	"spec_hydra", NULL, "spec_spec128", NULL, "1991",
	"Hydra (128K)\0", NULL, "Domark", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_RACING, 0,
	SpectrumGetZipName, SpecHydraRomInfo, SpecHydraRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ice Temple, The (48K)

static struct BurnRomInfo SpecIcetempleRomDesc[] = {
	{ "Ice Temple, The 48K (1986)(Bubble Bus).tap", 45017, 0x269dc9de, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecIcetemple, SpecIcetemple, Spectrum)
STD_ROM_FN(SpecIcetemple)

struct BurnDriver BurnSpecIcetemple = {
	"spec_icetemple", NULL, "spec_spectrum", NULL, "1986",
	"Ice Temple, The (48K)\0", NULL, "Bubble Bus", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_ADV, 0,
	SpectrumGetZipName, SpecIcetempleRomInfo, SpecIcetempleRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Intensity (48K)

static struct BurnRomInfo SpecIntensityRomDesc[] = {
	{ "Intensity 48K (1988)(Firebird).tap", 45643, 0x77a12914, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecIntensity, SpecIntensity, Spectrum)
STD_ROM_FN(SpecIntensity)

struct BurnDriver BurnSpecIntensity = {
	"spec_intensity", NULL, "spec_spectrum", NULL, "1988",
	"Intensity (48K)\0", "Press 1: one player option - Press 2: two player option", "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecIntensityRomInfo, SpecIntensityRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rescate Atlantida (Spanish) (128K)

static struct BurnRomInfo SpecRescatlanRomDesc[] = {
	{ "Rescate Atlantida ES 128K (1989)(Dinamic).z80", 109724, 0xc437d0e7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecRescatlan, SpecRescatlan, Spec128)
STD_ROM_FN(SpecRescatlan)

struct BurnDriver BurnSpecRescatlan = {
	"spec_rescatlan", NULL, "spec_spec128", NULL, "1989",
	"Rescate Atlantida (Spanish) (128K)\0", "Press SPACE to choose and ENTER to select options", "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecRescatlanRomInfo, SpecRescatlanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Total Eclipse 2 - The Sphinx Jinx (48K)

static struct BurnRomInfo SpecTeclipse2RomDesc[] = {
	{ "Total Eclipse 2 - The Sphinx Jinx 48K (1991)(Incentive).tap", 45514, 0x0c02705e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTeclipse2, SpecTeclipse2, Spectrum)
STD_ROM_FN(SpecTeclipse2)

struct BurnDriver BurnSpecTeclipse2 = {
	"spec_teclipse2", NULL, "spec_spectrum", NULL, "1991",
	"Total Eclipse 2 - The Sphinx Jinx (48K)\0", "P and L: look up and down - S: step size - SPACE: draw/replace gunsight", "Incentive", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_ADV, 0,
	SpectrumGetZipName, SpecTeclipse2RomInfo, SpecTeclipse2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// All-American Basketball (48K)

static struct BurnRomInfo SpecAabasketRomDesc[] = {
	{ "All-American Basketball 48K (1992)(Zeppelin Games).tap", 47298, 0xa6000491, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAabasket, SpecAabasket, Spectrum)
STD_ROM_FN(SpecAabasket)

struct BurnDriver BurnSpecAabasket = {
	"spec_aabasket", NULL, "spec_spectrum", NULL, "1992",
	"All-American Basketball (48K)\0", NULL, "Zeppelin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecAabasketRomInfo, SpecAabasketRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Colour of Magic, The (48K)

static struct BurnRomInfo SpecColmagicRomDesc[] = {
	{ "Colour of Magic, The 48K (1986)(Piranha).tap", 138647, 0x43a002ab, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecColmagic, SpecColmagic, Spectrum)
STD_ROM_FN(SpecColmagic)

struct BurnDriver BurnSpecColmagic = {
	"spec_colmagic", NULL, "spec_spectrum", NULL, "1986",
	"Colour of Magic, The (48K)\0", NULL, "Piranha", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecColmagicRomInfo, SpecColmagicRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Fist II - The Legend Continues (48K)

static struct BurnRomInfo SpecFist2RomDesc[] = {
	{ "Fist II - The Legend Continues 48K (1986)(Melbourne House).tap", 47879, 0x41d1677f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFist2, SpecFist2, Spectrum)
STD_ROM_FN(SpecFist2)

struct BurnDriver BurnSpecFist2 = {
	"spec_fist2", NULL, "spec_spectrum", NULL, "1986",
	"Fist II - The Legend Continues (48K)\0", "Press any key to choose options and ENTER to select one", "Melbourne House", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecFist2RomInfo, SpecFist2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Lone Wolf - The Mirror of Death (128K)

static struct BurnRomInfo SpecLonewolfRomDesc[] = {
	{ "Lone Wolf - The Mirror of Death 128K (1991)(Audiogenic).z80", 69519, 0xf598b404, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecLonewolf, SpecLonewolf, Spec128)
STD_ROM_FN(SpecLonewolf)

struct BurnDriver BurnSpecLonewolf = {
	"spec_lonewolf", NULL, "spec_spec128", NULL, "1990",
	"Lone Wolf - The Mirror of Death (128K)\0", NULL, "Audiogenic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecLonewolfRomInfo, SpecLonewolfRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Loopz (48K)

static struct BurnRomInfo SpecLoopzRomDesc[] = {
	{ "Loopz 48K (1991)(Audiogenic).tap", 44809, 0x27fbfd1e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecLoopz, SpecLoopz, Spectrum)
STD_ROM_FN(SpecLoopz)

struct BurnDriver BurnSpecLoopz = {
	"spec_loopz", NULL, "spec_spectrum", NULL, "1991",
	"Loopz (48K)\0", "SPACE: rotate the piece - ENTER: put the piece", "Audiogenic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecLoopzRomInfo, SpecLoopzRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecCursorKeysDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sabrina (Spanish) (48K)

static struct BurnRomInfo SpecSabrinaRomDesc[] = {
	{ "Sabrina ES 48K (1989)(Iber Soft).z80", 35718, 0x04ffed4b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSabrina, SpecSabrina, Spectrum)
STD_ROM_FN(SpecSabrina)

struct BurnDriver BurnSpecSabrina = {
	"spec_sabrina", NULL, "spec_spectrum", NULL, "1989",
	"Sabrina (Spanish) (48K)\0", NULL, "Iber Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecSabrinaRomInfo, SpecSabrinaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPMDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Tuareg (Spanish) (48K)

static struct BurnRomInfo SpecTuaregRomDesc[] = {
	{ "Tuareg ES 48K (1988)(Topo Soft).tap", 58215, 0x4fb0f564, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTuareg, SpecTuareg, Spectrum)
STD_ROM_FN(SpecTuareg)

struct BurnDriver BurnSpecTuareg = {
	"spec_tuareg", NULL, "spec_spectrum", NULL, "1988",
	"Tuareg (Spanish) (48K)\0", NULL, "Topo Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_ADV, 0,
	SpectrumGetZipName, SpecTuaregRomInfo, SpecTuaregRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hawk Storm (128K)

static struct BurnRomInfo SpecHawkstormRomDesc[] = {
	{ "Hawk Storm 128K (1990)(Players).z80", 41034, 0xf38d5e3c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHawkstorm, SpecHawkstorm, Spec128)
STD_ROM_FN(SpecHawkstorm)

struct BurnDriver BurnSpecHawkstorm = {
	"spec_hawkstorm", NULL, "spec_spec128", NULL, "1990",
	"Hawk Storm (128K)\0", NULL, "Players Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecHawkstormRomInfo, SpecHawkstormRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hollow, The (48K)

static struct BurnRomInfo SpecHollowRomDesc[] = {
	{ "Hollow, The 48K (1985)(Gilsoft International).tap", 47876, 0xd2d36c3a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHollow, SpecHollow, Spectrum)
STD_ROM_FN(SpecHollow)

struct BurnDriver BurnSpecHollow = {
	"spec_hollow", NULL, "spec_spectrum", NULL, "1985",
	"Hollow, The (48K)\0", NULL, "Gilsoft International", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecHollowRomInfo, SpecHollowRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Micronaut One (48K)

static struct BurnRomInfo SpecMicronautRomDesc[] = {
	{ "Micronaut One 48K (1987)(Nexus Productions).tap", 48156, 0xe8317652, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMicronaut, SpecMicronaut, Spectrum)
STD_ROM_FN(SpecMicronaut)

struct BurnDriver BurnSpecMicronaut = {
	"spec_micronaut", NULL, "spec_spectrum", NULL, "1987",
	"Micronaut One (48K)\0", "S, Z, O, P and SPACE to set options in main menu", "Nexus Productions", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SIM, 0,
	SpectrumGetZipName, SpecMicronautRomInfo, SpecMicronautRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mission Omega (48K)

static struct BurnRomInfo SpecMissomegaRomDesc[] = {
	{ "Mission Omega 48K (1983)(Pulsonic).tap", 34697, 0x7f5c1203, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMissomega, SpecMissomega, Spectrum)
STD_ROM_FN(SpecMissomega)

struct BurnDriver BurnSpecMissomega = {
	"spec_missomega", NULL, "spec_spectrum", NULL, "1983",
	"Mission Omega (48K)\0", NULL, "Pulsonic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecMissomegaRomInfo, SpecMissomegaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Misterio del Nilo, El (Spanish) (48K)

static struct BurnRomInfo SpecMisterioniloRomDesc[] = {
	{ "Misterio del Nilo, El 48K (1987)(Zigurat).tap", 50412, 0xfedd2fd4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMisterionilo, SpecMisterionilo, Spectrum)
STD_ROM_FN(SpecMisterionilo)

struct BurnDriver BurnSpecMisterionilo = {
	"spec_misterionilo", "spec_mysterynile", "spec_spectrum", NULL, "1987",
	"Misterio del Nilo, El (Spanish) (48K)\0", NULL, "Zigurat Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecMisterioniloRomInfo, SpecMisterioniloRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mystery of the Nile, The (English) (48K)

static struct BurnRomInfo SpecMysterynileRomDesc[] = {
	{ "Mystery of the Nile, The 48K (1987)(Firebird).tap", 50370, 0x3dad4b2a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMysterynile, SpecMysterynile, Spectrum)
STD_ROM_FN(SpecMysterynile)

struct BurnDriver BurnSpecMysterynile = {
	"spec_mysterynile", NULL, "spec_spectrum", NULL, "1987",
	"Mystery of the Nile, The (English) (48K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecMysterynileRomInfo, SpecMysterynileRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 1985 - The Day After (48K)

static struct BurnRomInfo Spec1985tdaRomDesc[] = {
	{ "1985 - The Day After (1985)(Mastertronic).tap", 32574, 0x2bee2cb2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec1985tda, Spec1985tda, Spectrum)
STD_ROM_FN(Spec1985tda)

struct BurnDriver BurnSpec1985tda = {
	"spec_1985tda", NULL, "spec_spectrum", NULL, "1985",
	"1985 - The Day After (48K)\0", NULL, "Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Spec1985tdaRomInfo, Spec1985tdaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// ACE 2088 (128K)

static struct BurnRomInfo SpecAce2088RomDesc[] = {
	{ "ACE 2088 128K (1989)(Cascade Games).tap", 55874, 0x4ebb46a9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAce2088, SpecAce2088, Spec128)
STD_ROM_FN(SpecAce2088)

struct BurnDriver BurnSpecAce2088 = {
	"spec_ace2088", NULL, "spec_spec128", NULL, "1989",
	"ACE 2088 (128K)\0", "SHIFT+J/K: speed up/down - L, R, T: change weapons - W: set display", "Cascade Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_SIM, 0,
	SpectrumGetZipName, SpecAce2088RomInfo, SpecAce2088RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// C.O.R.E. (48K)

static struct BurnRomInfo SpecCoreRomDesc[] = {
	{ "C.O.R.E. (1986)(A'n'F Software).tap", 48919, 0x716221c7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCore, SpecCore, Spectrum)
STD_ROM_FN(SpecCore)

struct BurnDriver BurnSpecCore = {
	"spec_core", NULL, "spec_spectrum", NULL, "1986",
	"C.O.R.E. (48K)\0", NULL, "A'n'F Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_ADV, 0,
	SpectrumGetZipName, SpecCoreRomInfo, SpecCoreRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Gremlins - The Adventure (48K)

static struct BurnRomInfo SpecGremlinsadvRomDesc[] = {
	{ "Gremlins - The Adventure (1985)(Adventure International).z80", 45332, 0x7049389f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGremlinsadv, SpecGremlinsadv, Spectrum)
STD_ROM_FN(SpecGremlinsadv)

struct BurnDriver BurnSpecGremlinsadv = {
	"spec_gremlinsadv", NULL, "spec_spectrum", NULL, "1985",
	"Gremlins - The Adventure (48K)\0", NULL, "Adventure International", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecGremlinsadvRomInfo, SpecGremlinsadvRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Motorbike Madness (48K)

static struct BurnRomInfo SpecMotomadnessRomDesc[] = {
	{ "Motorbike Madness 48K (1988)(Mastertronic).tap", 68303, 0x8346f6c5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMotomadness, SpecMotomadness, Spectrum)
STD_ROM_FN(SpecMotomadness)

struct BurnDriver BurnSpecMotomadness = {
	"spec_motomadness", NULL, "spec_spectrum", NULL, "1988",
	"Motorbike Madness (48K)\0", NULL, "Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecMotomadnessRomInfo, SpecMotomadnessRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 19 Part 1 - Boot Camp (128K)

static struct BurnRomInfo Spec19bootcampRomDesc[] = {
	{ "19 Part 1 - Boot Camp 128K (1988)(Cascade Games).tap", 213402, 0xae022874, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec19bootcamp, Spec19bootcamp, Spec128)
STD_ROM_FN(Spec19bootcamp)

struct BurnDriver BurnSpec19bootcamp = {
	"spec_19bootcamp", NULL, "spec_spec128", NULL, "1988",
	"19 Part 1 - Boot Camp (128K)\0", NULL, "Cascade Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 4, HARDWARE_SPECTRUM, GBF_ACTION | GBF_SHOOT, 0,
	SpectrumGetZipName, Spec19bootcampRomInfo, Spec19bootcampRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// ACE - Air Combat Emulator (48K)

static struct BurnRomInfo SpecAceRomDesc[] = {
	{ "ACE 48K (1986)(Cascade Games).tap", 46187, 0x8568a584, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAce, SpecAce, Spectrum)
STD_ROM_FN(SpecAce)

struct BurnDriver BurnSpecAce = {
	"spec_ace", NULL, "spec_spectrum", NULL, "1986",
	"ACE - Air Combat Emulator (48K)\0", "LSHIFT-Z: throttle - U: undercarriage - M: map - J: eject", "Cascade Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SIM, 0,
	SpectrumGetZipName, SpecAceRomInfo, SpecAceRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// ACE 2 - The Ultimate Head to Head Conflict (128K)

static struct BurnRomInfo SpecAce2RomDesc[] = {
	{ "ACE 2 The Ultimate Head to Head Conflict 128K (1987)(Cascade Games).tap", 52048, 0x4f850b86, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAce2, SpecAce2, Spec128)
STD_ROM_FN(SpecAce2)

struct BurnDriver BurnSpecAce2 = {
	"spec_ace2", NULL, "spec_spec128", NULL, "1987",
	"ACE 2 - The Ultimate Head to Head Conflict (128K)\0", "D-S/K-J: throttle - W/O: select weapon - C/B: map", "Cascade Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SIM, 0,
	SpectrumGetZipName, SpecAce2RomInfo, SpecAce2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Frightmare (128K)

static struct BurnRomInfo SpecFrightmareRomDesc[] = {
	{ "Frightmare 128K (1988)(Cascade Games).tap", 59861, 0xa5ba5804, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFrightmare, SpecFrightmare, Spec128)
STD_ROM_FN(SpecFrightmare)

struct BurnDriver BurnSpecFrightmare = {
	"spec_frightmare", NULL, "spec_spec128", NULL, "1988",
	"Frightmare (128K)\0", NULL, "Cascade Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecFrightmareRomInfo, SpecFrightmareRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Drazen Petrovic Basket (spanish) (128K)

static struct BurnRomInfo SpecDpbasketRomDesc[] = {
	{ "Drazen Petrovic Basket ES 128K (1989)(Topo Soft).tap", 64906, 0xf0d8a280, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDpbasket, SpecDpbasket, Spec128)
STD_ROM_FN(SpecDpbasket)

struct BurnDriver BurnSpecDpbasket = {
	"spec_dpbasket", NULL, "spec_spec128", NULL, "1989",
	"Drazen Petrovic Basket (Spanish) (128K)\0", NULL, "Topo Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecDpbasketRomInfo, SpecDpbasketRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dynamite Dux (128K)

static struct BurnRomInfo SpecDynamduxRomDesc[] = {
	{ "Dynamite Dux (1989)(Activision).tap", 150855, 0xc13fc9c0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDynamdux, SpecDynamdux, Spec128)
STD_ROM_FN(SpecDynamdux)

struct BurnDriver BurnSpecDynamdux = {
	"spec_dynamdux", NULL, "spec_spec128", NULL, "1989",
	"Dynamite Dux (128K)\0", "Press '2' to use controller", "Activision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecDynamduxRomInfo, SpecDynamduxRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Invaders (16K)

static struct BurnRomInfo SpecInvadersRomDesc[] = {
	{ "Invaders 16K (1983)(DK'Tronics).tap", 6795, 0xa6263eb9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecInvaders, SpecInvaders, Spectrum)
STD_ROM_FN(SpecInvaders)

struct BurnDriver BurnSpecInvaders = {
	"spec_invaders", NULL, "spec_spectrum", NULL, "1983",
	"Invaders (16K)\0", NULL, "DK'Tronics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecInvadersRomInfo, SpecInvadersRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecCursorKeysDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Kenny Dalglish Soccer Manager (48K)

static struct BurnRomInfo SpecKdmanagerRomDesc[] = {
	{ "Kenny Dalglish Soccer Manager 48K (1989)(Cognito).tap", 54337, 0xaaa8bbb3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecKdmanager, SpecKdmanager, Spectrum)
STD_ROM_FN(SpecKdmanager)

struct BurnDriver BurnSpecKdmanager = {
	"spec_kdmanager", NULL, "spec_spectrum", NULL, "1989",
	"Kenny Dalglish Soccer Manager (48K)\0", NULL, "Cognito", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_STRATEGY, 0,
	SpectrumGetZipName, SpecKdmanagerRomInfo, SpecKdmanagerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Kenny Dalglish Soccer Match (128K)

static struct BurnRomInfo SpecKdmatchRomDesc[] = {
	{ "Kenny Dalglish Soccer Match 128K (1990)(Impressions).tap", 55717, 0xf9447e26, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecKdmatch, SpecKdmatch, Spec128)
STD_ROM_FN(SpecKdmatch)

struct BurnDriver BurnSpecKdmatch = {
	"spec_kdmatch", NULL, "spec_spec128", NULL, "1990",
	"Kenny Dalglish Soccer Match (128K)\0", NULL, "Impressions", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSFOOTBALL, 0,
	SpectrumGetZipName, SpecKdmatchRomInfo, SpecKdmatchRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Pinball (16K)

static struct BurnRomInfo SpecPinballRomDesc[] = {
	{ "Pinball 16K (1983)(Sagittarian).tap", 16187, 0x26267d3e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPinball, SpecPinball, Spectrum)
STD_ROM_FN(SpecPinball)

struct BurnDriver BurnSpecPinball = {
	"spec_pinball", NULL, "spec_spectrum", NULL, "1983",
	"Pinball (16K)\0", "Q: left flipper - P: right flipper - 0: launch the ball", "Sagittarian Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PINBALL, 0,
	SpectrumGetZipName, SpecPinballRomInfo, SpecPinballRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Agent X (48K)

static struct BurnRomInfo SpecAgentxRomDesc[] = {
	{ "Agent X 48K (1986)(Mastertronic).tap", 73557, 0xcfac7765, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAgentx, SpecAgentx, Spectrum)
STD_ROM_FN(SpecAgentx)

struct BurnDriver BurnSpecAgentx = {
	"spec_agentx", NULL, "spec_spectrum", NULL, "1986",
	"Agent X (48K)\0", NULL, "Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecAgentxRomInfo, SpecAgentxRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Agent X II - The Mad Prof's Back! (128K)

static struct BurnRomInfo SpecAgentx2RomDesc[] = {
	{ "Agent X II - The Mad Prof's Back! 128K (1987)(Mastertronic).tap", 144693, 0x889a8e99, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAgentx2, SpecAgentx2, Spec128)
STD_ROM_FN(SpecAgentx2)

struct BurnDriver BurnSpecAgentx2 = {
	"spec_agentx2", NULL, "spec_spec128", NULL, "1987",
	"Agent X II - The Mad Prof's Back! (128K)\0", NULL, "Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecAgentx2RomInfo, SpecAgentx2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Alien Highway (48K)

static struct BurnRomInfo SpecAlienhwRomDesc[] = {
	{ "Alien Highway 48K (1986)(Vortex).tap", 48086, 0xf5a22737, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAlienhw, SpecAlienhw, Spectrum)
STD_ROM_FN(SpecAlienhw)

struct BurnDriver BurnSpecAlienhw = {
	"spec_alienhw", NULL, "spec_spectrum", NULL, "1986",
	"Alien Highway (48K)\0", NULL, "Vortex Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecAlienhwRomInfo, SpecAlienhwRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Automania (48K)

static struct BurnRomInfo SpecAutomaniaRomDesc[] = {
	{ "Automania 48K (1984)(Mikro-Gen).tap", 39462, 0xc637b38a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAutomania, SpecAutomania, Spectrum)
STD_ROM_FN(SpecAutomania)

struct BurnDriver BurnSpecAutomania = {
	"spec_automania", NULL, "spec_spectrum", NULL, "1984",
	"Automania (48K)\0", NULL, "Mikro-Gen", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecAutomaniaRomInfo, SpecAutomaniaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Havoc (128K)

static struct BurnRomInfo SpecHavocRomDesc[] = {
	{ "Havoc 128K (1990)(Players Premier).z80", 48092, 0x510c85ea, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHavoc, SpecHavoc, Spec128)
STD_ROM_FN(SpecHavoc)

struct BurnDriver BurnSpecHavoc = {
	"spec_havoc", NULL, "spec_spec128", NULL, "1990",
	"Havoc (128K)\0", NULL, "Players Premier", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecHavocRomInfo, SpecHavocRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Critical Mass (48K)

static struct BurnRomInfo SpecCriticmassRomDesc[] = {
	{ "Critical Mass 48K (1985)(Durell).tap", 44720, 0xba99bc3e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCriticmass, SpecCriticmass, Spectrum)
STD_ROM_FN(SpecCriticmass)

struct BurnDriver BurnSpecCriticmass = {
	"spec_criticmass", NULL, "spec_spectrum", NULL, "1985",
	"Critical Mass (48K)\0", NULL, "Durell Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecCriticmassRomInfo, SpecCriticmassRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Great Gurianos (48K)

static struct BurnRomInfo SpecGreatgurianRomDesc[] = {
	{ "Great Gurianos 48K (1991)(Hit-Pak).z80", 39089, 0x2a96de1d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGreatgurian, SpecGreatgurian, Spectrum)
STD_ROM_FN(SpecGreatgurian)

struct BurnDriver BurnSpecGreatgurian = {
	"spec_greatgurian", NULL, "spec_spectrum", NULL, "1991",
	"Great Gurianos (48K)\0", NULL, "Hit-Pak", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecGreatgurianRomInfo, SpecGreatgurianRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mot - Intro (Spanish) (128K)

static struct BurnRomInfo Specmot128introRomDesc[] = {
	{ "Mot - Intro ES 128K2a (1989)(Opera Soft).tap", 49127, 0x76c4cac9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmot128intro, Specmot128intro, Spec1282a)
STD_ROM_FN(Specmot128intro)

struct BurnDriver BurnSpecmot128intro = {
	"spec_mot128intro", NULL, "spec_spec1282a", NULL, "1989",
	"Mot - Intro (Spanish) (128K)\0", NULL, "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specmot128introRomInfo, Specmot128introRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KPlus2Init, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mot - Part 1 (Spanish) (128K)

static struct BurnRomInfo Specmot128aRomDesc[] = {
	{ "Mot - Part 1 ES 128K2a (1989)(Opera Soft).tap", 49127, 0x0713edec, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmot128a, Specmot128a, Spec1282a)
STD_ROM_FN(Specmot128a)

struct BurnDriver BurnSpecmot128a = {
	"spec_mot128a", "spec_mot128intro", "spec_spec1282a", NULL, "1989",
	"Mot - Part 1 (Spanish) (128K)\0", NULL, "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specmot128aRomInfo, Specmot128aRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KPlus2Init, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mot - Part 2 (Spanish) (128K)

static struct BurnRomInfo Specmot128bRomDesc[] = {
	{ "Mot - Part 2 ES 128K2a (1989)(Opera Soft).tap", 49127, 0xff0cb875, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmot128b, Specmot128b, Spec1282a)
STD_ROM_FN(Specmot128b)

struct BurnDriver BurnSpecmot128b = {
	"spec_mot128b", "spec_mot128intro", "spec_spec1282a", NULL, "1989",
	"Mot - Part 2 (Spanish) (128K)\0", NULL, "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specmot128bRomInfo, Specmot128bRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KPlus2Init, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mot - Part 3 (Spanish) (128K)

static struct BurnRomInfo Specmot128cRomDesc[] = {
	{ "Mot - Part 3 ES 128K2a (1989)(Opera Soft).tap", 49127, 0x43ed24bc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmot128c, Specmot128c, Spec1282a)
STD_ROM_FN(Specmot128c)

struct BurnDriver BurnSpecmot128c = {
	"spec_mot128c", "spec_mot128intro", "spec_spec1282a", NULL, "1989",
	"Mot - Part 3 (Spanish) (128K)\0", NULL, "Opera Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specmot128cRomInfo, Specmot128cRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KPlus2Init, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Overlander (128K)

static struct BurnRomInfo SpecOverlanderRomDesc[] = {
	{ "Overlander 128K (1988)(Elite Systems).tap", 58270, 0xf0e79999, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecOverlander, SpecOverlander, Spec128)
STD_ROM_FN(SpecOverlander)

struct BurnDriver BurnSpecOverlander = {
	"spec_overlander", NULL, "spec_spec128", NULL, "1988",
	"Overlander (128K)\0", NULL, "Elite Systems", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING | GBF_SHOOT, 0,
	SpectrumGetZipName, SpecOverlanderRomInfo, SpecOverlanderRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Beach-Head (48K)

static struct BurnRomInfo SpecBheadRomDesc[] = {
	{ "Beach-Head 48K (1984)(U.S. Gold).tap", 48048, 0x429ecb91, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBhead, SpecBhead, Spectrum)
STD_ROM_FN(SpecBhead)

struct BurnDriver BurnSpecBhead = {
	"spec_bhead", NULL, "spec_spectrum", NULL, "1984",
	"Beach-Head (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecBheadRomInfo, SpecBheadRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Beach-Head II - The Dictator Strikes Back! (48K)

static struct BurnRomInfo SpecBhead2RomDesc[] = {
	{ "Beach-Head II - The Dictator Strikes Back! 48K (1985)(U.S. Gold).tap", 48857, 0xe977cfce, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBhead2, SpecBhead2, Spectrum)
STD_ROM_FN(SpecBhead2)

struct BurnDriver BurnSpecBhead2 = {
	"spec_bhead2", NULL, "spec_spectrum", NULL, "1985",
	"Beach-Head II - The Dictator Strikes Back! (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecBhead2RomInfo, SpecBhead2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Battlezone (48K)

static struct BurnRomInfo SpecBtlzoneRomDesc[] = {
	{ "Battlezone 48K (1984)(Quicksilva).tap", 36210, 0xaf085e02, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBtlzone, SpecBtlzone, Spectrum)
STD_ROM_FN(SpecBtlzone)

struct BurnDriver BurnSpecBtlzone = {
	"spec_btlzone", NULL, "spec_spectrum", NULL, "1984",
	"Battlezone (48K)\0", NULL, "Quicksilva", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecBtlzoneRomInfo, SpecBtlzoneRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Chubby Gristle (128K)

static struct BurnRomInfo SpecChubbyRomDesc[] = {
	{ "Chubby Gristle 128K (1988)(Grandslam).tap", 52414, 0x46a7bdd3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecChubby, SpecChubby, Spec128)
STD_ROM_FN(SpecChubby)

struct BurnDriver BurnSpecChubby = {
	"spec_chubby", NULL, "spec_spec128", NULL, "1988",
	"Chubby Gristle (128K)\0", NULL, "Grandslam Entertainments", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecChubbyRomInfo, SpecChubbyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Crack Down (128K)

static struct BurnRomInfo SpecCrackdownRomDesc[] = {
	{ "Crack Down 128K (1990)(U.S. Gold).tap", 189590, 0xc82c010e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCrackdown, SpecCrackdown, Spec128)
STD_ROM_FN(SpecCrackdown)

struct BurnDriver BurnSpecCrackdown = {
	"spec_crackdown", NULL, "spec_spec128", NULL, "1990",
	"Crack Down (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecCrackdownRomInfo, SpecCrackdownRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Defcom (128K)

static struct BurnRomInfo SpecDefcomRomDesc[] = {
	{ "Defcom 128K (1986)(Quicksilva).tap", 55692, 0x0f2a7a1b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDefcom, SpecDefcom, Spec128)
STD_ROM_FN(SpecDefcom)

struct BurnDriver BurnSpecDefcom = {
	"spec_defcom", NULL, "spec_spec128", NULL, "1986",
	"Defcom (128K)\0", NULL, "Quicksilva", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecDefcomRomInfo, SpecDefcomRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Spectrum Cross (16K)

static struct BurnRomInfo SpecSpecrossRomDesc[] = {
	{ "Spectrum Cross 16K (1983)(Your Computer).tap", 8210, 0xf16574c6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSpecross, SpecSpecross, Spectrum)
STD_ROM_FN(SpecSpecross)

struct BurnDriver BurnSpecSpecross = {
	"spec_specross", NULL, "spec_spectrum", NULL, "1983",
	"Spectrum Cross (16K)\0", "1: up - 9: left - 0: right", "Your Computer", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecSpecrossRomInfo, SpecSpecrossRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Tapper (48K)

static struct BurnRomInfo SpecTapperRomDesc[] = {
	{ "Tapper 48K (1985)(U.S. Gold).tap", 48800, 0xdcc02c66, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTapper, SpecTapper, Spectrum)
STD_ROM_FN(SpecTapper)

struct BurnDriver BurnSpecTapper = {
	"spec_tapper", NULL, "spec_spectrum", NULL, "1985",
	"Tapper (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecTapperRomInfo, SpecTapperRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Blue Max (48K)

static struct BurnRomInfo SpecBluemaxRomDesc[] = {
	{ "Blue Max 48K (1984)(U.S. Gold).tap", 42721, 0x13b6bc71, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBluemax, SpecBluemax, Spectrum)
STD_ROM_FN(SpecBluemax)

struct BurnDriver BurnSpecBluemax = {
	"spec_bluemax", NULL, "spec_spectrum", NULL, "1984",
	"Blue Max (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecBluemaxRomInfo, SpecBluemaxRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Express Raider (48K)

static struct BurnRomInfo SpecExpraiderRomDesc[] = {
	{ "Express Raider 48K (1987)(U.S. Gold).z80", 42604, 0x78fa11ff, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecExpraider, SpecExpraider, Spectrum)
STD_ROM_FN(SpecExpraider)

struct BurnDriver BurnSpecExpraider = {
	"spec_expraider", NULL, "spec_spectrum", NULL, "1987",
	"Express Raider (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecExpraiderRomInfo, SpecExpraiderRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Final Assault (128K)

static struct BurnRomInfo SpecFinalassaultRomDesc[] = {
	{ "Final Assault 128K (1988)(U.S. Gold).z80", 82875, 0xbd8bec7a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFinalassault, SpecFinalassault, Spec128)
STD_ROM_FN(SpecFinalassault)

struct BurnDriver BurnSpecFinalassault = {
	"spec_finalassault", NULL, "spec_spec128", NULL, "1988",
	"Final Assault (128K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecFinalassaultRomInfo, SpecFinalassaultRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Heavy Metal (128K)

static struct BurnRomInfo SpecHeavymetalRomDesc[] = {
	{ "Heavy Metal 128K (1990)(U.S. Gold).tap", 86702, 0x67e11acf, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHeavymetal, SpecHeavymetal, Spec128)
STD_ROM_FN(SpecHeavymetal)

struct BurnDriver BurnSpecHeavymetal = {
	"spec_heavymetal", NULL, "spec_spec128", NULL, "1990",
	"Heavy Metal (128K)\0", "W/S: speed up/down - E: select weapon - D: smoke", "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecHeavymetalRomInfo, SpecHeavymetalRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Leader Board (48K)

static struct BurnRomInfo SpecLeaderboardRomDesc[] = {
	{ "Leader Board 48K (1987)(U.S. Gold).tap", 45712, 0x97794c06, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecLeaderboard, SpecLeaderboard, Spectrum)
STD_ROM_FN(SpecLeaderboard)

struct BurnDriver BurnSpecLeaderboard = {
	"spec_leaderboard", NULL, "spec_spectrum", NULL, "1987",
	"Leader Board (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 4, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecLeaderboardRomInfo, SpecLeaderboardRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Pssst (16K)

static struct BurnRomInfo SpecPssstRomDesc[] = {
	{ "Pssst 16K (1983)(Ultimate Play The Game).tap", 15927, 0x5f5272d5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPssst, SpecPssst, Spectrum)
STD_ROM_FN(SpecPssst)

struct BurnDriver BurnSpecPssst = {
	"spec_pssst", NULL, "spec_spectrum", NULL, "1983",
	"Pssst (16K)\0", NULL, "Ultimate Play The Game", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecPssstRomInfo, SpecPssstRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Road Runner (48K)

static struct BurnRomInfo SpecRoadrunnerRomDesc[] = {
	{ "Road Runner 48K (1987)(U.S. Gold).z80", 45123, 0x4b781d7c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecRoadrunner, SpecRoadrunner, Spectrum)
STD_ROM_FN(SpecRoadrunner)

struct BurnDriver BurnSpecRoadrunner = {
	"spec_roadrunner", NULL, "spec_spectrum", NULL, "1987",
	"Road Runner (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecRoadrunnerRomInfo, SpecRoadrunnerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// BMX Ninja (48K)

static struct BurnRomInfo SpecBmxninjaRomDesc[] = {
	{ "BMX Ninja 48K (1988)(Alternative).tap", 57376, 0x64ed0d8b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBmxninja, SpecBmxninja, Spectrum)
STD_ROM_FN(SpecBmxninja)

struct BurnDriver BurnSpecBmxninja = {
	"spec_bmxninja", NULL, "spec_spectrum", NULL, "1988",
	"BMX Ninja (48K)\0", NULL, "Alternative", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpecBmxninjaRomInfo, SpecBmxninjaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// BMX Racers (48K)

static struct BurnRomInfo SpecBmxraceRomDesc[] = {
	{ "BMX Racers 48K (1984)(Mastertronic).tap", 45509, 0x40ec8744, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBmxrace, SpecBmxrace, Spectrum)
STD_ROM_FN(SpecBmxrace)

struct BurnDriver BurnSpecBmxrace = {
	"spec_bmxrace", NULL, "spec_spectrum", NULL, "1984",
	"BMX Racers (48K)\0", NULL, "Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecBmxraceRomInfo, SpecBmxraceRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Championship Run (48K)

static struct BurnRomInfo SpecChamprunRomDesc[] = {
	{ "Championship Run 48K (1991)(Impulze).tap", 46687, 0x08eeabd6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecChamprun, SpecChamprun, Spectrum)
STD_ROM_FN(SpecChamprun)

struct BurnDriver BurnSpecChamprun = {
	"spec_champrun", NULL, "spec_spectrum", NULL, "1991",
	"Championship Run (48K)\0", NULL, "Impulze", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecChamprunRomInfo, SpecChamprunRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Planetoids (16K)

static struct BurnRomInfo SpecPlanetoidsRomDesc[] = {
	{ "Planetoids 16K (1982)(Sinclair Research).tap", 15635, 0x9c2a2474, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPlanetoids, SpecPlanetoids, Spectrum)
STD_ROM_FN(SpecPlanetoids)

struct BurnDriver BurnSpecPlanetoids = {
	"spec_planetoids", NULL, "spec_spectrum", NULL, "1982",
	"Planetoids (16K)\0", NULL, "Sinclair Research", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecPlanetoidsRomInfo, SpecPlanetoidsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2SwapDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Titanic Blinky (48K)

static struct BurnRomInfo SpecTitblinkyRomDesc[] = {
	{ "Titanic Blinky 48K (1991)(Zeppelin Games).tap", 46442, 0xdff53244, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTitblinky, SpecTitblinky, Spectrum)
STD_ROM_FN(SpecTitblinky)

struct BurnDriver BurnSpecTitblinky = {
	"spec_titblinky", NULL, "spec_spectrum", NULL, "1991",
	"Titanic Blinky (48K)\0", NULL, "Zeppelin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecTitblinkyRomInfo, SpecTitblinkyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// American Tag-Team Wrestling (48K)

static struct BurnRomInfo SpecAttwrestRomDesc[] = {
	{ "American Tag-Team Wrestling 48K (1992)(Zeppelin Games).z80", 38767, 0xafb4c0cc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAttwrest, SpecAttwrest, Spectrum)
STD_ROM_FN(SpecAttwrest)

struct BurnDriver BurnSpecAttwrest = {
	"spec_attwrest", NULL, "spec_spectrum", NULL, "1992",
	"American Tag-Team Wrestling (48K)\0", NULL, "Zeppelin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpecAttwrestRomInfo, SpecAttwrestRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bedlam (128K)

static struct BurnRomInfo SpecBedlamRomDesc[] = {
	{ "Bedlam 128K (1988)(Go!).tap", 106920, 0x733eb4ae, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBedlam, SpecBedlam, Spec128)
STD_ROM_FN(SpecBedlam)

struct BurnDriver BurnSpecBedlam = {
	"spec_bedlam", NULL, "spec_spec128", NULL, "1988",
	"Bedlam (128K)\0", NULL, "Go!", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecBedlamRomInfo, SpecBedlamRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Crime Busters Featuring Bennie the Burglar (48K)

static struct BurnRomInfo SpecCrimebustRomDesc[] = {
	{ "Crime Busters Featuring Bennie the Burglar (1988)(Players Software).tap", 88832, 0x6fa4d165, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCrimebust, SpecCrimebust, Spectrum)
STD_ROM_FN(SpecCrimebust)

struct BurnDriver BurnSpecCrimebust = {
	"spec_crimebust", NULL, "spec_spectrum", NULL, "1988",
	"Crime Busters Featuring Bennie the Burglar (48K)\0", "'Q', 'W' and Space to set options", "Players Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecCrimebustRomInfo, SpecCrimebustRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Lazer Tag (48K)

static struct BurnRomInfo SpecLazertagRomDesc[] = {
	{ "Lazer Tag 48K (1988)(Go!).tap", 29696, 0xa88a379d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecLazertag, SpecLazertag, Spectrum)
STD_ROM_FN(SpecLazertag)

struct BurnDriver BurnSpecLazertag = {
	"spec_lazertag", NULL, "spec_spectrum", NULL, "1988",
	"Lazer Tag (48K)\0", NULL, "Go!", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION | GBF_SHOOT, 0,
	SpectrumGetZipName, SpecLazertagRomInfo, SpecLazertagRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Neighbours (48K)

static struct BurnRomInfo SpecNeighboursRomDesc[] = {
	{ "Neighbours 48K (1992)(Impulze).tap", 52388, 0x1a852016, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecNeighbours, SpecNeighbours, Spectrum)
STD_ROM_FN(SpecNeighbours)

struct BurnDriver BurnSpecNeighbours = {
	"spec_neighbours", NULL, "spec_spectrum", NULL, "1992",
	"Neighbours (48K)\0", "Set '3 SINCLAIR' to use controller", "Impulze", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecNeighboursRomInfo, SpecNeighboursRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sharkey's Moll (128K)

static struct BurnRomInfo SpecSkmollRomDesc[] = {
	{ "Sharkey's Moll 128K (1991)(Zeppelin Games).tap", 44068, 0xc831bf3f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSkmoll, SpecSkmoll, Spec128)
STD_ROM_FN(SpecSkmoll)

struct BurnDriver BurnSpecSkmoll = {
	"spec_skmoll", NULL, "spec_spec128", NULL, "1991",
	"Sharkey's Moll (128K)\0", NULL, "Zeppelin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecSkmollRomInfo, SpecSkmollRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Stack Up (48K)

static struct BurnRomInfo SpecStackupRomDesc[] = {
	{ "Stack Up 48K (1991)(Zeppelin Games).tap", 43582, 0xf07a8de4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecStackup, SpecStackup, Spectrum)
STD_ROM_FN(SpecStackup)

struct BurnDriver BurnSpecStackup = {
	"spec_stackup", NULL, "spec_spectrum", NULL, "1991",
	"Stack Up (48K)\0", NULL, "Zeppelin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecStackupRomInfo, SpecStackupRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Submariner (48K)

static struct BurnRomInfo SpecSubmarinerRomDesc[] = {
	{ "Submariner 48K (1988)(Atlantis).tap", 37754, 0xbdaf3289, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSubmariner, SpecSubmariner, Spectrum)
STD_ROM_FN(SpecSubmariner)

struct BurnDriver BurnSpecSubmariner = {
	"spec_submariner", NULL, "spec_spectrum", NULL, "1988",
	"Submariner (48K)\0", NULL, "Atlantis Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecSubmarinerRomInfo, SpecSubmarinerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Zenji (48K)

static struct BurnRomInfo SpecZenjiRomDesc[] = {
	{ "Zenji 48K (1984)(Activision).tap", 23232, 0x62e604e5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecZenji, SpecZenji, Spectrum)
STD_ROM_FN(SpecZenji)

struct BurnDriver BurnSpecZenji = {
	"spec_zenji", NULL, "spec_spectrum", NULL, "1984",
	"Zenji (48K)\0", NULL, "Activision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecZenjiRomInfo, SpecZenjiRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Buffalo Bill's Wild West Show (128K)

static struct BurnRomInfo SpecBbwwsRomDesc[] = {
	{ "Buffalo Bill's Wild West Show 128K (1989)(Tynesoft).tap", 131092, 0xfc253b18, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBbwws, SpecBbwws, Spec128)
STD_ROM_FN(SpecBbwws)

struct BurnDriver BurnSpecBbwws = {
	"spec_bbwws", NULL, "spec_spec128", NULL, "1989",
	"Buffalo Bill's Wild West Show (128K)\0", "Set 'Sinclair port 1' to use controller", "Tynesoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 4, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecBbwwsRomInfo, SpecBbwwsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Circus Games (48K)

static struct BurnRomInfo SpecCircusgamesRomDesc[] = {
	{ "Circus Games 48K (1988)(Tynesoft).tap", 151897, 0x4d953275, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCircusgames, SpecCircusgames, Spectrum)
STD_ROM_FN(SpecCircusgames)

struct BurnDriver BurnSpecCircusgames = {
	"spec_circusgames", NULL, "spec_spectrum", NULL, "1988",
	"Circus Games (48K)\0", NULL, "Tynesoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecCircusgamesRomInfo, SpecCircusgamesRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Cop-Out (48K)

static struct BurnRomInfo SpecCopoutRomDesc[] = {
	{ "Cop-Out 48K (1986)(Mikro-Gen).tap", 49468, 0x27dd134e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCopout, SpecCopout, Spectrum)
STD_ROM_FN(SpecCopout)

struct BurnDriver BurnSpecCopout = {
	"spec_copout", NULL, "spec_spectrum", NULL, "1986",
	"Cop-Out (48K)\0", NULL, "Mikro-Gen", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecCopoutRomInfo, SpecCopoutRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Frank N Stein (48K)

static struct BurnRomInfo SpecFranknsteinRomDesc[] = {
	{ "Frank N Stein 48K (1984)(PSS).tap", 47860, 0x3ff60912, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFranknstein, SpecFranknstein, Spectrum)
STD_ROM_FN(SpecFranknstein)

struct BurnDriver BurnSpecFranknstein = {
	"spec_franknstein", NULL, "spec_spectrum", NULL, "1984",
	"Frank N Stein (48K)\0", NULL, "PSS", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecFranknsteinRomInfo, SpecFranknsteinRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sabotage (48K)

static struct BurnRomInfo SpecSabotageRomDesc[] = {
	{ "Sabotage 48K (1988)(Zeppelin Games).tap", 43408, 0xb83fb136, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSabotage, SpecSabotage, Spectrum)
STD_ROM_FN(SpecSabotage)

struct BurnDriver BurnSpecSabotage = {
	"spec_sabotage", NULL, "spec_spectrum", NULL, "1988",
	"Sabotage (48K)\0", "Select icon 'S' to use controller", "Zeppelin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecSabotageRomInfo, SpecSabotageRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Side Arms (48K)

static struct BurnRomInfo SpecSidearmsRomDesc[] = {
	{ "Side Arms 48K (1988)(Go!).z80", 47259, 0xcb9fcf76, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSidearms, SpecSidearms, Spectrum)
STD_ROM_FN(SpecSidearms)

struct BurnDriver BurnSpecSidearms = {
	"spec_sidearms", NULL, "spec_spectrum", NULL, "1988",
	"Side Arms (48K)\0", NULL, "Go!", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecSidearmsRomInfo, SpecSidearmsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Anarchy (48K)

static struct BurnRomInfo SpecAnarchyRomDesc[] = {
	{ "Anarchy 48K (1987)(Rack-It).tap", 39327, 0xa48348c3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAnarchy, SpecAnarchy, Spectrum)
STD_ROM_FN(SpecAnarchy)

struct BurnDriver BurnSpecAnarchy = {
	"spec_anarchy", NULL, "spec_spectrum", NULL, "1987",
	"Anarchy (48K)\0", NULL, "Rack-It - Hewson", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecAnarchyRomInfo, SpecAnarchyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bestial Warrior (Spanish) (48K)

static struct BurnRomInfo SpecBestialwrrRomDesc[] = {
	{ "Bestial Warrior 48K ES (1989)(Dinamic).tap", 55620, 0xf1fb3cb5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBestialwrr, SpecBestialwrr, Spectrum)
STD_ROM_FN(SpecBestialwrr)

struct BurnDriver BurnSpecBestialwrr = {
	"spec_bestialwrr", NULL, "spec_spectrum", NULL, "1989",
	"Bestial Warrior (Spanish) (48K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecBestialwrrRomInfo, SpecBestialwrrRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Battle Valley (128K)

static struct BurnRomInfo SpecBtlvalleyRomDesc[] = {
	{ "Battle Valley 128K (1988)(Rack-It).tap", 39021, 0x93b3ce85, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBtlvalley, SpecBtlvalley, Spec128)
STD_ROM_FN(SpecBtlvalley)

struct BurnDriver BurnSpecBtlvalley = {
	"spec_btlvalley", NULL, "spec_spec128", NULL, "1988",
	"Battle Valley (128K)\0", "Double click on fire button to stop/reverse", "Rack-It - Hewson", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecBtlvalleyRomInfo, SpecBtlvalleyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Capitan Sevilla (Spanish) (48K)

static struct BurnRomInfo SpecCaptsevillaRomDesc[] = {
	{ "Capitan Sevilla 48K ES (1988)(Dinamic).tap", 97238, 0x8f8c61e7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCaptsevilla, SpecCaptsevilla, Spectrum)
STD_ROM_FN(SpecCaptsevilla)

struct BurnDriver BurnSpecCaptsevilla = {
	"spec_captsevilla", NULL, "spec_spectrum", NULL, "1988",
	"Capitan Sevilla (Spanish) (48K)\0", "SPACE: change to Cap. Sevilla (if available)", "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecCaptsevillaRomInfo, SpecCaptsevillaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Cosmic Sheriff (Spanish) (48K)

static struct BurnRomInfo SpecCsheriffRomDesc[] = {
	{ "Cosmic Sheriff 48K ES (1990)(Dinamic).tap", 43690, 0x3bee0827, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCsheriff, SpecCsheriff, Spectrum)
STD_ROM_FN(SpecCsheriff)

struct BurnDriver BurnSpecCsheriff = {
	"spec_csheriff", NULL, "spec_spectrum", NULL, "1990",
	"Cosmic Sheriff (Spanish) (48K)\0", "ENTER: quit game, come back to options screen", "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecCsheriffRomInfo, SpecCsheriffRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Di-Lithium Lift (16K)

static struct BurnRomInfo SpecDilithiumRomDesc[] = {
	{ "Di-Lithium Lift 16K (1983)(Hewson Consultants).tap", 8878, 0xc04e50e2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDilithium, SpecDilithium, Spectrum)
STD_ROM_FN(SpecDilithium)

struct BurnDriver BurnSpecDilithium = {
	"spec_dilithium", NULL, "spec_spectrum", NULL, "1983",
	"Di-Lithium Lift (16K)\0", NULL, "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecDilithiumRomInfo, SpecDilithiumRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Draughts Genius (48K)

static struct BurnRomInfo SpecDrgeniusRomDesc[] = {
	{ "Draughts Genius 48K (1987)(Rack-It).tap", 48605, 0x3d91e93f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDrgenius, SpecDrgenius, Spectrum)
STD_ROM_FN(SpecDrgenius)

struct BurnDriver BurnSpecDrgenius = {
	"spec_drgenius", NULL, "spec_spectrum", NULL, "1987",
	"Draughts Genius (48K)\0", NULL, "Rack-It - Hewson", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecDrgeniusRomInfo, SpecDrgeniusRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Gutz (48K)

static struct BurnRomInfo SpecGutzRomDesc[] = {
	{ "Gutz 48K (1988)(Ocean).tap", 43037, 0xad8a6a47, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGutz, SpecGutz, Spectrum)
STD_ROM_FN(SpecGutz)

struct BurnDriver BurnSpecGutz = {
	"spec_gutz", NULL, "spec_spectrum", NULL, "1988",
	"Gutz (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE | GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecGutzRomInfo, SpecGutzRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Babaliba (48K)

static struct BurnRomInfo SpecBabalibaRomDesc[] = {
	{ "Babaliba 48K (1984)(Dinamic Software).tap", 47064, 0xfe7fc680, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBabaliba, SpecBabaliba, Spectrum)
STD_ROM_FN(SpecBabaliba)

struct BurnDriver BurnSpecBabaliba = {
	"spec_babaliba", NULL, "spec_spectrum", NULL, "1984",
	"Babaliba (48K)\0", NULL, "Dinamic Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE, 0,
	SpectrumGetZipName, SpecBabalibaRomInfo, SpecBabalibaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Beverly Hills Cop (128K)

static struct BurnRomInfo SpecBhcopRomDesc[] = {
	{ "Beverly Hills Cop 128K (1990)(Tynesoft).tap", 117310, 0x2b8a9a2f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBhcop, SpecBhcop, Spec128)
STD_ROM_FN(SpecBhcop)

struct BurnDriver BurnSpecBhcop = {
	"spec_bhcop", NULL, "spec_spec128", NULL, "1990",
	"Beverly Hills Cop (128K)\0", NULL, "Tynesoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_SHOOT, 0,
	SpectrumGetZipName, SpecBhcopRomInfo, SpecBhcopRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Danger Mouse in Double Trouble (48K)

static struct BurnRomInfo SpecDangmouseRomDesc[] = {
	{ "Danger Mouse in Double Trouble 48K (1984)(Creative Sparks).z80", 44510, 0x691c3c67, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDangmouse, SpecDangmouse, Spectrum)
STD_ROM_FN(SpecDangmouse)

struct BurnDriver BurnSpecDangmouse = {
	"spec_dangmouse", NULL, "spec_spectrum", NULL, "1984",
	"Danger Mouse in Double Trouble (48K)\0", NULL, "Creative Sparks", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecDangmouseRomInfo, SpecDangmouseRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Duck Out! (48K)

static struct BurnRomInfo SpecDuckoutRomDesc[] = {
	{ "Duck Out! 48K (1989)(Dro Soft).tap", 108898, 0x2d8955dc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDuckout, SpecDuckout, Spectrum)
STD_ROM_FN(SpecDuckout)

struct BurnDriver BurnSpecDuckout = {
	"spec_duckout", NULL, "spec_spectrum", NULL, "1989",
	"Duck Out! (48K)\0", NULL, "Dro Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecDuckoutRomInfo, SpecDuckoutRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Fallen Angel (48K)

static struct BurnRomInfo SpecFallenangRomDesc[] = {
	{ "Fallen Angel 48K (1989)(Screen 7).tap", 82545, 0x0067c07c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFallenang, SpecFallenang, Spectrum)
STD_ROM_FN(SpecFallenang)

struct BurnDriver BurnSpecFallenang = {
	"spec_fallenang", NULL, "spec_spectrum", NULL, "1989",
	"Fallen Angel (48K)\0", "Select '2) Sinclair SJS' to use controller", "Screen 7", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecFallenangRomInfo, SpecFallenangRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Post Mortem (Spanish) (48K)

static struct BurnRomInfo SpecPostmortemRomDesc[] = {
	{ "Post Mortem 48K ES (1988)(Iber Soft).z80", 39261, 0xfbdeb9b7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPostmortem, SpecPostmortem, Spectrum)
STD_ROM_FN(SpecPostmortem)

struct BurnDriver BurnSpecPostmortem = {
	"spec_postmortem", NULL, "spec_spectrum", NULL, "1987",
	"Post Mortem (Spanish) (48K)\0", NULL, "Iber Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecPostmortemRomInfo, SpecPostmortemRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Pyracurse (48K)

static struct BurnRomInfo SpecPyracurseRomDesc[] = {
	{ "Pyracurse 48K (1986)(Hewson Consultants).tap", 47935, 0x0b498400, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPyracurse, SpecPyracurse, Spectrum)
STD_ROM_FN(SpecPyracurse)

struct BurnDriver BurnSpecPyracurse = {
	"spec_pyracurse", NULL, "spec_spectrum", NULL, "1986",
	"Pyracurse (48K)\0", NULL, "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_ADV, 0,
	SpectrumGetZipName, SpecPyracurseRomInfo, SpecPyracurseRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dynasty Wars (48K)

static struct BurnRomInfo SpecDynawarsRomDesc[] = {
	{ "Dynasty Wars 48K (1990)(U.S. Gold).tap", 100253, 0xae3317af, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDynawars, SpecDynawars, Spectrum)
STD_ROM_FN(SpecDynawars)

struct BurnDriver BurnSpecDynawars = {
	"spec_dynawars", NULL, "spec_spectrum", NULL, "1990",
	"Dynasty Wars (48K)\0", "Press '1' for one player mode or '2' for two player mode", "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecDynawarsRomInfo, SpecDynawarsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Earthlight (128K)

static struct BurnRomInfo SpecEarthlightRomDesc[] = {
	{ "Earthlight 128K (1988)(Firebird).tap", 61597, 0x063d61f4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecEarthlight, SpecEarthlight, Spec128)
STD_ROM_FN(SpecEarthlight)

struct BurnDriver BurnSpecEarthlight = {
	"spec_earthlight", NULL, "spec_spec128", NULL, "1988",
	"Earthlight (128K)\0", "Q: increase altitude - W: decrease altitude", "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecEarthlightRomInfo, SpecEarthlightRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Planet 10 (48K)

static struct BurnRomInfo SpecPlanet10RomDesc[] = {
	{ "Planet 10 48K (1989)(Mastertronic).tap", 48937, 0x4ae2d7a9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPlanet10, SpecPlanet10, Spectrum)
STD_ROM_FN(SpecPlanet10)

struct BurnDriver BurnSpecPlanet10 = {
	"spec_planet10", NULL, "spec_spectrum", NULL, "1989",
	"Planet 10 (48K)\0", NULL, "Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE, 0,
	SpectrumGetZipName, SpecPlanet10RomInfo, SpecPlanet10RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Postman Pat - The Computer Game (48K)

static struct BurnRomInfo SpecPostmanRomDesc[] = {
	{ "Postman Pat - The Computer Game 48K (1989)(Alternative).z80", 43141, 0xc991a405, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPostman, SpecPostman, Spectrum)
STD_ROM_FN(SpecPostman)

struct BurnDriver BurnSpecPostman = {
	"spec_postman", NULL, "spec_spectrum", NULL, "1989",
	"Postman Pat - The Computer Game (48K)\0", NULL, "Alternative", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecPostmanRomInfo, SpecPostmanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Werewolves of London (48K)

static struct BurnRomInfo SpecWerewolvesRomDesc[] = {
	{ "Werewolves of London 48K (1989)(Mastertronic).tap", 47170, 0x8b9b9683, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecWerewolves, SpecWerewolves, Spectrum)
STD_ROM_FN(SpecWerewolves)

struct BurnDriver BurnSpecWerewolves = {
	"spec_werewolves", NULL, "spec_spectrum", NULL, "1989",
	"Werewolves of London (48K)\0", "O/P: select icon - H: pause - R: restart", "Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_ADV, 0,
	SpectrumGetZipName, SpecWerewolvesRomInfo, SpecWerewolvesRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// A-Team, The - El Equipo A - Part 1 (Spanish) (128K)

static struct BurnRomInfo SpecAteamp1RomDesc[] = {
	{ "A-Team, the - El Equipo A ES 128K - Part1 (1988)(Zafiro Software).tap", 48242, 0x9acf83a6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAteamp1, SpecAteamp1, Spec128)
STD_ROM_FN(SpecAteamp1)

struct BurnDriver BurnSpecAteamp1 = {
	"spec_ateamp1", NULL, "spec_spec128", NULL, "1988",
	"A-Team, The - El Equipo A - Part 1 (Spanish) (128K)\0", "SPACE: smart bomb", "Zafiro Software Division", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecAteamp1RomInfo, SpecAteamp1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// A-Team, The - El Equipo A - Part 2 (Spanish) (128K)

static struct BurnRomInfo SpecAteamp2RomDesc[] = {
	{ "A-Team, the - El Equipo A ES 128K - Part2 (1988)(Zafiro Software).tap", 49137, 0x8e139fd1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAteamp2, SpecAteamp2, Spec128)
STD_ROM_FN(SpecAteamp2)

struct BurnDriver BurnSpecAteamp2 = {
	"spec_ateamp2", "spec_ateamp1", "spec_spec128", NULL, "1988",
	"A-Team, The - El Equipo A - Part 2 (Spanish) (128K)\0", "Password: WHJONDPEAQR", "Zafiro Software Division", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecAteamp2RomInfo, SpecAteamp2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Eskimo Eddie (48K)

static struct BurnRomInfo SpecEskeddieRomDesc[] = {
	{ "Eskimo Eddie 48K (1984)(Ocean).tap", 36318, 0x4b532294, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecEskeddie, SpecEskeddie, Spectrum)
STD_ROM_FN(SpecEskeddie)

struct BurnDriver BurnSpecEskeddie = {
	"spec_eskeddie", NULL, "spec_spectrum", NULL, "1984",
	"Eskimo Eddie (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecEskeddieRomInfo, SpecEskeddieRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// LA Drugs Bust (128K)

static struct BurnRomInfo SpecLadbustRomDesc[] = {
	{ "LA Drugs Bust 128K (1990)(Players Software).tap", 50858, 0x20fbc4bd, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecLadbust, SpecLadbust, Spec128)
STD_ROM_FN(SpecLadbust)

struct BurnDriver BurnSpecLadbust = {
	"spec_ladbust", NULL, "spec_spec128", NULL, "1990",
	"Los Angeles Drugs Bust (128K)\0", NULL, "Players Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecLadbustRomInfo, SpecLadbustRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Riding the Rapids (48K)

static struct BurnRomInfo SpecRidrapidsRomDesc[] = {
	{ "Riding the Rapids 48K (1987)(Players Software).tap", 47675, 0x940cabf6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecRidrapids, SpecRidrapids, Spectrum)
STD_ROM_FN(SpecRidrapids)

struct BurnDriver BurnSpecRidrapids = {
	"spec_ridrapids", NULL, "spec_spectrum", NULL, "1987",
	"Riding the Rapids (48K)\0", NULL, "Players Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecRidrapidsRomInfo, SpecRidrapidsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Shao Lin's Road (48K)

static struct BurnRomInfo SpecShaolinroadRomDesc[] = {
	{ "Shao Lin's Road 48K (1986)(The Edge).tap", 43831, 0x100a17ad, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecShaolinroad, SpecShaolinroad, Spectrum)
STD_ROM_FN(SpecShaolinroad)

struct BurnDriver BurnSpecShaolinroad = {
	"spec_shaolinroad", NULL, "spec_spectrum", NULL, "1986",
	"Shao Lin's Road (48K)\0", NULL, "The Edge", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecShaolinroadRomInfo, SpecShaolinroadRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dark Sceptre (48K)

static struct BurnRomInfo SpecDarksceptreRomDesc[] = {
	{ "Dark Sceptre 48K (1987)(Firebird).tap", 54587, 0x364d1c34, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDarksceptre, SpecDarksceptre, Spectrum)
STD_ROM_FN(SpecDarksceptre)

struct BurnDriver BurnSpecDarksceptre = {
	"spec_darksceptre", NULL, "spec_spectrum", NULL, "1987",
	"Dark Sceptre (48K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_STRATEGY, 0,
	SpectrumGetZipName, SpecDarksceptreRomInfo, SpecDarksceptreRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hobgoblin (48K)

static struct BurnRomInfo SpecHobgoblRomDesc[] = {
	{ "Hobgoblin 48K (1991)(Atlantis).tap", 31817, 0x244df5e4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHobgobl, SpecHobgobl, Spectrum)
STD_ROM_FN(SpecHobgobl)

struct BurnDriver BurnSpecHobgobl = {
	"spec_hobgobl", NULL, "spec_spectrum", NULL, "1991",
	"Hobgoblin (48K)\0", NULL, "Atlantis Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecHobgoblRomInfo, SpecHobgoblRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPMDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Hotshot (128K)

static struct BurnRomInfo SpecHotshotRomDesc[] = {
	{ "Hotshot 128K (1988)(Addictive Games).tap", 30707, 0x4f9b3846, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHotshot, SpecHotshot, Spec128)
STD_ROM_FN(SpecHotshot)

struct BurnDriver BurnSpecHotshot = {
	"spec_hotshot", NULL, "spec_spec128", NULL, "1988",
	"Hotshot (128K)\0", NULL, "Addictive Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION | GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecHotshotRomInfo, SpecHotshotRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Magnetron (48K)

static struct BurnRomInfo SpecMagnetronRomDesc[] = {
	{ "Magnetron 48K (1988)(Firebird).tap", 48383, 0xf407317e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMagnetron, SpecMagnetron, Spectrum)
STD_ROM_FN(SpecMagnetron)

struct BurnDriver BurnSpecMagnetron = {
	"spec_magnetron", NULL, "spec_spectrum", NULL, "1988",
	"Magnetron (48K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE, 0,
	SpectrumGetZipName, SpecMagnetronRomInfo, SpecMagnetronRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Nigel Mansell's World Championship (128K)

static struct BurnRomInfo SpecMansellwcRomDesc[] = {
	{ "Nigel Mansell's World Championship 128K (1993)(Gremlin Graphics).tap", 224111, 0x36ee68e4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMansellwc, SpecMansellwc, Spec128)
STD_ROM_FN(SpecMansellwc)

struct BurnDriver BurnSpecMansellwc = {
	"spec_mansellwc", NULL, "spec_spec128", NULL, "1993",
	"Nigel Mansell's World Championship (128K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecMansellwcRomInfo, SpecMansellwcRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Peter Pack Rat (128K)

static struct BurnRomInfo SpecPpratRomDesc[] = {
	{ "Peter Pack Rat 128K (1989)(Silverbird).tap", 53285, 0x5d318fcd, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPprat, SpecPprat, Spec128)
STD_ROM_FN(SpecPprat)

struct BurnDriver BurnSpecPprat = {
	"spec_pprat", NULL, "spec_spec128", NULL, "1989",
	"Peter Pack Rat (128K)\0", NULL, "Silverbird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecPpratRomInfo, SpecPpratRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rebel (48K)

static struct BurnRomInfo SpecRebelRomDesc[] = {
	{ "Rebel 48K (1987)(Virgin Games).tap", 47276, 0xf7833b9c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecRebel, SpecRebel, Spectrum)
STD_ROM_FN(SpecRebel)

struct BurnDriver BurnSpecRebel = {
	"spec_rebel", NULL, "spec_spectrum", NULL, "1987",
	"Rebel (48K)\0", NULL, "Virgin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecRebelRomInfo, SpecRebelRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sherlock (48K)

static struct BurnRomInfo SpecSherlockRomDesc[] = {
	{ "Sherlock 48K (1984)(Melbourne House).z80", 43891, 0x91d7e211, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSherlock, SpecSherlock, Spectrum)
STD_ROM_FN(SpecSherlock)

struct BurnDriver BurnSpecSherlock = {
	"spec_sherlock", NULL, "spec_spectrum", NULL, "1984",
	"Sherlock (48K)\0", NULL, "Melbourne House", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecSherlockRomInfo, SpecSherlockRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Spooked (48K)

static struct BurnRomInfo SpecSpookedRomDesc[] = {
	{ "Spooked 48K (1989)(Players Premier).tap", 40963, 0xb3b2d241, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSpooked, SpecSpooked, Spectrum)
STD_ROM_FN(SpecSpooked)

struct BurnDriver BurnSpecSpooked = {
	"spec_spooked", NULL, "spec_spectrum", NULL, "1989",
	"Spooked (48K)\0", NULL, "Players Premier", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecSpookedRomInfo, SpecSpookedRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Stardust (48K)

static struct BurnRomInfo SpecStardustRomDesc[] = {
    { "Stardust 48K (1987)(Topo Soft).tap", 73651, 0xe6a65d7b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecStardust, SpecStardust, Spectrum)
STD_ROM_FN(SpecStardust)

struct BurnDriver BurnSpecStardust = {
    "spec_stardust", NULL, "spec_spectrum", NULL, "1987",
    "Stardust (48K)\0", NULL, "Topo Soft", "ZX Spectrum",
    NULL, NULL, NULL, NULL,
    BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
    SpectrumGetZipName, SpecStardustRomInfo, SpecStardustRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPMDIPInfo,
    SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
    &SpecRecalc, 0x10, 288, 224, 4, 3
};

// Twin World (128K)

static struct BurnRomInfo SpecTwinworldRomDesc[] = {
	{ "Twin World 128K (1990)(Ubi Soft).tap", 357327, 0x1b14d5c6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTwinworld, SpecTwinworld, Spec128)
STD_ROM_FN(SpecTwinworld)

struct BurnDriver BurnSpecTwinworld = {
	"spec_twinworld", NULL, "spec_spec128", NULL, "1990",
	"Twin World (128K)\0", NULL, "Ubi Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecTwinworldRomInfo, SpecTwinworldRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Galivan - Cosmo Police (48K)

static struct BurnRomInfo SpecGalivanRomDesc[] = {
	{ "Galivan - Cosmo Police 48K (1986)(Ocean).tap", 49664, 0x6987e5d8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGalivan, SpecGalivan, Spectrum)
STD_ROM_FN(SpecGalivan)

struct BurnDriver BurnSpecGalivan = {
	"spec_galivan", NULL, "spec_spectrum", NULL, "1986",
	"Galivan - Cosmo Police (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecGalivanRomInfo, SpecGalivanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Luna Crabs (16K)

static struct BurnRomInfo SpecLunacrabsRomDesc[] = {
	{ "Luna Crabs 16K (1983)(Micromega).tap", 16574, 0xffd2fb0b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecLunacrabs, SpecLunacrabs, Spectrum)
STD_ROM_FN(SpecLunacrabs)

struct BurnDriver BurnSpecLunacrabs = {
	"spec_lunacrabs", NULL, "spec_spectrum", NULL, "1983",
	"Luna Crabs (16K)\0", "1: left - 0: right - 9: fire", "Micromega", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecLunacrabsRomInfo, SpecLunacrabsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Konami's Ping Pong (48K)

static struct BurnRomInfo SpecPingpongRomDesc[] = {
	{ "Konami's Ping Pong 48K (1986)(Imagine).tap", 39121, 0xd484a79a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPingpong, SpecPingpong, Spectrum)
STD_ROM_FN(SpecPingpong)

struct BurnDriver BurnSpecPingpong = {
	"spec_pingpong", NULL, "spec_spectrum", NULL, "1986",
	"Konami's Ping Pong (48K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SPORTSMISC, 0,
	SpectrumGetZipName, SpecPingpongRomInfo, SpecPingpongRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Psycho Soldier (48K)

static struct BurnRomInfo SpecPsychosoldRomDesc[] = {
	{ "Psycho Soldier 48K (1987)(Imagine).tap", 85352, 0x4c6cacaa, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPsychosold, SpecPsychosold, Spectrum)
STD_ROM_FN(SpecPsychosold)

struct BurnDriver BurnSpecPsychosold = {
	"spec_psychosold", NULL, "spec_spectrum", NULL, "1987",
	"Psycho Soldier (48K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecPsychosoldRomInfo, SpecPsychosoldRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Steel Eagle (128K)

static struct BurnRomInfo SpecSteeleagleRomDesc[] = {
	{ "Steel Eagle 128K (1990)(Players Premier).tap", 52632, 0x0d55bee6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSteeleagle, SpecSteeleagle, Spec128)
STD_ROM_FN(SpecSteeleagle)

struct BurnDriver BurnSpecSteeleagle = {
	"spec_steeleagle", NULL, "spec_spec128", NULL, "1990",
	"Steel Eagle (128K)\0", NULL, "Players Premier", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecSteeleagleRomInfo, SpecSteeleagleRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Stomp (48K)

static struct BurnRomInfo SpecStompRomDesc[] = {
	{ "Stomp 48K (1985)(DK'Tronics).tap", 9669, 0x52926196, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecStomp, SpecStomp, Spectrum)
STD_ROM_FN(SpecStomp)

struct BurnDriver BurnSpecStomp = {
	"spec_stomp", NULL, "spec_spectrum", NULL, "1985",
	"Stomp (48K)\0", NULL, "DK'Tronics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecStompRomInfo, SpecStompRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Stonkers (48K)

static struct BurnRomInfo SpecStonkersRomDesc[] = {
	{ "Stonkers 48K (1983)(Imagine).tap", 47132, 0x8e2bfa6f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecStonkers, SpecStonkers, Spectrum)
STD_ROM_FN(SpecStonkers)

struct BurnDriver BurnSpecStonkers = {
	"spec_stonkers", NULL, "spec_spectrum", NULL, "1983",
	"Stonkers (48K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_STRATEGY, 0,
	SpectrumGetZipName, SpecStonkersRomInfo, SpecStonkersRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Cavelon (48K)

static struct BurnRomInfo SpecCavelonRomDesc[] = {
	{ "Cavelon 48K (1984)(Ocean).tap", 33603, 0xdf2fa332, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCavelon, SpecCavelon, Spectrum)
STD_ROM_FN(SpecCavelon)

struct BurnDriver BurnSpecCavelon = {
	"spec_cavelon", NULL, "spec_spectrum", NULL, "1984",
	"Cavelon (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_MAZE, 0,
	SpectrumGetZipName, SpecCavelonRomInfo, SpecCavelonRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Chinese Juggler, The (48K)

static struct BurnRomInfo SpecChinjugglerRomDesc[] = {
	{ "Chinese Juggler, The 48K (1984)(Ocean).tap", 44449, 0xc5956b3f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecChinjuggler, SpecChinjuggler, Spectrum)
STD_ROM_FN(SpecChinjuggler)

struct BurnDriver BurnSpecChinjuggler = {
	"spec_chinjuggler", NULL, "spec_spectrum", NULL, "1984",
	"Chinese Juggler, The (48K)\0", "Press 'J' to use controller", "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecChinjugglerRomInfo, SpecChinjugglerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Gilligan's Gold (48K)

static struct BurnRomInfo SpecGillgoldRomDesc[] = {
	{ "Gilligan's Gold 48K (1984)(Ocean).tap", 45750, 0x8623c1a1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGillgold, SpecGillgold, Spectrum)
STD_ROM_FN(SpecGillgold)

struct BurnDriver BurnSpecGillgold = {
	"spec_gillgold", NULL, "spec_spectrum", NULL, "1984",
	"Gilligan's Gold (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecGillgoldRomInfo, SpecGillgoldRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Knight Rider (48K)

static struct BurnRomInfo SpecKnightriderRomDesc[] = {
	{ "Knight Rider 48K (1986)(Ocean).tap", 49152, 0xdf4e4c85, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecKnightrider, SpecKnightrider, Spectrum)
STD_ROM_FN(SpecKnightrider)

struct BurnDriver BurnSpecKnightrider = {
	"spec_knightrider", NULL, "spec_spectrum", NULL, "1986",
	"Knight Rider (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_RACING, 0,
	SpectrumGetZipName, SpecKnightriderRomInfo, SpecKnightriderRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Moon Alert! (48K)

static struct BurnRomInfo SpecMoonalertRomDesc[] = {
	{ "Moon Alert! 48K (1984)(Ocean).tap", 47090, 0x01a888bf, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMoonalert, SpecMoonalert, Spectrum)
STD_ROM_FN(SpecMoonalert)

struct BurnDriver BurnSpecMoonalert = {
	"spec_moonalert", NULL, "spec_spectrum", NULL, "1984",
	"Moon Alert! (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecMoonalertRomInfo, SpecMoonalertRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Phantom Club (48K)

static struct BurnRomInfo SpecPhantclubRomDesc[] = {
	{ "Phantom Club 48K (1988)(Ocean).tap", 51960, 0xda40ea75, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPhantclub, SpecPhantclub, Spectrum)
STD_ROM_FN(SpecPhantclub)

struct BurnDriver BurnSpecPhantclub = {
	"spec_phantclub", NULL, "spec_spectrum", NULL, "1988",
	"Phantom Club (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_ADV, 0,
	SpectrumGetZipName, SpecPhantclubRomInfo, SpecPhantclubRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Pogo (48K)

static struct BurnRomInfo SpecPogoRomDesc[] = {
	{ "Pogo 48K (1984)(Ocean).tap", 42355, 0x858e042e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPogo, SpecPogo, Spectrum)
STD_ROM_FN(SpecPogo)

struct BurnDriver BurnSpecPogo = {
	"spec_pogo", NULL, "spec_spectrum", NULL, "1984",
	"Pogo (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecPogoRomInfo, SpecPogoRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Red Heat (48K)

static struct BurnRomInfo SpecRedheatRomDesc[] = {
	{ "Red Heat 48K (1989)(Ocean).tap", 119068, 0x4d1c484b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecRedheat, SpecRedheat, Spectrum)
STD_ROM_FN(SpecRedheat)

struct BurnDriver BurnSpecRedheat = {
	"spec_redheat", NULL, "spec_spectrum", NULL, "1989",
	"Red Heat (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecRedheatRomInfo, SpecRedheatRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Return of the Jedi (128K)

static struct BurnRomInfo SpecRetofjediRomDesc[] = {
	{ "Return of the Jedi 128K (1989)(Domark).tap", 37222, 0xdccfc93b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecRetofjedi, SpecRetofjedi, Spec128)
STD_ROM_FN(SpecRetofjedi)

struct BurnDriver BurnSpecRetofjedi = {
	"spec_retofjedi", NULL, "spec_spec128", NULL, "1989",
	"Return of the Jedi (128K)\0", NULL, "Domark", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_ACTION, 0,
	SpectrumGetZipName, SpecRetofjediRomInfo, SpecRetofjediRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Street Hawk (48K)

static struct BurnRomInfo SpecStreethawkRomDesc[] = {
	{ "Street Hawk 48K (1986)(Ocean).tap", 47745, 0x37c623ad, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecStreethawk, SpecStreethawk, Spectrum)
STD_ROM_FN(SpecStreethawk)

struct BurnDriver BurnSpecStreethawk = {
	"spec_streethawk", NULL, "spec_spectrum", NULL, "1986",
	"Street Hawk (48K)\0", NULL, "Ocean", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecStreethawkRomInfo, SpecStreethawkRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bloody Paws - Part 1 (Spanish) (128K)

static struct BurnRomInfo SpecBloodpaws1RomDesc[] = {
	{ "Bloody Paws ES Part1 (1990)(G.LL.).tap", 35835, 0x65d44dab, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBloodpaws1, SpecBloodpaws1, Spec128)
STD_ROM_FN(SpecBloodpaws1)

struct BurnDriver BurnSpecBloodpaws1 = {
	"spec_bloodpaws1", NULL, "spec_spec128", NULL, "1990",
	"Bloody Paws - Part 1 (Spanish) (128K)\0", "Set 'Sinclair' to use controller", "G.LL. Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecBloodpaws1RomInfo, SpecBloodpaws1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bloody Paws - Part 2 (Spanish) (128K)

static struct BurnRomInfo SpecBloodpaws2RomDesc[] = {
	{ "Bloody Paws ES Part2 (1990)(G.LL.).tap", 37263, 0x97af0b3a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBloodpaws2, SpecBloodpaws2, Spec128)
STD_ROM_FN(SpecBloodpaws2)

struct BurnDriver BurnSpecBloodpaws2 = {
	"spec_bloodpaws2", "spec_bloodpaws1", "spec_spec128", NULL, "1990",
	"Bloody Paws - Part 2 (Spanish) (128K)\0", "Set 'Sinclair' to use controller", "G.LL. Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecBloodpaws2RomInfo, SpecBloodpaws2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Crazy House (48K)

static struct BurnRomInfo SpecCrazyhouseRomDesc[] = {
	{ "Crazy House (1990)(Bulldog).z80", 18580, 0x876275b0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCrazyhouse, SpecCrazyhouse, Spectrum)
STD_ROM_FN(SpecCrazyhouse)

struct BurnDriver BurnSpecCrazyhouse = {
	"spec_crazyhouse", NULL, "spec_spectrum", NULL, "1990",
	"Crazy House (48K)\0", "Set 'Cursor' to use controller", "Bulldog Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecCrazyhouseRomInfo, SpecCrazyhouseRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecCursorKeysDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sonic Boom (128K)

static struct BurnRomInfo SpecSonicboomRomDesc[] = {
	{ "Sonic Boom 128K (1990)(Activision).tap", 89195, 0x8fff430d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSonicboom, SpecSonicboom, Spec128)
STD_ROM_FN(SpecSonicboom)

struct BurnDriver BurnSpecSonicboom = {
	"spec_sonicboom", NULL, "spec_spec128", NULL, "1990",
	"Sonic Boom (128K) (HB)\0", NULL, "Activision", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecSonicboomRomInfo, SpecSonicboomRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Twister (48K)

static struct BurnRomInfo SpecTwisterRomDesc[] = {
	{ "Twister 48K (1986)(System 3).tap", 25904, 0x3a54f2c8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTwister, SpecTwister, Spectrum)
STD_ROM_FN(SpecTwister)

struct BurnDriver BurnSpecTwister = {
	"spec_twister", NULL, "spec_spectrum", NULL, "1986",
	"Twister (48K)\0", NULL, "System 3", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecTwisterRomInfo, SpecTwisterRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Turbo Driver (16K)

static struct BurnRomInfo SpecTurbodriverRomDesc[] = {
	{ "Turbo Driver (1983)(B.O.S. Software).tap", 7006, 0x59b6a725, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTurbodriver, SpecTurbodriver, Spectrum)
STD_ROM_FN(SpecTurbodriver)

struct BurnDriver BurnSpecTurbodriver = {
	"spec_turbodriver", NULL, "spec_spectrum", NULL, "1983",
	"Turbo Driver (16K)\0", NULL, "B.O.S. Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING, 0,
	SpectrumGetZipName, SpecTurbodriverRomInfo, SpecTurbodriverRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Atlantis (48K)

static struct BurnRomInfo SpecAtlantisRomDesc[] = {
	{ "Atlantis 48K (1985)(Anirog).tap", 27049, 0x460ac502, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAtlantis, SpecAtlantis, Spectrum)
STD_ROM_FN(SpecAtlantis)

struct BurnDriver BurnSpecAtlantis = {
	"spec_atlantis", NULL, "spec_spectrum", NULL, "1985",
	"Atlantis (48K)\0", "Press 'B' to use controller and 'S' to start game", "Anirog", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecAtlantisRomInfo, SpecAtlantisRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dogsbody to the Rescue (48K)

static struct BurnRomInfo SpecDogsbodyRomDesc[] = {
	{ "Dogsbody to the Rescue 48K (1985)(Bug-Byte).tap", 49460, 0xaad92250, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDogsbody, SpecDogsbody, Spectrum)
STD_ROM_FN(SpecDogsbody)

struct BurnDriver BurnSpecDogsbody = {
	"spec_dogsbody", NULL, "spec_spectrum", NULL, "1985",
	"Dogsbody to the Rescue (48K)\0", NULL, "Bug-Byte", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE, 0,
	SpectrumGetZipName, SpecDogsbodyRomInfo, SpecDogsbodyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Frenzy (16K)

static struct BurnRomInfo SpecFrenzyRomDesc[] = {
	{ "Frenzy 16K (1982)(Quicksilva).z80", 7770, 0x434c69e1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFrenzy, SpecFrenzy, Spectrum)
STD_ROM_FN(SpecFrenzy)

struct BurnDriver BurnSpecFrenzy = {
	"spec_frenzy", NULL, "spec_spectrum", NULL, "1982",
	"Frenzy (16K)\0", "To play properly, invert 'up' and 'down' inputs", "Quicksilva", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecFrenzyRomInfo, SpecFrenzyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Gothik (48K)

static struct BurnRomInfo SpecGothikRomDesc[] = {
	{ "Gothik 48K (1987)(Firebird).tap", 37854, 0x77774db3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGothik, SpecGothik, Spectrum)
STD_ROM_FN(SpecGothik)

struct BurnDriver BurnSpecGothik = {
	"spec_gothik", NULL, "spec_spectrum", NULL, "1987",
	"Gothik (48K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE | GBF_ACTION, 0,
	SpectrumGetZipName, SpecGothikRomInfo, SpecGothikRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Pac Hic (48K)

static struct BurnRomInfo SpecPachicRomDesc[] = {
	{ "Pac Hic (1986)(Daniel Isoba).tap", 45730, 0x42bb1258, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPachic, SpecPachic, Spectrum)
STD_ROM_FN(SpecPachic)

struct BurnDriver BurnSpecPachic = {
	"spec_pachic", NULL, "spec_spectrum", NULL, "1986",
	"Pac Hic (48K)\0", NULL, "Daniel Isoba", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE | GBF_ACTION, 0,
	SpectrumGetZipName, SpecPachicRomInfo, SpecPachicRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Prison Riot (128K)

static struct BurnRomInfo SpecPrisriotRomDesc[] = {
	{ "Prison Riot 128K (1990)(Players Premier).tap", 64518, 0xb760f8bd, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPrisriot, SpecPrisriot, Spec128)
STD_ROM_FN(SpecPrisriot)

struct BurnDriver BurnSpecPrisriot = {
	"spec_prisriot", NULL, "spec_spec128", NULL, "1990",
	"Prison Riot (128K)\0", NULL, "Players Premier", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE | GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecPrisriotRomInfo, SpecPrisriotRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Tir Na Nog (48K)

static struct BurnRomInfo SpecTirnanogRomDesc[] = {
	{ "Tir Na Nog 48K (1984)(Gargoyle Games).tap", 46565, 0x8f5492ed, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTirnanog, SpecTirnanog, Spectrum)
STD_ROM_FN(SpecTirnanog)

struct BurnDriver BurnSpecTirnanog = {
	"spec_tirnanog", NULL, "spec_spectrum", NULL, "1984",
	"Tir Na Nog (48K)\0", NULL, "Gargoyle Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecTirnanogRomInfo, SpecTirnanogRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Demonslair (48K)

static struct BurnRomInfo SpecDemonslairRomDesc[] = {
	{ "Demonslair 48K (1987)(Gadtek Games).tap", 42898, 0xcb570271, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDemonslair, SpecDemonslair, Spectrum)
STD_ROM_FN(SpecDemonslair)

struct BurnDriver BurnSpecDemonslair = {
	"spec_demonslair", NULL, "spec_spectrum", NULL, "1987",
	"Demonslair (48K)\0", "Set 'd: sinclair.II' to use controller", "Gadtek Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecDemonslairRomInfo, SpecDemonslairRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Exploding Fist+ (48K)

static struct BurnRomInfo SpecFistplusRomDesc[] = {
	{ "Exploding Fist+ 48K (1988)(Firebird).tap", 52905, 0xd096a4be, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFistplus, SpecFistplus, Spectrum)
STD_ROM_FN(SpecFistplus)

struct BurnDriver BurnSpecFistplus = {
	"spec_fistplus", NULL, "spec_spectrum", NULL, "1988",
	"Exploding Fist+ (48K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 3, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpecFistplusRomInfo, SpecFistplusRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Manollo - El Cavernicola (Spanish) (48K)

static struct BurnRomInfo SpecManolloRomDesc[] = {
	{ "Manollo - El Cavernicola ES 48K (1988)(Iber Software).tap", 39236, 0x594d3ab5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecManollo, SpecManollo, Spectrum)
STD_ROM_FN(SpecManollo)

struct BurnDriver BurnSpecManollo = {
	"spec_manollo", NULL, "spec_spectrum", NULL, "1988",
	"Manollo - El Cavernicola (Spanish) (48K)\0", NULL, "Iber Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecManolloRomInfo, SpecManolloRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// MASK II (128K)

static struct BurnRomInfo SpecMask2RomDesc[] = {
	{ "MASK II 128K (1987)(Gremlin Graphics).tap", 79774, 0x972b746c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMask2, SpecMask2, Spec128)
STD_ROM_FN(SpecMask2)

struct BurnDriver BurnSpecMask2 = {
	"spec_mask2", NULL, "spec_spec128", NULL, "1987",
	"MASK II (128K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecMask2RomInfo, SpecMask2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// MASK III - VENOM Strikes Back (128K)

static struct BurnRomInfo SpecMask3RomDesc[] = {
	{ "MASK III - VENOM Strikes Back 128K (1988)(Gremlin Graphics).tap", 51714, 0xafcf673d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMask3, SpecMask3, Spec128)
STD_ROM_FN(SpecMask3)

struct BurnDriver BurnSpecMask3 = {
	"spec_mask3", NULL, "spec_spec128", NULL, "1988",
	"MASK III - VENOM Strikes Back (128K)\0", NULL, "Gremlin Graphics", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecMask3RomInfo, SpecMask3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Punk Star (Spanish) (48K)

static struct BurnRomInfo SpecPunkstarRomDesc[] = {
	{ "Punk Star ES 48K (1988)(Genesis Soft).tap", 27136, 0xca188147, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPunkstar, SpecPunkstar, Spectrum)
STD_ROM_FN(SpecPunkstar)

struct BurnDriver BurnSpecPunkstar = {
	"spec_punkstar", NULL, "spec_spectrum", NULL, "1988",
	"Punk Star (Spanish) (48K)\0", NULL, "Genesis Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecPunkstarRomInfo, SpecPunkstarRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Turbo the Tortoise - 30th Anniversary Edition (128K)

static struct BurnRomInfo SpecTurbothetortoise30thRomDesc[] = {
	{ "Turbo The Tortoise 30th Anniversary Edition (2022)(BiNMaN).tap", 74768, 0x0207ae31, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTurbothetortoise30th, SpecTurbothetortoise30th, Spec128)
STD_ROM_FN(SpecTurbothetortoise30th)

struct BurnDriver BurnSpecTurbothetortoise30th = {
	"spec_turbothetortoise30th", "spec_turbothetortoise", "spec_spec128", NULL, "2022",
	"Turbo the Tortoise - 30th Anniversary Edition (128K)\0", NULL, "BiNMaN", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecTurbothetortoise30thRomInfo, SpecTurbothetortoise30thRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Buccaneer (48K)

static struct BurnRomInfo SpecBuccaneerRomDesc[] = {
	{ "Buccaneer 48K (1985)(Firebird).tap", 49414, 0x9fdacef2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBuccaneer, SpecBuccaneer, Spectrum)
STD_ROM_FN(SpecBuccaneer)

struct BurnDriver BurnSpecBuccaneer = {
	"spec_buccaneer", NULL, "spec_spectrum", NULL, "1985",
	"Buccaneer (48K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecBuccaneerRomInfo, SpecBuccaneerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Doctor Doom's Revenge (48K)

static struct BurnRomInfo SpecDoomrevengeRomDesc[] = {
	{ "Doctor Doom's Revenge 48K (1989)(Empire).tap", 197516, 0x5f1b6cfc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDoomrevenge, SpecDoomrevenge, Spectrum)
STD_ROM_FN(SpecDoomrevenge)

struct BurnDriver BurnSpecDoomrevenge = {
	"spec_doomrevenge", NULL, "spec_spectrum", NULL, "1989",
	"Doctor Doom's Revenge (48K)\0", "The Amazing Spider-Man and Captain America in Dr. Doom's Revenge!","Empire", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_VSFIGHT, 0,
	SpectrumGetZipName, SpecDoomrevengeRomInfo, SpecDoomrevengeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Spirits (Spanish) (48K)

static struct BurnRomInfo SpecSpiritsRomDesc[] = {
	{ "Spirits ES 48K (1987)(Topo Soft).tap", 49605, 0x901fbdaa, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSpirits, SpecSpirits, Spectrum)
STD_ROM_FN(SpecSpirits)

struct BurnDriver BurnSpecSpirits = {
	"spec_spirits", NULL, "spec_spectrum", NULL, "1987",
	"Spirits (Spanish) (48K)\0", NULL, "Topo Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecSpiritsRomInfo, SpecSpiritsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPMDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Spirits - 30th anniversary (Spanish) (48K)

static struct BurnRomInfo SpecSpirits30thRomDesc[] = {
	{ "Spirits ES 30th anniversary 48K (1987-2017)(Topo Soft).tap", 57855, 0x521f32f4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSpirits30th, SpecSpirits30th, Spectrum)
STD_ROM_FN(SpecSpirits30th)

struct BurnDriver BurnSpecSpirits30th = {
	"spec_spirits30th", "spec_spirits", "spec_spectrum", NULL, "2017",
	"Spirits - 30th Anniversary (Spanish) (48K)\0", NULL, "Topo Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecSpirits30thRomInfo, SpecSpirits30thRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPMDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Wells & Fargo (Spanish) (48K)

static struct BurnRomInfo SpecWellsfargoRomDesc[] = {
	{ "Wells and Fargo ES 48K (1988)(Topo Soft).tap", 64570, 0x74b64095, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecWellsfargo, SpecWellsfargo, Spectrum)
STD_ROM_FN(SpecWellsfargo)

struct BurnDriver BurnSpecWellsfargo = {
	"spec_wellsfargo", NULL, "spec_spectrum", NULL, "1988",
	"Wells & Fargo (Spanish) (48K)\0", NULL, "Topo Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecWellsfargoRomInfo, SpecWellsfargoRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Downtown (48K)

static struct BurnRomInfo SpecDowntownRomDesc[] = {
	{ "Downtown 48K (1991)(Atlantis Software).tap", 48188, 0xcfbbf2fb, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDowntown, SpecDowntown, Spectrum)
STD_ROM_FN(SpecDowntown)

struct BurnDriver BurnSpecDowntown = {
	"spec_downtown", NULL, "spec_spectrum", NULL, "1991",
	"Downtown (48K)\0", NULL, "Atlantis Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecDowntownRomInfo, SpecDowntownRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Earth Shaker (48K)

static struct BurnRomInfo SpecEarthshakerRomDesc[] = {
	{ "Earth Shaker 48K (1990)(Your Sinclair).tap", 37729, 0x0c8c1e65, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecEarthshaker, SpecEarthshaker, Spectrum)
STD_ROM_FN(SpecEarthshaker)

struct BurnDriver BurnSpecEarthshaker = {
	"spec_earthshaker", NULL, "spec_spectrum", NULL, "1990",
	"Earth Shaker (48K)\0", NULL, "Your Sinclair", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecEarthshakerRomInfo, SpecEarthshakerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Friday the 13th (48K)

static struct BurnRomInfo SpecFriday13thRomDesc[] = {
	{ "Friday the 13th 48K (1986)(Domark).tap", 44032, 0xc8fd8929, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFriday13th, SpecFriday13th, Spectrum)
STD_ROM_FN(SpecFriday13th)

struct BurnDriver BurnSpecFriday13th = {
	"spec_friday13th", NULL, "spec_spectrum", NULL, "1986",
	"Friday the 13th (48K)\0", NULL, "Domark", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_ADV, 0,
	SpectrumGetZipName, SpecFriday13thRomInfo, SpecFriday13thRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Future Bike Simulator (128K)

static struct BurnRomInfo SpecFutbikesimRomDesc[] = {
	{ "Future Bike Simulator (1990)(Hi-Tec).tap", 95192, 0x3ced8a2d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFutbikesim, SpecFutbikesim, Spec128)
STD_ROM_FN(SpecFutbikesim)

struct BurnDriver BurnSpecFutbikesim = {
	"spec_futbikesim", NULL, "spec_spec128", NULL, "1990",
	"Future Bike Simulator (128K)\0", NULL, "Hi-Tec Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_RACING | GBF_SHOOT, 0,
	SpectrumGetZipName, SpecFutbikesimRomInfo, SpecFutbikesimRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Monty Python's Flying Circus (128K)

static struct BurnRomInfo SpecMpflycircusRomDesc[] = {
	{ "Monty Python's Flying Circus 128K (1990)(Virgin Games).tap", 136802, 0x3a99fb2d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMpflycircus, SpecMpflycircus, Spec128)
STD_ROM_FN(SpecMpflycircus)

struct BurnDriver BurnSpecMpflycircus = {
	"spec_mpflycircus", NULL, "spec_spec128", NULL, "1990",
	"Monty Python's Flying Circus (128K)\0", NULL, "Virgin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_ADV, 0,
	SpectrumGetZipName, SpecMpflycircusRomInfo, SpecMpflycircusRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mystery of Arkham Manor, The - Part 1 (48K)

static struct BurnRomInfo SpecMysteryarkham1RomDesc[] = {
	{ "Mystery of Arkham Manor, The - Part1 48K (1987)(Melbourne House).tap", 56277, 0x6bf7a6ea, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMysteryarkham1, SpecMysteryarkham1, Spectrum)
STD_ROM_FN(SpecMysteryarkham1)

struct BurnDriver BurnSpecMysteryarkham1 = {
	"spec_mysteryarkham1", NULL, "spec_spectrum", NULL, "1987",
	"Mystery of Arkham Manor, The - Part 1 (48K)\0", "In-game menu - 6: down - 7: up - Space: select", "Melbourne House", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecMysteryarkham1RomInfo, SpecMysteryarkham1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mystery of Arkham Manor, The - Part 2 (48K)

static struct BurnRomInfo SpecMysteryarkham2RomDesc[] = {
	{ "Mystery of Arkham Manor, The - Part2 48K (1987)(Melbourne House).tap", 49343, 0x85b9867c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMysteryarkham2, SpecMysteryarkham2, Spectrum)
STD_ROM_FN(SpecMysteryarkham2)

struct BurnDriver BurnSpecMysteryarkham2 = {
	"spec_mysteryarkham2", "spec_mysteryarkham1", "spec_spectrum", NULL, "1987",
	"Mystery of Arkham Manor, The - Part 2 (48K)\0", "In-game menu - 6: down - 7: up - Space: select", "Melbourne House", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecMysteryarkham2RomInfo, SpecMysteryarkham2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Senda Salvaje - Parte 1 (Spanish) (48K)

static struct BurnRomInfo SpecSendasalvaje1RomDesc[] = {
	{ "Senda Salvaje - Parte 1 ES 48K (1990)(Zigurat).tap", 48033, 0x62afcb2d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSendasalvaje1, SpecSendasalvaje1, Spectrum)
STD_ROM_FN(SpecSendasalvaje1)

struct BurnDriver BurnSpecSendasalvaje1 = {
	"spec_sendasalvaje1", NULL, "spec_spectrum", NULL, "1990",
	"Senda Salvaje - Parte 1 (Spanish) (48K)\0", NULL, "Zigurat Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecSendasalvaje1RomInfo, SpecSendasalvaje1RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Senda Salvaje - Parte 2 (Spanish) (48K)

static struct BurnRomInfo SpecSendasalvaje2RomDesc[] = {
	{ "Senda Salvaje - Parte 2 ES 48K (1990)(Zigurat).tap", 47701, 0xaccef9d1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSendasalvaje2, SpecSendasalvaje2, Spectrum)
STD_ROM_FN(SpecSendasalvaje2)

struct BurnDriver BurnSpecSendasalvaje2 = {
	"spec_sendasalvaje2", "spec_sendasalvaje1", "spec_spectrum", NULL, "1990",
	"Senda Salvaje - Parte 2 (Spanish) (48K)\0", "Password: PLATANON", "Zigurat Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecSendasalvaje2RomInfo, SpecSendasalvaje2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Spooky Castle (48K)

static struct BurnRomInfo SpecSpookycastleRomDesc[] = {
	{ "Spooky Castle 48K (1990)(Atlantis Software).z80", 38671, 0xfad004e5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSpookycastle, SpecSpookycastle, Spectrum)
STD_ROM_FN(SpecSpookycastle)

struct BurnDriver BurnSpecSpookycastle = {
	"spec_spookycastle", NULL, "spec_spectrum", NULL, "1990",
	"Spooky Castle (48K)\0", NULL, "Atlantis Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecSpookycastleRomInfo, SpecSpookycastleRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Underground (Spanish) (48K)

static struct BurnRomInfo SpecUndergroundRomDesc[] = {
	{ "Underground ES 48K (1988)(System 4).tap", 49575, 0x0470e56c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecUnderground, SpecUnderground, Spectrum)
STD_ROM_FN(SpecUnderground)

struct BurnDriver BurnSpecUnderground = {
	"spec_underground", NULL, "spec_spectrum", NULL, "1988",
	"Underground (Spanish) (48K)\0", NULL, "System 4", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecUndergroundRomInfo, SpecUndergroundRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bumpy (48K)

static struct BurnRomInfo SpecBumpyRomDesc[] = {
	{ "Bumpy (1989)(Loriciels).tap", 45878, 0xc79bed54, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBumpy, SpecBumpy, Spectrum)
STD_ROM_FN(SpecBumpy)

struct BurnDriver BurnSpecBumpy = {
	"spec_bumpy", NULL, "spec_spectrum", NULL, "1989",
	"Bumpy (48K)\0", NULL, "Loriciels", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecBumpyRomInfo, SpecBumpyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Eric And The Floaters (48K)

static struct BurnRomInfo SpecEricfloatersRomDesc[] = {
	{ "Eric And The Floaters (1983)(Hudson Soft).tap", 8492, 0x0f7826de, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecEricfloaters, SpecEricfloaters, Spectrum)
STD_ROM_FN(SpecEricfloaters)

struct BurnDriver BurnSpecEricfloaters = {
	"spec_ericfloaters", NULL, "spec_spectrum", NULL, "1983",
	"Eric And The Floaters (48K)\0", NULL, "Hudson Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE | GBF_ACTION, 0,
	SpectrumGetZipName, SpecEricfloatersRomInfo, SpecEricfloatersRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Glug Glug (48K)

static struct BurnRomInfo SpecGlugglugRomDesc[] = {
	{ "Glug Glug (1984)(CRL Ltd.).TAP", 18843, 0x0a096413, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGlugglug, SpecGlugglug, Spectrum)
STD_ROM_FN(SpecGlugglug)

struct BurnDriver BurnSpecGlugglug = {
	"spec_glugglug", NULL, "spec_spectrum", NULL, "1984",
	"Glug Glug (48K)\0", NULL, "CRL Ltd.", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecGlugglugRomInfo, SpecGlugglugRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Jumping Jack (48K)

static struct BurnRomInfo SpecJjackRomDesc[] = {
	{ "Jumping Jack (1983)(Imagine Software).tap", 11833, 0xc54ae2a8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecJjack, SpecJjack, Spectrum)
STD_ROM_FN(SpecJjack)

struct BurnDriver BurnSpecJjack = {
	"spec_jjack", NULL, "spec_spectrum", NULL, "1983",
	"Jumping Jack (48K)\0", NULL, "Imagine Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecJjackRomInfo, SpecJjackRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecJJackDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Chaos - The Battle of Wizards (48K)

static struct BurnRomInfo SpecChaosRomDesc[] = {
	{ "Chaos - The Battle of Wizards 48K (1985)(Games Workshop).tap", 41640, 0xa696e01a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecChaos, SpecChaos, Spectrum)
STD_ROM_FN(SpecChaos)

struct BurnDriver BurnSpecChaos = {
	"spec_chaos", NULL, "spec_spectrum", NULL, "1985",
	"Chaos - The Battle of Wizards (48K)\0", "Q, W, E, A, D, Z, X, C, to move cursor", "Games Workshop", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 8, HARDWARE_SPECTRUM, GBF_STRATEGY, 0,
	SpectrumGetZipName, SpecChaosRomInfo, SpecChaosRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Cyberun (48K)

static struct BurnRomInfo SpecCyberunRomDesc[] = {
	{ "Cyberun 48K (1986)(Ultimate Play The Game).tap", 47506, 0x8ffbbf90, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCyberun, SpecCyberun, Spectrum)
STD_ROM_FN(SpecCyberun)

struct BurnDriver BurnSpecCyberun = {
	"spec_cyberun", NULL, "spec_spectrum", NULL, "1986",
	"Cyberun (48K)\0", "Press '1' or '2' to set controls", "Ultimate Play The Game", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecCyberunRomInfo, SpecCyberunRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Martianoids (48K)

static struct BurnRomInfo SpecMartianoidsRomDesc[] = {
	{ "Martianoids 48K (1987)(Ultimate Play The Game).tap", 49438, 0xa1fe1d86, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMartianoids, SpecMartianoids, Spectrum)
STD_ROM_FN(SpecMartianoids)

struct BurnDriver BurnSpecMartianoids = {
	"spec_martianoids", NULL, "spec_spectrum", NULL, "1987",
	"Martianoids (48K)\0", NULL, "Ultimate Play The Game", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE, 0,
	SpectrumGetZipName, SpecMartianoidsRomInfo, SpecMartianoidsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Nightshade (48K)

static struct BurnRomInfo SpecNightshadeRomDesc[] = {
	{ "Nightshade 48K (1985)(Ultimate Play The Game).tap", 42456, 0xb421d1a8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecNightshade, SpecNightshade, Spectrum)
STD_ROM_FN(SpecNightshade)

struct BurnDriver BurnSpecNightshade = {
	"spec_nightshade", NULL, "spec_spectrum", NULL, "1985",
	"Nightshade (48K)\0", NULL, "Ultimate Play The Game", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_ADV, 0,
	SpectrumGetZipName, SpecNightshadeRomInfo, SpecNightshadeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Silent Service (48K)

static struct BurnRomInfo SpecSilentserviceRomDesc[] = {
	{ "Silent Service 48K (1986)(U.S. Gold).z80", 40554, 0x1822026f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSilentservice, SpecSilentservice, Spectrum)
STD_ROM_FN(SpecSilentservice)

struct BurnDriver BurnSpecSilentservice = {
	"spec_silentservice", NULL, "spec_spectrum", NULL, "1986",
	"Silent Service (48K)\0", NULL, "U.S. Gold", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SIM, 0,
	SpectrumGetZipName, SpecSilentserviceRomInfo, SpecSilentserviceRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Tau-Ceti (48K)

static struct BurnRomInfo SpecTaucetiRomDesc[] = {
	{ "Tau-Ceti 48K (1985)(CRL Group).tap", 56807, 0x309522ae, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTauceti, SpecTauceti, Spectrum)
STD_ROM_FN(SpecTauceti)

struct BurnDriver BurnSpecTauceti = {
	"spec_tauceti", "spec_taucetise", "spec_spectrum", NULL, "1985",
	"Tau-Ceti (48K)\0", "'SHIFT+SPACE' to begin, write 'HELP' to list commands", "CRL Group", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_SIM, 0,
	SpectrumGetZipName, SpecTaucetiRomInfo, SpecTaucetiRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Tau Ceti - The Special Edition (128K)

static struct BurnRomInfo SpecTaucetiseRomDesc[] = {
	{ "Tau Ceti - The Special Edition 128K (1987)(CRL Group).tap", 129527, 0x97369bcf, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTaucetise, SpecTaucetise, Spec128)
STD_ROM_FN(SpecTaucetise)

struct BurnDriver BurnSpecTaucetise = {
	"spec_taucetise", NULL, "spec_spec128", NULL, "1987",
	"Tau Ceti - The Special Edition (128K)\0", "'SHIFT+SPACE' to begin, write 'HELP' to list commands", "CRL Group", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT | GBF_SIM, 0,
	SpectrumGetZipName, SpecTaucetiseRomInfo, SpecTaucetiseRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Vectron (48K)

static struct BurnRomInfo SpecVectronRomDesc[] = {
	{ "Vectron 48K (1985)(Firebird).tap", 49343, 0xec6217eb, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecVectron, SpecVectron, Spectrum)
STD_ROM_FN(SpecVectron)

struct BurnDriver BurnSpecVectron = {
	"spec_vectron", NULL, "spec_spectrum", NULL, "1985",
	"Vectron (48K)\0", NULL, "Firebird Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecVectronRomInfo, SpecVectronRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Cannibal Island (48K)

static struct BurnRomInfo SpecCannislandRomDesc[] = {
	{ "Cannibal Island 48K (1986)(LiveWire Software).tap", 48016, 0x0a6738c4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCannisland, SpecCannisland, Spectrum)
STD_ROM_FN(SpecCannisland)

struct BurnDriver BurnSpecCannisland = {
	"spec_cannisland", NULL, "spec_spectrum", NULL, "1986",
	"Cannibal Island (48K)\0", NULL, "LiveWire Software", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_ADV, 0,
	SpectrumGetZipName, SpecCannislandRomInfo, SpecCannislandRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// City Connection (128K)

static struct BurnRomInfo SpecCityconn128RomDesc[] = {
	{ "City Connection 128K (1988-2023)(Manuel Lemos).tap", 112933, 0xba5fd65a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCityconn128, SpecCityconn128, Spec128)
STD_ROM_FN(SpecCityconn128)

struct BurnDriver BurnSpecCityconn128 = {
	"spec_cityconn128", NULL, "spec_spec128", NULL, "2023",
	"City Connection (128K)\0", NULL, "Manuel Lemos", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecCityconn128RomInfo, SpecCityconn128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// City Connection (48K)

static struct BurnRomInfo SpecCityconn48RomDesc[] = {
	{ "City Connection 48K (1988)(Manuel Lemos).z80", 41555, 0xcbd6d656, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCityconn48, SpecCityconn48, Spectrum)
STD_ROM_FN(SpecCityconn48)

struct BurnDriver BurnSpecCityconn48 = {
	"spec_cityconn48", "spec_cityconn128", "spec_spectrum", NULL, "1988",
	"City Connection (48K)\0", NULL, "Manuel Lemos", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecCityconn48RomInfo, SpecCityconn48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Shuriken (48K)

static struct BurnRomInfo SpecShurikenRomDesc[] = {
	{ "Shuriken 48K (1989)(Xplosive).tap", 48301, 0xcee18b6e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecShuriken, SpecShuriken, Spectrum)
STD_ROM_FN(SpecShuriken)

struct BurnDriver BurnSpecShuriken = {
	"spec_shuriken", NULL, "spec_spectrum", NULL, "1989",
	"Shuriken (48K)\0", "Set '3 - Sinclair' to use contoller, press 'SPACE' for missiles", "Xplosive", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecShurikenRomInfo, SpecShurikenRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2SwapDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Guardian, The (48K)

static struct BurnRomInfo SpecGuardianRomDesc[] = {
	{ "Guardian, The 48K (1983)(PSS).tap", 19190, 0xd7fec385, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGuardian, SpecGuardian, Spectrum)
STD_ROM_FN(SpecGuardian)

struct BurnDriver BurnSpecGuardian = {
	"spec_guardian", NULL, "spec_spectrum", NULL, "1983",
	"Guardian, The (48K)\0", NULL, "PSS", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecGuardianRomInfo, SpecGuardianRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Swords & Sorcery (48K)

static struct BurnRomInfo SpecSwordsnsorceryRomDesc[] = {
	{ "Swords & Sorcery 48K (1985)(PSS).tap", 48456, 0xee9f4073, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSwordsnsorcery, SpecSwordsnsorcery, Spectrum)
STD_ROM_FN(SpecSwordsnsorcery)

struct BurnDriver BurnSpecSwordsnsorcery = {
	"spec_swordsnsorcery", NULL, "spec_spectrum", NULL, "1985",
	"Swords & Sorcery (48K)\0", "Press '1-2-3' to move character and '8-9-0' to use menu", "PSS", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_MAZE | GBF_RPG, 0,
	SpectrumGetZipName, SpecSwordsnsorceryRomInfo, SpecSwordsnsorceryRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Xavior (48K)

static struct BurnRomInfo SpecXaviorRomDesc[] = {
	{ "Xavior 48K (1984)(PSS).tap", 50074, 0x2c36bcdb, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecXavior, SpecXavior, Spectrum)
STD_ROM_FN(SpecXavior)

struct BurnDriver BurnSpecXavior = {
	"spec_xavior", NULL, "spec_spectrum", NULL, "1984",
	"Xavior (48K)\0", NULL, "PSS", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecXaviorRomInfo, SpecXaviorRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Action Biker (48K)

static struct BurnRomInfo SpecActionbikerRomDesc[] = {
	{ "Action Biker 48K (1985)(Mastertronic).tap", 63742, 0x1a62419a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecActionbiker, SpecActionbiker, Spectrum)
STD_ROM_FN(SpecActionbiker)

struct BurnDriver BurnSpecActionbiker = {
	"spec_actionbiker", NULL, "spec_spectrum", NULL, "1985",
	"Action Biker (48K)\0", NULL, "Mastertronic", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecActionbikerRomInfo, SpecActionbikerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ant Attack (48K)

static struct BurnRomInfo SpecAntattackRomDesc[] = {
	{ "Ant Attack 48K (1983)(Quicksilva).tap", 41020, 0x91cbb1dd, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecAntattack, SpecAntattack, Spectrum)
STD_ROM_FN(SpecAntattack)

struct BurnDriver BurnSpecAntattack = {
	"spec_antattack", NULL, "spec_spectrum", NULL, "1983",
	"Ant Attack (48K)\0", NULL, "Quicksilva", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_MAZE, 0,
	SpectrumGetZipName, SpecAntattackRomInfo, SpecAntattackRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Exploding Wall (48K)

static struct BurnRomInfo SpecExplodwallRomDesc[] = {
	{ "Exploding Wall 48K (1989)(Byte Back).tap", 37401, 0xfe708ad6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecExplodwall, SpecExplodwall, Spectrum)
STD_ROM_FN(SpecExplodwall)

struct BurnDriver BurnSpecExplodwall = {
	"spec_explodwall", NULL, "spec_spectrum", NULL, "1989",
	"Exploding Wall (48K)\0", NULL, "Byte Back", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_BREAKOUT, 0,
	SpectrumGetZipName, SpecExplodwallRomInfo, SpecExplodwallRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// King Valley (128K)

static struct BurnRomInfo SpecKingvalleyRomDesc[] = {
	{ "King Valley 128K (1994)(WeByBelarus).tap", 39820, 0x1d1ebe53, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecKingvalley, SpecKingvalley, Spec128)
STD_ROM_FN(SpecKingvalley)

struct BurnDriver BurnSpecKingvalley = {
	"spec_kingvalley", NULL, "spec_spec128", NULL, "1994",
	"King Valley (128K)\0", NULL, "WE [BY] (Belarus)", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecKingvalleyRomInfo, SpecKingvalleyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Prince of Persia (Russian) (128K)

static struct BurnRomInfo SpecPrincepersiaRomDesc[] = {
	{ "Prince of Persia RU 128K (1996)(Magic Soft).tap", 630191, 0x07200508, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPrincepersia, SpecPrincepersia, Spec128)
STD_ROM_FN(SpecPrincepersia)

struct BurnDriver BurnSpecPrincepersia = {
	"spec_princepersia", NULL, "spec_spec128", NULL, "1996",
	"Prince of Persia (Russian) (128K)\0", NULL, "Magic Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecPrincepersiaRomInfo, SpecPrincepersiaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sir Lancelot (48K)

static struct BurnRomInfo SpecSirlancelotRomDesc[] = {
	{ "Sir Lancelot 48K (1984)(Melbourne House).tap", 16929, 0xd3b14ed5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSirlancelot, SpecSirlancelot, Spectrum)
STD_ROM_FN(SpecSirlancelot)

struct BurnDriver BurnSpecSirlancelot = {
	"spec_sirlancelot", NULL, "spec_spectrum", NULL, "1984",
	"Sir Lancelot (48K)\0", NULL, "Melbourne House", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecSirlancelotRomInfo, SpecSirlancelotRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sorcery Plus (48K)

static struct BurnRomInfo SpecSorceryplusRomDesc[] = {
	{ "Sorcery Plus 48K (1986)(Virgin Games Ltd.).tap", 45930, 0xfc628bd2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecSorceryplus, SpecSorceryplus, Spectrum)
STD_ROM_FN(SpecSorceryplus)

struct BurnDriver BurnSpecSorceryplus = {
	"spec_sorceryplus", NULL, "spec_spectrum", NULL, "1986",
	"Sorcery Plus (48K)\0", "Q: left - A: right - 0: jump", "Virgin Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecSorceryplusRomInfo, SpecSorceryplusRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Terramex (128K)

static struct BurnRomInfo SpecTerramexRomDesc[] = {
	{ "Terramex 128K (1987)(Grandslam Entertainments).tap", 82251, 0xeaf717b6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTerramex, SpecTerramex, Spec128)
STD_ROM_FN(SpecTerramex)

struct BurnDriver BurnSpecTerramex = {
	"spec_terramex", NULL, "spec_spec128", NULL, "1987",
	"Terramex (128K)\0", NULL, "Grandslam Entertainments", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpecTerramexRomInfo, SpecTerramexRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// 3D Seiddab Attack (16K)

static struct BurnRomInfo Spec3dseiatkRomDesc[] = {
	{ "3D Seiddab Attack 16K (1984)(Hewson Consultants).tap", 8581, 0x22e97ea1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spec3dseiatk, Spec3dseiatk, Spectrum)
STD_ROM_FN(Spec3dseiatk)

struct BurnDriver BurnSpec3dseiatk = {
	"spec_3dseiatk", NULL, "spec_spectrum", NULL, "1984",
	"3D Seiddab Attack (16K)\0", NULL, "Hewson Consultants", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, Spec3dseiatkRomInfo, Spec3dseiatkRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mach 3 (Spanish) (48K)

static struct BurnRomInfo SpecMach3RomDesc[] = {
	{ "Mach 3 ES 48K (1989)(Loricels).tap", 44944, 0xa7f9fe1b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMach3, SpecMach3, Spectrum)
STD_ROM_FN(SpecMach3)

struct BurnDriver BurnSpecMach3 = {
	"spec_mach3", NULL, "spec_spectrum", NULL, "1989",
	"Mach 3 (Spanish) (48K)\0", NULL, "Loricels - Proein Soft Line", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecMach3RomInfo, SpecMach3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Puzzled! (48K)

static struct BurnRomInfo SpecPuzzledRomDesc[] = {
	{ "Puzzled! 48K (1989)(Guild, The).tap", 47979, 0x5a86daa5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecPuzzled, SpecPuzzled, Spectrum)
STD_ROM_FN(SpecPuzzled)

struct BurnDriver BurnSpecPuzzled = {
	"spec_puzzled", NULL, "spec_spectrum", NULL, "1989",
	"Puzzled! (48K)\0", NULL, "Guild, The", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecPuzzledRomInfo, SpecPuzzledRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rocky Horror Show, The (128K)

static struct BurnRomInfo SpecRhs128RomDesc[] = {
	{ "Rocky Horror Show, The 128K (1985)(CRL Group).tap", 81739, 0x011957b4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecRhs128, SpecRhs128, Spec128)
STD_ROM_FN(SpecRhs128)

struct BurnDriver BurnSpecRhs128 = {
	"spec_rhs128", NULL, "spec_spec128", NULL, "1985",
	"Rocky Horror Show, The (128K)\0", "Press 'Space' to select and 'Enter' to set options", "CRL Group", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecRhs128RomInfo, SpecRhs128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rocky Horror Show, The (48K)

static struct BurnRomInfo SpecRhs48RomDesc[] = {
	{ "Rocky Horror Show, The 48K (1985)(CRL Group).tap", 49355, 0x12082319, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecRhs48, SpecRhs48, Spectrum)
STD_ROM_FN(SpecRhs48)

struct BurnDriver BurnSpecRhs48 = {
	"spec_rhs48", "spec_rhs128", "spec_spectrum", NULL, "1985",
	"Rocky Horror Show, The (48K)\0", "Press 'Space' to select and 'Enter' to set options", "CRL Group", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_SPECTRUM, GBF_ADV, 0,
	SpectrumGetZipName, SpecRhs48RomInfo, SpecRhs48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Zip-Zap (48K)

static struct BurnRomInfo SpecZipzapRomDesc[] = {
	{ "Zip-Zap 48K (1983)(Imagine).tap", 37207, 0xfcb30933, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecZipzap, SpecZipzap, Spectrum)
STD_ROM_FN(SpecZipzap)

struct BurnDriver BurnSpecZipzap = {
	"spec_zipzap", NULL, "spec_spectrum", NULL, "1983",
	"Zip-Zap (48K)\0", NULL, "Imagine", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecZipzapRomInfo, SpecZipzapRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Homebrew Games (Post-2000)
// Hereunder put only the HB ZX Spectrum games

// Black Star (48K) (HB)

static struct BurnRomInfo SpecblackstarRomDesc[] = {
	{ "Black Star (2015)(usebox.net).tap", 26349, 0xfd6cd75d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specblackstar, Specblackstar, Spectrum)
STD_ROM_FN(Specblackstar)

struct BurnDriver BurnSpecblackstar = {
	"spec_blackstar", NULL, "spec_spectrum", NULL, "2015",
	"Black Star (48K) (HB)\0", NULL, "usebox.net", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecblackstarRomInfo, SpecblackstarRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// BlockZ (128K) (HB)

static struct BurnRomInfo SpecblockzRomDesc[] = {
	{ "BlockZ (2021)(Raymond Russell).tap", 24913, 0x44b94d5e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specblockz, Specblockz, Spec128)
STD_ROM_FN(Specblockz)

struct BurnDriver BurnSpecblockz = {
	"spec_blockz", NULL, "spec_spec128", NULL, "2020",
	"BlockZ (128K) (HB)\0", NULL, "Raymond Russell", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecblockzRomInfo, SpecblockzRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPMDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Brick Rick Graveyard Shift (128K) (HB, v1.03)

static struct BurnRomInfo SpecbrickrickRomDesc[] = {
	{ "Brick Rick Graveyard Shift v1.03 (2021)(usebox.net).tap", 31751, 0x5cea272c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbrickrick, Specbrickrick, Spec128)
STD_ROM_FN(Specbrickrick)

struct BurnDriver BurnSpecbrickrick = {
	"spec_brickrick", NULL, "spec_spec128", NULL, "2021",
	"Brick Rick Graveyard Shift (128K) (HB, v1.03)\0", NULL, "usebox.net", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecbrickrickRomInfo, SpecbrickrickRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Crazy Blaster (128K) (HB)

static struct BurnRomInfo SpeccblasterRomDesc[] = {
	{ "Crazy Blaster (2019)(TFW8b).tap", 32951, 0x6797d22c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccblaster, Speccblaster, Spec128)
STD_ROM_FN(Speccblaster)

struct BurnDriver BurnSpeccblaster = {
	"spec_cblaster", NULL, "spec_spec128", NULL, "2019",
	"Crazy Blaster (128K) (HB)\0", NULL, "The Future Was 8bit", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpeccblasterRomInfo, SpeccblasterRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Danterrifik III - The Dark Age of Grotesque (128K) (HB)

static struct BurnRomInfo Specdanterrifik3RomDesc[] = {
	{ "Danterrifik 3 (2021)(TokuSoft).tap", 46898, 0x9041c4da, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdanterrifik3, Specdanterrifik3, Spec128)
STD_ROM_FN(Specdanterrifik3)

struct BurnDriver BurnSpecdanterrifik3 = {
	"spec_danterrifik3", NULL, "spec_spec128", NULL, "2021",
	"Danterrifik III - The Dark Age of Grotesque (128K) (HB)\0", NULL, "Toku Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Specdanterrifik3RomInfo, Specdanterrifik3RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Down the pipe (128K) (HB)

static struct BurnRomInfo Specdownpipe128RomDesc[] = {
	{ "Down the pipe 128K (2021)(DF Design).tap", 58634, 0x6ff07621, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdownpipe128, Specdownpipe128, Spec128)
STD_ROM_FN(Specdownpipe128)

struct BurnDriver BurnSpecdownpipe128 = {
	"spec_downpipe128", NULL, "spec_spec128", NULL, "2021",
	"Down the pipe (128K) (HB)\0", NULL, "DF Design", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, Specdownpipe128RomInfo, Specdownpipe128RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Down the pipe (48K) (HB)

static struct BurnRomInfo Specdownpipe48RomDesc[] = {
	{ "Down the pipe 48K (2021)(DF Design).tap", 46991, 0x71ce5665, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdownpipe48, Specdownpipe48, Spectrum)
STD_ROM_FN(Specdownpipe48)

struct BurnDriver BurnSpecdownpipe48 = {
	"spec_downpipe48", "spec_downpipe128", "spec_spectrum", NULL, "2021",
	"Down the pipe (48K) (HB)\0", NULL, "DF Design", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, Specdownpipe48RomInfo, Specdownpipe48RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Eugene - Lord of the Bathroom (48K) (HB)

static struct BurnRomInfo SpecmminrelbRomDesc[] = {
	{ "Manic Miner - Eugene - Lord of the Bathroom (1999)(Manic Miner Technologies).z80", 0x07792, 0x3062e7d8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmminrelb, Specmminrelb, Spectrum)
STD_ROM_FN(Specmminrelb)

struct BurnDriver BurnSpecmminrelb = {
	"spec_mminrelb", NULL, "spec_spectrum", NULL, "1999",
	"Eugene - Lord of the Bathroom (48K) (HB)\0", NULL, "Manic Miner Technologies", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecmminrelbRomInfo, SpecmminrelbRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Goody Recoloured (128K) (Hack)

static struct BurnRomInfo SpecgoodyrecolourRomDesc[] = {
	{ "Goody (1987)(Opera Soft)(Recoloured)(es).tap", 0x0dbb2, 0x8b02e960, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specgoodyrecolour, Specgoodyrecolour, Spec128)
STD_ROM_FN(Specgoodyrecolour)

struct BurnDriver BurnSpecgoodyrecolour = {
	"spec_goodyrecolour", "spec_goodyes", "spec_spec128", NULL, "2012",
	"Goody Recoloured (128K) (Hack)\0", NULL, "Rafal Miazga", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecgoodyrecolourRomInfo, SpecgoodyrecolourRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// L'Abbaye des Morts (128K) (HB)

static struct BurnRomInfo SpeclabbayeRomDesc[] = {
	{ "L'Abbaye des Morts (2014)(Darkhorace).z80", 58110, 0xdec96ae7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speclabbaye, Speclabbaye, Spec128)
STD_ROM_FN(Speclabbaye)

struct BurnDriver BurnSpeclabbaye = {
	"spec_labbaye", NULL, "spec_spec128", NULL, "2014",
	"L'Abbaye des Morts (128K) (HB)\0", "Port of original game by Locomalito and Gryzor87", "Darkhorace", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpeclabbayeRomInfo, SpeclabbayeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Legend of Traxtor (48K) (HB)

static struct BurnRomInfo SpectraxtorRomDesc[] = {
	{ "Legend of Traxtor (2014)(usebox.net).tap", 23986, 0xef7dc08c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spectraxtor, Spectraxtor, Spectrum)
STD_ROM_FN(Spectraxtor)

struct BurnDriver BurnSpectraxtor = {
	"spec_traxtor", NULL, "spec_spectrum", NULL, "2014",
	"Legend of Traxtor (48K) (HB)\0", NULL, "usebox.net", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpectraxtorRomInfo, SpectraxtorRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Pilot Attack (128K) (HB, v1.1)

static struct BurnRomInfo SpecpilotatkRomDesc[] = {
	{ "Pilot Attack v1.1 (2017)(TFW8b).tap", 30930, 0x8e0f7278, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specpilotatk, Specpilotatk, Spec128)
STD_ROM_FN(Specpilotatk)

struct BurnDriver BurnSpecpilotatk = {
	"spec_pilotatk", NULL, "spec_spec128", NULL, "2017",
	"Pilot Attack (128K) (HB, v1.1)\0", NULL, "The Future Was 8bit", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT, 0,
	SpectrumGetZipName, SpecpilotatkRomInfo, SpecpilotatkRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Return of Traxtor (48K) (HB)

static struct BurnRomInfo Spectraxtor2RomDesc[] = {
	{ "Return of Traxtor (2015)(usebox.net).tap", 27292, 0x6fac2cd6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spectraxtor2, Spectraxtor2, Spectrum)
STD_ROM_FN(Spectraxtor2)

struct BurnDriver BurnSpectraxtor2 = {
	"spec_traxtor2", NULL, "spec_spectrum", NULL, "2015",
	"Return of Traxtor (48K) (HB)\0", NULL, "usebox.net", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, Spectraxtor2RomInfo, Spectraxtor2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Valley of Rains (128K) (HB)

static struct BurnRomInfo SpecvalleyofrainsRomDesc[] = {
	{ "Valley of Rains (2019)(Zosya Entertainment)(128k).z80", 0x0b647, 0x7a79ca09, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specvalleyofrains, Specvalleyofrains, Spec128)
STD_ROM_FN(Specvalleyofrains)

struct BurnDriver BurnSpecvalleyofrains = {
	"spec_valleyofrains", NULL, "spec_spec128", NULL, "2019",
	"Valley of Rains (128K) (HB)\0", NULL, "Zosya Entertainment", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecvalleyofrainsRomInfo, SpecvalleyofrainsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Alter Ego (48K) (HB)

static struct BurnRomInfo SpecalteregoRomDesc[] = {
	{ "Alter Ego (2011)(RetroSouls).tap", 38926, 0x855c13dc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specalterego, Specalterego, Spectrum)
STD_ROM_FN(Specalterego)

struct BurnDriver BurnSpecalterego = {
	"spec_alterego", NULL, "spec_spectrum", NULL, "2011",
	"Alter Ego (48K) (HB)\0", NULL, "RetroSouls", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecalteregoRomInfo, SpecalteregoRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Cheril the Goddess (48K) (HB, v2)

static struct BurnRomInfo SpeccherilgodRomDesc[] = {
	{ "Cheril The Goddess V2 (2011)(Mojon Twins).tap", 43545, 0x94cc3db8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccherilgod, Speccherilgod, Spectrum)
STD_ROM_FN(Speccherilgod)

struct BurnDriver BurnSpeccherilgod = {
	"spec_cherilgod", NULL, "spec_spectrum", NULL, "2011",
	"Cheril the Goddess (48K) (HB, v2)\0", NULL, "The Mojon Twins", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpeccherilgodRomInfo, SpeccherilgodRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Danterrifik (128K) (HB, v1.1)

static struct BurnRomInfo SpecdanterrifikRomDesc[] = {
	{ "Danterrifik v1.1 (2020)(Toku Soft).tap", 49067, 0x6a088600, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdanterrifik, Specdanterrifik, Spec128)
STD_ROM_FN(Specdanterrifik)

struct BurnDriver BurnSpecdanterrifik = {
	"spec_danterrifik", NULL, "spec_spec128", NULL, "2020",
	"Danterrifik (128K) (HB, v1.1)\0", NULL, "Toku Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecdanterrifikRomInfo, SpecdanterrifikRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Danterrifik II - The Nightmare Continues (128K) (HB)

static struct BurnRomInfo Specdanterrifik2RomDesc[] = {
	{ "Danterrifik 2 (2020)(Toku Soft).tap", 49982, 0x478c7adf, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdanterrifik2, Specdanterrifik2, Spec128)
STD_ROM_FN(Specdanterrifik2)

struct BurnDriver BurnSpecdanterrifik2 = {
	"spec_danterrifik2", NULL, "spec_spec128", NULL, "2020",
	"Danterrifik II - The Nightmare Continues (128K) (HB)\0", NULL, "Toku Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Specdanterrifik2RomInfo, Specdanterrifik2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dreamwalker - Alter Ego 2 (128K) (HB, v1.1)

static struct BurnRomInfo SpecdrmwalkerRomDesc[] = {
	{ "Dreamwalker128k v1.1 (2014)(RetroSouls).tap", 92060, 0x89db756e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specdrmwalker, Specdrmwalker, Spec128)
STD_ROM_FN(Specdrmwalker)

struct BurnDriver BurnSpecdrmwalker = {
	"spec_drmwalker", NULL, "spec_spec128", NULL, "2014",
	"Dreamwalker - Alter Ego 2 (128K) (HB, v1.1)\0", NULL, "RetroSouls", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecdrmwalkerRomInfo, SpecdrmwalkerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPMDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Endless Forms Most Beautiful (48K) (HB)

static struct BurnRomInfo SpecefmbRomDesc[] = {
	{ "Endless Forms Most Beautiful (2012)(Stonechat Productions).tap", 31843, 0x4d866f79, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specefmb, Specefmb, Spectrum)
STD_ROM_FN(Specefmb)

struct BurnDriver BurnSpecefmb = {
	"spec_efmb", NULL, "spec_spectrum", NULL, "2012",
	"Endless Forms Most Beautiful (48K) (HB)\0", NULL, "Stonechat Productions", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecefmbRomInfo, SpecefmbRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Frantic Pengo (128K) (HB)

static struct BurnRomInfo Specfp128kRomDesc[] = {
	{ "frantic pengo 128k (2020)(gabam).tap", 45927, 0x66c3790a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specfp128k, Specfp128k, Spec128)
STD_ROM_FN(Specfp128k)

struct BurnDriver BurnSpecfp128k = {
	"spec_fp128k", NULL, "spec_spec128", NULL, "2020",
	"Frantic Pengo (128K) (HB)\0", NULL, "Gabriele Amore", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, Specfp128kRomInfo, Specfp128kRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPMDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Gluf (RetroSouls) (128K) (HB)

static struct BurnRomInfo SpecglufRomDesc[] = {
	{ "Gluf (2019)(RetroSouls).tap",  34669, 0x52be5093, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specgluf, Specgluf, Spec128)
STD_ROM_FN(Specgluf)

struct BurnDriver BurnSpecgluf = {
	"spec_gluf", NULL, "spec_spec128", NULL, "2019",
	"GLUF (128K) (HB)\0", NULL, "RetroSouls", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecglufRomInfo, SpecglufRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// GraviBots (RetroSouls) (128K) (HB)

static struct BurnRomInfo SpecgravibotsRomDesc[] = {
	{ "GraviBots (2014)(RetroSouls).z80",  29436, 0x375bfa1c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specgravibots, Specgravibots, Spec128)
STD_ROM_FN(Specgravibots)

struct BurnDriver BurnSpecgravibots = {
	"spec_gravibots", NULL, "spec_spec128", NULL, "2014",
	"GraviBots (128K) (HB)\0", "Press Space and Enter to select options", "RetroSouls", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecgravibotsRomInfo, SpecgravibotsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Jumpin' Jupiter (48K) (HB)

static struct BurnRomInfo SpecjumpjupiterRomDesc[] = {
	{ "Jumpin' Jupiter (2020)(Quantum Sheep).tap", 40858, 0x34a1f400, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specjumpjupiter, Specjumpjupiter, Spectrum)
STD_ROM_FN(Specjumpjupiter)

struct BurnDriver BurnSpecjumpjupiter = {
	"spec_jumpjupiter", NULL, "spec_spectrum", NULL, "2020",
	"Jumpin' Jupiter (48K) (HB)\0", NULL, "Quantum Sheep", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecjumpjupiterRomInfo, SpecjumpjupiterRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPMDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Lost Treasures of Tulum, The (English) (128K) (HB)

static struct BurnRomInfo SpeclostulumenRomDesc[] = {
	{ "Lost Treasures of Tulum, The EN 128K (2020)(RetroWorks).tap", 75999, 0xc64c38ad, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speclostulumen, Speclostulumen, Spec128)
STD_ROM_FN(Speclostulumen)

struct BurnDriver BurnSpeclostulumen = {
	"spec_lostulumen", NULL, "spec_spec128", NULL, "2020",
	"Lost Treasures of Tulum, The (English) (128K) (HB)\0", NULL, "RetroWorks", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpeclostulumenRomInfo, SpeclostulumenRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Lost Treasures of Tulum, The (Spanish) (128K) (HB)

static struct BurnRomInfo SpeclostulumesRomDesc[] = {
	{ "Lost Treasures of Tulum, The ES 128K (2020)(RetroWorks).tap", 75999, 0xc4087b0e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speclostulumes, Speclostulumes, Spec128)
STD_ROM_FN(Speclostulumes)

struct BurnDriver BurnSpeclostulumes = {
	"spec_lostulumes", "spec_lostulumen", "spec_spec128", NULL, "2020",
	"Lost Treasures of Tulum, The (Spanish) (128K) (HB)\0", NULL, "RetroWorks", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_ADV, 0,
	SpectrumGetZipName, SpeclostulumesRomInfo, SpeclostulumesRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Misifu Remeow Color (128K) (HB, v1.4.1)

static struct BurnRomInfo SpecmisifucolorRomDesc[] = {
	{ "misifu_COLOR(2021)(JSM Soft).tap",  73610, 0x387dd30a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmisifucolor, Specmisifucolor, Spec128)
STD_ROM_FN(Specmisifucolor)

struct BurnDriver BurnSpecmisifucolor = {
	"spec_misifucolor", NULL, "spec_spec128", NULL, "2021",
	"Misifu Remeow Color (128K) (HB, v1.4.1)\0", "Alley Cat demake for ZX Spectrum", "JSM Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecmisifucolorRomInfo, SpecmisifucolorRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3

};

// Misifu Remeow CGA (128K) (HB, v1.4.1)

static struct BurnRomInfo SpecmisifucgaRomDesc[] = {
	{ "misifu_CGA(2021)(JSM Soft).tap", 73469, 0x84550b57, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmisifucga, Specmisifucga, Spec128)
STD_ROM_FN(Specmisifucga)

struct BurnDriver BurnSpecmisifucga = {
	"spec_misifucga", "spec_misifucolor", "spec_spec128", NULL, "2021",
	"Misifu Remeow CGA (128K) (HB, v1.4.1)\0", "Alley Cat demake for ZX Spectrum", "JSM Soft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecmisifucgaRomInfo, SpecmisifucgaRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// MultiDude (128K) (HB)

static struct BurnRomInfo SpecmultidudeRomDesc[] = {
	{ "MultiDude128k (2014)(RetroSouls).tap",  47634, 0x0892872f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specmultidude, Specmultidude, Spec128)
STD_ROM_FN(Specmultidude)

struct BurnDriver BurnSpecmultidude = {
	"spec_multidude", NULL, "spec_spec128", NULL, "2014",
	"MultiDude (128K) (HB)\0", NULL, "RetroSouls", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecmultidudeRomInfo, SpecmultidudeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Old Tower (128K) (HB)

static struct BurnRomInfo SpecoldtowerRomDesc[] = {
	{ "OldTower128k (2018)(RetroSouls).tap",  97764, 0x837df93f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specoldtower, Specoldtower, Spec128)
STD_ROM_FN(Specoldtower)

struct BurnDriver BurnSpecoldtower = {
	"spec_oldtower", NULL, "spec_spec128", NULL, "2019",
	"Old Tower (128K) (HB)\0", NULL, "RetroSouls", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_ACTION | GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecoldtowerRomInfo, SpecoldtowerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Relic, The - aka Reliquia, La (English) (128K) (HB)

static struct BurnRomInfo SpecrelicRomDesc[] = {
	{ "The Relic 128K (2020)(Roolandoo).tap",  46629, 0x183c5158, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrelic, Specrelic, Spec128)
STD_ROM_FN(Specrelic)

struct BurnDriver BurnSpecrelic = {
	"spec_relic", NULL, "spec_spec128", NULL, "2020",
	"Reliquia, La (English) (128K) (HB)\0", NULL, "Angel Colaso", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecrelicRomInfo, SpecrelicRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sir Ababol 2 (English) (128K) (HB)

static struct BurnRomInfo Specsirababol2RomDesc[] = {
	{ "Sir Ababol 2 128k (2013)(mojon twins).tap", 79038, 0x61a75108, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsirababol2, Specsirababol2, Spec128)
STD_ROM_FN(Specsirababol2)

struct BurnDriver BurnSpecsirababol2 = {
	"spec_sirababol2", NULL, "spec_spec128", NULL, "2013",
	"Sir Ababol 2 (English) (128K) (HB)\0", NULL, "The Mojon Twins", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, Specsirababol2RomInfo, Specsirababol2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sir Ababol DX (48K) (HB)

static struct BurnRomInfo SpecsirababoldxRomDesc[] = {
	{ "Sir Ababol DX (2013)(mojon twins).tap", 43469, 0x06705a38, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsirababoldx, Specsirababoldx, Spectrum)
STD_ROM_FN(Specsirababoldx)

struct BurnDriver BurnSpecsirababoldx = {
	"spec_sirababoldx", NULL, "spec_spectrum", NULL, "2013",
	"Sir Ababol DX (48K) (HB)\0", NULL, "The Mojon Twins", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecsirababoldxRomInfo, SpecsirababoldxRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// That Sinking Feeling (128K) (HB)

static struct BurnRomInfo SpectsfeelingRomDesc[] = {
	{ "That Sinking Feeling(2017)(Sunteam).tap", 34567, 0xe7822c33, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spectsfeeling, Spectsfeeling, Spec128)
STD_ROM_FN(Spectsfeeling)

struct BurnDriver BurnSpectsfeeling = {
	"spec_tsfeeling", NULL, "spec_spec128", NULL, "2017",
	"That Sinking Feeling (128K) (HB)\0", NULL, "Sunteam", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpectsfeelingRomInfo, SpectsfeelingRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// The Witch v1.0.1 (128K) (HB)

static struct BurnRomInfo SpecthewitchRomDesc[] = {
	{ "The Witch (v1.0.1)(2020)(Serranito Online).tap",  48970, 0x4dacc7d8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specthewitch, Specthewitch, Spec128)
STD_ROM_FN(Specthewitch)

struct BurnDriver BurnSpecthewitch = {
	"spec_thewitch", NULL, "spec_spec128", NULL, "2020",
	"The Witch (128K) (HB, v1.0.1)\0", "The remake of classic game Cauldron", "Serranito Online", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecthewitchRomInfo, SpecthewitchRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3

};

// The Witch - Ending (English) (128K) (HB)

static struct BurnRomInfo SpecthewitchenRomDesc[] = {
	{ "The Witch (ending EN)(2020)(Serranito Online).tap", 36161, 0x1d18f64a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specthewitchen, Specthewitchen, Spec128)
STD_ROM_FN(Specthewitchen)

struct BurnDriver BurnSpecthewitchen = {
	"spec_thewitchen", "spec_thewitch", "spec_spec128", NULL, "2020",
	"The Witch - Ending (English) (128K) (HB, v1.0.1)\0", "The remake of classic game Cauldron", "Serranito Online", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecthewitchenRomInfo, SpecthewitchenRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// The Witch - Ending (Spanish) (128K) (HB)

static struct BurnRomInfo SpecthewitchesRomDesc[] = {
	{ "The Witch (ending ES)(2020)(Serranito Online).tap", 36165, 0x3c916134, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specthewitches, Specthewitches, Spec128)
STD_ROM_FN(Specthewitches)

struct BurnDriver BurnSpecthewitches = {
	"spec_thewitches", "spec_thewitch", "spec_spec128", NULL, "2020",
	"The Witch - Ending (Spanish) (128K) (HB, v1.0.1)\0", "The remake of classic game Cauldron", "Serranito Online", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_HORSHOOT | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecthewitchesRomInfo, SpecthewitchesRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Tourmaline (128K) (HB)

static struct BurnRomInfo SpectrmlineRomDesc[] = {
	{ "Tourmaline (2016)(RetroSouls).tap",  46613, 0xf573c97f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spectrmline, Spectrmline, Spec128)
STD_ROM_FN(Spectrmline)

struct BurnDriver BurnSpectrmline = {
	"spec_trmline", NULL, "spec_spec128", NULL, "2016",
	"Tourmaline (128K) (HB)\0", NULL, "RetroSouls", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpectrmlineRomInfo, SpectrmlineRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Yazzie (128K) (HB)

static struct BurnRomInfo SpecyazzieRomDesc[] = {
	{ "Yazzie128k (2019)(RetroSouls).tap",  70103, 0x68d21e49, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specyazzie, Specyazzie, Spec128)
STD_ROM_FN(Specyazzie)

struct BurnDriver BurnSpecyazzie = {
	"spec_yazzie", NULL, "spec_spec128", NULL, "2019",
	"Yazzie (128K) (HB)\0", NULL, "RetroSouls", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecyazzieRomInfo, SpecyazzieRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Willy The Wasp (48K-128K) (HB)

static struct BurnRomInfo SpecWwaspRomDesc[] = {
	{ "Willy The Wasp (2014)(The Death Squad).tap", 41193, 0x030e8442, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecWwasp, SpecWwasp, Spec128)
STD_ROM_FN(SpecWwasp)

struct BurnDriver BurnSpecWwasp = {
	"spec_wwasp", NULL, "spec_spec128", NULL, "2014",
	"Willy The Wasp (48K-128K) (HB)\0", NULL, "The Death Squad", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecWwaspRomInfo, SpecWwaspRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Willy The Wasp 2 (128K) (HB)

static struct BurnRomInfo SpecWwasp2RomDesc[] = {
	{ "Willy The Wasp 2 (2014)(The Death Squad).tap", 64604, 0x3e8b70fb, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecWwasp2, SpecWwasp2, Spec128)
STD_ROM_FN(SpecWwasp2)

struct BurnDriver BurnSpecWwasp2 = {
	"spec_wwasp2", NULL, "spec_spec128", NULL, "2014",
	"Willy The Wasp 2 (128K) (HB)\0", NULL, "The Death Squad", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecWwasp2RomInfo, SpecWwasp2RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecIntf2DIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Black & White (128K) (HB, v1.1)

static struct BurnRomInfo SpecbnwRomDesc[] = {
	{ "Black & White v1.1 (2020)(Pat Morita Team).tap", 72706, 0x6629deb1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specbnw, Specbnw, Spec128)
STD_ROM_FN(Specbnw)

struct BurnDriver BurnSpecbnw = {
	"spec_bnw", NULL, "spec_spec128", NULL, "2020",
	"Black & White (128K) (HB, v1.1)\0", "Use SPACE and ENTER to set options", "Pat Morita Team", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecbnwRomInfo, SpecbnwRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Circuitry (48K) (HB, v1.1)

static struct BurnRomInfo SpeccircuitryRomDesc[] = {
	{ "Circuitry48k v1.1 (2017)(Rucksack Games).tap", 31538, 0xc97061d1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccircuitry, Speccircuitry, Spectrum)
STD_ROM_FN(Speccircuitry)

struct BurnDriver BurnSpeccircuitry = {
	"spec_circuitry", NULL, "spec_spectrum", NULL, "2017",
	"Circuitry (48K) (HB, v1.1)\0", NULL, "Rucksack Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpeccircuitryRomInfo, SpeccircuitryRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Coloco (48K) (HB)

static struct BurnRomInfo SpeccolocoRomDesc[] = {
	{ "Coloco (2020)(Mojon Twins).tap", 46855, 0xdd07fa17, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Speccoloco, Speccoloco, Spectrum)
STD_ROM_FN(Speccoloco)

struct BurnDriver BurnSpeccoloco = {
	"spec_coloco", NULL, "spec_spectrum", NULL, "2020",
	"Coloco (48K) (HB)\0", NULL, "The Mojon Twins", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpeccolocoRomInfo, SpeccolocoRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Foggy's Quest (128K) (HB)

static struct BurnRomInfo SpecfoggyquestRomDesc[] = {
	{ "Foggy's Quest128k (2017)(Rucksack Games).tap", 43528, 0x0d624a1c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specfoggyquest, Specfoggyquest, Spec128)
STD_ROM_FN(Specfoggyquest)

struct BurnDriver BurnSpecfoggyquest = {
	"spec_foggyquest", NULL, "spec_spec128", NULL, "2017",
	"Foggy's Quest (128K) (HB)\0", NULL, "Rucksack Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecfoggyquestRomInfo, SpecfoggyquestRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// All Hallows Rise of Pumpkin (128K) (HB)

static struct BurnRomInfo SpechlwpumpkinRomDesc[] = {
	{ "ALL Hallows Rise of Pumpkin (2018)(Rucksack Games).tap", 46999, 0x76ecb3a0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spechlwpumpkin, Spechlwpumpkin, Spec128)
STD_ROM_FN(Spechlwpumpkin)

struct BurnDriver BurnSpechlwpumpkin = {
	"spec_hlwpumpkin", NULL, "spec_spec128", NULL, "2018",
	"All Hallows Rise of Pumpkin (128K) (HB)\0", NULL, "Rucksack Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpechlwpumpkinRomInfo, SpechlwpumpkinRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ninjajar v1.1 (128K) (HB)

static struct BurnRomInfo SpecninjajarRomDesc[] = {
	{ "Ninjajar v1.1 (2014)(Mojon Twins).tap", 122675, 0x617db06c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specninjajar, Specninjajar, Spec128)
STD_ROM_FN(Specninjajar)

struct BurnDriver BurnSpecninjajar = {
	"spec_ninjajar", NULL, "spec_spec128", NULL, "2014",
	"Ninjajar (128K) (HB, v1.1)\0", NULL, "The Mojon Twins", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecninjajarRomInfo, SpecninjajarRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Rubicon (128K) (HB, v1.1)

static struct BurnRomInfo SpecrubiconRomDesc[] = {
	{ "Rubicon v1.1 (2018)(Rucksack Games).tap", 48224, 0x99ec517e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrubicon, Specrubicon, Spec128)
STD_ROM_FN(Specrubicon)

struct BurnDriver BurnSpecrubicon = {
	"spec_rubicon", NULL, "spec_spec128", NULL, "2018",
	"Rubicon (128K) (HB, v1.1)\0", NULL, "Rucksack Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_MAZE | GBF_ACTION, 0,
	SpectrumGetZipName, SpecrubiconRomInfo, SpecrubiconRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Sector Invasion (128K) (HB)

static struct BurnRomInfo SpecsectinvnRomDesc[] = {
	{ "Sector Invasion (2014).tap", 38873, 0x275c56d5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specsectinvn, Specsectinvn, Spec128)
STD_ROM_FN(Specsectinvn)

struct BurnDriver BurnSpecsectinvn = {
	"spec_sectinvn", NULL, "spec_spec128", NULL, "2014",
	"Sector Invasion (128K) (HB)\0", NULL, "RetroSouls", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_VERSHOOT, 0,
	SpectrumGetZipName, SpecsectinvnRomInfo, SpecsectinvnRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Tenebra Macabre (48K) (HB)

static struct BurnRomInfo SpectmacabreRomDesc[] = {
	{ "Tenebra Macabre (2013)(Mojon Twins).tap", 39490, 0xbfa58b2c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Spectmacabre, Spectmacabre, Spectrum)
STD_ROM_FN(Spectmacabre)

struct BurnDriver BurnSpectmacabre = {
	"spec_tmacabre", NULL, "spec_spectrum", NULL, "2013",
	"Tenebra Macabre (48K) (HB)\0", NULL, "The Mojon Twins", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM | GBF_PUZZLE, 0,
	SpectrumGetZipName, SpectmacabreRomInfo, SpectmacabreRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Vampire Vengeance (128K) (HB, v1.1)

static struct BurnRomInfo SpecvampvengRomDesc[] = {
	{ "Vampire Vengeance v1.1 (2020)(Poe Games).tap", 58059, 0x2796871d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specvampveng, Specvampveng, Spec128)
STD_ROM_FN(Specvampveng)

struct BurnDriver BurnSpecvampveng = {
	"spec_vampveng", NULL, "spec_spec128", NULL, "2020",
	"Vampire Vengeance (128K) (HB, v1.1)\0", NULL, "Poe Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecvampvengRomInfo, SpecvampvengRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Cray-5 (128K) (HB)

static struct BurnRomInfo SpecCray5RomDesc[] = {
	{ "Cray-5 (2011)(Retroworks).tap", 53875, 0xa1467bf5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCray5, SpecCray5, Spec128)
STD_ROM_FN(SpecCray5)

struct BurnDriver BurnSpecCray5 = {
	"spec_cray5", NULL, "spec_spec128", NULL, "2011",
	"Cray-5 (128K) (HB)\0", NULL, "Retroworks", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_RUNGUN, 0,
	SpectrumGetZipName, SpecCray5RomInfo, SpecCray5RomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dead Flesh Boy (128K) (HB)

static struct BurnRomInfo SpecDfboyRomDesc[] = {
	{ "Dead Flesh Boy (2015)(VANB).tap", 40669, 0x3375d2c9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDfboy, SpecDfboy, Spec128)
STD_ROM_FN(SpecDfboy)

struct BurnDriver BurnSpecDfboy = {
	"spec_dfboy", NULL, "spec_spec128", NULL, "2015",
	"Dead Flesh Boy (128K) (HB)\0", NULL, "VANB", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecDfboyRomInfo, SpecDfboyRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Flype (48K) (HB)

static struct BurnRomInfo SpecFlypeRomDesc[] = {
	{ "Flype (2015)(Repixel8).tap", 41719, 0x2b9a90f2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecFlype, SpecFlype, Spectrum)
STD_ROM_FN(SpecFlype)

struct BurnDriver BurnSpecFlype = {
	"spec_flype", NULL, "spec_spectrum", NULL, "2015",
	"Flype (48K) (HB)\0", NULL, "Repixel8", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecFlypeRomInfo, SpecFlypeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Horace and the Robots (128K) (HB)

static struct BurnRomInfo SpecHoracerobotsRomDesc[] = {
	{ "Horace and the Robots (2017)(Steve Snake).tap", 25517, 0x05fd13e2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecHoracerobots, SpecHoracerobots, Spec128)
STD_ROM_FN(SpecHoracerobots)

struct BurnDriver BurnSpecHoracerobots = {
	"spec_horacerobots", NULL, "spec_spec128", NULL, "2017",
	"Horace and the Robots (128K) (HB)\0", NULL, "Steve Snake", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecHoracerobotsRomInfo, SpecHoracerobotsRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ooze (128K) (HB)

static struct BurnRomInfo SpecOozeRomDesc[] = {
	{ "Ooze (2017)(Bubblesoft).tap", 49790, 0x188d3e4e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecOoze, SpecOoze, Spec128)
STD_ROM_FN(SpecOoze)

struct BurnDriver BurnSpecOoze = {
	"spec_ooze", NULL, "spec_spec128", NULL, "2017",
	"Ooze (128K) (HB)\0", NULL, "Bubblesoft", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecOozeRomInfo, SpecOozeRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Restless Andre (128K) (HB)

static struct BurnRomInfo SpecRestlessandreRomDesc[] = {
	{ "Restless Andre (2019)(Jaime Grilo).tap", 48113, 0x0db7023a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecRestlessandre, SpecRestlessandre, Spec128)
STD_ROM_FN(SpecRestlessandre)

struct BurnDriver BurnSpecRestlessandre = {
	"spec_restlessandre", NULL, "spec_spec128", NULL, "2019",
	"Restless Andre (128K) (HB)\0", NULL, "Jaime Grilo", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_MAZE | GBF_ACTION, 0,
	SpectrumGetZipName, SpecRestlessandreRomInfo, SpecRestlessandreRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Terrahawks (128K) (HB)

static struct BurnRomInfo SpecTerrahawksRomDesc[] = {
	{ "Terrahawks (2014)(Gary James).tap", 24138, 0xd6f05303, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecTerrahawks, SpecTerrahawks, Spec128)
STD_ROM_FN(SpecTerrahawks)

struct BurnDriver BurnSpecTerrahawks = {
	"spec_terrahawks", NULL, "spec_spec128", NULL, "2014",
	"Terrahawks (128K) (HB)\0", NULL, "Gary James", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_SHOOT, 0,
	SpectrumGetZipName, SpecTerrahawksRomInfo, SpecTerrahawksRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Incredible Shrinking Professor, The (128K) (HB, v1.1)

static struct BurnRomInfo SpecincprofRomDesc[] = {
	{ "Incredible Shrinking Professor, the (2017)(Rucksack Games).tap", 44116, 0x03e3de19, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specincprof, Specincprof, Spec128)
STD_ROM_FN(Specincprof)

struct BurnDriver BurnSpecincprof = {
	"spec_incprof", NULL, "spec_spec128", NULL, "2017",
	"Incredible Shrinking Professor, The (128K) (HB, v1.1)\0", NULL, "Rucksack Games", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecincprofRomInfo, SpecincprofRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Robots Rumble (128K) (HB)

static struct BurnRomInfo SpecrrumbleRomDesc[] = {
	{ "Robots Rumble (2018)(Miguetelo).tap", 39461, 0xc7c5dc79, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(Specrrumble, Specrrumble, Spec128)
STD_ROM_FN(Specrrumble)

struct BurnDriver BurnSpecrrumble = {
	"spec_rrumble", NULL, "spec_spec128", NULL, "2018",
	"Robots Rumble (128K) (HB)\0", NULL, "Miguetelo", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecrrumbleRomInfo, SpecrrumbleRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Bobby Carrot (128K) (HB)

static struct BurnRomInfo SpecBobbycarrotRomDesc[] = {
	{ "Bobby Carrot (2018)(Diver, Quiet, Kyv, Zorba).tap", 34125, 0xaf5919ca, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecBobbycarrot, SpecBobbycarrot, Spec128)
STD_ROM_FN(SpecBobbycarrot)

struct BurnDriver BurnSpecBobbycarrot = {
	"spec_bobbycarrot", NULL, "spec_spec128", NULL, "2018",
	"Bobby Carrot (128K) (HB)\0", NULL, "Diver, Quiet, Kyv, Zorba", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecBobbycarrotRomInfo, SpecBobbycarrotRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Castle Capers (128K) (HB)

static struct BurnRomInfo SpecCastlecapersRomDesc[] = {
	{ "CastleCapers (2017)(Gabriele Amore).tap", 36751, 0xdb772c7c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecCastlecapers, SpecCastlecapers, Spec128)
STD_ROM_FN(SpecCastlecapers)

struct BurnDriver BurnSpecCastlecapers = {
	"spec_castlecapers", NULL, "spec_spec128", NULL, "2017",
	"Castle Capers (128K) (HB)\0", NULL, "Gabriele Amore", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecCastlecapersRomInfo, SpecCastlecapersRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Dirty Dozer (128K) (HB)

static struct BurnRomInfo SpecDirtydozerRomDesc[] = {
	{ "Dirty Dozer (2019)(Miguetelo).tap", 41391, 0xaf06dd37, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecDirtydozer, SpecDirtydozer, Spec128)
STD_ROM_FN(SpecDirtydozer)

struct BurnDriver BurnSpecDirtydozer = {
	"spec_dirtydozer", NULL, "spec_spec128", NULL, "2019",
	"Dirty Dozer (128K) (HB)\0", NULL, "Miguetelo", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PUZZLE, 0,
	SpectrumGetZipName, SpecDirtydozerRomInfo, SpecDirtydozerRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// El Stompo (48K) (HB)

static struct BurnRomInfo SpecElstompoRomDesc[] = {
	{ "ElStompo (2014)(Stonechat Productions).tap", 44449, 0xb2370ea9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecElstompo, SpecElstompo, Spectrum)
STD_ROM_FN(SpecElstompo)

struct BurnDriver BurnSpecElstompo = {
	"spec_elstompo", NULL, "spec_spectrum", NULL, "2014",
	"El Stompo (48K) (HB)\0", NULL, "Stonechat Productions", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_ACTION, 0,
	SpectrumGetZipName, SpecElstompoRomInfo, SpecElstompoRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecQAOPSpaceDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Gandalf Deluxe (128K) (HB)

static struct BurnRomInfo SpecGandalfRomDesc[] = {
	{ "Gandalf Deluxe (2018)(Speccy Nights).tap", 95019, 0x1f7f26c8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecGandalf, SpecGandalf, Spec128)
STD_ROM_FN(SpecGandalf)

struct BurnDriver BurnSpecGandalf = {
	"spec_gandalf", NULL, "spec_spec128", NULL, "2018",
	"Gandalf Deluxe (128K) (HB)\0", NULL, "Speccy Nights", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecGandalfRomInfo, SpecGandalfRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Metal Man (Russian) (48K) (HB)

static struct BurnRomInfo SpecMetalmanRomDesc[] = {
	{ "Metal Man (2014)(Oleg Origin).tap", 48298, 0xef74d54b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMetalman, SpecMetalman, Spectrum)
STD_ROM_FN(SpecMetalman)

struct BurnDriver BurnSpecMetalman = {
	"spec_metalman", NULL, "spec_spectrum", NULL, "1997-2010",
	"Metal Man (Russian) (48K) (HB)\0", NULL, "Oleg Origin", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_RUNGUN | GBF_PLATFORM, 0,
	SpectrumGetZipName, SpecMetalmanRomInfo, SpecMetalmanRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	SpecInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Mighty Final Fight (128K) (HB)

static struct BurnRomInfo SpecMightyffRomDesc[] = {
	{ "Mighty Final Fight (2018)(SaNchez).tap", 302096, 0x1697a6ed, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecMightyff, SpecMightyff, Spec128)
STD_ROM_FN(SpecMightyff)

struct BurnDriver BurnSpecMightyff = {
	"spec_mightyff", NULL, "spec_spec128", NULL, "2018",
	"Mighty Final Fight (128K) (HB)\0", "Q, A and SPACE to select options in title screen", "SaNchez", "ZX Spectrum",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_SPECTRUM, GBF_SCRFIGHT, 0,
	SpectrumGetZipName, SpecMightyffRomInfo, SpecMightyffRomName, NULL, NULL, NULL, NULL, SpecInputInfo, SpecDIPInfo,
	Spec128KInit, SpecExit, SpecFrame, SpecDraw, SpecScan,
	&SpecRecalc, 0x10, 288, 224, 4, 3
};

// Ninja Gaiden Shadow Warriors (128K) (HB)

static struct BurnRomInfo SpecNinjagaidenRomDesc[] = {
	{ "Ninja Gaiden Shadow Warriors (2018)(Jerri, DaRkHoRaCe, Diver).tap", 103645, 0x44081e87, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(SpecNinjagaiden, SpecNinjagaiden, Spec128)
STD_ROM_FN(SpecNinjagaiden)

struct BurnDriver BurnSpecNinjagaiden = {
	"spec_ninjagaiden", NULL, "spec_spec128", NULL, "2018",
	"Ninja Gaiden Shadow Warriors (128K) (HB)\0", NULL, "Jerri, DaRkHoRaCe, Di