// FBNEO NES/Famicom and FDS emulator
// (c)2019-2020 dink, kev, iq_132

#include "tiles_generic.h"
#include "bitswap.h"
#include "m6502_intf.h"
#include "nes_apu.h"
#include "ay8910.h"			// mapper 69 (sunsoft / gimmick!)
#include "burn_ym2413.h"	// mapper 85 (vrc7 / lagrange point)
#include "burn_gun.h"		// zapper games
#include "burn_led.h"		// fds disk status
#if !defined (_MSC_VER)
#include <unistd.h>         // for unlink()/rename() in ips_patch()/ips_make()
#endif
#include <errno.h>          // .. unlink()/rename()


static UINT8 DrvReset = 0;
static UINT32 DrvPalette[(0x40 * 8) /*nes*/ + 0x100 /*icons*/];
static UINT8 NESRecalc;

static UINT8 NESJoy1[8]     = { 0, 0, 0, 0, 0, 0, 0, 0 };
static UINT8 NESJoy2[8]     = { 0, 0, 0, 0, 0, 0, 0, 0 };
static UINT8 NESJoy3[8]     = { 0, 0, 0, 0, 0, 0, 0, 0 };
static UINT8 NESJoy4[8]     = { 0, 0, 0, 0, 0, 0, 0, 0 };
static UINT8 NESCoin[8]     = { 0, 0, 0, 0, 0, 0, 0, 0 };
static UINT8 DrvInputs[4]   = { 0, 0, 0, 0 };
static UINT8 DrvDips[4]     = { 0, 0, 0, 0 };

static UINT32 JoyShifter[2] = { 0, 0 };
static UINT8 JoyStrobe      = 0;

// Zapper emulation
static INT16 ZapperX;
static INT16 ZapperY;
static UINT8 ZapperFire;
static UINT8 ZapperReload;
static UINT8 ZapperReloadTimer;

// FDS-Buttons
static UINT8 FDSEject;
static UINT8 FDSSwap;

// FDS-Timer (for insert/swap robot)
static UINT32 FDSFrameCounter;
static UINT32 FDSFrameAction;
static UINT32 FDSFrameDiskIcon;

// game-specific options not found in header.  see bottom of cartridge_load()
static UINT32 NESMode = 0;
#define NO_WORKRAM		0x0001 // 6000-7fff reads data openbus
#define BUS_CONFLICTS	0x0002 // rom conflicts with writes, needs special handling
#define ALT_MMC3        0x0004 // alternate mmc3 scanline timing
#define BAD_HOMEBREW    0x0008 // writes OAMDATA while rendering (can't possibly work on a real NES/FC)
#define USE_4SCORE      0x0100 // 4-Player device (NES)
#define USE_HORI4P      0x0200 // 4-Player device Mode 2 (Famicom)
#define USE_ZAPPER      0x0400 // Zapper Gun device
#define MAPPER_NOCLEAR  0x0800 // preserve mapper regs on soft-reset
#define IS_PAL          0x1000 // PAL-mode (not fully supported)
#define IS_FDS          0x2000 // Famicom Disk System
#define SHOW_OVERSCAN   0x4000 // - for debugging -
#define ALT_TIMING      0x8000 // for games that use "BIT PPUSTATUS; BIT PPUSTATUS; BPL -"
							   // Assimilate, Star Wars, full_palette.nes, etc.
#define ALT_TIMING2     0x0080 // Don Doko Don 2 doesn't like the nmi delay that gunnac, b-wings, etc needs.
#define VS_ZAPPER		0x0010 // VS. UniSystem Zapper
#define VS_REVERSED     0x0020 // VS. p1/p2 -> p2/p1 (inputs swapped)
#define RAM_RANDOM      0x0040 // Init. ram w/random bytes (Go! Dizzy Go!)

// Usually for Multi-Cart mappers
static UINT32 RESETMode = 0;
#define RESET_POWER     0x0001
#define RESET_BUTTON    0x0002

// PPU types
enum {
	RP2C02 = 0, // NES/FC
	RP2C03,     // VS. (... and below!)
	RP2C04A,
	RP2C04B,
	RP2C04C,
	RP2C04D,
	RP2C05A,
	RP2C05B,
	RP2C05C,
	RP2C05D,
	RP2C05E
};

static const UINT8 PPUTypes[][6] = {
	{ "2C02" },     // NES/FC
	{ "2C03" },     // VS. (... and below!)
	{ "2C04A" },
	{ "2C04B" },
	{ "2C04C" },
	{ "2C04D" },
	{ "2C05A" },
	{ "2C05B" },
	{ "2C05C" },
	{ "2C05D" },
	{ "2C05E" }
};

static INT32 PPUType;

static struct BurnInputInfo NESInputList[] =
{
	{"P1 Select",	  BIT_DIGITAL,   NESJoy1 + 2, "p1 select" },
	{"P1 Start",	  BIT_DIGITAL,   NESJoy1 + 3, "p1 start"  },
	{"P1 Up",	  	  BIT_DIGITAL,   NESJoy1 + 4, "p1 up"     },
	{"P1 Down",	  	  BIT_DIGITAL,   NESJoy1 + 5, "p1 down"   },
	{"P1 Left",		  BIT_DIGITAL,   NESJoy1 + 6, "p1 left"   },
	{"P1 Right",	  BIT_DIGITAL,   NESJoy1 + 7, "p1 right"  },
	{"P1 Button B",	  BIT_DIGITAL,   NESJoy1 + 1, "p1 fire 1" },
	{"P1 Button A",	  BIT_DIGITAL,   NESJoy1 + 0, "p1 fire 2" },

	{"P2 Select",	  BIT_DIGITAL,   NESJoy2 + 2, "p2 select" },
	{"P2 Start",	  BIT_DIGITAL,   NESJoy2 + 3, "p2 start"  },
	{"P2 Up",	  	  BIT_DIGITAL,   NESJoy2 + 4, "p2 up"     },
	{"P2 Down",	  	  BIT_DIGITAL,   NESJoy2 + 5, "p2 down"   },
	{"P2 Left",		  BIT_DIGITAL,   NESJoy2 + 6, "p2 left"   },
	{"P2 Right",	  BIT_DIGITAL,   NESJoy2 + 7, "p2 right"  },
	{"P2 Button B",	  BIT_DIGITAL,   NESJoy2 + 1, "p2 fire 1" },
	{"P2 Button A",	  BIT_DIGITAL,   NESJoy2 + 0, "p2 fire 2" },

	{"Reset", 		  BIT_DIGITAL,   &DrvReset  , "reset"     },
	{"Dip A",		  BIT_DIPSWITCH, DrvDips + 0, "dip"       },
	{"Dip B",		  BIT_DIPSWITCH, DrvDips + 1, "dip"       },
	{"Dip C",		  BIT_DIPSWITCH, DrvDips + 2, "dip"       },
};

STDINPUTINFO(NES)

static struct BurnInputInfo NESVSInputList[] =
{
	{"P1 Coin",	  	  BIT_DIGITAL,   NESCoin + 0, "p1 coin"   },
	{"P1 Select",	  BIT_DIGITAL,   NESJoy1 + 3, "p1 select" },
	{"P1 Start",	  BIT_DIGITAL,   NESJoy1 + 2, "p1 start"  },
	{"P1 Up",	  	  BIT_DIGITAL,   NESJoy1 + 4, "p1 up"     },
	{"P1 Down",	  	  BIT_DIGITAL,   NESJoy1 + 5, "p1 down"   },
	{"P1 Left",		  BIT_DIGITAL,   NESJoy1 + 6, "p1 left"   },
	{"P1 Right",	  BIT_DIGITAL,   NESJoy1 + 7, "p1 right"  },
	{"P1 Button B",	  BIT_DIGITAL,   NESJoy1 + 1, "p1 fire 1" },
	{"P1 Button A",	  BIT_DIGITAL,   NESJoy1 + 0, "p1 fire 2" },

	{"P2 Select",	  BIT_DIGITAL,   NESJoy2 + 3, "p2 select" },
	{"P2 Start",	  BIT_DIGITAL,   NESJoy2 + 2, "p2 start"  },
	{"P2 Up",	  	  BIT_DIGITAL,   NESJoy2 + 4, "p2 up"     },
	{"P2 Down",	  	  BIT_DIGITAL,   NESJoy2 + 5, "p2 down"   },
	{"P2 Left",		  BIT_DIGITAL,   NESJoy2 + 6, "p2 left"   },
	{"P2 Right",	  BIT_DIGITAL,   NESJoy2 + 7, "p2 right"  },
	{"P2 Button B",	  BIT_DIGITAL,   NESJoy2 + 1, "p2 fire 1" },
	{"P2 Button A",	  BIT_DIGITAL,   NESJoy2 + 0, "p2 fire 2" },

	{"Reset", 		  BIT_DIGITAL,   &DrvReset  , "reset"     },
	{"Dip A",		  BIT_DIPSWITCH, DrvDips + 0, "dip"       },
	{"Dip B",		  BIT_DIPSWITCH, DrvDips + 1, "dip"       },
	{"Dip C",		  BIT_DIPSWITCH, DrvDips + 2, "dip"       },
	{"Dip D",		  BIT_DIPSWITCH, DrvDips + 3, "dip"       },
};

STDINPUTINFO(NESVS)

#define A(a, b, c, d) {a, b, (UINT8*)(c), d}
static struct BurnInputInfo NESZapperInputList[] =
{
	{"P1 Select",	  BIT_DIGITAL,   NESJoy1 + 2, "p1 select" },
	{"P1 Start",	  BIT_DIGITAL,   NESJoy1 + 3, "p1 start"  },
	{"P1 Up",	  	  BIT_DIGITAL,   NESJoy1 + 4, "p1 up"     },
	{"P1 Down",	  	  BIT_DIGITAL,   NESJoy1 + 5, "p1 down"   },
	{"P1 Left",		  BIT_DIGITAL,   NESJoy1 + 6, "p1 left"   },
	{"P1 Right",	  BIT_DIGITAL,   NESJoy1 + 7, "p1 right"  },
	{"P1 Button B",	  BIT_DIGITAL,   NESJoy1 + 1, "p1 fire 1" },
	{"P1 Button A",	  BIT_DIGITAL,   NESJoy1 + 0, "p1 fire 2" },

	A("P2 Gun X",     BIT_ANALOG_REL, &ZapperX,   "p2 x-axis" ),
	A("P2 Gun Y",     BIT_ANALOG_REL, &ZapperY,   "p2 y-axis" ),
	{"P2 Fire",       BIT_DIGITAL,  &ZapperFire,  "p2 fire 1" },
	{"P2 Reload",     BIT_DIGITAL,  &ZapperReload,"p2 fire 2" },

	{"Reset", 		  BIT_DIGITAL,   &DrvReset  , "reset"     },
	{"Dip A",		  BIT_DIPSWITCH, DrvDips + 0, "dip"       },
	{"Dip B",		  BIT_DIPSWITCH, DrvDips + 1, "dip"       },
	{"Dip C",		  BIT_DIPSWITCH, DrvDips + 2, "dip"       },
};

STDINPUTINFO(NESZapper)

static struct BurnInputInfo NESVSZapperInputList[] =
{
	{"P1 Coin",	  	  BIT_DIGITAL,   NESCoin + 0, "p1 coin"   },

	A("P1 Gun X",     BIT_ANALOG_REL, &ZapperX,   "p1 x-axis" ),
	A("P1 Gun Y",     BIT_ANALOG_REL, &ZapperY,   "p1 y-axis" ),
	{"P1 Fire",       BIT_DIGITAL,  &ZapperFire,  "p1 fire 1" },
	{"P1 Reload",     BIT_DIGITAL,  &ZapperReload,"p1 fire 2" },

	{"Reset", 		  BIT_DIGITAL,   &DrvReset  , "reset"     },
	{"Dip A",		  BIT_DIPSWITCH, DrvDips + 0, "dip"       },
	{"Dip B",		  BIT_DIPSWITCH, DrvDips + 1, "dip"       },
	{"Dip C",		  BIT_DIPSWITCH, DrvDips + 2, "dip"       },
	{"Dip D",		  BIT_DIPSWITCH, DrvDips + 3, "dip"       },
};
#undef A
STDINPUTINFO(NESVSZapper)

static struct BurnInputInfo NES4ScoreInputList[] =
{
	{"P1 Select",	  BIT_DIGITAL,   NESJoy1 + 2, "p1 select" },
	{"P1 Start",	  BIT_DIGITAL,   NESJoy1 + 3, "p1 start"  },
	{"P1 Up",	  	  BIT_DIGITAL,   NESJoy1 + 4, "p1 up"     },
	{"P1 Down",	  	  BIT_DIGITAL,   NESJoy1 + 5, "p1 down"   },
	{"P1 Left",		  BIT_DIGITAL,   NESJoy1 + 6, "p1 left"   },
	{"P1 Right",	  BIT_DIGITAL,   NESJoy1 + 7, "p1 right"  },
	{"P1 Button B",	  BIT_DIGITAL,   NESJoy1 + 1, "p1 fire 1" },
	{"P1 Button A",	  BIT_DIGITAL,   NESJoy1 + 0, "p1 fire 2" },

	{"P2 Select",	  BIT_DIGITAL,   NESJoy2 + 2, "p2 select" },
	{"P2 Start",	  BIT_DIGITAL,   NESJoy2 + 3, "p2 start"  },
	{"P2 Up",	  	  BIT_DIGITAL,   NESJoy2 + 4, "p2 up"     },
	{"P2 Down",	  	  BIT_DIGITAL,   NESJoy2 + 5, "p2 down"   },
	{"P2 Left",		  BIT_DIGITAL,   NESJoy2 + 6, "p2 left"   },
	{"P2 Right",	  BIT_DIGITAL,   NESJoy2 + 7, "p2 right"  },
	{"P2 Button B",	  BIT_DIGITAL,   NESJoy2 + 1, "p2 fire 1" },
	{"P2 Button A",	  BIT_DIGITAL,   NESJoy2 + 0, "p2 fire 2" },

	{"P3 Select",	  BIT_DIGITAL,   NESJoy3 + 2, "p3 select" },
	{"P3 Start",	  BIT_DIGITAL,   NESJoy3 + 3, "p3 start"  },
	{"P3 Up",	  	  BIT_DIGITAL,   NESJoy3 + 4, "p3 up"     },
	{"P3 Down",	  	  BIT_DIGITAL,   NESJoy3 + 5, "p3 down"   },
	{"P3 Left",		  BIT_DIGITAL,   NESJoy3 + 6, "p3 left"   },
	{"P3 Right",	  BIT_DIGITAL,   NESJoy3 + 7, "p3 right"  },
	{"P3 Button B",	  BIT_DIGITAL,   NESJoy3 + 1, "p3 fire 1" },
	{"P3 Button A",	  BIT_DIGITAL,   NESJoy3 + 0, "p3 fire 2" },

	{"P4 Select",	  BIT_DIGITAL,   NESJoy4 + 2, "p4 select" },
	{"P4 Start",	  BIT_DIGITAL,   NESJoy4 + 3, "p4 start"  },
	{"P4 Up",	  	  BIT_DIGITAL,   NESJoy4 + 4, "p4 up"     },
	{"P4 Down",	  	  BIT_DIGITAL,   NESJoy4 + 5, "p4 down"   },
	{"P4 Left",		  BIT_DIGITAL,   NESJoy4 + 6, "p4 left"   },
	{"P4 Right",	  BIT_DIGITAL,   NESJoy4 + 7, "p4 right"  },
	{"P4 Button B",	  BIT_DIGITAL,   NESJoy4 + 1, "p4 fire 1" },
	{"P4 Button A",	  BIT_DIGITAL,   NESJoy4 + 0, "p4 fire 2" },

	{"Reset", 		  BIT_DIGITAL,   &DrvReset  , "reset"     },
	{"Dip A",		  BIT_DIPSWITCH, DrvDips + 0, "dip"       },
	{"Dip B",		  BIT_DIPSWITCH, DrvDips + 1, "dip"       },
	{"Dip C",		  BIT_DIPSWITCH, DrvDips + 2, "dip"       },
};

STDINPUTINFO(NES4Score)

static struct BurnInputInfo NESFDSInputList[] =
{
	{"P1 Select",	  BIT_DIGITAL,   NESJoy1 + 2, "p1 select" },
	{"P1 Start",	  BIT_DIGITAL,   NESJoy1 + 3, "p1 start"  },
	{"P1 Up",	  	  BIT_DIGITAL,   NESJoy1 + 4, "p1 up"     },
	{"P1 Down",	  	  BIT_DIGITAL,   NESJoy1 + 5, "p1 down"   },
	{"P1 Left",		  BIT_DIGITAL,   NESJoy1 + 6, "p1 left"   },
	{"P1 Right",	  BIT_DIGITAL,   NESJoy1 + 7, "p1 right"  },
	{"P1 Button B",	  BIT_DIGITAL,   NESJoy1 + 1, "p1 fire 1" },
	{"P1 Button A",	  BIT_DIGITAL,   NESJoy1 + 0, "p1 fire 2" },

	{"P2 Microphone", BIT_DIGITAL,   NESJoy2 + 2, "p2 select" },
	{"P2 Up",	  	  BIT_DIGITAL,   NESJoy2 + 4, "p2 up"     },
	{"P2 Down",	  	  BIT_DIGITAL,   NESJoy2 + 5, "p2 down"   },
	{"P2 Left",		  BIT_DIGITAL,   NESJoy2 + 6, "p2 left"   },
	{"P2 Right",	  BIT_DIGITAL,   NESJoy2 + 7, "p2 right"  },
	{"P2 Button B",	  BIT_DIGITAL,   NESJoy2 + 1, "p2 fire 1" },
	{"P2 Button A",	  BIT_DIGITAL,   NESJoy2 + 0, "p2 fire 2" },

	{"Swap Disk Side",	  BIT_DIGITAL, &FDSSwap,  "swap"      },
	{"Eject/Insert Disk", BIT_DIGITAL, &FDSEject, "eject"     },

	{"Reset", 		  BIT_DIGITAL,   &DrvReset  , "reset"     },
	{"Dip A",		  BIT_DIPSWITCH, DrvDips + 0, "dip"       },
	{"Dip B",		  BIT_DIPSWITCH, DrvDips + 1, "dip"       },
	{"Dip C",		  BIT_DIPSWITCH, DrvDips + 2, "dip"       },
};

STDINPUTINFO(NESFDS)

static struct BurnDIPInfo NESDIPList[] =
{
	DIP_OFFSET(0x11)
	{0x00, 0xff, 0xff, 0x00, NULL			},
	{0x01, 0xff, 0xff, 0x00, NULL			},
	{0x02, 0xff, 0xff, 0x00, NULL			},

	{0   , 0xfe, 0   ,    2, "Sprite Limit"	},
	{0x00, 0x01, 0x01, 0x01, "Off"			},
	{0x00, 0x01, 0x01, 0x00, "On"			},

	{0   , 0xfe, 0   ,    2, "Aspect Ratio"	},
	{0x01, 0x01, 0x01, 0x00, "Off"			},
	{0x01, 0x01, 0x01, 0x01, "4:3"			},

	{0   , 0xfe, 0   ,    2, "Stereoizer"	},
	{0x01, 0x01, 0x02, 0x00, "Off"			},
	{0x01, 0x01, 0x02, 0x02, "On"			},

	{0   , 0xfe, 0   ,    2, "DMC Bit-Reversal"	},
	{0x01, 0x01, 0x08, 0x00, "Off"			},
	{0x01, 0x01, 0x08, 0x08, "On"			},

	{0   , 0xfe, 0   ,    2, "Palette"		},
	{0x02, 0x01, 0x01, 0x00, "Famicom"		},
	{0x02, 0x01, 0x01, 0x01, "RGB"			},
};

STDDIPINFO(NES)

static struct BurnDIPInfo NESZapperDIPList[] =
{
	DIP_OFFSET(0x0d)
	{0x00, 0xff, 0xff, 0x00, NULL			},
	{0x01, 0xff, 0xff, 0x00, NULL			},
	{0x02, 0xff, 0xff, 0x00, NULL			},

	{0   , 0xfe, 0   ,    2, "Sprite Limit"	},
	{0x00, 0x01, 0x01, 0x01, "Off"			},
	{0x00, 0x01, 0x01, 0x00, "On"			},

	{0   , 0xfe, 0   ,    2, "Aspect Ratio"	},
	{0x01, 0x01, 0x01, 0x00, "Off"			},
	{0x01, 0x01, 0x01, 0x01, "4:3"			},

	{0   , 0xfe, 0   ,    2, "Stereoizer"	},
	{0x01, 0x01, 0x02, 0x00, "Off"			},
	{0x01, 0x01, 0x02, 0x02, "On"			},

	{0   , 0xfe, 0   ,    2, "DMC Bit-Reversal"	},
	{0x01, 0x01, 0x08, 0x00, "Off"			},
	{0x01, 0x01, 0x08, 0x08, "On"			},

	{0   , 0xfe, 0   ,    2, "Palette"		},
	{0x02, 0x01, 0x01, 0x00, "Famicom"		},
	{0x02, 0x01, 0x01, 0x01, "RGB"			},
};

STDDIPINFO(NESZapper)

static struct BurnDIPInfo NESVS_DIPList[] =    		// VS. Unisystem
{
	DIP_OFFSET(0x12)

	{0   , 0xfe, 0   ,    2, "Sprite Limit"	},
	{0x00, 0x01, 0x01, 0x01, "Off"			},
	{0x00, 0x01, 0x01, 0x00, "On"			},

	{0   , 0xfe, 0   ,    2, "Aspect Ratio"	},
	{0x01, 0x01, 0x01, 0x00, "Off"			},
	{0x01, 0x01, 0x01, 0x01, "4:3"			},

	{0   , 0xfe, 0   ,    2, "Stereoizer"	},
	{0x01, 0x01, 0x02, 0x00, "Off"			},
	{0x01, 0x01, 0x02, 0x02, "On"			},

	{0   , 0xfe, 0   ,    2, "DMC Bit-Reversal"	},
	{0x01, 0x01, 0x08, 0x00, "Off"			},
	{0x01, 0x01, 0x08, 0x08, "On"			},

	{0   , 0xfe, 0   ,    2, "Dip 1"		},
	{0x03, 0x01, 0x01, 0x00, "Off"			},
	{0x03, 0x01, 0x01, 0x01, "On"			},

	{0   , 0xfe, 0   ,    2, "Dip 2"		},
	{0x03, 0x01, 0x02, 0x00, "Off"			},
	{0x03, 0x01, 0x02, 0x02, "On"			},

	{0   , 0xfe, 0   ,    2, "Dip 3"		},
	{0x03, 0x01, 0x04, 0x00, "Off"			},
	{0x03, 0x01, 0x04, 0x04, "On"			},

	{0   , 0xfe, 0   ,    2, "Dip 4"		},
	{0x03, 0x01, 0x08, 0x00, "Off"			},
	{0x03, 0x01, 0x08, 0x08, "On"			},

	{0   , 0xfe, 0   ,    2, "Dip 5"		},
	{0x03, 0x01, 0x10, 0x00, "Off"			},
	{0x03, 0x01, 0x10, 0x10, "On"			},

	{0   , 0xfe, 0   ,    2, "Dip 6"		},
	{0x03, 0x01, 0x20, 0x00, "Off"			},
	{0x03, 0x01, 0x20, 0x20, "On"			},

	{0   , 0xfe, 0   ,    2, "Dip 7"		},
	{0x03, 0x01, 0x40, 0x00, "Off"			},
	{0x03, 0x01, 0x40, 0x40, "On"			},

	{0   , 0xfe, 0   ,    2, "Dip 8"		},
	{0x03, 0x01, 0x80, 0x00, "Off"			},
	{0x03, 0x01, 0x80, 0x80, "On"			},
};

static struct BurnDIPInfo NESVS_DefaultsDIPList[] = // VS. Unisystem, default config
{
	{0x00, 0xff, 0xff, 0x00, NULL			},
	{0x01, 0xff, 0xff, 0x00, NULL			},
	{0x02, 0xff, 0xff, 0x00, NULL			},
	{0x03, 0xff, 0xff, 0x00, NULL			},
};

static struct BurnDIPInfo vsdip_bit_1_DIPList[]=
{
	{0x00, 0xff, 0xff, 0x00, NULL			},
	{0x01, 0xff, 0xff, 0x00, NULL			},
	{0x02, 0xff, 0xff, 0x00, NULL			},
	{0x03, 0xff, 0xff, 0x02, NULL			},
};

static struct BurnDIPInfo vsdip_bit_0_DIPList[]=
{
	{0x00, 0xff, 0xff, 0x00, NULL			},
	{0x01, 0xff, 0xff, 0x00, NULL			},
	{0x02, 0xff, 0xff, 0x00, NULL			},
	{0x03, 0xff, 0xff, 0x01, NULL			},
};

STDDIPINFOEXT(NESVS, NESVS_Defaults, NESVS_)  // NESVSDipInfo = NESVS_ + NESVS_Defaults
STDDIPINFOEXT(NESVSFromBelow, vsdip_bit_1_, NESVS_)  // vs. from below defaults
STDDIPINFOEXT(NESVSBit0High, vsdip_bit_0_, NESVS_)  // vs. pinball, stroke & match golf defaults

static struct BurnDIPInfo NESVSZapperDIPList[] =	// VS. Unisystem
{
	DIP_OFFSET(0x06)
	{0x00, 0xff, 0xff, 0x00, NULL			},
	{0x01, 0xff, 0xff, 0x00, NULL			},
	{0x02, 0xff, 0xff, 0x00, NULL			},
	{0x03, 0xff, 0xff, 0x00, NULL			},

	{0   , 0xfe, 0   ,    2, "Sprite Limit"	},
	{0x00, 0x01, 0x01, 0x01, "Off"			},
	{0x00, 0x01, 0x01, 0x00, "On"			},

	{0   , 0xfe, 0   ,    2, "Aspect Ratio"	},
	{0x01, 0x01, 0x01, 0x00, "Off"			},
	{0x01, 0x01, 0x01, 0x01, "4:3"			},

	{0   , 0xfe, 0   ,    2, "Stereoizer"	},
	{0x01, 0x01, 0x02, 0x00, "Off"			},
	{0x01, 0x01, 0x02, 0x02, "On"			},

	{0   , 0xfe, 0   ,    2, "DMC Bit-Reversal"	},
	{0x01, 0x01, 0x08, 0x00, "Off"			},
	{0x01, 0x01, 0x08, 0x08, "On"			},

	{0   , 0xfe, 0   ,    2, "Dip 1"		},
	{0x03, 0x01, 0x01, 0x00, "Off"			},
	{0x03, 0x01, 0x01, 0x01, "On"			},

	{0   , 0xfe, 0   ,    2, "Dip 2"		},
	{0x03, 0x01, 0x02, 0x00, "Off"			},
	{0x03, 0x01, 0x02, 0x02, "On"			},

	{0   , 0xfe, 0   ,    2, "Dip 3"		},
	{0x03, 0x01, 0x04, 0x00, "Off"			},
	{0x03, 0x01, 0x04, 0x04, "On"			},

	{0   , 0xfe, 0   ,    2, "Dip 4"		},
	{0x03, 0x01, 0x08, 0x00, "Off"			},
	{0x03, 0x01, 0x08, 0x08, "On"			},

	{0   , 0xfe, 0   ,    2, "Dip 5"		},
	{0x03, 0x01, 0x10, 0x00, "Off"			},
	{0x03, 0x01, 0x10, 0x10, "On"			},

	{0   , 0xfe, 0   ,    2, "Dip 6"		},
	{0x03, 0x01, 0x20, 0x00, "Off"			},
	{0x03, 0x01, 0x20, 0x20, "On"			},

	{0   , 0xfe, 0   ,    2, "Dip 7"		},
	{0x03, 0x01, 0x40, 0x00, "Off"			},
	{0x03, 0x01, 0x40, 0x40, "On"			},

	{0   , 0xfe, 0   ,    2, "Dip 8"		},
	{0x03, 0x01, 0x80, 0x00, "Off"			},
	{0x03, 0x01, 0x80, 0x80, "On"			},
};

STDDIPINFO(NESVSZapper)

static struct BurnDIPInfo NES4ScoreDIPList[] =
{
	DIP_OFFSET(0x21)
	{0x00, 0xff, 0xff, 0x00, NULL			},
	{0x01, 0xff, 0xff, 0x00, NULL			},
	{0x02, 0xff, 0xff, 0x00, NULL			},

	{0   , 0xfe, 0   ,    2, "Sprite Limit"	},
	{0x00, 0x01, 0x01, 0x01, "Off"			},
	{0x00, 0x01, 0x01, 0x00, "On"			},

	{0   , 0xfe, 0   ,    2, "Aspect Ratio"	},
	{0x01, 0x01, 0x01, 0x00, "Off"			},
	{0x01, 0x01, 0x01, 0x01, "4:3"			},

	{0   , 0xfe, 0   ,    2, "Stereoizer"	},
	{0x01, 0x01, 0x02, 0x00, "Off"			},
	{0x01, 0x01, 0x02, 0x02, "On"			},

	{0   , 0xfe, 0   ,    2, "DMC Bit-Reversal"	},
	{0x01, 0x01, 0x08, 0x00, "Off"			},
	{0x01, 0x01, 0x08, 0x08, "On"			},

	{0   , 0xfe, 0   ,    2, "Palette"		},
	{0x02, 0x01, 0x01, 0x00, "Famicom"		},
	{0x02, 0x01, 0x01, 0x01, "RGB"			},
};

STDDIPINFO(NES4Score)

static struct BurnDIPInfo NESFDSDIPList[] =
{
	DIP_OFFSET(0x12)
	{0x00, 0xff, 0xff, 0x00, NULL			},
	{0x01, 0xff, 0xff, 0x00, NULL			},
	{0x02, 0xff, 0xff, 0x00, NULL			},

	{0   , 0xfe, 0   ,    2, "Sprite Limit"	},
	{0x00, 0x01, 0x01, 0x01, "Off"			},
	{0x00, 0x01, 0x01, 0x00, "On"			},

	{0   , 0xfe, 0   ,    2, "Auto Insert/Eject" },
	{0x00, 0x01, 0x02, 0x02, "Off"			},
	{0x00, 0x01, 0x02, 0x00, "On"			},

	{0   , 0xfe, 0   ,    2, "Aspect Ratio"	},
	{0x01, 0x01, 0x01, 0x00, "Off"			},
	{0x01, 0x01, 0x01, 0x01, "4:3"			},

	{0   , 0xfe, 0   ,    2, "Stereoizer"	},
	{0x01, 0x01, 0x02, 0x00, "Off"			},
	{0x01, 0x01, 0x02, 0x02, "On"			},

	{0   , 0xfe, 0   ,    2, "DMC Bit-Reversal"	},
	{0x01, 0x01, 0x08, 0x00, "Off"			},
	{0x01, 0x01, 0x08, 0x08, "On"			},

	{0   , 0xfe, 0   ,    2, "Palette"		},
	{0x02, 0x01, 0x01, 0x00, "Famicom"		},
	{0x02, 0x01, 0x01, 0x01, "RGB"			},
};

STDDIPINFO(NESFDS)

#define SCREEN_WIDTH 256
#define SCREEN_HEIGHT (240-16) // top and bottom 8px is overscan
#define SCREEN_HEIGHT_PAL (SCREEN_HEIGHT + 8)

static UINT8* NES_CPU_RAM;

static UINT8* rom;
static INT32 cyc_counter = 0; // frame-based cycle counter
static UINT64 mega_cyc_counter = 0; // "from reset"-based cycle counter

static UINT8 cpu_open_bus;
static INT32 scanline, pixel;
static UINT8 (*read_nt)(UINT16 address);
static void (*write_nt)(UINT16 address, UINT8 data);

enum Scanline  { VISIBLE = 0, POSTRENDER, VBLANK, PRERENDER };
enum Mirroring { VERTICAL = 0, HORIZONTAL, SINGLE_LOW, SINGLE_HIGH, FOUR_SCREEN, SACHEN };

// Reference: https://wiki.nesdev.com/w/index.php?title=PPU_registers

union PPUCTRL
{
#ifdef LSB_FIRST
	struct {
		UINT8 nttab   : 2;  // Base nametable address (0 = $2000; 1 = $2400; 2 = $2800; 3 = $2C00)
		UINT8 incr    : 1;  // VRAM address increment (0 = 1, 1 = 32)
		UINT8 sprtab  : 1;  // Sprite Pattern table address for 8x8 sprites
		UINT8 bgtab   : 1;  // Background pattern table address
		UINT8 sprsize : 1;  // Sprite size, 8x8 / 8x16
		UINT8 slave   : 1;  // PPU master/slave select
		UINT8 nmi     : 1;  // Generate NMI @ VBL
	} bit;
	UINT8 reg;
#else
	UINT8 reg;
	struct {
		UINT8 nmi     : 1;  // Generate NMI @ VBL
		UINT8 slave   : 1;  // PPU master/slave select
		UINT8 sprsize : 1;  // Sprite size, 8x8 / 8x16
		UINT8 bgtab   : 1;  // Background pattern table address
		UINT8 sprtab  : 1;  // Sprite Pattern table address for 8x8 sprites
		UINT8 incr    : 1;  // VRAM address increment (0 = 1, 1 = 32)
		UINT8 nttab   : 2;  // Base nametable address (0 = $2000; 1 = $2400; 2 = $2800; 3 = $2C00)
	} bit;
#endif
};

union PPUMASK
{
#ifdef LSB_FIRST
	struct {
		UINT8 gray    : 1;  // Grayscale
		UINT8 bgLeft  : 1;  // Show background in left 8 px of screen
		UINT8 sprLeft : 1;  // Show sprite in left 8 px
		UINT8 bg      : 1;  // Show background
		UINT8 spr     : 1;  // Show sprites
		UINT8 red     : 1;  // Emphasize red
		UINT8 green   : 1;  // Emphasize green
		UINT8 blue    : 1;  // Emphasize blue
	} bit;
	UINT8 reg;
#else
	UINT8 reg;
	struct {
		UINT8 blue    : 1;  // Emphasize blue
		UINT8 green   : 1;  // Emphasize green
		UINT8 red     : 1;  // Emphasize red
		UINT8 spr     : 1;  // Show sprites
		UINT8 bg      : 1;  // Show background
		UINT8 sprLeft : 1;  // Show sprite in left 8 px
		UINT8 bgLeft  : 1;  // Show background in left 8 px of screen
		UINT8 gray    : 1;  // Grayscale
	} bit;
#endif
};

union PPUSTATUS
{
#ifdef LSB_FIRST
	struct {
		UINT8 bus     : 5;  // Open bus
		UINT8 spovrf  : 1;  // Sprite overflow (8+ sprites/scanline)
		UINT8 sp0hit  : 1;  // Sprite 0 Hit
		UINT8 VBlank  : 1;  // Vertical blanking
	} bit;
	UINT8 reg;
#else
	UINT8 reg;
	struct {
		UINT8 VBlank  : 1;  // Vertical blanking
		UINT8 sp0hit  : 1;  // Sprite 0 Hit
		UINT8 spovrf  : 1;  // Sprite overflow (8+ sprites/scanline)
		UINT8 bus     : 5;  // Open bus
	} bit;
#endif
};

struct OAMBUF
{
	UINT8 idx;   // # in OAM
	UINT8 x;     // following 4 bytes (x, y, tile, attr) mirror oam_ram
	UINT8 y;
	UINT8 tile;
	UINT8 attr;
	UINT8 tileL; // tile data from CHR
	UINT8 tileH;
};

struct cartridge {
	UINT8	*PRGRom;
	INT32	 PRGRomSize;
	INT32	 PRGRomMask;

	UINT8	*WorkRAM;
	INT32	 WorkRAMSize;
	INT32	 WorkRAMMask;
	INT32	 BatteryBackedSRAM;

	UINT8	*CHRRom;
	INT32	 CHRRomSize;
	UINT8	*CHRRam;
	INT32	 CHRRamSize;

	UINT8   *ExtData;
	INT32    ExtDataSize;

	INT32	 Mapper;
	INT32	 SubMapper;
	INT32	 Mirroring;
	INT32	 Trainer;
	UINT32	 Crc;

	INT32	 FDSMode;
	INT32	 FDSDiskSides;
	UINT8	*FDSDiskRawOrig;	// FDS Disk loaded (unmodified)
	INT32    FDSDiskRawOrigSize;

	UINT8	*FDSDiskRaw;		// FDS Disk loaded (possibly modified)
	INT32    FDSDiskRawSize;
	UINT8	*FDSDisk;			// Same as above, but ptr adjusted to skip the header (if avail.)
	INT32    FDSDiskSize;
};

static cartridge Cart;

// forward decl's (used in mappers)
static void set_mirroring(INT32 mode);
static void nametable_map(INT32 nt, INT32 ntbank);
static void nametable_mapraw(INT32 nt, UINT8 *ntraw, UINT8 type);
static void nametable_mapall(INT32 ntbank0, INT32 ntbank1, INT32 ntbank2, INT32 ntbank3);

enum { MEM_RAM = 0, MEM_RAM_RO = 1, MEM_ROM = 2 };
static void mapper_set_chrtype(INT32 type);

static INT32 mapper_init(INT32 mappernum);
static UINT8 (*mapper_prg_read)(UINT16 addr);
static UINT8 mapper_prg_read_int(UINT16 address); // mapper_prg_read points here by default
static void (*mapper_write)(UINT16 address, UINT8 data) = NULL;
static void (*mapper_map)() = NULL;
static void (*mapper_scanline)() = NULL;                // once per scanline
static void (*mapper_cycle)() = NULL;                   // once per cpu-cycle
static void (*mapper_ppu_clock)(UINT16 busaddr) = NULL; // called during visible & prerender
static void (*mapper_ppu_clockall)(UINT16 busaddr) = NULL; // called during every cycle
static void (*mapper_scan_cb)() = NULL;                 // state scanning
static void (*mapper_scan_cb_nvram)() = NULL;           // state scanning (nvram)

static INT32 cartridge_load(UINT8* ROMData, UINT32 ROMSize, UINT32 ROMCRC)
{
	if (ROMData == NULL || ROMSize < 16384 ) {
		bprintf(0, _T("Invalid ROM data!\n"));
		return 1;
	}

	if (memcmp("NES\x1a", &ROMData[0], 4)) {
		bprintf(0, _T("Invalid ROM header!\n"));
		return 1;
	}

	NESMode = 0;

	INT32 nes20 = (ROMData[7] & 0xc) == 0x8;

	memset(&Cart, 0, sizeof(Cart));

	Cart.Crc = ROMCRC;
	Cart.PRGRomSize = ROMData[4] * 0x4000;
	Cart.CHRRomSize = ROMData[5] * 0x2000;

	if (Cart.Crc == 0x2a798367) {
		// JY 45-in-1 can't be represented by regular nes header.
		Cart.CHRRomSize = 128 * 0x4000;
	}

	PPUType = RP2C02;

	if (nes20 && (ROMData[7] & 0x3) == 1) {
		switch (ROMData[13] & 0x0f) {
			case 0:
			case 1:
			case 6:
			case 7: PPUType = RP2C03; break;
			case 2: PPUType = RP2C04A; break;
			case 3: PPUType = RP2C04B; break;
			case 4: PPUType = RP2C04C; break;
			case 5: PPUType = RP2C04D; break;
			case 8: PPUType = RP2C05A; break;
			case 9: PPUType = RP2C05B; break;
			case 0xa: PPUType = RP2C05C; break;
			case 0xb: PPUType = RP2C05D; break;
			case 0xc: PPUType = RP2C05E; break;
		}
		bprintf(0, _T("VS. System - PPU %S (%x)\n"), PPUTypes[PPUType], PPUType);
		ROMData[6] |= 8; // "fix to 4-screen mode" - nesdev wiki
	}

	bprintf(0, _T("PRG Size: %d\n"), Cart.PRGRomSize);
	bprintf(0, _T("CHR Size: %d\n"), Cart.CHRRomSize);

	if (ROMData[6] & 0x8)
		Cart.Mirroring = 4;
	else
		Cart.Mirroring = ROMData[6] & 1;

	switch (Cart.Mirroring) {
		case 0: set_mirroring(HORIZONTAL); break;
		case 1: set_mirroring(VERTICAL); break;
		case 4: set_mirroring(FOUR_SCREEN); break;
	}

	// Parse MAPPER iNES + NES 2.0
	Cart.Mapper = (ROMData[6] >> 4) | (ROMData[7] & 0xf0);

	if (!memcmp("DiskDude!", &ROMData[7], 9)) {
		bprintf(PRINT_ERROR, _T("``DiskDude!'' spam, ignoring upper bits of mapper.\n"));

		Cart.Mapper &= 0x0f; // remove spam from upper bits of mapper
	}

	if (nes20) {
		Cart.Mapper |= (ROMData[8] & 0x0f) << 8;
		Cart.SubMapper = (ROMData[8] & 0xf0) >> 4;

		if (Cart.Mapper & 0xf00 || Cart.SubMapper != 0)
			bprintf(0, _T("NES 2.0 Extended Mapper: %d\tSub: %d\n"), Cart.Mapper, Cart.SubMapper);
	}

	// Mapper EXT-hardware inits
	// Initted here, because mapper_init() is called on reset
	if (Cart.Mapper == 69) { // SunSoft fme-7 (5b) audio expansion - ay8910
		AY8910Init(0, 1789773 / 2, 1);
		AY8910SetAllRoutes(0, 0.70, BURN_SND_ROUTE_BOTH);
		AY8910SetBuffered(M6502TotalCycles, 1789773);
	}

	if (Cart.Mapper == 85) { // VRC7 audio expansion - YM2413
		BurnYM2413Init(3579545, 1);
		BurnYM2413SetAllRoutes(2.00, BURN_SND_ROUTE_BOTH);
	}

	Cart.Trainer = (ROMData[6] & 0x4) >> 2;
	Cart.BatteryBackedSRAM = (ROMData[6] & 0x2) >> 1;

	Cart.PRGRom = ROMData + 0x10 + (Cart.Trainer ? 0x200 : 0);

	// Default CHR-Ram size (8k), always set-up (for advanced mappers, etc)
	Cart.CHRRamSize = 0x2000;

	if (nes20) {
		// NES2.0 header specifies CHR-Ram size (Nalle Land, Haunted Halloween '86, VS. Topgun, VS. Castlevania)
		Cart.CHRRamSize = 64 << (ROMData[0xb] & 0xf);

		// Detect Extended Data (Super Russian Roulette..)
		INT32 total_nes_rom = 0x10 + (Cart.Trainer ? 0x200 : 0) + Cart.PRGRomSize + Cart.CHRRomSize;
		Cart.ExtDataSize = (ROMSize > total_nes_rom) ? (ROMSize - total_nes_rom) : 0;

		if (Cart.ExtDataSize) {
			bprintf(0, _T("%x bytes Extended Data detected\n"), Cart.ExtDataSize);
			Cart.ExtData = ROMData + total_nes_rom;
		}
	}

	if (Cart.Mapper == 30 || Cart.Mapper == 111) { // UNROM-512, GTROM(Cheapocabra) defaults to 32k chr-ram
		Cart.CHRRamSize = 0x8000;
	}

	if (Cart.Crc == 0xf0847322) {
		Cart.CHRRamSize = 0x4000; // 16k chr-ram for Videomation
	}

	if (Cart.Crc == 0xdd65a6cc) { // Street Heroes 262
		Cart.Mapper = 262;
	}

	Cart.CHRRam = (UINT8*)BurnMalloc(Cart.CHRRamSize);

	if (Cart.CHRRomSize) {
		Cart.CHRRom = Cart.PRGRom + Cart.PRGRomSize;
		mapper_set_chrtype(MEM_ROM);
		bprintf(0, _T("Using ROM-supplied CHR data\n"));
	} else {
		mapper_set_chrtype(MEM_RAM);
		bprintf(0, _T("Using RAM CHR data (%dk bytes)\n"), Cart.CHRRamSize / 1024);
	}

	// set-up Cart PRG-RAM/WORK-RAM (6000-7fff)

	Cart.WorkRAMSize = (Cart.Mapper != 5) ? 0x2000 : (8 * 0x2000);

	if (nes20) {
		// NES2.0 header specifies NV-Ram size (ex. Nova the Squirrel)
		INT32 l_shift0 = ((ROMData[0xa] & 0xf0) >> 4); // sram
		INT32 l_shift1 = ((ROMData[0xa] & 0x0f) >> 0); // work-ram
		INT32 use = (l_shift0 > l_shift1) ? l_shift0 : l_shift1; // use the larger of the 2?
		Cart.WorkRAMSize = (use == 0) ? 0 : (64 << use);
	}

	if (PPUType > RP2C02) Cart.WorkRAMSize = 0x800; // VS. 6000-7fff 2k (mirrored)

	switch (ROMCRC) {
		case 0x478a04d8:
			Cart.WorkRAMSize = 0x4000; // Genghis Khan 16k SRAM/WRAM
			break;

		case 0x6f5f3bd2:
		case 0x6f8906ad:
			Cart.WorkRAMSize = 0x8000; // Final Fantasy 1 & 2 (Japan/T-eng) 32k SRAM/WRAM
			break;
	}

	bprintf(0, _T("Cartridge RAM: %d\n"), Cart.WorkRAMSize);
	Cart.WorkRAM = (UINT8*)BurnMalloc(Cart.WorkRAMSize);
	if (Cart.WorkRAMSize == 0) NESMode |= NO_WORKRAM;

	if (Cart.Trainer) {
		// This is not a trainer in the traditional sense.  It was a little
		// block of ram on early nes/fc copy-machines to simulate certain
		// mappers with code.
		bprintf(0, _T("ROM has Trainer code, mapping @ 0x7000.\n"));
		if (Cart.WorkRAMSize == 0x2000) {
			memcpy(Cart.WorkRAM + 0x1000, ROMData + 0x10, 0x200);
		} else {
			bprintf(PRINT_ERROR, _T("Invalid WorkRam size, can't use Trainer data.\n"));
		}
	}

	// set-up MAPPER
	bprintf(0, _T("Cartridge Mapper: %d   Mirroring: "), Cart.Mapper);
	switch (Cart.Mirroring) {
		case 0: bprintf(0, _T("Horizontal")); break;
		case 1: bprintf(0, _T("Vertical")); break;
		case 4: bprintf(0, _T("Four-Screen")); break;
		default: bprintf(0, _T("???")); break;
	}
	bprintf(0, _T(" (%d)\n"), Cart.Mirroring);

	if (mapper_init(Cart.Mapper) == -1) {
		bprintf(0, _T("Unsupported mapper\n"));
		return 1;
	}

	if (Cart.BatteryBackedSRAM) {
		bprintf(0, _T("Game has battery-backed sRAM\n"));
	}

	Cart.PRGRomMask = Cart.PRGRomSize - 1;
	Cart.WorkRAMMask = Cart.WorkRAMSize - 1;

	// Game-specific stuff:
	// Mapper 7 or 4-way mirroring usually gets no workram (6000-7fff)
	if (Cart.Mapper == 7 || (Cart.Mirroring == 4 && !(PPUType > RP2C02)))
		NESMode |= NO_WORKRAM; // VS. is exempt from this limitation.

	NESMode |= (ROMCRC == 0xab29ab28) ? BUS_CONFLICTS : 0; // Dropzone
	NESMode |= (ROMCRC == 0xe3a6d7f6) ? BUS_CONFLICTS : 0; // Cybernoid
	NESMode |= (ROMCRC == 0x552a903a) ? BUS_CONFLICTS : 0; // Huge Insect
	NESMode |= (ROMCRC == 0xb90a1ca1) ? NO_WORKRAM : 0; // Low G Man
	NESMode |= (ROMCRC == 0xa905cc12) ? NO_WORKRAM : 0; // Bill & Ted
	NESMode |= (ROMCRC == 0x585f3500) ? ALT_MMC3 : 0; // Darkwing Duck (T-Chi)
	NESMode |= (ROMCRC == 0x38f65b2d) ? BAD_HOMEBREW : 0; // Battler (HB)
	NESMode |= (ROMCRC == 0x6673e5da) ? ALT_TIMING : 0; // t.atom
	NESMode |= (ROMCRC == 0xd5e10c90) ? ALT_TIMING : 0; // astro boy
	NESMode |= (ROMCRC == 0xc9f3f439) ? ALT_TIMING : 0; // vs. freedom force
	NESMode |= (ROMCRC == 0x53eb8950) ? ALT_TIMING : 0; // freedom force
	NESMode |= (ROMCRC == 0x560142bc) ? ALT_TIMING2 : 0; // don doko don 2
	NESMode |= (ROMCRC == 0xe39e0be2) ? ALT_TIMING2 : 0; // laser invasion
	NESMode |= (ROMCRC == 0x47d22165) ? BAD_HOMEBREW : 0; // animal clipper
	NESMode |= (ROMCRC == 0x70eac605) ? ALT_TIMING : 0; // deblock
	NESMode |= (ROMCRC == 0x3616c7dd) ? ALT_TIMING : 0; // days of thunder
	NESMode |= (ROMCRC == 0xeb506bf9) ? ALT_TIMING : 0; // star wars
	NESMode |= (ROMCRC == 0xa2d504a8) ? ALT_TIMING : 0; // assimilate
	NESMode |= (ROMCRC == 0x4df9d7c8) ? ALT_TIMING : 0; // overlord
	NESMode |= (ROMCRC == 0x85784e11) ? ALT_TIMING : 0; // blargg full palette
	NESMode |= (ROMCRC == 0x5da28b4f) ? ALT_TIMING : 0; // cmc! wall demo
	NESMode |= (ROMCRC == 0xab862073) ? ALT_TIMING : 0; // ultimate frogger champ.
	NESMode |= (ROMCRC == 0x2a798367) ? ALT_TIMING : 0; // jy 45-in-1
	NESMode |= (ROMCRC == 0xb4255e99) ? (IS_PAL | SHOW_OVERSCAN) : 0; // Moonglow (HB)
	NESMode |= (ROMCRC == 0x78716f4f) ? RAM_RANDOM : 0; // Go! Dizzy Go!
	NESMode |= (ROMCRC == 0x8c4f37e2) ? RAM_RANDOM : 0; // Minna no Taabou no Nakayoshi Daisakusen (Japan)
	NESMode |= (ROMCRC == 0x17336a80) ? RAM_RANDOM : 0; // Minna no Taabou no Nakayoshi Daisakusen (T-Eng)
	NESMode |= (ROMCRC == 0xc0b4bce5) ? RAM_RANDOM : 0; // Terminator 2 (T2) - Judgement Day
	NESMode |= (ROMCRC == 0x4d58c832) ? IS_PAL : 0; // Hammerin' Harry
	NESMode |= (ROMCRC == 0x149e367f) ? IS_PAL : 0; // Lion King, The
	NESMode |= (ROMCRC == 0xbf80b241) ? IS_PAL : 0; // Mr. Gimmick
	NESMode |= (ROMCRC == 0x4cf12d39) ? IS_PAL : 0; // Elite
	NESMode |= (ROMCRC == 0x3a0b6dd2) ? IS_PAL : 0; // Hero Quest
	NESMode |= (ROMCRC == 0x6d1e30a7) ? IS_PAL : 0; // TMHT
	NESMode |= (ROMCRC == 0xa5bbb96b) ? IS_PAL : 0; // TMHTII
	NESMode |= (ROMCRC == 0x6453f65e) ? IS_PAL : 0; // Uforia
	NESMode |= (ROMCRC == 0x55cbc495) ? IS_PAL : 0; // Super Turrican
	NESMode |= (ROMCRC == 0x11a245a0) ? IS_PAL : 0; // Rod Land
	NESMode |= (ROMCRC == 0xa535e1be) ? IS_PAL : 0; // Rackets and Rivals
	NESMode |= (ROMCRC == 0x8236d3e0) ? IS_PAL : 0; // Probotector
	NESMode |= (ROMCRC == 0x1d2fb2b7) ? IS_PAL : 0; // Shadow Warriors
	NESMode |= (ROMCRC == 0x96b6a919) ? IS_PAL : 0; // Shadow Warriors II
	NESMode |= (ROMCRC == 0x22ad4753) ? IS_PAL : 0; // Parodius
	NESMode |= (ROMCRC == 0xdcd55bec) ? IS_PAL : 0; // Aladdin
	NESMode |= (ROMCRC == 0xe08a5881) ? IS_PAL : 0; // Beauty and the Beast
	NESMode |= (ROMCRC == 0xcbde707e) ? IS_PAL : 0; // International Cricket
	NESMode |= (ROMCRC == 0xe2685bbf) ? IS_PAL : 0; // Kick Off
	NESMode |= (ROMCRC == 0xab21ab5f) ? IS_PAL : 0; // Noah's Ark
	NESMode |= (ROMCRC == 0xab29ab28) ? IS_PAL : 0; // Dropzone
	NESMode |= (ROMCRC == 0xde7e6767) ? IS_PAL : 0; // Asterix
	NESMode |= (ROMCRC == 0xdc7a16e6) ? IS_PAL : 0; // Parasol Stars
	NESMode |= (ROMCRC == 0xfac97247) ? IS_PAL : 0; // Rainbow Islands (Ocean)
	NESMode |= (ROMCRC == 0x732b1a7a) ? IS_PAL : 0; // Smurfs, The

	if (NESMode) {
		bprintf(0, _T("Game-specific configuration:\n"));

		for (UINT16 i = 1; i != 0x0000; i <<= 1) {
			switch (NESMode & i) {
				case NO_WORKRAM:
					bprintf(0, _T("*  Disabling cart. work-ram (6000-7fff)\n"));
					break;

				case BUS_CONFLICTS:
					bprintf(0, _T("*  Enabling BUS-CONFLICTS.\n"));
					break;

				case ALT_TIMING:
					bprintf(0, _T("*  Enabling ALT-TIMING.\n"));
					break;

				case ALT_TIMING2:
					bprintf(0, _T("*  Enabling ALT-TIMING2.\n"));
					break;

				case IS_PAL:
					bprintf(0, _T("*  Enabling PAL weirdness.\n"));
					break;

				case RAM_RANDOM:
					bprintf(0, _T("*  Init RAM w/random bytes.\n"));
					break;
			}
		}
	}

	return 0;
}

static INT32 filediff(TCHAR *file1, TCHAR *file2)
{
	FILE *fp1, *fp2;
	INT32 len1, len2;
	UINT8 c1, c2;

	fp1 = _tfopen(file1, _T("rb"));
	if (!fp1) return -1;
	fp2 = _tfopen(file2, _T("rb"));
	if (!fp2) {
		fclose(fp1);
		return -2;
	}

	fseek(fp1, 0, SEEK_END);
	len1 = ftell(fp1);
	fseek(fp1, 0, SEEK_SET);

	fseek(fp2, 0, SEEK_END);
	len2 = ftell(fp2);
	fseek(fp2, 0, SEEK_SET);

	if (!len1 || !len2 || len1 != len2) {
		fclose(fp1);
		fclose(fp2);
		bprintf(0, _T("filediff(): length mismatch\n"));
		return -3;
	}

	for (INT32 i = 0; i < len1; i++) {
		fread(&c1, 1, 1, fp1);
		fread(&c2, 1, 1, fp2);
		if (c1 != c2) {
			fclose(fp1);
			fclose(fp2);
			bprintf(0, _T("filediff(): difference at offset $%x\n"));
			return -3;
		}
	}

	fclose(fp1);
	fclose(fp2);

	return 0; // file1 == file2
}

static INT32 ips_make(UINT8 *orig_data, UINT8 *new_data, INT32 size_data, TCHAR *ips_dir, TCHAR *ips_fn)
{
	#ifndef BUILD_WIN32
	#define _wunlink unlink
	#define _wrename rename
	#endif
	char iPATCH[6] = "PATCH";
	char iEOF[4] = "EOF";
	TCHAR ips_path_fn[MAX_PATH];
	TCHAR ips_path_fn_temp[MAX_PATH];
	INT32 ips_address;
	UINT8 ips_address_c[3];

	INT32 ips_length;
	UINT8 ips_length_c[2];

	INT32 rle_good;
	INT32 rle_length;
	UINT8 rle_byte;

	INT32 is_difference = 0;
	for (INT32 i = 0; i < size_data; i++)
	{
		if (orig_data[i] != new_data[i]) {
			is_difference = 1;
			break;
		}
	}

	if (is_difference == 0) return -2; // no change!

	_stprintf(ips_path_fn, _T("%s%s"), ips_dir, ips_fn);
	_stprintf(ips_path_fn_temp, _T("%s%s.temp"), ips_dir, ips_fn);

	bprintf(0, _T("ips_make() writing to [%s]\n"), ips_path_fn_temp);
	FILE *f = _tfopen(ips_path_fn_temp, _T("wb"));
	if (!f) return -1; // uhoh! (can't open file for writing)

	fwrite(&iPATCH, 1, 5, f);
	for (INT32 i = 0; i < size_data; i++)
	{
		if (orig_data[i] == new_data[i]) continue;

		ips_address = i;

		rle_good = 0;
		rle_length = 0;
		rle_byte = new_data[i];
		ips_length = 0;

		if (ips_address == 0x454f46) { // 'EOF'
			ips_length = 1; // Patch @ BAD ADDRESS (0x454f46), let's go back 1 byte and start here.
			ips_address--;
			if (rle_byte == new_data[ips_address]) {
				//printf("BAD ADDRESS-1 + RLE possible.\n");
				rle_length++;
			}
		}

		while ( (i < size_data) && (ips_length < 0xfce2)
			   && ((orig_data[i] != new_data[i])
				   || (rle_good && rle_byte == new_data[i])) ) // rle-opti: so we don't create multiple rle-chunks if a couple bytes are unchanged
		{
			if (rle_byte == new_data[i]) {
				if (rle_length == ips_length && rle_length > 3) {
					rle_good = 1;
				}
				rle_length++;
				if (rle_length > 5 && rle_good == 0) {
					// dump patch from before rle so the next contiguous block
					// is the rle block.
					if (i - rle_length == 0x454f46-1) {
						// this is not a good spot to rewind to.. abort rle for now.
						rle_length = 0;
					} else {
						i -= rle_length;
						ips_length -= rle_length;
						i--;
						break;
					}
				}
			} else if (rle_good) {
				i--;
				break;
			} else {
				rle_length = 0;
				rle_byte = new_data[i];
			}

			ips_length++;
			i++;
		}
		//printf("%spatchy @ %x   len %x  rle_len %x\n", ((rle_good)?"RLE-":""),ips_address, ips_length, rle_length);
		ips_address_c[0] = (ips_address >> 16) & 0xff;
		ips_address_c[1] = (ips_address >>  8) & 0xff;
		ips_address_c[2] = (ips_address >>  0) & 0xff;

		if (rle_good) ips_length = 0;

		ips_length_c[0] = (ips_length >> 8) & 0xff;
		ips_length_c[1] = (ips_length >> 0) & 0xff;

		fwrite(&ips_address_c, 1, 3, f);
		fwrite(&ips_length_c, 1, 2, f);
		if (rle_good) {
			ips_length_c[0] = (rle_length >> 8) & 0xff;
			ips_length_c[1] = (rle_length >> 0) & 0xff;
			fwrite(&ips_length_c, 1, 2, f);
			fwrite(&rle_byte, 1, 1, f);
		} else {
			fwrite(&new_data[ips_address], 1, ips_length, f);
		}
	}

	fwrite(&iEOF, 1, 3, f);

	fclose(f);

	// Check if the newly written patch is the same as previously written..
	if (filediff(ips_path_fn_temp, ips_path_fn)) {
		// Patch differs!
		// re-order backups
		const INT32 MAX_BACKUPS = 10;
		for (INT32 i = MAX_BACKUPS; i >= 0; i--) {
			TCHAR szBackupNameTo[MAX_PATH];
			TCHAR szBackupNameFrom[MAX_PATH];

			_stprintf(szBackupNameTo, _T("%s%s.backup%d"), ips_dir, ips_fn, i + 1);
			_stprintf(szBackupNameFrom, _T("%s%s.backup%d"), ips_dir, ips_fn, i);
			if (i == MAX_BACKUPS) {
				_wunlink(szBackupNameFrom); // make sure there is only MAX_BACKUPS :)
			} else {
				_wrename(szBackupNameFrom, szBackupNameTo); //derp.ips.backup0 -> derp.ips.backup1
				if (i == 0) {
					_wrename(ips_path_fn, szBackupNameFrom); //derp.ips -> derp.ips.backup0
				}
			}
		}
		// Rename temp patch filename to real patch filename
		int rc = _wrename(ips_path_fn_temp, ips_path_fn);
		bprintf(0, _T("wrename rc = %d   errno = %d\n"), rc, errno);
		bprintf(0, _T("-- lastly temp to non-temp -- rename %s %s\n"), ips_path_fn_temp, ips_path_fn);
		bprintf(0, _T("--- Fresh new IPS patch written: %s\n"), ips_path_fn);
	} else {
		bprintf(0, _T("--- IPS Patch from this session same as last - aborting write!\n"));
		return -2; // no change!
	}

	return 0;
}

static INT32 ips_patch(UINT8 *data, INT32 size_data, TCHAR *ips_fn)
{
	UINT8 buf[7] = "\0\0\0\0\0\0";

	UINT32 ips_address;
	UINT32 ips_length;

	INT32 rle_repeat;
	UINT8 rle_byte;

	FILE *f = _tfopen(ips_fn, _T("rb"));
	if (!f) return -1;

	fread(&buf, 1, 5, f);
	if (memcmp(&buf, "PATCH", 5) != 0) {
		fclose(f);
		return -2;
	}

	while (!feof(f))
	{
		memset(&buf, 0, 3);
		fread(buf, 1, 3, f);
		if (memcmp(&buf, "EOF", 3) == 0) {
			break;
		}
		ips_address = ((buf[0] << 16) & 0xff0000) | ((buf[1] << 8) & 0xff00) | (buf[2] & 0xff);

		memset(&buf, 0, 3);
		fread(buf, 1, 2, f);
		ips_length = ((buf[0] << 8) & 0xff00) | (buf[1] & 0xff);

		if (ips_length == 0) { // RLE chunk
			fread(buf, 1, 3, f);
			rle_repeat = ((buf[0] << 8) & 0xff00) | (buf[1] & 0xff);
			rle_byte = buf[2];

			if (ips_address + rle_repeat <= size_data) { // ok to patch
				memset(&data[ips_address], rle_byte, rle_repeat);
			} else {
				//printf("patch grow disabled, aborting.\n");
				break;
			}
		} else { // normal chunk
			if (ips_address + ips_length <= size_data) { // ok to patch
				fread(&data[ips_address], 1, ips_length, f);
			} else {
				//printf("patch grow disabled, aborting.\n");
				break;
			}
		}
	}

	fclose(f);

	return 0;
}

static INT32 fds_load(UINT8* ROMData, UINT32 ROMSize, UINT32 ROMCRC)
{
	bprintf(0, _T("FDS Loader\n"));
	if (NULL == ROMData || ROMSize < 8192 ) {
		bprintf(0, _T("Invalid ROM data!\n"));
		return 1;
	}

	// FDS mem map:
	// 6000 - dfff RAM
	// e000 - ffff ROM (FDS Bios)

	memset(&Cart, 0, sizeof(Cart));

	PPUType = RP2C02;

	Cart.FDSMode = 1;
	Cart.FDSDiskRaw = (UINT8*)BurnMalloc(0x100000);
	Cart.FDSDiskRawOrig = (UINT8*)BurnMalloc(0x100000);
	Cart.FDSDisk = Cart.FDSDiskRaw;
	Cart.FDSDiskRawSize = ROMSize;
	Cart.FDSDiskRawOrigSize = ROMSize;
	Cart.FDSDiskSize = ROMSize;
	Cart.FDSDiskSides = ROMSize / 65500;
	if (BurnLoadRom(Cart.FDSDiskRaw, 0, 1)) return 1; // load FDS Disk Image
	if (BurnLoadRom(Cart.FDSDiskRawOrig, 0, 1)) return 1; // load FDS Disk Image

	// Load IPS patch (aka: disk-saves @ exit)
	TCHAR szFilename[MAX_PATH];
	_stprintf(szFilename, _T("%s%s.ips"), szAppEEPROMPath, BurnDrvGetText(DRV_NAME));
	INT32 ips = ips_patch(Cart.FDSDiskRaw, Cart.FDSDiskRawSize, szFilename);
	bprintf(0, _T("* FDS DISK patch: %s\n"), (ips == 0) ? _T("Loaded") : _T("Can't Load/Not Found."));

	if (!memcmp("FDS\x1a", &Cart.FDSDiskRaw[0], 4) && ROMSize > 0x10) {
		Cart.FDSDisk += 0x10;
		Cart.FDSDiskSize -= 0x10;
		bprintf(0, _T("* Skipping useless fds header..\n"));
	}

	Cart.Crc = ROMCRC;
	Cart.PRGRomSize = 0x8000;
	Cart.CHRRomSize = 0; // ram only
	Cart.Mirroring = 1; // for bios, but fds-mapper-controlled
	Cart.Mapper = 0x808; // fake mapper# for fds

	bprintf(0, _T("Disk Size: %d (%d sides)\n"), ROMSize, Cart.FDSDiskSides);
	bprintf(0, _T("PRG Size: %d\n"), Cart.PRGRomSize);
	bprintf(0, _T("CHR Size: %d\n"), Cart.CHRRomSize);

	switch (Cart.Mirroring) {
		case 0: set_mirroring(HORIZONTAL); break;
		case 1: set_mirroring(VERTICAL); break;
		case 4: set_mirroring(FOUR_SCREEN); break;
	}

//	Cart.BatteryBackedSRAM = (ROMData[6] & 0x2) >> 1;
	Cart.PRGRom = ROMData;

	// Default CHR-Ram size (8k), always set-up (for advanced mappers, etc)
	Cart.CHRRamSize = 0x2000;
	Cart.CHRRam = (UINT8*)BurnMalloc(Cart.CHRRamSize);
	mapper_set_chrtype(MEM_RAM);
	bprintf(0, _T("Using RAM CHR data (%dk bytes)\n"), Cart.CHRRamSize / 1024);

	// set-up Cart PRG-RAM/WORK-RAM (6000-7fff)
	Cart.WorkRAMSize = 0x2000;
	bprintf(0, _T("Cartridge RAM: %d\n"), Cart.WorkRAMSize);
	Cart.WorkRAM = (UINT8*)BurnMalloc(Cart.WorkRAMSize);

	// set-up MAPPER
	bprintf(0, _T("Cartridge Mapper: %d   Mirroring: "), Cart.Mapper);
	switch (Cart.Mirroring) {
		case 0: bprintf(0, _T("Horizontal")); break;
		case 1: bprintf(0, _T("Vertical")); break;
		case 4: bprintf(0, _T("Four-Screen")); break;
		default: bprintf(0, _T("???")); break;
	}
	bprintf(0, _T(" (%d)\n"), Cart.Mirroring);

	if (mapper_init(Cart.Mapper) == -1) {
		bprintf(0, _T("Unsupported mapper\n"));
		return 1;
	}

	if (Cart.BatteryBackedSRAM) {
		bprintf(0, _T("Game has battery-backed sRAM\n"));
	}

	Cart.PRGRomMask = Cart.PRGRomSize - 1;
	Cart.WorkRAMMask = Cart.WorkRAMSize - 1;

	// Game-specific stuff:
	NESMode = IS_FDS;

	if (NESMode) {
		bprintf(0, _T("Game-specific configuration:\n"));

		switch (NESMode) {
			case NO_WORKRAM:
				bprintf(0, _T("*  Disabling cart. work-ram (6000-7fff)\n"));
				break;

			case BUS_CONFLICTS:
				bprintf(0, _T("*  Enabling BUS-CONFLICTS.\n"));
				break;

			case ALT_TIMING:
				bprintf(0, _T("*  Enabling ALT-TIMING.\n"));
				break;

			case IS_PAL:
				bprintf(0, _T("*  Enabling PAL weirdness.\n"));
				break;
		}
	}

	return 0;
}

// ---- mapper system begins! ----
static UINT32 PRGMap[4];
static UINT8  PRGType[4];
static UINT32 CHRMap[8];
static UINT8  CHRType[8]; // enum { MEM_RAM = 0, MEM_RAM_RO = 1, MEM_ROM = 2};
static UINT8  mapper_regs[0x20]; // General-purpose mapper registers (8bit)
static UINT16 mapper_regs16[0x20]; // General-purpose mapper registers (16bit)
static INT32 mapper_irq_exec; // cycle-delayed irq for mapper_irq();
static void mapper_irq(INT32 cyc); // forward

// mapping expansion ram/rom (6000 - 7fff)  refer to mapper69 for hookup info
static INT32 PRGExpMap;
static void  (*cart_exp_write)(UINT16 address, UINT8 data) = NULL;
static UINT8 (*cart_exp_read)(UINT16 address) = NULL;
static INT32 cart_exp_write_abort; // abort fallthrough - see mapper68
// mapping 4020 - 5fff, use these callbacks. refer to Sachen 8259
static void  (*psg_area_write)(UINT16 address, UINT8 data) = NULL;
static UINT8 (*psg_area_read)(UINT16 address) = NULL;
static UINT8 read_nt_int(UINT16 address);

static void mapper_map_prg(INT32 pagesz, INT32 slot, INT32 bank, INT32 type = MEM_ROM) // 8000 - ffff
{
	INT32 ramromsize = (type == MEM_ROM) ? Cart.PRGRomSize : Cart.WorkRAMSize;

	if (bank < 0) { // negative bank == map page from end of rom
		bank = (ramromsize / (pagesz * 1024)) + bank;
	}

	for (INT32 i = 0; i < (pagesz / 8); i++) {
		PRGMap[(pagesz / 8) * slot + i] = (pagesz * 1024 * bank + 0x2000 * i) % ramromsize;
		PRGType[(pagesz / 8) * slot + i] = type;
	}
}

static void mapper_map_exp_prg(INT32 bank, INT32 type = MEM_ROM) // 6000 - 7fff area (cartridge expansion ram/sram or battery backed ram)
{
	INT32 ramromsize = (type == MEM_ROM) ? Cart.PRGRomSize : Cart.WorkRAMSize;

	if (bank < 0) { // negative bank == map page from end of rom
		bank = (ramromsize / 0x2000) + bank;
	}

	PRGExpMap = (0x2000 * bank) % ramromsize;
}

static void mapper_set_chrtype(INT32 type)
{
	for (INT32 i = 0; i < 8; i++)
		CHRType[i] = type;
}

static void mapper_map_chr(INT32 pagesz, INT32 slot, INT32 bank)
{
	for (INT32 i = 0; i < pagesz; i++) {
		switch (CHRType[pagesz * slot + i]) {
			case MEM_ROM:
				CHRMap[pagesz * slot + i] = (pagesz * 1024 * bank + 1024 * i) % Cart.CHRRomSize;
				break;

			case MEM_RAM:
			case MEM_RAM_RO:
				CHRMap[pagesz * slot + i] = (pagesz * 1024 * bank + 1024 * i) % Cart.CHRRamSize;
				break;
		}
	}
}

#define MAP_CHR_RAMROM_DEBUG 0

static void mapper_map_chr_ramrom(INT32 pagesz, INT32 slot, INT32 bank, INT32 type)
{
	if (type > MEM_ROM) {
		bprintf(0, _T("mapper_map_chr_ramrom(): invalid type field!!\n"));
	}

#if MAP_CHR_RAMROM_DEBUG
	bprintf(0, _T("mapper_map_chr_ramrom(%x, %x, %x, %x)\n"), pagesz, slot, bank, type);
#endif

	for (INT32 i = 0; i < pagesz; i++) {
		switch (type) {
			case MEM_ROM:
				CHRMap[pagesz * slot + i] = (pagesz * 1024 * bank + 1024 * i) % Cart.CHRRomSize;
				CHRType[pagesz * slot + i] = MEM_ROM;
#if MAP_CHR_RAMROM_DEBUG
				bprintf(0, _T("ROM: CHRMap[%x] = %x\n"),pagesz * slot + i, (pagesz * 1024 * bank + 1024 * i) % Cart.CHRRomSize);
#endif
				break;

			case MEM_RAM:
			case MEM_RAM_RO:
				CHRMap[pagesz * slot + i] = (pagesz * 1024 * bank + 1024 * i) % Cart.CHRRamSize;
				CHRType[pagesz * slot + i] = type;
#if MAP_CHR_RAMROM_DEBUG
				bprintf(0, _T("RAM: CHRMap[%x] = %x\n"),pagesz * slot + i, (pagesz * 1024 * bank + 1024 * i) % Cart.CHRRamSize);
#endif
				break;
		}
	}
}

static UINT8 mapper_prg_read_int(UINT16 address) // mapper_prg_read points here
{
	switch (PRGType[(address & ~0x8000) / 0x2000]) {
		case MEM_ROM:
			return Cart.PRGRom[PRGMap[(address & ~0x8000) / 0x2000] + (address & 0x1fff)];
		case MEM_RAM:
			return Cart.WorkRAM[PRGMap[(address & ~0x8000) / 0x2000] + (address & 0x1fff)];
	}

	bprintf(0, _T("PRGType[] corruption?\n"));
	return 0x00;
}

static void mapper_prg_write(UINT16 address, UINT8 data)
{
	switch (PRGType[(address & ~0x8000) / 0x2000]) {
		case MEM_RAM:
			Cart.WorkRAM[PRGMap[(address & ~0x8000) / 0x2000] + (address & 0x1fff)] = data;
	}
}

static UINT8 mapper_chr_read(UINT16 address)
{
	switch (CHRType[address / 1024]) {
		case MEM_ROM:
			return Cart.CHRRom[CHRMap[address / 1024] + (address & (1024 - 1))];

		case MEM_RAM:
		case MEM_RAM_RO:
			return Cart.CHRRam[CHRMap[address / 1024] + (address & (1024 - 1))];
	}

	return 0x00;
}

static void mapper_chr_write(UINT16 address, UINT8 data)
{
	if (CHRType[address / 1024] == MEM_RAM) {
		Cart.CHRRam[CHRMap[address / 1024] + (address & (1024 - 1))] = data;
	}
}

// Flippy Disk ICON (thanks iq_132! :)
static UINT8 disk_ab[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x00, 0x00, 0x07, 0x07, 0x0f, 0x07, 0x07, 0x07, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x0f, 0x01, 0x0e, 0x01, 0x03, 0x09, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x0f, 0x07, 0x07, 0x00, 0x00, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x00, 0x09, 0x04, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x0c, 0x0d, 0x00, 0x0d, 0x0e, 0x0a, 0x00, 0x02, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x00, 0x00, 0x07, 0x07, 0x0f, 0x07, 0x07, 0x07, 0x00, 0x09, 0x04, 0x04, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x0d, 0x00, 0x00, 0x0c, 0x09, 0x0f, 0x00, 0x00, 0x06, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x00, 0x09, 0x04, 0x04, 0x04, 0x09, 0x09, 0x09, 0x09, 0x09, 0x08, 0x00, 0x00, 0x0d, 0x09, 0x09, 0x09, 0x0a, 0x00, 0x00, 0x03, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x00, 0x00, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x00, 0x09, 0x04, 0x04, 0x04, 0x04, 0x09, 0x09, 0x09, 0x09, 0x02, 0x00, 0x00, 0x06, 0x09, 0x09, 0x09, 0x0d, 0x00, 0x00, 0x06, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x03, 0x07, 0x00, 0x09, 0x09, 0x00, 0x07, 0x03, 0x07, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x09, 0x09, 0x09, 0x00, 0x00, 0x00, 0x0e, 0x09, 0x09, 0x09, 0x02, 0x00, 0x00, 0x0a, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x0f, 0x07, 0x00, 0x09, 0x09, 0x00, 0x07, 0x0f, 0x07, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x09, 0x03, 0x00, 0x00, 0x00, 0x01, 0x09, 0x09, 0x09, 0x02, 0x00, 0x00, 0x00, 0x0c, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x07, 0x03, 0x07, 0x00, 0x00, 0x07, 0x03, 0x07, 0x07, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x09, 0x09, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x09, 0x09, 0x09, 0x01, 0x00, 0x00, 0x00, 0x0d, 0x0e, 0x01, 0x0e, 0x05, 0x00, 0x00, 0x00, 0x08, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x07, 0x07, 0x03, 0x03, 0x03, 0x03, 0x07, 0x07, 0x07, 0x07, 0x07, 0x00, 0x09, 0x04, 0x04, 0x04, 0x09, 0x09, 0x09, 0x09, 0x06, 0x00, 0x00, 0x00, 0x01, 0x09, 0x09, 0x09, 0x0d, 0x00, 0x00, 0x00, 0x0b, 0x09,
	0x00, 0x03, 0x07, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x07, 0x03, 0x00, 0x09, 0x04, 0x04, 0x09, 0x09, 0x09, 0x09, 0x09, 0x02, 0x00, 0x00, 0x00, 0x01, 0x09, 0x09, 0x09, 0x02, 0x00, 0x00, 0x00, 0x01, 0x09,
	0x00, 0x03, 0x07, 0x07, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x07, 0x07, 0x03, 0x00, 0x09, 0x04, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x06, 0x00, 0x00, 0x00, 0x01, 0x09, 0x09, 0x09, 0x0d, 0x00, 0x00, 0x00, 0x01, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x07, 0x07, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x02, 0x00, 0x00, 0x00, 0x01, 0x09, 0x09, 0x09, 0x02, 0x00, 0x00, 0x00, 0x01, 0x09,
	0x00, 0x07, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x07, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x07, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x0f, 0x07, 0x07, 0x07, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x00, 0x00, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x0e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x0d, 0x0e, 0x0c, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x0f, 0x07, 0x07, 0x00, 0x00, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x00, 0x09, 0x04, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x06, 0x00, 0x00, 0x00, 0x0b, 0x03, 0x06, 0x00, 0x00, 0x00, 0x01, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x00, 0x00, 0x07, 0x07, 0x0f, 0x07, 0x07, 0x07, 0x00, 0x09, 0x04, 0x04, 0x09, 0x09, 0x09, 0x09, 0x09, 0x0e, 0x00, 0x00, 0x00, 0x03, 0x09, 0x03, 0x00, 0x00, 0x00, 0x00, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x00, 0x09, 0x04, 0x04, 0x04, 0x09, 0x09, 0x09, 0x09, 0x06, 0x00, 0x00, 0x00, 0x03, 0x09, 0x03, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x00, 0x00, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x00, 0x09, 0x04, 0x04, 0x04, 0x04, 0x09, 0x09, 0x09, 0x0e, 0x00, 0x00, 0x00, 0x03, 0x09, 0x0b, 0x00, 0x00, 0x00, 0x0a, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x03, 0x07, 0x00, 0x09, 0x09, 0x00, 0x07, 0x03, 0x07, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x09, 0x09, 0x06, 0x00, 0x00, 0x00, 0x0a, 0x0d, 0x0a, 0x00, 0x0a, 0x02, 0x03, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x0f, 0x07, 0x00, 0x09, 0x09, 0x00, 0x07, 0x0f, 0x07, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x09, 0x0e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x02, 0x0c, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x07, 0x03, 0x07, 0x00, 0x00, 0x07, 0x03, 0x07, 0x07, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x09, 0x09, 0x06, 0x00, 0x00, 0x00, 0x0f, 0x09, 0x0b, 0x00, 0x00, 0x00, 0x05, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x09, 0x09, 0x09, 0x0e, 0x00, 0x00, 0x00, 0x0c, 0x09, 0x09, 0x00, 0x00, 0x00, 0x00, 0x03, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x07, 0x07, 0x03, 0x03, 0x03, 0x03, 0x07, 0x07, 0x07, 0x07, 0x07, 0x00, 0x09, 0x04, 0x04, 0x04, 0x09, 0x09, 0x09, 0x09, 0x0e, 0x00, 0x00, 0x00, 0x0c, 0x09, 0x09, 0x00, 0x00, 0x00, 0x00, 0x03, 0x09, 0x09,
	0x00, 0x03, 0x07, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x07, 0x03, 0x00, 0x09, 0x04, 0x04, 0x09, 0x09, 0x09, 0x09, 0x09, 0x06, 0x00, 0x00, 0x00, 0x0c, 0x09, 0x09, 0x00, 0x00, 0x00, 0x0a, 0x09, 0x09, 0x09,
	0x00, 0x03, 0x07, 0x07, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x07, 0x07, 0x03, 0x00, 0x09, 0x04, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x0e, 0x00, 0x00, 0x00, 0x01, 0x03, 0x06, 0x00, 0x00, 0x0a, 0x0f, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x07, 0x07, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x0e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x02, 0x01, 0x0c, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x07, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x07, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09
};

static UINT8 disk_ab_pal[] = {
	0x01, 0x01, 0x01,
	0x87, 0x87, 0x87,
	0x4c, 0x4c, 0x4c,
	0xc4, 0xc4, 0xc4,
	0x25, 0x1d, 0xed,
	0x27, 0x27, 0x27,
	0x66, 0x66, 0x66,
	0xb1, 0xe5, 0xef,
	0xa6, 0xa6, 0xa6,
	0xff, 0xff, 0xff,
	0x19, 0x19, 0x19,
	0x98, 0x98, 0x98,
	0xe1, 0xe1, 0xe1,
	0x3b, 0x3b, 0x3b,
	0x75, 0x75, 0x75,
	0xba, 0xba, 0xba
};

// ---[ mapper FDS (Famicom Disk System) #0x808 (2056)
#define mapperFDS_ioenable		(mapper_regs[0x1f - 0]) // 4023
#define mapperFDS_control		(mapper_regs[0x1f - 1]) // 4025

#define mapperFDS_eighth_bit 	(mapper_regs[0x1f - 2])
#define mapperFDS_in_irq		(mapper_regs[0x1f - 3])

#define mapperFDS_irqenable		(mapper_regs[0x1f - 4])
#define mapperFDS_irqrepeat		(mapper_regs[0x1f - 5])
#define mapperFDS_irqcount		(mapper_regs16[0x1f - 0])
#define mapperFDS_irqlatch		(mapper_regs16[0x1f - 1])

#define mapperFDS_diskside		(mapper_regs[0x1f - 6])
#define mapperFDS_diskinsert	(mapper_regs[0x1f - 7])
#define mapperFDS_diskaccessed	(mapper_regs[0x1f - 8])
#define mapperFDS_diskaddr		(mapper_regs16[0x1f - 2])
#define mapperFDS_diskirqcount	(mapper_regs16[0x1f - 3])
#define mapperFDS_arm_timer() 	do { mapperFDS_diskirqcount = 150; } while (0)
#define mapperFDS_filesize      (mapper_regs16[0x1f - 4])
#define mapperFDS_blockid       (mapper_regs16[0x1f - 5])
#define mapperFDS_blockstart    (mapper_regs16[0x1f - 6])
#define mapperFDS_blocklen      (mapper_regs16[0x1f - 7])
#define fds_disk() 				(Cart.FDSDisk[(mapperFDS_diskside * 65500) + mapperFDS_blockstart + mapperFDS_diskaddr])

static char fdsfile[10]; // for debugging.. remove later!

enum FDS_FrameActions { FA_IDLE = 0, INSERT_DISK = 1, SWAP_DISK = 2, FA_FADEOUT};
enum FDS_DiskBlockIDs { DSK_INIT = 0, DSK_VOLUME, DSK_FILECNT, DSK_FILEHDR, DSK_FILEDATA };

static void FDS_FrameAction(INT32 todo);

static void FDS_Insert(INT32 status)
{
	static INT32 debounce = 0;
	if (status && debounce == 0) {
		mapperFDS_diskinsert ^= 1;
		bprintf(0, _T("FDS_Insert: %d\n"), mapperFDS_diskinsert);
	}
	debounce = status;
}

static void FDS_SwapSides(INT32 status, INT32 no_debounce = 0)
{
	static INT32 debounce = 0;
	if (status && (debounce == 0 || no_debounce) && mapperFDS_diskinsert == 0) {
		mapperFDS_diskside = (mapperFDS_diskside + 1) % Cart.FDSDiskSides;
		bprintf(0, _T("FDS_SwapSides: %d\n"), mapperFDS_diskside);
	}
	debounce = status;
}

static void FDS_SwapSidesAuto(INT32 status)
{
	static INT32 debounce = 0;
	if (status && debounce == 0) {
		FDS_FrameAction(SWAP_DISK);
	}
	debounce = status;
}

static void FDS_FrameAction(INT32 todo)
{
	FDSFrameCounter = 0;
	FDSFrameAction = todo;
}

static void FDS_FrameTicker()
{
	switch (FDSFrameAction) {
		case FA_IDLE:
			break;

		case INSERT_DISK:
			if (FDSFrameCounter == 5) {
				mapperFDS_diskinsert = 1;
				FDSFrameAction = FA_IDLE;
			}
			break;

		case SWAP_DISK:
			if (FDSFrameCounter == 0) {
				FDSFrameDiskIcon = ((mapperFDS_diskside + 1) % Cart.FDSDiskSides) & 1;
			}
			if (FDSFrameCounter == 5) {
				mapperFDS_diskinsert = 0;
			}
			if (FDSFrameCounter == 75) {
				bprintf(0, _T("(auto) "));
				FDS_SwapSides(1, 1);
			}
			if (FDSFrameCounter == 155) {
				mapperFDS_diskinsert = 1;
				FDSFrameAction = FA_FADEOUT;
			}
			break;
		case FA_FADEOUT:
			if (FDSFrameCounter == 155+20) {
				FDSFrameAction = FA_IDLE;
			}
			break;
	}
	FDSFrameCounter++;
}

struct fds_sound {
	UINT8 volmaster;
	UINT8 wavwren_hold;
	UINT8 volgain; // r:4090 w:4080
	UINT8 modgain; // r:4092 w:4084
	UINT8 waveram[0x40];
	UINT8 modwaveram[0x40];

	UINT8 env_master;

	UINT32 wavefreq;
	UINT32 wavepos;
	UINT32 modphase;
	UINT32 modfreq;
	INT32 mod_accu;

	UINT8 mod_counter;
	UINT8 env_all_stop;
	UINT8 wav_stop;
	UINT8 mod_cnt_stop;

	INT32 env_vol_accu;
	UINT8 env_vol_stop;
	UINT8 env_vol_direction;
	UINT8 env_vol_period;

	INT32 env_mod_accu;
	UINT8 env_mod_stop;
	UINT8 env_mod_direction;
	UINT8 env_mod_period;

	INT32 filt_prev;
	INT32 filt_sa;
	INT32 filt_sb;
};

static fds_sound fds;

static void mapperFDS_reset()
{
	BurnLEDReset();

	memset(&fds, 0, sizeof(fds));

	// Auto-"Insertion/Swap" robot
	if (~DrvDips[0] & 2)
		FDS_FrameAction(INSERT_DISK);

	// Init 2khz lp filter coefficient @ 1.786mhz
	double omeg = exp(-2.0 * 3.1415926 * 2000 / (29781 * 60));
    fds.filt_sa = (INT32)(omeg * (double)(1 << 12));
    fds.filt_sb = (1 << 12) - fds.filt_sa;
}

static void mapperFDS_scan()
{
	SCAN_VAR(fds);
	SCAN_VAR(FDSFrameCounter);
	SCAN_VAR(FDSFrameAction);

	// 6000-7fff WRAM(cart_ext) is scanned by system
	ScanVar(Cart.PRGRom, 0x6000, "PRGRAM 8000-dfff");
}

static UINT8 fds_sound_read(UINT16 address)
{
	if (address >= 0x4040 && address <= 0x407f) {
		return fds.waveram[address & 0x3f] | (cpu_open_bus & 0xc0);
	}

	switch (address) {
		case 0x4090: return (fds.volgain & 0x3f) | (cpu_open_bus & 0xc0);
		case 0x4092: return (fds.modgain & 0x3f) | (cpu_open_bus & 0xc0);
	}

	return cpu_open_bus;
}

static void fds_sound_write(UINT16 address, UINT8 data)
{
	if (address >= 0x4040 && address <= 0x407f) {
		if (fds.wavwren_hold) {
			fds.waveram[address & 0x3f] = data & 0x3f;
		}
		return;
	}

	switch (address) {
		case 0x4080:
			fds.env_vol_stop = (data & 0x80) >> 7;
			fds.env_vol_direction = (data & 0x40) >> 6;
			fds.env_vol_period = data & 0x3f;
			if (data & 0x80) {
				fds.volgain = data & 0x3f;
			}
			break;
		case 0x4082:
			fds.wavefreq = (fds.wavefreq & 0x0f00) | data;
			break;
		case 0x4083:
			fds.wavefreq = (fds.wavefreq & 0x00ff) | ((data & 0xf) << 8);
			fds.env_all_stop = (data & 0x40) >> 6;
			fds.wav_stop = (data & 0x80) >> 7;

			if (data & 0x80) {
				// phase reset
				fds.wavepos = 0;
			}
			break;
		case 0x4084:
			fds.env_mod_stop = (data & 0x80) >> 7;
			fds.env_mod_direction = (data & 0x40) >> 6;
			fds.env_mod_period = data & 0x3f;
			if (data & 0x80) {
				fds.modgain = data & 0x3f;
			}
			break;
		case 0x4085:
			fds.mod_counter = data & 0x7f;
			fds.modphase &= 0x3f0000;
			break;
		case 0x4086:
			fds.modfreq = (fds.modfreq & 0x0f00) | data;
			break;
		case 0x4087:
			fds.modfreq = (fds.modfreq & 0x00ff) | ((data & 0xf) << 8);
			fds.mod_cnt_stop = (data & 0x80) >> 7;
			if (data & 0x80) {
				fds.mod_accu = 0;
			}
			break;
		case 0x4088:
			if (fds.mod_cnt_stop) {
				fds.modwaveram[(fds.modphase >> 16) & 0x3f] = data & 7;
				fds.modphase = (fds.modphase + 0x10000) & 0x3fffff;
				fds.modwaveram[(fds.modphase >> 16) & 0x3f] = data & 7;
				fds.modphase = (fds.modphase + 0x10000) & 0x3fffff;
			}
			break;
		case 0x4089:
			fds.volmaster = data & 3;
			fds.wavwren_hold = (data & 0x80) >> 7;
			break;
		case 0x408a:
			fds.env_master = data;
			break;
	}
}

static void env_vol()
{
	if (fds.env_vol_stop == 0) {
		fds.env_vol_accu--;
		if (fds.env_vol_accu <= 0) {
			fds.env_vol_accu += ((fds.env_vol_period + 1) * fds.env_master) << 3;
			if (fds.env_vol_direction) {
				if (fds.volgain < 0x3f) {
					fds.volgain++;
				}
			} else {
				if (fds.volgain > 0) {
					fds.volgain--;
				}
			}
		}
	}
}

static void env_mod()
{
	if (fds.env_mod_stop == 0) {
		fds.env_mod_accu--;
		if (fds.env_mod_accu <= 0) {
			fds.env_mod_accu += ((fds.env_mod_period + 1) * fds.env_master) << 3;
			if (fds.env_mod_direction) {
				if (fds.modgain < 0x3f) {
					fds.modgain++;
				}
			} else {
				if (fds.modgain > 0) {
					fds.modgain--;
				}
			}
		}
	}
}

static void mod_sweep()
{
	INT32 mod = 0;
	if (fds.mod_cnt_stop == 0) {
		fds.mod_accu += fds.modfreq;
		if(fds.mod_accu >= 0x10000) {
			fds.mod_accu -= 0x10000;
			const INT32 BIASTBL[8] = { 0, 1, 2, 4, 0, -4, -2, -1 };
			UINT8 w = fds.modwaveram[(fds.modphase >> 16) & 0x3f];
			fds.mod_counter = (w == 4) ? 0 : (fds.mod_counter + BIASTBL[w]) & 0x7f;
		}
		fds.modphase = (fds.modphase + fds.modfreq) & 0x3fffff;
	}

	if (fds.wav_stop == 0) {
		if (fds.modgain != 0) {
			INT32 pos = (fds.mod_counter < 64) ? fds.mod_counter : (fds.mod_counter-128);

			// nesdev to the rescue! :) (yet again!)
			INT32 temp = pos * ((fds.modgain < 0x20) ? fds.modgain : 0x20);
			INT32 remainder = temp & 0x0f;
			temp >>= 4;
			if (remainder > 0 && ~temp & 0x80) {
				temp += (pos < 0) ? -1 : 2;
			}

			if (temp >= 192) temp -= 256;
			else if (temp < -64) temp += 256;

			temp = fds.wavefreq * temp;
			remainder = temp & 0x3f;
			temp >>= 6;
			if (remainder >= 32) temp += 1;
			mod = temp;
		}

		fds.wavepos = (fds.wavepos + fds.wavefreq + mod) & 0x3fffff;
	}
}

static INT16 mapperFDS_mixer()
{
	UINT32 sample = 0;

	if (fds.env_all_stop == 0 && fds.wav_stop == 0 && fds.env_master != 0) {
		env_vol();
		env_mod();
	}

	mod_sweep();

	if (fds.wavwren_hold == 0) {
		// render sample
		INT32 voltab[4] = { 0x173, 0xf7, 0xb9, 0x95 };
		UINT8 envvol = (fds.volgain > 0x20) ? 0x20 : fds.volgain;
		sample = (fds.waveram[(fds.wavepos >> 16) & 0x3f] * envvol) * voltab[fds.volmaster];
		sample >>= 7;

		// 2khz lp filter
		sample = ((fds.filt_prev * fds.filt_sa) + (sample * fds.filt_sb)) >> 12;
		fds.filt_prev = sample;
	} else {
		sample = fds.filt_prev;
	}

	return sample;
}

static void mapperFDS_cycle()
{
	if (mapperFDS_irqenable) {
		mapperFDS_irqcount--;
		// m6502 cpu core isn't great.. compensate by running irq for 5 cycles longer.
		// Testcases: Marchen Veil HUD, KaetteKita Mario Bros (white bricks cutscene)
		if (mapperFDS_irqcount == 0xffff-5) {
			mapperFDS_irqcount = mapperFDS_irqlatch;
			if (mapperFDS_irqrepeat == 0) {
				mapperFDS_irqenable = 0;
			}
			M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
			mapperFDS_in_irq = 1;
		}
	}

	if (mapperFDS_diskirqcount > 0) {
		mapperFDS_diskirqcount--;
		if (mapperFDS_diskirqcount == 0) {
			if (mapperFDS_control & 0x80) {
				M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
				mapperFDS_eighth_bit = 1;
			}
		}
	}
}

static void mapperFDS_map()
{
	set_mirroring((mapperFDS_control & 8) ? HORIZONTAL : VERTICAL);
}

static UINT8 mapperFDS_read(UINT16 address)
{
	UINT8 ret = 0;

	if (mapperFDS_ioenable & 2 && address >= 0x4040 && address <= 0x4097)
		return fds_sound_read(address);

	if (~mapperFDS_ioenable & 1) return cpu_open_bus; // io disabled.

	switch (address) {
		case 0x4030:
			if (mapperFDS_in_irq) ret |= 1;
			if (mapperFDS_eighth_bit) ret |= 2;

			mapperFDS_in_irq = mapperFDS_eighth_bit = 0;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;

		case 0x4031:
			ret = 0xff;
			if (mapperFDS_diskinsert && mapperFDS_control & 0x04) {
				mapperFDS_diskaccessed = 1;

				ret = 0;

				switch (mapperFDS_blockid) {
					case DSK_FILEHDR:
						if (mapperFDS_diskaddr < mapperFDS_blocklen) {
							ret = fds_disk();
							switch (mapperFDS_diskaddr) {
								case 13: mapperFDS_filesize = ret; break;
								case 14:
									mapperFDS_filesize |= ret << 8;
									strncpy(fdsfile, (char*)&Cart.FDSDisk[(mapperFDS_diskside * 65500) + mapperFDS_blockstart + 3], 8);
									bprintf(0, _T("Read file: %S (size: %d)\n"), fdsfile, mapperFDS_filesize);
									break;
							}
							mapperFDS_diskaddr++;
						}
						break;
					default:
						if (mapperFDS_diskaddr < mapperFDS_blocklen) {
							ret = fds_disk();
							mapperFDS_diskaddr++;
						}
						break;
				}

				mapperFDS_arm_timer();
				mapperFDS_eighth_bit = 0;
				M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			}
			break;

		case 0x4032:
			ret = cpu_open_bus & 0xf8;
			if (mapperFDS_diskinsert == 0) {
				ret |= 4 /*write protect or eject*/ | 1 /*disk not inserted*/;
			}
			if ((mapperFDS_diskinsert == 0) || (~mapperFDS_control & 1/*motor off*/) || (mapperFDS_control & 2/*transfer reset*/)) {
				ret |= 2 /*disk not ready*/;
			}
			break;

		case 0x4033: // battery status of drive unit
			ret |= 0x80;
			break;
	}

	return ret;
}

static void mapperFDS_prg_write(UINT16 address, UINT8 data)
{
	if (address >= 0x8000 && address <= 0xdfff) {
		Cart.PRGRom[address & 0x7fff] = data;
	}
}

static void mapperFDS_write(UINT16 address, UINT8 data)
{
	if (mapperFDS_ioenable & 2 && address >= 0x4040 && address <= 0x4097)
		return fds_sound_write(address, data);

	if ((~mapperFDS_ioenable & 1) && address >= 0x4024) return; // io disabled.

	switch (address) {
		case 0x4020:
			mapperFDS_irqlatch = (mapperFDS_irqlatch & 0xff00) | data;
			break;
		case 0x4021:
			mapperFDS_irqlatch = (mapperFDS_irqlatch & 0x00ff) | (data << 8);
			break;

		case 0x4022:
			mapperFDS_irqenable = (data & 2) && (mapperFDS_ioenable & 1);
			mapperFDS_irqrepeat = data & 1;
			if (mapperFDS_irqenable) {
				mapperFDS_irqcount = mapperFDS_irqlatch;
			} else {
				M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
				mapperFDS_in_irq = 0;
			}
			break;

		case 0x4023:
			mapperFDS_ioenable = data;
			if (~data & 1) {
				mapperFDS_irqenable = 0;
				M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
				mapperFDS_in_irq = mapperFDS_eighth_bit = 0;
			}
			break;

		case 0x4024:
			if (mapperFDS_diskinsert && ~mapperFDS_control & 0x04) {

				if (mapperFDS_diskaccessed == 0) {
					mapperFDS_diskaccessed = 1;
					break;
				}

				switch (mapperFDS_blockid) {
					case DSK_FILEHDR:
						if (mapperFDS_diskaddr < mapperFDS_blocklen) {
							fds_disk() = data;
							switch (mapperFDS_diskaddr) {
								case 13: mapperFDS_filesize = data; break;
								case 14:
									mapperFDS_filesize |= data << 8;
									fdsfile[0] = 0;
									strncpy(fdsfile, (char*)&Cart.FDSDisk[(mapperFDS_diskside * 65500) + mapperFDS_blockstart + 3], 8);
									bprintf(0, _T("Write file: %S (size: %d)\n"), fdsfile, mapperFDS_filesize);
									break;
							}
							mapperFDS_diskaddr++;
						}
						break;
					default:
						if (mapperFDS_diskaddr < mapperFDS_blocklen) {
							fds_disk() = data;
							mapperFDS_diskaddr++;
						}
						break;
				}

			}
			break;

		case 0x4025:
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);

			//bprintf(0, _T("0x4025: nCurrentFrame %d  param %x   ppc %x\n"), nCurrentFrame, data, M6502GetPrevPC(-1));
			if (mapperFDS_diskinsert) {
				if (data & 0x40 && ~mapperFDS_control & 0x40) {
					mapperFDS_diskaccessed = 0;

					mapperFDS_arm_timer();

					// blockstart  - address of block on disk
					// diskaddr    - address relative to blockstart
					mapperFDS_blockstart += mapperFDS_diskaddr;
					mapperFDS_diskaddr = 0;

					mapperFDS_blockid++;
					if (mapperFDS_blockid > DSK_FILEDATA)
						mapperFDS_blockid = DSK_FILEHDR;

					switch (mapperFDS_blockid) {
						case DSK_VOLUME:
							mapperFDS_blocklen = 0x38;
							break;
						case DSK_FILECNT:
							mapperFDS_blocklen = 0x02;
							break;
						case DSK_FILEHDR:
							mapperFDS_blocklen = 0x10;
							break;
						case DSK_FILEDATA:		 // [blockid]<filedata>
							mapperFDS_blocklen = 0x01 + mapperFDS_filesize;
							break;
					}
				}

				if (data & 0x02) { // transfer reset
					mapperFDS_blockid = DSK_INIT;
					mapperFDS_blockstart = 0;
					mapperFDS_blocklen = 0;
					mapperFDS_diskaddr = 0;
					mapperFDS_arm_timer();
				}
				if (data & 0x40 && ~data & 2) { // turn on motor
					mapperFDS_arm_timer();
					BurnLEDSetStatus(0, 1);
					BurnLEDSetColor((mapperFDS_control & 0x04) ? LED_COLOR_GREEN : LED_COLOR_RED);
				} else {
					BurnLEDSetStatus(0, 0);
				}
			}
			mapperFDS_control = data;

			mapper_map(); // set mirror

			break;
	}
}

#undef mapperFDS_ioenable
#undef mapperFDS_control
#undef mapperFDS_eighth_bit
#undef mapperFDS_in_irq
#undef mapperFDS_irqenable
#undef mapperFDS_irqrepeat
#undef mapperFDS_irqcount
#undef mapperFDS_irqlatch
#undef mapperFDS_diskside
#undef mapperFDS_diskinsert
#undef mapperFDS_diskaccessed
#undef mapperFDS_diskaddr
#undef mapperFDS_diskirqcount
#undef mapperFDS_arm_timer
#undef mapperFDS_filesize
#undef mapperFDS_blockid
#undef mapperFDS_blockstart

// ---[ mapper 01 (MMC1)
#define mapper01_serialbyte	(mapper_regs[0x1f - 0])
#define mapper01_bitcount	(mapper_regs[0x1f - 1])
#define mapper01_lastwrite  (mapper_regs[0x1f - 2])
#define mapper01_exbits		(mapper_regs[0x1f - 3])
#define mapper01_prg2x		(mapper_regs[0x1f - 4])
static INT32 *mapper01_last_cyc = (INT32*)&mapper_regs16[0];

static void mapper01_write(UINT16 address, UINT8 data)
{
	if (address & 0x8000) {
		if (mega_cyc_counter - mapper01_last_cyc[0] < 2) {
			// https://wiki.nesdev.com/w/index.php/MMC1
			// Bill & Ted's Excellent Adventure resets the MMC1 by doing INC
			// on a ROM location containing $FF; the MMC1 sees the $FF
			// written back and ignores the $00 written on the next cycle.
			mapper01_last_cyc[0] = mega_cyc_counter;
			return;
		}
        if (data & 0x80) { // bit 7, mapper reset
			mapper01_bitcount = 0;
			mapper01_serialbyte = 0;
			mapper_regs[0] |= 0x0c;
			if (mapper_map) mapper_map();
		} else {
			mapper01_serialbyte |= (data & 1) << mapper01_bitcount;
			mapper01_bitcount++;
			if (mapper01_bitcount == 5) {
				UINT8 reg = (address >> 13) & 0x3;
				mapper_regs[reg] = mapper01_serialbyte;
				//bprintf(0, _T("mmc1_reg[%x]  =  %x\n"), reg, mapper01_serialbyte);
				switch (reg) {
					case 1: mapper01_lastwrite = 0; break;
					case 2: mapper01_lastwrite = 1; break;
				}
                mapper01_bitcount = 0;
                mapper01_serialbyte = 0;
                if (mapper_map) mapper_map();
			}
		}
		mapper01_last_cyc[0] = mega_cyc_counter;
    }
}

static void mapper01_exp_write(UINT16 address, UINT8 data) // 6000 - 7fff
{
	if (~mapper_regs[3] & 0x10) {
		Cart.WorkRAM[PRGExpMap + (address & 0x1fff)] = data;
	}
	cart_exp_write_abort = 1; // don't fall-through after callback!
}

static UINT8 mapper01_exp_read(UINT16 address)             // 6000 - 7fff
{
	return (~mapper_regs[3] & 0x10) ? Cart.WorkRAM[PRGExpMap + (address & 0x1fff)] : cpu_open_bus;
}

static void mapper01_map()
{
	mapper01_exbits = ((mapper01_lastwrite == 1) && (mapper_regs[0] & 0x10)) ? mapper_regs[2] : mapper_regs[1];

	if (Cart.WorkRAMSize > 0x2000) {
		// MMC1 Bankable WRAM/SRAM
		UINT8 page = 0;
		switch (Cart.WorkRAMSize) {
			case 0x8000: // 32k (Final Fantasy 1 & 2)
				page = (mapper_regs[1] >> 2) & 3;
				break;
			default:
				page = (mapper_regs[1] >> 3) & 1;
				break;
		}
		mapper_map_exp_prg(page);
	}

	INT32 bigcart = 0;

	if (Cart.PRGRomSize >= 0x80000) {
		bigcart = 1;
		// SUROM (512K/1M rom), use extra mapping bit
		mapper01_prg2x = mapper01_exbits & 0x10;
	}

	if (mapper_regs[0] & 0x8) {
		if (mapper_regs[0] & 0x4) {
            mapper_map_prg(16, 0, (mapper_regs[3] & 0xf) | mapper01_prg2x);
			if (bigcart) {
				mapper_map_prg(16, 1, 0xf | mapper01_prg2x);
			} else {
				mapper_map_prg(16, 1, -1);
			}
        } else {
            mapper_map_prg(16, 0, 0 | mapper01_prg2x);
            mapper_map_prg(16, 1, (mapper_regs[3] & 0xf) | mapper01_prg2x);
        }
	} else {
		mapper_map_prg(32, 0, ((mapper_regs[3] & 0xf) | mapper01_prg2x) >> 1);
	}

	if (mapper_regs[0] & 0x10) {
        mapper_map_chr( 4, 0, mapper_regs[1]);
        mapper_map_chr( 4, 1, mapper_regs[2]);
	} else {
		mapper_map_chr( 8, 0, mapper_regs[1] >> 1);
	}

	switch (mapper_regs[0] & 3) {
        case 0: set_mirroring(SINGLE_LOW); break;
        case 1: set_mirroring(SINGLE_HIGH); break;
        case 2: set_mirroring(VERTICAL); break;
        case 3: set_mirroring(HORIZONTAL); break;
	}
}
#undef mapper01_serialbyte
#undef mapper01_bitcount

// ---[ mapper 02 (UxROM)
static void mapper02_write(UINT16 address, UINT8 data)
{
	if (address & 0x8000) {
		mapper_regs[0] = data;
		mapper_map();
	}
}

static void mapper02_map()
{
    mapper_map_prg(16, 0, mapper_regs[0] & 0xff);
    mapper_map_prg(16, 1, -1);
    mapper_map_chr( 8, 0, 0);
}

// ---[ mapper 41 Caltron 6-in-1
#define mapper41_prg	(mapper_regs[0])
#define mapper41_chr	(mapper_regs[1])
#define mapper41_mirror	(mapper_regs[2])

static void mapper41_write(UINT16 address, UINT8 data)
{
	if (address >= 0x6000 && address <= 0x67ff) {
		mapper41_prg = address & 0x07;
		mapper41_chr = (mapper41_chr & 0x03) | ((address & 0x18) >> 1);
		mapper41_mirror = (address & 0x20) >> 5;
	}
	if (address >= 0x8000 && address <= 0xffff) {
		if (mapper41_prg & 4) {
			mapper41_chr = (mapper41_chr & 0x0c) | (data & 0x03);
		}
	}

	mapper_map();
}

static void mapper41_map()
{
	mapper_map_prg(32, 0, mapper41_prg);
	mapper_map_chr( 8, 0, mapper41_chr);

	set_mirroring((mapper41_mirror & 0x01) ? HORIZONTAL : VERTICAL);
}
#undef mapper41_prg
#undef mapper41_chr
#undef mapper41_mirror

// ---[ mapper 53 SuperVision 16-in-1
#define mapper53_reg(x)	(mapper_regs[0 + (x)])

static void mapper53_write(UINT16 address, UINT8 data)
{
	mapper53_reg((address & 0x8000) >> 15) = data;

	mapper_map();
}

static UINT8 mapper53_exp_read(UINT16 address)
{
	return Cart.PRGRom[PRGExpMap + (address & 0x1fff)];
}

static void mapper53_map()
{
	UINT8 bank = (mapper53_reg(0) & 0xf) << 3;
	mapper_map_exp_prg((0xf | (bank << 1)) + 4);
	if (mapper53_reg(0) & 0x10) {
		mapper_map_prg(16, 0, ((bank | (mapper53_reg(1) & 7)) + 2));
		mapper_map_prg(16, 1, ((bank | 7) + 2));
	} else {
		mapper_map_prg(32, 0, 0);
	}

	mapper_map_chr( 8, 0, 0);

	set_mirroring((mapper53_reg(0) & 0x20) ? HORIZONTAL : VERTICAL);
}
#undef mapper53_reg

// ---[ mapper 104 Golden Five, Pegasus 5-in-1
#define mapper104_prg(x)	(mapper_regs[0 + (x)])

static void mapper104_write(UINT16 address, UINT8 data)
{
	if (address >= 0x8000 && address <= 0xbfff) {
		if (data & 0x08) {
			mapper104_prg(0) = (mapper104_prg(0) & 0x0f) | (data & 0x07) << 4;
			mapper104_prg(1) = (data & 0x07) << 4;
		}
	}
	if (address >= 0xc000 && address <= 0xffff) {
		mapper104_prg(0) = (mapper104_prg(0) & 0x70) | (data & 0x0f);
	}

	mapper_map();
}

static void mapper104_map()
{
	mapper_map_prg(16, 0, mapper104_prg(0));
	mapper_map_prg(16, 1, mapper104_prg(1) | 0x0f);
	mapper_map_chr( 8, 0, 0);
}
#undef mapper104_prg

// ---[ mapper 190 Magic Kid Googoo
#define mapper190_prg		(mapper_regs[0])
#define mapper190_chr(x)	(mapper_regs[1 + (x)])
static void mapper190_write(UINT16 address, UINT8 data)
{
	switch (address & ~0x1fff) {
		case 0x8000:
			mapper190_prg = data & 0x07;
			break;
		case 0xa000:
			mapper190_chr(address & 0x03) = data & 0x3f;
			break;
		case 0xc000:
			mapper190_prg = (data & 0x07) | 0x08;
			break;
	}

	mapper_map();
}

static void mapper190_map()
{
    mapper_map_prg(16, 0, mapper190_prg);
    mapper_map_prg(16, 1, 0); // second 16k bank mapped to first prg bank

	mapper_map_chr( 2, 0, mapper190_chr(0));
	mapper_map_chr( 2, 1, mapper190_chr(1));
	mapper_map_chr( 2, 2, mapper190_chr(2));
	mapper_map_chr( 2, 3, mapper190_chr(3));

	set_mirroring(VERTICAL);
}
#undef mapper190_prg
#undef mapper190_chr

// ---[ mapper 193 NTDEC TC-112 (War in the Gulf, Fighting Hero, ...)
#define mapper193_prg		(mapper_regs[0])
#define mapper193_chr(x)	(mapper_regs[1 + (x)])
#define mapper193_mirror	(mapper_regs[4])

static void mapper193_write(UINT16 address, UINT8 data)
{
	switch (address & 0xe007) {
		case 0x6000:
		case 0x6001:
		case 0x6002:
			mapper193_chr(address & 3) = data;
			break;
		case 0x6003:
			mapper193_prg = data;
			break;
		case 0x6004:
			mapper193_mirror = data;
			break;
	}

	mapper_map();
}

static void mapper193_map()
{
	mapper_map_prg( 8, 0, mapper193_prg);
	mapper_map_prg( 8, 1, -3);
	mapper_map_prg( 8, 2, -2);
	mapper_map_prg( 8, 3, -1);

	mapper_map_chr( 4, 0, mapper193_chr(0) >> 2);

	mapper_map_chr( 2, 2, mapper193_chr(1) >> 1);
	mapper_map_chr( 2, 3, mapper193_chr(2) >> 1);

	set_mirroring((mapper193_mirror & 0x01) ? HORIZONTAL : VERTICAL);
}
#undef mapper193_prg
#undef mapper193_chr
#undef mapper193_mirror

// ---[ mapper 15 Contra 168-in-1 Multicart
#define mapper15_prg		(mapper_regs[0])
#define mapper15_prgbit		(mapper_regs[1])
#define mapper15_prgmode	(mapper_regs[2])
#define mapper15_mirror		(mapper_regs[3])

static void mapper15_write(UINT16 address, UINT8 data)
{
	mapper15_mirror = data & 0x40;
	mapper15_prg = (data & 0x7f) << 1;
	mapper15_prgbit = (data & 0x80) >> 7;
	mapper15_prgmode = address & 0xff; // must ignore weird writes.

	mapper_map();
}

static void mapper15_map()
{
	switch (mapper15_prgmode) {
		case 0x00:
			mapper_map_prg( 8, 0, (mapper15_prg + 0) ^ mapper15_prgbit);
			mapper_map_prg( 8, 1, (mapper15_prg + 1) ^ mapper15_prgbit);
			mapper_map_prg( 8, 2, (mapper15_prg + 2) ^ mapper15_prgbit);
			mapper_map_prg( 8, 3, (mapper15_prg + 3) ^ mapper15_prgbit);
			break;
		case 0x01:
			mapper_map_prg( 8, 0, (mapper15_prg + 0) | mapper15_prgbit);
			mapper_map_prg( 8, 1, (mapper15_prg + 1) | mapper15_prgbit);
			mapper_map_prg( 8, 2, (mapper15_prg + 0) | 0x0e | mapper15_prgbit);
			mapper_map_prg( 8, 3, (mapper15_prg + 1) | 0x0e | mapper15_prgbit);
			break;
		case 0x02:
			mapper_map_prg( 8, 0, (mapper15_prg + 0) | mapper15_prgbit);
			mapper_map_prg( 8, 1, (mapper15_prg + 0) | mapper15_prgbit);
			mapper_map_prg( 8, 2, (mapper15_prg + 0) | mapper15_prgbit);
			mapper_map_prg( 8, 3, (mapper15_prg + 0) | mapper15_prgbit);
			break;
		case 0x03:
			mapper_map_prg( 8, 0, (mapper15_prg + 0) | mapper15_prgbit);
			mapper_map_prg( 8, 1, (mapper15_prg + 1) | mapper15_prgbit);
			mapper_map_prg( 8, 2, (mapper15_prg + 0) | mapper15_prgbit);
			mapper_map_prg( 8, 3, (mapper15_prg + 1) | mapper15_prgbit);
			break;
	}

	mapper_map_chr_ramrom( 8, 0, 0, (mapper15_prgmode == 3) ? MEM_RAM_RO : MEM_RAM);

	set_mirroring((mapper15_mirror & 0x40) ? HORIZONTAL : VERTICAL);
}

#undef mapper15_prg
#undef mapper15_prgbit
#undef mapper15_prgmode
#undef mapper15_mirror

// ---[ mapper 389 Caltron 9-in-1
#define mapper389_reg8  (mapper_regs[0])
#define mapper389_reg9  (mapper_regs[1])
#define mapper389_rega  (mapper_regs[2])

static void mapper389_write(UINT16 address, UINT8 data)
{
	switch (address & 0xf000) {
		case 0x8000: mapper389_reg8 = address & 0xff; break;
		case 0x9000: mapper389_reg9 = address & 0xff; break;
		default: mapper389_rega = address & 0xff; break;
	}

	mapper_map();
}

static void mapper389_map()
{
	if (mapper389_reg9 & 0x02) { // UNROM-064 mode
		mapper_map_prg(16, 0, (mapper389_reg8 >> 2) | ((mapper389_rega >> 2) & 0x03));
		mapper_map_prg(16, 1, (mapper389_reg8 >> 2) | 0x03);
	} else { // NROM-256 mode
		mapper_map_prg(32, 0, mapper389_reg8 >> 3);
	}
	mapper_map_chr( 8, 0, ((mapper389_reg9 & 0x38) >> 1) | (mapper389_rega & 0x03));

	set_mirroring((mapper389_reg8 & 0x01) ? HORIZONTAL : VERTICAL);
}

#undef mapper389_reg8
#undef mapper389_reg9
#undef mapper389_rega

// ---[ mapper 216 (??) Magic Jewelry 2
#define mapper216_prg   (mapper_regs[0])
#define mapper216_chr   (mapper_regs[1])
static void mapper216_write(UINT16 address, UINT8 data)
{
	if (address >= 0x8000) {
		mapper216_prg = address & 1;
		mapper216_chr = (address & 0x000e) >> 1;
		mapper_map();
	}
}

static UINT8 mapper216_read(UINT16 address)
{
	return 0; // must be 0 @ 0x5000
}

static void mapper216_map()
{
    mapper_map_prg(32, 0, mapper216_prg);
	mapper_map_chr( 8, 0, mapper216_chr);
}
#undef mapper216_prg
#undef mapper216_chr

// ---[ mapper 132 (TXC Micro Genius) Qi Wang
#define mapper132_reg(x)   (mapper_regs[0 + (x)])
#define mapper132_reghi    (mapper_regs[8])
static void mapper132_write(UINT16 address, UINT8 data)
{
	if (address >= 0x4100 && address <= 0x4103) {
		mapper132_reg(address & 3) = data;
		//mapper_map(); // only 8000+!
	}

	if (address >= 0x8000) {
		mapper132_reghi = data;
		mapper_map();
	}
}

static UINT8 mapper132_read(UINT16 address)
{
	if (address == 0x4100) {
		return (mapper132_reg(1) ^ mapper132_reg(2)) | 0x40;
	}
	return cpu_open_bus;
}

static void mapper132_map()
{
    mapper_map_prg(32, 0, (mapper132_reg(2) >> 2) & 1);
	mapper_map_chr( 8, 0, mapper132_reg(2) & 3);
}
#undef mapper132_reg
#undef mapper132_reghi

// ---[ mapper 30 (UNROM-512)
#define mapper30_mirroring_en   (mapper_regs[1])
static void mapper30_write(UINT16 address, UINT8 data)
{
	if (address >= 0xc000) {
		mapper_regs[0] = data;
		mapper_map();
	}
}

static void mapper30_map()
{
    mapper_map_prg(16, 0, mapper_regs[0] & 0x1f);
    mapper_map_prg(16, 1, -1);
	mapper_map_chr( 8, 0, (mapper_regs[0] >> 5) & 0x03);
	if (mapper30_mirroring_en) {
		set_mirroring((mapper_regs[0] & 0x80) ? SINGLE_HIGH : SINGLE_LOW);
	}
}

// ---[ mapper 03 (CNROM)
#define mapper03_need_update   (mapper_regs[1])
static void mapper03_write(UINT16 address, UINT8 data)
{
	if (address & 0x8000) {
		if (NESMode & BUS_CONFLICTS) {
			data &= mapper_prg_read(address);
		}
		mapper_regs[0] = data;
		mapper03_need_update = 1;

		if (Cart.Crc != 0xab29ab28) // dropzone gets delayed a bit.
			mapper_map();
	}
}

static void mapper03_map()
{
    mapper_map_chr( 8, 0, mapper_regs[0]);
}

// dropzone likes to change the chr bank too soon sometimes, this causes a line
// through the player when he is near the bottom of the screen.
// Solution: delay chr bank switches until after sprites load
static void mapper03_cycle()
{
	if (pixel > 321 && mapper03_need_update) {
		mapper03_need_update = 0;
		mapper_map();
	}

}
#undef mapper03_need_update

// ---[ mapper 04 (mmc3) & mmc3-based: 12, 76, 95, 108, 115, 118, 119, 189, 262
#define mapper4_banksel         (mapper_regs[0x1f - 0])
#define mapper4_mirror			(mapper_regs[0x1f - 1])
#define mapper4_irqlatch 		(mapper_regs[0x1f - 2])
#define mapper4_irqcount		(mapper_regs[0x1f - 3])
#define mapper4_irqenable		(mapper_regs[0x1f - 4])
#define mapper4_irqreload		(mapper_regs[0x1f - 5])
#define mapper4_writeprotect    (mapper_regs[0x1f - 6])
#define mapper12_lowchr			(mapper_regs16[0x1f - 0])
#define mapper12_highchr		(mapper_regs16[0x1f - 1])
#define mapper04_vs_prottype    (mapper_regs16[0x1f - 2])
#define mapper04_vs_protidx	    (mapper_regs16[0x1f - 3])
#define mapper115_prg           (mapper_regs[0x1f - 7])
#define mapper115_chr           (mapper_regs[0x1f - 8])
#define mapper115_prot          (mapper_regs[0x1f - 9])
#define mapper258_reg           (mapper_regs[0x1f - 0xa])
#define mapper262_reg           (mapper_regs[0x1f - 0xa])
#define mapper189_reg           (mapper_regs[0x1f - 0xa]) // same as 262
#define mapper268_reg(x)        (mapper_regs[(0x1f - 0xa) + (x)])
// mapper 165 mmc3 w/mmc4-like 4k chr banking latch
#define mapper165_chrlatch(x)   (mapper_regs[(0x1f - 0x0a) + (x)])
#define mapper165_update        (mapper_regs[0x1f - 0xb])

static UINT8 mapper04_vs_rbi_tko_prot(UINT16 address)
{
	static const UINT8 protdata[2][0x20] = {
		{
			0xff, 0xff, 0xff, 0xff, 0xb4, 0xff, 0xff, 0xff,
			0xff, 0x6f, 0xff, 0xff, 0xff, 0xff, 0x94, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff
		},
		{
			0xff, 0xbf, 0xb7, 0x97, 0x97, 0x17, 0x57, 0x4f,
			0x6f, 0x6b, 0xeb, 0xa9, 0xb1, 0x90, 0x94, 0x14,
			0x56, 0x4e, 0x6f, 0x6b, 0xeb, 0xa9, 0xb1, 0x90,
			0xd4, 0x5c, 0x3e, 0x26, 0x87, 0x83, 0x13, 0x00
		}

	};
	switch (address) {
		case 0x54ff:
			return 0x05;
		case 0x5678:
			return mapper04_vs_protidx ^ 1;
		case 0x578f:
			return 0x81 | ((mapper04_vs_protidx) ? 0x50 : 0x08);
		case 0x5567:
			mapper04_vs_protidx ^= 1;
			return 0x3e ^ ((mapper04_vs_protidx) ? 0x09 : 0x00);
		case 0x5e00:
			mapper04_vs_protidx = 0;
			break;
		case 0x5e01:
			return protdata[mapper04_vs_prottype][mapper04_vs_protidx++ & 0x1f];
	}
	return cpu_open_bus;
}

static void mapper04_write(UINT16 address, UINT8 data)
{
	if (address & 0x8000) {
        switch (address & 0xE001) {
            case 0x8000: mapper4_banksel = data; break;
            case 0x8001: mapper_regs[(mapper4_banksel & 0x7)] = data; break;
			case 0xA000: mapper4_mirror = ~data & 1; break;
			case 0xA001: mapper4_writeprotect = ~data & 1; break;
            case 0xC000: mapper4_irqlatch = data; break;
            case 0xC001: mapper4_irqreload = 1; break;
            case 0xE000: mapper4_irqenable = 0; M6502SetIRQLine(0, CPU_IRQSTATUS_NONE); break;
            case 0xE001: mapper4_irqenable = 1; break;
        }
		mapper_map();
    }
}

static void mapper04_map()
{
    mapper_map_prg(8, 1, mapper_regs[7]);

    if (~mapper4_banksel & 0x40) {
        mapper_map_prg(8, 0, mapper_regs[6]);
        mapper_map_prg(8, 2, -2);
    } else {
        mapper_map_prg(8, 0, -2);
        mapper_map_prg(8, 2, mapper_regs[6]);
    }

    if (~mapper4_banksel & 0x80) {
		mapper_map_chr(2, 0, (mapper_regs[0] + mapper12_lowchr) >> 1);
        mapper_map_chr(2, 1, (mapper_regs[1] + mapper12_lowchr) >> 1);

		mapper_map_chr(1, 4, mapper_regs[2] + mapper12_highchr);
		mapper_map_chr(1, 5, mapper_regs[3] + mapper12_highchr);
		mapper_map_chr(1, 6, mapper_regs[4] + mapper12_highchr);
		mapper_map_chr(1, 7, mapper_regs[5] + mapper12_highchr);
	} else {
		mapper_map_chr(1, 0, mapper_regs[2] + mapper12_lowchr);
		mapper_map_chr(1, 1, mapper_regs[3] + mapper12_lowchr);
		mapper_map_chr(1, 2, mapper_regs[4] + mapper12_lowchr);
		mapper_map_chr(1, 3, mapper_regs[5] + mapper12_lowchr);

		mapper_map_chr(2, 2, (mapper_regs[0] + mapper12_highchr) >> 1);
		mapper_map_chr(2, 3, (mapper_regs[1] + mapper12_highchr) >> 1);
	}

	if (Cart.Mirroring != 4)
		set_mirroring(mapper4_mirror ? VERTICAL : HORIZONTAL);
}

static void mapper258_map()
{
	if (mapper258_reg & 0x80) {
		// mmc3 prg-override (similar to mapper 115)
		if (mapper258_reg & 0x20) {
			mapper_map_prg(32, 0, (mapper258_reg & 0x7) >> 1);
		} else {
			mapper_map_prg(16, 0, mapper258_reg & 0x7);
			mapper_map_prg(16, 1, mapper258_reg & 0x7);
		}
	} else {
		mapper_map_prg(8, 1, mapper_regs[7] & 0xf);

		if (~mapper4_banksel & 0x40) {
			mapper_map_prg(8, 0, mapper_regs[6] & 0xf);
			mapper_map_prg(8, 2, -2);
		} else {
			mapper_map_prg(8, 0, -2);
			mapper_map_prg(8, 2, mapper_regs[6] & 0xf);
		}
		mapper_map_prg(8, 3, -1);
	}

    if (~mapper4_banksel & 0x80) {
		mapper_map_chr(2, 0, (mapper_regs[0] + mapper12_lowchr) >> 1);
        mapper_map_chr(2, 1, (mapper_regs[1] + mapper12_lowchr) >> 1);

		mapper_map_chr(1, 4, mapper_regs[2] + mapper12_highchr);
		mapper_map_chr(1, 5, mapper_regs[3] + mapper12_highchr);
		mapper_map_chr(1, 6, mapper_regs[4] + mapper12_highchr);
		mapper_map_chr(1, 7, mapper_regs[5] + mapper12_highchr);
	} else {
		mapper_map_chr(1, 0, mapper_regs[2] + mapper12_lowchr);
		mapper_map_chr(1, 1, mapper_regs[3] + mapper12_lowchr);
		mapper_map_chr(1, 2, mapper_regs[4] + mapper12_lowchr);
		mapper_map_chr(1, 3, mapper_regs[5] + mapper12_lowchr);

		mapper_map_chr(2, 2, (mapper_regs[0] + mapper12_highchr) >> 1);
		mapper_map_chr(2, 3, (mapper_regs[1] + mapper12_highchr) >> 1);
	}

	if (Cart.Mirroring != 4)
		set_mirroring(mapper4_mirror ? VERTICAL : HORIZONTAL);
}

static UINT8 mapper258_read(UINT16 address)
{
	const UINT8 lut[8] = { 0, 0, 0, 1, 2, 4, 0xf, 0 };

	return cpu_open_bus | lut[address & 7];
}

static void mapper258_write(UINT16 address, UINT8 data)
{
	if ((address & 0x7) == 0x0) {
		mapper258_reg = data;

		mapper_map();
	}
}

static void mapper76_map()
{
	mapper_map_prg(8, 0, mapper_regs[6]);
    mapper_map_prg(8, 1, mapper_regs[7]);
	mapper_map_prg(8, 2, -2);

	mapper_map_chr(2, 0, mapper_regs[2]);
	mapper_map_chr(2, 1, mapper_regs[3]);
	mapper_map_chr(2, 2, mapper_regs[4]);
	mapper_map_chr(2, 3, mapper_regs[5]);

	if (Cart.Mirroring != 4)
		set_mirroring(mapper4_mirror ? VERTICAL : HORIZONTAL);
}

static void mapper95_map()
{
    mapper_map_prg(8, 1, mapper_regs[7]);

	mapper_map_prg(8, 0, mapper_regs[6]);
	mapper_map_prg(8, 2, -2);

	mapper_map_chr(2, 0, mapper_regs[0] >> 1);
	mapper_map_chr(2, 1, mapper_regs[1] >> 1);

	nametable_mapall((mapper_regs[0] >> 5) & 1, (mapper_regs[0] >> 5) & 1, (mapper_regs[1] >> 5) & 1, (mapper_regs[1] >> 5) & 1);

	mapper_map_chr(1, 4, mapper_regs[2]);
	mapper_map_chr(1, 5, mapper_regs[3]);
	mapper_map_chr(1, 6, mapper_regs[4]);
	mapper_map_chr(1, 7, mapper_regs[5]);
}

static void mapper268_prg(INT32 slot, INT32 bank)
{
	UINT32 prgmask = ((~mapper268_reg(3) & 0x10) >> 4) * 0x0f; // prg a13, a14, a15, a16 (GNROM)
		   prgmask |= (~mapper268_reg(0) & 0x40) >> 2; // prg a17~
		   prgmask |= (~mapper268_reg(1) & 0x80) >> 2; // prg a18~
		   prgmask |=   mapper268_reg(1) & 0x40;       // prg a19
		   prgmask |= ( mapper268_reg(1) & 0x20) << 2; // prg a20

	UINT32 prgbase  = (mapper268_reg(3) & 0x0e);       // prg a16, a15, a14 (GNROM)
		   prgbase |= (mapper268_reg(0) & 0x07) << 4;  // prg a19, a18, a17
		   prgbase |= (mapper268_reg(1) & 0x10) << 3;  // prg a20
		   prgbase |= (mapper268_reg(1) & 0x0c) << 6;  // prg a22, a21
		   prgbase |= (mapper268_reg(0) & 0x30) << 6;  // prg a24, a23
		   prgbase &= ~prgmask;

	UINT32 gnprgmask = (mapper268_reg(3) & 0x10) ? ((mapper268_reg(1) & 0x02) | 0x01) : 0x00;

	mapper_map_prg(8, slot, prgbase | (bank & prgmask) | (slot & gnprgmask));
}

// mapper 268: coolboy / mindkids
static void mapper268_map()
{
	if (mapper268_reg(3) & (0x40)) {
		bprintf(0, _T(" **  Mapper 268: 'weird' modes not supported yet.\n"));
	}

	mapper268_prg(1, mapper_regs[7]);

    if (~mapper4_banksel & 0x40) {
		mapper268_prg(0, mapper_regs[6]);
		mapper268_prg(2, -2);
    } else {
		mapper268_prg(0, -2);
		mapper268_prg(2, mapper_regs[6]);
    }
	mapper268_prg(3, -1);

	// chr - only mmc3 for now! -dink feb3. 2022
	INT32 chrmask = (mapper268_reg(0) & 0x80) ? 0x7f : 0xff;
	INT32 chrbase = ((mapper268_reg(0) & 0x08) << 4) & ~chrmask;

	if (~mapper4_banksel & 0x80) {
		mapper_map_chr(2, 0, ((mapper_regs[0] & chrmask) + chrbase) >> 1);
        mapper_map_chr(2, 1, ((mapper_regs[1] & chrmask) + chrbase) >> 1);

		mapper_map_chr(1, 4, (mapper_regs[2] & chrmask) + chrbase);
		mapper_map_chr(1, 5, (mapper_regs[3] & chrmask) + chrbase);
		mapper_map_chr(1, 6, (mapper_regs[4] & chrmask) + chrbase);
		mapper_map_chr(1, 7, (mapper_regs[5] & chrmask) + chrbase);
	} else {
		mapper_map_chr(1, 0, (mapper_regs[2] & chrmask) + chrbase);
		mapper_map_chr(1, 1, (mapper_regs[3] & chrmask) + chrbase);
		mapper_map_chr(1, 2, (mapper_regs[4] & chrmask) + chrbase);
		mapper_map_chr(1, 3, (mapper_regs[5] & chrmask) + chrbase);

		mapper_map_chr(2, 2, ((mapper_regs[0] & chrmask) + chrbase) >> 1);
		mapper_map_chr(2, 3, ((mapper_regs[1] & chrmask) + chrbase) >> 1);
	}

	if (Cart.Mirroring != 4)
		set_mirroring(mapper4_mirror ? VERTICAL : HORIZONTAL);
}

static void mapper268_write(UINT16 address, UINT8 data)
{
	if (address < 0x8000) {
		cart_exp_write_abort = (mapper4_writeprotect & 0xc0) == 0x80;

		if ((Cart.SubMapper == 1 && address >= 0x5000 && address <= 0x5fff) ||
		   (Cart.SubMapper == 0 && address >= 0x6000 && address <= 0x6fff))
		{
			if ((mapper268_reg(3) & 0x90) != 0x80) {
				mapper268_reg(address & 3) = data;

				mapper_map();
			}
		}
	} else {
		mapper04_write(address, data);
	}
}

static void mapper12_write(UINT16 address, UINT8 data)
{
	if (address & 0x4000) {
		mapper12_lowchr  = (data & 0x01) << 8;
		mapper12_highchr = (data & 0x10) << 4;

		mapper_map();
	}
}

static UINT8 mapper12_read(UINT16 address)
{
	UINT8 ret = cpu_open_bus;

	if (address & 0x4000) {
		ret = mapper12_lowchr >> 8;
		ret |= mapper12_highchr >> 4;
	}
	return ret;
}

static void mapper95_write(UINT16 address, UINT8 data)
{
	if ((address & 0x8001) == 0x8000)
		data &= 0x3f;

	mapper04_write(address & 0x8001, data);

	mapper_map();
}

static void mapper115_map()
{
	if (mapper115_prg & 0x80) {
		// mmc3 prg-override
		if (mapper115_prg & 0x20) {
			mapper_map_prg(32, 0, (mapper115_prg & 0xf) >> 1);
		} else {
			mapper_map_prg(16, 0, mapper115_prg & 0xf);
			mapper_map_prg(16, 1, mapper115_prg & 0xf);
		}
	} else {
		mapper_map_prg(8, 1, mapper_regs[7]);

		if (~mapper4_banksel & 0x40) {
			mapper_map_prg(8, 0, mapper_regs[6]);
			mapper_map_prg(8, 2, -2);
		} else {
			mapper_map_prg(8, 0, -2);
			mapper_map_prg(8, 2, mapper_regs[6]);
		}
		mapper_map_prg( 8, 3, -1);
	}

    if (~mapper4_banksel & 0x80) {
		mapper_map_chr(2, 0, (mapper_regs[0] + mapper12_lowchr) >> 1);
        mapper_map_chr(2, 1, (mapper_regs[1] + mapper12_lowchr) >> 1);

		mapper_map_chr(1, 4, mapper_regs[2] + mapper12_highchr);
		mapper_map_chr(1, 5, mapper_regs[3] + mapper12_highchr);
		mapper_map_chr(1, 6, mapper_regs[4] + mapper12_highchr);
		mapper_map_chr(1, 7, mapper_regs[5] + mapper12_highchr);
	} else {
		mapper_map_chr(1, 0, mapper_regs[2] + mapper12_lowchr);
		mapper_map_chr(1, 1, mapper_regs[3] + mapper12_lowchr);
		mapper_map_chr(1, 2, mapper_regs[4] + mapper12_lowchr);
		mapper_map_chr(1, 3, mapper_regs[5] + mapper12_lowchr);

		mapper_map_chr(2, 2, (mapper_regs[0] + mapper12_highchr) >> 1);
		mapper_map_chr(2, 3, (mapper_regs[1] + mapper12_highchr) >> 1);
	}

	if (Cart.Mirroring != 4)
		set_mirroring(mapper4_mirror ? VERTICAL : HORIZONTAL);
}

static UINT8 mapper115_read(UINT16 address)
{
	UINT8 ret = cpu_open_bus;

	switch (address) {
		case 0x5080:
			ret = mapper115_prot;
			break;
		case 0x6000:
			ret = mapper115_prg;
			break;
		case 0x6001:
			ret = mapper115_chr;
			break;
	}
	return ret;
}

static void mapper115_write(UINT16 address, UINT8 data)
{
	switch (address) {
		case 0x5080:
			mapper115_prot = data;
			break;
		case 0x6000:
			mapper115_prg = data;
			break;
		case 0x6001:
			mapper115_chr = data;
			mapper12_highchr = (data & 1) << 8;
			mapper12_lowchr = (data & 1) << 8;
			break;
	}

	mapper_map();
}

static void mapper118_map()
{
    mapper_map_prg(8, 1, mapper_regs[7]);

    if (~mapper4_banksel & 0x40) {
        mapper_map_prg(8, 0, mapper_regs[6]);
        mapper_map_prg(8, 2, -2);
    } else {
        mapper_map_prg(8, 0, -2);
        mapper_map_prg(8, 2, mapper_regs[6]);
    }

    if (~mapper4_banksel & 0x80) {
		mapper_map_chr(2, 0, mapper_regs[0] >> 1);
        mapper_map_chr(2, 1, mapper_regs[1] >> 1);
		nametable_map(0, (mapper_regs[0] >> 7) & 1);
		nametable_map(1, (mapper_regs[0] >> 7) & 1);
		nametable_map(2, (mapper_regs[1] >> 7) & 1);
		nametable_map(3, (mapper_regs[1] >> 7) & 1);

		mapper_map_chr(1, 4, mapper_regs[2]);
		mapper_map_chr(1, 5, mapper_regs[3]);
		mapper_map_chr(1, 6, mapper_regs[4]);
		mapper_map_chr(1, 7, mapper_regs[5]);
	} else {
		mapper_map_chr(1, 0, mapper_regs[2]);
		mapper_map_chr(1, 1, mapper_regs[3]);
		mapper_map_chr(1, 2, mapper_regs[4]);
		mapper_map_chr(1, 3, mapper_regs[5]);
		nametable_map(0, (mapper_regs[2] >> 7) & 1);
		nametable_map(1, (mapper_regs[3] >> 7) & 1);
		nametable_map(2, (mapper_regs[4] >> 7) & 1);
		nametable_map(3, (mapper_regs[5] >> 7) & 1);

		mapper_map_chr(2, 2, mapper_regs[0] >> 1);
		mapper_map_chr(2, 3, mapper_regs[1] >> 1);
	}
}

static void mapper189_write(UINT16 address, UINT8 data)
{
	mapper189_reg = data | (data >> 4);
	mapper_map();
}

static void mapper189_map()
{
	mapper_map_prg(32, 0, mapper189_reg & 0x07);

    if (~mapper4_banksel & 0x80) {
		mapper_map_chr(2, 0, mapper_regs[0] >> 1);
        mapper_map_chr(2, 1, mapper_regs[1] >> 1);

		mapper_map_chr(1, 4, mapper_regs[2]);
		mapper_map_chr(1, 5, mapper_regs[3]);
		mapper_map_chr(1, 6, mapper_regs[4]);
		mapper_map_chr(1, 7, mapper_regs[5]);
	} else {
		mapper_map_chr(1, 0, mapper_regs[2]);
		mapper_map_chr(1, 1, mapper_regs[3]);
		mapper_map_chr(1, 2, mapper_regs[4]);
		mapper_map_chr(1, 3, mapper_regs[5]);

		mapper_map_chr(2, 2, mapper_regs[0] >> 1);
		mapper_map_chr(2, 3, mapper_regs[1] >> 1);
	}

	if (Cart.Mirroring != 4)
		set_mirroring(mapper4_mirror ? VERTICAL : HORIZONTAL);
}

static void mapper119_map()
{
    mapper_map_prg(8, 1, mapper_regs[7]);

    if (~mapper4_banksel & 0x40) {
        mapper_map_prg(8, 0, mapper_regs[6]);
        mapper_map_prg(8, 2, -2);
    } else {
        mapper_map_prg(8, 0, -2);
        mapper_map_prg(8, 2, mapper_regs[6]);
    }

    if (~mapper4_banksel & 0x80) {
		mapper_map_chr_ramrom(2, 0, (mapper_regs[0] >> 1) & 0x1f, ((mapper_regs[0]) & 0x40) ? MEM_RAM : MEM_ROM);
		mapper_map_chr_ramrom(2, 1, (mapper_regs[1] >> 1) & 0x1f, ((mapper_regs[1]) & 0x40) ? MEM_RAM : MEM_ROM);

		mapper_map_chr_ramrom(1, 4, mapper_regs[2] & 0x3f, (mapper_regs[2] & 0x40) ? MEM_RAM : MEM_ROM);
		mapper_map_chr_ramrom(1, 5, mapper_regs[3] & 0x3f, (mapper_regs[3] & 0x40) ? MEM_RAM : MEM_ROM);
		mapper_map_chr_ramrom(1, 6, mapper_regs[4] & 0x3f, (mapper_regs[4] & 0x40) ? MEM_RAM : MEM_ROM);
		mapper_map_chr_ramrom(1, 7, mapper_regs[5] & 0x3f, (mapper_regs[5] & 0x40) ? MEM_RAM : MEM_ROM);
	} else {
		mapper_map_chr_ramrom(1, 0, mapper_regs[2] & 0x3f, (mapper_regs[2] & 0x40) ? MEM_RAM : MEM_ROM);
		mapper_map_chr_ramrom(1, 1, mapper_regs[3] & 0x3f, (mapper_regs[3] & 0x40) ? MEM_RAM : MEM_ROM);
		mapper_map_chr_ramrom(1, 2, mapper_regs[4] & 0x3f, (mapper_regs[4] & 0x40) ? MEM_RAM : MEM_ROM);
		mapper_map_chr_ramrom(1, 3, mapper_regs[5] & 0x3f, (mapper_regs[5] & 0x40) ? MEM_RAM : MEM_ROM);

		mapper_map_chr_ramrom(2, 2, (mapper_regs[0] >> 1) & 0x1f, ((mapper_regs[0]) & 0x40) ? MEM_RAM : MEM_ROM);
		mapper_map_chr_ramrom(2, 3, (mapper_regs[1] >> 1) & 0x1f, ((mapper_regs[1]) & 0x40) ? MEM_RAM : MEM_ROM);
	}

	if (Cart.Mirroring != 4)
		set_mirroring((mapper4_mirror) ? VERTICAL : HORIZONTAL);
}

static void mapper165_ppu_clock(UINT16 address)
{
	if (mapper165_update) {
		mapper_map();
		mapper165_update = 0;
	}

	switch (address & 0x3ff8) {
		case 0x0fd0:
			mapper165_chrlatch(0) = 0;
			mapper165_update = 1;
			break;
		case 0x0fe8:
			mapper165_chrlatch(0) = 1;
			mapper165_update = 1;
			break;
		case 0x1fd0:
			mapper165_chrlatch(1) = 2;
			mapper165_update = 1;
			break;
		case 0x1fe8:
			mapper165_chrlatch(1) = 4;
			mapper165_update = 1;
			break;
	}
}

static void mapper165_chrmap(INT32 slot, INT32 bank)
{
	mapper_map_chr_ramrom(4, slot, bank >> 2, (bank == 0x00) ? MEM_RAM : MEM_ROM);
}

static void mapper165_map()
{
    mapper_map_prg(8, 1, mapper_regs[7]);

    if (~mapper4_banksel & 0x40) {
        mapper_map_prg(8, 0, mapper_regs[6]);
        mapper_map_prg(8, 2, -2);
    } else {
        mapper_map_prg(8, 0, -2);
        mapper_map_prg(8, 2, mapper_regs[6]);
    }

	mapper165_chrmap(0, mapper_regs[mapper165_chrlatch(0)]);
	mapper165_chrmap(1, mapper_regs[mapper165_chrlatch(1)]);

	if (Cart.Mirroring != 4)
		set_mirroring(mapper4_mirror ? VERTICAL : HORIZONTAL);
}

static void mapper192_chrmap(INT32 slot, INT32 bank)
{
	mapper_map_chr_ramrom(1, slot, bank, (bank >= 0x8 && bank <= 0xb) ? MEM_RAM : MEM_ROM);
}

static void mapper192_map()
{
    mapper_map_prg(8, 1, mapper_regs[7]);

    if (~mapper4_banksel & 0x40) {
        mapper_map_prg(8, 0, mapper_regs[6]);
        mapper_map_prg(8, 2, -2);
    } else {
        mapper_map_prg(8, 0, -2);
        mapper_map_prg(8, 2, mapper_regs[6]);
    }

    if (~mapper4_banksel & 0x80) {
		mapper192_chrmap(0, mapper_regs[0] & 0xfe);
		mapper192_chrmap(1, mapper_regs[0] | 0x01);
        mapper192_chrmap(2, mapper_regs[1] & 0xfe);
        mapper192_chrmap(3, mapper_regs[1] | 0x01);

		mapper192_chrmap(4, mapper_regs[2]);
		mapper192_chrmap(5, mapper_regs[3]);
		mapper192_chrmap(6, mapper_regs[4]);
		mapper192_chrmap(7, mapper_regs[5]);
	} else {
		mapper192_chrmap(0, mapper_regs[2]);
		mapper192_chrmap(1, mapper_regs[3]);
		mapper192_chrmap(2, mapper_regs[4]);
		mapper192_chrmap(3, mapper_regs[5]);

		mapper192_chrmap(4, mapper_regs[0] & 0xfe);
		mapper192_chrmap(5, mapper_regs[0] | 0x01);
		mapper192_chrmap(6, mapper_regs[1] & 0xfe);
		mapper192_chrmap(7, mapper_regs[1] | 0x01);
	}

	if (Cart.Mirroring != 4)
		set_mirroring(mapper4_mirror ? VERTICAL : HORIZONTAL);
}

// mapper 195: mmc3 + chrram banks 0-3, cpu r/w @ 5000-5fff)

static void mapper195_chrmap(INT32 slot, INT32 bank)
{
	mapper_map_chr_ramrom(1, slot, bank, (bank >= 0x0 && bank <= 0x3) ? MEM_RAM : MEM_ROM);
}

static void mapper195_map()
{
    mapper_map_prg(8, 1, mapper_regs[7]);

    if (~mapper4_banksel & 0x40) {
        mapper_map_prg(8, 0, mapper_regs[6]);
        mapper_map_prg(8, 2, -2);
    } else {
        mapper_map_prg(8, 0, -2);
        mapper_map_prg(8, 2, mapper_regs[6]);
    }

    if (~mapper4_banksel & 0x80) {
		mapper195_chrmap(0, mapper_regs[0] & 0xfe);
		mapper195_chrmap(1, mapper_regs[0] | 0x01);
        mapper195_chrmap(2, mapper_regs[1] & 0xfe);
        mapper195_chrmap(3, mapper_regs[1] | 0x01);

		mapper195_chrmap(4, mapper_regs[2]);
		mapper195_chrmap(5, mapper_regs[3]);
		mapper195_chrmap(6, mapper_regs[4]);
		mapper195_chrmap(7, mapper_regs[5]);
	} else {
		mapper195_chrmap(0, mapper_regs[2]);
		mapper195_chrmap(1, mapper_regs[3]);
		mapper195_chrmap(2, mapper_regs[4]);
		mapper195_chrmap(3, mapper_regs[5]);

		mapper195_chrmap(4, mapper_regs[0] & 0xfe);
		mapper195_chrmap(5, mapper_regs[0] | 0x01);
		mapper195_chrmap(6, mapper_regs[1] & 0xfe);
		mapper195_chrmap(7, mapper_regs[1] | 0x01);
	}

	if (Cart.Mirroring != 4)
		set_mirroring(mapper4_mirror ? VERTICAL : HORIZONTAL);
}

static void mapper195_write(UINT16 address, UINT8 data)
{
	if (address >= 0x5000 && address <= 0x5fff) {
		Cart.CHRRam[address&0xfff] = data;
	}
}

static UINT8 mapper195_read(UINT16 address)
{
	if (address >= 0x5000 && address <= 0x5fff) {
		return Cart.CHRRam[address&0xfff];
	}
	return cpu_open_bus;
}

static void mapper262_map()
{
    mapper_map_prg(8, 1, mapper_regs[7]);

    if (~mapper4_banksel & 0x40) {
        mapper_map_prg(8, 0, mapper_regs[6]);
        mapper_map_prg(8, 2, -2);
    } else {
        mapper_map_prg(8, 0, -2);
        mapper_map_prg(8, 2, mapper_regs[6]);
    }

	if (mapper262_reg & 0x40) {
	   mapper_map_chr_ramrom(8, 0, 0, MEM_RAM);
	} else {
		mapper_set_chrtype(MEM_ROM);

		if (~mapper4_banksel & 0x80) {
			mapper_map_chr(1, 0, (mapper_regs[0] & 0xfe) + ((mapper262_reg & 8) << 5));
			mapper_map_chr(1, 1, (mapper_regs[0] | 0x01) + ((mapper262_reg & 8) << 5));

			mapper_map_chr(1, 2, (mapper_regs[1] & 0xfe) + ((mapper262_reg & 4) << 6));
			mapper_map_chr(1, 3, (mapper_regs[1] | 0x01) + ((mapper262_reg & 4) << 6));

			mapper_map_chr(1, 4, mapper_regs[2] + ((mapper262_reg & 1) << 8));
			mapper_map_chr(1, 5, mapper_regs[3] + ((mapper262_reg & 1) << 8));
			mapper_map_chr(1, 6, mapper_regs[4] + ((mapper262_reg & 2) << 7));
			mapper_map_chr(1, 7, mapper_regs[5] + ((mapper262_reg & 2) << 7));
		} else {
			mapper_map_chr(1, 0, mapper_regs[2] + ((mapper262_reg & 8) << 5));
			mapper_map_chr(1, 1, mapper_regs[3] + ((mapper262_reg & 8) << 5));
			mapper_map_chr(1, 2, mapper_regs[4] + ((mapper262_reg & 4) << 6));
			mapper_map_chr(1, 3, mapper_regs[5] + ((mapper262_reg & 4) << 6));

			mapper_map_chr(1, 4, (mapper_regs[0] & 0xfe) + ((mapper262_reg & 1) << 8));
			mapper_map_chr(1, 5, (mapper_regs[0] | 0x01) + ((mapper262_reg & 1) << 8));

			mapper_map_chr(1, 6, (mapper_regs[1] & 0xfe) + ((mapper262_reg & 2) << 7));
			mapper_map_chr(1, 7, (mapper_regs[1] | 0x01) + ((mapper262_reg & 2) << 7));
		}
	}

	if (Cart.Mirroring != 4)
		set_mirroring(mapper4_mirror ? VERTICAL : HORIZONTAL);
}

static UINT8 mapper262_read(UINT16 address)
{
	if (address == 0x4100) {
		// 0 = Street heroes, 0xff = Shihun
		return 0;
	}

	return 0;
}

static void mapper262_write(UINT16 address, UINT8 data)
{
	if (address == 0x4100) {
		mapper262_reg = data;
		mapper_map();
	}
}

// ---[ mapper 451: Haratyler HG (AMIC flashrom)
#define mapper451_bank          (mapper_regs[0])

// flashrom simulator
#define flashrom_cmd            (mapper_regs[0x1f - 0x9]) // must not conflict with mmc3 for 406 (Haradius Zero)
#define flashrom_busy           (mapper_regs16[0x00])
#define flashrom_chiptype       (mapper_regs[0x1f - 0xa])
enum { AMIC = 0, MXIC };

static UINT8 flashrom_read(UINT16 address)
{
	if (flashrom_cmd == 0x90) { // flash chip identification
		//bprintf(0, _T("flashrom chip ID\n"));
		if (flashrom_chiptype == AMIC) {
			switch (address & 0x03) {
				case 0x00: return 0x37; // manufacturer ID
				case 0x01: return 0x86; // device ID
				case 0x03: return 0x7f; // Continuation ID
			}
		} else if (flashrom_chiptype == MXIC) {
			switch (address & 0x03) {
				case 0x00: return 0xc2; // manufacturer ID
				case 0x01: return 0xa4; // device ID
			}
		}
	}

	if (flashrom_busy > 0) { // flash chip program or "erasing sector or chip" mode (it takes time..)
		flashrom_busy--;

		UINT8 status = (flashrom_busy & 0x01) << 6; // toggle bit I
		switch (flashrom_cmd) {
			case 0x82: // embedded erase sector/chip
				status |= (flashrom_busy & 0x01) << 2; // toggle bit II
				status |= 1 << 3; // "erasing" status bit
				break;
			case 0xa0: // embedded program
				status |= ~mapper_prg_read_int(address) & 0x80;
				break;
		}
		//bprintf(0, _T("erase/pgm status  %x\n"), status);
		return status;
	}

	return mapper_prg_read_int(address);
}

static void flashrom_prg_write(UINT16 address, UINT8 data)
{
	Cart.PRGRom[PRGMap[(address & ~0x8000) / 0x2000] + (address & 0x1fff)] = data;
}

static void flashrom_write(UINT16 address, UINT8 data)
{
	if (data == 0xf0) {
		// read array / reset
		flashrom_cmd = 0;
		flashrom_busy = 0;
		return;
	}

	switch (flashrom_cmd) {
		case 0x00:
		case 0x80:
			if ((address & 0xfff) == 0x555 && data == 0xaa)
				flashrom_cmd++;
			break;
		case 0x01:
		case 0x81:
			if (((address & 0xfff) == 0x2aa ||
				 (address & 0xfff) == 0xaaa) && data == 0x55)
				flashrom_cmd++;
			break;
		case 0x02:
			if ((address & 0xfff) == 0x555)
				flashrom_cmd = data;
			break;
		case 0x82: {
			switch (data) {
				case 0x10:
					bprintf(0, _T("mapper %d: flashrom - full flash erase not impl. (will break game!)\n"), Cart.Mapper);
					flashrom_busy = Cart.PRGRomSize / 0x100; // fake it
					break;
				case 0x30:
					bprintf(0, _T("mapper %d: flashrom - sector erase.  addr %x [%x]\n"), Cart.Mapper, address, (PRGMap[(address & ~0x8000) / 0x2000] & 0x7f0000));
					for (INT32 i = 0; i < 0x10000; i++) {
						Cart.PRGRom[(PRGMap[(address & ~0x8000) / 0x2000] & 0x7f0000) + i] = 0xff;
					}
					flashrom_busy = 0xffff;
					break;
			}
			break;
		}
		case 0xa0:
			flashrom_prg_write(address, data);
			flashrom_busy = 8;
			flashrom_cmd = 0;
			break;
	}
}

static void mapper451_scan()
{
	ScanVar(&Cart.PRGRom[0x50000], 0x10000, "Mapper451 HighScore Sector");
}

static void mapper451_write(UINT16 address, UINT8 data)
{
	flashrom_write(address, data);

	if (address & 0x8000) {
        switch (address & 0xe000) {
			case 0xa000: {
				mapper4_mirror = ~address & 1;
				mapper_map();
				break;
			}
			case 0xc000: {
				mapper4_irqlatch = (address & 0xff) - 1;
				mapper4_irqreload = 0;
				if ((address & 0xff) == 0xff) {
					mapper4_irqlatch = 0;
					mapper4_irqenable = 0;
					M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
				} else {
					mapper4_irqenable = 1;
				}
				break;
			}
			case 0xe000: {
				mapper451_bank = address & 3;
				mapper_map();
				break;
			}
        }
    }
}

static void mapper451_map()
{
	mapper_map_prg(8, 0, (0x00) );
	mapper_map_prg(8, 1, (0x10 + (mapper451_bank & 1) + ((mapper451_bank & 2) << 2)) );
	mapper_map_prg(8, 2, (0x20 + (mapper451_bank & 1) + ((mapper451_bank & 2) << 2)) );
    mapper_map_prg(8, 3, (0x30) );

	mapper_map_chr(8, 0, mapper451_bank & 1);

	set_mirroring(mapper4_mirror ? VERTICAL : HORIZONTAL);
}

// mapper 406 - haradius zero (mmc3 + MXIC flash chip(prg,rw) + MXIC flash chip(chr,r)
// dev notes: https://pastebin.com/9UH8yGg6
static void mapper406_write(UINT16 address, UINT8 data)
{
	flashrom_write(address, data);
	mapper04_write((address & 0xfffe) | ((address >> 1) & 1), data);
}

static void mapper406_scan()
{
	ScanVar(&Cart.PRGRom[0x50000], 0x10000, "Mapper406 HighScore Sector");
}

#undef flashrom_cmd
#undef flashrom_busy

static UINT8 *mmc5_mask; // mmc3/mmc5 ppumask-sniffer // 0x18 = rendering
static UINT8 *mmc5_ctrl; // mmc3/mmc5 ppuctrl-sniffer // 0x20 = 8x16 spr

static void mapper04_scanline()
{
	if (NESMode & ALT_MMC3 && (mmc5_mask[0] & 0x18) == 0x00) {
		return;
	}

	{ // a12 counter stop
		const INT32 spr = (mmc5_ctrl[0] & 0x08) >> 3;
		const INT32 bg = (mmc5_ctrl[0] & 0x10) >> 4;
		const INT32 _16 = (mmc5_ctrl[0] & 0x20) >> 5;

		// if spr table == bg table data, a12 will never clock
		// in 16x8 sprite mode, a12 will "most likely" clock

		if (spr == bg && _16 == 0) return;
	}

	INT32 cnt = mapper4_irqcount;
	if (mapper4_irqcount == 0 || mapper4_irqreload) {
		mapper4_irqreload = 0;
		mapper4_irqcount = mapper4_irqlatch;
	} else {
		mapper4_irqcount--;
	}

	if (cnt && mapper4_irqenable && mapper4_irqcount == 0) {
		if ((~NESMode & ALT_MMC3 && mmc5_mask[0] & 0x18) || NESMode & ALT_MMC3) { // aka if (RENDERING)
			M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
		}
	}
}

//#undef mapper4_mirror // used in mapper_init()
#undef mapper4_irqlatch
#undef mapper4_irqcount
#undef mapper4_irqenable
#undef mapper4_irqreload
#undef mapper4_banksel

// ---[ mapper 05 (MMC5) Castlevania III, Uchuu Keibitai SDF
// PPU Hooks
static UINT8 *mmc5_nt_ram; // pointer to our ppu's nt ram
#define MMC5RENDERING (mmc5_mask[0] & 0x18)
#define MMC58x16 (mmc5_ctrl[0] & 0x20)

// Mapper registers & ram
static UINT8 mmc5_expram[1024];

#define mmc5_prgmode		(mapper_regs[0x0])
#define mmc5_chrmode		(mapper_regs[0x1])
#define mmc5_prgprot1		(mapper_regs[0x2])
#define mmc5_prgprot2		(mapper_regs[0x3])
#define mmc5_expram_mode	(mapper_regs[0x4])
#define mmc5_mirror(x)		(mapper_regs[0x1b + (x)])
#define mmc5_filltile		(mapper_regs[0x5])
#define mmc5_fillcolor		(mapper_regs[0x6])
#define mmc5_prgexp			(mapper_regs[0x7])

#define mmc5_prg(x)			(mapper_regs16[0x4 + (x)])
#define mmc5_chr(x)			(mapper_regs16[0x10 + (x)])
#define mmc5_upperchr		(mapper_regs[0x8])

#define mmc5_split			(mapper_regs[0x9])
#define mmc5_splitside		(mapper_regs[0xa])
#define mmc5_splittile		(mapper_regs[0xb])
#define mmc5_splitscroll	(mapper_regs[0xc])
#define mmc5_splitscrollmod (mapper_regs[0xd])
#define mmc5_splitchr		(mapper_regs[0xe])

#define mmc5_irqenable		(mapper_regs[0xf])
#define mmc5_irqcompare		(mapper_regs[0x10])
#define mmc5_irqpend		(mapper_regs[0x11])

#define mmc5_mult0			(mapper_regs[0x12])
#define mmc5_mult1			(mapper_regs[0x13])

#define mmc5_irqcount		(mapper_regs[0x14])
#define mmc5_inframe		(mapper_regs[0x15])

#define mmc5_lastchr		(mapper_regs[0x16])
#define mmc5_expramattr		(mapper_regs[0x17])

#define mmc5_pcmwrmode		(mapper_regs[0x18])
#define mmc5_pcmirq			(mapper_regs[0x19])
#define mmc5_pcmdata		(mapper_regs[0x1a])

enum { CHR_GUESS = 0, CHR_TILE, CHR_SPRITE, CHR_LASTREG };

static void mapper5_reset()
{
	memset(&mmc5_expram, 0xff, sizeof(mmc5_expram));

	mmc5_prgmode = 3;
	mmc5_chrmode = 3;
	mmc5_prgexp = 7;

	mmc5_prg(0) = 0x7f;
	mmc5_prg(1) = 0x7f;
	mmc5_prg(2) = 0x7f;
	mmc5_prg(3) = 0x7f;

	mmc5_filltile = 0xff;
	mmc5_fillcolor = 0xff;
	mmc5_mult0 = 0xff;
	mmc5_mult1 = 0xff; // default

	mmc5_pcmwrmode = 0x00;
	mmc5_pcmirq = 0x00;
	mmc5_pcmdata = 0x00;
}

static void mapper5_scan()
{
	SCAN_VAR(mmc5_expram);
}

static INT16 mapper5_mixer()
{
	return (INT16)(mmc5_pcmdata << 4);
}

static void mmc5_mapchr(UINT8 type)
{
	// https://wiki.nesdev.com/w/index.php/MMC5#CHR_mode_.28.245101.29
	// When using 8x8 sprites, only registers $5120-$5127 are used. Registers $5128-$512B are completely ignored.
	// When using 8x16 sprites, registers $5120-$5127 specify banks for sprites, registers $5128-$512B apply to background tiles, and the last set of registers written to (either $5120-$5127 or $5128-$512B) will be used for I/O via PPUDATA ($2007).

	if (~MMC58x16) mmc5_lastchr = 0;

	UINT8 bank = (MMC58x16 == 0) || (MMC58x16 && type == CHR_SPRITE);

	if (MMC58x16 && type == CHR_LASTREG)
		bank = mmc5_lastchr < 8;

	UINT8 banks[0x2][0xf] = {
/*bg*/  { 0xb, 0xb, 0xb, 0x9, 0xb, 0x9, 0xb, 0x8, 0x9, 0xa, 0xb, 0x8, 0x9, 0xa, 0xb },
/*spr*/ { 0x7, 0x3, 0x7, 0x1, 0x3, 0x5, 0x7, 0x0, 0x1, 0x2, 0x3, 0x4, 0x5, 0x6, 0x7 } };

	switch (mmc5_chrmode) {
		case 0:
			mapper_map_chr(8, 0, mmc5_chr(banks[bank][0x0]));
			break;

		case 1:
			mapper_map_chr(4, 0, mmc5_chr(banks[bank][0x1]));
			mapper_map_chr(4, 1, mmc5_chr(banks[bank][0x2]));
			break;

		case 2:
			mapper_map_chr(2, 0, mmc5_chr(banks[bank][0x3]));
			mapper_map_chr(2, 1, mmc5_chr(banks[bank][0x4]));
			mapper_map_chr(2, 2, mmc5_chr(banks[bank][0x5]));
			mapper_map_chr(2, 3, mmc5_chr(banks[bank][0x6]));
			break;

		case 3:
			mapper_map_chr(1, 0, mmc5_chr(banks[bank][0x7]));
			mapper_map_chr(1, 1, mmc5_chr(banks[bank][0x8]));
			mapper_map_chr(1, 2, mmc5_chr(banks[bank][0x9]));
			mapper_map_chr(1, 3, mmc5_chr(banks[bank][0xa]));
			mapper_map_chr(1, 4, mmc5_chr(banks[bank][0xb]));
			mapper_map_chr(1, 5, mmc5_chr(banks[bank][0xc]));
			mapper_map_chr(1, 6, mmc5_chr(banks[bank][0xd]));
			mapper_map_chr(1, 7, mmc5_chr(banks[bank][0xe]));
			break;
	}
}

static void mapper5_map()
{
	switch (mmc5_prgmode) {
		case 0:
			mapper_map_prg(32, 0, (mmc5_prg(3) & 0x7c) >> 2);
			break;

		case 1:
			mapper_map_prg(16, 0, (mmc5_prg(1) & 0x7e) >> 1, (mmc5_prg(1) & 0x80) ? MEM_ROM : MEM_RAM);
			mapper_map_prg(16, 1, (mmc5_prg(3) & 0x7e) >> 1);
			break;

		case 2:
			mapper_map_prg(16, 0, (mmc5_prg(1) & 0x7e) >> 1, (mmc5_prg(1) & 0x80) ? MEM_ROM : MEM_RAM);
			mapper_map_prg(8, 2, mmc5_prg(2) & 0x7f, (mmc5_prg(2) & 0x80) ? MEM_ROM : MEM_RAM);
			mapper_map_prg(8, 3, mmc5_prg(3) & 0x7f);
			break;

		case 3:
			mapper_map_prg(8, 0, mmc5_prg(0) & 0x7f, (mmc5_prg(0) & 0x80) ? MEM_ROM : MEM_RAM);
			mapper_map_prg(8, 1, mmc5_prg(1) & 0x7f, (mmc5_prg(1) & 0x80) ? MEM_ROM : MEM_RAM);
			mapper_map_prg(8, 2, mmc5_prg(2) & 0x7f, (mmc5_prg(2) & 0x80) ? MEM_ROM : MEM_RAM);
			mapper_map_prg(8, 3, mmc5_prg(3) & 0x7f);
			break;
	}

	mapper_map_exp_prg(mmc5_prgexp);

	// Note: mapper5_ppu_clk() takes care of chr banking
}

static UINT8 mapper5_read(UINT16 address)
{
	if (address >= 0x5000 && address <= 0x5015) {
		switch (address) {
			case 0x5010: {
				bprintf(0, _T("mmc5 irq ack\n"));
				UINT8 ret = ((mmc5_pcmirq & 1) << 7) | (~mmc5_pcmwrmode & 1);
				mmc5_pcmirq &= ~1; // clear flag
				M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
				return ret;
			}
			default:
				return nesapuRead(0, (address & 0x1f) | 0x80);
		}
	}

	if (address >= 0x5c00 && address <= 0x5fff) {
		return (mmc5_expram_mode & 2) ? mmc5_expram[address & 0x3ff] : cpu_open_bus;
	}

	switch (address) {
		case 0x5204: {
			UINT8 ret =	(mmc5_irqpend << 7) | (mmc5_inframe << 6) | (cpu_open_bus & 0x3f);
			mmc5_irqpend = 0;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			return ret;
		}

		case 0x5205:
		case 0x5206: return ((mmc5_mult0 * mmc5_mult1) >> ((address & 0x2) << 2)) & 0xff;
	}

	return cpu_open_bus;
}


static void mapper5_exp_write(UINT16 address, UINT8 data) // 6000 - 7fff
{
	Cart.WorkRAM[PRGExpMap + (address & 0x1fff)] = data;
	cart_exp_write_abort = 1;
}

static UINT8 mapper5_exp_read(UINT16 address)             // 6000 - 7fff
{
	return Cart.WorkRAM[PRGExpMap + (address & 0x1fff)];
}

static void mapper5_prg_write(UINT16 address, UINT8 data)
{
	if (mmc5_prgprot1 == 0x02 && mmc5_prgprot2 == 0x01) {
		mapper_prg_write(address, data);
	}
}

static void mapper5_write(UINT16 address, UINT8 data)
{
	if (address >= 0x5000 && address <= 0x5015) {
		// Audio section
		switch (address) {
			case 0x5010:
				mmc5_pcmwrmode = ~data & 1;
				mmc5_pcmirq = data & 0x80;
				break;
			case 0x5011:
				if (mmc5_pcmwrmode) {
					if (data == 0x00 && mmc5_pcmirq) {
						mapper_irq(0);
						mmc5_pcmirq |= 0x01;
					}
					mmc5_pcmdata = data;
				}
				break;
			default:
				nesapuWrite(0, (address & 0x1f) | 0x80, data);
				break;
		}
		return;
	}

	if (address >= 0x5c00 && address <= 0x5fff) {
		switch (mmc5_expram_mode) {
			case 0:
			case 1:
				mmc5_expram[address & 0x3ff] = (MMC5RENDERING) ? data : 0x00;
				break;
			case 2:
				mmc5_expram[address & 0x3ff] = data;
				break;
        }
	}

	switch (address) {
		case 0x5100: mmc5_prgmode = data & 3; break;
		case 0x5101: mmc5_chrmode = data & 3; break;
		case 0x5102: mmc5_prgprot1 = data; break;
		case 0x5103: mmc5_prgprot2 = data; break;
		case 0x5104: mmc5_expram_mode = data & 3; break;
		case 0x5105:
			mmc5_mirror(0) = (data >> 0) & 3;
			mmc5_mirror(1) = (data >> 2) & 3;
			mmc5_mirror(2) = (data >> 4) & 3;
			mmc5_mirror(3) = (data >> 6) & 3;
			break;

		case 0x5106: mmc5_filltile = data; break;
		case 0x5107: mmc5_fillcolor = ((data & 3) << 6) | ((data & 3) << 4) | ((data & 3) << 2) | (data & 3); break;

		case 0x5113: mmc5_prgexp = data & 7; break;
		case 0x5114:
		case 0x5115:
		case 0x5116:
		case 0x5117: mmc5_prg(address & 3) = data; break;

		case 0x5120:
		case 0x5121:
		case 0x5122:
		case 0x5123:
		case 0x5124:
		case 0x5125:
		case 0x5126:
		case 0x5127:
		case 0x5128:
		case 0x5129:
		case 0x512a:
		case 0x512b:
			mmc5_chr(address & 0xf) = (mmc5_upperchr << 8) | data;
			mmc5_lastchr = (MMC58x16) ? (address & 0xf) : 0;
			break;

		case 0x5130: mmc5_upperchr = data & 3; break;

		case 0x5200:
			mmc5_splittile = data & 0x1f;
			mmc5_splitside = data & 0x40;
			mmc5_split = (data & 0x80) >> 7;
			break;
		case 0x5201:
			mmc5_splitscroll = data >> 3;
			mmc5_splitscrollmod = (mmc5_splitscroll < 30 ? 30 : 32);
			break;
		case 0x5202: mmc5_splitchr = data; break;

		case 0x5203: mmc5_irqcompare = data; break;
		case 0x5204:
			mmc5_irqenable = (data & 0x80) >> 7;
			M6502SetIRQLine(0, (mmc5_irqenable && mmc5_irqpend) ? CPU_IRQSTATUS_ACK : CPU_IRQSTATUS_NONE);
			break;

		case 0x5205: mmc5_mult0 = data; break;
		case 0x5206: mmc5_mult1 = data; break;
	}

    mapper_map();
}

static UINT8 mapper5_ntread(UINT16 address)
{
	if (mmc5_expram_mode == 1) {
		if ((address & 0x3c0) < 0x3c0) {
			UINT8 temp = mmc5_expram[(32 * ((address >> 5) & 0x1f)) + (address & 0x1f)];
			mmc5_expramattr = (temp & 0xc0) >> 6;
            mapper_map_chr(4, 0, (mmc5_upperchr << 6) | (temp & 0x3f));
            mapper_map_chr(4, 1, (mmc5_upperchr << 6) | (temp & 0x3f));
        } else {
			return (mmc5_expramattr << 6) | (mmc5_expramattr << 4) | (mmc5_expramattr << 2) | mmc5_expramattr;
        }
    }

	if (mmc5_split && mmc5_expram_mode <= 1) {
        UINT8 cur_tile = ((pixel >> 3) + 2) % 32;

		if ((mmc5_splitside && cur_tile >= mmc5_splittile) || (!mmc5_splitside && cur_tile < mmc5_splittile)) {
                mapper_map_chr(4, 0, mmc5_splitchr);
                mapper_map_chr(4, 1, mmc5_splitchr);
                UINT8 y = ((scanline >> 3) + mmc5_splitscroll) % mmc5_splitscrollmod;
				address = (pixel & 2) ? ((y * 32) & 0x3e0) | cur_tile : 0x3c0 | ((y << 1) & 0x38) | (cur_tile >> 2);
                return mmc5_expram[address & 0x3ff];
		} else {
			mmc5_mapchr(CHR_GUESS);
		}
    }

	switch (mmc5_mirror((address & 0x1fff) >> 10)) {
		case 0: return mmc5_nt_ram[(address & 0x3ff)];
		case 1: return mmc5_nt_ram[(address & 0x3ff) + 0x400];
		case 2: return (mmc5_expram_mode <= 1) ? mmc5_expram[address & 0x3ff] : 0;
		case 3: return ((address & 0x3c0) == 0x3c0) ? mmc5_fillcolor : mmc5_filltile;
	}

	return 0x00;
}

static void mapper5_ntwrite(UINT16 address, UINT8 data)
{
	switch (mmc5_mirror((address & 0x1fff) >> 10)) {
		case 0: mmc5_nt_ram[(address & 0x3ff)] = data; break;
		case 1: mmc5_nt_ram[(address & 0x3ff) + 0x400] = data; break;
		case 2: if (mmc5_expram_mode <= 1) mmc5_expram[address & 0x3ff] = data; break;
		case 3: /* invalid */ break;
	}
}

static void mmc5_lineadvance()
{
	if (scanline <= 240) {
		if (mmc5_inframe == 0 && MMC5RENDERING) {
			mmc5_inframe = 1;
			mmc5_irqcount = 0;
			mmc5_irqpend = 0;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
		} else if (mmc5_inframe) {
			mmc5_irqcount++;
			if (mmc5_irqcount == mmc5_irqcompare) {
				mmc5_irqpend = 1;
				if (mmc5_irqenable) {
					M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
				}
			}
		}
	}
}

static void mapper5_ppu_clk(UINT16 address)
{
	if (MMC5RENDERING == 0 || scanline >= 240) {
		mmc5_inframe = 0;
		mmc5_mapchr(CHR_LASTREG);
		return;
	}

	switch (pixel) {
		case   1: if (~NESMode & ALT_TIMING2) mmc5_lineadvance(); break; // castelvania iii
		case  16: if (NESMode & ALT_TIMING2) mmc5_lineadvance(); break; // laser invasion prefers cycle 16 (grr..)
		case 257: mmc5_mapchr(CHR_SPRITE); break; // sprite bank switch
		case 321: mmc5_mapchr(CHR_TILE); break; // bg bank switch
	}
}
#undef mmc5_prgmode
#undef mmc5_chrmode
#undef mmc5_prgprot1
#undef mmc5_prgprot2
#undef mmc5_expram_mode
#undef mmc5_mirror
#undef mmc5_filltile
#undef mmc5_fillcolor
#undef mmc5_prgexp
#undef mmc5_prg
#undef mmc5_chr
#undef mmc5_upperchr
#undef mmc5_split
#undef mmc5_splitside
#undef mmc5_splittile
#undef mmc5_splitscroll
#undef mmc5_splitscrollmod
#undef mmc5_splitchr
#undef mmc5_irqenable
#undef mmc5_irqcompare
#undef mmc5_irqpend
#undef mmc5_mult0
#undef mmc5_mult1
#undef mmc5_irqcount
#undef mmc5_inframe
#undef mmc5_lastchr
#undef mmc5_expramattr

// ---[ mapper 07 (AxROM) Battle Toads, Marble Madness, RC Pro-Am
static void mapper07_write(UINT16 address, UINT8 data)
{
	if (address & 0x8000) {
		mapper_regs[0] = data;
		mapper_map();
	}
}

static void mapper07_map()
{
	set_mirroring((mapper_regs[0] & 0x10) ? SINGLE_HIGH : SINGLE_LOW);
    mapper_map_prg(32, 0, mapper_regs[0] & 0x7);
}

// ---[ mapper 09 (mmc2) / 10 (mmc4) Mike Tyson's Punch-Out!!, Fire Emblem
#define mapper9_prg				(mapper_regs[0xf - 0])
#define mapper9_chr_lo(x)		(mapper_regs[0xf - 1 - x])
#define mapper9_chr_hi(x)		(mapper_regs[0xf - 3 - x])
#define mapper9_chr_lo_C000		(mapper_regs[0xf - 5])
#define mapper9_chr_hi_E000		(mapper_regs[0xf - 6])
#define mapper9_mirror			(mapper_regs[0xf - 7])
#define mapper9_update          (mapper_regs[0xf - 8])

static void mapper09_write(UINT16 address, UINT8 data)
{
	if (address & 0x8000) {
		switch (address & 0xf000) {
			case 0xa000: mapper9_prg       = data & 0xf; break;
			case 0xb000: mapper9_chr_lo(0) = data & 0x1f; break;
			case 0xc000: mapper9_chr_lo(1) = data & 0x1f; break;
			case 0xd000: mapper9_chr_hi(0) = data & 0x1f; break;
			case 0xe000: mapper9_chr_hi(1) = data & 0x1f; break;
			case 0xf000: mapper9_mirror    = data & 0x1; break;
		}
		mapper_map();
	}
}

static void mapper09_map()
{
	set_mirroring((mapper9_mirror) ? HORIZONTAL : VERTICAL);
	mapper_map_prg( 8, 0, mapper9_prg);
	mapper_map_chr( 4, 0, mapper9_chr_lo(mapper9_chr_lo_C000));
	mapper_map_chr( 4, 1, mapper9_chr_hi(mapper9_chr_hi_E000));
}

static void mapper10_map()
{
	set_mirroring((mapper9_mirror) ? HORIZONTAL : VERTICAL);
	mapper_map_prg(16, 0, mapper9_prg);
	mapper_map_prg(16, 1, -1);
	mapper_map_chr( 4, 0, mapper9_chr_lo(mapper9_chr_lo_C000));
	mapper_map_chr( 4, 1, mapper9_chr_hi(mapper9_chr_hi_E000));
}

static void mapper09_ppu_clk(UINT16 busaddr)
{
	switch (busaddr & 0x3fff) {
		case 0x0fd8:
			mapper9_chr_lo_C000 = 0;
			mapper9_update = 1;
			break;
		case 0x0fe8:
			mapper9_chr_lo_C000 = 1;
			mapper9_update = 1;
			break;
	}

	switch (busaddr & 0x3ff8) {
		case 0x1fd8:
			mapper9_chr_hi_E000 = 0;
			mapper9_update = 1;
			break;
		case 0x1fe8:
			mapper9_chr_hi_E000 = 1;
			mapper9_update = 1;
			break;
	}

	if (mapper9_update) {
		// mmc2 needs update immediately on latch
		mapper9_update = 0;
		mapper_map();
	}
}

static void mapper10_ppu_clk(UINT16 busaddr)
{
	if (mapper9_update) {
		// mmc4 needs delayed update.  right window borders break in fire emblem
		// without
		mapper9_update = 0;
		mapper_map();
	}

	switch (busaddr & 0x3ff8) {
		case 0x0fd8:
			mapper9_chr_lo_C000 = 0;
			mapper9_update = 1;
			break;
		case 0x0fe8:
			mapper9_chr_lo_C000 = 1;
			mapper9_update = 1;
			break;
		case 0x1fd8:
			mapper9_chr_hi_E000 = 0;
			mapper9_update = 1;
			break;
		case 0x1fe8:
			mapper9_chr_hi_E000 = 1;
			mapper9_update = 1;
			break;
	}
}

#undef mapper9_prg
#undef mapper9_chr_lo
#undef mapper9_chr_hi
#undef mapper9_chr_lo_C000
#undef mapper9_chr_hi_E000
#undef mapper9_mirror
#undef mapper9_update

// ---[ mapper 99 (VS NES)
static void mapper99_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;

	mapper_map();
}

static void mapper99_map()
{
	mapper_map_chr(8, 0, (mapper_regs[0] >> 2) & 1);
	mapper_map_prg(32, 0, 0);
	if (Cart.PRGRomSize > 0x8000) {
		mapper_map_prg(8, 0, mapper_regs[0] & 4); // gumshoe
	}
}

// ---[ mapper 13 (CPROM) Videomation
static void mapper13_write(UINT16 address, UINT8 data)
{
	if (address & 0x8000) {
		mapper_regs[0] = data;
	}

	mapper_map();
}

static void mapper13_map()
{
	mapper_map_chr(4, 0, 0);
	mapper_map_chr(4, 1, mapper_regs[0] & 0x03);
}

// ---[ mapper 133 (Sachen ??) Jovial Race
static void mapper133_write(UINT16 address, UINT8 data)
{
	if ((address & 0x4100) == 0x4100) {
		mapper_regs[0] = data;
	}

	mapper_map();
}

static void mapper133_map()
{
	mapper_map_chr(8, 0, mapper_regs[0] & 0x03);
	mapper_map_prg(32, 0, (mapper_regs[0] >> 2) & 0x01);
}

// --[ mapper 16, 153, 159 - Bandai FCG
#define mapper16_mirror		(mapper_regs[0x1f - 0])
#define mapper16_irqenable	(mapper_regs[0x1f - 1])
#define mapper16_irqcount	(mapper_regs16[0x1f - 0])
#define mapper16_irqlatch	(mapper_regs16[0x1f - 1])

static void mapper16_write(UINT16 address, UINT8 data)
{
	switch (address & 0x000f) {
		case 0x00: case 0x01: case 0x02:
		case 0x03: case 0x04: case 0x05:
		case 0x06: case 0x07: case 0x08:
			mapper_regs[address & 0xf] = data;
			break; // mixed prg, chr

		case 0x09:
			mapper16_mirror = data & 0x3;
			break;

		case 0x0a:
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			mapper16_irqenable = data & 1;
			mapper16_irqcount = mapper16_irqlatch;
			break;

		case 0x0b:
			mapper16_irqlatch = (mapper16_irqlatch & 0xff00) | data;
			break;

		case 0x0c:
			mapper16_irqlatch = (mapper16_irqlatch & 0x00ff) | (data << 8);
			break;

		case 0x0d: // x24 i2c write
			return; // don't bother mapper_map(); below..
			break;
	}

	mapper_map();
}

static void mapper16_map()
{
	mapper_map_prg(16, 0, mapper_regs[8]);
	mapper_map_prg(16, 1, -1);

	mapper_map_chr( 1, 0, mapper_regs[0]);
	mapper_map_chr( 1, 1, mapper_regs[1]);
	mapper_map_chr( 1, 2, mapper_regs[2]);
	mapper_map_chr( 1, 3, mapper_regs[3]);
	mapper_map_chr( 1, 4, mapper_regs[4]);
	mapper_map_chr( 1, 5, mapper_regs[5]);
	mapper_map_chr( 1, 6, mapper_regs[6]);
	mapper_map_chr( 1, 7, mapper_regs[7]);

	switch (mapper16_mirror) {
		case 0: set_mirroring(VERTICAL); break;
		case 1: set_mirroring(HORIZONTAL); break;
		case 2: set_mirroring(SINGLE_LOW); break;
		case 3: set_mirroring(SINGLE_HIGH); break;
	}
}

static void mapper153_map()
{
	mapper_map_prg(16, 0, (mapper_regs[8] & 0xf) | ((mapper_regs[0] & 0x1) << 4));
	mapper_map_prg(16, 1, 0xf | ((mapper_regs[0] & 0x1) << 4));

	mapper_map_chr( 8, 0, 0);

	switch (mapper16_mirror) {
		case 0: set_mirroring(VERTICAL); break;
		case 1: set_mirroring(HORIZONTAL); break;
		case 2: set_mirroring(SINGLE_LOW); break;
		case 3: set_mirroring(SINGLE_HIGH); break;
	}
}

static void mapper16_cycle()
{
	if (mapper16_irqenable) {
		mapper16_irqcount--;

		if (mapper16_irqcount == 0xffff) {
			M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
			mapper16_irqenable = 0;
		}
	}
}
#undef mapper16_mirror
#undef mapper16_irqenable
#undef mapper16_irqcount

// --[ mapper 18 - Jaleco (Magical John, Pizza Pop etc)
#define mapper18_irqenable	(mapper_regs[0x1f - 0])
#define mapper18_mirror     (mapper_regs[0x1f - 1])
#define mapper18_prg(x)     (mapper_regs[0 + (x)])
#define mapper18_chr(x)     (mapper_regs[4 + (x)])
#define mapper18_irqcount	(mapper_regs16[0x1f - 0])
#define mapper18_irqlatch	(mapper_regs16[0x1f - 1])
#define mapper18_irqmask	(mapper_regs16[0x1f - 2])

static void nib2byte(UINT8 &byte, UINT8 nib, INT32 high)
{
	if (high == 0) {
		byte = (byte & 0xf0) | (nib & 0xf);
	} else {
		byte = (byte & 0x0f) | (nib & 0xf) << 4;
	}
}

static void mapper18_write(UINT16 address, UINT8 data)
{
	INT32 highbits = address & 1;

	switch (address & 0xf003) {
		case 0x8000: case 0x8001:
			nib2byte(mapper18_prg(0), data, highbits);
			break;
		case 0x8002: case 0x8003:
			nib2byte(mapper18_prg(1), data, highbits);
			break;
		case 0x9000: case 0x9001:
			nib2byte(mapper18_prg(2), data, highbits);
			break;

		case 0xa000: case 0xa001:
			nib2byte(mapper18_chr(0), data, highbits);
			break;
		case 0xa002: case 0xa003:
			nib2byte(mapper18_chr(1), data, highbits);
			break;

		case 0xb000: case 0xb001:
			nib2byte(mapper18_chr(2), data, highbits);
			break;
		case 0xb002: case 0xb003:
			nib2byte(mapper18_chr(3), data, highbits);
			break;

		case 0xc000: case 0xc001:
			nib2byte(mapper18_chr(4), data, highbits);
			break;
		case 0xc002: case 0xc003:
			nib2byte(mapper18_chr(5), data, highbits);
			break;

		case 0xd000: case 0xd001:
			nib2byte(mapper18_chr(6), data, highbits);
			break;
		case 0xd002: case 0xd003:
			nib2byte(mapper18_chr(7), data, highbits);
			break;

		case 0xe000: mapper18_irqlatch = (mapper18_irqlatch & 0xfff0) | (data & 0xf); break;
		case 0xe001: mapper18_irqlatch = (mapper18_irqlatch & 0xff0f) | ((data & 0xf) << 4); break;
		case 0xe002: mapper18_irqlatch = (mapper18_irqlatch & 0xf0ff) | ((data & 0xf) << 8); break;
		case 0xe003: mapper18_irqlatch = (mapper18_irqlatch & 0x0fff) | ((data & 0xf) << 12); break;
		case 0xf000:
			mapper18_irqcount = mapper18_irqlatch;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		case 0xf001: {
			const UINT16 masks[4] = { (1<<16) - 1, (1<<12) - 1, (1<<8) - 1, (1<<4) - 1 };
			UINT8 maskpos = 0;
			mapper18_irqenable = data & 1;
			switch (data & (2|4|8)) {
				case 8: maskpos = 3; break;
				case 4: maskpos = 2; break;
				case 2: maskpos = 1; break;
				default: maskpos = 0; break;
			}
			mapper18_irqmask = masks[maskpos];

			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		}
		case 0xf002: mapper18_mirror = data & 3; break;
	}

	mapper_map();
}

static void mapper18_map()
{
	mapper_map_prg( 8, 0, mapper18_prg(0));
	mapper_map_prg( 8, 1, mapper18_prg(1));
	mapper_map_prg( 8, 2, mapper18_prg(2));
	mapper_map_prg( 8, 3, -1);

	for (INT32 i = 0; i < 8; i++) {
		mapper_map_chr( 1, i, mapper18_chr(i));
	}

	switch (mapper18_mirror) {
		case 0: set_mirroring(HORIZONTAL); break;
		case 1: set_mirroring(VERTICAL); break;
		case 2: set_mirroring(SINGLE_LOW); break;
		case 3: set_mirroring(SINGLE_HIGH); break;
	}
}

static void mapper18_cycle()
{
	if (mapper18_irqenable) {
		UINT16 count = mapper18_irqcount & mapper18_irqmask;
		count--;
		if (count == 0) {
			mapper_irq(2); // 2 cyc delay fixes jeebies in pizza-pop (mid-hud) and magic john (intro)
		}
		mapper18_irqcount = (mapper18_irqcount & ~mapper18_irqmask) | (count & mapper18_irqmask);
	}
}
#undef mapper18_irqenable
#undef mapper18_mirror
#undef mapper18_prg
#undef mapper18_chr
#undef mapper18_irqcount
#undef mapper18_irqlatch
#undef mapper18_irqmask

// ---[ mapper 19 Namco 163 + WSG (wave sound gen)
#define mapper19_irqcount		(mapper_regs16[0x1f - 0])
#define mapper19_irqenable  	(mapper_regs[0x1f - 0])
#define mapper19_soundaddr		(mapper_regs[0x1f - 1])
#define mapper19_soundaddrinc	(mapper_regs[0x1f - 2])
#define mapper19_soundenable	(mapper_regs[0x1f - 3])
#define mapper19_chrram_lo		(mapper_regs[0x1f - 4])
#define mapper19_chrram_mid		(mapper_regs[0x1f - 5])
#define mapper19_prg(x)     	(mapper_regs[0 + (x)]) // 0 - 2
#define mapper19_mapper210      (mapper_regs[0x1f - 6])
#define namco340_mirror			(mapper_regs[0x1f - 7])

// since chr mapping is a bit "advanced", we use a little struct
// instead of the usual mapper registers. mapper19_scan() takes care
// of the state-stuff (as well as the WSG channel regs, etc)

struct namco163_chrstuff {
	UINT8 ram;
	UINT8 data;
};

static namco163_chrstuff n163_chr[12]; // chr mapping: 8 chr banks, 4 nt banks

static UINT8 mapper19_soundram[0x80];

struct namco163_channel {
	UINT32 enabled;
	UINT32 freq;
	UINT32 phase;
	UINT32 vol;
	UINT32 len;
	UINT32 address;
	UINT32 accum;
};

static namco163_channel n163_ch[8];
static INT32 n163_channels = 0;

static void namco163_process_channel(INT16 address, UINT8 data) {
	namco163_channel *c = &n163_ch[(address >> 3) & 7];

	switch (address & 7) {
		case 0x1:
			c->phase = (c->phase & ~0x0000ff) | data;
			break;
		case 0x3:
			c->phase = (c->phase & ~0x00ff00) | (data << 8);
			break;
		case 0x5:
			c->phase = (c->phase & ~0xff0000) | (data << 16);
			break;
		case 0x0:
			c->freq = (c->freq & ~0x0000ff) | data;
			break;
		case 0x2:
			c->freq = (c->freq & ~0x00ff00) | (data << 8);
			break;
		case 0x4:
			c->freq = (c->freq & ~0xff0000) | ((data & 3) << 16);
			c->len = 0x100 - (data & 0xfc);
			c->enabled = data >> 5;
#if 0
			bprintf(0, _T("ch %d enabled%X?\n"), (address / 8) & 7, c->enabled);
			bprintf(0, _T("len: %X\n"), c->len);
			bprintf(0, _T("phase: %X\n"), c->phase);
#endif
			break;
		case 0x6:
			c->address = data;
			break;
		case 0x7:
			c->vol = (data & 0xf) * 8;
			if (address == 0x7f) {
				n163_channels = (data >> 4) & 7;
			}
#if 0
			bprintf(0, _T("ch %d vol %X data %x.     channels %d    (cycle: %d)\n"), (address / 8) & 7, c->vol, data, n163_channels, M6502TotalCycles());
#endif
			break;
	}
}

static void namco163_update_phase(namco163_channel *c, INT32 ch)
{
	ch = 0x40 + ch * 8;
	mapper19_soundram[ch + 5] = (c->phase >> 16) & 0xff;
	mapper19_soundram[ch + 3] = (c->phase >>  8) & 0xff;
	mapper19_soundram[ch + 1] = (c->phase >>  0) & 0xff;
}

static UINT32 namco163_fetch_sample(namco163_channel *c)
{
	UINT32 wave_addr = c->address + (c->phase >> 16);
	UINT32 wave_sam = mapper19_soundram[(wave_addr & 0xff) >> 1];
	wave_sam >>= (wave_addr & 1) << 2; // LE-ordered nibble

	return ((wave_sam & 0xf) - 8) * c->vol;
}

static INT16 namco163_wavesynth()
{
	if (!mapper19_soundenable) return 0;

	INT32 sample = 0;

	for (INT32 ch = 7; ch >= (7 - n163_channels); ch--) {
		namco163_channel *c = &n163_ch[ch];

		if (c->enabled && c->vol) {
			sample += namco163_fetch_sample(c);

			if (c->accum == 0) {
				c->accum = (n163_channels + 1) * 0xf;
				c->phase = (c->phase + c->freq) % (c->len << 16);
				namco163_update_phase(c, ch);
			}

			c->accum--;
		}
	}

	return sample;
}

static void mapper19_reset()
{
	memset(&n163_chr, 0, sizeof(n163_chr));
	memset(&n163_ch, 0, sizeof(n163_ch));
	memset(&mapper19_soundram, 0, sizeof(mapper19_soundram));
}

static void mapper19_scan()
{
	SCAN_VAR(n163_chr); // chr/nt mapping regs

	SCAN_VAR(mapper19_soundram); // WSG soundram

	for (INT32 i = 0x40; i < 0x80; i++) { // rebuild n163_ch from soundram
		namco163_process_channel(i, mapper19_soundram[i]);
	}
}

static INT16 mapper19_mixer()
{
	INT32 sum = namco163_wavesynth();
	return (INT16)(sum * 1.75);
}

static void mapper19_write(UINT16 address, UINT8 data)
{
	switch (address & 0xf800) {
		case 0x4800:
			if (mapper19_soundaddr >= 0x40) {
				namco163_process_channel(mapper19_soundaddr, data);
			}
			mapper19_soundram[mapper19_soundaddr] = data;
			mapper19_soundaddr = (mapper19_soundaddr + mapper19_soundaddrinc) & 0x7f;
			return; // [sound core] avoid calling mapper_map() below
		case 0x5000:
			mapper19_irqcount = (mapper19_irqcount & 0xff00) | data;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		case 0x5800:
			mapper19_irqcount = (mapper19_irqcount & 0x00ff) | ((data & 0x7f) << 8);
			mapper19_irqenable = data & 0x80;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		case 0xe000:
			mapper19_prg(0) = data & 0x3f;
			mapper19_soundenable = !(data & 0x40);
			namco340_mirror = (data & 0xc0) >> 6;
			break;
		case 0xe800:
			mapper19_prg(1) = data & 0x3f;
			mapper19_chrram_lo = !(data & 0x40);
			mapper19_chrram_mid = !(data & 0x80);
			break;
		case 0xf000:
			mapper19_prg(2) = data & 0x3f;
			break;
		case 0xf800:
			mapper19_soundaddr = data & 0x7f;
			mapper19_soundaddrinc = (data >> 7) & 1;
			return; // [sound core] avoid calling mapper_map() below

		case 0x8000: case 0x8800:
		case 0x9000: case 0x9800: {
			INT32 bank = (address - 0x8000) >> 11;
			n163_chr[bank].ram = mapper19_chrram_lo;
			n163_chr[bank].data = data;
			break;
		}
		case 0xa000: case 0xa800:
		case 0xb000: case 0xb800: {
			INT32 bank = (address - 0x8000) >> 11;
			n163_chr[bank].ram = mapper19_chrram_mid;
			n163_chr[bank].data = data;
			break;
		}
		case 0xc000: case 0xc800:
		case 0xd000: case 0xd800: {
			INT32 nt = (address - 0xc000) >> 11;
			n163_chr[8 + nt].data = data;
			break;
		}
	}

	mapper_map();
}

static UINT8 mapper19_read(UINT16 address)
{
	UINT8 ret = 0;

	switch (address & 0xf800) {
		case 0x4800:
			ret = mapper19_soundram[mapper19_soundaddr];
			mapper19_soundaddr = (mapper19_soundaddr + mapper19_soundaddrinc) & 0x7f;
			break;

		case 0x5000:
			ret = mapper19_irqcount & 0xff;
			break;

		case 0x5800:
			ret = (mapper19_irqcount >> 8) & 0xff;
			break;
	}

	return ret;
}

static void mapper19_map()
{
	mapper_map_prg(8, 0, mapper19_prg(0));
	mapper_map_prg(8, 1, mapper19_prg(1));
	mapper_map_prg(8, 2, mapper19_prg(2));
	mapper_map_prg(8, 3, -1);

	for (INT32 i = 0; i < 8; i++) {
		if (mapper19_mapper210 == 0 && n163_chr[i].ram && n163_chr[i].data >= 0xe0) {
			mapper_set_chrtype(MEM_RAM);
			mapper_map_chr(1, i, n163_chr[i].data & 0x1f);
		} else {
			mapper_set_chrtype(MEM_ROM);
			mapper_map_chr(1, i, n163_chr[i].data);
		}
	}

	switch (mapper19_mapper210) {
		case 0:	// mapper 19 (namco163) only!
			for (INT32 i = 8; i < 12; i++) {
				if (n163_chr[i].data >= 0xe0) {
					nametable_map(i & 3, n163_chr[i].data & 1);
				} else {
					nametable_mapraw(i & 3, Cart.CHRRom + (n163_chr[i].data << 10), MEM_ROM);
				}
			}
			break;
		case 1: // mapper 210 submapper 1 (namco 175): hardwired h/v mirroring
			break;
		case 2: // mapper 210 submapper 2 (namco 340): selectable
			switch (namco340_mirror) {
				case 0: set_mirroring(SINGLE_LOW); break;
				case 1: set_mirroring(VERTICAL); break;
				case 2: set_mirroring(HORIZONTAL); break;
				case 3: set_mirroring(SINGLE_HIGH); break;
			}
			break;
	}
}

static void mapper19_cycle()
{
	if (mapper19_irqenable) {
		mapper19_irqcount++;

		if (mapper19_irqcount == 0x7fff) {
			M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
			mapper19_irqenable = 0;
		}
	}
}
#undef mapper19_irqcount
#undef mapper19_irqenable
#undef mapper19_soundaddr
#undef mapper19_soundaddrinc
#undef mapper19_soundenable
#undef mapper19_chrram_lo
#undef mapper19_chrram_mid
#undef mapper19_prg
//#undef mapper19_mapper210 -- used in mapper_init!
#undef namco340_mirror

// --[ mapper 21, 22, 23, 25 - Konami VRC2/VRC4
#define mapper23_prg(x)		(mapper_regs[0x00 + x])
#define mapper23_chr(x)		(mapper_regs[0x02 + x])
#define mapper23_chrhigh(x) (mapper_regs[0x0a + x])
#define mapper23_prgswap	(mapper_regs[0x12])
#define mapper23_irqrepeat	(mapper_regs[0x13])
#define mapper23_mirror		(mapper_regs[0x14])
#define vrc2and4_ines22		(mapper_regs[0x17])

#define mapper23_irqenable	(mapper_regs[0x18])
#define mapper23_irqlatch	(mapper_regs[0x19])
#define mapper23_irqmode	(mapper_regs[0x1a])
#define mapper23_irqcycle	(mapper_regs16[0x1f - 0])
#define mapper23_irqcount	(mapper_regs16[0x1f - 1])

static void vrc2vrc4_map()
{
	if (mapper23_prgswap & 2) {
		mapper_map_prg( 8, 0, -2);
		mapper_map_prg( 8, 1, mapper23_prg(1));
		mapper_map_prg( 8, 2, mapper23_prg(0));
		mapper_map_prg( 8, 3, -1);
	} else {
		mapper_map_prg( 8, 0, mapper23_prg(0));
		mapper_map_prg( 8, 1, mapper23_prg(1));
		mapper_map_prg( 8, 2, -2);
		mapper_map_prg( 8, 3, -1);
	}

	for (INT32 i = 0; i < 8; i++) {
		mapper_map_chr( 1, i, ((mapper23_chrhigh(i) << 4) | mapper23_chr(i)) >> vrc2and4_ines22);
	}

	switch (mapper23_mirror) {
		case 0: set_mirroring(VERTICAL); break;
		case 1: set_mirroring(HORIZONTAL); break;
		case 2: set_mirroring(SINGLE_LOW); break;
		case 3: set_mirroring(SINGLE_HIGH); break;
	}
}

static void vrc2vrc4_write(UINT16 address, UINT8 data)
{
	address &= 0xf003;
	if (address >= 0xb000 && address <= 0xe003) {
		UINT8 reg = ((address >> 1) & 1) | ((address - 0xb000) >> 11);
		if (address & 1) { // high 5-bits
			mapper23_chrhigh(reg) = data & 0x1f;
		} else { // low 4-bits
			mapper23_chr(reg) = data & 0xf;
		}
		mapper_map();
	} else {
		switch (address & 0xf003) {
			case 0x8000:
			case 0x8001:
			case 0x8002:
			case 0x8003:
				mapper23_prg(0) = data; // usually a 0x1f mask, some pirate carts/hacks want this unmasked to address larger prg than usual
				mapper_map();
				break;
			case 0xA000:
			case 0xA001:
			case 0xA002:
			case 0xA003:
				mapper23_prg(1) = data; // comment: same as above
				mapper_map();
				break;
			case 0x9000:
			case 0x9001:
				if (data != 0xff) { // Wai Wai World, what are you doing?
					mapper23_mirror = data & 3;
				}
				mapper_map();
				break;
			case 0x9002:
			case 0x9003:
				mapper23_prgswap = data;
				mapper_map();
				break;
			case 0xf000:
				mapper23_irqlatch = (mapper23_irqlatch & 0xf0) | (data & 0xf);
				M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
				break;
			case 0xf001:
				mapper23_irqlatch = (mapper23_irqlatch & 0x0f) | ((data & 0xf) << 4);
				M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
				break;
			case 0xf002:
				mapper23_irqrepeat = data & 1;
				mapper23_irqenable = data & 2;
				mapper23_irqmode = data & 4; // 1 cycle, 0 scanline
				if (data & 2) {
					mapper23_irqcycle = 0;
					mapper23_irqcount = mapper23_irqlatch;
				}
				M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
				break;
			case 0xf003:
				mapper23_irqenable = mapper23_irqrepeat;
				M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
				break;
		}
	}
}

static void vrc2vrc4_cycle()
{
	if (mapper23_irqenable) {
		if (mapper23_irqmode) { // cycle mode
			mapper23_irqcount++;
			if (mapper23_irqcount >= 0x100) {
				mapper_irq(5);
				mapper23_irqcount = mapper23_irqlatch;
			}
		} else {
			mapper23_irqcycle += 3; // scanline mode
			if (mapper23_irqcycle >= 341) {
				mapper23_irqcycle -= 341;
				mapper23_irqcount++;
				if (mapper23_irqcount == 0x100) {
					mapper_irq(5);
					mapper23_irqcount = mapper23_irqlatch;
				}
			}
		}
	}
}

static void mapper21_write(UINT16 address, UINT8 data)
{
	address = (address & 0xf000) | ((address >> 1) & 0x3);

	vrc2vrc4_write(address, data);
}

static void mapper22_write(UINT16 address, UINT8 data)
{
	address |= ((address >> 2) & 0x3);

	vrc2vrc4_write((address & 0xf000) | ((address >> 1) & 1) | ((address << 1) & 2), data);
}

static void mapper23_write(UINT16 address, UINT8 data)
{
	address |= ((address >> 2) & 0x3) | ((address >> 4) & 0x3) | ((address >> 6) & 0x3);

	vrc2vrc4_write(address, data);
}

#define mapper25_write		mapper22_write   // same address line hookup/swapping

// --[ mapper 24, 26 (Konami VRC6)
#define mapper24_prg(x)		(mapper_regs[0x00 + x])
#define mapper24_chr(x)		(mapper_regs[0x02 + x])
#define mapper24_mirror		(mapper_regs[0x10])
#define mapper24_irqenable	(mapper_regs[0x11])
#define mapper24_irqrepeat	(mapper_regs[0x12])
#define mapper24_irqlatch	(mapper_regs[0x13])
#define mapper24_irqmode	(mapper_regs[0x14])
#define mapper24_irqcount	(mapper_regs16[0x1f - 0])
#define mapper24_irqcycle	(mapper_regs16[0x1f - 1])
#define mapper26			(mapper_regs[0x15])

struct vrc6_channel {
	INT32 dutyacc;
	INT32 phaseacc;
	INT32 volacc;
	INT16 sample;
	UINT8 regs[4];
};

struct vrc6_master {
	UINT8 reg;
	INT32 disable;
	INT32 octave_shift;
};

static vrc6_channel vrc6_ch[3];
static vrc6_master vrc6_cntrl;

static void vrc6_reset()
{
	memset(&vrc6_ch, 0, sizeof(vrc6_ch));
	memset(&vrc6_cntrl, 0, sizeof(vrc6_cntrl));
}

static void vrc6_scan()
{
	SCAN_VAR(vrc6_ch);
	SCAN_VAR(vrc6_cntrl);
}

static INT16 vrc6_pulse(INT32 num)
{
	struct vrc6_channel *ch = &vrc6_ch[num];

	if (ch->regs[2] & 0x80 && vrc6_cntrl.disable == 0) { // channel enabled?
		ch->sample = 0;
		if (ch->regs[0] & 0x80) { // ignore duty
			ch->sample = (ch->regs[0] & 0xf) << 8;
		} else {
			if (ch->dutyacc > ((ch->regs[0] >> 4) & 7))
				ch->sample = (ch->regs[0] & 0xf) << 8;
			ch->phaseacc--;
			if (ch->phaseacc <= 0) {
				ch->phaseacc = ((((ch->regs[2] & 0xf) << 8) | ch->regs[1]) + 1) >> vrc6_cntrl.octave_shift;
				ch->dutyacc = (ch->dutyacc + 1) & 0xf;
			}
		}
	}
	return ch->sample;
}

static INT16 vrc6_saw()
{
	struct vrc6_channel *ch = &vrc6_ch[2];

	if (ch->regs[2] & 0x80 && vrc6_cntrl.disable == 0) { // channel enabled?
		ch->sample = ((ch->volacc >> 3) & 0x1f) << 8;
		ch->phaseacc--;
		if (ch->phaseacc <= 0) {
			ch->phaseacc = ( ((((ch->regs[2] & 0xf) << 8) | ch->regs[1]) + 1)
							 >> vrc6_cntrl.octave_shift ) * 2; // each phase takes 2 cycles (* 2)
			ch->volacc += ch->regs[0] & 0x3f;
			ch->dutyacc++;
			if (ch->dutyacc == 7) {
				ch->dutyacc = 0;
				ch->volacc = 0;
			}
		}
	}
	return ch->sample;
}

static INT16 vrc6_mixer()
{
	INT32 sum = vrc6_saw();
    sum += vrc6_pulse(0);
    sum += vrc6_pulse(1);
	return (INT16)(sum * 0.75);
}

static void vrc6_sound_write(UINT16 address, UINT8 data)
{
	switch (address) {
		case 0x9003:                           // master control reg
			vrc6_cntrl.reg = data;
			vrc6_cntrl.disable = data & 1;
			vrc6_cntrl.octave_shift = (data & 4) ? 8 : 0;
			if (vrc6_cntrl.octave_shift == 0)
				vrc6_cntrl.octave_shift = (data & 2) ? 4 : 0;
			break;
		case 0x9000: case 0x9001: case 0x9002: // pulse
		case 0xa000: case 0xa001: case 0xa002: // pulse 2
		case 0xb000: case 0xb001: case 0xb002: // saw
			vrc6_ch[((address >> 12) - 9) & 3].regs[address & 3] = data;
			break;
	}
}

static void vrc6_map()
{
	mapper_map_prg(16, 0, mapper24_prg(0));
	mapper_map_prg( 8, 2, mapper24_prg(1));
	mapper_map_prg( 8, 3, -1);

	for (INT32 i = 0; i < 8; i++)
		mapper_map_chr( 1, i, mapper24_chr(i));

	switch (mapper24_mirror) {
		case 0: set_mirroring(VERTICAL); break;
		case 1: set_mirroring(HORIZONTAL); break;
		case 2: set_mirroring(SINGLE_LOW); break;
		case 3: set_mirroring(SINGLE_HIGH); break;
	}
}

static void vrc6_write(UINT16 address, UINT8 data)
{
	if (mapper26) {
		// VRC6b has bits 0 & 1 swapped
		address = (address & 0xfffc) | ((address >> 1) & 1) | ((address << 1) & 2);
	}

	if (address >= 0x9000 && address <= 0xb002) {
		vrc6_sound_write(address & 0xf003, data);
		return;
	}

	switch (address & 0xf003) {
		case 0x8000: mapper24_prg(0) = data; break;
		case 0xb003: mapper24_mirror = (data & 0xc) >> 2; break;
		case 0xc000: mapper24_prg(1) = data; break;
		case 0xd000: mapper24_chr(0) = data; break;
		case 0xd001: mapper24_chr(1) = data; break;
		case 0xd002: mapper24_chr(2) = data; break;
		case 0xd003: mapper24_chr(3) = data; break;
		case 0xe000: mapper24_chr(4) = data; break;
		case 0xe001: mapper24_chr(5) = data; break;
		case 0xe002: mapper24_chr(6) = data; break;
		case 0xe003: mapper24_chr(7) = data; break;
		case 0xf000:
			mapper24_irqlatch = data;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		case 0xf001:
			mapper24_irqrepeat = data & 1;
			mapper24_irqenable = data & 2;
			mapper24_irqmode = data & 4;
			if (mapper24_irqenable) {
				mapper24_irqcycle = 0;
				mapper24_irqcount = mapper24_irqlatch;
			}
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		case 0xf002:
			mapper24_irqenable = mapper24_irqrepeat;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
	}

	mapper_map();
}

static void vrc6_cycle()
{
	if (mapper24_irqenable) {
		if (mapper24_irqmode) { // cycle mode
			mapper24_irqcount++;
			if (mapper24_irqcount >= 0x100) {
				mapper_irq(0);
				mapper24_irqcount = mapper24_irqlatch;
			}
		} else {
			mapper24_irqcycle += 3; // scanline mode
			if (mapper24_irqcycle >= 341) {
				mapper24_irqcycle -= 341;
				mapper24_irqcount++;
				if (mapper24_irqcount == 0x100) {
					mapper_irq(0);
					mapper24_irqcount = mapper24_irqlatch;
				}
			}
		}
	}
}
#undef mapper24_prg
#undef mapper24_chr
#undef mapper24_mirror
#undef mapper24_irqenable
#undef mapper24_irqrepeat
#undef mapper24_irqlatch
#undef mapper24_irqmode
#undef mapper24_irqcount
#undef mapper24_irqcycle

// ---[ mapper 227 xxx-in-1, Waixing Bio Hazard
#define mapper227_mirror	(mapper_regs[0])
#define mapper227_S         (mapper_regs[1])
#define mapper227_L         (mapper_regs[2])
#define mapper227_P         (mapper_regs[3])
#define mapper227_O         (mapper_regs[4])

static void mapper227_write(UINT16 address, UINT8 data)
{
	mapper227_P = ((address >> 2) & 0x1f) | ((address & 0x100) >> 3);
	mapper227_S = address & 0x01;
	mapper227_L = address & (1 << 9);
	mapper227_O = address & (1 << 7);

	mapper227_mirror = address & 0x02;

	mapper_map();
}

static void mapper227_map()
{
	if (mapper227_O) {
		if (mapper227_S) {
			mapper_map_prg(32, 0, mapper227_P >> 1);
		} else {
			mapper_map_prg(16, 0, mapper227_P);
			mapper_map_prg(16, 1, mapper227_P);
		}
	} else {
		if (mapper227_S) {
			if (mapper227_L) {
				mapper_map_prg(16, 0, mapper227_P & 0x3e);
				mapper_map_prg(16, 1, mapper227_P | 0x07);
			} else {
				mapper_map_prg(16, 0, mapper227_P & 0x3e);
				mapper_map_prg(16, 1, mapper227_P & 0x38);
			}
		} else {
			if (mapper227_L) {
				mapper_map_prg(16, 0, mapper227_P);
				mapper_map_prg(16, 1, mapper227_P | 0x07);
			} else {
				mapper_map_prg(16, 0, mapper227_P);
				mapper_map_prg(16, 1, mapper227_P & 0x38);
			}
		}
	}

	mapper_map_chr( 8, 0, 0);

	set_mirroring((mapper227_mirror & 0x02) ? HORIZONTAL : VERTICAL);
}

// ---[ mapper 172: 1991 Du Ma Racing
#define jv001_register			(mapper_regs[0x1f - 0])
#define jv001_invert			(mapper_regs[0x1f - 1])
#define jv001_mode				(mapper_regs[0x1f - 2])
#define jv001_input				(mapper_regs[0x1f - 3])
#define jv001_output			(mapper_regs[0x1f - 4])

#define jv001_d0d3_mask		(0x0f)
#define jv001_d4d5_mask		(0x30)

static UINT8 jv001_read()
{
	UINT8 ret;
	ret =  (jv001_register & jv001_d0d3_mask);
	// if (invert), bits d4 and d5 are inverted
	ret |= (jv001_register & jv001_d4d5_mask) ^ (jv001_invert * jv001_d4d5_mask);

	bprintf(0, _T("jv001_read:  %x\n"), ret);

	return ret;
}

static void jv001_write(UINT16 address, UINT8 data)
{
	if (address & 0x8000) {
		bprintf(0, _T("jv001_latch address (%x) - output: %x\n"), address, jv001_register);
		jv001_output = jv001_register;
	} else {
		switch (address & 0xe103) {
			case 0x4100:
				if (jv001_mode) {
					// increment d0-d3, leaving d4-d5 unchanged
					UINT8 before = jv001_register;
					jv001_register = ((jv001_register + 1) & jv001_d0d3_mask) | (jv001_register & jv001_d4d5_mask);
					bprintf(0, _T("jv001_inc: mode %x  before  %x  after  %x\n"), jv001_mode, before, jv001_register);
				} else {
					// load register.  if inverted invert d0-d3, leaving d4-d5 unchanged
					UINT8 before = jv001_register;
					jv001_register = (jv001_invert) ? ((~jv001_input & jv001_d0d3_mask) | (jv001_input & jv001_d4d5_mask)) : jv001_input;
					bprintf(0, _T("jv001_load(inc): mode %x  before  %x  after  %x   input  %x\n"), jv001_mode, before, jv001_register, jv001_input);
				}
				break;
			case 0x4101:
				bprintf(0, _T("invert  %x\n"), data);
				jv001_invert = (data & 0x10) >> 4;
				break;
			case 0x4102:
				bprintf(0, _T("input  %x\n"), data);
				jv001_input = data;
				break;
			case 0x4103:
				bprintf(0, _T("mode  %x\n"), data);
				jv001_mode = (data & 0x10) >> 4;
				break;
		}
	}
}

// mapper 172: jv001 chip is mounted upside-down thus flipping d0 - d5
static UINT8 mapper172_jv001_swap(UINT8 data)
{
	return ((data & 0x01) << 5) | ((data & 0x02) << 3) | ((data & 0x04) << 1) | ((data & 0x08) >> 1) | ((data & 0x10) >> 3) | ((data & 0x20) >> 5);
}

static UINT8 mapper172_read(UINT16 address)
{
	if ((address & 0xe100) == 0x4100) {
		return mapper172_jv001_swap(jv001_read()) | (cpu_open_bus & 0xc0);
	}

	return cpu_open_bus;
}

static void mapper172_write(UINT16 address, UINT8 data)
{
	jv001_write(address, mapper172_jv001_swap(data));

	if (address & 0x8000) mapper_map();
}

static void mapper172_map()
{
	mapper_map_prg(32, 0, 0);
	mapper_map_chr( 8, 0, jv001_output & 0x03);

	set_mirroring((jv001_invert) ? VERTICAL : HORIZONTAL);
}


// --[ mapper 228: Action52
#define mapper228_mirror	(mapper_regs[0x1f - 0])
#define mapper228_prg		(mapper_regs[0x1f - 1])
#define mapper228_prgtype	(mapper_regs[0x1f - 2])
#define mapper228_chr		(mapper_regs[0x1f - 3])
#define mapper228_weird(x)	(mapper_regs[0 + (x)])

static void mapper228_write(UINT16 address, UINT8 data)
{
	if (address & 0x8000) {
		UINT8 csel = (address >> 11) & 0x03;
		if (csel == 3) csel = 2;

		mapper228_prg = ((address >> 6) & 0x1f) | (csel << 5);
		mapper228_prgtype = address & 0x20;
		mapper228_chr = ((address & 0x0f) << 2) | (data & 0x03);
		mapper228_mirror = (~address & 0x2000) >> 13;

		mapper_map();
	}
}

static void mapper228_psg_write(UINT16 address, UINT8 data)
{
	if (address >= 0x5ff0) {
		mapper228_weird(address & 3) = data;
	}
}

static UINT8 mapper228_psg_read(UINT16 address)
{
	if (address >= 0x5ff0) {
		return mapper228_weird(address & 3);
	}

	return cpu_open_bus;
}

static void mapper228_map()
{
	if (mapper228_prgtype) {
		mapper_map_prg(16, 0, mapper228_prg);
		mapper_map_prg(16, 1, mapper228_prg);
	} else {
		mapper_map_prg(16, 0, (mapper228_prg & ~1));
		mapper_map_prg(16, 1, mapper228_prg | 1);
	}

	mapper_map_chr( 8, 0, mapper228_chr);
	set_mirroring((mapper228_mirror) ? VERTICAL : HORIZONTAL);
}
#undef mapper228_mirror
#undef mapper228_prg
#undef mapper228_prgtype
#undef mapper228_chr

// --[ mapper 90: jy company
#define mapper90_209                (mapper_regs[0x1f - 0x0])
#define mapper90_211                (mapper_regs[0x1f - 0x1])

// 5800&3: multiplier / accumulator
#define mapper90_mul0   			(mapper_regs[0x1f - 0x2])
#define mapper90_mul1   			(mapper_regs[0x1f - 0x3])
#define mapper90_accu   			(mapper_regs[0x1f - 0x4])
#define mapper90_testreg			(mapper_regs[0x1f - 0x5])

// c000 - cfff&7: IRQ
#define mapper90_irqenable			(mapper_regs[0x1f - 0x6])
#define mapper90_irqmode			(mapper_regs[0x1f - 0x7])
#define mapper90_irqcounter  		(mapper_regs[0x1f - 0x8])
#define mapper90_irqprescaler	  	(mapper_regs[0x1f - 0x9])
#define mapper90_irqxor  			(mapper_regs[0x1f - 0xa])
#define mapper90_irqprescalermask	(mapper_regs[0x1f - 0xb])
#define mapper90_irqunknown  		(mapper_regs[0x1f - 0xc])

// d000 - d7ff&3: mode
#define mapper90_mode				(mapper_regs[0x1f - 0xd])
#define mapper90_mirror				(mapper_regs[0x1f - 0xe])
#define mapper90_ppu				(mapper_regs[0x1f - 0xf])
#define mapper90_obank				(mapper_regs[0x1f - 0x10])

// 8000 - 87ff&3: PRG
#define mapper90_prg(x)				(mapper_regs[0x00 + (x)])

// 9000 - 97ff&7: CHRlo
#define mapper90_chrlo(x)			(mapper_regs[0x04 + (x)])

// a000 - a7ff&7: CHRhi (actually 8bit, ran out of 8bit regs)
#define mapper90_chrhi(x)			(mapper_regs16[0x00 + (x)])

// mmc4-like 4k chr banking latch
#define mapper90_chrlatch(x)        (mapper_regs16[0x08 + (x)])

// b000 - b7ff&3: nametable config (&4 = MSB)
#define mapper90_nt(x)              (mapper_regs16[0x0a + (x)])

// forward --
static void mapper90_ppu_clockmmc4(UINT16 address);

static void mapper90_write(UINT16 address, UINT8 data)
{
	if (address >= 0x8000 && address <= 0x87ff) {
		mapper90_prg(address & 3) = data & 0x3f;
	}

	if (address >= 0x9000 && address <= 0x97ff) {
		mapper90_chrlo(address & 7) = data;
	}

	if (address >= 0xa000 && address <= 0xa7ff) {
		mapper90_chrhi(address & 7) = data;
	}

	if (address >= 0xb000 && address <= 0xb7ff) {
		if (~address & 4) { // LSB
			mapper90_nt(address & 3) = (mapper90_nt(address & 3) & 0xff00) | (data << 0);
		} else { // MSB
			mapper90_nt(address & 3) = (mapper90_nt(address & 3) & 0x00ff) | (data << 8);
		}
	}

	if (address >= 0xc000 && address <= 0xcfff) {
		switch (address & 0x7) {
			case 0:
				mapper90_irqenable = data & 1;
				if (!mapper90_irqenable) {
					M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
				}
				break;
			case 1:
				mapper90_irqmode = data;
				mapper90_irqprescalermask = (data & 4) ? 0x7 : 0xff;
				break;
			case 2:
				mapper90_irqenable = 0;
				M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
				break;
			case 3:
				mapper90_irqenable = 1;
				break;
			case 4:
				mapper90_irqprescaler = data ^ mapper90_irqxor;
				break;
			case 5:
				mapper90_irqcounter = data ^ mapper90_irqxor;
				break;
			case 6:
				mapper90_irqxor = data;
				break;
			case 7:
				mapper90_irqunknown = data;
				break;
		}
		return;
	}

	if (address >= 0xd000 && address <= 0xd7ff) {
		switch (address & 0x0003) {
			case 0:
				mapper90_mode = data | ((mapper90_211) ? 0x20 : 0x00);
				break;
			case 1: mapper90_mirror = data; break;
			case 2: mapper90_ppu = data; break;
			case 3:
				mapper90_obank = data;

				if (mapper90_209 && mapper90_obank & 0x80) {
					mapper_ppu_clock = mapper90_ppu_clockmmc4; // chr latching. enabled dynamically
				} else {
					mapper_ppu_clock = NULL;
				}
				break;
		}
	}

	mapper_map();
}

static void mapper90_psg_write(UINT16 address, UINT8 data)
{
	switch (address & 0xfc03) {
		case 0x5800: mapper90_mul0 = data; break;
		case 0x5801: mapper90_mul1 = data; break;
		case 0x5802: mapper90_accu += data; break;
		case 0x5803: mapper90_testreg = data; mapper90_accu = 0; break;
	}
}

static UINT8 mapper90_psg_read(UINT16 address)
{
	switch (address & 0xfc03) {
		case 0x5800: return ((mapper90_mul0 * mapper90_mul1) >> 0) & 0xff;
		case 0x5801: return ((mapper90_mul0 * mapper90_mul1) >> 8) & 0xff;
		case 0x5802: return mapper90_accu;
		case 0x5803: return mapper90_testreg;
	}

	switch (address & 0xffff) {
		case 0x5000: // jumper/DIP register
		case 0x5400:
		case 0x5c00:
			return 0x00;
	}

	return cpu_open_bus;
}

static void mapper90_clockirq()
{
	switch (mapper90_irqmode & 0xc0) {
		case 0x40:
			mapper90_irqcounter++;
			if ((mapper90_irqcounter == 0) && mapper90_irqenable) {
				//bprintf(0, _T("irq+ (mode %x) @ SL %d\n"), mapper90_irqmode, scanline);
				mapper_irq(2);
			}
			break;
		case 0x80:
			mapper90_irqcounter--;
			if ((mapper90_irqcounter == 0xff) && mapper90_irqenable) {
				//bprintf(0, _T("irq- (mode %x) @ SL %d\n"), mapper90_irqmode, scanline);
				mapper_irq(2); // 2 - "super mario world (unl)" HUD shaking
			}
			break;
	}
}

static void mapper90_clockpre()
{
	switch (mapper90_irqmode & 0xc0) {
		case 0x40:
			mapper90_irqprescaler++;
			if ((mapper90_irqprescaler & mapper90_irqprescalermask) == 0) {
				mapper90_clockirq();
			}
			break;
		case 0x80:
			mapper90_irqprescaler--;
			if ((mapper90_irqprescaler & mapper90_irqprescalermask) == mapper90_irqprescalermask) {
				mapper90_clockirq();
			}
			break;
	}
}

static void mapper90_ppu_clock(UINT16 address)
{
	// ppu clock mode
	if ((mapper90_irqmode & 3) == 2) {
		mapper90_clockpre();
		mapper90_clockpre();
	}
}

static void mapper90_ppu_clockmmc4(UINT16 address)
{
	switch (address & 0x3ff8) {
		case 0x0fd8:
			mapper90_chrlatch(0) = 0;
			mapper_map();
			break;
		case 0x0fe8:
			mapper90_chrlatch(0) = 2;
			mapper_map();
			break;
		case 0x1fd8:
			mapper90_chrlatch(1) = 4;
			mapper_map();
			break;
		case 0x1fe8:
			mapper90_chrlatch(1) = 6;
			mapper_map();
			break;
	}
}

static void mapper90_scanline()
{
	if ((mapper90_irqmode & 3) == 1 && (mmc5_mask[0] & 0x18) /* rendering? */) {
		for (INT32 i = 0; i < 8; i++)
			mapper90_clockpre();
	}
}

static void mapper90_cycle()
{
	if ((mapper90_irqmode & 3) == 0)
		mapper90_clockpre();
}

static UINT8 mapper90_exp_read(UINT16 address)
{
	return (mapper90_mode & 0x80) ? Cart.PRGRom[PRGExpMap + (address & 0x1fff)] : Cart.WorkRAM[address & 0x1fff];
}

static void mapper90_exp_write(UINT16 address, UINT8 data) // 6000 - 7fff
{
	if (mapper90_mode & 0x80) {
		// prg-rom mode, abort write to wram
		cart_exp_write_abort = 1; // don't fall-through after callback!
	}
}

static UINT8 mapper90_ntread(UINT16 address) // this only gets mapped for 209, 211!
{
	if (mapper90_mode & 0x20) {
		INT32 nt = (address & 0xfff) / 0x400;
		if (mapper90_mode & 0x40 || ((mapper90_ppu & 0x80) ^ (mapper90_nt(nt) & 0x80))) {
			return Cart.CHRRom[(mapper90_nt(nt)) * 0x400 + (address & 0x3ff)];
		}
	}

	return read_nt_int(address); // fall back to internal
}

static UINT16 mapper90_getchr(INT32 num)
{
	UINT16 bank = 0;
	UINT16 mask = 0xffff;

	if (~mapper90_obank & 0x20) { // outer-bank mode
		bank = (mapper90_obank & 1) | ((mapper90_obank >> 2) & 6);

		switch (mapper90_mode & 0x18) {
			case 0x00: bank <<= 5; mask = 0x1f; break;
			case 0x08: bank <<= 6; mask = 0x3f; break;
			case 0x10: bank <<= 7; mask = 0x7f; break;
			case 0x18: bank <<= 8; mask = 0xff; break;
		}
	}

	return ((mapper90_chrlo(num) | (mapper90_chrhi(num) << 8)) & mask) | bank;
}

static UINT8 mapper90_bitswap06(UINT8 data)
{
	return (data & 0x40) >> 6 | (data & 0x20) >> 4 | (data & 0x10) >> 2 | (data & 0x08) | (data & 0x04) << 2 | (data & 0x02) << 4 | (data & 0x01) << 6;
}

static void mapper90_map()
{
	// prg
	INT32 prg8_obank  = (mapper90_obank & 6) << 5;
	INT32 prg16_obank = (mapper90_obank & 6) << 4;
	INT32 prg32_obank = (mapper90_obank & 6) << 3;
	switch (mapper90_mode & 3) {
		case 0x00:
			mapper_map_prg(32, 0, prg32_obank | ((mapper90_mode & 0x04) ? (mapper90_prg(3) & 0xf) : 0xf));
			if (mapper90_mode & 0x80) {
				mapper_map_exp_prg(prg8_obank | (((mapper90_prg(3) << 2) + 3) & 0x3f));
			}
			break;
		case 0x01:
			//bprintf(0, _T("Mapper: JyCompany - 16k prg mode. *untested*\n"));
			mapper_map_prg(16, 0, prg16_obank | (mapper90_prg(1) & 0x1f));
			mapper_map_prg(16, 1, prg16_obank | ((mapper90_mode & 0x04) ? (mapper90_prg(3) & 0x1f) : 0x1f));
			if (mapper90_mode & 0x80) {
				mapper_map_exp_prg(prg8_obank | (((mapper90_prg(3) << 1) + 1) & 0x3f));
			}
			break;
		case 0x02:
			mapper_map_prg(8, 0, prg8_obank | (mapper90_prg(0) & 0x3f));
			mapper_map_prg(8, 1, prg8_obank | (mapper90_prg(1) & 0x3f));
			mapper_map_prg(8, 2, prg8_obank | (mapper90_prg(2) & 0x3f));
			mapper_map_prg(8, 3, prg8_obank | ((mapper90_mode & 0x04) ? (mapper90_prg(3) & 0x3f) : 0x3f));
			if (mapper90_mode & 0x80) {
				mapper_map_exp_prg(prg8_obank | (mapper90_prg(3) & 0x3f));
			}
			break;
		case 0x03: // same as 0x02, but with inverted bits
			//bprintf(0, _T("Mapper: JyCompany - inverted bits. *untested*\n"));
			mapper_map_prg(8, 0, prg8_obank | (mapper90_bitswap06(mapper90_prg(0)) & 0x3f));
			mapper_map_prg(8, 1, prg8_obank | (mapper90_bitswap06(mapper90_prg(1)) & 0x3f));
			mapper_map_prg(8, 2, prg8_obank | (mapper90_bitswap06(mapper90_prg(2)) & 0x3f));
			mapper_map_prg(8, 3, prg8_obank | ((mapper90_mode & 0x04) ? (mapper90_bitswap06(mapper90_prg(3)) & 0x3f) : 0x3f));
			if (mapper90_mode & 0x80) {
				mapper_map_exp_prg(prg8_obank | (mapper90_bitswap06(mapper90_prg(3)) & 0x3f));
			}
			break;
	}

	// chr
	switch (mapper90_mode & 0x18) {
		case 0x00:
			mapper_map_chr( 8, 0, mapper90_getchr(0));
			break;
		case 0x08:
			if (~mapper90_obank & 0x80) { // normal 4k banking
				mapper_map_chr( 4, 0, mapper90_getchr(0));
				mapper_map_chr( 4, 1, mapper90_getchr(4));
			} else {                      // mmc4-like latch 4k banking
				mapper_map_chr( 4, 0, mapper90_getchr(mapper90_chrlatch(0)));
				mapper_map_chr( 4, 1, mapper90_getchr(mapper90_chrlatch(1)));
			}
			break;
		case 0x10:
			mapper_map_chr( 2, 0, mapper90_getchr(0));
			mapper_map_chr( 2, 1, mapper90_getchr(2));
			mapper_map_chr( 2, 2, mapper90_getchr(4));
			mapper_map_chr( 2, 3, mapper90_getchr(6));
			break;
		case 0x18:
			mapper_map_chr( 1, 0, mapper90_getchr(0));
			mapper_map_chr( 1, 1, mapper90_getchr(1));
			mapper_map_chr( 1, 2, mapper90_getchr(2));
			mapper_map_chr( 1, 3, mapper90_getchr(3));
			mapper_map_chr( 1, 4, mapper90_getchr(4));
			mapper_map_chr( 1, 5, mapper90_getchr(5));
			mapper_map_chr( 1, 6, mapper90_getchr(6));
			mapper_map_chr( 1, 7, mapper90_getchr(7));
			break;
	}

	// nametable config - if rom nt's are selected, they will be fed via mapper90_ntread()
	// a RAM nt is always selected for writing, though. (re: Tiny Toon Adv. 6 intro)
	if (mapper90_209 && mapper90_mode & 0x20) {
		nametable_map(0, mapper90_nt(0) & 1);
		nametable_map(1, mapper90_nt(1) & 1);
		nametable_map(2, mapper90_nt(2) & 1);
		nametable_map(3, mapper90_nt(3) & 1);
	} else {
		//standard nt config
		switch (mapper90_mirror & 0x3) {
			case 0: set_mirroring(VERTICAL); break;
			case 1: set_mirroring(HORIZONTAL); break;
			case 2: set_mirroring(SINGLE_LOW); break;
			case 3: set_mirroring(SINGLE_HIGH); break;
		}
	}
}
#undef mapper90_irqenable
#undef mapper90_irqmode
#undef mapper90_irqcounter
#undef mapper90_irqprescaler
#undef mapper90_irqxor
#undef mapper90_irqprescalermask
#undef mapper90_irqunknown
#undef mapper90_mode
#undef mapper90_mirror
#undef mapper90_ppu
#undef mapper90_obank
#undef mapper90_prg
#undef mapper90_chrlo
#undef mapper90_chrhi
#undef mapper90_nt

// --[ mapper 91: older JyCompany/Hummer
#define mapper91_prg(x)		(mapper_regs[0x00 + (x)])
#define mapper91_chr(x)		(mapper_regs[0x04 + (x)])
#define mapper91_irqcount   (mapper_regs[0x1f - 0x00])
#define mapper91_irqenable	(mapper_regs[0x1f - 0x01])

static void mapper91_write(UINT16 address, UINT8 data)
{
	switch (address & 0xf000) {
		case 0x6000:
			mapper91_chr(address & 3) = data;
			break;
		case 0x7000:
			switch (address & 3) {
				case 0:
				case 1:
					mapper91_prg(address & 1) = data;
					break;
				case 2:
					mapper91_irqenable = 0;
					mapper91_irqcount = 0;
					M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
					break;
				case 3:
					mapper91_irqenable = 1;
					break;
			}
			break;
	}

	mapper_map();
}

static void mapper91_scanline()
{
	if (mapper91_irqenable && (mmc5_mask[0] & 0x18) /* rendering? */) {
		mapper91_irqcount++;
		if (mapper91_irqcount == 8) {
			mapper_irq(0x14); // 0x14 - gets rid of artefacts in "dragon ball z - super butouden 2"
			mapper91_irqenable = 0;
		}
	}
}

static void mapper91_map()
{
	mapper_map_prg( 8, 0, mapper91_prg(0));
	mapper_map_prg( 8, 1, mapper91_prg(1));
	mapper_map_prg( 8, 2, -2);
	mapper_map_prg( 8, 3, -1);

	mapper_map_chr( 2, 0, mapper91_chr(0));
	mapper_map_chr( 2, 1, mapper91_chr(1));
	mapper_map_chr( 2, 2, mapper91_chr(2));
	mapper_map_chr( 2, 3, mapper91_chr(3));
}
#undef mapper91_prg
#undef mapper91_chr
#undef mapper91_irqcount
#undef mapper91_irqenable

// --[ mapper 17: FFE / Front Far East SMC (type 17)
#define mapper17_prg(x)		(mapper_regs[0x00 + (x)])
#define mapper17_chr(x)		(mapper_regs[0x04 + (x)])
#define mapper17_irqcount   (mapper_regs16[0x00])
#define mapper17_irqenable	(mapper_regs[0x1f - 0x00])
#define mapper17_mirror		(mapper_regs[0x1f - 0x01])

static void mapper17_write(UINT16 address, UINT8 data)
{
	switch (address) {
		case 0x42fe:
			switch (data & 0x10) {
				case 0x00: mapper17_mirror = 2; break;
				case 0x10: mapper17_mirror = 3; break;
			}
			break;
		case 0x42ff:
			switch (data & 0x10) {
				case 0x00: mapper17_mirror = 0; break;
				case 0x10: mapper17_mirror = 1; break;
			}
			break;

		case 0x4501:
			mapper17_irqenable = 0;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		case 0x4502:
			mapper17_irqcount = (mapper17_irqcount & 0xff00) | data;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		case 0x4503:
			mapper17_irqcount = (mapper17_irqcount & 0x00ff) | (data << 8);
			mapper17_irqenable = 1;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;

		case 0x4504:
		case 0x4505:
		case 0x4506:
		case 0x4507:
			mapper17_prg(address & 3) = data;
			break;

		case 0x4510:
		case 0x4511:
		case 0x4512:
		case 0x4513:
		case 0x4514:
		case 0x4515:
		case 0x4516:
		case 0x4517:
			mapper17_chr(address & 7) = data;
			break;
	}

	mapper_map();
}

static void mapper17_cycle()
{
	if (mapper17_irqenable) {
		mapper17_irqcount++;
		if (mapper17_irqcount == 0x0000) {
			mapper_irq(0);
			mapper17_irqenable = 0;
		}
	}
}

static void mapper17_map()
{
	mapper_map_prg( 8, 0, mapper17_prg(0));
	mapper_map_prg( 8, 1, mapper17_prg(1));
	mapper_map_prg( 8, 2, mapper17_prg(2));
	mapper_map_prg( 8, 3, mapper17_prg(3));

	mapper_map_chr( 1, 0, mapper17_chr(0));
	mapper_map_chr( 1, 1, mapper17_chr(1));
	mapper_map_chr( 1, 2, mapper17_chr(2));
	mapper_map_chr( 1, 3, mapper17_chr(3));
	mapper_map_chr( 1, 4, mapper17_chr(4));
	mapper_map_chr( 1, 5, mapper17_chr(5));
	mapper_map_chr( 1, 6, mapper17_chr(6));
	mapper_map_chr( 1, 7, mapper17_chr(7));

	switch (mapper17_mirror & 0x3) {
		case 0: set_mirroring(VERTICAL); break;
		case 1: set_mirroring(HORIZONTAL); break;
		case 2: set_mirroring(SINGLE_LOW); break;
		case 3: set_mirroring(SINGLE_HIGH); break;
	}
}
#undef mapper17_prg
#undef mapper17_chr
#undef mapper17_irqcount
#undef mapper17_irqenable
#undef mapper17_mirror

// ---[ mapper 400 8-Bit X-Mas 2017
#define mapper400_reg(x)		(mapper_regs[(x)]) // 0 - 3

static void mapper400_write(UINT16 address, UINT8 data)
{
	if (address >= 0x7800 && address <= 0x7fff) {
		mapper400_reg(0) = data;
	}
	if (address >= 0x8000 && address <= 0xbfff) {
		mapper400_reg(1) = data;
	}
	if (address >= 0xc000 && address <= 0xffff) {
		mapper400_reg(2) = data;
	}

	mapper_map();
}

static void mapper400_map()
{
	mapper_map_prg(16, 0, (mapper400_reg(0) & 0x78) | (mapper400_reg(2) & 0x07));
	mapper_map_prg(16, 1, (mapper400_reg(0) & 0x78) | 0x07);

	mapper_map_chr( 8, 0, (mapper400_reg(2) >> 5) & 0x03);

	if (mapper400_reg(0) != 0x80) {
		set_mirroring((mapper400_reg(0) & 0x20) ? HORIZONTAL : VERTICAL);
	}
}

#undef mapper400_reg

// ---[ mapper 413 Super Russian Roulette
#define mapper413_reg(x)		(mapper_regs[(x)]) // 0 - 3
#define mapper413_irq_latch		(mapper_regs[4])
#define mapper413_irq_count		(mapper_regs[5])
#define mapper413_irq_enable	(mapper_regs[6])
#define mapper413_ext_control   (mapper_regs[7])

static UINT32 *mapper413_ext_addr = (UINT32*)&mapper_regs16[0];

static UINT8 mapper413_read_pcm()
{
	UINT8 data = Cart.ExtData[mapper413_ext_addr[0] & (Cart.ExtDataSize - 1)];
	if (mapper413_ext_control & 0x02) mapper413_ext_addr[0]++;
	return data;
}

static UINT8 mapper413_prg_read(UINT16 address)
{
	if ((address & 0xf000) == 0xc000) {
		return mapper413_read_pcm();
	}

	return mapper_prg_read_int(address);
}

static void mapper413_prg_write(UINT16 address, UINT8 data)
{
	switch (address & 0xf000) {
		case 0x8000: mapper413_irq_latch = data; break;
		case 0x9000: mapper413_irq_count = 0; break;
		case 0xa000: mapper413_irq_enable = 0; M6502SetIRQLine(0, CPU_IRQSTATUS_NONE); break;
		case 0xb000: mapper413_irq_enable = 1; break;
		case 0xc000: mapper413_ext_addr[0] = (mapper413_ext_addr[0] << 1) | (data >> 7); break;
		case 0xd000: mapper413_ext_control = data; break;
		case 0xe000:
		case 0xf000: mapper413_reg((data >> 6)) = data & 0x3f; break;
	}

	mapper_map();
}

static UINT8 mapper413_psg_read(UINT16 address)
{
	if ((address & 0xf800) == 0x4800) {
		return mapper413_read_pcm();
	}
	if (address >= 0x5000) {
		return Cart.PRGRom[0x1000 | (address & 0x1fff)];
	}

	return cpu_open_bus;
}
static UINT8 mapper413_exp_read(UINT16 address)
{
	return Cart.PRGRom[PRGExpMap + (address & 0x1fff)];
}

static void mapper413_map()
{
	mapper_map_exp_prg(mapper413_reg(0)); // 6000 - 7fff
	mapper_map_prg( 8, 0, mapper413_reg(1));
	mapper_map_prg( 8, 1, mapper413_reg(2));
	mapper_map_prg( 8, 2, 3);
	mapper_map_prg( 8, 3, 4);

	mapper_map_chr( 4, 0, mapper413_reg(3));
	mapper_map_chr( 4, 1, 0x3d);
}

static void mapper413_scanline()
{
	if (mapper413_irq_count == 0) {
		mapper413_irq_count = mapper413_irq_latch;
	} else {
		mapper413_irq_count--;
	}

	if (mapper413_irq_enable && mapper413_irq_count == 0) {
		if (mmc5_mask[0] & 0x18) { // aka if (RENDERING)
			mapper_irq(0);
		}
	}
}

#undef mapper413_reg
#undef mapper413_irq_latch
#undef mapper413_irq_count
#undef mapper413_irq_enable
#undef mapper413_ext_control

// --[ mapper 28: Action53 Home-brew multicart
#define mapper28_mirror		(mapper_regs[0x1f - 0])
#define mapper28_mirrorbit  (mapper_regs[0x1f - 1])
#define mapper28_cmd		(mapper_regs[0x1f - 2])
#define mapper28_regs(x)	(mapper_regs[(x)])

static void mapper28_write(UINT16 address, UINT8 data)
{
	if (address >= 0x5000 && address <= 0x5fff) {
		mapper28_cmd = ((data >> 6) & 2) | (data & 1);
	} else if (address >= 0x8000) {
		switch (mapper28_cmd) {
			case 0:
			case 1:
				mapper28_mirrorbit = (data >> 4) & 1;
				break;
			case 2:
				mapper28_mirrorbit = data & 1;
				break;
		}
		mapper28_regs(mapper28_cmd & 3) = data;

		mapper_map();
	}
}

static void mapper28_map()
{
	UINT8 slot = (mapper28_regs(2) & 0x04) >> 2;
	UINT8 outerbanksize = (mapper28_regs(2) & 0x30) >> 4;
	UINT8 outerbank = mapper28_regs(3) << 1;
	UINT8 prgbank = (mapper28_regs(1) & 0xf) << ((~mapper28_regs(2) & 0x08) >> 3);
	UINT8 page = (outerbank & ~((1 << (outerbanksize+1))-1)) | (prgbank & ((1 << (outerbanksize+1))-1));

	if (mapper28_regs(2) & 0x08) {
		mapper_map_prg(16, slot ^ 1, page);
		mapper_map_prg(16, slot    , (outerbank & 0x1fe) | slot);
	} else {
		mapper_map_prg(16, 0, page);
		mapper_map_prg(16, 1, page | 1);
	}

	mapper_map_chr( 8, 0, mapper28_regs(0) & 0x03);

	UINT8 mirror = mapper28_regs(2) & 0x03;
	if (~mirror & 0x02) {
		mirror = mapper28_mirrorbit;
	}

	switch (mirror) {
		case 0: set_mirroring(SINGLE_LOW); break;
		case 1: set_mirroring(SINGLE_HIGH); break;
		case 2: set_mirroring(VERTICAL); break;
		case 3: set_mirroring(HORIZONTAL); break;
	}
}
#undef mapper28_mirror
#undef mapper28_mirrorbit
#undef mapper28_cmd
#undef mapper28_regs

// --[ mapper 33, 48: taito
#define mapper33_prg(x)		(mapper_regs[0 + x])
#define mapper33_chr(x)		(mapper_regs[2 + x])
#define mapper33_mirror		(mapper_regs[0x1f - 0])
#define mapper33_irqenable	(mapper_regs[0x1f - 1])
#define mapper48            (mapper_regs[0x1f - 2])
#define mapper33_irqcount	(mapper_regs16[0x1f - 0])
#define mapper33_irqlatch	(mapper_regs16[0x1f - 1])
#define mapper33_irqreload  (mapper_regs16[0x1f - 2])
#define mapper48_flintstones (mapper_regs16[0x1f - 3])

static void mapper33_write(UINT16 address, UINT8 data)
{
	switch (address & 0xf003) {
		case 0x8000:
			mapper33_prg(0) = data & 0x3f;
			if (mapper48 == 0) mapper33_mirror = data & 0x40;
			break;
		case 0x8001: mapper33_prg(1) = data & 0x3f; break;

		case 0x8002: mapper33_chr(0) = data; break;
		case 0x8003: mapper33_chr(1) = data; break;
		case 0xa000: mapper33_chr(2) = data; break;
		case 0xa001: mapper33_chr(3) = data; break;
		case 0xa002: mapper33_chr(4) = data; break;
		case 0xa003: mapper33_chr(5) = data; break;
	}

	if (mapper48) {
		switch (address & 0xf003) {
			case 0xc000: mapper33_irqlatch = (data ^ 0xff) + ((mapper48_flintstones) ? 0 : 1); M6502SetIRQLine(0, CPU_IRQSTATUS_NONE); break;
			case 0xc001: mapper33_irqreload = 1; mapper33_irqcount = 0; M6502SetIRQLine(0, CPU_IRQSTATUS_NONE); break;
			case 0xc002: mapper33_irqenable = 1; break;
			case 0xc003: mapper33_irqenable = 0; M6502SetIRQLine(0, CPU_IRQSTATUS_NONE); break;
			case 0xe000: if (mapper48) mapper33_mirror = data & 0x40; break;
		}
	}

	mapper_map();
}

static void mapper33_map()
{
	mapper_map_prg( 8, 0, mapper33_prg(0));
	mapper_map_prg( 8, 1, mapper33_prg(1));
	mapper_map_prg( 8, 2, -2);
	mapper_map_prg( 8, 3, -1);

	mapper_map_chr( 2, 0, mapper33_chr(0));
	mapper_map_chr( 2, 1, mapper33_chr(1));

	mapper_map_chr( 1, 4, mapper33_chr(2));
	mapper_map_chr( 1, 5, mapper33_chr(3));
	mapper_map_chr( 1, 6, mapper33_chr(4));
	mapper_map_chr( 1, 7, mapper33_chr(5));

	set_mirroring((mapper33_mirror) ? HORIZONTAL : VERTICAL);
}

static void mapper33_scanline()
{
	INT32 cnt = mapper33_irqenable;
	if (mapper33_irqcount == 0 || mapper33_irqreload) {
		mapper33_irqreload = 0;
		mapper33_irqcount = mapper33_irqlatch;
	} else {
		mapper33_irqcount--;
	}

	if (cnt && mapper33_irqenable && mapper33_irqcount == 0) {
		mapper_irq((mapper48_flintstones) ? 0x13 : 0x06);
	}
}
#undef mapper33_mirror
#undef mapper33_irqenable
#undef mapper33_prg
#undef mapper33_chr
#undef mapper33_irqcount
#undef mapper33_irqlatch
#undef mapper33_irqreload

// --[ mapper 36 - TXC (Policeman, Strike Wolf)
#define mapper36_prg		(mapper_regs[0x1f - 0])
#define mapper36_chr		(mapper_regs[0x1f - 1])
#define mapper36_RR			(mapper_regs[0x1f - 2])
#define mapper36_PP			(mapper_regs[0x1f - 3])
#define mapper36_invert		(mapper_regs[0x1f - 4])
#define mapper36_mode		(mapper_regs[0x1f - 5])

static UINT8 mapper36_read(UINT16 address)
{
	if ((address & 0xe100) == 0x4100)
		return (cpu_open_bus & 0xcf) | (mapper36_RR << 4);
	else
		return cpu_open_bus;
}

static void mapper36_write(UINT16 address, UINT8 data)
{
	if (address > 0x7fff) {
		mapper36_prg = mapper36_RR;
	} else {
		switch (address & 0xe103) {
			case 0x4100:
				if (mapper36_mode) {
					mapper36_RR++;
				} else {
					mapper36_RR = mapper36_PP;
				}
				break;
			case 0x4101:
				mapper36_invert = data & 0x10;
				break;
			case 0x4102:
				mapper36_PP = (data >> 4) & 0x3;
				break;
			case 0x4103:
				mapper36_mode = data & 0x10;
				break;
		}
		switch (address & 0xe200) {
			case 0x4200:
				mapper36_chr = data & 0xf;
				break;
		}
	}

	mapper_map();
}

static void mapper36_map()
{
	mapper_map_prg(32, 0, mapper36_prg);

	mapper_map_chr(8, 0, mapper36_chr);
}

#undef mapper36_prg
#undef mapper36_chr
#undef mapper36_RR
#undef mapper36_mode

// --[ mapper 42 Ai Senshi Nicol FDS conversion
#define mapper42_mirror		(mapper_regs[0x1f - 0])
#define mapper42_prg		(mapper_regs[0x1f - 1])
#define mapper42_chr		(mapper_regs[0x1f - 2])
#define mapper42_irqenable	(mapper_regs[0x1f - 3])
#define mapper42_irqcount   (mapper_regs16[0])

static void mapper42_write(UINT16 address, UINT8 data)
{
	switch (address & 0xe003) {
		case 0x8000: mapper42_chr = data; break;
		case 0xe000: mapper42_prg = data & 0xf; break;
		case 0xe001: mapper42_mirror = data; break;
		case 0xe002:
			mapper42_irqenable = data & 2;
			if (!mapper42_irqenable) {
				mapper42_irqcount = 0;
				M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			}
			break;
	}

	mapper_map();
}

static void mapper42_map()
{
	// set prg-rom @ 6000 - 7fff, fed by mapper42_exp_read()
	mapper_map_exp_prg(mapper42_prg);
	// normal prg-rom @ 8000 - ffff
	mapper_map_prg(32, 0, -1);

	mapper_map_chr(8, 0, mapper42_chr);

	set_mirroring((mapper42_mirror & 0x8) ? HORIZONTAL : VERTICAL);
}

static UINT8 mapper42_exp_read(UINT16 address)
{
	return Cart.PRGRom[PRGExpMap + (address & 0x1fff)];
}

static void mapper42_cycle()
{
	if (mapper42_irqenable) {
		mapper42_irqcount++;
		if (mapper42_irqcount >= 0x8000) {
			mapper42_irqcount -= 0x8000;
		}
		if (mapper42_irqcount >= 0x6000) {
			M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
		} else {
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
		}
	}
}

#undef mapper42_mirror
#undef mapper42_prg
#undef mapper42_chr
#undef mapper42_irqenable
#undef mapper42_irqcount

// --[ mapper 108 Meikyuu Jiin Dababa (Japan) FDS conversion
#define mapper108_prg		(mapper_regs[0x1f - 0])

static void mapper108_write(UINT16 address, UINT8 data)
{
	if ((address >= 0x8000 && address <= 0x8fff) || (address >= 0xf000 && address <= 0xffff)) {
		mapper108_prg = data;

		mapper_map();
	}
}

static void mapper108_map()
{
	// set prg-rom @ 6000 - 7fff, fed by mapper108_exp_read()
	mapper_map_exp_prg(mapper108_prg);
	// normal prg-rom @ 8000 - ffff
	mapper_map_prg(32, 0, -1);

	mapper_map_chr(8, 0, 0);
}

static UINT8 mapper108_exp_read(UINT16 address)
{
	return Cart.PRGRom[PRGExpMap + (address & 0x1fff)];
}

#undef mapper108_prg

// --[ mapper 111 Cheapocabra (GTROM)
#define mapper111_reg		(mapper_regs[0x1f - 0])

static void mapper111_write(UINT16 address, UINT8 data)
{
	if ((address >= 0x5000 && address <= 0x5fff) || (address >= 0x7000 && address <= 0x7fff)) {
		mapper111_reg = data;
		mapper_map();
	}
}

static void mapper111_map()
{
	mapper_map_prg(32, 0, mapper111_reg & 0xf);
	mapper_map_chr(8, 0, (mapper111_reg >> 4) & 0x1);

	INT32 nt = (mapper111_reg & 0x20) ? (0x4000 + 0x2000) : (0x4000 + 0x0000);

	for (INT32 i = 0; i < 4; i++) {
		nametable_mapraw(i & 3, Cart.CHRRam + nt + (i * 0x400), MEM_RAM);
	}
}

#undef mapper111_reg

// --[ mapper 120 Tobidase Daisakusen (Japan) FDS Conv. *game is buggy*
#define mapper120_prg		(mapper_regs[0x1f - 0])

static void mapper120_write(UINT16 address, UINT8 data)
{
	if (address == 0x41ff) {
		mapper120_prg = data & 0x7;
	}

	mapper_map();
}

static void mapper120_map()
{
	// set prg-rom @ 6000 - 7fff, fed by mapper120_exp_read()
	mapper_map_exp_prg(mapper120_prg);
	// normal prg-rom @ 8000 - ffff
	mapper_map_prg(32, 0, 2);

	mapper_map_chr(8, 0, 0);
}

static UINT8 mapper120_exp_read(UINT16 address)
{
	return Cart.PRGRom[PRGExpMap + (address & 0x1fff)];
}

#undef mapper120_prg

// --[ mapper 64 - Tengen (Atari)
#define mapper64_mirror			(mapper_regs[0x1f - 0])
#define mapper64_regnum			(mapper_regs[0x1f - 1])
#define mapper64_reload			(mapper_regs[0x1f - 2])
#define mapper64_irqmode		(mapper_regs[0x1f - 3])
#define mapper64_irqenable		(mapper_regs[0x1f - 4])
#define mapper64_irqlatch		(mapper_regs[0x1f - 5])
#define mapper64_irqcount		(mapper_regs[0x1f - 6])
#define mapper64_irqprescale	(mapper_regs[0x1f - 7])
#define mapper64_cycles         (mapper_regs16[0])

static void mapper64_write(UINT16 address, UINT8 data)
{
	if (address & 0x8000) {
        switch (address & 0xe001) {
			case 0xA000: mapper64_mirror = data & 1; mapper_map(); break;
            case 0x8000: mapper64_regnum = data; break;
			case 0x8001:
				mapper_regs[(mapper64_regnum & 0xf)] = data;
				mapper_map();
				break;

			case 0xC000:
				mapper64_irqlatch = data;
				break;
			case 0xC001:
				mapper64_reload = 1;
				mapper64_irqprescale = 0;
				mapper64_irqmode = data & 1;
				break;
			case 0xE000:
				mapper64_irqenable = 0;
				M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
				break;
			case 0xE001:
				mapper64_irqenable = 1;
				M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
				break;
        }
	}
}

static void mapper64_map()
{
	if (mapper64_regnum & 0x20) {
		mapper_map_chr(1, 0, mapper_regs[0]);
		mapper_map_chr(1, 1, mapper_regs[8]);
		mapper_map_chr(1, 2, mapper_regs[1]);
		mapper_map_chr(1, 3, mapper_regs[9]);
	} else {
		mapper_map_chr(2, 0, mapper_regs[0] >> 1);
		mapper_map_chr(2, 1, mapper_regs[1] >> 1);
	}
	mapper_map_chr(1, 4, mapper_regs[2]);
	mapper_map_chr(1, 5, mapper_regs[3]);
	mapper_map_chr(1, 6, mapper_regs[4]);
	mapper_map_chr(1, 7, mapper_regs[5]);

	mapper_map_prg(8, 0, mapper_regs[6]);
	mapper_map_prg(8, 1, mapper_regs[7]);
	mapper_map_prg(8, 2, mapper_regs[0xf]);
	mapper_map_prg(8, 3, -1);
	set_mirroring(mapper64_mirror ? HORIZONTAL : VERTICAL);
}

static void mapper64_irq_reload_logic()
{
	if (mapper64_reload) {
		mapper64_irqcount = (mapper64_irqlatch) ? mapper64_irqlatch | 1 : 0;
		if (mapper64_irqcount == 0 && mapper64_cycles > 0x10)
			mapper64_irqcount = 1;
		mapper64_reload = 0;
		mapper64_cycles = 0;
	} else if (mapper64_irqcount == 0) {
		mapper64_irqcount = mapper64_irqlatch;
		if (mapper64_cycles > 0x10)
			mapper64_cycles = 0;
	} else mapper64_irqcount --;
}

static void mapper64_scanline()
{
	if (mmc5_mask[0] & 0x18) { // aka if (RENDERING) ...
		if (mapper64_irqmode == 0) {

			mapper64_irq_reload_logic();

			if (mapper64_irqcount == 0 && mapper64_irqenable) {
				mapper_irq(1); // assert irq in 1 m2 cycle
			}
		}
	}
}

static void mapper64_cycle()
{
	if (mapper64_cycles == 0xffff) mapper64_cycles = 0x10;
	mapper64_cycles++;
	if (mapper64_irqmode == 1) {
		mapper64_irqprescale++;
		while (mapper64_irqprescale == 4) {
			mapper64_irqprescale = 0;

			mapper64_irq_reload_logic();

			if (mapper64_irqcount == 0 && mapper64_irqenable) {
				mapper_irq(4); // assert irq in 4 m2 cycles
			}
		}
	}
}

#undef mapper64_mirror
#undef mapper64_regnum
#undef mapper64_reload
#undef mapper64_irqmode
#undef mapper64_irqenable
#undef mapper64_irqlatch
#undef mapper64_irqcount
#undef mapper64_irqprescale

// --[ mapper 65 - irem h3001(?): Spartan X2, Kaiketsu Yanchamaru 3: Taiketsu! Zouringen,
#define mapper65_mirror		(mapper_regs[0x1f - 0])
#define mapper65_irqenable	(mapper_regs[0x1f - 1])
#define mapper65_prg(x)		(mapper_regs[0 + x])
#define mapper65_chr(x)		(mapper_regs[3 + x])
#define mapper65_irqcount	(mapper_regs16[0x1f - 0])
#define mapper65_irqlatch	(mapper_regs16[0x1f - 1])


static void mapper65_write(UINT16 address, UINT8 data)
{
	switch (address) {
		case 0x8000: mapper65_prg(0) = data; break;
		case 0xa000: mapper65_prg(1) = data; break;
		case 0xc000: mapper65_prg(2) = data; break;
		case 0x9001: mapper65_mirror = (~data >> 7) & 1; break;
		case 0x9003: mapper65_irqenable = data & 0x80; M6502SetIRQLine(0, CPU_IRQSTATUS_NONE); break;
		case 0x9004: mapper65_irqcount = mapper65_irqlatch; break;
		case 0x9005: mapper65_irqlatch = (mapper65_irqlatch & 0x00ff) | (data << 8); break;
		case 0x9006: mapper65_irqlatch = (mapper65_irqlatch & 0xff00) | data; break;
		case 0xb000: mapper65_chr(0) = data; break;
		case 0xb001: mapper65_chr(1) = data; break;
		case 0xb002: mapper65_chr(2) = data; break;
		case 0xb003: mapper65_chr(3) = data; break;
		case 0xb004: mapper65_chr(4) = data; break;
		case 0xb005: mapper65_chr(5) = data; break;
		case 0xb006: mapper65_chr(6) = data; break;
		case 0xb007: mapper65_chr(7) = data; break;
	}
	mapper_map();
}

static void mapper65_map()
{
	mapper_map_prg( 8, 0, mapper65_prg(0));
	mapper_map_prg( 8, 1, mapper65_prg(1));
	mapper_map_prg( 8, 2, mapper65_prg(2));
	mapper_map_prg( 8, 3, -1);

	mapper_map_chr( 1, 0, mapper65_chr(0));
	mapper_map_chr( 1, 1, mapper65_chr(1));
	mapper_map_chr( 1, 2, mapper65_chr(2));
	mapper_map_chr( 1, 3, mapper65_chr(3));
	mapper_map_chr( 1, 4, mapper65_chr(4));
	mapper_map_chr( 1, 5, mapper65_chr(5));
	mapper_map_chr( 1, 6, mapper65_chr(6));
	mapper_map_chr( 1, 7, mapper65_chr(7));

	set_mirroring(mapper65_mirror ? VERTICAL : HORIZONTAL);
}

static void mapper65_cycle()
{
	if (mapper65_irqenable) {
		mapper65_irqcount--;

		if (mapper65_irqcount == 0) {
			M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
			mapper65_irqenable = 0;
		}
	}
}
#undef mapper65_mirror
#undef mapper65_irqenable
#undef mapper65_prg
#undef mapper65_chr
#undef mapper65_irqcount
#undef mapper65_irqlatch


// --[ mapper 67 - Sunsoft-3 (Fantasy Zone II, Mito Koumon II: Sekai Manyuu Ki)
#define mapper67_mirror		(mapper_regs[0x1f - 0])
#define mapper67_irqenable	(mapper_regs[0x1f - 1])
#define mapper67_irqtoggle  (mapper_regs[0x1f - 2])
#define mapper67_prg		(mapper_regs[0])
#define mapper67_chr(x)		(mapper_regs[1 + x])
#define mapper67_irqcount	(mapper_regs16[0x1f - 0])

static void mapper67_write(UINT16 address, UINT8 data)
{
	switch (address & 0xf800) {
		case 0x8800: mapper67_chr(0) = data; break;
		case 0x9800: mapper67_chr(1) = data; break;
		case 0xa800: mapper67_chr(2) = data; break;
		case 0xb800: mapper67_chr(3) = data; break;
		case 0xc000:
		case 0xc800:
			if (mapper67_irqtoggle == 0) {
				mapper67_irqcount = (mapper67_irqcount & 0x00ff) | (data << 8);
			} else {
				mapper67_irqcount = (mapper67_irqcount & 0xff00) | (data << 0);
			}
			mapper67_irqtoggle ^= 1;
			break;
		case 0xd800:
			mapper67_irqtoggle = 0;
			mapper67_irqenable = data & 0x10;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		case 0xe800:
			mapper67_mirror = data & 3;
			break;
		case 0xf800:
			mapper67_prg = data;
			break;
	}
	mapper_map();
}

static void mapper67_map()
{
	mapper_map_prg(16, 0, mapper67_prg);
	mapper_map_prg(16, 1, -1);

	mapper_map_chr( 2, 0, mapper67_chr(0));
	mapper_map_chr( 2, 1, mapper67_chr(1));
	mapper_map_chr( 2, 2, mapper67_chr(2));
	mapper_map_chr( 2, 3, mapper67_chr(3));

	switch (mapper67_mirror) {
		case 0: set_mirroring(VERTICAL); break;
		case 1: set_mirroring(HORIZONTAL); break;
		case 2: set_mirroring(SINGLE_LOW); break;
		case 3: set_mirroring(SINGLE_HIGH); break;
	}
}

static void mapper67_cycle()
{
	if (mapper67_irqenable) {
		mapper67_irqcount--;

		if (mapper67_irqcount == 0xffff) {
			M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
			mapper67_irqenable = 0;
		}
	}
}
#undef mapper67_mirror
#undef mapper67_irqenable
#undef mapper67_irqtoggle
#undef mapper67_prg
#undef mapper67_chr
#undef mapper67_irqcount

// --[ mapper 68 - Sunsoft-4 (After Burner 1 & 2, Nantettatte!! Baseball, Maharaja)
// Notes: Nantettatte!! Baseball ext.roms not supported yet.
#define mapper68_mirror		(mapper_regs[0x1f - 0])
#define mapper68_prg		(mapper_regs[0x1f - 1])
#define mapper68_wram_en    (mapper_regs[0x1f - 2])
#define mapper68_nt0		(mapper_regs[0x1f - 3])
#define mapper68_nt1		(mapper_regs[0x1f - 4])
#define mapper68_chr(x)		(mapper_regs[0 + (x)])

static INT32 *mapper68_timer = (INT32*)&mapper_regs16[0];

static UINT8 mapper68_exp_read(UINT16 address) // 6000-7fff
{
	if (mapper68_wram_en) {
		return Cart.WorkRAM[address & 0x1fff];
	} else {
		return cpu_open_bus;
	}
}

static void mapper68_exp_write(UINT16 address, UINT8 data) // 6000-7fff
{
	if (mapper68_wram_en) {
		mapper68_timer[0] = 107520;
	} else {
		cart_exp_write_abort = 1; // wram/sram area disabled (disable writing on return from this callback)
	}
}

static void mapper68_write(UINT16 address, UINT8 data)
{
	switch (address & 0xf000) {
		case 0x8000: mapper68_chr(0) = data; break;
		case 0x9000: mapper68_chr(1) = data; break;
		case 0xa000: mapper68_chr(2) = data; break;
		case 0xb000: mapper68_chr(3) = data; break;
		case 0xc000: mapper68_nt0 = data | 0x80; break;
		case 0xd000: mapper68_nt1 = data | 0x80; break;
		case 0xe000: mapper68_mirror = data; break;
		case 0xf000:
			mapper68_prg = data & 7;
			mapper68_wram_en = data & 0x10;
			break;
	}
	mapper_map();
}

static void mapper68_map()
{
	mapper_map_prg(16, 0, mapper68_prg & 7);
	mapper_map_prg(16, 1, -1);

	mapper_map_chr( 2, 0, mapper68_chr(0));
	mapper_map_chr( 2, 1, mapper68_chr(1));
	mapper_map_chr( 2, 2, mapper68_chr(2));
	mapper_map_chr( 2, 3, mapper68_chr(3));

	if (mapper68_mirror & 0x10) {
		switch (mapper68_mirror & 3) {
			case 0:
				nametable_mapraw(0, Cart.CHRRom + (mapper68_nt0 << 10), MEM_ROM);
				nametable_mapraw(2, Cart.CHRRom + (mapper68_nt0 << 10), MEM_ROM);
				nametable_mapraw(1, Cart.CHRRom + (mapper68_nt1 << 10), MEM_ROM);
				nametable_mapraw(3, Cart.CHRRom + (mapper68_nt1 << 10), MEM_ROM);
				break;
			case 1:
				nametable_mapraw(0, Cart.CHRRom + (mapper68_nt0 << 10), MEM_ROM);
				nametable_mapraw(1, Cart.CHRRom + (mapper68_nt0 << 10), MEM_ROM);
				nametable_mapraw(2, Cart.CHRRom + (mapper68_nt1 << 10), MEM_ROM);
				nametable_mapraw(3, Cart.CHRRom + (mapper68_nt1 << 10), MEM_ROM);
				break;
			case 2:
				nametable_mapraw(0, Cart.CHRRom + (mapper68_nt0 << 10), MEM_ROM);
				nametable_mapraw(1, Cart.CHRRom + (mapper68_nt0 << 10), MEM_ROM);
				nametable_mapraw(2, Cart.CHRRom + (mapper68_nt0 << 10), MEM_ROM);
				nametable_mapraw(3, Cart.CHRRom + (mapper68_nt0 << 10), MEM_ROM);
				break;
			case 3:
				nametable_mapraw(0, Cart.CHRRom + (mapper68_nt1 << 10), MEM_ROM);
				nametable_mapraw(1, Cart.CHRRom + (mapper68_nt1 << 10), MEM_ROM);
				nametable_mapraw(2, Cart.CHRRom + (mapper68_nt1 << 10), MEM_ROM);
				nametable_mapraw(3, Cart.CHRRom + (mapper68_nt1 << 10), MEM_ROM);
				break;
		}

	} else {
		switch (mapper68_mirror & 3) {
			case 0: set_mirroring(VERTICAL); break;
			case 1: set_mirroring(HORIZONTAL); break;
			case 2: set_mirroring(SINGLE_LOW); break;
			case 3: set_mirroring(SINGLE_HIGH); break;
		}
	}
}

static void mapper68_cycle()
{
	if (mapper68_timer[0] > 0) {
		mapper68_timer[0]--;

		if (mapper68_timer[0] == 0) {
		}
	}
}

#undef mapper68_mirror
#undef mapper68_prg
#undef mapper68_chr
#undef mapper68_wram_en
#undef mapper68_nt0
#undef mapper68_nt1

// --[ mapper 69 (sunsoft fme-7: Gimmick!, Hebereke, Pyokotan)
#define mapper69_mirror		(mapper_regs[0x1f - 0])
#define mapper69_regnum		(mapper_regs[0x1f - 1])
#define mapper69_irqenable	(mapper_regs[0x1f - 2])
#define mapper69_irqcount   (mapper_regs16[0])

static void mapper69_write(UINT16 address, UINT8 data)
{
	switch (address & 0xe000) {
		case 0x8000: mapper69_regnum = data & 0xf; break;
		case 0xa000:
			switch (mapper69_regnum) {
				case 0x0:
				case 0x1:
				case 0x2:
				case 0x3:
				case 0x4:
				case 0x5:
				case 0x6:
				case 0x7: mapper_regs[(mapper69_regnum & 0xf)] = data; break;
				case 0x8: mapper_regs[0x8 + 3] = data; break;
				case 0x9: mapper_regs[0x8 + 0] = data; break;
				case 0xa: mapper_regs[0x8 + 1] = data; break;
				case 0xb: mapper_regs[0x8 + 2] = data; break;
				case 0xc: mapper69_mirror = data & 3; break;
				case 0xd: mapper69_irqenable = data; M6502SetIRQLine(0, CPU_IRQSTATUS_NONE); break;
				case 0xe: mapper69_irqcount = (mapper69_irqcount & 0xff00) | data; M6502SetIRQLine(0, CPU_IRQSTATUS_NONE); break;
				case 0xf: mapper69_irqcount = (mapper69_irqcount & 0x00ff) | (data << 8); M6502SetIRQLine(0, CPU_IRQSTATUS_NONE); break;
			}
			if (mapper69_regnum < 0xd)
				mapper_map();
			break;

		case 0xc000:
			AY8910Write(0, 0, data);
			break;
		case 0xe000:
			AY8910Write(0, 1, data);
			break;
	}
}

static void mapper69_map()
{
	if ((mapper_regs[8 + 3] & 0xc0) == 0xc0) {
		// set prg-ram @ 6000 - 7fff
		// handled in mapper69_exp_write/read
	} else {
		// set prg-rom @ 6000 - 7fff
		mapper_map_exp_prg(mapper_regs[8 + 3] & 0x3f);
	}

	mapper_map_prg( 8, 0, mapper_regs[8 + 0]);
	mapper_map_prg( 8, 1, mapper_regs[8 + 1]);
	mapper_map_prg( 8, 2, mapper_regs[8 + 2]);
	mapper_map_prg( 8, 3, -1);

	mapper_map_chr(1, 0, mapper_regs[0]);
	mapper_map_chr(1, 1, mapper_regs[1]);
	mapper_map_chr(1, 2, mapper_regs[2]);
	mapper_map_chr(1, 3, mapper_regs[3]);
	mapper_map_chr(1, 4, mapper_regs[4]);
	mapper_map_chr(1, 5, mapper_regs[5]);
	mapper_map_chr(1, 6, mapper_regs[6]);
	mapper_map_chr(1, 7, mapper_regs[7]);

	switch (mapper69_mirror) {
        case 0: set_mirroring(VERTICAL); break;
        case 1: set_mirroring(HORIZONTAL); break;
        case 2: set_mirroring(SINGLE_LOW); break;
        case 3: set_mirroring(SINGLE_HIGH); break;
	}
}

static void mapper69_exp_write(UINT16 address, UINT8 data)
{
	if ((mapper_regs[8 + 3] & 0xc0) == 0xc0) {
		Cart.WorkRAM[address & Cart.WorkRAMMask] = data;
	}
	cart_exp_write_abort = 1;
}

static UINT8 mapper69_exp_read(UINT16 address)
{
	if ((mapper_regs[8 + 3] & 0xc0) == 0x40) { // ram selected, but disabled
		return cpu_open_bus;
	} if ((mapper_regs[8 + 3] & 0xc0) == 0xc0) { // ram selected and enabled
		return Cart.WorkRAM[address & Cart.WorkRAMMask];
	} else { // rom selected
		return Cart.PRGRom[PRGExpMap + (address & 0x1fff)];
	}
}

static void mapper69_cycle()
{
	if (mapper69_irqenable) {
		mapper69_irqcount--;
		if (mapper69_irqcount == 0xffff) {
			M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
			mapper69_irqenable = 0;
		}
	}
}

#undef mapper69_mirror
#undef mapper69_regnum
#undef mapper69_irqenable

// --[ mapper 70: (Kamen Rider Club)
static void mapper70_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper70_map()
{
	mapper_map_prg(16, 0, mapper_regs[0] >> 4);
	mapper_map_prg(16, 1, -1);
	mapper_map_chr( 8, 0, mapper_regs[0] & 0xf);
}

// --[ mapper 71: Camerica (Bee 52, Big Nose, Fire Hawk, Micro Machines, ...)
static void mapper71_write(UINT16 address, UINT8 data)
{
	if ((address & 0xf000) == 0x9000) {
		mapper_regs[1] = ((data & 0x10) ? SINGLE_HIGH : SINGLE_LOW);
	} else {
		mapper_regs[0] = data;
	}

	mapper_map();
}

static void mapper71_map()
{
	mapper_map_prg(16, 0, mapper_regs[0]);
	mapper_map_prg(16, 1, -1);
	mapper_map_chr( 8, 0, 0);

	if (mapper_regs[1]) {
		// only fire hawk sets this, everything else uses hard-mirroring
		set_mirroring(mapper_regs[1]);
	}
}

// --[ mapper 73 - Konami vrc3 (Salamander)
// note: the flickery pixels on the top-right of HUD also happens on FC HW -dink
#define mapper73_irqenable	(mapper_regs[0x1f - 1])
#define mapper73_irqmode	(mapper_regs[0x1f - 2])
#define mapper73_irqreload  (mapper_regs[0x1f - 3])
#define mapper73_prg		(mapper_regs[0])
#define mapper73_irqcount	(mapper_regs16[0x1f - 0])
#define mapper73_irqlatch	(mapper_regs16[0x1f - 1])

static void mapper73_write(UINT16 address, UINT8 data)
{
	switch (address & 0xf000) {
		case 0x8000: mapper73_irqlatch = (mapper73_irqlatch & 0xfff0) | (data & 0xf); break;
		case 0x9000: mapper73_irqlatch = (mapper73_irqlatch & 0xff0f) | ((data & 0xf) << 4); break;
		case 0xa000: mapper73_irqlatch = (mapper73_irqlatch & 0xf0ff) | ((data & 0xf) << 8); break;
		case 0xb000: mapper73_irqlatch = (mapper73_irqlatch & 0x0fff) | ((data & 0xf) << 12); break;
		case 0xc000:
			mapper73_irqreload = data & 1;
			mapper73_irqenable = data & 2;
			mapper73_irqmode = data & 4;

			if (mapper73_irqenable) {
				if (mapper73_irqmode) {
					// when is this mode used? I can't make it happen -dink
					mapper73_irqcount = mapper73_irqlatch & 0xff;
				} else {
					mapper73_irqcount = mapper73_irqlatch;
				}
			}
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		case 0xd000:
			mapper73_irqenable = mapper73_irqreload;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		case 0xf000:
			mapper73_prg = data;
			break;
	}
	mapper_map();
}

static void mapper73_map()
{
	mapper_map_prg(16, 0, mapper73_prg);
	mapper_map_prg(16, 1, -1);
	mapper_map_chr( 8, 0, 0);
}

static void mapper73_cycle()
{
	if (mapper73_irqenable) {
		if ( (mapper73_irqmode && (mapper73_irqcount & 0xff) == 0xff) ||
			((mapper73_irqmode == 0) && mapper73_irqcount == 0xffff) ) {
			mapper73_irqcount = mapper73_irqlatch;
			M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
		} else {
			mapper73_irqcount++;
		}
	}
}
#undef mapper73_irqenable
#undef mapper73_irqmode
#undef mapper73_irqreload
#undef mapper73_prg
#undef mapper73_irqcount
#undef mapper73_irqlatch

// --[ mapper 75: Konami VRC1 (King Kong 2, Tetsuwan Atom, ...)
#define mapper75_prg(x)     (mapper_regs[0 + x]) // 0 - 2
#define mapper75_chr(x)     (mapper_regs[3 + x]) // 3 - 4
#define mapper75_ext        (mapper_regs[8])
static void mapper75_write(UINT16 address, UINT8 data)
{
	switch (address & 0xf000) {
		case 0x8000: mapper75_prg(0) = data; break;
		case 0x9000: mapper75_ext = data; break;
		case 0xa000: mapper75_prg(1) = data; break;
		case 0xc000: mapper75_prg(2) = data; break;
		case 0xe000: mapper75_chr(0) = data; break;
		case 0xf000: mapper75_chr(1) = data; break;
	}

	mapper_map();
}

static void mapper75_map()
{
	mapper_map_prg( 8, 0, mapper75_prg(0));
	mapper_map_prg( 8, 1, mapper75_prg(1));
	mapper_map_prg( 8, 2, mapper75_prg(2));
	mapper_map_prg( 8, 3, -1);

	mapper_map_chr( 4, 0, (mapper75_chr(0) & 0xf) | ((mapper75_ext & 2) << 3));
	mapper_map_chr( 4, 1, (mapper75_chr(1) & 0xf) | ((mapper75_ext & 4) << 2));

	if (Cart.Mirroring != 4) {
		set_mirroring((mapper75_ext & 1) ? HORIZONTAL : VERTICAL);
	}
}
#undef mapper75_prg
#undef mapper75_chr
#undef mapper75_ext

// --[ mapper 85 (VRC7) Lagrange Point, Tiny Toon Adventures 2 (Japan)
#define mapper85_prg(x)		(mapper_regs[0x00 + x])
#define mapper85_chr(x)		(mapper_regs[0x03 + x])
#define mapper85_mirror		(mapper_regs[0x10])
#define mapper85_wramen     (mapper_regs[0x11])
#define mapper85_irqenable	(mapper_regs[0x12])
#define mapper85_irqrepeat	(mapper_regs[0x13])
#define mapper85_irqlatch	(mapper_regs[0x14])
#define mapper85_irqmode	(mapper_regs[0x15])
#define mapper85_irqcount	(mapper_regs16[0x1f - 0])
#define mapper85_irqcycle	(mapper_regs16[0x1f - 1])
#define vrc7_mute           (mapper_regs[0x16])

static void vrc7_map()
{
	mapper_map_prg( 8, 0, mapper85_prg(0));
	mapper_map_prg( 8, 1, mapper85_prg(1));
	mapper_map_prg( 8, 2, mapper85_prg(2));
	mapper_map_prg( 8, 3, -1);

	for (INT32 i = 0; i < 8; i++)
		mapper_map_chr( 1, i, mapper85_chr(i));

	switch (mapper85_mirror) {
		case 0: set_mirroring(VERTICAL); break;
		case 1: set_mirroring(HORIZONTAL); break;
		case 2: set_mirroring(SINGLE_LOW); break;
		case 3: set_mirroring(SINGLE_HIGH); break;
	}

	if (mapper85_wramen) {
		NESMode &= ~NO_WORKRAM;
	} else {
		NESMode |= NO_WORKRAM;
	}
}

static void vrc7_write(UINT16 address, UINT8 data)
{
	if (address & 0x18) {
		// VRC7a uses xx10, VRC7b uses xxx8
		address = (address & ~0x0008) | 0x0010;
	}

	switch (address & 0xf030) {
		case 0x8000: mapper85_prg(0) = data & 0x3f; break;
		case 0x8010: mapper85_prg(1) = data & 0x3f; break;
		case 0x9000: mapper85_prg(2) = data & 0x3f; break;

		case 0x9010: BurnYM2413Write(0, data); break;
		case 0x9030: BurnYM2413Write(1, data); break;

		case 0xa000: mapper85_chr(0) = data; break;
		case 0xa010: mapper85_chr(1) = data; break;
		case 0xb000: mapper85_chr(2) = data; break;
		case 0xb010: mapper85_chr(3) = data; break;
		case 0xc000: mapper85_chr(4) = data; break;
		case 0xc010: mapper85_chr(5) = data; break;
		case 0xd000: mapper85_chr(6) = data; break;
		case 0xd010: mapper85_chr(7) = data; break;

		case 0xe000:
			mapper85_mirror = data & 0x3;
			mapper85_wramen = data & 0x80;
			vrc7_mute = data & 0x40;
			break;

		case 0xe010:
			mapper85_irqlatch = data;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		case 0xf000:
			mapper85_irqrepeat = data & 1;
			mapper85_irqenable = data & 2;
			mapper85_irqmode = data & 4;
			if (mapper85_irqenable) {
				mapper85_irqcycle = 0;
				mapper85_irqcount = mapper85_irqlatch;
			}
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		case 0xf010:
			mapper85_irqenable = mapper85_irqrepeat;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
	}

	mapper_map();
}

static void vrc7_cycle()
{
	if (mapper85_irqenable) {
		if (mapper85_irqmode) { // cycle mode
			mapper85_irqcount++;
			if (mapper85_irqcount >= 0x100) {
				M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
				mapper_irq(4);
				mapper85_irqcount = mapper85_irqlatch;
			}
		} else {
			mapper85_irqcycle += 3; // scanline mode
			if (mapper85_irqcycle >= 341) {
				mapper85_irqcycle -= 341;
				mapper85_irqcount++;
				if (mapper85_irqcount == 0x100) {
					M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
					mapper_irq(4);
					mapper85_irqcount = mapper85_irqlatch;
				}
			}
		}
	}
}
#undef mapper85_prg
#undef mapper85_chr
#undef mapper85_mirror
#undef mapper85_wramen
#undef mapper85_irqenable
#undef mapper85_irqrepeat
#undef mapper85_irqlatch
#undef mapper85_irqmode
#undef mapper85_irqcount
#undef mapper85_irqcycle

// --[ mapper 116: Somari (AV Girl Fighting)
// it's a frankensteinian mess-mapper -dink
#define mapper116_vrc2_prg(x)		(mapper_regs[0x00 + (x)])
#define mapper116_vrc2_chr(x)		(mapper_regs[0x02 + (x)])
#define mapper116_vrc2_mirror		(mapper_regs[0x0a])
#define mapper116_mode				(mapper_regs[0x0b])

#define mapper116_mmc3_banksel      (mapper_regs[0x0c])
#define mapper116_mmc3_mirror		(mapper_regs[0x0d])
#define mapper116_mmc3_irqlatch 	(mapper_regs[0x0e])
#define mapper116_mmc3_irqcount		(mapper_regs[0x0f])
#define mapper116_mmc3_irqenable	(mapper_regs[0x10])
#define mapper116_mmc3_irqreload	(mapper_regs[0x11])
#define mapper116_mmc3_regs(x)		(mapper_regs16[0x12 + (x)]) // must be regs16!

#define mapper116_mmc1_regs(x)		(mapper_regs[0x1b + (x)])
#define mapper116_mmc1_serialbyte	(mapper_regs16[0])
#define mapper116_mmc1_bitcount		(mapper_regs16[1])

static void mapper116_defaults()
{
	mapper116_vrc2_prg(0) = 0;
	mapper116_vrc2_prg(1) = 1;
	mapper116_vrc2_chr(0) = 0xff;
	mapper116_vrc2_chr(1) = 0xff;
	mapper116_vrc2_chr(2) = 0xff;
	mapper116_vrc2_chr(3) = 0xff;
	mapper116_vrc2_chr(4) = 4;
	mapper116_vrc2_chr(5) = 5;
	mapper116_vrc2_chr(6) = 6;
	mapper116_vrc2_chr(7) = 7;

	mapper116_mmc1_regs(0) = 0x0c;

	mapper116_mmc3_regs(0) = 0;
	mapper116_mmc3_regs(1) = 2;
	mapper116_mmc3_regs(2) = 4;
	mapper116_mmc3_regs(3) = 5;
	mapper116_mmc3_regs(4) = 6;
	mapper116_mmc3_regs(5) = 7;
	mapper116_mmc3_regs(6) = -4;
	mapper116_mmc3_regs(7) = -3;
	mapper116_mmc3_regs(8) = -2;
	mapper116_mmc3_regs(9) = -1;
}

static void mapper116_write(UINT16 address, UINT8 data)
{
	if (address < 0x8000) {
		if ((address & 0x4100) == 0x4100) {
			// Someri
			mapper116_mode = data;
			if (address & 1) {
				mapper116_mmc1_bitcount = 0;
				mapper116_mmc1_serialbyte = 0;
				mapper116_mmc1_regs(0) = 0x0c;
				mapper116_mmc1_regs(3) = 0x00;
			}
			mapper_map();
		}
	} else {
		if (address == 0xa131) {
			// Gouder SL-1632, mode & 2 == mmc3, ~mode & 2 == vrc2
			mapper116_mode = (data & ~3) | ((data & 2) >> 1);
		}
		switch (mapper116_mode & 3) {
			case 0:	{
				if (address >= 0xb000 && address <= 0xe003) {
					INT32 reg = ((((address & 2) | (address >> 10)) >> 1) + 2) & 7;

					if (~address & 1) {
						mapper116_vrc2_chr(reg) = (mapper116_vrc2_chr(reg) & 0xf0) | (data & 0x0f);
					} else {
						mapper116_vrc2_chr(reg) = (mapper116_vrc2_chr(reg) & 0x0f) | ((data & 0x0f) << 4);
					}
				} else {
					switch (address & 0xf000) {
						case 0x8000: mapper116_vrc2_prg(0) = data; break;
						case 0x9000: mapper116_vrc2_mirror = data & 1; break;
						case 0xa000: mapper116_vrc2_prg(1) = data; break;
					}
				}
				mapper_map();
			}
			break;
			case 1: {
				switch (address & 0xe001) {
					case 0x8000: mapper116_mmc3_banksel = data; break;
					case 0x8001: mapper116_mmc3_regs(mapper116_mmc3_banksel & 0x7) = data; break;
					case 0xa000: mapper116_mmc3_mirror = data & 1; break;
					case 0xc000: mapper116_mmc3_irqlatch = data; break;
					case 0xc001: mapper116_mmc3_irqreload = 1; break;
					case 0xe000: mapper116_mmc3_irqenable = 0; M6502SetIRQLine(0, CPU_IRQSTATUS_NONE); break;
					case 0xe001: mapper116_mmc3_irqenable = 1; break;
				}
				mapper_map();
			}
			break;
			case 2:
			case 3: {
				if (address & 0x8000) {
					if (data & 0x80) { // bit 7, mapper reset
						mapper116_mmc1_bitcount = 0;
						mapper116_mmc1_serialbyte = 0;
						mapper116_mmc1_regs(0) |= 0x0c;
						if (mapper_map) mapper_map();
					} else {
						mapper116_mmc1_serialbyte |= (data & 1) << mapper116_mmc1_bitcount;
						mapper116_mmc1_bitcount++;
						if (mapper116_mmc1_bitcount == 5) {
							UINT8 reg = (address >> 13) & 0x3;
							mapper116_mmc1_regs(reg) = mapper116_mmc1_serialbyte;
							mapper116_mmc1_bitcount = 0;
							mapper116_mmc1_serialbyte = 0;
							if (mapper_map) mapper_map();
						}
					}
				}
			}
			break;
		}
	}
}

static void mapper116_map()
{
	INT32 bank = (mapper116_mode & 4) << 6;
	switch (mapper116_mode & 3) {
		case 0: {
			mapper_map_chr( 1, 0, mapper116_vrc2_chr(0) | bank);
			mapper_map_chr( 1, 1, mapper116_vrc2_chr(1) | bank);
			mapper_map_chr( 1, 2, mapper116_vrc2_chr(2) | bank);
			mapper_map_chr( 1, 3, mapper116_vrc2_chr(3) | bank);
			mapper_map_chr( 1, 4, mapper116_vrc2_chr(4) | bank);
			mapper_map_chr( 1, 5, mapper116_vrc2_chr(5) | bank);
			mapper_map_chr( 1, 6, mapper116_vrc2_chr(6) | bank);
			mapper_map_chr( 1, 7, mapper116_vrc2_chr(7) | bank);

			mapper_map_prg( 8, 0, mapper116_vrc2_prg(0));
			mapper_map_prg( 8, 1, mapper116_vrc2_prg(1));
			mapper_map_prg( 8, 2, -2);
			mapper_map_prg( 8, 3, -1);

			set_mirroring((mapper116_vrc2_mirror) ? HORIZONTAL : VERTICAL);
		}
		break;
		case 1: {
			mapper_map_prg(8, ((mapper116_mmc3_banksel & 0x40) >> 5), mapper116_mmc3_regs(6));
			mapper_map_prg(8, 1, mapper116_mmc3_regs(7));
			mapper_map_prg(8, 2 ^ ((mapper116_mmc3_banksel & 0x40) >> 5), mapper116_mmc3_regs(8));
			mapper_map_prg(8, 3, mapper116_mmc3_regs(9));

			INT32 swap = (mapper116_mmc3_banksel & 0x80) >> 5;
			mapper_map_chr( 1, 0 ^ swap, (mapper116_mmc3_regs(0) & 0xfe) | bank);
			mapper_map_chr( 1, 1 ^ swap, (mapper116_mmc3_regs(0) | 0x01) | bank);
			mapper_map_chr( 1, 2 ^ swap, (mapper116_mmc3_regs(1) & 0xfe) | bank);
			mapper_map_chr( 1, 3 ^ swap, (mapper116_mmc3_regs(1) | 0x01) | bank);
			mapper_map_chr( 1, 4 ^ swap, mapper116_mmc3_regs(2) | bank);
			mapper_map_chr( 1, 5 ^ swap, mapper116_mmc3_regs(3) | bank);
			mapper_map_chr( 1, 6 ^ swap, mapper116_mmc3_regs(4) | bank);
			mapper_map_chr( 1, 7 ^ swap, mapper116_mmc3_regs(5) | bank);

			set_mirroring((mapper116_mmc3_mirror) ? HORIZONTAL : VERTICAL);
		}
		break;
		case 2:
		case 3: {
			if (mapper116_mmc1_regs(0) & 0x8) {
				if (mapper116_mmc1_regs(0) & 0x4) {
					mapper_map_prg(16, 0, mapper116_mmc1_regs(3) & 0xf);
					mapper_map_prg(16, 1, 0x0f);
				} else {
					mapper_map_prg(16, 0, 0);
					mapper_map_prg(16, 1, mapper116_mmc1_regs(3) & 0xf);
				}
			} else {
				mapper_map_prg(32, 0, (mapper116_mmc1_regs(3) & 0xf) >> 1);
			}

			if (mapper116_mmc1_regs(0) & 0x10) {
				mapper_map_chr( 4, 0, mapper116_mmc1_regs(1));
				mapper_map_chr( 4, 1, mapper116_mmc1_regs(2));
			} else {
				mapper_map_chr( 8, 0, mapper116_mmc1_regs(1) >> 1);
			}

			switch (mapper116_mmc1_regs(0) & 3) {
				case 0: set_mirroring(SINGLE_LOW); break;
				case 1: set_mirroring(SINGLE_HIGH); break;
				case 2: set_mirroring(VERTICAL); break;
				case 3: set_mirroring(HORIZONTAL); break;
			}
		}
		break;
	}
}

static void mapper14_map()
{
	switch (mapper116_mode & 3) {
		case 0: {
			mapper_map_chr( 1, 0, mapper116_vrc2_chr(0));
			mapper_map_chr( 1, 1, mapper116_vrc2_chr(1));
			mapper_map_chr( 1, 2, mapper116_vrc2_chr(2));
			mapper_map_chr( 1, 3, mapper116_vrc2_chr(3));
			mapper_map_chr( 1, 4, mapper116_vrc2_chr(4));
			mapper_map_chr( 1, 5, mapper116_vrc2_chr(5));
			mapper_map_chr( 1, 6, mapper116_vrc2_chr(6));
			mapper_map_chr( 1, 7, mapper116_vrc2_chr(7));

			mapper_map_prg( 8, 0, mapper116_vrc2_prg(0));
			mapper_map_prg( 8, 1, mapper116_vrc2_prg(1));
			mapper_map_prg( 8, 2, -2);
			mapper_map_prg( 8, 3, -1);

			set_mirroring((mapper116_vrc2_mirror) ? HORIZONTAL : VERTICAL);
		}
		break;
		case 1: {
			mapper_map_prg(8, ((mapper116_mmc3_banksel & 0x40) >> 5), mapper116_mmc3_regs(6));
			mapper_map_prg(8, 1, mapper116_mmc3_regs(7));
			mapper_map_prg(8, 2 ^ ((mapper116_mmc3_banksel & 0x40) >> 5), mapper116_mmc3_regs(8));
			mapper_map_prg(8, 3, mapper116_mmc3_regs(9));

			INT32 swap = (mapper116_mmc3_banksel & 0x80) >> 5;
			INT32 bank0 = (mapper116_mode & 0x08) << 5;
			INT32 bank1 = (mapper116_mode & 0x20) << 3;
			INT32 bank2 = (mapper116_mode & 0x80) << 1;
			mapper_map_chr( 1, 0 ^ swap, (mapper116_mmc3_regs(0) & 0xfe) | bank0);
			mapper_map_chr( 1, 1 ^ swap, (mapper116_mmc3_regs(0) | 0x01) | bank0);
			mapper_map_chr( 1, 2 ^ swap, (mapper116_mmc3_regs(1) & 0xfe) | bank0);
			mapper_map_chr( 1, 3 ^ swap, (mapper116_mmc3_regs(1) | 0x01) | bank0);
			mapper_map_chr( 1, 4 ^ swap, mapper116_mmc3_regs(2) | bank1);
			mapper_map_chr( 1, 5 ^ swap, mapper116_mmc3_regs(3) | bank1);
			mapper_map_chr( 1, 6 ^ swap, mapper116_mmc3_regs(4) | bank2);
			mapper_map_chr( 1, 7 ^ swap, mapper116_mmc3_regs(5) | bank2);

			set_mirroring((mapper116_mmc3_mirror) ? HORIZONTAL : VERTICAL);
		}
		break;
	}
}

static void mapper116_mmc3_scanline()
{
	if ((mapper116_mode & 0x03) != 1) return;

	if (mapper116_mmc3_irqcount == 0 || mapper116_mmc3_irqreload) {
		mapper116_mmc3_irqreload = 0;
		mapper116_mmc3_irqcount = mapper116_mmc3_irqlatch;
	} else {
		mapper116_mmc3_irqcount--;
	}

	if (mapper116_mmc3_irqcount == 0 && mapper116_mmc3_irqenable && (mmc5_mask[0] & 0x18)) {
		mapper_irq(0);
	}
}
#undef mapper116_vrc2_prg
#undef mapper116_vrc2_chr
#undef mapper116_vrc2_mirror
#undef mapper116_mode

#undef mapper116_mmc3_banksel
#undef mapper116_mmc3_mirror
#undef mapper116_mmc3_irqlatch
#undef mapper116_mmc3_irqcount
#undef mapper116_mmc3_irqenable
#undef mapper116_mmc3_irqreload
#undef mapper116_mmc3_regs

#undef mapper116_mmc1_regs
#undef mapper116_mmc1_serialbyte
#undef mapper116_mmc1_bitcount

// --[ mapper 80: Taito X1-005
// --[ mapper 207: with advanced mirroring
#define mapper207_prg(x)     (mapper_regs[0 + x]) // 0 - 2
#define mapper207_chr(x)     (mapper_regs[3 + x]) // 3 - 9
#define mapper207_mirror     (mapper_regs[0x1f - 0])
#define mapper207_ram_en     (mapper_regs[0x1f - 1])
#define mapper207_adv_mirror (mapper_regs[0x1f - 2]) // mapper 207 yes, 80 no

static UINT8 mapper80_ram[0x80];

static void mapper207_scan()
{
	SCAN_VAR(mapper80_ram);
}

static UINT8 mapper207_read(UINT16 address)
{
	if (address >= 0x7f00 && address <= 0x7fff) {
		if (mapper207_ram_en == 0xa3) {
			return mapper80_ram[address & 0x7f];
		}
	}
	return 0xff;
}

static void mapper207_write(UINT16 address, UINT8 data)
{
	if (address >= 0x7f00 && address <= 0x7fff) {
		if (mapper207_ram_en == 0xa3) {
			mapper80_ram[address & 0x7f] = data;
		}
	}

	switch (address) {
		case 0x7ef0: mapper207_chr(0) = data; break;
		case 0x7ef1: mapper207_chr(1) = data; break;
		case 0x7ef2: mapper207_chr(2) = data; break;
		case 0x7ef3: mapper207_chr(3) = data; break;
		case 0x7ef4: mapper207_chr(4) = data; break;
		case 0x7ef5: mapper207_chr(5) = data; break;
		case 0x7ef6:
		case 0x7ef7: mapper207_mirror = data & 1; break; // for mapper80
		case 0x7ef8:
		case 0x7ef9: mapper207_ram_en = data; break; // for mapper80
		case 0x7efa:
		case 0x7efb: mapper207_prg(0) = data; break;
		case 0x7efc:
		case 0x7efd: mapper207_prg(1) = data; break;
		case 0x7efe:
		case 0x7eff: mapper207_prg(2) = data; break;
	}

	mapper_map();
}

static void mapper207_map()
{
	mapper_map_prg( 8, 0, mapper207_prg(0));
	mapper_map_prg( 8, 1, mapper207_prg(1));
	mapper_map_prg( 8, 2, mapper207_prg(2));
	mapper_map_prg( 8, 3, -1);

	mapper_map_chr( 2, 0, (mapper207_chr(0) >> 1) & 0x3f);
	mapper_map_chr( 2, 1, (mapper207_chr(1) >> 1) & 0x3f);
	mapper_map_chr( 1, 4, mapper207_chr(2));
	mapper_map_chr( 1, 5, mapper207_chr(3));
	mapper_map_chr( 1, 6, mapper207_chr(4));
	mapper_map_chr( 1, 7, mapper207_chr(5));

	if (mapper207_adv_mirror == 0) {
		set_mirroring((mapper207_mirror & 1) ? VERTICAL : HORIZONTAL);
	} else {
		nametable_mapall((mapper207_chr(0) >> 7) & 1, (mapper207_chr(0) >> 7) & 1, (mapper207_chr(1) >> 7) & 1, (mapper207_chr(1) >> 7) & 1);
	}
}
#undef mapper207_prg
#undef mapper207_chr
#undef mapper207_mirror
#undef mapper207_ram_en

// ---[ mapper 81 NTDEC Super Gun (Caltron)
static void mapper81_write(UINT16 address, UINT8 data)
{
	if (address & 0x8000) {
		mapper_regs[0] = data;
		mapper_map();
	}
}

static void mapper81_map()
{
    mapper_map_prg(16, 0, (mapper_regs[0] >> 2) & 0x03);
    mapper_map_prg(16, 1, -1);
    mapper_map_chr( 8, 0, mapper_regs[0] & 0x03);
}

// --[ mapper 82: Taito X1-017
#define mapper82_prg(x)     (mapper_regs[0 + (x)]) // 0 - 2
#define mapper82_chr(x)     (mapper_regs[3 + (x)]) // 3 - 8
#define mapper82_mirror     (mapper_regs[0x1f - 0])
#define mapper82_a12inv     (mapper_regs[0x1f - 1])
#define mapper82_ram_en(x)  (mapper_regs[9 + (x)])

static UINT8 mapper82_read(UINT16 address)
{
	if ( ((address >= 0x6000 && address <= 0x67ff) && mapper82_ram_en(0)) ||
		 ((address >= 0x6800 && address <= 0x6fff) && mapper82_ram_en(1)) ||
		 ((address >= 0x7000 && address <= 0x73ff) && mapper82_ram_en(2))
	   )
		return Cart.WorkRAM[(address & 0x1fff)];

	return cpu_open_bus;
}


static void mapper82_write(UINT16 address, UINT8 data)
{
	cart_exp_write_abort = 1; // don't fall-through after callback!

	if ( ((address >= 0x6000 && address <= 0x67ff) && mapper82_ram_en(0)) ||
		 ((address >= 0x6800 && address <= 0x6fff) && mapper82_ram_en(1)) ||
		 ((address >= 0x7000 && address <= 0x73ff) && mapper82_ram_en(2))
	   )
		Cart.WorkRAM[(address & 0x1fff)] = data;

	switch (address) {
		case 0x7ef0: mapper82_chr(0) = data; break;
		case 0x7ef1: mapper82_chr(1) = data; break;
		case 0x7ef2: mapper82_chr(2) = data; break;
		case 0x7ef3: mapper82_chr(3) = data; break;
		case 0x7ef4: mapper82_chr(4) = data; break;
		case 0x7ef5: mapper82_chr(5) = data; break;
		case 0x7ef6: mapper82_mirror = data & 1; mapper82_a12inv = data & 2; break;
		case 0x7ef7: mapper82_ram_en(0) = (data == 0xca); break;
		case 0x7ef8: mapper82_ram_en(1) = (data == 0x69); break;
		case 0x7ef9: mapper82_ram_en(2) = (data == 0x84); break;
		case 0x7efa: mapper82_prg(0) = data >> 2; break;
		case 0x7efb: mapper82_prg(1) = data >> 2; break;
		case 0x7efc: mapper82_prg(2) = data >> 2; break;
		case 0x7efd: // unused irq latch
		case 0x7efe: // unused irq control
		case 0x7eff: // unused irq ack
			break;
	}

	mapper_map();
}

static void mapper82_map()
{
	mapper_map_prg( 8, 0, mapper82_prg(0));
	mapper_map_prg( 8, 1, mapper82_prg(1));
	mapper_map_prg( 8, 2, mapper82_prg(2));
	mapper_map_prg( 8, 3, -1);

	if (mapper82_a12inv == 0) {
		mapper_map_chr( 2, 0, (mapper82_chr(0) >> 1) & 0x7f);
		mapper_map_chr( 2, 1, (mapper82_chr(1) >> 1) & 0x7f);
		mapper_map_chr( 1, 4, mapper82_chr(2));
		mapper_map_chr( 1, 5, mapper82_chr(3));
		mapper_map_chr( 1, 6, mapper82_chr(4));
		mapper_map_chr( 1, 7, mapper82_chr(5));
	} else {
		mapper_map_chr( 2, 2, (mapper82_chr(0) >> 1) & 0x7f);
		mapper_map_chr( 2, 3, (mapper82_chr(1) >> 1) & 0x7f);
		mapper_map_chr( 1, 0, mapper82_chr(2));
		mapper_map_chr( 1, 1, mapper82_chr(3));
		mapper_map_chr( 1, 2, mapper82_chr(4));
		mapper_map_chr( 1, 3, mapper82_chr(5));
	}

	set_mirroring((mapper82_mirror) ? VERTICAL : HORIZONTAL);
}
#undef mapper82_prg
#undef mapper82_chr
#undef mapper82_mirror
#undef mapper82_a12inv
#undef mapper82_ram_en


// --[ mapper 232: Camerica (Quattro Adventure, Arcade, Sports)
static void mapper232_write(UINT16 address, UINT8 data)
{
	if (address >= 0x8000 && address <= 0xbfff) {
		mapper_regs[0] = data;
	}

	if (address >= 0xc000 && address <= 0xffff) {
		mapper_regs[1] = data;
	}

	mapper_map();
}

static void mapper232_map()
{
	UINT32 block = (mapper_regs[0] & 0x18) >> 1;
	mapper_map_prg(16, 0, block | (mapper_regs[1] & 3));
	mapper_map_prg(16, 1, block | 3);
	mapper_map_chr( 8, 0, 0);
}

// --[ mapper 77: Irem? Napoleon Senki
static void mapper77_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper77_map()
{
	mapper_map_prg(32, 0, mapper_regs[0] & 0x0f);
	mapper_map_chr_ramrom( 2, 0, (mapper_regs[0] >> 4) & 0x0f, MEM_ROM);
	mapper_map_chr_ramrom( 2, 1, 1, MEM_RAM);
	mapper_map_chr_ramrom( 2, 2, 2, MEM_RAM);
	mapper_map_chr_ramrom( 2, 3, 3, MEM_RAM);
}

// --[ mapper 78: Irem? (Holy Diver, Uchuusen: Cosmo Carrier)
static void mapper78_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper78_map()
{
	mapper_map_prg(16, 0, mapper_regs[0] & 7);
	mapper_map_prg(16, 1, -1);
	mapper_map_chr( 8, 0, mapper_regs[0] >> 4);
	if (Cart.Mirroring == 4) // Holy Diver
		set_mirroring(((mapper_regs[0] >> 3) & 1) ? VERTICAL : HORIZONTAL);
	else
		set_mirroring(((mapper_regs[0] >> 3) & 1) ? SINGLE_HIGH : SINGLE_LOW);
}

// --[ mapper 32: Irem G101 (Image Fight, Kaiketsu Yanchamaru 2, Major League, Meikyuu Jima, ...)
#define mapper32_mirror  (mapper_regs[0x1f - 0])
#define mapper32_prg(x)  (mapper_regs[0 + x])
#define mapper32_chr(x)  (mapper_regs[2 + (x)])

static void mapper32_write(UINT16 address, UINT8 data)
{
	switch (address & 0xf000) {
		case 0x8000:
			mapper32_prg(0) = data;
			break;
		case 0x9000:
			mapper32_mirror = data;
			break;
		case 0xa000:
			mapper32_prg(1) = data;
			break;
		case 0xb000:
			mapper32_chr((address & 7)) = data;
			break;
	}
	mapper_map();
}

static void mapper32_map()
{
	mapper_map_prg( 8, 0 ^ (mapper32_mirror & 2), mapper32_prg(0));
	mapper_map_prg( 8, 1, mapper32_prg(1));
	mapper_map_prg( 8, 2 ^ (mapper32_mirror & 2), -2);
	mapper_map_prg( 8, 3, -1);

	for (INT32 i = 0; i < 8; i++)
		mapper_map_chr( 1, i, mapper32_chr(i));

	if (Cart.Crc == 0xd8dfd3d1) {
		// Major League (Japan) hardwired to SINGLE_LOW
		set_mirroring(SINGLE_LOW);
	} else {
		set_mirroring((mapper32_mirror & 1) ? HORIZONTAL : VERTICAL);
	}
}

#undef mapper32_mirror
#undef mapper32_prg
#undef mapper32_chr

// --[ mapper 34: BNROM, NINA-001 (Deadly Towers, Darkseed) (Impossible Mission 2)
static void mapper34_write(UINT16 address, UINT8 data)
{
	if (address >= 0x8000) { // BNROM
		mapper_regs[0] = data;
	} else {
		switch (address) {   // NINA-001
			case 0x7ffd:
			case 0x7ffe:
			case 0x7fff:
				mapper_regs[(address & 0x3) - 1] = data;
				break;
		}
	}
	mapper_map();
}

static void mapper34_map()
{
	mapper_map_prg(32, 0, mapper_regs[0]);

	mapper_map_chr( 4, 0, mapper_regs[1]);
	mapper_map_chr( 4, 1, mapper_regs[2]);
}

// --[ mapper 87: Jaleco (Argus, Ninja JajaMaru Kun, City Connection)
static void mapper87_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = ((data >> 1) & 1) | ((data << 1) & 2);
	mapper_map();
}

static void mapper87_map()
{
	mapper_map_prg(32, 0, 0);
	mapper_map_chr( 8, 0, mapper_regs[0]);
}

// --[ mapper 88,154: Namcot (Quinty, Dragon Spirit..)
#define mapper88_cmd	(mapper_regs[0x1f - 0])
#define mapper88_mirror	(mapper_regs[0x1f - 1])
#define mapper154       (mapper_regs[0x1f - 2])

static void mapper88_write(UINT16 address, UINT8 data)
{
	switch (address & 0x8001) {
		case 0x8000:
			mapper88_cmd = data & 7;
			mapper88_mirror = (data >> 6) & 1;
			break;
		case 0x8001:
			mapper_regs[mapper88_cmd] = data;
			break;
	}

	mapper_map();
}

static void mapper88_map()
{
	mapper_map_chr( 2, 0, (mapper_regs[0] >> 1) & 0x1f);
	mapper_map_chr( 2, 1, (mapper_regs[1] >> 1) & 0x1f);

	mapper_map_chr( 1, 4, mapper_regs[2] | 0x40);
	mapper_map_chr( 1, 5, mapper_regs[3] | 0x40);
	mapper_map_chr( 1, 6, mapper_regs[4] | 0x40);
	mapper_map_chr( 1, 7, mapper_regs[5] | 0x40);

	mapper_map_prg( 8, 0, mapper_regs[6]);
	mapper_map_prg( 8, 1, mapper_regs[7]);
	mapper_map_prg( 8, 2, -2);
	mapper_map_prg( 8, 3, -1);

	if (mapper154) {
		switch (mapper88_mirror) {
			case 0: set_mirroring(SINGLE_LOW); break;
			case 1: set_mirroring(SINGLE_HIGH); break;
		}
	}
}
#undef mapper88_cmd
#undef mapper88_mirror

// --[ mapper 89: (Tenka no Goikenban: Mito Koumon)
static void mapper89_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper89_map()
{
	mapper_map_prg(16, 0, (mapper_regs[0] >> 4) & 7);
	mapper_map_prg(16, 1, -1);
	mapper_map_chr( 8, 0, (mapper_regs[0] & 7) | ((mapper_regs[0] >> 4) & 8));
	set_mirroring((mapper_regs[0] & 8) ? SINGLE_HIGH : SINGLE_LOW);
}

// --[ mapper 93: Sunsoft-1/2 PCB: Fantasy Zone, Shanghai
static void mapper93_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper93_map()
{
	mapper_map_prg(16, 0, mapper_regs[0] >> 4);
	mapper_map_prg(16, 1, -1);
	mapper_map_chr( 8, 0, 0);
}

// --[ mapper 94: Senjou no Ookami (Commando)
static void mapper94_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper94_map()
{
	mapper_map_prg(16, 0, mapper_regs[0] >> 2);
	mapper_map_prg(16, 1, -1);
	mapper_map_chr( 8, 0, 0);
}

// --[ mapper 97: Kaiketsu Yanchamaru
static void mapper97_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper97_map()
{
	mapper_map_prg(16, 0, -1);
	mapper_map_prg(16, 1, mapper_regs[0] & 0xf);
	mapper_map_chr( 8, 0, ((mapper_regs[0] >> 1) & 1) | ((mapper_regs[0] << 1) & 2));
	set_mirroring((mapper_regs[0] & 0x80) ? VERTICAL : HORIZONTAL);
}

// --[ mapper 107: (Magic Dragon)
static void mapper107_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper107_map()
{
	mapper_map_prg(32, 0, (mapper_regs[0] >> 1) & 3);
	mapper_map_chr( 8, 0, mapper_regs[0] & 7);
}

// --[ mapper 101: (Urusei Yatsura - Lum no Wedding Bell (Japan))
static void mapper101_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper101_map()
{
	mapper_map_prg(32, 0, 0);
	mapper_map_chr( 8, 0, mapper_regs[0]);
}

// --[ mapper 112: ASDER (Huang Di, ...)
#define mapper112_mirror		(mapper_regs[0x1f - 0])
#define mapper112_regnum		(mapper_regs[0x1f - 1])
#define mapper112_chrbank		(mapper_regs[0x1f - 2])

static void mapper112_write(UINT16 address, UINT8 data)
{
	switch (address & 0xe001) {
		case 0x8000: mapper112_regnum = data & 7; break;
		case 0xa000: mapper_regs[mapper112_regnum] = data; break;
		case 0xc000: mapper112_chrbank = data; break;
		case 0xe000: mapper112_mirror = data & 1; break;
	}

	mapper_map();
}

static void mapper112_map()
{
	mapper_map_prg( 8, 0, mapper_regs[0]);
	mapper_map_prg( 8, 1, mapper_regs[1]);

	mapper_map_chr( 2, 0, mapper_regs[2] >> 1);
	mapper_map_chr( 2, 1, mapper_regs[3] >> 1);

	mapper_map_chr( 1, 4, ((mapper112_chrbank & 0x10) << 4) | mapper_regs[4]);
	mapper_map_chr( 1, 5, ((mapper112_chrbank & 0x20) << 3) | mapper_regs[5]);
	mapper_map_chr( 1, 6, ((mapper112_chrbank & 0x40) << 2) | mapper_regs[6]);
	mapper_map_chr( 1, 7, ((mapper112_chrbank & 0x80) << 1) | mapper_regs[7]);

	set_mirroring(mapper112_mirror ? HORIZONTAL : VERTICAL);
}

// --[ mapper 113 NINA-03/NINA-06 ext. (Papillon Gals) / Mojon Twins Multicart
static void mapper113_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper113_map()
{
	mapper_map_prg(32, 0, (mapper_regs[0] >> 3) & 7);
	mapper_map_chr( 8, 0, ((mapper_regs[0] >> 3) & 8) | (mapper_regs[0] & 7));
	set_mirroring((mapper_regs[0] & 0x80) ? VERTICAL : HORIZONTAL);
}

// --[ mapper 79, 146: NINA-03/NINA-06 (Death Bots), Sachen (146 / Metal Fighter)
static void mapper79_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper79_map()
{
	mapper_map_prg(32, 0, (mapper_regs[0] >> 3) & 1);
	mapper_map_chr( 8, 0, (mapper_regs[0] & 7));
}

// --[ mapper 137, 138, 139, 141: Sachen 8259x (Super Pang, etc)
static void mapper8259_write(UINT16 address, UINT8 data)
{
	if (address < 0x4100) return;

	if ((address & 0x4101) == 0x4100) {
		mapper_regs[8] = data; // command
	} else {
		mapper_regs[mapper_regs[8] & 7] = data;
		mapper_map();
	}
}

static void mapper8259_map()
{
	mapper_map_prg(32, 0, mapper_regs[5] & 7);

	if (Cart.CHRRomSize != 0) {
		for (INT32 i = 0; i < 4; i++) {
			INT32 bank = 0;

			if (mapper_regs[7] & 1)
				bank = (mapper_regs[0] & 7) | ((mapper_regs[4] & 7) << 3);
			else
				bank = (mapper_regs[i] & 7) | ((mapper_regs[4] & 7) << 3);

			switch (mapper_regs[0x1f]) {
				case 137:                         // 137 8259D
					bank = mapper_regs[i] & 7;
					switch (i & 3) {
						case 1:
							bank |= (mapper_regs[4] & 1) << 4;
							break;
						case 2:
							bank |= (mapper_regs[4] & 2) << 3;
							break;
						case 3:
							bank |= ((mapper_regs[4] & 4) << 2) | ((mapper_regs[6] & 1) << 3);
							break;
					}
					mapper_map_chr( 1, i, bank);
					mapper_map_chr( 4, 1, -1);
					break;
				case 138:                         // 138 8259B
					mapper_map_chr( 2, i, bank);
					break;
				case 139:
					bank = (bank << 2) | (i & 3); // 139 8259C
					mapper_map_chr( 2, i, bank);
					break;
				case 141:
					bank = (bank << 1) | (i & 1); // 141 8259A
					mapper_map_chr( 2, i, bank);
					break;
			}
		}
	}

	if (mapper_regs[7] & 1) {
		set_mirroring(VERTICAL);
	} else {
		switch ((mapper_regs[7] >> 1) & 3) {
			case 0: set_mirroring(VERTICAL); break;
			case 1: set_mirroring(HORIZONTAL); break;
			case 2: set_mirroring(SACHEN); break;
			case 3: set_mirroring(SINGLE_LOW); break;
		}
	}
}

// --[ mapper 150: Sachen 74LS374XX (Tasac)
#define mapper150_cmd       (mapper_regs[0x1f - 0])
#define mapper150_prg       (mapper_regs[0x1f - 1])

static void mapper150_write(UINT16 address, UINT8 data)
{
	switch (address & 0xc101) {
		case 0x4100:
			mapper150_cmd = data & 0x7;
			break;
		case 0x4101:
			switch (mapper150_cmd) {
				case 2:
					mapper150_prg = data & 1;
					break;
				case 5:
					mapper150_prg = data & 7;
					break;
				default:
					mapper_regs[mapper150_cmd] = data;
					break;
			}
			mapper_map();
			break;
	}
}

static UINT8 mapper150_read(UINT16 address)
{
	if ((address & 0xc101) == 0x4000) {
		return (~mapper150_cmd) ^ 1; // '1' (or 0) is switch for Sachen / Sachen Hacker
	}

	return 0;
}

static void mapper150_map()
{
	mapper_map_prg(32, 0, mapper150_prg);

	mapper_map_chr( 8, 0, ((mapper_regs[2] & 0x1) << 3) | ((mapper_regs[4] & 0x1) << 2) | (mapper_regs[6] & 0x03));

	switch ((mapper_regs[7] >> 1) & 3) {
		case 0: set_mirroring(VERTICAL); break;
		case 1: set_mirroring(HORIZONTAL); break;
		case 2: set_mirroring(SACHEN); break;
		case 3: set_mirroring(SINGLE_LOW); break;
	}
}

// ---[ mapper 162 Zelda Triforce of the Gods
#define mapper162_reg(x)			(mapper_regs[0x00 + (x)])

static void mapper162_write(UINT16 address, UINT8 data)
{
	if (address >= 0x5000 && address <= 0x5fff) {
		mapper162_reg((address >> 8) & 0x03) = data;

		mapper_map();
	}
}

static void mapper162_map()
{
	switch (mapper162_reg(3) & 0x05) {
		case 0:
			mapper_map_prg(32, 0, (mapper162_reg(1) & 0x02) | ((mapper162_reg(2) & 0x0f) << 4) | (mapper162_reg(0) & 0x0c));
			break;
		case 1:
			mapper_map_prg(32, 0, ((mapper162_reg(2) & 0x0f) << 4) | (mapper162_reg(0) & 0x0c));
			break;
		case 4:
			mapper_map_prg(32, 0, ((mapper162_reg(1) & 0x02) >> 1) | ((mapper162_reg(2) & 0x0f) << 4) | (mapper162_reg(0) & 0x0e));
			break;
		case 5:
			mapper_map_prg(32, 0, ((mapper162_reg(2) & 0x0f) << 4) | (mapper162_reg(0) & 0x0f));
			break;
	}
}

// ---[ mapper 163 Final Fantasy VII (NJ063) + Advent Children ver. (&etc.)
#define mapper163_reg(x)			(mapper_regs[0x00 + (x)])
#define mapper163_chr(x)			(mapper_regs[0x08 + (x)])
#define mapper163_prg				(mapper_regs[0x1f - 0])
#define mapper163_prot_flip			(mapper_regs[0x1f - 1])
#define mapper163_raster_split		(mapper_regs[0x1f - 2])

static UINT8 mapper163_read(UINT16 address)
{
	switch (address & 0x7700) {
		case 0x5100:
			return mapper163_reg(3) | mapper163_reg(1) | mapper163_reg(0) | (mapper163_reg(2) ^ 0xff);
		case 0x5500:
			return (mapper163_prot_flip == 0) ? (mapper163_reg(3) | mapper163_reg(0)) : 0;
	}

	return 0x4;
}

static void mapper163_write(UINT16 address, UINT8 data)
{
	if ((address & 0xf000) == 0x5000) {
		if (address == 0x5101) {
			if (mapper163_reg(4) != 0 && data == 0) {
				mapper163_prot_flip ^= 1;
			}
			mapper163_reg(4) = data;
		} else if (address == 0x5100 && data == 0x6) {
			mapper163_prg = 0x3;
		} else {
			switch (address & 0x7300) {
				case 0x5000:
					mapper163_reg(0) = data;
					if (!(data & 0x80) && scanline < 128) {
						mapper163_chr(0) = 0;
						mapper163_chr(1) = 1;
					}
					mapper163_raster_split = data & 0x80;
					mapper163_prg = (mapper163_reg(0) & 0xf) | ((mapper163_reg(2) & 0xf) << 4);
					break;
				case 0x5100:
					mapper163_reg(1) = data;
					if (data == 0x6) {
						mapper163_prg = 3;
					}
					break;
				case 0x5200:
					mapper163_reg(2) = data;
					mapper163_prg = (mapper163_reg(0) & 0xf) | ((mapper163_reg(2) & 0xf) << 4);
					break;
				case 0x5300:
					mapper163_reg(3) = data;
					break;
			}
		}
		mapper_map();
	}
}

static void mapper163_map()
{
	mapper_map_prg(32, 0, mapper163_prg);
	mapper_map_chr(4, 0, mapper163_chr(0));
	mapper_map_chr(4, 1, mapper163_chr(1));
}

static void mapper163_cycle(UINT16 address)
{
	static INT32 last_127 = 0;
	static INT32 last_239 = 0;

	if (mapper163_raster_split && pixel > 257) {
		switch (scanline) {
			case 127:
				if (nCurrentFrame != last_127) {
					mapper_map_chr(4, 0, 1);
					mapper_map_chr(4, 1, 1);
					mapper163_chr(0) = 1;
					mapper163_chr(1) = 1;
				}
				last_127 = nCurrentFrame;
				break;
			case 239:
				if (nCurrentFrame != last_239) {
					mapper_map_chr(4, 0, 0);
					mapper_map_chr(4, 1, 0);
					mapper163_chr(0) = 0;
					mapper163_chr(1) = 0;
				}
				last_239 = nCurrentFrame;
				break;
		}
	}
}

#undef mapper163_reg
#undef mapper163_prg
#undef mapper163_chr
#undef mapper163_prot_flip
#undef mapper163_raster_split

// --[ mapper 38: (Crime Busters)
static void mapper38_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper38_map()
{
	mapper_map_prg(32, 0, mapper_regs[0] & 0x3);
	mapper_map_chr( 8, 0, (mapper_regs[0] >> 2) & 0x3);
}

// --[ mapper 66,140: (Bio Senshi Dan: Increaser tono Tatakai, Youkai Club, ..)
static void mapper140_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper140_map()
{
	mapper_map_prg(32, 0, mapper_regs[0] >> 4);
	mapper_map_chr( 8, 0, mapper_regs[0] & 0xf);
}

// --[ mapper 11, 144: ColorDreams, Death Race(144)
static void mapper11_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper144_write(UINT16 address, UINT8 data)
{
	if (address >= 0x8001) {
		mapper_regs[0] = data;
		mapper_map();
	}
}

static void mapper11_map()
{
	mapper_map_prg(32, 0, mapper_regs[0] & 0xf);
	mapper_map_chr( 8, 0, mapper_regs[0] >> 4);
}

// --[ mapper 152: (Arkanoid II, Pocket Zaurus: Juu Ouken no Nazo, ...)
static void mapper152_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper152_map()
{
	mapper_map_prg(16, 0, (mapper_regs[0] >> 4) & 7);
	mapper_map_prg(16, 1, -1);
	mapper_map_chr( 8, 0, mapper_regs[0] & 0xf);
	set_mirroring((mapper_regs[0] & 0x80) ? SINGLE_HIGH : SINGLE_LOW);
}

// --[ mapper 156: Open (Metal Force, Buzz & Waldog, Koko Adv., Janggun-ui Adeul)
#define mapper156_chr_lo(x)     (mapper_regs[0 + (x)])  // x = 0 - 7
#define mapper156_chr_hi(x)     (mapper_regs[8 + (x)])  // x = 0 - 7
#define mapper156_prg           (mapper_regs[0x1f - 0])
#define mapper156_mirror        (mapper_regs[0x1f - 1])

static void mapper156_write(UINT16 address, UINT8 data)
{
	switch (address) {
		case 0xc000:
		case 0xc001:
		case 0xc002:
		case 0xc003:
		case 0xc008:
		case 0xc009:
		case 0xc00a:
		case 0xc00b:
			mapper156_chr_lo((address & 0x3) | ((address & 0x8) >> 1)) = data;
			break;
		case 0xc004:
		case 0xc005:
		case 0xc006:
		case 0xc007:
		case 0xc00c:
		case 0xc00d:
		case 0xc00e:
		case 0xc00f:
			mapper156_chr_hi((address & 0x3) | ((address & 0x8) >> 1)) = data;
			break;
		case 0xc010:
			mapper156_prg = data;
			break;
		case 0xc014:
			mapper156_mirror = 0x10 | (data & 1);
			break;
	}

	mapper_map();
}

static void mapper156_map()
{
	mapper_map_prg(16, 0, mapper156_prg);
	mapper_map_prg(16, 1, -1);

	for (INT32 i = 0; i < 8; i++) {
		mapper_map_chr( 1, i, (mapper156_chr_hi(i) << 8) | mapper156_chr_lo(i));
	}

	switch (mapper156_mirror) {
		case 0: set_mirroring(SINGLE_LOW); break;
		case 0x10: set_mirroring(VERTICAL); break;
		case 0x11: set_mirroring(HORIZONTAL); break;
	}
}

// --[ mapper 180: Crazy Climber
static void mapper180_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper180_map()
{
	mapper_map_prg(16, 0, 0);
	mapper_map_prg(16, 1, mapper_regs[0] & 0x7);
	mapper_map_chr( 8, 0, 0);
}

// --[ mapper 184: Atlantis no Nazo, Kanshakudama Nage Kantarou no Toukaidou Gojuusan Tsugi, Wing of Madoola
static void mapper184_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper184_map()
{
	mapper_map_prg(32, 0, 0);
	mapper_map_chr( 4, 0, mapper_regs[0] & 0xf);
	mapper_map_chr( 4, 1, mapper_regs[0] >> 4);
}

// --[ mapper 185 CNROM+prot
static void mapper185_write(UINT16 address, UINT8 data)
{
	if (address & 0x8000) {
		mapper_regs[0] = data;
		mapper_map();
	}
}

static void mapper185_map()
{
	if ( (mapper_regs[0] & 3) && (mapper_regs[0] != 0x13) ) { // map rom
		mapper_map_chr_ramrom( 8, 0, 0, MEM_ROM);
	} else { // map ram
		mapper_map_chr_ramrom( 8, 0, 0, MEM_RAM);
	}
}

// --[ end mappers

static void mapper_reset()
{
	mapper_init(Cart.Mapper);
}

static void mapper_scan(INT32 nAction, INT32 *pnMin)
{
	SCAN_VAR(PRGMap);
	SCAN_VAR(CHRMap);
	SCAN_VAR(CHRType);
	SCAN_VAR(PRGExpMap);

	SCAN_VAR(mapper_regs);
	SCAN_VAR(mapper_regs16);
	SCAN_VAR(mapper_irq_exec);

	if (mapper_scan_cb) {
		mapper_scan_cb();
	}

	if (nAction & ACB_WRITE) {
		if (mapper_map) mapper_map();
	}
}

static INT32 mapper_init(INT32 mappernum)
{
	INT32 retval = -1;

	if (NESMode & MAPPER_NOCLEAR && RESETMode == RESET_BUTTON) {
		// do nothing! certain mappers need regs preserved on RESET_BUTTON
	} else {
		memset(mapper_regs, 0, sizeof(mapper_regs));
		memset(mapper_regs16, 0, sizeof(mapper_regs16));
	}

	mapper_write = NULL; // 8000 - ffff
	mapper_map = NULL;
	mapper_scanline = NULL; // scanline cb
	mapper_cycle = NULL; // cycle cb (1x per cycle)
	mapper_ppu_clock = NULL; // called after busaddress change (see mapper09) (only in visible & prerender!)
	mapper_ppu_clockall = NULL; // called every ppu clock
	mapper_scan_cb = NULL; // savestate cb (see vrc6)
	mapper_scan_cb_nvram = NULL; // savestate cb (nvram, mapper 406, 451)

	mapper_prg_read = mapper_prg_read_int; // 8000-ffff (read)

	PRGExpMap = 0;
	cart_exp_write = NULL; // 6000 - 7fff
	cart_exp_read = NULL;

	psg_area_write = NULL; // 4020 - 5fff
	psg_area_read = NULL;

	bprintf(0, _T("Mapper #%d init/reset!\n"), mappernum);

	switch (mappernum) {
		case 0x808: { // Famicom Disk System
			NESMode |= MAPPER_NOCLEAR; // preserve mapper regs on soft-reset
			mapper_map_prg(32, 0, 0);
			mapper_map_chr( 8, 0, 0);
			psg_area_write = mapperFDS_write; // 4000 - 5fff
			psg_area_read = mapperFDS_read;
			mapper_write = mapperFDS_prg_write; // 8000 - ffff
			mapper_cycle = mapperFDS_cycle;
			mapper_map   = mapperFDS_map;
			nes_ext_sound_cb = mapperFDS_mixer;
			mapper_scan_cb  = mapperFDS_scan;

			BurnLEDInit(1, LED_POSITION_BOTTOM_RIGHT, LED_SIZE_4x4, LED_COLOR_GREEN, 80);

			mapperFDS_reset();
			mapper_map();
			retval = 0;
			break;
		}

		case 400: { // 8 Bit X-Mas 2017 (RET-X7-GBL)
			cart_exp_write  = mapper400_write;  // 6000 - 7fff
			mapper_write    = mapper400_write;  // 8000 - ffff
			mapper_map      = mapper400_map;

			mapper_regs[0] = 0x80; // default

			mapper_map();
			retval = 0;
			break;
		}

		case 413: { // Super Russian Roulette (BATLAB-SRR-X PCB)
			psg_area_read   = mapper413_psg_read;  // 4000 - 5fff
			cart_exp_read   = mapper413_exp_read;  // 6000 - 7fff
			mapper_prg_read = mapper413_prg_read;  // 8000 - ffff
			mapper_write    = mapper413_prg_write; // 8000 - ffff
			mapper_scanline = mapper413_scanline;
			mapper_map      = mapper413_map;

			mapper_map();
			retval = 0;
			break;
		}

		case 0: { // no mapper
			mapper_map_prg(32, 0, 0);
			mapper_map_chr( 8, 0, 0);
			retval = 0;
			break;
		}

		case 155: // 155 is mmc1 w/wram always mapped @ 6000-7fff
		case 1: { // mmc1
			if (mappernum != 155) {
				cart_exp_write = mapper01_exp_write; // 6000 - 7fff
				cart_exp_read = mapper01_exp_read;
			}
			mapper_write = mapper01_write;
			mapper_map   = mapper01_map;
			mapper_regs[0] = 0xc;
			mapper_map();
			retval = 0;
			break;
		}

		case 2: { // UxROM
			mapper_write = mapper02_write;
			mapper_map   = mapper02_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 15: { // Contra 168-in-1 Multicart
			mapper_write = mapper15_write;
			mapper_map   = mapper15_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 190: { // Magic Kid Googoo
			mapper_write = mapper190_write;
			mapper_map   = mapper190_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 193: { // NTDEC TC-112 (War in the Gulf, Fighting Hero, ..)
			cart_exp_write = mapper193_write;
			mapper_map   = mapper193_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 41: { // Caltron 6-in-1
			cart_exp_write = mapper41_write;
			mapper_write = mapper41_write;
			mapper_map   = mapper41_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 53: { // SuperVision 16-in-1
			NESMode |= MAPPER_NOCLEAR; // preserve mapper regs on soft-reset
			cart_exp_read = mapper53_exp_read;
			cart_exp_write = mapper53_write;
			mapper_write = mapper53_write;
			mapper_map   = mapper53_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 104: { // Golden Five / Pegasus 5-in-1
			mapper_write = mapper104_write;
			mapper_map   = mapper104_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 389: { // Caltron 9-in-1
			mapper_write = mapper389_write;
			mapper_map   = mapper389_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 216: { // Magic Jewelry 2
			mapper_write = mapper216_write;
			mapper_map   = mapper216_map;
			psg_area_read = mapper216_read;
			mapper_map();
			retval = 0;
			break;
		}

		case 132: { // Qi Wang
			mapper_write = mapper132_write;
			mapper_map   = mapper132_map;
			psg_area_read = mapper132_read;
			psg_area_write = mapper132_write;
			mapper_map();
			retval = 0;
			break;
		}

		case 3: { // CNROM
			mapper_write = mapper03_write;
			mapper_map   = mapper03_map;
			if (Cart.Crc == 0xab29ab28) // dropzone
				mapper_cycle = mapper03_cycle;
			mapper_map_prg(32, 0, 0);
			mapper_map();
			retval = 0;
			break;
		}

		case 5: { // MMC5
			cart_exp_write = mapper5_exp_write; // 6000 - 7fff
			cart_exp_read = mapper5_exp_read;
			psg_area_write = mapper5_write; // 4000 - 5fff
			psg_area_read = mapper5_read;
			mapper_write = mapper5_prg_write; // 8000 - ffff
			mapper_map   = mapper5_map;
			mapper_scan_cb  = mapper5_scan;
			mapper_ppu_clockall = mapper5_ppu_clk;
			nes_ext_sound_cb = mapper5_mixer;

			read_nt = &mapper5_ntread;
			write_nt = &mapper5_ntwrite;

			mapper5_reset();

			mapper_map();
			retval = 0;
			break;
		}

		case 7: { // AxROM: Battle Toads, Marble Madness, RC Pro-Am
			mapper_write = mapper07_write;
			mapper_map   = mapper07_map;
		    mapper_map_chr( 8, 0, 0);
			mapper_map();
			retval = 0;
			break;
		}

		case 9: { // mmc2: punch out!
			mapper_write = mapper09_write;
			mapper_map   = mapper09_map;
			mapper_ppu_clock = mapper09_ppu_clk;
			mapper_map_prg( 8, 1, -3);
			mapper_map_prg( 8, 2, -2);
			mapper_map_prg( 8, 3, -1);
			mapper_map();
			retval = 0;
			break;
		}

		case 10: { // mmc4: fire emblem (mmc2 + sram + different prg mapping)
			mapper_write = mapper09_write;
			mapper_map   = mapper10_map;
			mapper_ppu_clock = mapper10_ppu_clk;
			mapper_map();
			retval = 0;
			break;
		}

		case 99: {
			mapper_map = mapper99_map;
			mapper_map();
			if (Cart.Crc == 0x84ad422e) {
				bprintf(0, _T("Patching: VS. Raid on Bungling Bay...\n"));
				// VS. Raid on Bungling Bay patch
				bprintf(0, _T("%02x %02x %02x\n"), Cart.PRGRom[0x16df], Cart.PRGRom[0x16e0], Cart.PRGRom[0x16e1]);
				Cart.PRGRom[0x16df] = 0xea; // game stuck in loop @ 96df - possibly waiting for second cpu?
				Cart.PRGRom[0x16e0] = 0xea;
				Cart.PRGRom[0x16e1] = 0xea;

				nesapuSetMode4017(0x00); // enable frame-irq
			}

			retval = 0;
			break;
		}

		case 13: {
			mapper_write = mapper13_write;
			mapper_map = mapper13_map;
			mapper_map_prg(32, 0, 0);
			mapper_map();
			retval = 0;
			break;
		}

		case 133: {
			psg_area_write = mapper133_write;
			mapper_map = mapper133_map;
			mapper_map_prg(32, 0, 0);
			mapper_map();
			retval = 0;
			break;
		}

		case 159: // 128byte eeprom
		case 16: // 256byte eeprom
			{ // Bandai fcg
			mapper_write = mapper16_write;
			cart_exp_write = mapper16_write;
			mapper_map   = mapper16_map;
			mapper_cycle = mapper16_cycle;
			mapper_map();
			retval = 0;
			break;
		}

		case 153: { // Bandai fcg (Famicom Jump II)
			mapper_write = mapper16_write;
			mapper_map   = mapper153_map;
			mapper_cycle = mapper16_cycle;
			mapper_map();
			retval = 0;
			break;
		}

		case 18: { // Jaleco (Magical John, Pizza Pop, etc)
			mapper_write = mapper18_write;
			mapper_map   = mapper18_map;
			mapper_cycle = mapper18_cycle;
			mapper_map();
			retval = 0;
			break;
		}

		case 210:
		case 19: { // Namco n106/163
			psg_area_read  = mapper19_read;
			psg_area_write = mapper19_write; // 4020 - 5fff
			mapper_write = mapper19_write; // 8000 - ffff
			mapper_map   = mapper19_map;
			mapper_cycle = mapper19_cycle;
			nes_ext_sound_cb = mapper19_mixer;
			mapper_scan_cb = mapper19_scan;
			mapper19_mapper210 = (mappernum == 210);

			//waygan 2  0x8aa76b0b   wagyan 3 0x7aff2388  splatterh 0xb7eeb48b
			if (Cart.Crc == 0x8aa76b0b || Cart.Crc == 0x7aff2388 || Cart.Crc == 0xb7eeb48b) {
				mapper19_mapper210 = 2; // namco 340 (210 submapper 2)
				bprintf(0, _T("Namco 210 submapper 2 (Namco 340)\n"));
			}

			mapper19_reset();
			mapper_map();
			retval = 0;
			break;
		}

		case 21: { // vrc2
			mapper_write = mapper21_write;
			mapper_map   = vrc2vrc4_map;
			mapper_cycle = vrc2vrc4_cycle;
			mapper_map();
			retval = 0;
			break;
		}

		case 22: { // vrc2
			mapper_write = mapper22_write;
			mapper_map   = vrc2vrc4_map;
			vrc2and4_ines22 = 1;
			mapper_map();
			retval = 0;
			break;
		}

		case 23: { // vrc4
			mapper_write = mapper23_write;
			mapper_map   = vrc2vrc4_map;
			mapper_cycle = vrc2vrc4_cycle;
			mapper_map();
			retval = 0;
			break;
		}

		case 25: { // vrc4
			mapper_write = mapper22_write;
			mapper_map   = vrc2vrc4_map;
			mapper_cycle = vrc2vrc4_cycle;
			mapper_map();
			retval = 0;
			break;
		}

		case 24: { // vrc6
			mapper_write = vrc6_write;
			mapper_map   = vrc6_map;
			mapper_cycle = vrc6_cycle;
			vrc6_reset();
			mapper_map();
			nes_ext_sound_cb = vrc6_mixer; // vrc6 sound
			mapper_scan_cb = vrc6_scan;
			retval = 0;
			break;
		}

		case 26: { // vrc6
			mapper_write = vrc6_write;
			mapper_map   = vrc6_map;
			mapper_cycle = vrc6_cycle;
			mapper26 = 1;
			vrc6_reset();
			mapper_map();
			nes_ext_sound_cb = vrc6_mixer; // vrc6 sound
			mapper_scan_cb = vrc6_scan;
			retval = 0;
			break;
		}

		case 227: { // xxx-in-1, Waixing Bio Hazard
			mapper_write = mapper227_write;
			mapper_map   = mapper227_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 172: { // 1991 Du Ma Racing
			psg_area_write = mapper172_write;	// 4020 - 5fff
			psg_area_read = mapper172_read;		// 4020 - 5fff
			mapper_write = mapper172_write;
			mapper_map   = mapper172_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 228: { // Action 52, Cheetah Men II
			mapper_write = mapper228_write;
			psg_area_write = mapper228_psg_write;
			psg_area_read = mapper228_psg_read;
			mapper_map   = mapper228_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 91: { // Older JY Company/Hummer
			cart_exp_write = mapper91_write; // 6000 - 7fff
			mapper_map = mapper91_map;
			mapper_scanline = mapper91_scanline;
			mapper_map();
			retval = 0;
			break;
		}

		case 17: { // Front Far East FFE SMC (Type 17)
			psg_area_write = mapper17_write; // 4020 - 5fff
			mapper_map = mapper17_map;
			mapper_cycle = mapper17_cycle;
			mapper_regs[0] = ~3;
			mapper_regs[1] = ~2;
			mapper_regs[2] = ~1;
			mapper_regs[3] = ~0;
			mapper_map();
			retval = 0;
			break;
		}

		case 211: mapper90_211 = 1;
		case 209: mapper90_209 = 1;
		case 90: { // JY Company
			mapper_write = mapper90_write;       // 8000 - ffff
			psg_area_write = mapper90_psg_write; // 4020 - 5fff
			psg_area_read = mapper90_psg_read;
			cart_exp_write = mapper90_exp_write; // 6000 - 7fff
			cart_exp_read = mapper90_exp_read;

			mapper_scanline = mapper90_scanline;
			mapper_cycle = mapper90_cycle;
			mapper_ppu_clockall = mapper90_ppu_clock; // irq
			if (mapper90_209) {
				read_nt = &mapper90_ntread;
			}

			mapper_map   = mapper90_map;

			// mapper defaults
			mapper90_mul0 = 0xff;
			mapper90_mul1 = 0xff;
			mapper90_accu = 0xff;
			mapper90_testreg = 0xff;
			mapper90_chrlatch(0) = 0;
			mapper90_chrlatch(1) = 4;

			mapper_map();
			retval = 0;
			break;
		}

		case 28: { // Action53 / Home-brew multicart
			mapper_write   = mapper28_write;
			psg_area_write = mapper28_write;
			mapper_map     = mapper28_map;
			//mapper_map()
			NESMode |= MAPPER_NOCLEAR; // preserve mapper regs on soft-reset
			if (RESETMode == RESET_POWER)
				mapper_map_prg(16, 1, -1);
			retval = 0;
			break;
		}

		case 30: { // UNROM-512
			mapper_write = mapper30_write;
			mapper_map   = mapper30_map;
			switch (rom[6] & (1|8)) {
				case 0: set_mirroring(HORIZONTAL); break;
				case 1: set_mirroring(VERTICAL); break;
				case 8: set_mirroring(SINGLE_LOW); mapper30_mirroring_en = 1; break;
				case 9: set_mirroring(FOUR_SCREEN); break;
			}
			mapper_map();
			retval = 0;
			break;
		}

		case 33: { // Taito TC0190
			mapper_write = mapper33_write;
			mapper_map   = mapper33_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 36: { // TXC (Policeman, Strike Wolf)
			psg_area_read  = mapper36_read;
			psg_area_write = mapper36_write;
			mapper_write   = mapper36_write;
			mapper_map     = mapper36_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 42: { // Ai Senshi Nicol FDS Conversion
			mapper_write = mapper42_write;
			mapper_map   = mapper42_map;
			mapper_cycle = mapper42_cycle;
			cart_exp_read = mapper42_exp_read;
			mapper_map();
			retval = 0;
			break;
		}

		case 108: { // Meikyuu Jiin Dababa FDS conversion
			mapper_write = mapper108_write;
			psg_area_write = mapper108_write; // just to silent debug msgs
			mapper_map   = mapper108_map;
			cart_exp_read = mapper108_exp_read;
			mapper_map();
			retval = 0;
			break;
		}

		case 111: { // Cheapocabra (GTROM)
			cart_exp_write = mapper111_write;
			psg_area_write = mapper111_write;
			mapper_map   = mapper111_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 120: { // Tobidase Daisakusen FDS Conversion - garbage! (game crashes after lv1)
			psg_area_write = mapper120_write;
			mapper_map   = mapper120_map;
			cart_exp_read = mapper120_exp_read;
			mapper_map();
			retval = 0;
			break;
		}

		case 48: { // Taito TC0690
			mapper_write = mapper33_write;
			mapper_map   = mapper33_map;
			mapper_scanline = mapper33_scanline;
			mapper48 = 1;
			mapper48_flintstones = (Cart.Crc == 0x12f38740);
			bprintf(0, _T("mapper48 - flintstones? %x\n"), mapper48_flintstones);
			mapper_map();
			retval = 0;
			break;
		}

		case 185: { // CNROM + prot
			mapper_write = mapper185_write;
			mapper_map   = mapper185_map;
			memset(Cart.CHRRam, 0xff, Cart.CHRRamSize);
			mapper_map_prg(32, 0, 0);
			mapper_map();
			retval = 0;
			break;
		}

		case 32: { // Irem G101
			mapper_write = mapper32_write;
			mapper_map = mapper32_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 34: { // BNROM/NINA-001
			mapper_write = mapper34_write;
			cart_exp_write = mapper34_write;
			mapper_map = mapper34_map;
			mapper_regs[1] = 0; // defaults to retain compatibility with BNROM
			mapper_regs[2] = 1;
			mapper_map();
			retval = 0;
			break;
		}

		case 65: { // Irem h3001
			mapper_write = mapper65_write;
			mapper_map = mapper65_map;
			mapper_cycle = mapper65_cycle;
			mapper_regs[2] = -2;
			mapper_map_prg( 8, 3, -1);
			mapper_map();
			retval = 0;
			break;
		}

		case 67: { // Sunsoft-3
			mapper_write = mapper67_write;
			mapper_map = mapper67_map;
			mapper_cycle = mapper67_cycle;
			mapper_map();
			retval = 0;
			break;
		}

		case 68: { // Sunsoft-4
			mapper_write = mapper68_write;
			mapper_map = mapper68_map;
			mapper_cycle = mapper68_cycle;
			cart_exp_read = mapper68_exp_read;
			cart_exp_write = mapper68_exp_write;
			mapper_map();
			retval = 0;
			break;
		}

		case 64: { // Tengen RAMBO-1
			mapper_write = mapper64_write;
			mapper_map   = mapper64_map;
			mapper_scanline = mapper64_scanline;
			mapper_cycle = mapper64_cycle;
			mapper_map_prg( 8, 3, -1);
		    mapper_map();
			retval = 0;
			break;
		}

		case 69: { // sunsoft
			mapper_write = mapper69_write;
			mapper_map   = mapper69_map;
			mapper_cycle = mapper69_cycle;
			cart_exp_read = mapper69_exp_read;
			cart_exp_write = mapper69_exp_write;
			mapper_map();
			AY8910Reset(0);
			retval = 0;
			break;
		}

		case 70: {
			mapper_write = mapper70_write; // 8000 - ffff
			mapper_map   = mapper70_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 71: { // camerica (codemasters) misc
			mapper_write = mapper71_write; // 8000 - ffff
			mapper_map   = mapper71_map;
		    mapper_map();
			if (Cart.Crc == 0xa64e926a && Cart.PRGRom[0xc4c1] == 0xa5) {
				bprintf(0, _T("Applied Hot-Patch: Quattro Sports / BMX Simulator start fix..\n"));
				Cart.PRGRom[0xc4c1] = 0xa9; // make ctrlr-debounce read happy
			}
			retval = 0;
			break;
		}

		case 73: { // Konami vrc3 (Salamander)
			mapper_write = mapper73_write;
			mapper_map   = mapper73_map;
			mapper_cycle = mapper73_cycle;
		    mapper_map();
			retval = 0;
			break;
		}

		case 75: { // Konami vrc1
			mapper_write = mapper75_write; // 8000 - ffff
			mapper_map   = mapper75_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 85: { // Konami vrc7 (Lagrange Point, Tiny Toons Adventures 2 (Japan))
			mapper_write = vrc7_write;
			mapper_map   = vrc7_map;
			mapper_cycle = vrc7_cycle;
			BurnYM2413VRC7Reset();
		    mapper_map();
			retval = 0;
			break;
		}

		case 232: { // camerica (codemasters) quattro
			mapper_write = mapper232_write;
			mapper_map   = mapper232_map;
			mapper_map();
			if (Cart.Crc == 0x9adc4130 && Cart.PRGRom[0x2c4c1] == 0xa5) {
				bprintf(0, _T("Applied Hot-Patch: Quattro Sports / BMX Simulator start fix..\n"));
				Cart.PRGRom[0x2c4c1] = 0xa9; // make ctrlr-debounce read happy
			}
			retval = 0;
			break;
		}

		case 77: { // Irem? Napoleon Senki
			mapper_write = mapper77_write; // 8000 - ffff
			mapper_map   = mapper77_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 78: { // Jaleco JF16
			mapper_write = mapper78_write; // 8000 - ffff
			mapper_map   = mapper78_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 80: { // Taito x1005
			cart_exp_write = mapper207_write;
			cart_exp_read  = mapper207_read;
			mapper_map     = mapper207_map;
			mapper_scan_cb = mapper207_scan;
			memset(mapper80_ram, 0, sizeof(mapper80_ram));
			mapper207_adv_mirror = 0;
			mapper_map();
			retval = 0;
			break;
		}

		case 82: { // Taito x1017
			cart_exp_write = mapper82_write;
			cart_exp_read  = mapper82_read;
			mapper_map     = mapper82_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 81: { // NTDEC Super Gun (Caltron)
			mapper_write = mapper81_write;
			mapper_map   = mapper81_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 207: { // Taito x1005 w/advanced mirroring
			cart_exp_write = mapper207_write;
			cart_exp_read  = mapper207_read;
			mapper_map     = mapper207_map;
			mapper_scan_cb = mapper207_scan;
			memset(mapper80_ram, 0, sizeof(mapper80_ram));
			mapper207_adv_mirror = 1;
			mapper_map();
			retval = 0;
			break;
		}

		case 87: {
			cart_exp_write = mapper87_write; // 6000 - 7fff!
			mapper_map   = mapper87_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 88: {
			mapper_write = mapper88_write;
			mapper_map   = mapper88_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 154: {
			mapper_write = mapper88_write;
			mapper_map   = mapper88_map;
			mapper154 = 1;
		    mapper_map();
			retval = 0;
			break;
		}

		case 101: {
			cart_exp_write = mapper101_write; // 6000 - 7fff!
			mapper_map   = mapper101_map;
			mapper_regs[0] = 0xff;
		    mapper_map();
			retval = 0;
			break;
		}

		case 107: {
			mapper_write = mapper107_write;
			mapper_map   = mapper107_map;
			mapper_regs[0] = 0xff;
		    mapper_map();
			retval = 0;
			break;
		}

		case 38: {
			cart_exp_write = mapper38_write; // 8000 - ffff
			mapper_map   = mapper38_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 66: {
			mapper_write = mapper140_write; // 8000 - ffff
			mapper_map   = mapper140_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 11: {
			mapper_write = mapper11_write; // 8000 - ffff
			mapper_map   = mapper11_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 144: { // similar to 11
			mapper_write = mapper144_write; // 8000 - ffff
			mapper_map   = mapper11_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 112: {
			mapper_write = mapper112_write;
			psg_area_write = mapper112_write;
			mapper_map   = mapper112_map;
			mapper_map_prg(8, 2, -2);
			mapper_map_prg(8, 3, -1);
		    mapper_map();
			retval = 0;
			break;
		}

		case 146: // Sachen Metal Fighter
		case 79: { // NINA-03 / NINA-06
			psg_area_write = mapper79_write; // 4020 - 5fff
			mapper_map     = mapper79_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 113: { // NINA-03 / NINA-06 extended / Mojon Twins Multicart
			psg_area_write = mapper113_write; // 4020 - 5fff
			mapper_map     = mapper113_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 137:
		case 138:
		case 139:
		case 141: { // Sachen 8259A/B/C/D
			psg_area_write = mapper8259_write; // 4020 - 5fff -and-
			cart_exp_write = mapper8259_write; // 6000 - 7fff
			mapper_map     = mapper8259_map;
			mapper_regs[0x1f] = mappernum;
			mapper_map_chr( 8, 0, 0);
		    mapper_map();
			retval = 0;
			break;
		}

		case 150: { // Sachen LS
			psg_area_write = mapper150_write; // 4020 - 5fff -and-
			psg_area_read  = mapper150_read;
			cart_exp_write = mapper150_write; // 6000 - 7fff
			cart_exp_read  = mapper150_read;
			mapper_map     = mapper150_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 162: { // (Waixing) Zelda - Triforce of the Gods
			psg_area_write = mapper162_write; // 4020 - 5fff
			mapper_map     = mapper162_map;
			mapper162_reg(0) = 0x03;
			mapper162_reg(3) = 0x07;
			mapper_map_chr( 8, 0, 0);
		    mapper_map();
			retval = 0;
			break;
		}

		case 163: { // Final Fantasy VII (NJ063)
			psg_area_write = mapper163_write; // 4020 - 5fff
			psg_area_read  = mapper163_read;
			mapper_map     = mapper163_map;
			mapper_ppu_clock = mapper163_cycle;
		    mapper_map();
			retval = 0;
			break;
		}

		case 140: {
			cart_exp_write = mapper140_write; // 6000 - 7fff!
			mapper_map     = mapper140_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 180: { // crazy climber
			mapper_write = mapper180_write;
			mapper_map   = mapper180_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 184: {
			cart_exp_write = mapper184_write; // 6000 - 7fff!
			mapper_map     = mapper184_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 152: {
			mapper_write = mapper152_write;
			mapper_map   = mapper152_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 156: {
			mapper_write = mapper156_write;
			mapper_map   = mapper156_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 89: {
			mapper_write = mapper89_write;
			mapper_map   = mapper89_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 93: { // Sunsoft-1/2: Fantasy Zone, Shanghai
			mapper_write = mapper93_write;
			mapper_map   = mapper93_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 94: { // Senjou no Ookami (Commando)
			mapper_write = mapper94_write;
			mapper_map   = mapper94_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 97: { // Kaiketsu Yanchamaru (Kid Niki)
			mapper_write = mapper97_write;
			mapper_map   = mapper97_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 206: // mmc3 w/no irq (Tengen mimic-1, Namcot 118)
		case 4: { // mmc3
			mapper_write = mapper04_write;
			mapper_map   = mapper04_map;
			mapper_scanline = mapper04_scanline;
			mapper4_mirror = Cart.Mirroring; // wagyan land doesn't set the mapper bit!

			{
				// VS. Mapper 4 / 206 Protection
				mapper04_vs_prottype = 0;
				switch (Cart.Crc) {
					default: break;
					case 0x9f2251e4: // super xevious
						mapper04_vs_prottype++;         // fallthrough!
					case 0xd81d33da: // tko boxing
						mapper04_vs_prottype++;         // fallthrough!
					case 0xac95e2c9: // rbi baseball
						psg_area_read = mapper04_vs_rbi_tko_prot;
						break;
				}
			}

			// default mmc3 regs:
			// chr
			mapper_regs[0] = 0;
			mapper_regs[1] = 2;
			mapper_regs[2] = 4;
			mapper_regs[3] = 5;
			mapper_regs[4] = 6;
			mapper_regs[5] = 7;
			// prg
			mapper_regs[6] = 0;
			mapper_regs[7] = 1;

			mapper_map_prg( 8, 3, -1);
		    mapper_map();
			retval = 0;
			break;
		}

		case 406: { // Haradius Zero (mmc3 + 2xMXIC flash)
			flashrom_chiptype = MXIC;

			mapper_prg_read = flashrom_read;
			mapper_write    = mapper406_write;
			mapper_scan_cb_nvram = mapper406_scan;

			mapper_map      = mapper04_map;
			mapper_scanline = mapper04_scanline;
			mapper4_mirror  = Cart.Mirroring;

			mapper_map_prg( 8, 3, -1);
			mapper_map();
			retval = 0;
			break;
		}

		case 451: { // Haratyler pseudo-mmc3 + AMIC flash
			flashrom_chiptype = AMIC;

			mapper_prg_read = flashrom_read;
			mapper_write    = mapper451_write;
			mapper_scan_cb_nvram = mapper451_scan;

			mapper_map      = mapper451_map;
			mapper_scanline = mapper04_scanline;
			mapper4_mirror  = Cart.Mirroring;

			mapper_map();
			retval = 0;
			break;
		}

		case 12: { // mmc3 variant (dbz 5, mk4, umk3)
			psg_area_write = mapper12_write;
			psg_area_read = mapper12_read;
			mapper_write = mapper04_write;
			mapper_map   = mapper04_map;
			mapper_scanline = mapper04_scanline;
			mapper4_mirror = Cart.Mirroring; // wagyan land doesn't set the mapper bit!
			mapper_map_prg( 8, 3, -1);
		    mapper_map();
			retval = 0;
			break;
		}

		case 76: { // mmc3 variant (digital devil m.)
			mapper_write = mapper04_write;
			mapper_map   = mapper76_map;
			mapper_scanline = mapper04_scanline;
			mapper4_mirror = Cart.Mirroring;
			mapper_map_prg( 8, 3, -1);
		    mapper_map();
			retval = 0;
			break;
		}

		case 268: { // mmc3 variant (COOLBOY / MINDKIDS)
			mapper_write = mapper268_write;

			if (Cart.SubMapper == 1) psg_area_write = mapper268_write; // 5000-5fff
			else cart_exp_write = mapper268_write;                     // 6000-6fff

			mapper_map   = mapper268_map;
			mapper_scanline = mapper04_scanline;
			mapper4_mirror = Cart.Mirroring;

			mapper_map();
			retval = 0;
			break;
		}

		case 95: { // mmc3 variant (dragon buster)
			mapper_write = mapper95_write;
			mapper_map   = mapper95_map;
			mapper_scanline = mapper04_scanline;
			mapper_map_prg( 8, 3, -1);
		    mapper_map();
			retval = 0;
			break;
		}

		case 115: { // mmc3 variant (thunderbolt 2)
			mapper_write = mapper04_write;

			psg_area_write = mapper115_write;
			psg_area_read = mapper115_read;
			cart_exp_write = mapper115_write;
			cart_exp_read = mapper115_read;

			mapper_map   = mapper115_map;
			mapper_scanline = mapper04_scanline;
			mapper4_mirror = Cart.Mirroring; // wagyan land doesn't set the mapper bit!
		    mapper_map();
			retval = 0;
			break;
		}

		case 116: { // Somari (AV Girl Fighter)
			mapper_write = mapper116_write;
			psg_area_write = mapper116_write;
			cart_exp_write = mapper116_write;
			mapper_map   = mapper116_map;
			mapper_scanline = mapper116_mmc3_scanline;
			mapper116_defaults();
		    mapper_map();
			retval = 0;
			break;
		}

		case 14: { // Gouder SL-1632 (Samurai Spirits)
			mapper_write = mapper116_write;
			psg_area_write = mapper116_write;
			cart_exp_write = mapper116_write;
			mapper_map   = mapper14_map; // difference from 116
			mapper_scanline = mapper116_mmc3_scanline;
			mapper116_defaults();
		    mapper_map();
			retval = 0;
			break;
		}

		case 118: { // mmc3-derivative TKSROM/TLSROM (Alien Syndrome, Armadillo)
			mapper_write = mapper04_write;
			mapper_map   = mapper118_map;
			mapper_scanline = mapper04_scanline;
			mapper_map_prg( 8, 3, -1);
		    mapper_map();
			retval = 0;
			break;
		}

		case 119: { // mmc3-derivative w/char ram+rom & ability to map both into diff. banks! (pin-bot, high speed)
			mapper_write = mapper04_write;
			mapper_map   = mapper119_map;
			mapper_scanline = mapper04_scanline;
			mapper_set_chrtype(MEM_RAM);
			mapper_map_prg( 8, 3, -1);
		    mapper_map();
			retval = 0;
			break;
		}

		case 165: { // mmc3-derivative w/mmc4-style char ram(bank0)+rom(others)
			mapper_write = mapper04_write;
			mapper_map   = mapper165_map;
			mapper_ppu_clock = mapper165_ppu_clock;
			mapper_scanline = mapper04_scanline;
			mapper_set_chrtype(MEM_RAM);
			mapper_map_prg( 8, 3, -1);
		    mapper_map();
			retval = 0;
			break;
		}

		case 192: { // mmc3-derivative w/char ram+rom, ram mapped to chr banks 8, 9, a, b
			mapper_write = mapper04_write;
			mapper_map   = mapper192_map;
			mapper_scanline = mapper04_scanline;
			mapper_set_chrtype(MEM_RAM);
			mapper_map_prg( 8, 3, -1);
		    mapper_map();
			retval = 0;
			break;
		}

		case 195: { // mmc3-derivative w/char ram+rom, ram mapped to chr banks 0-3, cpu accessable via 5000-5fff
			mapper_write = mapper04_write;
			mapper_map   = mapper195_map;
			psg_area_write  = mapper195_write;	// 5000 - 5fff r/w chr ram
			psg_area_read   = mapper195_read;	// 5000 - 5fff
			mapper_scanline = mapper04_scanline;
			mapper_set_chrtype(MEM_RAM);
			mapper_map_prg( 8, 3, -1);
		    mapper_map();
			retval = 0;
			break;
		}

		case 189: { // mmc3-derivative
			psg_area_write = mapper189_write; // 4020 - 5fff
			cart_exp_write = mapper189_write; // 6000 - 7fff
			mapper_write = mapper04_write;    // 8000 - ffff
			mapper_map   = mapper189_map;
			mapper_scanline = mapper04_scanline;
			mapper4_mirror = Cart.Mirroring; // wagyan land doesn't set the mapper bit!
		    mapper_map();
			retval = 0;
			break;
		}

		case 258: { // mmc3-derivative (blood of jurassic)
			psg_area_write = mapper258_write;	// 4020 - 5fff
			psg_area_read = mapper258_read;		// 4020 - 5fff
			mapper_write = mapper04_write;		// 8000 - ffff
			mapper_map   = mapper258_map;
			mapper_scanline = mapper04_scanline;
			mapper4_mirror = Cart.Mirroring; // wagyan land doesn't set the mapper bit!

			// default mmc3 regs:
			// chr
			mapper_regs[0] = 0;
			mapper_regs[1] = 2;
			mapper_regs[2] = 4;
			mapper_regs[3] = 5;
			mapper_regs[4] = 6;
			mapper_regs[5] = 7;
			// prg
			mapper_regs[6] = 0;
			mapper_regs[7] = 1;

			mapper_map_prg( 8, 3, -1);
		    mapper_map();
			retval = 0;
			break;
		}

		case 262: { // mmc3-derivative (Street Heroes)
			psg_area_write = mapper262_write;
			psg_area_read = mapper262_read;
			mapper_write = mapper04_write;
			mapper_map   = mapper262_map;
			mapper_scanline = mapper04_scanline;
			mapper4_mirror = Cart.Mirroring; // wagyan land doesn't set the mapper bit!
			mapper_map_prg( 8, 3, -1);
		    mapper_map();
			retval = 0;
			break;
		}
	}
	return retval;
}

static void mapper_irq(INT32 delay_cyc)
{
	if (delay_cyc == 0) { // irq now
		M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
	} else { // irq later (after 'cyc' m2 cycles)
		mapper_irq_exec = delay_cyc;
	}
}

static void mapper_run()
{
	if (mapper_irq_exec) {
		mapper_irq_exec--;
		if (mapper_irq_exec == 0) {
			M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
		}
	}

	if (mapper_cycle) mapper_cycle();
}

// --------[ end Mappers

static UINT32 nes_palette[2][0x40] = {
{ // dink-fami
	0x5c5c5c, 0x00237e, 0x100e9e, 0x33009b, 0x520074, 0x630037, 0x610500, 0x4c1700, 0x2b2c00, 0x093e00, 0x004700, 0x004505, 0x003744, 0x000000, 0x000000, 0x000000,
	0xa7a7a7, 0x1157d6, 0x3b38ff, 0x6d21fe, 0x9916c8, 0xb11973, 0xae2a1a, 0x904500, 0x626400, 0x307d00, 0x0a8a00, 0x00862a, 0x007385, 0x000000, 0x000000, 0x000000,
	0xfeffff, 0x5ba9ff, 0x8a88ff, 0xc16eff, 0xef61ff, 0xff65c7, 0xff7866, 0xe6961b, 0xb4b700, 0x7fd200, 0x53e027, 0x3cdc79, 0x3fc7da, 0x454545, 0x000000, 0x000000,
	0xfeffff, 0xbadbff, 0xcecdff, 0xe5c2ff, 0xf8bcff, 0xffbee7, 0xffc6be, 0xf4d39c, 0xe0e18a, 0xc9ec8c, 0xb6f2a2, 0xacf0c6, 0xade8ef, 0xb0b0b0, 0x000000, 0x000000,
},
{ // rgb
	0x7c7c7c, 0x0000fc, 0x0000bc, 0x4428bc, 0x940084, 0xa80020, 0xa81000, 0x881400,	0x503000, 0x007800, 0x006800, 0x005800, 0x004058, 0x000000, 0x000000, 0x000000,
	0xbcbcbc, 0x0078f8, 0x0058f8, 0x6844fc, 0xd800cc, 0xe40058, 0xf83800, 0xe45c10,	0xac7c00, 0x00b800, 0x00a800, 0x00a844, 0x008888, 0x000000, 0x000000, 0x000000,
	0xf8f8f8, 0x3cbcfc, 0x6888fc, 0x9878f8, 0xf878f8, 0xf85898, 0xf87858, 0xfca044,	0xf8b800, 0xb8f818, 0x58d854, 0x58f898, 0x00e8d8, 0x787878, 0x000000, 0x000000,
	0xfcfcfc, 0xa4e4fc, 0xb8b8f8, 0xd8b8f8, 0xf8b8f8, 0xf8a4c0, 0xf0d0b0, 0xfce0a8,	0xf8d878, 0xd8f878, 0xb8f8b8, 0xb8f8d8, 0x00fcfc, 0xf8d8f8, 0x000000, 0x000000
} };

static UINT32 vs_palette[5][0x40] = {
{ // 2C03, 2C05
	0x6d6d6d, 0x002492, 0x0000db, 0x6d49db, 0x92006d, 0xb6006d, 0xb62400, 0x924900, 0x6d4900, 0x244900, 0x006d24, 0x009200, 0x004949, 0x000000, 0x000000, 0x000000,
	0xb6b6b6, 0x006ddb, 0x0049ff, 0x9200ff, 0xb600ff, 0xff0092, 0xff0000, 0xdb6d00, 0x926d00, 0x249200, 0x009200, 0x00b66d, 0x009292, 0x000000, 0x000000, 0x000000,
	0xffffff, 0x6db6ff, 0x9292ff, 0xdb6dff, 0xff00ff, 0xff6dff, 0xff9200, 0xffb600, 0xdbdb00, 0x6ddb00, 0x00ff00, 0x49ffdb, 0x00ffff, 0x000000, 0x000000, 0x000000,
	0xffffff, 0xb6dbff, 0xdbb6ff, 0xffb6ff, 0xff92ff, 0xffb6b6, 0xffdb92, 0xffff49, 0xffff6d, 0xb6ff49, 0x92ff6d, 0x49ffdb, 0x92dbff, 0x000000, 0x000000, 0x000000
},
{ // 2C04A
	0xffb6b6, 0xdb6dff, 0xff0000, 0x9292ff, 0x009292, 0x244900, 0x494949, 0xff0092, 0xffffff, 0x6d6d6d, 0xffb600, 0xb6006d, 0x92006d, 0xdbdb00, 0x6d4900, 0xffffff,
	0x6db6ff, 0xdbb66d, 0x6d2400, 0x6ddb00, 0x92dbff, 0xdbb6ff, 0xffdb92, 0x0049ff, 0xffdb00, 0x49ffdb, 0x000000, 0x490000, 0xdbdbdb, 0x929292, 0xff00ff, 0x002492,
	0x00006d, 0xb6dbff, 0xffb6ff, 0x00ff00, 0x00ffff, 0x004949, 0x00b66d, 0xb600ff, 0x000000, 0x924900, 0xff92ff, 0xb62400, 0x9200ff, 0x0000db, 0xff9200, 0x000000,
	0x000000, 0x249200, 0xb6b6b6, 0x006d24, 0xb6ff49, 0x6d49db, 0xffff00, 0xdb6d00, 0x004900, 0x006ddb, 0x009200, 0x242424, 0xffff6d, 0xff6dff, 0x926d00, 0x92ff6d
},
{ // 2C04B
	0x000000, 0xffb600, 0x926d00, 0xb6ff49, 0x92ff6d, 0xff6dff, 0x009292, 0xb6dbff, 0xff0000, 0x9200ff, 0xffff6d, 0xff92ff, 0xffffff, 0xdb6dff, 0x92dbff, 0x009200,
	0x004900, 0x6db6ff, 0xb62400, 0xdbdbdb, 0x00b66d, 0x6ddb00, 0x490000, 0x9292ff, 0x494949, 0xff00ff, 0x00006d, 0x49ffdb, 0xdbb6ff, 0x6d4900, 0x000000, 0x6d49db,
	0x92006d, 0xffdb92, 0xff9200, 0xffb6ff, 0x006ddb, 0x6d2400, 0xb6b6b6, 0x0000db, 0xb600ff, 0xffdb00, 0x6d6d6d, 0x244900, 0x0049ff, 0x000000, 0xdbdb00, 0xffffff,
	0xdbb66d, 0x242424, 0x00ff00, 0xdb6d00, 0x004949, 0x002492, 0xff0092, 0x249200, 0x000000, 0x00ffff, 0x924900, 0xffff00, 0xffb6b6, 0xb6006d, 0x006d24, 0x929292
},
{ // 2C04C
	0xb600ff, 0xff6dff, 0x92ff6d, 0xb6b6b6, 0x009200, 0xffffff, 0xb6dbff, 0x244900, 0x002492, 0x000000, 0xffdb92, 0x6d4900, 0xff0092, 0xdbdbdb, 0xdbb66d, 0x92dbff,
	0x9292ff, 0x009292, 0xb6006d, 0x0049ff, 0x249200, 0x926d00, 0xdb6d00, 0x00b66d, 0x6d6d6d, 0x6d49db, 0x000000, 0x0000db, 0xff0000, 0xb62400, 0xff92ff, 0xffb6b6,
	0xdb6dff, 0x004900, 0x00006d, 0xffff00, 0x242424, 0xffb600, 0xff9200, 0xffffff, 0x6ddb00, 0x92006d, 0x6db6ff, 0xff00ff, 0x006ddb, 0x929292, 0x000000, 0x6d2400,
	0x00ffff, 0x490000, 0xb6ff49, 0xffb6ff, 0x924900, 0x00ff00, 0xdbdb00, 0x494949, 0x006d24, 0x000000, 0xdbb6ff, 0xffff6d, 0x9200ff, 0x49ffdb, 0xffdb00, 0x004949
},
{ // 2C04D
	0x926d00, 0x6d49db, 0x009292, 0xdbdb00, 0x000000, 0xffb6b6, 0x002492, 0xdb6d00, 0xb6b6b6, 0x6d2400, 0x00ff00, 0x00006d, 0xffdb92, 0xffff00, 0x009200, 0xb6ff49,
	0xff6dff, 0x490000, 0x0049ff, 0xff92ff, 0x000000, 0x494949, 0xb62400, 0xff9200, 0xdbb66d, 0x00b66d, 0x9292ff, 0x249200, 0x92006d, 0x000000, 0x92ff6d, 0x6db6ff,
	0xb6006d, 0x006d24, 0x924900, 0x0000db, 0x9200ff, 0xb600ff, 0x6d6d6d, 0xff0092, 0x004949, 0xdbdbdb, 0x006ddb, 0x004900, 0x242424, 0xffff6d, 0x929292, 0xff00ff,
	0xffb6ff, 0xffffff, 0x6d4900, 0xff0000, 0xffdb00, 0x49ffdb, 0xffffff, 0x92dbff, 0x000000, 0xffb600, 0xdb6dff, 0xb6dbff, 0x6ddb00, 0xdbb6ff, 0x00ffff, 0x244900
} };

#define DIP_PALETTE (DrvDips[2] & 1)
static INT32 last_palette;
UINT32 *our_palette = NULL;

static void UpdatePalettePointer()
{
	switch (PPUType) {
		default:
		case RP2C02: // NES/FC palette (dip-selectable)
			our_palette = nes_palette[DIP_PALETTE];
			break;
		case RP2C03:	// VS. palettes (... and below!)
		case RP2C05A:
		case RP2C05B:
		case RP2C05C:
		case RP2C05D:
		case RP2C05E:
			our_palette = vs_palette[0];
			break;
		case RP2C04A:
		case RP2C04B:
		case RP2C04C:
		case RP2C04D:
			our_palette = vs_palette[1 + (PPUType - RP2C04A)];
			break;
	}
}

static UINT8 GetAvgBrightness(INT32 x, INT32 y)
{
	// Zapper Detection
	const UINT32 rgbcolor = our_palette[pTransDraw[(y) * 256 + x] & 0x3f];

	return ((rgbcolor & 0xff) + ((rgbcolor >> 8) & 0xff) + ((rgbcolor >> 16) & 0xff)) / 3;
}

static INT32 nes_frame_cycles;
static float nes_ppu_cyc_mult;
static INT32 prerender_line; // ntsc 261, pal 311

static INT32 ppu_frame;
static INT32 ppu_framecycles;
static INT32 mirroring;
static UINT8 nt_ram[0x400 * 4]; // nt (2k std. / 4k four-screen [2k expansion is usually on cart])
static UINT8 pal_ram[0x20];   // palette ram
static UINT8 oam_ram[0x100];  // oamram (sprite)
static OAMBUF oam[0x40];   // "to be drawn" oam
static INT32 oam_cnt;
static OAMBUF oam2[0x40];  // secondary oam (after eval)
static INT32 oam2_cnt;
static UINT8 oamAddr;

static UINT16 vAddr, tAddr; // vram addr (loopy-v), temp vram addr (loopy-t)
static UINT16 vAddr_update;
static INT32 vAddr_update_delay;
static UINT8 fine_x; // fine-x (scroll)

static PPUCTRL ctrl;
static PPUMASK mask;
static PPUSTATUS status;
#define RENDERING (mask.reg & 0x18) // show bg (0x08) + show sprites (0x10)
static INT32 sprite_height; // 8, 16. set on PPUCTRL
static INT32 v_incr; // 1, 32. set on PPUCTRL
static INT32 bgtable_start; // 0, 0x1000. set on PPUCTRL
static INT32 sprtable_start; // 0, 0x1000. set on PPUCTRL
static INT32 ppu_pal_mask; // 0x3f (color) 0x30 (gray). set on PPUMASK
static INT32 ppu_pal_emphasis; // emphasis palette index

static INT32 ppu_no_nmi_this_frame;
static INT32 ppu_odd;
static INT32 ppu_startup; // Start-up delay
static UINT16 ppu_bus_address;
static INT32 ppu_over; // #cycles we've run over/under to be compensated for on next frame
static UINT8 ppu_dbus; // ppu data-bus
static UINT8 ppu_buffer; // VRAM read buffer.

static UINT8 write_latch;
static UINT8 nt_byte;

static UINT8 bgL, bgH;
static UINT16 bg_shiftL, bg_shiftH;

static UINT8 at_byte;
static UINT8 at_shiftL, at_shiftH;
static UINT8 at_latchL, at_latchH;

#define get_bit(x, n) (((x) >> (n)) & 1)
static UINT8 bitrev_table[0x100];

// Nametable mirroring/mapping
static UINT8 *NTMap[4];
static INT32 NTType[4];

static void nametable_map(INT32 nt, INT32 ntbank)
{
	NTMap[nt & 3] = &nt_ram[0x400 * (ntbank & 3)];
	NTType[nt & 3] = MEM_RAM;
}

static void nametable_mapraw(INT32 nt, UINT8 *ntraw, UINT8 type)
{
	NTMap[nt & 3] = ntraw;
	NTType[nt & 3] = type;
}

static void nametable_mapall(INT32 ntbank0, INT32 ntbank1, INT32 ntbank2, INT32 ntbank3)
{
	nametable_map(0, ntbank0);
	nametable_map(1, ntbank1);
	nametable_map(2, ntbank2);
	nametable_map(3, ntbank3);
}

static void set_mirroring(INT32 mode)
{
	switch (mode)
    {
        case HORIZONTAL:  nametable_mapall(0, 0, 1, 1); break;
        case VERTICAL:	  nametable_mapall(0, 1, 0, 1); break;
		case SINGLE_LOW:  nametable_mapall(0, 0, 0, 0); break;
		case SINGLE_HIGH: nametable_mapall(1, 1, 1, 1); break;
		case FOUR_SCREEN: nametable_mapall(0, 1, 2, 3); break;
		case SACHEN:	  nametable_mapall(0, 1, 1, 1); break;
	}

	mirroring = mode;
}

static UINT8 read_nt_int(UINT16 address)
{
	return NTMap[(address & 0xfff) >> 10][address & 0x3ff];
}

static void write_nt_int(UINT16 address, UINT8 data)
{
	if (NTType[(address & 0xfff) >> 10] == MEM_RAM)
		NTMap[(address & 0xfff) >> 10][address & 0x3ff] = data;
}

// The internal PPU-Bus
static UINT8 ppu_bus_read(UINT16 address)
{
	if (ppu_startup) return 0; // ignore reads until line 261 on first frame.

	if (address >= 0x0000 && address <= 0x1fff)
		return mapper_chr_read(address);

	if (address >= 0x2000 && address <= 0x3eff)
		return read_nt(address);

	if (address >= 0x3f00 && address <= 0x3fff) {
		if ((address & 0x13) == 0x10) // 0x10, 0x14, 0x18, 0x1c -> 0x00, 0x04, 0x08, 0x0c
			address &= ~0x10;
		return pal_ram[address & 0x1f] & ppu_pal_mask;
	}

	return 0;
}

static void ppu_bus_write(UINT16 address, UINT8 data)
{
	if (address >= 0x0000 && address <= 0x1fff)
		mapper_chr_write(address, data);

	if (address >= 0x2000 && address <= 0x3eff)
		write_nt(address, data);

	if (address >= 0x3f00 && address <= 0x3fff) {
		if ((address & 0x13) == 0x10)
			address &= ~0x10;
		pal_ram[address & 0x1f] = data;
	}
}

static void h_scroll();
static void v_scroll();

static void ppu_inc_v_addr()
{
	if (RENDERING && (scanline < 241 || scanline == prerender_line)) {
		// no updates while rendering!
		//bprintf(0, _T("no updates while rendering! frame: %X  scanline: %d    pixel: %d\n"), nCurrentFrame, scanline, pixel);

		// Burai Fighter (U) statusbar fix
		h_scroll();
		v_scroll();
		return;
	}

	vAddr = (vAddr + v_incr) & 0x7fff;
	ppu_bus_address = vAddr & 0x3fff;
}

static void get_vsystem_prot(UINT8 &dbus, UINT8 status_reg)
{	// 2C05A-E(D) returns a special ID in the usually-unused PPU status bits
	// as a form of EEPROM-Swap/Copy protection.
	const UINT8 _2c05_ids[5] = { 0x1b, 0x3d, 0x1c, 0x1b, 0x00 };
	if (PPUType >= RP2C05A && PPUType <= RP2C05E) {
		dbus = _2c05_ids[PPUType - RP2C05A] | status_reg;
	}
}

static UINT8 ppu_read(UINT16 reg)
{
	reg &= 7;

	switch (reg)
	{
		case 2: // PPUSTATUS
			if (scanline == 241 && pixel < 3) {
				// re: https://wiki.nesdev.com/w/index.php/PPU_frame_timing
				// PPUSTATUS read: 1 cycle before vbl (pixel == 0) skips status & nmi
				// on or 1 pixel after (pixel == 1 or 2) returns vblank status, but skips nmi
				//bprintf(0, _T("PPUSTATUS: CANCEL NMI.  scanline %d  pixel %d\n"), scanline, pixel);
				ppu_no_nmi_this_frame = 1;
				if (pixel == 0) status.bit.VBlank = 0;
			}
			ppu_dbus = (ppu_dbus & 0x1f) | status.reg;
			get_vsystem_prot(ppu_dbus, status.reg);
			status.bit.VBlank = 0;
			write_latch = 0;
			//bprintf(0, _T("PPUSTATUS - frame: %d   scanline: %d     pixel: %d    res: %X   PC: %X\n"), nCurrentFrame, scanline, pixel, ppu_dbus, M6502GetPrevPC(-1));
			break;

		case 4: // OAMDATA
			if (RENDERING && scanline < 241) {
				// quirks to keep micromachines happy
				if (pixel >= 1 && pixel <= 64) {
					ppu_dbus = 0xff;
					break;
				}
				if (pixel >= 65 && pixel <= 256) {
					ppu_dbus = 0x00;
					break;
				}

				if (pixel >= 257 && pixel <= 320) {
					ppu_dbus = oam2[((pixel & 0xff) >> 3)].x;
					break;
				}

				if ((pixel >= 321 && pixel <= 340) || pixel == 0) {
					ppu_dbus = oam2[0].y;
					break;
				}
			}
			ppu_dbus = oam_ram[oamAddr];
			break;

		case 7: // PPUDATA
			if ((ppu_bus_address & 0x3fff) < 0x3f00) {
				// vram buffer delay
				ppu_dbus = ppu_buffer;
				ppu_buffer = ppu_bus_read(ppu_bus_address);
				//mapper_ppu_clock(ppu_bus_address);
			} else {
				// palette has no buffer delay, buffer gets stuffed with vram though (ppu quirk)
				ppu_dbus = ppu_bus_read(ppu_bus_address);
				ppu_buffer = ppu_bus_read(ppu_bus_address - 0x1000);
			}

			ppu_inc_v_addr();
			break;

	}

	return ppu_dbus;
}

static void ppu_write(UINT16 reg, UINT8 data)
{
	reg &= 7;

	if (PPUType >= RP2C05A && reg < 2) {
		// PPUCTRL / PPUMASK swapped on RP2C05x
		reg ^= 1;
	}

	ppu_dbus = data;

	switch (reg)
	{
		case 0: // PPUCTRL
			if (ppu_startup) return; // ignore writes until line 261
			if (data & 0x80) {
				if ((~ctrl.reg & 0x80) && status.bit.VBlank) {
					//--Note: If NMI is fired here, it will break:
					//Bram Stokers Dracula, Galaxy 5000, GLUK The Thunder aka Thunder Warrior
					//Animal Clipper (HB) - nmi clobbers A register during bootup.
					//Dragon Power scrolling goes bad
					//Solution: Delay NMI by 1(?) opcode
					//bprintf(0, _T("PPUCTRL: toggle-nmi-arm! scanline %d  pixel %d    frame: %d   PPC %X\n"), scanline, pixel, nCurrentFrame, M6502GetPC(-1));
					m6502_set_nmi_hold2();
				}
			} else {
				//bprintf(0, _T("PPUCTRL: %X  cancel-nmi?  scanline %d  pixel %d   frame %d\n"), data, scanline, pixel, nCurrentFrame);
			}

			ctrl.reg = data;
			//bprintf(0, _T("PPUCTRL reg: %X   scanline %d  pixel %d   frame %d  PC  %X\n"), ctrl.reg, scanline, pixel, nCurrentFrame, M6502GetPC(-1));
			tAddr = (tAddr & 0x73ff) | ((data & 0x3) << 10);

			sprite_height = ctrl.bit.sprsize ? 16 : 8;
			v_incr = ctrl.bit.incr ? 32 : 1;
			bgtable_start = ctrl.bit.bgtab * 0x1000;
			sprtable_start = ctrl.bit.sprtab * 0x1000;
			break;

		case 1: // PPUMASK
			if (ppu_startup) return; // ignore writes until line 261
			if ((data & 0x18)==0 && RENDERING) { // rendering ON -> OFF assert ppu_bus_address
				ppu_bus_address = vAddr & 0x3fff;
			}
			mask.reg = data;
			ppu_pal_emphasis = (data >> 5) * 0x40;
			if (NESMode & IS_PAL) {
				ppu_pal_emphasis = (((data & 0x80) | ((data & 0x40) >> 1) | ((data & 0x20) << 1)) >> 5) * 0x40;
			}
			//bprintf(0, _T("mask   %x\n"), data);
			ppu_pal_mask = (mask.bit.gray ? 0x30 : 0x3f);
			break;

		case 3: // OAMADDR
			oamAddr = data;
			break;

		case 4: // OAMDATA
			//bprintf(0, _T("Frame %04x:  OAMDATA[%X]  %X      scanline %d  pixel %d\n"), nCurrentFrame, oamAddr, data, scanline, pixel);
			if (RENDERING && (scanline < 241 || scanline == prerender_line) && (~NESMode & BAD_HOMEBREW)) {
				bprintf(0, _T("write OAM prohibited.  scanline %d\n"), scanline);
				return;
			}
			if ((oamAddr & 3) == 2) data &= 0xe3; // attr mask
			oam_ram[oamAddr] = data;
			oamAddr = (oamAddr + 1) & 0xff;
			break;

		case 5: // PPUSCROLL
			if (ppu_startup) return; // ignore writes until line 261

			if (!write_latch) {      // First write.
				fine_x = data & 7;
				tAddr = (tAddr & 0x7fe0) | ((data & 0xf8) >> 3);
			} else  {                // Second write.
				tAddr = (tAddr & 0x0c1f) | ((data & 0xf8) << 2) | ((data & 0x07) << 12);
			}
			write_latch = !write_latch;
			break;
		case 6: // PPUADDR
			if (ppu_startup) return; // ignore writes until line 261
			if (!write_latch) {		// First write.
				tAddr = (tAddr & 0x00ff) | ((data & 0x3f) << 8);
			} else {                // Second write.
				tAddr = (tAddr & 0x7f00) | data;
				vAddr_update = tAddr;
				vAddr_update_delay = 2;
			}
			write_latch = !write_latch;
			break;
		case 7: // PPUDATA
			ppu_bus_write(ppu_bus_address, data);
			//mapper_ppu_clock(ppu_bus_address);

			ppu_inc_v_addr();
			break;
	}
}

static void h_scroll()
{
	if (RENDERING) {
		if ((vAddr & 0x1f) == 0x1f) {
			vAddr ^= 0x041f;
		} else {
			vAddr++;
		}
	}
}

static void v_scroll()
{
	if (RENDERING) {
		if ((vAddr & 0x7000) == 0x7000) {
			switch (vAddr & 0x03e0) {
				case 0x03a0: vAddr ^= 0x7800 | 0x03a0; break;
				case 0x03e0: vAddr &= ~0x73e0; break;
				default: vAddr = (vAddr & ~0x7000) + 0x0020;
			}
		} else {
			vAddr += 0x1000;
		}
	}
}

static void reload_shifters()
{
	at_latchL = at_byte & 1;
    at_latchH = (at_byte & 2) >> 1;

	bg_shiftL = (bg_shiftL & 0xff00) | bgL;
    bg_shiftH = (bg_shiftH & 0xff00) | bgH;
}

static void evaluate_sprites()
{
	INT32 cur_line = (scanline == prerender_line) ? -1 : scanline;
	oam2_cnt = 0;

    for (INT32 i = oamAddr; i < 0x40; i++) {
        INT32 line = cur_line - oam_ram[(i << 2) + 0];
        if (line >= 0 && line < sprite_height) {
			if (oam2_cnt == 8) {
				status.bit.spovrf = 1;
				if (~DrvDips[0] & 1) // DIP option: disable sprite limit
					break;
			}
			oam2[oam2_cnt].idx  = i;
            oam2[oam2_cnt].y    = oam_ram[(i << 2) + 0];
            oam2[oam2_cnt].tile = oam_ram[(i << 2) + 1];
            oam2[oam2_cnt].attr = oam_ram[(i << 2) + 2];
			oam2[oam2_cnt++].x  = oam_ram[(i << 2) + 3];
		}
	}
}

static void load_sprites()
{
	oam_cnt = 0;
	for (INT32 i = 0; i < oam2_cnt; i++) {
		oam[oam_cnt++] = oam2[i];

		UINT32 sprY = scanline - oam[i].y;
		sprY = (oam[i].attr & 0x80) ? ~sprY : sprY; // invert y-bits if y-flipped

		if (sprite_height == 16) {
			ppu_bus_address = ((oam[i].tile & 1) * 0x1000) + ((oam[i].tile & 0xfe) * 16)
				+ ((sprY & 8) << 1) + (sprY & 7);
		} else {
			ppu_bus_address = sprtable_start + (oam[i].tile * 16) + (sprY & 7);
		}

		oam[i].tileL = ppu_bus_read(ppu_bus_address);
		ppu_bus_address += 8;
		oam[i].tileH = ppu_bus_read(ppu_bus_address);

		if (mapper_ppu_clock) mapper_ppu_clock(ppu_bus_address); // Punch-Out (mapper 9) latches on high-byte read (+8)

		if (oam[i].attr & 0x40) { // reverse bits if x-flipped
			oam[i].tileL = bitrev_table[oam[i].tileL];
			oam[i].tileH = bitrev_table[oam[i].tileH];
		}
    }
}

static void draw_and_shift()
{
    INT32 x = pixel - 2;

	if (scanline < 240 && x >= 0 && x < 256) {
		UINT8 pix = 0;
		UINT8 sprPal = 0;
		UINT8 sprPri = 0;
		UINT8 spr = 0;
		UINT16 eff_x = 0;

		if (!RENDERING && (vAddr & 0x3f00) == 0x3f00) {
			// https://wiki.nesdev.com/w/index.php/PPU_palettes "The background palette hack"
			pix = vAddr & 0x1f;
		}

		if (mask.bit.bg && (mask.bit.bgLeft == 0 && x < 8) == 0) {
			pix = (get_bit(bg_shiftH, 15 - fine_x) << 1) |
				  (get_bit(bg_shiftL, 15 - fine_x) << 0);

			if (pix) {
				pix |= ((get_bit(at_shiftH, 7 - fine_x) << 1) |
						(get_bit(at_shiftL, 7 - fine_x) << 0)) << 2;
			}
        }

		if (mask.bit.spr && (mask.bit.sprLeft == 0 && x < 8) == 0) {
			for (INT32 i = oam_cnt - 1; i >= 0; i--) {
				if (oam[i].idx == 0xff) // no sprite
					continue;

				eff_x = x - oam[i].x;
				if (eff_x >= 8) // sprite out of view
					continue;

				spr = (get_bit(oam[i].tileH, 7 - eff_x) << 1) |
					  (get_bit(oam[i].tileL, 7 - eff_x) << 0);

				if (spr == 0) // transparent sprite, ignore
					continue;

				if (oam[i].idx == 0 && pix && x != 0xff) {
					status.bit.sp0hit = 1;
				}

                spr |= (oam[i].attr & 3) << 2; // add color (attr), 2bpp shift
                sprPal = spr + 0x10; // add sprite color palette-offset
                sprPri = ~oam[i].attr & 0x20; // sprite over bg?
            }
		}

		if (~nBurnLayer & 1) pix = 0; // if tile layer disabled, clear pixel.
        if (sprPal && (pix == 0 || sprPri) && nSpriteEnable & 1) pix = sprPal;

		if (NESMode & SHOW_OVERSCAN) {
			pTransDraw[scanline * 256 + x] = ppu_bus_read(0x3f00 + pix) | ppu_pal_emphasis;
		} else {
			if (scanline >= 8 && scanline < (nScreenHeight + 8)/*232*/)
				pTransDraw[(scanline - 8) * 256 + x] = ppu_bus_read(0x3f00 + pix) | ppu_pal_emphasis;
		}
    }

	bg_shiftL <<= 1;
	bg_shiftH <<= 1;
    at_shiftL = (at_shiftL << 1) | at_latchL;
    at_shiftH = (at_shiftH << 1) | at_latchH;
}

static void scanlinestate(INT32 state)
{
	if (state == VBLANK) {

		switch (pixel) {
			case 1: // enter VBlank
				ppu_bus_address = vAddr & 0x3fff;
				status.bit.VBlank = 1;
				ppu_startup = 0;

				if (NESMode & ALT_TIMING2) {
					if (ctrl.bit.nmi && ppu_no_nmi_this_frame == 0) {
						//bprintf(0, _T("nmi @ frame %d  scanline %d  pixel %d  PPUCTRL %x\n"), nCurrentFrame, scanline, pixel, ctrl.reg);
						M6502SetIRQLine(CPU_IRQLINE_NMI, CPU_IRQSTATUS_AUTO);
					}
					ppu_no_nmi_this_frame = 0;
				}
				break;

			case (6 * 3):
				// 6 CPU-cycles later, do nmi.  fixes boot w/b-wings, bad dudes, gunnac
				// crap on screen with marble madness. (Also passes blargg timing tests)
				if (~NESMode & ALT_TIMING2) {
					if (ctrl.bit.nmi && ppu_no_nmi_this_frame == 0) {
						//bprintf(0, _T("nmi @ frame %d  scanline %d  pixel %d  PPUCTRL %x\n"), nCurrentFrame, scanline, pixel, ctrl.reg);
						M6502SetIRQLine(CPU_IRQLINE_NMI, CPU_IRQSTATUS_AUTO);
					}
					ppu_no_nmi_this_frame = 0;
				}
				break;
		}
	}
	else if (state == VISIBLE || state == PRERENDER) {

		// Sprites
		switch (pixel) {
			case 1:
				memset(oam2, 0xff, sizeof(oam2)); // clear secondary oam

				if (state == PRERENDER) {
					status.reg &= 0x1f; // clear vbl, spr0 hit & overflow bits

					if (oamAddr > 7) { // 2c02 oamram corruption (Huge Insect, Tatakai no Banka pre-revA)
						for (INT32 i = 0; i < 8; i++) {
							oam_ram[i] = oam_ram[(oamAddr & 0xf8) + i];
						}
					}
				}
				break;
			case 257:
				evaluate_sprites();
				if (RENDERING) {
					oamAddr = 0;
				}
				break;
			case 321:
				if (RENDERING) {
					load_sprites();
				} else {
					oam_cnt = 0;
				}
				break;
		}

		// Tiles
		if ( (pixel >= 1 && pixel <= 257) || (pixel >= 321 && pixel <= 337) ) {
			if (pixel != 1) draw_and_shift();
			if (RENDERING) {
				switch (pixel & 7) {
					case 1:
						ppu_bus_address = 0x2000 | (vAddr & 0x0fff); // nametable address
						reload_shifters();
						if (pixel == 257 && RENDERING) {
							// copy horizontal bits from loopy-T to loopy-V
							vAddr = (vAddr & ~0x041f) | (tAddr & 0x041f);
						}
						break;
					case 2:
						nt_byte = ppu_bus_read(ppu_bus_address);
						break;
					case 3:
						ppu_bus_address = 0x23c0 | (vAddr & 0x0c00) | ((vAddr >> 4) & 0x38) | ((vAddr >> 2) & 7); // attribute address
						break;
					case 4:
						at_byte = ppu_bus_read(ppu_bus_address);
						if (vAddr & (1 << 6) /* coarse Y & 2 */) at_byte >>= 4;
						if (vAddr & (1 << 1) /* coarse X & 2 */) at_byte >>= 2;
						break;
					case 5:
						ppu_bus_address = bgtable_start + (nt_byte * 16) + (vAddr >> 12); // background address
						break;
					case 6:
						bgL = ppu_bus_read(ppu_bus_address);
						break;
					case 7:
						ppu_bus_address = 8 + bgtable_start + (nt_byte * 16) + (vAddr >> 12); // background address
						break;
					case 0:
						bgH = ppu_bus_read(ppu_bus_address);
						if (pixel == 256)
							v_scroll();
						else
							h_scroll();
						break;
				}
			}
		}

		if (state == PRERENDER && pixel >= 280 && pixel <= 304 && RENDERING) {
			// copy vertical bits from loopy-T to loopy-V
			vAddr = (vAddr & ~0x7be0) | (tAddr & 0x7be0);
		}

		if (pixel >= 337 && pixel <= 340 && RENDERING) {
			// dummy nt fetches
			ppu_bus_address = 0x2000 | (vAddr & 0x0fff);
		}

		// Signal scanline to mapper: (+18 gets rid of jitter in Kirby, Yume Penguin Monogatari, Klax, ...)
		// Note: was 20, but caused occasional flickering in the rasterized water in "Kira Kira Star Night GOLD" (maybe the others in the series, too?)
		if (pixel == (260+18) /*&& RENDERING*/ && mapper_scanline) mapper_scanline();
		// Mapper callback w/ppu_bus_address - used by mmc2/mmc4 (mapper9/mapper10)
		if (mapper_ppu_clock) mapper_ppu_clock(ppu_bus_address); // ppu callback (visible lines)
    }
}

void ppu_cycle()
{
	ppu_framecycles++;	// keep track of cycles/frame since reset
	pixel++; 			// cycle (pixel)/line

	if (pixel > 340) {
		pixel = 0;
		scanline++;
		if (scanline > prerender_line) {
			scanline = 0;
			ppu_frame++;
			ppu_odd ^= 1;
			if (RENDERING && ppu_odd) {
				// this real hw hack was added to the ppu back in the day for better
				// rf/composite output.  since it messes with current cpu:ppu
				// sync. and isn't really necessary w/emu, let's disable it. -dink
				//pixel++;
				//ppu_framecycles++;
			}
        }
	}

	if (scanline >= 0 && scanline <= 239)
		scanlinestate(VISIBLE);
	else if (scanline == 241)
		scanlinestate(VBLANK);
	else if (scanline == prerender_line) {
		scanlinestate(PRERENDER);
	}

	if (mapper_ppu_clockall) mapper_ppu_clockall(ppu_bus_address); // ppu callback (every line)

	if (vAddr_update_delay > 0) {
		vAddr_update_delay--;
		if (vAddr_update_delay == 0) {
			vAddr = vAddr_update;
			if ((scanline >= 240 && scanline < prerender_line) || !RENDERING) {
				ppu_bus_address = vAddr & 0x3fff;
			}
		}
	}
}

static void ppu_run(INT32 cyc)
{
	while (ppu_over < 0) { // we didn't run enough cycles last frame, catch-up!
		cyc++;
		ppu_over++;
	}

	if ((NESMode & IS_PAL) && (mega_cyc_counter % 5) == 0) {
		cyc++;
	}

	while (cyc) {
		if (ppu_over > 0) { // if we're over some cycles on the start of next:
			ppu_over--;     // frame - idle them away
		} else {
			ppu_cycle();
		}
		cyc--;
	}
}

static void ppu_scan(INT32 nAction)
{
	SCAN_VAR(mirroring);
	SCAN_VAR(scanline);
	SCAN_VAR(pixel);
	SCAN_VAR(ppu_frame);

	SCAN_VAR(vAddr);
	SCAN_VAR(tAddr);
	SCAN_VAR(vAddr_update);
	SCAN_VAR(vAddr_update_delay);

	SCAN_VAR(fine_x);
	SCAN_VAR(oamAddr);

	SCAN_VAR(ctrl);
	SCAN_VAR(mask);
	SCAN_VAR(status);

	SCAN_VAR(sprite_height);
	SCAN_VAR(v_incr);
	SCAN_VAR(bgtable_start);
	SCAN_VAR(sprtable_start);
	SCAN_VAR(ppu_pal_mask);

	SCAN_VAR(ppu_no_nmi_this_frame);
	SCAN_VAR(ppu_odd);
	SCAN_VAR(ppu_startup);
	SCAN_VAR(ppu_bus_address);
	SCAN_VAR(ppu_over);
	SCAN_VAR(ppu_dbus);
	SCAN_VAR(ppu_buffer);
	SCAN_VAR(write_latch);

	SCAN_VAR(nt_byte);
	SCAN_VAR(bgL);
	SCAN_VAR(bgH);
	SCAN_VAR(bg_shiftL);
	SCAN_VAR(bg_shiftH);

	SCAN_VAR(at_byte);
	SCAN_VAR(at_shiftL);
	SCAN_VAR(at_shiftH);
	SCAN_VAR(at_latchL);
	SCAN_VAR(at_latchH);

	// ppu-ram
	SCAN_VAR(nt_ram);
	SCAN_VAR(pal_ram);
	SCAN_VAR(oam_ram);

	if (nAction & ACB_WRITE) {
		set_mirroring(mirroring);
	}
}

static void ppu_reset()
{
	mmc5_nt_ram = &nt_ram[0];

	// start at (around) vblank to remove 1 full frame of input lag
	scanline = 239; // line on titlescreen of micromachines if starts on 240
	if (NESMode & BAD_HOMEBREW) scanline = 0; // animal clipper, enables nmi via ppuctrl, if happens during vblank, game will go nuts
	pixel = 0;
	ppu_frame = 0;

	vAddr = 0;
	tAddr = 0;
	fine_x = 0;
	oamAddr = 0;

	ctrl.reg = 0;
	mask.reg = 0;
	status.reg = 0;

	mmc5_mask = &mask.reg; // point to ppumask register
	mmc5_ctrl = &ctrl.reg; // point to ppuctrl register

	sprite_height = 8;
	v_incr = 1;
	bgtable_start = 0;
	sprtable_start = 0;
	ppu_pal_mask = 0x3f;
	ppu_pal_emphasis = 0;

	ppu_odd = 0;
	ppu_no_nmi_this_frame = 0;
	ppu_startup = 0; // ignore some reads/writes until ppu "warms up" *disabled, see next line*
	// ppu_startup breaks Magic John - but why??  A: games writes ppu registers too early..
	ppu_bus_address = 0;
	ppu_over = 0; // cycles overrun this frame
	ppu_dbus = 0;
	ppu_buffer = 0;

	write_latch = 0;

	ppu_framecycles = 0; // total ran cycles this frame

    memset(nt_ram, 0xff, sizeof(nt_ram));
	memset(pal_ram, 0x00, sizeof(pal_ram));
    memset(oam_ram, 0xff, sizeof(oam_ram));

	memset(oam, 0xff, sizeof(oam));
	memset(oam2, 0xff, sizeof(oam2));

	nt_byte = 0;
	bgL = bgH = 0;
	bg_shiftL = bg_shiftH = 0;

	at_byte = 0;
	at_shiftL = at_shiftH = 0;
	at_latchL = at_latchH = 0;

	oam_cnt = 0;
	oam2_cnt = 0;

	last_palette = DIP_PALETTE;

	UpdatePalettePointer();
}

static void ppu_init(INT32 is_pal)
{
	read_nt = &read_nt_int;
	write_nt = &write_nt_int;
	for (INT32 i = 0; i < 0x100; i++)
		bitrev_table[i] = BITSWAP08(i, 0, 1, 2, 3, 4, 5, 6, 7);

	if (is_pal) {
		nes_frame_cycles = 33248; // pal
		prerender_line = 311;
		nes_ppu_cyc_mult = 3.2;
	} else {
		nes_frame_cycles = 29781; // ntsc (default)
		nes_ppu_cyc_mult = 3.0;
		prerender_line = 261;
	}

	ppu_reset();
}

enum {
	ZAP_SENSE		= 0x00,
	ZAP_NONSENSE	= 0x08,
	ZAP_TRIGGER		= 0x10
};

static UINT8 nes_read_zapper()
{
	if (RENDERING == 0 || scanline < 8 || scanline > 240)
		return ZAP_NONSENSE;

	INT32 in_y = ((BurnGunReturnY(0) * 224) / 255);
	INT32 in_x = BurnGunReturnX(0);
	INT32 real_sl = scanline - 8;

	// offscreen check
	if (in_y == 0 || in_y == 224 || in_x == 0 || in_x == 255) {
		return ZAP_NONSENSE;
	}

	for (INT32 yy = in_y - 2; yy < in_y + 2; yy++) {
		if (yy < real_sl-8 || yy > real_sl || yy < 0 || yy > 224) continue;

		for (INT32 xx = in_x - 2; xx < in_x + 2; xx++) {
			if (xx < 0 || xx > 255) continue;
			if (yy == real_sl && xx >= pixel) break; // <- timing is everything, here.
			if (GetAvgBrightness(xx, yy) > 0x88) { // + flux capacitor makes time travel possible
				return ZAP_SENSE;
			}
		}
	}

	return ZAP_NONSENSE;
}

static UINT8 nes_read_joy(INT32 port)
{
	UINT8 ret = 0;

	if ((NESMode & USE_ZAPPER) && port == 1) {
		// NES Zapper on second port (0x4017)
		ret = nes_read_zapper(); // light sensor
		ret |= (ZapperFire) ? ZAP_TRIGGER : 0x00; // trigger

		if (ZapperReload) {
			ZapperReloadTimer = 10;
			ret = ZAP_TRIGGER | ZAP_NONSENSE;
		} else if (ZapperReloadTimer) {
			ret = ZAP_NONSENSE;
		}

	} else {
		if (NESMode & USE_HORI4P) {
			UINT8 a = (JoyShifter[port&1] >> 0) & 0xff; // joy 1,3
			UINT8 b = (JoyShifter[port&1] >> 8) & 0xff; // joy 2,4
			ret  = (a & 1);			// joy 1,2 d0 (FC hardwired controllers)
			ret |= (b & 1) << 1;	// joy 3,4 d1 (controllers connected to HORI4P)
			a >>= 1;                // shift next bit
			b >>= 1;                // "
			a |= 0x80;              // feed empty high bit (active low)
			b |= 0x80;              // "

			JoyShifter[port&1] = (a & 0xff) | ((b << 8) & 0xff00);
		} else {
			ret = JoyShifter[port&1]&1;
			JoyShifter[port&1] >>= 1;   // shift next
			JoyShifter[port&1] |= 0x80000000; // feed high bit so reads past our data return high
		}

		// MIC "blow" hack (read on 1p controller address, yet, mic is on 2p controller!)
		if (NESMode & IS_FDS && port == 0) {
			if (DrvInputs[1] & ((1<<2) | (1<<3))) { // check if 2p select or start pressed. note: famicom 2p controller doesn't have start or select, we use it as a MIC button.
				ret |= 4;
			}
		}

		// note, some games (f.ex: paperboy) relies on the open bus bits in the high nibble
		ret |= cpu_open_bus & 0xf0;
	}

	return ret;
}

static UINT8 nesvs_read_joy(INT32 port)
{ // VS. Unisystem, VS. Zapper handled in psg_io_write() ctrlr strobe
	UINT8 joy = nes_read_joy(port) & 1;

	switch (port) {
		case 0: // 4016: joy + bottom 3 bits of dip + coin
			joy = (joy & 0x01) | ((DrvDips[3] & 3) << 3) | (NESCoin[0] << 2);
			break;
		case 1: // 4017: joy + top 6 bits of dip
			joy = (joy & 0x01) | (DrvDips[3] & 0xfc);
			break;
	}

	return joy;
}

static UINT8 psg_io_read(UINT16 address)
{
	if (address == 0x4016 || address == 0x4017)
	{	// Gamepad 1 & 2 / Zapper
		if (PPUType > RP2C02) {
			return nesvs_read_joy(address & 1);
		}
		return nes_read_joy(address & 1);
	}

	if (address == 0x4015)
	{
		return nesapuRead(0, address & 0x1f, cpu_open_bus);
	}

	if (address >= 0x4020 && psg_area_read) {
		return psg_area_read(address);
	}

	//bprintf(0, _T("psg_io_read(unmapped) %X\n"), address);

	return cpu_open_bus;
}

static void psg_io_write(UINT16 address, UINT8 data)
{
	if (address == 0x4016)
	{
		if (Cart.Mapper == 99) {
			mapper99_write(address, data);
		}

		if ((JoyStrobe & 1) && (~data & 1)) {
			switch (NESMode & (USE_4SCORE | USE_HORI4P | VS_ZAPPER | VS_REVERSED)) {
				case USE_4SCORE:
					// "Four Score" 4-player adapter (NES / World)
					JoyShifter[0] = DrvInputs[0] | (DrvInputs[2] << 8) | (bitrev_table[0x10] << 16);
					JoyShifter[1] = DrvInputs[1] | (DrvInputs[3] << 8) | (bitrev_table[0x20] << 16);
					break;
				case USE_HORI4P: // mode 2
					// "Hori" 4-player adapter Mode 4 (Japanese / Famicom)
					JoyShifter[0] = DrvInputs[0] | (DrvInputs[2] << 8);
					JoyShifter[1] = DrvInputs[1] | (DrvInputs[3] << 8);
					break;
				case VS_ZAPPER: { // VS. UniSystem Zapper
					UINT8 zap = 0x10;
					zap |= ((~nes_read_zapper() << 3) & 0x40) | (ZapperFire << 7);
					zap |= (ZapperReload) ? 0xc0 : 0x00;
					JoyShifter[0] = zap | 0xffffff00;
					JoyShifter[1] = zap | 0xffffff00;
					break;
				}
				case VS_REVERSED: {
					// some VS. games swap p1/p2 inputs (not select/start aka 0xc0)
					UINT8 in1 = (DrvInputs[1] & 0x0c) | (DrvInputs[0] & ~0x0c);
					UINT8 in0 = (DrvInputs[0] & 0x0c) | (DrvInputs[1] & ~0x0c);
					JoyShifter[0] = in0 | 0xffffff00;
					JoyShifter[1] = in1 | 0xffffff00;
					break;
				}
				default:
					// standard nes controllers
					JoyShifter[0] = DrvInputs[0] | 0xffffff00;
					JoyShifter[1] = DrvInputs[1] | 0xffffff00;
					break;
			}
		}
		JoyStrobe = data;
		return;
	}

	if (address >= 0x4000 && address <= 0x4017)
	{
		nesapuWrite(0, address & 0x1f, data);
		return;
	}

	if (address >= 0x4020 && psg_area_write) {
		psg_area_write(address, data);
		return;
	}

	if (address == 0x4020 && PPUType > RP2C02) return; // 0x4020: ignore coin counter writes on VS. Unisystem

	bprintf(0, _T("psg_io_write(unmapped) %X    %x\n"), address, data);
}

static UINT8 cpu_ram_read(UINT16 address)
{
	return NES_CPU_RAM[address & 0x7FF];
}

static void cpu_ram_write(UINT16 address, UINT8 data)
{
	NES_CPU_RAM[address & 0x7FF] = data;
}

static UINT8 prg_ram_read(UINT16 address) // expansion ram / rom on the cartridge.
{
	if (cart_exp_read) {
		return cart_exp_read(address);
	}

	// some games get irritated (low g-man, mapper 7: double dragon, battle toads) if prg ram exists
	return (NESMode & NO_WORKRAM) ? cpu_open_bus : Cart.WorkRAM[address & Cart.WorkRAMMask];
}

static void prg_ram_write(UINT16 address, UINT8 data)
{
	if (cart_exp_write) {
		cart_exp_write_abort = 0;
		cart_exp_write(address, data);
		if (cart_exp_write_abort) return;
		// fallthrough if abort wasn't set! :)
	}

	if (NESMode & NO_WORKRAM) return;
	Cart.WorkRAM[address & Cart.WorkRAMMask] = data;
}

// cheat system
static UINT8 gg_bit(UINT8 g)
{
	const UINT8 gg_str[0x11] = "APZLGITYEOXUKSVN";

	for (UINT8 i = 0; i < 0x10; i++) {
		if (g == gg_str[i]) {
			return i;
		}
	}
	return 0;
}

static INT32 gg_decode(char *gg_code, UINT16 &address, UINT8 &value, INT32 &compare)
{
	INT32 type = strlen(gg_code);

	if (type != 6 && type != 8) {
		// bad code!
		return 1;
	}

	UINT8 str_bits[8];

	for (UINT8 i = 0; i < type; i++) {
		str_bits[i] = gg_bit(gg_code[i]);
	}

	address = 0x8000 | ((str_bits[1] & 8) << 4) | ((str_bits[2] & 7) << 4) | ((str_bits[3] & 7) << 12) | ((str_bits[3] & 8) << 0) | ((str_bits[4] & 7) << 0) | ((str_bits[4] & 8) << 8) | ((str_bits[5] & 7) << 8);
	value = ((str_bits[0] & 7) << 0) | ((str_bits[0] & 8) << 4) | ((str_bits[1] & 7) << 4);
	compare = -1;

	switch (type) {
		case 6: {
			value |= ((str_bits[5] & 8) << 0);
			break;
		}
		case 8: {
			value |= ((str_bits[7] & 8) << 0);
			compare = ((str_bits[5] & 8) << 0) | ((str_bits[6] & 7) << 0) | ((str_bits[6] & 8) << 4) | ((str_bits[7] & 7) << 4);
			break;
		}
	}

	return 0;
}

const INT32 cheat_MAX = 0x100;
static INT32 cheats_active = 0;

struct cheat_struct {
	char code[0x10]; // gamegenie code
	UINT16 address;
	UINT8 value;
	INT32 compare; // -1, compare disabled.
};

static cheat_struct cheats[cheat_MAX];

void nes_add_cheat(char *code) // 6/8 character game genie codes allowed
{
	UINT16 address;
	UINT8 value;
	INT32 compare;

	if (!gg_decode(code, address, value, compare) && cheats_active < (cheat_MAX-1)) {
		strncpy(cheats[cheats_active].code, code, 9);
		cheats[cheats_active].address = address;
		cheats[cheats_active].value = value;
		cheats[cheats_active].compare = compare;
		bprintf(0, _T("cheat #%d (%S) added.  (%x, %x, %d)\n"), cheats_active, cheats[cheats_active].code, address, value, compare);
		cheats_active++;
	} else {
		if (cheats_active < (cheat_MAX-1)) {
			bprintf(0, _T("nes cheat engine: bad GameGenie code %S\n"), code);
		} else {
			bprintf(0, _T("nes cheat engine: too many active!\n"));
		}
	}
}

void nes_remove_cheat(char *code)
{
	cheat_struct cheat_temp[cheat_MAX];
	INT32 temp_num = 0;

	for (INT32 i = 0; i < cheats_active; i++) {
		if (strcmp(code, cheats[i].code) != 0) {
			memcpy(&cheat_temp[temp_num], &cheats[i], sizeof(cheat_struct));
			temp_num++;
		} else {
			bprintf(0, _T("cheat %S disabled.\n"), cheats[i].code);
		}
	}

	memcpy(cheats, cheat_temp, sizeof(cheats));
	cheats_active = temp_num;
}

static inline UINT8 cheat_check(UINT16 address, UINT8 value)
{
	for (INT32 i = 0; i < cheats_active; i++) {
		if (cheats[i].address == address && (cheats[i].compare == -1 || cheats[i].compare == value)) {
		 //   bprintf(0, _T("."));
			return cheats[i].value;
		}
	}

	return value;
}


static UINT8 cpu_bus_read(UINT16 address)
{
	UINT8 ret = 0;
	switch (address & 0xe000)
	{
		case 0x0000:  //    0 - 1fff
			ret = cpu_ram_read(address); break;
		case 0x2000:  // 2000 - 3fff
			ret = ppu_read(address); break;
		case 0x4000:  // 4000 - 5fff
			ret = psg_io_read(address); break;
		case 0x6000:  // 6000 - 7fff
			ret = prg_ram_read(address); break;
		default:      // 8000 - ffff
			ret = mapper_prg_read(address); break;
	}

	ret = cheat_check(address, ret);

	cpu_open_bus = ret;

	return ret;
}


#define DEBUG_DMA 0

static void cpu_bus_write(UINT16 address, UINT8 data)
{
	cpu_open_bus = data;

	if (address == 0x4014) { // OAM_DMA
#if DEBUG_DMA
		bprintf(0, _T("DMA, tcyc %d   scanline: %d    pixel: %d\n"), M6502TotalCycles(), scanline, pixel);
#endif
		UINT8 byte = 0;
		M6502Stall(1);		// 1 dummy cyc

		if (M6502TotalCycles() & 1) {
			M6502Stall(1);	// 1 if starts on odd cycle
		}

		for (INT32 i = 0; i < 256; i++) {
			M6502Stall(1); // 1 for cpu_read
			byte = cpu_bus_read(data * 0x100 + i);
			M6502Stall(1); // 1 for cpu_write to ppu
			ppu_write(0x2004, byte);
		}
#if DEBUG_DMA
		bprintf(0, _T("..end-DMA, tcyc %d   scanline: %d    pixel: %d\n"), M6502TotalCycles(), scanline, pixel);
#endif
		return;
	}

	switch (address & 0xe000) {
		case 0x0000: //    0 - 1fff
			cpu_ram_write(address, data); break;
		case 0x2000: // 2000 - 3fff
			ppu_write(address, data); break;
		case 0x4000: // 4000 - 5fff
			psg_io_write(address, data); break;
		case 0x6000: // 6000 - 7fff
			prg_ram_write(address, data); break;
		default:     // 8000 - ffff
			if (mapper_write) mapper_write(address, data);
			return;
	}
}

static INT32 DrvDoReset()
{
	if (RESETMode == RESET_POWER) {
		memset(NES_CPU_RAM, 0x00, 0x800);  // only cleared @ power-on

		if (NESMode & RAM_RANDOM) { // Some games prefer random RAM @ power-on
			UINT8 Pattern[0x08] = { 0x00, 0x5a, 0x01, 0x49, 0xe5, 0xf8, 0xa5, 0x10 };

			for (INT32 i = 0; i < 0x800; i++) {
				NES_CPU_RAM[i] = Pattern[i & 0x07];
			}
		}
	}
	M6502Open(0);
	M6502Reset();
	M6502Close();
	nesapuReset();
	//	if (RESETMode == RESET_POWER) {
	// Nebs 'n Debs thinks we're in 50hz/PAL mode on reset if we don't reset PPU here..
	// note: This is actually due to a bug in the game
		ppu_reset();
	//	}
	mapper_reset();

	JoyShifter[0] = JoyShifter[1] = 0xffffffff;
	JoyStrobe = 0;
	ZapperReloadTimer = 0;

	cyc_counter = 0;
	mega_cyc_counter = 0;

	{
		INT32 nAspectX, nAspectY;
		BurnDrvGetAspect(&nAspectX, &nAspectY);
		if (DrvDips[1] & 1) { // 4:3
			if (nAspectX != 4) {
				bprintf(0, _T("*  NES: Changing to 4:3 aspect\n"));
				BurnDrvSetAspect(4, 3);
				Reinitialise();
			}
		} else { // no aspect ratio handling
			if (nAspectX != SCREEN_WIDTH) {
				bprintf(0, _T("*  NES: Changing to pixel aspect\n"));
				BurnDrvSetAspect(SCREEN_WIDTH, SCREEN_HEIGHT);
				Reinitialise();
			}
		}
	}

	if (PPUType > RP2C02) {
		HiscoreReset();
	}

	return 0;
}

static INT32 NESInit()
{
	GenericTilesInit();

	NES_CPU_RAM = (UINT8*)BurnMalloc(0x800);

	cheats_active = 0;

	struct BurnRomInfo ri;
	char *romname = NULL;
	BurnDrvGetRomInfo(&ri, 0);
	BurnDrvGetRomName(&romname, 0, 0);
	UINT32 length = ri.nLen;
	UINT32 is_FDS = (strstr(romname, ".fds") != 0);
	bprintf(0, _T("ROM Name: %S\n"), romname);
	rom = BurnMalloc((length < 0x100000) ? 0x100000 : length);

	bprintf(0, _T("ROM Length: %d\n"), ri.nLen);
	RESETMode = RESET_POWER;

	if (is_FDS) {
		if (BurnLoadRom(rom + 0x6000, 0x80, 1)) return 1; // bios @ 0xe000
		if (fds_load(rom, ri.nLen, ri.nCrc)) return 1;
	} else {
		if (BurnLoadRom(rom, 0, 1)) return 1;
		if (cartridge_load(rom, ri.nLen, ri.nCrc)) return 1;
	}

	BurnSetRefreshRate((NESMode & IS_PAL) ? 50.00 : 60.00);

	M6502Init(0, TYPE_N2A03);
	M6502Open(0);
	M6502SetWriteHandler(cpu_bus_write);
	M6502SetReadHandler(cpu_bus_read);
	M6502Close();

	ppu_init((NESMode & IS_PAL) ? 1 : 0);

	if (NESMode & IS_PAL) {
		nesapuInitPal(0, 1798773, 0);
	} else {
		nesapuInit(0, 1798773, 0);
	}
	nesapuSetAllRoutes(0, 2.40, BURN_SND_ROUTE_BOTH);

	DrvDoReset();

	return 0;
}

static INT32 NES4ScoreInit()
{
	INT32 rc = NESInit();

	NESMode |= USE_4SCORE;

	bprintf(0, _T("*  FourScore 4 Player device.\n"));

	return rc;
}

static INT32 NESHori4pInit()
{
	INT32 rc = NESInit();

	NESMode |= USE_HORI4P;

	bprintf(0, _T("*  Hori 4 Player device (Mode 2).\n"));

	return rc;
}

static INT32 NESReversedInit()
{
	INT32 rc = NESInit();

	NESMode |= VS_REVERSED;

	bprintf(0, _T("*  Inputs reversed (p1/p2 -> p2/p1).\n"));

	return rc;
}

static INT32 NESZapperInit()
{
	INT32 rc = NESInit();

	NESMode |= (PPUType == RP2C02) ? USE_ZAPPER : VS_ZAPPER;
	BurnGunInit(1, true);

	bprintf(0, _T("*  Zapper on Port #2.\n"));

	return rc;
}

// stereo effector (delay line+comb filter)
struct ms_ring {
	INT16 *l;
	INT16 *r;
	INT32 ring_size;
	INT32 inpos;
	INT32 outpos_l;
	INT32 outpos_r;

	void exit() {
		if (ring_size || l || r) {
			BurnFree(l);
			BurnFree(r);
			ring_size = 0;
			bprintf(0, _T("ms_ring exited.\n"));
		}
	}

	void init() {
		ring_size = (INT32)((double)14 / 1000 * nBurnSoundRate); // 14ms ring buffer

		l = (INT16*)BurnMalloc(ring_size * sizeof(INT16));
		r = (INT16*)BurnMalloc(ring_size * sizeof(INT16));

		for (INT32 i = 0; i < ring_size; i++) {
			write(0, 0);
		}
		inpos = 0; // position in @ beginning of ring, out @ end
		outpos_l = 1;
		outpos_r = 1;
		bprintf(0, _T("ms_ring initted (%d entry ringbuffer)\n"), ring_size);
	}

	INT32 needs_init() {
		return (l == NULL || r == NULL || ring_size == 0);
	}

	void write(INT16 sam_l, INT16 sam_r) {
		l[inpos] = sam_l;
		r[inpos] = sam_r;
		inpos = (inpos + 1) % ring_size;
	}

	INT16 read_r() {
		INT16 temp = r[outpos_r];
		outpos_r = (outpos_r + 1) % ring_size;
		return temp;
	}

	INT16 read_l() {
		INT16 temp = l[outpos_l];
		outpos_l = (outpos_l + 1) % ring_size;
		return temp;
	}

	void process(INT16 *buffer, INT32 samples) {
		if (needs_init()) {
			init();
		}

		for (INT32 i = 0; i < samples; i++) {
			write(buffer[i * 2 + 0], buffer[i * 2 + 1]);
			INT16 sam_mask = ((read_l(), read_r()) / 2) * 0.75;
			INT16 sam_now = (buffer[i * 2 + 0] + buffer[i * 2 + 1]) / 2;
			buffer[i * 2 + 0] = sam_now + sam_mask;
			buffer[i * 2 + 1] = sam_now - sam_mask;
		}
	}
};

static ms_ring ms_delay;

// end stereoeffect

static INT32 NESExit()
{
	GenericTilesExit();
	M6502Exit();
	nesapuExit();

	// Mapper EXT-hardware exits
	if (Cart.Mapper == 69) { // SunSoft fme-7 (5b) audio expansion - ay8910. mapper69
		AY8910Exit(0);
	}

	if (Cart.Mapper == 85) { // VRC7 audio expansion - ym2413
		BurnYM2413Exit();
	}

	if (Cart.FDSMode) {
		// exit saver:
		TCHAR fds_saves_patch[MAX_PATH];
		_stprintf(fds_saves_patch, _T("%s.ips"), BurnDrvGetText(DRV_NAME));
		INT32 ips = ips_make(Cart.FDSDiskRawOrig, Cart.FDSDiskRaw, Cart.FDSDiskRawSize, szAppEEPROMPath, fds_saves_patch);
		bprintf(0, _T("* FDS DISK patch: "));
		switch (ips) {
			case  0: bprintf(0, _T("Saved.\n")); break;
			case -1: bprintf(0, _T("Can't Save (File I/O Error).\n")); break;
			case -2: bprintf(0, _T("No Change.\n")); break;
		}

		BurnFree(Cart.FDSDiskRaw);
		BurnFree(Cart.FDSDiskRawOrig);

		BurnLEDExit();
	}

	if (NESMode & (USE_ZAPPER | VS_ZAPPER))
		BurnGunExit();

	BurnFree(rom);
	BurnFree(NES_CPU_RAM);
	BurnFree(Cart.WorkRAM);
	BurnFree(Cart.CHRRam);

	ms_delay.exit();

	return 0;
}

static UINT32 EmphRGB(INT32 pal_idx, UINT8 type, UINT32 *palette)
{
/*
	dink ppu color emphasis notes
	ppumask	register	(reg >> 5) (r/g swapped for PAL)

	red		20		 	1
	green	40			2
	blue	80			4

	// All possible combinations, palette offset, (reg >> 5)
	normal	0-3f		0
	r		40-7f		1
	g		80-bf		2
	rg		c0-ff		3
	b		100 - 13f	4
	rb		140 - 17f	5
	gb		180 - 1bf	6
	rgb		1c0 - 1ff	7
*/

	UINT64 er = 1.0 * (1 << 16);
	UINT64 eg = 1.0 * (1 << 16);
	UINT64 eb = 1.0 * (1 << 16);
	UINT32 EMPH   = 1.2 * (1 << 16);
	UINT32 DEEMPH = 0.8 * (1 << 16);
	if (type & 1) {
		er = (er *   EMPH) >> 16;
		eg = (eg * DEEMPH) >> 16;
		eb = (eb * DEEMPH) >> 16;
	}
	if (type & 2) {
		er = (er * DEEMPH) >> 16;
		eg = (eg *   EMPH) >> 16;
		eb = (eb * DEEMPH) >> 16;
	}
	if (type & 4) {
		er = (er * DEEMPH) >> 16;
		eg = (eg * DEEMPH) >> 16;
		eb = (eb *   EMPH) >> 16;
	}

	UINT32 r = (((palette[pal_idx & 0x3f] >> 16) & 0xff) * er) >> 16;
	if (r > 0xff) r = 0xff;
	UINT32 g = (((palette[pal_idx & 0x3f] >>  8) & 0xff) * eg) >> 16;
	if (g > 0xff) g = 0xff;
	UINT32 b = (((palette[pal_idx & 0x3f] >>  0) & 0xff) * eb) >> 16;
	if (b > 0xff) b = 0xff;

	return BurnHighCol(r, g, b, 0);
}

static void DrvCalcPalette()
{
	UpdatePalettePointer();

	// Normal NES Palette
	for (INT32 i = 0; i < 0x40; i++) {
		DrvPalette[i] = BurnHighCol((our_palette[i] >> 16) & 0xff, (our_palette[i] >> 8) & 0xff, our_palette[i] & 0xff, 0);
	}

	// Emphasized Palettes (all combinations, see comments-table in EmphRGB)
	for (INT32 i = 0x40; i < 0x200; i++) {
		DrvPalette[i] = EmphRGB(i, i >> 6, our_palette);
	}

	// Palette for the FDS Swap Disk icon
	for (INT32 fadelv = 0; fadelv < 0x10; fadelv++) {
		for (INT32 i = 0x200; i < 0x210; i++) {
			DrvPalette[i + (fadelv * 0x10)] = BurnHighCol(disk_ab_pal[(i & 0xf) * 3 + 2] / (fadelv + 1), disk_ab_pal[(i & 0xf) * 3 + 1] / (fadelv + 1), disk_ab_pal[(i & 0xf) * 3 + 0] / (fadelv + 1), 0);
		}
	}
}

static INT32 NESDraw()
{
	if (NESRecalc || last_palette != DIP_PALETTE) {
		DrvCalcPalette();
		NESRecalc = 0;
		last_palette = DIP_PALETTE;
	}

	if ((NESMode & IS_FDS) && (FDSFrameAction == SWAP_DISK || FDSFrameAction == FA_FADEOUT)) {
		static INT32 fader = 0;
		if (FDSFrameAction == SWAP_DISK) fader = 0;
		if (FDSFrameAction == FA_FADEOUT && ((FDSFrameCounter%2)==0)) {
			if (fader < 0x8)
				fader++;
		}

		switch (FDSFrameDiskIcon) {
			case 0:
				RenderCustomTile(pTransDraw, 38, 18, 0, 109/*x*/, 100/*y*/, 0, 8, 0x200 + (fader * 0x10), &disk_ab[0]);
				break;
			case 1:
				RenderCustomTile(pTransDraw, 38, 18, 0, 109/*x*/, 100/*y*/, 0, 8, 0x200 + (fader * 0x10), &disk_ab[18*38]);
				break;
		}
	}

	BurnTransferCopy(DrvPalette);

	if (NESMode & (USE_ZAPPER | VS_ZAPPER))
		BurnGunDrawTargets();

	if (NESMode & IS_FDS)
		BurnLEDRender();

	return 0;
}

static void clear_opposites(UINT8 &inpt)
{
	// some games will straight-up crash or go berzerk if up+down or left+right
	// is pressed.  This can easily happen when playing via kbd or severly
	// worn gamepad.

	if ((inpt & ( (1 << 4) | (1 << 5) )) == ((1 << 4) | (1 << 5)) )
		inpt &= ~((1 << 4) | (1 << 5)); // up + down pressed, cancel both

	if ((inpt & ( (1 << 6) | (1 << 7) )) == ((1 << 6) | (1 << 7)) )
		inpt &= ~((1 << 6) | (1 << 7)); // left + right pressed, cancel both
}

#define DEBUG_CYC 0

//#define nes_frame_cycles 29781(ntsc) 33248(pal)

INT32 nes_scanline()
{
	return scanline;
}

static INT32 NESFrame()
{
#if DEBUG_CYC
	bprintf(0, _T("enter frame:  scanline %d   pixel %d  ppucyc %d    cyc_counter %d\n"), scanline, pixel, ppu_framecycles, cyc_counter);
#endif
	if (DrvReset)
	{
		RESETMode = RESET_BUTTON;
		DrvDoReset();
	}

	{	// compile inputs!
		DrvInputs[0] = DrvInputs[1] = 0x00;
		DrvInputs[2] = DrvInputs[3] = 0x00;
		for (INT32 i = 0; i < 8; i++) {
			DrvInputs[0] ^= (NESJoy1[i] & 1) << i;
			DrvInputs[1] ^= (NESJoy2[i] & 1) << i;
			DrvInputs[2] ^= (NESJoy3[i] & 1) << i;
			DrvInputs[3] ^= (NESJoy4[i] & 1) << i;
		}

		clear_opposites(DrvInputs[0]);
		clear_opposites(DrvInputs[1]);
		clear_opposites(DrvInputs[2]);
		clear_opposites(DrvInputs[3]);

		if (NESMode & (USE_ZAPPER | VS_ZAPPER)) {
			BurnGunMakeInputs(0, ZapperX, ZapperY);

			if (ZapperReloadTimer) {
				ZapperReloadTimer--;
			}
		}

		if (NESMode & IS_FDS) {
			FDS_Insert(FDSEject);

			if (~DrvDips[0] & 2) {
				FDS_FrameTicker();
				FDS_SwapSidesAuto(FDSSwap);
			} else {
				FDS_SwapSides(FDSSwap);
			}
		}
	}

	M6502Open(0);
	M6502NewFrame();
	M6502Idle(cyc_counter);
	cyc_counter = 0;
#if DEBUG_CYC
	INT32 last_ppu = ppu_framecycles;
#endif
	if (ppu_over > 0) { // idle away extra ppu cycles
		ppu_framecycles = ppu_over;
		ppu_over = 0;
	} else {
		ppu_framecycles = 0;
	}

	for (INT32 i = 0; i < nes_frame_cycles; i++)
	{
		cyc_counter ++; // frame-based
		mega_cyc_counter ++; // "since reset"-based

		nesapu_runclock(cyc_counter - 1);  // clock dmc & external sound (n163, vrc, fds)

		if ((cyc_counter - M6502TotalCycles()) > 0)	{
			M6502Run(cyc_counter - M6502TotalCycles());
			// to debug game hard-lock: uncomment and press n 4 times to get pc.
			// tip: 99.9% its either a mapper bug or game needs ALT_TIMING flag
#if defined (FBNEO_DEBUG)
			extern int counter;
			if (counter == -4) bprintf(0, _T("PC:  %x   tc: %d   cyc_ctr: %d\n"), M6502GetPC(-1), M6502TotalCycles(), cyc_counter);
#endif
		}

		mapper_run();

		INT32 p_cyc = (cyc_counter * nes_ppu_cyc_mult) - ppu_framecycles;
		if (p_cyc > 0) {
			ppu_run(p_cyc);
		}
	}

	if (NESMode & ALT_TIMING) {
		ppu_framecycles--;
	}
	if (NESMode & IS_PAL) {
		ppu_over = ppu_framecycles - 106392;
	} else {
		ppu_over = ppu_framecycles - ((~NESMode & ALT_TIMING) ? 89342 : 89343);
	}

#if DEBUG_CYC
	INT32 cc = cyc_counter;
#endif

	cyc_counter = M6502TotalCycles() - nes_frame_cycles; // the overflow of cycles for next frame to idle away

#if DEBUG_CYC
	bprintf(0, _T("6502 cycles ran: %d   cyc_counter %d   rollover: %d    ppu.over %d   ppu.framecyc %d    last_ppu %d\n"), M6502TotalCycles(), cc, cyc_counter, ppu_over, ppu_framecycles, last_ppu);
#endif

	if (pBurnSoundOut) {
		nesapuSetDMCBitDirection(DrvDips[1] & 8);
		nesapuUpdate(0, pBurnSoundOut, nBurnSoundLen);
		BurnSoundDCFilter();

		// Mapper EXT-hardware sound renders
		if (Cart.Mapper == 69) { // SunSoft fme-7 (5b) audio expansion - ay8910. mapper69
			AY8910Render(pBurnSoundOut, nBurnSoundLen);
		}

		if (Cart.Mapper == 85) { // VRC7 audio expansion - YM2413
			BurnYM2413Render(pBurnSoundOut, nBurnSoundLen);
		}

		if (DrvDips[1] & 0x02) {
			ms_delay.process(pBurnSoundOut, nBurnSoundLen);
		}
	}
	M6502Close();

	if (pBurnDraw)
	{
		NESDraw();
	}
	return 0;
}

static INT32 NESScan(INT32 nAction, INT32 *pnMin)
{
	if (pnMin) {
		*pnMin = 0x029708;
	}

	if (nAction & ACB_VOLATILE) {
		M6502Scan(nAction);
		nesapuScan(nAction, pnMin);

		SCAN_VAR(cpu_open_bus);
		SCAN_VAR(cyc_counter);
		SCAN_VAR(JoyShifter);
		SCAN_VAR(JoyStrobe);
		SCAN_VAR(ZapperReloadTimer);

		ScanVar(NES_CPU_RAM, 0x800, "CPU Ram");
		ScanVar(Cart.WorkRAM, Cart.WorkRAMSize, "Work Ram");
		ScanVar(Cart.CHRRam, Cart.CHRRamSize, "CHR Ram");

		mapper_scan(nAction, pnMin);

		ppu_scan(nAction);

		// Mapper EXT-hardware sound scans
		if (Cart.Mapper == 69) { // SunSoft fme-7 (5b) audio expansion - ay8910. mapper69
			AY8910Scan(nAction, pnMin);
		}
		if (Cart.Mapper == 85) { // VRC7 audio expansion - YM2413
			BurnYM2413Scan(nAction, pnMin);
		}

		if (NESMode & (USE_ZAPPER | VS_ZAPPER))
			BurnGunScan();

		if (nAction & ACB_WRITE) {
			// save for later?
		}
	}

	if (nAction & ACB_NVRAM) {
		if (Cart.BatteryBackedSRAM) {
			ScanVar(Cart.WorkRAM, Cart.WorkRAMSize, "s-ram");
		}

		if (mapper_scan_cb_nvram) {
			mapper_scan_cb_nvram();
		}
	}

	return 0;
}

static INT32 NESGetZipName(char** pszName, UINT32 i)
{
	static char szFilename[MAX_PATH];
	char* pszGameName = NULL;

	if (pszName == NULL) {
		return 1;
	}

	if (i == 0) {
		pszGameName = BurnDrvGetTextA(DRV_NAME);
	} else {
		if (i == 1 && BurnDrvGetTextA(DRV_BOARDROM)) {
			pszGameName = BurnDrvGetTextA(DRV_BOARDROM);
		} else {
			pszGameName = BurnDrvGetTextA(DRV_PARENT);
		}
	}

	if (pszGameName == NULL || i > 2) {
		*pszName = NULL;
		return 1;
	}

	// remove nes_
	memset(szFilename, 0, MAX_PATH);
	for (UINT32 j = 0; j < (strlen(pszGameName) - 4); j++) {
		szFilename[j] = pszGameName[j+4];
	}

	*pszName = szFilename;

	return 0;
}

// FDS WIP.
static struct BurnRomInfo emptyRomDesc[] = {
	{ "",                    0,          0, 0 },
};

static struct BurnRomInfo fds_fdsbiosRomDesc[] = {
    { "fdsbios.nes",     0x2000, 0x5e607dcf, BRF_BIOS }, // 0x80 - standard bios
};

STD_ROM_PICK(fds_fdsbios)
STD_ROM_FN(fds_fdsbios)

struct BurnDriver BurnDrvfds_fdsbios = {
	"fds_fdsbios", NULL, NULL, NULL, "198x",
	"FDS System BIOS\0", "BIOS only", "NES", "NES",
	NULL, NULL, NULL, NULL,
	BDF_BOARDROM, 0, HARDWARE_FDS, GBF_BIOS, 0,
	NESGetZipName, fds_fdsbiosRomInfo, fds_fdsbiosRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_fdsirqtestRomDesc[] = {
	{ "fdsirqtests.fds",          65516, 0xa5a63b30, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_fdsirqtest, fds_fdsirqtest, fds_fdsbios)
STD_ROM_FN(fds_fdsirqtest)

struct BurnDriver BurnDrvfds_fdsirqtest = {
	"fds_fdsirqtest", NULL, "fds_fdsbios", NULL, "1989?",
	"FDS IRQ Test (Sour, v7)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_fdsirqtestRomInfo, fds_fdsirqtestRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_19neunzehnjRomDesc[] = {
	{ "19 - Neunzehn (Japan).fds",          131016, 0xe6e06610, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_19neunzehnj, fds_19neunzehnj, fds_fdsbios)
STD_ROM_FN(fds_19neunzehnj)

struct BurnDriver BurnDrvfds_19neunzehnj = {
	"fds_19neunzehnj", "fds_19neunzehn", "fds_fdsbios", NULL, "1988",
	"19 - Neunzehn (Japan)\0", NULL, "Soft Pro", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_STRATEGY, 0,
	NESGetZipName, fds_19neunzehnjRomInfo, fds_19neunzehnjRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_19neunzehnRomDesc[] = {
	{ "19 - Neunzehn (T-Eng).fds",          131000, 0xda11432a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_19neunzehn, fds_19neunzehn, fds_fdsbios)
STD_ROM_FN(fds_19neunzehn)

struct BurnDriver BurnDrvfds_19neunzehn = {
	"fds_19neunzehn", NULL, "fds_fdsbios", NULL, "2015",
	"19 - Neunzehn (T-Eng)\0", NULL, "MrRichard999", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_STRATEGY, 0,
	NESGetZipName, fds_19neunzehnRomInfo, fds_19neunzehnRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_adiannotsueRomDesc[] = {
	{ "Adian no Tsue (Japan).fds",          131000, 0xdcc7a8d6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_adiannotsue, fds_adiannotsue, fds_fdsbios)
STD_ROM_FN(fds_adiannotsue)

struct BurnDriver BurnDrvfds_adiannotsue = {
	"fds_adiannotsue", NULL, "fds_fdsbios", NULL, "1986",
	"Adian no Tsue (Japan)\0", NULL, "Sunsoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_QUIZ, 0,
	NESGetZipName, fds_adiannotsueRomInfo, fds_adiannotsueRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_aisenshinicolRomDesc[] = {
	{ "Ai Senshi Nicol (Japan).fds",          131000, 0x21a1ad6e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_aisenshinicol, fds_aisenshinicol, fds_fdsbios)
STD_ROM_FN(fds_aisenshinicol)

struct BurnDriver BurnDrvfds_aisenshinicol = {
	"fds_aisenshinicol", NULL, "fds_fdsbios", NULL, "1987",
	"Ai Senshi Nicol (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_RUNGUN | GBF_ADV, 0,
	NESGetZipName, fds_aisenshinicolRomInfo, fds_aisenshinicolRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_akitsunofusnokaRomDesc[] = {
	{ "Aki to Tsukasa no Fushigi no Kabe (Japan).fds",          131016, 0xd7b4a01e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_akitsunofusnoka, fds_akitsunofusnoka, fds_fdsbios)
STD_ROM_FN(fds_akitsunofusnoka)

struct BurnDriver BurnDrvfds_akitsunofusnoka = {
	"fds_akitsunofusnoka", NULL, "fds_fdsbios", NULL, "1988",
	"Aki to Tsukasa no Fushigi no Kabe (Japan)\0", NULL, "Super Pig", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_BREAKOUT | GBF_PUZZLE, 0,
	NESGetZipName, fds_akitsunofusnokaRomInfo, fds_akitsunofusnokaRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_akumajoudraculajRomDesc[] = {
	{ "Akumajou Dracula (Japan).fds",          131016, 0xffed3158, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_akumajoudraculaj, fds_akumajoudraculaj, fds_fdsbios)
STD_ROM_FN(fds_akumajoudraculaj)

struct BurnDriver BurnDrvfds_akumajoudraculaj = {
	"fds_akumajoudraculaj", "fds_akumajoudracula", "fds_fdsbios", NULL, "1986",
	"Akumajou Dracula (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_SCRFIGHT | GBF_PLATFORM, 0,
	NESGetZipName, fds_akumajoudraculajRomInfo, fds_akumajoudraculajRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_akumajoudraculaRomDesc[] = {
	{ "Akumajou Dracula (T-Eng).fds",          131016, 0xf72f22af, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_akumajoudracula, fds_akumajoudracula, fds_fdsbios)
STD_ROM_FN(fds_akumajoudracula)

struct BurnDriver BurnDrvfds_akumajoudracula = {
	"fds_akumajoudracula", NULL, "fds_fdsbios", NULL, "2020",
	"Akumajou Dracula (T-Eng)\0", NULL, "DvD Translations", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_SCRFIGHT | GBF_PLATFORM, 0,
	NESGetZipName, fds_akumajoudraculaRomInfo, fds_akumajoudraculaRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_akuusenraiRomDesc[] = {
	{ "Akuu Senki Raijin (Japan).fds",          131016, 0x5117e64d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_akuusenrai, fds_akuusenrai, fds_fdsbios)
STD_ROM_FN(fds_akuusenrai)

struct BurnDriver BurnDrvfds_akuusenrai = {
	"fds_akuusenrai", NULL, "fds_fdsbios", NULL, "1988",
	"Akuu Senki Raijin (Japan)\0", NULL, "Square", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_RUNGUN | GBF_VERSHOOT, 0,
	NESGetZipName, fds_akuusenraiRomInfo, fds_akuusenraiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_allnignipsupmabrRomDesc[] = {
	{ "All Night Nippon Super Mario Brothers (Japan).fds",          65500, 0x43e7fe95, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_allnignipsupmabr, fds_allnignipsupmabr, fds_fdsbios)
STD_ROM_FN(fds_allnignipsupmabr)

struct BurnDriver BurnDrvfds_allnignipsupmabr = {
	"fds_allnignipsupmabr", NULL, "fds_fdsbios", NULL, "1986",
	"All Night Nippon Super Mario Brothers (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PLATFORM, 0,
	NESGetZipName, fds_allnignipsupmabrRomInfo, fds_allnignipsupmabrRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_appletowmonRomDesc[] = {
	{ "Apple Town Monogatari - Little Computer People (Japan).fds",          131016, 0xbf89dd4a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_appletowmon, fds_appletowmon, fds_fdsbios)
STD_ROM_FN(fds_appletowmon)

struct BurnDriver BurnDrvfds_appletowmon = {
	"fds_appletowmon", NULL, "fds_fdsbios", NULL, "1987",
	"Apple Town Monogatari - Little Computer People (Japan)\0", NULL, "Square", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_ADV, 0,
	NESGetZipName, fds_appletowmonRomInfo, fds_appletowmonRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_armananokisekijRomDesc[] = {
	{ "Armana no Kiseki (Japan).fds",          131016, 0x0bad64a5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_armananokisekij, fds_armananokisekij, fds_fdsbios)
STD_ROM_FN(fds_armananokisekij)

struct BurnDriver BurnDrvfds_armananokisekij = {
	"fds_armananokisekij", "fds_armananokiseki", "fds_fdsbios", NULL, "1987",
	"Armana no Kiseki (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_PLATFORM, 0,
	NESGetZipName, fds_armananokisekijRomInfo, fds_armananokisekijRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_armananokisekiRomDesc[] = {
	{ "Armana no Kiseki (T-Eng).fds",          131000, 0x8b61e398, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_armananokiseki, fds_armananokiseki, fds_fdsbios)
STD_ROM_FN(fds_armananokiseki)

struct BurnDriver BurnDrvfds_armananokiseki = {
	"fds_armananokiseki", NULL, "fds_fdsbios", NULL, "2012",
	"Armana no Kiseki (T-Eng)\0", NULL, "DvD Translations", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_PLATFORM, 0,
	NESGetZipName, fds_armananokisekiRomInfo, fds_armananokisekiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_aspicjRomDesc[] = {
	{ "Aspic - Majaou no Noroi (J)(1988)(Bothtec).fds",          131000, 0x1999addb, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_aspicj, fds_aspicj, fds_fdsbios)
STD_ROM_FN(fds_aspicj)

struct BurnDriver BurnDrvfds_aspicj = {
	"fds_aspicj", "fds_aspic", "fds_fdsbios", NULL, "1988",
	"Aspic - Majaou no Noroi (Japan)\0", NULL, "Bothtec", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_RPG, 0,
	NESGetZipName, fds_aspicjRomInfo, fds_aspicjRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_aspicRomDesc[] = {
	{ "Aspic - Curse of the Snakelord T-Eng (2023)(BlackPaladin).fds",          131000, 0x9f1976dd, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_aspic, fds_aspic, fds_fdsbios)
STD_ROM_FN(fds_aspic)

struct BurnDriver BurnDrvfds_aspic = {
	"fds_aspic", NULL, "fds_fdsbios", NULL, "2023",
	"Aspic - Curse of the Snakelord (T-Eng)\0", NULL, "BlackPaladin", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_RPG, 0,
	NESGetZipName, fds_aspicRomInfo, fds_aspicRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_bakutpatkunRomDesc[] = {
	{ "Bakutoushi Patton-kun (Japan).fds",          131016, 0xbdb5104d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_bakutpatkun, fds_bakutpatkun, fds_fdsbios)
STD_ROM_FN(fds_bakutpatkun)

struct BurnDriver BurnDrvfds_bakutpatkun = {
	"fds_bakutpatkun", NULL, "fds_fdsbios", NULL, "1988",
	"Bakutoushi Patton-kun (Japan)\0", NULL, "Soft Pro", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 4, HARDWARE_FDS, GBF_ACTION, 0,
	NESGetZipName, fds_bakutpatkunRomInfo, fds_bakutpatkunRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_bigchadogspiRomDesc[] = {
	{ "Big Challenge! Dogfight Spirit (Japan).fds",          131016, 0xfcdb9fc9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_bigchadogspi, fds_bigchadogspi, fds_fdsbios)
STD_ROM_FN(fds_bigchadogspi)

struct BurnDriver BurnDrvfds_bigchadogspi = {
	"fds_bigchadogspi", NULL, "fds_fdsbios", NULL, "1988",
	"Big Challenge! Dogfight Spirit (Japan)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_VERSHOOT, 0,
	NESGetZipName, fds_bigchadogspiRomInfo, fds_bigchadogspiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_bigchagogoboRomDesc[] = {
	{ "Big Challenge! Go! Go! Bowling (Japan).fds",          131016, 0x11519f4a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_bigchagogobo, fds_bigchagogobo, fds_fdsbios)
STD_ROM_FN(fds_bigchagogobo)

struct BurnDriver BurnDrvfds_bigchagogobo = {
	"fds_bigchagogobo", NULL, "fds_fdsbios", NULL, "1989",
	"Big Challenge! Go! Go! Bowling (Japan)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 4, HARDWARE_FDS, GBF_SPORTSMISC, 0,
	NESGetZipName, fds_bigchagogoboRomInfo, fds_bigchagogoboRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_bigchagunfigRomDesc[] = {
	{ "Big Challenge! Gun Fighter (Japan).fds",          131016, 0x691cb0c1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_bigchagunfig, fds_bigchagunfig, fds_fdsbios)
STD_ROM_FN(fds_bigchagunfig)

struct BurnDriver BurnDrvfds_bigchagunfig = {
	"fds_bigchagunfig", NULL, "fds_fdsbios", NULL, "1989",
	"Big Challenge! Gun Fighter (Japan)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, fds_bigchagunfigRomInfo, fds_bigchagunfigRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_bigchajuusenRomDesc[] = {
	{ "Big Challenge! Juudou Senshuken (Japan).fds",          131016, 0xa65712a7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_bigchajuusen, fds_bigchajuusen, fds_fdsbios)
STD_ROM_FN(fds_bigchajuusen)

struct BurnDriver BurnDrvfds_bigchajuusen = {
	"fds_bigchajuusen", NULL, "fds_fdsbios", NULL, "1988",
	"Big Challenge! Juudou Senshuken (Japan)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_VSFIGHT, 0,
	NESGetZipName, fds_bigchajuusenRomInfo, fds_bigchajuusenRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_biomirbokupaRomDesc[] = {
	{ "Bio Miracle Bokutte Upa (Japan).fds",          131016, 0x75e44b0d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_biomirbokupa, fds_biomirbokupa, fds_fdsbios)
STD_ROM_FN(fds_biomirbokupa)

struct BurnDriver BurnDrvfds_biomirbokupa = {
	"fds_biomirbokupa", NULL, "fds_fdsbios", NULL, "1988",
	"Bio Miracle Bokutte Upa (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PLATFORM, 0,
	NESGetZipName, fds_biomirbokupaRomInfo, fds_biomirbokupaRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_bombermanRomDesc[] = {
	{ "Bomber Man (Japan).fds",          65516, 0x4f7bee06, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_bomberman, fds_bomberman, fds_fdsbios)
STD_ROM_FN(fds_bomberman)

struct BurnDriver BurnDrvfds_bomberman = {
	"fds_bomberman", NULL, "fds_fdsbios", NULL, "1990",
	"Bomber Man (Japan)\0", NULL, "Hudson Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_MAZE | GBF_ACTION, 0,
	NESGetZipName, fds_bombermanRomInfo, fds_bombermanRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_bubblebobbleRomDesc[] = {
	{ "Bubble Bobble (Japan).fds",          131000, 0x8c8d153e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_bubblebobble, fds_bubblebobble, fds_fdsbios)
STD_ROM_FN(fds_bubblebobble)

struct BurnDriver BurnDrvfds_bubblebobble = {
	"fds_bubblebobble", NULL, "fds_fdsbios", NULL, "1988",
	"Bubble Bobble (Japan)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_PLATFORM, 0,
	NESGetZipName, fds_bubblebobbleRomInfo, fds_bubblebobbleRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_burgertimeRomDesc[] = {
	{ "BurgerTime (Japan).fds",          65516, 0x44e41aba, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_burgertime, fds_burgertime, fds_fdsbios)
STD_ROM_FN(fds_burgertime)

struct BurnDriver BurnDrvfds_burgertime = {
	"fds_burgertime", NULL, "fds_fdsbios", NULL, "1987",
	"BurgerTime (Japan)\0", NULL, "Data East", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_ACTION, 0,
	NESGetZipName, fds_burgertimeRomInfo, fds_burgertimeRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_chitetaiordRomDesc[] = {
	{ "Chitei Tairiku Ordola (Japan).fds",          131000, 0xaacd4f64, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_chitetaiord, fds_chitetaiord, fds_fdsbios)
STD_ROM_FN(fds_chitetaiord)

struct BurnDriver BurnDrvfds_chitetaiord = {
	"fds_chitetaiord", NULL, "fds_fdsbios", NULL, "1987",
	"Chitei Tairiku Ordola (Japan)\0", NULL, "Sunsoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_chitetaiordRomInfo, fds_chitetaiordRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_cluclulandRomDesc[] = {
	{ "Clu Clu Land (Japan).fds",          65516, 0x72d0bde6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_clucluland, fds_clucluland, fds_fdsbios)
STD_ROM_FN(fds_clucluland)

struct BurnDriver BurnDrvfds_clucluland = {
	"fds_clucluland", NULL, "fds_fdsbios", NULL, "1988",
	"Clu Clu Land (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_ACTION, 0,
	NESGetZipName, fds_cluclulandRomInfo, fds_cluclulandRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_coconaworldRomDesc[] = {
	{ "Cocona World (Japan).fds",          131016, 0x0b429e6c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_coconaworld, fds_coconaworld, fds_fdsbios)
STD_ROM_FN(fds_coconaworld)

struct BurnDriver BurnDrvfds_coconaworld = {
	"fds_coconaworld", NULL, "fds_fdsbios", NULL, "1987",
	"Cocona World (Japan)\0", NULL, "Sofel", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_ADV, 0,
	NESGetZipName, fds_coconaworldRomInfo, fds_coconaworldRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_dandyjRomDesc[] = {
	{ "Dandy - Zeuon no Fukkatsu (Japan).fds",          131016, 0xa2bfcbcc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_dandyj, fds_dandyj, fds_fdsbios)
STD_ROM_FN(fds_dandyj)

struct BurnDriver BurnDrvfds_dandyj = {
	"fds_dandyj", "fds_dandy", "fds_fdsbios", NULL, "1988",
	"Dandy - Zeuon no Fukkatsu (Japan)\0", NULL, "Pony Canyon", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_RPG, 0,
	NESGetZipName, fds_dandyjRomInfo, fds_dandyjRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_dandyRomDesc[] = {
	{ "Dandy - Zeuon no Fukkatsu (T-Eng).fds",          131383, 0x29a1613f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_dandy, fds_dandy, fds_fdsbios)
STD_ROM_FN(fds_dandy)

struct BurnDriver BurnDrvfds_dandy = {
	"fds_dandy", NULL, "fds_fdsbios", NULL, "2021",
	"Dandy - Zeuon no Fukkatsu (T-Eng)\0", NULL, "Stardust Crusaders", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_RPG, 0,
	NESGetZipName, fds_dandyRomInfo, fds_dandyRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_deadzonejRomDesc[] = {
	{ "Dead Zone (Japan).fds",          131000, 0xde673254, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_deadzonej, fds_deadzonej, fds_fdsbios)
STD_ROM_FN(fds_deadzonej)

struct BurnDriver BurnDrvfds_deadzonej = {
	"fds_deadzonej", "fds_deadzone", "fds_fdsbios", NULL, "1986",
	"Dead Zone (Japan)\0", NULL, "Sunsoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_ADV, 0,
	NESGetZipName, fds_deadzonejRomInfo, fds_deadzonejRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_deadzoneRomDesc[] = {
	{ "Dead Zone (T-Eng).fds",          131000, 0x1bf2409a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_deadzone, fds_deadzone, fds_fdsbios)
STD_ROM_FN(fds_deadzone)

struct BurnDriver BurnDrvfds_deadzone = {
	"fds_deadzone", NULL, "fds_fdsbios", NULL, "2019",
	"Dead Zone (T-Eng)\0", NULL, "Stardust Crusaders", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_ADV, 0,
	NESGetZipName, fds_deadzoneRomInfo, fds_deadzoneRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_digdugRomDesc[] = {
	{ "Dig Dug (Japan).fds",          65516, 0xeb126143, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_digdug, fds_digdug, fds_fdsbios)
STD_ROM_FN(fds_digdug)

struct BurnDriver BurnDrvfds_digdug = {
	"fds_digdug", NULL, "fds_fdsbios", NULL, "1990",
	"Dig Dug (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_ACTION, 0,
	NESGetZipName, fds_digdugRomInfo, fds_digdugRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_digdugiiRomDesc[] = {
	{ "Dig Dug II (Japan).fds",          65516, 0x1b2b1342, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_digdugii, fds_digdugii, fds_fdsbios)
STD_ROM_FN(fds_digdugii)

struct BurnDriver BurnDrvfds_digdugii = {
	"fds_digdugii", NULL, "fds_fdsbios", NULL, "1990",
	"Dig Dug II (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_ACTION, 0,
	NESGetZipName, fds_digdugiiRomInfo, fds_digdugiiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_dirtypaijRomDesc[] = {
	{ "Dirty Pair - Project Eden (Japan).fds",          131016, 0x0bba3177, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_dirtypaij, fds_dirtypaij, fds_fdsbios)
STD_ROM_FN(fds_dirtypaij)

struct BurnDriver BurnDrvfds_dirtypaij = {
	"fds_dirtypaij", "fds_dirtypai", "fds_fdsbios", NULL, "1987",
	"Dirty Pair - Project Eden (Japan)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_HORSHOOT | GBF_PLATFORM, 0,
	NESGetZipName, fds_dirtypaijRomInfo, fds_dirtypaijRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_dirtypaiRomDesc[] = {
	{ "Dirty Pair - Project Eden (T-Eng).fds",          131000, 0x5707b40a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_dirtypai, fds_dirtypai, fds_fdsbios)
STD_ROM_FN(fds_dirtypai)

struct BurnDriver BurnDrvfds_dirtypai = {
	"fds_dirtypai", NULL, "fds_fdsbios", NULL, "2021",
	"Dirty Pair - Project Eden (T-Eng)\0", NULL, "Stardust Crusaders", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 2, HARDWARE_FDS, GBF_HORSHOOT | GBF_PLATFORM, 0,
	NESGetZipName, fds_dirtypaiRomInfo, fds_dirtypaiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_dokidokipanicRomDesc[] = {
	{ "Doki Doki Panic (Japan).fds",          131016, 0xb36dfcd2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_dokidokipanic, fds_dokidokipanic, fds_fdsbios)
STD_ROM_FN(fds_dokidokipanic)

struct BurnDriver BurnDrvfds_dokidokipanic = {
	"fds_dokidokipanic", NULL, "fds_fdsbios", NULL, "1987",
	"Doki Doki Panic (Japan)\0", NULL, "Nintendo - Fuji TV", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PLATFORM, 0,
	NESGetZipName, fds_dokidokipanicRomInfo, fds_dokidokipanicRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_donkeykongRomDesc[] = {
	{ "Donkey Kong (Japan).fds",          65516, 0x6c0740e9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_donkeykong, fds_donkeykong, fds_fdsbios)
STD_ROM_FN(fds_donkeykong)

struct BurnDriver BurnDrvfds_donkeykong = {
	"fds_donkeykong", NULL, "fds_fdsbios", NULL, "1988",
	"Donkey Kong (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_ACTION, 0,
	NESGetZipName, fds_donkeykongRomInfo, fds_donkeykongRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_donkeykongjrRomDesc[] = {
	{ "Donkey Kong Jr. (Japan).fds",          65516, 0x3046966a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_donkeykongjr, fds_donkeykongjr, fds_fdsbios)
STD_ROM_FN(fds_donkeykongjr)

struct BurnDriver BurnDrvfds_donkeykongjr = {
	"fds_donkeykongjr", NULL, "fds_fdsbios", NULL, "1988",
	"Donkey Kong Jr. (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_ACTION, 0,
	NESGetZipName, fds_donkeykongjrRomInfo, fds_donkeykongjrRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_doremikkoRomDesc[] = {
	{ "Doremikko (Japan)(1987)(Konami).fds",          131000, 0xbee7a47b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_doremikko, fds_doremikko, fds_fdsbios)
STD_ROM_FN(fds_doremikko)

struct BurnDriver BurnDrvfds_doremikko = {
	"fds_doremikko", NULL, "fds_fdsbios", NULL, "1987",
	"Doremikko (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_doremikkoRomInfo, fds_doremikkoRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_drchaRomDesc[] = {
	{ "Dr. Chaos - Jigoku no Tobira (Japan).fds",          131000, 0x7954bd8f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_drcha, fds_drcha, fds_fdsbios)
STD_ROM_FN(fds_drcha)

struct BurnDriver BurnDrvfds_drcha = {
	"fds_drcha", NULL, "fds_fdsbios", NULL, "1987",
	"Dr. Chaos - Jigoku no Tobira (Japan)\0", NULL, "Pony Canyon", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, fds_drchaRomInfo, fds_drchaRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_dracuiiRomDesc[] = {
	{ "Dracula II - Noroi no Fuuin (Japan).fds",          131016, 0x2d1ec77c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_dracuii, fds_dracuii, fds_fdsbios)
STD_ROM_FN(fds_dracuii)

struct BurnDriver BurnDrvfds_dracuii = {
	"fds_dracuii", NULL, "fds_fdsbios", NULL, "1987",
	"Dracula II - Noroi no Fuuin (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PLATFORM | GBF_ADV, 0,
	NESGetZipName, fds_dracuiiRomInfo, fds_dracuiiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_druidRomDesc[] = {
	{ "Druid - Kyoufu no Tobira (Japan).fds",          131016, 0xb5d4e955, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_druid, fds_druid, fds_fdsbios)
STD_ROM_FN(fds_druid)

struct BurnDriver BurnDrvfds_druid = {
	"fds_druid", "fds_druiden", "fds_fdsbios", NULL, "1988",
	"Druid - Kyoufu no Tobira (Japan)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_MAZE | GBF_RUNGUN, 0,
	NESGetZipName, fds_druidRomInfo, fds_druidRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// http://www.romhacking.net/translations/6470/
static struct BurnRomInfo fds_druidenRomDesc[] = {
	{ "Druid - Kyoufu no Tobira (T-Eng)(2022)(BlackPaladin).fds",          131000, 0x1e9dc522, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_druiden, fds_druiden, fds_fdsbios)
STD_ROM_FN(fds_druiden)

struct BurnDriver BurnDrvfds_druiden = {
	"fds_druiden", NULL, "fds_fdsbios", NULL, "2022",
	"Druid - The Gate of Fear (T-Eng)\0", NULL, "BlackPaladin", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_MAZE | GBF_RUNGUN, 0,
	NESGetZipName, fds_druidenRomInfo, fds_druidenRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_eggerlandRomDesc[] = {
	{ "Egger Land (Japan).fds",          131016, 0x84f75912, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_eggerland, fds_eggerland, fds_fdsbios)
STD_ROM_FN(fds_eggerland)

struct BurnDriver BurnDrvfds_eggerland = {
	"fds_eggerland", NULL, "fds_fdsbios", NULL, "1987",
	"Egger Land (Japan)\0", NULL, "HAL Laboratory", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PUZZLE, 0,
	NESGetZipName, fds_eggerlandRomInfo, fds_eggerlandRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_eggerlansouenotaRomDesc[] = {
	{ "Egger Land - Souzouhe no Tabidachi (Japan).fds",          131016, 0x2441b3b9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_eggerlansouenota, fds_eggerlansouenota, fds_fdsbios)
STD_ROM_FN(fds_eggerlansouenota)

struct BurnDriver BurnDrvfds_eggerlansouenota = {
	"fds_eggerlansouenota", NULL, "fds_fdsbios", NULL, "1988",
	"Egger Land - Souzouhe no Tabidachi (Japan)\0", NULL, "HAL Laboratory", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PUZZLE, 0,
	NESGetZipName, fds_eggerlansouenotaRomInfo, fds_eggerlansouenotaRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_electrician1RomDesc[] = {
	{ "Electrician (Japan).fds",          131016, 0x9c37e289, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_electrician1, fds_electrician1, fds_fdsbios)
STD_ROM_FN(fds_electrician1)

struct BurnDriver BurnDrvfds_electrician1 = {
	"fds_electrician1", "fds_electrician", "fds_fdsbios", NULL, "1986",
	"Electrician (Japan)\0", NULL, "Kemco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_ACTION, 0,
	NESGetZipName, fds_electrician1RomInfo, fds_electrician1RomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_electricianRomDesc[] = {
	{ "Electrician (T-eng).fds",          131016, 0x4aa51f48, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_electrician, fds_electrician, fds_fdsbios)
STD_ROM_FN(fds_electrician)

struct BurnDriver BurnDrvfds_electrician = {
	"fds_electrician", NULL, "fds_fdsbios", NULL, "2001",
	"Electrician (T-eng)\0", NULL, "King Mike", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_ACTION, 0,
	NESGetZipName, fds_electricianRomInfo, fds_electricianRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_esperdreamjRomDesc[] = {
	{ "Esper Dream (Japan).fds",          131000, 0x1d40f83f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_esperdreamj, fds_esperdreamj, fds_fdsbios)
STD_ROM_FN(fds_esperdreamj)

struct BurnDriver BurnDrvfds_esperdreamj = {
	"fds_esperdreamj", "fds_esperdream", "fds_fdsbios", NULL, "1987",
	"Esper Dream (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_ADV, 0,
	NESGetZipName, fds_esperdreamjRomInfo, fds_esperdreamjRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_esperdreamRomDesc[] = {
	{ "Esper Dream (T-Eng).fds",          131000, 0x51d8c9f1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_esperdream, fds_esperdream, fds_fdsbios)
STD_ROM_FN(fds_esperdream)

struct BurnDriver BurnDrvfds_esperdream = {
	"fds_esperdream", NULL, "fds_fdsbios", NULL, "2003",
	"Esper Dream (T-Eng)\0", NULL, "Mute", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_ADV, 0,
	NESGetZipName, fds_esperdreamRomInfo, fds_esperdreamRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_excitsocRomDesc[] = {
	{ "Exciting Soccer - Konami Cup (Japan).fds",          131016, 0x43ccd4ba, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_excitsoc, fds_excitsoc, fds_fdsbios)
STD_ROM_FN(fds_excitsoc)

struct BurnDriver BurnDrvfds_excitsoc = {
	"fds_excitsoc", NULL, "fds_fdsbios", NULL, "1988",
	"Exciting Soccer - Konami Cup (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_SPORTSFOOTBALL, 0,
	NESGetZipName, fds_excitsocRomInfo, fds_excitsocRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_fairypinRomDesc[] = {
	{ "Fairy Pinball - Yousei Tachi no Pinball (Japan).fds",          131016, 0x1ce10f77, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_fairypin, fds_fairypin, fds_fdsbios)
STD_ROM_FN(fds_fairypin)

struct BurnDriver BurnDrvfds_fairypin = {
	"fds_fairypin", NULL, "fds_fdsbios", NULL, "1987",
	"Fairy Pinball - Yousei Tachi no Pinball (Japan)\0", NULL, "Hacker International", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PINBALL, 0,
	NESGetZipName, fds_fairypinRomInfo, fds_fairypinRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_falsionRomDesc[] = {
	{ "Falsion (Japan).fds",          131016, 0x398bc9dc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_falsion, fds_falsion, fds_fdsbios)
STD_ROM_FN(fds_falsion)

struct BurnDriver BurnDrvfds_falsion = {
	"fds_falsion", NULL, "fds_fdsbios", NULL, "1987",
	"Falsion (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_SHOOT, 0,
	NESGetZipName, fds_falsionRomInfo, fds_falsionRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_famicgrapriRomDesc[] = {
	{ "Famicom Grand Prix - F1 Race (Japan).fds",          131016, 0x7040bf35, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_famicgrapri, fds_famicgrapri, fds_fdsbios)
STD_ROM_FN(fds_famicgrapri)

struct BurnDriver BurnDrvfds_famicgrapri = {
	"fds_famicgrapri", NULL, "fds_fdsbios", NULL, "1987",
	"Famicom Grand Prix - F1 Race (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_RACING, 0,
	NESGetZipName, fds_famicgrapriRomInfo, fds_famicgrapriRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_famicgrapriiiRomDesc[] = {
	{ "Famicom Grand Prix II - 3D Hot Rally (Japan).fds",          131016, 0x012267d8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_famicgrapriii, fds_famicgrapriii, fds_fdsbios)
STD_ROM_FN(fds_famicgrapriii)

struct BurnDriver BurnDrvfds_famicgrapriii = {
	"fds_famicgrapriii", NULL, "fds_fdsbios", NULL, "1988",
	"Famicom Grand Prix II - 3D Hot Rally (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_RACING, 0,
	NESGetZipName, fds_famicgrapriiiRomInfo, fds_famicgrapriiiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_famimdisvol1RomDesc[] = {
	{ "Famimaga Disk Vol. 1 - Hong Kong (Japan).fds",          65516, 0xc6f69987, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_famimdisvol1, fds_famimdisvol1, fds_fdsbios)
STD_ROM_FN(fds_famimdisvol1)

struct BurnDriver BurnDrvfds_famimdisvol1 = {
	"fds_famimdisvol1", NULL, "fds_fdsbios", NULL, "1990",
	"Famimaga Disk Vol. 1 - Hong Kong (Japan)\0", NULL, "Tokuma Shoten", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PUZZLE, 0,
	NESGetZipName, fds_famimdisvol1RomInfo, fds_famimdisvol1RomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_famimdisvol2RomDesc[] = {
	{ "Famimaga Disk Vol. 2 - Panic Space (Japan).fds",          131016, 0x1274645d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_famimdisvol2, fds_famimdisvol2, fds_fdsbios)
STD_ROM_FN(fds_famimdisvol2)

struct BurnDriver BurnDrvfds_famimdisvol2 = {
	"fds_famimdisvol2", NULL, "fds_fdsbios", NULL, "1990",
	"Famimaga Disk Vol. 2 - Panic Space (Japan)\0", NULL, "Tokuma Shoten", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PUZZLE, 0,
	NESGetZipName, fds_famimdisvol2RomInfo, fds_famimdisvol2RomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_famimdisvol3RomDesc[] = {
	{ "Famimaga Disk Vol. 3 - All 1 (Japan).fds",          65516, 0x936c8017, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_famimdisvol3, fds_famimdisvol3, fds_fdsbios)
STD_ROM_FN(fds_famimdisvol3)

struct BurnDriver BurnDrvfds_famimdisvol3 = {
	"fds_famimdisvol3", NULL, "fds_fdsbios", NULL, "1991",
	"Famimaga Disk Vol. 3 - All 1 (Japan)\0", NULL, "Tokuma Shoten", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PUZZLE, 0,
	NESGetZipName, fds_famimdisvol3RomInfo, fds_famimdisvol3RomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_famimdisvol4RomDesc[] = {
	{ "Famimaga Disk Vol. 4 - Clox (Japan).fds",          65516, 0xe46ffd9b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_famimdisvol4, fds_famimdisvol4, fds_fdsbios)
STD_ROM_FN(fds_famimdisvol4)

struct BurnDriver BurnDrvfds_famimdisvol4 = {
	"fds_famimdisvol4", NULL, "fds_fdsbios", NULL, "1991",
	"Famimaga Disk Vol. 4 - Clox (Japan)\0", NULL, "Tokuma Shoten", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_PUZZLE, 0,
	NESGetZipName, fds_famimdisvol4RomInfo, fds_famimdisvol4RomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_famimdisvol5RomDesc[] = {
	{ "Famimaga Disk Vol. 5 - Puyo Puyo (Japan).fds",          131016, 0xe84c7927, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_famimdisvol5, fds_famimdisvol5, fds_fdsbios)
STD_ROM_FN(fds_famimdisvol5)

struct BurnDriver BurnDrvfds_famimdisvol5 = {
	"fds_famimdisvol5", NULL, "fds_fdsbios", NULL, "1991",
	"Famimaga Disk Vol. 5 - Puyo Puyo (Japan)\0", NULL, "Tokuma Shoten", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_PUZZLE, 0,
	NESGetZipName, fds_famimdisvol5RomInfo, fds_famimdisvol5RomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_famimdisvol6RomDesc[] = {
	{ "Famimaga Disk Vol. 6 - Janken Disk Jou (Japan).fds",          131016, 0x251ddeaa, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_famimdisvol6, fds_famimdisvol6, fds_fdsbios)
STD_ROM_FN(fds_famimdisvol6)

struct BurnDriver BurnDrvfds_famimdisvol6 = {
	"fds_famimdisvol6", NULL, "fds_fdsbios", NULL, "1992",
	"Famimaga Disk Vol. 6 - Janken Disk Jou (Japan)\0", NULL, "Tokuma Shoten", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, fds_famimdisvol6RomInfo, fds_famimdisvol6RomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_finalcomRomDesc[] = {
	{ "Final Command - Akai Yousai (Japan).fds",          131016, 0xb89d5131, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_finalcom, fds_finalcom, fds_fdsbios)
STD_ROM_FN(fds_finalcom)

struct BurnDriver BurnDrvfds_finalcom = {
	"fds_finalcom", NULL, "fds_fdsbios", NULL, "1988",
	"Final Command - Akai Yousai (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_RUNGUN, 0,
	NESGetZipName, fds_finalcomRomInfo, fds_finalcomRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_firebamRomDesc[] = {
	{ "Fire Bam (Japan).fds",          131016, 0x9e5c5ede, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_firebam, fds_firebam, fds_fdsbios)
STD_ROM_FN(fds_firebam)

struct BurnDriver BurnDrvfds_firebam = {
	"fds_firebam", NULL, "fds_fdsbios", NULL, "1988",
	"Fire Bam (Japan)\0", NULL, "HAL Laboratory", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, fds_firebamRomInfo, fds_firebamRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_firerockRomDesc[] = {
	{ "Fire Rock (Japan).fds",          131016, 0xe44c7313, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_firerock, fds_firerock, fds_fdsbios)
STD_ROM_FN(fds_firerock)

struct BurnDriver BurnDrvfds_firerock = {
	"fds_firerock", NULL, "fds_fdsbios", NULL, "1988",
	"Fire Rock (Japan)\0", NULL, "System Sacom", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PLATFORM, 0,
	NESGetZipName, fds_firerockRomInfo, fds_firerockRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_fuuunankokuRomDesc[] = {
	{ "Fuuun Shourinken - Ankoku no Maou (Japan)(1988)(Jaleco).fds",          131000, 0x3c5f6468, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_fuuunankoku, fds_fuuunankoku, fds_fdsbios)
STD_ROM_FN(fds_fuuunankoku)

struct BurnDriver BurnDrvfds_fuuunankoku = {
	"fds_fuuunankoku", NULL, "fds_fdsbios", NULL, "1988",
	"Fuuun Shourin Ken - Ankoku no Maou (Japan)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_VSFIGHT | GBF_ADV, 0,
	NESGetZipName, fds_fuuunankokuRomInfo, fds_fuuunankokuRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_fuuunshokenRomDesc[] = {
	{ "Fuuun Shourin Ken (Japan).fds",          131016, 0xb5b75934, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_fuuunshoken, fds_fuuunshoken, fds_fdsbios)
STD_ROM_FN(fds_fuuunshoken)

struct BurnDriver BurnDrvfds_fuuunshoken = {
	"fds_fuuunshoken", NULL, "fds_fdsbios", NULL, "1987",
	"Fuuun Shourin Ken (Japan)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_VSFIGHT, 0,
	NESGetZipName, fds_fuuunshokenRomInfo, fds_fuuunshokenRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_galagaRomDesc[] = {
	{ "Galaga (Japan).fds",          65516, 0xa9ba7801, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_galaga, fds_galaga, fds_fdsbios)
STD_ROM_FN(fds_galaga)

struct BurnDriver BurnDrvfds_galaga = {
	"fds_galaga", NULL, "fds_fdsbios", NULL, "1990",
	"Galaga (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_SHOOT, 0,
	NESGetZipName, fds_galagaRomInfo, fds_galagaRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_galaxianRomDesc[] = {
	{ "Galaxian (Japan).fds",          65516, 0x429169f5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_galaxian, fds_galaxian, fds_fdsbios)
STD_ROM_FN(fds_galaxian)

struct BurnDriver BurnDrvfds_galaxian = {
	"fds_galaxian", NULL, "fds_fdsbios", NULL, "1990",
	"Galaxian (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_SHOOT, 0,
	NESGetZipName, fds_galaxianRomInfo, fds_galaxianRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_gallforjRomDesc[] = {
	{ "Gall Force - Eternal Story (Japan).fds",          131016, 0x51ef74fb, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_gallforj, fds_gallforj, fds_fdsbios)
STD_ROM_FN(fds_gallforj)

struct BurnDriver BurnDrvfds_gallforj = {
	"fds_gallforj", "fds_gallfor", "fds_fdsbios", NULL, "1986",
	"Gall Force - Eternal Story (Japan)\0", NULL, "HAL Laboratory", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_VERSHOOT, 0,
	NESGetZipName, fds_gallforjRomInfo, fds_gallforjRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_gallforRomDesc[] = {
	{ "Gall Force - Eternal Story (T-Eng).fds",          131000, 0xd66e007f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_gallfor, fds_gallfor, fds_fdsbios)
STD_ROM_FN(fds_gallfor)

struct BurnDriver BurnDrvfds_gallfor = {
	"fds_gallfor", NULL, "fds_fdsbios", NULL, "2008",
	"Gall Force - Eternal Story (T-Eng)\0", NULL, "Gil Galad", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_VERSHOOT, 0,
	NESGetZipName, fds_gallforRomInfo, fds_gallforRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_gingadenRomDesc[] = {
	{ "Ginga Denshou - Galaxy Odyssey (Japan).fds",          131016, 0xd38240da, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_gingaden, fds_gingaden, fds_fdsbios)
STD_ROM_FN(fds_gingaden)

struct BurnDriver BurnDrvfds_gingaden = {
	"fds_gingaden", NULL, "fds_fdsbios", NULL, "1986",
	"Ginga Denshou - Galaxy Odyssey (Japan)\0", NULL, "Imagineer", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_VERSHOOT | GBF_RPG, 0,
	NESGetZipName, fds_gingadenRomInfo, fds_gingadenRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_golfRomDesc[] = {
	{ "Golf (Japan).fds",          65516, 0x4667eafa, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_golf, fds_golf, fds_fdsbios)
STD_ROM_FN(fds_golf)

struct BurnDriver BurnDrvfds_golf = {
	"fds_golf", NULL, "fds_fdsbios", NULL, "1985",
	"Golf (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_SPORTSMISC, 0,
	NESGetZipName, fds_golfRomInfo, fds_golfRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_golfjapcouRomDesc[] = {
	{ "Golf - Japan Course (Japan).fds",          131016, 0xdbacfaaa, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_golfjapcou, fds_golfjapcou, fds_fdsbios)
STD_ROM_FN(fds_golfjapcou)

struct BurnDriver BurnDrvfds_golfjapcou = {
	"fds_golfjapcou", NULL, "fds_fdsbios", NULL, "1987",
	"Golf - Japan Course (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_SPORTSMISC, 0,
	NESGetZipName, fds_golfjapcouRomInfo, fds_golfjapcouRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_golfuscourseRomDesc[] = {
	{ "Golf - US Course (Japan).fds",          131016, 0x0ef5dae4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_golfuscourse, fds_golfuscourse, fds_fdsbios)
STD_ROM_FN(fds_golfuscourse)

struct BurnDriver BurnDrvfds_golfuscourse = {
	"fds_golfuscourse", NULL, "fds_fdsbios", NULL, "1987",
	"Golf - US Course (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_SPORTSMISC, 0,
	NESGetZipName, fds_golfuscourseRomInfo, fds_golfuscourseRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_gooniesRomDesc[] = {
	{ "Goonies (Japan).fds",          65516, 0x81b760a2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_goonies, fds_goonies, fds_fdsbios)
STD_ROM_FN(fds_goonies)

struct BurnDriver BurnDrvfds_goonies = {
	"fds_goonies", NULL, "fds_fdsbios", NULL, "1988",
	"Goonies, The (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PLATFORM, 0,
	NESGetZipName, fds_gooniesRomInfo, fds_gooniesRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_greenberetRomDesc[] = {
	{ "Green Beret (Japan).fds",          131016, 0xe7f0577b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_greenberet, fds_greenberet, fds_fdsbios)
STD_ROM_FN(fds_greenberet)

struct BurnDriver BurnDrvfds_greenberet = {
	"fds_greenberet", NULL, "fds_fdsbios", NULL, "1987",
	"Green Beret (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, fds_greenberetRomInfo, fds_greenberetRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_gunsmokeRomDesc[] = {
	{ "Gun.Smoke (Japan).fds",          131016, 0xbe761b95, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_gunsmoke, fds_gunsmoke, fds_fdsbios)
STD_ROM_FN(fds_gunsmoke)

struct BurnDriver BurnDrvfds_gunsmoke = {
	"fds_gunsmoke", NULL, "fds_fdsbios", NULL, "1988",
	"Gun.Smoke (Japan)\0", NULL, "Capcom", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_RUNGUN, 0,
	NESGetZipName, fds_gunsmokeRomInfo, fds_gunsmokeRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_gyruss1RomDesc[] = {
	{ "Gyruss (Japan).fds",          131016, 0x4552202c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_gyruss1, fds_gyruss1, fds_fdsbios)
STD_ROM_FN(fds_gyruss1)

struct BurnDriver BurnDrvfds_gyruss1 = {
	"fds_gyruss1", "fds_gyruss", "fds_fdsbios", NULL, "1988",
	"Gyruss (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_SHOOT, 0,
	NESGetZipName, fds_gyruss1RomInfo, fds_gyruss1RomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_gyrussRomDesc[] = {
	{ "Gyruss (T-eng).fds",          131000, 0xf17f2728, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_gyruss, fds_gyruss, fds_fdsbios)
STD_ROM_FN(fds_gyruss)

struct BurnDriver BurnDrvfds_gyruss = {
	"fds_gyruss", NULL, "fds_fdsbios", NULL, "2021",
	"Gyruss (T-eng)\0", NULL, "Stardust Crusader", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_SHOOT, 0,
	NESGetZipName, fds_gyrussRomInfo, fds_gyrussRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_halleywarsRomDesc[] = {
	{ "Halley Wars (Japan).fds",          131016, 0x583b11ca, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_halleywars, fds_halleywars, fds_fdsbios)
STD_ROM_FN(fds_halleywars)

struct BurnDriver BurnDrvfds_halleywars = {
	"fds_halleywars", NULL, "fds_fdsbios", NULL, "1989",
	"Halley Wars (Japan)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_VERSHOOT, 0,
	NESGetZipName, fds_halleywarsRomInfo, fds_halleywarsRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_haokunnofusnataRomDesc[] = {
	{ "Hao-kun no Fushigi na Tabi (Japan).fds",          131016, 0x085f5f88, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_haokunnofusnata, fds_haokunnofusnata, fds_fdsbios)
STD_ROM_FN(fds_haokunnofusnata)

struct BurnDriver BurnDrvfds_haokunnofusnata = {
	"fds_haokunnofusnata", NULL, "fds_fdsbios", NULL, "1987",
	"Hao-kun no Fushigi na Tabi (Japan)\0", NULL, "Square", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PLATFORM, 0,
	NESGetZipName, fds_haokunnofusnataRomInfo, fds_haokunnofusnataRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_hikarshiRomDesc[] = {
	{ "Hikari Shinwa - Palutena no Kagami (Japan).fds",          131000, 0x32516fcf, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_hikarshi, fds_hikarshi, fds_fdsbios)
STD_ROM_FN(fds_hikarshi)

struct BurnDriver BurnDrvfds_hikarshi = {
	"fds_hikarshi", NULL, "fds_fdsbios", NULL, "1986",
	"Hikari Shinwa - Palutena no Kagami (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PLATFORM, 0,
	NESGetZipName, fds_hikarshiRomInfo, fds_hikarshiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_icehockeyRomDesc[] = {
	{ "Ice Hockey (Japan).fds",          65516, 0x7bb390de, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_icehockey, fds_icehockey, fds_fdsbios)
STD_ROM_FN(fds_icehockey)

struct BurnDriver BurnDrvfds_icehockey = {
	"fds_icehockey", NULL, "fds_fdsbios", NULL, "1988",
	"Ice Hockey (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_SPORTSMISC, 0,
	NESGetZipName, fds_icehockeyRomInfo, fds_icehockeyRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_idolhljRomDesc[] = {
	{ "Idol Hotline - Nakayama Miho no Tokimeki High School (Japan)(DV10)(1987)(Nintendo).fds",          131000, 0x9c9616b8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_idolhlj, fds_idolhlj, fds_fdsbios)
STD_ROM_FN(fds_idolhlj)

struct BurnDriver BurnDrvfds_idolhlj = {
	"fds_idolhlj", "fds_idolhl", "fds_fdsbios", NULL, "1987",
	"Idol Hotline - Nakayama Miho no Tokimeki High School (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_ADV, 0,
	NESGetZipName, fds_idolhljRomInfo, fds_idolhljRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_idolhlRomDesc[] = {
	{ "Idol Hotline - Nakayama Miho no Tokimeki High School (T-Eng)(2022)(FCandChill).fds",          131000, 0x772a6bcc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_idolhl, fds_idolhl, fds_fdsbios)
STD_ROM_FN(fds_idolhl)

struct BurnDriver BurnDrvfds_idolhl = {
	"fds_idolhl", NULL, "fds_fdsbios", NULL, "2022",
	"Idol Hotline - Nakayama Miho no Tokimeki High School (T-Eng)\0", NULL, "FCandChill", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_ADV, 0,
	NESGetZipName, fds_idolhlRomInfo, fds_idolhlRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_ishidoujRomDesc[] = {
	{ "Ishidou (Japan).fds",          131000, 0x0c54a19e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_ishidouj, fds_ishidouj, fds_fdsbios)
STD_ROM_FN(fds_ishidouj)

struct BurnDriver BurnDrvfds_ishidouj = {
	"fds_ishidouj", "fds_ishidou", "fds_fdsbios", NULL, "1990",
	"Ishidou (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_PUZZLE, 0,
	NESGetZipName, fds_ishidoujRomInfo, fds_ishidoujRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_ishidouRomDesc[] = {
	{ "Ishidou (T-Eng).fds",          131000, 0xebfcf648, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_ishidou, fds_ishidou, fds_fdsbios)
STD_ROM_FN(fds_ishidou)

struct BurnDriver BurnDrvfds_ishidou = {
	"fds_ishidou", NULL, "fds_fdsbios", NULL, "2011",
	"Ishidou (T-Eng)\0", NULL, "Django", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_PUZZLE, 0,
	NESGetZipName, fds_ishidouRomInfo, fds_ishidouRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_jikaishometmagRomDesc[] = {
	{ "Jikai Shounen Met Mag (Japan).fds",          131016, 0xc8ce0200, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_jikaishometmag, fds_jikaishometmag, fds_fdsbios)
STD_ROM_FN(fds_jikaishometmag)

struct BurnDriver BurnDrvfds_jikaishometmag = {
	"fds_jikaishometmag", NULL, "fds_fdsbios", NULL, "1987",
	"Jikai Shounen Met Mag (Japan)\0", NULL, "Square - Thinking Rabbit", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PLATFORM | GBF_PUZZLE, 0,
	NESGetZipName, fds_jikaishometmagRomInfo, fds_jikaishometmagRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_kaettmarbroRomDesc[] = {
	{ "Kaettekita Mario Brothers (Japan).fds",          131016, 0x12358ded, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_kaettmarbro, fds_kaettmarbro, fds_fdsbios)
STD_ROM_FN(fds_kaettmarbro)

struct BurnDriver BurnDrvfds_kaettmarbro = {
	"fds_kaettmarbro", NULL, "fds_fdsbios", NULL, "1988",
	"Kaettekita Mario Brothers (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_ACTION, 0,
	NESGetZipName, fds_kaettmarbroRomInfo, fds_kaettmarbroRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_kalinnotsurugijRomDesc[] = {
	{ "Kalin no Tsurugi (Japan).fds",          131000, 0xc63a0059, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_kalinnotsurugij, fds_kalinnotsurugij, fds_fdsbios)
STD_ROM_FN(fds_kalinnotsurugij)

struct BurnDriver BurnDrvfds_kalinnotsurugij = {
	"fds_kalinnotsurugij", "fds_kalinnotsurugi", "fds_fdsbios", NULL, "1987",
	"Kalin no Tsurugi (Japan)\0", NULL, "Square", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_ACTION | GBF_RPG, 0,
	NESGetZipName, fds_kalinnotsurugijRomInfo, fds_kalinnotsurugijRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_kalinnotsurugiRomDesc[] = {
	{ "Kalin no Tsurugi (T-Eng).fds",          131000, 0x7825c974, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_kalinnotsurugi, fds_kalinnotsurugi, fds_fdsbios)
STD_ROM_FN(fds_kalinnotsurugi)

struct BurnDriver BurnDrvfds_kalinnotsurugi = {
	"fds_kalinnotsurugi", NULL, "fds_fdsbios", NULL, "2005",
	"Kalin no Tsurugi (T-Eng)\0", NULL, "filler, KingMike, satsu", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_ACTION | GBF_RPG, 0,
	NESGetZipName, fds_kalinnotsurugiRomInfo, fds_kalinnotsurugiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_kamenridblaRomDesc[] = {
	{ "Kamen Rider Black - Taiketsu Shadow Moon (Japan).fds",          131016, 0x0db848ea, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_kamenridbla, fds_kamenridbla, fds_fdsbios)
STD_ROM_FN(fds_kamenridbla)

struct BurnDriver BurnDrvfds_kamenridbla = {
	"fds_kamenridbla", "fds_kamenridblaen", "fds_fdsbios", NULL, "1988",
	"Kamen Rider Black - Taiketsu Shadow Moon (Japan)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_SCRFIGHT | GBF_PLATFORM, 0,
	NESGetZipName, fds_kamenridblaRomInfo, fds_kamenridblaRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_kamenridblaenRomDesc[] = {
	{ "Kamen Rider Black - Taiketsu Shadow Moon T-Eng (2022)(BlackPaladin).fds",          131016, 0x0ca67b03, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_kamenridblaen, fds_kamenridblaen, fds_fdsbios)
STD_ROM_FN(fds_kamenridblaen)

struct BurnDriver BurnDrvfds_kamenridblaen = {
	"fds_kamenridblaen", NULL, "fds_fdsbios", NULL, "2022",
	"Kamen Rider Black - Taiketsu Shadow Moon (T-Eng)\0", NULL, "BlackPaladin", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_SCRFIGHT | GBF_PLATFORM, 0,
	NESGetZipName, fds_kamenridblaenRomInfo, fds_kamenridblaenRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_kickandrunRomDesc[] = {
	{ "Kick and Run (Japan).fds",          131016, 0xa37d4c85, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_kickandrun, fds_kickandrun, fds_fdsbios)
STD_ROM_FN(fds_kickandrun)

struct BurnDriver BurnDrvfds_kickandrun = {
	"fds_kickandrun", NULL, "fds_fdsbios", NULL, "1988",
	"Kick and Run (Japan)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_SPORTSFOOTBALL, 0,
	NESGetZipName, fds_kickandrunRomInfo, fds_kickandrunRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_kickchaRomDesc[] = {
	{ "Kick Challenger - Air Foot (Japan).fds",          131016, 0xc83ef278, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_kickcha, fds_kickcha, fds_fdsbios)
STD_ROM_FN(fds_kickcha)

struct BurnDriver BurnDrvfds_kickcha = {
	"fds_kickcha", NULL, "fds_fdsbios", NULL, "1987",
	"Kick Challenger - Air Foot (Japan)\0", NULL, "Vap", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_ACTION, 0,
	NESGetZipName, fds_kickchaRomInfo, fds_kickchaRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_kietaprincessRomDesc[] = {
	{ "Kieta Princess (Japan).fds",          131000, 0x60d55cea, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_kietaprincess, fds_kietaprincess, fds_fdsbios)
STD_ROM_FN(fds_kietaprincess)

struct BurnDriver BurnDrvfds_kietaprincess = {
	"fds_kietaprincess", NULL, "fds_fdsbios", NULL, "1986",
	"Kieta Princess (Japan)\0", NULL, "Imagineer", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, fds_kietaprincessRomInfo, fds_kietaprincessRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_kikikaiRomDesc[] = {
	{ "Kiki Kaikai - Dotou Hen (Japan).fds",          131000, 0x7f446c09, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_kikikai, fds_kikikai, fds_fdsbios)
STD_ROM_FN(fds_kikikai)

struct BurnDriver BurnDrvfds_kikikai = {
	"fds_kikikai", NULL, "fds_fdsbios", NULL, "1987",
	"Kiki Kaikai - Dotou Hen (Japan)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_RUNGUN, 0,
	NESGetZipName, fds_kikikaiRomInfo, fds_kikikaiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_kinniRomDesc[] = {
	{ "Kinnikuman - Kinnikusei Oui Soudatsusen (Japan).fds",          131016, 0x5149b663, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_kinni, fds_kinni, fds_fdsbios)
STD_ROM_FN(fds_kinni)

struct BurnDriver BurnDrvfds_kinni = {
	"fds_kinni", NULL, "fds_fdsbios", NULL, "1987",
	"Kinnikuman - Kinnikusei Oui Soudatsusen (Japan)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_SCRFIGHT, 0,
	NESGetZipName, fds_kinniRomInfo, fds_kinniRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_knighlorRomDesc[] = {
	{ "Knight Lore - Majou no Ookami Otoko (Japan).fds",          131016, 0xc07a8b04, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_knighlor, fds_knighlor, fds_fdsbios)
STD_ROM_FN(fds_knighlor)

struct BurnDriver BurnDrvfds_knighlor = {
	"fds_knighlor", NULL, "fds_fdsbios", NULL, "1986",
	"Knight Lore - Majou no Ookami Otoko (Japan)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, fds_knighlorRomInfo, fds_knighlorRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_knightmovejRomDesc[] = {
	{ "Knight Move (Japan).fds",          65500, 0x6d365e2f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_knightmovej, fds_knightmovej, fds_fdsbios)
STD_ROM_FN(fds_knightmovej)

struct BurnDriver BurnDrvfds_knightmovej = {
	"fds_knightmovej", "fds_knightmove", "fds_fdsbios", NULL, "1990",
	"Knight Move (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_PUZZLE, 0,
	NESGetZipName, fds_knightmovejRomInfo, fds_knightmovejRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_knightmoveRomDesc[] = {
	{ "Knight Move (T-Eng).fds",          65500, 0x474b83c4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_knightmove, fds_knightmove, fds_fdsbios)
STD_ROM_FN(fds_knightmove)

struct BurnDriver BurnDrvfds_knightmove = {
	"fds_knightmove", NULL, "fds_fdsbios", NULL, "2003",
	"Knight Move (T-Eng)\0", NULL, "PentarouZero", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 2, HARDWARE_FDS, GBF_PUZZLE, 0,
	NESGetZipName, fds_knightmoveRomInfo, fds_knightmoveRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_konamictennisRomDesc[] = {
	{ "Konamic Tennis (Japan).fds",          131000, 0xedbaee55, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_konamictennis, fds_konamictennis, fds_fdsbios)
STD_ROM_FN(fds_konamictennis)

struct BurnDriver BurnDrvfds_konamictennis = {
	"fds_konamictennis", NULL, "fds_fdsbios", NULL, "1988",
	"Konamic Tennis (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_SPORTSMISC, 0,
	NESGetZipName, fds_konamictennisRomInfo, fds_konamictennisRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_konekmonRomDesc[] = {
	{ "Koneko Monogatari - The Adventures of Chatran (Japan).fds",          131016, 0xe3387bf0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_konekmon, fds_konekmon, fds_fdsbios)
STD_ROM_FN(fds_konekmon)

struct BurnDriver BurnDrvfds_konekmon = {
	"fds_konekmon", NULL, "fds_fdsbios", NULL, "1986",
	"Koneko Monogatari - The Adventures of Chatran (Japan)\0", NULL, "Pony Canyon", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PLATFORM, 0,
	NESGetZipName, fds_konekmonRomInfo, fds_konekmonRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_linknoboujRomDesc[] = {
	{ "Link no Bouken - The Legend of Zelda 2 (Japan).fds",          131016, 0xa118214f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_linknobouj, fds_linknobouj, fds_fdsbios)
STD_ROM_FN(fds_linknobouj)

struct BurnDriver BurnDrvfds_linknobouj = {
	"fds_linknobouj", "fds_linknobou", "fds_fdsbios", NULL, "1987",
	"Link no Bouken - The Legend of Zelda 2 (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, fds_linknoboujRomInfo, fds_linknoboujRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_linknobouRomDesc[] = {
	{ "Link no Bouken - The Legend of Zelda 2 (T-Eng).fds",          131000, 0x359d3f2b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_linknobou, fds_linknobou, fds_fdsbios)
STD_ROM_FN(fds_linknobou)

struct BurnDriver BurnDrvfds_linknobou = {
	"fds_linknobou", NULL, "fds_fdsbios", NULL, "2017",
	"Link no Bouken - The Legend of Zelda 2 (T-Eng)\0", NULL, "columbo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, fds_linknobouRomInfo, fds_linknobouRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_lutterRomDesc[] = {
	{ "Lutter (Japan).fds",          131016, 0x54335232, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_lutter, fds_lutter, fds_fdsbios)
STD_ROM_FN(fds_lutter)

struct BurnDriver BurnDrvfds_lutter = {
	"fds_lutter", NULL, "fds_fdsbios", NULL, "1989",
	"Lutter (Japan)\0", NULL, "Athena", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_ACTION | GBF_RPG, 0,
	NESGetZipName, fds_lutterRomInfo, fds_lutterRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_maerchenveilRomDesc[] = {
	{ "Maerchen Veil (Japan).fds",          131016, 0x618d10a2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_maerchenveil, fds_maerchenveil, fds_fdsbios)
STD_ROM_FN(fds_maerchenveil)

struct BurnDriver BurnDrvfds_maerchenveil = {
	"fds_maerchenveil", NULL, "fds_fdsbios", NULL, "1987",
	"Maerchen Veil (Japan)\0", NULL, "Sunsoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_ACTION | GBF_RPG, 0,
	NESGetZipName, fds_maerchenveilRomInfo, fds_maerchenveilRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_magmaproRomDesc[] = {
	{ "Magma Project - Hacker (Japan).fds",          131016, 0x59d51115, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_magmapro, fds_magmapro, fds_fdsbios)
STD_ROM_FN(fds_magmapro)

struct BurnDriver BurnDrvfds_magmapro = {
	"fds_magmapro", NULL, "fds_fdsbios", NULL, "1989",
	"Magma Project - Hacker (Japan)\0", NULL, "Bits Laboratory", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_RPG, 0,
	NESGetZipName, fds_magmaproRomInfo, fds_magmaproRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_matounohouRomDesc[] = {
	{ "Matou no Houkai - The Hero of Babel (Japan).fds",          131016, 0x0596b36a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_matounohou, fds_matounohou, fds_fdsbios)
STD_ROM_FN(fds_matounohou)

struct BurnDriver BurnDrvfds_matounohou = {
	"fds_matounohou", NULL, "fds_fdsbios", NULL, "1988",
	"Matou no Houkai - The Hero of Babel (Japan)\0", NULL, "Pony Canyon", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PLATFORM, 0,
	NESGetZipName, fds_matounohouRomInfo, fds_matounohouRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_meikyjiidabjRomDesc[] = {
	{ "Meikyuu Jiin Dababa (Japan).fds",          131000, 0x1526d32e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_meikyjiidabj, fds_meikyjiidabj, fds_fdsbios)
STD_ROM_FN(fds_meikyjiidabj)

struct BurnDriver BurnDrvfds_meikyjiidabj = {
	"fds_meikyjiidabj", "fds_meikyjiidab", "fds_fdsbios", NULL, "1987",
	"Meikyuu Jiin Dababa (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_RUNGUN, 0,
	NESGetZipName, fds_meikyjiidabjRomInfo, fds_meikyjiidabjRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_meikyjiidabRomDesc[] = {
	{ "Meikyuu Jiin Dababa (T-Eng).fds",          131000, 0x4d13e6b7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_meikyjiidab, fds_meikyjiidab, fds_fdsbios)
STD_ROM_FN(fds_meikyjiidab)

struct BurnDriver BurnDrvfds_meikyjiidab = {
	"fds_meikyjiidab", NULL, "fds_fdsbios", NULL, "2017",
	"Meikyuu Jiin Dababa (T-Eng)\0", NULL, "FlashPV", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_RUNGUN, 0,
	NESGetZipName, fds_meikyjiidabRomInfo, fds_meikyjiidabRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_metroidjRomDesc[] = {
	{ "Metroid (Japan).fds",          131000, 0x5e4f013d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_metroidj, fds_metroidj, fds_fdsbios)
STD_ROM_FN(fds_metroidj)

struct BurnDriver BurnDrvfds_metroidj = {
	"fds_metroidj", "fds_metroid", "fds_fdsbios", NULL, "1986",
	"Metroid (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, fds_metroidjRomInfo, fds_metroidjRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_metroidRomDesc[] = {
	{ "Metroid (T-Eng).fds",          131000, 0x2711adba, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_metroid, fds_metroid, fds_fdsbios)
STD_ROM_FN(fds_metroid)

struct BurnDriver BurnDrvfds_metroid = {
	"fds_metroid", NULL, "fds_fdsbios", NULL, "2016",
	"Metroid (T-Eng)\0", NULL, "Jordiway73", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, fds_metroidRomInfo, fds_metroidRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_michaengdaiRomDesc[] = {
	{ "Michael English Daibouken (Japan).fds",          131000, 0xe5ab0e6c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_michaengdai, fds_michaengdai, fds_fdsbios)
STD_ROM_FN(fds_michaengdai)

struct BurnDriver BurnDrvfds_michaengdai = {
	"fds_michaengdai", NULL, "fds_fdsbios", NULL, "1987",
	"Michael English Daibouken (Japan)\0", NULL, "Scorpion Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_ACTION, 0,
	NESGetZipName, fds_michaengdaiRomInfo, fds_michaengdaiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_moerotwiRomDesc[] = {
	{ "Moero TwinBee - Cinnamon Hakase wo Sukue! (Japan).fds",          131016, 0x0d684aa7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_moerotwi, fds_moerotwi, fds_fdsbios)
STD_ROM_FN(fds_moerotwi)

struct BurnDriver BurnDrvfds_moerotwi = {
	"fds_moerotwi", NULL, "fds_fdsbios", NULL, "1986",
	"Moero TwinBee - Cinnamon Hakase wo Sukue! (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 3, HARDWARE_FDS, GBF_HORSHOOT, 0,
	NESGetZipName, fds_moerotwiRomInfo, fds_moerotwiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_montyonrunjRomDesc[] = {
	{ "Monty on the Run - Monty no Doki Doki Dai Dassou (Japan).fds",          131016, 0xbb7ab325, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_montyonrunj, fds_montyonrunj, fds_fdsbios)
STD_ROM_FN(fds_montyonrunj)

struct BurnDriver BurnDrvfds_montyonrunj = {
	"fds_montyonrunj", "fds_montyonrun", "fds_fdsbios", NULL, "1987",
	"Monty on the Run - Monty no Doki Doki Dai Dassou (Japan)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_PLATFORM, 0,
	NESGetZipName, fds_montyonrunjRomInfo, fds_montyonrunjRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_montyonrunRomDesc[] = {
	{ "Monty on the Run - Monty no Doki Doki Dai Dassou (T-Eng).fds",          131001, 0xd7242483, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_montyonrun, fds_montyonrun, fds_fdsbios)
STD_ROM_FN(fds_montyonrun)

struct BurnDriver BurnDrvfds_montyonrun = {
	"fds_montyonrun", NULL, "fds_fdsbios", NULL, "2021",
	"Monty on the Run - Monty no Doki Doki Dai Dassou (T-Eng)\0", NULL, "Stardust Crusaders", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_PLATFORM, 0,
	NESGetZipName, fds_montyonrunRomInfo, fds_montyonrunRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_moonballmagicRomDesc[] = {
	{ "Moonball Magic (Japan).fds",          131016, 0x99a87df8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_moonballmagic, fds_moonballmagic, fds_fdsbios)
STD_ROM_FN(fds_moonballmagic)

struct BurnDriver BurnDrvfds_moonballmagic = {
	"fds_moonballmagic", NULL, "fds_fdsbios", NULL, "1988",
	"Moonball Magic (Japan)\0", NULL, "System Sacom", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PINBALL, 0,
	NESGetZipName, fds_moonballmagicRomInfo, fds_moonballmagicRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_mystemurcasRomDesc[] = {
	{ "Mysterious Murasame Castle (T-Eng).fds",          131000, 0x5da99951, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_mystemurcas, fds_mystemurcas, fds_fdsbios)
STD_ROM_FN(fds_mystemurcas)

struct BurnDriver BurnDrvfds_mystemurcas = {
	"fds_mystemurcas", NULL, "fds_fdsbios", NULL, "2011",
	"Mysterious Murasame Castle (T-Eng)\0", NULL, "Spinner 8 and friends", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_ACTION, 0,
	NESGetZipName, fds_mystemurcasRomInfo, fds_mystemurcasRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_namidaRomDesc[] = {
	{ "Namida no Soukoban Special (Japan).fds",          131000, 0x0c5b12d6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_namida, fds_namida, fds_fdsbios)
STD_ROM_FN(fds_namida)

struct BurnDriver BurnDrvfds_namida = {
	"fds_namida", NULL, "fds_fdsbios", NULL, "1986",
	"Namida no Soukoban Special (Japan)\0", NULL, "ASCII", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PUZZLE, 0,
	NESGetZipName, fds_namidaRomInfo, fds_namidaRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_nazonokabRomDesc[] = {
	{ "Nazo no Kabe - Block Kuzushi (Japan).fds",          131000, 0x09292f11, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_nazonokab, fds_nazonokab, fds_fdsbios)
STD_ROM_FN(fds_nazonokab)

struct BurnDriver BurnDrvfds_nazonokab = {
	"fds_nazonokab", NULL, "fds_fdsbios", NULL, "1986",
	"Nazo no Kabe - Block Kuzushi (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_BREAKOUT, 0,
	NESGetZipName, fds_nazonokabRomInfo, fds_nazonokabRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_nazonomurRomDesc[] = {
	{ "Nazo no Murasamejou (Japan).fds",          131016, 0x2acde825, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_nazonomur, fds_nazonomur, fds_fdsbios)
STD_ROM_FN(fds_nazonomur)

struct BurnDriver BurnDrvfds_nazonomur = {
	"fds_nazonomur", "fds_mystemurcas", "fds_fdsbios", NULL, "1986",
	"Nazo no Murasamejou (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_ACTION, 0,
	NESGetZipName, fds_nazonomurRomInfo, fds_nazonomurRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_otockyRomDesc[] = {
	{ "Otocky (Japan).fds",          131016, 0xd7a49861, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_otocky, fds_otocky, fds_fdsbios)
STD_ROM_FN(fds_otocky)

struct BurnDriver BurnDrvfds_otocky = {
	"fds_otocky", NULL, "fds_fdsbios", NULL, "1987",
	"Otocky (Japan)\0", NULL, "ASCII", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_HORSHOOT, 0,
	NESGetZipName, fds_otockyRomInfo, fds_otockyRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_patlaRomDesc[] = {
	{ "Patlabor - The Mobile Police - Dai 2 Shoutai Shutsudou Seyo! (Japan).fds",          131016, 0xb5639859, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_patla, fds_patla, fds_fdsbios)
STD_ROM_FN(fds_patla)

struct BurnDriver BurnDrvfds_patla = {
	"fds_patla", "fds_patlaen", "fds_fdsbios", NULL, "1989",
	"Patlabor - The Mobile Police - Dai 2 Shoutai Shutsudou Seyo! (Japan)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_SHOOT | GBF_PLATFORM, 0,
	NESGetZipName, fds_patlaRomInfo, fds_patlaRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_patlaenRomDesc[] = {
	{ "Patlabor - The Mobile Police - Dai 2 Shoutai Shutsudou Seyo! T-Eng (2003)(Vice Translations).fds",          131016, 0x9d8236f8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_patlaen, fds_patlaen, fds_fdsbios)
STD_ROM_FN(fds_patlaen)

struct BurnDriver BurnDrvfds_patlaen = {
	"fds_patlaen", NULL, "fds_fdsbios", NULL, "2003",
	"Patlabor - The Mobile Police - Dai 2 Shoutai Shutsudou Seyo! (T-Eng)\0", NULL, "Vice Translations", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_SHOOT | GBF_PLATFORM, 0,
	NESGetZipName, fds_patlaenRomInfo, fds_patlaenRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_progolsarkagnotoRomDesc[] = {
	{ "Pro Golfer Saru - Kage no Tournament (Japan).fds",          131016, 0x8261910e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_progolsarkagnoto, fds_progolsarkagnoto, fds_fdsbios)
STD_ROM_FN(fds_progolsarkagnoto)

struct BurnDriver BurnDrvfds_progolsarkagnoto = {
	"fds_progolsarkagnoto", NULL, "fds_fdsbios", NULL, "1987",
	"Pro Golfer Saru - Kage no Tournament (Japan)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_SPORTSMISC, 0,
	NESGetZipName, fds_progolsarkagnotoRomInfo, fds_progolsarkagnotoRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_puttputtgolfRomDesc[] = {
	{ "Putt Putt Golf (Japan).fds",          131016, 0x0ac52ae0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_puttputtgolf, fds_puttputtgolf, fds_fdsbios)
STD_ROM_FN(fds_puttputtgolf)

struct BurnDriver BurnDrvfds_puttputtgolf = {
	"fds_puttputtgolf", NULL, "fds_fdsbios", NULL, "1989",
	"Putt Putt Golf (Japan)\0", NULL, "Pack-In-Video", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_SPORTSMISC, 0,
	NESGetZipName, fds_puttputtgolfRomInfo, fds_puttputtgolfRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_puzzleboysjRomDesc[] = {
	{ "Puzzle Boys (Japan).fds",          131000, 0xb9f22094, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_puzzleboysj, fds_puzzleboysj, fds_fdsbios)
STD_ROM_FN(fds_puzzleboysj)

struct BurnDriver BurnDrvfds_puzzleboysj = {
	"fds_puzzleboysj", "fds_puzzleboys", "fds_fdsbios", NULL, "1990",
	"Puzzle Boys (Japan)\0", NULL, "Atlus", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_PUZZLE, 0,
	NESGetZipName, fds_puzzleboysjRomInfo, fds_puzzleboysjRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_puzzleboysRomDesc[] = {
	{ "Puzzle Boys (T-Eng).fds",          131000, 0xfa03eba9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_puzzleboys, fds_puzzleboys, fds_fdsbios)
STD_ROM_FN(fds_puzzleboys)

struct BurnDriver BurnDrvfds_puzzleboys = {
	"fds_puzzleboys", NULL, "fds_fdsbios", NULL, "2011",
	"Puzzle Boys (T-Eng)\0", NULL, "DvD Translations", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 2, HARDWARE_FDS, GBF_PUZZLE, 0,
	NESGetZipName, fds_puzzleboysRomInfo, fds_puzzleboysRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_radicbomjirkunRomDesc[] = {
	{ "Radical Bomber!! Jirai-kun (Japan).fds",          131016, 0x2b41056b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_radicbomjirkun, fds_radicbomjirkun, fds_fdsbios)
STD_ROM_FN(fds_radicbomjirkun)

struct BurnDriver BurnDrvfds_radicbomjirkun = {
	"fds_radicbomjirkun", NULL, "fds_fdsbios", NULL, "1988",
	"Radical Bomber!! Jirai-kun (Japan)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PUZZLE | GBF_ADV, 0,
	NESGetZipName, fds_radicbomjirkunRomInfo, fds_radicbomjirkunRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_reflectworldRomDesc[] = {
	{ "Reflect World (Japan).fds",          131016, 0x2c24cd2d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_reflectworld, fds_reflectworld, fds_fdsbios)
STD_ROM_FN(fds_reflectworld)

struct BurnDriver BurnDrvfds_reflectworld = {
	"fds_reflectworld", NULL, "fds_fdsbios", NULL, "1987",
	"Reflect World (Japan)\0", NULL, "East Cube", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_SHOOT | GBF_RPG, 0,
	NESGetZipName, fds_reflectworldRomInfo, fds_reflectworldRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_relicsRomDesc[] = {
	{ "Relics - Ankoku Yousai (Japan).fds",          131016, 0xfb0ae491, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_relics, fds_relics, fds_fdsbios)
STD_ROM_FN(fds_relics)

struct BurnDriver BurnDrvfds_relics = {
	"fds_relics", NULL, "fds_fdsbios", NULL, "1987",
	"Relics - Ankoku Yousai (Japan)\0", NULL, "Bothtec", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_SCRFIGHT | GBF_PLATFORM, 0,
	NESGetZipName, fds_relicsRomInfo, fds_relicsRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_replicartRomDesc[] = {
	{ "Replicart (Japan).fds",          131016, 0xc3cbc847, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_replicart, fds_replicart, fds_fdsbios)
STD_ROM_FN(fds_replicart)

struct BurnDriver BurnDrvfds_replicart = {
	"fds_replicart", NULL, "fds_fdsbios", NULL, "1988",
	"Replicart (Japan)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_ACTION, 0,
	NESGetZipName, fds_replicartRomInfo, fds_replicartRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_rogerrabbitRomDesc[] = {
	{ "Roger Rabbit (Japan).fds",          131016, 0xb43f9b16, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_rogerrabbit, fds_rogerrabbit, fds_fdsbios)
STD_ROM_FN(fds_rogerrabbit)

struct BurnDriver BurnDrvfds_rogerrabbit = {
	"fds_rogerrabbit", NULL, "fds_fdsbios", NULL, "1989",
	"Roger Rabbit (Japan)\0", NULL, "Kemco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PLATFORM, 0,
	NESGetZipName, fds_rogerrabbitRomInfo, fds_rogerrabbitRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_samswordRomDesc[] = {
	{ "Samurai Sword (Japan)(1988)(Capcom).fds",          131016, 0x5ca021ff, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_samsword, fds_samsword, fds_fdsbios)
STD_ROM_FN(fds_samsword)

struct BurnDriver BurnDrvfds_samsword = {
	"fds_samsword", "fds_samsworden", "fds_fdsbios", NULL, "1988",
	"Samurai Sword (Japan)\0", NULL, "Capcom", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_ADV, 0,
	NESGetZipName, fds_samswordRomInfo, fds_samswordRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_samswordenRomDesc[] = {
	{ "Samurai Sword (T-Eng)(2003)(Mute).fds",          131016, 0x45ddd0f4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_samsworden, fds_samsworden, fds_fdsbios)
STD_ROM_FN(fds_samsworden)

struct BurnDriver BurnDrvfds_samsworden = {
	"fds_samsworden", NULL, "fds_fdsbios", NULL, "2003",
	"Samurai Sword (T-Eng)\0", NULL, "Mute", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_ADV, 0,
	NESGetZipName, fds_samswordenRomInfo, fds_samswordenRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_sdgunworgacsescwaRomDesc[] = {
	{ "SD Gundam World - Gachapon Senshi - Scramble Wars (Japan).fds",          131000, 0x9d287c3f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_sdgunworgacsescwa, fds_sdgunworgacsescwa, fds_fdsbios)
STD_ROM_FN(fds_sdgunworgacsescwa)

struct BurnDriver BurnDrvfds_sdgunworgacsescwa = {
	"fds_sdgunworgacsescwa", NULL, "fds_fdsbios", NULL, "1987",
	"SD Gundam World - Gachapon Senshi - Scramble Wars (Japan)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_STRATEGY, 0,
	NESGetZipName, fds_sdgunworgacsescwaRomInfo, fds_sdgunworgacsescwaRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_sectionzRomDesc[] = {
	{ "Section-Z (Japan).fds",          131016, 0x392488e2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_sectionz, fds_sectionz, fds_fdsbios)
STD_ROM_FN(fds_sectionz)

struct BurnDriver BurnDrvfds_sectionz = {
	"fds_sectionz", NULL, "fds_fdsbios", NULL, "1987",
	"Section-Z (Japan)\0", NULL, "Capcom", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_HORSHOOT, 0,
	NESGetZipName, fds_sectionzRomInfo, fds_sectionzRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_seikepsyjRomDesc[] = {
	{ "Seiken Psychocalibur - Majuu no Mori Densetsu (Japan).fds",          131000, 0xaea5691a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_seikepsyj, fds_seikepsyj, fds_fdsbios)
STD_ROM_FN(fds_seikepsyj)

struct BurnDriver BurnDrvfds_seikepsyj = {
	"fds_seikepsyj", "fds_seikepsy", "fds_fdsbios", NULL, "1987",
	"Seiken Psychocalibur - Majuu no Mori Densetsu (Japan)\0", NULL, "Imagineer", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_ACTION | GBF_RPG, 0,
	NESGetZipName, fds_seikepsyjRomInfo, fds_seikepsyjRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_seikepsyRomDesc[] = {
	{ "Seiken Psychocalibur - Majuu no Mori Densetsu (T-Eng).fds",          131000, 0x1ec1552b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_seikepsy, fds_seikepsy, fds_fdsbios)
STD_ROM_FN(fds_seikepsy)

struct BurnDriver BurnDrvfds_seikepsy = {
	"fds_seikepsy", NULL, "fds_fdsbios", NULL, "20??",
	"Seiken Psychocalibur - Majuu no Mori Densetsu (T-Eng)\0", NULL, "The Spoony Bard", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_ACTION | GBF_RPG, 0,
	NESGetZipName, fds_seikepsyRomInfo, fds_seikepsyRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_sexyinvadersRomDesc[] = {
	{ "Sexy Invaders (Japan).fds",          131000, 0x413e49da, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_sexyinvaders, fds_sexyinvaders, fds_fdsbios)
STD_ROM_FN(fds_sexyinvaders)

struct BurnDriver BurnDrvfds_sexyinvaders = {
	"fds_sexyinvaders", NULL, "fds_fdsbios", NULL, "1990",
	"Sexy Invaders (Japan)\0", NULL, "Super Pig", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_SHOOT, 0,
	NESGetZipName, fds_sexyinvadersRomInfo, fds_sexyinvadersRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_silvijRomDesc[] = {
	{ "Silviana - Ai Ippai no Little Angel (Japan).fds",          131016, 0x4d721d90, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_silvij, fds_silvij, fds_fdsbios)
STD_ROM_FN(fds_silvij)

struct BurnDriver BurnDrvfds_silvij = {
	"fds_silvij", "fds_silvi", "fds_fdsbios", NULL, "1988",
	"Silviana - Ai Ippai no Little Angel (Japan)\0", NULL, "Pack-In-Video", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, fds_silvijRomInfo, fds_silvijRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_silviRomDesc[] = {
	{ "Silviana - Ai Ippai no Little Angel (T-Eng).fds",          131000, 0xc5b1bff7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_silvi, fds_silvi, fds_fdsbios)
STD_ROM_FN(fds_silvi)

struct BurnDriver BurnDrvfds_silvi = {
	"fds_silvi", NULL, "fds_fdsbios", NULL, "2002",
	"Silviana - Ai Ippai no Little Angel (T-Eng)\0", NULL, "Mute", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, fds_silviRomInfo, fds_silviRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_smashpingpongRomDesc[] = {
	{ "Smash Ping Pong (Japan).fds",          131016, 0xcc1eb0dc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_smashpingpong, fds_smashpingpong, fds_fdsbios)
STD_ROM_FN(fds_smashpingpong)

struct BurnDriver BurnDrvfds_smashpingpong = {
	"fds_smashpingpong", NULL, "fds_fdsbios", NULL, "1987",
	"Smash Ping Pong (Japan)\0", NULL, "Nintendo - Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_SPORTSMISC, 0,
	NESGetZipName, fds_smashpingpongRomInfo, fds_smashpingpongRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_solomonnokagiRomDesc[] = {
	{ "Solomon no Kagi (Japan).fds",          65500, 0x5e298400, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_solomonnokagi, fds_solomonnokagi, fds_fdsbios)
STD_ROM_FN(fds_solomonnokagi)

struct BurnDriver BurnDrvfds_solomonnokagi = {
	"fds_solomonnokagi", NULL, "fds_fdsbios", NULL, "1987",
	"Solomon no Kagi (Japan)\0", NULL, "Tecmo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PLATFORM | GBF_PUZZLE, 0,
	NESGetZipName, fds_solomonnokagiRomInfo, fds_solomonnokagiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_superboyallanRomDesc[] = {
	{ "Super Boy Allan (Japan).fds",          131016, 0xfdc53496, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_superboyallan, fds_superboyallan, fds_fdsbios)
STD_ROM_FN(fds_superboyallan)

struct BurnDriver BurnDrvfds_superboyallan = {
	"fds_superboyallan", NULL, "fds_fdsbios", NULL, "1987",
	"Super Boy Allan (Japan)\0", NULL, "Sunsoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_ACTION | GBF_PUZZLE, 0,
	NESGetZipName, fds_superboyallanRomInfo, fds_superboyallanRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_superlodrunRomDesc[] = {
	{ "Super Lode Runner (Japan).fds",          131000, 0xe66f4682, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_superlodrun, fds_superlodrun, fds_fdsbios)
STD_ROM_FN(fds_superlodrun)

struct BurnDriver BurnDrvfds_superlodrun = {
	"fds_superlodrun", NULL, "fds_fdsbios", NULL, "1987",
	"Super Lode Runner (Japan)\0", NULL, "Irem", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_ACTION, 0,
	NESGetZipName, fds_superlodrunRomInfo, fds_superlodrunRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_superlodruniiRomDesc[] = {
	{ "Super Lode Runner II (Japan).fds",          131000, 0x758888f4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_superlodrunii, fds_superlodrunii, fds_fdsbios)
STD_ROM_FN(fds_superlodrunii)

struct BurnDriver BurnDrvfds_superlodrunii = {
	"fds_superlodrunii", NULL, "fds_fdsbios", NULL, "1987",
	"Super Lode Runner II (Japan)\0", NULL, "Irem", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_ACTION, 0,
	NESGetZipName, fds_superlodruniiRomInfo, fds_superlodruniiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_supermarbroRomDesc[] = {
	{ "Super Mario Brothers (Japan).fds",          65500, 0xe0890cf2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_supermarbro, fds_supermarbro, fds_fdsbios)
STD_ROM_FN(fds_supermarbro)

struct BurnDriver BurnDrvfds_supermarbro = {
	"fds_supermarbro", NULL, "fds_fdsbios", NULL, "1986",
	"Super Mario Brothers (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_PLATFORM, 0,
	NESGetZipName, fds_supermarbroRomInfo, fds_supermarbroRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_supermarbro2RomDesc[] = {
	{ "Super Mario Brothers 2 (Japan).fds",          131016, 0x443ed7b3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_supermarbro2, fds_supermarbro2, fds_fdsbios)
STD_ROM_FN(fds_supermarbro2)

struct BurnDriver BurnDrvfds_supermarbro2 = {
	"fds_supermarbro2", NULL, "fds_fdsbios", NULL, "1986",
	"Super Mario Brothers 2 (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PLATFORM, 0,
	NESGetZipName, fds_supermarbro2RomInfo, fds_supermarbro2RomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_tamafriRomDesc[] = {
	{ "Tama & Friends - 3 Choume Daibouken (Japan).fds",          131016, 0x8751a949, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_tamafri, fds_tamafri, fds_fdsbios)
STD_ROM_FN(fds_tamafri)

struct BurnDriver BurnDrvfds_tamafri = {
	"fds_tamafri", NULL, "fds_fdsbios", NULL, "1989",
	"Tama & Friends - 3 Choume Daibouken (Japan)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PLATFORM, 0,
	NESGetZipName, fds_tamafriRomInfo, fds_tamafriRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_tennisRomDesc[] = {
	{ "Tennis (Japan).fds",          65516, 0x136c64ed, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_tennis, fds_tennis, fds_fdsbios)
STD_ROM_FN(fds_tennis)

struct BurnDriver BurnDrvfds_tennis = {
	"fds_tennis", NULL, "fds_fdsbios", NULL, "1986",
	"Tennis (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_SPORTSMISC, 0,
	NESGetZipName, fds_tennisRomInfo, fds_tennisRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_tobiddaiRomDesc[] = {
	{ "Tobidase Daisakusen (Japan).fds",          131016, 0x95546107, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_tobiddai, fds_tobiddai, fds_fdsbios)
STD_ROM_FN(fds_tobiddai)

struct BurnDriver BurnDrvfds_tobiddai = {
	"fds_tobiddai", NULL, "fds_fdsbios", NULL, "1987",
	"Tobidase Daisakusen (Japan)\0", NULL, "Square", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_ACTION, 0,
	NESGetZipName, fds_tobiddaiRomInfo, fds_tobiddaiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_topplezipRomDesc[] = {
	{ "Topple Zip (Japan).fds",          131016, 0x7bf73c75, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_topplezip, fds_topplezip, fds_fdsbios)
STD_ROM_FN(fds_topplezip)

struct BurnDriver BurnDrvfds_topplezip = {
	"fds_topplezip", NULL, "fds_fdsbios", NULL, "1987",
	"Topple Zip (Japan)\0", NULL, "Bothtec", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_VERSHOOT, 0,
	NESGetZipName, fds_topplezipRomInfo, fds_topplezipRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_transRomDesc[] = {
	{ "Transformers - The Headmasters (Japan).fds",          131016, 0x8bd904cd, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_trans, fds_trans, fds_fdsbios)
STD_ROM_FN(fds_trans)

struct BurnDriver BurnDrvfds_trans = {
	"fds_trans", NULL, "fds_fdsbios", NULL, "1987",
	"Transformers - The Headmasters (Japan)\0", NULL, "Takara", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_HORSHOOT, 0,
	NESGetZipName, fds_transRomInfo, fds_transRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_twinbeeRomDesc[] = {
	{ "TwinBee (Japan).fds",          65500, 0x517be396, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_twinbee, fds_twinbee, fds_fdsbios)
STD_ROM_FN(fds_twinbee)

struct BurnDriver BurnDrvfds_twinbee = {
	"fds_twinbee", NULL, "fds_fdsbios", NULL, "1988",
	"TwinBee (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_VERSHOOT, 0,
	NESGetZipName, fds_twinbeeRomInfo, fds_twinbeeRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_ultraRomDesc[] = {
	{ "Ultraman - Kaijuu Teikoku no Gyakushuu (Japan).fds",          131016, 0x2ab0c9bd, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_ultra, fds_ultra, fds_fdsbios)
STD_ROM_FN(fds_ultra)

struct BurnDriver BurnDrvfds_ultra = {
	"fds_ultra", NULL, "fds_fdsbios", NULL, "1986",
	"Ultraman - Kaijuu Teikoku no Gyakushuu (Japan)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PLATFORM | GBF_SCRFIGHT, 0,
	NESGetZipName, fds_ultraRomInfo, fds_ultraRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_ultra2RomDesc[] = {
	{ "Ultraman 2 - Shutsugeki Katoku Tai (Japan).fds",          131016, 0x4010e726, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_ultra2, fds_ultra2, fds_fdsbios)
STD_ROM_FN(fds_ultra2)

struct BurnDriver BurnDrvfds_ultra2 = {
	"fds_ultra2", NULL, "fds_fdsbios", NULL, "1987",
	"Ultraman 2 - Shutsugeki Katoku Tai (Japan)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_ACTION, 0,
	NESGetZipName, fds_ultra2RomInfo, fds_ultra2RomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_ultracluRomDesc[] = {
	{ "Ultraman Club - Chikyuu Dakkan Sakusen (Japan).fds",          131000, 0xfd150e81, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_ultraclu, fds_ultraclu, fds_fdsbios)
STD_ROM_FN(fds_ultraclu)

struct BurnDriver BurnDrvfds_ultraclu = {
	"fds_ultraclu", NULL, "fds_fdsbios", NULL, "1988",
	"Ultraman Club - Chikyuu Dakkan Sakusen (Japan)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_ACTION | GBF_RPG, 0,
	NESGetZipName, fds_ultracluRomInfo, fds_ultracluRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_volleyballRomDesc[] = {
	{ "Volleyball (Japan).fds",          65516, 0x0533c149, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_volleyball, fds_volleyball, fds_fdsbios)
STD_ROM_FN(fds_volleyball)

struct BurnDriver BurnDrvfds_volleyball = {
	"fds_volleyball", NULL, "fds_fdsbios", NULL, "1986",
	"Volleyball (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_SPORTSMISC, 0,
	NESGetZipName, fds_volleyballRomInfo, fds_volleyballRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_vsexcitebikeRomDesc[] = {
	{ "Vs. Excitebike (Japan).fds",          131016, 0xb8a7c0a9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_vsexcitebike, fds_vsexcitebike, fds_fdsbios)
STD_ROM_FN(fds_vsexcitebike)

struct BurnDriver BurnDrvfds_vsexcitebike = {
	"fds_vsexcitebike", NULL, "fds_fdsbios", NULL, "1988",
	"Vs. Excitebike (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_ACTION, 0,
	NESGetZipName, fds_vsexcitebikeRomInfo, fds_vsexcitebikeRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_wakusatogaiRomDesc[] = {
	{ "Wakusei Aton Gaiden (Japan).fds",          131016, 0x94a825fa, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_wakusatogai, fds_wakusatogai, fds_fdsbios)
STD_ROM_FN(fds_wakusatogai)

struct BurnDriver BurnDrvfds_wakusatogai = {
	"fds_wakusatogai", NULL, "fds_fdsbios", NULL, "1990",
	"Wakusei Aton Gaiden (Japan)\0", NULL, "Kokuzeichou", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_VERSHOOT, 0,
	NESGetZipName, fds_wakusatogaiRomInfo, fds_wakusatogaiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_wardnernomoriRomDesc[] = {
	{ "Wardner no Mori (Japan).fds",          131016, 0xb5a04f7e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_wardnernomori, fds_wardnernomori, fds_fdsbios)
STD_ROM_FN(fds_wardnernomori)

struct BurnDriver BurnDrvfds_wardnernomori = {
	"fds_wardnernomori", NULL, "fds_fdsbios", NULL, "1988",
	"Wardner no Mori (Japan)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_PLATFORM, 0,
	NESGetZipName, fds_wardnernomoriRomInfo, fds_wardnernomoriRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_wintergamesRomDesc[] = {
	{ "Winter Games (Japan).fds",          131000, 0x6ab5e1d6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_wintergames, fds_wintergames, fds_fdsbios)
STD_ROM_FN(fds_wintergames)

struct BurnDriver BurnDrvfds_wintergames = {
	"fds_wintergames", NULL, "fds_fdsbios", NULL, "1987",
	"Winter Games (Japan)\0", NULL, "Pony Canyon", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_SPORTSMISC, 0,
	NESGetZipName, fds_wintergamesRomInfo, fds_wintergamesRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_wreckingcrewRomDesc[] = {
	{ "Wrecking Crew (Japan).fds",          65516, 0xf341f135, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_wreckingcrew, fds_wreckingcrew, fds_fdsbios)
STD_ROM_FN(fds_wreckingcrew)

struct BurnDriver BurnDrvfds_wreckingcrew = {
	"fds_wreckingcrew", NULL, "fds_fdsbios", NULL, "1988",
	"Wrecking Crew (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_PLATFORM, 0,
	NESGetZipName, fds_wreckingcrewRomInfo, fds_wreckingcrewRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_youkaiyashikijRomDesc[] = {
	{ "Youkai Yashiki (Japan).fds",          131016, 0xa888c6df, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_youkaiyashikij, fds_youkaiyashikij, fds_fdsbios)
STD_ROM_FN(fds_youkaiyashikij)

struct BurnDriver BurnDrvfds_youkaiyashikij = {
	"fds_youkaiyashikij", "fds_youkaiyashiki", "fds_fdsbios", NULL, "1987",
	"Youkai Yashiki (Japan)\0", NULL, "Irem", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_PLATFORM, 0,
	NESGetZipName, fds_youkaiyashikijRomInfo, fds_youkaiyashikijRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_youkaiyashikiRomDesc[] = {
	{ "Youkai Yashiki (T-Eng).fds",          131000, 0x546589ab, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_youkaiyashiki, fds_youkaiyashiki, fds_fdsbios)
STD_ROM_FN(fds_youkaiyashiki)

struct BurnDriver BurnDrvfds_youkaiyashiki = {
	"fds_youkaiyashiki", NULL, "fds_fdsbios", NULL, "2008",
	"Youkai Yashiki (T-Eng)\0", NULL, "KingMike's Translations", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_PLATFORM, 0,
	NESGetZipName, fds_youkaiyashikiRomInfo, fds_youkaiyashikiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_yuumazeRomDesc[] = {
	{ "Yuu Maze (Japan).fds",          131016, 0xde297106, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_yuumaze, fds_yuumaze, fds_fdsbios)
STD_ROM_FN(fds_yuumaze)

struct BurnDriver BurnDrvfds_yuumaze = {
	"fds_yuumaze", NULL, "fds_fdsbios", NULL, "1988",
	"Yuu Maze (Japan)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MAZE | GBF_ACTION, 0,
	NESGetZipName, fds_yuumazeRomInfo, fds_yuumazeRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_zanacRomDesc[] = {
	{ "Zanac (Japan).fds",          65516, 0xdaa74bc4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_zanac, fds_zanac, fds_fdsbios)
STD_ROM_FN(fds_zanac)

struct BurnDriver BurnDrvfds_zanac = {
	"fds_zanac", NULL, "fds_fdsbios", NULL, "1986",
	"Zanac (Japan)\0", NULL, "Pony Canyon", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_FDS, GBF_VERSHOOT, 0,
	NESGetZipName, fds_zanacRomInfo, fds_zanacRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_zeldanodenjRomDesc[] = {
	{ "Zelda no Densetsu - The Hyrule Fantasy (Japan).fds",          131016, 0x3fbdddcd, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_zeldanodenj, fds_zeldanodenj, fds_fdsbios)
STD_ROM_FN(fds_zeldanodenj)

struct BurnDriver BurnDrvfds_zeldanodenj = {
	"fds_zeldanodenj", "fds_zeldanoden", "fds_fdsbios", NULL, "1986",
	"Zelda no Densetsu - The Hyrule Fantasy (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_FDS, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, fds_zeldanodenjRomInfo, fds_zeldanodenjRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_zeldanodenRomDesc[] = {
	{ "Zelda no Densetsu - The Hyrule Fantasy (T-Eng).fds",          131016, 0xee11aa63, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_zeldanoden, fds_zeldanoden, fds_fdsbios)
STD_ROM_FN(fds_zeldanoden)

struct BurnDriver BurnDrvfds_zeldanoden = {
	"fds_zeldanoden", NULL, "fds_fdsbios", NULL, "2016",
	"Zelda no Densetsu - The Hyrule Fantasy (T-Eng)\0", NULL, "Jordiway73", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, fds_zeldanodenRomInfo, fds_zeldanodenRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};


// FDS END ---------------------------------------------------------------


// CPU TESTS
/*
#define WINNERS_DONT_USE_DRUGS 0
#if WINNERS_DONT_USE_DRUGS

static struct BurnRomInfo nestestRomDesc[] = {
	{ "nestest.nes",          24592, 0x9e179d92, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(nestest)
STD_ROM_FN(nestest)

struct BurnDriver BurnDrvnestest = {
	"nes_nestest", NULL, NULL, NULL, "1986",
	"nestest\0", NULL, "Kevtris", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName, nestestRomInfo, nestestRomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};


// PPU Tests

static struct BurnRomInfo nes_fullRomDesc[] = {
	{ "full_nes_palette2.nes",          16400, 0x85784e11, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_full)
STD_ROM_FN(nes_full)

struct BurnDriver BurnDrvnes_full = {
	"nes_full", NULL, NULL, NULL, "1989?",
	"full nes palette (Blargg)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_fullRomInfo, nes_fullRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo bpaltestRomDesc[] = {
	{ "palette_ram.nes",          16400, 0x5d447e3c, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bpaltest)
STD_ROM_FN(bpaltest)

struct BurnDriver BurnDrvbpaltest = {
	"nes_blarggppu", NULL, NULL, NULL, "1986",
	"Blargg pal ram test\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName, bpaltestRomInfo, bpaltestRomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo bpalpowertestRomDesc[] = {
	{ "power_up_palette.nes",          16400, 0x156f41f0, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bpalpowertest)
STD_ROM_FN(bpalpowertest)

struct BurnDriver BurnDrvbpalpowertest = {
	"nes_blarggppu", NULL, NULL, NULL, "1986",
	"Blargg pal power on test\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName, bpalpowertestRomInfo, bpalpowertestRomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo bspriteramRomDesc[] = {
	{ "sprite_ram.nes",          16400, 0xd8d42111, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bspriteram)
STD_ROM_FN(bspriteram)

struct BurnDriver BurnDrvbspriteram = {
	"nes_blarggppu", NULL, NULL, NULL, "1986",
	"Blargg sprite ram\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName, bspriteramRomInfo, bspriteramRomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo bvblRomDesc[] = {
	{ "vbl_clear_time.nes",          16400, 0x1e381801, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bvbl)
STD_ROM_FN(bvbl)

struct BurnDriver BurnDrvbvbl = {
	"nes_blarggppu", NULL, NULL, NULL, "1986",
	"VBL test\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName,bvblRomInfo, bvblRomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo bvramRomDesc[] = {
	{ "vram_access.nes",          16400, 0xee115c9a, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bvram)
STD_ROM_FN(bvram)

struct BurnDriver BurnDrvbbvram = {
	"nes_blarggppu", NULL, NULL, NULL, "1986",
	"VRAM test\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName,bvramRomInfo, bvramRomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};


static struct BurnRomInfo bsprite1RomDesc[] = {
	{ "01.basics.nes",          16400, 0xee115c9a, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bsprite1)
STD_ROM_FN(bsprite1)

struct BurnDriver BurnDrvbsprite1 = {
	"nes_bsprites", NULL, NULL, NULL, "1986",
	"sprite test 1\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName,bsprite1RomInfo, bsprite1RomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo bsprite2RomDesc[] = {
	{ "02.alignment.nes",          16400, 0xee115c9a, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bsprite2)
STD_ROM_FN(bsprite2)

struct BurnDriver BurnDrvbsprite2 = {
	"nes_bsprites", NULL, NULL, NULL, "1986",
	"sprite test 2\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName,bsprite2RomInfo, bsprite2RomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo bsprite3RomDesc[] = {
	{ "03.corners.nes",          16400, 0xee115c9a, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bsprite3)
STD_ROM_FN(bsprite3)

struct BurnDriver BurnDrvbsprite3 = {
	"nes_bsprites", NULL, NULL, NULL, "1986",
	"sprite test 3\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName,bsprite3RomInfo, bsprite3RomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};


static struct BurnRomInfo bsprite4RomDesc[] = {
	{ "04.flip.nes",          16400, 0xee115c9a, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bsprite4)
STD_ROM_FN(bsprite4)

struct BurnDriver BurnDrvbsprite4 = {
	"nes_bsprites", NULL, NULL, NULL, "1986",
	"sprite test 4\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName,bsprite4RomInfo, bsprite4RomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo bsprite5RomDesc[] = {
	{ "05.left_clip.nes",          16400, 0xee115c9a, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bsprite5)
STD_ROM_FN(bsprite5)

struct BurnDriver BurnDrvbsprite5 = {
	"nes_bsprites", NULL, NULL, NULL, "1986",
	"sprite test 5\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName,bsprite5RomInfo, bsprite5RomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo bsprite6RomDesc[] = {
	{ "06.right_edge.nes",          16400, 0xee115c9a, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bsprite6)
STD_ROM_FN(bsprite6)

struct BurnDriver BurnDrvbsprite6 = {
	"nes_bsprites", NULL, NULL, NULL, "1986",
	"sprite test 6\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName,bsprite6RomInfo, bsprite6RomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo bcputimeRomDesc[] = {
	{ "cpu_timing_test.nes",          16400, 0x161bc1fe, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bcputime)
STD_ROM_FN(bcputime)

struct BurnDriver BurnDrvbcputime = {
	"nes_bcputime", NULL, NULL, NULL, "1986",
	"bcputime\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName,bcputimeRomInfo, bcputimeRomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo bcpuallRomDesc[] = {
	{ "all_instrs.nes",          262160, 0xA4400963, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bcpuall)
STD_ROM_FN(bcpuall)

struct BurnDriver BurnDrvbcpuall = {
	"nes_bcputime", NULL, NULL, NULL, "1986",
	"cpu all instructions\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName,bcpuallRomInfo,bcpuallRomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ruderRomDesc[] = {
	{ "ruder (HB).nes",          24592, 0x4c5c6008, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ruder)
STD_ROM_FN(nes_ruder)

struct BurnDriver BurnDrvnes_ruder = {
	"nes_ruder", NULL, NULL, NULL, "1989?",
	"Zapper ruder (test) (HB)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_ruderRomInfo, nes_ruderRomName, NULL, NULL, NULL, NULL, NESZapperInputInfo, NESZapperDIPInfo,
	NESZapperInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_allpadsRomDesc[] = {
	{ "allpads.nes",          32784, 0x9180a163, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_allpads)
STD_ROM_FN(nes_allpads)

struct BurnDriver BurnDrvnes_allpads = {
	"nes_allpads", NULL, NULL, NULL, "2016",
	"allpads\0", NULL, "tepples", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_allpadsRomInfo, nes_allpadsRomName, NULL, NULL, NULL, NULL, NES4ScoreInputInfo, NES4ScoreDIPInfo,
	NESHori4pInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_apudinknoiseRomDesc[] = {
	{ "apu_dink_noise.nes",          32784, 0xb19c71f8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_apudinknoise)
STD_ROM_FN(nes_apudinknoise)

struct BurnDriver BurnDrvnes_apudinknoise = {
	"nes_apudinknoise", NULL, NULL, NULL, "1989?",
	"apu_dink_noise\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_apudinknoiseRomInfo, nes_apudinknoiseRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

#endif
*/
// VS NES games (hand-added!)

static struct BurnRomInfo nes_vsraidbbayRomDesc[] = {
	{ "VS. Raid on Bungling Bay (1985)(Broderbund).nes",          49168, 0x84ad422e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vsraidbbay)
STD_ROM_FN(nes_vsraidbbay)

struct BurnDriver BurnDrvnes_vsraidbbay = {
	"vsraidbbay", NULL, NULL, NULL, "1985",
	"VS. Raid on Bungling Bay\0", NULL, "Broderbund", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 2, HARDWARE_NVS, GBF_SHOOT, 0,
	NULL, nes_vsraidbbayRomInfo, nes_vsraidbbayRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};


static struct BurnRomInfo nes_vsrbibbalRomDesc[] = {
	{ "VS. Atari RBI Baseball (1987)(Atari Games).nes",          98320, 0xac95e2c9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vsrbibbal)
STD_ROM_FN(nes_vsrbibbal)

struct BurnDriver BurnDrvnes_vsrbibbal = {
	"vsrbibbal", NULL, NULL, NULL, "1987",
	"VS. Atari RBI Baseball\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 2, HARDWARE_NVS, GBF_SPORTSMISC, 0,
	NULL, nes_vsrbibbalRomInfo, nes_vsrbibbalRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSDIPInfo,
	NESReversedInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vsbattlecityRomDesc[] = {
	{ "VS. Battle City (1985)(Namco).nes",          49168, 0x750048c0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vsbattlecity)
STD_ROM_FN(nes_vsbattlecity)

struct BurnDriver BurnDrvnes_vsbattlecity = {
	"vsbattlecity", NULL, NULL, NULL, "1985",
	"VS. Battle City\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 2, HARDWARE_NVS, GBF_MAZE | GBF_SHOOT, 0,
	NULL, nes_vsbattlecityRomInfo, nes_vsbattlecityRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vscastlevaniaRomDesc[] = {
	{ "VS. Castlevania (1987)(Konami).nes",          131088, 0x584c758b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vscastlevania)
STD_ROM_FN(nes_vscastlevania)

struct BurnDriver BurnDrvnes_vscastlevania = {
	"vscastlevania", NULL, NULL, NULL, "1987",
	"VS. Castlevania\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 1, HARDWARE_NVS, GBF_PLATFORM, 0,
	NULL, nes_vscastlevaniaRomInfo, nes_vscastlevaniaRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vscluclulandRomDesc[] = {
	{ "VS. Clu Clu Land (1984)(Nintendo).nes",          49168, 0x8c68556d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vsclucluland)
STD_ROM_FN(nes_vsclucluland)

struct BurnDriver BurnDrvnes_vsclucluland = {
	"vsclucluland", NULL, NULL, NULL, "1984",
	"VS. Clu Clu Land\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 2, HARDWARE_NVS, GBF_MAZE | GBF_ACTION, 0,
	NULL, nes_vscluclulandRomInfo, nes_vscluclulandRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vsdrmarioRomDesc[] = {
	{ "VS. Dr. Mario (1990)(Nintendo).nes",          98320, 0xce165ad2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vsdrmario)
STD_ROM_FN(nes_vsdrmario)

struct BurnDriver BurnDrvnes_vsdrmario = {
	"vsdrmario", NULL, NULL, NULL, "1990",
	"VS. Dr. Mario\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 2, HARDWARE_NVS, GBF_PUZZLE, 0,
	NULL, nes_vsdrmarioRomInfo, nes_vsdrmarioRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSDIPInfo,
	NESReversedInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vsduckhuntRomDesc[] = {
	{ "VS. Duck Hunt (1984)(Nintendo).nes",          49168, 0xf08f6590, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vsduckhunt)
STD_ROM_FN(nes_vsduckhunt)

struct BurnDriver BurnDrvnes_vsduckhunt = {
	"vsduckhunt", NULL, NULL, NULL, "1984",
	"VS. Duck Hunt\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 2, HARDWARE_NVS, GBF_SHOOT, 0,
	NULL, nes_vsduckhuntRomInfo, nes_vsduckhuntRomName, NULL, NULL, NULL, NULL, NESVSZapperInputInfo, NESVSZapperDIPInfo,
	NESZapperInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vsexcitebikeRomDesc[] = {
	{ "VS. Excitebike (1985)(Nintendo).nes",          49168, 0x0dc357ae, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vsexcitebike)
STD_ROM_FN(nes_vsexcitebike)

struct BurnDriver BurnDrvnes_vsexcitebike = {
	"vsexcitebike", NULL, NULL, NULL, "1985",
	"VS. Excitebike\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 1, HARDWARE_NVS, GBF_RACING, 0,
	NULL, nes_vsexcitebikeRomInfo, nes_vsexcitebikeRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vsfreedomforceRomDesc[] = {
	{ "VS. Freedom Force (1988)(Sunsoft).nes",          196624, 0xc9f3f439, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vsfreedomforce)
STD_ROM_FN(nes_vsfreedomforce)

struct BurnDriver BurnDrvnes_vsfreedomforce = {
	"vsfreedomforce", NULL, NULL, NULL, "1988",
	"VS. Freedom Force\0", NULL, "Sunsoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 2, HARDWARE_NVS, GBF_SHOOT, 0,
	NULL, nes_vsfreedomforceRomInfo, nes_vsfreedomforceRomName, NULL, NULL, NULL, NULL, NESVSZapperInputInfo, NESVSZapperDIPInfo,
	NESZapperInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vsgooniesRomDesc[] = {
	{ "VS. Goonies, The (1986)(Konami).nes",          131088, 0x5fefc97b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vsgoonies)
STD_ROM_FN(nes_vsgoonies)

struct BurnDriver BurnDrvnes_vsgoonies = {
	"vsgoonies", NULL, NULL, NULL, "1986",
	"VS. Goonies, The\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 1, HARDWARE_NVS, GBF_PLATFORM, 0,
	NULL, nes_vsgooniesRomInfo, nes_vsgooniesRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vsgradiusRomDesc[] = {
	{ "VS. Gradius (1986)(Konami).nes",          98320, 0x066ba9ab, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vsgradius)
STD_ROM_FN(nes_vsgradius)

struct BurnDriver BurnDrvnes_vsgradius = {
	"vsgradius", NULL, NULL, NULL, "1986",
	"VS. Gradius\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 2, HARDWARE_NVS, GBF_HORSHOOT, 0,
	NULL, nes_vsgradiusRomInfo, nes_vsgradiusRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vsgumshoeRomDesc[] = {
	{ "VS. Gumshoe (1986)(Nintendo).nes",          65552, 0xee7be019, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vsgumshoe)
STD_ROM_FN(nes_vsgumshoe)

struct BurnDriver BurnDrvnes_vsgumshoe = {
	"vsgumshoe", NULL, NULL, NULL, "1986",
	"VS. Gumshoe\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 1, HARDWARE_NVS, GBF_SHOOT, 0,
	NULL, nes_vsgumshoeRomInfo, nes_vsgumshoeRomName, NULL, NULL, NULL, NULL, NESVSZapperInputInfo, NESVSZapperDIPInfo,
	NESZapperInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vshogansalleyRomDesc[] = {
	{ "VS. Hogan's Alley (1985)(Nintendo).nes",          49168, 0x023bd2d5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vshogansalley)
STD_ROM_FN(nes_vshogansalley)

struct BurnDriver BurnDrvnes_vshogansalley = {
	"vshogansalley", NULL, NULL, NULL, "1985",
	"VS. Hogan's Alley\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 2, HARDWARE_NVS, GBF_SHOOT, 0,
	NULL, nes_vshogansalleyRomInfo, nes_vshogansalleyRomName, NULL, NULL, NULL, NULL, NESVSZapperInputInfo, NESVSZapperDIPInfo,
	NESZapperInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vsiceclimberRomDesc[] = {
	{ "VS. Ice Climber (1985)(Nintendo).nes",          49168, 0x6cb18497, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vsiceclimber)
STD_ROM_FN(nes_vsiceclimber)

struct BurnDriver BurnDrvnes_vsiceclimber = {
	"vsiceclimber", NULL, NULL, NULL, "1985",
	"VS. Ice Climber\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 2, HARDWARE_NVS, GBF_ACTION, 0,
	NULL, nes_vsiceclimberRomInfo, nes_vsiceclimberRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vsmachriderRomDesc[] = {
	{ "VS. Mach Rider (1985)(Nintendo).nes",          49168, 0xdbdc31ba, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vsmachrider)
STD_ROM_FN(nes_vsmachrider)

struct BurnDriver BurnDrvnes_vsmachrider = {
	"vsmachrider", NULL, NULL, NULL, "1985",
	"VS. Mach Rider\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 1, HARDWARE_NVS, GBF_RACING, 0,
	NULL, nes_vsmachriderRomInfo, nes_vsmachriderRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vsmightybombjackRomDesc[] = {
	{ "VS. Mighty Bomb Jack (1986)(Tecmo).nes",          40976, 0x483d5d6d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vsmightybombjack)
STD_ROM_FN(nes_vsmightybombjack)

struct BurnDriver BurnDrvnes_vsmightybombjack = {
	"vsmightybomjack", NULL, NULL, NULL, "1986",
	"VS. Mighty Bomb Jack\0", NULL, "Tecmo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 2, HARDWARE_NVS, GBF_PLATFORM, 0,
	NULL, nes_vsmightybombjackRomInfo, nes_vsmightybombjackRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vsninjajkunRomDesc[] = {
	{ "VS. Ninja Jajamaru-kun (1986)(Jaleco).nes",          49168, 0x1d489460, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vsninjajkun)
STD_ROM_FN(nes_vsninjajkun)

struct BurnDriver BurnDrvnes_vsninjajkun = {
	"vsninjajkun", NULL, NULL, NULL, "1986",
	"VS. Ninja Jajamaru-kun\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 2, HARDWARE_NVS, GBF_ACTION, 0,
	NULL, nes_vsninjajkunRomInfo, nes_vsninjajkunRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSDIPInfo,
	NESReversedInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vspinballRomDesc[] = {
	{ "VS. Pinball (1984)(Nintendo).nes",          49168, 0xbe0baa9e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vspinball)
STD_ROM_FN(nes_vspinball)

struct BurnDriver BurnDrvnes_vspinball = {
	"vspinball", NULL, NULL, NULL, "1984",
	"VS. Pinball\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 2, HARDWARE_NVS, GBF_PINBALL, 0,
	NULL, nes_vspinballRomInfo, nes_vspinballRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSBit0HighDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vsplatoonRomDesc[] = {
	{ "VS. Platoon (1988)(Sunsoft).nes",          262160, 0x4d9aa57f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vsplatoon)
STD_ROM_FN(nes_vsplatoon)

struct BurnDriver BurnDrvnes_vsplatoon = {
	"vsplatoon", NULL, NULL, NULL, "1988",
	"VS. Platoon\0", NULL, "Sunsoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 1, HARDWARE_NVS, GBF_SHOOT | GBF_SCRFIGHT, 0,
	NULL, nes_vsplatoonRomInfo, nes_vsplatoonRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vsslalomRomDesc[] = {
	{ "VS. Slalom (1986)(Nintendo).nes",          40976, 0xf7953678, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vsslalom)
STD_ROM_FN(nes_vsslalom)

struct BurnDriver BurnDrvnes_vsslalom = {
	"vsslalom", NULL, NULL, NULL, "1986",
	"VS. Slalom\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 2, HARDWARE_NVS, GBF_SPORTSMISC, 0,
	NULL, nes_vsslalomRomInfo, nes_vsslalomRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vssoccerRomDesc[] = {
	{ "VS. Soccer (1985)(Nintendo).nes",          49168, 0xb2fc5aaa, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vssoccer)
STD_ROM_FN(nes_vssoccer)

struct BurnDriver BurnDrvnes_vssoccer = {
	"vssoccer", NULL, NULL, NULL, "1985",
	"VS. Soccer\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 2, HARDWARE_NVS, GBF_SPORTSFOOTBALL, 0,
	NULL, nes_vssoccerRomInfo, nes_vssoccerRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSDIPInfo,
	NESReversedInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vsstarlusterRomDesc[] = {
	{ "VS. Star Luster (1985)(Namco).nes",          49168, 0x2e549bef, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vsstarluster)
STD_ROM_FN(nes_vsstarluster)

struct BurnDriver BurnDrvnes_vsstarluster = {
	"vsstarluster", NULL, NULL, NULL, "1985",
	"VS. Star Luster\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 1, HARDWARE_NVS, GBF_SIM | GBF_SHOOT, 0,
	NULL, nes_vsstarlusterRomInfo, nes_vsstarlusterRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vssmgolfRomDesc[] = {
	{ "VS. Stroke & Match Golf (1984)(Nintendo).nes",          49168, 0xaca28dce, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vssmgolf)
STD_ROM_FN(nes_vssmgolf)

struct BurnDriver BurnDrvnes_vssmgolf = {
	"vssmgolf", NULL, NULL, NULL, "1984",
	"VS. Stroke & Match Golf\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 2, HARDWARE_NVS, GBF_SPORTSMISC, 0,
	NULL, nes_vssmgolfRomInfo, nes_vssmgolfRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSBit0HighDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vssmgolflaRomDesc[] = {
	{ "VS. Stroke & Match Golf Ladies (1984)(Nintendo).nes",          49168, 0x50c5c8a4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vssmgolfla)
STD_ROM_FN(nes_vssmgolfla)

struct BurnDriver BurnDrvnes_vssmgolfla = {
	"vssmgolfla", NULL, NULL, NULL, "1984",
	"VS. Stroke & Match Golf Ladies\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 2, HARDWARE_NVS, GBF_SPORTSMISC, 0,
	NULL, nes_vssmgolflaRomInfo, nes_vssmgolflaRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSBit0HighDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vssmbRomDesc[] = {
	{ "VS. Super Mario Bros (1986)(Nintendo).nes",          49168, 0x7b8c1d89, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vssmb)
STD_ROM_FN(nes_vssmb)

struct BurnDriver BurnDrvnes_vssmb = {
	"vssmb", NULL, NULL, NULL, "1986",
	"VS. Super Mario Bros\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 2, HARDWARE_NVS, GBF_PLATFORM, 0,
	NULL, nes_vssmbRomInfo, nes_vssmbRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vssuperskykidRomDesc[] = {
	{ "VS. Super Sky Kid (1986)(Namco).nes",          65552, 0xeb4452c9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vssuperskykid)
STD_ROM_FN(nes_vssuperskykid)

struct BurnDriver BurnDrvnes_vssuperskykid = {
	"vssuperskykid", NULL, NULL, NULL, "1986",
	"VS. Super Sky Kid\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 2, HARDWARE_NVS, GBF_HORSHOOT, 0,
	NULL, nes_vssuperskykidRomInfo, nes_vssuperskykidRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSDIPInfo,
	NESReversedInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vssuperxeviousRomDesc[] = {
	{ "VS. Super Xevious - Gump no Nazo (1986)(Namco).nes",          163856, 0x9f2251e4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vssuperxevious)
STD_ROM_FN(nes_vssuperxevious)

struct BurnDriver BurnDrvnes_vssuperxevious = {
	"vssuperxevious", NULL, NULL, NULL, "1986",
	"VS. Super Xevious - Gump no Nazo\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 1, HARDWARE_NVS, GBF_VERSHOOT, 0,
	NULL, nes_vssuperxeviousRomInfo, nes_vssuperxeviousRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vstetrisRomDesc[] = {
	{ "VS. Tetris (1988)(Atari Games).nes",          40976, 0x131afe42, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vstetris)
STD_ROM_FN(nes_vstetris)

struct BurnDriver BurnDrvnes_vstetris = {
	"vstetris", NULL, NULL, NULL, "1988",
	"VS. Tetris\0", NULL, "Atari Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 2, HARDWARE_NVS, GBF_PUZZLE, 0,
	NULL, nes_vstetrisRomInfo, nes_vstetrisRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSDIPInfo,
	NESReversedInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vstkoboxingRomDesc[] = {
	{ "VS. TKO Boxing (1987)(Data East).nes",          131088, 0xd81d33da, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vstkoboxing)
STD_ROM_FN(nes_vstkoboxing)

struct BurnDriver BurnDrvnes_vstkoboxing = {
	"vstkoboxing", NULL, NULL, NULL, "1987",
	"VS. TKO Boxing\0", NULL, "Data East", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 2, HARDWARE_NVS, GBF_VSFIGHT, 0,
	NULL, nes_vstkoboxingRomInfo, nes_vstkoboxingRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vstopgunRomDesc[] = {
	{ "VS. Top Gun (1987)(Konami).nes",          131088, 0xa64c6e99, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vstopgun)
STD_ROM_FN(nes_vstopgun)

struct BurnDriver BurnDrvnes_vstopgun = {
	"vstopgun", NULL, NULL, NULL, "1987",
	"VS. Top Gun\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 1, HARDWARE_NVS, GBF_SIM | GBF_SHOOT, 0,
	NULL, nes_vstopgunRomInfo, nes_vstopgunRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Non Homebrew (hand-added!)

static struct BurnRomInfo nes_minnanotaaboujRomDesc[] = {
	{ "Minna no Taabou no Nakayoshi Daisakusen (Japan).nes",          65552, 0x8c4f37e2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_minnanotaabouj)
STD_ROM_FN(nes_minnanotaabouj)

struct BurnDriver BurnDrvnes_minnanotaabouj = {
	"nes_minnanotaabouj", "nes_minnanotaabou", NULL, NULL, "1991",
	"Minna no Taabou no Nakayoshi Daisakusen (Japan)\0", NULL, "Character Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_ACTION | GBF_PUZZLE, 0,
	NESGetZipName, nes_minnanotaaboujRomInfo, nes_minnanotaaboujRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_minnanotaabouRomDesc[] = {
	{ "Minna no Taabou no Nakayoshi Daisakusen (T-Eng).nes",          65552, 0x17336a80, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_minnanotaabou)
STD_ROM_FN(nes_minnanotaabou)

struct BurnDriver BurnDrvnes_minnanotaabou = {
	"nes_minnanotaabou", NULL, NULL, NULL, "2017",
	"Minna no Taabou no Nakayoshi Daisakusen (T-Eng, v1.01)\0", NULL, "Suicidal Translations", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 2, HARDWARE_NES, GBF_ACTION | GBF_PUZZLE, 0,
	NESGetZipName, nes_minnanotaabouRomInfo, nes_minnanotaabouRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static INT32 topriderInit()
{
	INT32 rc = NESInit();

	if (!rc) {
		// Top Rider size / crc: 163856, 0xca1a395a
		// Patch in standard controller -dink feb.25, 2021
		// Game Genie code: IUEOKOAL
		// lda $494
		// ora $311
		// and #$10
		//*bne 99c5 ; start pressed, next screen
		// jsr 9f68 ; check if code entered on controller #2
		// bcs 99c2 ; yes? start game with normal controller in port #1
		// ...
		//*d0 38  bne 99c5 -> d0 35  bne 99c2

		Cart.PRGRom[0x1998c] = 0x35;
	}

	return rc;
}

static struct BurnRomInfo nes_topriderRomDesc[] = {
	{ "Top Rider (Japan).nes",          163856, 0xca1a395a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_toprider)
STD_ROM_FN(nes_toprider)

struct BurnDriver BurnDrvnes_toprider = {
	"nes_toprider", NULL, NULL, NULL, "1988",
	"Top Rider (Japan)\0", NULL, "Varie", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_RACING, 0,
	NESGetZipName, nes_topriderRomInfo, nes_topriderRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	topriderInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Willow (Hack, No Flash)
// https://www.romhacking.net/hacks/4992/

static struct BurnRomInfo nes_willownfhRomDesc[] = {
	{ "Willow (Hack, No Flash).nes",          262160, 0x956ea1a6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_willownfh)
STD_ROM_FN(nes_willownfh)

struct BurnDriver BurnDrvnes_willownfh = {
	"nes_willownfh", "nes_willow", NULL, NULL, "2020",
	"Willow (No Flash ed.) (Hack, v1.1)\0", NULL, "Jigglysaint", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_willownfhRomInfo, nes_willownfhRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_xiaomaliRomDesc[] = {
	{ "Xiao Ma Li (Unl).nes",          24592, 0xc8f9a5ab, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_xiaomali)
STD_ROM_FN(nes_xiaomali)

struct BurnDriver BurnDrvnes_xiaomali = {
	"nes_xiaomali", NULL, NULL, NULL, "1989-90",
	"Xiao Ma Li (Unl)\0", NULL, "Nei-Hu Electronics", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	0, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_xiaomaliRomInfo, nes_xiaomaliRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_1991dumaracRomDesc[] = {
	{ "1991 Du Ma Racing (Unl).nes",          65552, 0x8cd7f9b1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_1991dumarac)
STD_ROM_FN(nes_1991dumarac)

struct BurnDriver BurnDrvnes_1991dumarac = {
	"nes_1991dumarac", NULL, NULL, NULL, "1991",
	"1991 Du Ma Racing (Unl)\0", NULL, "Idea-tek", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_SPORTSMISC | GBF_MISC, 0,
	NESGetZipName, nes_1991dumaracRomInfo, nes_1991dumaracRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_mahjongblockRomDesc[] = {
	{ "Mahjong Block (Idea-tek) (Unl).nes",          65552, 0x30820b60, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mahjongblock)
STD_ROM_FN(nes_mahjongblock)

struct BurnDriver BurnDrvnes_mahjongblock = {
	"nes_mahjongblock", NULL, NULL, NULL, "1991",
	"Mahjong Block (Unl)\0", NULL, "Idea-tek", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_mahjongblockRomInfo, nes_mahjongblockRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_samuraispiritsRomDesc[] = {
	{ "Samurai Spirits (Unl).nes",          786448, 0x9b7305f7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_samuraispirits)
STD_ROM_FN(nes_samuraispirits)

struct BurnDriver BurnDrvnes_samuraispirits = {
	"nes_samuraispirits", NULL, NULL, NULL, "199?",
	"Samurai Spirits (Unl)\0", NULL, "Rex Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_VSFIGHT, 0,
	NESGetZipName, nes_samuraispiritsRomInfo, nes_samuraispiritsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_avpregirfigRomDesc[] = {
	{ "AV Pretty Girl Fight (Unl).nes",          655376, 0xc3d2b090, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_avpregirfig)
STD_ROM_FN(nes_avpregirfig)

struct BurnDriver BurnDrvnes_avpregirfig = {
	"nes_avpregirfig", NULL, NULL, NULL, "1994",
	"AV Pretty Girl Fight (Unl)\0", NULL, "Someri Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_VSFIGHT, 0,
	NESGetZipName, nes_avpregirfigRomInfo, nes_avpregirfigRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_janggunuiadeulRomDesc[] = {
	{ "Janggun-ui Adeul (Korea) (Unl).nes",          655376, 0x54171ca4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_janggunuiadeul)
STD_ROM_FN(nes_janggunuiadeul)

struct BurnDriver BurnDrvnes_janggunuiadeul = {
	"nes_janggunuiadeul", NULL, NULL, NULL, "1992",
	"Janggun-ui Adeul (Korea) (Unl)\0", NULL, "Daou", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_VSFIGHT, 0,
	NESGetZipName, nes_janggunuiadeulRomInfo, nes_janggunuiadeulRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Famista '93 (T-Eng)
// https://www.romhacking.net/
static struct BurnRomInfo nes_famista93eRomDesc[] = {
	{ "famista '93 (t-eng).nes",          262160, 0xd9301c12, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_famista93e)
STD_ROM_FN(nes_famista93e)

struct BurnDriver BurnDrvnes_famista93e = {
	"nes_famista93e", "nes_famista93", NULL, NULL, "20??",
	"Famista '93 (T-Eng)\0", NULL, "FlashPV", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_SPORTSMISC, 0,
	NESGetZipName, nes_famista93eRomInfo, nes_famista93eRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ultimexoremRomDesc[] = {
	{ "Ultima - Exodus Remastered (USA)(Hack).nes",          262160, 0x8afe467a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ultimexorem)
STD_ROM_FN(nes_ultimexorem)

struct BurnDriver BurnDrvnes_ultimexorem = {
	"nes_ultimexorem", "nes_ultimaexodus", NULL, NULL, "2020",
	"Ultima - Exodus Remastered (USA) (Hack)\0", NULL, "Fox Cunning", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_ultimexoremRomInfo, nes_ultimexoremRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_supermarallstanesRomDesc[] = {
	{ "Super Mario All Stars NES (Hack).nes",          2097168, 0xbe155d3e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_supermarallstanes)
STD_ROM_FN(nes_supermarallstanes)

struct BurnDriver BurnDrvnes_supermarallstanes = {
	"nes_supermarioallst", NULL, NULL, NULL, "2020",
	"Super Mario All Stars NES (Hack)\0", NULL, "infidelity", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_supermarallstanesRomInfo, nes_supermarallstanesRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_smbchrediRomDesc[] = {
	{ "Super Mario Bros. Christmas Edition (Hack).nes",          73744, 0xb293a7c4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smbchredi)
STD_ROM_FN(nes_smbchredi)

struct BurnDriver BurnDrvnes_smbchredi = {
	"nes_smbchredi", "nes_smb", NULL, NULL, "2020",
	"Super Mario Bros. Christmas Edition (Hack)\0", NULL, "U Retro", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_smbchrediRomInfo, nes_smbchrediRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// https://www.romhacking.net/hacks/5245/
static struct BurnRomInfo nes_starlightmarioRomDesc[] = {
	{ "Starlight Mario (2020)(TPWC Hacks).nes",          40976, 0x7b64b618, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_starlightmario)
STD_ROM_FN(nes_starlightmario)

struct BurnDriver BurnDrvnes_starlightmario = {
	"nes_starlightmario", "nes_smb", NULL, NULL, "2020",
	"Starlight Mario (Hack)\0", NULL, "TPWC Hacks", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_starlightmarioRomInfo, nes_starlightmarioRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// https://www.romhacking.net/hacks/7253/
static struct BurnRomInfo nes_starlightmariouRomDesc[] = {
	{ "Starlight Mario Underworld (2022)(TPWC Hacks).nes",          65552, 0xa7a313b5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_starlightmariou)
STD_ROM_FN(nes_starlightmariou)

struct BurnDriver BurnDrvnes_starlightmariou = {
	"nes_starlightmariou", "nes_smb", NULL, NULL, "2022",
	"Starlight Mario Underworld (Hack)\0", NULL, "TPWC Hacks", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_starlightmariouRomInfo, nes_starlightmariouRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// https://mail.romhacking.net/hacks/6922/
static struct BurnRomInfo nes_smbenRomDesc[] = {
	{ "Super Mario Bros Enhanced (2022)(NesInvent).nes",          40976, 0xb1e5b547, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smben)
STD_ROM_FN(nes_smben)

struct BurnDriver BurnDrvnes_smben = {
	"nes_smben", "nes_smb", NULL, NULL, "2022",
	"Super Mario Bros. Enhanced (Hack)\0", NULL, "NesInvent", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_smbenRomInfo, nes_smbenRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_smbendssmbha1RomDesc[] = {
	{ "Super Mario Bros. Ende's SMB Hack 1 (Hack).nes",          40976, 0x1eeab4b1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smbendssmbha1)
STD_ROM_FN(nes_smbendssmbha1)

struct BurnDriver BurnDrvnes_smbendssmbha1 = {
	"nes_smbendssmbha1", "nes_smb", NULL, NULL, "2004",
	"Super Mario Bros. Ende's SMB (Hack)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_smbendssmbha1RomInfo, nes_smbendssmbha1RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_kamimariodxRomDesc[] = {
	{ "Kamikaze Mario DX+ v1.9 (2021)(79).nes",          147472, 0xb7518c16, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_kamimariodx)
STD_ROM_FN(nes_kamimariodx)

struct BurnDriver BurnDrvnes_kamimariodx = {
	"nes_kamimariodx", "nes_smb", NULL, NULL, "2021",
	"Kamikaze Mario DX+ (Hack, v1.9)\0", NULL, "79", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_kamimariodxRomInfo, nes_kamimariodxRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_marioquestRomDesc[] = {
	{ "Mario Quest (2021)(Kitsunefox).nes",          40976, 0xed5be20d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_marioquest)
STD_ROM_FN(nes_marioquest)

struct BurnDriver BurnDrvnes_marioquest = {
	"nes_marioquest", "nes_smb", NULL, NULL, "2021",
	"Mario Quest (Hack)\0", NULL, "Kitsunefox", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_marioquestRomInfo, nes_marioquestRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// https://www.romhacking.net/hacks/7784/
static struct BurnRomInfo nes_maniacmanuncRomDesc[] = {
	{ "Maniac Mansion Uncensored v2.0 (2023)(gzip).nes",          524304, 0x483e38f2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_maniacmanunc)
STD_ROM_FN(nes_maniacmanunc)

struct BurnDriver BurnDrvnes_maniacmanunc = {
	"nes_maniacmanunc", "nes_maniacmansion", NULL, NULL, "2023",
	"Maniac Mansion Uncensored (Hack, v2.0)\0", NULL, "gzip", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_maniacmanuncRomInfo, nes_maniacmanuncRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_blockoutRomDesc[] = {
	{ "Block Out (USA).nes",          131088, 0xdeff01f6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_blockout)
STD_ROM_FN(nes_blockout)

struct BurnDriver BurnDrvnes_blockout = {
	"nes_blockout", NULL, NULL, NULL, "1990",
	"Block Out (USA)\0", NULL, "California Dreams / Technos", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_blockoutRomInfo, nes_blockoutRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};


static struct BurnRomInfo nes_bloodjurRomDesc[] = {
	{ "Blood of Jurassic (1997)(gd-98).nes",          262160, 0x76ea1e9f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bloodjur)
STD_ROM_FN(nes_bloodjur)

struct BurnDriver BurnDrvnes_bloodjur = {
	"nes_bloodjur", NULL, NULL, NULL, "1997",
	"Blood of Jurassic (Unl)\0", NULL, "Shanghai PS&T", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bloodjurRomInfo, nes_bloodjurRomName, NULL, NULL, NULL, NULL, NESZapperInputInfo, NESZapperDIPInfo,
	NESZapperInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};


static struct BurnRomInfo nes_bookymanRomDesc[] = {
	{ "Bookyman (1991)(Mega Soft).nes",          24592, 0xd8f11b78, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bookyman)
STD_ROM_FN(nes_bookyman)

struct BurnDriver BurnDrvnes_bookyman = {
	"nes_bookyman", NULL, NULL, NULL, "1991",
	"Bookyman (Unl)\0", NULL, "Mega Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_MAZE | GBF_ACTION, 0,
	NESGetZipName, nes_bookymanRomInfo, nes_bookymanRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_destroyerRomDesc[] = {
	{ "Destroyer (1992)(Mega Soft).nes",          65552, 0xd9362123, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_destroyer)
STD_ROM_FN(nes_destroyer)

struct BurnDriver BurnDrvnes_destroyer = {
	"nes_destroyer", NULL, NULL, NULL, "1992",
	"Destroyer (Unl)\0", NULL, "Mega Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_VERSHOOT, 0,
	NESGetZipName, nes_destroyerRomInfo, nes_destroyerRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_somariRomDesc[] = {
	{ "Somari (Unl).nes",          524304, 0x51589daf, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_somari)
STD_ROM_FN(nes_somari)

struct BurnDriver BurnDrvnes_somari = {
	"nes_somari", NULL, NULL, NULL, "1994",
	"Somari (Unl)\0", NULL, "Hummer Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_somariRomInfo, nes_somariRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Sonic Improvement vol.2
// http://www.romhacking.net/hacks/5026/
static struct BurnRomInfo nes_sonicRomDesc[] = {
	{ "Sonic the Hedgehog (Hack).nes",          786448, 0xb9ebd682, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sonic)
STD_ROM_FN(nes_sonic)

struct BurnDriver BurnDrvnes_sonic = {
	"nes_sonic", "nes_somari", NULL, NULL, "2020",
	"Sonic the Hedgehog (Hack)\0", NULL, "Terwilf", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_sonicRomInfo, nes_sonicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Bowsette 2 (Hack, Red version)
// https://www.romhacking.net/hacks/4326/

static struct BurnRomInfo nes_bowsette2RomDesc[] = {
	{ "Bowsette 2 (Hack).nes",          262160, 0x94390680, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bowsette2)
STD_ROM_FN(nes_bowsette2)

struct BurnDriver BurnDrvnes_bowsette2 = {
	"nes_bowsette2", "nes_smb2", NULL, NULL, "2019",
	"Bowsette 2 (Hack, Red version)\0", NULL, "Zynk Oxhyde", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_bowsette2RomInfo, nes_bowsette2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Super Mario Unlimited Deluxe
// http://www.romhacking.net/hacks/5546/
static struct BurnRomInfo nes_smbunldlxRomDesc[] = {
	{ "Super Mario Unlimited Deluxe (Hack).nes",          196624, 0x716c9eb4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smbunldlx)
STD_ROM_FN(nes_smbunldlx)

struct BurnDriver BurnDrvnes_smbunldlx = {
	"nes_smbunldlx", "nes_smb", NULL, NULL, "2020",
	"Super Mario Unlimited Deluxe (Hack)\0", NULL, "Frantik", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_smbunldlxRomInfo, nes_smbunldlxRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Little Samson
// Restoration Hack by SCD @ https://www.romhacking.net/hacks/5413/
// Pleasant Shot Sound hack by Nesrocks @ https://www.romhacking.net/hacks/5189/

static struct BurnRomInfo nes_littlesamsonhRomDesc[] = {
	{ "Little Samson (Hack).nes",          393232, 0x2178190e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_littlesamsonh)
STD_ROM_FN(nes_littlesamsonh)

struct BurnDriver BurnDrvnes_littlesamsonh = {
	"nes_littlesamsonh", "nes_littlesamson", NULL, NULL, "2020",
	"Little Samson (Restoration & Sound Hack)\0", NULL, "SCD - Nesrocks", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_littlesamsonhRomInfo, nes_littlesamsonhRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Aliens - Alien 2 (Fix Hack)
// Hack by lancuster
// https://www.romhacking.net/hacks/3790/
static struct BurnRomInfo fds_aliensfixRomDesc[] = {
	{ "Aliens - Alien 2 (Prototype)(Fix Hack).fds",          131016, 0x0d487b19, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_aliensfix, fds_aliensfix, fds_fdsbios)
STD_ROM_FN(fds_aliensfix)

struct BurnDriver BurnDrvfds_aliensfix = {
	"fds_aliensfix", NULL, "fds_fdsbios", NULL, "2017",
	"Aliens - Alien 2 (Fix Hack)\0", NULL, "lancuster", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_FDS, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, fds_aliensfixRomInfo, fds_aliensfixRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Aliens - Alien 2 (Japan) (Prototype)
static struct BurnRomInfo fds_aliensRomDesc[] = {
	{ "Aliens - Alien 2 (J) (Prototype).fds",          131000, 0x2b779ee3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_aliens, fds_aliens, fds_fdsbios)
STD_ROM_FN(fds_aliens)

struct BurnDriver BurnDrvfds_aliens = {
	"fds_aliens", "fds_aliensfix", "fds_fdsbios", NULL, "1987",
	"Aliens - Alien 2 (Japan) (Prototype)\0", NULL, "Square", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_PROTOTYPE, 1, HARDWARE_FDS, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, fds_aliensRomInfo, fds_aliensRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Bubble Bath Babes (USA) (Unl)
static struct BurnRomInfo nes_bbbabesRomDesc[] = {
	{ "Bubble Bath Babes (USA) (Unl).nes",          98320, 0xf36edcc5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bbbabes)
STD_ROM_FN(nes_bbbabes)

struct BurnDriver BurnDrvnes_bbbabes = {
	"nes_bbbabes", NULL, NULL, NULL, "1991",
	"Bubble Bath Babes (USA) (Unl)\0", NULL, "Panesian", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_bbbabesRomInfo, nes_bbbabesRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Jungle Book, The (Brightness fix)
// Hacked by Ilya Indigo
// https://www.romhacking.net/hacks/5323/

static struct BurnRomInfo nes_junglebookthebRomDesc[] = {
	{ "jungle book, the (2020)(brightness fix).nes",          262160, 0x1c9a2869, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_junglebooktheb)
STD_ROM_FN(nes_junglebooktheb)

struct BurnDriver BurnDrvnes_junglebooktheb = {
	"nes_junglebooktheb", "nes_junglebookthe", NULL, NULL, "2020",
	"Jungle Book, The (Brightness fix)\0", NULL, "Ilya Indigo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_junglebookthebRomInfo, nes_junglebookthebRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Super Royal Pals (Hack)
// Hacked by QUXNLADY
// https://www.romhacking.net/hacks/5320/

static struct BurnRomInfo nes_srpRomDesc[] = {
	{ "super royal pals (2020)(QUXNLADY).nes",          40976, 0x62797e6e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_srp)
STD_ROM_FN(nes_srp)

struct BurnDriver BurnDrvnes_srp = {
	"nes_srp", "nes_smb", NULL, NULL, "2020",
	"Super Royal Pals (Hack)\0", NULL, "QUXNLADY", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_srpRomInfo, nes_srpRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_rockm4miRomDesc[] = {
	{ "Rockman 4 - Minus Infinity (Hack).nes",          1048592, 0xe9887ba6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_rockm4mi)
STD_ROM_FN(nes_rockm4mi)

struct BurnDriver BurnDrvnes_rockm4mi = {
	"nes_rockm4mi", "nes_megaman4", NULL, NULL, "2016",
	"Rockman 4 - Minus Infinity (Hack)\0", NULL, "Puresabe", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_rockm4miRomInfo, nes_rockm4miRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Metal Storm Re-Release (Mapper 268)
static struct BurnRomInfo nes_metalstormrrRomDesc[] = {
	{ "Metal Storm (2019).nes",          524304, 0x33af86d9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_metalstormrr)
STD_ROM_FN(nes_metalstormrr)

struct BurnDriver BurnDrvnes_metalstormrr = {
	"nes_metalstormrr", "nes_metalstorm", NULL, NULL, "2019",
	"Metal Storm (Re-Release)\0", NULL, "Irem - Retro-bit", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_metalstormrrRomInfo, nes_metalstormrrRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Pacman CE
static struct BurnRomInfo nes_pacmanceRomDesc[] = {
	{ "pacman-ce (2020).nes",          262160, 0xb86c09af, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_pacmance)
STD_ROM_FN(nes_pacmance)

struct BurnDriver BurnDrvnes_pacmance = {
	"nes_pacmance", NULL, NULL, NULL, "2020",
	"Pac-Man Championship Edition\0", NULL, "Namco - BNEI", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MAZE | GBF_ACTION, 0,
	NESGetZipName, nes_pacmanceRomInfo, nes_pacmanceRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Gaplus
static struct BurnRomInfo nes_gaplusRomDesc[] = {
	{ "gaplus (bnei).nes",          262160, 0x60811720, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_gaplus)
STD_ROM_FN(nes_gaplus)

struct BurnDriver BurnDrvnes_gaplus = {
	"nes_gaplus", NULL, NULL, NULL, "2020",
	"Gaplus\0", NULL, "Namco - BNEI", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_SHOOT, 0,
	NESGetZipName, nes_gaplusRomInfo, nes_gaplusRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Happily Ever After (USA) (Prototype)
static struct BurnRomInfo nes_happilyeaRomDesc[] = {
	{ "Happily Ever After (USA) (Prototype).nes",          262160, 0x57f5c01d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_happilyea)
STD_ROM_FN(nes_happilyea)

struct BurnDriver BurnDrvnes_happilyea = {
	"nes_happilyea", NULL, NULL, NULL, "1991",
	"Happily Ever After (USA) (Prototype)\0", NULL, "SOFEL", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_PROTOTYPE, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_happilyeaRomInfo, nes_happilyeaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_magickidsdorpRomDesc[] = {
	{ "Magical Kid's Doropie (Prototype).nes",          262160, 0x84209fee, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_magickidsdorp)
STD_ROM_FN(nes_magickidsdorp)

struct BurnDriver BurnDrvnes_magickidsdorp = {
	"nes_magickidsdorp", "nes_krioncon", NULL, NULL, "1990",
	"Magical Kid's Doropie (Prototype)\0", NULL, "Vic Tokai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_PROTOTYPE, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_magickidsdorpRomInfo, nes_magickidsdorpRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Rockman 3 - Dr. Wily no Saigo! (T-Chi) 20230315
// Modified by NOKOH
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=719402
static struct BurnRomInfo nes_rockm3cRomDesc[] = {
	{ "Rockman 3 - Dr. Wily no Saigo! (T-Chi).nes",          786448, 0x8125b0a4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_rockm3c)
STD_ROM_FN(nes_rockm3c)

struct BurnDriver BurnDrvnes_rockm3c = {
	"nes_rockm3c", "nes_megaman3", NULL, NULL, "2023",
	"Rockman 3 - Dr. Wily no Saigo! (T-Chi)\0", NULL, "NOKOH", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_rockm3cRomInfo, nes_rockm3cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Jackal - Endless Mission (v1.2) 20210313
// Modified by HHNM Team - ZENG GE / 
// https://www.nesbbs.com/bbs/thread-51662-1-1.html
static struct BurnRomInfo nes_jackalemRomDesc[] = {
	{ "Jackal - Endless Mission (v1.2).nes",          262160, 0x833c6bc4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_jackalem)
STD_ROM_FN(nes_jackalem)

struct BurnDriver BurnDrvnes_jackalem = {
	"nes_jackalem", "nes_jackal", NULL, NULL, "2021",
	"Jackal - Endless Mission (v1.2)\0", NULL, "HHNM Team - ZENG GE", "Miscellaneous",
	L"Jackal - Endless Mission (v1.2)\0\u8d64\u8272\u8981\u585e - \u65e0\u5c3d\u7684\u4efb\u52a1 (v1.2)", NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_jackalemRomInfo, nes_jackalemRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Teenage Mutant Ninja Turtles II - The Arcade Game (T-Chi)
// Modified by NOKOH
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=720874
static struct BurnRomInfo nes_tmntiiarcgamtcRomDesc[] = {
	{ "Teenage Mutant Ninja Turtles II - The Arcade Game (T-Chi).nes",          786448, 0x35e378fc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tmntiiarcgamtc)
STD_ROM_FN(nes_tmntiiarcgamtc)

struct BurnDriver BurnDrvnes_tmntiiarcgamtc = {
	"nes_tmntiiarcgamtc", "nes_tmntiiarcgam", NULL, NULL, "2023",
	"Teenage Mutant Ninja Turtles II - The Arcade Game (T-Chi)\0", NULL, "NOKOH", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_tmntiiarcgamtcRomInfo, nes_tmntiiarcgamtcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// RoboCop (T-Chi)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=718366
static struct BurnRomInfo nes_robocopscRomDesc[] = {
	{ "RoboCop (T-Chi).nes",          393232, 0xa568590b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_robocopsc)
STD_ROM_FN(nes_robocopsc)

struct BurnDriver BurnDrvnes_robocopsc = {
	"nes_robocopsc", "nes_robocop", NULL, NULL, "2023",
	"RoboCop (T-Chi)\0", NULL, "LSP Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_robocopscRomInfo, nes_robocopscRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Kyouryuu Sentai Juuranger (T-Chi)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=718391
static struct BurnRomInfo nes_kyoursenjuucRomDesc[] = {
	{ "Kyouryuu Sentai Juuranger (T-Chi).nes",          393232, 0x2cebf5e7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_kyoursenjuuc)
STD_ROM_FN(nes_kyoursenjuuc)

struct BurnDriver BurnDrvnes_kyoursenjuuc = {
	"nes_kyoursenjuuc", "nes_kyoursenjuu", NULL, NULL, "2023",
	"Kyouryuu Sentai Juuranger (T-Chi)\0", NULL, "LSP Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_RUNGUN | GBF_MINIGAMES, 0,
	NESGetZipName, nes_kyoursenjuucRomInfo, nes_kyoursenjuucRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Power Blade 2 (T-Chi)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=718809
static struct BurnRomInfo nes_powerblade2cRomDesc[] = {
	{ "Power Blade 2 (T-Chi).nes",          524304, 0x4cd7666a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_powerblade2c)
STD_ROM_FN(nes_powerblade2c)

struct BurnDriver BurnDrvnes_powerblade2c = {
	"nes_powerblade2c", "nes_powerblade2", NULL, NULL, "2023",
	"Power Blade 2 (T-Chi)\0", NULL, "Ni You De Wu Cai Yu", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_powerblade2cRomInfo, nes_powerblade2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Power Blade (T-Chi, v1.0)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=717589
static struct BurnRomInfo nes_powerbladecRomDesc[] = {
	{ "Power Blade (T-Chi, v1.0).nes",          393232, 0xa368fcee, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_powerbladec)
STD_ROM_FN(nes_powerbladec)

struct BurnDriver BurnDrvnes_powerbladec = {
	"nes_powerbladec", "nes_powerblade", NULL, NULL, "2022",
	"Power Blade (T-Chi, v1.0)\0", NULL, "Ni You De Wu Cai Yu", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_powerbladecRomInfo, nes_powerbladecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Matendouji (T-Chi)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=713661
static struct BurnRomInfo nes_matendoujicRomDesc[] = {
	{ "Matendouji (T-Chi).nes",          524304, 0x919e863f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_matendoujic)
STD_ROM_FN(nes_matendoujic)

struct BurnDriver BurnDrvnes_matendoujic = {
	"nes_matendoujic", "nes_matendouji", NULL, NULL, "2022",
	"Matendouji (T-Chi)\0", NULL, "Ni You De Wu Cai Yu", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_matendoujicRomInfo, nes_matendoujicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Flintstones, The - The Rescue of Dino & Hoppy (T-Chi)
// https://www.nesbbs.com/bbs/thread-53671-1-1.html
static struct BurnRomInfo nes_flintresdinhopcRomDesc[] = {
	{ "Flintstones, The - The Rescue of Dino & Hoppy (T-Chi).nes",          655376, 0x773ffd89, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_flintresdinhopc)
STD_ROM_FN(nes_flintresdinhopc)

struct BurnDriver BurnDrvnes_flintresdinhopc = {
	"nes_flintresdinhopc", "nes_flintresdinhop", NULL, NULL, "2022",
	"Flintstones, The - The Rescue of Dino & Hoppy (T-Chi)\0", NULL, "HHNM Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_flintresdinhopcRomInfo, nes_flintresdinhopcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Captain America and the Avengers (T-Chi)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=712633
static struct BurnRomInfo nes_captaameandavecRomDesc[] = {
	{ "Captain America and the Avengers (T-Chi).nes",          393232, 0xcfdf0dc5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_captaameandavec)
STD_ROM_FN(nes_captaameandavec)

struct BurnDriver BurnDrvnes_captaameandavec = {
	"nes_captaameandavec", "nes_captaameandave", NULL, NULL, "2022",
	"Captain America and the Avengers (T-Chi)\0", NULL, "LSP Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_captaameandavecRomInfo, nes_captaameandavecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Zelda no Densetsu - The Hyrule Fantasy (T-Chi) - (FDS to NES)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=711227
static struct BurnRomInfo nes_zeldanodencRomDesc[] = {
	{ "Zelda no Densetsu - The Hyrule Fantasy (T-Chi).nes",          131088, 0x4522aa89, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_zeldanodenc)
STD_ROM_FN(nes_zeldanodenc)

struct BurnDriver BurnDrvnes_zeldanodenc = {
	"nes_zeldanodenc", "nes_legenzel", NULL, NULL, "2022",
	"Zelda no Densetsu - The Hyrule Fantasy (T-Chi)\0", "FDS to NES Hack", "NOKOH", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, nes_zeldanodencRomInfo, nes_zeldanodencRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Gomoku Narabe Renju (T-Eng)
// https://www.romhacking.net/translations/3073/
static struct BurnRomInfo nes_gomokuRomDesc[] = {
	{ "Gomoku Narabe Renju (T-Eng).nes",          24592, 0x18513cfc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_gomoku)
STD_ROM_FN(nes_gomoku)

struct BurnDriver BurnDrvnes_gomoku = {
	"nes_gomoku", NULL, NULL, NULL, "2017",
	"Gomoku Narabe Renju (T-Eng)\0", NULL, "Psyklax", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_STRATEGY, 0,
	NESGetZipName, nes_gomokuRomInfo, nes_gomokuRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_gomokujRomDesc[] = {
	{ "Gomoku Narabe Renju (Japan).nes",          24592, 0x28843bb1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_gomokuj)
STD_ROM_FN(nes_gomokuj)

struct BurnDriver BurnDrvnes_gomokuj = {
	"nes_gomokuj", "nes_gomoku", NULL, NULL, "1983",
	"Gomoku Narabe Renju (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_STRATEGY, 0,
	NESGetZipName, nes_gomokujRomInfo, nes_gomokujRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Gomoku Narabe Renju (T-Chi)
// https://tieba.baidu.com/p/7443334447
static struct BurnRomInfo nes_gomokucRomDesc[] = {
	{ "Gomoku Narabe Renju (T-Chi).nes",          24592, 0xbccf3f36, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_gomokuc)
STD_ROM_FN(nes_gomokuc)

struct BurnDriver BurnDrvnes_gomokuc = {
	"nes_gomokuc", "nes_gomoku", NULL, NULL, "2020",
	"Gomoku Narabe Renju (T-Chi)\0", NULL, "Yan Yu Mao Mao", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_STRATEGY, 0,
	NESGetZipName, nes_gomokucRomInfo, nes_gomokucRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Road Fighter (T-Chi, v1.2)
static struct BurnRomInfo nes_roadfightercRomDesc[] = {
	{ "Road Fighter (T-Chi, v1.2).nes",          24592, 0xb5d69e4c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_roadfighterc)
STD_ROM_FN(nes_roadfighterc)

struct BurnDriver BurnDrvnes_roadfighterc = {
	"nes_roadfighterc", "nes_roadfighter", NULL, NULL, "2009",
	"Road Fighter (T-Chi, v1.2)\0", NULL, "Jiu Ban", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RACING, 0,
	NESGetZipName, nes_roadfightercRomInfo, nes_roadfightercRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Kung Fu (T-Chi)
static struct BurnRomInfo nes_kungfucRomDesc[] = {
	{ "Kung Fu (T-Chi).nes",          24592, 0x480a29b7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_kungfuc)
STD_ROM_FN(nes_kungfuc)

struct BurnDriver BurnDrvnes_kungfuc = {
	"nes_kungfuc", "nes_kungfu", NULL, NULL, "2010",
	"Kung Fu (T-Chi)\0", NULL, "Jiu Ban", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_kungfucRomInfo, nes_kungfucRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Rad Racer (T-Chi)
static struct BurnRomInfo nes_radracercRomDesc[] = {
	{ "Rad Racer (T-Chi).nes",          131088, 0x13cfd9ca, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_radracerc)
STD_ROM_FN(nes_radracerc)

struct BurnDriver BurnDrvnes_radracerc = {
	"nes_radracerc", "nes_radracer", NULL, NULL, "2010",
	"Rad Racer (T-Chi)\0", NULL, "Jiu Ban", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RACING, 0,
	NESGetZipName, nes_radracercRomInfo, nes_radracercRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Rad Racer II (T-Chi)
static struct BurnRomInfo nes_radraceriicRomDesc[] = {
	{ "Rad Racer II (T-Chi).nes",          131088, 0x1d7e1802, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_radraceriic)
STD_ROM_FN(nes_radraceriic)

struct BurnDriver BurnDrvnes_radraceriic = {
	"nes_radraceriic", "nes_radracerii", NULL, NULL, "2010",
	"Rad Racer II (T-Chi)\0", NULL, "Jiu Ban", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RACING, 0,
	NESGetZipName, nes_radraceriicRomInfo, nes_radraceriicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Zen - Intergalactic Ninja (T-Chi)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=711394
/* It is recommended to use overclocking to reduce the flicker generated during screen switching */
static struct BurnRomInfo nes_zencRomDesc[] = {
	{ "Zen - Intergalactic Ninja (T-Chi).nes",          524304, 0xb8e36aee, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_zenc)
STD_ROM_FN(nes_zenc)

struct BurnDriver BurnDrvnes_zenc = {
	"nes_zenc", "nes_zen", NULL, NULL, "2022",
	"Zen - Intergalactic Ninja (T-Chi)\0", NULL, "Ni You De Wu Cai Yu", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_zencRomInfo, nes_zencRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Kick Master (T-Chi)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=711509
static struct BurnRomInfo nes_kickmastercRomDesc[] = {
	{ "Kick Master (T-Chi).nes",          393232, 0x85f44191, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_kickmasterc)
STD_ROM_FN(nes_kickmasterc)

struct BurnDriver BurnDrvnes_kickmasterc = {
	"nes_kickmasterc", "nes_kickmaster", NULL, NULL, "2022",
	"Kick Master (T-Chi)\0", NULL, "Ninsonytendog & Managus", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_kickmastercRomInfo, nes_kickmastercRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Makaimura (T-Chi)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=702939
static struct BurnRomInfo nes_makaimuracRomDesc[] = {
	{ "Makaimura (T-Chi).nes",          131088, 0x415d7c05, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_makaimurac)
STD_ROM_FN(nes_makaimurac)

struct BurnDriver BurnDrvnes_makaimurac = {
	"nes_makaimurac", "nes_ghostsngoblins", NULL, NULL, "2020",
	"Makaimura (T-Chi)\0", NULL, "Advance Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_makaimuracRomInfo, nes_makaimuracRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Double Dragon III - The Sacred Stones (T-Chi)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=711718
static struct BurnRomInfo nes_doubldraiiiscRomDesc[] = {
	{ "Double Dragon III - The Sacred Stones (T-Chi).nes",          262160, 0x4b992f63, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_doubldraiiisc)
STD_ROM_FN(nes_doubldraiiisc)

struct BurnDriver BurnDrvnes_doubldraiiisc = {
	"nes_doubldraiiisc", "nes_doubldraiii", NULL, NULL, "2022",
	"Double Dragon III - The Sacred Stones (T-Chi)\0", NULL, "asiwish", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_doubldraiiiscRomInfo, nes_doubldraiiiscRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Downtown - Nekketsu Monogatari (T-Chi)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=711718
static struct BurnRomInfo nes_downtnekmoncRomDesc[] = {
	{ "Downtown - Nekketsu Monogatari (T-Chi).nes",          262160, 0x60a9ee52, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_downtnekmonc)
STD_ROM_FN(nes_downtnekmonc)

struct BurnDriver BurnDrvnes_downtnekmonc = {
	"nes_downtnekmonc", "nes_downtnekmon", NULL, NULL, "2022",
	"Downtown - Nekketsu Monogatari (T-Chi)\0", NULL, "asiwish", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_downtnekmoncRomInfo, nes_downtnekmoncRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Mission Impossible (T-Chi)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=711906
static struct BurnRomInfo nes_missiimpcRomDesc[] = {
	{ "Mission Impossible (T-Chi).nes",          524304, 0xf930b169, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_missiimpc)
STD_ROM_FN(nes_missiimpc)

struct BurnDriver BurnDrvnes_missiimpc = {
	"nes_missiimpc", "nes_missiimp", NULL, NULL, "2022",
	"Mission Impossible (T-Chi)\0", NULL, "LSP Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN | GBF_ADV, 0,
	NESGetZipName, nes_missiimpcRomInfo, nes_missiimpcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Final Mission (T-Chi)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=708048
static struct BurnRomInfo nes_finalmissionjcRomDesc[] = {
	{ "Final Mission (T-Chi).nes",          393232, 0x9e25f63f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_finalmissionjc)
STD_ROM_FN(nes_finalmissionjc)

struct BurnDriver BurnDrvnes_finalmissionjc = {
	"nes_finalmissionjc", "nes_scat", NULL, NULL, "2021",
	"Final Mission (T-Chi)\0", NULL, "Ni You De Wu Cai Yu", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_HORSHOOT, 0,
	NESGetZipName, nes_finalmissionjcRomInfo, nes_finalmissionjcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Spider-Man - Return of the Sinister Six (T-Chi)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=708048
static struct BurnRomInfo nes_spidemancRomDesc[] = {
	{ "Spider-Man - Return of the Sinister Six (T-Chi).nes",          393232, 0xa116d626, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_spidemanc)
STD_ROM_FN(nes_spidemanc)

struct BurnDriver BurnDrvnes_spidemanc = {
	"nes_spidemanc", "nes_spideman", NULL, NULL, "2021",
	"Spider-Man - Return of the Sinister Six (T-Chi)\0", NULL, "Ni You De Wu Cai Yu", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_spidemancRomInfo, nes_spidemancRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Son Son (T-Chi)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=710703
static struct BurnRomInfo nes_sonsoncRomDesc[] = {
	{ "Son Son (T-Chi).nes",          81936, 0xe05a2451, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sonsonc)
STD_ROM_FN(nes_sonsonc)

struct BurnDriver BurnDrvnes_sonsonc = {
	"nes_sonsonc", "nes_sonson", NULL, NULL, "2022",
	"Son Son (T-Chi)\0", NULL, "NOKOH", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_sonsoncRomInfo, nes_sonsoncRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Kage no Densetsu (T-Chi)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=710703
static struct BurnRomInfo nes_kagenodensetsucRomDesc[] = {
	{ "Kage no Densetsu (T-Chi).nes",          106512, 0x170884af, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_kagenodensetsuc)
STD_ROM_FN(nes_kagenodensetsuc)

struct BurnDriver BurnDrvnes_kagenodensetsuc = {
	"nes_kagenodensetsuc", "nes_legenkag", NULL, NULL, "2022",
	"Kage no Densetsu (T-Chi)\0", NULL, "NOKOH", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_kagenodensetsucRomInfo, nes_kagenodensetsucRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Gyruss (T-Chi)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=710703
static struct BurnRomInfo nes_gyrusscRomDesc[] = {
	{ "Gyruss (T-Chi).nes",          188432, 0xb82b4f29, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_gyrussc)
STD_ROM_FN(nes_gyrussc)

struct BurnDriver BurnDrvnes_gyrussc = {
	"nes_gyrussc", "nes_gyruss", NULL, NULL, "2022",
	"Gyruss (T-Chi)\0", NULL, "NOKOH", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_SHOOT, 0,
	NESGetZipName, nes_gyrusscRomInfo, nes_gyrusscRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Downtown - Nekketsu Koushinkyoku - Soreyuke Daiundoukai (T-Chi)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=710703
static struct BurnRomInfo nes_downtnekkousordacRomDesc[] = {
	{ "Downtown - Nekketsu Koushinkyoku - Soreyuke Daiundoukai (T-Chi).nes",          483344, 0xd1728e58, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_downtnekkousordac)
STD_ROM_FN(nes_downtnekkousordac)

struct BurnDriver BurnDrvnes_downtnekkousordac = {
	"nes_downtnekkousordac", "nes_downtnekkousorda", NULL, NULL, "2022",
	"Downtown - Nekketsu Koushinkyoku - Soreyuke Daiundoukai (T-Chi)\0", NULL, "NOKOH", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_SPORTSMISC, 0,
	NESGetZipName, nes_downtnekkousordacRomInfo, nes_downtnekkousordacRomName, NULL, NULL, NULL, NULL, NES4ScoreInputInfo, NES4ScoreDIPInfo,
	NESHori4pInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Teenage Mutant Ninja Turtles - Tournament Fighters (T-Chi)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=710779
static struct BurnRomInfo nes_tmnttoufigscRomDesc[] = {
	{ "Teenage Mutant Ninja Turtles - Tournament Fighters (T-Chi).nes",          294928, 0x51ac1eb7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tmnttoufigsc)
STD_ROM_FN(nes_tmnttoufigsc)

struct BurnDriver BurnDrvnes_tmnttoufigsc = {
	"nes_tmnttoufigsc", "nes_tmnttoufig", NULL, NULL, "2022",
	"Teenage Mutant Ninja Turtles - Tournament Fighters (T-Chi)\0", NULL, "NOKOH", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_VSFIGHT, 0,
	NESGetZipName, nes_tmnttoufigscRomInfo, nes_tmnttoufigscRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Takahashi Meijin no Bouken-jima IV (T-Chi)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=710779
static struct BurnRomInfo nes_takameivicRomDesc[] = {
	{ "Takahashi Meijin no Bouken-jima IV (T-Chi).nes",          524304, 0x5d164133, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_takameivic)
STD_ROM_FN(nes_takameivic)

struct BurnDriver BurnDrvnes_takameivic = {
	"nes_takameivic", "nes_advenisliv", NULL, NULL, "2022",
	"Takahashi Meijin no Bouken-jima IV (T-Chi)\0", NULL, "NOKOH", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_takameivicRomInfo, nes_takameivicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Wai Wai World 2 - SOS!! Paseri Jou (T-Chi, v1.1)
// https://www.nesbbs.com/bbs/thread-53323-1-1.html
static struct BurnRomInfo nes_waiwaiwor2cRomDesc[] = {
	{ "Wai Wai World 2 - SOS!! Paseri Jou (T-Chi, v1.1).nes",          786448, 0xb03f3551, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_waiwaiwor2c)
STD_ROM_FN(nes_waiwaiwor2c)

struct BurnDriver BurnDrvnes_waiwaiwor2c = {
	"nes_waiwaiwor2c", "nes_waiwaiwor2", NULL, NULL, "2022",
	"Wai Wai World 2 - SOS!! Paseri Jou (T-Chi, v1.1)\0", NULL, "DMG Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_waiwaiwor2cRomInfo, nes_waiwaiwor2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// 1942 (T-Chi)
// https://www.nesbbs.com/bbs/thread-49846-1-1.html
static struct BurnRomInfo nes_1942cRomDesc[] = {
	{ "1942 (T-Chi).nes",          81936, 0x555741fc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_1942c)
STD_ROM_FN(nes_1942c)

struct BurnDriver BurnDrvnes_1942c = {
	"nes_1942c", "nes_1942", NULL, NULL, "2020",
	"1942 (T-Chi)\0", NULL, "6502", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_VERSHOOT, 0,
	NESGetZipName, nes_1942cRomInfo, nes_1942cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Little Nemo - The Dream Master (T-Chi)
// https://www.nesbbs.com/bbs/thread-43984-1-1.html [Logo by ZARD]
static struct BurnRomInfo nes_littlnemcRomDesc[] = {
	{ "Little Nemo - The Dream Master (T-Chi).nes",          327696, 0x841cbcd1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_littlnemc)
STD_ROM_FN(nes_littlnemc)

struct BurnDriver BurnDrvnes_littlnemc = {
	"nes_littlnemc", "nes_littlnem", NULL, NULL, "2017",
	"Little Nemo - The Dream Master (T-Chi)\0", NULL, "Ice Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_littlnemcRomInfo, nes_littlnemcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Little Magic (Japan)
static struct BurnRomInfo nes_littlemagicRomDesc[] = {
	{ "Little Magic (Japan).nes",          262160, 0xfc3c18eb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_littlemagic)
STD_ROM_FN(nes_littlemagic)

struct BurnDriver BurnDrvnes_littlemagic = {
	"nes_littlemagic", NULL, NULL, NULL, "1990",
	"Little Magic (Japan)\0", NULL, "Data East", "Miscellaneous",
	L"Little Magic (Japan)\0\u30ea\u30c8\u30eb\u30de\u30b8\u30c3\u30af\0", NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_littlemagicRomInfo, nes_littlemagicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Little Magic (T-Chi, MS Team 2nd Rev)
// MS Team 2nd Rev - 20071214
// http://fcpic.nesbbs.com/0zt/ms/Rom.asp?page=3
static struct BurnRomInfo nes_littlemagiccRomDesc[] = {
	{ "Little Magic (T-Chi, MS Team 2nd Rev).nes",          524304, 0xb0c1cd73, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_littlemagicc)
STD_ROM_FN(nes_littlemagicc)

struct BurnDriver BurnDrvnes_littlemagicc = {
	"nes_littlemagicc", "nes_littlemagic", NULL, NULL, "2007",
	"Little Magic (T-Chi, MS Team 2nd Rev)\0", NULL, "Waixing", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_littlemagiccRomInfo, nes_littlemagiccRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ninja Crusaders - Ryuuga (T-Chi)
// https://www.nesbbs.com/bbs/thread-48213-1-1.html
static struct BurnRomInfo nes_ninjacrusaderscRomDesc[] = {
	{ "Ninja Crusaders - Ryuuga (T-Chi).nes",          262160, 0x05fac5d5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ninjacrusadersc)
STD_ROM_FN(nes_ninjacrusadersc)

struct BurnDriver BurnDrvnes_ninjacrusadersc = {
	"nes_ninjacrusadersc", "nes_ninjacrusaders", NULL, NULL, "2019",
	"Ninja Crusaders - Ryuuga (T-Chi) \0", NULL, "HHNM Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_ninjacrusaderscRomInfo, nes_ninjacrusaderscRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Side Pocket (T-Chi, v2)
// https://www.nesbbs.com/bbs/thread-49290-1-1.html [Logo by ZARD]
static struct BurnRomInfo nes_sidepocketcRomDesc[] = {
	{ "Side Pocket (T-Chi, v2).nes",          188432, 0xe34da569, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sidepocketc)
STD_ROM_FN(nes_sidepocketc)

struct BurnDriver BurnDrvnes_sidepocketc = {
	"nes_sidepocketc", "nes_sidepocket", NULL, NULL, "2020",
	"Side Pocket (T-Chi, v2)\0", NULL, "Xing Kong", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_SPORTSMISC, 0,
	NESGetZipName, nes_sidepocketcRomInfo, nes_sidepocketcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Don Doko Don 2 (T-Chi)
// https://www.nesbbs.com/bbs/thread-50857-1-1.html
static struct BurnRomInfo nes_dondokodon2cRomDesc[] = {
	{ "Don Doko Don 2 (T-Chi).nes",          393232, 0x64bab6de, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dondokodon2c)
STD_ROM_FN(nes_dondokodon2c)

struct BurnDriver BurnDrvnes_dondokodon2c = {
	"nes_dondokodon2c", "nes_dondokodon2", NULL, NULL, "2020",
	"Don Doko Don 2 (T-Chi)\0", NULL, "Advance Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_dondokodon2cRomInfo, nes_dondokodon2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Tokkyuu Shirei Solbrain (T-Chi)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=705238
static struct BurnRomInfo nes_tokkyshisolcRomDesc[] = {
	{ "Tokkyuu Shirei Solbrain (T-Chi).nes",          524304, 0xe006a59f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tokkyshisolc)
STD_ROM_FN(nes_tokkyshisolc)

struct BurnDriver BurnDrvnes_tokkyshisolc = {
	"nes_tokkyshisolc", "nes_shatterhand", NULL, NULL, "2021",
	"Tokkyuu Shirei Solbrain (T-Chi)\0", NULL, "Ni You De Wu Cai Yu", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_tokkyshisolcRomInfo, nes_tokkyshisolcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Astyanax (T-Chi, v1.1)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=705238
static struct BurnRomInfo nes_astyanaxcRomDesc[] = {
	{ "Astyanax (T-Chi, v1.1).nes",          524304, 0x1517f03a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_astyanaxc)
STD_ROM_FN(nes_astyanaxc)

struct BurnDriver BurnDrvnes_astyanaxc = {
	"nes_astyanaxc", "nes_astyanax", NULL, NULL, "2021",
	"Astyanax (T-Chi, v1.1)\0", NULL, "Ni You De Wu Cai Yu", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_astyanaxcRomInfo, nes_astyanaxcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Chiki Chiki Machine Mou Race (T-Chi)
// https://www.nesbbs.com/bbs/thread-46582-1-1.html
static struct BurnRomInfo nes_chikichimacmouracRomDesc[] = {
	{ "Chiki Chiki Machine Mou Race (T-Chi).nes",          393232, 0x37e57151, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chikichimacmourac)
STD_ROM_FN(nes_chikichimacmourac)

struct BurnDriver BurnDrvnes_chikichimacmourac = {
	"nes_chikichimacmourac", "nes_wackyraces", NULL, NULL, "2018",
	"Chiki Chiki Machine Mou Race (T-Chi)\0", NULL, "Nonamed Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_chikichimacmouracRomInfo, nes_chikichimacmouracRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Bucky O'Hare (T-Chi)
// https://www.nesbbs.com/bbs/thread-44657-1-1.html [Logo by ZARD]
static struct BurnRomInfo nes_buckyoharecRomDesc[] = {
	{ "Bucky O'Hare (T-Chi).nes",          426000, 0xfaa6e27c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_buckyoharec)
STD_ROM_FN(nes_buckyoharec)

struct BurnDriver BurnDrvnes_buckyoharec = {
	"nes_buckyoharec", "nes_buckyohare", NULL, NULL, "2018",
	"Bucky O'Hare (T-Chi)\0", NULL, "Axi", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_buckyoharecRomInfo, nes_buckyoharecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Fester's Quest (T-Chi)
// https://www.nesbbs.com/bbs/thread-51920-1-1.html
static struct BurnRomInfo nes_festersquestcRomDesc[] = {
	{ "Fester's Quest (T-Chi).nes",          393232, 0x56a5335e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_festersquestc)
STD_ROM_FN(nes_festersquestc)

struct BurnDriver BurnDrvnes_festersquestc = {
	"nes_festersquestc", "nes_festersquest", NULL, NULL, "2021",
	"Fester's Quest (T-Chi)\0", NULL, "DMG Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN | GBF_ADV, 0,
	NESGetZipName, nes_festersquestcRomInfo, nes_festersquestcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Metal Gear (T-Chi)
// https://www.nesbbs.com/bbs/thread-31084-1-1.html
// https://www.nesbbs.com/bbs/thread-52305-1-1.html [mirroring fix]
static struct BurnRomInfo nes_metalgearscRomDesc[] = {
	{ "Metal Gear (T-Chi).nes",          262160, 0x3e2e5276, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_metalgearsc)
STD_ROM_FN(nes_metalgearsc)

struct BurnDriver BurnDrvnes_metalgearsc = {
	"nes_metalgearsc", "nes_metalgear", NULL, NULL, "2015",
	"Metal Gear (T-Chi)\0", NULL, "Axi", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN | GBF_ADV, 0,
	NESGetZipName, nes_metalgearscRomInfo, nes_metalgearscRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Golgo 13 - Daiisshou - Kamigami no Tasogare (T-Chi)
// https://www.nesbbs.com/bbs/thread-51921-1-1.html
static struct BurnRomInfo nes_golgo13scRomDesc[] = {
	{ "Golgo 13 - Daiisshou - Kamigami no Tasogare (T-Chi).nes",          524304, 0x9671760e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_golgo13sc)
STD_ROM_FN(nes_golgo13sc)

struct BurnDriver BurnDrvnes_golgo13sc = {
	"nes_golgo13sc", "nes_golgo13", NULL, NULL, "2021",
	"Golgo 13 - Daiisshou - Kamigami no Tasogare (T-Chi)\0", NULL, "DMG Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_SHOOT, 0,
	NESGetZipName, nes_golgo13scRomInfo, nes_golgo13scRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Sangokushi II - Haou no Tairiku (Japan)
static struct BurnRomInfo nes_sangokushiiihanouRomDesc[] = {
	{ "Sangokushi II - Haou no Tairiku (Japan).nes",          524304, 0x5c09bcaf, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sangokushiiihanou)
STD_ROM_FN(nes_sangokushiiihanou)

struct BurnDriver BurnDrvnes_sangokushiiihanou = {
	"nes_sangokushiiihanou", NULL, NULL, NULL, "1992",
	"Sangokushi II - Haou no Tairiku (Japan)\0", NULL, "Namco", "Miscellaneous",
	L"Sangokushi II - Haou no Tairiku (Japan)\0\u4e09\u56fd\u5fd7\u2161 - \u8987\u738b\u306e\u5927\u9678\0", NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_sangokushiiihanouRomInfo, nes_sangokushiiihanouRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Sangokushi II - Haou no Tairiku (T-Chi)
// https://www.nesbbs.com/bbs/thread-52813-1-1.html
static struct BurnRomInfo nes_sangokushiiihanoucRomDesc[] = {
	{ "Sangokushi II - Haou no Tairiku (T-Chi).nes",          524304, 0x611d79ff, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sangokushiiihanouc)
STD_ROM_FN(nes_sangokushiiihanouc)

struct BurnDriver BurnDrvnes_sangokushiiihanouc = {
	"nes_sangokushiiihanouc", "nes_sangokushiiihanou", NULL, NULL, "2017",
	"Sangokushi II - Haou no Tairiku (T-Chi)\0", NULL, "Unkown", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_sangokushiiihanoucRomInfo, nes_sangokushiiihanoucRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Mappy (T-Chi, v2.0)
// https://www.nesbbs.com/bbs/thread-43662-1-1.html
static struct BurnRomInfo nes_mappycRomDesc[] = {
	{ "Mappy (T-Chi, v2.0).nes",          40976, 0x7c922692, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mappyc)
STD_ROM_FN(nes_mappyc)

struct BurnDriver BurnDrvnes_mappyc = {
	"nes_mappyc", "nes_mappy", NULL, NULL, "2017",
	"Mappy (T-Chi, v2.0)\0", NULL, "Jiu Ban", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_mappycRomInfo, nes_mappycRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Kirby's Adventure (T-Chi)
// https://www.nesbbs.com/bbs/thread-50985-1-1.html
static struct BurnRomInfo nes_kirbysadvcRomDesc[] = {
	{ "Kirby's Adventure (T-Chi).nes",          1048592, 0x0efa7d0f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_kirbysadvc)
STD_ROM_FN(nes_kirbysadvc)

struct BurnDriver BurnDrvnes_kirbysadvc = {
	"nes_kirbysadvc", "nes_kirbysadv", NULL, NULL, "2020",
	"Kirby's Adventure (T-Chi)\0", NULL, "DMG Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_kirbysadvcRomInfo, nes_kirbysadvcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Fire Emblem Gaiden (T-Chi, v2.0)
// Modified by µ, άڶ, ѩʹ
// https://www.nesbbs.com/bbs/thread-51529-1-1.html
static struct BurnRomInfo nes_fireembgaicRomDesc[] = {
	{ "Fire Emblem Gaiden (T-Chi, v2.0).nes",          663568, 0x1df6fb24, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_fireembgaic)
STD_ROM_FN(nes_fireembgaic)

struct BurnDriver BurnDrvnes_fireembgaic = {
	"nes_fireembgaic", "nes_fireembgai", NULL, NULL, "2021",
	"Fire Emblem Gaiden (T-Chi, v2.0)\0", NULL, "Han Xue Shi Zhe", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_STRATEGY | GBF_RPG, 0,
	NESGetZipName, nes_fireembgaicRomInfo, nes_fireembgaicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Gimmick! (T-Chi)
// https://www.nesbbs.com/bbs/thread-52222-1-2.html
static struct BurnRomInfo nes_gimmickcRomDesc[] = {
	{ "Gimmick! (T-Chi).nes",          524304, 0xa442efa6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_gimmickc)
STD_ROM_FN(nes_gimmickc)

struct BurnDriver BurnDrvnes_gimmickc = {
	"nes_gimmickc", "nes_gimmick", NULL, NULL, "2021",
	"Gimmick! (T-Chi)\0", NULL, "Lei Jing Ling", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_gimmickcRomInfo, nes_gimmickcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Chip 'n Dale - Rescue Rangers (T-Chi, v1.1)
// https://www.nesbbs.com/bbs/thread-34058-1-1.html
static struct BurnRomInfo nes_chipndalresrascRomDesc[] = {
	{ "Chip 'n Dale - Rescue Rangers (T-Chi, v1.1).nes",          393232, 0x1c63b350, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chipndalresrasc)
STD_ROM_FN(nes_chipndalresrasc)

struct BurnDriver BurnDrvnes_chipndalresrasc = {
	"nes_chipndalresrasc", "nes_chipndalresra", NULL, NULL, "2015",
	"Chip 'n Dale - Rescue Rangers (T-Chi, v1.1)\0", NULL, "HHNM Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_chipndalresrascRomInfo, nes_chipndalresrascRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Chip 'n Dale - Rescue Rangers 2 (T-Chi, v1.13)
// https://www.nesbbs.com/bbs/thread-33949-1-1.html
static struct BurnRomInfo nes_chipndalresra2scRomDesc[] = {
	{ "Chip 'n Dale - Rescue Rangers 2 (T-Chi, v1.13).nes",          393232, 0x4acc32ba, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chipndalresra2sc)
STD_ROM_FN(nes_chipndalresra2sc)

struct BurnDriver BurnDrvnes_chipndalresra2sc = {
	"nes_chipndalresra2sc", "nes_chipndalresra2", NULL, NULL, "2015",
	"Chip 'n Dale - Rescue Rangers 2 (T-Chi, v1.13)\0", NULL, "HHNM Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_chipndalresra2scRomInfo, nes_chipndalresra2scRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Fire Emblem - Ankoku Ryuu to Hikari no Tsurugi
// Modified by µ, άڶ, ZARD/, ѩʹ
// https://www.nesbbs.com/bbs/thread-53027-1-1.html
static struct BurnRomInfo nes_fireembcRomDesc[] = {
	{ "Fire Emblem - Ankoku Ryuu to Hikari no Tsurugi (T-Chi).nes",          655376, 0xa3126533, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_fireembc)
STD_ROM_FN(nes_fireembc)

struct BurnDriver BurnDrvnes_fireembc = {
	"nes_fireembc", "nes_fireemb", NULL, NULL, "2022",
	"Fire Emblem - Ankoku Ryuu to Hikari no Tsurugi (T-Chi)\0", NULL, "Han Xue Shi Zhe", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_STRATEGY | GBF_RPG, 0,
	NESGetZipName, nes_fireembcRomInfo, nes_fireembcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Tiny Toon Adventures (T-Chi)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=709671&fromuid=17888
static struct BurnRomInfo nes_tinytooadvscRomDesc[] = {
	{ "Tiny Toon Adventures (T-Chi).nes",          344080, 0xfbf2bb98, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tinytooadvsc)
STD_ROM_FN(nes_tinytooadvsc)

struct BurnDriver BurnDrvnes_tinytooadvsc = {
	"nes_tinytooadvsc", "nes_tinytooadv", NULL, NULL, "2022",
	"Tiny Toon Adventures (T-Chi)\0", NULL, "Ni You De Wu Cai Yu", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_tinytooadvscRomInfo, nes_tinytooadvscRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Lupin Sansei - Pandora no Isan (T-Chi)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=705303&fromuid=17888
static struct BurnRomInfo nes_lupinsanscRomDesc[] = {
	{ "Lupin Sansei - Pandora no Isan (T-Chi).nes",          524304, 0x1fd2e927, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_lupinsansc)
STD_ROM_FN(nes_lupinsansc)

struct BurnDriver BurnDrvnes_lupinsansc = {
	"nes_lupinsansc", "nes_lupinsan", NULL, NULL, "2021",
	"Lupin Sansei - Pandora no Isan (T-Chi)\0", NULL, "Ni You De Wu Cai Yu", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_lupinsanscRomInfo, nes_lupinsanscRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Saint Seiya - Ougon Densetsu (T-Chi, v2.0)
// https://www.nesbbs.com/bbs/thread-45841-1-1.html
static struct BurnRomInfo nes_saintseiyaodcRomDesc[] = {
	{ "Saint Seiya - Ougon Densetsu (T-Chi, v2.0).nes",          1310736, 0xf7df4b56, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_saintseiyaodc)
STD_ROM_FN(nes_saintseiyaodc)

struct BurnDriver BurnDrvnes_saintseiyaodc = {
	"nes_saintseiyaodc", "nes_saintseiougden", NULL, NULL, "2018",
	"Saint Seiya - Ougon Densetsu (T-Chi, v2.0)\0", NULL, "HHNM Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM | GBF_RPG, 0,
	NESGetZipName, nes_saintseiyaodcRomInfo, nes_saintseiyaodcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Saint Seiya - Ougon Densetsu Kanketsu Hen (T-Chi)
// https://www.nesbbs.com/bbs/thread-40836-1-1.html
static struct BurnRomInfo nes_saintseiyaodkhcRomDesc[] = {
	{ "Saint Seiya - Ougon Densetsu Kanketsu Hen (T-Chi).nes",          524304, 0xb1651b70, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_saintseiyaodkhc)
STD_ROM_FN(nes_saintseiyaodkhc)

struct BurnDriver BurnDrvnes_saintseiyaodkhc = {
	"nes_saintseiyaodkhc", "nes_saintseiougdenkahen", NULL, NULL, "2017",
	"Saint Seiya - Ougon Densetsu Kanketsu Hen (T-Chi)\0", NULL, "HHNM Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM | GBF_RPG, 0,
	NESGetZipName, nes_saintseiyaodkhcRomInfo, nes_saintseiyaodkhcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// AV Mahjong Club (T-Chi) (Unl)
// https://www.nesbbs.com/bbs/thread-45883-1-1.html
static struct BurnRomInfo nes_avmahjongclubcRomDesc[] = {
	{ "AV Mahjong Club (T-Chi) (Unl).nes",          262160, 0x663870dd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_avmahjongclubc)
STD_ROM_FN(nes_avmahjongclubc)

struct BurnDriver BurnDrvnes_avmahjongclubc = {
	"nes_avmahjongclubc", NULL, NULL, NULL, "2018",
	"AV Mahjong Club (T-Chi) (Unl)\0", NULL, "HHNM Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_NES, GBF_MAHJONG | GBF_ADV, 0,
	NESGetZipName, nes_avmahjongclubcRomInfo, nes_avmahjongclubcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Jackal (T-Chi, v1.1)
// https://www.nesbbs.com/bbs/thread-49033-1-1.html
static struct BurnRomInfo nes_jackalcRomDesc[] = {
	{ "Jackal (T-Chi, v1.1).nes",          262160, 0x0eca3358, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_jackalc)
STD_ROM_FN(nes_jackalc)

struct BurnDriver BurnDrvnes_jackalc = {
	"nes_jackalc", "nes_jackal", NULL, NULL, "2020",
	"Jackal (T-Chi, v1.1)\0", NULL, "Lei Jing Ling", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 4, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_jackalcRomInfo, nes_jackalcRomName, NULL, NULL, NULL, NULL, NES4ScoreInputInfo, NES4ScoreDIPInfo,
	NESHori4pInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Darkwing Duck (T-Chi)
// https://www.nesbbs.com/bbs/thread-50681-1-1.html
static struct BurnRomInfo nes_darkwingduckcRomDesc[] = {
	{ "Darkwing Duck (T-Chi).nes",          393232, 0x585f3500, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_darkwingduckc)
STD_ROM_FN(nes_darkwingduckc)

struct BurnDriver BurnDrvnes_darkwingduckc = {
	"nes_darkwingduckc", "nes_darkwingduck", NULL, NULL, "2020",
	"Darkwing Duck (T-Chi)\0", NULL, "HHNM Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_darkwingduckcRomInfo, nes_darkwingduckcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Guardic Gaiden (T-Chi, v1.1)
// https://www.nesbbs.com/bbs/thread-42503-1-1.html
static struct BurnRomInfo nes_guardicgaidencRomDesc[] = {
	{ "Guardic Gaiden (T-Chi, v1.1).nes",          262160, 0x77df8d83, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_guardicgaidenc)
STD_ROM_FN(nes_guardicgaidenc)

struct BurnDriver BurnDrvnes_guardicgaidenc = {
	"nes_guardicgaidenc", "nes_guardleg", NULL, NULL, "2017",
	"Guardic Gaiden (T-Chi, v1.1)\0", NULL, "HHNM Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_SHOOT | GBF_ADV, 0,
	NESGetZipName, nes_guardicgaidencRomInfo, nes_guardicgaidencRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ninja Ryuuken Den III - Yomi no Hakobune (T-Chi)
// Translation by TPU
static struct BurnRomInfo nes_ninjaryudeniiicRomDesc[] = {
	{ "Ninja Ryuuken Den III - Yomi no Hakobune (T-Chi).nes",          344080, 0xba865119, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ninjaryudeniiic)
STD_ROM_FN(nes_ninjaryudeniiic)

struct BurnDriver BurnDrvnes_ninjaryudeniiic = {
	"nes_ninjaryudeniiic", "nes_ninjagaiiii", NULL, NULL, "2002",
	"Ninja Ryuuken Den III - Yomi no Hakobune (T-Chi)\0", NULL, "TPU", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_SCRFIGHT | GBF_PLATFORM, 0,
	NESGetZipName, nes_ninjaryudeniiicRomInfo, nes_ninjaryudeniiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ninja Ryuuken Den II - Ankoku no Jashin Ken (T-Chi)
// Translation by TPU
static struct BurnRomInfo nes_ninjaryudeniicRomDesc[] = {
	{ "Ninja Ryuuken Den II - Ankoku no Jashin Ken (T-Chi).nes",          335888, 0xf7d5d63b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ninjaryudeniic)
STD_ROM_FN(nes_ninjaryudeniic)

struct BurnDriver BurnDrvnes_ninjaryudeniic = {
	"nes_ninjaryudeniic", "nes_ninjagaiii", NULL, NULL, "200?",
	"Ninja Ryuuken Den II - Ankoku no Jashin Ken (T-Chi)\0", NULL, "TPU", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_SCRFIGHT | GBF_PLATFORM, 0,
	NESGetZipName, nes_ninjaryudeniicRomInfo, nes_ninjaryudeniicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ninja Ryuuken Den (T-Chi)
// Translation by FlameCyclone
static struct BurnRomInfo nes_ninjaryudencRomDesc[] = {
	{ "Ninja Ryuuken Den (T-Chi).nes",          524304, 0x7724d3d0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ninjaryudenc)
STD_ROM_FN(nes_ninjaryudenc)

struct BurnDriver BurnDrvnes_ninjaryudenc = {
	"nes_ninjaryudenc", "nes_ninjagaiden", NULL, NULL, "2017",
	"Ninja Ryuuken Den (T-Chi)\0", NULL, "FlameCyclone", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_SCRFIGHT | GBF_PLATFORM, 0,
	NESGetZipName, nes_ninjaryudencRomInfo, nes_ninjaryudencRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ningen Heiki - Dead Fox (T-Chi, v2.0)
// Translation by Xing Kong
static struct BurnRomInfo nes_deadfoxcRomDesc[] = {
	{ "Ningen Heiki - Dead Fox (T-Chi, v2.0).nes",          303120, 0x3cf5df56, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_deadfoxc)
STD_ROM_FN(nes_deadfoxc)

struct BurnDriver BurnDrvnes_deadfoxc = {
	"nes_deadfoxc", "nes_codenam", NULL, NULL, "2010",
	"Ningen Heiki - Dead Fox (T-Chi, v2.0)\0", NULL, "Xing Kong", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_deadfoxcRomInfo, nes_deadfoxcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Kage (T-Chi)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=706290
static struct BurnRomInfo nes_kagecRomDesc[] = {
	{ "Kage (T-Chi).nes",          270352, 0x2ff63b35, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_kagec)
STD_ROM_FN(nes_kagec)

struct BurnDriver BurnDrvnes_kagec = {
	"nes_kagec", "nes_shadonin", NULL, NULL, "2021",
	"Kage (T-Chi)\0", NULL, "Ni You De Wu Cai Yu", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_SCRFIGHT | GBF_PLATFORM, 0,
	NESGetZipName, nes_kagecRomInfo, nes_kagecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Super Mario Bros. 3 (T-Chi)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=710779
static struct BurnRomInfo nes_smb3cRomDesc[] = {
	{ "Super Mario Bros. 3 (T-Chi).nes",          458768, 0xe39b725d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smb3c)
STD_ROM_FN(nes_smb3c)

struct BurnDriver BurnDrvnes_smb3c = {
	"nes_smb3c", "nes_smb3", NULL, NULL, "2016",
	"Super Mario Bros. 3 (T-Chi)\0", NULL, "NOKOH", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_smb3cRomInfo, nes_smb3cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Akumajou Dracula (T-Chi)
// Translation by Fo Yi Team - Xi Xue Nan Jue
static struct BurnRomInfo nes_akumajoudraculacRomDesc[] = {
	{ "Akumajou Dracula (T-Chi).nes",          131088, 0xdc749e67, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_akumajoudraculac)
STD_ROM_FN(nes_akumajoudraculac)

struct BurnDriver BurnDrvnes_akumajoudraculac = {
	"nes_akumajoudraculac", "nes_castlevania", NULL, NULL, "199?",
	"Akumajou Dracula (T-Chi)\0", NULL, "Fo Yi Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_akumajoudraculacRomInfo, nes_akumajoudraculacRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Castlevania - The Holy Relics (Hack)
// Modified by Optomon, Setz, Tessa Montoya, Dr. Mario, Boneless Ivar
// https://www.romhacking.net/hacks/3759/ - 20171201
static struct BurnRomInfo nes_castlevaniahrRomDesc[] = {
	{ "Castlevania - The Holy Relics (Hack).nes",          131088, 0x8603cbbb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_castlevaniahr)
STD_ROM_FN(nes_castlevaniahr)

struct BurnDriver BurnDrvnes_castlevaniahr = {
	"nes_castlevaniahr", "nes_castlevania", NULL, NULL, "2017",
	"Castlevania - The Holy Relics (Hack)\0", NULL, "Optomon", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_castlevaniahrRomInfo, nes_castlevaniahrRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Castlevania - Chorus of Mysteries (v1.1, Hack)
// Modified by Dr. Mario, Optomon
// https://www.romhacking.net/hacks/263/ - 20070618
static struct BurnRomInfo nes_castlevaniacomRomDesc[] = {
	{ "Castlevania - Chorus of Mysteries (v1.1, Hack).nes",          131088, 0xbad05634, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_castlevaniacom)
STD_ROM_FN(nes_castlevaniacom)

struct BurnDriver BurnDrvnes_castlevaniacom = {
	"nes_castlevaniacom", "nes_castlevania", NULL, NULL, "2007",
	"Castlevania - Chorus of Mysteries (Hack, v1.1)\0", NULL, "Optomon & Dr. Mario", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_castlevaniacomRomInfo, nes_castlevaniacomRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Castlevania - The Last Tear (Hack, v2.78.5)
// Modified by Sky Flare
// https://mail.romhacking.net/hacks/7215/ - 20230223
static struct BurnRomInfo nes_castlevanialtRomDesc[] = {
	{ "Castlevania - The Last Tear v2.78.5 (2023)(Sky Flare).nes",          131088, 0x0839e2d4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_castlevanialt)
STD_ROM_FN(nes_castlevanialt)

struct BurnDriver BurnDrvnes_castlevanialt = {
	"nes_castlevanialt", "nes_castlevania", NULL, NULL, "2023",
	"Castlevania - The Last Tear (Hack, v2.78.5)\0", NULL, "Sky Flare", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_castlevanialtRomInfo, nes_castlevanialtRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Contra (T-Chi)
// Translation by FlameCyclone
static struct BurnRomInfo nes_contracRomDesc[] = {
	{ "Contra (T-Chi).nes",          393232, 0x2a39628d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_contrac)
STD_ROM_FN(nes_contrac)

struct BurnDriver BurnDrvnes_contrac = {
	"nes_contrac", "nes_contra", NULL, NULL, "2017",
	"Contra (T-Chi)\0", NULL, "FlameCyclone", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_contracRomInfo, nes_contracRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Wai Wai World (T-Chi)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=710779
static struct BurnRomInfo nes_waiwaiworldcRomDesc[] = {
	{ "Wai Wai World (T-Chi).nes",          524304, 0x7fe5c9f1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_waiwaiworldc)
STD_ROM_FN(nes_waiwaiworldc)

struct BurnDriver BurnDrvnes_waiwaiworldc = {
	"nes_waiwaiworldc", "nes_waiwaiworld", NULL, NULL, "2022",
	"Wai Wai World (T-Chi)\0", NULL, "NOKOH", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_waiwaiworldcRomInfo, nes_waiwaiworldcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Saiyuuki World (T-Chi, v1.04)
// https://www.nesbbs.com/bbs/thread-34610-1-1.html
static struct BurnRomInfo nes_saiyuukiworldcRomDesc[] = {
	{ "Saiyuuki World (T-Chi, v1.04).nes",          262160, 0x6e66cf6f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_saiyuukiworldc)
STD_ROM_FN(nes_saiyuukiworldc)

struct BurnDriver BurnDrvnes_saiyuukiworldc = {
	"nes_saiyuukiworldc", "nes_saiyuukiworld", NULL, NULL, "2016",
	"Saiyuuki World (T-Chi, v1.04)\0", NULL, "HHNM Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM | GBF_ADV, 0,
	NESGetZipName, nes_saiyuukiworldcRomInfo, nes_saiyuukiworldcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Mad City (T-Chi, v2)
// Translation by Tan Xiao Feng Sheng
static struct BurnRomInfo nes_madcitycRomDesc[] = {
	{ "Mad City (T-Chi, v2).nes",          393232, 0x3b2ad28b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_madcityc)
STD_ROM_FN(nes_madcityc)

struct BurnDriver BurnDrvnes_madcityc = {
	"nes_madcityc", "nes_advenbaybil", NULL, NULL, "2018",
	"Mad City (T-Chi, v2)\0", NULL, "Tan Xiao Feng Sheng", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_SCRFIGHT | GBF_SHOOT, 0,
	NESGetZipName, nes_madcitycRomInfo, nes_madcitycRomName, NULL, NULL, NULL, NULL, NESZapperInputInfo, NESZapperDIPInfo,
	NESZapperInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Contra Force (2nd option, T-Chi)
// Translation by Tan Xiao Feng Sheng
static struct BurnRomInfo nes_contraforcecRomDesc[] = {
	{ "Contra Force (T-Chi).nes",          524304, 0xcf2a15fb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_contraforcec)
STD_ROM_FN(nes_contraforcec)

struct BurnDriver BurnDrvnes_contraforcec = {
	"nes_contraforcec", "nes_contraforce", NULL, NULL, "2018",
	"Contra Force (2nd option, T-Chi)\0", NULL, "Tan Xiao Feng Sheng", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_contraforcecRomInfo, nes_contraforcecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Gun Nac (T-Chi)
static struct BurnRomInfo nes_gunnaccRomDesc[] = {
	{ "Gun Nac (T-Chi).nes",          393232, 0x3c63f85a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_gunnacc)
STD_ROM_FN(nes_gunnacc)

struct BurnDriver BurnDrvnes_gunnacc = {
	"nes_gunnacc", "nes_gunnac", NULL, NULL, "2013",
	"Gun Nac (T-Chi)\0", NULL, "Jiu Ban", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_VERSHOOT, 0,
	NESGetZipName, nes_gunnaccRomInfo, nes_gunnaccRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Mitsume ga Tooru (T-Chi, v3)
// https://www.ppxclub.com/forum.php?mod=viewthread&tid=710666
static struct BurnRomInfo nes_mitsumegatoorucRomDesc[] = {
	{ "Mitsume ga Tooru (T-Chi, v3).nes",          393232, 0x5087a64d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mitsumegatooruc)
STD_ROM_FN(nes_mitsumegatooruc)

struct BurnDriver BurnDrvnes_mitsumegatooruc = {
	"nes_mitsumegatooruc", "nes_mitsumegatooru", NULL, NULL, "2022",
	"Mitsume ga Tooru (T-Chi, v3)\0", NULL, "Ni You De Wu Cai Yu", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_mitsumegatoorucRomInfo, nes_mitsumegatoorucRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Akumajou Densetsu (T-Chi, v2.3)
// https://www.nesbbs.com/bbs/thread-46102-1-1.html
static struct BurnRomInfo nes_akumadencRomDesc[] = {
	{ "Akumajou Densetsu (T-Chi, v2.3).nes",          524304, 0xb810b79c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_akumadenc)
STD_ROM_FN(nes_akumadenc)

struct BurnDriver BurnDrvnes_akumadenc = {
	"nes_akumadenc", "nes_castliii", NULL, NULL, "2018",
	"Akumajou Densetsu (T-Chi, v2.3)\0", NULL, "HHNM Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_akumadencRomInfo, nes_akumadencRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Batman - The Video Game (T-Chi)
// https://www.nesbbs.com/bbs/thread-48898-1-1.html
static struct BurnRomInfo nes_batmavidgamcRomDesc[] = {
	{ "Batman - The Video Game (T-Chi).nes",          393232, 0x4b2de665, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_batmavidgamc)
STD_ROM_FN(nes_batmavidgamc)

struct BurnDriver BurnDrvnes_batmavidgamc = {
	"nes_batmavidgamc", "nes_batmavidgam", NULL, NULL, "2020",
	"Batman - The Video Game (T-Chi)\0", NULL, "HHNM Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_batmavidgamcRomInfo, nes_batmavidgamcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Salamander (T-Chi, v1.2)
// https://www.nesbbs.com/bbs/thread-49514-1-1.html
static struct BurnRomInfo nes_salamandercRomDesc[] = {
	{ "Salamander (T-Chi, v1.2).nes",          393232, 0xfbe84b97, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_salamanderc)
STD_ROM_FN(nes_salamanderc)

struct BurnDriver BurnDrvnes_salamanderc = {
	"nes_salamanderc", "nes_salamander", NULL, NULL, "2020",
	"Salamander (T-Chi, v1.2)\0", NULL, "Lei Jing Ling", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_HORSHOOT | GBF_VERSHOOT, 0,
	NESGetZipName, nes_salamandercRomInfo, nes_salamandercRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Gun-Dec (T-Chi)
// https://www.nesbbs.com/bbs/thread-47884-1-1.html
static struct BurnRomInfo nes_gundeccRomDesc[] = {
	{ "Gun-Dec (T-Chi).nes",          393232, 0x647eccfa, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_gundecc)
STD_ROM_FN(nes_gundecc)

struct BurnDriver BurnDrvnes_gundecc = {
	"nes_gundecc", "nes_vice", NULL, NULL, "2019",
	"Gun-Dec (T-Chi)\0", NULL, "HHNM Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_SHOOT | GBF_PLATFORM, 0,
	NESGetZipName, nes_gundeccRomInfo, nes_gundeccRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Teenage Mutant Ninja Turtles (T-Chi, v1.1)
// Modified by MEZARANSU, άڶ, 
// https://www.nesbbs.com/bbs/thread-49880-1-1.html
static struct BurnRomInfo nes_tmntcRomDesc[] = {
	{ "Teenage Mutant Ninja Turtles (T-Chi, v1.1).nes",          393232, 0xa02316dc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tmntc)
STD_ROM_FN(nes_tmntc)

struct BurnDriver BurnDrvnes_tmntc = {
	"nes_tmntc", "nes_tmnt", NULL, NULL, "2021",
	"Teenage Mutant Ninja Turtles (T-Chi, v1.1)\0", NULL, "MEZARANSU", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_tmntcRomInfo, nes_tmntcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Super Chinese II - Dragon Kid (T-Chi, v1.4)
// https://www.nesbbs.com/bbs/thread-46648-1-1.html
static struct BurnRomInfo nes_superchiiRomDesc[] = {
	{ "Super Chinese II - Dragon Kid (T-Chi, v1.4).nes",          524304, 0xa9e92ef3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_superchii)
STD_ROM_FN(nes_superchii)

struct BurnDriver BurnDrvnes_superchii = {
	"nes_superchii", "nes_littlninbro", NULL, NULL, "2019",
	"Super Chinese II - Dragon Kid (T-Chi, v1.4)\0", NULL, "HHNM Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_RUNGUN | GBF_RPG, 0,
	NESGetZipName, nes_superchiiRomInfo, nes_superchiiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Majou Densetsu II - Daimashikyou Galious (T-Chi, v2.2)
// https://www.nesbbs.com/bbs/thread-45465-1-2.html
static struct BurnRomInfo nes_majoudeniicRomDesc[] = {
	{ "Majou Densetsu II - Daimashikyou Galious (T-Chi, v2.2).nes",          262160, 0xa032170a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_majoudeniic)
STD_ROM_FN(nes_majoudeniic)

struct BurnDriver BurnDrvnes_majoudeniic = {
	"nes_majoudeniic", "nes_majoudenii", NULL, NULL, "2018",
	"Majou Densetsu II - Daimashikyou Galious (T-Chi, v2.2)\0", NULL, "HHNM Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_majoudeniicRomInfo, nes_majoudeniicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Silk Worm (T-Chi, v1.1)
// https://www.nesbbs.com/bbs/thread-48769-1-2.html
static struct BurnRomInfo nes_silkwormcRomDesc[] = {
	{ "Silk Worm (T-Chi, v1.1).nes",          393232, 0x3d53c8ad, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_silkwormc)
STD_ROM_FN(nes_silkwormc)

struct BurnDriver BurnDrvnes_silkwormc = {
	"nes_silkwormc", "nes_silkworm", NULL, NULL, "2020",
	"Silk Worm (T-Chi, v1.1)\0", NULL, "HHNM Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_HORSHOOT, 0,
	NESGetZipName, nes_silkwormcRomInfo, nes_silkwormcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Argos no Senshi (T-Chi, v1.11)
// https://www.nesbbs.com/bbs/thread-34446-1-1.html
static struct BurnRomInfo nes_argosnosenshicRomDesc[] = {
	{ "Argos no Senshi (T-Chi, v1.11).nes",          262160, 0xa1a22c10, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_argosnosenshic)
STD_ROM_FN(nes_argosnosenshic)

struct BurnDriver BurnDrvnes_argosnosenshic = {
	"nes_argosnosenshic", "nes_rygar", NULL, NULL, "2016",
	"Argos no Senshi (T-Chi, v1.11)\0", NULL, "HHNM Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_argosnosenshicRomInfo, nes_argosnosenshicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Super Contra (T-Chi)
// http://flamecyclone.ysepan.com/
static struct BurnRomInfo nes_supercontracRomDesc[] = {
	{ "Super Contra (T-Chi).nes",          393232, 0x8ccbab43, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_supercontrac)
STD_ROM_FN(nes_supercontrac)

struct BurnDriver BurnDrvnes_supercontrac = {
	"nes_supercontrac", "nes_superc", NULL, NULL, "2018",
	"Super Contra (T-Chi)\0", NULL, "FlameCyclone", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_supercontracRomInfo, nes_supercontracRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Hebereke (T-Chi)
// https://www.nesbbs.com/bbs/thread-53253-1-1.html
static struct BurnRomInfo nes_heberekecRomDesc[] = {
	{ "Hebereke (T-Chi).nes",          393232, 0x0691129d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_heberekec)
STD_ROM_FN(nes_heberekec)

struct BurnDriver BurnDrvnes_heberekecc= {
	"nes_heberekec", "nes_ufouria", NULL, NULL, "2022",
	"Hebereke (T-Chi)\0", NULL, "HHNM Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM | GBF_ADV, 0,
	NESGetZipName, nes_heberekecRomInfo, nes_heberekecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Gun.Smoke (T-Chi)
// http://www.dmgrom.com/cnpage.html
static struct BurnRomInfo nes_gunsmokecRomDesc[] = {
	{ "Gun.Smoke (T-Chi).nes",          163856, 0x4e0b7a7d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_gunsmokec)
STD_ROM_FN(nes_gunsmokec)

struct BurnDriver BurnDrvnes_gunsmokec = {
	"nes_gunsmokec", "nes_gunsmoke", NULL, NULL, "2019",
	"Gun.Smoke (T-Chi)\0", NULL, "DMG Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_gunsmokecRomInfo, nes_gunsmokecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Mappy Kids (T-Chi, v1.1)
// https://www.nesbbs.com/bbs/thread-48358-1-1.html
static struct BurnRomInfo nes_mappykidscRomDesc[] = {
	{ "Mappy Kids (T-Chi, v1.1).nes",          524304, 0x159bc653, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mappykidsc)
STD_ROM_FN(nes_mappykidsc)

struct BurnDriver BurnDrvnes_mappykidsc = {
	"nes_mappykidsc", "nes_mappykids", NULL, NULL, "2019",
	"Mappy Kids (T-Chi, v1.1)\0", NULL, "Lei Jing Ling", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_mappykidscRomInfo, nes_mappykidscRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Daiku no Gen-san 2 - Akage no Dan no Gyakushuu (T-Chi)
// https://www.nesbbs.com/bbs/thread-47859-1-1.html
static struct BurnRomInfo nes_daikunogensan2cRomDesc[] = {
	{ "Daiku no Gen-san 2 - Akage no Dan no Gyakushuu (T-Chi).nes",          524304, 0x66c08cd8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_daikunogensan2c)
STD_ROM_FN(nes_daikunogensan2c)

struct BurnDriver BurnDrvnes_daikunogensan2c = {
	"nes_daikunogensan2c", "nes_daikunogensan2", NULL, NULL, "2019",
	"Daiku no Gen-san 2 - Akage no Dan no Gyakushuu (T-Chi)\0", NULL, "Advance Team & DMG Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_daikunogensan2cRomInfo, nes_daikunogensan2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Saiyuuki World 2 - Tenjoukai no Majin (T-Chi)
// https://www.nesbbs.com/bbs/thread-48369-1-1.html
static struct BurnRomInfo nes_saiyuwor2cRomDesc[] = {
	{ "Saiyuuki World 2 - Tenjoukai no Majin (T-Chi).nes",          262160, 0x784fc411, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_saiyuwor2c)
STD_ROM_FN(nes_saiyuwor2c)

struct BurnDriver BurnDrvnes_saiyuwor2c = {
	"nes_saiyuwor2c", "nes_whompem", NULL, NULL, "2019",
	"Saiyuuki World 2 - Tenjoukai no Majin (T-Chi)\0", NULL, "nineswords", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_saiyuwor2cRomInfo, nes_saiyuwor2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// DuckTales 2 (T-Chi, v1.2)
// https://www.nesbbs.com/bbs/thread-47555-1-1.html
static struct BurnRomInfo nes_ducktales2cRomDesc[] = {
	{ "DuckTales 2 (T-Chi, v1.2).nes",          262160, 0x26815134, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ducktales2c)
STD_ROM_FN(nes_ducktales2c)

struct BurnDriver BurnDrvnes_ducktales2c = {
	"nes_ducktales2c", "nes_ducktales2", NULL, NULL, "2021",
	"DuckTales 2 (T-Chi, v1.2)\0", NULL, "DMG Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_ducktales2cRomInfo, nes_ducktales2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Daiku no Gen-san (T-Chi)
// https://www.nesbbs.com/bbs/thread-47118-1-1.html
static struct BurnRomInfo nes_daikunogensancRomDesc[] = {
	{ "Daiku no Gen-san (T-Chi).nes",          393232, 0xf0543c45, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_daikunogensanc)
STD_ROM_FN(nes_daikunogensanc)

struct BurnDriver BurnDrvnes_daikunogensanc = {
	"nes_daikunogensanc", "nes_daikunogensan", NULL, NULL, "2019",
	"Daiku no Gen-san (T-Chi)\0", NULL, "DMG Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_daikunogensancRomInfo, nes_daikunogensancRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Jigoku Gokuraku Maru (T-Chi)
// https://www.nesbbs.com/bbs/thread-48502-1-1.html
static struct BurnRomInfo nes_jigokgokmarcRomDesc[] = {
	{ "Jigoku Gokuraku Maru (T-Chi).nes",          393232, 0x375bf357, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_jigokgokmarc)
STD_ROM_FN(nes_jigokgokmarc)

struct BurnDriver BurnDrvnes_jigokgokmarc = {
	"nes_jigokgokmarc", "nes_kabuki", NULL, NULL, "2019",
	"Jigoku Gokuraku Maru (T-Chi)\0", NULL, "DMG Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_jigokgokmarcRomInfo, nes_jigokgokmarcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_triforcegcRomDesc[] = {
	{ "Legend of Zelda - Triforce of the Gods (China).nes",          2097168, 0x3a1b4502, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_triforcegc)
STD_ROM_FN(nes_triforcegc)

struct BurnDriver BurnDrvnes_triforcegc = {
	"nes_triforcegc", "nes_triforceg", NULL, NULL, "2004",
	"Legend of Zelda - Triforce of the Gods (China)\0", NULL, "Waixing", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, nes_triforcegcRomInfo, nes_triforcegcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_triforcegRomDesc[] = {
	{ "Legend of Zelda - Triforce of the Gods (T-Eng).nes",          2097168, 0x0be2d8fd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_triforceg)
STD_ROM_FN(nes_triforceg)

struct BurnDriver BurnDrvnes_triforceg = {
	"nes_triforceg", NULL, NULL, NULL, "2018",
	"Legend of Zelda - Triforce of the Gods (T-Eng)\0", NULL, "pacnsacdave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_NES, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, nes_triforcegRomInfo, nes_triforcegRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Wanpaku Duck Yume Bouken (T-Chi, v1.1)
// https://www.nesbbs.com/bbs/thread-51129-1-1.html
static struct BurnRomInfo nes_wanpakuduckcRomDesc[] = {
	{ "Wanpaku Duck Yume Bouken (T-Chi, v1.1).nes",          196624, 0x7c952f46, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_wanpakuduckc)
STD_ROM_FN(nes_wanpakuduckc)

struct BurnDriver BurnDrvnes_wanpakuduckc = {
	"nes_wanpakuduckc", "nes_ducktales", NULL, NULL, "2021",
	"Wanpaku Duck Yume Bouken (T-Chi, v1.1)\0", NULL, "Advance Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_wanpakuduckcRomInfo, nes_wanpakuduckcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Snow Brothers (T-Chi, v1.1)
// https://www.nesbbs.com/bbs/thread-53262-1-1.html
static struct BurnRomInfo nes_snowbrothersscRomDesc[] = {
	{ "Snow Brothers (T-Chi, v1.1).nes",          393232, 0xdac8e3dc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_snowbrotherssc)
STD_ROM_FN(nes_snowbrotherssc)

struct BurnDriver BurnDrvnes_snowbrotherssc = {
	"nes_snowbrotherssc", "nes_snowbrothers", NULL, NULL, "2022",
	"Snow Brothers (T-Chi, v1.1)\0", NULL, "LSP Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_snowbrothersscRomInfo, nes_snowbrothersscRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//  Addams Family, The - Pugsley's Scavenger Hunt (USA) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_addamfampugsschucRomDesc[] = {
	{ "addams family, the - pugsley's scavenger hunt (usa) - castellano v1.1.nes",          262160, 0x54bbec54, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_addamfampugsschuc)
STD_ROM_FN(nes_addamfampugsschuc)

struct BurnDriver BurnDrvnes_addamfampugsschuc = {
	"nes_addamfampugsschuc", "nes_addamfampugsschu", NULL, NULL, "2020",
	"Addams Family, The - Pugsley's Scavenger Hunt (Hack, Spanish v1.1)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_addamfampugsschucRomInfo, nes_addamfampugsschucRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//  Addams Family, The (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_addamfamcRomDesc[] = {
	{ "addams family, the (usa) - castellano v1.0.nes",          262160, 0x6bd3e973, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_addamfamc)
STD_ROM_FN(nes_addamfamc)

struct BurnDriver BurnDrvnes_addamfamc = {
	"nes_addamfamc", "nes_addamfam", NULL, NULL, "2020",
	"Addams Family, The (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_addamfamcRomInfo, nes_addamfamcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//  Adventure Island (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_adventureislandcRomDesc[] = {
	{ "adventure island (usa) - castellano v1.0.nes",          65552, 0xabf6745d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_adventureislandc)
STD_ROM_FN(nes_adventureislandc)

struct BurnDriver BurnDrvnes_adventureislandc = {
	"nes_adventureislandc", "nes_adventureisland", NULL, NULL, "2017",
	"Adventure Island (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_adventureislandcRomInfo, nes_adventureislandcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Adventure Island II (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_advenisliicRomDesc[] = {
	{ "adventure island ii (usa) - castellano v1.0.nes",          262160, 0x92f21250, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advenisliic)
STD_ROM_FN(nes_advenisliic)

struct BurnDriver BurnDrvnes_advenisliic = {
	"nes_advenisliic", "nes_advenislii", NULL, NULL, "2017",
	"Adventure Island II (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_advenisliicRomInfo, nes_advenisliicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Adventure Island 3 (USA) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_advenisl3cRomDesc[] = {
	{ "adventure island iii (usa) - castellano v1.1.nes",          262160, 0x31c3056e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advenisl3c)
STD_ROM_FN(nes_advenisl3c)

struct BurnDriver BurnDrvnes_advenisl3c = {
	"nes_advenisl3c", "nes_advenisl3", NULL, NULL, "2017",
	"Adventure Island 3 (Hack, Spanish v1.1)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_advenisl3cRomInfo, nes_advenisl3cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Alien 3 (USA) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_alien3cRomDesc[] = {
	{ "alien 3 (usa) - castellano v1.1.nes",          262160, 0xd16b91b7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_alien3c)
STD_ROM_FN(nes_alien3c)

struct BurnDriver BurnDrvnes_alien3c = {
	"nes_alien3c", "nes_alien3", NULL, NULL, "2020",
	"Alien 3 (Hack, Spanish v1.1)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_alien3cRomInfo, nes_alien3cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Advanced Dungeons & Dragons - Dragon Strike (USA)
static struct BurnRomInfo nes_advdddsRomDesc[] = {
	{ "advanced dungeons & dragons - dragon strike (usa).nes",          524304, 0x4c87cdde, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advddds)
STD_ROM_FN(nes_advddds)

struct BurnDriver BurnDrvnes_advddds = {
	"nes_advddds", NULL, NULL, NULL, "1990",
	"Advanced Dungeons & Dragons - Dragon Strike (USA)\0", NULL, "Natsume", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_SHOOT, 0,
	NESGetZipName, nes_advdddsRomInfo, nes_advdddsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Advanced Dungeons & Dragons - Dragon Strike (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_advdddscRomDesc[] = {
	{ "advanced dungeons & dragons - dragon strike (usa) - castellano v1.0.nes",          524304, 0x22cfb91d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advdddsc)
STD_ROM_FN(nes_advdddsc)

struct BurnDriver BurnDrvnes_advdddsc = {
	"nes_advdddsc", "nes_advddds", NULL, NULL, "2019",
	"Advanced Dungeons & Dragons - Dragon Strike (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_SHOOT, 0,
	NESGetZipName, nes_advdddscRomInfo, nes_advdddscRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Advanced Dungeons & Dragons - Dragons of Flame (T-eng)
// https://www.romhacking.net/
static struct BurnRomInfo nes_advdddfRomDesc[] = {
	{ "advanced dungeons & dragons - dragons of flame (t-eng).nes",          262160, 0x8196b978, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advdddf)
STD_ROM_FN(nes_advdddf)

struct BurnDriver BurnDrvnes_advdddf = {
	"nes_advdddf", NULL, NULL, NULL, "2003",
	"Advanced Dungeons & Dragons - Dragons of Flame (T-Eng, v1.03)\0", NULL, "DvD Translations", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_advdddfRomInfo, nes_advdddfRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Advanced Dungeons & Dragons - Dragons of Flame (Japan)
static struct BurnRomInfo nes_advdddfjRomDesc[] = {
	{ "advanced dungeons & dragons - dragons of flame (japan).nes",          262160, 0x8ba5de24, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advdddfj)
STD_ROM_FN(nes_advdddfj)

struct BurnDriver BurnDrvnes_advdddfj = {
	"nes_advdddfj", "nes_advdddf", NULL, NULL, "1992",
	"Advanced Dungeons & Dragons - Dragons of Flame (Japan)\0", NULL, "Pony Canyon", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_advdddfjRomInfo, nes_advdddfjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Advanced Dungeons & Dragons - Dragons of Flame (Hack, Spanish v1.0)
// https://www.romhacking.net/translations/6889/
static struct BurnRomInfo nes_advdddfsRomDesc[] = {
	{ "Advanced Dungeons & Dragons - Dragons of Flame T-Spa (2023)(Wave).nes",          262160, 0x9e431abb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advdddfs)
STD_ROM_FN(nes_advdddfs)

struct BurnDriver BurnDrvnes_advdddfs = {
	"nes_advdddfs", "nes_advdddf", NULL, NULL, "2023",
	"Advanced Dungeons & Dragons - Dragons of Flame (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_advdddfsRomInfo, nes_advdddfsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Advanced Dungeons & Dragons - Heroes of the Lance (USA)
static struct BurnRomInfo nes_advddhlRomDesc[] = {
	{ "advanced dungeons & dragons - heroes of the lance (usa).nes",          262160, 0xe880d426, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advddhl)
STD_ROM_FN(nes_advddhl)

struct BurnDriver BurnDrvnes_advddhl = {
	"nes_advddhl", NULL, NULL, NULL, "1990",
	"Advanced Dungeons & Dragons - Heroes of the Lance (USA)\0", NULL, "FCI Inc.", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_advddhlRomInfo, nes_advddhlRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Advanced Dungeons & Dragons - Heroes of the Lance (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_advddhlcRomDesc[] = {
	{ "advanced dungeons & dragons - heroes of the lance (usa) - castellano v1.0.nes",          262160, 0x365f9624, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advddhlc)
STD_ROM_FN(nes_advddhlc)

struct BurnDriver BurnDrvnes_advddhlc = {
	"nes_advddhlc", "nes_advddhl", NULL, NULL, "2019",
	"Advanced Dungeons & Dragons - Heroes of the Lance (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_advddhlcRomInfo, nes_advddhlcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Advanced Dungeons & Dragons - Hillsfar (USA)
static struct BurnRomInfo nes_advddhllfrRomDesc[] = {
	{ "advanced dungeons & dragons - hillsfar (usa).nes",          262160, 0x3849d0ee, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advddhllfr)
STD_ROM_FN(nes_advddhllfr)

struct BurnDriver BurnDrvnes_advddhllfr = {
	"nes_advddhllfr", NULL, NULL, NULL, "1992",
	"Advanced Dungeons & Dragons - Hillsfar (USA)\0", NULL, "FCI Inc.", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_advddhllfrRomInfo, nes_advddhllfrRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Advanced Dungeons & Dragons - Pool of Radiance (USA)
static struct BurnRomInfo nes_advddporRomDesc[] = {
	{ "advanced dungeons & dragons - pool of radiance (usa).nes",          655376, 0xbbea8d23, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advddpor)
STD_ROM_FN(nes_advddpor)

struct BurnDriver BurnDrvnes_advddpor = {
	"nes_advddpor", NULL, NULL, NULL, "1992",
	"Advanced Dungeons & Dragons - Pool of Radiance (USA)\0", NULL, "FCI Inc.", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_advddporRomInfo, nes_advddporRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Battle of Olympus, The (Europe) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_battlolycRomDesc[] = {
	{ "battle of olympus, the (europe) - castellano v1.1.nes",          131088, 0xab9d1a2e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_battlolyc)
STD_ROM_FN(nes_battlolyc)

struct BurnDriver BurnDrvnes_battlolyc = {
	"nes_battlolyc", "nes_battloly", NULL, NULL, "2018",
	"Battle of Olympus, The (Hack, Spanish v1.1)\0", NULL, "dmlan2000", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_battlolycRomInfo, nes_battlolycRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Battleship (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_battleshipcRomDesc[] = {
	{ "battleship (usa) - castellano v1.0.nes",          65552, 0xcf1ee9d6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_battleshipc)
STD_ROM_FN(nes_battleshipc)

struct BurnDriver BurnDrvnes_battleshipc = {
	"nes_battleshipc", "nes_battleship", NULL, NULL, "2019",
	"Battleship (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_STRATEGY, 0,
	NESGetZipName, nes_battleshipcRomInfo, nes_battleshipcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};							

// Bionic Commando (Japan) - Castellano v1.03
// https://www.romhacking.net/
static struct BurnRomInfo nes_bioniccommandocRomDesc[] = {
	{ "bionic commando (japan) - castellano v1.03.nes",          262160, 0xec5f3e79, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bioniccommandoc)
STD_ROM_FN(nes_bioniccommandoc)

struct BurnDriver BurnDrvnes_bioniccommandoc = {
	"nes_bioniccommandoc", "nes_bioniccommando", NULL, NULL, "2013",
	"Bionic Commando (Hack, Spanish v1.03)\0", NULL, "TheFireRed", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_bioniccommandocRomInfo, nes_bioniccommandocRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Blaster Master (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_blastermastercRomDesc[] = {
	{ "blaster master (usa) - castellano v1.0.nes",          262160, 0xb63123db, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_blastermasterc)
STD_ROM_FN(nes_blastermasterc)

struct BurnDriver BurnDrvnes_blastermasterc = {
	"nes_blastermasterc", "nes_blastermaster", NULL, NULL, "2018",
	"Blaster Master (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_blastermastercRomInfo, nes_blastermastercRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Bram Stoker's Dracula (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_bramstosdracRomDesc[] = {
	{ "bram stoker's dracula (usa) - castellano v1.0.nes",          262160, 0xd6eb5f83, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bramstosdrac)
STD_ROM_FN(nes_bramstosdrac)

struct BurnDriver BurnDrvnes_bramstosdrac = {
	"nes_bramstosdrac", "nes_bramstosdra", NULL, NULL, "2017",
	"Bram Stoker's Dracula (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_bramstosdracRomInfo, nes_bramstosdracRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Captain Tsubasa Vol. II - Super Striker (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_captatsuvoliicRomDesc[] = {
	{ "captain tsubasa vol. ii - super striker (japan) - castellano v1.0.nes",          393233, 0x49c692c5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_captatsuvoliic)
STD_ROM_FN(nes_captatsuvoliic)

struct BurnDriver BurnDrvnes_captatsuvoliic = {
	"nes_captatsuvoliic", "nes_captatsuvolii", NULL, NULL, "2013",
	"Captain Tsubasa Vol. II - Super Striker (Hack, Spanish v1.0)\0", NULL, "pepodmc", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_SPORTSMISC | GBF_RPG, 0,
	NESGetZipName, nes_captatsuvoliicRomInfo, nes_captatsuvoliicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Castlevania (USA) - Castellano 1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_castlevaniacRomDesc[] = {
	{ "castlevania (usa) - castellano 1.0.nes",          131088, 0xde2e6b55, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_castlevaniac)
STD_ROM_FN(nes_castlevaniac)

struct BurnDriver BurnDrvnes_castlevaniac = {
	"nes_castlevaniac", "nes_castlevania", NULL, NULL, "2012",
	"Castlevania (Hack, Spanish v1.0)\0", NULL, "Julian L.", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_castlevaniacRomInfo, nes_castlevaniacRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Castlevania II - Simon's Quest (USA) - Castellano v2.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_castliicRomDesc[] = {
	{ "castlevania ii - simon's quest (usa) - castellano v2.0.nes",          262160, 0x38d393a9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_castliic)
STD_ROM_FN(nes_castliic)

struct BurnDriver BurnDrvnes_castliic = {
	"nes_castliic", "nes_castlii", NULL, NULL, "2013",
	"Castlevania II - Simon's Quest (Hack, Spanish v2.0)\0", NULL, "Julian L.", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM | GBF_RPG, 0,
	NESGetZipName, nes_castliicRomInfo, nes_castliicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Akumajou Densetsu (Japan) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_akumadenjcRomDesc[] = {
	{ "akumajou densetsu (japan) - castellano v1.1.nes",          393232, 0x33ead706, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_akumadenjc)
STD_ROM_FN(nes_akumadenjc)

struct BurnDriver BurnDrvnes_akumadenjc = {
	"nes_akumadenjc", "nes_castliii", NULL, NULL, "2017",
	"Akumajou Densetsu (Hack, Spanish v1.1)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_akumadenjcRomInfo, nes_akumadenjcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Saint Seiya - Ougon Densetsu - Castellano v1.0.nes
// https://www.romhacking.net/
static struct BurnRomInfo nes_saintseiougdencRomDesc[] = {
	{ "saint seiya - ougon densetsu - castellano v1.0.nes.nes",          262160, 0x2abdcc86, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_saintseiougdenc)
STD_ROM_FN(nes_saintseiougdenc)

struct BurnDriver BurnDrvnes_saintseiougdenc = {
	"nes_saintseiougdenc", "nes_saintseiougden", NULL, NULL, "2004",
	"Saint Seiya - Ougon Densetsu (Hack, Spanish v1.0)\0", NULL, "Juan", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM | GBF_RPG, 0,
	NESGetZipName, nes_saintseiougdencRomInfo, nes_saintseiougdencRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Chip 'n Dale - Rescue Rangers (USA) - Castellano v0.9
// https://www.romhacking.net/
static struct BurnRomInfo nes_chipndalresracRomDesc[] = {
	{ "chip 'n dale - rescue rangers (usa) - castellano v0.9.nes",          262160, 0x7c6aed46, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chipndalresrac)
STD_ROM_FN(nes_chipndalresrac)

struct BurnDriver BurnDrvnes_chipndalresrac = {
	"nes_chipndalresrac", "nes_chipndalresra", NULL, NULL, "20??",
	"Chip 'n Dale - Rescue Rangers (Hack, Spanish v0.9)\0", NULL, "Unknown", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_chipndalresracRomInfo, nes_chipndalresracRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Chip 'n Dale - Rescue Rangers 2 (USA) - Castellano v0.99
// https://www.romhacking.net/
static struct BurnRomInfo nes_chipndalresra2cRomDesc[] = {
	{ "chip 'n dale - rescue rangers 2 (usa) - castellano v0.99.nes",          262160, 0xaa6e37e6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chipndalresra2c)
STD_ROM_FN(nes_chipndalresra2c)

struct BurnDriver BurnDrvnes_chipndalresra2c = {
	"nes_chipndalresra2c", "nes_chipndalresra2", NULL, NULL, "2019",
	"Chip 'n Dale - Rescue Rangers 2 (Hack, Spanish v0.99)\0", NULL, "jackic", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_chipndalresra2cRomInfo, nes_chipndalresra2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Contra (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_contracastRomDesc[] = {
	{ "contra (usa) - castellano v1.0.nes",          131088, 0x6a56fc08, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_contracast)
STD_ROM_FN(nes_contracast)

struct BurnDriver BurnDrvnes_contracast = {
	"nes_contracast", "nes_contra", NULL, NULL, "2019",
	"Contra (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_contracastRomInfo, nes_contracastRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Crystalis (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_crystaliscRomDesc[] = {
	{ "crystalis (usa) - castellano v1.0.nes",          393232, 0xb23c737c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_crystalisc)
STD_ROM_FN(nes_crystalisc)

struct BurnDriver BurnDrvnes_crystalisc = {
	"nes_crystalisc", "nes_crystalis", NULL, NULL, "2014",
	"Crystalis (Hack, Spanish v1.0)\0", NULL, "Teterilla", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_crystaliscRomInfo, nes_crystaliscRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Darkman (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_darkmancRomDesc[] = {
	{ "darkman (usa) - castellano v1.0.nes",          262160, 0xe2cb1c36, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_darkmanc)
STD_ROM_FN(nes_darkmanc)

struct BurnDriver BurnDrvnes_darkmanc = {
	"nes_darkmanc", "nes_darkman", NULL, NULL, "2017",
	"Darkman (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_darkmancRomInfo, nes_darkmancRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Die Hard (USA) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_diehardcRomDesc[] = {
	{ "die hard (usa) - castellano v1.1.nes",          262160, 0x602b9a29, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_diehardc)
STD_ROM_FN(nes_diehardc)

struct BurnDriver BurnDrvnes_diehardc = {
	"nes_diehardc", "nes_diehard", NULL, NULL, "2019",
	"Die Hard (Hack, Spanish v1.1)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_diehardcRomInfo, nes_diehardcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Disney's Aladdin (Europe) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_aladdincRomDesc[] = {
	{ "disney's aladdin (europe) - castellano v1.0.nes",          262160, 0xb857fcf2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_aladdinc)
STD_ROM_FN(nes_aladdinc)

struct BurnDriver BurnDrvnes_aladdinc = {
	"nes_aladdinc", "nes_aladdin", NULL, NULL, "2018",
	"Disney's Aladdin (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_aladdincRomInfo, nes_aladdincRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Double Dragon (USA) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_doubledragoncRomDesc[] = {
	{ "double dragon (usa) - castellano v1.1.nes",          262160, 0xf5e19c32, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_doubledragonc)
STD_ROM_FN(nes_doubledragonc)

struct BurnDriver BurnDrvnes_doubledragonc = {
	"nes_doubledragonc", "nes_doubledragon", NULL, NULL, "2021",
	"Double Dragon (Hack, Spanish v1.1)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_doubledragoncRomInfo, nes_doubledragoncRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Double Dragon II - The Revenge (USA) (Rev A) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_doubldraiicRomDesc[] = {
	{ "double dragon ii - the revenge (usa) (rev a) - castellano v1.0.nes",          262160, 0x67a0d781, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_doubldraiic)
STD_ROM_FN(nes_doubldraiic)

struct BurnDriver BurnDrvnes_doubldraiic = {
	"nes_doubldraiic", "nes_doubldraii", NULL, NULL, "2017",
	"Double Dragon II - The Revenge (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_doubldraiicRomInfo, nes_doubldraiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Double Dragon III - The Sacred Stones (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_doubldraiiicRomDesc[] = {
	{ "double dragon iii - the sacred stones (usa) - castellano v1.0.nes",          262160, 0xaa40b251, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_doubldraiiic)
STD_ROM_FN(nes_doubldraiiic)

struct BurnDriver BurnDrvnes_doubldraiiic = {
	"nes_doubldraiiic", "nes_doubldraiii", NULL, NULL, "2018",
	"Double Dragon III - The Sacred Stones (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_doubldraiiicRomInfo, nes_doubldraiiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Dragon Ball Z - Super Butouden 2 (Unl) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_dragobalzsupbu2cRomDesc[] = {
	{ "dragon ball z - super butouden 2 (unl) - castellano v1.1.nes",          393232, 0x8e8ebd23, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragobalzsupbu2c)
STD_ROM_FN(nes_dragobalzsupbu2c)

struct BurnDriver BurnDrvnes_dragobalzsupbu2c = {
	"nes_dragobalzsupbu2c", "nes_dragobalzsupbu2", NULL, NULL, "2017",
	"Dragon Ball Z - Super Butouden 2 (Hack, Spanish v1.1)\0", NULL, "Terwilf", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_VSFIGHT, 0,
	NESGetZipName, nes_dragobalzsupbu2cRomInfo, nes_dragobalzsupbu2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Dragon Ball Z III - Ressen Jinzou Ningen (Japan) - Castellano v1.0a
// https://www.romhacking.net/
static struct BurnRomInfo nes_dragobalziiirejinicRomDesc[] = {
	{ "dragon ball z iii - ressen jinzou ningen (japan) - castellano v1.0a.nes",          524304, 0xff1d3cf3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragobalziiirejinic)
STD_ROM_FN(nes_dragobalziiirejinic)

struct BurnDriver BurnDrvnes_dragobalziiirejinic = {
	"nes_dragobalziiirejinic", "nes_dragobalziiirejini", NULL, NULL, "2016",
	"Dragon Ball Z III - Ressen Jinzou Ningen (Hacl, Spanish v1.0a)\0", NULL, "TransGen", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_STRATEGY | GBF_ADV, 0,
	NESGetZipName, nes_dragobalziiirejinicRomInfo, nes_dragobalziiirejinicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Dragon Buster (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_dragonbustercRomDesc[] = {
	{ "dragon buster (japan) - castellano v1.0.nes",          163856, 0xe718a85d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragonbusterc)
STD_ROM_FN(nes_dragonbusterc)

struct BurnDriver BurnDrvnes_dragonbusterc = {
	"nes_dragonbusterc", "nes_dragonbuster", NULL, NULL, "2019",
	"Dragon Buster (Hack, Spanish v1.0)\0", NULL, "Max1323", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_dragonbustercRomInfo, nes_dragonbustercRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Dragon Warrior (USA) (Rev A) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_dragonwarriorcRomDesc[] = {
	{ "dragon warrior (usa) (rev a) - castellano v1.0.nes",          81936, 0x7442bd9f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragonwarriorc)
STD_ROM_FN(nes_dragonwarriorc)

struct BurnDriver BurnDrvnes_dragonwarriorc = {
	"nes_dragonwarriorc", "nes_dragonwarrior", NULL, NULL, "2019",
	"Dragon Warrior (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_dragonwarriorcRomInfo, nes_dragonwarriorcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Dragon Warrior III (USA) - Castellano v0.9
// https://www.romhacking.net/
static struct BurnRomInfo nes_dragowariiicRomDesc[] = {
	{ "dragon warrior iii (usa) - castellano v0.9.nes",          524304, 0xc3d2f589, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragowariiic)
STD_ROM_FN(nes_dragowariiic)

struct BurnDriver BurnDrvnes_dragowariiic = {
	"nes_dragowariiic", "nes_dragowariii", NULL, NULL, "2017",
	"Dragon Warrior III (Hack, Spanish v0.9)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_dragowariiicRomInfo, nes_dragowariiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Dragon Warrior IV (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_dragowarivcRomDesc[] = {
	{ "dragon warrior iv (usa) - castellano v1.0.nes",          524304, 0xf5577473, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragowarivc)
STD_ROM_FN(nes_dragowarivc)

struct BurnDriver BurnDrvnes_dragowarivc = {
	"nes_dragowarivc", "nes_dragowariv", NULL, NULL, "2016",
	"Dragon Warrior IV (Hack, Spanish v1.0)\0", NULL, "Butz Klauser", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_dragowarivcRomInfo, nes_dragowarivcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Dragon's Lair (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_dragonslaircRomDesc[] = {
	{ "dragon's lair (usa) - castellano v1.0.nes",          131088, 0xdb3d2678, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragonslairc)
STD_ROM_FN(nes_dragonslairc)

struct BurnDriver BurnDrvnes_dragonslairc = {
	"nes_dragonslairc", "nes_dragonslair", NULL, NULL, "2019",
	"Dragon's Lair (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_dragonslaircRomInfo, nes_dragonslaircRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// DuckTales (USA) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_ducktalescRomDesc[] = {
	{ "duck tales (usa) - castellano v1.1.nes",          131088, 0xfcb58532, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ducktalesc)
STD_ROM_FN(nes_ducktalesc)

struct BurnDriver BurnDrvnes_ducktalesc = {
	"nes_ducktalesc", "nes_ducktales", NULL, NULL, "2017",
	"DuckTales (Hack, Spanish v1.1)\0", NULL, "Darkchaosblast", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_ducktalescRomInfo, nes_ducktalescRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// DuckTales 2 (USA) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_ducktales2castRomDesc[] = {
	{ "duck tales 2 (usa) - castellano v1.1.nes",          131088, 0x908c28ff, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ducktales2cast)
STD_ROM_FN(nes_ducktales2cast)

struct BurnDriver BurnDrvnes_ducktales2cast = {
	"nes_ducktales2cast", "nes_ducktales2", NULL, NULL, "2017",
	"DuckTales 2 (Hack, Spanish v1.1)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_ducktales2castRomInfo, nes_ducktales2castRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Earth Bound (USA) (Proto) - Castellano v2.2
// https://www.romhacking.net/
static struct BurnRomInfo nes_earthboundcRomDesc[] = {
	{ "earth bound (usa) (proto) - castellano v2.2.nes",          524304, 0xd80e3217, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_earthboundc)
STD_ROM_FN(nes_earthboundc)

struct BurnDriver BurnDrvnes_earthboundc = {
	"nes_earthboundc", "nes_earthbound", NULL, NULL, "2016",
	"Earth Bound (Hack, Spanish v2.2)\0", NULL, "OAD", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_earthboundcRomInfo, nes_earthboundcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Elevator Action (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_elevatoractioncRomDesc[] = {
	{ "elevator action (usa) - castellano v1.0.nes",          40976, 0xcb844314, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_elevatoractionc)
STD_ROM_FN(nes_elevatoractionc)

struct BurnDriver BurnDrvnes_elevatoractionc = {
	"nes_elevatoractionc", "nes_elevatoraction", NULL, NULL, "2019",
	"Elevator Action (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_elevatoractioncRomInfo, nes_elevatoractioncRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Faxanadu (USA) (Rev A) - Castellano v0.7
// https://www.romhacking.net/
static struct BurnRomInfo nes_faxanaducRomDesc[] = {
	{ "faxanadu (usa) (rev a) - castellano v0.7.nes",          262160, 0x181b0549, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_faxanaduc)
STD_ROM_FN(nes_faxanaduc)

struct BurnDriver BurnDrvnes_faxanaduc = {
	"nes_faxanaduc", "nes_faxanadu", NULL, NULL, "2016",
	"Faxanadu (Hack, Spanish v0.7)\0", NULL, "Dirak", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM | GBF_ADV, 0,
	NESGetZipName, nes_faxanaducRomInfo, nes_faxanaducRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Final Fantasy (USA) - Castellano v1.0
// https://www.romhacking.net/translations/3419/
static struct BurnRomInfo nes_finalfantasycRomDesc[] = {
	{ "final fantasy (usa) - castellano v1.0.nes",          262160, 0x83a1074c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_finalfantasyc)
STD_ROM_FN(nes_finalfantasyc)

struct BurnDriver BurnDrvnes_finalfantasyc = {
	"nes_finalfantasyc", "nes_finalfantasy", NULL, NULL, "2002",
	"Final Fantasy (Hack, Spanish v1.0)\0", NULL, "Cunaaao", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_finalfantasycRomInfo, nes_finalfantasycRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Final Fantasy (USA) - Italian v1.0
// https://www.romhacking.net/translations/5633/
static struct BurnRomInfo nes_finalfantasyiRomDesc[] = {
	{ "Final Fantasy T-Ita (2020)(FIUMI).nes",          262160, 0x74201d71, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_finalfantasyi)
STD_ROM_FN(nes_finalfantasyi)

struct BurnDriver BurnDrvnes_finalfantasyi = {
	"nes_finalfantasyi", "nes_finalfantasy", NULL, NULL, "2020",
	"Final Fantasy (Hack, Italian v1.0)\0", NULL, "FIUMI", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_finalfantasyiRomInfo, nes_finalfantasyiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Final Fantasy II (Japan) (Headered) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_finalfantasyiicRomDesc[] = {
	{ "final fantasy ii (japan) (headered) - castellano v1.0.nes",          262160, 0x3fba77eb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_finalfantasyiic)
STD_ROM_FN(nes_finalfantasyiic)

struct BurnDriver BurnDrvnes_finalfantasyiic = {
	"nes_finalfantasyiic", "nes_finalfantasyii", NULL, NULL, "20XX?",
	"Final Fantasy II (Hack, Spanish v1.0)\0", NULL, "Unknown", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_finalfantasyiicRomInfo, nes_finalfantasyiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Final Fantasy III (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_finalfaniiicRomDesc[] = {
	{ "final fantasy iii (japan) - castellano v1.0.nes",          1048592, 0xb328a638, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_finalfaniiic)
STD_ROM_FN(nes_finalfaniiic)

struct BurnDriver BurnDrvnes_finalfaniiic = {
	"nes_finalfaniiic", "nes_finalfaniii", NULL, NULL, "2015",
	"Final Fantasy III (Hack, Spanish v1.0)\0", NULL, "Maeson", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_finalfaniiicRomInfo, nes_finalfaniiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Friday the 13th (USA) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_fridaythe13thcRomDesc[] = {
	{ "friday the 13th (usa) - castellano v1.1.nes",          65552, 0xabecfd68, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_fridaythe13thc)
STD_ROM_FN(nes_fridaythe13thc)

struct BurnDriver BurnDrvnes_fridaythe13thc = {
	"nes_fridaythe13thc", "nes_fridaythe13th", NULL, NULL, "2017",
	"Friday the 13th (Hack, Spanish v1.1)\0", NULL, "JONNYVILLA2088", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM | GBF_ADV, 0,
	NESGetZipName, nes_fridaythe13thcRomInfo, nes_fridaythe13thcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ganbare Goemon 2 (Japan) - Castellano v0.99
// https://www.romhacking.net/
static struct BurnRomInfo nes_ganbaregoemon2cRomDesc[] = {
	{ "ganbare goemon 2 (japan) - castellano v0.99.nes",          262160, 0xeca0d0e1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ganbaregoemon2c)
STD_ROM_FN(nes_ganbaregoemon2c)

struct BurnDriver BurnDrvnes_ganbaregoemon2c = {
	"nes_ganbaregoemon2c", "nes_ganbaregoemon2", NULL, NULL, "2021",
	"Ganbare Goemon 2 (Hack, Spanish v0.99)\0", NULL, "jackic", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_ACTION | GBF_RPG, 0,
	NESGetZipName, nes_ganbaregoemon2cRomInfo, nes_ganbaregoemon2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ganbare Goemon Gaiden 2 - Tenka no Zaihou (Japan) - Castellano v0.98
// https://www.romhacking.net/
static struct BurnRomInfo nes_ganbagoegai2cRomDesc[] = {
	{ "ganbare goemon gaiden 2 - tenka no zaihou (japan) - castellano v0.98.nes",          524304, 0xc2e96c08, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ganbagoegai2c)
STD_ROM_FN(nes_ganbagoegai2c)

struct BurnDriverD BurnDrvnes_ganbagoegai2c = {
	"nes_ganbagoegai2c", "nes_ganbagoegai2", NULL, NULL, "2020",
	"Ganbare Goemon Gaiden 2 - Tenka no Zaihou (Hack, Spanish v0.98)\0", NULL, "jackic", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_ganbagoegai2cRomInfo, nes_ganbagoegai2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ganbare Goemon! - Karakuri Douchuu (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_ganbagoecRomDesc[] = {
	{ "ganbare goemon! - karakuri douchuu (japan) - castellano v1.0.nes",          262160, 0xbf99d7e4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ganbagoec)
STD_ROM_FN(nes_ganbagoec)

struct BurnDriver BurnDrvnes_ganbagoec = {
	"nes_ganbagoec", "nes_ganbagoe", NULL, NULL, "2018",
	"Ganbare Goemon! - Karakuri Douchuu (Hack, Spanish v1.0)\0", NULL, "jackic", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, nes_ganbagoecRomInfo, nes_ganbagoecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Golgo 13 - Top Secret Episode (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_golgo13cRomDesc[] = {
	{ "golgo 13 - top secret episode (usa) - castellano v1.0.nes",          262160, 0x47cd6b0f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_golgo13c)
STD_ROM_FN(nes_golgo13c)

struct BurnDriver BurnDrvnes_golgo13c = {
	"nes_golgo13c", "nes_golgo13", NULL, NULL, "2021",
	"Golgo 13 - Top Secret Episode (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_SHOOT, 0,
	NESGetZipName, nes_golgo13cRomInfo, nes_golgo13cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Goonies (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_gooniescRomDesc[] = {
	{ "goonies (japan) - castellano v1.0.nes",          49168, 0xb74b6dc4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_gooniesc)
STD_ROM_FN(nes_gooniesc)

struct BurnDriver BurnDrvnes_gooniesc = {
	"nes_gooniesc", "nes_goonies", NULL, NULL, "2018",
	"Goonies (Hack, Spanish v1.0)\0", NULL, "Darkchaosblast", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_gooniescRomInfo, nes_gooniescRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Goonies II, The (USA) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_gooniesiithecRomDesc[] = {
	{ "goonies ii, the (usa) - castellano v1.1.nes",          131088, 0x6bc02b52, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_gooniesiithec)
STD_ROM_FN(nes_gooniesiithec)

struct BurnDriver BurnDrvnes_gooniesiithec = {
	"nes_gooniesiithec", "nes_gooniesiithe", NULL, NULL, "2018",
	"Goonies II, The (Hack, Spanish v1.1)\0", NULL, "Darkchaosblast", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_gooniesiithecRomInfo, nes_gooniesiithecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Hokuto no Ken 2 (Japan) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_hokutonoken2cRomDesc[] = {
	{ "hokuto no ken 2 (japan) - castellano v1.1.nes",          131088, 0x3a3cd6df, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_hokutonoken2c)
STD_ROM_FN(nes_hokutonoken2c)

struct BurnDriver BurnDrvnes_hokutonoken2c = {
	"nes_hokutonoken2c", "nes_hokutonoken2", NULL, NULL, "2018",
	"Hokuto no Ken 2 (Hack, Spanish v1.1)\0", NULL, "Jonathan", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE |  BDF_HACK, 1, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_hokutonoken2cRomInfo, nes_hokutonoken2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Hydlide (USA)
static struct BurnRomInfo nes_hydlideRomDesc[] = {
	{ "hydlide (usa).nes",          40976, 0x8962aebf, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_hydlide)
STD_ROM_FN(nes_hydlide)

struct BurnDriver BurnDrvnes_hydlide = {
	"nes_hydlide", NULL, NULL, NULL, "1989",
	"Hydlide (USA)\0", NULL, "Pony Canyon", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_hydlideRomInfo, nes_hydlideRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Hydlide Special (Japan)
static struct BurnRomInfo nes_hydlidejRomDesc[] = {
	{ "Hydlide Special (J) (1986)(T&E Soft).nes",          40976, 0x6207ba15, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_hydlidej)
STD_ROM_FN(nes_hydlidej)

struct BurnDriver BurnDrvnes_hydlidej = {
	"nes_hydlidej", "nes_hydlide", NULL, NULL, "1986",
	"Hydlide Special (Japan)\0", NULL, "T&E Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_hydlidejRomInfo, nes_hydlidejRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Hydlide (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_hydlidecRomDesc[] = {
	{ "hydlide (usa) - castellano v1.0.nes",          40976, 0x9b4f06c0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_hydlidec)
STD_ROM_FN(nes_hydlidec)

struct BurnDriver BurnDrvnes_hydlidec = {
	"nes_hydlidec", "nes_hydlide", NULL, NULL, "2019",
	"Hydlide (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_hydlidecRomInfo, nes_hydlidecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Hokuto no Ken 3 - Shin Seiki Souzou Seiken Restuden (Japan)
static struct BurnRomInfo nes_hokutonoken3RomDesc[] = {
	{ "hokuto no ken 3 - shin seiki souzou seiken restuden (japan).nes",          262160, 0x12b32580, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_hokutonoken3)
STD_ROM_FN(nes_hokutonoken3)

struct BurnDriver BurnDrvnes_hokutonoken3 = {
	"nes_hokutonoken3", NULL, NULL, NULL, "1989",
	"Hokuto no Ken 3 - Shin Seiki Souzou Seiken Restuden (Japan)\0", NULL, "TOEI Animation", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_hokutonoken3RomInfo, nes_hokutonoken3RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Hokuto no Ken 3 - Shin Seiki Souzou Seiken Restuden (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_hokutonoken3cRomDesc[] = {
	{ "hokuto no ken 3 - shin seiki souzou seiken restuden (japan) - castellano v1.0.nes",          262160, 0x9e7cc5dc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_hokutonoken3c)
STD_ROM_FN(nes_hokutonoken3c)

struct BurnDriver BurnDrvnes_hokutonoken3c = {
	"nes_hokutonoken3c", "nes_hokutonoken3", NULL, NULL, "20??",
	"Hokuto no Ken 3 - Shin Seiki Souzou Seiken Restuden (Hack, Spanish v1.0)\0", NULL, "Unknown", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_hokutonoken3cRomInfo, nes_hokutonoken3cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Immortal, The (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_immortalthecRomDesc[] = {
	{ "immortal, the (usa) - castellano v1.0.nes",          393232, 0xbf87b488, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_immortalthec)
STD_ROM_FN(nes_immortalthec)

struct BurnDriver BurnDrvnes_immortalthec = {
	"nes_immortalthec", "nes_immortalthe", NULL, NULL, "2004",
	"Immortal, The (Hack, Spanish v1.0)\0", NULL, "R. Merida", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG | GBF_RUNGUN, 0,
	NESGetZipName, nes_immortalthecRomInfo, nes_immortalthecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Jungle Book, The (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_junglebookthecRomDesc[] = {
	{ "jungle book, the (usa) - castellano v1.0.nes",          262160, 0x996a8677, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_junglebookthec)
STD_ROM_FN(nes_junglebookthec)

struct BurnDriver BurnDrvnes_junglebookthec = {
	"nes_junglebookthec", "nes_junglebookthe", NULL, NULL, "2018",
	"Jungle Book, The (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_junglebookthecRomInfo, nes_junglebookthecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Just Breed (Japan) - Castellano v2.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_justbreedcRomDesc[] = {
	{ "just breed (japan) - castellano v2.0.nes",          786448, 0xe415a13f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_justbreedc)
STD_ROM_FN(nes_justbreedc)

struct BurnDriver BurnDrvnes_justbreedc = {
	"nes_justbreedc", "nes_justbreed", NULL, NULL, "20??",
	"Just Breed (Hack, Spanish v2.0)\0", NULL, "Unknown", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG | GBF_STRATEGY, 0,
	NESGetZipName, nes_justbreedcRomInfo, nes_justbreedcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Karate Kid, The (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_karatekidthecRomDesc[] = {
	{ "karate kid, the (usa) - castellano v1.0.nes",          65552, 0x99cc2044, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_karatekidthec)
STD_ROM_FN(nes_karatekidthec)

struct BurnDriver BurnDrvnes_karatekidthec = {
	"nes_karatekidthec", "nes_karatekidthe", NULL, NULL, "2020",
	"Karate Kid, The (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_VSFIGHT, 0,
	NESGetZipName, nes_karatekidthecRomInfo, nes_karatekidthecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Karateka (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_karatekacRomDesc[] = {
	{ "karateka (japan) - castellano v1.0.nes",          24592, 0x3839edb9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_karatekac)
STD_ROM_FN(nes_karatekac)

struct BurnDriver BurnDrvnes_karatekac = {
	"nes_karatekac", "nes_karateka", NULL, NULL, "2020",
	"Karateka (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_VSFIGHT, 0,
	NESGetZipName, nes_karatekacRomInfo, nes_karatekacRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Krusty's Fun House (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_krustsfunhoucRomDesc[] = {
	{ "krusty's fun house (usa) - castellano v1.0.nes",          393232, 0xb09308d4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_krustsfunhouc)
STD_ROM_FN(nes_krustsfunhouc)

struct BurnDriver BurnDrvnes_krustsfunhouc = {
	"nes_krustsfunhouc", "nes_krustsfunhou", NULL, NULL, "2019",
	"Krusty's Fun House (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM | GBF_PUZZLE, 0,
	NESGetZipName, nes_krustsfunhoucRomInfo, nes_krustsfunhoucRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Legacy of the Wizard (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_legacwizcRomDesc[] = {
	{ "legacy of the wizard (usa) - castellano v1.0.nes",          196624, 0xe87e3d09, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_legacwizc)
STD_ROM_FN(nes_legacwizc)

struct BurnDriver BurnDrvnes_legacwizc = {
	"nes_legacwizc", "nes_legacwiz", NULL, NULL, "2020",
	"Legacy of the Wizard (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM | GBF_ADV, 0,
	NESGetZipName, nes_legacwizcRomInfo, nes_legacwizcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Legend of Zelda, The (Europe) - Italian v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_legenzeliRomDesc[] = {
	{ "Legend of Zelda, The (it) (2019)(Stich991).nes",          131088, 0xe0811f91, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_legenzeli)
STD_ROM_FN(nes_legenzeli)

struct BurnDriver BurnDrvnes_legenzeli = {
	"nes_legenzeli", "nes_legenzel", NULL, NULL, "2019",
	"Legend of Zelda, The (Hack, Italian v1.0)\0", NULL, "Stich991", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, nes_legenzeliRomInfo, nes_legenzeliRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Legend of Zelda Redux, The (Hack, v3.1)
// https://www.romhacking.net/hacks/5752/
static struct BurnRomInfo nes_legenzelrdRomDesc[] = {
	{ "Legend of Zelda Redux, The v3.1 (2022)(ShadowOne333).nes",          131088, 0x183f0f76, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_legenzelrd)
STD_ROM_FN(nes_legenzelrd)

struct BurnDriver BurnDrvnes_legenzelrd = {
	"nes_legenzelrd", "nes_legenzel", NULL, NULL, "2022",
	"Legend of Zelda Redux, The (Hack, v3.1)\0", NULL, "ShadowOne333", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, nes_legenzelrdRomInfo, nes_legenzelrdRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Lion King, The (Europe) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_lionkingthecRomDesc[] = {
	{ "lion king, the (europe) - castellano v1.0.nes",          262160, 0x562245ae, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_lionkingthec)
STD_ROM_FN(nes_lionkingthec)

struct BurnDriver BurnDrvnes_lionkingthec = {
	"nes_lionkingthec", "nes_lionkingthe", NULL, NULL, "2019",
	"Lion King, The (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_lionkingthecRomInfo, nes_lionkingthecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Lupin Sansei - Pandora no Isan (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_lupinsancRomDesc[] = {
	{ "lupin sansei - pandora no isan (japan) - castellano v1.0.nes",          196624, 0x0bb24755, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_lupinsanc)
STD_ROM_FN(nes_lupinsanc)

struct BurnDriver BurnDrvnes_lupinsanc = {
	"nes_lupinsanc", "nes_lupinsan", NULL, NULL, "2018",
	"Lupin Sansei - Pandora no Isan (Hack, Spanish v1.0)\0", NULL, "jackic", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_lupinsancRomInfo, nes_lupinsancRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Mafat Conspiracy - Golgo 13 (USA) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_mafatconcRomDesc[] = {
	{ "mafat conspiracy - golgo 13 (usa) - castellano v1.1.nes",          262160, 0xb527ad27, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mafatconc)
STD_ROM_FN(nes_mafatconc)

struct BurnDriver BurnDrvnes_mafatconc = {
	"nes_mafatconc", "nes_mafatcon", NULL, NULL, "2018",
	"Mafat Conspiracy - Golgo 13 (Hack, Spanish v1.1)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_SCRFIGHT | GBF_SHOOT, 0,
	NESGetZipName, nes_mafatconcRomInfo, nes_mafatconcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Magic of Scheherazade, The (USA) - Castellano v0.99
// https://www.romhacking.net/
static struct BurnRomInfo nes_magicschcRomDesc[] = {
	{ "magic of scheherazade, the (usa) - castellano v0.99.nes",          262160, 0x9d080895, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_magicschc)
STD_ROM_FN(nes_magicschc)

struct BurnDriver BurnDrvnes_magicschc = {
	"nes_magicschc", "nes_magicsch", NULL, NULL, "2019",
	"Magic of Scheherazade, The (Hack, Spanish v0.99)\0", NULL, "jackic", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_magicschcRomInfo, nes_magicschcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Megaman (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_megamancRomDesc[] = {
	{ "megaman (usa) - castellano v1.0.nes",          131088, 0x53db1d74, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_megamanc)
STD_ROM_FN(nes_megamanc)

struct BurnDriver BurnDrvnes_megamanc = {
	"nes_megamanc", "nes_megaman", NULL, NULL, "2017",
	"Megaman (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_megamancRomInfo, nes_megamancRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Megaman II (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_megaman2cRomDesc[] = {
	{ "megaman ii (usa) - castellano v1.0.nes",          262160, 0x35babd12, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_megaman2c)
STD_ROM_FN(nes_megaman2c)

struct BurnDriver BurnDrvnes_megaman2c = {
	"nes_megaman2c", "nes_megaman2", NULL, NULL, "2017",
	"Megaman II (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_megaman2cRomInfo, nes_megaman2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Megaman III (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_megaman3cRomDesc[] = {
	{ "megaman iii (usa) - castellano v1.0.nes",          393232, 0x7e1471b2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_megaman3c)
STD_ROM_FN(nes_megaman3c)

struct BurnDriver BurnDrvnes_megaman3c = {
	"nes_megaman3c", "nes_megaman3", NULL, NULL, "2017",
	"Megaman III (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_megaman3cRomInfo, nes_megaman3cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Megaman IV (USA) (Rev A) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_megaman4cRomDesc[] = {
	{ "megaman iv (usa) (rev a) - castellano v1.0.nes",          524304, 0xd4141bb2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_megaman4c)
STD_ROM_FN(nes_megaman4c)

struct BurnDriver BurnDrvnes_megaman4c = {
	"nes_megaman4c", "nes_megaman4", NULL, NULL, "2017",
	"Megaman IV (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_megaman4cRomInfo, nes_megaman4cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Megaman V (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_megaman5cRomDesc[] = {
	{ "megaman v (usa) - castellano v1.0.nes",          524304, 0xc46d7cd6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_megaman5c)
STD_ROM_FN(nes_megaman5c)

struct BurnDriver BurnDrvnes_megaman5c = {
	"nes_megaman5c", "nes_megaman5", NULL, NULL, "2017",
	"Megaman V (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_megaman5cRomInfo, nes_megaman5cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Megaman VI (USA) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_megaman6cRomDesc[] = {
	{ "megaman vi (usa) - castellano v1.1.nes",          524304, 0x1b902805, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_megaman6c)
STD_ROM_FN(nes_megaman6c)

struct BurnDriver BurnDrvnes_megaman6c = {
	"nes_megaman6c", "nes_megaman6", NULL, NULL, "2016",
	"Megaman VI (Hack, Spanish v1.1)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_megaman6cRomInfo, nes_megaman6cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Metal Gear (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_metalgearcRomDesc[] = {
	{ "metal gear (usa) - castellano v1.0.nes",          131088, 0xbabc19e2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_metalgearc)
STD_ROM_FN(nes_metalgearc)

struct BurnDriver BurnDrvnes_metalgearc = {
	"nes_metalgearc", "nes_metalgear", NULL, NULL, "2020",
	"Metal Gear (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN | GBF_ADV, 0,
	NESGetZipName, nes_metalgearcRomInfo, nes_metalgearcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Metroid (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_metroidcRomDesc[] = {
	{ "metroid (usa) - castellano v1.0.nes",          131088, 0x12767f2e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_metroidc)
STD_ROM_FN(nes_metroidc)

struct BurnDriver BurnDrvnes_metroidc = {
	"nes_metroidc", "nes_metroid", NULL, NULL, "2019",
	"Metroid (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_metroidcRomInfo, nes_metroidcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Mighty Final Fight (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_mightfinfigcRomDesc[] = {
	{ "mighty final fight (usa) - castellano v1.0.nes",          262160, 0x79b72603, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mightfinfigc)
STD_ROM_FN(nes_mightfinfigc)

struct BurnDriver BurnDrvnes_mightfinfigc = {
	"nes_mightfinfigc", "nes_mightfinfig", NULL, NULL, "2016",
	"Mighty Final Fight (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_mightfinfigcRomInfo, nes_mightfinfigcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Maniac Mansion (Europe)
static struct BurnRomInfo nes_maniacmansioneRomDesc[] = {
	{ "maniac mansion (europe).nes",          262160, 0x90333aea, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_maniacmansione)
STD_ROM_FN(nes_maniacmansione)

struct BurnDriver BurnDrvnes_maniacmansione = {
	"nes_maniacmansione", "nes_maniacmansion", NULL, NULL, "1989",
	"Maniac Mansion (Europe)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_maniacmansioneRomInfo, nes_maniacmansioneRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Maniac Mansion (France)
static struct BurnRomInfo nes_maniacmansionfRomDesc[] = {
	{ "maniac mansion (france).nes",          262160, 0x9118cd29, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_maniacmansionf)
STD_ROM_FN(nes_maniacmansionf)

struct BurnDriver BurnDrvnes_maniacmansionf = {
	"nes_maniacmansionf", "nes_maniacmansion", NULL, NULL, "1989",
	"Maniac Mansion (France)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_maniacmansionfRomInfo, nes_maniacmansionfRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Maniac Mansion (Germany)
static struct BurnRomInfo nes_maniacmansiongRomDesc[] = {
	{ "maniac mansion (germany).nes",          262160, 0x05455e77, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_maniacmansiong)
STD_ROM_FN(nes_maniacmansiong)

struct BurnDriver BurnDrvnes_maniacmansiong = {
	"nes_maniacmansiong", "nes_maniacmansion", NULL, NULL, "1989",
	"Maniac Mansion (Germany)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_maniacmansiongRomInfo, nes_maniacmansiongRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Maniac Mansion (Italy)
static struct BurnRomInfo nes_maniacmansioniRomDesc[] = {
	{ "maniac mansion (italy).nes",          262160, 0xb9fd5255, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_maniacmansioni)
STD_ROM_FN(nes_maniacmansioni)

struct BurnDriver BurnDrvnes_maniacmansioni = {
	"nes_maniacmansioni", "nes_maniacmansion", NULL, NULL, "1989",
	"Maniac Mansion (Italy)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_maniacmansioniRomInfo, nes_maniacmansioniRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Maniac Mansion (Japan)
static struct BurnRomInfo nes_maniacmansionjRomDesc[] = {
	{ "maniac mansion (japan).nes",          262160, 0x101f9604, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_maniacmansionj)
STD_ROM_FN(nes_maniacmansionj)

struct BurnDriver BurnDrvnes_maniacmansionj = {
	"nes_maniacmansionj", "nes_maniacmansion", NULL, NULL, "1989",
	"Maniac Mansion (Japan)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_maniacmansionjRomInfo, nes_maniacmansionjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Maniac Mansion (Spain)
static struct BurnRomInfo nes_maniacmansionsRomDesc[] = {
	{ "maniac mansion (spain).nes",          262160, 0x901d691d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_maniacmansions)
STD_ROM_FN(nes_maniacmansions)

struct BurnDriver BurnDrvnes_maniacmansions = {
	"nes_maniacmansions", "nes_maniacmansion", NULL, NULL, "1989",
	"Maniac Mansion (Spain)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_maniacmansionsRomInfo, nes_maniacmansionsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Maniac Mansion (Sweden)
static struct BurnRomInfo nes_maniacmansionswRomDesc[] = {
	{ "maniac mansion (sweden).nes",          262160, 0x5a841cb2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_maniacmansionsw)
STD_ROM_FN(nes_maniacmansionsw)

struct BurnDriver BurnDrvnes_maniacmansionsw = {
	"nes_maniacmansionsw", "nes_maniacmansion", NULL, NULL, "1989",
	"Maniac Mansion (Sweden)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_maniacmansionswRomInfo, nes_maniacmansionswRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Maniac Mansion (USA) (Prototype)
static struct BurnRomInfo nes_maniacmansionuprotoRomDesc[] = {
	{ "maniac mansion (usa) (prototype).nes",          262160, 0xbba3ef7e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_maniacmansionuproto)
STD_ROM_FN(nes_maniacmansionuproto)

struct BurnDriver BurnDrvnes_maniacmansionuproto = {
	"nes_maniacmansionuproto", "nes_maniacmansion", NULL, NULL, "1988",
	"Maniac Mansion (USA) (Prototype)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_PROTOTYPE, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_maniacmansionuprotoRomInfo, nes_maniacmansionuprotoRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Maniac Mansion (USA) (Prototype) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_mancmansuprocRomDesc[] = {
	{ "maniac mansion (usa) (prototype) - castellano v1.0.nes",          262160, 0x3be70f25, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mancmansuproc)
STD_ROM_FN(nes_mancmansuproc)

struct BurnDriver BurnDrvnes_mancmansuproc = {
	"nes_mancmansuproc", "nes_maniacmansion", NULL, NULL, "2018",
	"Maniac Mansion (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_PROTOTYPE | BDF_HACK, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_mancmansuprocRomInfo, nes_mancmansuprocRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Nekketsu! Street Basket - Ganbare Dunk Heroes (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_nekkestrbascastcRomDesc[] = {
	{ "nekketsu! street basket - ganbare dunk heroes (japan) - castellano v1.0.nes",          262160, 0x3a2487c3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_nekkestrbascastc)
STD_ROM_FN(nes_nekkestrbascastc)

struct BurnDriver BurnDrvnes_nekkestrbascastc = {
	"nes_nekkestrbascastc", "nes_nekkestrbas", NULL, NULL, "2020",
	"Nekketsu! Street Basket - Ganbare Dunk Heroes (Hack, Spanish v1.0)\0", NULL, "Max1323", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_SPORTSMISC, 0,
	NESGetZipName, nes_nekkestrbascastcRomInfo, nes_nekkestrbascastcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ninja Gaiden (USA) - Castellano v1.01 + Addendum
// https://www.romhacking.net/
static struct BurnRomInfo nes_ninjagaidencRomDesc[] = {
	{ "ninja gaiden (usa) - castellano v1.01.nes",          262160, 0x7da08225, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ninjagaidenc)
STD_ROM_FN(nes_ninjagaidenc)

struct BurnDriver BurnDrvnes_ninjagaidenc = {
	"nes_ninjagaidenc", "nes_ninjagaiden", NULL, NULL, "2016",
	"Ninja Gaiden (Hack, Spanish v1.0 + Addendum)\0", NULL, "Terwilf", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_SCRFIGHT | GBF_PLATFORM, 0,
	NESGetZipName, nes_ninjagaidencRomInfo, nes_ninjagaidencRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ninja Gaiden II - The Dark Sword of Chaos (USA) - Castellano v1.0 + Addendum
// https://www.romhacking.net/
static struct BurnRomInfo nes_ninjagaiiicRomDesc[] = {
	{ "ninja gaiden ii - the dark sword of chaos (usa) - castellano v1.0.nes",          262160, 0x5742d6c0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ninjagaiiic)
STD_ROM_FN(nes_ninjagaiiic)

struct BurnDriver BurnDrvnes_ninjagaiiic = {
	"nes_ninjagaiiic", "nes_ninjagaiii", NULL, NULL, "2016",
	"Ninja Gaiden II - The Dark Sword of Chaos (Hack, Spanish v1.0 + Addendum)\0", NULL, "Terwilf", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_SCRFIGHT | GBF_PLATFORM, 0,
	NESGetZipName, nes_ninjagaiiicRomInfo, nes_ninjagaiiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ninja Gaiden III - The Ancient Ship of Doom (USA) - Castellano v1.0 + Addendum
// https://www.romhacking.net/
static struct BurnRomInfo nes_ninjagaiiiicRomDesc[] = {
	{ "ninja gaiden iii - the ancient ship of doom (usa) - castellano v1.0.nes",          262160, 0xdfb2692b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ninjagaiiiic)
STD_ROM_FN(nes_ninjagaiiiic)

struct BurnDriver BurnDrvnes_ninjagaiiiic = {
	"nes_ninjagaiiiic", "nes_ninjagaiiii", NULL, NULL, "2016",
	"Ninja Gaiden III - The Ancient Ship of Doom (Hack, Spanish v1.0 + Addendum)\0", NULL, "Terwilf", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_SCRFIGHT | GBF_PLATFORM, 0,
	NESGetZipName, nes_ninjagaiiiicRomInfo, nes_ninjagaiiiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// North & South (USA) - Castellano v2.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_northsouthcRomDesc[] = {
	{ "north & south (usa) - castellano v2.0.nes",          262160, 0x98907558, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_northsouthc)
STD_ROM_FN(nes_northsouthc)

struct BurnDriver BurnDrvnes_northsouthc = {
	"nes_northsouthc", "nes_northsouth", NULL, NULL, "2013",
	"North & South (Hack, Spanish v2.0)\0", NULL, "Julian L.", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_ACTION | GBF_STRATEGY, 0,
	NESGetZipName, nes_northsouthcRomInfo, nes_northsouthcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// North & South (USA) - Italian v2.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_northsouthiRomDesc[] = {
	{ "North & South (it) v2.0 (2014)(Julian L.).nes",          262160, 0x155fa336, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_northsouthi)
STD_ROM_FN(nes_northsouthi)

struct BurnDriver BurnDrvnes_northsouthi = {
	"nes_northsouthi", "nes_northsouth", NULL, NULL, "2014",
	"North & South (Hack, Italian v2.0)\0", NULL, "Julian L.", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_ACTION | GBF_STRATEGY, 0,
	NESGetZipName, nes_northsouthiRomInfo, nes_northsouthiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Parodius Da! (Japan) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_parodiusdacRomDesc[] = {
	{ "parodius da! (japan) - castellano v1.1.nes",          393232, 0x746bb8a8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_parodiusdac)
STD_ROM_FN(nes_parodiusdac)

struct BurnDriver BurnDrvnes_parodiusdac = {
	"nes_parodiusdac", "nes_parodiusda", NULL, NULL, "2018",
	"Parodius Da! (Hack, Spanish v1.1)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_HORSHOOT, 0,
	NESGetZipName, nes_parodiusdacRomInfo, nes_parodiusdacRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// P.O.W. - Prisoners of War (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_powcRomDesc[] = {
	{ "p.o.w. - prisoners of war (usa) - castellano v1.0.nes",          262160, 0xc24c8bcb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_powc)
STD_ROM_FN(nes_powc)

struct BurnDriver BurnDrvnes_powc = {
	"nes_powc", "nes_pow", NULL, NULL, "2017",
	"P.O.W. - Prisoners of War (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_powcRomInfo, nes_powcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Probotector (Europe) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_probotectorcRomDesc[] = {
	{ "probotector (europe) - castellano v1.0.nes",          131088, 0x9e9a6477, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_probotectorc)
STD_ROM_FN(nes_probotectorc)

struct BurnDriver BurnDrvnes_probotectorc = {
	"nes_probotectorc", "nes_contra", NULL, NULL, "2019",
	"Probotector (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_probotectorcRomInfo, nes_probotectorcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Probotector II - Return of the Evil Forces (Europe) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_probotectorretcRomDesc[] = {
	{ "probotector ii - return of the evil forces (europe) - castellano v1.0.nes",          262160, 0xf8d46d0b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_probotectorretc)
STD_ROM_FN(nes_probotectorretc)

struct BurnDriver BurnDrvnes_probotectorretc = {
	"nes_probotectorretc", "nes_superc", NULL, NULL, "2019",
	"Probotector II - Return of the Evil Forces (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_probotectorretcRomInfo, nes_probotectorretcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Puyo Puyo (Japan) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_puyopuyocRomDesc[] = {
	{ "puyo puyo (japan) - castellano v1.1.nes",          131088, 0xf55b69bd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_puyopuyoc)
STD_ROM_FN(nes_puyopuyoc)

struct BurnDriver BurnDrvnes_puyopuyoc = {
	"nes_puyopuyoc", "nes_puyopuyo", NULL, NULL, "2020",
	"Puyo Puyo (Hack, Spanish v1.1)\0", NULL, "Max1323", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_puyopuyocRomInfo, nes_puyopuyocRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Prince of Persia (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_princeofpersiacRomDesc[] = {
	{ "prince of persia (usa) - castellano v1.0.nes",          131088, 0x3f11d7a3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_princeofpersiac)
STD_ROM_FN(nes_princeofpersiac)

struct BurnDriver BurnDrvnes_princeofpersiac = {
	"nes_princeofpersiac", "nes_princeofpersia", NULL, NULL, "2019",
	"Prince of Persia (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM | GBF_ADV, 0,
	NESGetZipName, nes_princeofpersiacRomInfo, nes_princeofpersiacRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Prince of Persia (Europe)
static struct BurnRomInfo nes_princeofpersiaeRomDesc[] = {
	{ "prince of persia (europe).nes",          131088, 0x84fa4a07, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_princeofpersiae)
STD_ROM_FN(nes_princeofpersiae)

struct BurnDriver BurnDrvnes_princeofpersiae = {
	"nes_princeofpersiae", "nes_princeofpersia", NULL, NULL, "1992",
	"Prince of Persia (Europe)\0", NULL, "Virgin Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_PLATFORM | GBF_ADV, 0,
	NESGetZipName, nes_princeofpersiaeRomInfo, nes_princeofpersiaeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Prince of Persia (France)
static struct BurnRomInfo nes_princeofpersiafRomDesc[] = {
	{ "prince of persia (france).nes",          131088, 0xb19a530e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_princeofpersiaf)
STD_ROM_FN(nes_princeofpersiaf)

struct BurnDriver BurnDrvnes_princeofpersiaf = {
	"nes_princeofpersiaf", "nes_princeofpersia", NULL, NULL, "1992",
	"Prince of Persia (France)\0", NULL, "Virgin Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_PLATFORM | GBF_ADV, 0,
	NESGetZipName, nes_princeofpersiafRomInfo, nes_princeofpersiafRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Prince of Persia (Germany)
static struct BurnRomInfo nes_princeofpersiagRomDesc[] = {
	{ "Prince of persia (germany).nes",          131088, 0x53a349d4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_princeofpersiag)
STD_ROM_FN(nes_princeofpersiag)

struct BurnDriver BurnDrvnes_princeofpersiag = {
	"nes_princeofpersiag", "nes_princeofpersia", NULL, NULL, "1992",
	"Prince of Persia (Germany)\0", NULL, "Virgin Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_PLATFORM | GBF_ADV, 0,
	NESGetZipName, nes_princeofpersiagRomInfo, nes_princeofpersiagRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Prince of Persia (Russia)
static struct BurnRomInfo nes_princeofpersiarRomDesc[] = {
	{ "prince of persia (russia).nes",          131088, 0x094e6961, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_princeofpersiar)
STD_ROM_FN(nes_princeofpersiar)

struct BurnDriver BurnDrvnes_princeofpersiar = {
	"nes_princeofpersiar", "nes_princeofpersia", NULL, NULL, "1992",
	"Prince of Persia (Russia)\0", NULL, "Virgin Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_PLATFORM | GBF_ADV, 0,
	NESGetZipName, nes_princeofpersiarRomInfo, nes_princeofpersiarRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Prince of Persia (Spain)
static struct BurnRomInfo nes_princeofpersiasRomDesc[] = {
	{ "prince of persia (spain).nes",          131088, 0x7817051a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_princeofpersias)
STD_ROM_FN(nes_princeofpersias)

struct BurnDriver BurnDrvnes_princeofpersias = {
	"nes_princeofpersias", "nes_princeofpersia", NULL, NULL, "1992",
	"Prince of Persia (Spain)\0", NULL, "Virgin Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_PLATFORM | GBF_ADV, 0,
	NESGetZipName, nes_princeofpersiasRomInfo, nes_princeofpersiasRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Rambo (USA) (Rev 1) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_rambocRomDesc[] = {
	{ "rambo (usa) (rev 1) - castellano v1.0.nes",          131088, 0xf28cc969, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ramboc)
STD_ROM_FN(nes_ramboc)

struct BurnDriver BurnDrvnes_ramboc = {
	"nes_ramboc", "nes_rambo", NULL, NULL, "2018",
	"Rambo (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_rambocRomInfo, nes_rambocRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Renegade (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_renegadecRomDesc[] = {
	{ "renegade (usa) - castellano v1.0.nes",          131088, 0xacac361a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_renegadec)
STD_ROM_FN(nes_renegadec)

struct BurnDriver BurnDrvnes_renegadec = {
	"nes_renegadec", "nes_renegade", NULL, NULL, "2017",
	"Renegade (Hack, Spanish v1.0)\0", NULL, "JonnyVilla2088", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_renegadecRomInfo, nes_renegadecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Rescue - The Embassy Mission (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_rescuecRomDesc[] = {
	{ "rescue - the embassy mission (usa) - castellano v1.0.nes",          262160, 0xab441e08, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_rescuec)
STD_ROM_FN(nes_rescuec)

struct BurnDriver BurnDrvnes_rescuec = {
	"nes_rescuec", "nes_rescue", NULL, NULL, "2018",
	"Rescue - The Embassy Mission (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_SHOOT | GBF_ADV, 0,
	NESGetZipName, nes_rescuecRomInfo, nes_rescuecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// RoboCop (USA) - Castellano v0.91
// https://www.romhacking.net/
static struct BurnRomInfo nes_robocopcRomDesc[] = {
	{ "robocop (usa) - castellano v0.91.nes",          262160, 0x0f97e1e8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_robocopc)
STD_ROM_FN(nes_robocopc)

struct BurnDriver BurnDrvnes_robocopc = {
	"nes_robocopc", "nes_robocop", NULL, NULL, "2016",
	"RoboCop (Hack, Spanish v0.91)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_robocopcRomInfo, nes_robocopcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// RoboCop 2 (USA) (Rev 1) - Castellano v0.93
// https://www.romhacking.net/
static struct BurnRomInfo nes_robocop2cRomDesc[] = {
	{ "robocop 2 (usa) (rev 1) - castellano v0.93.nes",          262160, 0x4b3e87d9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_robocop2c)
STD_ROM_FN(nes_robocop2c)

struct BurnDriver BurnDrvnes_robocop2c = {
	"nes_robocop2c", "nes_robocop2", NULL, NULL, "2017",
	"RoboCop 2 (Hack, Spanish v0.93)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_robocop2cRomInfo, nes_robocop2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Robocop 3 (USA) - Castellano v0.9
// https://www.romhacking.net/
static struct BurnRomInfo nes_robocop3cRomDesc[] = {
	{ "robocop 3 (usa) - castellano v0.9.nes",          262160, 0xeda2dcb2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_robocop3c)
STD_ROM_FN(nes_robocop3c)

struct BurnDriver BurnDrvnes_robocop3c = {
	"nes_robocop3c", "nes_robocop3", NULL, NULL, "2017",
	"Robocop 3 (Hack, Spanish v0.9)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_robocop3cRomInfo, nes_robocop3cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Rush'n Attack (USA) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_rushnattackcRomDesc[] = {
	{ "rush'n attack (usa) - castellano v1.1.nes",          131088, 0xb6bdb5e5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_rushnattackc)
STD_ROM_FN(nes_rushnattackc)

struct BurnDriver BurnDrvnes_rushnattackc = {
	"nes_rushnattackc", "nes_rushnattack", NULL, NULL, "2021",
	"Rush'n Attack (Hack, Spanish v1.1)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_rushnattackcRomInfo, nes_rushnattackcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Smash T.V. (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_smashtvcRomDesc[] = {
	{ "smash t.v. (usa) - castellano v1.0.nes",          262160, 0x658cbecc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smashtvc)
STD_ROM_FN(nes_smashtvc)

struct BurnDriver BurnDrvnes_smashtvc = {
	"nes_smashtvc", "nes_smashtv", NULL, NULL, "2019",
	"Smash T.V. (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_smashtvcRomInfo, nes_smashtvcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Snake's Revenge (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_snakesrevengecRomDesc[] = {
	{ "snake's revenge (usa) - castellano v1.0.nes",          262160, 0x3d4a2507, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_snakesrevengec)
STD_ROM_FN(nes_snakesrevengec)

struct BurnDriver BurnDrvnes_snakesrevengec = {
	"nes_snakesrevengec", "nes_snakesrevenge", NULL, NULL, "2018",
	"Snake's Revenge (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN | GBF_ADV, 0,
	NESGetZipName, nes_snakesrevengecRomInfo, nes_snakesrevengecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Snow Brothers (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_snowbrotherscRomDesc[] = {
	{ "snow brothers (usa) - castellano v1.0.nes",          262160, 0x29c40f70, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_snowbrothersc)
STD_ROM_FN(nes_snowbrothersc)

struct BurnDriver BurnDrvnes_snowbrothersc = {
	"nes_snowbrothersc", "nes_snowbrothers", NULL, NULL, "2018",
	"Snow Brothers (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_snowbrotherscRomInfo, nes_snowbrotherscRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Spy Vs Spy (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_spyvsspycRomDesc[] = {
	{ "spy vs spy (usa) - castellano v1.0.nes",          40976, 0x0c4a3d91, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_spyvsspyc)
STD_ROM_FN(nes_spyvsspyc)

struct BurnDriver BurnDrvnes_spyvsspyc = {
	"nes_spyvsspyc", "nes_spyvsspy", NULL, NULL, "2019",
	"Spy Vs Spy (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_ACTION | GBF_STRATEGY, 0,
	NESGetZipName, nes_spyvsspycRomInfo, nes_spyvsspycRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Star Wars (Japan) (Namco) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_starwarsjcRomDesc[] = {
	{ "star wars (japan) (namco) - castellano v1.0.nes",          262160, 0xeef6974a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_starwarsjc)
STD_ROM_FN(nes_starwarsjc)

struct BurnDriver BurnDrvnes_starwarsjc = {
	"nes_starwarsjc", "nes_starwars", NULL, NULL, "2017",
	"Star Wars (Namco) (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_starwarsjcRomInfo, nes_starwarsjcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Star Wars (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_starwarscRomDesc[] = {
	{ "star wars (usa) - castellano v1.0.nes",          262160, 0x0e6e000c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_starwarsc)
STD_ROM_FN(nes_starwarsc)

struct BurnDriver BurnDrvnes_starwarsc = {
	"nes_starwarsc", "nes_starwars", NULL, NULL, "2019",
	"Star Wars (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_starwarscRomInfo, nes_starwarscRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Street Fighter 2010 - The Final Fight (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_streefig201cRomDesc[] = {
	{ "street fighter 2010 - the final fight (usa) - castellano v1.0.nes",          262160, 0x0c963bf7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_streefig201c)
STD_ROM_FN(nes_streefig201c)

struct BurnDriver BurnDrvnes_streefig201c = {
	"nes_streefig201c", "nes_streefig201", NULL, NULL, "2017",
	"Street Fighter 2010 - The Final Fight (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_streefig201cRomInfo, nes_streefig201cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Street Fighter II - The World Warrior (Unl) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_streefigiicRomDesc[] = {
	{ "street fighter ii - the world warrior (unl) - castellano v1.0.nes",          262160, 0x3861489, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_streefigiic)
STD_ROM_FN(nes_streefigiic)

struct BurnDriver BurnDrvnes_streefigiic = {
	"nes_streefigiic", "nes_streefigii", NULL, NULL, "201?",
	"Street Fighter II - The World Warrior (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_VSFIGHT, 0,
	NESGetZipName, nes_streefigiicRomInfo, nes_streefigiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Super Mario Bros. (World) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_smbcRomDesc[] = {
	{ "super mario bros. (world) - castellano v1.0.nes",          40976, 0x54de5344, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smbc)
STD_ROM_FN(nes_smbc)

struct BurnDriver BurnDrvnes_smbc = {
	"nes_smbc", "nes_smb", NULL, NULL, "2018",
	"Super Mario Bros. (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_smbcRomInfo, nes_smbcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Super Mario Bros. 2 (USA) (Rev 1) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_smb2cRomDesc[] = {
	{ "super mario bros. 2 (usa) (rev 1) - castellano v1.0.nes",          262160, 0x36114415, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smb2c)
STD_ROM_FN(nes_smb2c)

struct BurnDriver BurnDrvnes_smb2c = {
	"nes_smb2c", "nes_smb2", NULL, NULL, "2017",
	"Super Mario Bros. 2 (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_smb2cRomInfo, nes_smb2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Super Mario Bros. 3 (USA) (Rev 1) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_smb3castRomDesc[] = {
	{ "super mario bros. 3 (usa) (rev 1) - castellano v1.0.nes",          393232, 0x2e094f49, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smb3cast)
STD_ROM_FN(nes_smb3cast)

struct BurnDriver BurnDrvnes_smb3cast = {
	"nes_smb3cast", "nes_smb3", NULL, NULL, "2018",
	"Super Mario Bros. 3 (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_smb3castRomInfo, nes_smb3castRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Sword Master (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_swordmastercRomDesc[] = {
	{ "sword master (usa) - castellano v1.0.nes",          262160, 0x0f7fcbaf, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_swordmasterc)
STD_ROM_FN(nes_swordmasterc)

struct BurnDriver BurnDrvnes_swordmasterc = {
	"nes_swordmasterc", "nes_swordmaster", NULL, NULL, "2019",
	"Sword Master (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_swordmastercRomInfo, nes_swordmastercRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Target Renegade (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_targetrenegadecRomDesc[] = {
	{ "target - renegade (usa) - castellano v1.0.nes",          262160, 0x6754acae, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_targetrenegadec)
STD_ROM_FN(nes_targetrenegadec)

struct BurnDriver BurnDrvnes_targetrenegadec = {
	"nes_targetrenegadec", "nes_targetrenegade", NULL, NULL, "2017",
	"Target Renegade (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_targetrenegadecRomInfo, nes_targetrenegadecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Teenage Mutant Ninja Turtles - Tournament Fighters (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_tmnttoufigcRomDesc[] = {
	{ "teenage mutant ninja turtles - tournament fighters (usa) - castellano v1.0.nes",          262160, 0x32eac8c5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tmnttoufigc)
STD_ROM_FN(nes_tmnttoufigc)

struct BurnDriver BurnDrvnes_tmnttoufigc = {
	"nes_tmnttoufigc", "nes_tmnttoufig", NULL, NULL, "2020",
	"Teenage Mutant Ninja Turtles - Tournament Fighters (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_VSFIGHT, 0,
	NESGetZipName, nes_tmnttoufigcRomInfo, nes_tmnttoufigcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Teenage Mutant Ninja Turtles (USA) - Castellano v0.9
// https://www.romhacking.net/
static struct BurnRomInfo nes_tmntcastRomDesc[] = {
	{ "teenage mutant ninja turtles (usa) - castellano v0.9.nes",          262160, 0x30cd0997, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tmntcast)
STD_ROM_FN(nes_tmntcast)

struct BurnDriver BurnDrvnes_tmntcast = {
	"nes_tmntcast", "nes_tmnt", NULL, NULL, "2016",
	"Teenage Mutant Ninja Turtles (Hack, Spanish v0.9)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_tmntcastRomInfo, nes_tmntcastRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Teenage Mutant Ninja Turtles II - The Arcade Game (USA) - Castellano v0.9
// https://www.romhacking.net/
static struct BurnRomInfo nes_tmntiiarcgamcRomDesc[] = {
	{ "teenage mutant ninja turtles ii - the arcade game (usa) - castellano v0.9.nes",          524304, 0xb83bcec8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tmntiiarcgamc)
STD_ROM_FN(nes_tmntiiarcgamc)

struct BurnDriver BurnDrvnes_tmntiiarcgamc = {
	"nes_tmntiiarcgamc", "nes_tmntiiarcgam", NULL, NULL, "2016",
	"Teenage Mutant Ninja Turtles II - The Arcade Game (Hack, Spanish v0.9)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_tmntiiarcgamcRomInfo, nes_tmntiiarcgamcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Teenage Mutant Ninja Turtles III - The Manhattan Project (USA) - Castellano v0.91
// https://www.romhacking.net/
static struct BurnRomInfo nes_tmntiiicRomDesc[] = {
	{ "teenage mutant ninja turtles iii - the manhattan project (usa) - castellano v0.91.nes",          524304, 0x3e9ddb5c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tmntiiic)
STD_ROM_FN(nes_tmntiiic)

struct BurnDriver BurnDrvnes_tmntiiic = {
	"nes_tmntiiic", "nes_tmntiii", NULL, NULL, "2017",
	"Teenage Mutant Ninja Turtles III - The Manhattan Project (Hack, Spanish v0.91)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_tmntiiicRomInfo, nes_tmntiiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Terminator 2 - Judgment Day (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_termi2cRomDesc[] = {
	{ "terminator 2 - judgment day (usa) - castellano v1.0.nes",          262160, 0x43d08e31, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_termi2c)
STD_ROM_FN(nes_termi2c)

struct BurnDriver BurnDrvnes_termi2c = {
	"nes_termi2c", "nes_termi2", NULL, NULL, "2016",
	"Terminator 2 - Judgment Day (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_SCRFIGHT | GBF_RUNGUN, 0,
	NESGetZipName, nes_termi2cRomInfo, nes_termi2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Thexder (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_thexdercRomDesc[] = {
	{ "Thexder (japan) - castellano v1.0.nes",          40976, 0x52ac30c1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_thexderc)
STD_ROM_FN(nes_thexderc)

struct BurnDriver BurnDrvnes_thexderc = {
	"nes_thexderc", "nes_thexder", NULL, NULL, "2019",
	"Thexder (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_thexdercRomInfo, nes_thexdercRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Tiny Toon Adventures (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_tinytooadvcRomDesc[] = {
	{ "tiny toon adventures (usa) - castellano v1.0.nes",          262160, 0xbcd94fa2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tinytooadvc)
STD_ROM_FN(nes_tinytooadvc)

struct BurnDriver BurnDrvnes_tinytooadvc = {
	"nes_tinytooadvc", "nes_tinytooadv", NULL, NULL, "2019",
	"Tiny Toon Adventures (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_tinytooadvcRomInfo, nes_tinytooadvcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Tiny Toon Adventures 6 (Unl) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_tinytooadv6cRomDesc[] = {
	{ "tiny toon adventures 6 (unl) - castellano v1.0.nes",          524304, 0x93db8d83, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tinytooadv6c)
STD_ROM_FN(nes_tinytooadv6c)

struct BurnDriver BurnDrvnes_tinytooadv6c = {
	"nes_tinytooadv6c", "nes_tinytooadv6", NULL, NULL, "2018",
	"Tiny Toon Adventures 6 (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_tinytooadv6cRomInfo, nes_tinytooadv6cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Tom & Jerry - The Ultimate Game of Cat and Mouse! (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_tomjerultgamcatandmcRomDesc[] = {
	{ "tom & jerry - the ultimate game of cat and mouse! (usa) - castellano v1.0.nes",          262160, 0x9c6a9bcc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tomjerultgamcatandmc)
STD_ROM_FN(nes_tomjerultgamcatandmc)

struct BurnDriver BurnDrvnes_tomjerultgamcatandmc = {
	"nes_tomjerultgamcatandmc", "nes_tomjerultgamcatandm", NULL, NULL, "2018",
	"Tom & Jerry - The Ultimate Game of Cat and Mouse! (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_tomjerultgamcatandmcRomInfo, nes_tomjerultgamcatandmcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Top Gun - The Second Mission (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_topgunsecmiscRomDesc[] = {
	{ "top gun - the second mission (usa) - castellano v1.0.nes",          262160, 0x350f534b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_topgunsecmisc)
STD_ROM_FN(nes_topgunsecmisc)

struct BurnDriver BurnDrvnes_topgunsecmisc = {
	"nes_topgunsecmisc", "nes_topgunsecmis", NULL, NULL, "2019",
	"Top Gun - The Second Mission (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_SIM | GBF_SHOOT, 0,
	NESGetZipName, nes_topgunsecmiscRomInfo, nes_topgunsecmiscRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Top Gun (USA) (Rev 1) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_topguncRomDesc[] = {
	{ "top gun (usa) (rev 1) - castellano v1.0.nes",          131088, 0x879481af, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_topgunc)
STD_ROM_FN(nes_topgunc)

struct BurnDriver BurnDrvnes_topgunc = {
	"nes_topgunc", "nes_topgun", NULL, NULL, "2019",
	"Top Gun (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_SIM | GBF_SHOOT, 0,
	NESGetZipName, nes_topguncRomInfo, nes_topguncRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};							

// Total Recall (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_totalrecallcRomDesc[] = {
	{ "Total Recall (USA) - Castellano v1.0.nes",          131088, 0x93a02a67, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_totalrecallc)
STD_ROM_FN(nes_totalrecallc)

struct BurnDriver BurnDrvnes_totalrecallc = {
	"nes_totalrecallc", "nes_totalrecall", NULL, NULL, "2019",
	"Total Recall (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_totalrecallcRomInfo, nes_totalrecallcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Uncanny X-Men, The (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_uncanxmencRomDesc[] = {
	{ "uncanny x-men, the (usa) - castellano v1.0.nes",          131088, 0x21a9db24, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_uncanxmenc)
STD_ROM_FN(nes_uncanxmenc)

struct BurnDriver BurnDrvnes_uncanxmenc = {
	"nes_uncanxmenc", "nes_uncanxmen", NULL, NULL, "2019",
	"Uncanny X-Men, The (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_uncanxmencRomInfo, nes_uncanxmencRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Valkyrie no Bouken - Toki no Kagi Densetsu (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_valkynoboucRomDesc[] = {
	{ "valkyrie no bouken - toki no kagi densetsu (japan) - castellano v1.0.nes",          65552, 0x9e222fab, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_valkynobouc)
STD_ROM_FN(nes_valkynobouc)

struct BurnDriver BurnDrvnes_valkynobouc = {
	"nes_valkynobouc", "nes_valkynobou", NULL, NULL, "2019",
	"Valkyrie no Bouken - Toki no Kagi Densetsu (Hack, Spanish v1.0)\0", NULL, "Max1323", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_valkynoboucRomInfo, nes_valkynoboucRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Wizards & Warriors III - Kuros - Visions of Power (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_wizarwariiicRomDesc[] = {
	{ "wizards & warriors iii - kuros - visions of power (usa) - castellano v1.0.nes",          262160, 0x7135c442, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_wizarwariiic)
STD_ROM_FN(nes_wizarwariiic)

struct BurnDriver BurnDrvnes_wizarwariiic = {
	"nes_wizarwariiic", "nes_wizarwariii", NULL, NULL, "2020",
	"Wizards & Warriors III - Kuros - Visions of Power (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM | GBF_ADV, 0,
	NESGetZipName, nes_wizarwariiicRomInfo, nes_wizarwariiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Wolverine (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_wolverinecRomDesc[] = {
	{ "wolverine (usa) - castellano v1.0.nes",          262160, 0xdb5ee5cb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_wolverinec)
STD_ROM_FN(nes_wolverinec)

struct BurnDriver BurnDrvnes_wolverinec = {
	"nes_wolverinec", "nes_wolverine", NULL, NULL, "2019",
	"Wolverine (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_wolverinecRomInfo, nes_wolverinecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Yie Ar Kung-Fu (Japan) (V1.4) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_yiearkungfucRomDesc[] = {
	{ "yie ar kung-fu (japan) (v1.4) - castellano v1.0.nes",          24592, 0x032019a7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_yiearkungfuc)
STD_ROM_FN(nes_yiearkungfuc)

struct BurnDriver BurnDrvnes_yiearkungfuc = {
	"nes_yiearkungfuc", "nes_yiearkungfu", NULL, NULL, "2019",
	"Yie Ar Kung-Fu (V1.4) (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_VSFIGHT, 0,
	NESGetZipName, nes_yiearkungfucRomInfo, nes_yiearkungfucRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ys (Japan) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_yscRomDesc[] = {
	{ "ys (japan) - castellano v1.1.nes",          262160, 0x024dd1cf, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ysc)
STD_ROM_FN(nes_ysc)

struct BurnDriver BurnDrvnes_ysc = {
	"nes_ysc", "nes_ys", NULL, NULL, "2017",
	"Ys (Hack, Spanish v1.1)\0", NULL, "Takkun57", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_yscRomInfo, nes_yscRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ys II - Ancient Ys Vanished - The Final Chapter (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_ysiicRomDesc[] = {
	{ "ys ii - ancient ys vanished - the final chapter (japan) - castellano v1.0.nes",          393232, 0x20770bf0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ysiic)
STD_ROM_FN(nes_ysiic)

struct BurnDriver BurnDrvnes_ysiic = {
	"nes_ysiic", "nes_ysii", NULL, NULL, "2017",
	"Ys II - Ancient Ys Vanished - The Final Chapter (Hack, Spanish v1.0)\0", NULL, "Takkun57", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_ysiicRomInfo, nes_ysiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};	

// Zelda II - The Adventure of Link (Europe) (Rev 2) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_zeldaiicRomDesc[] = {
	{ "zelda ii - the adventure of link (europe) (rev 2) - castellano v1.0.nes",          262160, 0xd31d09ae, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_zeldaiic)
STD_ROM_FN(nes_zeldaiic)

struct BurnDriver BurnDrvnes_zeldaiic = {
	"nes_zeldaiic", "nes_zeldaii", NULL, NULL, "2016",
	"Zelda II - The Adventure of Link (Rev 2) (Hack, Spanish v1.0)\0", NULL, "jackic", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM | GBF_ADV, 0,
	NESGetZipName, nes_zeldaiicRomInfo, nes_zeldaiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Zelda II Redux - The Adventure of Link (Hack, v3.0)
// https://www.romhacking.net/hacks/5440/
static struct BurnRomInfo nes_zeldaiirdRomDesc[] = {
	{ "Zelda II - The Adventure of Link Redux v3.0 (2022)(ShadowOne333).nes",          262160, 0xf4bdec40, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_zeldaiird)
STD_ROM_FN(nes_zeldaiird)

struct BurnDriver BurnDrvnes_zeldaiird = {
	"nes_zeldaiird", "nes_zeldaii", NULL, NULL, "2022",
	"Zelda II Redux - The Adventure of Link (Hack, v3.0)\0", NULL, "ShadowOne333", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM | GBF_ADV, 0,
	NESGetZipName, nes_zeldaiirdRomInfo, nes_zeldaiirdRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Adventure Island IV (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_advenislivcRomDesc[] = {
	{ "adventure island iv (japan) - castellano v1.0.nes",          393232, 0x9ba3eb86, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advenislivc)
STD_ROM_FN(nes_advenislivc)

struct BurnDriver BurnDrvnes_advenislivc = {
	"nes_advenislivc", "nes_advenisliv", NULL, NULL, "2016",
	"Adventure Island IV (Hack, Spanish v1.0)\0", NULL, "Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_advenislivcRomInfo, nes_advenislivcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ganbare Goemon Gaiden - Kieta Ougon Kiseru (Japan) - Castellano v0.99
// https://www.romhacking.net/
static struct BurnRomInfo nes_ganbagoegaicRomDesc[] = {
	{ "ganbare goemon gaiden - kieta ougon kiseru (japan) - castellano v0.99.nes",          524304, 0x90ac0bda, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ganbagoegaic)
STD_ROM_FN(nes_ganbagoegaic)

struct BurnDriver BurnDrvnes_ganbagoegaic = {
	"nes_ganbagoegaic", "nes_ganbagoegai", NULL, NULL, "2021",
	"Ganbare Goemon Gaiden - Kieta Ougon Kiseru (Hack, Spanish v0.99)\0", NULL, "jackic", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_ganbagoegaicRomInfo, nes_ganbagoegaicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Battle Oozumou - Heisei Hero Basho (Japan)
static struct BurnRomInfo nes_sdbttloozumouRomDesc[] = {
	{ "sd battle oozumou - heisei hero basho (japan).nes",          262160, 0xcd1931de, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdbttloozumou)
STD_ROM_FN(nes_sdbttloozumou)

struct BurnDriver BurnDrvnes_sdbttloozumou = {
	"nes_sdbttloozumou", NULL, NULL, NULL, "1990",
	"SD Battle Oozumou - Heisei Hero Basho (Japan)\0", NULL, "Banpresto", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_VSFIGHT, 0,
	NESGetZipName, nes_sdbttloozumouRomInfo, nes_sdbttloozumouRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};								

// SD Gundam - Gachapon Senshi 2 - Capsule Senki (Japan)
static struct BurnRomInfo nes_sdgundgs2RomDesc[] = {
	{ "sd gundam - gachapon senshi 2 - capsule senki (japan).nes",          262160, 0x88641b52, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdgundgs2)
STD_ROM_FN(nes_sdgundgs2)

struct BurnDriver BurnDrvnes_sdgundgs2 = {
	"nes_sdgundgs2", NULL, NULL, NULL, "1989",
	"SD Gundam - Gachapon Senshi 2 - Capsule Senki (Japan)\0", NULL, "Bandai Shinsei", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_STRATEGY, 0,
	NESGetZipName, nes_sdgundgs2RomInfo, nes_sdgundgs2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Gundam - Gachapon Senshi 2 - Capsule Senki (Japan) - Castellano v1.0
static struct BurnRomInfo nes_sdgundgs2cRomDesc[] = {
	{ "sd gundam - gachapon senshi 2 - capsule senki (japan) - castellano v1.0.nes",          262160, 0x86274ffd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdgundgs2c)
STD_ROM_FN(nes_sdgundgs2c)

struct BurnDriver BurnDrvnes_sdgundgs2c = {
	"nes_sdgundgs2c", "nes_sdgundgs2", NULL, NULL, "2020",
	"SD Gundam - Gachapon Senshi 2 - Capsule Senki (Hack, Spanish v1.0)\0", NULL, "Max1323", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_STRATEGY, 0,
	NESGetZipName, nes_sdgundgs2cRomInfo, nes_sdgundgs2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Gundam - Gachapon Senshi 3 - Eiyuu Senki (Japan)
static struct BurnRomInfo nes_sdgundgs3RomDesc[] = {
	{ "sd gundam - gachapon senshi 3 - eiyuu senki (japan).nes",          262160, 0x3af5beff, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdgundgs3)
STD_ROM_FN(nes_sdgundgs3)

struct BurnDriver BurnDrvnes_sdgundgs3 = {
	"nes_sdgundgs3", NULL, NULL, NULL, "1990",
	"SD Gundam - Gachapon Senshi 3 - Eiyuu Senki (Japan)\0", NULL, "Yutaka", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_STRATEGY, 0,
	NESGetZipName, nes_sdgundgs3RomInfo, nes_sdgundgs3RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};						

// SD Gundam - Gachapon Senshi 4 - New Type Story (Japan)
static struct BurnRomInfo nes_sdgundgs4RomDesc[] = {
	{ "sd gundam - gachapon senshi 4 - new type story (japan).nes",          524304, 0x078b916e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdgundgs4)
STD_ROM_FN(nes_sdgundgs4)

struct BurnDriver BurnDrvnes_sdgundgs4 = {
	"nes_sdgundgs4", NULL, NULL, NULL, "1991",
	"SD Gundam - Gachapon Senshi 4 - New Type Story (Japan)\0", NULL, "Yutaka", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 3, HARDWARE_NES, GBF_STRATEGY, 0,
	NESGetZipName, nes_sdgundgs4RomInfo, nes_sdgundgs4RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Gundam - Gachapon Senshi 5 - Battle of Universal Century (Japan)
static struct BurnRomInfo nes_sdgachapon5RomDesc[] = {
	{ "sd gundam - gachapon senshi 5 - battle of universal century (japan).nes",          262160, 0x971d0527, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdgachapon5)
STD_ROM_FN(nes_sdgachapon5)

struct BurnDriver BurnDrvnes_sdgachapon5 = {
	"nes_sdgachapon5", NULL, NULL, NULL, "1992",
	"SD Gundam - Gachapon Senshi 5 - Battle of Universal Century (Japan)\0", NULL, "Yutaka", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 4, HARDWARE_NES, GBF_STRATEGY, 0,
	NESGetZipName, nes_sdgachapon5RomInfo, nes_sdgachapon5RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Gundam Gaiden - Knight Gundam Monogatari (Japan) (V1.0) (T-Eng)
static struct BurnRomInfo nes_sdgunkgmRomDesc[] = {
	{ "sd gundam gaiden - knight gundam monogatari (japan) (v1.0) (t-eng).nes",          393232, 0x2c91773e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdgunkgm)
STD_ROM_FN(nes_sdgunkgm)

struct BurnDriver BurnDrvnes_sdgunkgm = {
	"nes_sdgunkgm", NULL, NULL, NULL, "2020",
	"SD Gundam Gaiden - Knight Gundam Monogatari (T-Eng, v1.0)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_sdgunkgmRomInfo, nes_sdgunkgmRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Gundam Gaiden - Knight Gundam Monogatari (Japan) (V1.1)
static struct BurnRomInfo nes_sdgunkgmjRomDesc[] = {
	{ "sd gundam gaiden - knight gundam monogatari (japan) (v1.1).nes",          393232, 0xb6cafb31, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdgunkgmj)
STD_ROM_FN(nes_sdgunkgmj)

struct BurnDriver BurnDrvnes_sdgunkgmj = {
	"nes_sdgunkgmj", "nes_sdgunkgm", NULL, NULL, "1990",
	"SD Gundam Gaiden - Knight Gundam Monogatari (Japan, v1.1)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_sdgunkgmjRomInfo, nes_sdgunkgmjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Gundam Gaiden - Knight Gundam Monogatari 2 - Hikari no Kishi (Japan)
static struct BurnRomInfo nes_sdgunkgm2RomDesc[] = {
	{ "sd gundam gaiden - knight gundam monogatari 2 - hikari no kishi (japan).nes",          524304, 0x0c1319dd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdgunkgm2)
STD_ROM_FN(nes_sdgunkgm2)

struct BurnDriver BurnDrvnes_sdgunkgm2 = {
	"nes_sdgunkgm2", NULL, NULL, NULL, "1991",
	"SD Gundam Gaiden - Knight Gundam Monogatari 2 - Hikari no Kishi (Japan)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_sdgunkgm2RomInfo, nes_sdgunkgm2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Gundam Gaiden - Knight Gundam Monogatari 3 - Densetsu no Kishi Dan (Japan)
static struct BurnRomInfo nes_sdgunkgm3RomDesc[] = {
	{ "sd gundam gaiden - knight gundam monogatari 3 - densetsu no kishi dan (japan).nes",          524304, 0xf02a0aab, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdgunkgm3)
STD_ROM_FN(nes_sdgunkgm3)

struct BurnDriver BurnDrvnes_sdgunkgm3 = {
	"nes_sdgunkgm3", NULL, NULL, NULL, "1992",
	"SD Gundam Gaiden - Knight Gundam Monogatari 3 - Densetsu no Kishi Dan (Japan)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_sdgunkgm3RomInfo, nes_sdgunkgm3RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Hero Soukessen - Taose! Aku no Gundan (Japan)
static struct BurnRomInfo nes_sdhstangRomDesc[] = {
	{ "SD Hero Soukessen - Taose! Aku no Gundan (Japan).nes",          262160, 0xf17fefdd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdhstang)
STD_ROM_FN(nes_sdhstang)

struct BurnDriver BurnDrvnes_sdhstang = {
	"nes_sdhstang", NULL, NULL, NULL, "1990",
	"SD Hero Soukessen - Taose! Aku no Gundan (Japan)\0", NULL, "Banpresto", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_sdhstangRomInfo, nes_sdhstangRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Hero Soukessen - Taose! Aku no Gundan (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_sdhstangcRomDesc[] = {
	{ "sd hero soukessen - taose! aku no gundan (japan) - castellano v1.0.nes",          262160, 0x7c0a15a4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdhstangc)
STD_ROM_FN(nes_sdhstangc)

struct BurnDriver BurnDrvnes_sdhstangc = {
	"nes_sdhstangc", "nes_sdhstang", NULL, NULL, "2020",
	"SD Hero Soukessen - Taose! Aku no Gundan (Hack, Spanish v1.0)\0", NULL, "Max1323", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_sdhstangcRomInfo, nes_sdhstangcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Keiji - Blader (Japan) [T-Eng]
static struct BurnRomInfo nes_sdkbladeRomDesc[] = {
	{ "sd keiji - blader (japan) [t-eng].nes",          393232, 0x1fa8a57d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdkblade)
STD_ROM_FN(nes_sdkblade)

struct BurnDriverD BurnDrvnes_sdkblade = {
	"nes_sdkblade", NULL, NULL, NULL, "2021",
	"SD Keiji - Blader (T-Eng)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_sdkbladeRomInfo, nes_sdkbladeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Keiji - Blader (Japan)
static struct BurnRomInfo nes_sdkbladejRomDesc[] = {
	{ "sd keiji - blader (japan).nes",          262160, 0x452c373c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdkbladej)
STD_ROM_FN(nes_sdkbladej)

struct BurnDriverD BurnDrvnes_sdkbladej = {
	"nes_sdkbladej", "nes_sdkblade", NULL, NULL, "1991",
	"SD Keiji - Blader (Japan)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_sdkbladejRomInfo, nes_sdkbladejRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Gundam World - Gachapon Senshi - Scramble Wars (Japan) [b] - Castellano v1.0
static struct BurnRomInfo fds_sdgunworgacsescwacRomDesc[] = {
	{ "sd gundam world - gachapon senshi - scramble wars (japan) [b] - castellano v1.0.fds",          131016, 0x53f73319, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_sdgunworgacsescwac, fds_sdgunworgacsescwac, fds_fdsbios)
STD_ROM_FN(fds_sdgunworgacsescwac)

struct BurnDriver BurnDrvfds_sdgunworgacsescwac = {
	"fds_sdgunworgacsescwac", "fds_sdgunworgacsescwa", "fds_fdsbios", NULL, "2020",
	"SD Gundam World - Gachapon Senshi - Scramble Wars (Hack, Spanish v1.0)\0", NULL, "Max1323", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_FDS, GBF_STRATEGY, 0,
	NESGetZipName, fds_sdgunworgacsescwacRomInfo, fds_sdgunworgacsescwacRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Super Mario Brothers (Japan) - Castellano v1.0
static struct BurnRomInfo fds_supermarbrocRomDesc[] = {
	{ "super mario brothers (japan) - castellano v1.0.fds",          65516, 0x42d4a7c5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_supermarbroc, fds_supermarbroc, fds_fdsbios)
STD_ROM_FN(fds_supermarbroc)

struct BurnDriver BurnDrvfds_supermarbroc = {
	"fds_supermarbroc", "fds_supermarbro", "fds_fdsbios", NULL, "2020",
	"Super Mario Brothers (Hack, Spanish v1.0)\0", NULL, "Noishe", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_FDS, GBF_PLATFORM, 0,
	NESGetZipName, fds_supermarbrocRomInfo, fds_supermarbrocRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Link no Bouken - The Legend of Zelda 2 (Japan) (Rev 1) - Castellano v1.0
static struct BurnRomInfo fds_linknoboucRomDesc[] = {
	{ "link no bouken - the legend of zelda 2 (japan) (rev 1) - castellano v1.0.fds",          131000, 0x60a26198, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_linknobouc, fds_linknobouc, fds_fdsbios)
STD_ROM_FN(fds_linknobouc)

struct BurnDriver BurnDrvfds_linknobouc = {
	"fds_linknobouc", "fds_linknobou", "fds_fdsbios", NULL, "2019",
	"Link no Bouken - The Legend of Zelda 2 (Rev 1) (Hack, Spanish v1.0)\0", NULL, "jackic", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_FDS, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, fds_linknoboucRomInfo, fds_linknoboucRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Earth Bound Zero (USA)
// https://www.romhacking.net/
static struct BurnRomInfo nes_earthboundzeroRomDesc[] = {
	{ "earth bound zero (usa).nes",          524304, 0x6a5e39e2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_earthboundzero)
STD_ROM_FN(nes_earthboundzero)

struct BurnDriver BurnDrvnes_earthboundzero = {
	"nes_earthboundzero", "nes_earthbound", NULL, NULL, "2014",
	"Earth Bound Zero (USA)\0", NULL, "Giygas999", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_earthboundzeroRomInfo, nes_earthboundzeroRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Earth Bound Zero (USA) - Castellano v0.99
// https://www.romhacking.net/
static struct BurnRomInfo nes_earthboundzerocRomDesc[] = {
	{ "earth bound zero (usa) - castellano v0.99.nes",          524304, 0x2e24c315, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_earthboundzeroc)
STD_ROM_FN(nes_earthboundzeroc)

struct BurnDriver BurnDrvnes_earthboundzeroc = {
	"nes_earthboundzeroc", "nes_earthbound", NULL, NULL, "201?",
	"Earth Bound Zero (Hack, Spanish v0.99)\0", NULL, "jackic", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_earthboundzerocRomInfo, nes_earthboundzerocRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Dai-2-ji Super Robot Taisen (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_dai2jisuprotajcRomDesc[] = {
	{ "Dai-2-ji Super Robot Taisen (Japan) - Castellano v1.0.nes",          524304, 0x5b7ab480, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dai2jisuprotajc)
STD_ROM_FN(nes_dai2jisuprotajc)

struct BurnDriver BurnDrvnes_dai2jisuprotajc = {
	"nes_dai2jisuprotajc", "nes_dai2jisuprota", NULL, NULL, "2021",
	"Dai-2-ji Super Robot Taisen (Hack, Spanish v1.0)\0", NULL, "Max1323", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_STRATEGY, 0,
	NESGetZipName, nes_dai2jisuprotajcRomInfo, nes_dai2jisuprotajcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Dai-2-ji Super Robot Taisen (Japan) - Italian v0.98
// https://www.romhacking.net/
static struct BurnRomInfo nes_dai2jisuprotajiRomDesc[] = {
	{ "Super Robot Taisen (it) v0.98i (2019)(Wolfgare).nes",          524304, 0xdfd59e4e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dai2jisuprotaji)
STD_ROM_FN(nes_dai2jisuprotaji)

struct BurnDriver BurnDrvnes_dai2jisuprotaji = {
	"nes_dai2jisuprotaji", "nes_dai2jisuprota", NULL, NULL, "2019",
	"Dai-2-ji Super Robot Taisen (Hack, Italian v0.98)\0", NULL, "Wolfgare", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_STRATEGY, 0,
	NESGetZipName, nes_dai2jisuprotajiRomInfo, nes_dai2jisuprotajiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Shin 4 Nin Uchi Mahjong (T-Chi)
// http://wangyixiao.ysepan.com/
static struct BurnRomInfo nes_shin4ninuchmacRomDesc[] = {
	{ "Shin 4 Nin Uchi Mahjong (T-Chi).nes",          393232, 0x975f7379, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_shin4ninuchmac)
STD_ROM_FN(nes_shin4ninuchmac)

struct BurnDriver BurnDrvnes_shin4ninuchmac = {
	"nes_shin4ninuchmac", "nes_shin4ninuchma", NULL, NULL, "2013",
	"Shin 4 Nin Uchi Mahjong (T-Chi)\0", NULL, "Wang Yixiao", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 4, HARDWARE_NES, GBF_MAHJONG, 0,
	NESGetZipName, nes_shin4ninuchmacRomInfo, nes_shin4ninuchmacRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Alien 3 (T-Chi, v1.1)
// http://www.dmgrom.com/cnpage.html
static struct BurnRomInfo nes_alien3scRomDesc[] = {
	{ "Alien 3 (T-Chi, v1.1).nes",          262160, 0x70506f6f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_alien3sc)
STD_ROM_FN(nes_alien3sc)

struct BurnDriver BurnDrvnes_alien3sc = {
	"nes_alien3sc", "nes_alien3", NULL, NULL, "2020",
	"Alien 3 (T-Chi, v1.1)\0", NULL, "DMG Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_alien3scRomInfo, nes_alien3scRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Grand Master (T-Chi, v1.1)
// https://www.nesbbs.com/bbs/thread-51304-1-1.html
static struct BurnRomInfo nes_grandmastercRomDesc[] = {
	{ "Grand Master (T-Chi, v1.1).nes",          1048592, 0x505d11c0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_grandmasterc)
STD_ROM_FN(nes_grandmasterc)

struct BurnDriver BurnDrvnes_grandmasterc = {
	"nes_grandmasterc", "nes_grandmaster", NULL, NULL, "2021",
	"Grand Master (T-Chi, v1.1)\0", NULL, "HHNM Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_grandmastercRomInfo, nes_grandmastercRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Nekketsu! Street Basket - Ganbare Dunk Heroes (T-Chi, v2)
// https://www.nesbbs.com/bbs/forum.php?mod=viewthread&tid=50722 
static struct BurnRomInfo nes_nekkestrbascRomDesc[] = {
	{ "nekketsu! street basket - ganbare dunk heroes (T-Chi, v2).nes",          524304, 0x8f359b4f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_nekkestrbasc)
STD_ROM_FN(nes_nekkestrbasc)

struct BurnDriver BurnDrvnes_nekkestrbasc = {
	"nes_nekkestrbasc", "nes_nekkestrbas", NULL, NULL, "2020",
	"Nekketsu! Street Basket - Ganbare Dunk Heroes (T-Chi, v2)\0", NULL, "HHNM Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_SPORTSMISC, 0,
	NESGetZipName, nes_nekkestrbascRomInfo, nes_nekkestrbascRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Grand Master (Japan) (Castellano)
// https://www.romhacking.net/
static struct BurnRomInfo nes_grandmasterjcRomDesc[] = {
	{ "grand master (japan) - castellano.nes",          393232, 0x3ad512ce, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_grandmasterjc)
STD_ROM_FN(nes_grandmasterjc)

struct BurnDriver BurnDrvnes_grandmasterjc = {
	"nes_grandmasterjc", "nes_grandmaster", NULL, NULL, "2021",
	"Grand Master (Hack, Spanish v1.0)\0", NULL, "Max1323", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_grandmasterjcRomInfo, nes_grandmasterjcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// A Ressha de Ikou (T-Eng)
// https://www.romhacking.net/
static struct BurnRomInfo nes_atrainRomDesc[] = {
	{ "a ressha de ikou (t-eng).nes",          147472, 0xfd7bc532, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_atrain)
STD_ROM_FN(nes_atrain)

struct BurnDriver BurnDrvnes_atrain = {
	"nes_atrain", NULL, NULL, NULL, "2021",
	"A Ressha de Ikou (T-Eng)\0", NULL, "AgentOrange & Jink640", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_NES, GBF_STRATEGY, 0,
	NESGetZipName, nes_atrainRomInfo, nes_atrainRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// A Ressha de Ikou (Japan)
static struct BurnRomInfo nes_atrainjRomDesc[] = {
	{ "a ressha de ikou (japan).nes",          147472, 0x58fe5467, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_atrainj)
STD_ROM_FN(nes_atrainj)

	struct BurnDriver BurnDrvnes_atrainj = {
	"nes_atrainj", "nes_atrain", NULL, NULL, "1991",
	"A Ressha de Ikou (Japan)\0", NULL, "Pony Canyon", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_STRATEGY, 0,
	NESGetZipName, nes_atrainjRomInfo, nes_atrainjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Law of the West (T-Eng)
static struct BurnRomInfo nes_lawesteRomDesc[] = {
	{ "Law of the West T-Eng (2020)(GAFF Translations).nes",          262160, 0x34437445, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_laweste)
STD_ROM_FN(nes_laweste)

	struct BurnDriver BurnDrvnes_laweste = {
	"nes_laweste", NULL, NULL, NULL, "2020",
	"Law of the West (T-Eng)\0", NULL, "GAFF Translations", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_NES, GBF_SHOOT | GBF_ADV, 0,
	NESGetZipName, nes_lawesteRomInfo, nes_lawesteRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Law of the West (Japan)
static struct BurnRomInfo nes_lawestjRomDesc[] = {
	{ "Law of the West Jpn (1987)(Pony Canyon).nes",          131088, 0x9d1b5c98, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_lawestj)
STD_ROM_FN(nes_lawestj)

	struct BurnDriver BurnDrvnes_lawestj = {
	"nes_lawestj", "nes_laweste", NULL, NULL, "1987",
	"Law of the West (Japan)\0", NULL, "Pony Canyon", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_SHOOT | GBF_ADV, 0,
	NESGetZipName, nes_lawestjRomInfo, nes_lawestjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// War on Wheels (USA) (Proto)
static struct BurnRomInfo nes_wowheelsRomDesc[] = {
	{ "War on Wheels (Proto)(2006)(Jaleco).nes",          393232, 0xc957dcb0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_wowheels)
STD_ROM_FN(nes_wowheels)

	struct BurnDriver BurnDrvnes_wowheels = {
	"nes_wowheels", NULL, NULL, NULL, "1991-2006",
	"War on Wheels (USA) (Prototype)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_PROTOTYPE, 1, HARDWARE_NES, GBF_SPORTSMISC, 0,
	NESGetZipName, nes_wowheelsRomInfo, nes_wowheelsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Super Maruo (Unknown)
static struct BurnRomInfo nes_smaruoRomDesc[] = {
	{ "Super Maruo (Unknown).nes",          49168, 0xe5334399, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smaruo)
STD_ROM_FN(nes_smaruo)

	struct BurnDriver BurnDrvnes_smaruo = {
	"nes_smaruo", NULL, NULL, NULL, "199?",
	"Super Maruo\0", NULL, "Unknown", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_smaruoRomInfo, nes_smaruoRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Hokuto no Ken 4 - Shichisei Haken Den (Japan)
static struct BurnRomInfo nes_hokutonoken4RomDesc[] = {
	{ "Hokuto no Ken 4 - Shichisei Haken Den Jpn (1991)(Shouei System).nes",          262160, 0x06e95541, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_hokutonoken4)
STD_ROM_FN(nes_hokutonoken4)

struct BurnDriver BurnDrvnes_hokutonoken4 = {
	"nes_hokutonoken4", NULL, NULL, NULL, "1991",
	"Hokuto no Ken 4 - Shichisei Haken Den (Japan)\0", NULL, "TOEI Animation", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_hokutonoken4RomInfo, nes_hokutonoken4RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Smurfs, The (Europe)
static struct BurnRomInfo nes_smurfsRomDesc[] = {
	{ "Smurfs, The PAL (1994)(Infogrames).nes",          131088, 0x732b1a7a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smurfs)
STD_ROM_FN(nes_smurfs)

struct BurnDriver BurnDrvnes_smurfs = {
	"nes_smurfs", NULL, NULL, NULL, "1994",
	"Smurfs, The (Europe)\0", NULL, "Infogrames", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_smurfsRomInfo, nes_smurfsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT_PAL, SCREEN_WIDTH, SCREEN_HEIGHT_PAL
};

//Seikima II - Akuma no Gyakushuu (Japan)
static struct BurnRomInfo nes_seikima2RomDesc[] = {
	{ "Seikima II - Akuma no Gyakushuu (J)(1986)(CBS-Sony).nes",          65552, 0xb546589d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_seikima2)
STD_ROM_FN(nes_seikima2)

struct BurnDriver BurnDrvnes_seikima2 = {
	"nes_seikima2", NULL, NULL, NULL, "1986",
	"Seikima II - Akuma no Gyakushuu (Japan)\0", NULL, "CBS-Sony", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_seikima2RomInfo, nes_seikima2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Hoppin' Mad (Prototype) (1989) (Nowhere Prod.)
static struct BurnRomInfo nes_hoppinmadRomDesc[] = {
	{ "Hoppin' Mad (Proto) (1989)(Nowhere Prod.).nes",          40976, 0xac45ada9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_hoppinmad)
STD_ROM_FN(nes_hoppinmad)

struct BurnDriver BurnDrvnes_hoppinmad = {
	"nes_hoppinmad", NULL, NULL, NULL, "1989",
	"Hoppin' Mad (Prototype)\0", NULL, "Nowhere Productions", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_PROTOTYPE, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_hoppinmadRomInfo, nes_hoppinmadRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Hyaku no Sekai no Monogatari - The 100 World Story (T-Eng)
static struct BurnRomInfo nes_hyakuenRomDesc[] = {
	{ "Hyaku no Sekai no Monogatari - The 100 World Story (T-Eng)(2007)(AlanMidas).nes",	262160, 0x1385e8a5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_hyakuen)
STD_ROM_FN(nes_hyakuen)

struct BurnDriver BurnDrvnes_hyakuen = {
	"nes_hyakuen", NULL, NULL, NULL, "2007",
	"Hyaku no Sekai no Monogatari - The 100 World Story (T-Eng)\0", NULL, "AlanMidas", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 4, HARDWARE_NES, GBF_ADV | GBF_MISC, 0,
	NESGetZipName, nes_hyakuenRomInfo, nes_hyakuenRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Hyaku no Sekai no Monogatari - The 100 World Story (T-Ita)
static struct BurnRomInfo nes_hyakuitRomDesc[] = {
	{ "Hyaku no Sekai no Monogatari - The 100 World Story (T-Ita)(2021)(Mog).nes",	262160, 0x4d740153, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_hyakuit)
STD_ROM_FN(nes_hyakuit)

struct BurnDriver BurnDrvnes_hyakuit = {
	"nes_hyakuit", "nes_hyakuen", NULL, NULL, "2021",
	"Hyaku no Sekai no Monogatari - The 100 World Story (Hack, Italian v1.0)\0", NULL, "Mog", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 4, HARDWARE_NES, GBF_ADV | GBF_MISC, 0,
	NESGetZipName, nes_hyakuitRomInfo, nes_hyakuitRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Hyaku no Sekai no Monogatari - The 100 World Story (Japan)
static struct BurnRomInfo nes_hyakujpRomDesc[] = {
	{ "Hyaku no Sekai no Monogatari - The 100 World Story (J)(1991)(ASK).nes",	262160, 0x84ed70ce, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_hyakujp)
STD_ROM_FN(nes_hyakujp)

struct BurnDriver BurnDrvnes_hyakujp = {
	"nes_hyakujp", "nes_hyakuen", NULL, NULL, "1991",
	"Hyaku no Sekai no Monogatari - The 100 World Story (Japan)\0", NULL, "ASK", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 4, HARDWARE_NES, GBF_ADV | GBF_MISC, 0,
	NESGetZipName, nes_hyakujpRomInfo, nes_hyakujpRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//CJ's Elephant Antics (Codemasters)
static struct BurnRomInfo nes_cjeleantRomDesc[] = {
	{ "CJ's Elephant Antics (1992)(Codemasters).nes",          65552, 0x875855f1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cjeleant)
STD_ROM_FN(nes_cjeleant)

struct BurnDriver BurnDrvnes_cjeleant = {
	"nes_cjeleant", NULL, NULL, NULL, "1992",
	"CJ's Elephant Antics (USA) (Unl)\0", NULL, "Camerica", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_cjeleantRomInfo, nes_cjeleantRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Go! Dizzy Go! (Codemasters)
static struct BurnRomInfo nes_godizzygoRomDesc[] = {
	{ "Go! Dizzy Go! (1992)(Codemasters, Camerica).nes",          65552, 0x78716f4f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_godizzygo)
STD_ROM_FN(nes_godizzygo)

struct BurnDriver BurnDrvnes_godizzygo = {
	"nes_godizzygo", NULL, NULL, NULL, "1992",
	"Go! Dizzy Go! (USA) (Unl)\0", NULL, "Camerica", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MAZE | GBF_ACTION, 0,
	NESGetZipName, nes_godizzygoRomInfo, nes_godizzygoRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Tecmo Super Bowl 2020 v3.0 (Hack)
static struct BurnRomInfo nes_tsb20v3RomDesc[] = {
	{ "Tecmo Super Bowl 2020 v3.0 (2021)(SBlueman).nes",          524304, 0x24de7ae7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tsb20v3)
STD_ROM_FN(nes_tsb20v3)

struct BurnDriver BurnDrvnes_tsb20v3 = {
	"nes_tsb20v3", "nes_tecmosuperbowl", NULL, NULL, "2021",
	"Tecmo Super Bowl 2020 (Hack, v3.0)\0", NULL, "SBlueman", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_SPORTSMISC, 0,
	NESGetZipName, nes_tsb20v3RomInfo, nes_tsb20v3RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Tecmo Super Bowl 2021 Final Edition (Hack)
static struct BurnRomInfo nes_tsb21feRomDesc[] = {
	{ "Tecmo Super Bowl 2021 Final Edition (2022)(SBlueman).nes",          524304, 0x0557f756, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tsb21fe)
STD_ROM_FN(nes_tsb21fe)

struct BurnDriver BurnDrvnes_tsb21fe = {
	"nes_tsb21fe", "nes_tecmosuperbowl", NULL, NULL, "2022",
	"Tecmo Super Bowl 2021 Final Edition (Hack)\0", NULL, "SBlueman", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_SPORTSMISC, 0,
	NESGetZipName, nes_tsb21feRomInfo, nes_tsb21feRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Tecmo Super Bowl 2022 NCAA D.1 Edition (Hack)
static struct BurnRomInfo nes_tsb22ncaaRomDesc[] = {
	{ "Tecmo Super Bowl - 2022 NCAA D.1 Ed. (2022)(SBlueman).nes",          524304, 0x54133719, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tsb22ncaa)
STD_ROM_FN(nes_tsb22ncaa)

struct BurnDriver BurnDrvnes_tsb22ncaa = {
	"nes_tsb22ncaa", "nes_tecmosuperbowl", NULL, NULL, "2022",
	"Tecmo Super Bowl 2022 NCAA D.1 Edition (Hack)\0", NULL, "SBlueman", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_SPORTSMISC, 0,
	NESGetZipName, nes_tsb22ncaaRomInfo, nes_tsb22ncaaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Adam & Eve (Unl)
static struct BurnRomInfo nes_adameveRomDesc[] = {
	{ "Adam & Eve (1991)(Mega Soft)(Unl).nes",          49168, 0x9e2d9a66, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_adameve)
STD_ROM_FN(nes_adameve)

struct BurnDriver BurnDrvnes_adameve = {
	"nes_adameve", NULL, NULL, NULL, "1991",
	"Adam & Eve (Unl)\0", NULL, "Mega Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_adameveRomInfo, nes_adameveRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Creatom (Unl)
static struct BurnRomInfo nes_creatomRomDesc[] = {
	{ "Creatom (1991)(Micro Genius)(Unl).nes",          98320, 0x1a1328bc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_creatom)
STD_ROM_FN(nes_creatom)

struct BurnDriver BurnDrvnes_creatom = {
	"nes_creatom", NULL, NULL, NULL, "1991",
	"Creatom (Unl)\0", NULL, "Micro Genius", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_creatomRomInfo, nes_creatomRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Frog Adventure (Unl)
static struct BurnRomInfo nes_frogadvRomDesc[] = {
	{ "Frog Adventure (Unl)(1993)(Sachen).nes",          98320, 0x842d3e9f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_frogadv)
STD_ROM_FN(nes_frogadv)

struct BurnDriver BurnDrvnes_frogadv = {
	"nes_frogadv", NULL, NULL, NULL, "1993",
	"Frog Adventure (Unl)\0", NULL, "Sachen", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_frogadvRomInfo, nes_frogadvRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Depth Bomb v1 (Unl)
static struct BurnRomInfo nes_depbomb1RomDesc[] = {
	{ "Depth Bomb v1 (Unl)(2004)(Nice Code).nes",          24592, 0xa1d97a3e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_depbomb1)
STD_ROM_FN(nes_depbomb1)

struct BurnDriver BurnDrvnes_depbomb1 = {
	"nes_depbomb1", NULL, NULL, NULL, "2004",
	"Depth Bomb v1 (Unl)\0", NULL, "Nice Code Software", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_STRATEGY, 0,
	NESGetZipName, nes_depbomb1RomInfo, nes_depbomb1RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Depth Bomb v2 (Unl)
static struct BurnRomInfo nes_depbomb2RomDesc[] = {
	{ "Depth Bomb v2 (Unl)(2004)(Nice Code).nes",          24592, 0x4cf58383, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_depbomb2)
STD_ROM_FN(nes_depbomb2)

struct BurnDriver BurnDrvnes_depbomb2 = {
	"nes_depbomb2", "nes_depbomb1", NULL, NULL, "2004",
	"Depth Bomb v2 (Unl)\0", NULL, "Nice Code Software", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_STRATEGY, 0,
	NESGetZipName, nes_depbomb2RomInfo, nes_depbomb2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Ushio to Tora - Shinen no Taiyou (T-Eng)
static struct BurnRomInfo nes_ushiotoraRomDesc[] = {
	{ "Ushio to Tora - Shinen no Taiyou T-Eng (2022)(BlackPaladin).nes",          393232, 0x9472f0af, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ushiotora)
STD_ROM_FN(nes_ushiotora)

struct BurnDriver BurnDrvnes_ushiotora = {
	"nes_ushiotora", NULL, NULL, NULL, "2022",
	"Ushio to Tora - Shinen no Taiyou (T-Eng)\0", NULL, "BlackPaladin", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_ushiotoraRomInfo, nes_ushiotoraRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Ushio to Tora - Shinen no Taiyou (Japan)
static struct BurnRomInfo nes_ushiotorajRomDesc[] = {
	{ "Ushio to Tora - Shinen no Taiyou (J)(1993)(Yutaka).nes",          393232, 0x7aa8f7d0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ushiotoraj)
STD_ROM_FN(nes_ushiotoraj)

struct BurnDriver BurnDrvnes_ushiotoraj = {
	"nes_ushiotoraj", "nes_ushiotora", NULL, NULL, "1993",
	"Ushio to Tora - Shinen no Taiyou (Japan)\0", NULL, "Yutaka", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_ushiotorajRomInfo, nes_ushiotorajRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Xybots (USA) (Prototype)
static struct BurnRomInfo nes_xybotsRomDesc[] = {
	{ "Xybots (proto)(1990)(Tengen).nes",          196624, 0x59dae478, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_xybots)
STD_ROM_FN(nes_xybots)

struct BurnDriver BurnDrvnes_xybots = {
	"nes_xybots", NULL, NULL, NULL, "1990",
	"Xybots (USA) (Prototype)\0", NULL, "Tengen", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_PROTOTYPE, 2, HARDWARE_NES, GBF_SHOOT | GBF_MAZE, 0,
	NESGetZipName, nes_xybotsRomInfo, nes_xybotsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Virus (USA) (Prototype)
static struct BurnRomInfo nes_virusRomDesc[] = {
	{ "Virus (U)(Proto)(1990)(Nintendo).nes",          49168, 0x409c55f5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_virus)
STD_ROM_FN(nes_virus)

struct BurnDriver BurnDrvnes_virus = {
	"nes_virus", "nes_drmario", NULL, NULL, "1990",
	"Virus (USA) (Prototype)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_PROTOTYPE, 2, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_virusRomInfo, nes_virusRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Kaguya Hime Densetsu (T-Eng)
static struct BurnRomInfo nes_kaguyaeRomDesc[] = {
	{ "Kaguya Hime Densetsu (T-Eng)(2022)(FCandChill).nes",          524304, 0x80de5290, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_kaguyae)
STD_ROM_FN(nes_kaguyae)

struct BurnDriver BurnDrvnes_kaguyae = {
	"nes_kaguyae", NULL, NULL, NULL, "2022",
	"Kaguya Hime Densetsu (T-Eng)\0", NULL, "FCandChill", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_kaguyaeRomInfo, nes_kaguyaeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Kaguya Hime Densetsu (Japan)
static struct BurnRomInfo nes_kaguyajRomDesc[] = {
	{ "Kaguya Hime Densetsu (J)(1988)(Victor).nes",          262160, 0x69565d0e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_kaguyaj)
STD_ROM_FN(nes_kaguyaj)

struct BurnDriver BurnDrvnes_kaguyaj = {
	"nes_kaguyaj", "nes_kaguyae", NULL, NULL, "1988",
	"Kaguya Hime Densetsu (Japan)\0", NULL, "Victor", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_kaguyajRomInfo, nes_kaguyajRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Echo Chamber (Unl)
static struct BurnRomInfo nes_echocambRomDesc[] = {
	{ "Echo Chamber (Unl)(200X)(Nice Code).nes",          40976, 0xc2aa9dc8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_echocamb)
STD_ROM_FN(nes_echocamb)

struct BurnDriver BurnDrvnes_echocamb = {
	"nes_echocamb", NULL, NULL, NULL, "200?",
	"Echo Chamber (Unl)\0", NULL, "Nice Code", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_echocambRomInfo, nes_echocambRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Eeeck! A Mouse! (Unl)
static struct BurnRomInfo nes_eeeckRomDesc[] = {
	{ "Eeeck! A Mouse! (Unl)(200X)(Nice Code).nes",          40976, 0x9efb7a7d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_eeeck)
STD_ROM_FN(nes_eeeck)

struct BurnDriver BurnDrvnes_eeeck = {
	"nes_eeeck", NULL, NULL, NULL, "200?",
	"Eeeck! A Mouse! (Unl)\0", NULL, "Nice Code", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_eeeckRomInfo, nes_eeeckRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Sara Parker's Pool Challenge
static struct BurnRomInfo nes_spoolchalRomDesc[] = {
	{ "Sara Parker's Pool Challenge (2010)(RetroUSB).nes",          98320, 0x5a34e197, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_spoolchal)
STD_ROM_FN(nes_spoolchal)

struct BurnDriver BurnDrvnes_spoolchal = {
	"nes_spoolchal", "nes_rpoolchal", NULL, NULL, "2010",
	"Sara Parker's Pool Challenge\0", NULL, "retroUSB", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 4, HARDWARE_NES, GBF_SPORTSMISC, 0,
	NESGetZipName, nes_spoolchalRomInfo, nes_spoolchalRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Robert Byrne's Pool Challenge (Prototype)
static struct BurnRomInfo nes_rpoolchalRomDesc[] = {
	{ "Robert Byrne's Pool Challenge Prototype (1996)(Odyssey Software).nes",          98320, 0x9b273e2b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_rpoolchal)
STD_ROM_FN(nes_rpoolchal)

struct BurnDriver BurnDrvnes_rpoolchal = {
	"nes_rpoolchal", NULL, NULL, NULL, "1996",
	"Robert Byrne's Pool Challenge (Prototype)\0", NULL, "Odyssey Software", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_PROTOTYPE, 4, HARDWARE_NES, GBF_SPORTSMISC, 0,
	NESGetZipName, nes_rpoolchalRomInfo, nes_rpoolchalRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Titanic (T-Eng, v2.1)
static struct BurnRomInfo nes_titanicRomDesc[] = {
	{ "Titanic (T-Eng, v2.1)(2022(pacnsacdave).nes",          524304, 0x4f0bef8a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_titanic)
STD_ROM_FN(nes_titanic)

struct BurnDriver BurnDrvnes_titanic = {
	"nes_titanic", NULL, NULL, NULL, "2022",
	"Titanic (T-Eng, v2.1)\0", NULL, "pacnsacdave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM | GBF_ADV, 0,
	NESGetZipName, nes_titanicRomInfo, nes_titanicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Titanic (China) (Unl)
static struct BurnRomInfo nes_titaniccRomDesc[] = {
	{ "Titanic (Unl)(Chi)(2005)(Shenzhen Nanjing Technology).nes",          524304, 0xc63a373f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_titanicc)
STD_ROM_FN(nes_titanicc)

struct BurnDriver BurnDrvnes_titanicc = {
	"nes_titanicc", "nes_titanic", NULL, NULL, "2005",
	"Titanic (China) (Unl)\0", NULL, "Shenzhen Nanjing Technology", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_PLATFORM | GBF_ADV, 0,
	NESGetZipName, nes_titaniccRomInfo, nes_titaniccRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Tombs & Treasure (USA)
static struct BurnRomInfo nes_tmbtreasureRomDesc[] = {
	{ "Tombs & Treasure (U)(1989)(Infocom).nes",          262160, 0x013bd19f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tmbtreasure)
STD_ROM_FN(nes_tmbtreasure)

struct BurnDriver BurnDrvnes_tmbtreasure = {
	"nes_tmbtreasure", NULL, NULL, NULL, "1989",
	"Tombs & Treasure (USA)\0", NULL, "Infocom", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_RPG | GBF_ADV, 0,
	NESGetZipName, nes_tmbtreasureRomInfo, nes_tmbtreasureRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Taiyou no Shinden - Asteka 2 (Japan)
static struct BurnRomInfo nes_asteka2RomDesc[] = {
	{ "Taiyou no Shinden - Asteka 2 (J)(1988)(Tokyo Shoseki).nes",          262160, 0x01259d1f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_asteka2)
STD_ROM_FN(nes_asteka2)

struct BurnDriver BurnDrvnes_asteka2 = {
	"nes_asteka2", "nes_tmbtreasure", NULL, NULL, "1988",
	"Taiyou no Shinden - Asteka 2 (Japan)\0", NULL, "Tokyo Shoseki", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_RPG | GBF_ADV, 0,
	NESGetZipName, nes_asteka2RomInfo, nes_asteka2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Monkey King (China) (Unl)
static struct BurnRomInfo nes_monkeykingRomDesc[] = {
	{ "Monkey King (Unl)(2005)(Waixing).nes",          524304, 0xe5119eed, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_monkeyking)
STD_ROM_FN(nes_monkeyking)

struct BurnDriver BurnDrvnes_monkeyking = {
	"nes_monkeyking", NULL, NULL, NULL, "2005",
	"Monkey King (China) (Unl)\0", NULL, "Waixing", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_monkeykingRomInfo, nes_monkeykingRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Honoo no Doukyuuji - Dodge Danpei (Japan)
static struct BurnRomInfo nes_ddballjRomDesc[] = {
	{ "Honoo no Doukyuuji - Dodge Danpei (J)(1992)(Sunsoft).nes",          393232, 0xaea2a320, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ddballj)
STD_ROM_FN(nes_ddballj)

struct BurnDriver BurnDrvnes_ddballj = {
	"nes_ddballj", "nes_ddball", NULL, NULL, "1992",
	"Honoo no Doukyuuji - Dan's Dodge Ball (Japan)\0", NULL, "Sunsoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_ADV | GBF_STRATEGY, 0,
	NESGetZipName, nes_ddballjRomInfo, nes_ddballjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Honoo no Doukyuuji - Dodge Danpei (T-Eng, v1.0b)
// https://www.romhacking.net/translations/1255/
static struct BurnRomInfo nes_ddballRomDesc[] = {
	{ "Honoo no Doukyuuji - Dodge Danpei T-Eng v1.0.b (2017)(TransGen).nes",          393232, 0x47325872, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ddball)
STD_ROM_FN(nes_ddball)

struct BurnDriver BurnDrvnes_ddball = {
	"nes_ddball", NULL, NULL, NULL, "2017",
	"Honoo no Doukyuuji - Dan's Dodge Ball (T-Eng, v1.0b)\0", NULL, "TransGen", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_NES, GBF_ADV | GBF_STRATEGY, 0,
	NESGetZipName, nes_ddballRomInfo, nes_ddballRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Honoo no Doukyuuji - Dodge Danpei 2 (Japan)
static struct BurnRomInfo nes_ddball2jRomDesc[] = {
	{ "Honoo no Doukyuuji - Dodge Danpei 2 (J)(1993)(Sunsoft).nes",          524304, 0x646666f2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ddball2j)
STD_ROM_FN(nes_ddball2j)

struct BurnDriver BurnDrvnes_ddball2j = {
	"nes_ddball2j", "nes_ddball2", NULL, NULL, "1993",
	"Honoo no Doukyuuji - Dan's Dodge Ball 2 (Japan)\0", NULL, "Sunsoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_ADV | GBF_STRATEGY, 0,
	NESGetZipName, nes_ddball2jRomInfo, nes_ddball2jRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Honoo no Doukyuuji - Dodge Danpei 2 (T-Eng, v1.0a)
// https://www.romhacking.net/translations/1269/
static struct BurnRomInfo nes_ddball2RomDesc[] = {
	{ "Honoo no Doukyuuji - Dodge Danpei 2 T-Eng v1.0a (2017)(TransGen).nes",          524304, 0x95469699, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ddball2)
STD_ROM_FN(nes_ddball2)

struct BurnDriver BurnDrvnes_ddball2 = {
	"nes_ddball2", NULL, NULL, NULL, "2017",
	"Honoo no Doukyuuji - Dan's Dodge Ball 2 (T-Eng, v1.0a)\0", NULL, "TransGen", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_NES, GBF_ADV | GBF_STRATEGY, 0,
	NESGetZipName, nes_ddball2RomInfo, nes_ddball2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// NES Wars (T-Eng)
// https://www.romhacking.net/translations/6828/
static struct BurnRomInfo nes_neswarsRomDesc[] = {
	{ "NES Wars T-Eng (2023)(Stardust Crusaders).nes",          196624, 0x35da4b73, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_neswars)
STD_ROM_FN(nes_neswars)

struct BurnDriver BurnDrvnes_neswars = {
	"nes_neswars", "nes_famicomwars", NULL, NULL, "2023",
	"NES Wars (T-Eng)\0", NULL, "Stardust Crusaders", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_STRATEGY, 0,
	NESGetZipName, nes_neswarsRomInfo, nes_neswarsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ripple Island (Japan)
static struct BurnRomInfo nes_ripislandjRomDesc[] = {
	{ "Ripple Island (J)(1988)(Sunsoft).nes",          131088, 0x7148c342, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ripislandj)
STD_ROM_FN(nes_ripislandj)

struct BurnDriver BurnDrvnes_ripislandj = {
	"nes_ripislandj", "nes_ripisland", NULL, NULL, "1988",
	"Ripple Island (Japan)\0", NULL, "Sunsoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_ripislandjRomInfo, nes_ripislandjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ripple Island (T-Eng)
// https://www.romhacking.net/translations/3769/
static struct BurnRomInfo nes_ripislandRomDesc[] = {
	{ "Ripple Island T-Eng (2018)(LIPEMCO! Translations).nes",          262160, 0x51f5e289, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ripisland)
STD_ROM_FN(nes_ripisland)

struct BurnDriver BurnDrvnes_ripisland = {
	"nes_ripisland", NULL, NULL, NULL, "2018",
	"Ripple Island (T-Eng)\0", NULL, "LIPEMCO! Translations", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_ripislandRomInfo, nes_ripislandRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Zoids - Chuuou Tairiku no Tatakai (Japan)
static struct BurnRomInfo nes_zoidsjRomDesc[] = {
	{ "Zoids - Chuuou Tairiku no Tatakai (J)(1987)(Toshiba-EMI).nes",          131088, 0xe3d6a15a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_zoidsj)
STD_ROM_FN(nes_zoidsj)

struct BurnDriver BurnDrvnes_zoidsj = {
	"nes_zoidsj", "nes_zoids", NULL, NULL, "1987",
	"Zoids - Chuuou Tairiku no Tatakai (Japan)\0", NULL, "Toshiba-EMI", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 1, HARDWARE_NES, GBF_ACTION | GBF_RPG, 0,
	NESGetZipName, nes_zoidsjRomInfo, nes_zoidsjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Zoids - Battle of Central Continent (T-Eng)
// https://www.romhacking.net/translations/6845/
static struct BurnRomInfo nes_zoidsRomDesc[] = {
	{ "Zoids - Chuuou Tairiku no Tatakai T-Eng (2023)(Life With Matthew).nes",          131088, 0x1528e6c0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_zoids)
STD_ROM_FN(nes_zoids)

struct BurnDriver BurnDrvnes_zoids = {
	"nes_zoids", NULL, NULL, NULL, "2023",
	"Zoids - Battle of Central Continent (T-Eng)\0", NULL, "Life With Matthew", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK, 1, HARDWARE_NES, GBF_ACTION | GBF_RPG, 0,
	NESGetZipName, nes_zoidsRomInfo, nes_zoidsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Dino Hockey (Prototype)
static struct BurnRomInfo nes_dinohockeyRomDesc[] = {
	{ "Dino Hockey (Proto)(1991)(Virgin Games).nes",          262160, 0xc3710b73, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dinohockey)
STD_ROM_FN(nes_dinohockey)

struct BurnDriver BurnDrvnes_dinohockey = {
	"nes_dinohockey", NULL, NULL, NULL, "1991",
	"Dino Hockey (Prototype)\0", NULL, "Sunsoft - Virgin Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_PROTOTYPE, 2, HARDWARE_NES, GBF_SPORTSMISC, 0,
	NESGetZipName, nes_dinohockeyRomInfo, nes_dinohockeyRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Quarter Back Scramble (Prototype)
static struct BurnRomInfo nes_qbscrambleRomDesc[] = {
	{ "Quarter Back Scramble (USA)(Proto)(1990)(Pony Canyon).nes",          262160, 0x0a55b11f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_qbscramble)
STD_ROM_FN(nes_qbscramble)

struct BurnDriver BurnDrvnes_qbscramble = {
	"nes_qbscramble", NULL, NULL, NULL, "1990",
	"Quarter Back Scramble (Prototype)\0", NULL, "Pony Canyon", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_PROTOTYPE, 2, HARDWARE_NES, GBF_SPORTSMISC, 0,
	NESGetZipName, nes_qbscrambleRomInfo, nes_qbscrambleRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Quarter Back Scramble (Japan)
static struct BurnRomInfo nes_qbscramblejRomDesc[] = {
	{ "Quarter Back Scramble (J)(1990)(Pony Canyon).nes",          262160, 0xc373aeec, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_qbscramblej)
STD_ROM_FN(nes_qbscramblej)

struct BurnDriver BurnDrvnes_qbscramblej = {
	"nes_qbscramblej", "nes_qbscramble", NULL, NULL, "1990",
	"Quarter Back Scramble (Japan)\0", NULL, "Pony Canyon", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_SPORTSMISC, 0,
	NESGetZipName, nes_qbscramblejRomInfo, nes_qbscramblejRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Mike Ditka's Big Play Football (Unl)
static struct BurnRomInfo nes_mdfootballRomDesc[] = {
	{ "Mike Ditka's Big Play Football (Unl)(2008)(NGD).nes",          262160, 0x6bf165f3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mdfootball)
STD_ROM_FN(nes_mdfootball)

struct BurnDriver BurnDrvnes_mdfootball = {
	"nes_mdfootball", "nes_qbscramble", NULL, NULL, "2008",
	"Mike Ditka's Big Play Football (Unl)\0", NULL, "NGD", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_SPORTSMISC, 0,
	NESGetZipName, nes_mdfootballRomInfo, nes_mdfootballRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Airball
static struct BurnRomInfo nes_airballRomDesc[] = {
	{ "Airball (2007)(Retrozone).nes",          131088, 0x9a535a93, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_airball)
STD_ROM_FN(nes_airball)

struct BurnDriver BurnDrvnes_airball = {
	"nes_airball", NULL, NULL, NULL, "2007",
	"Airball\0", NULL, "Retrozone - Tengen", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_MAZE | GBF_PUZZLE, 0,
	NESGetZipName, nes_airballRomInfo, nes_airballRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Airball (Prototype)
static struct BurnRomInfo nes_airballpRomDesc[] = {
	{ "Airball (Proto)(1990)(Tengen).nes",          131088, 0x8adb80e1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_airballp)
STD_ROM_FN(nes_airballp)

struct BurnDriver BurnDrvnes_airballp = {
	"nes_airballp", "nes_airball", NULL, NULL, "1990",
	"Airball (Prototype)\0", NULL, "Tengen", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_PROTOTYPE, 1, HARDWARE_NES, GBF_MAZE | GBF_PUZZLE, 0,
	NESGetZipName, nes_airballpRomInfo, nes_airballpRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// END of "Non Homebrew (hand-added!)"

// Homebrew (hand-added)

static struct BurnRomInfo nes_5starRomDesc[] = {
	{ "5-star Restaurant (2023)(Wendel Scardua).nes",          81936, 0x42658f47, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_5star)
STD_ROM_FN(nes_5star)

struct BurnDriver BurnDrvnes_5star = {
	"nes_5star", NULL, NULL, NULL, "2023",
	"5-star Restaurant (HB)\0", NULL, "Wendel Scardua", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_5starRomInfo, nes_5starRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_blazblocksRomDesc[] = {
	{ "Blazing Blocks (2023)(FG Software).nes",          65552, 0x3a8f9933, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_blazblocks)
STD_ROM_FN(nes_blazblocks)

struct BurnDriver BurnDrvnes_blazblocks = {
	"nes_blazblocks", NULL, NULL, NULL, "2023",
	"Blazing Blocks (HB)\0", NULL, "FG Software", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_blazblocksRomInfo, nes_blazblocksRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bowbeastRomDesc[] = {
	{ "Bowels of the Beast (2021)(T-bone).nes",          524304, 0x2002b239, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bowbeast)
STD_ROM_FN(nes_bowbeast)

struct BurnDriver BurnDrvnes_bowbeast = {
	"nes_bowbeast", NULL, NULL, NULL, "2021",
	"Bowels of the Beast (HB)\0", NULL, "T-Bone", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_bowbeastRomInfo, nes_bowbeastRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_convoyRomDesc[] = {
	{ "Convoy, The (2023)(FG Software).nes",          24592, 0xe187eabf, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_convoy)
STD_ROM_FN(nes_convoy)

struct BurnDriver BurnDrvnes_convoy = {
	"nes_convoy", NULL, NULL, NULL, "2023",
	"Convoy, The (HB)\0", NULL, "FG Software", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_convoyRomInfo, nes_convoyRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_mystpillarsRomDesc[] = {
	{ "Mystic Pillars (2008)(Sivak Games).nes",          524304, 0x420f1b1a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mystpillars)
STD_ROM_FN(nes_mystpillars)

struct BurnDriver BurnDrvnes_mystpillars = {
	"nes_mystpillars", NULL, NULL, NULL, "2008",
	"Mystic Pillars (HB)\0", NULL, "Sivak Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_mystpillarsRomInfo, nes_mystpillarsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_pplanev2RomDesc[] = {
	{ "Paper Plane, The v2 (2022)(5cinq).nes",          524304, 0x9fe3d96e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_pplanev2)
STD_ROM_FN(nes_pplanev2)

struct BurnDriver BurnDrvnes_pplanev2 = {
	"nes_pplanev2", NULL, NULL, NULL, "2022",
	"Paper Plane, The (HB, v2)\0", NULL, "5cinq", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_pplanev2RomInfo, nes_pplanev2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_pentablocatRomDesc[] = {
	{ "Pentablocat (2023)(Pineberry fox).nes",          24592, 0x27a14751, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_pentablocat)
STD_ROM_FN(nes_pentablocat)

struct BurnDriver BurnDrvnes_pentablocat = {
	"nes_pentablocat", NULL, NULL, NULL, "2023",
	"Pentablocat (HB)\0", NULL, "Pineberry fox", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_pentablocatRomInfo, nes_pentablocatRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_spacecatRomDesc[] = {
	{ "Space Cat (2023).nes",          524304, 0x809a6b97, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_spacecat)
STD_ROM_FN(nes_spacecat)

struct BurnDriver BurnDrvnes_spacecat = {
	"nes_spacecat", NULL, NULL, NULL, "2023",
	"Space Cat (HB)\0", NULL, "unknown", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_spacecatRomInfo, nes_spacecatRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_spacemutdgRomDesc[] = {
	{ "Space Mutants DG (2023)(T-Bone).nes",          524304, 0x3f73e619, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_spacemutdg)
STD_ROM_FN(nes_spacemutdg)

struct BurnDriver BurnDrvnes_spacemutdg = {
	"nes_spacemutdg", NULL, NULL, NULL, "2023",
	"Space Mutants - DG BGM (HB)\0", NULL, "T-Bone", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_spacemutdgRomInfo, nes_spacemutdgRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_spacemutjkRomDesc[] = {
	{ "Space Mutants JK (2023)(T-Bone).nes",          524304, 0xd202f6b6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_spacemutjk)
STD_ROM_FN(nes_spacemutjk)

struct BurnDriver BurnDrvnes_spacemutjk = {
	"nes_spacemutjk", "nes_spacemutdg", NULL, NULL, "2023",
	"Space Mutants - JK BGM (HB)\0", NULL, "T-Bone", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_spacemutjkRomInfo, nes_spacemutjkRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_spaceraftRomDesc[] = {
	{ "Space Raft (2020)(Raftronaut).nes",          524304, 0x03841119, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_spaceraft)
STD_ROM_FN(nes_spaceraft)

struct BurnDriver BurnDrvnes_spaceraft = {
	"nes_spaceraft", NULL, NULL, NULL, "2020",
	"Space Raft (HB)\0", NULL, "Raftronaut", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_spaceraftRomInfo, nes_spaceraftRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_splatoodRomDesc[] = {
	{ "SplatooD v1.0.6 (2016)(John Carmackerel).nes",          40976, 0x93d17b02, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_splatood)
STD_ROM_FN(nes_splatood)

struct BurnDriver BurnDrvnes_splatood = {
	"nes_splatood", NULL, NULL, NULL, "2016",
	"SplatooD (HB, v1.0.6)\0", NULL, "John Carmackerel", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION | GBF_MAZE, 0,
	NESGetZipName, nes_splatoodRomInfo, nes_splatoodRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_sudoku23RomDesc[] = {
	{ "Sudoku 23 (2023)(Johnybot).nes",          40976, 0x914d683d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sudoku23)
STD_ROM_FN(nes_sudoku23)

struct BurnDriver BurnDrvnes_sudoku23 = {
	"nes_sudoku23", NULL, NULL, NULL, "2023",
	"Sudoku 23 (HB)\0", NULL, "Johnybot", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_sudoku23RomInfo, nes_sudoku23RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// https://www.romhacking.net/hacks/7583/
static struct BurnRomInfo nes_stetrislandRomDesc[] = {
	{ "Super Tetris Land - GB edition (2022)(CAndiman).nes",          163856, 0x4aaca4c2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_stetrisland)
STD_ROM_FN(nes_stetrisland)

struct BurnDriver BurnDrvnes_stetrisland = {
	"nes_stetrisland", "nes_tetris", NULL, NULL, "2022",
	"Super Tetris Land - GB skin (Hack)\0", NULL, "CAndiman", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_stetrislandRomInfo, nes_stetrislandRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_tenebraRomDesc[] = {
	{ "Tenebra (2023)(Haplo).nes",          40976, 0x213d756f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tenebra)
STD_ROM_FN(nes_tenebra)

struct BurnDriver BurnDrvnes_tenebra = {
	"nes_tenebra", NULL, NULL, NULL, "2023",
	"Tenebra (HB)\0", NULL, "Haplo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_MAZE, 0,
	NESGetZipName, nes_tenebraRomInfo, nes_tenebraRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_chickenfarmRomDesc[] = {
	{ "Chicken Of The Farm (2019)(mitch3a and link_7777).nes",          40976, 0x60e65140, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chickenfarm)
STD_ROM_FN(nes_chickenfarm)

struct BurnDriver BurnDrvnes_chickenfarm = {
	"nes_chickenfarm", NULL, NULL, NULL, "2019",
	"Chicken Of The Farm (HB)\0", NULL, "mitch3a, link_7777", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_chickenfarmRomInfo, nes_chickenfarmRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// https://www.romhacking.net/hacks/4723/
static struct BurnRomInfo nes_deadpoolRomDesc[] = {
	{ "Deadpool Globalhack (2019)(Techmoon).nes",          786448, 0x5e36e542, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_deadpool)
STD_ROM_FN(nes_deadpool)

struct BurnDriver BurnDrvnes_deadpool = {
	"nes_deadpool", "nes_ninjagaiden", NULL, NULL, "2019",
	"Deadpool (GlobalHack)\0", NULL, "Techmoon", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW | BDF_HACK, 1, HARDWARE_NES, GBF_SCRFIGHT | GBF_PLATFORM, 0,
	NESGetZipName, nes_deadpoolRomInfo, nes_deadpoolRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_fullquietrvRomDesc[] = {
	{ "Full Quiet (2022)(Retrotainment Games).nes",          524304, 0x45d166dd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_fullquietrv)
STD_ROM_FN(nes_fullquietrv)

struct BurnDriver BurnDrvnes_fullquietrv = {
	"nes_fullquietrv", NULL, NULL, NULL, "2022",
	"Full Quiet (HB)\0", NULL, "Retrotainment Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM | GBF_ADV, 0,
	NESGetZipName, nes_fullquietrvRomInfo, nes_fullquietrvRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_spookysoccerRomDesc[] = {
	{ "Spooky Soccer (Proto)(2022)(Dullahan Software).nes",          40976, 0xfd22b3f3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_spookysoccer)
STD_ROM_FN(nes_spookysoccer)

struct BurnDriver BurnDrvnes_spookysoccer = {
	"nes_spookysoccer", NULL, NULL, NULL, "2022",
	"Spooky Soccer (HB, Prototype)\0", NULL, "Dullahan Software", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW | BDF_PROTOTYPE, 2, HARDWARE_NES, GBF_ACTION | GBF_SPORTSFOOTBALL, 0,
	NESGetZipName, nes_spookysoccerRomInfo, nes_spookysoccerRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_uchusenRomDesc[] = {
	{ "Uchusen - Ultimate PLOID Battle (2021)(Nape Games).nes",          524304, 0x3ec46930, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_uchusen)
STD_ROM_FN(nes_uchusen)

struct BurnDriver BurnDrvnes_uchusen = {
	"nes_uchusen", NULL, NULL, NULL, "2021",
	"Uchusen - Ultimate PLOID Battle (HB)\0", NULL, "NAPE Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_HORSHOOT, 0,
	NESGetZipName, nes_uchusenRomInfo, nes_uchusenRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_uchusen2RomDesc[] = {
	{ "Uchusen 2 - Blood Invasion Battle (2021)(Nape Games).nes",          524304, 0xa9623021, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_uchusen2)
STD_ROM_FN(nes_uchusen2)

struct BurnDriver BurnDrvnes_uchusen2 = {
	"nes_uchusen2", NULL, NULL, NULL, "2021",
	"Uchusen 2 - Blood Invasion Battle (HB)\0", NULL, "NAPE Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_HORSHOOT, 0,
	NESGetZipName, nes_uchusen2RomInfo, nes_uchusen2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_2048RomDesc[] = {
	{ "2048 (2014)(Tsone).nes",          16400, 0x50b7ee6f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_2048)
STD_ROM_FN(nes_2048)

struct BurnDriver BurnDrvnes_2048 = {
	"nes_2048", NULL, NULL, NULL, "2014",
	"2048 (HB)\0", NULL, "Tsone", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_2048RomInfo, nes_2048RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ballstandRomDesc[] = {
	{ "Balloon Stand (2022)(Spazzylemons).nes",          24592, 0x1bb52cb6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ballstand)
STD_ROM_FN(nes_ballstand)

struct BurnDriver BurnDrvnes_ballstand = {
	"nes_ballstand", NULL, NULL, NULL, "2022",
	"Balloon Stand (HB)\0", NULL, "Spazzylemons", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_ballstandRomInfo, nes_ballstandRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ddreamddiverRomDesc[] = {
	{ "Ddream Ddiver (2022)(Game Losers).nes",          524304, 0x26943006, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ddreamddiver)
STD_ROM_FN(nes_ddreamddiver)

struct BurnDriver BurnDrvnes_ddreamddiver = {
	"nes_ddreamddiver", NULL, NULL, NULL, "2022",
	"Ddream Ddiver (HB)\0", NULL, "Game Losers", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_ddreamddiverRomInfo, nes_ddreamddiverRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_karatekickRomDesc[] = {
	{ "Karate Kick (2017)(Mike Moffitt).nes",          40976, 0xda688ee1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_karatekick)
STD_ROM_FN(nes_karatekick)

struct BurnDriver BurnDrvnes_karatekick = {
	"nes_karatekick", NULL, NULL, NULL, "2017",
	"Karate Kick (HB)\0", NULL, "Mike Moffitt", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_karatekickRomInfo, nes_karatekickRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_tctwhRomDesc[] = {
	{ "They Came To Wreak Hell v0.3a (2022)(Team Disposable).nes",          65552, 0x83c4b5a3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tctwh)
STD_ROM_FN(nes_tctwh)

struct BurnDriver BurnDrvnes_tctwh = {
	"nes_tctwh", NULL, NULL, NULL, "2022",
	"They Came To Wreak Hell (HB, v0.3a)\0", NULL, "Team Disposable", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_RUNGUN | GBF_MAZE, 0,
	NESGetZipName, nes_tctwhRomInfo, nes_tctwhRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bobsixsevenRomDesc[] = {
	{ "Bobby Six Seven (Rev2)(2022)(calgames).nes",          524304, 0x81920aa8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bobsixseven)
STD_ROM_FN(nes_bobsixseven)

struct BurnDriver BurnDrvnes_bobsixseven = {
	"nes_bobsixseven", NULL, NULL, NULL, "2022",
	"Bobby Six Seven: Time Travelling Detective (HB, Rev2)\0", NULL, "CalGames", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_bobsixsevenRomInfo, nes_bobsixsevenRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_coinworldRomDesc[] = {
	{ "Coin World (2021)(CMYKSoft).nes",          40976, 0x6775b5d1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_coinworld)
STD_ROM_FN(nes_coinworld)

struct BurnDriver BurnDrvnes_coinworld = {
	"nes_coinworld", NULL, NULL, NULL, "2021",
	"Coin World (HB)\0", NULL, "CMYKSoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_coinworldRomInfo, nes_coinworldRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_perkeleRomDesc[] = {
	{ "PERKELE! Suomi 100 vuotta 5th anniversary edition (2022)(AasiPelit).nes",          393232, 0xbec64871, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_perkele)
STD_ROM_FN(nes_perkele)

struct BurnDriver BurnDrvnes_perkele = {
	"nes_perkele", NULL, NULL, NULL, "2022",
	"PERKELE! Suomi 100 vuotta - 5th anniversary edition (HB)\0", NULL, "AasiPelit", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_perkeleRomInfo, nes_perkeleRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_rapidcyclingRomDesc[] = {
	{ "Rapid Cycling (2022)(Game Losers).nes",          524304, 0x4abb94f0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_rapidcycling)
STD_ROM_FN(nes_rapidcycling)

struct BurnDriver BurnDrvnes_rapidcycling = {
	"nes_rapidcycling", NULL, NULL, NULL, "2022",
	"Rapid Cycling (HB)\0", NULL, "Game Losers", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_rapidcyclingRomInfo, nes_rapidcyclingRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cmbr4wlRomDesc[] = {
	{ "Chibi Monster Br4wl (2022)(dalecoop Studio).nes",          524304, 0x0eaa7806, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cmbr4wl)
STD_ROM_FN(nes_cmbr4wl)

struct BurnDriver BurnDrvnes_cmbr4wl = {
	"nes_cmbr4wl", NULL, NULL, NULL, "2022",
	"Chibi Monster Br4wl (HB)\0", NULL, "dalecoop Studio", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 4, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_cmbr4wlRomInfo, nes_cmbr4wlRomName, NULL, NULL, NULL, NULL, NES4ScoreInputInfo, NES4ScoreDIPInfo,
	NES4ScoreInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_guntnerRomDesc[] = {
	{ "GunTneR (2022)(langel).nes",          24592, 0x170e5f53, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_guntner)
STD_ROM_FN(nes_guntner)

struct BurnDriver BurnDrvnes_guntner = {
	"nes_guntner", NULL, NULL, NULL, "2022",
	"GunTneR (HB)\0", NULL, "langel", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_HORSHOOT, 0,
	NESGetZipName, nes_guntnerRomInfo, nes_guntnerRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_heistRomDesc[] = {
	{ "Heist (2022)(Johnybot).nes",          40976, 0x75c942f5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_heist)
STD_ROM_FN(nes_heist)

struct BurnDriver BurnDrvnes_heist = {
	"nes_heist", NULL, NULL, NULL, "2022",
	"Heist (HB)\0", NULL, "Johnybot", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_SHOOT, 0,
	NESGetZipName, nes_heistRomInfo, nes_heistRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_irritshipRomDesc[] = {
	{ "Irritating Ship (2022)(Fiskbit).nes",          40976, 0xacab29f4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_irritship)
STD_ROM_FN(nes_irritship)

struct BurnDriver BurnDrvnes_irritship = {
	"nes_irritship", NULL, NULL, NULL, "2022",
	"Irritating Ship (HB)\0", NULL, "Fiskbit", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_irritshipRomInfo, nes_irritshipRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_sensecityRomDesc[] = {
	{ "Senseless City (2022)(Rani Baker).nes",          40976, 0x70c92e3b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sensecity)
STD_ROM_FN(nes_sensecity)

struct BurnDriver BurnDrvnes_sensecity = {
	"nes_sensecity", NULL, NULL, NULL, "2022",
	"Senseless City (HB)\0", NULL, "Rani Baker", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_sensecityRomInfo, nes_sensecityRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_tensecondsRomDesc[] = {
	{ "Tense Conds (2022)(Wendel Scardua).nes",          65552, 0x3e385cb0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tenseconds)
STD_ROM_FN(nes_tenseconds)

struct BurnDriver BurnDrvnes_tenseconds = {
	"nes_tenseconds", NULL, NULL, NULL, "2022",
	"Tense Conds (HB)\0", NULL, "Wendel Scardua", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_MAZE | GBF_ACTION, 0,
	NESGetZipName, nes_tensecondsRomInfo, nes_tensecondsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bloblandRomDesc[] = {
	{ "Blob Land (2020)(Wendel Scardua).nes",          40976, 0x89b129ef, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_blobland)
STD_ROM_FN(nes_blobland)

struct BurnDriver BurnDrvnes_blobland = {
	"nes_blobland", NULL, NULL, NULL, "2020",
	"Blob Land (HB)\0", NULL, "Wendel Scardua", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_bloblandRomInfo, nes_bloblandRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_hbcpballRomDesc[] = {
	{ "HBC Phutball (2022)(Wendel Scardua).nes",          65552, 0xb2f98884, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_hbcpball)
STD_ROM_FN(nes_hbcpball)

struct BurnDriver BurnDrvnes_hbcpball = {
	"nes_hbcpball", NULL, NULL, NULL, "2022",
	"HBC Phutball (HB)\0", NULL, "Wendel Scardua", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_hbcpballRomInfo, nes_hbcpballRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_maurimursuRomDesc[] = {
	{ "Mauri Mursu's Hangover (2022)(Nervanti).nes",          524304, 0xa5fac69c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_maurimursu)
STD_ROM_FN(nes_maurimursu)

struct BurnDriver BurnDrvnes_maurimursu = {
	"nes_maurimursu", NULL, NULL, NULL, "2022",
	"Mauri Mursu's Hangover (HB)\0", NULL, "Nervanti", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, nes_maurimursuRomInfo, nes_maurimursuRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_mkmadnessRomDesc[] = {
	{ "Minekart Madness (2022)(Matt Hughson).nes",          65552, 0x165519c7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mkmadness)
STD_ROM_FN(nes_mkmadness)

struct BurnDriver BurnDrvnes_mkmadness = {
	"nes_mkmadness", NULL, NULL, NULL, "2022",
	"Minekart Madness (HB)\0", NULL, "Matt Hughson", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_mkmadnessRomInfo, nes_mkmadnessRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_mkmadnessmmc1RomDesc[] = {
	{ "Minekart Madness mmc1 (2022)(Matt Hughson).nes",          262160, 0x14297e1e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mkmadnessmmc1)
STD_ROM_FN(nes_mkmadnessmmc1)

struct BurnDriver BurnDrvnes_mkmadnessmmc1 = {
	"nes_mkmadnessmmc1", "nes_mkmadness", NULL, NULL, "2022",
	"Minekart Madness (HB, MMC1 ver.)\0", NULL, "Matt Hughson", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_mkmadnessmmc1RomInfo, nes_mkmadnessmmc1RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ravensgateRomDesc[] = {
	{ "Ravens Gate (2022)(MercuryBD).nes",          65552, 0x3911f904, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ravensgate)
STD_ROM_FN(nes_ravensgate)

struct BurnDriver BurnDrvnes_ravensgate = {
	"nes_ravensgate", NULL, NULL, NULL, "2022",
	"Raven's Gate (HB)\0", NULL, "MercuryBD", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, nes_ravensgateRomInfo, nes_ravensgateRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ravensgatemmc1RomDesc[] = {
	{ "Raven's Gate mmc1 (2022)(MercuryBD).nes",          262160, 0xd2389326, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ravensgatemmc1)
STD_ROM_FN(nes_ravensgatemmc1)

struct BurnDriver BurnDrvnes_ravensgatemmc1 = {
	"nes_ravensgatemmc1", "nes_ravensgate", NULL, NULL, "2022",
	"Raven's Gate (HB, MMC1 ver.)\0", NULL, "MercuryBD", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, nes_ravensgatemmc1RomInfo, nes_ravensgatemmc1RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_steelmoonsRomDesc[] = {
	{ "Steel Moons (2022)(Mazionach).nes",          65552, 0x3dc97b51, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_steelmoons)
STD_ROM_FN(nes_steelmoons)

struct BurnDriver BurnDrvnes_steelmoons = {
	"nes_steelmoons", NULL, NULL, NULL, "2022",
	"Steel Moons (HB)\0", NULL, "Mazionach", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_SHOOT, 0,
	NESGetZipName, nes_steelmoonsRomInfo, nes_steelmoonsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_tactusRomDesc[] = {
	{ "Tactus Prototype (2022)(zeta0134).nes",          65552, 0x51876b54, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tactus)
STD_ROM_FN(nes_tactus)

struct BurnDriver BurnDrvnes_tactus = {
	"nes_tactus", NULL, NULL, NULL, "2022",
	"Tactus (HB, Prototype)\0", NULL, "zeta134", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW | BDF_PROTOTYPE, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_tactusRomInfo, nes_tactusRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_garbpailkidsRomDesc[] = {
	{ "Garbage Pail Kids Mad Mike and the Quest for Stale Gum (2022)(iam8bit).nes",          526080, 0x3d110c09, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_garbpailkids)
STD_ROM_FN(nes_garbpailkids)

struct BurnDriver BurnDrvnes_garbpailkids = {
	"nes_garbpailkids", NULL, NULL, NULL, "2022",
	"Garbage Pail Kids Mad Mike and the Quest for Stale Gum (HB)\0", NULL, "iam8bit", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_garbpailkidsRomInfo, nes_garbpailkidsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_mrmayclRomDesc[] = {
	{ "Mr. Maymunshines Christmas Land (2020)(Calgames).nes",          524304, 0x36fe2a47, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mrmaycl)
STD_ROM_FN(nes_mrmaycl)

struct BurnDriver BurnDrvnes_mrmaycl = {
	"nes_mrmaycl", NULL, NULL, NULL, "2020",
	"Mr. Maymunshines Christmas Land (HB)\0", NULL, "Calgames", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_mrmayclRomInfo, nes_mrmayclRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_nightelmstreetdmkRomDesc[] = {
	{ "Nightmare on Elm Street, A - Son of a Hundred Maniacs Demake (1999-2022)(Fista Productions).nes",          524304, 0x7c6b0b6c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_nightelmstreetdmk)
STD_ROM_FN(nes_nightelmstreetdmk)

struct BurnDriver BurnDrvnes_nightelmstreetdmk = {
	"nes_nightelmstreetdmk", NULL, NULL, NULL, "1999-2022",
	"Nightmare on Elm Street, A - Son of a Hundred Maniacs Demake (HB)\0", NULL, "Fista Productions", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, nes_nightelmstreetdmkRomInfo, nes_nightelmstreetdmkRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_orebodybtRomDesc[] = {
	{ "Orebody - Binder's Tale (2022)(John Vanderhoef).nes",          524304, 0x45e91f5a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_orebodybt)
STD_ROM_FN(nes_orebodybt)

struct BurnDriver BurnDrvnes_orebodybt = {
	"nes_orebodybt", NULL, NULL, NULL, "2022",
	"Orebody - Binder's Tale (HB, v1.x)\0", NULL, "John Vanderhoef", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_orebodybtRomInfo, nes_orebodybtRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_orebodysrRomDesc[] = {
	{ "Orebody - Sand Ripples (2022)(Orebody Inc.).nes",          40976, 0xb7917a5f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_orebodysr)
STD_ROM_FN(nes_orebodysr)

struct BurnDriver BurnDrvnes_orebodysr = {
	"nes_orebodysr", NULL, NULL, NULL, "2022",
	"Orebody - Sand Ripples (HB)\0", NULL, "Orebody Inc.", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_orebodysrRomInfo, nes_orebodysrRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_squirchaserRomDesc[] = {
	{ "Squirrel Chaser (2022)(Brad Hammond).nes",          524304, 0x3a48f1bd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_squirchaser)
STD_ROM_FN(nes_squirchaser)

struct BurnDriver BurnDrvnes_squirchaser = {
	"nes_squirchaser", NULL, NULL, NULL, "2022",
	"Squirrel Chaser (HB)\0", NULL, "Brad Hammond", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, nes_squirchaserRomInfo, nes_squirchaserRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_gremlinsRomDesc[] = {
	{ "Gremlins GlobalHack (2016)(pacnsacdave).nes",          262160, 0xf6d5d3e3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_gremlins)
STD_ROM_FN(nes_gremlins)

struct BurnDriver BurnDrvnes_gremlins = {
	"nes_gremlins", "nes_hudsonhawk", NULL, NULL, "2016",
	"Gremlins (GlobalHack)\0", NULL, "pacnsacdave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_gremlinsRomInfo, nes_gremlinsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_leglutlivrRomDesc[] = {
	{ "Leggite Luta Livre Remaster (2022)(John Vanderhoef).nes",          524304, 0x73bf3d14, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_leglutlivr)
STD_ROM_FN(nes_leglutlivr)

struct BurnDriver BurnDrvnes_leglutlivr = {
	"nes_leglutlivr", NULL, NULL, NULL, "2021-22",
	"Leggite Luta Livre Remaster (HB)\0", NULL, "John Vanderhoef", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_leglutlivrRomInfo, nes_leglutlivrRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_megamanxRomDesc[] = {
	{ "Mega Man Age of Wily GlobalHack (2015)(pacnsacdave).nes",          524304, 0x9414c14f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_megamanx)
STD_ROM_FN(nes_megamanx)

struct BurnDriver BurnDrvnes_megamanx = {
	"nes_megamanx", "nes_megaman", NULL, NULL, "2015",
	"Mega Man Age of Wily (GlobalHack)\0", NULL, "pacnsacdave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_megamanxRomInfo, nes_megamanxRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_mogminesRomDesc[] = {
	{ "Mogura Mines (2022)(8Bitalo).nes",          524304, 0xf8d0825d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mogmines)
STD_ROM_FN(nes_mogmines)

struct BurnDriver BurnDrvnes_mogmines = {
	"nes_mogmines", NULL, NULL, NULL, "2022",
	"Mogura Mines (HB)\0", NULL, "8Bitalo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_mogminesRomInfo, nes_mogminesRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NES4ScoreInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_snrRomDesc[] = {
	{ "Swords and Runes (2014)(Sole Goose Productions).nes",          524304, 0x6f596f4b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_snr)
STD_ROM_FN(nes_snr)

struct BurnDriver BurnDrvnes_snr = {
	"nes_snr", NULL, NULL, NULL, "2014",
	"Swords and Runes (HB)\0", NULL, "Sole Goose Productions", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_RPG | GBF_STRATEGY, 0,
	NESGetZipName, nes_snrRomInfo, nes_snrRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_snrleRomDesc[] = {
	{ "Swords and Runes LE (2014)(Sole Goose Productions).nes",          24592, 0xea0b4a06, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_snrle)
STD_ROM_FN(nes_snrle)

struct BurnDriver BurnDrvnes_snrle = {
	"nes_snrle", "nes_snr", NULL, NULL, "2014",
	"Swords and Runes Limited Edition (HB)\0", NULL, "Sole Goose Productions", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_RPG | GBF_STRATEGY, 0,
	NESGetZipName, nes_snrleRomInfo, nes_snrleRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_snr3RomDesc[] = {
	{ "Swords and Runes III (2020)(Sole Goose Productions).nes",          524304, 0x5738826e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_snr3)
STD_ROM_FN(nes_snr3)

struct BurnDriver BurnDrvnes_snr3 = {
	"nes_snr3", NULL, NULL, NULL, "2020",
	"Swords and Runes III (HB)\0", NULL, "Sole Goose Productions", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_RPG | GBF_STRATEGY, 0,
	NESGetZipName, nes_snr3RomInfo, nes_snr3RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_zombhuntbobRomDesc[] = {
	{ "Zombiehunter Bob (2022)(board-b).nes",          524304, 0x00927241, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_zombhuntbob)
STD_ROM_FN(nes_zombhuntbob)

struct BurnDriver BurnDrvnes_zombhuntbob = {
	"nes_zombhuntbob", NULL, NULL, NULL, "2022",
	"Zombiehunter Bob (HB)\0", NULL, "board-b", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_zombhuntbobRomInfo, nes_zombhuntbobRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NES4ScoreInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_micromages2RomDesc[] = {
	{ "Micro Mages - Second Quest (2022)(Morphcat Games).nes",          40976, 0xbbc0ef28, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_micromages2)
STD_ROM_FN(nes_micromages2)

struct BurnDriver BurnDrvnes_micromages2 = {
	"nes_micromages2", NULL, NULL, NULL, "2022",
	"Micro Mages - Second Quest (HB)\0", NULL, "Morphcat Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 4, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_micromages2RomInfo, nes_micromages2RomName, NULL, NULL, NULL, NULL, NES4ScoreInputInfo, NES4ScoreDIPInfo,
	NES4ScoreInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_roniustaleRomDesc[] = {
	{ "Roniu's Tale (2022)(Mega Cat Studios).nes",          262160, 0xd1d75042, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_roniustale)
STD_ROM_FN(nes_roniustale)

struct BurnDriver BurnDrvnes_roniustale = {
	"nes_roniustale", NULL, NULL, NULL, "2022",
	"Roniu's Tale (HB)\0", NULL, "Mega Cat Studios", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_roniustaleRomInfo, nes_roniustaleRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NES4ScoreInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_advpanzer2RomDesc[] = {
	{ "Adventures of Panzer 2, The v1.1 (2022)(PixelCraft).nes",          524304, 0x29750f4f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advpanzer2)
STD_ROM_FN(nes_advpanzer2)

struct BurnDriver BurnDrvnes_advpanzer2 = {
	"nes_advpanzer2", NULL, NULL, NULL, "2022",
	"Adventures of Panzer 2, The (HB, v1.1)\0", NULL, "PixelCraft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM | GBF_ADV, 0,
	NESGetZipName, nes_advpanzer2RomInfo, nes_advpanzer2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_gemventureRomDesc[] = {
	{ "GemVenture (2010)(Tom Livac).nes",          32784, 0x2b8eae32, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_gemventure)
STD_ROM_FN(nes_gemventure)

struct BurnDriver BurnDrvnes_gemventure = {
	"nes_gemventure", NULL, NULL, NULL, "2010",
	"GemVenture (HB)\0", NULL, "Tom Livac", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_gemventureRomInfo, nes_gemventureRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_snakkyRomDesc[] = {
	{ "Snakky (2016)(Shiru).nes",          40976, 0xc1b8427e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_snakky)
STD_ROM_FN(nes_snakky)

struct BurnDriver BurnDrvnes_snakky = {
	"nes_snakky", NULL, NULL, NULL, "2016",
	"Snakky (HB)\0", NULL, "Shiru", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_snakkyRomInfo, nes_snakkyRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_8bitxmas17RomDesc[] = {
	{ "8-bit XMAS 2017 (2017)(RetroUSB).nes",          2097168, 0x330da716, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_8bitxmas17)
STD_ROM_FN(nes_8bitxmas17)

struct BurnDriver BurnDrvnes_8bitxmas17 = {
	"nes_8bitxmas17", NULL, NULL, NULL, "2017",
	"8-bit XMAS 2017 (HB)\0", "8-bit XMAS 2008-2016 Compilation", "retroUSB", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_8bitxmas17RomInfo, nes_8bitxmas17RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_8bitxmas18RomDesc[] = {
	{ "8-bit XMAS 2018 (2018)(RetroUSB).nes",          131088, 0x9c8d171e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_8bitxmas18)
STD_ROM_FN(nes_8bitxmas18)

struct BurnDriver BurnDrvnes_8bitxmas18 = {
	"nes_8bitxmas18", NULL, NULL, NULL, "2018",
	"8-bit XMAS 2018 (HB)\0", NULL, "retroUSB", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_8bitxmas18RomInfo, nes_8bitxmas18RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_8bitxmas19RomDesc[] = {
	{ "8-bit XMAS 2019 (2019)(RetroUSB).nes",          131088, 0x6a33f365, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_8bitxmas19)
STD_ROM_FN(nes_8bitxmas19)

struct BurnDriver BurnDrvnes_8bitxmas19 = {
	"nes_8bitxmas19", NULL, NULL, NULL, "2019",
	"8-bit XMAS 2019 (HB)\0", NULL, "retroUSB", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_8bitxmas19RomInfo, nes_8bitxmas19RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_8bitxmas20RomDesc[] = {
	{ "8-bit XMAS 2020 (2020)(RetroUSB).nes",          131088, 0x4dacc860, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_8bitxmas20)
STD_ROM_FN(nes_8bitxmas20)

struct BurnDriver BurnDrvnes_8bitxmas20 = {
	"nes_8bitxmas20", NULL, NULL, NULL, "2020",
	"8-bit XMAS 2020 (HB)\0", NULL, "retroUSB", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_8bitxmas20RomInfo, nes_8bitxmas20RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_8bitxmas21RomDesc[] = {
	{ "8-bit XMAS 2021 (2021)(RetroUSB).nes",          131088, 0x94c3e8a3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_8bitxmas21)
STD_ROM_FN(nes_8bitxmas21)

struct BurnDriver BurnDrvnes_8bitxmas21 = {
	"nes_8bitxmas21", NULL, NULL, NULL, "2021",
	"8-bit XMAS 2021 (HB)\0", NULL, "retroUSB", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_8bitxmas21RomInfo, nes_8bitxmas21RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_8bitxmas22RomDesc[] = {
	{ "8-bit XMAS 2022 (2022)(RetroUSB).nes",          131088, 0x81bbc4a5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_8bitxmas22)
STD_ROM_FN(nes_8bitxmas22)

struct BurnDriver BurnDrvnes_8bitxmas22 = {
	"nes_8bitxmas22", NULL, NULL, NULL, "2022",
	"8-bit XMAS 2022 (HB)\0", NULL, "retroUSB", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_8bitxmas22RomInfo, nes_8bitxmas22RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dethcomplex2RomDesc[] = {
	{ "Deth Complex 2 (2022)(T-bone).nes",          524304, 0xaf32771c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dethcomplex2)
STD_ROM_FN(nes_dethcomplex2)

struct BurnDriver BurnDrvnes_dethcomplex2 = {
	"nes_dethcomplex2", NULL, NULL, NULL, "2022",
	"Deth Complex 2 (HB)\0", NULL, "T-Bone Studios", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_dethcomplex2RomInfo, nes_dethcomplex2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_gravedudeRomDesc[] = {
	{ "Graveyard Dude (2022)(Rani Baker).nes",          40976, 0xb0c5ff55, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_gravedude)
STD_ROM_FN(nes_gravedude)

struct BurnDriver BurnDrvnes_gravedude = {
	"nes_gravedude", NULL, NULL, NULL, "2022",
	"Graveyard Dude (HB)\0", NULL, "Rani Baker", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_gravedudeRomInfo, nes_gravedudeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_orpheaRomDesc[] = {
	{ "Orphea v1.1 (2021)(SkyBoy Games).nes",          40976, 0x7d8a4906, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_orphea)
STD_ROM_FN(nes_orphea)

struct BurnDriver BurnDrvnes_orphea = {
	"nes_orphea", NULL, NULL, NULL, "2021",
	"Orphea (HB, v1.1)\0", NULL, "SkyBoy Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_HORSHOOT, 0,
	NESGetZipName, nes_orpheaRomInfo, nes_orpheaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_fullquietRomDesc[] = {
	{ "Full Quiet (Test Cart)(2021)(Retrotainment Games).nes",          524304, 0x03055116, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_fullquiet)
STD_ROM_FN(nes_fullquiet)

struct BurnDriver BurnDrvnes_fullquiet = {
	"nes_fullquiet", "nes_fullquietrv", NULL, NULL, "2021",
	"Full Quiet (Test Cart) (HB, Prototype)\0", NULL, "Retrotainment Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW | BDF_PROTOTYPE, 1, HARDWARE_NES, GBF_PLATFORM | GBF_ADV, 0,
	NESGetZipName, nes_fullquietRomInfo, nes_fullquietRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_monterevRomDesc[] = {
	{ "Montezuma's Revenge (2022)(2nd Dimension).nes",          40976, 0x02cfc8a7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_monterev)
STD_ROM_FN(nes_monterev)

struct BurnDriver BurnDrvnes_monterev = {
	"nes_monterev", NULL, NULL, NULL, "2022",
	"Montezuma's Revenge (HB)\0", NULL, "2nd Dimension", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_monterevRomInfo, nes_monterevRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_oopiquestRomDesc[] = {
	{ "Oopi's Quest v1.2.0 (2022)(Kenney).nes",          40976, 0x34de6da0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_oopiquest)
STD_ROM_FN(nes_oopiquest)

struct BurnDriver BurnDrvnes_oopiquest = {
	"nes_oopiquest", NULL, NULL, NULL, "2022",
	"Oopi's Quest (HB, v1.2.0)\0", NULL, "Kenney", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_oopiquestRomInfo, nes_oopiquestRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_rickrolledRomDesc[] = {
	{ "Rick Rolled! Tech-Demo (2009) (Frantik).nes",          262160, 0xc1be91f9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_rickrolled)
STD_ROM_FN(nes_rickrolled)

struct BurnDriver BurnDrvnes_rickrolled = {
	"nes_rickrolled", NULL, NULL, NULL, "2009",
	"Rick Rolled! (HB, Tech-Demo)\0", NULL, "Frantik", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW | BDF_DEMO, 1, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_rickrolledRomInfo, nes_rickrolledRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dethcomplexRomDesc[] = {
	{ "Deth Complex (2022)(T-bone).nes",          524304, 0x7b213380, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dethcomplex)
STD_ROM_FN(nes_dethcomplex)

struct BurnDriver BurnDrvnes_dethcomplex = {
	"nes_dethcomplex", NULL, NULL, NULL, "2022",
	"Deth Complex! (HB)\0", NULL, "T-Bone Studios", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_dethcomplexRomInfo, nes_dethcomplexRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_picopicoRomDesc[] = {
	{ "Pico Pico Grand Adventure (J)(2019)(Broke Studio).nes",          524304, 0x3950140d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_picopico)
STD_ROM_FN(nes_picopico)

struct BurnDriver BurnDrvnes_picopico = {
	"nes_picopico", "nes_bassedefadv", NULL, NULL, "2019-20",
	"Pico Pico Grand Adventure (HB)\0", NULL, "Columbus Circle", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_picopicoRomInfo, nes_picopicoRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_quadralordsRomDesc[] = {
	{ "Quadralords (2016)(RetroUSB).nes",          524304, 0x88b407f7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_quadralords)
STD_ROM_FN(nes_quadralords)

struct BurnDriver BurnDrvnes_quadralords = {
	"nes_quadralords", NULL, NULL, NULL, "2016",
	"Quadralords (HB)\0", NULL, "retroUSB", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 4, HARDWARE_NES, GBF_ACTION | GBF_BREAKOUT, 0,
	NESGetZipName, nes_quadralordsRomInfo, nes_quadralordsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_snesnake2RomDesc[] = {
	{ "Super NeSnake 2 (2008)(RetroUSB).nes",          131088, 0xa8d9d559, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_snesnake2)
STD_ROM_FN(nes_snesnake2)

struct BurnDriver BurnDrvnes_snesnake2 = {
	"nes_snesnake2", NULL, NULL, NULL, "2008",
	"Super NeSnake 2 (HB)\0", NULL, "retroUSB", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_snesnake2RomInfo, nes_snesnake2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_armwrestlRomDesc[] = {
	{ "Arm Wrestling Classic, The (2022)(Fista Games).nes",          524304, 0x5c62c55a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_armwrestl)
STD_ROM_FN(nes_armwrestl)

struct BurnDriver BurnDrvnes_armwrestl = {
	"nes_armwrestl", NULL, NULL, NULL, "2022",
	"Arm Wrestling Classic, The (HB)\0", NULL, "Fista Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_armwrestlRomInfo, nes_armwrestlRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_drgarfieldRomDesc[] = {
	{ "Dr. Garfield hack v22 (2022)(Pangent Technologies).nes",          114704, 0xfdd0a135, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_drgarfield)
STD_ROM_FN(nes_drgarfield)

struct BurnDriver BurnDrvnes_drgarfield = {
	"nes_drgarfield", "nes_drmario", NULL, NULL, "2022",
	"Dr. Garfield (Hack, v22)\0", NULL, "Pangent Technologies", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_drgarfieldRomInfo, nes_drgarfieldRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_halloween86roeRomDesc[] = {
	{ "Haunted Halloween '86 - Remastered Onslaught Edition (2018)(Retrotainment Games).nes",          524304, 0x94f92f7c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_halloween86roe)
STD_ROM_FN(nes_halloween86roe)

struct BurnDriver BurnDrvnes_halloween86roe = {
	"nes_halloween86roe", NULL, NULL, NULL, "2018",
	"Haunted Halloween'86 - Remastered Onslaught Edition (HB)\0", NULL, "Retrotainment Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_halloween86roeRomInfo, nes_halloween86roeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_mawthorneRomDesc[] = {
	{ "Mawthorne (USA, Europe)(2022)(CGT Games).nes",          524304, 0x447bec5d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mawthorne)
STD_ROM_FN(nes_mawthorne)

struct BurnDriver BurnDrvnes_mawthorne = {
	"nes_mawthorne", NULL, NULL, NULL, "2022",
	"Mawthorne (USA) (HB)\0", NULL, "CGT Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_mawthorneRomInfo, nes_mawthorneRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_mawthornejRomDesc[] = {
	{ "Mawthorne (Japan)(2022)(CGT Games).nes",          524304, 0x471ddcbf, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mawthornej)
STD_ROM_FN(nes_mawthornej)

struct BurnDriver BurnDrvnes_mawthornej = {
	"nes_mawthornej", "nes_mawthorne", NULL, NULL, "2022",
	"Mawthorne (Japan) (HB)\0", NULL, "CGT Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_mawthornejRomInfo, nes_mawthornejRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_tolaRomDesc[] = {
	{ "Tola (2022)(OmniRetro Games).nes",          40976, 0x315309f6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tola)
STD_ROM_FN(nes_tola)

struct BurnDriver BurnDrvnes_tola = {
	"nes_tola", NULL, NULL, NULL, "2022",
	"Tola (HB)\0", NULL, "OmniRetro Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_tolaRomInfo, nes_tolaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_tortoisesRomDesc[] = {
	{ "Tortoises (2017)(FG Software).nes",          65552, 0x05f4f4e2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tortoises)
STD_ROM_FN(nes_tortoises)

struct BurnDriver BurnDrvnes_tortoises = {
	"nes_tortoises", NULL, NULL, NULL, "2017",
	"Tortoises (HB)\0", NULL, "FG Software", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION | GBF_MAZE, 0,
	NESGetZipName, nes_tortoisesRomInfo, nes_tortoisesRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_awakeningRomDesc[] = {
	{ "Alwa's Awakening (2021)(Elden Pixels).nes",          524304, 0x5cc1abc7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_awakening)
STD_ROM_FN(nes_awakening)

struct BurnDriver BurnDrvnes_awakening = {
	"nes_awakening", NULL, NULL, NULL, "2021",
	"Alwa's Awakening (HB)\0", NULL, "Elden Pixels", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_awakeningRomInfo, nes_awakeningRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_nemulesisRomDesc[] = {
	{ "neMULEsis v2.25.23 (2023)(John Vanderhoef).nes",          524304, 0x9982f1b7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_nemulesis)
STD_ROM_FN(nes_nemulesis)

struct BurnDriver BurnDrvnes_nemulesis = {
	"nes_nemulesis", NULL, NULL, NULL, "2021-23",
	"neMULEsis (HB, v2.25.23)\0", NULL, "John Vanderhoef", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_nemulesisRomInfo, nes_nemulesisRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_pixelpoopsRomDesc[] = {
	{ "Pixel Poops Number Two (2021)(Techmaster Games).nes",          524304, 0x17812952, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_pixelpoops)
STD_ROM_FN(nes_pixelpoops)

struct BurnDriver BurnDrvnes_pixelpoops = {
	"nes_pixelpoops", NULL, NULL, NULL, "2021",
	"Pixel Poops Number Two (HB)\0", NULL, "Techmaster Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_pixelpoopsRomInfo, nes_pixelpoopsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_beyopinsRomDesc[] = {
	{ "Beyond the Pins (2021)(KHAN Games).nes",          524304, 0xd5a3f7a0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_beyopins)
STD_ROM_FN(nes_beyopins)

struct BurnDriver BurnDrvnes_beyopins = {
	"nes_beyopins", NULL, NULL, NULL, "2021",
	"Beyond the Pins (HB)\0", NULL, "Khan Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_SPORTSMISC | GBF_ADV, 0,
	NESGetZipName, nes_beyopinsRomInfo, nes_beyopinsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_germsquashRomDesc[] = {
	{ "Germ Squashers (2015)(8bit Evolution).nes",          65552, 0xcd6349b4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_germsquash)
STD_ROM_FN(nes_germsquash)

struct BurnDriver BurnDrvnes_germsquash = {
	"nes_germsquash", NULL, NULL, NULL, "2015",
	"Germ Squashers (HB)\0", NULL, "8bit Evolution", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_germsquashRomInfo, nes_germsquashRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_madwizsdsRomDesc[] = {
	{ "The Mad Wizard - a Candelabra Chronicle (2014)(Sly Dog Studios).nes",          131088, 0x5481d022, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_madwizsds)
STD_ROM_FN(nes_madwizsds)

struct BurnDriver BurnDrvnes_madwizsds = {
	"nes_madwizsds", NULL, NULL, NULL, "2014",
	"Mad Wizard, The - A Candelabra Chronicle (HB) (Sly Dog Studios)\0", NULL, "Sly Dog Studios", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_madwizsdsRomInfo, nes_madwizsdsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_moregliderRomDesc[] = {
	{ "More Glider (2009)(Retro USB).nes",          524304, 0x4ed588ce, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_moreglider)
STD_ROM_FN(nes_moreglider)

struct BurnDriver BurnDrvnes_moreglider = {
	"nes_moreglider", NULL, NULL, NULL, "2009",
	"More Glider (HB)\0", NULL, "Retro USB", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_moregliderRomInfo, nes_moregliderRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ramondusRomDesc[] = {
	{ "Rise of Amondus, The (2015)(Sly Dog Studios).nes",          131088, 0xc609eb0b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ramondus)
STD_ROM_FN(nes_ramondus)

struct BurnDriver BurnDrvnes_ramondus = {
	"nes_ramondus", NULL, NULL, NULL, "2015",
	"Rise of Amondus, The (HB) (Sly Dog Studios)\0", NULL, "Sly Dog Studios", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_HORSHOOT, 0,
	NESGetZipName, nes_ramondusRomInfo, nes_ramondusRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_beeeRomDesc[] = {
	{ "Beat 'Em & Eat 'Em v1.0 (2014)(FG Software).nes",          40976, 0x11175864, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_beee)
STD_ROM_FN(nes_beee)

struct BurnDriver BurnDrvnes_beee = {
	"nes_beee", NULL, NULL, NULL, "2014",
	"Beat 'Em & Eat 'Em (HB)\0", "Adult game", "FG Software", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_beeeRomInfo, nes_beeeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_stingRomDesc[] = {
	{ "STING (2008)(smkdan).nes",          40976, 0xf0c8191b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sting)
STD_ROM_FN(nes_sting)

struct BurnDriver BurnDrvnes_sting = {
	"nes_sting", NULL, NULL, NULL, "2008",
	"STING (HB)\0", NULL, "smkdan", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_SHOOT, 0,
	NESGetZipName, nes_stingRomInfo, nes_stingRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_tdilemmaRomDesc[] = {
	{ "Temple Dilemma (2019)(Oldschool Interactive).nes",          524304, 0x5e2a4eeb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tdilemma)
STD_ROM_FN(nes_tdilemma)

struct BurnDriver BurnDrvnes_tdilemma = {
	"nes_tdilemma", NULL, NULL, NULL, "2019",
	"Temple Dilemma (HB)\0", NULL, "Oldschool Interactive", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_tdilemmaRomInfo, nes_tdilemmaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_pinkyRomDesc[] = {
	{ "Pinky (2020)(Axbakk Games).nes",          524304, 0xcb0bcb5d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_pinky)
STD_ROM_FN(nes_pinky)

struct BurnDriver BurnDrvnes_pinky = {
	"nes_pinky", NULL, NULL, NULL, "2020",
	"Pinky (HB)\0", NULL, "Axbakk Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_pinkyRomInfo, nes_pinkyRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_pinkyxmasRomDesc[] = {
	{ "Pinky Xmas (2020)(Axbakk Games).nes",          524304, 0xc1d80a1d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_pinkyxmas)
STD_ROM_FN(nes_pinkyxmas)

struct BurnDriver BurnDrvnes_pinkyxmas = {
	"nes_pinkyxmas", NULL, NULL, NULL, "2020",
	"Pinky Xmas (HB)\0", NULL, "Axbakk Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_pinkyxmasRomInfo, nes_pinkyxmasRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ralph4RomDesc[] = {
	{ "Ralph 4 (2016)(Pubby).nes",          24592, 0xaf85d620, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ralph4)
STD_ROM_FN(nes_ralph4)

struct BurnDriver BurnDrvnes_ralph4 = {
	"nes_ralph4", NULL, NULL, NULL, "2016",
	"Ralph 4 (HB)\0", NULL, "Pubby", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_ralph4RomInfo, nes_ralph4RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_trouble2a03RomDesc[] = {
	{ "Trouble in 2a03 (2021)(Team Disposable).nes",          65552, 0x5559e594, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_trouble2a03)
STD_ROM_FN(nes_trouble2a03)

struct BurnDriver BurnDrvnes_trouble2a03 = {
	"nes_trouble2a03", NULL, NULL, NULL, "2021",
	"Trouble in 2a03 (HB)\0", NULL, "Team Disposable", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_RUNGUN | GBF_MAZE, 0,
	NESGetZipName, nes_trouble2a03RomInfo, nes_trouble2a03RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_waddlesRomDesc[] = {
	{ "Waddles the Duck (2017)(CP Programs).nes",          65552, 0xe6b45c0e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_waddles)
STD_ROM_FN(nes_waddles)

struct BurnDriver BurnDrvnes_waddles = {
	"nes_waddles", NULL, NULL, NULL, "2017",
	"Waddles the Duck (HB)\0", NULL, "CP Programs", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_waddlesRomInfo, nes_waddlesRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_animclipRomDesc[] = {
	{ "Animal Clipper (2021)(Okunyon).nes",          40976, 0x47d22165, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_animclip)
STD_ROM_FN(nes_animclip)

struct BurnDriver BurnDrvnes_animclip = {
	"nes_animclip", NULL, NULL, NULL, "2021",
	"Animal Clipper (HB)\0", "If the board completely empties the music crashes", "Okunyon", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_animclipRomInfo, nes_animclipRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_famdeponRomDesc[] = {
	{ "Famicom De Pon (2020)(Okunyon).nes",          40976, 0xe2f88d81, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_famdepon)
STD_ROM_FN(nes_famdepon)

struct BurnDriver BurnDrvnes_famdepon = {
	"nes_famdepon", NULL, NULL, NULL, "2020",
	"Famicom De Pon (HB)\0", NULL, "Okunyon", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_famdeponRomInfo, nes_famdeponRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_flyawayRomDesc[] = {
	{ "Fly a Way (2020)(Okunyon).nes",          40976, 0x81e30194, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_flyaway)
STD_ROM_FN(nes_flyaway)

struct BurnDriver BurnDrvnes_flyaway = {
	"nes_flyaway", NULL, NULL, NULL, "2020",
	"Fly a Way (HB)\0", NULL, "Okunyon", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_flyawayRomInfo, nes_flyawayRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_biocatsRomDesc[] = {
	{ "Biocats (2022)(PB Games).nes",          524304, 0x9fd3fcfe, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_biocats)
STD_ROM_FN(nes_biocats)

struct BurnDriver BurnDrvnes_biocats = {
	"nes_biocats", NULL, NULL, NULL, "2022",
	"Biocats (HB)\0", NULL, "PB Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_biocatsRomInfo, nes_biocatsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_inttransRomDesc[] = {
	{ "Interglactic Transmissing (2018)(GGJ).nes",          40976, 0xe13adc0c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_inttrans)
STD_ROM_FN(nes_inttrans)

struct BurnDriver BurnDrvnes_inttrans = {
	"nes_inttrans", NULL, NULL, NULL, "2018",
	"Interglactic Transmissing (HB)\0", NULL, "GGJ", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_inttransRomInfo, nes_inttransRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_lanmasterRomDesc[] = {
	{ "Lan Master (2011)(Shiru).nes",          40976, 0x32d113b5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_lanmaster)
STD_ROM_FN(nes_lanmaster)

struct BurnDriver BurnDrvnes_lanmaster = {
	"nes_lanmaster", NULL, NULL, NULL, "2011",
	"Lan Master (HB)\0", NULL, "Shiru", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_lanmasterRomInfo, nes_lanmasterRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_lightsoutRomDesc[] = {
	{ "Lights Out (2017)(Mika Haarahiltunen).nes",          24592, 0x3d4ab74b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_lightsout)
STD_ROM_FN(nes_lightsout)

struct BurnDriver BurnDrvnes_lightsout = {
	"nes_lightsout", NULL, NULL, NULL, "2017",
	"Lights Out (HB)\0", NULL, "Mika Haarahiltunen", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_lightsoutRomInfo, nes_lightsoutRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_hangmanRomDesc[] = {
	{ "NES Hangman (2015)(FG Software).nes",          40976, 0xb0500cd4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_hangman)
STD_ROM_FN(nes_hangman)

struct BurnDriver BurnDrvnes_hangman = {
	"nes_hangman", NULL, NULL, NULL, "2015",
	"NES Hangman (HB)\0", NULL, "FG Software", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_hangmanRomInfo, nes_hangmanRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_roshamboRomDesc[] = {
	{ "Rock Paper Scissors Lizard Sbock v0.9b (2017)(Doug Fraker).nes",          24592, 0x8c17a094, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_roshambo)
STD_ROM_FN(nes_roshambo)

struct BurnDriver BurnDrvnes_roshambo = {
	"nes_roshambo", NULL, NULL, NULL, "2017",
	"Rock Paper Scissors Lizard Sbock (HB, v0.9b)\0", NULL, "Doug Fraker", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_roshamboRomInfo, nes_roshamboRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_wordleRomDesc[] = {
	{ "Wordle v1.1 (2022)(FG Software).nes",          65552, 0x604ff03b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_wordle)
STD_ROM_FN(nes_wordle)

struct BurnDriver BurnDrvnes_wordle = {
	"nes_wordle", NULL, NULL, NULL, "2022",
	"Wordle (HB, v1.1)\0", NULL, "FG Software", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_wordleRomInfo, nes_wordleRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_almikaRomDesc[] = {
	{ "Almika The Star Rider Densetsu Gaiden (2021)(Mouse Spirit).nes",          524304, 0x3b5a7745, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_almika)
STD_ROM_FN(nes_almika)

struct BurnDriver BurnDrvnes_almika = {
	"nes_almika", NULL, NULL, NULL, "2021",
	"Almika The Star Rider Densetsu Gaiden (HB)\0", NULL, "Mouse Spirit", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_almikaRomInfo, nes_almikaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bandanaRomDesc[] = {
	{ "Bandana (2021)(Rodnade).nes",          524304, 0x292de1a8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bandana)
STD_ROM_FN(nes_bandana)

struct BurnDriver BurnDrvnes_bandana = {
	"nes_bandana", NULL, NULL, NULL, "2021",
	"Bandana (HB)\0", NULL, "Rodnade", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_bandanaRomInfo, nes_bandanaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cavedrillRomDesc[] = {
	{ "Cave Driller v2.2 (2021)(Mazin).nes",          40976, 0xf11b4a70, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cavedrill)
STD_ROM_FN(nes_cavedrill)

struct BurnDriver BurnDrvnes_cavedrill = {
	"nes_cavedrill", NULL, NULL, NULL, "2021",
	"Cave Driller (HB, v2.2)\0", NULL, "Mazin", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_cavedrillRomInfo, nes_cavedrillRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dangbridgeRomDesc[] = {
	{ "Danger Bridge (200X)(Power Joy).nes",          49168, 0x5e59a8b1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dangbridge)
STD_ROM_FN(nes_dangbridge)

struct BurnDriver BurnDrvnes_dangbridge = {
	"nes_dangbridge", NULL, NULL, NULL, "200?",
	"Danger Bridge (HB)\0", NULL, "Power Joy", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_dangbridgeRomInfo, nes_dangbridgeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ninninRomDesc[] = {
	{ "Nin-Nin v1 (2020)(The Mojon Twins).nes",          65552, 0xd46b4955, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ninnin)
STD_ROM_FN(nes_ninnin)

struct BurnDriver BurnDrvnes_ninnin = {
	"nes_ninnin", NULL, NULL, NULL, "2020",
	"Nin-Nin (HB, v1)\0", NULL, "The Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_ninninRomInfo, nes_ninninRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_pandaworldRomDesc[] = {
	{ "Panda World (Unl)(20XX).nes",          262160, 0xdff2fe93, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_pandaworld)
STD_ROM_FN(nes_pandaworld)

struct BurnDriver BurnDrvnes_pandaworld = {
	"nes_pandaworld", NULL, NULL, NULL, "20??",
	"Panda World (HB)\0", NULL, "Unknown", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_pandaworldRomInfo, nes_pandaworldRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_td1990RomDesc[] = {
	{ "Tower Defense 1990 (2018)(Ryan Lloyd).nes",          131088, 0x403af207, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_td1990)
STD_ROM_FN(nes_td1990)

struct BurnDriver BurnDrvnes_td1990 = {
	"nes_td1990", NULL, NULL, NULL, "2018",
	"Tower Defense 1990 (HB)\0", NULL, "Ryan Lloyd", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_STRATEGY, 0,
	NESGetZipName, nes_td1990RomInfo, nes_td1990RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_enormadvRomDesc[] = {
	{ "Enormous Adventures (2021)(Game Losers).nes",          524304, 0xfc0a42ee, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_enormadv)
STD_ROM_FN(nes_enormadv)

struct BurnDriver BurnDrvnes_enormadv = {
	"nes_enormadv", NULL, NULL, NULL, "2021",
	"Enormous Adventures (HB)\0", NULL, "Game Losers", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_enormadvRomInfo, nes_enormadvRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Yeah Yeah Beebiss II (HB)
// https://store.steampowered.com/app/1871700/Yeah_Yeah_Beebiss_II/
static struct BurnRomInfo nes_yybeebissiiRomDesc[] = {
	{ "Yeah Yeah Beebiss II (2022).nes",          65552, 0x4a8eed90, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_yybeebissii)
STD_ROM_FN(nes_yybeebissii)

struct BurnDriver BurnDrvnesnes_yybeebissii = {
	"nes_yybeebissii", NULL, NULL, NULL, "2022",
	"Yeah Yeah Beebiss II (HB)\0", NULL, "Rigg'd Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM | GBF_ACTION, 0,
	NESGetZipName, nes_yybeebissiiRomInfo, nes_yybeebissiiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_heohdemoRomDesc[] = {
	{ "HEOH demo (2019)(shiru).nes",          524304, 0x4fcd5116, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_heohdemo)
STD_ROM_FN(nes_heohdemo)

struct BurnDriver BurnDrvnes_heohdemo = {
	"nes_heohdemo", NULL, NULL, NULL, "2019",
	"HEOH demo (HB, Tech-Demo)\0", NULL, "shiru", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW | BDF_DEMO, 1, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_heohdemoRomInfo, nes_heohdemoRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_animalblkRomDesc[] = {
	{ "Animal Blockes (200X)(Unknown).nes",          49168, 0xfc2b860c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_animalblk)
STD_ROM_FN(nes_animalblk)

struct BurnDriver BurnDrvnes_animalblk = {
	"nes_animalblk", NULL, NULL, NULL, "200?",
	"Animal Blockes (HB)\0", NULL, "Unknown", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_animalblkRomInfo, nes_animalblkRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_archerRomDesc[] = {
	{ "Archer, The (200X)(Power Joy).nes",          24592, 0x911e0365, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_archer)
STD_ROM_FN(nes_archer)

struct BurnDriver BurnDrvnes_archer = {
	"nes_archer", NULL, NULL, NULL, "200?",
	"Archer, The (HB)\0", NULL, "Power Joy", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_archerRomInfo, nes_archerRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_awakeningpRomDesc[] = {
	{ "Alwa's Awakening - Prototype (2021)(Elden Pixels).nes",          524304, 0xc07533c0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_awakeningp)
STD_ROM_FN(nes_awakeningp)

struct BurnDriver BurnDrvnes_awakeningp = {
	"nes_awakeningp", "nes_awakening", NULL, NULL, "2021",
	"Alwa's Awakening (HB, Prototype)\0", NULL, "Elden Pixels", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW | BDF_PROTOTYPE, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_awakeningpRomInfo, nes_awakeningpRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_rainbriteRomDesc[] = {
	{ "Rainbow Brite - Journey to Rainbow Land (2017)(Hallmark).nes",          524304, 0x8902cf08, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_rainbrite)
STD_ROM_FN(nes_rainbrite)

struct BurnDriver BurnDrvnes_rainbrite = {
	"nes_rainbrite", NULL, NULL, NULL, "2017",
	"Rainbow Brite - Journey to Rainbow Land (HB)\0", NULL, "Hallmark", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, nes_rainbriteRomInfo, nes_rainbriteRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_tombiceRomDesc[] = {
	{ "Tomb of Ice (2020)(HB)(FG Software).nes",          40976, 0x426d3439, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tombice)
STD_ROM_FN(nes_tombice)

struct BurnDriver BurnDrvnes_tombice = {
	"nes_tombice", NULL, NULL, NULL, "2020",
	"Tomb of Ice (HB)\0", "No sound", "FG Software", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_tombiceRomInfo, nes_tombiceRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vanguardRomDesc[] = {
	{ "Vanguard (20XX)(Power Joy).nes",          163856, 0xb6c0c49e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vanguard)
STD_ROM_FN(nes_vanguard)

struct BurnDriver BurnDrvnes_vanguard = {
	"nes_vanguard", NULL, NULL, NULL, "200?",
	"Vanguard (HB)\0", NULL, "Power Joy", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_vanguardRomInfo, nes_vanguardRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dveelngRomDesc[] = {
	{ "D'Veel'Ng (2015)(The Mojon Twins).nes",          40976, 0x7a810742, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dveelng)
STD_ROM_FN(nes_dveelng)

struct BurnDriver BurnDrvnes_dveelng = {
	"nes_dveelng", NULL, NULL, NULL, "2015",
	"D'Veel'Ng (HB)\0", NULL, "The Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_dveelngRomInfo, nes_dveelngRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_lunarlimitswRomDesc[] = {
	{ "Lunar Limit - Star Wars Edition (2021)(Pubby).nes",          40976, 0xc130d230, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_lunarlimitsw)
STD_ROM_FN(nes_lunarlimitsw)

struct BurnDriver BurnDrvnes_lunarlimitsw = {
	"nes_lunarlimitsw", "nes_lunarlimit", NULL, NULL, "2021",
	"Lunar Limit - Star Wars Edition (HB)\0", NULL, "Pubby", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_SHOOT, 0,
	NESGetZipName, nes_lunarlimitswRomInfo, nes_lunarlimitswRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_moonglowRomDesc[] = {
	{ "Moonglow (2021)(Yurt Fiend).nes",          40976, 0xb4255e99, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_moonglow)
STD_ROM_FN(nes_moonglow)

struct BurnDriver BurnDrvnes_moonglow = {
	"nes_moonglow", NULL, NULL, NULL, "2021",
	"Moonglow (HB)\0", NULL, "Yurt Fiend", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_moonglowRomInfo, nes_moonglowRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT_PAL+8, SCREEN_WIDTH, SCREEN_HEIGHT_PAL+8
};

static struct BurnRomInfo nes_deadtombRomDesc[] = {
	{ "Dead Tomb (2020)(CollectorVision).nes",          262160, 0x0ab3480b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_deadtomb)
STD_ROM_FN(nes_deadtomb)

struct BurnDriver BurnDrvnes_deadtomb = {
	"nes_deadtomb", NULL, NULL, NULL, "2020",
	"Dead Tomb (HB)\0", NULL, "CollectorVision Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_deadtombRomInfo, nes_deadtombRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_enigmacoreRomDesc[] = {
	{ "Enigmacore (2015)(J. Kane & K. Hanley).nes",          40976, 0xcf680195, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_enigmacore)
STD_ROM_FN(nes_enigmacore)

struct BurnDriver BurnDrvnes_enigmacore = {
	"nes_enigmacore", NULL, NULL, NULL, "2015",
	"Enigmacore (HB)\0", NULL, "J. Kane & K. Hanley", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_enigmacoreRomInfo, nes_enigmacoreRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_weedstiff2RomDesc[] = {
	{ "Weed N' Stiff 2 Bobsons Revenge (2021)(CalGames).nes",          524304, 0x8fd50f91, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_weedstiff2)
STD_ROM_FN(nes_weedstiff2)

struct BurnDriver BurnDrvnes_weedstiff2 = {
	"nes_weedstiff2", NULL, NULL, NULL, "2021",
	"Weed N' Stiff 2 Bobsons Revenge (HB)\0", NULL, "CalGames", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_SCRFIGHT | GBF_ADV, 0,
	NESGetZipName, nes_weedstiff2RomInfo, nes_weedstiff2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_superrusrouRomDesc[] = {
	{ "Super Russian Roulette (HB).nes",          8912912, 0xdde4cf55, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_superrusrou)
STD_ROM_FN(nes_superrusrou)

struct BurnDriver BurnDrvnes_superrusrou = {
	"nes_srroulette", NULL, NULL, NULL, "2017",
	"Super Russian Roulette (HB)\0", NULL, "Andrew Reitano", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 3, HARDWARE_NES, GBF_MISC | GBF_SHOOT, 0,
	NESGetZipName, nes_superrusrouRomInfo, nes_superrusrouRomName, NULL, NULL, NULL, NULL, NESZapperInputInfo, NESZapperDIPInfo,
	NESZapperInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_armedbatleRomDesc[] = {
	{ "Armed for Battle - Limited Edition (2014) (1010 Howe).nes",          131088, 0x542bbace, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_armedbatle)
STD_ROM_FN(nes_armedbatle)

struct BurnDriver BurnDrvnes_armedbatle = {
	"nes_armedbatle", NULL, NULL, NULL, "2014",
	"Armed for Battle - Limited Edition (HB)\0", NULL, "1010 Howe", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_STRATEGY, 0,
	NESGetZipName, nes_armedbatleRomInfo, nes_armedbatleRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_armedbatRomDesc[] = {
	{ "Armed for Battle (2014)(1010 Howe).nes",          131088, 0xa7deed94, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_armedbat)
STD_ROM_FN(nes_armedbat)

struct BurnDriver BurnDrvnes_armedbat = {
	"nes_armedbat", "nes_armedbatle", NULL, NULL, "2014",
	"Armed for Battle (HB)\0", NULL, "1010 Howe", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_STRATEGY, 0,
	NESGetZipName, nes_armedbatRomInfo, nes_armedbatRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_boblv2RomDesc[] = {
	{ "BoBL v1.2 (2021)(Morphcat Games).nes",          131088, 0x7a2efc9d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_boblv2)
STD_ROM_FN(nes_boblv2)

struct BurnDriver BurnDrvnes_boblv2 = {
	"nes_boblv2", NULL, NULL, NULL, "2021",
	"BoBL (HB, v1.2)\0", NULL, "Morphcat Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_boblv2RomInfo, nes_boblv2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_legwnssdxRomDesc[] = {
	{ "Legend of Weed N' Stiff SDX, The (2021)(CalGames).nes",          524304, 0xe59b7fa0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_legwnssdx)
STD_ROM_FN(nes_legwnssdx)

struct BurnDriver BurnDrvnes_legwnssdx = {
	"nes_legwnssdx", NULL, NULL, NULL, "2021",
	"Legend of Weed N' Stiff SDX, The (HB)\0", NULL, "CalGames", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, nes_legwnssdxRomInfo, nes_legwnssdxRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_legwnsdxRomDesc[] = {
	{ "Legend of Weed N' Stiff DX, The (2021)(CalGames).nes",          524304, 0xf2e59dc7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_legwnsdx)
STD_ROM_FN(nes_legwnsdx)

struct BurnDriver BurnDrvnes_legwnsdx = {
	"nes_legwnsdx", "nes_legwnssdx", NULL, NULL, "2020",
	"Legend of Weed N' Stiff DX, The (HB)\0", NULL, "CalGames", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, nes_legwnsdxRomInfo, nes_legwnsdxRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_legwnsRomDesc[] = {
	{ "Legend of Weed N' Stiff, The (2020)(CalGames).nes",          524304, 0x3e8c0fe5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_legwns)
STD_ROM_FN(nes_legwns)

struct BurnDriver BurnDrvnes_legwns = {
	"nes_legwns", "nes_legwnssdx", NULL, NULL, "2020",
	"Legend of Weed N' Stiff, The (HB)\0", NULL, "CalGames", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, nes_legwnsRomInfo, nes_legwnsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_mariokartRomDesc[] = {
	{ "Mario Kart GlobHack (2017)(pacnsacdave).nes",          65552, 0xca29d3f0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mariokart)
STD_ROM_FN(nes_mariokart)

struct BurnDriver BurnDrvnes_mariokart = {
	"nes_mariokart", "nes_rcproam", NULL, NULL, "2017",
	"Mario Kart (HB, GlobalHack)\0", NULL, "pacnsacdave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW | BDF_HACK, 1, HARDWARE_NES, GBF_RACING, 0,
	NESGetZipName, nes_mariokartRomInfo, nes_mariokartRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_reknum1RomDesc[] = {
	{ "Reknum - The Awakening (2019).nes",          524304, 0xe2e0ea31, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_reknum1)
STD_ROM_FN(nes_reknum1)

struct BurnDriver BurnDrvnes_reknum1 = {
	"nes_reknum1", NULL, NULL, NULL, "2019",
	"Reknum - The Awakening (HB)\0", "No Sound", "NAPE Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_reknum1RomInfo, nes_reknum1RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_reknum2RomDesc[] = {
	{ "Reknum - Souls Adventure (2021).nes",          524304, 0xccdc9c66, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_reknum2)
STD_ROM_FN(nes_reknum2)

struct BurnDriver BurnDrvnes_reknum2 = {
	"nes_reknum2", NULL, NULL, NULL, "2021",
	"Reknum - Souls Adventure (HB)\0", NULL, "NAPE Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, nes_reknum2RomInfo, nes_reknum2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_angunaRomDesc[] = {
	{ "Anguna - Scourge of the Goblin King (2021)(The 6502 Collective).nes",          524304, 0x1119efff, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_anguna)
STD_ROM_FN(nes_anguna)

struct BurnDriver BurnDrvnes_anguna = {
	"nes_anguna", NULL, NULL, NULL, "2021",
	"Anguna - Scourge of the Goblin King (HB)\0", NULL, "The 6502 Collective", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, nes_angunaRomInfo, nes_angunaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_aoRomDesc[] = {
	{ "AO (2016)(2nd Dimension).nes",          24592, 0xbc0a5f02, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ao)
STD_ROM_FN(nes_ao)

struct BurnDriver BurnDrvnes_ao = {
	"nes_ao", NULL, NULL, NULL, "2016",
	"AO (HB)\0", NULL, "2nd Dimension", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_aoRomInfo, nes_aoRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_aodRomDesc[] = {
	{ "AO (2016)(2nd Dimension, Digital).nes",          24592, 0xae018214, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_aod)
STD_ROM_FN(nes_aod)

struct BurnDriver BurnDrvnes_aod = {
	"nes_aod", "nes_ao", NULL, NULL, "2016",
	"AO (HB, Digital Edition)\0", NULL, "2nd Dimension", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_aodRomInfo, nes_aodRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_plumchalRomDesc[] = {
	{ "Plummet Challenge Game (2021)(Fista Productions).nes",          524304, 0x7ec3675e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_plumchal)
STD_ROM_FN(nes_plumchal)

struct BurnDriver BurnDrvnes_plumchal = {
	"nes_plumchal", NULL, NULL, NULL, "2021",
	"Plummet Challenge Game (HB)\0", NULL, "Fista Productions", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_plumchalRomInfo, nes_plumchalRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_wampusdxRomDesc[] = {
	{ "Wampus DX (2020)(John Vanderhoef).nes",          524304, 0x690f19e0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_wampusdx)
STD_ROM_FN(nes_wampusdx)

struct BurnDriver BurnDrvnes_wampusdx = {
	"nes_wampusdx", NULL, NULL, NULL, "2020",
	"Wampus DX (HB)\0", NULL, "John Vanderhoef", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_wampusdxRomInfo, nes_wampusdxRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dungdoomRomDesc[] = {
	{ "Dungeons & Doomknights (2021)(Artix Ent.).nes",          524304, 0xd4e96569, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dungdoom)
STD_ROM_FN(nes_dungdoom)

struct BurnDriver BurnDrvnes_dungdoom = {
	"nes_dungdoom", NULL, NULL, NULL, "2021",
	"Dungeons & Doomknights (HB)\0", NULL, "Artix Entertainment", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, nes_dungdoomRomInfo, nes_dungdoomRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_espiteneRomDesc[] = {
	{ "Espitene (2018)(The Mojon Twins).nes",          65552, 0x1f80c94e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_espitene)
STD_ROM_FN(nes_espitene)

struct BurnDriver BurnDrvnes_espitene = {
	"nes_espitene", NULL, NULL, NULL, "2018",
	"Espitene (HB)\0", NULL, "The Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_espiteneRomInfo, nes_espiteneRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ghoulgrindRomDesc[] = {
	{ "Ghoul Grind Night of the Necromancer (2021)(WoogWorx).nes",          524304, 0x3557252e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ghoulgrind)
STD_ROM_FN(nes_ghoulgrind)

struct BurnDriver BurnDrvnes_ghoulgrind = {
	"nes_ghoulgrind", NULL, NULL, NULL, "2021",
	"Ghoul Grind Night of the Necromancer (HB)\0", NULL, "WoogWorx", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_ghoulgrindRomInfo, nes_ghoulgrindRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_wartwormRomDesc[] = {
	{ "Wart Worm Wingding (2020)(John Vanderhoef).nes",          524304, 0x8b9d8827, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_wartworm)
STD_ROM_FN(nes_wartworm)

struct BurnDriver BurnDrvnes_wartworm = {
	"nes_wartworm", NULL, NULL, NULL, "2020",
	"Wart Worm Wingding (HB)\0", NULL, "John Vanderhoef", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION | GBF_PLATFORM, 0,
	NESGetZipName, nes_wartwormRomInfo, nes_wartwormRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_beatlesadvRomDesc[] = {
	{ "The Beatles Adventures in Pepperland GlobHack (2021)(NesDraug).nes",          524304, 0xfa5e3fa2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_beatlesadv)
STD_ROM_FN(nes_beatlesadv)

struct BurnDriver BurnDrvnes_beatlesadv = {
	"nes_beatlesadv", "nes_smb2", NULL, NULL, "2021",
	"The Beatles Adventures in Pepperland (GlobalHack)\0", NULL, "NesDraug", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW | BDF_CLONE | BDF_HOMEBREW | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_beatlesadvRomInfo, nes_beatlesadvRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_firerescueRomDesc[] = {
	{ "Fire and Rescue v1.1 (2021)(Skyboy Games).nes",          40976, 0x3a233cd9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_firerescue)
STD_ROM_FN(nes_firerescue)

struct BurnDriver BurnDrvnes_firerescue = {
	"nes_firerescue", NULL, NULL, NULL, "2021",
	"Fire and Rescue (HB, v1.1)\0", NULL, "Skyboy Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_firerescueRomInfo, nes_firerescueRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_zeldaiiacRomDesc[] = {
	{ "Zelda II Amida's Curse GlobHack v1.7 (2021)(Ok Impala!).nes",          262160, 0xa6ccfea4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_zeldaiiac)
STD_ROM_FN(nes_zeldaiiac)

struct BurnDriver BurnDrvnes_zeldaiiac = {
	"nes_zeldaiiac", "nes_zeldaii", NULL, NULL, "2021",
	"Zelda II Amida's Curse (GlobalHack, v1.7)\0", NULL, "Ok Impala!", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW | BDF_CLONE | BDF_HOMEBREW | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM | GBF_ADV, 0,
	NESGetZipName, nes_zeldaiiacRomInfo, nes_zeldaiiacRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_upsadownRomDesc[] = {
	{ "Upsad Down v0.6 (2021)(Fadest).nes",          40976, 0xf90302bc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_upsadown)
STD_ROM_FN(nes_upsadown)

struct BurnDriver BurnDrvnes_upsadown = {
	"nes_upsadown", NULL, NULL, NULL, "2020",
	"Upsad Down (HB, v0.6)\0", "NesDev Competition build", "Fadest", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_upsadownRomInfo, nes_upsadownRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_blazrangRomDesc[] = {
	{ "Blazing Rangers (2021)(Karu Gamo).nes",          40976, 0xf11f92b4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_blazrang)
STD_ROM_FN(nes_blazrang)

struct BurnDriver BurnDrvnes_blazrang = {
	"nes_blazrang", NULL, NULL, NULL, "2021",
	"Blazing Rangers (HB)\0", NULL, "Karu Gamo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_blazrangRomInfo, nes_blazrangRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_antennaRomDesc[] = {
	{ "Antenna Antics (2021)(FG Software).nes",          65552, 0x0f5050e6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_antenna)
STD_ROM_FN(nes_antenna)

struct BurnDriver BurnDrvnes_antenna = {
	"nes_antenna", NULL, NULL, NULL, "2021",
	"Antenna Antics (HB)\0", NULL, "FG Software", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_antennaRomInfo, nes_antennaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dgimmickRomDesc[] = {
	{ "Dark Gimmick v1.1 (2014).nes",          393232, 0xd7ddb1d6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dgimmick)
STD_ROM_FN(nes_dgimmick)

struct BurnDriver BurnDrvnes_dgimmick = {
	"nes_dgimmick", "nes_gimmick", NULL, NULL, "2014",
	"Dark Gimmick (HB, v1.1)\0", NULL, "Berion - Dizzy9", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_dgimmickRomInfo, nes_dgimmickRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_oozerdxRomDesc[] = {
	{ "Ooze Redux (2021)(Hippokittie Prod.).nes",          524304, 0x047a83d5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_oozerdx)
STD_ROM_FN(nes_oozerdx)

struct BurnDriver BurnDrvnes_oozerdx = {
	"nes_oozerdx", NULL, NULL, NULL, "2021",
	"Ooze Redux (HB)\0", NULL, "Hippokittie Productions", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_oozerdxRomInfo, nes_oozerdxRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_fluffyRomDesc[] = {
	{ "Fluffy Space Escape (2018)(Ludum Dare 40).nes",          81936, 0xddd06400, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_fluffy)
STD_ROM_FN(nes_fluffy)

struct BurnDriver BurnDrvnes_fluffy = {
	"nes_fluffy", NULL, NULL, NULL, "2018",
	"Fluffy Space Escape (HB)\0", NULL, "Ludum Dare 40", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_fluffyRomInfo, nes_fluffyRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_fthetaRomDesc[] = {
	{ "F-Theta (2021)(Little Sound)_MirroringFix.nes",          262160, 0x7bd2813c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ftheta)
STD_ROM_FN(nes_ftheta)

struct BurnDriver BurnDrvnes_ftheta = {
	"nes_ftheta", NULL, NULL, NULL, "2021",
	"F-Theta (HB)\0", NULL, "Little Sound Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_RACING, 0,
	NESGetZipName, nes_fthetaRomInfo, nes_fthetaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_sheraRomDesc[] = {
	{ "Shera & the 40 Thieves (2021)(CGT Games).nes",          524304, 0x40da8715, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_shera)
STD_ROM_FN(nes_shera)

struct BurnDriver BurnDrvnes_shera = {
	"nes_shera", NULL, NULL, NULL, "2021",
	"Shera & the 40 Thieves (HB)\0", NULL, "CGT Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_sheraRomInfo, nes_sheraRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_shootufoRomDesc[] = {
	{ "Shoot UFO (2015)(GlobHack).nes",          40976, 0xb5c91a2a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_shootufo)
STD_ROM_FN(nes_shootufo)

struct BurnDriver BurnDrvnes_shootufo = {
	"nes_shootufo", NULL, NULL, NULL, "2015",
	"Shoot UFO (HB)\0", "A GlobalHack of Duck Hunt", "Unknown", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW | BDF_HACK, 1, HARDWARE_NES, GBF_SHOOT, 0,
	NESGetZipName, nes_shootufoRomInfo, nes_shootufoRomName, NULL, NULL, NULL, NULL, NESZapperInputInfo, NESZapperDIPInfo,
	NESZapperInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_vsfrombelowRomDesc[] = {
	{ "VS. From Below (HB, v0.10).nes",          40976, 0xd1616b88, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_vsfrombelow)
STD_ROM_FN(nes_vsfrombelow)

struct BurnDriver BurnDrvnes_vsfrombelow = {
	"vsfrombelow", NULL, NULL, NULL, "2021",
	"VS. From Below (HB, v0.10)\0", NULL, "Goose2k", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NVS, GBF_PUZZLE, 0,
	NULL, nes_vsfrombelowRomInfo, nes_vsfrombelowRomName, NULL, NULL, NULL, NULL, NESVSInputInfo, NESVSFromBelowDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_filthyktcnRomDesc[] = {
	{ "Filthy Kitchen (2016)(Dustmop).nes",          65552, 0xa22cdf8a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_filthyktcn)
STD_ROM_FN(nes_filthyktcn)

struct BurnDriver BurnDrvnes_filthyktcn = {
	"nes_filthyktcn", NULL, NULL, NULL, "2016",
	"Filthy Kitchen (HB)\0", NULL, "Dustmop", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_filthyktcnRomInfo, nes_filthyktcnRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_tapewormRomDesc[] = {
	{ "Tapeworm Disco Puzzle v05 (2022)(Lowtek Games).nes",          524304, 0x3cf42935, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tapeworm)
STD_ROM_FN(nes_tapeworm)

struct BurnDriver BurnDrvnes_tapeworm = {
	"nes_tapeworm", NULL, NULL, NULL, "2021-22",
	"Tapeworm Disco Puzzle (HB, v05)\0", NULL, "Lowtek Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_tapewormRomInfo, nes_tapewormRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_wampusRomDesc[] = {
	{ "Wampus (2019)(John Vanderhoef).nes",          524304, 0xf67d26a2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_wampus)
STD_ROM_FN(nes_wampus)

struct BurnDriver BurnDrvnes_wampus = {
	"nes_wampus", "nes_wampusdx", NULL, NULL, "2019",
	"Wampus (HB)\0", NULL, "John Vanderhoef", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_wampusRomInfo, nes_wampusRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_advpanzerRomDesc[] = {
	{ "Adventures of Panzer, The v1.5 (2021-22)(PixelCraft).nes",          524304, 0x93ba17c9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advpanzer)
STD_ROM_FN(nes_advpanzer)

struct BurnDriver BurnDrvnes_advpanzer = {
	"nes_advpanzer", NULL, NULL, NULL, "2021-22",
	"Adventures of Panzer, The (HB, v1.5)\0", NULL, "PixelCraft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM | GBF_ADV, 0,
	NESGetZipName, nes_advpanzerRomInfo, nes_advpanzerRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_explodingfistpiRomDesc[] = {
	{ "Exploding Fist (2018)(Piko Interactive).nes",          65552, 0x4017b398, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_explodingfistpi)
STD_ROM_FN(nes_explodingfistpi)

struct BurnDriver BurnDrvnes_explodingfistpi = {
	"nes_explodingfistpi", "nes_explodingfist", NULL, NULL, "2018",
	"Exploding Fist (HB)\0", NULL, "Piko Interactive", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 3, HARDWARE_NES, GBF_VSFIGHT, 0,
	NESGetZipName, nes_explodingfistpiRomInfo, nes_explodingfistpiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_hauntedadvRomDesc[] = {
	{ "Haunted House Adventure (2019)(M. Stave).nes",          524304, 0xd0c3acbc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_hauntedadv)
STD_ROM_FN(nes_hauntedadv)

struct BurnDriver BurnDrvnes_hauntedadv = {
	"nes_hauntedadv", NULL, NULL, NULL, "2019",
	"Haunted House Adventure (HB)\0", NULL, "M. Stave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_hauntedadvRomInfo, nes_hauntedadvRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_labbayeRomDesc[] = {
	{ "L'Abbaye des Morts (2021)(Broke Studios).nes",          131088, 0xa032a1f6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_labbaye)
STD_ROM_FN(nes_labbaye)

struct BurnDriver BurnDrvnes_labbaye = {
	"nes_labbaye", NULL, NULL, NULL, "2021",
	"L'Abbaye des Morts (HB)\0", NULL, "Broke Studio", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM | GBF_ADV, 0,
	NESGetZipName, nes_labbayeRomInfo, nes_labbayeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_etdxRomDesc[] = {
	{ "E.T. The Extra Terrestrial DX (2019)(pacnsacdave).nes",          524304, 0xac82d2b4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_etdx)
STD_ROM_FN(nes_etdx)

struct BurnDriver BurnDrvnes_etdx = {
	"nes_etdx", "nes_et", NULL, NULL, "2019",
	"E.T. - The Extra Terrestrial DX (HB)\0", NULL, "Khan Games - pacnsacdave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW | BDF_HACK, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_etdxRomInfo, nes_etdxRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_minibrixRomDesc[] = {
	{ "Mini Brix Battle (2016)(Aleff Correa).nes",          40976, 0x07781bee, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_minibrix)
STD_ROM_FN(nes_minibrix)

struct BurnDriver BurnDrvnes_minibrix = {
	"nes_minibrix", NULL, NULL, NULL, "2016",
	"Mini Brix Battle (HB)\0", NULL, "Aleff Correa", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_BREAKOUT, 0,
	NESGetZipName, nes_minibrixRomInfo, nes_minibrixRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_nessyrobotRomDesc[] = {
	{ "Nessy the NES Robot v1.30 (2021)(Daniel T. Gaming).nes",          524304, 0x709d7dcf, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_nessyrobot)
STD_ROM_FN(nes_nessyrobot)

struct BurnDriver BurnDrvnes_nessyrobot = {
	"nes_nessyrobot", NULL, NULL, NULL, "2021",
	"Nessy the NES Robot (HB, v1.30)\0", NULL, "Daniel T. Gaming", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM | GBF_ADV, 0,
	NESGetZipName, nes_nessyrobotRomInfo, nes_nessyrobotRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_berserkRomDesc[] = {
	{ "Berzerk (2019)(Parisoft).nes",          65552, 0x25a46b45, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_berserk)
STD_ROM_FN(nes_berserk)

struct BurnDriver BurnDrvnes_berserk = {
	"nes_berserk", NULL, NULL, NULL, "2019",
	"Berserk (HB)\0", NULL, "Parisoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_berserkRomInfo, nes_berserkRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cowligamlostadvRomDesc[] = {
	{ "Cowlitz Gamers Lost Adventure (2020)(Cowlitz).nes",          65552, 0x6ded904f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cowligamlostadv)
STD_ROM_FN(nes_cowligamlostadv)

struct BurnDriver BurnDrvnes_cowligamlostadv = {
	"nes_cowligamlostadv", NULL, NULL, NULL, "2020",
	"Cowlitz Gamers Lost Adventure (HB)\0", NULL, "Cowlitz", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_cowligamlostadvRomInfo, nes_cowligamlostadvRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_splitsecondRomDesc[] = {
	{ "Split Second (2019)(Skadiet).nes",          24592, 0xa26a6c89, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_splitsecond)
STD_ROM_FN(nes_splitsecond)

struct BurnDriver BurnDrvnes_splitsecond = {
	"nes_splitsecond", NULL, NULL, NULL, "2019",
	"Split Second (HB)\0", NULL, "Skadiet", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_splitsecondRomInfo, nes_splitsecondRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_smunlimdxRomDesc[] = {
	{ "Super Mario Unlimited Deluxe v2.4 (2021)(Frantik).nes",          196624, 0xe4198a92, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smunlimdx)
STD_ROM_FN(nes_smunlimdx)

struct BurnDriver BurnDrvnes_smunlimdx = {
	"nes_smunlimdx", NULL, NULL, NULL, "2021",
	"Super Mario Unlimited Deluxe (GlobalHack, v2.4)\0", NULL, "Frantik", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_smunlimdxRomInfo, nes_smunlimdxRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_teslavsedisonRomDesc[] = {
	{ "Tesla vs Edison (2018)(Radtek).nes",          40976, 0x7d646297, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_teslavsedison)
STD_ROM_FN(nes_teslavsedison)

struct BurnDriver BurnDrvnes_teslavsedison = {
	"nes_teslavsedison", NULL, NULL, NULL, "2018",
	"Tesla vs Edison (HB)\0", NULL, "Radnek", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION | GBF_BALLPADDLE, 0,
	NESGetZipName, nes_teslavsedisonRomInfo, nes_teslavsedisonRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_carpetsharkRomDesc[] = {
	{ "Carpet Shark (2021)(Fista Productions).nes",          524304, 0x8ab5f2e0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_carpetshark)
STD_ROM_FN(nes_carpetshark)

struct BurnDriver BurnDrvnes_carpetshark = {
	"nes_carpetshark", NULL, NULL, NULL, "2021",
	"Carpet Shark (HB)\0", NULL, "Fista Productions", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_carpetsharkRomInfo, nes_carpetsharkRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bubblebobbledxRomDesc[] = {
	{ "Bubble Bobble Deluxe (2021)(Polar Hacker).nes",          163856, 0x198125ce, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bubblebobbledx)
STD_ROM_FN(nes_bubblebobbledx)

struct BurnDriver BurnDrvnes_bubblebobbledx = {
	"nes_bubblebobbledx", "nes_bubblebobble", NULL, NULL, "2018",
	"Bubble Bobble Deluxe (Hack, HB)\0", NULL, "Polar Hacker", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_bubblebobbledxRomInfo, nes_bubblebobbledxRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_gruniRomDesc[] = {
	{ "Gruniozerca (2017)(Lukasz Kur).nes",          40976, 0x2ea66761, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_gruni)
STD_ROM_FN(nes_gruni)

struct BurnDriver BurnDrvnes_gruni = {
	"nes_gruni", NULL, NULL, NULL, "2017",
	"Gruniozerca (HB)\0", NULL, "Lukasz Kur", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_gruniRomInfo, nes_gruniRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_mkrotfsRomDesc[] = {
	{ "Micro Knight RotFS (2017)(Unknown).nes",          131088, 0xbc44af32, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mkrotfs)
STD_ROM_FN(nes_mkrotfs)

struct BurnDriver BurnDrvnes_mkrotfs = {
	"nes_mkrotfs", NULL, NULL, NULL, "2017",
	"Micro Knight - Revenge of the Four Skulls (HB)\0", NULL, "Unknown", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_mkrotfsRomInfo, nes_mkrotfsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_nesertgolfRomDesc[] = {
	{ "NESert Golfing (2019)(Unknown).nes",          32784, 0xcb43ec95, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_nesertgolf)
STD_ROM_FN(nes_nesertgolf)

struct BurnDriver BurnDrvnes_nesertgolf = {
	"nes_nesertgolf", NULL, NULL, NULL, "2019",
	"NESert Golfing (HB)\0", NULL, "Unknown", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 4, HARDWARE_NES, GBF_SPORTSMISC, 0,
	NESGetZipName, nes_nesertgolfRomInfo, nes_nesertgolfRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_nesertgolfteRomDesc[] = {
	{ "NESert Golfing - Tournament Edition (2019)(Unknown).nes",          32784, 0xdb28a4e8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_nesertgolfte)
STD_ROM_FN(nes_nesertgolfte)

struct BurnDriver BurnDrvnes_nesertgolfte = {
	"nes_nesertgolfte", "nes_nesertgolf", NULL, NULL, "2019",
	"NESert Golfing - Tournament Edition (HB)\0", NULL, "Unknown", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 4, HARDWARE_NES, GBF_SPORTSMISC, 0,
	NESGetZipName, nes_nesertgolfteRomInfo, nes_nesertgolfteRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_rockm5mpRomDesc[] = {
	{ "Rockman 5 - Metropolis (2021)(Stalkermaestro).nes",          524304, 0x0911f8bf, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_rockm5mp)
STD_ROM_FN(nes_rockm5mp)

struct BurnDriver BurnDrvnes_rockm5mp = {
	"nes_rockm5mp", "nes_megaman5", NULL, NULL, "2021",
	"Rockman 5 - Metropolis (Hack, HB)\0", NULL, "Stalkermaestro", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_rockm5mpRomInfo, nes_rockm5mpRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_magnilocaseRomDesc[] = {
	{ "Magnilo Case, The v1.1 (2021)(DalyenRetroGames).nes",          524304, 0x3b14782a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_magnilocase)
STD_ROM_FN(nes_magnilocase)

struct BurnDriver BurnDrvnes_magnilocase = {
	"nes_magnilocase", NULL, NULL, NULL, "2021",
	"Magnilo Case, The (HB, v1.1)\0", NULL, "DalyenRetroGames", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, nes_magnilocaseRomInfo, nes_magnilocaseRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_evildeadRomDesc[] = {
	{ "Evil Dead Ash lives! (2020)(Crying Onion).nes",          262160, 0x2ce2ead0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_evildead)
STD_ROM_FN(nes_evildead)

struct BurnDriver BurnDrvnes_evildead = {
	"nes_evildead", NULL, NULL, NULL, "2020",
	"Evil Dead - Ash lives! (Hack)\0", NULL, "Crying Onion", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_evildeadRomInfo, nes_evildeadRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_gyromiteseRomDesc[] = {
	{ "Gyromite Special Edition (2014)(the jabu).nes",          40976, 0x1495d355, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_gyromitese)
STD_ROM_FN(nes_gyromitese)

struct BurnDriver BurnDrvnes_gyromitese = {
	"nes_gyromitese", "nes_gyromite", NULL, NULL, "2014",
	"Gyromite Special Edition (Hack)\0", "This hack allows to play the game on a traditional controller", "the Jabu", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM | GBF_PUZZLE, 0,
	NESGetZipName, nes_gyromiteseRomInfo, nes_gyromiteseRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_pacmanxRomDesc[] = {
	{ "Pac-Man Xtreme (2016)(KaBooM!).nes",          40976, 0xee653ac8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_pacmanx)
STD_ROM_FN(nes_pacmanx)

struct BurnDriver BurnDrvnes_pacmanx = {
	"nes_pacmanx", NULL, NULL, NULL, "2016",
	"Pac-Man Xtreme (Hack)\0", NULL, "KaBooM!", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_MAZE | GBF_ACTION, 0,
	NESGetZipName, nes_pacmanxRomInfo, nes_pacmanxRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_smb8enRomDesc[] = {
	{ "Super Mario Bros 8 hack eng (2021)(Crying Onion).nes",          393232, 0x1f5d9f05, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smb8en)
STD_ROM_FN(nes_smb8en)

struct BurnDriver BurnDrvnes_smb8en = {
	"nes_smb8en", NULL, NULL, NULL, "2021",
	"Super Mario Bros 8 (English) (Hack)\0", NULL, "Crying Onion", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_smb8enRomInfo, nes_smb8enRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_smb8esRomDesc[] = {
	{ "Super Mario Bros 8 hack spa (2021)(Crying Onion).nes",          393232, 0xef9c0a4f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smb8es)
STD_ROM_FN(nes_smb8es)

struct BurnDriver BurnDrvnes_smb8es = {
	"nes_smb8es", "nes_smb8en", NULL, NULL, "2021",
	"Super Mario Bros 8 (Spanish) (Hack)\0", NULL, "Crying Onion", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_smb8esRomInfo, nes_smb8esRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_smb8seRomDesc[] = {
	{ "Super Mario Bros 8 SE hack (2021)(Crying Onion).nes",          393232, 0xaa5822c0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smb8se)
STD_ROM_FN(nes_smb8se)

struct BurnDriver BurnDrvnes_smb8se = {
	"nes_smb8se", "nes_smb8en", NULL, NULL, "2021",
	"Super Mario Bros 8 Special Edition (Hack)\0", "Don Doko Don 2 tribute edition", "Crying Onion", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_smb8seRomInfo, nes_smb8seRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_spiritimpelRomDesc[] = {
	{ "Spirit Impel (2020)(Passe Gaming).nes",          524304, 0x146f8f6c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_spiritimpel)
STD_ROM_FN(nes_spiritimpel)

struct BurnDriver BurnDrvnes_spiritimpel = {
	"nes_spiritimpel", NULL, NULL, NULL, "2020",
	"Spirit Impel (HB)\0", NULL, "Passe Gaming", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_HORSHOOT, 0,
	NESGetZipName, nes_spiritimpelRomInfo, nes_spiritimpelRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_batterychadRomDesc[] = {
	{ "Battery Chad - Shock the World (2019)(Formula Fanboy).nes",          524304, 0x030c4355, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_batterychad)
STD_ROM_FN(nes_batterychad)

struct BurnDriver BurnDrvnes_batterychad = {
	"nes_batterychad", NULL, NULL, NULL, "2019",
	"Battery Chad - Shock the World (HB)\0", NULL, "Formula Fanboy", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_batterychadRomInfo, nes_batterychadRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_frogRomDesc[] = {
	{ "Frog (2020)(Boston Breams).nes",          393232, 0x06d96b55, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_frog)
STD_ROM_FN(nes_frog)

struct BurnDriver BurnDrvnes_frog = {
	"nes_frog", NULL, NULL, NULL, "2020",
	"Frog (HB)\0", NULL, "Boston Breams", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_frogRomInfo, nes_frogRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_harrydwarfRomDesc[] = {
	{ "Harry Dwarf (2019)(Byte-off).nes",          524304, 0xf51e13d4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_harrydwarf)
STD_ROM_FN(nes_harrydwarf)

struct BurnDriver BurnDrvnes_harrydwarf = {
	"nes_harrydwarf", NULL, NULL, NULL, "2019",
	"Harry Dwarf (HB)\0", NULL, "Byte-off", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM | GBF_PUZZLE, 0,
	NESGetZipName, nes_harrydwarfRomInfo, nes_harrydwarfRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_slowmoleRomDesc[] = {
	{ "Slow Mole v1.2b (2021)(Erik Rosenlund).nes",          524304, 0x61226e5f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_slowmole)
STD_ROM_FN(nes_slowmole)

struct BurnDriver BurnDrvnes_slowmole = {
	"nes_slowmole", NULL, NULL, NULL, "2021",
	"Slow Mole (HB, v1.2b)\0", NULL, "Erik Rosenlund", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_slowmoleRomInfo, nes_slowmoleRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_zeldakongRomDesc[] = {
	{ "Zelda Kong (2021)(Zero Meaning).nes",          24592, 0x4c801a11, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_zeldakong)
STD_ROM_FN(nes_zeldakong)

struct BurnDriver BurnDrvnes_zeldakong = {
	"nes_zeldakong", "nes_donkeykong", NULL, NULL, "2021",
	"Zelda Kong (HB, Global Hack)\0", NULL, "Zero Meaning", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_zeldakongRomInfo, nes_zeldakongRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_zdeyRomDesc[] = {
	{ "Zdey the Game (2020)(Art'Cade).nes",          524304, 0xfd4036e1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_zdey)
STD_ROM_FN(nes_zdey)

struct BurnDriver BurnDrvnes_zdey = {
	"nes_zdey", NULL, NULL, NULL, "2020",
	"Zdey the Game (HB)\0", NULL, "Art'Cade", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_zdeyRomInfo, nes_zdeyRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_battlerRomDesc[] = {
	{ "Battler (2015)(Karu Gamo).nes",          40976, 0x38f65b2d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_battler)
STD_ROM_FN(nes_battler)

struct BurnDriver BurnDrvnes_battler = {
	"nes_battler", NULL, NULL, NULL, "2015",
	"Battler (HB)\0", NULL, "Karu Gamo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_SHOOT, 0,
	NESGetZipName, nes_battlerRomInfo, nes_battlerRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bomberboxartRomDesc[] = {
	{ "Bomberman - BoxArt v1.02 (2017)(Corpse Grinder).nes",          32784, 0xcb68a7fc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bomberboxart)
STD_ROM_FN(nes_bomberboxart)

struct BurnDriver BurnDrvnes_bomberboxart = {
	"nes_bomberboxart", NULL, NULL, NULL, "2017",
	"Bomberman - BoxArt (HB, v1.02)\0", NULL, "Corpse Grinder", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HACK | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION | GBF_MAZE, 0,
	NESGetZipName, nes_bomberboxartRomInfo, nes_bomberboxartRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ggggRomDesc[] = {
	{ "Gold Guardian Gun Girl (2020)(Good Tune).nes",          40976, 0xa0e59e67, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_gggg)
STD_ROM_FN(nes_gggg)

struct BurnDriver BurnDrvnes_gggg = {
	"nes_gggg", NULL, NULL, NULL, "2020",
	"Gold Guardian Gun Girl (HB)\0", NULL, "Good Tune", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_SHOOT, 0,
	NESGetZipName, nes_ggggRomInfo, nes_ggggRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_justiceRomDesc[] = {
	{ "Justice (2020)(Unknown).nes",          196624, 0x5a8b8f13, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_justice)
STD_ROM_FN(nes_justice)

struct BurnDriver BurnDrvnes_justice = {
	"nes_justice", NULL, NULL, NULL, "2020",
	"Justice (HB)\0", NULL, "Unknown", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_SCRFIGHT | GBF_PLATFORM, 0,
	NESGetZipName, nes_justiceRomInfo, nes_justiceRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_altercationRomDesc[] = {
	{ "Altercation v0.1 (2021)(Scottlowe).nes",          40976, 0x27288df5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_altercation)
STD_ROM_FN(nes_altercation)

struct BurnDriver BurnDrvnes_altercation = {
	"nes_altercation", NULL, NULL, NULL, "2021",
	"Altercation (HB, v0.1)\0", NULL, "Scottlowe", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_altercationRomInfo, nes_altercationRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_crapjobRomDesc[] = {
	{ "Crap Job (2021)(Oniric Factor).nes",          524304, 0xcfa265ab, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_crapjob)
STD_ROM_FN(nes_crapjob)

struct BurnDriver BurnDrvnes_crapjob = {
	"nes_crapjob", NULL, NULL, NULL, "2021",
	"Crap Job (HB)\0", NULL, "Oniric Factor", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_crapjobRomInfo, nes_crapjobRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_blackholRomDesc[] = {
	{ "Black Hole - Czarna Dziura (HB, v0.20)(SIUDYM).nes",          131088, 0x8fd3a9a7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_blackhol)
STD_ROM_FN(nes_blackhol)

struct BurnDriver BurnDrvnes_blackhol = {
	"nes_blackhol", NULL, NULL, NULL, "2021",
	"Black Hole - Czarna Dziura (HB, v0.20)\0", NULL, "Siudym", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_SHOOT, 0,
	NESGetZipName, nes_blackholRomInfo, nes_blackholRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_hackmatchRomDesc[] = {
	{ "Hack Match (2021)(Zachtronics).nes",          40976, 0x94554d49, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_hackmatch)
STD_ROM_FN(nes_hackmatch)

struct BurnDriver BurnDrvnes_hackmatch = {
	"nes_hackmatch", NULL, NULL, NULL, "2021",
	"Hack*Match (HB)\0", "Original game by Juice (1990)", "Zachtronics", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_hackmatchRomInfo, nes_hackmatchRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_mnmayhemRomDesc[] = {
	{ "Mine Mayhem (2021)(Vectrex28).nes",          65552, 0xc6b47cca, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mnmayhem)
STD_ROM_FN(nes_mnmayhem)

struct BurnDriver BurnDrvnes_mnmayhem = {
	"nes_mnmayhem", NULL, NULL, NULL, "2021",
	"Mine Mayhem (HB)\0", NULL, "FG Software", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_mnmayhemRomInfo, nes_mnmayhemRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_luckypenguinRomDesc[] = {
	{ "Lucky Penguin (2019)(Mega Cat Studios).nes",          65552, 0x2a4c310b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_luckypenguin)
STD_ROM_FN(nes_luckypenguin)

struct BurnDriver BurnDrvnes_luckypenguin = {
	"nes_luckypenguin", NULL, NULL, NULL, "2019",
	"Lucky Penguin (HB)\0", NULL, "Mega Cat Studios", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_luckypenguinRomInfo, nes_luckypenguinRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_batlizardRomDesc[] = {
	{ "Bat Lizard Bonanza (2020)(John Vanderhoef).nes",          524304, 0x4418f0f3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_batlizard)
STD_ROM_FN(nes_batlizard)

struct BurnDriver BurnDrvnes_batlizard = {
	"nes_batlizard", NULL, NULL, NULL, "2020",
	"Bat Lizard Bonanza (HB)\0", NULL, "John Vanderhoef", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_HORSHOOT, 0,
	NESGetZipName, nes_batlizardRomInfo, nes_batlizardRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_eyraRomDesc[] = {
	{ "Eyra the Crow Maiden (2020)(2nd Dimension).nes",          40976, 0x1415153d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_eyra)
STD_ROM_FN(nes_eyra)

struct BurnDriver BurnDrvnes_eyra = {
	"nes_eyra", NULL, NULL, NULL, "2020",
	"Eyra - The Crow Maiden (HB)\0", NULL, "2nd Dimension", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_eyraRomInfo, nes_eyraRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ghostbrmRomDesc[] = {
	{ "Ghostbusters Remastered v1.1 (2019)(Nesrocks).nes",          98320, 0x29c2e0d0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ghostbrm)
STD_ROM_FN(nes_ghostbrm)

struct BurnDriver BurnDrvnes_ghostbrm = {
	"nes_ghostbrm", NULL, NULL, NULL, "2019",
	"Ghostbusters Remastered (HB, v1.1)\0", NULL, "NesRocks", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW | BDF_HACK, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_ghostbrmRomInfo, nes_ghostbrmRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_littlelancRomDesc[] = {
	{ "Little Lancelot (2019)(Piko Int).nes",          262160, 0x65e1bc64, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_littlelanc)
STD_ROM_FN(nes_littlelanc)

struct BurnDriver BurnDrvnes_littlelanc = {
	"nes_littlelanc", NULL, NULL, NULL, "2019",
	"Little Lancelot (HB)\0", "Developed by Ocean", "Piko Interactive", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_littlelancRomInfo, nes_littlelancRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_pyramidsraRomDesc[] = {
	{ "Pyramids of Ra (2019)(Piko Int.).nes",          393232, 0x0bcee664, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_pyramidsra)
STD_ROM_FN(nes_pyramidsra)

struct BurnDriver BurnDrvnes_pyramidsra = {
	"nes_pyramidsra", NULL, NULL, NULL, "2019",
	"Pyramids of Ra (HB)\0", NULL, "Piko Interactive", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_pyramidsraRomInfo, nes_pyramidsraRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_starversusRomDesc[] = {
	{ "Star Versus (2015)(Dustmop).nes",          196624, 0x33e83fa5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_starversus)
STD_ROM_FN(nes_starversus)

struct BurnDriver BurnDrvnes_starversus = {
	"nes_starversus", NULL, NULL, NULL, "2015",
	"Star Versus (HB)\0", NULL, "Dustmop", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_SHOOT, 0,
	NESGetZipName, nes_starversusRomInfo, nes_starversusRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_8bitpicsRomDesc[] = {
	{ "8bit Illustrations (2015)(RIKI).nes",          524304, 0x19ca739b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_8bitpics)
STD_ROM_FN(nes_8bitpics)

struct BurnDriver BurnDrvnes_8bitpics = {
	"nes_8bitpics", NULL, NULL, NULL, "2015",
	"8-bit Illustrations (HB)\0", NULL, "RiKi", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_8bitpicsRomInfo, nes_8bitpicsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_8bitmusicRomDesc[] = {
	{ "8bit Music Power (2017)(RIKI).nes",          786448, 0x351426ab, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_8bitmusic)
STD_ROM_FN(nes_8bitmusic)

struct BurnDriver BurnDrvnes_8bitmusic = {
	"nes_8bitmusic", NULL, NULL, NULL, "2017",
	"8-bit Music Power (HB)\0", NULL, "RiKi", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_8bitmusicRomInfo, nes_8bitmusicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_8bitmusiceRomDesc[] = {
	{ "8bit Music Power ENCORE (2017)(RIKI).nes",          524304, 0x8b381de2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_8bitmusice)
STD_ROM_FN(nes_8bitmusice)

struct BurnDriver BurnDrvnes_8bitmusice = {
	"nes_8bitmusice", "nes_8bitmusic", NULL, NULL, "2017",
	"8-bit Music Power ENCORE (HB)\0", NULL, "RiKi", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW | BDF_CLONE, 1, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_8bitmusiceRomInfo, nes_8bitmusiceRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_chknightRomDesc[] = {
	{ "Chrono Knight (2020)(Artix Games).nes",          524304, 0x88a47918, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chknight)
STD_ROM_FN(nes_chknight)

struct BurnDriver BurnDrvnes_chknight = {
	"nes_chknight", NULL, NULL, NULL, "2020",
	"Chrono Knight (HB)\0", NULL, "Artix Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_chknightRomInfo, nes_chknightRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_chumleeadvRomDesc[] = {
	{ "Chumlee Adventure (2020)(Khan Games).nes",          131088, 0x0f81f876, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chumleeadv)
STD_ROM_FN(nes_chumleeadv)

struct BurnDriver BurnDrvnes_chumleeadv = {
	"nes_chumleeadv", NULL, NULL, NULL, "2020",
	"Chumlee's Adventure - the quest for Pinky (HB)\0", NULL, "Khan Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_chumleeadvRomInfo, nes_chumleeadvRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_doodleworldRomDesc[] = {
	{ "Doodle World (2020)(Nate Peters).nes",          524304, 0x6c71a681, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_doodleworld)
STD_ROM_FN(nes_doodleworld)

struct BurnDriver BurnDrvnes_doodleworld = {
	"nes_doodleworld", NULL, NULL, NULL, "2020",
	"Doodle World (HB)\0", NULL, "Nate Peters", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_doodleworldRomInfo, nes_doodleworldRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_frenzyRomDesc[] = {
	{ "Frenzy A7800 (2016)(Bob Decrescenzo).nes",          40976, 0x82b0f8f4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_frenzy)
STD_ROM_FN(nes_frenzy)

struct BurnDriver BurnDrvnes_frenzy = {
	"nes_frenzy", NULL, NULL, NULL, "2016",
	"Frenzy A7800 (HB)\0", "The Atari 7800 Frenzy on NES (conversion)", "Bob Decrescenzo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_frenzyRomInfo, nes_frenzyRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_jimpowerRomDesc[] = {
	{ "Jim Power - The Lost Dimension (2020)(Piko).nes",          524304, 0x3f088114, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_jimpower)
STD_ROM_FN(nes_jimpower)

struct BurnDriver BurnDrvnes_jimpower = {
	"nes_jimpower", NULL, NULL, NULL, "2020",
	"Jim Power - The Lost Dimension (HB)\0", NULL, "Piko Interactive", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM | GBF_RUNGUN, 0,
	NESGetZipName, nes_jimpowerRomInfo, nes_jimpowerRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_leglutlivRomDesc[] = {
	{ "Leggite Luta Livre (2021)(John Vanderhoef).nes",          524304, 0xbcbc972c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_leglutliv)
STD_ROM_FN(nes_leglutliv)

struct BurnDriver BurnDrvnes_leglutliv = {
	"nes_leglutliv", "nes_leglutlivr", NULL, NULL, "2021",
	"Leggite Luta Livre (HB)\0", NULL, "John Vanderhoef", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_leglutlivRomInfo, nes_leglutlivRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ninja4sRomDesc[] = {
	{ "Ninja of the 4 Seasons (2021)(Kool Doob).nes",          524304, 0x3943cf5f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ninja4s)
STD_ROM_FN(nes_ninja4s)

struct BurnDriver BurnDrvnes_ninja4s = {
	"nes_ninja4s", NULL, NULL, NULL, "2021",
	"Ninja of the 4 Seasons (HB)\0", NULL, "Kool Doob", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_ninja4sRomInfo, nes_ninja4sRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ploidRomDesc[] = {
	{ "Ploid (2020)(NAPE Games).nes",          524304, 0xc7567823, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ploid)
STD_ROM_FN(nes_ploid)

struct BurnDriver BurnDrvnes_ploid = {
	"nes_ploid", NULL, NULL, NULL, "2020",
	"Ploid (HB)\0", NULL, "NAPE Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_ploidRomInfo, nes_ploidRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_astronmdxRomDesc[] = {
	{ "Astro Ninja Man DX (2022)(RiKi).nes",          786448, 0x27d5371f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_astronmdx)
STD_ROM_FN(nes_astronmdx)

struct BurnDriver BurnDrvnes_astronmdx = {
	"nes_astronmdx", NULL, NULL, NULL, "2022",
	"Astro Ninja Man DX (HB)\0", NULL, "RiKi", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_VERSHOOT, 0,
	NESGetZipName, nes_astronmdxRomInfo, nes_astronmdxRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_astronmRomDesc[] = {
	{ "Astro Ninja Man (2020)(RiKi).nes",          786448, 0xf709e1e0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_astronm)
STD_ROM_FN(nes_astronm)

struct BurnDriver BurnDrvnes_astronm = {
	"nes_astronm", "nes_astronmdx", NULL, NULL, "2020",
	"Astro Ninja Man (HB)\0", NULL, "RiKi", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_VERSHOOT, 0,
	NESGetZipName, nes_astronmRomInfo, nes_astronmRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_porunchanRomDesc[] = {
	{ "Porun-chan no onigiri daisuki (v1.1-2019)(mook-tv).nes",          262160, 0xe15026cd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_porunchan)
STD_ROM_FN(nes_porunchan)

struct BurnDriver BurnDrvnes_porunchan = {
	"nes_porunchan", NULL, NULL, NULL, "2019",
	"Porun-chan No Onigiri Daisuki (HB, v1.1)\0", NULL, "Mook-TV", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_porunchanRomInfo, nes_porunchanRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_porunamabieRomDesc[] = {
	{ "Porun-chan no Onigiri Daisuki - Amabie (2020)(Mook-TV).nes",          262160, 0xc2c2e17a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_porunamabie)
STD_ROM_FN(nes_porunamabie)

struct BurnDriver BurnDrvnes_porunamabie = {
	"nes_porunamabie", NULL, NULL, NULL, "2020",
	"Porun-chan no Onigiri Daisuki - Amabie (HB)\0", NULL, "Mook-TV", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_porunamabieRomInfo, nes_porunamabieRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_trophyRomDesc[] = {
	{ "Trophy (2020)(the 6502 collective).nes",          786448, 0x94c16bfc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_trophy)
STD_ROM_FN(nes_trophy)

struct BurnDriver BurnDrvnes_trophy = {
	"nes_trophy", NULL, NULL, NULL, "2020",
	"Trophy (HB)\0", NULL, "the 6502 collective", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_trophyRomInfo, nes_trophyRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_arkaderushRomDesc[] = {
	{ "Arkade Rush (HB)(2021)(Nathan Tolbert).nes",          32784, 0xd40c212e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_arkaderush)
STD_ROM_FN(nes_arkaderush)

struct BurnDriver BurnDrvnes_arkaderush = {
	"nes_arkaderush", NULL, NULL, NULL, "2021",
	"Arkade Rush (HB)\0", NULL, "Nathan Tolbert", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 4, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_arkaderushRomInfo, nes_arkaderushRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bmetal2RomDesc[] = {
	{ "bare metal 2.0(2021)(scarduadev).nes",          49168, 0xe37e0c59, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bmetal2)
STD_ROM_FN(nes_bmetal2)

struct BurnDriver BurnDrvnes_bmetal2 = {
	"nes_bmetal2", NULL, NULL, NULL, "2021",
	"Bare Metal 2.0 (HB)\0", "NesDev Competition build", "ScarduaDev", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM | GBF_PUZZLE, 0,
	NESGetZipName, nes_bmetal2RomInfo, nes_bmetal2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_blobquestRomDesc[] = {
	{ "Blob Quest (HB)(2021)(Tom Livak).nes",          32784, 0xe3ac2b6d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_blobquest)
STD_ROM_FN(nes_blobquest)

struct BurnDriver BurnDrvnes_blobquest = {
	"nes_blobquest", NULL, NULL, NULL, "2021",
	"Blob Quest (HB)\0", NULL, "Tom Livak", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_blobquestRomInfo, nes_blobquestRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_spacegullsRomDesc[] = {
	{ "Spacegulls (HB)(2021)(Morphcat Games).nes",          65552, 0x539f87d4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_spacegulls)
STD_ROM_FN(nes_spacegulls)

struct BurnDriver BurnDrvnes_spacegulls = {
	"nes_spacegulls", NULL, NULL, NULL, "2021",
	"Spacegulls (HB, v1.1)\0", NULL, "Morphcat Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_spacegullsRomInfo, nes_spacegullsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_witchnwizRomDesc[] = {
	{ "Witch n' Wiz (2021)(mhughson).nes",          262160, 0x4ac76d3f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_witchnwiz)
STD_ROM_FN(nes_witchnwiz)

struct BurnDriver BurnDrvnes_witchnwiz = {
	"nes_witchnwiz", NULL, NULL, NULL, "2021",
	"Witch n' Wiz (HB)\0", NULL, "Matt Hughson", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE | GBF_ADV, 0,
	NESGetZipName, nes_witchnwizRomInfo, nes_witchnwizRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_witchnwizndRomDesc[] = {
	{ "Witch n' Wiz (2020-21)(NESDev Ed.).nes",          65552, 0xbddd6cba, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_witchnwiznd)
STD_ROM_FN(nes_witchnwiznd)

struct BurnDriver BurnDrvnes_witchnwiznd = {
	"nes_witchnwiznd", "nes_witchnwiz", NULL, NULL, "2020-21",
	"Witch n' Wiz (HB, NESDev Ed.)\0", NULL, "Matt Hughson", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_witchnwizndRomInfo, nes_witchnwizndRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_wolfspiritRomDesc[] = {
	{ "Wolf Spirit (HB)(2021)(Valdir Salgueiro).nes",          65552, 0x74574809, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_wolfspirit)
STD_ROM_FN(nes_wolfspirit)

struct BurnDriver BurnDrvnes_wolfspirit = {
	"nes_wolfspirit", NULL, NULL, NULL, "2021",
	"Wolf Spirit (HB, Demo)\0", NULL, "Valdir Salgueiro", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_wolfspiritRomInfo, nes_wolfspiritRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_tapewormdRomDesc[] = {
	{ "Tapeworm Disco Puzzle (HB)(Demo).nes",          65552, 0x30e0689f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tapewormd)
STD_ROM_FN(nes_tapewormd)

struct BurnDriver BurnDrvnes_tapewormd = {
	"nes_tapewormd", "nes_tapeworm", NULL, NULL, "2020",
	"Tapeworm Disco Puzzle (HB, Demo)\0", NULL, "Lowtek Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_tapewormdRomInfo, nes_tapewormdRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_whatremainsRomDesc[] = {
	{ "What Remains (HB).nes",          524304, 0x2f2d03e5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_whatremains)
STD_ROM_FN(nes_whatremains)

struct BurnDriver BurnDrvnes_whatremains = {
	"nes_whatremains", NULL, NULL, NULL, "2019-20",
	"What Remains (HB, v1.0.2)\0", NULL, "Iodine Dynamics", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_whatremainsRomInfo, nes_whatremainsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_galfRomDesc[] = {
	{ "Galf (2018)(Limited Run Games).nes",          262160, 0x52340070, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_galf)
STD_ROM_FN(nes_galf)

struct BurnDriver BurnDrvnes_galf = {
	"nes_galf", NULL, NULL, NULL, "2018",
	"Galf (HB)\0", NULL, "Limited Run Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_SPORTSMISC, 0,
	NESGetZipName, nes_galfRomInfo, nes_galfRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_sharksharkRomDesc[] = {
	{ "Shark! Shark! (HB).nes",          24592, 0x5f33e5b4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sharkshark)
STD_ROM_FN(nes_sharkshark)

struct BurnDriver BurnDrvnes_sharkshark = {
	"nes_sharkshark", NULL, NULL, NULL, "2014",
	"Shark! Shark! (HB)\0", "Original title developed for Intellivision in 1982", "Mattel Electronics", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_sharksharkRomInfo, nes_sharksharkRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_caveRomDesc[] = {
	{ "cave (2017)(mojon twins).nes",          49168, 0xb86ce86b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cave)
STD_ROM_FN(nes_cave)

struct BurnDriver BurnDrvnes_cave = {
	"nes_cave", NULL, NULL, NULL, "2017",
	"Cave (HB)\0", NULL, "The Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_caveRomInfo, nes_caveRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_nixrelicRomDesc[] = {
	{ "Nix Paradox Relic v1.9(2020)(Dustin Huddleston).nes",          524304, 0xc85e7bcd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_nixrelic)
STD_ROM_FN(nes_nixrelic)

struct BurnDriver BurnDrvnes_nixrelic = {
	"nes_nixrelic", NULL, NULL, NULL, "2020",
	"Nix: the Paradox Relic (HB, v1.9)\0", NULL, "Dustin Huddleston", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_nixrelicRomInfo, nes_nixrelicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_rollieRomDesc[] = {
	{ "Rollie (HB).nes",          40976, 0x8165c356, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_rollie)
STD_ROM_FN(nes_rollie)

struct BurnDriver BurnDrvnes_rollie = {
	"nes_rollie", NULL, NULL, NULL, "2019",
	"Rollie (HB)\0", NULL, "Optovania", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_rollieRomInfo, nes_rollieRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_kosatakRomDesc[] = {
	{ "Kosmity Atakujo (HB, v10.07.2020).nes",          131088, 0x0570865c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_kosatak)
STD_ROM_FN(nes_kosatak)

struct BurnDriver BurnDrvnes_kosatak = {
	"nes_kosatak", NULL, NULL, NULL, "2020",
	"Kosmity Atakujo (HB, v10.07.2020)\0", NULL, "sdm2", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_kosatakRomInfo, nes_kosatakRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_jaysilentRomDesc[] = {
	{ "Jay and Silent Bob Mall Brawl.nes",          393232, 0xf167590d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_jaysilent)
STD_ROM_FN(nes_jaysilent)

struct BurnDriver BurnDrvnes_jaysilent = {
	"nes_jaysilent", NULL, NULL, NULL, "2020",
	"Jay and Silent Bob Mall Brawl (HB)\0", NULL, "Interabang Entertainment", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_jaysilentRomInfo, nes_jaysilentRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_journeyalRomDesc[] = {
	{ "journey to the center of the alien (2019)(mojon twins).nes",          73744, 0xd4b8eaa5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_journeyal)
STD_ROM_FN(nes_journeyal)

struct BurnDriver BurnDrvnes_journeyal = {
	"nes_journeyal", NULL, NULL, NULL, "2019",
	"Journey to the Center of the Alien (HB)\0", NULL, "The Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_journeyalRomInfo, nes_journeyalRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_blockageRomDesc[] = {
	{ "Blockage (HB, v0.3.2).nes",          40976, 0x1c5f7bcf, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_blockage)
STD_ROM_FN(nes_blockage)

struct BurnDriver BurnDrvnes_blockage = {
	"nes_blockage", NULL, NULL, NULL, "2019",
	"Blockage (HB, v0.3.2)\0", NULL, "Scott Lowe", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_blockageRomInfo, nes_blockageRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_frombelowRomDesc[] = {
	{ "From Below (HB, v1.0).nes",          40976, 0x8ce49adb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_frombelow)
STD_ROM_FN(nes_frombelow)

struct BurnDriver BurnDrvnes_frombelow = {
	"nes_frombelow", NULL, NULL, NULL, "2020",
	"From Below (HB, v1.0)\0", NULL, "Goose2k", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_frombelowRomInfo, nes_frombelowRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_legenzeldxRomDesc[] = {
	{ "legend of zelda dx (2014)(pacnsacdave).nes",          131088, 0x0698579d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_legenzeldx)
STD_ROM_FN(nes_legenzeldx)

struct BurnDriver BurnDrvnes_legenzeldx = {
	"nes_legenzeldx", "nes_legenzel", NULL, NULL, "2014",
	"Legend of Zelda DX, The (HB, Graphics Hack)\0", NULL, "pacnsacdave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_ACTION | GBF_ADV, 0,
	NESGetZipName, nes_legenzeldxRomInfo, nes_legenzeldxRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_multidudeRomDesc[] = {
	{ "multidude (2014)(retrosouls).nes",          40976, 0x154af940, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_multidude)
STD_ROM_FN(nes_multidude)

struct BurnDriver BurnDrvnes_multidude = {
	"nes_multidude", NULL, NULL, NULL, "2014",
	"Multidude (HB)\0", NULL, "RetroSouls", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_multidudeRomInfo, nes_multidudeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_nimnomRomDesc[] = {
	{ "nim and nom (2018)(metakrill) v1.2.nes",          40976, 0xe0096736, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_nimnom)
STD_ROM_FN(nes_nimnom)

struct BurnDriver BurnDrvnes_nimnom = {
	"nes_nimnom", NULL, NULL, NULL, "2018",
	"Nim & Nom (HB, v1.2)\0", NULL, "Metakrill", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_nimnomRomInfo, nes_nimnomRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// https://www.romhacking.net/hacks/144/
static struct BurnRomInfo nes_smbdxRomDesc[] = {
	{ "Super Mario Bros. DX - Anniversary Edition (Hack, v4.2).nes",          40976, 0x03df519c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smbdx)
STD_ROM_FN(nes_smbdx)

struct BurnDriver BurnDrvnes_smbdx = {
	"nes_smbdx", "nes_smb", NULL, NULL, "2022",
	"Super Mario Bros. DX - Anniversary Edition (Hack, v4.2)\0", NULL, "Flamephanter", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_smbdxRomInfo, nes_smbdxRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// https://www.romhacking.net/hacks/7615/
static struct BurnRomInfo nes_smbroopRomDesc[] = {
	{ "Super Mario Bro-Op v1.1 (2023)(NesDraug).nes",          73744, 0x1ce829ca, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smbroop)
STD_ROM_FN(nes_smbroop)

struct BurnDriver BurnDrvnes_smbroop = {
	"nes_smbroop", "nes_smb", NULL, NULL, "2023",
	"Super Mario Bro-Op (Hack, v1.1)\0", NULL, "NesDraug", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_smbroopRomInfo, nes_smbroopRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_musheavenRomDesc[] = {
	{ "Mushroom Heaven (2022)(sinksadventure).nes",          40976, 0xd5ce9457, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_musheaven)
STD_ROM_FN(nes_musheaven)

struct BurnDriver BurnDrvnes_musheaven = {
	"nes_musheaven", "nes_smb", NULL, NULL, "2022",
	"Mushroom Heaven (Hack)\0", NULL, "sinksadventure", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_musheavenRomInfo, nes_musheavenRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_yunr5RomDesc[] = {
	{ "yun r5 (2018)(mojon twins).nes",          65552, 0x04940713, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_yunr5)
STD_ROM_FN(nes_yunr5)

struct BurnDriver BurnDrvnes_yunr5 = {
	"nes_yunr5", NULL, NULL, NULL, "2018",
	"Yun R5 (HB)\0", NULL, "The Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_yunr5RomInfo, nes_yunr5RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_demondistrictRomDesc[] = {
	{ "Demon District (HB).nes",          319504, 0x33e22dcb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_demondistrict)
STD_ROM_FN(nes_demondistrict)

struct BurnDriver BurnDrvnes_demondistrict = {
	"nes_demondistrict", NULL, NULL, NULL, "2019",
	"Demon District (HB)\0", NULL, "Klonoa", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_demondistrictRomInfo, nes_demondistrictRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_badappleRomDesc[] = {
	{ "bad_apple (2015 v2.5)(Little Limit).nes",          786448, 0xf96a5a8b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_badapple)
STD_ROM_FN(nes_badapple)

struct BurnDriver BurnDrvnes_badapple = {
	"nes_badapple", NULL, NULL, NULL, "2015",
	"Bad Apple (HB, Tech-Demo)\0", NULL, "Little Limit", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW | BDF_DEMO, 1, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_badappleRomInfo, nes_badappleRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cadaverionRomDesc[] = {
	{ "Cadaverion (2018)(Mojon Twins).nes",          40976, 0x6a45c96a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cadaverion)
STD_ROM_FN(nes_cadaverion)

struct BurnDriver BurnDrvnes_cadaverion = {
	"nes_cadaverion", NULL, NULL, NULL, "2018",
	"Cadaverion (HB)\0", NULL, "The Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE | GBF_ADV, 0,
	NESGetZipName, nes_cadaverionRomInfo, nes_cadaverionRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_chaseRomDesc[] = {
	{ "Chase (2012)(shiru).nes",          24592, 0xb887529d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chase)
STD_ROM_FN(nes_chase)

struct BurnDriver BurnDrvnes_chase = {
	"nes_chase", NULL, NULL, NULL, "2012",
	"Chase (HB)\0", NULL, "Shiru", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION | GBF_MAZE, 0,
	NESGetZipName, nes_chaseRomInfo, nes_chaseRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cherilpcRomDesc[] = {
	{ "Cheril Perils Classic (2018)(Mojon Twins).nes",          40976, 0x8132afd5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cherilpc)
STD_ROM_FN(nes_cherilpc)

struct BurnDriver BurnDrvnes_cherilpc = {
	"nes_cherilpc", NULL, NULL, NULL, "2018",
	"Cheril Perils Classic (HB)\0", NULL, "The Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_cherilpcRomInfo, nes_cherilpcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_driarRomDesc[] = {
	{ "Driar (2012)(Eriksson & Adolfsson).nes",          131088, 0x2c31ce23, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_driar)
STD_ROM_FN(nes_driar)

struct BurnDriver BurnDrvnes_driar = {
	"nes_driar", NULL, NULL, NULL, "2018",
	"Driar (HB)\0", NULL, "Eriksson & Adolfsson", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_driarRomInfo, nes_driarRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dushlanRomDesc[] = {
	{ "Dushlan (2016)(Peter McQuillan).nes",          40976, 0x07614f71, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dushlan)
STD_ROM_FN(nes_dushlan)

struct BurnDriver BurnDrvnes_dushlan = {
	"nes_dushlan", NULL, NULL, NULL, "2016",
	"Dushlan (HB)\0", NULL, "Peter McQuillan", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_dushlanRomInfo, nes_dushlanRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_sirababolrRomDesc[] = {
	{ "Sir Ababol Remastered Edition (2016)(Mojon Twins).nes",          40976, 0x36d8ca2b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sirababolr)
STD_ROM_FN(nes_sirababolr)

struct BurnDriver BurnDrvnes_sirababolr = {
	"nes_sirababolr", "nes_sirababol", NULL, NULL, "2016",
	"Sir Ababol Remastered Edition (HB)\0", NULL, "The Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_sirababolrRomInfo, nes_sirababolrRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_underadvRomDesc[] = {
	{ "Underground Adventure (2019)(Dale Coop).nes",          524304, 0xeaf67029, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_underadv)
STD_ROM_FN(nes_underadv)

struct BurnDriver BurnDrvnes_underadv = {
	"nes_underadv", NULL, NULL, NULL, "2019",
	"Underground Adventure (HB)\0", NULL, "Dale Coop", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_underadvRomInfo, nes_underadvRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_wxniaoRomDesc[] = {
	{ "Wo Xiang Niao Niao (2016)(Xin Xin).nes",          40976, 0x7063779c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_wxniao)
STD_ROM_FN(nes_wxniao)

struct BurnDriver BurnDrvnes_wxniao = {
	"nes_wxniao", NULL, NULL, NULL, "2016",
	"Wo Xiang Niao Niao (HB)\0", NULL, "Xin Xin", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_wxniaoRomInfo, nes_wxniaoRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_aspectstarnRomDesc[] = {
	{ "Aspect Star N (2019)(Nicole Express).nes",          24592, 0x14b70f98, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_aspectstarn)
STD_ROM_FN(nes_aspectstarn)

struct BurnDriver BurnDrvnes_aspectstarn = {
	"nes_aspectstarn", NULL, NULL, NULL, "2019",
	"Aspect Star N (HB)\0", NULL, "Nicole Express", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_aspectstarnRomInfo, nes_aspectstarnRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bloodfallRomDesc[] = {
	{ "Bloodfall (HB).nes",          40976, 0x3c5548ef, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bloodfall)
STD_ROM_FN(nes_bloodfall)

struct BurnDriver BurnDrvnes_bloodfall = {
	"nes_bloodfall", NULL, NULL, NULL, "2019",
	"Bloodfall (HB)\0", NULL, "Dr. Ludos", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_bloodfallRomInfo, nes_bloodfallRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_superhomwarRomDesc[] = {
	{ "Super Homebrew War (HB).nes",          65552, 0x16b8d50a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_superhomwar)
STD_ROM_FN(nes_superhomwar)

struct BurnDriver BurnDrvnes_superhomwar = {
	"nes_superhomwar", NULL, NULL, NULL, "2019",
	"Super Homebrew War (HB)\0", NULL, "gauauu", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 4, HARDWARE_NES, GBF_VSFIGHT, 0,
	NESGetZipName, nes_superhomwarRomInfo, nes_superhomwarRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_veggieinvadersRomDesc[] = {
	{ "Veggie Invaders (HB).nes",          65552, 0x6276c0a0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_veggieinvaders)
STD_ROM_FN(nes_veggieinvaders)

struct BurnDriver BurnDrvnes_veggieinvaders = {
	"nes_veggieinvaders", NULL, NULL, NULL, "2019",
	"Veggie Invaders (HB)\0", NULL, "dustmop", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_veggieinvadersRomInfo, nes_veggieinvadersRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Project Blue ROM @ https://toggleswitch.itch.io/projectblue
static struct BurnRomInfo nes_projectblueRomDesc[] = {
	{ "Project Blue (HB).nes",          524304, 0x43b9e57e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_projectblue)
STD_ROM_FN(nes_projectblue)

struct BurnDriver BurnDrvnes_projectblue = {
	"nes_projectblue", NULL, NULL, NULL, "2019",
	"Project Blue (HB)\0", NULL, "toggle switch", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_projectblueRomInfo, nes_projectblueRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_chemanRomDesc[] = {
	{ "Che-Man (HB).nes",          40976, 0x9677c8a1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cheman)
STD_ROM_FN(nes_cheman)

struct BurnDriver BurnDrvnes_cheman = {
	"nes_cheman", NULL, NULL, NULL, "2018",
	"Che-Man (HB)\0", NULL, "The Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_chemanRomInfo, nes_chemanRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_eeeaaaoooRomDesc[] = {
	{ "EEEAAAOOO (HB).nes",          589840, 0xea1cb7b6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_eeeaaaooo)
STD_ROM_FN(nes_eeeaaaooo)

struct BurnDriver BurnDrvnes_eeeaaaooo = {
	"nes_eeeaaaooo", NULL, NULL, NULL, "2020",
	"EEEAAAOOO - aka AAAEEEOOO (HB, Tech-Demo)\0", NULL, "N.K", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW | BDF_DEMO, 1, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_eeeaaaoooRomInfo, nes_eeeaaaoooRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_fallingtilesRomDesc[] = {
	{ "Falling Tiles (HB, rev.B).nes",          40976, 0x5b8fcd00, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_fallingtiles)
STD_ROM_FN(nes_fallingtiles)

struct BurnDriver BurnDrvnes_fallingtiles = {
	"nes_fallingtiles", NULL, NULL, NULL, "2020",
	"Falling Tiles (HB, rev.B)\0", NULL, "Doug Fraker", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_SHOOT, 0,
	NESGetZipName, nes_fallingtilesRomInfo, nes_fallingtilesRomName, NULL, NULL, NULL, NULL, NESZapperInputInfo, NESZapperDIPInfo,
	NESZapperInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_machinecaveRomDesc[] = {
	{ "Machine Cave (HB, v1.9).nes",          40976, 0x2d904baf, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_machinecave)
STD_ROM_FN(nes_machinecave)

struct BurnDriver BurnDrvnes_machinecave = {
	"nes_machinecave", NULL, NULL, NULL, "2018",
	"Machine Cave (HB, v1.9)\0", NULL, "Simo Sievanen", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_SHOOT | GBF_PUZZLE, 0,
	NESGetZipName, nes_machinecaveRomInfo, nes_machinecaveRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_wolflingRomDesc[] = {
	{ "Wolfling (HB, v1.4).nes",          65552, 0xf22ae9b8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_wolfling)
STD_ROM_FN(nes_wolfling)

struct BurnDriver BurnDrvnes_wolfling = {
	"nes_wolfling", NULL, NULL, NULL, "2018",
	"Wolfling (HB, v1.4)\0", NULL, "Lazy Cow", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_wolflingRomInfo, nes_wolflingRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_scrambleRomDesc[] = {
	{ "Scramble (HB).nes",          131088, 0xca6130a6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_scramble)
STD_ROM_FN(nes_scramble)

struct BurnDriver BurnDrvnes_scramble = {
	"nes_scramble", NULL, NULL, NULL, "2017",
	"Scramble (HB)\0", NULL, "Khan Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_HORSHOOT, 0,
	NESGetZipName, nes_scrambleRomInfo, nes_scrambleRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ultimfrochaRomDesc[] = {
	{ "Ultimate Frogger Champion (HB).nes",          49168, 0xab862073, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ultimfrocha)
STD_ROM_FN(nes_ultimfrocha)

struct BurnDriver BurnDrvnes_ultimfrocha = {
	"nes_ultimfrocha", NULL, NULL, NULL, "2019",
	"Ultimate Frogger Champion (HB)\0", NULL, "Khan Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_ultimfrochaRomInfo, nes_ultimfrochaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cowligam2ndadvRomDesc[] = {
	{ "Cowlitz Gamers 2nd Adventure (HB).nes",          131088, 0x0db2dd59, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cowligam2ndadv)
STD_ROM_FN(nes_cowligam2ndadv)

struct BurnDriver BurnDrvnes_cowligam2ndadv = {
	"nes_cowligam2ndadv", NULL, NULL, NULL, "2017",
	"Cowlitz Gamers 2nd Adventure (HB)\0", NULL, "Cowlitz", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_cowligam2ndadvRomInfo, nes_cowligam2ndadvRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cowligamadvRomDesc[] = {
	{ "Cowlitz Gamers Adventure (HB).nes",          32784, 0x0ae2354b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cowligamadv)
STD_ROM_FN(nes_cowligamadv)

struct BurnDriver BurnDrvnes_cowligamadv = {
	"nes_cowligamadv", NULL, NULL, NULL, "2016",
	"Cowlitz Gamers Adventure (HB)\0", NULL, "Cowlitz", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_cowligamadvRomInfo, nes_cowligamadvRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_madwizRomDesc[] = {
	{ "The Mad Wizard - A Candelabra Chronicle (HB).nes",          131088, 0x7750dac6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_madwiz)
STD_ROM_FN(nes_madwiz)

struct BurnDriver BurnDrvnes_madwiz = {
	"nes_madwiz", "nes_madwizsds", NULL, NULL, "2014",
	"Mad Wizard, The - A Candelabra Chronicle (HB)\0", NULL, "Robert L. Bryant", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_madwizRomInfo, nes_madwizRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_riseamoRomDesc[] = {
	{ "The Rise of Amondus (HB).nes",          131088, 0xb15ac0fe, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_riseamo)
STD_ROM_FN(nes_riseamo)

struct BurnDriver BurnDrvnes_riseamo = {
	"nes_riseamo", "nes_ramondus", NULL, NULL, "2015",
	"Rise of Amondus, The (HB)\0", NULL, "Robert L. Bryant", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_HORSHOOT, 0,
	NESGetZipName, nes_riseamoRomInfo, nes_riseamoRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_towerofturmoilRomDesc[] = {
	{ "Tower of Turmoil (HB, v1.03).nes",          524304, 0x89cbd840, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_towerofturmoil)
STD_ROM_FN(nes_towerofturmoil)

struct BurnDriver BurnDrvnes_towerofturmoil = {
	"nes_towerofturmoil", NULL, NULL, NULL, "2019",
	"Tower of Turmoil (HB, v1.03)\0", NULL, "CutterCross", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_towerofturmoilRomInfo, nes_towerofturmoilRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_jamgRomDesc[] = {
	{ "JAMG - The Lost Coins (HB).nes",          262160, 0x7a8877aa, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_jamg)
STD_ROM_FN(nes_jamg)

struct BurnDriver BurnDrvnes_jamg = {
	"nes_jamg", NULL, NULL, NULL, "2018",
	"JAMG - The Lost Coins (HB)\0", NULL, "OCDreproductions", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_jamgRomInfo, nes_jamgRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_pogocatsRomDesc[] = {
	{ "Pogo Cats (HB).nes",          24592, 0x57fc5391, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_pogocats)
STD_ROM_FN(nes_pogocats)

struct BurnDriver BurnDrvnes_pogocats = {
	"nes_pogocats", NULL, NULL, NULL, "2012",
	"Pogo Cats (HB)\0", NULL, "YGGI", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_pogocatsRomInfo, nes_pogocatsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_purplecapemanRomDesc[] = {
	{ "Purple Cape Man (HB).nes",          24592, 0xd8a7825b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_purplecapeman)
STD_ROM_FN(nes_purplecapeman)

struct BurnDriver BurnDrvnes_purplecapeman = {
	"nes_purplecapeman", NULL, NULL, NULL, "2015",
	"Purple Cape Man (HB)\0", NULL, "Vigilante Ninja", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_purplecapemanRomInfo, nes_purplecapemanRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_sgthelRomDesc[] = {
	{ "Sgt. Helmet - Training Day (HB).nes",          40976, 0x44c6ca4d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sgthel)
STD_ROM_FN(nes_sgthel)

struct BurnDriver BurnDrvnes_sgthel = {
	"nes_sgthel", NULL, NULL, NULL, "2013",
	"Sgt. Helmet - Training Day (HB)\0", NULL, "The Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_sgthelRomInfo, nes_sgthelRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_blackboxchaRomDesc[] = {
	{ "Black Box Challenge (HB).nes",          524304, 0x6b30c83b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_blackboxcha)
STD_ROM_FN(nes_blackboxcha)

struct BurnDriver BurnDrvnes_blackboxcha = {
	"nes_blackboxcha", NULL, NULL, NULL, "2017",
	"Black Box Challenge (HB)\0", NULL, "Sly Dog Studios", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_blackboxchaRomInfo, nes_blackboxchaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_flappybirdRomDesc[] = {
	{ "Flappy Bird (HB).nes",          24592, 0x3653c07d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_flappybird)
STD_ROM_FN(nes_flappybird)

struct BurnDriver BurnDrvnes_flappybird = {
	"nes_flappybird", NULL, NULL, NULL, "2014",
	"Flappy Bird (HB)\0", NULL, "Nioreh", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_flappybirdRomInfo, nes_flappybirdRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// https://lowtekgames.itch.io/fleanes
static struct BurnRomInfo nes_fleaRomDesc[] = {
	{ "flea! (HB).nes",          524304, 0xd208c6da, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_flea)
STD_ROM_FN(nes_flea)

struct BurnDriver BurnDrvnes_flea = {
	"nes_flea", NULL, NULL, NULL, "2020",
	"FLEA! (HB)\0", NULL, "Lowtek", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_fleaRomInfo, nes_fleaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_fleademoRomDesc[] = {
	{ "flea! demo (HB).nes",          524304, 0x6d18f7c4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_fleademo)
STD_ROM_FN(nes_fleademo)

struct BurnDriver BurnDrvnes_fleademo = {
	"nes_fleademo", "nes_flea", NULL, NULL, "2019",
	"FLEA! demo (HB)\0", NULL, "Lowtek", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_fleademoRomInfo, nes_fleademoRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_supertiltbroRomDesc[] = {
	{ "Super Tilt Bro (HB).nes",          524304, 0x58829008, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_supertiltbro)
STD_ROM_FN(nes_supertiltbro)

struct BurnDriver BurnDrvnes_supertiltbro = {
	"nes_supertiltbro", NULL, NULL, NULL, "2018",
	"Super Tilt Bro (HB, v2 alpha1)\0", NULL, "Broke Studio", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_VSFIGHT, 0,
	NESGetZipName, nes_supertiltbroRomInfo, nes_supertiltbroRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cmc80sdemoRomDesc[] = {
	{ "CMC 80's Demo (HB).nes",          49168, 0x0135b0e8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cmc80sdemo)
STD_ROM_FN(nes_cmc80sdemo)

struct BurnDriver BurnDrvnes_cmc80sdemo = {
	"nes_cmc80sdemo", NULL, NULL, NULL, "2000",
	"CMC 80's Demo (HB, Tech-Demo)\0", NULL, "Chris Covell", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW | BDF_DEMO, 1, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_cmc80sdemoRomInfo, nes_cmc80sdemoRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_etRomDesc[] = {
	{ "E.T. (2014)(Khan Games).nes",          131088, 0xc1878830, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_et)
STD_ROM_FN(nes_et)

struct BurnDriver BurnDrvnes_et = {
	"nes_et", NULL, NULL, NULL, "2014",
	"E.T. - The Extra Terrestrial (HB)\0", NULL, "Khan Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_etRomInfo, nes_etRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_larryaRomDesc[] = {
	{ "Larry (HB, alt).nes",          524304, 0x2c8d188e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_larrya)
STD_ROM_FN(nes_larrya)

struct BurnDriver BurnDrvnes_larrya = {
	"nes_larrya", "nes_larry", NULL, NULL, "2014",
	"Larry (HB, alt)\0", NULL, "Khan Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_larryaRomInfo, nes_larryaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_larryRomDesc[] = {
	{ "Larry (HB).nes",          524304, 0x96f31fb9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_larry)
STD_ROM_FN(nes_larry)

struct BurnDriver BurnDrvnes_larry = {
	"nes_larry", NULL, NULL, NULL, "2014",
	"Larry (HB)\0", NULL, "Khan Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_larryRomInfo, nes_larryRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_mrsplashRomDesc[] = {
	{ "Mr. Splash (HB).nes",          750052, 0x8b86978b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mrsplash)
STD_ROM_FN(nes_mrsplash)

struct BurnDriver BurnDrvnes_mrsplash = {
	"nes_mrsplash", NULL, NULL, NULL, "2008",
	"Mr. Splash (HB)\0", "No one player mode", "Project F", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_mrsplashRomInfo, nes_mrsplashRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_nescapeRomDesc[] = {
	{ "NEScape (HB).nes",          524304, 0x60b6ad99, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_nescape)
STD_ROM_FN(nes_nescape)

struct BurnDriver BurnDrvnes_nescape = {
	"nes_nescape", NULL, NULL, NULL, "2019",
	"NEScape (HB)\0", NULL, "Khan Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_nescapeRomInfo, nes_nescapeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_steinsgateRomDesc[] = {
	{ "Steins Gate (HB).nes",          786448, 0x547c4f1d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_steinsgate)
STD_ROM_FN(nes_steinsgate)

struct BurnDriver BurnDrvnes_steinsgate = {
	"nes_steinsgate", NULL, NULL, NULL, "2018",
	"Steins Gate (HB)\0", NULL, "Chiyomaru", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_steinsgateRomInfo, nes_steinsgateRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_boblRomDesc[] = {
	{ "bobl (HB).nes",          65552, 0x831956ea, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bobl)
STD_ROM_FN(nes_bobl)

struct BurnDriver BurnDrvnes_bobl = {
	"nes_bobl", "nes_boblv2", NULL, NULL, "2020",
	"BoBL (HB, v1.1)\0", NULL, "Morphcat Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_boblRomInfo, nes_boblRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_novasquRomDesc[] = {
	{ "Nova the Squirrel (HB).nes",          262160, 0x0b752bee, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_novasqu)
STD_ROM_FN(nes_novasqu)

struct BurnDriver BurnDrvnes_novasqu = {
	"nes_novasqu", NULL, NULL, NULL, "2018",
	"Nova the Squirrel (HB)\0", NULL, "NovaSquirrel", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_novasquRomInfo, nes_novasquRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_nallelandRomDesc[] = {
	{ "Nalle Land (HB, v040).nes",          49168, 0xc1a447b0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_nalleland)
STD_ROM_FN(nes_nalleland)

struct BurnDriver BurnDrvnes_nalleland = {
	"nes_nalleland", NULL, NULL, NULL, "2020",
	"Nalle Land (HB, v0.4.0)\0", NULL, "Benjamin Larsson", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_nallelandRomInfo, nes_nallelandRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_mysticoriginsRomDesc[] = {
	{ "Mystic Origins (HB).nes",          524304, 0x249c666e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mysticorigins)
STD_ROM_FN(nes_mysticorigins)

struct BurnDriver BurnDrvnes_mysticorigins = {
	"nes_mysticorigins", NULL, NULL, NULL, "2019",
	"Mystic Origins (HB)\0", NULL, "New 8-bit Heroes", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_mysticoriginsRomInfo, nes_mysticoriginsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_amazorundiejRomDesc[] = {
	{ "Amazons Running Diet (Japan).nes",          524304, 0x1eced454, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_amazorundiej)
STD_ROM_FN(nes_amazorundiej)

struct BurnDriver BurnDrvnes_amazorundiej = {
	"nes_amazorundiej", "nes_amazorundie", NULL, NULL, "2017",
	"Amazons Running Diet (HB, Japan)\0", NULL, "Ancient", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_amazorundiejRomInfo, nes_amazorundiejRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_amazorundieRomDesc[] = {
	{ "Amazons Running Diet (USA).nes",          524304, 0x4db6acdd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_amazorundie)
STD_ROM_FN(nes_amazorundie)

struct BurnDriver BurnDrvnes_amazorundie = {
	"nes_amazorundie", NULL, NULL, NULL, "2017",
	"Amazons Running Diet (HB, USA)\0", NULL, "Ancient", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_amazorundieRomInfo, nes_amazorundieRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_haradiuszeroRomDesc[] = {
	{ "Haradius Zero (HB).nes",          786448, 0x5ae4ccff, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_haradiuszero)
STD_ROM_FN(nes_haradiuszero)

struct BurnDriver BurnDrvnes_haradiuszero = {
	"nes_haradiuszero", NULL, NULL, NULL, "2019",
	"Haradius Zero (HB, v1.2a)\0", NULL, "Impact Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_HORSHOOT, 0,
	NESGetZipName, nes_haradiuszeroRomInfo, nes_haradiuszeroRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan,