/*	$NetBSD: md.c,v 1.11 2024/01/04 18:09:37 tsutsui Exp $ */

/*
 * Copyright 1997 Piermont Information Systems Inc.
 * All rights reserved.
 *
 * Based on code written by Philip A. Nelson for Piermont Information
 * Systems Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of Piermont Information Systems Inc. may not be used to endorse
 *    or promote products derived from this software without specific prior
 *    written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY PIERMONT INFORMATION SYSTEMS INC. ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL PIERMONT INFORMATION SYSTEMS INC. BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* md.c -- atari machine specific routines */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/param.h>
#include <sys/sysctl.h>
#include <sys/utsname.h>

#include "defs.h"
#include "md.h"
#include "msg_defs.h"
#include "menu_defs.h"

void
md_init(void)
{
}

void
md_init_set_status(int flags)
{
	struct utsname instsys;

	(void)flags;

	/*
	 * Get the name of the kernel we are running under and
	 * enable the installation of the corresponding GENERIC-like kernel.
	 */
	uname(&instsys);
	if (strstr(instsys.version, "(ATARITT"))
		set_kernel_set(SET_KERNEL_1);
	else if (strstr(instsys.version, "(FALCON"))
		set_kernel_set(SET_KERNEL_2);
	else if (strstr(instsys.version, "(SMALL030"))
		set_kernel_set(SET_KERNEL_3);
	else if (strstr(instsys.version, "(HADES"))
		set_kernel_set(SET_KERNEL_4);
	else if (strstr(instsys.version, "(MILAN-ISAIDE"))
		set_kernel_set(SET_KERNEL_5);
	else if (strstr(instsys.version, "(MILAN-PCIIDE"))
		set_kernel_set(SET_KERNEL_6);
}

bool
md_get_info(struct install_partition_desc *install)
{
	set_default_sizemult(pm->diskdev, MEG, pm->sectorsize);
	return true;
}

/*
 * md back-end code for menu-driven BSD disklabel editor.
 */
int
md_make_bsd_partitions(struct install_partition_desc *install)
{
	msg_fmt_display(MSG_infoahdilabel, "%s", pm->diskdev);
	if (ask_noyes(NULL)) {
		run_program(RUN_DISPLAY, "ahdilabel /dev/r%sc", pm->diskdev);
	}
	return make_bsd_partitions(install);
}

/*
 * any additional partition validation
 */
bool
md_check_partitions(struct install_partition_desc *install)
{
	return true;
}

/*
 * hook called before writing new disklabel.
 */
bool
md_pre_disklabel(struct install_partition_desc *install,
    struct disk_partitions *parts)
{
	return true;
}

/*
 * hook called after writing disklabel to new target disk.
 */
bool
md_post_disklabel(struct install_partition_desc *install,
    struct disk_partitions *parts)
{
	return true;
}

/*
 * hook called after upgrade() or install() has finished setting
 * up the target disk but immediately before the user is given the
 * ``disks are now set up'' message.
 */
int
md_post_newfs(struct install_partition_desc *install)
{
	static const int mib[2] = {CTL_HW, HW_MODEL};
	size_t len;
	char *cpu_model;
	int milan;
	char bootpath[MAXPATHLEN];
	int rv;

	/* check machine type via sysctl to select appropriate bootloaders */
	milan = 0;	/* std is default */
	sysctl(mib, 2, NULL, &len, NULL, 0);
	cpu_model = malloc(len);
	sysctl(mib, 2, cpu_model, &len, NULL, 0);
	/* XXX model strings should be a common macro to sync with kernel */
	if (strstr(cpu_model, "Milan") != NULL)
		milan = 1;
	free(cpu_model);

	/* copy tertiary boot and install boot blocks */
	msg_fmt_display(MSG_dobootblks, "%s", pm->diskdev);
	snprintf(bootpath, sizeof(bootpath), "/usr/mdec/%s/boot.atari",
	    milan ? "milan" : "std");
	rv = cp_to_target(bootpath, "/");
	if (rv != 0)
		return rv;

	rv = run_program(RUN_DISPLAY, "/usr/mdec/installboot -v%s /dev/r%sc",
	    milan ? "m" : "", pm->diskdev);

	return rv;
}

int
md_post_extract(struct install_partition_desc *install, bool upgrade)
{
	return 0;
}

void
md_cleanup_install(struct install_partition_desc *install)
{
#ifndef DEBUG
	enable_rc_conf();
#endif
}

int
md_pre_update(struct install_partition_desc *install)
{
	return 1;
}

/* Upgrade support */
int
md_update(struct install_partition_desc *install)
{
	md_post_newfs(install);
	return 1;
}

int
md_pre_mount(struct install_partition_desc *install, size_t ndx)
{
	return 0;
}

bool
md_parts_use_wholedisk(struct disk_partitions *parts)
{
	return parts_use_wholedisk(parts, 0, NULL);
}

#ifdef HAVE_GPT
bool
md_gpt_post_write(struct disk_partitions *parts, part_id root_id,
    bool root_is_new, part_id efi_id, bool efi_is_new)
{
	/* no GPT boot support, no special checks needed */
	return true;
}
#endif

